/***********************************************************************************
 File:			HwQManager.c
 Module:		Hw Q Manager
 Purpose:		
 Description:	FW driver for Hw Q Manager Module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "HwQManager_API.h"
#include "DlmRegs.h"
#include "RegAccess_Api.h"
#include "PacketDescriptor.h"
#include "HostInterface_API.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ShramPacketDescriptors.h"
#include "HwMemoryMap.h"
#include "shramTxDesc.h"
#include "QManagerUmacRegs.h"
#include "QManagerLmacRegs.h"
#include "stringLibApi.h"
#include "ShramDataTxDescriptors.h"
#include "int_gen.h"
#include "Pac_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 8

typedef enum RegInterfaceIndex
{
	HW_Q_MANAGER_CONTROL_REG_INDEX,
	HW_Q_MANAGER_ADDR_REG_INDEX,
	HW_Q_MANAGER_RETURN_ADDR_REG_INDEX,
	HW_Q_MANAGER_FIFO_STATUS_REG_INDEX,
	HW_Q_MANAGER_TOTAL_NUM_OF_REG_INDEXES,
} RegInterfaceIndex_e;

typedef enum
{
	HW_Q_MANAGER_DISABLE_LIST_HW_0_MASTER = 0,
	HW_Q_MANAGER_DISABLE_LIST_HW_1_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_TX_GENRISC_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_HOST_GENRISC_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_RX_GENRISC_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_UMAC_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_LMAC_MASTER,
	HW_Q_MANAGER_NUM_OF_DISABLE_LISTS_MASTERS,
	HW_Q_MANAGER_MAX_NUM_OF_DISABLE_LISTS_MASTERS = 0xFF,
} HwQManagerDisableListMaster_e;


#define DLM_RAM_INIT_VALUE			(0xFFFFFFFF)
#ifndef TX_DATA_UM_PATH
#define HW_Q_MANAGER_TX_ENABLED_Q  ((1 << HW_Q_MANAGER_DONE_LIST_HOST_DATA) |\
									(1 << HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS) |\
									(1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW) |\
                                    (1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST) |\
                                	(1 << HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST) |\
									(1 << HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW) |\
                                	(1 << HW_Q_MANAGER_LOW_PR_READY_LIST_AGER) |\
									(1 << HW_Q_MANAGER_READY_LIST_MULTICAST_PD) |\
									(1 << HW_Q_MANAGER_READY_LIST_UNICAST_PD))
#else
#define HW_Q_MANAGER_TX_ENABLED_Q  ((1 << HW_Q_MANAGER_DONE_LIST_HOST_DATA) |\
									(1 << HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS) |\
									(1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW) |\
                                    (1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST) |\
                                	(1 << HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST) |\
									(1 << HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW) |\
                                	(1 << HW_Q_MANAGER_LOW_PR_READY_LIST_AGER) |\
									(1 << HW_Q_MANAGER_READY_LIST_UNICAST_PD))
#endif

                                	

#define HW_Q_MANAGER_TX_DISABLE_LISTS_ENABLED_Q ((1 << HW_Q_MANAGER_DISABLE_LIST_HW_0_MASTER) |\
												 (1 << HW_Q_MANAGER_DISABLE_LIST_HW_1_MASTER) |\
												 (1 << HW_Q_MANAGER_DISABLE_LIST_TX_GENRISC_MASTER) |\
												 (1 << HW_Q_MANAGER_DISABLE_LIST_HOST_GENRISC_MASTER) |\
												 (1 << HW_Q_MANAGER_DISABLE_LIST_RX_GENRISC_MASTER) |\
												 (1 << HW_Q_MANAGER_DISABLE_LIST_UMAC_MASTER) |\
												 (1 << HW_Q_MANAGER_DISABLE_LIST_LMAC_MASTER))

#if defined (RX_LIST_DEBUG)
#define HW_Q_MANAGER_RX_ENABLED_Q  ((1<< HW_Q_MANAGER_RX_READY_LIST_FW)\
                                	|(1<<HW_Q_MANAGER_RX_READY_LIST_DRIVER)\
                               	    |(1<<HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW)\
                                    |(1<<HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS)\
                                    |(1<<HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS)\
                                    |(1<<HW_Q_MANAGER_RX_READY_LIST_DEBUG)\
									|(1<<HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP))
#else
#define HW_Q_MANAGER_RX_ENABLED_Q  ((1<< HW_Q_MANAGER_RX_READY_LIST_FW)\
                                	|(1<<HW_Q_MANAGER_RX_READY_LIST_DRIVER)\
                               	    |(1<<HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW)\
                                    |(1<<HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS)\
                                    |(1<<HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS))
        
#endif

#define HW_Q_MANAGER_CLEAR_INT		(1)

#define SET_NULL_TO_NEXT (1)
#define DO_NOT_SET_NULL_TO_NEXT (0)
#define DLM_SIZE_WITHOUT_RERGISTERS (0x2700)
#define DLM_TOTAL_SIZE_IN_WORDS (CONVERT_BYTES_TO_WORDS(DLM_SIZE_WITHOUT_RERGISTERS))

/* Ager section */
#define AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_CLIENTS 8
#define AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_SHRAM_ACCESS (50) /* 4 */
#define AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT 7
#define AGER_SINGLE_CHECK_IS_VAP_QUEUE_MASK (1 << AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT) 
#define AGER_CLEAR_DISCARDED_PACKETS_TO_UMAC_COUNTER_MASK 0x1
#define AGER_CLEAR_DISCARDED_PACKETS_TO_LIBERATOR_COUNTER_MASK 0x2
#define AGER_CLEAR_LOCK_FAILED_COUNTER_MASK 0x4
#define AGER_CLEAR_ALL_COUNTERS_MASK (AGER_CLEAR_DISCARDED_PACKETS_TO_UMAC_COUNTER_MASK |\
									  AGER_CLEAR_DISCARDED_PACKETS_TO_LIBERATOR_COUNTER_MASK |\
									  AGER_CLEAR_LOCK_FAILED_COUNTER_MASK)

#define HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL 1000

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void hwQManagerPushToQ(HwQMangerRequestType_e requestType, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams);
void hwQManagerPop(HwQMangerRequestType_e requestType, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams);
void hwQManagerFillControlRegFields(RegQManagerUmacSwMaster3Ctl_u *controlReg, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, HwQMangerRequestType_e requestType);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
const uint32 regInterfaceAddresses[HW_Q_MANAGER_TOTAL_NUM_OF_REG_IF][HW_Q_MANAGER_TOTAL_NUM_OF_REG_INDEXES] =
{
#if defined (ENET_INC_UMAC)
	{REG_Q_MANAGER_UMAC_SW_MASTER3_CTL,REG_Q_MANAGER_UMAC_SW_MASTER3_ADDR,REG_Q_MANAGER_UMAC_SW_MASTER3_RETURN_ADDR,REG_Q_MANAGER_UMAC_SW_MASTER3_FIFO_STATUS},
	{REG_Q_MANAGER_UMAC_SW_MASTER3_CTL,REG_Q_MANAGER_UMAC_SW_MASTER3_ADDR,REG_Q_MANAGER_UMAC_SW_MASTER3_RETURN_ADDR,REG_Q_MANAGER_UMAC_SW_MASTER3_FIFO_STATUS},
#else 
	{REG_Q_MANAGER_LMAC_SW_MASTER4_CTL,REG_Q_MANAGER_LMAC_SW_MASTER4_ADDR,REG_Q_MANAGER_LMAC_SW_MASTER4_RETURN_ADDR,REG_Q_MANAGER_LMAC_SW_MASTER4_FIFO_STATUS},
	{REG_Q_MANAGER_LMAC_SW_MASTER4_CTL,REG_Q_MANAGER_LMAC_SW_MASTER4_ADDR,REG_Q_MANAGER_LMAC_SW_MASTER4_RETURN_ADDR,REG_Q_MANAGER_LMAC_SW_MASTER4_FIFO_STATUS},
#endif //ENET_INC_UMAC
};

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

hwQManagerPushToQ 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQMangerRequestType_e requestType
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerPushToQ(HwQMangerRequestType_e requestType,HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	TX_INTERRUPT_SAVE_AREA;
	/* These registers are identical for LMAC and UMAC */
	RegQManagerUmacSwMaster3Ctl_u controlReg;
	RegQManagerUmacSwMaster3Addr_u addrReg;
	RegQManagerUmacSwMaster3FifoStatus_u fifoStatusReg;
	uint32 controlRegAddress = 0;
	uint32 addrRegAddress = 0;
	uint32 fifoStatusRegAddress = 0;
	uint32 timeStamp=0;
	controlReg.val = 0;
	addrReg.val = 0;

	/* Fill Control Register values*/
	hwQManagerFillControlRegFields(&controlReg,hwQueueManagerRequestParams,requestType);
	/* Fill Address Register - Holds the pointer of the required PD to enqueue */
	addrReg.bitFields.topAddr = ((uint32*)(hwQueueManagerRequestParams->pHeadDesc)) - PacketDescriptorsPtr;
	
	if(hwQueueManagerRequestParams->pTailDesc != NULL)
	{
		addrReg.bitFields.bottomAddr = ((uint32*)(hwQueueManagerRequestParams->pTailDesc)) - PacketDescriptorsPtr;//For Single Packet Should Be Null
	}

	/* Set the required registers addresses */
	controlRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_CONTROL_REG_INDEX];
	addrRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_ADDR_REG_INDEX];
	fifoStatusRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_FIFO_STATUS_REG_INDEX];

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	//Debug asserts:
	//1. Make sure head pointer is not 0xFFFF
	DEBUG_ASSERT(addrReg.bitFields.topAddr != NEXT_PD_NULL);

	//2. Make sure that in case we are trying to push a list the tail pointer is not 0xFFFF
	DEBUG_ASSERT(((requestType != HW_Q_MANAGER_PUSH_PACKET_LIST_TO_TAIL) &&	(requestType != HW_Q_MANAGER_PUSH_PACKET_LIST_TO_HEAD)) || (addrReg.bitFields.bottomAddr != NEXT_PD_NULL));

	RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);

	timeStamp = GET_TSF_TIMER_LOW();
	while (fifoStatusReg.bitFields.swFifoFull == 1)
	{
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL); 
		RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);
	}

	RegAccess_Write(addrRegAddress,addrReg.val);
	RegAccess_Write(controlRegAddress,controlReg.val);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);

}

/**********************************************************************************

hwQManagerPop 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQMangerRequestType_e requestType
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerPop(HwQMangerRequestType_e requestType,HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	TX_INTERRUPT_SAVE_AREA;
	/* These registers are identical for LMAC and UMAC */
	RegQManagerUmacSwMaster3Ctl_u controlReg;
	RegQManagerUmacSwMaster3Addr_u addrReg;
	RegQManagerUmacSwMaster3ReturnAddr_u returnAddrReg;
	RegQManagerUmacSwMaster3FifoStatus_u fifoStatusReg;
	uint32 controlRegAddress = 0;
	uint32 returnAddrRegAddress = 0;
	uint32 addrRegAddress = 0;
	uint32 fifoStatusRegAddress = 0;
	uint32 timeStamp = 0;
	/* Fill Control Register values*/
	hwQManagerFillControlRegFields(&controlReg,hwQueueManagerRequestParams,requestType);

	/* Set the required registers addresses */
	addrRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_ADDR_REG_INDEX];
	controlRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_CONTROL_REG_INDEX];
	returnAddrRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_RETURN_ADDR_REG_INDEX]; 
	fifoStatusRegAddress = regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_FIFO_STATUS_REG_INDEX];

	addrReg.bitFields.topAddr = ((uint32*)(hwQueueManagerRequestParams->pHeadDesc)) - PacketDescriptorsPtr;

	if(hwQueueManagerRequestParams->pTailDesc != NULL)
	{
		addrReg.bitFields.bottomAddr = ((uint32*)(hwQueueManagerRequestParams->pTailDesc)) - PacketDescriptorsPtr;//For Single Packet Should Be Null
	}

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);
	timeStamp = GET_TSF_TIMER_LOW();
	while (fifoStatusReg.bitFields.swFifoFull == 1)
	{
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL);
		RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);		 
	}
	RegAccess_Write(addrRegAddress,addrReg.val);
	RegAccess_Write(controlRegAddress,controlReg.val);
	RegAccess_Read(returnAddrRegAddress, &returnAddrReg.val);
	OSAL_ENABLE_INTERRUPTS(interrupt_save); 
	
	/* Fill Addr Register - Holds the pointer of the required PD to enqueue */
	/* In order to calculate the address - the base adress is added with word offset whose result is the exact address since the base address in uint32* pointer
	- no further manipulation is needed */
	hwQueueManagerRequestParams->pHeadDesc  = PacketDescriptorsPtr  + returnAddrReg.bitFields.rtnHeadPdAddr;
	hwQueueManagerRequestParams->pTailDesc  = PacketDescriptorsPtr + returnAddrReg.bitFields.rtnTailPdAddr; /* For single PD value is unknown*/
}

/**********************************************************************************

hwQManagerFillControlRegFields 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	RegHwQManagerControl_u * controlReg
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerFillControlRegFields(RegQManagerUmacSwMaster3Ctl_u *controlReg,HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, HwQMangerRequestType_e requestType)
{
	/* Fill Control Register values - the register is identical for LMAC and UMAC */
	controlReg->bitFields.ctlPrimaryAddr = hwQueueManagerRequestParams->primaryAddr;
	controlReg->bitFields.ctlSecondaryAddr = hwQueueManagerRequestParams->secondaryAddr;

	controlReg->bitFields.ctlDplIdx = hwQueueManagerRequestParams->dplIndex;
	controlReg->bitFields.ctlDlmIdx = hwQueueManagerRequestParams->dlmNum;
	controlReg->bitFields.ctlSetNull = hwQueueManagerRequestParams->setNull;
	controlReg->bitFields.ctlReq = requestType;
}

#if defined(ENET_INC_UMAC)
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void hwQManagerInitDlm(void)
{
	/* The register is identical for Tx and Rx */
	RegDlmDlm0DescBaseAddr_u descBaseAddr;
	uint8 *ramAddr;

	descBaseAddr.val = 0;

	/* Configure all dlm sections - all dlm sections are sequential */
	ramAddr = (uint8 *)DLM_BASE_ADDR;
	memset32(ramAddr,DLM_RAM_INIT_VALUE,DLM_TOTAL_SIZE_IN_WORDS); /* Set DLM's ram to 0xFF*/

	/* Configure descriptors base address - identical for Rx and Tx */
	descBaseAddr.val = 0;
	descBaseAddr.bitFields.dlm0DescBaseAddr = CONVERT_BYTES_TO_WORDS((uint32)PacketDescriptorsPtr);
		
	RegAccess_Write(REG_DLM_DLM0_DESC_BASE_ADDR,descBaseAddr.val);
	RegAccess_Write(REG_DLM_DLM2_DESC_BASE_ADDR,descBaseAddr.val);

}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

#endif /* #if defined(ENET_INC_UMAC)  */
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

HwQManager_PushPacketToTail 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketToTail(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->pTailDesc = NULL;
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_TO_TAIL,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PushPacketListToTail 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketListToTail(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_LIST_TO_TAIL,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PushPacketToHead 

Description:
------------
	Push PD to the head of the required Q

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketToHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->pTailDesc = NULL;
	hwQueueManagerRequestParams->setNull = DO_NOT_SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_TO_HEAD,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PushPacketlistToHead 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketlistToHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = DO_NOT_SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_LIST_TO_HEAD,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PopPacket 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PopPacket(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPop(HW_Q_MANAGER_POP_PACKET,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PopListFromHead 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams - Request Parametes (DLM Index, Queue Index,Request Type etc)
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PopListFromHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPop(HW_Q_MANAGER_POP_PACKET_LIST,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PeekHeadPacket 

Description:
------------
	Returns the First PD address on the required Q without removing the PD from Q

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams - Request Parametes (DLM Index, Queue Index,Request Type etc)
Output:
-------
	Head pointer value
Returns:
--------
	void 
**********************************************************************************/
void HwQManager_PeekHeadPacket(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQManagerPop(HW_Q_MANAGER_PEEK_PACKET,hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_FlushQ 

Description:
------------
	Remove all PD from required Queue  

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	Head & Tail pointers to the list flushed from the Q 
Returns:
--------
	void 
**********************************************************************************/
void HwQManager_FlushQ(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQManagerPop(HW_Q_MANAGER_FLUSH,hwQueueManagerRequestParams);
}


/**********************************************************************************

HwQManager_GetLowPriorityTxPdReadyListsStatus 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	uint32 * highPriorityTxPdReadylistStatus
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_GetLowPriorityTxPdReadyListsStatus(uint32 *highPriorityTxPdReadylistStatus)
{
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LO_PRI_TX_READY_INT,highPriorityTxPdReadylistStatus);
}

/**********************************************************************************

HwQManager_GetHighPriorityTxPdReadyListsStatus 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	uint32 * lowPriorityTxPdReadylistStatus
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_GetHighPriorityTxPdReadyListsStatus(uint32 *lowPriorityTxPdReadylistStatus)
{
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_HI_PRI_TX_READY_INT,lowPriorityTxPdReadylistStatus);
}

/**********************************************************************************

HwQManager_ClearTxListsInt 

Description:
------------
	clear interrupt of a specific Tx list

Input:
-----
	txListIndex - the index of the list

Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_ClearTxListsInt(uint32 txListIndex)
{
	RegQManagerUmacSwMaster3TxIntClr_u txIntClr;

	txIntClr.val = 1 << txListIndex;

	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_TX_INT_CLR,txIntClr.val);
}

/**********************************************************************************

HwQManager_ClearDisableListsInt 

Description:
------------
clear interrupt of a specific disable list

Input:
-----
disableListIndex - the index of the list

Output:
-------
	
Returns:
--------
	
**********************************************************************************/
void HwQManager_ClearDisableListsInt(uint32 disableListIndex)
{
	RegQManagerUmacSwMaster3DisableIntClr_u disableListClearInterruptRegister;
	
	disableListClearInterruptRegister.val = 0;

	disableListClearInterruptRegister.val = 1 << disableListIndex;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_DISABLE_INT_CLR,disableListClearInterruptRegister.val);
}


/**********************************************************************************

HwQManager_ClearRxListsInt 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQManagerIntId_e intId
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_ClearRxListsInt(uint32  rxListIndex)
{
	RegQManagerUmacSwMaster3RxIntClr_u rxIntClr;

	rxIntClr.val = 1 << rxListIndex;

	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_RX_INT_CLR,rxIntClr.val);
}


/**********************************************************************************

HwQManager_ErrorEvent 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	void
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_ErrorEvent(void)
{
	RegQManagerUmacSwMaster3Err_u errorReg;

	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_ERR, &errorReg.val);

	ILOG2_D("HwQManager_ErrorEvent: 0x%X ",errorReg.val);
	
	FATAL("HW Q Manager Error Event");
}

/**********************************************************************************

HwQManager_GetEmptyTxDoneLists 

Description:
------------
	return bitmap of tx done list that became empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became empty
**********************************************************************************/
uint16 HwQManager_GetEmptyTxDoneLists(void)
{
	RegQManagerUmacSwMaster3TxDoneInt_u doneListRegister;
	doneListRegister.val = 0;
	
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_TX_DONE_INT, &doneListRegister.val);
	DEBUG_ASSERT(!doneListRegister.bitFields.txDoneListsEmptyInt);

	return doneListRegister.bitFields.txDoneListsEmptyInt;
}


/**********************************************************************************

HwQManager_GetNotEmptyTxDoneLists 

Description:
------------
	return bitmap of tx done list that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxDoneLists(void)
{
	RegQManagerUmacSwMaster3TxDoneInt_u doneListsRegister;
	doneListsRegister.val = 0;
		
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_TX_DONE_INT, &doneListsRegister.val);

	DEBUG_ASSERT(doneListsRegister.bitFields.txDoneListsNotEmptyInt);

	return doneListsRegister.bitFields.txDoneListsNotEmptyInt;
}

/**********************************************************************************

HwQManager_GetNotEmptyTxDisableLists 

Description:
------------
return bitmap of tx disable lists that became not empty

Input:
-----
	
Output:
-------
	
Returns:
--------
bitmap of tx disable lists that became not empty

**********************************************************************************/

uint16 HwQManager_GetNotEmptyTxDisableLists(void)
{
	RegQManagerUmacSwMaster3DisableInt_u disableListsRegister;
	disableListsRegister.val = 0;
		
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_DISABLE_INT, &disableListsRegister.val);

    //  DEBUG_ASSERT(disableListsRegister.bitFields.umacDisQueueNotEmptyInt == (disableListsRegister.bitFields.umacDisQueueNotEmptyInt & HW_Q_MANAGER_TX_DISABLE_LISTS_ENABLED_Q));
    //  Due to FW W/A for HW bug (WLANSW-3026), the only master that can assert Tx disabled List interrupt is the UMAC.
        DEBUG_ASSERT(disableListsRegister.bitFields.umacDisQueueNotEmptyInt == (disableListsRegister.bitFields.umacDisQueueNotEmptyInt & (1 << HW_Q_MANAGER_DISABLE_LIST_UMAC_MASTER)));
	
	return disableListsRegister.bitFields.umacDisQueueNotEmptyInt;
}


/**********************************************************************************

HwQManager_GetNotEmptyTxReadyLoPriLists 

Description:
------------
	return bitmap of tx ready low priority list that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxReadyLoPriLists(void)
{
	RegQManagerUmacSwMaster3LoPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
		
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LO_PRI_TX_READY_INT, &readyListRegister.val);
	return readyListRegister.bitFields.txLoPriReadyNotEmptyInt;
}

/**********************************************************************************

HwQManager_GetEmptyTxReadyLoPriLists 

Description:
------------
return bitmap of tx ready low priority lists

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of txready low priority lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetEmptyTxReadyLoPriLists(void)
{
	RegQManagerUmacSwMaster3LoPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
				   
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LO_PRI_TX_READY_INT, &readyListRegister.val);
	return readyListRegister.bitFields.txLoPriReadyEmptyInt;
}

/**********************************************************************************

HwQManager_GetNotEmptyTxReadyHiPriLists 

Description:
------------
	return bitmap of tx ready high priority list that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxReadyHiPriLists(void)
{
	RegQManagerUmacSwMaster3HiPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
		
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_HI_PRI_TX_READY_INT, &readyListRegister.val);
	return readyListRegister.bitFields.txHiPriReadyNotEmptyInt;
}

/**********************************************************************************

HwQManager_GetEmptyTxReadyLowPriLists 

Description:
------------
return bitmap of tx ready low priority lists

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of txready low priority lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetEmptyTxReadyHiPriLists(void)
{
	RegQManagerUmacSwMaster3HiPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
				   
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_HI_PRI_TX_READY_INT, &readyListRegister.val);
	return readyListRegister.bitFields.txHiPriReadyEmptyInt;
}


/**********************************************************************************

HwQManager_GetNotEmptyRxReadyLists 

Description:
------------
return bitmap of rx ready lists (RDs that were handled to FW by HW) that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of rx ready lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyRxReadyLists(void)
{
	RegQManagerUmacSwMaster3HiPriRxReadyInt_u rxReadyRdListRegister;
	rxReadyRdListRegister.val = 0;
				   
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_HI_PRI_RX_READY_INT, &rxReadyRdListRegister.val);
	return rxReadyRdListRegister.bitFields.rxHiPriReadyNotEmptyInt;
}

/**********************************************************************************

HwQManager_RemovePdFromQueue 

Description:
------------
removes provided pd from queue specified in hwQueueManagerRequestParams.

Input:
-----
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams - contains queue description
	TxPd_t* pd - pointer to pd that must be removed from queue
	
Output:
-------
	
Returns:
--------
	Returns TRUE if pd was found in queue and successfully removed, FALSE otherwise
**********************************************************************************/
bool HwQManager_RemovePdFromQueue(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, TxPd_t* removePd)
{
	TxPd_t*				prevPd = (TxPd_t *)NULL_PD;
	TxPd_t*				pd;
	bool							pdFound = FALSE;

	HwQManager_FlushQ(hwQueueManagerRequestParams);

	// Try to find the PD in the list
	if (hwQueueManagerRequestParams->pHeadDesc != (TxPd_t *)NULL_PD)
	{
		pd = hwQueueManagerRequestParams->pHeadDesc;
		while (pd != (TxPd_t *)NULL_PD)
		{
			if (pd == removePd)
			{
				// PD was found
				pdFound = TRUE;
				if (prevPd != (TxPd_t *)NULL_PD)
				{
					prevPd->nextPdPointer = SET_NEXT_PD(GET_NEXT_PD(pd));
				}
				else
				{
					// PD was head of queue
					hwQueueManagerRequestParams->pHeadDesc = (void *)GET_NEXT_PD(pd);
				}
				if (pd->nextPdPointer == NEXT_PD_NULL)
				{
					// PD was tail of queue
					hwQueueManagerRequestParams->pTailDesc = (void *)prevPd;
				}
				if(!((pd->pdType == PD_TYPE_CONTROL_UNENC) && (pd->ctrlSubtype == PD_CONTROL_SUBTYPE_CTS)))

				{
					// Restore DRAM pointer for all Pds except Control Unenc - this pd has no Dram field
					pd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(pd->packetPointer);  
				}

				// Quit the loop
				break;
			}
			prevPd = pd;
			pd = (TxPd_t *)GET_NEXT_PD(pd);
		}

		// Push the list back to STA queue - we may have removed the only PD so check we have something in list
		if (hwQueueManagerRequestParams->pHeadDesc != (TxPd_t *)NULL_PD)
		{
			HwQManager_PushPacketlistToHead(hwQueueManagerRequestParams); 
		}
	}

	return pdFound;
}

#ifdef ENET_INC_UMAC
/**********************************************************************************

HwQManagerAger_ConfigureAllAndActivatePeriodicCheck 


Description:
------------
Configure all parameters and activate periodic check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the periodic check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_ConfigureAllAndActivatePeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters)
{
	RegQManagerUmacSwMaster3AgerCfg_u globalParametersRegister;
	RegQManagerUmacSwMaster3AgerAcLeaveCriteria_u acsCriteriaRegister;
	RegQManagerUmacSwMaster3AgerAcThreshCriteria01_u ac01MinimumPdsThreshold;
	RegQManagerUmacSwMaster3AgerAcThreshCriteria23_u ac23MinimumPdsThreshold;
	RegQManagerUmacSwMaster3AgerEn_u  periodicCheckControlRegister;
	
	globalParametersRegister.val = 0;
	acsCriteriaRegister.val = 0;
	ac01MinimumPdsThreshold.val = 0;
	ac23MinimumPdsThreshold.val = 0;
	periodicCheckControlRegister.val = 0;

    /* Update only thr Ager period (in TTL units) */
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CFG, &globalParametersRegister.val);
    globalParametersRegister.bitFields.agerPeriod = agerCheckParameters->agerPeriod;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CFG, globalParametersRegister.val);
	
    /* Update all the values together */
	//acsCriteriaRegister.val = *((uint32 *)(agerCheckParameters->generalParameters.acTtlCriteria));
	memcpy(&acsCriteriaRegister.val, &agerCheckParameters->generalParameters.acTtlCriteria[0], sizeof(uint32));
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_LEAVE_CRITERIA, acsCriteriaRegister.val);

    /* Update minimum PDs threshold */
	ac01MinimumPdsThreshold.bitFields.beThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BE];
	ac01MinimumPdsThreshold.bitFields.bkThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BK];
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_THRESH_CRITERIA_0_1, ac01MinimumPdsThreshold.val);
	
	ac23MinimumPdsThreshold.bitFields.viThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI];
	ac23MinimumPdsThreshold.bitFields.voThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VO];
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_THRESH_CRITERIA_2_3, ac23MinimumPdsThreshold.val);
	
	periodicCheckControlRegister.bitFields.agerEn = TRUE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_EN, periodicCheckControlRegister.val);	
}

/**********************************************************************************

HwQManagerAger_ReconfigurePartialPeriodicCheck 


Description:
------------
Stop the periodic check and reconfigure only the TTL criteria and the ager period 
and then activate the Ager periodic check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the periodic check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_ReconfigurePartialPeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters)
{
	RegQManagerUmacSwMaster3AgerCfg_u globalParametersRegister;
	RegQManagerUmacSwMaster3AgerAcLeaveCriteria_u acsCriteriaRegister;
	RegQManagerUmacSwMaster3AgerEn_u  periodicCheckControlRegister;
	
	globalParametersRegister.val = 0;
	acsCriteriaRegister.val = 0;
	periodicCheckControlRegister.val = 0;

    /* First stop the periodic check if needed */ 
	HwQManagerAger_StopPeriodicCheck();

    /* Update only the Ager period (in TTL units) */
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CFG, &globalParametersRegister.val);
    globalParametersRegister.bitFields.agerPeriod = agerCheckParameters->agerPeriod;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CFG, globalParametersRegister.val);
	
    /* Update all the values together */
	//acsCriteriaRegister.val = *((uint32 *)(agerCheckParameters->generalParameters.acTtlCriteria));
	memcpy(&acsCriteriaRegister.val, &agerCheckParameters->generalParameters.acTtlCriteria[0], sizeof(uint32));
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_LEAVE_CRITERIA, acsCriteriaRegister.val);

	periodicCheckControlRegister.bitFields.agerEn = TRUE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_EN, periodicCheckControlRegister.val);	
}



/**********************************************************************************

HwQManagerAger_StopPeriodicCheck 


Description:
------------
Stops Ager periodic check

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_StopPeriodicCheck(void)
{
	RegQManagerUmacSwMaster3AgerEn_u  periodicCheckControlRegister;
	RegQManagerUmacSwMaster3AgerCheckStatus_u statusRegister;
	
	periodicCheckControlRegister.val = 0;
	statusRegister.val = 0;

	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_EN, &periodicCheckControlRegister.val);

	periodicCheckControlRegister.bitFields.agerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_EN, periodicCheckControlRegister.val);	

	statusRegister.bitFields.singleCheckInWait = TRUE;
	while(statusRegister.bitFields.singleCheckInWait)
	{
		RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CHECK_STATUS, &statusRegister.val);
	}
}


/**********************************************************************************

HwQManagerAger_StartSingleCheck 


Description:
------------
Activates Ager single check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the single check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_StartSingleCheck(AgerSingleCheckParameters_t  *agerCheckParameters, bool restrictedAc)
{
	RegQManagerUmacSwMaster3AgerAcLeaveCriteria_u acsCriteriaRegister;
	RegQManagerUmacSwMaster3AgerAcThreshCriteria01_u ac01MinimumPdsThreshold;
	RegQManagerUmacSwMaster3AgerAcThreshCriteria23_u ac23MinimumPdsThreshold;
	RegQManagerUmacSwMaster3AgerSingleCheckCfg_u singleCheckConfigurationRegister;
	RegQManagerUmacSwMaster3AgerSinglePdLimit_u singleCheckPdsLimitRegister;
	RegQManagerUmacSwMaster3AgerSingleCheckActivate_u singleCheckControlRegister;
	RegQManagerUmacSwMaster3AgerCounterClr_u clearCountersRegister;
	
	acsCriteriaRegister.val = 0;
	ac01MinimumPdsThreshold.val = 0;
	ac23MinimumPdsThreshold.val = 0;
	singleCheckConfigurationRegister.val = 0;
	singleCheckPdsLimitRegister.val = 0;
	singleCheckControlRegister.val = 0;	
	clearCountersRegister.val = 0;

    clearCountersRegister.val = AGER_CLEAR_ALL_COUNTERS_MASK;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_COUNTER_CLR, clearCountersRegister.val);
	
    /* Update all the values together */
	//acsCriteriaRegister.val = *((uint32 *)(agerCheckParameters->generalParameters.acTtlCriteria));
	memcpy(&acsCriteriaRegister.val, &agerCheckParameters->generalParameters.acTtlCriteria[0], sizeof(uint32));
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_LEAVE_CRITERIA, acsCriteriaRegister.val);

    /* Update minimum PDs threshold */
	ac01MinimumPdsThreshold.bitFields.beThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BE];
	ac01MinimumPdsThreshold.bitFields.bkThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BK];
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_THRESH_CRITERIA_0_1, ac01MinimumPdsThreshold.val);
	
	ac23MinimumPdsThreshold.bitFields.viThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI];;
	ac23MinimumPdsThreshold.bitFields.voThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VO];;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_THRESH_CRITERIA_2_3, ac23MinimumPdsThreshold.val);
	
    singleCheckConfigurationRegister.bitFields.forceLock = (restrictedAc == TRUE) ? 1 : agerCheckParameters->forceLock;
	singleCheckConfigurationRegister.bitFields.startStaIdx = agerCheckParameters->stationOrVapIndexToStart;
	singleCheckConfigurationRegister.bitFields.startStaIdx |=  agerCheckParameters->isStartingFromVap << AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_SINGLE_CHECK_CFG, singleCheckConfigurationRegister.val);

	singleCheckPdsLimitRegister.bitFields.singlePdDiscardLimit = agerCheckParameters->maximumPdsToDiscard;
	singleCheckPdsLimitRegister.bitFields.singlePdDiscardLimitEn = agerCheckParameters->isEnabledMaximumLimit;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_SINGLE_PD_LIMIT, singleCheckPdsLimitRegister.val);

	singleCheckControlRegister.bitFields.ageSingleCheckWaitRelease = TRUE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_SINGLE_CHECK_ACTIVATE, singleCheckControlRegister.val);
}

/**********************************************************************************

HwQManagerAger_StopPeriodicCheckAndStartSingleCheck 


Description:
------------
Stops the periodic check and then activates the Ager single check 

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the single check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_StopPeriodicCheckAndStartSingleCheck(AgerSingleCheckParameters_t  *agerCheckParameters, bool restrictedAc)
{
	HwQManagerAger_StopPeriodicCheck();
	HwQManagerAger_StartSingleCheck(agerCheckParameters, restrictedAc);
}


/**********************************************************************************

HwQManagerAger_GetLastCheckParameters 


Description:
------------
Returns the parameters of the last Ager check

Input:
-----
lastCheckParameters - pointer to the strcutre to fill


Output:
-------
The structure contains the needed parameters	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_GetLastCheckParameters(AgerLastCheckParametes_t *lastCheckParameters)
{
	RegQManagerUmacSwMaster3AgerCheckStatus_u statusRegister;
	
	statusRegister.val = 0;

	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CHECK_STATUS, &statusRegister.val);
	
	lastCheckParameters->tid =  statusRegister.bitFields.lastTidChecked;
	lastCheckParameters->isVapQueue = statusRegister.bitFields.lastStaChecked >> AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT;
	statusRegister.bitFields.lastStaChecked &= ~AGER_SINGLE_CHECK_IS_VAP_QUEUE_MASK;
	lastCheckParameters->index =  statusRegister.bitFields.lastStaChecked;
	lastCheckParameters->isSingleCheckDone = statusRegister.bitFields.singleCheckDone;
}


/**********************************************************************************

HwQManagerAger_GetLastCheckCounters 


Description:
------------
Returns the counters of the last Ager check


Input:
-----
lastCheckCounters - pointer to the strcutre to fill


Output:
-------
The structure contains the counters	

Returns:
--------

	
**********************************************************************************/
void HwQManagerAger_GetLastCheckCounters(AgerLastCheckCounters_t *lastCheckCounters)
{
	RegQManagerUmacSwMaster3AgerTryFailCounter_u lockFailuresCounterRegister;
	RegQManagerUmacSwMaster3AgerDoneCounter_u handledPdsCountersRegister;
	
	lockFailuresCounterRegister.val = 0;
	handledPdsCountersRegister.val = 0;

	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_TRY_FAIL_COUNTER, &lockFailuresCounterRegister.val);
	lastCheckCounters->numberOfLockFailures = lockFailuresCounterRegister.bitFields.tryFailCounter;

	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_DONE_COUNTER, &handledPdsCountersRegister.val);
	lastCheckCounters->numberOfPdsHandledToLiberator = handledPdsCountersRegister.bitFields.libCounter;
	lastCheckCounters->numberOfPdsHandledToUmac = handledPdsCountersRegister.bitFields.umacCounter;
}

/**********************************************************************************

HwQManagerAger_ClearSingleCheckDoneInterrupt 


Description:
------------
Returns the counters of the last Ager check


Input:
-----
lastCheckCounters - pointer to the strcutre to fill


Output:
-------
The structure contains the counters	

Returns:
--------

	
**********************************************************************************/
void HwQManagerAger_ClearSingleCheckDoneInterrupt()
{
	RegQManagerUmacSwMaster3AgerIrqClr_u clearSingleCheckDoneInterruptRegister;

	
	clearSingleCheckDoneInterruptRegister.val = 0;

	clearSingleCheckDoneInterruptRegister.bitFields.agerDoneIrqClr = TRUE;

	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_IRQ_CLR, clearSingleCheckDoneInterruptRegister.val);
}


/**********************************************************************************

HwQManagerAger_Initialize 


Description:
------------
Initialize Ager internal registers

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void HwQManagerAger_Initialize(void)
{
	RegQManagerUmacSwMaster3AgerCfg_u globalParametersRegister;
	RegQManagerUmacSwMaster3AgerListsCfg_u listsConfigurationRegister;
	RegQManagerUmacSwMaster3AgerAcThreshCriteria01_u pdsLimitThresholdAc01;
	RegQManagerUmacSwMaster3AgerAcThreshCriteria23_u pdsLimitThresholdAc23;
	RegQManagerUmacSwMaster3AgerTmpListCfg_u tempListConfigurationRegister;

	globalParametersRegister.val = 0;
	listsConfigurationRegister.val = 0;
	pdsLimitThresholdAc01.val = 0; 
	pdsLimitThresholdAc23.val = 0;
	tempListConfigurationRegister.val = 0;

	globalParametersRegister.bitFields.shramWaitState = AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_SHRAM_ACCESS;
	globalParametersRegister.bitFields.clientWaitState = AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_CLIENTS;
//	globalParametersRegister.bitFields.dropRetryPdListEn = TRUE;
	globalParametersRegister.bitFields.retryDiscardLock = FALSE;
    RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_CFG, globalParametersRegister.val);

	listsConfigurationRegister.bitFields.liberatorListIdx = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	listsConfigurationRegister.bitFields.umacListIdx = HW_Q_MANAGER_LOW_PR_READY_LIST_AGER;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_LISTS_CFG, listsConfigurationRegister.val);

	tempListConfigurationRegister.bitFields.umacTmpListIdx = HW_Q_MANAGER_READY_LIST_AGER_TEMPORARY_LIST;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_TMP_LIST_CFG, tempListConfigurationRegister.val); 

	/* The starting PDs limit threshold is 0 */
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_THRESH_CRITERIA_0_1, pdsLimitThresholdAc01.val);
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_AGER_AC_THRESH_CRITERIA_2_3, pdsLimitThresholdAc23.val);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
#endif /* #ifdef ENET_INC_UMAC */

/**********************************************************************************

HwQManager_Init 

Description:
------------
	Init DLM Rams (Um Only)
	Init interrupt configuration 

Input:
-----
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void HwQManager_Init(void)
{
#if defined(ENET_INC_UMAC)
	RegQManagerUmacSwMaster3TxListsNotEmptyIntEn_u txNotEmptyInterruptEnableRegister;
	RegQManagerUmacSwMaster3RxListsNotEmptyIntEn_u rxNotEmptyInterruptEnableRegister;
	RegQManagerUmacSwMaster3DisableNotEmptyIntEn_u disableListsinterruptEnableRegister;

    txNotEmptyInterruptEnableRegister.val = 0;
	rxNotEmptyInterruptEnableRegister.val = 0;
	disableListsinterruptEnableRegister.val = 0;

	hwQManagerInitDlm();
	
	/* Configure UM Register Interface (IF 4) */
	txNotEmptyInterruptEnableRegister.val =  HW_Q_MANAGER_TX_ENABLED_Q;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_TX_LISTS_NOT_EMPTY_INT_EN,txNotEmptyInterruptEnableRegister.val);

	rxNotEmptyInterruptEnableRegister.val = HW_Q_MANAGER_RX_ENABLED_Q;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_RX_LISTS_NOT_EMPTY_INT_EN,rxNotEmptyInterruptEnableRegister.val);
	
	disableListsinterruptEnableRegister.val = HW_Q_MANAGER_TX_DISABLE_LISTS_ENABLED_Q;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_DISABLE_NOT_EMPTY_INT_EN, disableListsinterruptEnableRegister.val);
#endif

#if defined(ENET_INC_LMAC) 
	/* Configure LM Register Interface (IF 3) */
	/* Currently there are no interrupts directed to LM from HW Q Manager */
#endif
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif




