/***********************************************************************************
 File:			HwQManager.c
 Module:		Hw Q Manager
 Purpose:		
 Description:	FW driver for Hw Q Manager Module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "HwQManager_API.h"
#include "DlmRegs.h"
#include "RegAccess_Api.h"
#include "PacketDescriptor.h"
#include "HostInterface_API.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ShramPacketDescriptors.h"
#include "HwMemoryMap.h"
#include "shramTxDesc.h"
#include "QManagerCpu0Regs.h" // LM BAND 1
#include "QManagerCpu1Regs.h" // LM BAND 0
#include "QManagerCpu2Regs.h" // UM
#include "stringLibApi.h"
#include "ShramDataTxDescriptors.h"
#include "int_gen.h"
#include "Pac_Api.h"
#include "ConfigurationManager_api.h"
#include "RxMpduDescriptor_Descriptors.h"
#include "TxMpduDescriptor_Descriptors.h"
#include "DmacWrapper_Descriptors.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 9

typedef enum RegInterfaceIndex
{
	HW_Q_MANAGER_CONTROL_REG_INDEX,
	HW_Q_MANAGER_ADDR_TOP_REG_INDEX,
	HW_Q_MANAGER_ADDR_BOTTOM_REG_INDEX,
	HW_Q_MANAGER_RETURN_ADDR_TOP_REG_INDEX,
	HW_Q_MANAGER_RETURN_ADDR_BOTTOM_REG_INDEX,
	HW_Q_MANAGER_FIFO_STATUS_REG_INDEX,
	HW_Q_MANAGER_TOTAL_NUM_OF_REG_INDEXES,
} RegInterfaceIndex_e;

typedef enum InterruptRegInterfaceIndex
{
	HW_Q_MANAGER_TX_DONE_INT_REG_INDEX,
	HW_Q_MANAGER_LO_PRI_TX_READY_INT_REG_INDEX,
	HW_Q_MANAGER_HI_PRI_TX_READY_INT_REG_INDEX,
	HW_Q_MANAGER_HI_PRI_RX_READY_INT_REG_INDEX,
	HW_Q_MANAGER_TX_INT_CLR_REG_INDEX,
	HW_Q_MANAGER_RX_INT_CLR_REG_INDEX,
	HW_Q_MANAGER_DISABLE_INT_CLR_REG_INDEX,
	HW_Q_MANAGER_DISABLE_INT_REG_INDEX,	
	HW_Q_MANAGER_ERR_REG_INDEX,
	HW_Q_MANAGER_TOTAL_NUM_OF_INTERRUPT_REG_INDEXES,
} InterruptRegInterfaceIndex_e;

typedef enum
{
	HW_Q_MANAGER_DISABLE_LIST_HW_MASTER_BAND_0 = 0,
	HW_Q_MANAGER_DISABLE_LIST_HW_MASTER_BAND_1,
	HW_Q_MANAGER_DISABLE_LIST_TX_GENRISC_MASTER_BAND_0,
	HW_Q_MANAGER_DISABLE_LIST_HOST_GENRISC_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_RX_GENRISC_MASTER_BAND_0,
	HW_Q_MANAGER_DISABLE_LIST_UMAC_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_LMAC0_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_LMAC1_MASTER,
	HW_Q_MANAGER_DISABLE_LIST_TX_GENRISC_MASTER_BAND_1,
	HW_Q_MANAGER_DISABLE_LIST_RX_GENRISC_MASTER_BAND_1,
	HW_Q_MANAGER_NUM_OF_DISABLE_LISTS_MASTERS,
	HW_Q_MANAGER_MAX_NUM_OF_DISABLE_LISTS_MASTERS = 0xFF,
} HwQManagerDisableListMaster_e;

#define DLM_RAM_INIT_VALUE			(0xFFFFFFFF)

#if defined (AGER_STATUS_WORKAROUND)
#define HW_Q_MANAGER_TX_ENABLED_Q_0_31	((1 << HW_Q_MANAGER_DONE_LIST_HOST_DATA) 			|\
										 (1 << HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS) 	|\
										 (1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW) 	|\
                                    	 (1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST) |\
                                		 (1 << HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST) 	|\
										 (1 << HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW) 		|\
										 (1 << HW_Q_MANAGER_DONE_LIST_AGER_LIBERATOR) 		|\
										 (1 << HW_Q_MANAGER_DONE_LIST_ERROR)				|\
                                		 (1 << HW_Q_MANAGER_LOW_PR_READY_LIST_AGER))
#else
#define HW_Q_MANAGER_TX_ENABLED_Q_0_31	((1 << HW_Q_MANAGER_DONE_LIST_HOST_DATA) 			|\
										 (1 << HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS) 	|\
										 (1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW) 	|\
                                    	 (1 << HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST) |\
                                		 (1 << HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST) 	|\
										 (1 << HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW) 		|\
										 (1 << HW_Q_MANAGER_DONE_LIST_ERROR)				|\
                                		 (1 << HW_Q_MANAGER_LOW_PR_READY_LIST_AGER))
#endif                                		 

#define HW_Q_MANAGER_TX_ENABLED_Q_32_63	((1 << (HW_Q_MANAGER_READY_LIST_MULTICAST_PD & 0x1F)) |\
                                		 (1 << (HW_Q_MANAGER_READY_LIST_UNICAST_PD & 0x1F)))
                                	
#define HW_Q_MANAGER_TX_DISABLE_LISTS_ENABLED_Q (	(1 << HW_Q_MANAGER_DISABLE_LIST_HW_MASTER_BAND_0)			| (1 << HW_Q_MANAGER_DISABLE_LIST_HW_MASTER_BAND_1) 		|\
													(1 << HW_Q_MANAGER_DISABLE_LIST_TX_GENRISC_MASTER_BAND_0)	| (1 << HW_Q_MANAGER_DISABLE_LIST_TX_GENRISC_MASTER_BAND_1) |\
													(1 << HW_Q_MANAGER_DISABLE_LIST_RX_GENRISC_MASTER_BAND_0)	| (1 << HW_Q_MANAGER_DISABLE_LIST_RX_GENRISC_MASTER_BAND_1) |\
													(1 << HW_Q_MANAGER_DISABLE_LIST_HOST_GENRISC_MASTER)		| (1 << HW_Q_MANAGER_DISABLE_LIST_UMAC_MASTER) 				|\
													(1 << HW_Q_MANAGER_DISABLE_LIST_LMAC0_MASTER)				| (1 << HW_Q_MANAGER_DISABLE_LIST_LMAC1_MASTER))
	
#if defined (RX_LIST_DEBUG)
#define HW_Q_MANAGER_RX_ENABLED_Q_0_31	(0)

#define HW_Q_MANAGER_RX_ENABLED_Q_32_63	((1 << (HW_Q_MANAGER_RX_READY_LIST_FW & 0x1F)) 								|\
										 (1 << (HW_Q_MANAGER_RX_READY_LIST_DRIVER & 0x1F)) 							|\
										 (1 << (HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW & 0x1F))	|\
                                    	 (1 << (HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS & 0x1F)) 						|\
                                    	 (1 << (HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS & 0x1F)) 			|\
                                    	 (1 << (HW_Q_MANAGER_RX_READY_LIST_DEBUG & 0x1F)) 							|\
                                		 (1 << (HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP & 0x1F)))
#else
#define HW_Q_MANAGER_RX_ENABLED_Q_0_31	(0)

#define HW_Q_MANAGER_RX_ENABLED_Q_32_63	((1 << (HW_Q_MANAGER_RX_READY_LIST_FW & 0x1F)) 								|\
										 (1 << (HW_Q_MANAGER_RX_READY_LIST_DRIVER & 0x1F)) 							|\
										 (1 << (HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW & 0x1F))	|\
										 (1 << (HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS & 0x1F))						|\
                                    	 (1 << (HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS & 0x1F)))


#endif

#define SET_NULL_TO_NEXT (1)
#define DO_NOT_SET_NULL_TO_NEXT (0)
#define DLM_SIZE_WITHOUT_RERGISTERS (0xABE8) // Taken from HW Q Manager document (memory map chapter)
#define DLM_TOTAL_SIZE_IN_WORDS (CONVERT_BYTES_TO_WORDS(DLM_SIZE_WITHOUT_RERGISTERS))

/* Ager section */
#define AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_CLIENTS 8
#define AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_SHRAM_ACCESS (50) /* 4 */
#define AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT 8 /* (max index bit num of startStaIdx) */

#define AGER_SINGLE_CHECK_IS_VAP_QUEUE_MASK (1 << AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT) 
#define AGER_CLEAR_DISCARDED_PACKETS_TO_UMAC_COUNTER_MASK 0x1
#define AGER_CLEAR_DISCARDED_PACKETS_TO_LIBERATOR_COUNTER_MASK 0x2
#define AGER_CLEAR_LOCK_FAILED_COUNTER_MASK 0x4
#define AGER_CLEAR_ALL_COUNTERS_MASK (AGER_CLEAR_DISCARDED_PACKETS_TO_UMAC_COUNTER_MASK |\
									  AGER_CLEAR_DISCARDED_PACKETS_TO_LIBERATOR_COUNTER_MASK |\
									  AGER_CLEAR_LOCK_FAILED_COUNTER_MASK)

#define HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL 1000

#define NEXT_DESC_NULL (0xFFFF)


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void hwQManagerPushToQ(HwQMangerRequestType_e requestType, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams);
void hwQManagerPop(HwQMangerRequestType_e requestType, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams);
void hwQManagerFillControlRegFields(RegQManagerCpu2Ctl_u* controlReg, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, HwQMangerRequestType_e requestType);
void hwQManager_InitDmaDlmList(void);
void hwQManager_Init_B0_MpduDlmList(void);
void hwQManager_Init_B1_MpduDlmList(void);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
const uint32 regInterfaceAddresses[HW_Q_MANAGER_TOTAL_NUM_OF_REG_IF][HW_Q_MANAGER_TOTAL_NUM_OF_REG_INDEXES] =
{
#if defined (ENET_INC_UMAC)
	{
		REG_Q_MANAGER_CPU2_CTL,
		REG_Q_MANAGER_CPU2_PUSH_ADDR_TOP,
		REG_Q_MANAGER_CPU2_PUSH_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU2_POP_ADDR_TOP,
		REG_Q_MANAGER_CPU2_POP_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU2_FIFO_STATUS
	},
	{	
		REG_Q_MANAGER_CPU2_INT_CTL,
		REG_Q_MANAGER_CPU2_INT_PUSH_ADDR_TOP,
		REG_Q_MANAGER_CPU2_INT_PUSH_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU2_INT_POP_ADDR_TOP,
		REG_Q_MANAGER_CPU2_INT_POP_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU2_FIFO_STATUS
	},
#endif
#if defined (ENET_INC_LMAC0)
	{
		REG_Q_MANAGER_CPU0_CTL,
		REG_Q_MANAGER_CPU0_PUSH_ADDR_TOP,
		REG_Q_MANAGER_CPU0_PUSH_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU0_POP_ADDR_TOP,
		REG_Q_MANAGER_CPU0_POP_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU0_FIFO_STATUS
	},
	{	
		REG_Q_MANAGER_CPU0_INT_CTL,
		REG_Q_MANAGER_CPU0_INT_PUSH_ADDR_TOP,
		REG_Q_MANAGER_CPU0_INT_PUSH_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU0_INT_POP_ADDR_TOP,
		REG_Q_MANAGER_CPU0_INT_POP_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU0_FIFO_STATUS
	},
#endif
#if defined (ENET_INC_LMAC1)
	{
		REG_Q_MANAGER_CPU1_CTL,
		REG_Q_MANAGER_CPU1_PUSH_ADDR_TOP,
		REG_Q_MANAGER_CPU1_PUSH_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU1_POP_ADDR_TOP,
		REG_Q_MANAGER_CPU1_POP_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU1_FIFO_STATUS
	},
	{	
		REG_Q_MANAGER_CPU1_INT_CTL,
		REG_Q_MANAGER_CPU1_INT_PUSH_ADDR_TOP,
		REG_Q_MANAGER_CPU1_INT_PUSH_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU1_INT_POP_ADDR_TOP,
		REG_Q_MANAGER_CPU1_INT_POP_ADDR_BOTTOM,
		REG_Q_MANAGER_CPU1_FIFO_STATUS
	},
#endif

};

const uint32 interruptsRegInterfaceAddresses[HW_Q_MANAGER_TOTAL_NUM_OF_INTERRUPT_REG_INDEXES] =
{
#if defined (ENET_INC_UMAC)	
	REG_Q_MANAGER_CPU2_TX_DONE_INT,
	REG_Q_MANAGER_CPU2_LO_PRI_TX_READY_INT,
	REG_Q_MANAGER_CPU2_HI_PRI_TX_READY_INT,
	REG_Q_MANAGER_CPU2_HI_PRI_RX_READY_INT,
	REG_Q_MANAGER_CPU2_TX_INT_CLR0_31,
	REG_Q_MANAGER_CPU2_RX_INT_CLR0_31,
	REG_Q_MANAGER_CPU2_DISABLE_INT_CLR,
	REG_Q_MANAGER_CPU2_DISABLE_INT,
	REG_Q_MANAGER_CPU2_ERR,
#endif
#if defined (ENET_INC_LMAC0)	
	REG_Q_MANAGER_CPU0_TX_DONE_INT,
	REG_Q_MANAGER_CPU0_LO_PRI_TX_READY_INT,
	REG_Q_MANAGER_CPU0_HI_PRI_TX_READY_INT,
	REG_Q_MANAGER_CPU0_HI_PRI_RX_READY_INT,
	REG_Q_MANAGER_CPU0_TX_INT_CLR0_31,
	REG_Q_MANAGER_CPU0_RX_INT_CLR0_31,
	REG_Q_MANAGER_CPU0_DISABLE_INT_CLR,
	REG_Q_MANAGER_CPU0_DISABLE_INT,
	REG_Q_MANAGER_CPU0_ERR,
#endif
#if defined (ENET_INC_LMAC1)	
	REG_Q_MANAGER_CPU1_TX_DONE_INT,
	REG_Q_MANAGER_CPU1_LO_PRI_TX_READY_INT,
	REG_Q_MANAGER_CPU1_HI_PRI_TX_READY_INT,
	REG_Q_MANAGER_CPU1_HI_PRI_RX_READY_INT,
	REG_Q_MANAGER_CPU1_TX_INT_CLR0_31,
	REG_Q_MANAGER_CPU1_RX_INT_CLR0_31,
	REG_Q_MANAGER_CPU1_DISABLE_INT_CLR,
	REG_Q_MANAGER_CPU1_DISABLE_INT,
	REG_Q_MANAGER_CPU1_ERR,
#endif
};

MAP_TO_SECTION(bss, ".DMAC_LL_MEMORY")
volatile DmacWrapper_t sDmaDescriptor[DMAC_NUM_OF_DESCRIPTORS];
MAP_TO_DEFAULT_SECTION(bss)


MAP_TO_SECTION(bss, ".B0_MPDU_LL_MEMORY")
b0RxTxMpduDesc_u	B0_MpduDescriptors;
MAP_TO_DEFAULT_SECTION(bss)


MAP_TO_SECTION(bss, ".B1_MPDU_LL_MEMORY")
b1RxTxMpduDesc_u	B1_MpduDescriptors;
MAP_TO_DEFAULT_SECTION(bss)




/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

hwQManagerPushToQ 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQMangerRequestType_e requestType
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerPushToQ(HwQMangerRequestType_e requestType,HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	TX_INTERRUPT_SAVE_AREA;
	/* These registers are identical for LMACs and UMAC */
	RegQManagerCpu2Ctl_u 			controlReg;
	RegQManagerCpu2PushAddrTop_u 	addrTopReg;
	RegQManagerCpu2PushAddrBottom_u addrBottomReg;
	volatile RegQManagerCpu2FifoStatus_u 	fifoStatusReg;
	uint32 							controlRegAddress = 0;
	uint32 							addrTopRegAddress = 0;
	uint32 							addrBottomRegAddress = 0;	
	uint32 							fifoStatusRegAddress = 0;
	uint32 							timeStamp = 0;

	controlReg.val = 0;
	addrTopReg.val = 0;
	addrBottomReg.val = 0;

	/* Fill Control Register values*/
	hwQManagerFillControlRegFields(&controlReg, hwQueueManagerRequestParams, requestType);

	/* Fill Address Registers - Holds the pointer of the required PD/RD to enqueue  (PDs and RDs share the same base address) */
	addrTopReg.bitFields.pushAddrTop = CONVERT_PD_TO_OFFSET(hwQueueManagerRequestParams->pHeadDesc);
	if(hwQueueManagerRequestParams->pTailDesc != NULL)
	{
		addrBottomReg.bitFields.pushAddrBottom = CONVERT_PD_TO_OFFSET(hwQueueManagerRequestParams->pTailDesc); //For Single Packet Should Be Null
	}


	/* Set the required registers addresses */
	controlRegAddress 		= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_CONTROL_REG_INDEX];
	addrTopRegAddress 		= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_ADDR_TOP_REG_INDEX];
	addrBottomRegAddress 	= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_ADDR_BOTTOM_REG_INDEX];	
	fifoStatusRegAddress 	= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_FIFO_STATUS_REG_INDEX];

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	//Debug asserts:
	//1. Make sure head pointer is not 0xFFFFFF (NULL)
	ASSERT((addrTopReg.bitFields.pushAddrTop < NEXT_PD_NULL) || (hwQueueManagerRequestParams->dlmNum > HW_Q_MANAGER_TX_LISTS_DLM));
	ASSERT((addrTopReg.bitFields.pushAddrTop < NEXT_RD_NULL) || (hwQueueManagerRequestParams->dlmNum != HW_Q_MANAGER_RX_LISTS_DLM));

	//2. Make sure that in case we are trying to push a list the tail pointer is not 0xFFFFFF (NULL)
	ASSERT(((requestType != HW_Q_MANAGER_PUSH_PACKET_LIST_TO_TAIL) && (requestType != HW_Q_MANAGER_PUSH_PACKET_LIST_TO_HEAD)) || ((addrBottomReg.bitFields.pushAddrBottom < NEXT_PD_NULL) || (hwQueueManagerRequestParams->dlmNum > HW_Q_MANAGER_TX_LISTS_DLM)));
	ASSERT(((requestType != HW_Q_MANAGER_PUSH_PACKET_LIST_TO_TAIL) && (requestType != HW_Q_MANAGER_PUSH_PACKET_LIST_TO_HEAD)) || ((addrBottomReg.bitFields.pushAddrBottom < NEXT_RD_NULL) || (hwQueueManagerRequestParams->dlmNum != HW_Q_MANAGER_RX_LISTS_DLM)));

	// Make sure the fifo of the Hw Queue Manager is not full (can't accept more requests)
	RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);
#ifndef ARC_SIM
	timeStamp = GET_TSF_TIMER_LOW();

	
	//KW_FIX_FW_M - pragma ignore as register read & timeout check with an assert
	while (fifoStatusReg.bitFields.swFifoFull == 1)
	{
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL); 
		RegAccess_Read(fifoStatusRegAddress, (uint32 *)&fifoStatusReg.val);
	}
#endif

	// Configure the HW (writing the control register "kick" the push command)
	RegAccess_Write(addrTopRegAddress,		addrTopReg.val);
	RegAccess_Write(addrBottomRegAddress,	addrBottomReg.val);
	RegAccess_Write(controlRegAddress,		controlReg.val);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);

}

/**********************************************************************************

hwQManagerPop 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQMangerRequestType_e requestType
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerPop(HwQMangerRequestType_e requestType,HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	TX_INTERRUPT_SAVE_AREA;
	/* These registers are identical for LMACs and UMAC */
	RegQManagerCpu2Ctl_u 			controlReg;
	RegQManagerCpu2PushAddrTop_u 	addrTopReg;
	RegQManagerCpu2PushAddrBottom_u addrBottomReg;
	RegQManagerCpu2PopAddrTop_u		returnAddrTopReg;
	RegQManagerCpu2PopAddrBottom_u	returnAddrBottomReg;
	volatile RegQManagerCpu2FifoStatus_u 	fifoStatusReg;
	uint32 							controlRegAddress = 0;
	uint32 							addrTopRegAddress = 0;
	uint32 							addrBottomRegAddress = 0;	
	uint32 							returnAddrTopRegAddress = 0;
	uint32 							returnAddrBottomRegAddress = 0;	
	uint32 							fifoStatusRegAddress = 0;
	uint32 							timeStamp = 0;
	
	/* Fill Control Register values*/
	hwQManagerFillControlRegFields(&controlReg, hwQueueManagerRequestParams, requestType);

	/* Set the required registers addresses */
	controlRegAddress 			= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_CONTROL_REG_INDEX];
	addrTopRegAddress 			= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_ADDR_TOP_REG_INDEX];
	addrBottomRegAddress 		= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_ADDR_BOTTOM_REG_INDEX];	
	returnAddrTopRegAddress		= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_RETURN_ADDR_TOP_REG_INDEX];
	returnAddrBottomRegAddress 	= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_RETURN_ADDR_BOTTOM_REG_INDEX];
	fifoStatusRegAddress 		= regInterfaceAddresses[hwQueueManagerRequestParams->regIfNum][HW_Q_MANAGER_FIFO_STATUS_REG_INDEX];

	/* Fill Address Registers - Holds the pointer of the required PD/RD to dequeue  (PDs and RDs share the same base address) */	
	addrTopReg.bitFields.pushAddrTop = CONVERT_PD_TO_OFFSET(hwQueueManagerRequestParams->pHeadDesc);
	if(hwQueueManagerRequestParams->pTailDesc != NULL)
	{
		addrBottomReg.bitFields.pushAddrBottom = CONVERT_PD_TO_OFFSET(hwQueueManagerRequestParams->pTailDesc); //For Single Packet Should Be Null
	}

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Make sure the fifo of the Hw Queue Manager is not full (can't accept more requests)
	RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);
	timeStamp = GET_TSF_TIMER_LOW();
	
	//KW_FIX_FW_M - pragma ignore as register read & timeout check with an assert
	while (fifoStatusReg.bitFields.swFifoFull == 1)
	{
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL);
		RegAccess_Read(fifoStatusRegAddress, (uint32 *)&fifoStatusReg.val);		 
	}

	// Configure the HW (writing the control register "kick" the push command)
	RegAccess_Write(addrTopRegAddress,		addrTopReg.val);
	RegAccess_Write(addrBottomRegAddress,	addrBottomReg.val);
	RegAccess_Write(controlRegAddress,		controlReg.val);

	// Read back the results
	RegAccess_Read(returnAddrTopRegAddress, 	&returnAddrTopReg.val);
	RegAccess_Read(returnAddrBottomRegAddress, 	&returnAddrBottomReg.val);	

	OSAL_ENABLE_INTERRUPTS(interrupt_save); 
	
	// return the head and tail (optional) that we popped.
	hwQueueManagerRequestParams->pHeadDesc  = CONVERT_OFFSET_TO_PD(returnAddrTopReg.bitFields.popAddrTop);
	hwQueueManagerRequestParams->pTailDesc  = CONVERT_OFFSET_TO_PD(returnAddrBottomReg.bitFields.popAddrBottom); /* For single PD value is unknown*/
}

/**********************************************************************************

hwQManagerFillControlRegFields 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	RegHwQManagerControl_u * controlReg
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerFillControlRegFields(RegQManagerCpu2Ctl_u* controlReg, HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, HwQMangerRequestType_e requestType)
{	
	controlReg->val = 0;
	
	/* Fill Control Register values - the register is identical for LMACs and UMAC */
	controlReg->bitFields.ctlPrimaryAddr	= hwQueueManagerRequestParams->primaryAddr;
	controlReg->bitFields.ctlSecondaryAddr	= hwQueueManagerRequestParams->secondaryAddr;
	controlReg->bitFields.ctlSetNull 		= hwQueueManagerRequestParams->setNull;
	controlReg->bitFields.ctlDplIdx 		= hwQueueManagerRequestParams->dplIndex;
	controlReg->bitFields.ctlDlmIdx 		= hwQueueManagerRequestParams->dlmNum;
	controlReg->bitFields.ctlReq 			= requestType;
	controlReg->bitFields.ctlMgmt 			= FALSE;

	//if this is tx/rx data dlm & we want to use mgmt tid we need to set ctlMgmt
	if ((controlReg->bitFields.ctlDlmIdx == HW_Q_MANAGER_TX_DATA_DLM) || (controlReg->bitFields.ctlDlmIdx == HW_Q_MANAGER_RX_DATA_DLM))
	{
		if (hwQueueManagerRequestParams->secondaryAddr == MANAGEMENT_TID)
		{
			controlReg->bitFields.ctlMgmt = TRUE;
		}
	}
}

/**********************************************************************************

hwQManager_Init_B0_MpduDlmList 

Description:
------------
	Initialize the Band0 MPDU Descriptors DLM with descriptors. Link them and push them to the free lists

Input:
-----
	None
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManager_Init_B0_MpduDlmList(void)
{
	uint32 				i;
	uint32 				actualNumOfRxDesc;
	uint32 				actualNumOfTxDesc;
	RxMpduDescriptor_t*	pRxMpduDescriptor;
	TxMpduDescriptor_t*	pTxMpduDescriptor;


	ASSERT((uint32)&B0_MpduDescriptors == B0_MPDU_DESC_MEM_BASE_ADDR);

	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_SINGLE_BAND)
	{
		actualNumOfRxDesc = NUM_OF_RX_MPDU_DESCRIPTORS_B0_SINGLE_BAND;
		actualNumOfTxDesc = NUM_OF_TX_MPDU_DESCRIPTORS_B0_SINGLE_BAND;
		pRxMpduDescriptor = &(B0_MpduDescriptors.singleBandDescriptors.sB0_RxMpduDescriptor[0]);
		pTxMpduDescriptor = &(B0_MpduDescriptors.singleBandDescriptors.sB0_TxMpduDescriptor[0]);
	}
	else
	{
		actualNumOfRxDesc = NUM_OF_RX_MPDU_DESCRIPTORS_B0_DUAL_BAND;
		actualNumOfTxDesc = NUM_OF_TX_MPDU_DESCRIPTORS_B0_DUAL_BAND;
		pRxMpduDescriptor = &(B0_MpduDescriptors.dualBandDescriptors.sB0_RxMpduDescriptor[0]);
		pTxMpduDescriptor = &(B0_MpduDescriptors.dualBandDescriptors.sB0_TxMpduDescriptor[0]);
	}

	memset(pRxMpduDescriptor, 0, sizeof(RxMpduDescriptor_t)*actualNumOfRxDesc);
	memset(pTxMpduDescriptor, 0, sizeof(TxMpduDescriptor_t)*actualNumOfTxDesc);



	// Init the RX MPDU descriptors
	for(i = 0; i < actualNumOfRxDesc; i++)
	{
		pRxMpduDescriptor[i].nextPointer= B0_SET_NEXT_MPDU_DESC(&pRxMpduDescriptor[i+1]);
	}
	i --;

	pRxMpduDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_RX_MPDU_DESCRIPTOR_BAND0_LISTS_DLM, B0_CONVERT_MPDU_DESC_TO_OFFSET(&pRxMpduDescriptor[0]), B0_CONVERT_MPDU_DESC_TO_OFFSET(&pRxMpduDescriptor[i]), HW_Q_MANAGER_RX_MPDU_RXC_IN_LIST);

	// Init the TX MPDU descriptors
	for(i = 0; i < actualNumOfTxDesc; i++)
	{
		pTxMpduDescriptor[i].nextMpduDescriptor= B0_SET_NEXT_MPDU_DESC(&pTxMpduDescriptor[i+1]);
	}
	i --;

	pTxMpduDescriptor[i].nextMpduDescriptor = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_TX_MPDU_DESCRIPTOR_BAND0_LISTS_DLM, B0_CONVERT_MPDU_DESC_TO_OFFSET(&pTxMpduDescriptor[0]), B0_CONVERT_MPDU_DESC_TO_OFFSET(&pTxMpduDescriptor[i]), HW_Q_MANAGER_TX_MPDU_FREE_LIST);
}


/**********************************************************************************

hwQManager_Init_B1_MpduDlmList 

Description:
------------
	Initialize the Band1 MPDU Descriptors DLM with descriptors. Link them and push them to the free lists

Input:
-----
	None
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManager_Init_B1_MpduDlmList(void)
{
	uint32 				i;
	uint32 				actualNumOfRxDesc;
	uint32 				actualNumOfTxDesc;
	RxMpduDescriptor_t*	pRxMpduDescriptor;
	TxMpduDescriptor_t*	pTxMpduDescriptor;


	ASSERT((uint32)&B1_MpduDescriptors == B1_MPDU_DESC_MEM_BASE_ADDR);

	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_SINGLE_BAND)
	{
		actualNumOfRxDesc = NUM_OF_RX_MPDU_DESCRIPTORS_B1_SINGLE_BAND;
		actualNumOfTxDesc = NUM_OF_TX_MPDU_DESCRIPTORS_B1_SINGLE_BAND;
		pRxMpduDescriptor = &(B1_MpduDescriptors.singleBandDescriptors.sB1_RxMpduDescriptor[0]);
		pTxMpduDescriptor = &(B1_MpduDescriptors.singleBandDescriptors.sB1_TxMpduDescriptor[0]);
	}
	else
	{
		actualNumOfRxDesc = NUM_OF_RX_MPDU_DESCRIPTORS_B1_DUAL_BAND;
		actualNumOfTxDesc = NUM_OF_TX_MPDU_DESCRIPTORS_B1_DUAL_BAND;
		pRxMpduDescriptor = &(B1_MpduDescriptors.dualBandDescriptors.sB1_RxMpduDescriptor[0]);
		pTxMpduDescriptor = &(B1_MpduDescriptors.dualBandDescriptors.sB1_TxMpduDescriptor[0]);
	}

	memset(pRxMpduDescriptor, 0, sizeof(RxMpduDescriptor_t)*actualNumOfRxDesc);
	memset(pTxMpduDescriptor, 0, sizeof(TxMpduDescriptor_t)*actualNumOfTxDesc);



	// Init the RX MPDU descriptors
	for(i = 0; i < actualNumOfRxDesc; i++)
	{
		pRxMpduDescriptor[i].nextPointer= B1_SET_NEXT_MPDU_DESC(&pRxMpduDescriptor[i+1]);
	}
	i --;

	pRxMpduDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_RX_MPDU_DESCRIPTOR_BAND1_LISTS_DLM, B1_CONVERT_MPDU_DESC_TO_OFFSET(&pRxMpduDescriptor[0]), B1_CONVERT_MPDU_DESC_TO_OFFSET(&pRxMpduDescriptor[i]), HW_Q_MANAGER_RX_MPDU_RXC_IN_LIST);

	// Init the TX MPDU descriptors
	for(i = 0; i < actualNumOfTxDesc; i++)
	{
		pTxMpduDescriptor[i].nextMpduDescriptor= B1_SET_NEXT_MPDU_DESC(&pTxMpduDescriptor[i+1]);
	}
	i --;

	pTxMpduDescriptor[i].nextMpduDescriptor = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_TX_MPDU_DESCRIPTOR_BAND1_LISTS_DLM, B1_CONVERT_MPDU_DESC_TO_OFFSET(&pTxMpduDescriptor[0]), B1_CONVERT_MPDU_DESC_TO_OFFSET(&pTxMpduDescriptor[i]), HW_Q_MANAGER_TX_MPDU_FREE_LIST);
}

/**********************************************************************************

hwQManager_InitDmaDlmList 

Description:
------------
	Initialize the DMA DLM with descriptors. Link them and push them to the free lists

Input:
-----
	None
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManager_InitDmaDlmList(void)
{
	uint16 i;
	uint16 lastListIndex=0;

	ASSERT((uint32)&sDmaDescriptor[0] == WLAN_ARM_LL_MEM_BASE_ADDR);

	//init the TX0 free DPL (80 descriptors)
	for(i = 0; i < DMAC_NUM_OF_TX_0_DESCRIPTORS; i++)
	{
		sDmaDescriptor[i].nextPointer = SET_NEXT_DMA_DESC(&sDmaDescriptor[i+1]);
	}
	i --;
	lastListIndex =lastListIndex + DMAC_NUM_OF_TX_0_DESCRIPTORS;
	sDmaDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_DMA_LISTS_DLM,CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[0]),CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[i]),HW_Q_MANAGER_DLM_DMA_TX_0_FREE_LIST);


	//init the TX1 free DPL (80 descriptors)
	for(i = lastListIndex; i < (lastListIndex + DMAC_NUM_OF_TX_1_DESCRIPTORS) ; i++)
	{
		sDmaDescriptor[i].nextPointer = SET_NEXT_DMA_DESC(&sDmaDescriptor[i+1]);
	}
	i --;
	
	sDmaDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_DMA_LISTS_DLM,CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[lastListIndex]),CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[i]),HW_Q_MANAGER_DLM_DMA_TX_1_FREE_LIST);
	lastListIndex =lastListIndex +DMAC_NUM_OF_TX_1_DESCRIPTORS;


	//init the RX0 free DPL (80 descriptors)
	for(i = lastListIndex; i < (lastListIndex + DMAC_NUM_OF_RX_0_DESCRIPTORS); i++)
	{
		sDmaDescriptor[i].nextPointer = SET_NEXT_DMA_DESC(&sDmaDescriptor[i+1]);
	}
	i --;
	
	sDmaDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_DMA_LISTS_DLM,CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[lastListIndex]),CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[i]),HW_Q_MANAGER_DLM_DMA_RX_0_FREE_LIST);
	lastListIndex =lastListIndex +DMAC_NUM_OF_RX_0_DESCRIPTORS;


	//init the RX1 free DPL (80 descriptors)
	for(i = lastListIndex; i < (lastListIndex + DMAC_NUM_OF_RX_1_DESCRIPTORS); i++)
	{
		sDmaDescriptor[i].nextPointer = SET_NEXT_DMA_DESC(&sDmaDescriptor[i+1]);
	}
	i --;
	
	sDmaDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_DMA_LISTS_DLM,CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[lastListIndex]),CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[i]),HW_Q_MANAGER_DLM_DMA_RX_1_FREE_LIST);
	lastListIndex =lastListIndex +DMAC_NUM_OF_RX_1_DESCRIPTORS;


	//init the HOST free DPL (32 descriptors)
	for(i = lastListIndex; i < (lastListIndex + DMAC_NUM_OF_HOST_DESCRIPTORS); i++)
	{
		sDmaDescriptor[i].nextPointer = SET_NEXT_DMA_DESC(&sDmaDescriptor[i+1]);
	}
	i --;
	
	sDmaDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_DMA_LISTS_DLM,CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[lastListIndex]),CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[i]),HW_Q_MANAGER_DLM_DMA_HOST_FREE_LIST);
	lastListIndex =lastListIndex + DMAC_NUM_OF_HOST_DESCRIPTORS;


	//init the FW free DPL (32 descriptors)
	for(i = lastListIndex; i < (lastListIndex + DMAC_NUM_OF_FW_DESCRIPTORS); i++)
	{
		sDmaDescriptor[i].nextPointer = SET_NEXT_DMA_DESC(&sDmaDescriptor[i+1]);
	}
	i --;
	sDmaDescriptor[i].nextPointer = NEXT_DESC_NULL;
	HwQManagerPushToDlmQ(HW_Q_MANAGER_DMA_LISTS_DLM,CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[lastListIndex]),CONVERT_DMA_DESC_TO_OFFSET(&sDmaDescriptor[i]),HW_Q_MANAGER_DLM_DMA_FW_FREE_LIST);
}



#if defined(ENET_INC_UMAC)
/**********************************************************************************

hwQManagerInitDlm 

Description:
------------
	Initialize the DLMs

Input:
-----
	None
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void hwQManagerInitDlm(void)
{
	/* The register is identical for Tx and Rx */
	uint8 *ramAddr;

	/* Configure all dlm sections - all dlm sections are sequential */
	ramAddr = (uint8 *)DLM_BASE_ADDR;
	memset32(ramAddr, DLM_RAM_INIT_VALUE, DLM_TOTAL_SIZE_IN_WORDS); /* Set DLM's ram to 0xFF*/

	// Init the DLM memory of the DMA with descriptors from the DMA LL MEMORY.
	hwQManager_InitDmaDlmList();
	// Init Band0 TX & RX MPDU descriptors 
	hwQManager_Init_B0_MpduDlmList();
	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
	{
		// Init Band1 TX & RX MPDU descriptors 
		hwQManager_Init_B1_MpduDlmList();
	}

	// We don't need to configure the base address for the HW DLMs (DMA, MPDU descriptors) since it's hard wired to the memory.
	// We also don't need to configure the RX & TX anymore since it's in a dedicated memory.
}

#endif /* #if defined(ENET_INC_UMAC) */
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

HwQManager_PushPacketToTail 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketToTail(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->pTailDesc = NULL;
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_TO_TAIL, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PushPacketListToTail 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketListToTail(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_LIST_TO_TAIL, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PushPacketToHead 

Description:
------------
	Push PD to the head of the required Q

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketToHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->pTailDesc = NULL;
	hwQueueManagerRequestParams->setNull = DO_NOT_SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_TO_HEAD, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PushPacketlistToHead 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PushPacketlistToHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = DO_NOT_SET_NULL_TO_NEXT;
	hwQManagerPushToQ(HW_Q_MANAGER_PUSH_PACKET_LIST_TO_HEAD, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PopPacket 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PopPacket(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPop(HW_Q_MANAGER_POP_PACKET, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PopListFromHead 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams - Request Parametes (DLM Index, Queue Index,Request Type etc)
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_PopListFromHead(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQueueManagerRequestParams->setNull = SET_NULL_TO_NEXT;
	hwQManagerPop(HW_Q_MANAGER_POP_PACKET_LIST, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_PeekHeadPacket 

Description:
------------
	Returns the First PD address on the required Q without removing the PD from Q

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams - Request Parametes (DLM Index, Queue Index,Request Type etc)
Output:
-------
	Head pointer value
Returns:
--------
	void 
**********************************************************************************/
void HwQManager_PeekHeadPacket(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQManagerPop(HW_Q_MANAGER_PEEK_PACKET, hwQueueManagerRequestParams);
}

/**********************************************************************************

HwQManager_FlushQ 

Description:
------------
	Remove all PD from required Queue  

Input:
-----
	HwQueueManagerRequestParams_t * hwQueueManagerRequestParams
Output:
-------
	Head & Tail pointers to the list flushed from the Q 
Returns:
--------
	void 
**********************************************************************************/
void HwQManager_FlushQ(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams)
{
	hwQManagerPop(HW_Q_MANAGER_FLUSH, hwQueueManagerRequestParams);
}


/**********************************************************************************

HwQManager_GetLowPriorityTxPdReadyListsStatus 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	uint32 * highPriorityTxPdReadylistStatus
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_GetLowPriorityTxPdReadyListsStatus(uint32 *highPriorityTxPdReadylistStatus)
{
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_LO_PRI_TX_READY_INT_REG_INDEX], highPriorityTxPdReadylistStatus);
}

/**********************************************************************************

HwQManager_GetHighPriorityTxPdReadyListsStatus 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	uint32 * lowPriorityTxPdReadylistStatus
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_GetHighPriorityTxPdReadyListsStatus(uint32 *lowPriorityTxPdReadylistStatus)
{
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_HI_PRI_TX_READY_INT_REG_INDEX], lowPriorityTxPdReadylistStatus);
}

/**********************************************************************************

HwQManager_ClearTxListsInt 

Description:
------------
	clear interrupt of a specific Tx list

Input:
-----
	txListIndex - the index of the list

Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_ClearTxListsInt(uint32 txListIndex)
{
	RegQManagerCpu2TxIntClr031_u 	txIntClr;
	uint8 							regOffset;

	regOffset = CONVERT_WORDS_TO_BYTES(txListIndex >> 5); // Choose if the first register is used or the second. Each register holds 32 lists. Offset is in words so it's converted to bytes.
	txIntClr.val = (1 << (txListIndex & 0x1F)); // Get the correct bit in the chosen index

	RegAccess_Write((interruptsRegInterfaceAddresses[HW_Q_MANAGER_TX_INT_CLR_REG_INDEX] + regOffset), txIntClr.val);
}

/**********************************************************************************

HwQManager_ClearDisableListsInt 

Description:
------------
clear interrupt of a specific disable list

Input:
-----
disableListIndex - the index of the list

Output:
-------
	
Returns:
--------
	
**********************************************************************************/
void HwQManager_ClearDisableListsInt(uint32 disableListIndex)
{
	RegQManagerCpu2DisableIntClr_u disableListClearInterruptRegister;
	
	disableListClearInterruptRegister.val = 1 << disableListIndex;
	RegAccess_Write(interruptsRegInterfaceAddresses[HW_Q_MANAGER_DISABLE_INT_CLR_REG_INDEX], disableListClearInterruptRegister.val);
}


/**********************************************************************************

HwQManager_ClearRxListsInt 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	HwQManagerIntId_e intId
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_ClearRxListsInt(uint32  rxListIndex)
{
	RegQManagerCpu2RxIntClr031_u	rxIntClr;
	uint8							regOffset;

	regOffset = CONVERT_WORDS_TO_BYTES(rxListIndex >> 5); // Choose if the first register is used or the second. Each register holds 32 lists. Offset is in words so it's converted to bytes.
	rxIntClr.val = (1 << (rxListIndex & 0x1F)); // Get the correct bit in the chosen index

	RegAccess_Write((interruptsRegInterfaceAddresses[HW_Q_MANAGER_RX_INT_CLR_REG_INDEX] + regOffset), rxIntClr.val);
}


/**********************************************************************************

HwQManager_ErrorEvent 

Description:
------------
	<Description of the purpose of the function>

Input:
-----
	void
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_ErrorEvent(void)
{
	RegQManagerCpu2Err_u 	errorReg;
	RegDlmDlm0ErrStatus_u	Dlm0ErrStatusReg;
	RegDlmDlm1ErrStatus_u	Dlm1ErrStatusReg;
	RegDlmDlm2ErrStatus_u	Dlm2ErrStatusReg;
	RegDlmDlm3ErrStatus_u	Dlm3ErrStatusReg;
	RegDlmDlm4ErrStatus_u	Dlm4ErrStatusReg;
	RegDlmDlm5ErrStatus_u	Dlm5ErrStatusReg;
	RegDlmDlm6ErrStatus_u	Dlm6ErrStatusReg;
	RegDlmDlm7ErrStatus_u	Dlm7ErrStatusReg;
	RegDlmDlm8ErrStatus_u	Dlm8ErrStatusReg;

	// Read error reg of this CPU
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_ERR_REG_INDEX], &errorReg.val);
	ILOG2_D("HwQManager_ErrorEvent: errorReg = 0x%X",errorReg.val);

	// Read all DLMs error status regs
	RegAccess_Read(REG_DLM_DLM0_ERR_STATUS, &Dlm0ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM1_ERR_STATUS, &Dlm1ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM2_ERR_STATUS, &Dlm2ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM3_ERR_STATUS, &Dlm3ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM4_ERR_STATUS, &Dlm4ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM5_ERR_STATUS, &Dlm5ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM6_ERR_STATUS, &Dlm6ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM7_ERR_STATUS, &Dlm7ErrStatusReg.val);
	RegAccess_Read(REG_DLM_DLM8_ERR_STATUS, &Dlm8ErrStatusReg.val);

	ILOG2_DDDD("DLM status registers: dlm0=0x%X, dlm1=0x%X, dlm2=0x%X, dlm3=0x%X",
		Dlm0ErrStatusReg.val, Dlm1ErrStatusReg.val, Dlm2ErrStatusReg.val, Dlm3ErrStatusReg.val);
	ILOG2_DDDD("DLM status registers: dlm4=0x%X, dlm5=0x%X, dlm6=0x%X, dlm7=0x%X",
		Dlm4ErrStatusReg.val, Dlm5ErrStatusReg.val, Dlm6ErrStatusReg.val, Dlm7ErrStatusReg.val);
	ILOG2_D("DLM status registers: dlm8=0x%X",
		Dlm8ErrStatusReg.val);

	FATAL("HW Q Manager Error Event");
}

/**********************************************************************************

HwQManager_GetEmptyTxDoneLists 

Description:
------------
	return bitmap of tx done list that became empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became empty
**********************************************************************************/
uint16 HwQManager_GetEmptyTxDoneLists(void)
{
	RegQManagerCpu2TxDoneInt_u doneListRegister;
	doneListRegister.val = 0;
	
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_TX_DONE_INT_REG_INDEX], &doneListRegister.val);
	DEBUG_ASSERT(!doneListRegister.bitFields.txDoneListsEmptyInt);

	return doneListRegister.bitFields.txDoneListsEmptyInt;
}


/**********************************************************************************

HwQManager_GetNotEmptyTxDoneLists 

Description:
------------
	return bitmap of tx done list that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxDoneLists(void)
{
	RegQManagerCpu2TxDoneInt_u doneListsRegister;
	doneListsRegister.val = 0;
		
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_TX_DONE_INT_REG_INDEX], &doneListsRegister.val);

	DEBUG_ASSERT(doneListsRegister.bitFields.txDoneListsNotEmptyInt);

	return doneListsRegister.bitFields.txDoneListsNotEmptyInt;
}

/**********************************************************************************

HwQManager_GetNotEmptyTxDisableLists 

Description:
------------
return bitmap of tx disable lists that became not empty

Input:
-----
	
Output:
-------
	
Returns:
--------
bitmap of tx disable lists that became not empty

**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxDisableLists(void)
{
	RegQManagerCpu2DisableInt_u disableListsRegister;
	disableListsRegister.val = 0;
		
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_DISABLE_INT_REG_INDEX], &disableListsRegister.val);

	DEBUG_ASSERT(disableListsRegister.bitFields.disQueueNotEmptyInt == (disableListsRegister.bitFields.disQueueNotEmptyInt & HW_Q_MANAGER_TX_DISABLE_LISTS_ENABLED_Q));
	
	return disableListsRegister.bitFields.disQueueNotEmptyInt;
}


/**********************************************************************************

HwQManager_GetNotEmptyTxReadyLoPriLists 

Description:
------------
	return bitmap of tx ready low priority list that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxReadyLoPriLists(void)
{
	RegQManagerCpu2LoPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
		
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_LO_PRI_TX_READY_INT_REG_INDEX], &readyListRegister.val);
	return readyListRegister.bitFields.txLoPriReadyNotEmptyInt;
}

/**********************************************************************************

HwQManager_GetEmptyTxReadyLoPriLists 

Description:
------------
return bitmap of tx ready low priority lists

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of txready low priority lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetEmptyTxReadyLoPriLists(void)
{
	RegQManagerCpu2LoPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
				   
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_LO_PRI_TX_READY_INT_REG_INDEX], &readyListRegister.val);
	return readyListRegister.bitFields.txLoPriReadyEmptyInt;
}

/**********************************************************************************

HwQManager_GetNotEmptyTxReadyHiPriLists 

Description:
------------
	return bitmap of tx ready high priority list that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of tx done lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyTxReadyHiPriLists(void)
{
	RegQManagerCpu2HiPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
		
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_HI_PRI_TX_READY_INT_REG_INDEX], &readyListRegister.val);
	return readyListRegister.bitFields.txHiPriReadyNotEmptyInt;
}

/**********************************************************************************

HwQManager_GetEmptyTxReadyLowPriLists 

Description:
------------
return bitmap of tx ready low priority lists

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of txready low priority lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetEmptyTxReadyHiPriLists(void)
{
	RegQManagerCpu2HiPriTxReadyInt_u readyListRegister;
	readyListRegister.val = 0;
				   
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_HI_PRI_TX_READY_INT_REG_INDEX], &readyListRegister.val);
	return readyListRegister.bitFields.txHiPriReadyEmptyInt;
}


/**********************************************************************************

HwQManager_GetNotEmptyRxReadyLists 

Description:
------------
return bitmap of rx ready lists (RDs that were handled to FW by HW) that became not empty

Input:
-----
	void
Output:
-------
	
Returns:
--------
	bitmap of rx ready lists that became not empty
**********************************************************************************/
uint16 HwQManager_GetNotEmptyRxReadyLists(void)
{
	RegQManagerCpu2HiPriRxReadyInt_u rxReadyRdListRegister;
	rxReadyRdListRegister.val = 0;
				   
	RegAccess_Read(interruptsRegInterfaceAddresses[HW_Q_MANAGER_HI_PRI_RX_READY_INT_REG_INDEX], &rxReadyRdListRegister.val);
	return rxReadyRdListRegister.bitFields.rxHiPriReadyNotEmptyInt;
}



/**********************************************************************************

HwQManager_RemovePdFromQueue 

Description:
------------
removes provided pd from queue specified in hwQueueManagerRequestParams.

Input:
-----
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams - contains queue description
	TxPd_t* pd - pointer to pd that must be removed from queue
	
Output:
-------
	
Returns:
--------
	Returns TRUE if pd was found in queue and successfully removed, FALSE otherwise
**********************************************************************************/
bool HwQManager_RemovePdFromQueue(HwQueueManagerRequestParams_t *hwQueueManagerRequestParams, TxPd_t* removePd)
{
	TxPd_t*				prevPd = (TxPd_t *)NULL_PD;
	TxPd_t*				pd;
	bool				pdFound = FALSE;


	ILOG2_V("HwQManager_RemovePdFromQueue");

	HwQManager_FlushQ(hwQueueManagerRequestParams);

	// Try to find the PD in the list
	if (hwQueueManagerRequestParams->pHeadDesc != (TxPd_t *)NULL_PD)
	{
		pd = hwQueueManagerRequestParams->pHeadDesc;
		while (pd != (TxPd_t *)NULL_PD)
		{
			if (pd == removePd)
			{
				// PD was found
				pdFound = TRUE;
				if (prevPd != (TxPd_t *)NULL_PD)
				{
					prevPd->nextPdPointer = SET_NEXT_PD(GET_NEXT_PD(pd));
				}
				else
				{
					// PD was head of queue
					hwQueueManagerRequestParams->pHeadDesc = (void *)GET_NEXT_PD(pd);
				}
				if (pd->nextPdPointer == NEXT_PD_NULL)
				{
					// PD was tail of queue
					hwQueueManagerRequestParams->pTailDesc = (void *)prevPd;
				}
				if(!((pd->pdType == PD_TYPE_CONTROL_UNENC) && (pd->ctrlFrameSubtype == CTRL_FRAME_SUBTYPE_CTS_2_SELF)))
				{
					// Restore DRAM pointer for all Pds except Control Unenc - this pd has no Dram field
					pd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(pd->packetPointer);  
				}

				// Quit the loop
				break;
			}
			prevPd = pd;
			pd = (TxPd_t *)GET_NEXT_PD(pd);
		}

		// Push the list back to STA queue - we may have removed the only PD so check we have something in list
		if (hwQueueManagerRequestParams->pHeadDesc != (TxPd_t *)NULL_PD)
		{
			HwQManager_PushPacketlistToHead(hwQueueManagerRequestParams); 
		}
	}

	return pdFound;
}


/**********************************************************************************

HwQManagerPushToDlmQ

Description:
------------
	Push descriptors to the DLM of the DMA

Input:
-----
	head and tail of the DMA descriptors list + List index
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManagerPushToDlmQ(uint8 dlmIndex, uint32 headAddr,uint32 tailAddr,uint8 dplIndex)
{
		TX_INTERRUPT_SAVE_AREA;
		/* These registers are identical for LMACs and UMAC */
		RegQManagerCpu2Ctl_u			controlReg;
		RegQManagerCpu2PushAddrTop_u	addrTopReg;
		RegQManagerCpu2PushAddrBottom_u addrBottomReg;
		volatile RegQManagerCpu2FifoStatus_u 	fifoStatusReg;
		uint32							controlRegAddress = 0;
		uint32							addrTopRegAddress = 0;
		uint32							addrBottomRegAddress = 0;	
		uint32							fifoStatusRegAddress = 0;
		uint32							timeStamp = 0;
	
		controlReg.val = 0;
		addrTopReg.val = 0;
		addrBottomReg.val = 0;
	
		/* Fill Control Register values*/
		controlReg.bitFields.ctlSetNull		= TRUE;
		controlReg.bitFields.ctlDplIdx 		= dplIndex;
		controlReg.bitFields.ctlDlmIdx 		= dlmIndex;
		controlReg.bitFields.ctlReq			= HW_Q_MANAGER_PUSH_PACKET_LIST_TO_TAIL;
	
		/* Fill Address Registers */
		addrTopReg.bitFields.pushAddrTop = headAddr;
		addrBottomReg.bitFields.pushAddrBottom = tailAddr;
	
	
		/* Set the required registers addresses */
		controlRegAddress		= regInterfaceAddresses[HW_Q_MANAGER_REG_IF_NUM_ONE][HW_Q_MANAGER_CONTROL_REG_INDEX];
		addrTopRegAddress		= regInterfaceAddresses[HW_Q_MANAGER_REG_IF_NUM_ONE][HW_Q_MANAGER_ADDR_TOP_REG_INDEX];
		addrBottomRegAddress	= regInterfaceAddresses[HW_Q_MANAGER_REG_IF_NUM_ONE][HW_Q_MANAGER_ADDR_BOTTOM_REG_INDEX]; 
		fifoStatusRegAddress	= regInterfaceAddresses[HW_Q_MANAGER_REG_IF_NUM_ONE][HW_Q_MANAGER_FIFO_STATUS_REG_INDEX];
	
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
		// Make sure the fifo of the Hw Queue Manager is not full (can't accept more requests)
		RegAccess_Read(fifoStatusRegAddress, &fifoStatusReg.val);
#ifndef ARC_SIM
		timeStamp = GET_TSF_TIMER_LOW();

		//KW_FIX_FW_M - pragma ignore as register read & timeout check with an assert
		while (fifoStatusReg.bitFields.swFifoFull == 1)
		{
			RegAccess_Read(fifoStatusRegAddress, (uint32 *)&fifoStatusReg.val);
			ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= HW_Q_MANAGER_MAX_MICROSECONDS_FOR_FIFO_NOT_FULL); 
			
		}
#endif
	
		// Configure the HW (writing the control register "kick" the push command)
		RegAccess_Write(addrTopRegAddress,		addrTopReg.val);
		RegAccess_Write(addrBottomRegAddress,	addrBottomReg.val);
		RegAccess_Write(controlRegAddress,		controlReg.val);
		
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
}




#ifdef ENET_INC_UMAC
/**********************************************************************************

HwQManagerAger_ConfigureAllAndActivatePeriodicCheck 


Description:
------------
Configure all parameters and activate periodic check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the periodic check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_ConfigureAllAndActivatePeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters)
{
	RegQManagerCpu2AgerCfg_u globalParametersRegister;
	RegQManagerCpu2AgerAcThreshCriteria0_u ac0MinimumPdsThreshold;
	RegQManagerCpu2AgerAcThreshCriteria1_u ac1MinimumPdsThreshold;
	RegQManagerCpu2AgerAcThreshCriteria2_u ac2MinimumPdsThreshold;
	RegQManagerCpu2AgerAcThreshCriteria3_u ac3MinimumPdsThreshold;
	RegQManagerCpu2AgerEn_u  periodicCheckControlRegister;
	
	//ILOG0_V("[HwQManagerAger_ConfigureAllAndActivatePeriodicCheck]");
	globalParametersRegister.val = 0;
	ac0MinimumPdsThreshold.val = 0;
	ac1MinimumPdsThreshold.val = 0;
	ac2MinimumPdsThreshold.val = 0;
	ac3MinimumPdsThreshold.val = 0;	
	periodicCheckControlRegister.val = 0;

    /* Update only thr Ager period (in TTL units) */
	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_CFG, &globalParametersRegister.val);
    globalParametersRegister.bitFields.agerPeriod = agerCheckParameters->agerPeriod;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_CFG, globalParametersRegister.val);
	
    /* Update minimum PDs threshold */
	ac0MinimumPdsThreshold.bitFields.beThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BE];
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA0, ac0MinimumPdsThreshold.val);

	ac1MinimumPdsThreshold.bitFields.bkThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BK];
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA1, ac1MinimumPdsThreshold.val);
	
	ac2MinimumPdsThreshold.bitFields.viThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI];
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA2, ac2MinimumPdsThreshold.val);
	
	ac3MinimumPdsThreshold.bitFields.voThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VO];
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA3, ac3MinimumPdsThreshold.val);
	
	periodicCheckControlRegister.bitFields.agerEn = TRUE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_EN, periodicCheckControlRegister.val);	
}

/**********************************************************************************

HwQManagerAger_ReconfigurePartialPeriodicCheck 


Description:
------------
Stop the periodic check and reconfigure only the TTL criteria and the ager period 
and then activate the Ager periodic check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the periodic check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_ReconfigurePartialPeriodicCheck(AgerPeriodicCheckParameters_t  *agerCheckParameters)
{
	RegQManagerCpu2AgerCfg_u globalParametersRegister;
	RegQManagerCpu2AgerEn_u  periodicCheckControlRegister;
	//ILOG0_V("[HwQManagerAger_ReconfigurePartialPeriodicCheck]");
	globalParametersRegister.val = 0;
	periodicCheckControlRegister.val = 0;

    /* First stop the periodic check if needed */ 
	HwQManagerAger_StopPeriodicCheck();

    /* Update only the Ager period (in TTL units) */
	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_CFG, &globalParametersRegister.val);
    globalParametersRegister.bitFields.agerPeriod = agerCheckParameters->agerPeriod;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_CFG, globalParametersRegister.val);

	periodicCheckControlRegister.bitFields.agerEn = TRUE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_EN, periodicCheckControlRegister.val);	
}



/**********************************************************************************

HwQManagerAger_StopPeriodicCheck 


Description:
------------
Stops Ager periodic check

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_StopPeriodicCheck(void)
{
	RegQManagerCpu2AgerEn_u  periodicCheckControlRegister;
	volatile RegQManagerCpu2AgerCheckStatus_u statusRegister;
	//ILOG0_V("[HwQManagerAger_StopPeriodicCheck]");
	periodicCheckControlRegister.val = 0;
	statusRegister.val = 0;

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_EN, &periodicCheckControlRegister.val);

	periodicCheckControlRegister.bitFields.agerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_EN, periodicCheckControlRegister.val);	

	statusRegister.bitFields.singleCheckInWait = TRUE;
	while(statusRegister.bitFields.singleCheckInWait)
	{
		RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_CHECK_STATUS, &statusRegister.val);
	}
}


/**********************************************************************************

HwQManagerAger_StartSingleCheck 


Description:
------------
Activates Ager single check

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the single check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_StartSingleCheck(AgerSingleCheckParameters_t  *agerCheckParameters, bool restrictedAc)
{
	RegQManagerCpu2AgerAcLeaveCriteria_u acsCriteriaRegister;
	RegQManagerCpu2AgerAcThreshCriteria0_u ac0MinimumPdsThreshold;
	RegQManagerCpu2AgerAcThreshCriteria1_u ac1MinimumPdsThreshold;	
	RegQManagerCpu2AgerAcThreshCriteria2_u ac2MinimumPdsThreshold;
	RegQManagerCpu2AgerAcThreshCriteria3_u ac3MinimumPdsThreshold;
	RegQManagerCpu2AgerSingleCheckCfg_u singleCheckConfigurationRegister;
	RegQManagerCpu2AgerSinglePdLimit_u singleCheckPdsLimitRegister;
	RegQManagerCpu2AgerSingleCheckActivate_u singleCheckControlRegister;
	RegQManagerCpu2AgerCounterClr_u clearCountersRegister;
	
	//ILOG0_V("[HwQManagerAger_StartSingleCheck]");
	
	acsCriteriaRegister.val = 0;
	ac0MinimumPdsThreshold.val = 0;	
	ac1MinimumPdsThreshold.val = 0;
	ac2MinimumPdsThreshold.val = 0;
	ac3MinimumPdsThreshold.val = 0;
	singleCheckConfigurationRegister.val = 0;
	singleCheckPdsLimitRegister.val = 0;
	singleCheckControlRegister.val = 0;	
	clearCountersRegister.val = 0;

    clearCountersRegister.val = AGER_CLEAR_ALL_COUNTERS_MASK;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_COUNTER_CLR, clearCountersRegister.val);
	
    /* Update all the values together */
	//acsCriteriaRegister.val = *((uint32 *)(agerCheckParameters->generalParameters.acTtlCriteria));
	MEMCPY(&acsCriteriaRegister.val, &agerCheckParameters->generalParameters.acTtlCriteria[0], sizeof(uint32));
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_LEAVE_CRITERIA, acsCriteriaRegister.val);

    /* Update minimum PDs threshold */
	ac0MinimumPdsThreshold.bitFields.beThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BE];	
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA0, ac0MinimumPdsThreshold.val);
	
	ac1MinimumPdsThreshold.bitFields.bkThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BK];
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA1, ac1MinimumPdsThreshold.val);
	
	ac2MinimumPdsThreshold.bitFields.viThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI];;	
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA2, ac2MinimumPdsThreshold.val);
	
	ac3MinimumPdsThreshold.bitFields.voThreshCriteria = agerCheckParameters->generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VO];;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA3, ac3MinimumPdsThreshold.val);
	
    singleCheckConfigurationRegister.bitFields.forceLock = (restrictedAc == TRUE) ? 1 : agerCheckParameters->forceLock;
	singleCheckConfigurationRegister.bitFields.startStaIdx = agerCheckParameters->stationOrVapIndexToStart;
	singleCheckConfigurationRegister.bitFields.startStaIdx |=  agerCheckParameters->isStartingFromVap << AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_SINGLE_CHECK_CFG, singleCheckConfigurationRegister.val);

	singleCheckPdsLimitRegister.bitFields.singlePdDiscardLimit = agerCheckParameters->maximumPdsToDiscard;
	singleCheckPdsLimitRegister.bitFields.singlePdDiscardLimitEn = agerCheckParameters->isEnabledMaximumLimit;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_SINGLE_PD_LIMIT, singleCheckPdsLimitRegister.val);

	singleCheckControlRegister.bitFields.ageSingleCheckWaitRelease = TRUE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_SINGLE_CHECK_ACTIVATE, singleCheckControlRegister.val);
}

/**********************************************************************************

HwQManagerAger_StopPeriodicCheckAndStartSingleCheck 


Description:
------------
Stops the periodic check and then activates the Ager single check 

Input:
-----
agerCheckParameters- a pointer to the structure that contains all the single check 
parameters


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_StopPeriodicCheckAndStartSingleCheck(AgerSingleCheckParameters_t  *agerCheckParameters, bool restrictedAc)
{
	//ILOG0_V("[HwQManagerAger_StopPeriodicCheckAndStartSingleCheck]");
	HwQManagerAger_StopPeriodicCheck();
	HwQManagerAger_StartSingleCheck(agerCheckParameters, restrictedAc);
}


/**********************************************************************************

HwQManagerAger_GetLastCheckParameters 


Description:
------------
Returns the parameters of the last Ager check

Input:
-----
lastCheckParameters - pointer to the strcutre to fill


Output:
-------
The structure contains the needed parameters	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_GetLastCheckParameters(AgerLastCheckParametes_t *lastCheckParameters)
{
	RegQManagerCpu2AgerCheckMostStatus_u regQManagerCpu2AgerCheckMostStatus;
	RegQManagerCpu2AgerCheckStatus_u regQManagerCpu2AgerCheckStatus;
	//ILOG0_V("[HwQManagerAger_GetLastCheckParameters]");
	regQManagerCpu2AgerCheckMostStatus.val = 0;
	regQManagerCpu2AgerCheckStatus.val = 0;

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_CHECK_MOST_STATUS, &regQManagerCpu2AgerCheckMostStatus.val);
	lastCheckParameters->tid =  regQManagerCpu2AgerCheckMostStatus.bitFields.lastTidChecked;
	lastCheckParameters->isVapQueue = regQManagerCpu2AgerCheckMostStatus.bitFields.lastStaChecked >> AGER_SINGLE_CHECK_STARTING_FROM_VAP_BIT;
	regQManagerCpu2AgerCheckMostStatus.bitFields.lastStaChecked &= ~AGER_SINGLE_CHECK_IS_VAP_QUEUE_MASK;
	lastCheckParameters->index =  regQManagerCpu2AgerCheckMostStatus.bitFields.lastStaChecked;

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_CHECK_STATUS, &regQManagerCpu2AgerCheckStatus.val);
	lastCheckParameters->isSingleCheckDone = regQManagerCpu2AgerCheckStatus.bitFields.singleCheckDone;
}


/**********************************************************************************

HwQManagerAger_GetLastCheckCounters 


Description:
------------
Returns the counters of the last Ager check


Input:
-----
lastCheckCounters - pointer to the strcutre to fill


Output:
-------
The structure contains the counters	

Returns:
--------

	
**********************************************************************************/
void HwQManagerAger_GetLastCheckCounters(AgerLastCheckCounters_t *lastCheckCounters)
{
	RegQManagerCpu2AgerTryFailCounter_u lockFailuresCounterRegister;
	RegQManagerCpu2AgerUmacDoneCounter_u regQManagerCpu2AgerUmacDoneCounter;
	RegQManagerCpu2AgerLibDoneCounter_u regQManagerCpu2AgerLibDoneCounter;
	//ILOG0_V("[HwQManagerAger_GetLastCheckCounters]");
	lockFailuresCounterRegister.val = 0;
	regQManagerCpu2AgerUmacDoneCounter.val = 0;
	regQManagerCpu2AgerLibDoneCounter.val = 0;

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_TRY_FAIL_COUNTER, &lockFailuresCounterRegister.val);
	lastCheckCounters->numberOfLockFailures = lockFailuresCounterRegister.bitFields.tryFailCounter;

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_UMAC_DONE_COUNTER, &regQManagerCpu2AgerUmacDoneCounter.val);
	lastCheckCounters->numberOfPdsHandledToUmac = regQManagerCpu2AgerUmacDoneCounter.bitFields.umacCounter;

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_LIB_DONE_COUNTER, &regQManagerCpu2AgerLibDoneCounter.val);
	lastCheckCounters->numberOfPdsHandledToLiberator = regQManagerCpu2AgerLibDoneCounter.bitFields.libCounter;
}

/**********************************************************************************

HwQManagerAger_ClearSingleCheckDoneInterrupt 


Description:
------------
Returns the counters of the last Ager check


Input:
-----
lastCheckCounters - pointer to the strcutre to fill


Output:
-------
The structure contains the counters	

Returns:
--------

	
**********************************************************************************/
void HwQManagerAger_ClearSingleCheckDoneInterrupt()
{
	RegQManagerCpu2AgerIrqClr_u clearSingleCheckDoneInterruptRegister;
	//ILOG0_V("[HwQManagerAger_ClearSingleCheckDoneInterrupt]");

	
	clearSingleCheckDoneInterruptRegister.val = 0;

	clearSingleCheckDoneInterruptRegister.bitFields.agerDoneIrqClr = TRUE;

	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_IRQ_CLR, clearSingleCheckDoneInterruptRegister.val);
}

/**********************************************************************************

HwQManagerAger_SetStaTtlVal 


Description:
------------



Input:
-----



Output:
-------


Returns:
--------

	
**********************************************************************************/

void HwQManagerAger_SetStaTtlVal(StaId staInd, uint16 Value)
{

	RegQManagerCpu2AgerStaTtl_u regQManagerCpu2AgerStaTtl;
	//ILOG0_DD("[HwQManagerAger_SetStaTtlVal] - staInd 0x%x  Value 0x%x ",staInd,Value);

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_STA_TTL,&regQManagerCpu2AgerStaTtl.val);
	
	ASSERT(regQManagerCpu2AgerStaTtl.bitFields.staTtlDone == TRUE);

	regQManagerCpu2AgerStaTtl.bitFields.staTtlValueWr = Value;
	regQManagerCpu2AgerStaTtl.bitFields.staTtlStaIdx = staInd;
	regQManagerCpu2AgerStaTtl.bitFields.staTtlWr = TRUE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_STA_TTL, regQManagerCpu2AgerStaTtl.val);
}
/**********************************************************************************

HwQManagerAger_GetStaTtlVal 


Description:
------------



Input:
-----



Output:
-------


Returns:
--------

	
**********************************************************************************/

uint16 HwQManagerAger_GetStaTtlVal(StaId staInd)
{

	RegQManagerCpu2AgerStaTtl_u regQManagerCpu2AgerStaTtl;
	//ILOG0_D("[HwQManagerAger_GetStaTtlVal] - staInd 0x%x",staInd);

	RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_STA_TTL,&regQManagerCpu2AgerStaTtl.val);
	
	ASSERT(regQManagerCpu2AgerStaTtl.bitFields.staTtlDone == TRUE);

	regQManagerCpu2AgerStaTtl.bitFields.staTtlStaIdx = staInd;
	regQManagerCpu2AgerStaTtl.bitFields.staTtlWr = FALSE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_STA_TTL, regQManagerCpu2AgerStaTtl.val);

	do
	{		
		RegAccess_Read(REG_Q_MANAGER_CPU2_AGER_STA_TTL,&regQManagerCpu2AgerStaTtl.val);
	} while (regQManagerCpu2AgerStaTtl.bitFields.staTtlDone == FALSE);
	
	return regQManagerCpu2AgerStaTtl.bitFields.staTtlValueRd;
	
}

/**********************************************************************************

HwQManagerAger_Initialize 


Description:
------------
Initialize Ager internal registers

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwQManagerAger_Initialize(void)
{
	RegQManagerCpu2AgerCfg_u globalParametersRegister;
	RegQManagerCpu2AgerListsCfg_u listsConfigurationRegister;
	RegQManagerCpu2AgerAcThreshCriteria0_u pdsLimitThresholdAc0;	
	RegQManagerCpu2AgerAcThreshCriteria1_u pdsLimitThresholdAc1;
	RegQManagerCpu2AgerAcThreshCriteria2_u pdsLimitThresholdAc2;
	RegQManagerCpu2AgerAcThreshCriteria3_u pdsLimitThresholdAc3;
	RegQManagerCpu2AgerTmpListCfg_u tempListConfigurationRegister;

	globalParametersRegister.val = 0;
	listsConfigurationRegister.val = 0;
	pdsLimitThresholdAc0.val = 0;	
	pdsLimitThresholdAc1.val = 0;
	pdsLimitThresholdAc2.val = 0;
	pdsLimitThresholdAc3.val = 0;
	tempListConfigurationRegister.val = 0;

	globalParametersRegister.bitFields.shramWaitState = AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_SHRAM_ACCESS;
	globalParametersRegister.bitFields.clientWaitState = AGER_NUMBER_OF_WAIT_CYCLES_BETWEEN_CLIENTS;
//	globalParametersRegister.bitFields.dropRetryPdListEn = TRUE;
	globalParametersRegister.bitFields.retryDiscardLock = FALSE;
    RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_CFG, globalParametersRegister.val);

#if defined (AGER_STATUS_WORKAROUND)
	listsConfigurationRegister.bitFields.liberatorListIdx = HW_Q_MANAGER_DONE_LIST_AGER_LIBERATOR;
#else			
	listsConfigurationRegister.bitFields.liberatorListIdx = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
#endif
	listsConfigurationRegister.bitFields.umacListIdx = HW_Q_MANAGER_LOW_PR_READY_LIST_AGER;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_LISTS_CFG, listsConfigurationRegister.val);

	tempListConfigurationRegister.bitFields.umacTmpListIdx = HW_Q_MANAGER_READY_LIST_AGER_TEMPORARY_LIST;
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_TMP_LIST_CFG, tempListConfigurationRegister.val); 

	/* The starting PDs limit threshold is 0 */
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA0, pdsLimitThresholdAc0.val);	
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA1, pdsLimitThresholdAc1.val);
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA2, pdsLimitThresholdAc2.val);
	RegAccess_Write(REG_Q_MANAGER_CPU2_AGER_AC_THRESH_CRITERIA3, pdsLimitThresholdAc3.val);
}

#endif /* #ifdef ENET_INC_UMAC */

/**********************************************************************************

HwQManager_Init 

Description:
------------
	Init DLM Rams (Um Only)
	Init interrupt configuration 

Input:
-----
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwQManager_Init(void)
{
#if defined(ENET_INC_UMAC)
	RegQManagerCpu2TxListsNotEmptyIntEn031_u 	txNotEmptyInterruptEnable031Register;
	RegQManagerCpu2TxListsNotEmptyIntEn3263_u	txNotEmptyInterruptEnable3263Register;
	RegQManagerCpu2RxListsNotEmptyIntEn031_u 	rxNotEmptyInterruptEnable031Register;
	RegQManagerCpu2RxListsNotEmptyIntEn3263_u	rxNotEmptyInterruptEnable3263Register;
	RegQManagerCpu2DisableNotEmptyIntEn_u 		disableListsinterruptEnableRegister;
	

    txNotEmptyInterruptEnable031Register.val = 0;
	rxNotEmptyInterruptEnable031Register.val = 0;
	disableListsinterruptEnableRegister.val  = 0;

	hwQManagerInitDlm();
	
	/* Configure UM Register Interface. Currently we don't want interrupts in LM0/1 */
	txNotEmptyInterruptEnable031Register.val =	HW_Q_MANAGER_TX_ENABLED_Q_0_31; 
	RegAccess_Write(REG_Q_MANAGER_CPU2_TX_LISTS_NOT_EMPTY_INT_EN0_31,txNotEmptyInterruptEnable031Register.val);
	txNotEmptyInterruptEnable3263Register.val = HW_Q_MANAGER_TX_ENABLED_Q_32_63;
	RegAccess_Write(REG_Q_MANAGER_CPU2_TX_LISTS_NOT_EMPTY_INT_EN32_63,txNotEmptyInterruptEnable3263Register.val);

	rxNotEmptyInterruptEnable031Register.val = HW_Q_MANAGER_RX_ENABLED_Q_0_31;
	RegAccess_Write(REG_Q_MANAGER_CPU2_RX_LISTS_NOT_EMPTY_INT_EN0_31,rxNotEmptyInterruptEnable031Register.val);
	rxNotEmptyInterruptEnable3263Register.val = HW_Q_MANAGER_RX_ENABLED_Q_32_63;
	RegAccess_Write(REG_Q_MANAGER_CPU2_RX_LISTS_NOT_EMPTY_INT_EN32_63,rxNotEmptyInterruptEnable3263Register.val);
	
	disableListsinterruptEnableRegister.val = 	HW_Q_MANAGER_TX_DISABLE_LISTS_ENABLED_Q;
	RegAccess_Write(REG_Q_MANAGER_CPU2_DISABLE_NOT_EMPTY_INT_EN, disableListsinterruptEnableRegister.val);
#endif

}





