/***********************************************************************************
 File:		HwSemaphore.c
 Module:		Semaphore
 Purpose:		
 Description:	FW driver for HW semaphore
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwSemaphore_API.h"
#include "OSAL_Api.h"
#include "RegAccess_Api.h"
#include "MacHtExtensionsRegs.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_UTILS   
#define LOG_LOCAL_FID 0


#define HW_SEMAPHORE_LOOP_COUNTER 32 /* Get 1 microSecond delay: one write to SHRAM is 1/32 microSecond */
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
static TX_INTERRUPT_SAVE_AREA;
uint32 delayCounter;
uint32 hwSemaphoreShadow = 0;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
HwSemaphore_Lock  

Description:
------------
Try to lock a semaphore
Loop until success
User may keep the lock up to 4 microseconds
	 
Input:
-----
	hwSemaphoreResourceType: which semaphore should be locked
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void HwSemaphore_Lock (HwSemaphoreResourceType_e hwSemaphoreResourceType)
{
	uint32 loop = 0;
	uint32 readValue = 0;
	uint32 writeValue = (1 << hwSemaphoreResourceType);
	
	/* Reset delayCounter */
	delayCounter = 0;

	/* Check that we are not trying to lock a locked semaphore */
	ASSERT((hwSemaphoreShadow & writeValue) != writeValue);	

	//ILOG0_DDDD("[HwSemaphore_Lock 1] hwSemaphoreResourceType is %d writeValue is %d interrupt_save = %d hwSemaphoreShadow = %d", hwSemaphoreResourceType, writeValue, interrupt_save, hwSemaphoreShadow);
	
	while(1)
	{		
		/* Disable interrupts */
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		/* WRITE: try to lock the semaphore */
		RegAccess_Write(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, writeValue);

		/* READ: get the semaphore status */
		RegAccess_Read(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, &readValue);
		
		/* Check: if the semaphore is locked - exit, else - loop (try to lock again) */
		if ((readValue & writeValue) != writeValue)
		{			
			/* Enable interrupts */
			OSAL_ENABLE_INTERRUPTS(interrupt_save);
			
			/* Delay */
			for(loop = 0; loop < HW_SEMAPHORE_LOOP_COUNTER ;loop++)
			{
				/* Do nothing */
				
				/* Increment delay counter */			
				delayCounter++;
			}

			/* Loop again */
			continue;
		}
		
		break;
	}

	/* Update shadow register */
	hwSemaphoreShadow |= writeValue;
	
	//ILOG0_DDDD("[HwSemaphore_Lock 2] hwSemaphoreResourceType is %d writeValue is %d interrupt_save = %d hwSemaphoreShadow = %d", hwSemaphoreResourceType, writeValue, interrupt_save, hwSemaphoreShadow);
}
/**********************************************************************************
HwSemaphore_Free  

Description:
------------
Unlock a semaphore
	 
Input:
-----
	hwSemaphoreResourceType: which semaphore should be unlocked
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/

void HwSemaphore_Free (HwSemaphoreResourceType_e hwSemaphoreResourceType)
{
	uint32 writeValue = (1 << hwSemaphoreResourceType);

	//ILOG0_DDDD("[HwSemaphore_Free 1] hwSemaphoreResourceType is %d writeValue is %d interrupt_save = %d hwSemaphoreShadow = %d", hwSemaphoreResourceType, writeValue, interrupt_save, hwSemaphoreShadow);

	/* Check that we are not trying to free a free semaphore */	
	ASSERT((hwSemaphoreShadow & writeValue) == writeValue);	

	/* WRITE: unlock the semaphore */
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, writeValue);

	/* Update shadow register */
	hwSemaphoreShadow &= ~(writeValue);
	
	/* Enable interrupts */
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	
	//ILOG0_DDDD("[HwSemaphore_Free 2] hwSemaphoreResourceType is %d writeValue is %d interrupt_save = %d hwSemaphoreShadow = %d", hwSemaphoreResourceType, writeValue, interrupt_save, hwSemaphoreShadow);
}
/**********************************************************************************
HwSemaphore_TryLock 

Description:
------------
Try to lock a semaphore
Function returns immediately
User may keep the lock up to 4 microseconds 
	 
Input:
-----
	hwSemaphoreResourceType: which semaphore should be locked
Output:
-------
	None
Returns:
--------
	Fail/Success status
**********************************************************************************/

bool HwSemaphore_TryLock (HwSemaphoreResourceType_e hwSemaphoreResourceType)
{
	bool locked = TRUE;
	uint32 readValue = 0;
	uint32 writeValue = (1 << hwSemaphoreResourceType);	

	ILOG0_DDDD("[HwSemaphore_TryLock 1] hwSemaphoreResourceType is %d writeValue is %d interrupt_save = %d hwSemaphoreShadow = %d", hwSemaphoreResourceType, writeValue, interrupt_save, hwSemaphoreShadow);

	/* Check that we are not trying to lock a locked semaphore */
	ASSERT((hwSemaphoreShadow & writeValue) != writeValue);
	
	/* Disable interrupts */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* WRITE: try to lock the semaphore */
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, writeValue);
	
	/* READ: get the semaphore status */
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, &readValue);
	
	/* Check: if the semaphore is unlocked enable interrupts */
	if ((readValue & writeValue) != writeValue)
	{
		locked = FALSE;
		
		/* Enable interrupts */
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
	else
	{
		/* Update shadow register */
		hwSemaphoreShadow |= (writeValue);
	}
	
	ILOG0_DDDD("[HwSemaphore_TryLock 2] hwSemaphoreResourceType is %d writeValue is %d interrupt_save = %d hwSemaphoreShadow = %d", hwSemaphoreResourceType, writeValue, interrupt_save, hwSemaphoreShadow);	

	return (locked);
}
#if defined (HW_SEMAPHORE_DEBUG)
/**********************************************************************************
HwSemaphore_Read  

Description:
------------
Read HW semaphore register (debug)
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	HW semaphore status
**********************************************************************************/

uint32 HwSemaphore_Read(void)
{	
	uint32 readValue = 0;	

	/* READ: get the semaphore status */
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, &readValue);

	ILOG0_D("[HwSemaphore_Read 1] readValue is %d ", readValue);	
	
	return (readValue);
}

/**********************************************************************************
HwSemaphore_FreeAllDebug  

Description:
------------
Free all locked semaphore (debug)
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void HwSemaphore_FreeAllDebug(void)
{
	uint32 readValue = 0;	
	uint32 loop = 0;	

	ILOG0_D("[HwSemaphore_FreeAllDebug 1] hwSemaphoreShadow = %d", hwSemaphoreShadow);	
	
	/* Disable interrupts */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* READ: get the semaphore status */
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, &readValue);
	
	for (loop = 0; loop < 32; loop++)
	{
		if ((readValue & (1 << loop)) == 1)
		{
			uint32 writeValue = 1 << loop;
			
			/* WRITE: try to unlock the semaphore */			
			RegAccess_Write(REG_MAC_HT_EXTENSIONS_HW_SEMAPHORE, writeValue);
		}
	}
	/* Reset shadow register */
	hwSemaphoreShadow = 0;

	/* Enable interrupts */
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}
#endif // HW_SEMAPHORE_DEBUG 
