/***********************************************************************************
 File:		HwSemaphore.c
 Module:		Semaphore
 Purpose:		
 Description:	FW driver for HW semaphore
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwSemaphore_API.h"
#include "OSAL_Api.h"
#include "RegAccess_Api.h"
#include "MacGeneralRegs.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "HostIfRegs.h"
#endif
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_UTILS   
#define LOG_LOCAL_FID 1

#define WRITE_REG_SHIFT	(5)		//remove the 5 LSB 
#define READ_REG_SHIFT	(4)		//remove the 4 LSB 
#define WRITE_VAL_MASK	(0x1F) 	//take the 5 LSB 
#define TAKEN_VAL_MASK	(0xF)  	//take the 4 LSB 
#define NUM_OF_PARAMS_IN_HW_SEMAPHORE_STATUS	(2) //taken and pending
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
static TX_INTERRUPT_SAVE_AREA;
uint32 delayCounter;
uint32 hwSemaphoreShadow[2] = {0};

#if defined (ENET_INC_UMAC)	
#ifdef ENET_INC_ARCH_WAVE600D2
uint32 hwSemaphoreToggleRegSet[2] = {
										REG_HOST_IF_CLIENT_0_SMPHR_TOGGLE_31TO00,	
										REG_HOST_IF_CLIENT_0_SMPHR_TOGGLE_63TO32	
									};
uint32 hwSemaphoreStatusRegSet[4] = {
										REG_HOST_IF_CLIENT_0_SMPHR_STAT_15TO00,
										REG_HOST_IF_CLIENT_0_SMPHR_STAT_31TO16,
										REG_HOST_IF_CLIENT_0_SMPHR_STAT_47TO32,
										REG_HOST_IF_CLIENT_0_SMPHR_STAT_63TO48
									};
#else
uint32 hwSemaphoreToggleRegSet[2] = {
										REG_MAC_GENERAL_CLIENT_0_SMPHR_TOGGLE_31TO00,	
										REG_MAC_GENERAL_CLIENT_0_SMPHR_TOGGLE_63TO32	
									};
uint32 hwSemaphoreStatusRegSet[4] = {
										REG_MAC_GENERAL_CLIENT_0_SMPHR_STAT_15TO00,
										REG_MAC_GENERAL_CLIENT_0_SMPHR_STAT_31TO16,
										REG_MAC_GENERAL_CLIENT_0_SMPHR_STAT_47TO32,
										REG_MAC_GENERAL_CLIENT_0_SMPHR_STAT_63TO48
									};
#endif
#endif
#if defined (ENET_INC_LMAC0)	
#ifdef ENET_INC_ARCH_WAVE600D2
uint32 hwSemaphoreToggleRegSet[2] = {	
										REG_HOST_IF_CLIENT_1_SMPHR_TOGGLE_31TO00,
										REG_HOST_IF_CLIENT_1_SMPHR_TOGGLE_63TO32
									};
uint32 hwSemaphoreStatusRegSet[4] = {
										REG_HOST_IF_CLIENT_1_SMPHR_STAT_15TO00,
										REG_HOST_IF_CLIENT_1_SMPHR_STAT_31TO16,
										REG_HOST_IF_CLIENT_1_SMPHR_STAT_47TO32,
										REG_HOST_IF_CLIENT_1_SMPHR_STAT_63TO48
									};
#else
uint32 hwSemaphoreToggleRegSet[2] = {	
										REG_MAC_GENERAL_CLIENT_1_SMPHR_TOGGLE_31TO00,
										REG_MAC_GENERAL_CLIENT_1_SMPHR_TOGGLE_63TO32
									};
uint32 hwSemaphoreStatusRegSet[4] = {
										REG_MAC_GENERAL_CLIENT_1_SMPHR_STAT_15TO00,
										REG_MAC_GENERAL_CLIENT_1_SMPHR_STAT_31TO16,
										REG_MAC_GENERAL_CLIENT_1_SMPHR_STAT_47TO32,
										REG_MAC_GENERAL_CLIENT_1_SMPHR_STAT_63TO48
									};
#endif
#endif

#if defined (ENET_INC_LMAC1)	
#ifdef ENET_INC_ARCH_WAVE600D2
uint32 hwSemaphoreToggleRegSet[2] = {	
										REG_HOST_IF_CLIENT_2_SMPHR_TOGGLE_31TO00,	
										REG_HOST_IF_CLIENT_2_SMPHR_TOGGLE_63TO32};
uint32 hwSemaphoreStatusRegSet[4] = {
										REG_HOST_IF_CLIENT_2_SMPHR_STAT_15TO00,
										REG_HOST_IF_CLIENT_2_SMPHR_STAT_31TO16,
										REG_HOST_IF_CLIENT_2_SMPHR_STAT_47TO32,
										REG_HOST_IF_CLIENT_2_SMPHR_STAT_63TO48
									};
#else
uint32 hwSemaphoreToggleRegSet[2] = {	
										REG_MAC_GENERAL_CLIENT_2_SMPHR_TOGGLE_31TO00,	
										REG_MAC_GENERAL_CLIENT_2_SMPHR_TOGGLE_63TO32};
uint32 hwSemaphoreStatusRegSet[4] = {
										REG_MAC_GENERAL_CLIENT_2_SMPHR_STAT_15TO00,
										REG_MAC_GENERAL_CLIENT_2_SMPHR_STAT_31TO16,
										REG_MAC_GENERAL_CLIENT_2_SMPHR_STAT_47TO32,
										REG_MAC_GENERAL_CLIENT_2_SMPHR_STAT_63TO48
									};
#endif
#endif
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
HwSemaphore_Lock  

Description:
------------
Try to lock a semaphore
Loop until success
User may keep the lock up to 4 microseconds

HW registers:
toggle - one bit for each semaphore 						
status - tow bits for each semaphore - taken and pending 
 
toggle -   |   0  |   1  |   2  |   3  |.......|   62    |   63    |
status -   | 0  1 | 2  3 | 4  5 | 6  7 |.......| 124 125 | 127 128 |

Input:
-----
	hwSemaphoreResourceType: which semaphore should be locked
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void HwSemaphore_Lock (HwSemaphoreResourceType_e hwSemaphoreResourceType)
{
	uint32 readValue = 0;
	uint32 writeRegSet = hwSemaphoreResourceType>>WRITE_REG_SHIFT;
	uint32 readRegset = hwSemaphoreResourceType>>READ_REG_SHIFT ;
	uint32 writeValue = (1 << (hwSemaphoreResourceType & WRITE_VAL_MASK));
	uint32 takenReadValue = (1 << ((hwSemaphoreResourceType & TAKEN_VAL_MASK) * NUM_OF_PARAMS_IN_HW_SEMAPHORE_STATUS));
		
	/* Reset delayCounter */
	delayCounter = 0;	

	/* Check that we are not trying to lock a locked semaphore */
	ASSERT((hwSemaphoreShadow[writeRegSet] & writeValue) != writeValue);	

	/* Disable interrupts */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/* WRITE: try to lock the semaphore */
	RegAccess_Write(hwSemaphoreToggleRegSet[writeRegSet], writeValue);
	while(1)
	{	
		/* READ: get the semaphore status */
		RegAccess_Read(hwSemaphoreStatusRegSet[readRegset], &readValue);
	
		/* Check: if the semaphore is locked - exit, else - loop  */
		if ((readValue & takenReadValue) == takenReadValue)
		{	
			//the semaphore is locked		
			break;
		}
		/* Loop again */
		delayCounter++;;
	}

	/* Update shadow register */
	hwSemaphoreShadow[writeRegSet] |= writeValue;
	
}
/**********************************************************************************
HwSemaphore_Free  

Description:
------------
Unlock a semaphore
	 
Input:
-----
	hwSemaphoreResourceType: which semaphore should be unlocked
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/

void HwSemaphore_Free (HwSemaphoreResourceType_e hwSemaphoreResourceType)
{
	uint32 writeRegSet = hwSemaphoreResourceType>>WRITE_REG_SHIFT;
	uint32 writeValue = (1 << (hwSemaphoreResourceType & WRITE_VAL_MASK));
	
	/* Check that we are not trying to free a free semaphore */	
	ASSERT((hwSemaphoreShadow[writeRegSet] & writeValue) == writeValue);	

	/* WRITE: unlock the semaphore */
	RegAccess_Write(hwSemaphoreToggleRegSet[writeRegSet], writeValue);

	/* Update shadow register */
	hwSemaphoreShadow[writeRegSet] &= ~(writeValue);
	
	/* Enable interrupts */
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}
/**********************************************************************************
HwSemaphore_TryLock 

Description:
------------
Try to lock a semaphore
Function returns immediately
User may keep the lock up to 4 microseconds 
	 
Input:
-----
	hwSemaphoreResourceType: which semaphore should be locked
Output:
-------
	None
Returns:
--------
	Fail/Success status
**********************************************************************************/

bool HwSemaphore_TryLock (HwSemaphoreResourceType_e hwSemaphoreResourceType)
{
	UNUSED_PARAM(hwSemaphoreResourceType);	
	return (FALSE);
}
