/***********************************************************************************
 File:			RadarDetection.h
 Module:		Interferer detection 
 Purpose: 		Detect none wifi interferers
 Description:	This file contains all the definitions and structures of the radar 
 				detection which is responsible to detect presence of radars in the 
 				current channel
************************************************************************************/
#ifndef RADAR_DETECTION_H_
#define RADAR_DETECTION_H_

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "lmi.h"
#include "HdkCdbManagerTask_api.h"

//---------------------------------------------------------------------------------
//						Defines						
//--------------------------------------------------------------------------------- 
#define NO_RADAR						0

#define RADAR_ETSI_MEDIUM_MIN_PW_TH	20

#define RADAR_LONG_CLUSTER_NUM		0  /*Cluster num for radar type5 (long pulse)*/
#define RADAR_SHORT_CLUSTER_NUM		1  /*Cluster num for radar type0 (0.5 usec)*/

#define MAX_SHORT_PW_VAL			104 //max val of pulse width 

#define PW_RESOLUTION				2

#define SHORT_PULSE_AGED_OUT		40000	 //short pulses(pw = 1-20) that are older than 40ms are aged out
#define MEDIUM_PULSE_AGED_OUT		12000	 //short pulses(pw = 20-30) that are older than 12ms are aged out
#define LONG_PULSE_AGED_OUT			8000000 //longpulses (pw > 50) that are older than 8sec are agfed out

#define MIN_NUM_OF_SHORT_PULSES		2	//The minimal number of short &medium  pulses we need in order to try detect radar
#define MIN_NUM_OF_ZERO_PW_PULSES	2   //The minimal number of zero pw  pulses we need in order to try detect radar
#define MIN_NUM_OF_LONG_PULSES		7   //The minimal number of long pulses we need in order to try detect radar

#define MAX_NUM_OF_PULSE_RECORDS	32  
#define NUM_OF_PULSE_CLUSTERS		MAX_NUM_OF_PULSE_RECORDS
#define FIRST_NON_RESERVED_CLUSTER  2
#define MAX_PRI_RATIO				2
#define MAX_PRI_REMAIN				2
#define MIN_PRI_COUNT				3 
#define MAX_PRI_ANALYSIS			32	

#define MIN_SHORT_PRI_LONG_RADAR_TH	2
#define MIN_LONG_PRI_LONG_RADAR_TH	3

#define PW_INVALID 					0xFF
#define PRI_INVALID 				0xFFFF

/*********** FCC radar parameters ***********/
#define FCC_RADAR_MIN_PW_TH			2
#define FCC_IGNORED_RADAR_MIN_PW	24
#define FCC_IGNORED_RADAR_MAX_PW	53
#define FCC_RADAR_MAX_PW_TH			104
#define FCC_MIN_PRI_SHORT_RADAR		148
#define FCC_MAX_PRI_SHORT_RADAR		3068
#define FCC_MIN_PRI_LONG_RADAR		998
#define FCC_MAX_PRI_LONG_RADAR		2002

/*********** ETSI (and China) radar parameters ***********/
#define ETSI_RADAR_MIN_PW_TH		2
#define ETSI_IGNORED_RADAR_MIN_PW	34
#define ETSI_IGNORED_RADAR_MAX_PW	PW_INVALID
#define ETSI_RADAR_MAX_PW_TH		PW_INVALID
#define ETSI_MIN_PRI_SHORT_RADAR	248
#define ETSI_MAX_PRI_SHORT_RADAR	5002
#define ETSI_MIN_PRI_LONG_RADAR		PRI_INVALID
#define ETSI_MAX_PRI_LONG_RADAR		PRI_INVALID

/*********** JAPAN radar parameters ***********/
#define JP_RADAR_MIN_PW_TH			2
#define JP_IGNORED_RADAR_MIN_PW		24
#define JP_IGNORED_RADAR_MAX_PW		53
#define JP_RADAR_MAX_PW_TH			104
#define JP_MIN_PRI_SHORT_RADAR		148
#define JP_MAX_PRI_SHORT_RADAR		4002
#define JP_MIN_PRI_LONG_RADAR		998
#define JP_MAX_PRI_LONG_RADAR		2002

/*********** KOREA radar parameters ***********/
#define KR_RADAR_MIN_PW_TH			2
#define KR_IGNORED_RADAR_MIN_PW		4
#define KR_IGNORED_RADAR_MAX_PW		PW_INVALID
#define KR_RADAR_MAX_PW_TH			PW_INVALID
#define KR_MIN_PRI_SHORT_RADAR		331
#define KR_MAX_PRI_SHORT_RADAR		3032
#define KR_MIN_PRI_LONG_RADAR		PRI_INVALID
#define KR_MAX_PRI_LONG_RADAR		PRI_INVALID

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
typedef struct RadarDetectionPulseRecord RadarDetectionPulseRecord_t;

struct RadarDetectionPulseRecord 
{
	RadarDetectionPulseRecord_t *nextPulseRecord;		/*next pulse record in cluster*/
	RadarDetectionPulseRecord_t *nextPulseGlobalrecord;	/*next pulse record according to time of arrival*/
	uint32 pulseTimeStamp;
	uint8  pulseWidth;
	uint8  ampSubBandBitMap;
	uint8  clusterIndex;								/*What cluster the pulse record is on*/
};

typedef struct RadarDetectionPulseCluster
{
	QPB pulseQueue;
	uint8 avgPulseWidth;
	uint8 pulseCount;
}RadarDetectionPulseCluster_t;

typedef struct RadarDetectionBlockManagement
{
	uint32 newestPulseTimeStamp;
	uint8 biggestClusterIndex;/* the largest cluster index currently in use(to minimize the search) */
	uint8 numOfNewPulseRec;
	uint8 reserved[2];
}RadarDetectionBlockManagement_t;

typedef struct RadarDetectionPulses
{
	RadarDetectionPulseCluster_t    freePulseCluster;
	RadarDetectionPulseCluster_t    pulseClustersControl[NUM_OF_PULSE_CLUSTERS]; 
	QPB								pulseRecordsQueue;	/*Need to store the records in the order they arrived, regradless of what cluster they are put on*/
	RadarDetectionPulseRecord_t	    pulseRecords[MAX_NUM_OF_PULSE_RECORDS];
	RadarDetectionBlockManagement_t blockManagement;
}RadarDetectionPulses_t;

typedef struct RadarDetectionGlobalParameters_s
{
	RadarDetectionPulses_t    pulses;
	uint8                     isIndicationEnabled;
	uint8                     channel;
	uint8                     wasRadarIndicationSent;
	uint8                     ageingTimerActive;
	uint8					  shortRadarPriTh;
	uint8					  longRadarPriTh;
	uint8					  radioState;
	uint8					  minShortPwTh;
	uint8					  maxShortPwTh;
	uint8					  minLongPwTh;
	uint8					  maxLongPwTh;
 	uint16					  shortRadarMinPriTh;
	uint16					  shortRadarMaxPriTh;
	uint16					  longRadarMinPriTh;
	uint16					  longRadarMaxPriTh;
}RadarDetectionGlobalParameters_t;
typedef struct RadarAdaptiveSensParams
{
	uint8 nonAssoDataPacketsTh;
	uint8 InterferernceRadarRatio;
	bool disableFeature;
}RadarAdaptiveSensParams_t;


typedef enum
{
	RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_ADDED_RECORD = 0,
	RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_DELETED_RECORD,
	RADAR_DETECTION_NUM_OF_AVERAGE_CALCULATION_TYPES,
	RADAR_DETECTION_MAX_NUM_OF_AVERAGE_CALCULATION_TYPES = 0xFF,	
}RadarDetectionAverageCalculationType_e;

typedef enum
{
	RADAR_ADAPTIVE_SENS_EVENT_SET_RADAR_CHANNEL_IN_CAC,
	RADAR_ADAPTIVE_SENS_EVENT_SET_RADAR_CHANNEL_IN_SERVICE,
	RADAR_ADAPTIVE_SENS_EVENT_TIMER_EXPIRED,
	RADAR_ADAPTIVE_SENS_EVENT_SET_NO_RADAR_CHANNEL,
	RADAR_ADAPTIVE_SENS_EVENT_RADAR_DETECTED,
	RADAR_ADAPTIVE_SENS_EVENTS_NUM,
	
} AdaptiveSensEvent_e;

//---------------------------------------------------------------------------------
//						Public Function Declaration				
//---------------------------------------------------------------------------------
void RadarDetection_SetChannel(uint8 channel, uint8 isRadarDetectionEnabled);
void RadarDetection_ProcessSamples(K_MSG *radarDetectionMessage);
void RadarDetection_EnableRadarIndication(K_MSG *radarDetectionMessage);
void RadarDetection_AgingTimer(K_MSG *radarDetectionMessage);
void RadarDetection_SetRadioState(uint8 radioState);
void RadarDetection_Initialize(void);
void RadarDetection_SetRegDomainParams(uint8 RegulationType);
void radarAdaptiveSensStateMachineEnterEvent(AdaptiveSensEvent_e event);
void radarDetectionAdaptiveSensUpdateParams(bool  disableFeture, uint8 nonAssoDataPacketsTh, uint8 InterferernceRadarRatio);
void radarAdaptiveSensInit(void);
void sendSerialTraceToHdkCdbMan(K_MSG_TYPE msgType, HdkRadarDetectionSerialTrace_e serialTraceType, uint32 val1, uint32 val2, uint32 val3);
void RadarAdaptiveSensEnableSerialTrace(bool enable);

#endif /* RADAR_DETECTION_H_ */

