/***********************************************************************************
 File:			ContinuousInterfererDetection.c
 Module:		Interferer detection 
 Purpose: 		Detect none wifi interferers
 Description:   This file is the implementation of the continuous interferer 
 				detection module which is responsible to detect continuous
 				interferers (when energy is always high)
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"	
#include "InterfererDetection_Api.h"
#include "InterfererDetection.h"
#include "ContinuousInterfererDetection.h"
#include "mhi_umi.h"
#include "OSAL_Api.h"
#include "init_ifmsg.h"
#include "um_interface.h"
#include "loggerAPI.h"
#include "Pac_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_INTERFERER_DETECTION
#define LOG_LOCAL_FID 2


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void continuousInterfererDetectionSortNewMetric(int8 newMetric);
static void continuousInterfererDetectionResetSamples(void);
static void continuousInterfererDetectionResetMaxMeasurement(void);



/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static ContinuousInterfererDetectionGlobalParameters_t ContinuousInterfererDetectionGlobalParameters;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
ContinuousInterfererDetectionIndicationBuffer_t ContinuousInterfererDetectionIndicationBuffer;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

continuousInterfererDetectionSortNewMetric


Description:
------------
Check if new metric should be inserted and if so sort the metrics

Input: 
-----
newMetric - the measurement from the PHY genrisc

Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void continuousInterfererDetectionSortNewMetric(int8 newMetric)
{
	uint8 index = 0;
	int8 newMaxMeasurement = 0;
	uint8 isSortNeeded = FALSE;
	
	if(ContinuousInterfererDetectionGlobalParameters.numberOfValidSamples < CONTINUOUS_INTERFERER_DETECTION_NUMBER_OF_METRICS)
	{
		isSortNeeded = TRUE;
		ContinuousInterfererDetectionGlobalParameters.metricsArray[ContinuousInterfererDetectionGlobalParameters.numberOfValidSamples] = newMetric;
		ContinuousInterfererDetectionGlobalParameters.numberOfValidSamples ++;		
	}
	else if(ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet > newMetric)
	{
		isSortNeeded = TRUE;
		ContinuousInterfererDetectionGlobalParameters.metricsArray[ContinuousInterfererDetectionGlobalParameters.maxMeasurementIndex] = newMetric;
	}
	
	if(isSortNeeded)
	{
		newMaxMeasurement = newMetric;
		for(index = 0; index < ContinuousInterfererDetectionGlobalParameters.numberOfValidSamples; index++)
		{
			if(ContinuousInterfererDetectionGlobalParameters.metricsArray[index] >= newMaxMeasurement)
			{
				newMaxMeasurement = ContinuousInterfererDetectionGlobalParameters.metricsArray[index];
				ContinuousInterfererDetectionGlobalParameters.maxMeasurementIndex = index;
			}
		}
		ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet = newMaxMeasurement;
#if defined (LOG_CONT_INT_DETECTION)
		ILOG0_D("ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet = %d", ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet);
#endif
	}
}


/**********************************************************************************

continuousInterfererDetectionResetSamples 


Description:
------------
Reset all continuous interferer detection metrics

Input: 
-----

Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void continuousInterfererDetectionResetSamples()
{
	/* Intialize all the entries in the array together instead of for loop */
	memset(ContinuousInterfererDetectionGlobalParameters.metricsArray, 0, CONTINUOUS_INTERFERER_DETECTION_NUMBER_OF_METRICS);
	ContinuousInterfererDetectionGlobalParameters.numberOfValidSamples = 0;
}

/**********************************************************************************

continuousInterfererDetectionResetMaxMeasurement 


Description:
------------
Reset max measurements

Input: 
-----

Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void continuousInterfererDetectionResetMaxMeasurement()
{
	ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet = CONTINUOUS_INTERFERER_DETECTION_MINIMAL_VALUE;
	ContinuousInterfererDetectionGlobalParameters.maxMeasurementIndex = 0;
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

ContinuousInterfererDetection_ProcessSamples 


Description:
------------
Process the samples sent by the HDK module


Input:
-----
continuousInterfererDetectionMessage - the message that contains the contimuos interferer 
detection samples


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
void ContinuousInterfererDetection_ProcessSamples(K_MSG * continuousInterfererDetectionMessage)
{
	InterfererDetectionBtInterfererDetectionSamplesMessageParameters *continuousInterfererDetectionSamplesMessage = NULL;
 	uint8 numberOfSamples = 0;
	uint8 currentMeasurementIndex = 0; 
	UMI_CONTINUOUS_INTERFERER*  continuousInterfererMessageParameters;
	K_MSG* pMsg = NULL;

 	continuousInterfererDetectionSamplesMessage = (InterfererDetectionBtInterfererDetectionSamplesMessageParameters *)continuousInterfererDetectionMessage->abData; 
    numberOfSamples = continuousInterfererDetectionSamplesMessage->numberOfSamples;
	//ILOG0_DD("ContinuousInterfererDetection_ProcessSamples  numberOfSamples = %d continuousInterfererDetectionSamplesMessage->isNotificationNeeded: %d", numberOfSamples, continuousInterfererDetectionSamplesMessage->isNotificationNeeded);
	for(currentMeasurementIndex = 0; currentMeasurementIndex < numberOfSamples; currentMeasurementIndex++)
	{
		continuousInterfererDetectionSortNewMetric(continuousInterfererDetectionSamplesMessage->measurements[currentMeasurementIndex]);	
	}

	//ILOG0_DD("continuousInterfererDetectionSamplesMessage->isNotificationNeeded = %d ContinuousInterfererDetectionGlobalParameters.interferenceThreshold: %d", continuousInterfererDetectionSamplesMessage->isNotificationNeeded, ContinuousInterfererDetectionGlobalParameters.interferenceThreshold);
	if(continuousInterfererDetectionSamplesMessage->isNotificationNeeded)
	{	
		if(ContinuousInterfererDetectionGlobalParameters.numberOfValidSamples)
		{
			/* There are samples to be checked */
			if ((ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet > ContinuousInterfererDetectionGlobalParameters.interferenceThreshold) &&
				(ContinuousInterfererDetectionGlobalParameters.radioState == ENABLE_RADIO))
			{
				if (ContinuousInterfererDetectionGlobalParameters.driverResponseArrived == TRUE)
				{
					pMsg = OSAL_GET_MESSAGE(sizeof(UMI_CONTINUOUS_INTERFERER));
					continuousInterfererMessageParameters = (UMI_CONTINUOUS_INTERFERER*)pK_MSG_DATA(pMsg);
					continuousInterfererMessageParameters->maximumValue = ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet;
					continuousInterfererMessageParameters->channel = ContinuousInterfererDetectionGlobalParameters.channel;
#if defined (LOG_CONT_INT_DETECTION)				
					ILOG0_DD("continuousInterfererMessageParameters->maximumValue = %d continuousInterfererMessageParameters->channel: %d", continuousInterfererMessageParameters->maximumValue, continuousInterfererMessageParameters->channel);
#endif
					OSAL_SEND_MESSAGE(UMI_MC_MAN_CONTINUOUS_INTERFERER_IND, TASK_UM_IF_TASK, pMsg, GET_DEFAULT_VAP_FOR_MY_BAND());

					ContinuousInterfererDetectionGlobalParameters.driverResponseArrived = FALSE;
				}
				else
				{
					if (ContinuousInterfererDetectionIndicationBuffer.numberOfPendingIndications == CONTINUOUS_INTERFERER_DETECTION_IND_BUFFER_SIZE)
					{
						FATAL("hContinuousInterfererDetection_ProcessSampleswe -we do not get driver response");
					}
					
					ContinuousInterfererDetectionIndicationBuffer.continuousInterfererDetectionIndication[ContinuousInterfererDetectionIndicationBuffer.continuousInterfererWriteIndex].channel = ContinuousInterfererDetectionGlobalParameters.channel;
					ContinuousInterfererDetectionIndicationBuffer.continuousInterfererDetectionIndication[ContinuousInterfererDetectionIndicationBuffer.continuousInterfererWriteIndex].maxMeasuremenet= ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet;
					ContinuousInterfererDetectionIndicationBuffer.continuousInterfererDetectionIndication[ContinuousInterfererDetectionIndicationBuffer.continuousInterfererWriteIndex].tsf = GET_TSF_TIMER_LOW();
					ContinuousInterfererDetectionIndicationBuffer.continuousInterfererWriteIndex++;

					if (ContinuousInterfererDetectionIndicationBuffer.continuousInterfererWriteIndex == CONTINUOUS_INTERFERER_DETECTION_IND_BUFFER_SIZE )
					{
						ContinuousInterfererDetectionIndicationBuffer.continuousInterfererWriteIndex = 0;
					}

					ContinuousInterfererDetectionIndicationBuffer.numberOfPendingIndications++;
				}
			}
			continuousInterfererDetectionResetSamples();	
		}
	}
}

/**********************************************************************************

ContinuousInterfererDetection_DrivrResponseArrived 


Description:
------------


Input:
-----


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/

void ContinuousInterfererDetection_DrivrResponseArrived (void)
{
	UMI_CONTINUOUS_INTERFERER*  continuousInterfererMessageParameters;
	K_MSG* pMsg = NULL;
	
	ContinuousInterfererDetectionGlobalParameters.driverResponseArrived = TRUE;
	//send the saved ind to the driver
	if (ContinuousInterfererDetectionIndicationBuffer.numberOfPendingIndications !=0 )
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(UMI_CONTINUOUS_INTERFERER));
		continuousInterfererMessageParameters = (UMI_CONTINUOUS_INTERFERER*)pK_MSG_DATA(pMsg);
		continuousInterfererMessageParameters->maximumValue = ContinuousInterfererDetectionIndicationBuffer.continuousInterfererDetectionIndication[ContinuousInterfererDetectionIndicationBuffer.continuousInterfererReadIndex].maxMeasuremenet;
		continuousInterfererMessageParameters->channel = ContinuousInterfererDetectionIndicationBuffer.continuousInterfererDetectionIndication[ContinuousInterfererDetectionIndicationBuffer.continuousInterfererReadIndex].channel;
#if defined (LOG_CONT_INT_DETECTION)				
		ILOG0_DD("ContinuousInterfererDetection_DrivrResponseArrived - continuousInterfererMessageParameters->maximumValue = %d continuousInterfererMessageParameters->channel: %d", continuousInterfererMessageParameters->maximumValue, continuousInterfererMessageParameters->channel);
#endif
		OSAL_SEND_MESSAGE(UMI_MC_MAN_CONTINUOUS_INTERFERER_IND, TASK_UM_IF_TASK, pMsg, GET_DEFAULT_VAP_FOR_MY_BAND());

		ContinuousInterfererDetectionGlobalParameters.driverResponseArrived = FALSE;

		ContinuousInterfererDetectionIndicationBuffer.numberOfPendingIndications--;

		ContinuousInterfererDetectionIndicationBuffer.continuousInterfererReadIndex ++;
		if (ContinuousInterfererDetectionIndicationBuffer.continuousInterfererReadIndex == CONTINUOUS_INTERFERER_DETECTION_IND_BUFFER_SIZE)
		{
			ContinuousInterfererDetectionIndicationBuffer.continuousInterfererReadIndex = 0;
		}
	}
}


/**********************************************************************************

ContinuousInterfererDetection_GetMaxMeasuremenet 


Description:
------------
Returns the current MAX measurement


Input:
-----


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
int8 ContinuousInterfererDetection_GetMaxMeasuremenet(void)
{
	int8 maxMeasuremenet = MIN_INT8;

	if (ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet != 0)
	{
		maxMeasuremenet = ContinuousInterfererDetectionGlobalParameters.maxMeasuremenet;
	}
	return maxMeasuremenet;
}

/**********************************************************************************

ContinuousInterfererDetection_ConfigureParameters 


Description:
------------
Configures the continuous interferer detection parameters


Input:
-----
continuousInterfererDetectionMessage - the message that contains the continuous 
interferer detection parameters


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
void ContinuousInterfererDetection_ConfigureParameters(K_MSG * continuousInterfererDetectionMessage)
{
	UMI_INTERFERER_DETECTION_PARAMS *InterfererDetectionParametersMessage = NULL;

 	InterfererDetectionParametersMessage = (UMI_INTERFERER_DETECTION_PARAMS *)continuousInterfererDetectionMessage->abData; 
	if (InterfererDetectionParametersMessage->getSetOperation == API_GET_OPERATION)
	{
		InterfererDetectionParametersMessage->threshold = ContinuousInterfererDetectionGlobalParameters.interferenceThreshold;
	}
	else
	{
		ContinuousInterfererDetectionGlobalParameters.interferenceThreshold = InterfererDetectionParametersMessage->threshold;
	}
}


/**********************************************************************************

ContinuousInterfererDetection_SetChannel 


Description:
------------
Set new channel in the continuous interferer detection module - Reset all 
continuous interferer detection metrics

Input: 
-----
channel - the new channel to set

Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
void ContinuousInterfererDetection_SetChannel(uint8 channel)
{
	continuousInterfererDetectionResetSamples();
	continuousInterfererDetectionResetMaxMeasurement();
	ContinuousInterfererDetectionGlobalParameters.channel = channel;
}

/**********************************************************************************

ContinuousInterfererDetection_SetRadioState 


Description:
------------
Set Radio State 

Input: 
-----
radioState - radio State

Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
void ContinuousInterfererDetection_SetRadioState(uint8 radioState)
{
	ContinuousInterfererDetectionGlobalParameters.radioState = radioState;
}

/**********************************************************************************

ContinuousInterfererDetection_Initialize 


Description:
------------
Initialize the Continuous interferer detection module internal parameters and structures

Input:
-----


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void ContinuousInterfererDetection_Initialize()
{
	memset(&ContinuousInterfererDetectionGlobalParameters, 0, sizeof(ContinuousInterfererDetectionGlobalParameters_t));
	memset(&ContinuousInterfererDetectionIndicationBuffer, 0, sizeof(ContinuousInterfererDetectionIndicationBuffer_t));

	continuousInterfererDetectionResetSamples();
	continuousInterfererDetectionResetMaxMeasurement();
	
	ContinuousInterfererDetectionGlobalParameters.interferenceThreshold = CONTINUOUS_INTERFERER_DETECTION_DEFAULT_VALUE;  
	ContinuousInterfererDetectionGlobalParameters.driverResponseArrived = TRUE;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


