/***********************************************************************************
 File:			InterfererDetection.h
 Module:		Interferer detection 
 Purpose: 		Detect none wifi interferers
 Description:   This file is the implementation of the Interferer detection module 
 				which is responsible to detect none wifi interferers such as radar
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"	
#include "InterfererDetection_Api.h"
#include "InterfererDetection.h"
#include "queue_utility.h"
#include "RadarDetection.h"
#include "ContinuousInterfererDetection.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_INTERFERER_DETECTION
#define LOG_LOCAL_FID 0


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void interfererDetectionSetChannel(K_MSG *interfererDetectionMessage); 
#if defined RADAR_DETECTION_ENABLED
static void interfererDetectionRadarDetectionSamples(K_MSG *interfererDetectionMessage);
#endif
static void interfererDetectionContinuousInterfererDetectionSamples(K_MSG *interfererDetectionMessage);
static void interfererDetectionSetInterfererDetectionParameters(K_MSG *interfererDetectionMessage);
static void interfererDetectionEnableRadarIndication(K_MSG *interfererDetectionMessage);
#if defined RADAR_DETECTION_ENABLED
static void interfererDetectionRadarAgingTimer(K_MSG *interfererDetectionMessage);
#endif
static void interfererDetectionSetRadioState(K_MSG *interfererDetectionMessage);
static void interfererDetectionContinuousInterfererDriverResArrived(K_MSG *interfererDetectionMessage);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static const FunctionEntry_t afpTaskTable[TASK_INTERFERER_DETECTION_END - TASK_INTERFERER_DETECTION_START]=
{
	{interfererDetectionSetChannel, 							DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_SET_CHANNEL)},
#if defined RADAR_DETECTION_ENABLED
	{interfererDetectionRadarDetectionSamples,					DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_RADAR_DETECTION_SAMPLES)},
#endif		
	{interfererDetectionContinuousInterfererDetectionSamples,	DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_CONTINUOUS_INTERFERER_DETECTION_SAMPLES)},
	{interfererDetectionSetInterfererDetectionParameters,		DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_SET_INTERFERER_DETECTION_PARAMETERS)},		
	{interfererDetectionEnableRadarIndication,					DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_ENABLE_RADAR_INDICATION)},
#if defined RADAR_DETECTION_ENABLED
	{interfererDetectionRadarAgingTimer,						DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_RADAR_AGING_TIMER)},
#endif		
	{interfererDetectionSetRadioState,							DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_SET_RADIO_STATE)},
	{interfererDetectionContinuousInterfererDriverResArrived,	DOUBLE_CHECK_MSG_TYPE(INTERFERER_DETECTION_CONTINUOUS_INTERFERER_RES_ARRIVED)},	
};


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

interfererDetectionSetChannel 


Description:
------------
Set new channel in the interferer detection module. This message is sent by the 
HDK module as part of the set channle process

Input: 
-----
interfererDetectionMessage - pointer to the message that contains the channel index	
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/

static void interfererDetectionSetChannel(K_MSG *interfererDetectionMessage)
{
	uint8 channel = 0;
	InterfererDetectionSetChannelMessageParameters_t *setChannelMessageParameters = NULL;
	
	setChannelMessageParameters = ((InterfererDetectionSetChannelMessageParameters_t *)interfererDetectionMessage->abData);
	channel = setChannelMessageParameters->channel;

#if defined RADAR_DETECTION_ENABLED
	RadarDetection_SetChannel(channel, setChannelMessageParameters->isRadarDetectionEnabled);
#endif
	ContinuousInterfererDetection_SetChannel(channel); 
}

/**********************************************************************************

interfererDetectionRadarDetectionSamples 


Description:
------------
Process radar detection samples from the PHY genrisc. This message is sent by the 
HDK module after processing the radar detection FIFO in the PHY genrisc.

Input: 
-----
interfererDetectionMessage - pointer to the message that contains the radar samples	
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
#if defined RADAR_DETECTION_ENABLED
static void interfererDetectionRadarDetectionSamples(K_MSG *interfererDetectionMessage)
{
    RadarDetection_ProcessSamples(interfererDetectionMessage);
}
#endif

/**********************************************************************************

interfererDetectionContinuousInterfererDetectionSamples 


Description:
------------
Process continuous interferer detection samples from the PHY genrisc. This message 
is sent by the HDK module after processing the continuous interferer detection FIFO 
in the PHY genrisc.

Input: 
-----
interfererDetectionMessage - pointer to the message that contains the continuous 
interferer samples	
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void interfererDetectionContinuousInterfererDetectionSamples(K_MSG *interfererDetectionMessage)
{
	ContinuousInterfererDetection_ProcessSamples(interfererDetectionMessage);
}

/**********************************************************************************

interfererDetectionContinuousInterfererDverResArrived 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void interfererDetectionContinuousInterfererDriverResArrived(K_MSG *interfererDetectionMessage)
{
	UNUSED_PARAM(interfererDetectionMessage);
	ContinuousInterfererDetection_DrivrResponseArrived();
}

/**********************************************************************************

interfererDetectionSetInterfererDetectionParameters 


Description:
------------
Configures the interferer detection parameters.

Input: 
-----
interfererDetectionMessage - pointer to the message that contains the interferer
detection parameters
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void interfererDetectionSetInterfererDetectionParameters(K_MSG *interfererDetectionMessage)
{
	ContinuousInterfererDetection_ConfigureParameters(interfererDetectionMessage);
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_INTERFERER_DETECTION_PARAMS_CFM, TASK_UM_IF_TASK, interfererDetectionMessage, interfererDetectionMessage->header.vapId);
}


/**********************************************************************************

interfererDetectionEnableRadarIndication 


Description:
------------
Enable/Disable the radar indication - only for sending messages to driver

Input: 
-----
interfererDetectionMessage - pointer to the message that contains the interferer
detection parameters
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void interfererDetectionEnableRadarIndication(K_MSG *interfererDetectionMessage)
{
#if defined RADAR_DETECTION_ENABLED
	RadarDetection_EnableRadarIndication(interfererDetectionMessage); 
#endif
	OSAL_SEND_MESSAGE(UMI_MC_MAN_ENABLE_RADAR_INDICATION_CFM, TASK_UM_IF_TASK, interfererDetectionMessage, interfererDetectionMessage->header.vapId);
}


/**********************************************************************************

interfererDetectionRadarAgingTimer 


Description:
------------
Radar Ageing timeout

Input: 
-----
interfererDetectionMessage - pointer to the message that contains the interferer
detection parameters
		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
#if defined RADAR_DETECTION_ENABLED
static void interfererDetectionRadarAgingTimer(K_MSG *interfererDetectionMessage)
{
	RadarDetection_AgingTimer(interfererDetectionMessage);
}
#endif

/**********************************************************************************

interfererDetectionSetRadioState 


Description:
------------
Set Radio State

Input: 
-----

		
Output:
-------
	
Returns:
--------	 
	
**********************************************************************************/
static void interfererDetectionSetRadioState(K_MSG *interfererDetectionMessage)
{
	InterfererDetectionSetRadioState_t *setRadioState = (InterfererDetectionSetRadioState_t *)pK_MSG_DATA(interfererDetectionMessage);
	
#if defined RADAR_DETECTION_ENABLED
	RadarDetection_SetRadioState(setRadioState->radioState);
#endif
	ContinuousInterfererDetection_SetRadioState(setRadioState->radioState);
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

InterfererDetection_TaskEntry 


Description:
------------
The entry point of the interferer detection task

Input: 
-----
interfererDetectionMessage - pointer to the message to handle	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterfererDetection_TaskEntry(K_MSG *interfererDetectionMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(interfererDetectionMessage, afpTaskTable, TASK_INTERFERER_DETECTION_START, TASK_INTERFERER_DETECTION_END);
}


/**********************************************************************************

InterfererDetection_Initialize 


Description:
------------
Initialize the Interferer detection module internal parameters and structures

Input:
-----


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void InterfererDetection_Initialize()
{
#if defined RADAR_DETECTION_ENABLED
	RadarDetection_Initialize();
#endif
	ContinuousInterfererDetection_Initialize();
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


