/***********************************************************************************
 File:			RadarDetection.c
 Module:		Interferer detection 
 Purpose: 		Detect none wifi interferers
 Description:   This file is the implementation of the radar detection which is 
 				responsible to detect presence of radars in the	current channel
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "InterfererDetection_Api.h"
#include "InterfererDetection.h"
#include "RadarDetection.h"
#include "Pac_Api.h"
#include "mhi_umi.h"
#include "Utils_Api.h"
#include "loggerAPI.h"
#include "init_ifmsg.h"
#include "um_interface.h"
#include "HdkTask.h"
#include "PhyRxTdRegs.h"
#include "RegAccess_Api.h"
#include "fast_mem_psd2mips.h"
#include "HdkCdbManagerTask.h"
#if defined RADAR_DETECTION_ENABLED

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_INTERFERER_DETECTION
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
/*Age out every second*/
#define RADAR_AGING_TIMER_IN_MILLISECONDS	(1000)	
/*Adaptive sensetivity thresholds*/
#define CAC_STATISTICS_PERIOD_5MSEC_UNIT 200 // 1 sec
#define IN_SERVICE_STATISTICS_PERIOD_5MSEC_UNIT 200 // 1 sec
#define CAC_NON_ASSOC_DATA_PACKET_TH 5 
#define IN_SERVICE_NON_ASSOC_DATA_PACKET_TH 5 
#define RADAR_INTERFERER_RATIO_TH 12
#define INTERFERE_COUNTER_FACTOR 10

#ifndef ENET_INC_ARCH_WAVE600
/*Phy counters address*/
#define PHY_INTERFERER_COUNTER_ADDRESS 0x203EC
#define PHY_PACKET_DETECT_COUNTER_ADDRESS 0x20788
#define PHY_PACKET_RADAR_COUNTERS1_ADDRESS 0x207DC
#define PHY_PACKET_RADAR_COUNTERS2_ADDRESS 0x207EC
#define PHY_PACKET_RADAR_COUNTERS3_ADDRESS 0x207F4
#define PHY_PACKET_RADAR_COUNTERS4_ADDRESS 0x207F8
#define PHY_PACKET_RADAR_COUNTERS5_ADDRESS 0x207FC
#endif





/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum
{
	RADAR_ADAPTIVE_SENS_STATE_DISABLE,
	RADAR_ADAPTIVE_SENS_STATE_CAC,
	RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE,	
	RADAR_ADAPTIVE_SENS_STATES_NUM,		
} AdaptiveSensState_e;
typedef struct radarAdaptiveSensDb_t
{
	AdaptiveSensState_e state;
	uint16 timerCounter;
	uint32 radarCounter;
	uint32 interfererCounter;
	uint32 dataCounterNonAssoc;
	UMI_RADAR_DETECTION lastRadarDetectionParams;
	bool isRadarFiltered;
	bool radarInFilteringPeriod;
	bool featureDisabled;
	bool enableSerialTrace;
}radarAdaptiveSensDb_t;
typedef struct radarAdaptiveSensConfig
{
	uint16 stateTimeCounterTh; //5msec units
	uint8 nonAssoDataPacketsTh;
	uint8 InterferernceRadarRatio;
}radarAdaptiveSensConfig_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void radarDetectionResetSamples(void);
static RadarDetectionPulseRecord_t *radarDetectionGetPulseRecord(RadarDetectionPulseCluster_t *pulseRecQueue);
static void radarDetectionPutPulseRecord(RadarDetectionPulseCluster_t *pulseRecQueue,RadarDetectionPulseRecord_t *pulseRecord);
static void radarDetectionCalculateNewAverageValue(RadarDetectionPulseCluster_t *pulseCluster,RadarDetectionPulseRecord_t *newPulseRecord,RadarDetectionAverageCalculationType_e calcType);
static void radarDetectionSortPulseRecord(RadarDetectionPulseRecord_t *pulseDetails);
static void radarDetectionCopyPulseDetails(InterfererDetectionRadarDetectionSamplesMessageParameters_t *radarDetectionSamplesMessage,RadarDetectionPulseCluster_t *waitingPulseRecordsCluster);
static void radarDetectionResetAgedOutRecords(void);
static uint8 radarDetectionCheckLongRadarPri(RadarDetectionPulseCluster_t *pulseCluster);
static uint8 radarDetectionCheckShortRadarPri(RadarDetectionPulseCluster_t *pulseCluster);
static bool radarDetectionCheckValidPri(RadarDetectionPulseRecord_t *pulseRecord,RadarDetectionPulseCluster_t *pulseCluster);
static bool radarDetectionCheckLongValidPri(RadarDetectionPulseRecord_t *pulseRecord,RadarDetectionPulseCluster_t *pulseCluster);
static uint32 radarDetectionCalculateRemainder(uint32 numerator ,uint32 denominator,uint32 ratio);
static void radarDetectionPutPulseRecordGlobal(QPB *recordsQueue, RadarDetectionPulseRecord_t *pulseRecord);
static RadarDetectionPulseRecord_t* radarDetectionGetPulseRecordGlobal(QPB *recordsQueue);
static bool isValidPri(uint32 pri, uint8 pw);
void radarDetectionStateMachineChangeState(AdaptiveSensState_e newState);
void radarAdaptSensSetChannelEvent(radarAdaptiveSensConfig_t* pConfig, AdaptiveSensState_e newState);
void radarAdaptSensDetectionEvent(radarAdaptiveSensConfig_t* pConfig);
void radarAdaptSensTimerExpiredEvent(radarAdaptiveSensConfig_t* pConfig);
void radarAdaptSensCalculateCountersDiff(uint32* diffRadarCounter, uint32* difftInterfererCounter, uint32* difftDataNonAssocCounter, bool updateCountersInDb);
void radarAdaptSensDisableEvent(void);
static void updateCurrentRadarCountersInDb(uint32 currentRadarCounter, uint32 currentInterfererCounter,uint32 currentDataCounterNonAssoc);
static void readCurrentRadarCountersFromPhy(uint32* pCurrentRadarCounter, uint32* pCurrentInterfererCounter,uint32* pCurrentDataCounterNonAssoc);
static bool isRadarFiltering(radarAdaptiveSensConfig_t* pConfig, uint32 diffRadarCounter, uint32 difftInterfererCounter, uint32 difftDataNonAssocCounter);
static void radarDetectionSetPRIsThresholds(uint16 shortRadarMinPriTh, uint16 shortRadarMaxPriTh, uint16 longRadarMinPriTh, uint16 longRadarMaxPriTh);
static void radarDetectionSetPWsThresholds(uint8 minShortPwTh, uint8 maxShortPwTh, uint8 minLongPwTh, uint8 maxLongPwTh);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static RadarDetectionGlobalParameters_t RadarDetectionGlobalParameters;
static radarAdaptiveSensConfig_t CacConfigurations = 
{
	CAC_STATISTICS_PERIOD_5MSEC_UNIT,
	CAC_NON_ASSOC_DATA_PACKET_TH,
	RADAR_INTERFERER_RATIO_TH
};
static radarAdaptiveSensConfig_t InServiceConfigurations = 
{
	IN_SERVICE_STATISTICS_PERIOD_5MSEC_UNIT,
	IN_SERVICE_NON_ASSOC_DATA_PACKET_TH,
	RADAR_INTERFERER_RATIO_TH
};
static radarAdaptiveSensDb_t RadarAdaptiveSensDataBase;

//#define RADAR_DEBUG_TRACE
#ifdef RADAR_DEBUG_TRACE
#define RADAR_DEBUG_TRACE_SIZE 100
uint32 radarDebugPulseList[RADAR_DEBUG_TRACE_SIZE] = {0};
uint32 radarDebugPulseCounter = 0;
#endif // RADAR_DEBUG_TRACE


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

radarDetectionResetSamples

Description:
------------
Reset all samples and database

Input: 
-----	

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void radarDetectionResetSamples(void)
{
	uint8 recordIndex = 0;

	memset(&RadarDetectionGlobalParameters.pulses, 0, sizeof(RadarDetectionPulses_t));
	for (recordIndex = 0; recordIndex < MAX_NUM_OF_PULSE_RECORDS; recordIndex++)
	{
		/*Put record in free list*/
		radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,&RadarDetectionGlobalParameters.pulses.pulseRecords[recordIndex]);	
	}
}


/**********************************************************************************

radarDetectionGetPulseRecord 


Description:
------------


Input: 
-----	
pulseCluster
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static RadarDetectionPulseRecord_t *radarDetectionGetPulseRecord(RadarDetectionPulseCluster_t *pulseCluster)
{
	RadarDetectionPulseRecord_t *queuedPulseRecord = (RadarDetectionPulseRecord_t *)pulseCluster->pulseQueue.pvHead;
	
	DEBUG_ASSERT(queuedPulseRecord);
	/* move head onto next so removing this item from the queue */
	pulseCluster->pulseQueue.pvHead = (void *)queuedPulseRecord->nextPulseRecord;
	/* check if queue is now empty */
	if( pulseCluster->pulseQueue.pvHead == NULL )
	{
		/* queue is empty - tail should also point to NULL */
		pulseCluster->pulseQueue.pvTail = NULL;
	}
	pulseCluster->pulseCount--;	
	return queuedPulseRecord;
}

/**********************************************************************************

radarDetectionPutPulseRecord 


Description:
------------


Input: 
-----	
pulseCluster - 
pulseRecord - 	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionPutPulseRecord(RadarDetectionPulseCluster_t *pulseCluster,RadarDetectionPulseRecord_t *pulseRecord)
{
	pulseRecord->nextPulseRecord = NULL;
    /* Check for queue empty */
    if( pulseCluster->pulseQueue.pvHead == NULL )
    {
        /* queue is empty */
        pulseCluster->pulseQueue.pvHead = (void *)pulseRecord;
        pulseCluster->pulseQueue.pvTail = (void *)pulseRecord;
    }
    else
    {
        DEBUG_ASSERT(pulseRecord != (RadarDetectionPulseRecord_t *)pulseCluster->pulseQueue.pvTail);
        /* Place item at the tail of the queue */
        ((RadarDetectionPulseRecord_t *)(pulseCluster->pulseQueue.pvTail))->nextPulseRecord = pulseRecord;
        pulseCluster->pulseQueue.pvTail = (void *)pulseRecord;
    }
	pulseCluster->pulseCount++;
}

/**********************************************************************************

radarDetectionPutPulseRecordGlobal 


Description:
------------


Input: 
-----	
recordsQueue - 
pulseRecord - 	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionPutPulseRecordGlobal(QPB *recordsQueue, RadarDetectionPulseRecord_t *pulseRecord)
{
	pulseRecord->nextPulseGlobalrecord = NULL;
    /* Check for queue empty */
    if( recordsQueue->pvHead == NULL )
    {
        /* queue is empty */
        recordsQueue->pvHead = (void *)pulseRecord;
        recordsQueue->pvTail = (void *)pulseRecord;
    }
    else
    {
        DEBUG_ASSERT(pulseRecord != (RadarDetectionPulseRecord_t *)recordsQueue->pvTail);
        /* Place item at the tail of the queue */
        ((RadarDetectionPulseRecord_t *)(recordsQueue->pvTail))->nextPulseGlobalrecord = pulseRecord;
        recordsQueue->pvTail = (void *)pulseRecord;
    }
}

/**********************************************************************************

radarDetectionGetPulseRecordGlobal 


Description:
------------


Input: 
-----	
pulseCluster
	
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static RadarDetectionPulseRecord_t *radarDetectionGetPulseRecordGlobal(QPB *recordsQueue)
{
	RadarDetectionPulseRecord_t *queuedPulseRecord = (RadarDetectionPulseRecord_t *)recordsQueue->pvHead;
	
	DEBUG_ASSERT(queuedPulseRecord);
	/* move head onto next so removing this item from the queue */
	recordsQueue->pvHead = (void *)queuedPulseRecord->nextPulseGlobalrecord;
	/* check if queue is now empty */
	if( recordsQueue->pvHead == NULL )
	{
		/* queue is empty - tail should also point to NULL */
		recordsQueue->pvTail = NULL;
	}
	return queuedPulseRecord;
}

/**********************************************************************************

radarDetectionCalculateNewAverageValue 


Description:
------------


Input: 
-----	
pulseCluster - 
newPulseRecord -
calcType -

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionCalculateNewAverageValue(RadarDetectionPulseCluster_t *pulseCluster,RadarDetectionPulseRecord_t *newPulseRecord,RadarDetectionAverageCalculationType_e calcType) 
{
	RadarDetectionPulseRecord_t* pHead = (RadarDetectionPulseRecord_t*)pulseCluster->pulseQueue.pvHead;
	
	switch(calcType)
	{
		case RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_ADDED_RECORD:
			DEBUG_ASSERT(pulseCluster->pulseCount != 0);
			pulseCluster->avgPulseWidth = (pulseCluster->avgPulseWidth * (pulseCluster->pulseCount - 1) + newPulseRecord->pulseWidth)/(pulseCluster->pulseCount);
			break;
		case RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_DELETED_RECORD:
			if (pulseCluster->pulseCount == 0)
			{
				pulseCluster->avgPulseWidth = 0;
			}
			else if (pulseCluster->pulseCount == 1)
			{
				/* avgPulseWidth must be equal to the cluster head pulse width */
				pulseCluster->avgPulseWidth = pHead->pulseWidth;
			}
			else
			{
				pulseCluster->avgPulseWidth = (pulseCluster->avgPulseWidth * (pulseCluster->pulseCount + 1) - newPulseRecord->pulseWidth)/(pulseCluster->pulseCount);
			}
			break;
		default:
			break;
	}
}

/**********************************************************************************

radarDetectionSortPulseRecord 


Description:
------------


Input: 
-----	 
newPulseRecord -


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionSortPulseRecord(RadarDetectionPulseRecord_t *pulseRecord)
{
	uint8  clusterIndex = 0;
	uint8 index,emptyClusterIndex;
	uint32 minVal;

	minVal = MAX_SHORT_PW_VAL;
	emptyClusterIndex = (RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex >= FIRST_NON_RESERVED_CLUSTER ? RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex + 1:FIRST_NON_RESERVED_CLUSTER);
	for( index = FIRST_NON_RESERVED_CLUSTER; index <= RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex ; index++ )
	{
		uint32 val;
		
		if(RadarDetectionGlobalParameters.pulses.pulseClustersControl[index].pulseCount > 0) //check if cluster is not empty
		{
			val = ABS_DIFFERENCE(RadarDetectionGlobalParameters.pulses.pulseClustersControl[index].avgPulseWidth ,pulseRecord->pulseWidth);
			if(val < minVal)
			{
				minVal = val;
				clusterIndex = index;
			}
		}
		else
		{
			emptyClusterIndex = index;
		}
	}
	/*There are no clusters that are suitable to our current PW, Use  an empty cluster*/
	if(minVal >= PW_RESOLUTION)
	{
		ASSERT(emptyClusterIndex < NUM_OF_PULSE_CLUSTERS);
		clusterIndex = emptyClusterIndex;	
		if(clusterIndex > RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex)
		{
			RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex = clusterIndex ;
			ASSERT(RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex < NUM_OF_PULSE_CLUSTERS);
		}
	}
	else 
	{
		if(!radarDetectionCheckValidPri(pulseRecord,&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex]))
		{
			/*The PRI is not valid ignore the pulse details & return free pulse cluster */
			radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,pulseRecord);
			return;
		}
	}
	/*Store cluster index in pulse record*/
	pulseRecord->clusterIndex = clusterIndex;
	/*Put record on global queue*/
	radarDetectionPutPulseRecordGlobal(&RadarDetectionGlobalParameters.pulses.pulseRecordsQueue,pulseRecord);
	radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex],pulseRecord);
	/*Calculate new average of the cluster*/
	radarDetectionCalculateNewAverageValue(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex],pulseRecord,RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_ADDED_RECORD);
}

/**********************************************************************************

radarDetectionCopyPulseDetails 


Description:
------------
copy all the pulses details from the LM block to a temporary cluster in order to 
free the resource for the LM.


Input: 
-----	
radarDetectionSamplesMessage - message from the HDK that contains the samples
waitingPulseRecordsCluster - temporary pulse cluster

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionCopyPulseDetails(InterfererDetectionRadarDetectionSamplesMessageParameters_t *radarDetectionSamplesMessage,RadarDetectionPulseCluster_t *waitingPulseRecordsCluster)
{
	RadarDetectionPulseRecord_t *pulseRecord;
	RadarDetectionSample_t *pulses = radarDetectionSamplesMessage->pulses;
	uint32 pulseIndex;

	DEBUG_ASSERT(radarDetectionSamplesMessage->numberOfSamples <= RadarDetectionGlobalParameters.pulses.freePulseCluster.pulseCount);
	for(pulseIndex = 0; pulseIndex <radarDetectionSamplesMessage->numberOfSamples; pulseIndex++)
	{
		pulseRecord = radarDetectionGetPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster);
		pulseRecord->pulseTimeStamp = pulses->timeStamp;
		pulseRecord->pulseWidth = pulses->pulseWidth;
		pulseRecord->ampSubBandBitMap   = pulses->ampSubBandBitMap;
		radarDetectionPutPulseRecord(waitingPulseRecordsCluster,pulseRecord);
		pulses ++;
	}
}

/**********************************************************************************

radarDetectionCopyPulseDetails 


Description:
------------
Delete aged old pulse records from cluster in order to free space for new records 



Input: 
-----	
radarDetectionSamplesMessage - 
waitingPulseRecordsCluster -

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionResetAgedOutRecords(void)
{
	RadarDetectionPulseRecord_t *pulseRecord;  
	uint32 currentPulseTimeStamp;
	uint32 pulseAgedOut;
	uint8  clusterIndex;
	RadarDetectionPulseRecord_t *pvHeadTemp, *prevTemp = NULL;

	/* Reset records aged out , or the oldest records if all the pulse_records 
	 * are in use(and we need space for the new records)*/
	/*First drop aged out*/
	pvHeadTemp = (RadarDetectionPulseRecord_t *)RadarDetectionGlobalParameters.pulses.pulseRecordsQueue.pvHead;
	while(pvHeadTemp)
	{
		currentPulseTimeStamp = pvHeadTemp->pulseTimeStamp;
		clusterIndex = pvHeadTemp->clusterIndex;
		/*A cluster may contain pulses that could fall under different thresholds below, therefore to determine the threshold
		we use the cluster average pulse width to ensure that pulese are aged out from a cluster in order. When a pulse is removed 
		the average width is recalculated and the threshold used for the next time may be different but we will stay remove pulses
		from a cluster in order*/
		if (RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex].avgPulseWidth >= RadarDetectionGlobalParameters.minLongPwTh)
		{
			pulseAgedOut = LONG_PULSE_AGED_OUT;
		}
		else if (RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex].avgPulseWidth >= RADAR_ETSI_MEDIUM_MIN_PW_TH)
		{
			pulseAgedOut = MEDIUM_PULSE_AGED_OUT;
		}
		else
		{
			pulseAgedOut = SHORT_PULSE_AGED_OUT;
		}
		if ((RadarDetectionGlobalParameters.pulses.blockManagement.newestPulseTimeStamp - currentPulseTimeStamp) > pulseAgedOut)
		{
			/*When aging out we may remove from the middle ofthe global queue*/
			if (pvHeadTemp == (RadarDetectionPulseRecord_t *)RadarDetectionGlobalParameters.pulses.pulseRecordsQueue.pvHead)
			{
				pulseRecord = radarDetectionGetPulseRecordGlobal(&RadarDetectionGlobalParameters.pulses.pulseRecordsQueue);
			}
			else
			{
				ASSERT(prevTemp!=NULL);
				prevTemp->nextPulseGlobalrecord = pvHeadTemp->nextPulseGlobalrecord;
				if (pvHeadTemp->nextPulseGlobalrecord == NULL)
				{
					RadarDetectionGlobalParameters.pulses.pulseRecordsQueue.pvTail = prevTemp;
				}
			}
			/*Must be head of cluster*/
			ASSERT(pvHeadTemp == RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex].pulseQueue.pvHead);
			/*From cluster queue always remove from head*/
			pulseRecord = radarDetectionGetPulseRecord(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex]);
			radarDetectionCalculateNewAverageValue(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex],pulseRecord,RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_DELETED_RECORD);					
			radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,pulseRecord);
		}						
		else
		{
			/*Set prev*/
			prevTemp = pvHeadTemp;
		}
		/*Move to next*/
		pvHeadTemp = (void *)pvHeadTemp->nextPulseGlobalrecord;
	}
	/*Now drop the oldest as needed*/
	while (RadarDetectionGlobalParameters.pulses.blockManagement.numOfNewPulseRec > RadarDetectionGlobalParameters.pulses.freePulseCluster.pulseCount)
	{
		/*When removing the oldes we remove from the head from both the global queue and the cluster queue*/
		/*Remove from global queue*/
		pulseRecord = radarDetectionGetPulseRecordGlobal(&RadarDetectionGlobalParameters.pulses.pulseRecordsQueue);
		/*get cluster index from record*/
		clusterIndex = pulseRecord->clusterIndex;
		/*Must be head of cluster*/
		ASSERT(pulseRecord == RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex].pulseQueue.pvHead);
		/*remove from cluster list*/
		pulseRecord = radarDetectionGetPulseRecord(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex]);
		radarDetectionCalculateNewAverageValue(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex],pulseRecord,RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_DELETED_RECORD);					
		radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,pulseRecord);
	}
	/*Recalculate the bigggest Cluster Index with pulses on it*/
	RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex = 0;
	for (clusterIndex = 0; clusterIndex < NUM_OF_PULSE_CLUSTERS; clusterIndex++)
	{
		if (RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex].pulseCount > 0)
		{
			RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex = clusterIndex;
		}
	}
}

/**********************************************************************************

radarDetectionCheckLongRadarPri 


Description:
------------


Input: 
-----	
pulseCluster -

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
uint8 radarDetectionCheckLongRadarPri(RadarDetectionPulseCluster_t *pulseCluster)
{
	RadarDetectionPulseRecord_t *pulseRecord;
	uint32 rawPri;
	uint8 priCnt  = 0,longPri= 0,shortPri = 0;
	uint8 pulse_index;
	uint8 ampSubBandBitMap = 0;

	pulseRecord = (RadarDetectionPulseRecord_t *) pulseCluster->pulseQueue.pvHead;
	for (pulse_index = 0;pulse_index < ((pulseCluster->pulseCount) - 1);pulse_index++)
	{	
		rawPri = pulseRecord->nextPulseRecord->pulseTimeStamp -  pulseRecord->pulseTimeStamp;
		if(rawPri >= RadarDetectionGlobalParameters.longRadarMinPriTh)
		{
			priCnt ++;
			ampSubBandBitMap = (uint8)(ampSubBandBitMap |(pulseRecord->ampSubBandBitMap |pulseRecord->nextPulseRecord->ampSubBandBitMap));			
			if(rawPri >= RadarDetectionGlobalParameters.longRadarMaxPriTh)
			{
				longPri++;
			}
			else
			{
				shortPri++;
			}
		}
		pulseRecord = pulseRecord->nextPulseRecord;
	}
	
	ILOG0_DDD("radarDetectionCheckLongRadarPri: pri count %d, long %d, short %d", priCnt, longPri, shortPri);
	if ((priCnt >= (RadarDetectionGlobalParameters.longRadarPriTh)) && (longPri >= MIN_LONG_PRI_LONG_RADAR_TH) && (shortPri >= MIN_SHORT_PRI_LONG_RADAR_TH))
	{
		if(ampSubBandBitMap == 0)
		{
			ampSubBandBitMap = 0xff;
		}
		return ampSubBandBitMap;
	}
	return FALSE;
}


/**********************************************************************************

radarDetectionCheckShortRadarPri 


Description:
------------


Input: 
-----	
pulseCluster -
shortRadarType -

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static uint8 radarDetectionCheckShortRadarPri(RadarDetectionPulseCluster_t *pulseCluster)
{
	RadarDetectionPulseRecord_t *pulseRecord =  (RadarDetectionPulseRecord_t *) pulseCluster->pulseQueue.pvHead;
	uint32 rawPri[MAX_NUM_OF_PULSE_RECORDS];  //KW ignore as rawPri is always initialized
	uint8 rawSubBandBitMap[MAX_NUM_OF_PULSE_RECORDS];
	uint32 start_pri_index = 0;
	uint32 priRatio,priRemain;
	uint8  pri_j_index ,pri_index = 0,pri_cnt;
	uint8 ampSubBandBitMap;
	
	DEBUG_ASSERT(pulseRecord);
	DEBUG_ASSERT(pulseRecord->nextPulseRecord);
	/*Calculate  pri's in this cluster*/
 	while (pulseRecord->nextPulseRecord)
	{
		ASSERT(pri_index < MAX_NUM_OF_PULSE_RECORDS);
		rawPri[pri_index] = (pulseRecord->nextPulseRecord->pulseTimeStamp -  pulseRecord->pulseTimeStamp);
		rawSubBandBitMap[pri_index] = pulseRecord->ampSubBandBitMap;
		pulseRecord = pulseRecord->nextPulseRecord;
		pri_index++;
	}
	//since pri_index is incremented above, rawsubbandbitmap may use 1-22 instead of 0-21.. review by Yossi
	ASSERT(pri_index < MAX_NUM_OF_PULSE_RECORDS);
	rawSubBandBitMap[pri_index] = pulseRecord->ampSubBandBitMap;
	
	DEBUG_ASSERT(pulseCluster->pulseCount >= pri_index)
	start_pri_index = ((pri_index >= MAX_PRI_ANALYSIS) ? (pri_index - MAX_PRI_ANALYSIS):0);
	for(pri_index = start_pri_index;pri_index < (pulseCluster->pulseCount - 1);pri_index ++ )
	{
		DEBUG_ASSERT(rawPri[pri_index]); //KW_IGNORE UNINIT.STACK.ARRAY.MIGHT rawPri is always initialized
		pri_cnt = 0;
		ampSubBandBitMap = 0;
		if (isValidPri(rawPri[pri_index], pulseCluster->avgPulseWidth) == TRUE) //ignore (pw, pri) combinations which are not in DFS standard
		{
			for (pri_j_index = start_pri_index;pri_j_index < (pulseCluster->pulseCount - 1);pri_j_index ++ )
			{
				priRatio = rawPri[pri_index] / rawPri[pri_j_index];
				priRemain = radarDetectionCalculateRemainder(rawPri[pri_index],rawPri[pri_j_index],priRatio);
				if ((priRatio <= MAX_PRI_RATIO) && (priRemain <= MAX_PRI_REMAIN ))
				{
					
					ampSubBandBitMap |= rawSubBandBitMap[pri_index];
					pri_cnt++;	
					if(pri_cnt >= RadarDetectionGlobalParameters.shortRadarPriTh)
					{
						if(ampSubBandBitMap == 0)
						{
							ampSubBandBitMap = 0xff;
						}
						ILOG0_D("enough PRIs %d", pri_cnt);
						return ampSubBandBitMap;
					}
				}
			}
		}
	}
	return FALSE;
}

/**********************************************************************************

radarDetectionCheckValidPri 


Description:
------------


Input: 
-----	
pulseCluster -
pulseCluster -
shortRadarType -

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static bool radarDetectionCheckValidPri(RadarDetectionPulseRecord_t *pulseRecord,RadarDetectionPulseCluster_t *pulseCluster)
{
	uint32 pri;
	
	if (pulseCluster->pulseQueue.pvTail != NULL) 
	{
		pri = pulseRecord->pulseTimeStamp - ((RadarDetectionPulseRecord_t *)pulseCluster->pulseQueue.pvTail)->pulseTimeStamp; 
		ILOG0_D("radarDetectionCheckValidPri: pri %d", pri);
		if(pri <= RadarDetectionGlobalParameters.shortRadarMinPriTh)
		{
			/*There is no valid pri smaller than the TH, so we ignored this pulse*/
			return FALSE;
		}
	}
	return TRUE;
}

/**********************************************************************************

radarDetectionCheckLongValidPri 


Description:
------------


Input: 
-----	
pulseCluster -
pulseCluster -

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static bool radarDetectionCheckLongValidPri(RadarDetectionPulseRecord_t *pulseRecord,RadarDetectionPulseCluster_t *pulseCluster)
{
	uint32 pri,pwDiff;

	if(pulseCluster->pulseQueue.pvTail != NULL) 
	{
		pri = pulseRecord->pulseTimeStamp - ((RadarDetectionPulseRecord_t *)pulseCluster->pulseQueue.pvTail)->pulseTimeStamp; 
		pwDiff = ABS_DIFFERENCE(pulseRecord->pulseWidth,((RadarDetectionPulseRecord_t *)pulseCluster->pulseQueue.pvTail)->pulseWidth);
		ILOG0_DD("radarDetectionCheckLongValidPri: pri %d, pwdiff %d", pri, pwDiff);
		/*1.There no valid pri smaller than the TH, so we ignored this pulse 
		  2.Incase the pri is between 1000 - 2000 the pulses should have the same PW*/
		if ((pri <= RadarDetectionGlobalParameters.longRadarMinPriTh) || ((pri <= RadarDetectionGlobalParameters.longRadarMaxPriTh) && (pwDiff > PW_RESOLUTION) ))
		{
			return FALSE;

		}
	}
	return TRUE;
}

/**********************************************************************************

radarDetectionCalculateRemainder 


Description:
------------
Calculate the absolute distance between the denominator and the remainer of 2 integers.


Input: 
-----	
numerator -
denominator -
ratio - 

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static uint32 radarDetectionCalculateRemainder(uint32 numerator ,uint32 denominator,uint32 ratio)
{
	uint32 remainder;

	remainder =  numerator % denominator;
	if((ratio < MAX_PRI_RATIO) && (remainder > (denominator>>1)))
	{
		remainder = denominator - remainder; 
	}
	return remainder;			
}



/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

RadarDetection_SetChannel


Description:
------------
Set new channel in the radar detection module

Input: 
-----	
channel - the new channel to set
		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void RadarDetection_SetChannel(uint8 channel, uint8 isRadarDetectionEnabled)
{
	RadarDetectionGlobalParameters.channel = channel;
	RadarDetectionGlobalParameters.wasRadarIndicationSent = FALSE;
	radarDetectionResetSamples();
	if (RadarDetectionGlobalParameters.ageingTimerActive == TRUE)
	{
		RadarDetectionGlobalParameters.ageingTimerActive = FALSE;
		OSAL_RESET_TIMER_EXPLICIT(INTERFERER_DETECTION_RADAR_AGING_TIMER, TASK_INTERFERER_DETECTION);
	}
	if (isRadarDetectionEnabled == TRUE)
	{
		RadarDetectionGlobalParameters.ageingTimerActive = TRUE;
		OSAL_SET_TIMER_EXPLICIT(INTERFERER_DETECTION_RADAR_AGING_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(RADAR_AGING_TIMER_IN_MILLISECONDS), TASK_INTERFERER_DETECTION);
	}
}

/**********************************************************************************

RadarDetection_ProcessSamples 


Description:
------------
Process the samples sent by the HDK module

Input:
-----
radarDetectionMessage - the message that contians the radar detection samples


		
Output:
-------
	

Returns:
--------

   	
**********************************************************************************/
void RadarDetection_ProcessSamples(K_MSG *radarDetectionMessage)
{
	InterfererDetectionRadarDetectionSamplesMessageParameters_t *radarDetectionSamplesMessage = (InterfererDetectionRadarDetectionSamplesMessageParameters_t *)radarDetectionMessage->abData; 
	uint8 numberOfSamples = 0;
	RadarDetectionPulseRecord_t *pulseRecord;
	RadarDetectionPulseCluster_t waitingPulseRecordsCluster; //pulse records waiting for sorting
	uint32 clusterIndex;	
	uint32 diffRadarCounter;
	uint32 difftInterfererCounter;
	uint32 difftDataNonAssocCounter;
	uint8  MT11h_RadarType = NO_RADAR;
	UMI_RADAR_DETECTION*  radarDetectionMessageParameters;
	K_MSG* pMsg = NULL;
	uint32 tempBitMapForSubBand = 0;
#ifdef RADAR_DEBUG_TRACE
	RadarDetectionPulseCluster_t* RadarDetectionPulseClusterForPrint = NULL;
#endif // RADAR_DEBUG_TRACE
    memset(&waitingPulseRecordsCluster, 0, sizeof(RadarDetectionPulseCluster_t));
	numberOfSamples = radarDetectionSamplesMessage->numberOfSamples;
	DEBUG_ASSERT(numberOfSamples <= MAX_NUMBER_OF_RADAR_SAMPLES_IN_MESSAGE); 
	DEBUG_ASSERT(numberOfSamples != 0);
   	//SLOG0(0, 0, InterfererDetectionRadarDetectionSamplesMessageParameters_t, radarDetectionSamplesMessage);
	RadarDetectionGlobalParameters.pulses.blockManagement.newestPulseTimeStamp = radarDetectionSamplesMessage->pulses[numberOfSamples - 1].timeStamp;
	RadarDetectionGlobalParameters.pulses.blockManagement.numOfNewPulseRec = numberOfSamples;
	radarDetectionResetAgedOutRecords();
	radarDetectionCopyPulseDetails(radarDetectionSamplesMessage,&waitingPulseRecordsCluster);

	ILOG0_V("================================== RadarDetection_ProcessSamples ======================================");
	/*Sort the pulses received from the phy */
	while(waitingPulseRecordsCluster.pulseQueue.pvHead)
	{	
		/*Get pulse record from temporary cluster*/
		pulseRecord = radarDetectionGetPulseRecord(&waitingPulseRecordsCluster);
		ILOG0_DD("@@@@@ record: pw %d, timestamp %u @@@@@", pulseRecord->pulseWidth, pulseRecord->pulseTimeStamp);
		//ILOG0_DDDD("RADAR PWs minshort=%d maxshort=%d minlong=%d maxlong=%d",RadarDetectionGlobalParameters.minShortPwTh, RadarDetectionGlobalParameters.maxShortPwTh, RadarDetectionGlobalParameters.minLongPwTh, RadarDetectionGlobalParameters.maxLongPwTh);
		//ILOG0_DDDD("RADAR PRIs minshort=%d, maxshort=%d minlong=%d maxlong=%d", RadarDetectionGlobalParameters.shortRadarMinPriTh, RadarDetectionGlobalParameters.shortRadarMaxPriTh, RadarDetectionGlobalParameters.longRadarMinPriTh, RadarDetectionGlobalParameters.longRadarMaxPriTh);

		/* PW is out of range*/
		if(((pulseRecord->pulseWidth >= RadarDetectionGlobalParameters.maxShortPwTh) && (pulseRecord->pulseWidth <= RadarDetectionGlobalParameters.minLongPwTh))||
			(pulseRecord->pulseWidth > RadarDetectionGlobalParameters.maxLongPwTh))
		{
			radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,pulseRecord);
			continue; 
		}
		
		/* PW is long*/
		else if((pulseRecord->pulseWidth > RadarDetectionGlobalParameters.minLongPwTh) && (pulseRecord->pulseWidth <= RadarDetectionGlobalParameters.maxLongPwTh))
		{	
			if(radarDetectionCheckLongValidPri(pulseRecord,&RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_LONG_CLUSTER_NUM]))	
			{
				pulseRecord->clusterIndex = RADAR_LONG_CLUSTER_NUM;
				radarDetectionPutPulseRecordGlobal(&RadarDetectionGlobalParameters.pulses.pulseRecordsQueue,pulseRecord);
				radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_LONG_CLUSTER_NUM],pulseRecord);
				radarDetectionCalculateNewAverageValue(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_LONG_CLUSTER_NUM],pulseRecord,RADAR_DETECTION_AVERAGE_CALCULATION_TYPE_ADDED_RECORD);
			}
			else
			{
				radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,pulseRecord);
			}
		}
		/* PW is short*/
		else if(pulseRecord->pulseWidth == 0)
		{
			if(radarDetectionCheckValidPri(pulseRecord,&RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_SHORT_CLUSTER_NUM]))
			{
				pulseRecord->clusterIndex = RADAR_SHORT_CLUSTER_NUM;
				radarDetectionPutPulseRecordGlobal(&RadarDetectionGlobalParameters.pulses.pulseRecordsQueue,pulseRecord);
				radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_SHORT_CLUSTER_NUM],pulseRecord);
				if(RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex < RADAR_SHORT_CLUSTER_NUM)
				{
					RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex = RADAR_SHORT_CLUSTER_NUM;
				}
			}
			else
			{
				radarDetectionPutPulseRecord(&RadarDetectionGlobalParameters.pulses.freePulseCluster,pulseRecord);
			}
		}
		else
		{
			/*sort  & cluster pulse by PW - */
			radarDetectionSortPulseRecord(pulseRecord);
		}
	}//while
	for(clusterIndex = FIRST_NON_RESERVED_CLUSTER; clusterIndex <= RadarDetectionGlobalParameters.pulses.blockManagement.biggestClusterIndex; clusterIndex++ )
	{	
		tempBitMapForSubBand = 0;
		/*Check if we have enough pulse records to continue the analysis */
		if((RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex].pulseCount >= MIN_NUM_OF_SHORT_PULSES))
			{
				tempBitMapForSubBand = radarDetectionCheckShortRadarPri(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex]);
			
				/*Determine the type of radar detected */
				
				//ILOG0_D("$$$$$$$$$$$$$$$$$$$$$$$$$$ Radar detected 5555%d $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$", MT11h_RadarType);
			
				if(tempBitMapForSubBand)
				{
					MT11h_RadarType = UMI_BSS_RADAR_NORM;
#ifdef RADAR_DEBUG_TRACE
					RadarDetectionPulseClusterForPrint = &RadarDetectionGlobalParameters.pulses.pulseClustersControl[clusterIndex];
#endif //RADAR_DEBUG_TRACE
				}
				break;
					
			}
	}
	/*Check if we reached minimal number of pulse to declare radar detection*/
	if ((RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_LONG_CLUSTER_NUM].pulseCount >= MIN_NUM_OF_LONG_PULSES)&&(MT11h_RadarType == NO_RADAR))
	{	
			
			tempBitMapForSubBand = radarDetectionCheckLongRadarPri(RadarDetectionGlobalParameters.pulses.pulseClustersControl);
			//ILOG0_D("$$$$$$$$$$$$$$$$$$$$$$$$$$ Radar detected 3333%d $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$", MT11h_RadarType);
			if(tempBitMapForSubBand)
			{
				MT11h_RadarType = UMI_BSS_RADAR_HOP;	
#ifdef RADAR_DEBUG_TRACE
				RadarDetectionPulseClusterForPrint = RadarDetectionGlobalParameters.pulses.pulseClustersControl;
#endif //RADAR_DEBUG_TRACE
			}
	}	
	if ((RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_SHORT_CLUSTER_NUM].pulseCount >= MIN_NUM_OF_ZERO_PW_PULSES)&&(MT11h_RadarType == NO_RADAR))
	{	
			
			//ILOG0_D("$$$$$$$$$$$$$$$$$$$$$$$$$$ Radar detected 22222%d $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$", MT11h_RadarType);
			tempBitMapForSubBand = radarDetectionCheckShortRadarPri(&RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_SHORT_CLUSTER_NUM]);
			if(tempBitMapForSubBand)
			{
				MT11h_RadarType = UMI_BSS_RADAR_NORM;
#ifdef RADAR_DEBUG_TRACE
				RadarDetectionPulseClusterForPrint = &RadarDetectionGlobalParameters.pulses.pulseClustersControl[RADAR_TYPE0_CLUSTER_NUM];
#endif //RADAR_DEBUG_TRACE
			}
	}
	if (MT11h_RadarType)
	{
		ILOG0_D("$$$$$$$$$$$$$$$$$$$$$$$$$$ Radar detected %d $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$", MT11h_RadarType);
		ILOG0_D("$$$$$$$$$$$$$$$$$$$$$$$$$$ Radar detected SubBandBitmap = %d $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$", tempBitMapForSubBand);
		/*Initialize module - reset all records*/
#ifndef RADAR_DEBUG_TRACE
		// For debug, don't delete samples here
		radarDetectionResetSamples();
#endif // !RADAR_DEBUG_TRACE
		if((RadarDetectionGlobalParameters.isIndicationEnabled == TRUE) && 
			(RadarDetectionGlobalParameters.radioState == ENABLE_RADIO) && 
		   ((!RadarDetectionGlobalParameters.wasRadarIndicationSent)||(getChannelNotification() != CHANNEL_NOTIFICATION_NORMAL)))
		{
			// When noise immunity feature enabled and we are not in filtering mode, check counter to prevent false detection
			if((RadarAdaptiveSensDataBase.featureDisabled == FALSE) && (RadarAdaptiveSensDataBase.radarInFilteringPeriod == FALSE))
			{
				radarAdaptiveSensStateMachineEnterEvent(RADAR_ADAPTIVE_SENS_EVENT_RADAR_DETECTED);
			}
			/*Adaptive sensetivity - filter radar according to phy statistics*/
			if ((RadarAdaptiveSensDataBase.featureDisabled == TRUE) || (RadarAdaptiveSensDataBase.radarInFilteringPeriod == FALSE))
			{
				ILOG0_V("RadarDetection_ProcessSamples, RADAR NOT FILTERED");

				RadarAdaptiveSensDataBase.isRadarFiltered = FALSE;
				RadarDetectionGlobalParameters.wasRadarIndicationSent = TRUE;

#ifdef RADAR_DEBUG_TRACE
				if(RadarDetectionPulseClusterForPrint != NULL)
				{
					//For each detection, first entry is the first sample timestamp, then list of samples in: 0x<PRI><PW> and after last sample 0xFFFFFFFF
					pulseRecord = (RadarDetectionPulseRecord_t *)RadarDetectionPulseClusterForPrint->pulseQueue.pvHead;

					DEBUG_ASSERT(pulseRecord);
					DEBUG_ASSERT(pulseRecord->nextPulseRecord);

					if(radarDebugPulseCounter == RADAR_DEBUG_TRACE_SIZE)
					{
						radarDebugPulseCounter = 0;
					}
					radarDebugPulseList[radarDebugPulseCounter++] = pulseRecord->pulseTimeStamp;

					while(pulseRecord->nextPulseRecord)
					{
						if(radarDebugPulseCounter == RADAR_DEBUG_TRACE_SIZE)
						{
							radarDebugPulseCounter = 0;
						}
						radarDebugPulseList[radarDebugPulseCounter++] = pulseRecord->pulseWidth + ((pulseRecord->nextPulseRecord->pulseTimeStamp -	pulseRecord->pulseTimeStamp)<<16);
						pulseRecord = pulseRecord->nextPulseRecord;
					}

					if(radarDebugPulseCounter == RADAR_DEBUG_TRACE_SIZE)
					{
						radarDebugPulseCounter = 0;
					}
					radarDebugPulseList[radarDebugPulseCounter++] = 0xFFFFFFFF;
				}
#endif //RADAR_DEBUG_TRACE

				pMsg = OSAL_GET_MESSAGE(sizeof(UMI_RADAR_DETECTION));
				radarDetectionMessageParameters = (UMI_RADAR_DETECTION*)pK_MSG_DATA(pMsg);
				radarDetectionMessageParameters->radarType = MT11h_RadarType;
				radarDetectionMessageParameters->channel = RadarDetectionGlobalParameters.channel;
				radarDetectionMessageParameters->subBandBitmap = tempBitMapForSubBand;
	 			OSAL_SEND_MESSAGE(UMI_MC_MAN_RADAR_IND, TASK_UM_IF_TASK, pMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
				if (RadarAdaptiveSensDataBase.enableSerialTrace == TRUE)
				{
					radarAdaptSensCalculateCountersDiff(&diffRadarCounter, &difftInterfererCounter, &difftDataNonAssocCounter, FALSE);
					sendSerialTraceToHdkCdbMan(HDK_RADAR_SERIAL_TRACE, RADAR_SERIAL_TRACE_RADAR_NOT_FILERED, difftDataNonAssocCounter, difftInterfererCounter, diffRadarCounter);
				}			
			}
			else
			{
				if (RadarAdaptiveSensDataBase.enableSerialTrace == TRUE)
				{
					radarAdaptSensCalculateCountersDiff(&diffRadarCounter, &difftInterfererCounter, &difftDataNonAssocCounter, FALSE);
					sendSerialTraceToHdkCdbMan(HDK_RADAR_SERIAL_TRACE, RADAR_SERIAL_TRACE_RADAR_FILTERED, difftDataNonAssocCounter, difftInterfererCounter, diffRadarCounter);
				}			
				ILOG0_V("RadarDetection_ProcessSamples, RADAR FILTERED");
				/*Fill radar indication parameters, message will be sent depending on radar adaptive sensetivity algorithm*/
				RadarAdaptiveSensDataBase.isRadarFiltered = TRUE;
				RadarAdaptiveSensDataBase.lastRadarDetectionParams.radarType = MT11h_RadarType;
				RadarAdaptiveSensDataBase.lastRadarDetectionParams.channel = RadarDetectionGlobalParameters.channel;
				RadarAdaptiveSensDataBase.lastRadarDetectionParams.subBandBitmap = tempBitMapForSubBand;
			}
 		}
#ifdef RADAR_DEBUG_TRACE
		radarDetectionResetSamples();
#endif //RADAR_DEBUG_TRACE
	}
}

/**********************************************************************************

RadarDetection_EnableRadarIndication 


Description:
------------
Configure the flag of radar indication - this flag is used for deciding if to send
indication to the driver in case radar is detected.

Input:
-----
radarDetectionMessage - the message that contians the value of the flag


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void RadarDetection_EnableRadarIndication(K_MSG *radarDetectionMessage)
{
	UMI_ENABLE_RADAR_INDICATION *enableRadarIndicationMessage = NULL;

 	enableRadarIndicationMessage = (UMI_ENABLE_RADAR_INDICATION *)radarDetectionMessage->abData; 
	RadarDetectionGlobalParameters.isIndicationEnabled = enableRadarIndicationMessage->enableIndication;

}

/**********************************************************************************

RadarDetection_AgingTimer 


Description:
------------
Aging Timeout

Input:
-----
radarDetectionMessage - the message that contians the value of the flag


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void RadarDetection_AgingTimer(K_MSG *radarDetectionMessage)
{	
#ifdef ENET_INC_ARCH_WAVE600
	RegPhyRxTdPhyTsfTimer_u phyTsfReg;
#else
	RegPhyRxTdPhyRxtdReg2B7_u phyTsfReg;
#endif //ENET_INC_ARCH_WAVE600
	UNUSED_PARAM(radarDetectionMessage);

	if (RadarDetectionGlobalParameters.ageingTimerActive == TRUE)
	{
#ifdef ENET_INC_ARCH_WAVE600
		RegAccess_Read(REG_PHY_RX_TD_PHY_TSF_TIMER,&phyTsfReg.val);
#else
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG2B7,&phyTsfReg.val);
#endif //ENET_INC_ARCH_WAVE600
		RadarDetectionGlobalParameters.pulses.blockManagement.newestPulseTimeStamp = phyTsfReg.bitFields.phyTsfTimer;
		RadarDetectionGlobalParameters.pulses.blockManagement.numOfNewPulseRec = 0;
		radarDetectionResetAgedOutRecords();
		OSAL_SET_TIMER_EXPLICIT(INTERFERER_DETECTION_RADAR_AGING_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(RADAR_AGING_TIMER_IN_MILLISECONDS), TASK_INTERFERER_DETECTION);
	}
}

/**********************************************************************************

RadarDetection_SetRadioState 


Description:
------------


Input:
-----



		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void RadarDetection_SetRadioState(uint8 radioState)
{
	RadarDetectionGlobalParameters.radioState = radioState;
}

/**********************************************************************************

isValidPri 


Description:
------------
return TRUE is (pw, pri) combination exist in DFS SPEC

Input:
-----


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static bool isValidPri(uint32 pri,uint8 pw)
{
	bool retVal = FALSE;
	
	if ((pw <= RadarDetectionGlobalParameters.maxShortPwTh) && (pri <= RadarDetectionGlobalParameters.shortRadarMaxPriTh))
	{
		retVal = TRUE;
	}
	
	return retVal;
}

/**********************************************************************************

RadarDetection_SetPRIsThresholds 


Description:
------------
Set threshold PRI count for short and long radar types detection

Input:
-----


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionSetPRIsThresholds(uint16 shortRadarMinPriTh, uint16 shortRadarMaxPriTh, uint16 longRadarMinPriTh, uint16 longRadarMaxPriTh)
{
	RadarDetectionGlobalParameters.shortRadarMinPriTh = shortRadarMinPriTh;
	RadarDetectionGlobalParameters.shortRadarMaxPriTh = shortRadarMaxPriTh;
	RadarDetectionGlobalParameters.longRadarMinPriTh = longRadarMinPriTh;
	RadarDetectionGlobalParameters.longRadarMaxPriTh = longRadarMaxPriTh;
}
/**********************************************************************************

RadarDetection_SetPWsThresholds 


Description:
------------
Set threshold PW for short and long radar types detection

Input:
-----


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void radarDetectionSetPWsThresholds(uint8 minShortPwTh, uint8 maxShortPwTh, uint8 minLongPwTh, uint8 maxLongPwTh)
{
	RadarDetectionGlobalParameters.minShortPwTh	= minShortPwTh;
 	RadarDetectionGlobalParameters.maxShortPwTh	= maxShortPwTh;
	RadarDetectionGlobalParameters.minLongPwTh	= minLongPwTh;
	RadarDetectionGlobalParameters.maxLongPwTh	= maxLongPwTh;
}

/**********************************************************************************

RadarDetection_SetRegDomainParams 


Description:
------------
Set PW and PRI thresholds according to new regulatory domain

Input:
-----


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void RadarDetection_SetRegDomainParams(uint8 RegulationType)
{
	switch(RegulationType)
	{
		case REG_DOMAIN_ETSI:
		case REG_DOMAIN_CHINA:
		{
			radarDetectionSetPRIsThresholds(ETSI_MIN_PRI_SHORT_RADAR, ETSI_MAX_PRI_SHORT_RADAR, ETSI_MIN_PRI_LONG_RADAR, ETSI_MAX_PRI_LONG_RADAR);
			radarDetectionSetPWsThresholds(ETSI_RADAR_MIN_PW_TH, ETSI_IGNORED_RADAR_MIN_PW, ETSI_IGNORED_RADAR_MAX_PW, ETSI_RADAR_MAX_PW_TH);
			break;
		}
		case REG_DOMAIN_JAPAN:
		{
			radarDetectionSetPRIsThresholds(JP_MIN_PRI_SHORT_RADAR, JP_MAX_PRI_SHORT_RADAR, JP_MIN_PRI_LONG_RADAR, JP_MAX_PRI_LONG_RADAR);
			radarDetectionSetPWsThresholds(JP_RADAR_MIN_PW_TH, JP_IGNORED_RADAR_MIN_PW, JP_IGNORED_RADAR_MAX_PW, JP_RADAR_MAX_PW_TH);
			break;
		}
		case REG_DOMAIN_KOREA:
		{
			radarDetectionSetPRIsThresholds(KR_MIN_PRI_SHORT_RADAR, KR_MAX_PRI_SHORT_RADAR, KR_MIN_PRI_LONG_RADAR, KR_MAX_PRI_LONG_RADAR);
			radarDetectionSetPWsThresholds(KR_RADAR_MIN_PW_TH, KR_IGNORED_RADAR_MIN_PW, KR_IGNORED_RADAR_MAX_PW, KR_RADAR_MAX_PW_TH);
			break;
		}
		case REG_DOMAIN_FCC:
		case REG_DOMAIN_DEFAULT:
		default:
		{
			radarDetectionSetPRIsThresholds(FCC_MIN_PRI_SHORT_RADAR, FCC_MAX_PRI_SHORT_RADAR, FCC_MIN_PRI_LONG_RADAR, FCC_MAX_PRI_LONG_RADAR);
			radarDetectionSetPWsThresholds(FCC_RADAR_MIN_PW_TH, FCC_IGNORED_RADAR_MIN_PW, FCC_IGNORED_RADAR_MAX_PW, FCC_RADAR_MAX_PW_TH);
			break;
		}
	}
}

void radarAdaptiveSensStateMachineEnterEvent(AdaptiveSensEvent_e event)
{
	switch (event)
	{
		case RADAR_ADAPTIVE_SENS_EVENT_SET_RADAR_CHANNEL_IN_CAC:
			switch (RadarAdaptiveSensDataBase.state)
			{
				case RADAR_ADAPTIVE_SENS_STATE_DISABLE:
				case RADAR_ADAPTIVE_SENS_STATE_CAC:
				case RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE:
					radarAdaptSensSetChannelEvent(&CacConfigurations, RADAR_ADAPTIVE_SENS_STATE_CAC);			
					break;
				default:
					ASSERT(0);
			}
			break;
		case RADAR_ADAPTIVE_SENS_EVENT_SET_RADAR_CHANNEL_IN_SERVICE:
			switch (RadarAdaptiveSensDataBase.state)
			{
				case RADAR_ADAPTIVE_SENS_STATE_DISABLE:
				case RADAR_ADAPTIVE_SENS_STATE_CAC:
				case RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE:
					radarAdaptSensSetChannelEvent(&InServiceConfigurations, RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE);			
					break;
				default:
					ASSERT(0);
			}
			break;
		case RADAR_ADAPTIVE_SENS_EVENT_TIMER_EXPIRED:
			switch (RadarAdaptiveSensDataBase.state)
			{
				case RADAR_ADAPTIVE_SENS_STATE_CAC:
					radarAdaptSensTimerExpiredEvent(&CacConfigurations);			
					break;
				case RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE:
					radarAdaptSensTimerExpiredEvent(&InServiceConfigurations);			
					break;
				default:
					ASSERT(0);
			}
			break;

		case RADAR_ADAPTIVE_SENS_EVENT_RADAR_DETECTED:
			switch (RadarAdaptiveSensDataBase.state)
			{
				case RADAR_ADAPTIVE_SENS_STATE_CAC:
					radarAdaptSensDetectionEvent(&CacConfigurations);			
					break;
				case RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE:
					radarAdaptSensDetectionEvent(&InServiceConfigurations);			
					break;
				default:
					ASSERT(0);
			}
			break;
			
		case RADAR_ADAPTIVE_SENS_EVENT_SET_NO_RADAR_CHANNEL:
			switch (RadarAdaptiveSensDataBase.state)
			{
				case RADAR_ADAPTIVE_SENS_STATE_DISABLE:
				case RADAR_ADAPTIVE_SENS_STATE_CAC:
				case RADAR_ADAPTIVE_SENS_STATE_IN_SERVICE:
					radarAdaptSensDisableEvent();			
					break;
				default:
					ASSERT(0);
			}
			break;
		default: 
			ASSERT(0);
	}
			
}
void radarDetectionStateMachineChangeState(AdaptiveSensState_e newState)
{
	ILOG0_D("radarDetectionStateMachineChangeState, newState = %d",newState);
	RadarAdaptiveSensDataBase.state = newState;
}
void radarAdaptSensSetChannelEvent(radarAdaptiveSensConfig_t* pConfig, AdaptiveSensState_e newState)
{
	uint32 currentRadarCounter;
	uint32 currentInterfererCounter;
	uint32 currentDataCounterNonAssoc;
	UNUSED_PARAM(pConfig);
	
	readCurrentRadarCountersFromPhy(&currentRadarCounter, &currentInterfererCounter, &currentDataCounterNonAssoc);
	updateCurrentRadarCountersInDb(currentRadarCounter, currentInterfererCounter, currentDataCounterNonAssoc);
	RadarAdaptiveSensDataBase.timerCounter = 0;
	radarDetectionStateMachineChangeState(newState);
	ILOG0_V("radarAdaptSensSetChannelEvent");
	SLOG0(0, 0, radarAdaptiveSensDb_t, &RadarAdaptiveSensDataBase);
}

/**********************************************************************************

radarAdaptSensCalculateCountersDiff 


Description:
------------
Read current noise counters from PHY and calculate diff from last read
Every 1sec. - update the saved counters.

Input:
-----
updateCountersInDb - update counters DB or not
		
Output:
-------
pDiffRadarCounter - diff between current radar counter and db radar counter
pDifftInterfererCounter - diff between current interferers counter and db interferers counter
pDifftDataNonAssocCounter - diff between current non associated packets counter and the same db counter

Returns:
--------

	
**********************************************************************************/
void radarAdaptSensCalculateCountersDiff(uint32* pDiffRadarCounter, uint32* pDifftInterfererCounter, uint32* pDifftDataNonAssocCounter, bool updateCountersInDb)
{
	
	uint32 currentRadarCounter;
	uint32 currentInterfererCounter;
	uint32 currentDataCounterNonAssoc;
	
	readCurrentRadarCountersFromPhy(&currentRadarCounter, &currentInterfererCounter, &currentDataCounterNonAssoc);
	*pDiffRadarCounter = currentRadarCounter -  RadarAdaptiveSensDataBase.radarCounter;
	*pDifftInterfererCounter = currentInterfererCounter - RadarAdaptiveSensDataBase.interfererCounter;
	*pDifftDataNonAssocCounter = currentDataCounterNonAssoc - RadarAdaptiveSensDataBase.dataCounterNonAssoc;

	if(TRUE == updateCountersInDb)
	{
		updateCurrentRadarCountersInDb(currentRadarCounter, currentInterfererCounter,currentDataCounterNonAssoc);
	}
}

/**********************************************************************************

radarAdaptSensDetectionEvent 


Description:
------------
Handle detection event, in case not in filtering period check counter for noise bursts and prevent false detction

Input:
-----
pConfig - current configuration params: in CAC or in service
		
Output:
-------

Returns:
--------

	
**********************************************************************************/
void radarAdaptSensDetectionEvent(radarAdaptiveSensConfig_t* pConfig)
{
	uint32 diffRadarCounter;
	uint32 difftInterfererCounter;
	uint32 difftDataNonAssocCounter;

	ILOG0_V("RadarDetection_ProcessSamples - Radar adaptive sens check burst");
	radarAdaptSensCalculateCountersDiff(&diffRadarCounter, &difftInterfererCounter, &difftDataNonAssocCounter, FALSE);
	/*Check filtering criteria*/
	RadarAdaptiveSensDataBase.radarInFilteringPeriod = isRadarFiltering(pConfig, diffRadarCounter, difftInterfererCounter, difftDataNonAssocCounter);
}

void radarAdaptSensTimerExpiredEvent(radarAdaptiveSensConfig_t* pConfig)
{
	uint32 diffRadarCounter;
	uint32 difftInterfererCounter;
	uint32 difftDataNonAssocCounter;
	UMI_RADAR_DETECTION*  radarDetectionMessageParameters;
	K_MSG* pMsg = NULL;
	

	RadarAdaptiveSensDataBase.timerCounter++;
	if (RadarAdaptiveSensDataBase.timerCounter >= pConfig->stateTimeCounterTh)
	{
		
		ILOG0_V("radarAdaptSensTimerExpiredEvent - Radar adaptive sens timer expired");
		RadarAdaptiveSensDataBase.timerCounter = 0;
		radarAdaptSensCalculateCountersDiff(&diffRadarCounter, &difftInterfererCounter, &difftDataNonAssocCounter, TRUE);
		
//***************DEBUG*********
		if (RadarAdaptiveSensDataBase.enableSerialTrace == TRUE)
		{
			sendSerialTraceToHdkCdbMan(HDK_RADAR_SERIAL_TRACE, RADAR_SERIAL_TRACE_TIMER_EXPIRED, difftDataNonAssocCounter, difftInterfererCounter, diffRadarCounter);
		}
//*****************************
		SLOG0(0, 0, radarAdaptiveSensDb_t, &RadarAdaptiveSensDataBase);
		SLOG0(0, 0, radarAdaptiveSensConfig_t, &CacConfigurations);
		SLOG0(0, 0, radarAdaptiveSensConfig_t, &InServiceConfigurations);

		/*Check filtering criteria*/
		RadarAdaptiveSensDataBase.radarInFilteringPeriod = isRadarFiltering(pConfig, diffRadarCounter,difftInterfererCounter,difftDataNonAssocCounter);
		ILOG0_D("RADAR FILTERED = %d", RadarAdaptiveSensDataBase.radarInFilteringPeriod);
		if ((RadarAdaptiveSensDataBase.radarInFilteringPeriod == FALSE) && (RadarAdaptiveSensDataBase.isRadarFiltered == TRUE))
		{
			ILOG0_V("radarAdaptSensTimerExpiredEvent, REPORT RADAR");

			pMsg = OSAL_GET_MESSAGE(sizeof(UMI_RADAR_DETECTION));
			radarDetectionMessageParameters = (UMI_RADAR_DETECTION*)pK_MSG_DATA(pMsg);
			radarDetectionMessageParameters->radarType = RadarAdaptiveSensDataBase.lastRadarDetectionParams.radarType;
			radarDetectionMessageParameters->channel = RadarAdaptiveSensDataBase.lastRadarDetectionParams.channel;
			radarDetectionMessageParameters->subBandBitmap = RadarAdaptiveSensDataBase.lastRadarDetectionParams.subBandBitmap;
			OSAL_SEND_MESSAGE(UMI_MC_MAN_RADAR_IND, TASK_UM_IF_TASK, pMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
		}
		RadarAdaptiveSensDataBase.isRadarFiltered = FALSE;
	}
}
static bool isRadarFiltering(radarAdaptiveSensConfig_t* pConfig, uint32 diffRadarCounter, uint32 difftInterfererCounter, uint32 difftDataNonAssocCounter)
{
	bool filterRadar = FALSE;

	ILOG0_DDD("isRadarFiltering, diffRadarCounter = %d, difftInterfererCounter = %d, difftDataNonAssocCounter = %d", diffRadarCounter, difftInterfererCounter, difftDataNonAssocCounter);
	SLOG0(0, 0, radarAdaptiveSensConfig_t, pConfig);
	if ((difftDataNonAssocCounter > pConfig->nonAssoDataPacketsTh) && ((INTERFERE_COUNTER_FACTOR*difftInterfererCounter) > ((pConfig->InterferernceRadarRatio)*diffRadarCounter)))
	{
		filterRadar = TRUE;
	}

	return filterRadar;
}

void radarAdaptSensDisableEvent()
{
	radarDetectionStateMachineChangeState(RADAR_ADAPTIVE_SENS_STATE_DISABLE);

}
static void updateCurrentRadarCountersInDb(uint32 currentRadarCounter, uint32 currentInterfererCounter,uint32 currentDataCounterNonAssoc)
{
	RadarAdaptiveSensDataBase.radarCounter = currentRadarCounter;
	RadarAdaptiveSensDataBase.interfererCounter = currentInterfererCounter;
	RadarAdaptiveSensDataBase.dataCounterNonAssoc = currentDataCounterNonAssoc;
}
static void readCurrentRadarCountersFromPhy(uint32* pCurrentRadarCounter, uint32* pCurrentInterfererCounter,uint32* pCurrentDataCounterNonAssoc)
{
	uint32 tempRadarCountr1;
	uint32 tempRadarCountr2;
	uint32 tempRadarCountr3;
	uint32 tempRadarCountr4;
#if !defined (ENET_INC_ARCH_WAVE600)
	uint32 tempRadarCountr5;
#endif
#if defined (ENET_INC_ARCH_WAVE600)
	RegAccess_Read(MEM_INTERFERER_COUNTER, pCurrentInterfererCounter);
	RegAccess_Read(OFDM_PACKET_COUNT_5G_NOT_TO_MY_AP, pCurrentDataCounterNonAssoc);
	RegAccess_Read(RADAR_COUNTER_0_2U, &tempRadarCountr1);
	RegAccess_Read(RADAR_COUNTER_2_6U, &tempRadarCountr2);
	RegAccess_Read(RADAR_COUNTER_6_7U, &tempRadarCountr3);
	RegAccess_Read(RADAR_COUNTER_LONG, &tempRadarCountr4);
#else
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_INTERFERER_COUNTER_ADDRESS, pCurrentInterfererCounter);
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_PACKET_DETECT_COUNTER_ADDRESS, pCurrentDataCounterNonAssoc);
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_PACKET_RADAR_COUNTERS1_ADDRESS, &tempRadarCountr1);
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_PACKET_RADAR_COUNTERS2_ADDRESS, &tempRadarCountr2);
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_PACKET_RADAR_COUNTERS3_ADDRESS, &tempRadarCountr3);
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_PACKET_RADAR_COUNTERS4_ADDRESS, &tempRadarCountr4);
	RegAccess_Read(PHY_TD_BASE_ADDR + PHY_PACKET_RADAR_COUNTERS5_ADDRESS, &tempRadarCountr5);
#endif
#if defined (ENET_INC_ARCH_WAVE600)
	*pCurrentRadarCounter = tempRadarCountr1 + tempRadarCountr2 + tempRadarCountr3 + tempRadarCountr4;
#else
	*pCurrentRadarCounter = tempRadarCountr1 + tempRadarCountr2 + tempRadarCountr3 + tempRadarCountr4 + tempRadarCountr5;
#endif
		
}
void radarDetectionAdaptiveSensUpdateParams(bool  disableFeture, uint8 nonAssoDataPacketsTh, uint8 InterferernceRadarRatio)
{
	uint32 currentRadarCounter;
	uint32 currentInterfererCounter;
	uint32 currentDataCounterNonAssoc;

	RadarAdaptiveSensDataBase.featureDisabled = disableFeture;
	CacConfigurations.nonAssoDataPacketsTh = nonAssoDataPacketsTh;
	CacConfigurations.InterferernceRadarRatio = InterferernceRadarRatio;
	InServiceConfigurations.nonAssoDataPacketsTh = nonAssoDataPacketsTh;
	InServiceConfigurations.InterferernceRadarRatio = InterferernceRadarRatio;
	
	if (disableFeture == FALSE)
	{
		
		readCurrentRadarCountersFromPhy(&currentRadarCounter, &currentInterfererCounter, &currentDataCounterNonAssoc);
		updateCurrentRadarCountersInDb(currentRadarCounter, currentInterfererCounter, currentDataCounterNonAssoc);
		RadarAdaptiveSensDataBase.isRadarFiltered = 0;
		RadarAdaptiveSensDataBase.radarInFilteringPeriod = TRUE;
		RadarAdaptiveSensDataBase.timerCounter = 0;
	}
}
void radarAdaptiveSensInit(void)
{
	memset(&RadarAdaptiveSensDataBase,0, sizeof(radarAdaptiveSensDb_t));
	RadarAdaptiveSensDataBase.radarInFilteringPeriod = TRUE; //start with filtering until statistics will be collected for the first time
}
void sendSerialTraceToHdkCdbMan(K_MSG_TYPE msgType, HdkRadarDetectionSerialTrace_e serialTraceType, uint32 val1, uint32 val2, uint32 val3)

{
	K_MSG* pRadarSerialTraceMsg;
	RadarSerialTraceMsg_t* radarSerialTraceParams;
	pRadarSerialTraceMsg = OSAL_GET_MESSAGE(sizeof(RadarSerialTraceMsg_t));
	radarSerialTraceParams = (RadarSerialTraceMsg_t*)pK_MSG_DATA(pRadarSerialTraceMsg);
	radarSerialTraceParams->val1 = val1;
	radarSerialTraceParams->val2 = val2;
	radarSerialTraceParams->val3 = val3;
	radarSerialTraceParams->serialTraceType = serialTraceType;
		
	OSAL_SEND_MESSAGE(msgType, TASK_HDK_CDB_MANAGER, pRadarSerialTraceMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
}
void RadarAdaptiveSensEnableSerialTrace(bool enable)
{
	RadarAdaptiveSensDataBase.enableSerialTrace = enable;
}
/**********************************************************************************

RadarDetection_Initialize 


Description:
------------
Initialize the radar detection internal parameters and structures

Input:
-----


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void RadarDetection_Initialize(void)
{
	radarDetectionResetSamples();
	RadarDetectionGlobalParameters.isIndicationEnabled = FALSE;
	RadarDetectionGlobalParameters.ageingTimerActive = FALSE;
	RadarDetectionGlobalParameters.shortRadarPriTh = MIN_PRI_COUNT;
	RadarDetectionGlobalParameters.longRadarPriTh = MIN_NUM_OF_LONG_PULSES - 1;
	radarAdaptiveSensInit();

#ifdef RADAR_DEBUG_TRACE
		memset((void*)radarDebugPulseList, 0, sizeof(radarDebugPulseList));
		radarDebugPulseCounter = 0;
#endif //RADAR_DEBUG_TRACE
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

 
#endif //RADAR_DETECTION_ENABLED

