/***********************************************************************************
 File:		InterruptManager_Api.h
 Module:		Interrupt Manager
 Purpose: 		
 Description:	
************************************************************************************/
#ifndef INTERRUPTS_API_H_
#define INTERRUPTS_API_H_
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/

#if defined (CPU_ARC)
//#define ISR_VOID	_Fast_Interrupt void
#define ISR_VOID	void // I removed the _Fast_interrupt since it seems it can't work due to the assembler wrapper.

#endif




/**********************************************************************************
Description:
------------
	Gen 6 Definitions
	
**********************************************************************************/

/****************************************************/
/*	Mapped Interrupt definitions for Upper Mac Gen6	*/
/****************************************************/
#if defined(ENET_INC_UMAC)
typedef enum
{  
	MAPPED_INT_HW_EVENTS_BAND0_A = 1,
	MAPPED_INT_HW_EVENTS_BAND1_A,
	MAPPED_INT_HW_ERRORS_COMMON,
	MAPPED_INT_HW_EVENTS_COMMON_A,
	MAPPED_INT_DMA,
	MAPPED_INT_RAB_INTER_CPU,
	MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY,
	MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY,
	MAPPED_INT_LOCKER_BAND0,
	MAPPED_INT_LOCKER_BAND1,
	MAPPED_INT_HW_EVENTS_BAND0_B,
	MAPPED_INT_HW_EVENTS_BAND1_B,
#ifdef ATOMIC_COUNTER_ENABLER	
	MAPPED_INT_HOST_MAILBOX_NOT_EMPTY,
#endif
	MAPPED_INT_SYSTEM_IF_TIMERS_0,
	MAPPED_INT_SYSTEM_IF_TIMERS_1,
	MAPPED_INT_TOTAL_NUM = MAPPED_INT_SYSTEM_IF_TIMERS_1,
	MAPPED_INT_MAX_NUM = 0xFF,
} MappedIntId_e;
#endif /* ENET_INC_UMAC*/    


/****************************************************/
/*    Mapped Interrupt definitions for Lower Gen6   */
/****************************************************/
#if defined(ENET_INC_LMAC) 
#if defined(ENET_INC_LMAC0) 

typedef enum
{
	MAPPED_INT_HW_ERRORS_BAND0 = 1,
	MAPPED_INT_HW_EVENTS_BAND0_A,
	MAPPED_INT_RAB_INTER_CPU,
	MAPPED_INT_SYSTEM_IF_TIMERS_0,
	MAPPED_INT_SYSTEM_IF_TIMERS_1,
	MAPPED_INT_SYSTEM_IF_TIMERS_2,
	MAPPED_INT_SYSTEM_IF_TIMERS_3,
	MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY_BAND0,
	MAPPED_INT_LOCKER_BAND0,
	MAPPED_INT_BAA_RA_FIFO_NOT_EMPTY_BAND0,
	MAPPED_INT_PAUSER_BAND0,	// Interrupt is in UM for gen5 and in LM for gen6
	MAPPED_INT_ULPR_FIFO_NOT_EMPTY_BAND0,
	MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND0,
	MAPPED_INT_HW_EVENTS_BAND0_B,
	MAPPED_INT_TOTAL_NUM = MAPPED_INT_HW_EVENTS_BAND0_B,
	MAPPED_INT_MAX_NUM   = 0xFF,
} MappedIntId_e;
#endif
#if defined(ENET_INC_LMAC1) 
typedef enum
{
	MAPPED_INT_HW_ERRORS_BAND1 = 1,
	MAPPED_INT_HW_EVENTS_BAND1_A,
	MAPPED_INT_RAB_INTER_CPU,
	MAPPED_INT_SYSTEM_IF_TIMERS_0,
	MAPPED_INT_SYSTEM_IF_TIMERS_1,
	MAPPED_INT_SYSTEM_IF_TIMERS_2,
	MAPPED_INT_SYSTEM_IF_TIMERS_3,
	MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY_BAND1,
	MAPPED_INT_LOCKER_BAND1,
	MAPPED_INT_BAA_RA_FIFO_NOT_EMPTY_BAND1,
	MAPPED_INT_PAUSER_BAND1,	// Interrupt is in UM for gen5 and in LM for gen6
	MAPPED_INT_ULPR_FIFO_NOT_EMPTY_BAND1,
	MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND1,
	MAPPED_INT_HW_EVENTS_BAND1_B,
	MAPPED_INT_TOTAL_NUM = MAPPED_INT_HW_EVENTS_BAND1_B,
	MAPPED_INT_MAX_NUM   = 0xFF,
} MappedIntId_e;

#endif
#endif /* ENET_INC_LMAC*/

#define NUM_OF_ARC_INTERNAL_INTERRUPTS	(3)	// 2 timers + 1 performance counters interrupts
#define NUM_OF_ARC_INTERRUPTS			(32 + NUM_OF_ARC_INTERNAL_INTERRUPTS)


/********************************************************/
/*	Edged Triggered Interrupt definitions Gen6			*/
/********************************************************/
/* The following interrupts are converted from edge to level by the MAC_HT_EXT block */
typedef enum 
{
	EDGED_TRIGGERED_INT_DEL_TO_GENRISCTXENDIRQ			= 8,  // Delia 0/1 to genrisc Tx end IRQ
	EDGED_TRIGGERED_INT_AUTO_FILL_END					= 15, // Delia 0 auto fil end IRQ	
	EDGED_TRIGGERED_INT_PAC_TICK_TIMER 					= 18, // PAS PAC Tick timer
	EDGED_TRIGGERED_INT_MAC_PHY_GP_IF_STATUS			= 19, // MAC_PHY GP IF IRQ status
	EDGED_TRIGGERED_INT_TX_ERROR 						= 20, // PAS PAC 0 TX Error
	EDGED_TRIGGERED_DEL_TIMER_EXPIRED					= 25, // del limit timer expired IRQ
} EdgedTriggeredIntId_e;

/* The following interrupts are converted from edge to level by the MAC_GENERAL block */

typedef enum 
{
	EDGED_TRIGGERED_INT_GPIO_0 =	0,  // External IRQ
	EDGED_TRIGGERED_INT_GPIO_1 = 	1,	//
	EDGED_TRIGGERED_INT_GPIO_10 =	2,	// 	
	EDGED_TRIGGERED_INT_GPIO_11 =	3,	//
} EdgedTriggeredIntMacGeneralId_e;


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/* Define polarity for REG_MAC_GENERAL_EXTERNAL_IRQ_INVERT */
#define EXTERNAL_IRQ_INVERT_LOW		(1) /* Interrupt is triggered when signal changes from HIGH to LOW */
#define EXTERNAL_IRQ_INVERT_HIGH	(0) /* Interrupt is triggered when signal changes from LOW to HIGH */
/*---------------------------------------------------------------------------------
/						Data Type Definition						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Function Declaration						
/----------------------------------------------------------------------------------*/

#if defined (ANALYZE_DISABLE_INTERRUPT)
void InterruptManager_DisableAllInterrupts(unsigned int* pInterrupt_save, char* fileName, uint32 line);
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save, char* fileName, uint32 line);
#else
void InterruptManager_DisableAllInterrupts(unsigned int* pInterrupt_save);
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save);
#endif	
bool InterruptManager_IsInInterruptContext(void);
void InterruptManager_MaskInterrupt(MappedIntId_e mappedIntId);
void InterruptManager_UnMaskInterrupt(MappedIntId_e mappedIntId);
void InterruptManager_ClearEdgedTriggeredInterrupt(EdgedTriggeredIntId_e intId);
void InterruptManager_ClearMacGeneralEdgedTriggeredInterrupt(EdgedTriggeredIntMacGeneralId_e intId);
void InterruptManagerConfigureMacGeneralPolarity(EdgedTriggeredIntMacGeneralId_e intId, uint8 polarity);
void InterruptManager_Configure(void);
void InterruptManager_Reset(void);
void InterruptManager_EnableUpiRabInt(uint8 intIndex);
void InterruptManager_DisableUpiRabInt(uint8 intIndex);
void InterruptManager_EnableLpiRabInt(uint8 intIndex);
void InterruptManager_DisableLpiRabInt(uint8 intIndex);
void InterruptManager_EnablePhiRabInt(uint8 intIndex);
void InterruptManager_DisablePhiRabInt(uint8 intIndex);
void InterruptManager_EnableNpuRabInt(uint8 intIndex);
void InterruptManager_DisableNpuRabInt(uint8 intIndex);
void InterruptManagerConfigureMacGeneraExternalIrqInvert(EdgedTriggeredIntMacGeneralId_e intId, uint8 polarity);





#if defined (ENET_INC_LMAC)
ISR_VOID vLM_pac_tsf_timer_int(void);
void LM_ClearAllInterrupts(void);
#endif


void InterruptManager_RabInit(void);
void vIGEN_SendToUm(uint32 u32Source);
void vIGEN_SendToLm0(uint32 u32Source);
#if defined (ENET_INC_ARCH_WAVE600)
void vIGEN_SendToLm1(uint32 u32Source);
#endif


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/
#if defined (ANALYZE_DISABLE_INTERRUPT)
#define AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE	(2048) //a power of 2 in order to use modulus and ....
	
	
extern uint32 AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE][7];
extern uint16 AAA_DebugTrace_analyzeDisableInterrupts_idx;
	
	
	
#define AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_MACRO(x,fileName,l) \
	{\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][0] = GET_TSF_TIMER_LOW();\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][1] = x;\
		RegAccess_Read(REG_EMERALD_ENV_IRQ_MASK,    &AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][2]);\
		RegAccess_Read(REG_EMERALD_ENV_IRQ_MAPPED,  &AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][3]);\
		RegAccess_Read(REG_EMERALD_ENV_IRQ_UNMAPPED,&AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][4]);\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][5] = (uint32)fileName;\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][6] = l;\
		AAA_DebugTrace_analyzeDisableInterrupts_idx = (AAA_DebugTrace_analyzeDisableInterrupts_idx + 1) & (AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE - 1);\
	}
#else
#define AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_MACRO(x,fileName,l) {}
#endif


#endif /* INTERRUPTS_API_H_*/

