/***********************************************************************************
 File:		InterruptManager_Api.h
 Module:		Interrupt Manager
 Purpose: 		
 Description:	
************************************************************************************/
#ifndef INTERRUPTS_API_H_
#define INTERRUPTS_API_H_
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/

#if defined (CPU_MIPS)
#define ISR_VOID	void
#endif


/**********************************************************************************
Description:
------------
	Gen 5 Definitions
	
**********************************************************************************/

/*******************************************/
/*	Mapped Interrupt definitions for Upper Mac Gen5	*/
/*******************************************/
#if defined(ENET_INC_UMAC)
typedef enum
{  
	MAPPED_INT_HW_ERRORS = 1,
	MAPPED_INT_HW_EVENTS,
	MAPPED_INT_PAC_TICK_TIMER,
	MAPPED_INT_TX_DONE_LISTS,
	MAPPED_INT_DMA,
	MAPPED_INT_INTER_CPU,
	MAPPED_INT_GP_TIMERS_A,
	MAPPED_INT_TX_READY_PD_LIST,
	MAPPED_INT_RX_RD_LIST_NOT_EMPTY,
	MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY,
	MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY,
	MAPPED_INT_SHRAM_ADDR_INT,
	MAPPED_INT_GENRISC_RX_HANDLER,
	MAPPED_INT_GENRISC_HOST_IF,
	MAPPED_INT_STATUS_FIFO,
	MAPPED_INT_LOCKER,
	MAPPED_INT_PAUSER,	// Interrupt is in UM for gen5 and in LM for gen6
	MAPPED_INT_DISABLE_LIST_NOT_EMPTY,
	MAPPED_INT_GENRISC_SENDER,
#ifdef ATOMIC_COUNTER_ENABLER
	MAPPED_INT_HOST_MAILBOX_NOT_EMPTY,
	MAPPED_INT_TOTAL_NUM = MAPPED_INT_HOST_MAILBOX_NOT_EMPTY,
#else
	MAPPED_INT_TOTAL_NUM = 	MAPPED_INT_GENRISC_SENDER,
#endif
	MAPPED_INT_MAX_NUM   = 0xFF,
} MappedIntId_e;
#endif /* ENET_INC_UMAC*/    


/******************************************/
/*    Mapped Interrupt definitions for Lower mac Gen5   */
/******************************************/
#if defined(ENET_INC_LMAC) 

typedef enum
{
	MAPPED_INT_HW_ERRORS = 1,
	MAPPED_INT_HW_EVENTS,
	MAPPED_INT_INTER_CPU,
	MAPPED_INT_TSF_TIMER,
	MAPPED_INT_GP_TIMERS_A,
	MAPPED_INT_GP_TIMERS_B,
	MAPPED_INT_AGGREGATION_BUILDER,
	MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY,
	MAPPED_INT_LOCKER,
#if defined(CALCULATE_CPU_IDLE_TIME) 
	MAPPED_INT_PAC_TICK_TIMER,
#endif //CALCULATE_CPU_IDLE_TIME
	MAPPED_INT_BAA_FIFO_NOT_EMPTY,
	MAPPED_INT_TOTAL_NUM = MAPPED_INT_BAA_FIFO_NOT_EMPTY,
	MAPPED_INT_MAX_NUM   = 0xFF,
} MappedIntId_e;
#endif /* ENET_INC_LMAC*/


/*******************************************/
/*	Edged Triggered Interrupt definitions Gen5			*/
/*******************************************/
typedef enum 
{
	EDGED_TRIGGERED_INT_PAS_PAC_1_TXERROR				= 2,	// PAS PAC 1 TX Error
	EDGED_TRIGGERED_INT_PAS_PAC_2_TXERROR				= 3,	// PAS PAC 2 TX Error
	EDGED_TRIGGERED_INT_PAS_PAC_3_TXERROR				= 4,	// PAS PAC 3 TX Error
	EDGED_TRIGGERED_INT_DEL_1_AUTO_FILL_CTRLIMITIRQ		= 5,	// Delia 1 auto fill ctr limit IRQ
	EDGED_TRIGGERED_INT_DEL_2_AUTO_FILL_CTRLIMITIRQ		= 6,	// Delia 2 auto fill ctr limit IRQ
	EDGED_TRIGGERED_INT_DEL_3_AUTO_FILL_CTRLIMITIRQ		= 7,	// Delia 3 auto fill ctr limit IRQ
	EDGED_TRIGGERED_INT_DEL_TO_GENRISC0TXENDIRQ			= 8,	// Delia 0 to genrisc Tx end IRQ
	EDGED_TRIGGERED_INT_DEL_TO_GENRISC1TXENDIRQ			= 9,	// Delia 1 to genrisc Tx end IRQ
	EDGED_TRIGGERED_INT_DEL_TO_GENRISC2TXENDIRQ			= 10,	// Delia 2 to genrisc Tx end IRQ
	EDGED_TRIGGERED_INT_DEL_TO_GENRISC3TXENDIRQ			= 11,	// Delia 3 to genrisc Tx end IRQ
	EDGED_TRIGGERED_INT_DEL_TO_GENRISC_START			= 14, // All Delias to genrisc Tx end IRQ
	EDGED_TRIGGERED_INT_AUTO_FILL_END					= 15, // Delia 0 auto fil end IRQ	
	EDGED_TRIGGERED_INT_PAC_TICK_TIMER 					= 18, // PAS PAC Tick timer
	EDGED_TRIGGERED_INT_TX_ERROR 						= 20, // PAS PAC 0 TX Error
	EDGED_TRIGGERED_INT_RX_AGGR_ERROR					= 21, // Rx Aggregation error indication
	EDGED_TRIGGERED_INT_RX_EOF_ERROR					= 22, // Rx EOF error indication
	EDGED_TRIGGERED_INT_BLOCK_ACK_ERROR					= 23, // Block ack error indication
	EDGED_TRIGGERED_INT_EXTERNAL						= 24, // external_irq_reg
	EDGED_TRIGGERED_DEL_TIMER_EXPIRED					= 25, // del limit timer expired IRQ
	EDGED_TRIGGERED_INT_DEL_RX_HALT						= 27, // Delineator Rx Halt interrupt
	EDGED_TRIGGERED_INT_NPU_ARB_BUS_ERROR				= 28, // NPU arb bus err IRQ
	EDGED_TRIGGERED_INT_REDUCED_FREQ					= 29, // reduced freq IRQ	
	EDGED_TRIGGERED_INT_AUTO_FILL_COUNTER_LIMIT			= 30, // Delia 0 auto fill ctr limit IRQ
	EDGED_TRIGGERED_INT_DELINEATOR_FIFO_FULL			= 31, // Delineator FIFO Full IRQ
} EdgedTriggeredIntId_e;



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Function Declaration						
/----------------------------------------------------------------------------------*/

#if defined (ANALYZE_DISABLE_INTERRUPT)
void InterruptManager_DisableAllInterrupts(unsigned int* pInterrupt_save, char* fileName, uint32 line);
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save, char* fileName, uint32 line);
#else
void InterruptManager_DisableAllInterrupts(unsigned int* pInterrupt_save);
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save);
#endif	
bool InterruptManager_IsInInterruptContext(void);
void InterruptManager_MaskInterrupt(MappedIntId_e mappedIntId);
void InterruptManager_UnMaskInterrupt(MappedIntId_e mappedIntId);
void InterruptManager_ClearEdgedTriggeredInterrupt(EdgedTriggeredIntId_e intId);
void InterruptManager_Configure(void);
void InterruptManager_Reset(void);
void InterruptManager_EnableUpiRabInt(uint8 intIndex);
void InterruptManager_DisableUpiRabInt(uint8 intIndex);
void InterruptManager_EnableLpiRabInt(uint8 intIndex);
void InterruptManager_DisableLpiRabInt(uint8 intIndex);
void InterruptManager_EnablePhiRabInt(uint8 intIndex);
void InterruptManager_DisablePhiRabInt(uint8 intIndex);
void InterruptManager_EnableNpuRabInt(uint8 intIndex);
void InterruptManager_DisableNpuRabInt(uint8 intIndex);





#if defined (ENET_INC_LMAC)
ISR_VOID vLM_pac_tsf_timer_int(void);
void LM_ClearAllInterrupts(void);
#endif


void InterruptManager_RabInit(void);
void vIGEN_SendToUm(uint32 u32Source);
void vIGEN_SendToLm0(uint32 u32Source);
#if defined (ENET_INC_ARCH_WAVE600)
void vIGEN_SendToLm1(uint32 u32Source);
#endif


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/
#if defined (ANALYZE_DISABLE_INTERRUPT)
#define AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE	(2048) //a power of 2 in order to use modulus and ....
	
	
extern uint32 AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE][7];
extern uint16 AAA_DebugTrace_analyzeDisableInterrupts_idx;
	
	
	
#define AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_MACRO(x,fileName,l) \
	{\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][0] = GET_TSF_TIMER_LOW();\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][1] = x;\
		RegAccess_Read(REG_EMERALD_ENV_IRQ_MASK,    &AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][2]);\
		RegAccess_Read(REG_EMERALD_ENV_IRQ_MAPPED,  &AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][3]);\
		RegAccess_Read(REG_EMERALD_ENV_IRQ_UNMAPPED,&AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][4]);\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][5] = (uint32)fileName;\
		AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DebugTrace_analyzeDisableInterrupts_idx][6] = l;\
		AAA_DebugTrace_analyzeDisableInterrupts_idx = (AAA_DebugTrace_analyzeDisableInterrupts_idx + 1) & (AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE - 1);\
	}
#else
#define AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_MACRO(x,fileName,l) {}
#endif


#endif /* INTERRUPTS_API_H_*/

