/***********************************************************************************
 File:			InterruptManager.c			
 Module:			Interrupt Module		
 Purpose:			
 Description:		
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_Timers.h"
#include "arc_intrinsics.h"
#include "System_CoreTimersApi.h"
#include "InterruptManager_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "ErrorHandler_Api.h"
#include "int_gen.h"
#include "encryptor.h"
#include "SenderInterface_Api.h"
#include "HostGenRiscInterface_Api.h"
#if defined (ENET_INC_LMAC)
#include "PreAggregator_Api.h"
#endif
#include "Utils_Api.h"
#include "Pac_Api.h"
#include "RegAccess_Api.h"
#include "MacHtExtensionsRegs.h"
#include "EmeraldEnvRegs.h"
#include "Debug_Trace.h"
#include "CpuLoad_Api.h"
#include "shram_interruptSection.h"
#include "TxSelector_Api.h"
#include "MacGeneralRegs.h"
#if defined (ENET_INC_UMAC)
#include "HostInterface_API.h"
#include "TxPacketsClassifier_API.h"
#include "TxManager_Api.h"
#include "lm_interface.h"
#include "tx_port.h"
#include "BaAnalyzer_Api.h"
#endif
#include "RxManager_Api.h"
#include "linkAdaptation_api.h"
#if defined (ENET_INC_LMAC)
#include "lmtimerUtility.h"
#endif
#include "linkAdaptation_api.h"
#include "TxSequencer_Api.h"
#include "loggerAPI.h"
#include "DmaManager_Api.h"
#include "TpcClbrHndlr.h"
#include "lmtimerUtility.h"
#include "Pauser_Api.h"
#include "ipc_api.h"
#include "TxManager_Api.h"
#include "lm_interface.h"
#include "tx_port.h"
#include "BaAnalyzer_Api.h"
#include "PacManager_api.h"
#include "performanceCounters_api.h"



/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_SYS
#define LOG_LOCAL_FID 8

#define PIC_MAPPER_NUM_OF_BITS		(6)
#define CLEAR_ALL_INTERRUPTS		(0)

#define OS_STATUS_REG_RB_BITS		(0x00070000)/* Register Bank bits */


/**********************************************************************************
Description:
------------
	Gen 6 Definitions  
	- Unmapped Interrupt definitions for Gen 6
	
**********************************************************************************/
typedef enum
{
	UN_MAPPED_INT_SEQUENCER_RA_FIFO_BAND0				= 1,
	UN_MAPPED_INT_SEQUENCER_RA_FIFO_BAND1				= 2,
	UN_MAPPED_INT_BA_ANALYZER_RA_FIFO_BAND0				= 3,
	UN_MAPPED_INT_BA_ANALYZER_RA_FIFO_BAND1				= 4,
	UN_MAPPED_INT_TXH_PAUSER_BAND0						= 5,
	UN_MAPPED_INT_TXH_PAUSER_BAND1						= 6,
	UN_MAPPED_INT_RAB_INTER_CPU							= 7,
	UN_MAPPED_INT_HOST_MAILBOX_NOT_EMPTY				= 8,
	UN_MAPPED_INT_HOST_IF_RX_MNGMT_COUNT_NOT_EMPTY		= 9,
	UN_MAPPED_INT_HOST_IF_TX_MNGMT_COUNT_NOT_EMPTY		= 10,
	UN_MAPPED_INT_LOCKER_BAND0							= 11,
	UN_MAPPED_INT_LOCKER_BAND1							= 12,
	UN_MAPPED_INT_SYSTEM_IF_TIMERS_0					= 13,
	UN_MAPPED_INT_SYSTEM_IF_TIMERS_1					= 14,
	UN_MAPPED_INT_SYSTEM_IF_TIMERS_2					= 15,
	UN_MAPPED_INT_SYSTEM_IF_TIMERS_3					= 16,
	UN_MAPPED_INT_DMA_DONE								= 17,
	UN_MAPPED_INT_HW_ERRORS_BAND0						= 18,
	UN_MAPPED_INT_HW_ERRORS_BAND1						= 19,
	UN_MAPPED_INT_HW_ERRORS_COMMON						= 20,
	UN_MAPPED_INT_HW_EVENTS_BAND0_B						= 21,
	UN_MAPPED_INT_HW_EVENTS_BAND0_A						= 22,
	UN_MAPPED_INT_HW_EVENTS_BAND1_B						= 23,
	UN_MAPPED_INT_HW_EVENTS_BAND1_A						= 24,
	UN_MAPPED_INT_HW_EVENTS_COMMON_B					= 25,
	UN_MAPPED_INT_HW_EVENTS_COMMON_A					= 26,
    UN_MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND1	        = 27,
    UN_MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND0	        = 28,
    UN_MAPPED_INT_ULPR_REPORT_FIFO_NOT_EMPTY_BAND0		= 29,
	UN_MAPPED_INT_ULPR_REPORT_FIFO_NOT_EMPTY_BAND1		= 30,
    RESERVED_31	                                    	= 31,
}UnMappedIntId_e;														  

#if defined(ENET_INC_LMAC) 
#define EDGED_TRIGGERED_ACTIVE_INT_MAC_HT_MASK		((1 << EDGED_TRIGGERED_INT_DEL_TO_GENRISCTXENDIRQ)	| \
													 (1 << EDGED_TRIGGERED_INT_AUTO_FILL_END)			| \
													 (1 << EDGED_TRIGGERED_INT_PAC_TICK_TIMER)			| \
													 (1 << EDGED_TRIGGERED_INT_TX_ERROR )				| \
													 (1 << EDGED_TRIGGERED_INT_MAC_PHY_GP_IF_STATUS)	| \
													 (1 << EDGED_TRIGGERED_DEL_TIMER_EXPIRED))

#define EDGED_TRIGGERED_ACTIVE_INT_MAC_GENERAL_MASK	(0x0)

#endif /*ENET_INC_LMAC */
			
#if defined(ENET_INC_UMAC) 
#define EDGED_TRIGGERED_ACTIVE_INT_MAC_HT_MASK		(0x0)

#define EDGED_TRIGGERED_ACTIVE_INT_MAC_GENERAL_MASK	(1 << EDGED_TRIGGERED_INT_GPIO_10)

#endif /*#if defined(ENET_INC_UMAC) */



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define PIC_REG_OFFSET(unMappedInt)				(4*((unMappedInt-1)/PIC_MAPPER_NUM_OF_BITS))
#define PIC_REG_BIT_OFFSET(unMappedInt)			(5*((unMappedInt-1)%PIC_MAPPER_NUM_OF_BITS))
#define PIC_UNMAPPED_INT_VALUES(unMappedInt)	unMappedInt,PIC_REG_OFFSET(unMappedInt),PIC_REG_BIT_OFFSET(unMappedInt)

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef ISR_VOID (*IntHandler_t)(void);

typedef void (*IntFromOtherCpuHandler_t)(void);


typedef struct PicConfiguration
{
    uint8 		  mappedInterruptId;
    uint8         unMappedInterruptId;
    uint8         unMappedIntPicRegNum;
    uint8         unMappedIntPicRegOffset;
}PicConfiguration_t;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
ISR_VOID isr_InterruptManagerSpuriousEvent(void);
ISR_VOID isr_arcSim(void);
ISR_VOID vINT_fromOtherCore(void);

static void interruptManagerConfigureEdgedTriggeredInterrupt(void);
static void interruptManagerConfigureMacGeneralEdgedTriggeredInterrupt(void);



/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
#if defined(ENET_INC_LMAC)
#if defined(ENET_INC_LMAC0)

PicConfiguration_t interruptManagerPicConfiguration[MAPPED_INT_TOTAL_NUM] = 
{
	/* mappedInterruptId						unMappedInterruptId, unMappedIntPicRegNum, unMappedIntPicRegOffset*/	    
	{MAPPED_INT_HW_ERRORS_BAND0,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_ERRORS_BAND0)},
	{MAPPED_INT_HW_EVENTS_BAND0_A,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND0_A)},
	{MAPPED_INT_RAB_INTER_CPU,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_RAB_INTER_CPU)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_0, 			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_0)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_1,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_1)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_2,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_2)},	
	{MAPPED_INT_SYSTEM_IF_TIMERS_3,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_3)},	
	{MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY_BAND0, PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SEQUENCER_RA_FIFO_BAND0)},
	{MAPPED_INT_LOCKER_BAND0,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_LOCKER_BAND0)},
	{MAPPED_INT_BAA_RA_FIFO_NOT_EMPTY_BAND0, 	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_BA_ANALYZER_RA_FIFO_BAND0)},
	{MAPPED_INT_PAUSER_BAND0,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_TXH_PAUSER_BAND0)},  // Interrupt is in UM for gen5 and in LM for gen6
	{MAPPED_INT_ULPR_FIFO_NOT_EMPTY_BAND0,		PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_ULPR_REPORT_FIFO_NOT_EMPTY_BAND0)},
	{MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND0,	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND0)},
	{MAPPED_INT_HW_EVENTS_BAND0_B,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND0_B)},
};


// ARC interrupt handlers (ISRs) - LMAC0
IntHandler_t InterruptManagerIsrHandlersArc[NUM_OF_ARC_INTERRUPTS] =
{
	// ISR													number		offset in IVT 	(hex/dec)					
				
	System_CoreTimersIsrHandlerTimer0,	                    // 16		0x40    		64				Timer0				
	System_CoreTimersIsrHandlerTimer1,	                    // 17		0x44    		68				Timer1
	isr_HwErrors_Band0,					                	// 18		0x48    		72				MAPPED_INT_HW_ERRORS_BAND0
	isr_HwEvents_A_Band0,                              		// 19		0x4c    		76				MAPPED_INT_HW_EVENTS_BAND0_A
	isr_PerformanceCounters_CountLimitReached,				// 20		0x50    		80				Performance Counters
	vINT_fromOtherCore,										// 21		0x54   		84				MAPPED_INT_RAB_INTER_CPU
	isr_Timers_Timer0,										// 22		0x58			88				MAPPED_INT_SYSTEM_IF_TIMERS_0
	isr_Timer1,												// 23		0x5c			92				MAPPED_INT_SYSTEM_IF_TIMERS_1
	isr_Timer2,			           							// 24		0x60    		96				MAPPED_INT_SYSTEM_IF_TIMERS_2
	isr_Timer3, 											// 25		0x64    		100				MAPPED_INT_SYSTEM_IF_TIMERS_3
	isr_linkAdaptation_SequencerFifoNotEmpty,         		// 26		0x68    		104				MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY_BAND0
	isr_TxSelector_LockerRequestServiced_Band0,            	// 27		0x6c   		108				MAPPED_INT_LOCKER_BAND0
	isr_linkAdaptation_BaaFifoNotEmpty,              		// 28		0x70    		112 				MAPPED_INT_BAA_RA_FIFO_NOT_EMPTY_BAND0
	isr_TxPauser_PauseExecuted,								// 29		0x74    		116				MAPPED_INT_PAUSER_BAND0
	isr_PacManager_UlprFifoNotEmpty,	                	// 30		0x78    		120				MAPPED_INT_ULPR_FIFO_NOT_EMPTY_BAND0
	isr_PacManager_PreAggFifoNotEmpty,						// 31		0x7c			124				MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND0
	isr_HwEvents_B_Band0,					                // 32		0x80   		128					MAPPED_INT_HW_EVENTS_BAND0_B
	isr_InterruptManagerSpuriousEvent,		             	// 33		0x84   		132
	isr_InterruptManagerSpuriousEvent,	    	            // 34		0x88   		136
	isr_InterruptManagerSpuriousEvent,  	    	        // 35		0x8c    		140
	isr_InterruptManagerSpuriousEvent,			            // 36		0x90			144
	isr_InterruptManagerSpuriousEvent, 						// 37		0x94			148
	isr_InterruptManagerSpuriousEvent,                      // 38		0x98    		152
	isr_InterruptManagerSpuriousEvent,                      // 39		0x9c   		156
	isr_InterruptManagerSpuriousEvent,                      // 40		0xA0   		160
	isr_InterruptManagerSpuriousEvent,                      // 41		0xA4   		164
	isr_InterruptManagerSpuriousEvent,                      // 42		0xA8   		168
	isr_InterruptManagerSpuriousEvent,                      // 43		0xAc   		172
	isr_InterruptManagerSpuriousEvent,                      // 44		0xB0   		176
	isr_InterruptManagerSpuriousEvent,                      // 45		0xB4  		180
	isr_InterruptManagerSpuriousEvent,                      // 46		0xB8  		184
	isr_InterruptManagerSpuriousEvent,                      // 47		0xBc  		188
	isr_InterruptManagerSpuriousEvent,                      // 48		0xC0  		192
	isr_InterruptManagerSpuriousEvent,                      // 49		0xC4   		196
	isr_InterruptManagerSpuriousEvent,                      // 50		0xC8   		200
};
#endif //#if defined(ENET_INC_LMAC0)

#if defined(ENET_INC_LMAC1)

PicConfiguration_t interruptManagerPicConfiguration[MAPPED_INT_TOTAL_NUM] = 
{
	/* mappedInterruptId						unMappedInterruptId, unMappedIntPicRegNum, unMappedIntPicRegOffset*/	    
	{MAPPED_INT_HW_ERRORS_BAND1,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_ERRORS_BAND1)},
	{MAPPED_INT_HW_EVENTS_BAND1_A,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND1_A)},
	{MAPPED_INT_RAB_INTER_CPU,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_RAB_INTER_CPU)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_0, 			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_0)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_1,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_1)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_2,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_2)},	
	{MAPPED_INT_SYSTEM_IF_TIMERS_3,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_3)},	
	{MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY_BAND1, PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SEQUENCER_RA_FIFO_BAND1)},
	{MAPPED_INT_LOCKER_BAND1,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_LOCKER_BAND1)},
	{MAPPED_INT_BAA_RA_FIFO_NOT_EMPTY_BAND1, 	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_BA_ANALYZER_RA_FIFO_BAND1)},
	{MAPPED_INT_PAUSER_BAND1,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_TXH_PAUSER_BAND1)},  // Interrupt is in UM for gen5 and in LM for gen6
	{MAPPED_INT_ULPR_FIFO_NOT_EMPTY_BAND1,		PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_ULPR_REPORT_FIFO_NOT_EMPTY_BAND1)},
	{MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND1,	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND1)},
	{MAPPED_INT_HW_EVENTS_BAND1_B,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND1_B)},
};

// ARC interrupt handlers (ISRs) - LMAC1
IntHandler_t InterruptManagerIsrHandlersArc[NUM_OF_ARC_INTERRUPTS] =
{
	// ISR													number		offset in IVT 	(hex/dec)					
				
	System_CoreTimersIsrHandlerTimer0,	                    // 16		0x40    		64				Timer0				
	System_CoreTimersIsrHandlerTimer1,	                    // 17		0x44    		68				Timer1
	isr_HwErrors_Band1,					                	// 18		0x48    		72				MAPPED_INT_HW_ERRORS_BAND1
	isr_HwEvents_A_Band1,                              		// 19		0x4c    		76				MAPPED_INT_HW_EVENTS_BAND1_A
	isr_PerformanceCounters_CountLimitReached,				// 20		0x50    		80				Performance Counters
	vINT_fromOtherCore,										// 21		0x54   		84				MAPPED_INT_RAB_INTER_CPU
	isr_Timers_Timer0,										// 22		0x58			88				MAPPED_INT_SYSTEM_IF_TIMERS_0
	isr_Timer1,												// 23		0x5c			92				MAPPED_INT_SYSTEM_IF_TIMERS_1
	isr_Timer2,			           							// 24		0x60    		96				MAPPED_INT_SYSTEM_IF_TIMERS_2
	isr_Timer3, 											// 25		0x64    		100				MAPPED_INT_SYSTEM_IF_TIMERS_3
	isr_linkAdaptation_SequencerFifoNotEmpty,         		// 26		0x68    		104				MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY_BAND1
	isr_TxSelector_LockerRequestServiced_Band1,            	// 27		0x6c   		108				MAPPED_INT_LOCKER_BAND1
	isr_linkAdaptation_BaaFifoNotEmpty,              		// 28		0x70    		112 				MAPPED_INT_BAA_RA_FIFO_NOT_EMPTY_BAND1
	isr_TxPauser_PauseExecuted,								// 29		0x74    		116				MAPPED_INT_PAUSER_BAND1
	isr_PacManager_UlprFifoNotEmpty,	                	// 30		0x78    		120				MAPPED_INT_ULPR_FIFO_NOT_EMPTY_BAND1
	isr_PacManager_PreAggFifoNotEmpty,						// 31		0x7c			124				MAPPED_INT_PRE_AGG_FIFO_NOT_EMPTY_BAND1
	isr_HwEvents_B_Band1,					               	// 32		0x80   		128					MAPPED_INT_HW_EVENTS_BAND1_B
	isr_InterruptManagerSpuriousEvent,		             	// 33		0x84   		132
	isr_InterruptManagerSpuriousEvent,	    	            // 34		0x88   		136
	isr_InterruptManagerSpuriousEvent,  	    	        // 35		0x8c    		140
	isr_InterruptManagerSpuriousEvent,			            // 36		0x90			144
	isr_InterruptManagerSpuriousEvent, 						// 37		0x94			148
	isr_InterruptManagerSpuriousEvent,                      // 38		0x98    		152
	isr_InterruptManagerSpuriousEvent,                      // 39		0x9c   		156
	isr_InterruptManagerSpuriousEvent,                      // 40		0xA0   		160
	isr_InterruptManagerSpuriousEvent,                      // 41		0xA4   		164
	isr_InterruptManagerSpuriousEvent,                      // 42		0xA8   		168
	isr_InterruptManagerSpuriousEvent,                      // 43		0xAc   		172
	isr_InterruptManagerSpuriousEvent,                      // 44		0xB0   		176
	isr_InterruptManagerSpuriousEvent,                      // 45		0xB4  		180
	isr_InterruptManagerSpuriousEvent,                      // 46		0xB8  		184
	isr_InterruptManagerSpuriousEvent,                      // 47		0xBc  		188
	isr_InterruptManagerSpuriousEvent,                      // 48		0xC0  		192
	isr_InterruptManagerSpuriousEvent,                      // 49		0xC4   		196
	isr_InterruptManagerSpuriousEvent,                      // 50		0xC8   		200
};
#endif //#if defined(ENET_INC_LMAC1)
#endif /* (defined(ENET_INC_LMAC) */

#if defined(ENET_INC_UMAC)
const PicConfiguration_t interruptManagerPicConfiguration[MAPPED_INT_TOTAL_NUM] = 
{
	/* mappedInterruptId						unMappedInterruptId, unMappedIntPicRegNum, unMappedIntPicRegOffset*/	    
	{MAPPED_INT_HW_EVENTS_BAND0_A,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND0_A)},
	{MAPPED_INT_HW_EVENTS_BAND1_A,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND1_A)},
	{MAPPED_INT_HW_ERRORS_COMMON,	            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_ERRORS_COMMON)},
	{MAPPED_INT_HW_EVENTS_COMMON_A,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_COMMON_A)},
	{MAPPED_INT_DMA,					        PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_DMA_DONE)},
	{MAPPED_INT_RAB_INTER_CPU,			        PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_RAB_INTER_CPU)},
	{MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY,	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HOST_IF_TX_MNGMT_COUNT_NOT_EMPTY)},
	{MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY,	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HOST_IF_RX_MNGMT_COUNT_NOT_EMPTY)},
	{MAPPED_INT_LOCKER_BAND0,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_LOCKER_BAND0)},
	{MAPPED_INT_LOCKER_BAND1,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_LOCKER_BAND1)},
	{MAPPED_INT_HW_EVENTS_BAND0_B,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND0_B)},
	{MAPPED_INT_HW_EVENTS_BAND1_B,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS_BAND1_B)},
#ifdef ATOMIC_COUNTER_ENABLER	
	{MAPPED_INT_HOST_MAILBOX_NOT_EMPTY,			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HOST_MAILBOX_NOT_EMPTY)},
#endif
	{MAPPED_INT_SYSTEM_IF_TIMERS_0,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_0)},
	{MAPPED_INT_SYSTEM_IF_TIMERS_1,				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SYSTEM_IF_TIMERS_1)},		

	
};

// ARC interrupt handlers (ISRs) - UMAC
IntHandler_t InterruptManagerIsrHandlersArc[NUM_OF_ARC_INTERRUPTS] =
{
	// ISR													number		offset in IVT 	(hex/dec)					
				
	System_CoreTimersIsrHandlerTimer0,	                    // 16		0x40    	64				Timer0				
	System_CoreTimersIsrHandlerTimer1,	                    // 17		0x44    	68				Timer1
	isr_HwEvents_A_Band0,							       	// 18		0x48    	72				MAPPED_INT_HW_EVENTS_BAND0_A
	isr_HwEvents_A_Band1,	                               	// 19		0x4c    	76				MAPPED_INT_HW_EVENTS_BAND1_A
	isr_PerformanceCounters_CountLimitReached,//isr_arcSim	// 20		0x50    	80				Performance Counters
	isr_HwErrors_Common,							        // 21		0x54    	84				MAPPED_INT_HW_ERRORS_COMMON
	isr_HwEvents_Common_A,	                    			// 22		0x58   	88				MAPPED_INT_HW_EVENTS_COMMON_A
	isr_DmaDone,											// 23		0x5c		92				MAPPED_INT_DMA
	vINT_fromOtherCore,										// 24		0x60		96				MAPPED_INT_RAB_INTER_CPU
	isr_HostIfTxInMangCountNotZero,							// 25		0x64    	100				MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY
	isr_HostIfRxInMangCountNotZero, 		       			// 26		0x68    	104				MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY				
	isr_TxSelector_LockerRequestServiced_Band0,			   	// 27		0x6c    	108				MAPPED_INT_LOCKER_BAND0				
	isr_TxSelector_LockerRequestServiced_Band1,        		// 28		0x70    	112				MAPPED_INT_LOCKER_BAND1			
	isr_HwEvents_B_Band0,									// 29		0x74		116 				MAPPED_INT_HW_EVENTS_BAND0_B			
	isr_HwEvents_B_Band1,		        					// 30		0x78   	120				MAPPED_INT_HW_EVENTS_BAND1_B
#ifdef ATOMIC_COUNTER_ENABLER
	vHIM_HostISR,											// 31		0x7c   	124				MAPPED MAPPED_INT_MAILBOX_NOT_EMPTY
#else	
	isr_InterruptManagerSpuriousEvent,						// 31		0x7c   	124
#endif
	isr_Timer0,												// 32		0x80   	128				MAPPED_INT_SYSTEM_IF_TIMERS_0
	isr_Timer1,					   							// 33		0x84   	132				MAPPED_INT_SYSTEM_IF_TIMERS_1
	isr_InterruptManagerSpuriousEvent,          	  		// 34		0x88    	136
	isr_InterruptManagerSpuriousEvent,             			// 35		0x8c		140
	isr_InterruptManagerSpuriousEvent,				   		// 36		0x90    	144
	isr_InterruptManagerSpuriousEvent,						// 37		0x94    	144
	isr_InterruptManagerSpuriousEvent,             			// 38		0x98   	152
	isr_InterruptManagerSpuriousEvent,             			// 39		0x9c   	156
	isr_InterruptManagerSpuriousEvent,            			// 40		0xA0   	160
	isr_InterruptManagerSpuriousEvent,            			// 41		0xA4   	164
	isr_InterruptManagerSpuriousEvent,            			// 42		0xA8   	168
	isr_InterruptManagerSpuriousEvent,             			// 43		0xAc   	172
	isr_InterruptManagerSpuriousEvent,             			// 44		0xB0  	176
	isr_InterruptManagerSpuriousEvent,             			// 45		0xB4  	180
	isr_InterruptManagerSpuriousEvent,             			// 46		0xB8  	184
	isr_InterruptManagerSpuriousEvent,             			// 47		0xBc  	188
	isr_InterruptManagerSpuriousEvent,             			// 48		0xC0   	192
	isr_InterruptManagerSpuriousEvent,             			// 49		0xC4   	196
};

#endif /* defined(ENET_INC_UMAC) */ 

uint32 interruptManagerIntCntxtCounter = 0; /* Should be removed when new disable int method will be adopted (ThreadX)*/

/****************************************************************************************************/
/*					This Section  should be moved to Inter CPU Module when it will be merged					*/
/*****************************************************************************************************/
/* Second level callbacks from inter CPU interrupts */

#if defined(ENET_INC_LMAC0) 

const IntFromOtherCpuHandler_t apOtherCpuISR[OTHER_CPU_TABLE_SIZE] =
{
    vTLM_IPC_TO_LM_FROM_UM_ISR, 					/* IGEN_EVT_IPC_TO_LM0_FROM_UM		(1U << 0) 	*/
    vTLM_IPC_TO_LM0_FROM_LM1_ISR, 					/* IGEN_EVT_IPC_TO_LM0_FROM_LM1   	(1U << 1) 	*/
	vTLM_ASSERT_TO_LM0_FROM_UM_ISR,					/* IGEN_EVT_ASSERT_TO_LM0_FROM_UM	(1U << 2)	*/
    vTLM_ASSERT_TO_LM0_FROM_LM1_ISR,				/* IGEN_EVT_ASSERT_TO_LM0_FROM_LM1	(1U << 3) 	*/
	NULL,				/* 					       	(1U << 4) 	*/
    NULL,   			/* 				        	(1U << 5) 	*/
    NULL,               /* 					       	(1U << 6) 	*/
    NULL,               /* 					       	(1U << 7) 	*/
    NULL, 				/* 					      	(1U << 8) 	*/
    NULL,               /*                          (1U << 9) 	*/
    NULL,               /*                          (1U << 10) 	*/
    NULL,               /*                          (1U << 11) 	*/
    NULL,               /*                          (1U << 12) 	*/
    NULL,               /*                          (1U << 13) 	*/
    NULL,               /*                          (1U << 14) 	*/
    NULL,               /*                          (1U << 15) 	*/
    NULL,               /*                          (1U << 16) 	*/
    NULL,               /*                          (1U << 17) 	*/
    NULL,               /*                          (1U << 18) 	*/
    NULL,               /*                          (1U << 19) 	*/
	NULL,               /*                          (1U << 20) 	*/
	NULL,               /*                          (1U << 21) 	*/
	NULL,               /*                          (1U << 22) 	*/
	NULL,               /*                          (1U << 23) 	*/
};
#endif

#if defined(ENET_INC_LMAC1)
const IntFromOtherCpuHandler_t apOtherCpuISR[OTHER_CPU_TABLE_SIZE] =
{
    vTLM_IPC_TO_LM1_FROM_UM_ISR,					/* IGEN_EVT_IPC_TO_LM1_FROM_UM		(1U << 0) 	*/
    vTLM_IPC_TO_LM1_FROM_LM0_ISR,					/* IGEN_EVT_IPC_TO_LM1_FROM_LM0		(1U << 1) 	*/
    vTLM_ASSERT_TO_LM1_FROM_UM_ISR,					/* IGEN_EVT_ASSERT_TO_LM1_FROM_UM	(1U << 2)	*/
    vTLM_ASSERT_TO_LM1_FROM_LM0_ISR,				/* IGEN_EVT_ASSERT_TO_LM1_FROM_LM0	(1U << 3) 	*/
    NULL,				/* 					       	(1U << 4) 	*/
    NULL,   			/* 				        	(1U << 5) 	*/
    NULL,               /* 					       	(1U << 6) 	*/
    NULL,   			/* 				         	(1U << 7) 	*/
    NULL, 				/* 					      	(1U << 8) 	*/
    NULL,               /*                          (1U << 9) 	*/
    NULL,               /*                          (1U << 10) 	*/
    NULL,               /*                          (1U << 11) 	*/
    NULL,               /*                          (1U << 12) 	*/
    NULL,               /*                          (1U << 13) 	*/
    NULL,               /*                          (1U << 14) 	*/
    NULL,               /*                          (1U << 15) 	*/
    NULL,               /*                          (1U << 16) 	*/
    NULL,               /*                          (1U << 17) 	*/
    NULL,               /*                          (1U << 18) 	*/
    NULL,               /*                          (1U << 19) 	*/
    NULL,               /*                          (1U << 20) 	*/
    NULL,               /*                          (1U << 21) 	*/
    NULL,               /*                          (1U << 22) 	*/
    NULL,               /*                          (1U << 23) 	*/
};
#endif

#if defined(ENET_INC_UMAC)

const IntFromOtherCpuHandler_t apOtherCpuISR[OTHER_CPU_TABLE_SIZE] =
{

    vLMI_ISR_IPC_TO_UM_FROM_LM0,				/* IGEN_EVT_IPC_TO_UM_FROM_LM0		(1U << 0) 	*/
    vLMI_ISR_IPC_TO_UM_FROM_LM1,				/* IGEN_EVT_IPC_TO_UM_FROM_LM1		(1U << 1) 	*/
    vTLM_ASSERT_TO_UM_FROM_LM0_ISR,				/* IGEN_EVT_ASSERT_TO_UM_FROM_LM0	(1U << 2) 	*/
    vTLM_ASSERT_TO_UM_FROM_LM1_ISR,				/* IGEN_EVT_ASSERT_TO_UM_FROM_LM1	(1U << 3) 	*/ 
	NULL,               /*                         	(1U << 4) 	*/
    NULL,               /*                         	(1U << 5) 	*/
    NULL,               /*                         	(1U << 6) 	*/
    NULL,               /*                         	(1U << 7) 	*/
    NULL,               /*                         	(1U << 8) 	*/
    NULL,				/*							(1U << 9) 	*/
	NULL, 				/*							(1U << 10)	*/ 
    NULL, 				/* 					        (1U << 11) 	*/
	NULL, 				/* 							(1U << 12) 	*/
	NULL,				/* 							(1U << 13) 	*/
	NULL,				/* 							(1U << 14) 	*/
    NULL,				/* 							(1U << 15) 	*/
    NULL,				/*           				(1U << 16) 	*/
	NULL,				/*					       	(1U << 17) 	*/
    NULL, 				/* 							(1U << 18) 	*/
    NULL,  				/* 							(1U << 19) 	*/
    NULL,  				/* 							(1U << 20) 	*/
    NULL,  				/* 							(1U << 21) 	*/
	NULL,     			/*							(1U << 22) 	*/
	NULL,   			/*							(1U << 23) 	*/
};
#endif


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/
#if defined (ANALYZE_DISABLE_INTERRUPT)
		
uint32 AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE][7] = {0x0};
uint16 AAA_DebugTrace_analyzeDisableInterrupts_idx = 0;
		

#endif

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
#ifdef ENET_INC_UMAC

extern uint32 trigger_int_g;

/**********************************************************************************

isr_arcSim 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
extern void UmDbgCliReq(K_MSG *kMsg_p);
ISR_VOID isr_arcSim(void)
{ 
	K_MSG *triggerUmDbgCliReq_p = PNULL;
	UmiDbgCliReq_t *umiDbgCliReq_p = PNULL;

	ASSERT(0);
	
	triggerUmDbgCliReq_p = OSAL_GET_MESSAGE(sizeof(uint32));	
	umiDbgCliReq_p = (UmiDbgCliReq_t *)pK_MSG_DATA(triggerUmDbgCliReq_p);
	umiDbgCliReq_p->action = 31;
	umiDbgCliReq_p->data1 = 1;
	umiDbgCliReq_p->data2 = trigger_int_g;
	trigger_int_g = 0;
	_sr(0, AUX_IRQ_HINT);
	UmDbgCliReq(triggerUmDbgCliReq_p);
}
#endif

/**********************************************************************************

isr_InterruptManagerSpuriousEvent 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
ISR_VOID isr_InterruptManagerSpuriousEvent(void)
{ 
	FATAL("Spurious Interrupt"); 
}

/**********************************************************************************

vINT_fromOtherCore 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
ISR_VOID vINT_fromOtherCore()
{
	uint32 u32Index;

#if defined(ENET_INC_UMAC)
	RegMacGeneralUpiInterrupt_u 	 upiInterruptReg;
	RegMacGeneralUpiInterruptClear_u upiInterruptClearReg;
#endif
#if defined(ENET_INC_LMAC0)
	RegMacGeneralLpiInterrupt_u 	 lpiInterruptReg;
	RegMacGeneralLpiInterruptClear_u lpiInterruptClearReg;
#endif
#if defined(ENET_INC_LMAC1)
	RegMacGeneralLpi1Interrupt_u 	  lpi1InterruptReg;
	RegMacGeneralLpi1InterruptClear_u lpi1InterruptClearReg;
#endif

	ACCUMULATE_CPU_IDLE_TIME();
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vINT_OtherCore,0,0);

#if defined(ENET_INC_UMAC)
	RegAccess_Read(REG_MAC_GENERAL_UPI_INTERRUPT, &upiInterruptReg.val);
	//make sure an event exists
	ASSERT(upiInterruptReg.val);
	u32Index = Utils_CountTrailingZeros(upiInterruptReg.val);
	upiInterruptClearReg.bitFields.upiInterruptClear = 1 << u32Index;
	RegAccess_Write(REG_MAC_GENERAL_UPI_INTERRUPT_CLEAR, upiInterruptClearReg.val);
#endif

#if defined(ENET_INC_LMAC0)	
	RegAccess_Read(REG_MAC_GENERAL_LPI_INTERRUPT, &lpiInterruptReg.val);
	//make sure an event exists
	ASSERT(lpiInterruptReg.val);
	u32Index = Utils_CountTrailingZeros(lpiInterruptReg.val);
	lpiInterruptClearReg.bitFields.lpiInterruptClear = 1 << u32Index;
	RegAccess_Write(REG_MAC_GENERAL_LPI_INTERRUPT_CLEAR, lpiInterruptClearReg.val);
#endif

#if defined(ENET_INC_LMAC1)	
	RegAccess_Read(REG_MAC_GENERAL_LPI1_INTERRUPT, &lpi1InterruptReg.val);
	//make sure an event exists
	ASSERT(lpi1InterruptReg.val);
	u32Index = Utils_CountTrailingZeros(lpi1InterruptReg.val);
	lpi1InterruptClearReg.bitFields.lpi1InterruptClear = 1 << u32Index;
	RegAccess_Write(REG_MAC_GENERAL_LPI1_INTERRUPT_CLEAR, lpi1InterruptClearReg.val);
#endif	

	ASSERT(apOtherCpuISR[u32Index] != NULL);
	apOtherCpuISR[u32Index]();
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vINT_OtherCore,1,1);	
}

/**********************************************************************************

interruptManagerConfigureEdgedTriggeredInterrupt 

Description:
------------
	This function is called to configure interrupts in the unit that was added to convert those interrupts 
	that were originally edge triggered to be level triggered

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
static void interruptManagerConfigureEdgedTriggeredInterrupt(void)
{
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_WL_MAC_IRQ_CLR,   EDGED_TRIGGERED_ACTIVE_INT_MAC_HT_MASK);
}


/**********************************************************************************

interruptManagerConfigureMacGeneralEdgedTriggeredInterrupt 

Description:
------------
	This function is called to configure interrupts in the unit that was added to convert those interrupts 
	that were originally edge triggered to be level triggered

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
static void interruptManagerConfigureMacGeneralEdgedTriggeredInterrupt(void)
{
	RegAccess_Write(REG_MAC_GENERAL_WL_MAC_GENERAL_IRQ_CLR, EDGED_TRIGGERED_ACTIVE_INT_MAC_GENERAL_MASK);
}


    
/***********************************************************************
* InterruptManagerConfigureMacGeneraExternalIrqInvert
* 
* Description:
* ------------
* Set the polarity of the signal triggering the external interrupt (interrupt is triggered when the signal changes from 0->1 or from 1->0).
* 0 - interrupt is triggered when signal changes from LOW to HIGH
* 1 - interrupt is triggered when signal changes from HIGH to LOW
*
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void InterruptManagerConfigureMacGeneraExternalIrqInvert(EdgedTriggeredIntMacGeneralId_e intId, uint8 polarity)
{
	// old RegAccess_Write(REG_MAC_GENERAL_EXTERNAL_IRQ_INVERT, (polarity << intId));

	RegMacGeneralExternalIrqInvert_u regMacGeneralExternalIrqInvert;


	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_EXTERNAL_IRQ_INVERT, &regMacGeneralExternalIrqInvert.val);
	regMacGeneralExternalIrqInvert.bitFields.externalIrqInvert = (polarity == 0) ?	(regMacGeneralExternalIrqInvert.bitFields.externalIrqInvert & ~(1 << intId)):
																					(regMacGeneralExternalIrqInvert.bitFields.externalIrqInvert |  (1 << intId));

	RegAccess_Write(REG_MAC_GENERAL_EXTERNAL_IRQ_INVERT, regMacGeneralExternalIrqInvert.val);	
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

InterruptManager_RabInit 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/	
void InterruptManager_RabInit(void)
{
	RegMacGeneralNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
	RegMacGeneralUpiInterruptClear_u 	 upiInterruptClearReg;
	RegMacGeneralLpiInterruptClear_u 	 lpiInterruptClearReg;
	RegMacGeneralLpi1InterruptClear_u 	 lpi1InterruptClearReg;

	//Clear interrupts from MAC IP that might be pending
	
	upiInterruptClearReg.bitFields.upiInterruptClear = 0xFFFFFF;  // RAB/ucpu clear reg
	RegAccess_Write(REG_MAC_GENERAL_UPI_INTERRUPT_CLEAR, upiInterruptClearReg.val);

	lpiInterruptClearReg.bitFields.lpiInterruptClear = 0xFFFFFF;  // RAB/lcpu clear reg
	RegAccess_Write(REG_MAC_GENERAL_LPI_INTERRUPT_CLEAR, lpiInterruptClearReg.val);

	lpi1InterruptClearReg.bitFields.lpi1InterruptClear = 0xFFFFFF;	// RAB/lcpu1 clear reg 
	RegAccess_Write(REG_MAC_GENERAL_LPI1_INTERRUPT_CLEAR, lpi1InterruptClearReg.val);

	npu2UpiInterruptClearReg.bitFields.npu2UpiInterruptClear = 0xFFFFFF;	// RAB/npu clear reg
	RegAccess_Write(REG_MAC_GENERAL_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);
}

/**********************************************************************************

InterruptManager_DisableAllInterrupts 

Description:
------------
	Disable the interrupts of the MIPS, Used for shared variables protection.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
#if defined (ANALYZE_DISABLE_INTERRUPT)
void InterruptManager_DisableAllInterrupts(unsigned int* interrupt_save, char* fileName,unsigned long int line)
#else
void InterruptManager_DisableAllInterrupts(unsigned int* interrupt_save)
#endif
{
	uint32 statusReg;

	//check if we are using shadow set registers. If so - we are in the middle of interrupt. We don't need to disable interrupts (it's already disabled)
	statusReg = _lr(STATUS32);
	if (statusReg & OS_STATUS_REG_RB_BITS)
	{
		return;
	}

	*interrupt_save = _clri();

	// Increment global interrupt counter. it's ok to do it after _clri since if 
	// we are already with disabled interrupt it will remain in disable interrupts.
	// We want to a avoid - "disable->disable->enable->enable" which will enable on the first one.
	interruptManagerIntCntxtCounter++; 
}

/**********************************************************************************

InterruptManager_EnableAllInterrupts 

Description:
------------
	Enable the interrupts of the MIPS

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
#if defined(ANALYZE_DISABLE_INTERRUPT)
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save, char* fileName, unsigned long int line)
#else
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save)
#endif
{
	uint32 statusReg;

	//check if we are using shadow set registers. If so - we are in the middle of interrupt. We must not enable interrupts.
	statusReg = _lr(STATUS32);
	if (statusReg & OS_STATUS_REG_RB_BITS)
	{
		return;
	}

	ASSERT(interruptManagerIntCntxtCounter != 0);
	//	decrement global interrupt counter
	interruptManagerIntCntxtCounter--; 

	// in case of nested disable/enable, enable the interrupts only on the
	// "last enable" that goes with the "first disable" 
	if (interruptManagerIntCntxtCounter == 0)
	{
		_seti(interrupt_save);
	}
}

/**********************************************************************************

InterruptManager_IsInInterruptContext 

Description:
------------
	

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
bool InterruptManager_IsInInterruptContext(void)
{
	uint32 u32StatusReg;
	u32StatusReg = _lr(STATUS32);

	/*	check if we are using shadow set registers */
	if (u32StatusReg & OS_STATUS_REG_RB_BITS)
	{
		return TRUE;
	}
	return FALSE;
}

/**********************************************************************************

InterruptManager_MaskInterrupt 

Description:
------------
	Mask(Disable) specific source in the external interrupt controller

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
void InterruptManager_MaskInterrupt(MappedIntId_e mappedIntId)
{
	RegAccess_WriteMasked(REG_EMERALD_ENV_IRQ_MASK, (1 << mappedIntId), 0);
}

/**********************************************************************************

InterruptManger_UnMaskInterrupt 

Description:
------------
	Enable specific source in interrupt controller

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_UnMaskInterrupt(MappedIntId_e mappedIntId)
{
	RegAccess_WriteMasked(REG_EMERALD_ENV_IRQ_MASK, (1 << mappedIntId),(1 << mappedIntId));
}

/**********************************************************************************

InterruptManager_EnableUpiRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_EnableUpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnableUpiInterrupt_u enableUpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_UPI_INTERRUPT, &enableUpiInterruptReg.val);
	enableUpiInterruptReg.bitFields.enableUpiInterrupt |= (1 << intIndex);

	RegAccess_Write(REG_MAC_GENERAL_ENABLE_UPI_INTERRUPT, enableUpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_DisableUpiRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_DisableUpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnableUpiInterrupt_u enableUpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_UPI_INTERRUPT, &enableUpiInterruptReg.val);
	enableUpiInterruptReg.bitFields.enableUpiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_GENERAL_ENABLE_UPI_INTERRUPT, enableUpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_EnableLpiRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_EnableLpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnableLpiInterrupt_u enableLpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_LPI_INTERRUPT, &enableLpiInterruptReg.val);
	enableLpiInterruptReg.bitFields.enableLpiInterrupt |= (1 << intIndex);

	RegAccess_Write(REG_MAC_GENERAL_ENABLE_LPI_INTERRUPT, enableLpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_DisableLpiRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_DisableLpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnableLpiInterrupt_u enableLpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_LPI_INTERRUPT, &enableLpiInterruptReg.val);
	enableLpiInterruptReg.bitFields.enableLpiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_GENERAL_ENABLE_LPI_INTERRUPT, enableLpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_EnablePhiRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_EnablePhiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnablePhiInterrupt_u enablePhiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_PHI_INTERRUPT, &enablePhiInterruptReg.val);
	enablePhiInterruptReg.bitFields.enablePhiInterrupt |= (1 << intIndex);
	RegAccess_Write(REG_MAC_GENERAL_ENABLE_PHI_INTERRUPT, enablePhiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_DisablePhiRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_DisablePhiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnablePhiInterrupt_u enablePhiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_PHI_INTERRUPT, &enablePhiInterruptReg.val);
	enablePhiInterruptReg.bitFields.enablePhiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_GENERAL_ENABLE_PHI_INTERRUPT, enablePhiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_EnableNpuRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_EnableNpuRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnableNpu2UpiInterrupt_u enableNpu2UpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_NPU2UPI_INTERRUPT, &enableNpu2UpiInterruptReg.val);
	enableNpu2UpiInterruptReg.bitFields.enableNpu2UpiInterrupt |= (1 << intIndex);
	RegAccess_Write(REG_MAC_GENERAL_ENABLE_NPU2UPI_INTERRUPT, enableNpu2UpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_DisableNpuRabInt 

Description:
------------

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_DisableNpuRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacGeneralEnableNpu2UpiInterrupt_u enableNpu2UpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_GENERAL_ENABLE_NPU2UPI_INTERRUPT, &enableNpu2UpiInterruptReg.val);
	enableNpu2UpiInterruptReg.bitFields.enableNpu2UpiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_GENERAL_ENABLE_NPU2UPI_INTERRUPT, enableNpu2UpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************

InterruptManager_ClearEdgedTriggeredInterrupt 

Description:
------------
	Clear/disable interrupts in the unit that was added to convert 
	those interrupts that were originally edge triggered to be level triggered
Input:
-----

		
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void InterruptManager_ClearEdgedTriggeredInterrupt(EdgedTriggeredIntId_e intId)
{
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_WL_MAC_IRQ_CLR,1 << intId);
}

/**********************************************************************************

InterruptManager_ClearMacGeneralEdgedTriggeredInterrupt 

Description:
------------
	Clear/disable interrupts in the unit that was added to convert 
	those interrupts that were originally edge triggered to be level triggered
Input:
-----

		
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void InterruptManager_ClearMacGeneralEdgedTriggeredInterrupt(EdgedTriggeredIntMacGeneralId_e intId)
{
	RegAccess_Write(REG_MAC_GENERAL_WL_MAC_GENERAL_IRQ_CLR, 1 << intId);
}

/**********************************************************************************

InterruptManager_Configure 

Description:
------------
	This function should be called once during system initialization to initialise the Interrupt Controller Driver.
	It should be called before the operating system is initialised.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void InterruptManager_Configure(void)
{
	uint32 	picConfTableIndex;
	uint32 	regAddr;
	uint32	regValue;
	uint32	regMask;
	MappedIntId_e mappedInterruptId;

#if defined(ENET_INC_LMAC)
	TX_INTERRUPT_SAVE_AREA;
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#endif 	
	/*Remap all relevant interrupts to priority*/	
	for (picConfTableIndex = 0; picConfTableIndex < MAPPED_INT_TOTAL_NUM; picConfTableIndex++) 
	{
		mappedInterruptId = (MappedIntId_e)interruptManagerPicConfiguration[picConfTableIndex].mappedInterruptId;

		/* Configure PIC Registers*/
		regAddr = REG_EMERALD_ENV_IRQ_MAP_00 + interruptManagerPicConfiguration[picConfTableIndex].unMappedIntPicRegNum;
		regValue = interruptManagerPicConfiguration[picConfTableIndex].mappedInterruptId << interruptManagerPicConfiguration[picConfTableIndex].unMappedIntPicRegOffset;		
		regMask =  0x1F << interruptManagerPicConfiguration[picConfTableIndex].unMappedIntPicRegOffset;
		RegAccess_WriteMasked(regAddr,regMask,regValue);
		
		InterruptManager_UnMaskInterrupt(mappedInterruptId);
	}

#if defined (ENET_INC_HW_FPGA) && defined (ENET_INC_LMAC1)
	if(ConfigurationManager_GetBandConfigurationMode() != CONFIGURATION_MODE_SINGLE_BAND)
#endif
	{
		interruptManagerConfigureEdgedTriggeredInterrupt();
		interruptManagerConfigureMacGeneralEdgedTriggeredInterrupt();
	}
	HwEvents_Init();
	HwErrors_Init();
	/*Set All interrupt to work with second register file*/

#if defined(ENET_INC_LMAC)
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	/*All interrupts are disabled, we've finished mapping them so now enable all Int's*/
	_seti(0xF); // TBD - do we need priority 15 or something else here? We could have saved it when we disabled interrupts (but we didn't)
#endif 	
}

/**********************************************************************************

InterruptManager_Reset 

Description:
------------
	This function should be called once during system initialization to initialise the Interrupt Controller Driver.
	It should be called before the operating system is initialised.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void InterruptManager_Reset(void)
{
	uint32 regAddr;
	RegAccess_Write(REG_EMERALD_ENV_IRQ_MASK,CLEAR_ALL_INTERRUPTS);
		
	/* Remap all interrupts to an unused line - then open only those that are used */ 
	for(regAddr = REG_EMERALD_ENV_IRQ_MAP_00; regAddr <= REG_EMERALD_ENV_IRQ_MAP_05; regAddr += 4)
	{
		RegAccess_Write(regAddr,0);
	}
}

/***************************************************************************
**
** NAME         vIGEN_SendToUm
**
** PARAMETERS   u32Source      The interrupt source to generate.
**
** RETURNS      na
**
** DESCRIPTION  This function generates an interrupt to the Upper MAC.
**
****************************************************************************/
void vIGEN_SendToUm(uint32 u32Source)
{
	RegMacGeneralUpiInterruptSet_u upiInterruptSetReg;

	upiInterruptSetReg.bitFields.upiInterruptSet = u32Source; 
	RegAccess_Write(REG_MAC_GENERAL_UPI_INTERRUPT_SET, upiInterruptSetReg.val);
}

/***************************************************************************
**
** NAME         vIGEN_SendToLm0
**
** PARAMETERS   u32Source      The interrupt source to generate.
**
** RETURNS      na
**
** DESCRIPTION  This function generates an interrupt to the Lower MAC.
**
****************************************************************************/
void vIGEN_SendToLm0(uint32 u32Source)
{
	RegMacGeneralLpiInterruptSet_u lpiInterruptSetReg;
	lpiInterruptSetReg.bitFields.lpiInterruptSet = u32Source; 
	RegAccess_Write(REG_MAC_GENERAL_LPI_INTERRUPT_SET, lpiInterruptSetReg.val);
}

/***************************************************************************
**
** NAME         vIGEN_SendToLm1
**
** PARAMETERS   u32Source      The interrupt source to generate.
**
** RETURNS      na
**
** DESCRIPTION  This function generates an interrupt to the Lower MAC.
**
****************************************************************************/
void vIGEN_SendToLm1(uint32 u32Source)
{
	RegMacGeneralLpi1InterruptSet_u lpi1InterruptSetReg;
	lpi1InterruptSetReg.bitFields.lpi1InterruptSet = u32Source;	
	RegAccess_Write(REG_MAC_GENERAL_LPI1_INTERRUPT_SET, lpi1InterruptSetReg.val);
}

