/***********************************************************************************
 File:			InterruptManager.c			
 Module:			Interrupt Module		
 Purpose:			
 Description:		
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_Timers.h"
#include "MT_osutl.h"

#include "InterruptManager_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "ErrorHandler_Api.h"
#include "int_gen.h"
#include "encryptor.h"
#include "SenderInterface_Api.h"
#include "HostGenRiscInterface_Api.h"
#if defined (ENET_INC_UMAC)
#include "RxHandler_Api.h"
#endif
#include "AggregationBuilder_Api.h"
#include "Utils_Api.h"
#include "Pac_Api.h"
#include "RegAccess_Api.h"
#include "MacHtExtensionsRegs.h"
#include "EmeraldEnvRegs.h"
#include "Debug_Trace.h"
#include "CpuLoad_Api.h"
#include "shram_interruptSection.h"
#include "TxSelector_Api.h"
#include "MacRabRegs.h"
#if defined (ENET_INC_UMAC)
#include "HostInterface_API.h"
#include "TxPacketsClassifier_API.h"
#include "TxManager_Api.h"
#include "lm_interface.h"
#include "tx_port.h"
#include "BaAnalyzer_Api.h"
#endif

#include "RxManager_Api.h"
//#include "DmaManager_Api.h"
#include "linkAdaptation_api.h"
#if defined (ENET_INC_LMAC)
#include "lmtimerUtility.h"
#endif

#include "linkAdaptation_api.h"
#include "TxSequencer_Api.h"
#include "loggerAPI.h"
#include "DmaManager_Api.h"
#include "TpcClbrHndlr.h"
#include "lmtimerUtility.h"
#include "Pauser_Api.h"
#include "ipc_api.h"



/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_SYS
#define LOG_LOCAL_FID 0

#define PIC_MAPPER_NUM_OF_BITS		(6)
#define CLEAR_ALL_INTERRUPTS		(0)

#define OS_STATUS_REG_EXL_BIT			(0x02)/* M_StatusEXL */



/**********************************************************************************
Description:
------------
	Gen 5 Definitions  
	- Unmapped Interrupt definitions for Gen 5
	
**********************************************************************************/
typedef enum
{
	UN_MAPPED_INT_DMA = 1,
	UN_MAPPED_INT_TX_DONE_LISTS,
	UN_MAPPED_INT_RX_DONE_LISTS,
	UN_MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY,
	UN_MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY,
	UN_MAPPED_INT_HIGH_PRI_TX_PD_READY,
	UN_MAPPED_INT_HIGH_PRI_RX_PD_READY,
	UN_MAPPED_INT_DISABLE_LISTS,
	UN_MAPPED_INT_GENRISC_RX_HANDLER,
	UN_MAPPED_INT_TSF_TIMER,
	UN_MAPPED_INT_PROTECTED_DB,
	UN_MAPPED_INT_SHRAM_ADDR,
	UN_MAPPED_INT_AGGREGATION_BUILDER,
	UN_MAPPED_INT_GENRISC_SENDER,    
	UN_MAPPED_INT_TX_SEQUENCER,
	UN_MAPPED_INT_PAC_TICK_TIMER,
	UN_MAPPED_INT_BA_ANALYZER,
	UN_MAPPED_INT_DEL_2_GENRISC_TX_END,
	UN_MAPPED_INT_INTER_CPU,
	UN_MAPPED_INT_TXH_PAUSER,
	UN_MAPPED_INT_BA_ANALYZER_RA_FIFO,
	UN_MAPPED_INT_HW_ERRORS,
	UN_MAPPED_INT_HW_EVENTS,
	UN_MAPPED_INT_GP_TIMER_A,
	UN_MAPPED_INT_GP_TIMER_B,
	UN_MAPPED_INT_LOCKER,
	UN_MAPPED_INT_HOST_MAILBOX_NOT_EMPTY,
	UN_MAPPED_INT_SEQUENCER_RA_FIFO,
	UN_MAPPED_INT_GENRISC_HOST_IF,
	UN_MAPPED_INT_SEQUENCER_RA_FILLER,
	UN_MAPPED_INT_DEL_2_GENRISC_TX_START,
}UnMappedIntId_e;



#if defined(ENET_INC_LMAC) 

#define EDGED_TRIGGERED_ACTIVE_INT_MASK		((1 << EDGED_TRIGGERED_INT_BLOCK_ACK_ERROR)				| \
											 (1 << EDGED_TRIGGERED_INT_AUTO_FILL_COUNTER_LIMIT)		| \
											 (1 << EDGED_TRIGGERED_INT_DELINEATOR_FIFO_FULL)		| \
											 (1 << EDGED_TRIGGERED_INT_TX_ERROR)					| \
											 (1 << EDGED_TRIGGERED_INT_RX_AGGR_ERROR)				| \
											 (1 << EDGED_TRIGGERED_INT_RX_EOF_ERROR)				| \
											 (1 << EDGED_TRIGGERED_INT_AUTO_FILL_END)				| \
											 (1 << EDGED_TRIGGERED_INT_DEL_1_AUTO_FILL_CTRLIMITIRQ) | \
											 (1 << EDGED_TRIGGERED_INT_DEL_2_AUTO_FILL_CTRLIMITIRQ) | \
											 (1 << EDGED_TRIGGERED_INT_DEL_3_AUTO_FILL_CTRLIMITIRQ))
#endif /*ENET_INC_LMAC */
			
#if defined(ENET_INC_UMAC) 
#define EDGED_TRIGGERED_ACTIVE_INT_MASK		((1 << EDGED_TRIGGERED_INT_DEL_RX_HALT) | ((1 << EDGED_TRIGGERED_INT_PAC_TICK_TIMER) ))
#endif /*#if defined(ENET_INC_UMAC) */



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define PIC_REG_OFFSET(unMappedInt)				(4*((unMappedInt-1)/PIC_MAPPER_NUM_OF_BITS))
#define PIC_REG_BIT_OFFSET(unMappedInt)			(5*((unMappedInt-1)%PIC_MAPPER_NUM_OF_BITS))
#define PIC_UNMAPPED_INT_VALUES(unMappedInt)	unMappedInt,PIC_REG_OFFSET(unMappedInt),PIC_REG_BIT_OFFSET(unMappedInt)

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef ISR_VOID (*IntHandler_t)(void);

typedef void (*IntFromOtherCpuHandler_t)(void);


typedef struct PicConfiguration
{
    IntHandler_t  isrHandler;
    uint8 		  mappedInterruptId;
    uint8         unMappedInterruptId;
    uint8         unMappedIntPicRegNum;
    uint8         unMappedIntPicRegOffset;
}PicConfiguration_t;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
ISR_VOID isr_InterruptManagerSpuriousEvent(void);
ISR_VOID vINT_fromOtherCore(void);

static void interruptManagerConfigureEdgedTriggeredInterrupt(void);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
IntHandler_t InterruptManagerIsrHandlers[MAPPED_INT_TOTAL_NUM+1];

#if defined(ENET_INC_LMAC)
PicConfiguration_t interruptManagerPicConfiguration[MAPPED_INT_TOTAL_NUM] = 
{
	/*		ISR									mappedInterruptId				unMappedInterruptId, unMappedIntPicRegNum, unMappedIntPicRegOffset*/	    
	{isr_HwErrors,							MAPPED_INT_HW_ERRORS,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_ERRORS)},
	{isr_HwEvents,							MAPPED_INT_HW_EVENTS,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS)},
	{vINT_fromOtherCore,					MAPPED_INT_INTER_CPU,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_INTER_CPU)},
	{vLM_pac_tsf_timer_int,					MAPPED_INT_TSF_TIMER,   				PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_TSF_TIMER)},
	{MT_vINTTimer1_PromptScheduler,			MAPPED_INT_GP_TIMERS_A,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_GP_TIMER_A)},
	{GpTimerB_ISR,							MAPPED_INT_GP_TIMERS_B,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_GP_TIMER_B)},
	{isr_AggregationBuilder_ErrorOrDone,	MAPPED_INT_AGGREGATION_BUILDER,		 	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_AGGREGATION_BUILDER)},
	{isr_linkAdaptation_SequencerFifoNotEmpty,MAPPED_INT_SEQUENCER_FIFO_NOT_EMPTY, 	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SEQUENCER_RA_FIFO)},
	{isr_TxSelector_LockerRequestServiced,	MAPPED_INT_LOCKER, 						PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_LOCKER)},
#if defined(CALCULATE_CPU_IDLE_TIME) 
	{LmvKNLintrTimer,						MAPPED_INT_PAC_TICK_TIMER,		        PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_PAC_TICK_TIMER)},
#endif		
	{isr_linkAdaptation_BaaFifoNotEmpty,	MAPPED_INT_BAA_FIFO_NOT_EMPTY, 			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_BA_ANALYZER_RA_FIFO)},
};
#endif /* (defined(ENET_INC_LMAC) */

#if defined(ENET_INC_UMAC)
const PicConfiguration_t interruptManagerPicConfiguration[MAPPED_INT_TOTAL_NUM] = 
{
	{isr_HwErrors,							MAPPED_INT_HW_ERRORS,			            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_ERRORS)},
	{isr_HwEvents,							MAPPED_INT_HW_EVENTS,		 	            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HW_EVENTS)},
	{vKNLintrTimer,							MAPPED_INT_PAC_TICK_TIMER,		            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_PAC_TICK_TIMER)},
	{isr_TxManager_TransmissionComplete,	MAPPED_INT_TX_DONE_LISTS,		            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_TX_DONE_LISTS)},
	{isr_DmaDone, 							MAPPED_INT_DMA,					            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_DMA)},
	{vINT_fromOtherCore, 					MAPPED_INT_INTER_CPU,			            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_INTER_CPU)},
	{isr_Timers_TimerA, 					MAPPED_INT_GP_TIMERS_A,			            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_GP_TIMER_A)},
	{isr_TxPacketsClassifier_ReadyPds, 		MAPPED_INT_TX_READY_PD_LIST,	            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HIGH_PRI_TX_PD_READY)},
	{isr_RxManager_RdReady, 				MAPPED_INT_RX_RD_LIST_NOT_EMPTY,            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HIGH_PRI_RX_PD_READY)},
	{isr_HostIfTxInMangCountNotZero,		MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY,	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_TX_IN_MNGMNT_COUNT_NOT_EMPTY)},
	{isr_HostIfRxInMangCountNotZero,		MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY,	PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_RX_IN_MNGMNT_COUNT_NOT_EMPTY)},
	{SharedRamInt_ISR,						MAPPED_INT_SHRAM_ADDR_INT,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_SHRAM_ADDR)},
	{isr_RxHandler,							MAPPED_INT_GENRISC_RX_HANDLER, 	  			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_GENRISC_RX_HANDLER)},
	{isr_HostGenRisc,						MAPPED_INT_GENRISC_HOST_IF,		            PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_GENRISC_HOST_IF)},
	{isr_BaAnalyzer_UmacStatusFifoNotEmpty,	MAPPED_INT_STATUS_FIFO,		                PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_BA_ANALYZER)},
	{isr_TxSelector_LockerRequestServiced,	MAPPED_INT_LOCKER,							PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_LOCKER)},
	{isr_TxPauser_PauseExecuted,			MAPPED_INT_PAUSER,							PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_TXH_PAUSER)},
	{isr_TxManager_DisableListNotEmpty,	    MAPPED_INT_DISABLE_LIST_NOT_EMPTY,			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_DISABLE_LISTS)},
	{isr_Sender,	    					MAPPED_INT_GENRISC_SENDER,					PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_GENRISC_SENDER)},
#ifdef ATOMIC_COUNTER_ENABLER
	{vHIM_HostISR,					   		MAPPED_INT_HOST_MAILBOX_NOT_EMPTY,			PIC_UNMAPPED_INT_VALUES(UN_MAPPED_INT_HOST_MAILBOX_NOT_EMPTY)},			
#endif
};

#endif /* defined(ENET_INC_UMAC) */ 

uint32 interruptManagerIntCntxtCounter = 0; /* Should be removed when new disable int method will be adopted (ThreadX)*/

/****************************************************************************************************/
/*					This Section  should be moved to Inter CPU Module when it will be merged					*/
/*****************************************************************************************************/
/* Second level callbacks from inter CPU interrupts */

#if defined(ENET_INC_LMAC) 

const IntFromOtherCpuHandler_t apOtherCpuISR[OTHER_CPU_TABLE_SIZE] =
{
    vTLM_IPC_TO_LM_FROM_UM_ISR, 	/* IGEN_EVT_IPC_TO_LM0_FROM_UM		(1U << 0) 	*/
	NULL,				/*  						(1U << 1)	*/
	vTLM_ASSERT_TO_LM0_FROM_UM_ISR,	/* IGEN_EVT_ASSERT_TO_LM0_FROM_UM	(1U << 2)	*/
	NULL,				/* 							(1U << 3)	*/
	NULL,				/* 					       	(1U << 4) 	*/
    NULL,   			/* 				        	(1U << 5) 	*/
    NULL,               /* 					       	(1U << 6) 	*/
    NULL,               /* 					       	(1U << 7) 	*/
    NULL, 				/* 					      	(1U << 8) 	*/
    NULL,               /*                          (1U << 9) 	*/
    NULL,               /*                          (1U << 10) 	*/
    NULL,               /*                          (1U << 11) 	*/
    NULL,               /*                          (1U << 12) 	*/
    NULL,               /*                          (1U << 13) 	*/
    NULL,               /*                          (1U << 14) 	*/
    NULL,               /*                          (1U << 15) 	*/
    NULL,               /*                          (1U << 16) 	*/
    NULL,               /*                          (1U << 17) 	*/
    NULL,               /*                          (1U << 18) 	*/
    NULL,               /*                          (1U << 19) 	*/
	NULL,               /*                          (1U << 20) 	*/
	NULL,               /*                          (1U << 21) 	*/
	NULL,               /*                          (1U << 22) 	*/
	NULL,               /*                          (1U << 23) 	*/
};
#endif


#if defined(ENET_INC_UMAC)

const IntFromOtherCpuHandler_t apOtherCpuISR[OTHER_CPU_TABLE_SIZE] =
{

    vLMI_ISR_IPC_TO_UM_FROM_LM0,				/* IGEN_EVT_IPC_TO_UM_FROM_LM0		(1U << 0) 	*/
	NULL,				/*							(1U << 1)	*/ 
    vTLM_ASSERT_TO_UM_FROM_LM0_ISR,				/* IGEN_EVT_ASSERT_TO_UM_FROM_LM0	(1U << 2) 	*/
	NULL, 				/* 							(1U << 3)	*/
	NULL,               /*                         	(1U << 4) 	*/
    NULL,               /*                         	(1U << 5) 	*/
    NULL,               /*                         	(1U << 6) 	*/
    NULL,               /*                         	(1U << 7) 	*/
    NULL,               /*                         	(1U << 8) 	*/
    NULL,				/*							(1U << 9) 	*/
	NULL, 				/*							(1U << 10)	*/ 
    NULL, 				/* 					        (1U << 11) 	*/
	NULL, 				/* 							(1U << 12) 	*/
	NULL,				/* 							(1U << 13) 	*/
	NULL,				/* 							(1U << 14) 	*/
    NULL,				/* 							(1U << 15) 	*/
    NULL,				/*           				(1U << 16) 	*/
	NULL,				/*					       	(1U << 17) 	*/
    NULL, 				/* 							(1U << 18) 	*/
    NULL,  				/* 							(1U << 19) 	*/
    NULL,  				/* 							(1U << 20) 	*/
    NULL,  				/* 							(1U << 21) 	*/
	NULL,     			/*							(1U << 22) 	*/
	NULL,   			/*							(1U << 23) 	*/
};
#endif


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/
#if defined (ANALYZE_DISABLE_INTERRUPT)
		
uint32 AAA_DebugTrace_analyzeDisableInterrupts_array[AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_SIZE][7] = {0x0};
uint16 AAA_DebugTrace_analyzeDisableInterrupts_idx = 0;
		

#endif

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

isr_InterruptManagerSpuriousEvent 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
ISR_VOID isr_InterruptManagerSpuriousEvent(void)
{ 
	FATAL("Spurious Interrupt"); 
}

/**********************************************************************************

vINT_fromOtherCore 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
ISR_VOID vINT_fromOtherCore()
{
	uint32 u32Index;

#if defined(ENET_INC_UMAC)
	RegMacRabUpiInterrupt_u upiInterruptReg;
	RegMacRabUpiInterruptClear_u upiInterruptClearReg;
#endif
#if defined(ENET_INC_LMAC)
	RegMacRabLpiInterrupt_u lpiInterruptReg;
	RegMacRabLpiInterruptClear_u lpiInterruptClearReg;
#endif

	ACCUMULATE_CPU_IDLE_TIME();
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vINT_OtherCore,0,0);

#if defined(ENET_INC_UMAC)
	RegAccess_Read(REG_MAC_RAB_UPI_INTERRUPT, &upiInterruptReg.val);
	//make sure an event exists
	ASSERT(upiInterruptReg.val);
	u32Index = Utils_CountTrailingZeros(upiInterruptReg.val);
	upiInterruptClearReg.bitFields.upiInterruptClear = 1 << u32Index;
	RegAccess_Write(REG_MAC_RAB_UPI_INTERRUPT_CLEAR, upiInterruptClearReg.val);
#endif

#if defined(ENET_INC_LMAC)	
	RegAccess_Read(REG_MAC_RAB_LPI_INTERRUPT, &lpiInterruptReg.val);
	//make sure an event exists
	ASSERT(lpiInterruptReg.val);
	u32Index = Utils_CountTrailingZeros(lpiInterruptReg.val);
	lpiInterruptClearReg.bitFields.lpiInterruptClear = 1 << u32Index;
	RegAccess_Write(REG_MAC_RAB_LPI_INTERRUPT_CLEAR, lpiInterruptClearReg.val);
#endif

	ASSERT(apOtherCpuISR[u32Index] != NULL);
	apOtherCpuISR[u32Index]();
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vINT_OtherCore,1,1);	
}


/**********************************************************************************

interruptManagerConfigureEdgedTriggeredInterrupt 

Description:
------------
	This function is called to configure interrupts in the unit that was added to convert those interrupts 
	that were originally edge triggered to be level triggered

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
static void interruptManagerConfigureEdgedTriggeredInterrupt(void)
{
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_WL_MAC_IRQ, EDGED_TRIGGERED_ACTIVE_INT_MASK);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

InterruptManager_RabInit 

Description:
------------

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
#if defined (ENET_INC_UMAC) || defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif

void InterruptManager_RabInit(void)
{
	RegMacRabNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
	RegMacRabUpiInterruptClear_u upiInterruptClearReg;
	RegMacRabLpiInterruptClear_u lpiInterruptClearReg;

	//Clear interrupts from MAC IP that might be pending
	
	upiInterruptClearReg.bitFields.upiInterruptClear = 0xFFFFFF;  // RAB/ucpu clear reg
	RegAccess_Write(REG_MAC_RAB_UPI_INTERRUPT_CLEAR, upiInterruptClearReg.val);

	lpiInterruptClearReg.bitFields.lpiInterruptClear = 0xFFFFFF;  // RAB/lcpu clear reg
	RegAccess_Write(REG_MAC_RAB_LPI_INTERRUPT_CLEAR, lpiInterruptClearReg.val);

	npu2UpiInterruptClearReg.bitFields.npu2UpiInterruptClear = 0xFFFFFF;	// RAB/npu clear reg
	RegAccess_Write(REG_MAC_RAB_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);

}
#if ((defined (ENET_INC_UMAC)  || defined (ENET_INC_LMAC)) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

/**********************************************************************************

InterruptManager_DisableAllInterrupts 

Description:
------------
	Disable the interrupts of the MIPS, Used for shared variables protection.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
#if defined (ANALYZE_DISABLE_INTERRUPT)
void InterruptManager_DisableAllInterrupts(unsigned int* interrupt_save, char* fileName,unsigned long int line)
#else
void InterruptManager_DisableAllInterrupts(unsigned int* interrupt_save)
#endif
{
#if defined (ENET_INC_LMAC)
	uint32 u32StatusReg;


	u32StatusReg = __GetStatusReg();

	/*	check if disable interrupts was called when using shadow set registers.			
		in this case all interrupts are already disabled.	*/
	if (u32StatusReg & OS_STATUS_REG_EXL_BIT) 
	{
		return;
	}

 	interruptManagerIntCntxtCounter++; /* Increment global interrupt counter */

	if (interruptManagerIntCntxtCounter > 1)
	{
		return;
	}

	asm("di $3");
	asm("nop");


	AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_MACRO(1,fileName,line);
#else 
	*interrupt_save =  _tx_thread_interrupt_control(TX_INT_DISABLE);
#endif 
}

/**********************************************************************************

InterruptManager_EnableAllInterrupts 

Description:
------------
	Enable the interrupts of the MIPS

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
#if defined(ANALYZE_DISABLE_INTERRUPT)
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save, char* fileName, unsigned long int line)
#else
void InterruptManager_EnableAllInterrupts(unsigned int interrupt_save)
#endif
{
#if defined (ENET_INC_LMAC)
	uint32 u32StatusReg;
	u32StatusReg = __GetStatusReg();

	/*	check if enable interrupts was called when using shadow set registers.
		in this case no need to enable interrupts because we didn't call disable interrupts.*/
	if (u32StatusReg & OS_STATUS_REG_EXL_BIT)
	{
		return;
	}

	ASSERT(interruptManagerIntCntxtCounter != 0);

	interruptManagerIntCntxtCounter--; /*  decrement global interrupt counter */

	if (interruptManagerIntCntxtCounter == 0)
		/* -------------------------------------------------------------------------------------------------------------- */
		/* we enable interrupts only if we insure that we are not in a critical section                                   */
		/* -------------------------------------------------------------------------------------------------------------- */
	{
		AAA_DEBUGTRACE_ANALYZE_DISABLE_INTERRUPTS_MACRO(2,fileName,line);

		asm("ei $3");
		asm("nop");
	}	
#else
	_tx_thread_interrupt_control(interrupt_save);
#endif
}

/**********************************************************************************

InterruptManager_IsInInterruptContext 

Description:
------------
	

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
bool InterruptManager_IsInInterruptContext(void)
{
	uint32 u32StatusReg;

	u32StatusReg = __GetStatusReg();

	/*	check if we are using shadow set registers */
	if (u32StatusReg & OS_STATUS_REG_EXL_BIT) 
	{
		return TRUE;
	}

	return FALSE;
}

/**********************************************************************************

InterruptManager_MaskInterrupt 

Description:
------------
	Mask(Disable) specific source in the external interrupt controller

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
	
**********************************************************************************/
void InterruptManager_MaskInterrupt(MappedIntId_e mappedIntId)
{
	RegAccess_WriteMasked(REG_EMERALD_ENV_IRQ_MASK, (1 << mappedIntId), 0);
}

/**********************************************************************************

InterruptManger_UnMaskInterrupt 

Description:
------------
	Enable specific source in interrupt controller

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void InterruptManager_UnMaskInterrupt(MappedIntId_e mappedIntId)
{
	RegAccess_WriteMasked(REG_EMERALD_ENV_IRQ_MASK, (1 << mappedIntId),(1 << mappedIntId));
}


void InterruptManager_EnableUpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnableUpiInterrupt_u enableUpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_UPI_INTERRUPT, &enableUpiInterruptReg.val);
	enableUpiInterruptReg.bitFields.enableUpiInterrupt |= (1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_UPI_INTERRUPT, enableUpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

void InterruptManager_DisableUpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnableUpiInterrupt_u enableUpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_UPI_INTERRUPT, &enableUpiInterruptReg.val);
	enableUpiInterruptReg.bitFields.enableUpiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_UPI_INTERRUPT, enableUpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

void InterruptManager_EnableLpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnableLpiInterrupt_u enableLpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_LPI_INTERRUPT, &enableLpiInterruptReg.val);
	enableLpiInterruptReg.bitFields.enableLpiInterrupt |= (1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_LPI_INTERRUPT, enableLpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

void InterruptManager_DisableLpiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnableLpiInterrupt_u enableLpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_LPI_INTERRUPT, &enableLpiInterruptReg.val);
	enableLpiInterruptReg.bitFields.enableLpiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_LPI_INTERRUPT, enableLpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

void InterruptManager_EnablePhiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnablePhiInterrupt_u enablePhiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_PHI_INTERRUPT, &enablePhiInterruptReg.val);
	enablePhiInterruptReg.bitFields.enablePhiInterrupt |= (1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_PHI_INTERRUPT, enablePhiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

void InterruptManager_DisablePhiRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnablePhiInterrupt_u enablePhiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_PHI_INTERRUPT, &enablePhiInterruptReg.val);
	enablePhiInterruptReg.bitFields.enablePhiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_PHI_INTERRUPT, enablePhiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}


void InterruptManager_EnableNpuRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnableNpu2UpiInterrupt_u enableNpu2UpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_NPU2UPI_INTERRUPT, &enableNpu2UpiInterruptReg.val);
	enableNpu2UpiInterruptReg.bitFields.enableNpu2UpiInterrupt |= (1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_NPU2UPI_INTERRUPT, enableNpu2UpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}


void InterruptManager_DisableNpuRabInt(uint8 intIndex)
{
	TX_INTERRUPT_SAVE_AREA;
	RegMacRabEnableNpu2UpiInterrupt_u enableNpu2UpiInterruptReg;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// Read - Modify - Write
	RegAccess_Read(REG_MAC_RAB_ENABLE_NPU2UPI_INTERRUPT, &enableNpu2UpiInterruptReg.val);
	enableNpu2UpiInterruptReg.bitFields.enableNpu2UpiInterrupt &= ~(1 << intIndex);
	RegAccess_Write(REG_MAC_RAB_ENABLE_NPU2UPI_INTERRUPT, enableNpu2UpiInterruptReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}


/**********************************************************************************

InterruptManager_ClearEdgedTriggeredInterrupt 

Description:
------------
	Clear/disable interrupts in the unit that was added to convert 
	those interrupts that were originally edge triggered to be level triggered
Input:
-----

		
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void InterruptManager_ClearEdgedTriggeredInterrupt(EdgedTriggeredIntId_e intId)
{
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_WL_MAC_IRQ,1 << intId);
}

/**********************************************************************************

InterruptManager_Configure 

Description:
------------
	This function should be called once during system initialization to initialise the Interrupt Controller Driver.
	It should be called before the operating system is initialised.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
#if defined (ENET_INC_UMAC) || defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void InterruptManager_Configure(void)
{
	uint32 	picConfTableIndex;
	uint32 	regAddr;
	uint32	regValue;
	uint32	regMask;
	MappedIntId_e mappedInterruptId;

#if defined(ENET_INC_LMAC)
	TX_INTERRUPT_SAVE_AREA;
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#endif 	
	/*Remap all relevant interrupts to priority*/	
	for (picConfTableIndex = 0; picConfTableIndex < MAPPED_INT_TOTAL_NUM; picConfTableIndex++) 
	{
		mappedInterruptId = (MappedIntId_e)interruptManagerPicConfiguration[picConfTableIndex].mappedInterruptId;
		InterruptManagerIsrHandlers[mappedInterruptId] = interruptManagerPicConfiguration[picConfTableIndex].isrHandler;

		/* Confure PIC Registers*/
		regAddr = REG_EMERALD_ENV_IRQ_MAP_00 + interruptManagerPicConfiguration[picConfTableIndex].unMappedIntPicRegNum;
		regValue = interruptManagerPicConfiguration[picConfTableIndex].mappedInterruptId << interruptManagerPicConfiguration[picConfTableIndex].unMappedIntPicRegOffset;		
		regMask =  0x1F << interruptManagerPicConfiguration[picConfTableIndex].unMappedIntPicRegOffset;
		RegAccess_WriteMasked(regAddr,regMask,regValue);
		
		InterruptManager_UnMaskInterrupt(mappedInterruptId);
	}

	interruptManagerConfigureEdgedTriggeredInterrupt();
	HwEvents_Init();
	HwErrors_Init();
	/*Set All interrupt to work with second register file*/
	RegAccess_Write(REG_EMERALD_ENV_IRQ_SEC_RF,0xFFFFFFFF);	
#if defined(ENET_INC_LMAC)
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	/*All interrupts are disabled, we've finished mapping them so now enable all Int's*/
	__EnableInt(0);//mask changed from 0xff - EIC mode
#endif 	
}



/**********************************************************************************

InterruptManager_Reset 

Description:
------------
	This function should be called once during system initialization to initialise the Interrupt Controller Driver.
	It should be called before the operating system is initialised.
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void InterruptManager_Reset(void)
{

	uint32 regAddr;
	uint32 mappedIntid;
	RegAccess_Write(REG_EMERALD_ENV_IRQ_MASK,CLEAR_ALL_INTERRUPTS);
		
	/* Remap all interrupts to an unused line - then open only those that are used */ 
	for(regAddr = REG_EMERALD_ENV_IRQ_MAP_00; regAddr <= REG_EMERALD_ENV_IRQ_MAP_05; regAddr += 4)
	{
		RegAccess_Write(regAddr,0);
	}

	/* Clear ISR table*/
	for (mappedIntid = 0; mappedIntid < MAPPED_INT_TOTAL_NUM; mappedIntid++) 
	{
		InterruptManagerIsrHandlers[mappedIntid] = isr_InterruptManagerSpuriousEvent;
	}
}
#if ((defined (ENET_INC_UMAC)  || defined (ENET_INC_LMAC)) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
/***************************************************************************
**
** NAME         vIGEN_SendToUm
**
** PARAMETERS   u32Source      The interrupt source to generate.
**
** RETURNS      na
**
** DESCRIPTION  This function generates an interrupt to the Upper MAC.
**
****************************************************************************/

void vIGEN_SendToUm(uint32 u32Source)
{
	RegMacRabUpiInterruptSet_u upiInterruptSetReg;

	upiInterruptSetReg.bitFields.upiInterruptSet = u32Source; 
	RegAccess_Write(REG_MAC_RAB_UPI_INTERRUPT_SET, upiInterruptSetReg.val);
}

/***************************************************************************
**
** NAME         vIGEN_SendToLm0
**
** PARAMETERS   u32Source      The interrupt source to generate.
**
** RETURNS      na
**
** DESCRIPTION  This function generates an interrupt to the Lower MAC.
**
****************************************************************************/
void vIGEN_SendToLm0(uint32 u32Source)
{
	RegMacRabLpiInterruptSet_u lpiInterruptSetReg;
	lpiInterruptSetReg.bitFields.lpiInterruptSet = u32Source; 
	RegAccess_Write(REG_MAC_RAB_LPI_INTERRUPT_SET, lpiInterruptSetReg.val);
}





