/***********************************************************************************
 File:		AntennaSelection.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "HdkGlobalDefs.h"
#include "HwGlobalDefinitions.h"
#include "lm_VapDatabase.h"
#include "RateAdaptation.h"
#include "PowerAdaptation.h"
#include "AntennaSelection.h"
#include "LinkAdaptationCommon.h"
#include "LinkAdaptation_StateMachine.h"
#include "Estimators.h"
#include "CDD.h"
#include "Locker_Api.h"
#include "GroupDb.h"
#include "ShramGroupDatabase.h"
#include "ConfigurationManager_api.h"
#include "CommonRamLinkAdaptation.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "PreAggregator_Api.h"
#endif
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 1


//#define ANTENNA_SELECTION_DEBUG
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void setAntennaCount(StaId staIndex, uint8 antennaSelectionMask);
static void setActivatedAntennas(void);
static void setAntSelParams(StaId staIndex, uint8 numOfAntennas, bool isLimitingState, bool changeWorkingPoint);
static uint8 getPermutation(uint8 antennasIndex, uint8 permutationIndex);
static void updateStaAndVapAntennaSelection(void);
static void antennaSelectionSetAutoReplyAntMask(uint8 antennaSelectionMask);
static bool setAntSelectionInStaTcr(StaId staIndex, uint8 antSelectionMask, bool changeProbingPoint, LaPacketType_e packetType, bool setRatePlusOne);
static void AntSelectionModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static uint8 increaseAntSelectionProbingIndices(StaId staIndex);

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
static uint8 ConnectedAntennasBitmap; 	//bitmap for physical antennas, every 2 bits resemble one antenna LSB is antenna #0
static uint8 EnabledAntennasBitmap;		//bitmap for enabled antennas by CoC, every 2 bits resemble one antenna LSB is antenna #0
static uint8 ActivatedAntennasBitmap;	//bitmap for enabled antnenas by Antenna selection, every 2 bits resemble one antenna LSB is antenna #0
static uint8 ActivatedAntennasCount;			//number of ActivatedAntennasBitmap
static StationBitmap_t AntSelProbeWithRatePlus1;
static uint8 AntennaSelectionPermutationsDB[ANT_SEL_MAX_NUMBER_OF_ANTENNAS][ANT_SEL_MAX_NUMBER_OF_PERMUTATIONS + 1] = 
{
	{0x1,  0x2,  0x4,  0x8,  0x10, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x5,  0x6,  0x9,  0xA,  0x11, 0x12, 0x21, 0x22, 0x14, 0x18, 0x24, 0x28, 0x41, 0x42, 0x81, 0x82, 0x44, 0x48, 0x84, 0x88, 0x50, 0x60, 0x90, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x15, 0x16, 0x19, 0x1A, 0x25, 0x26, 0x29, 0x2A, 0x45, 0x46, 0x49, 0x4A, 0x85, 0x86, 0x89, 0x8A, 0x51, 0x52, 0x91, 0x92, 0x61, 0x62, 0xA1, 0xA2, 0x54, 0x64, 0x58, 0x68, 0x94, 0xA4, 0x98, 0xA8, 0x00},
	{0x55, 0x56, 0x59, 0x5A, 0x65, 0x66, 0x69, 0x6A, 0x95, 0x96, 0x99, 0x9A, 0xA5, 0xA6, 0xA9, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
};

/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

AntennaSelectionInit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void AntennaSelectionInit (void)
{
	ConnectedAntennasBitmap = ANTENNA_CONNECTED_DEFAULT_BITMAP;
	EnabledAntennasBitmap = ANTENNA_CONNECTED_DEFAULT_BITMAP;
	setActivatedAntennas();
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************

AntennaSelectionAddSta 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void AntennaSelectionAddSta (StaId staIndex)
{
	AntennaSelectionDatabase_t* pAntennaSelectionDb = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.antennaSelectionDb);
	uint8 antSelMask = AntennaSelectionGetDefaultBitmap();
	
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_DD("AntennaSelectionAddSta. ActivatedAntennasCount %d antSelMask %d", ActivatedAntennasCount, antSelMask);
	SLOG0(0, 0, AntennaSelectionDatabase_t, pAntennaSelectionDb);
#endif		

	LA_CLR_BIT_IN_BITMAP(AntSelProbeWithRatePlus1.staBitMap,staIndex);

	// Update working point params
	pAntennaSelectionDb->workingPointAntennaCount = ActivatedAntennasCount;
	pAntennaSelectionDb->workingPointAntennaMask = antSelMask;
	
	// Update probing point params
	pAntennaSelectionDb->probingIndices.numOfAntennasIndex = ActivatedAntennasCount - 1;
	pAntennaSelectionDb->probingIndices.permutationIndex = 0;

#ifdef ANTENNA_SELECTION_DEBUG	
	SLOG0(0, 0, AntennaSelectionDatabase_t, pAntennaSelectionDb);
#endif	

	// Set Default Antenna Selection mask in station TCR
	setAntSelectionInStaTcr(staIndex, antSelMask, FALSE, LA_PACKET_TYPE_DATA, FALSE);
	setAntSelectionInStaTcr(staIndex, antSelMask, FALSE, LA_PACKET_TYPE_MANAGEMENT, FALSE);
}

/**********************************************************************************

AntennaSelectionSetNextProbingPoint 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	AntennaSelectionDatabase_t* pAntennaSelectionDb = &(laDbDistributionParameter->laStationUnique->antennaSelectionDb);
	bool probeWithRatePlusOne;
	uint8 nextprobingMask = ANT_SEL_INVALID_PERMUTATION;
	bool isCurrentSlowProbingMaskEmpty = FALSE;
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_V("AntennaSelectionSetNextProbingPoint");
	SLOG0(0, 0, AntennaSelectionDatabase_t, pAntennaSelectionDb);
#endif	

	UNUSED_PARAM(stayWithLastProbingPoint);

	setCurrentSlowProbingTaskInDb(laDbDistributionParameter, SLOW_PROBING_ANT_SEL);

	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(AntSelProbeWithRatePlus1.staBitMap, laDbDistributionParameter->stationOrGroupIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	nextprobingMask = getPermutation(pAntennaSelectionDb->probingIndices.numOfAntennasIndex, pAntennaSelectionDb->probingIndices.permutationIndex);
	
	if ((pAntennaSelectionDb->workingPointAntennaMask == nextprobingMask) || ((nextprobingMask & ActivatedAntennasBitmap) != nextprobingMask))
	{
		nextprobingMask = increaseAntSelectionProbingIndices(laDbDistributionParameter->stationOrGroupIndex);
	}
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_D("AntennaSelectionSetNextProbingPoint. nextprobingMask 0x%x", nextprobingMask);
#endif		

	if (nextprobingMask != ANT_SEL_INVALID_PERMUTATION)
	{
		// Set Antenna Selection in TCR
		setAntSelectionInStaTcr(laDbDistributionParameter->stationOrGroupIndex, nextprobingMask, TRUE, LA_PACKET_TYPE_DATA, probeWithRatePlusOne);
		laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_SLOW_PROBE_VALID);
	}
	else
	{
		isCurrentSlowProbingMaskEmpty = disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_ANT_SEL);
		if(isCurrentSlowProbingMaskEmpty == TRUE)
		{
			//Don't use probing,   update the state of RA accodingly 
			laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
		}
	}
	
}

/**********************************************************************************

AntennaSelectionProcessFeedback 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t *pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	AntennaSelectionDatabase_t* pAntSelDb = &(laDbDistributionParameter->laStationUnique->antennaSelectionDb);
	Bandwidth_e mainBw;
	SlowProbingPointEstimators_t* pSlowProbingEst;
	FastProbingPointEstimators_t* pWorkingPointEst;
	int8 workingPointPer;
	uint8 slowProbingPointPer;
	bool probeWithRatePlusOne;
	uint8 slowProbingRateIndex;
	uint8 currentRateindex;
	uint16 losslessTpWp;
	uint16 losslessTpPp;
	uint16 effectiveTpWp;
	uint16 effectiveTpProbing;
	uint8 antSelMask;
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	StaId staIndex;
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_V("AntennaSelectionProcessFeedback");
#endif	
	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_ANT_SEL);

	if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
	{
		return; 
	}
	staIndex = laDbDistributionParameter->stationOrGroupIndex;
	mainBw = GetDataBwLimit(staIndex,INVALID_MU_USP_INDEX, FALSE);
	pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);
	pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter);
	slowProbingRateIndex = pWorkingPointEst->rateIndex;
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(AntSelProbeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	//KW_FIX_FW_M
	ASSERT(mainBw < MAX_POSSIBLE_NUM_OF_BW);

	if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff) &&
		AlphaFilter_isFilterResultValid(&pWorkingPointEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
	{

		workingPointPer = AlphaFilter_GetFilterResult(&pWorkingPointEst->averagePer);
		slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);
		workingPointPer = MAX(workingPointPer - WP_PER_BIAS, 0);
		
		/*Get current rate index*/
		currentRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);

		if (probeWithRatePlusOne == FALSE)
		{
			/*Instead of calculate the effective TP, calculate the success ratio instead of PER, so the comparition later will be correct*/
			effectiveTpWp = HUNDRED_PERCENT - workingPointPer;
			effectiveTpProbing = HUNDRED_PERCENT - slowProbingPointPer ;
		}
		else
		{
			/*Extract lossless TP according to bandwidth*/
			losslessTpWp = ratesTable[currentRateindex].rateTableBwParmas[mainBw].shortCpPhyRate;
			losslessTpPp = ratesTable[slowProbingRateIndex].rateTableBwParmas[mainBw].shortCpPhyRate;
			
			/*Calc effective TP for working point and probing point*/
			effectiveTpWp = estimatorsCalcEffectiveTp(losslessTpWp,workingPointPer);
			effectiveTpProbing = estimatorsCalcEffectiveTp(losslessTpPp,slowProbingPointPer);
		}
		/*probing per was better than WP per, and probing step != 0*/
		if (effectiveTpProbing > effectiveTpWp)
		{
			// PB is better than WP - change WP
#ifdef ANTENNA_SELECTION_DEBUG
			ILOG0_V("AntennaSelectionProcessFeedback. PB is better than WP - change WP!");
#endif		
			antSelMask = getPermutation(pAntSelDb->probingIndices.numOfAntennasIndex, pAntSelDb->probingIndices.permutationIndex);
			
			AntennaSelectionSetStaAntSelMask(staIndex, antSelMask, FALSE);
			
			// Increase Antenna Selection index for the next probing
			increaseAntSelectionProbingIndices(staIndex);
			
			/*Reset thresholds*/
			rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
			/*Reset estimators*/
			estimatorsResetRatesEstimators(laDbDistributionParameter);
			/*Reset fast probing counter so fast probing can collect statistics before next fast probing event*/	
			resetFastProbingCounters(laDbDistributionParameter);
			changeTaskToDefaultPriority(laDbDistributionParameter, SLOW_PROBING_ANT_SEL);
			resetNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_ANT_SEL);
		}
		else
		{
			/*Probing point did not improve the TP*/
			if ((probeWithRatePlusOne == FALSE) &&
				(workingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh) &&
				(slowProbingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh))
			{
				/*If WP PER is lower than threshold, try to probe with higher rate in the next probing cycle*/
				LA_SET_BIT_IN_BITMAP(AntSelProbeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
#ifdef ANTENNA_SELECTION_DEBUG
			ILOG0_V("AntennaSelectionProcessFeedback - SET RATE + 1");
#endif	
			}
			else
			{
				// Unsuccessful probing.
				
				/*Clear probe with rate plus one indication*/
				LA_CLR_BIT_IN_BITMAP(AntSelProbeWithRatePlus1.staBitMap, staIndex);
				
				// Get next Antenna Selection permutation for the next probing
				increaseAntSelectionProbingIndices(staIndex);
				updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_ANT_SEL);
			}
		}
	}
	else
	{
		// Get next Antenna Selection permutation for the next probing
		increaseAntSelectionProbingIndices(staIndex);
		
		/*Clear probe with rate plus one indication*/
		LA_CLR_BIT_IN_BITMAP(AntSelProbeWithRatePlus1.staBitMap, staIndex);
	}
	return;
}

/**********************************************************************************

AntennaSelectionGetAntennaSelectionBitmap 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 AntennaSelectionGetAntennaSelectionBitmap(StaId staIndex)
{
	AntennaSelectionDatabase_t* pAntennaSelectionDb = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.antennaSelectionDb);
	
	return pAntennaSelectionDb->workingPointAntennaMask;
}

/**********************************************************************************

AntennaSelectionGetActivatedAntennasBitmap 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 AntennaSelectionGetActivatedAntennasBitmap()
{
	return ActivatedAntennasBitmap;
}

/**********************************************************************************

setActivatedAntennas 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setActivatedAntennas(void)
{
	ActivatedAntennasBitmap = EnabledAntennasBitmap & ConnectedAntennasBitmap;
	ActivatedAntennasCount = AntennaSelectionCalcAntennaCount(ActivatedAntennasBitmap);

#ifdef ENET_INC_ARCH_WAVE600D2
	/* Configure TF Generator for new max num of Antennas, for starting SS calc (used in UL MU-MIMO) */
	PreAggregator_SetTfGenMaxNumOfStartingSS(CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(ActivatedAntennasCount));
#endif
}

/**********************************************************************************

setAntSelParams 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setAntSelParams(StaId staIndex, uint8 numOfAntennas, bool isLimitingState, bool changeWorkingPoint)
{
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	AntennaSelectionDatabase_t* pAntennaSelectionDb = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.antennaSelectionDb);
	uint8 rateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(&(LinkAdaptationStaDatabase[staIndex].laStaUspCommon));
	uint8 nss = ratesTable[rateIndex].numberOfNss;
	uint8 index;
	
	if (changeWorkingPoint && (pAntennaSelectionDb->workingPointAntennaCount != numOfAntennas))
	{
		// Update working point params
		pAntennaSelectionDb->workingPointAntennaCount = numOfAntennas;
		pAntennaSelectionDb->workingPointAntennaMask = AntennaSelectionGetLowerAntSelection(numOfAntennas);
		
		// Update probing point params
		pAntennaSelectionDb->probingIndices.numOfAntennasIndex = numOfAntennas - 1;
		pAntennaSelectionDb->probingIndices.permutationIndex = 0;
		
#ifndef ENET_INC_ARCH_WAVE600
		// Update CDD adaptation loop
		CddStaNumOfAntsIsChanged(staIndex);
#endif
	}
	
	// Update valid antennas bitmap
	if (isLimitingState)
	{
		pAntennaSelectionDb->validAntennas = (1 << (numOfAntennas - 1));
	}
	else
	{
		pAntennaSelectionDb->validAntennas = 0;
		for (index = nss; index < numOfAntennas; index++)
		{
			pAntennaSelectionDb->validAntennas |= (1 << index);
		}
	}

#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_DDD("setAntSelParams. numOfAntennas %d isLimitingState %d validAntennas %d ",numOfAntennas, isLimitingState, pAntennaSelectionDb->validAntennas);
	SLOG0(0, 0, AntennaSelectionDatabase_t, pAntennaSelectionDb);
#endif
}

/**********************************************************************************

getPermutation 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static uint8 getPermutation(uint8 antennasIndex, uint8 permutationIndex)
{
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_DD("getPermutation. antennasIndex %d permutationIndex %d ", antennasIndex, permutationIndex);
#endif
	DEBUG_ASSERT(antennasIndex < ANT_SEL_MAX_NUMBER_OF_ANTENNAS);
	DEBUG_ASSERT(permutationIndex <= ANT_SEL_MAX_NUMBER_OF_PERMUTATIONS);
	
	return AntennaSelectionPermutationsDB[antennasIndex][permutationIndex];
}

/**********************************************************************************

increaseAntSelectionProbingIndices 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static uint8 increaseAntSelectionProbingIndices(StaId staIndex)
{
	AntennaSelectionDatabase_t* pAntSelDb = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.antennaSelectionDb);
	uint8 loopIndex;
	uint8 nextPermutation;
	uint8 permutationIndex = pAntSelDb->probingIndices.permutationIndex;
	uint8 antennaIndex = pAntSelDb->probingIndices.numOfAntennasIndex;
	bool validPermutationFound = FALSE;
	
	permutationIndex++;
	// Get permutation from Database
	nextPermutation = getPermutation(antennaIndex, permutationIndex);

	for (loopIndex = 0; loopIndex < ANT_SEL_MAX_NUMBER_OF_ANTENNAS; loopIndex++)
	{
		if (pAntSelDb->validAntennas & (1 << (antennaIndex)))
		{
			while (nextPermutation != ANT_SEL_INVALID_PERMUTATION)
			{
#ifdef ANTENNA_SELECTION_DEBUG				
				ILOG0_DDD("increaseAntSelectionProbingIndices. nextPermutation 0x%x antennaIndex %d permutationIndex %d ",nextPermutation, antennaIndex, permutationIndex);
#endif	
				if (((nextPermutation & ActivatedAntennasBitmap) == nextPermutation) && (nextPermutation != pAntSelDb->workingPointAntennaMask))
				{
					// Valid permutation is found for the next probing
					validPermutationFound = TRUE;
					break;
				}
				permutationIndex++;
				// Get permutation from Database
				nextPermutation = getPermutation(antennaIndex, permutationIndex);
			};
			
			if (!validPermutationFound)
			{			
				// End of Antenna Selection permutations group
				antennaIndex++;
				if (antennaIndex == ActivatedAntennasCount)
				{
					antennaIndex = 0;
				}
				permutationIndex = 0;
				// Get permutation from Database
				nextPermutation = getPermutation(antennaIndex, permutationIndex);
			}
			else
			{
				break;
			}
		}
	}
	
	if (validPermutationFound)
	{	
		//Update probing point params
		
		pAntSelDb->probingIndices.numOfAntennasIndex = antennaIndex;
		pAntSelDb->probingIndices.permutationIndex = permutationIndex;
	}
	else
	{
		nextPermutation = ANT_SEL_INVALID_PERMUTATION;
		
		// Disable loop
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_ANT_SEL, DISABLE_LOOP);
	}	

#ifdef ANTENNA_SELECTION_DEBUG
		ILOG0_DD("increaseAntSelectionProbingIndices. validPermutationFound %d ActivatedAntennasBitmap 0x%x ",validPermutationFound, ActivatedAntennasBitmap);
#endif	
	return nextPermutation;
}

/**********************************************************************************

AntennaSelectionCalcAntennaCount


Description:
------------
Count number of activate antennas according to antenna selection bitmask

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 AntennaSelectionCalcAntennaCount(uint8 antennaSelctionBitmap)
{
	uint8 antennaCount = 0;
	
	while (antennaSelctionBitmap != 0)
	{
		if (antennaSelctionBitmap & 0x3)
		{
			antennaCount++;
		}
		antennaSelctionBitmap >>= 2;
	}
	
	DEBUG_ASSERT(antennaCount <= MAX_NUM_OF_ANTENNAS);

	return antennaCount;
}

/**********************************************************************************

setAntennaCount


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setAntennaCount(StaId staIndex, uint8 antennaSelectionMask)
{
	LinkAdaptationStaDatabase[staIndex].laStationUnique.antennaSelectionDb.workingPointAntennaCount = AntennaSelectionCalcAntennaCount(antennaSelectionMask);
}

/**********************************************************************************

AntennaSelectionGetAntennaCount


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 AntennaSelectionGetAntennaCount(StaId staIndex)
{
	uint8 antennaCount = LinkAdaptationStaDatabase[staIndex].laStationUnique.antennaSelectionDb.workingPointAntennaCount;

	ASSERT(antennaCount > 0);
	
	return antennaCount;
}

/**********************************************************************************

AntennaSelectionSetChannel


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void AntennaSelectionSetChannel(void)
{
	updateStaAndVapAntennaSelection();
}

/**********************************************************************************

AntennaSelectionConvertAntMaskToEnabledAnt


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 AntennaSelectionConvertAntMaskToEnabledAnt(uint8 antennaMask)
{
	uint8 antennaSelction =0;
	uint8 antennaSelctionMaskTwoAnt = 0x3;
	
	while (antennaMask != 0)
	{
		if (antennaMask & 0x1)
		{
			antennaSelction |= antennaSelctionMaskTwoAnt;
		}
		antennaSelctionMaskTwoAnt <<= 2;
		antennaMask >>= 1;
	}
	DEBUG_ASSERT(antennaSelction != 0);

	return antennaSelction;
}

/**********************************************************************************

AntennaSelectionSetEnabledAntMask


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionSetEnabledAntMask(uint8 enabledAntennaBitmap)
{
	EnabledAntennasBitmap = enabledAntennaBitmap;
	setActivatedAntennas();
	updateStaAndVapAntennaSelection();
}

/**********************************************************************************

AntennaSelection_GetConnectedAntennasBitmap 


Description:
------------


Input: 
-----

Returns:
--------
	ConnectedAntennasBitmap - bitmap of physically connected antennas
	
**********************************************************************************/
uint8 AntennaSelection_GetConnectedAntennasBitmap(void)
{
	return(ConnectedAntennasBitmap);
}

/**********************************************************************************

updateStaAndVapAntennaSelection


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void updateStaAndVapAntennaSelection(void)
{
	uint8 vapId;
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	uint8 vapAntennaSelection;
	uint8 defaultAntennaSelection;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	uint8 staNssNumber;
	uint8 staLimitNss;
	uint8 groupId;
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();
	const RateObj_t* ratesTable;
	

	defaultAntennaSelection = AntennaSelectionGetDefaultBitmap();

	/*Run over all connected stations and update anetnna selection*/
	while (nextSid != DB_ASYNC_SID)
	{
		DEBUG_ASSERT(nextSid < HW_NUM_OF_STATIONS);
		setAntSelParams(nextSid, AntennaSelectionCalcAntennaCount(defaultAntennaSelection), FALSE, TRUE);
		// Enable loop
		rateAdaptationEnableDisableStaSlowLoop(nextSid, SLOW_PROBING_ANT_SEL, ENABLE_LOOP);
		AntennaSelectionSetStaAntSelMask(nextSid, defaultAntennaSelection, TRUE);
		updateLaDbDistributionParam(&laDbDistributionParameter, nextSid, INVALID_MU_USP_INDEX, FALSE);
		ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);		
		staNssNumber =  ratesTable[estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter.laStaUspCommon)].numberOfNss;
		staLimitNss = ((SpatialStreamNum_e)CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(AntennaSelectionGetActivatedAntennasCount()));
		UpdateStationMaxSupportedNss(nextSid,(SpatialStreamNum_e)(MIN(staNssNumber, staLimitNss)));
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}

	/*Vap already been connected , so we need to go over all vaps and change antenna selection mask & max nss*/
	for (vapId = firstVapInBand; vapId < (firstVapInBand + numOfVapsInBand); vapId++)
	{
		vapAntennaSelection = defaultAntennaSelection;
		AntennaSelectionSetVapAntSelMask(vapId, vapAntennaSelection);

		//no tx at this time (called under process) - we can access db directly
		UpdateVapMaxNssNdp(vapId, (SpatialStreamNum_e)CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(AntennaSelectionGetActivatedAntennasCount()));
	}
	/*Update all active VHT MU groups*/
	for (groupId = 0; groupId < TX_MU_GROUPS; groupId++)
	{
		if(linkAdaptationIsGroupActive(groupId, FALSE))
		{
			antennaSelectionSetMuGroupAntenna(groupId,defaultAntennaSelection, FALSE);
		}
	}
#ifdef ENET_INC_ARCH_WAVE600
	/*Update all active HE MU groups*/
	for (groupId = 0; groupId < NUM_OF_LA_HE_MU_DB_ENTRIES; groupId++)
	{
		if(linkAdaptationIsGroupActive(groupId, TRUE))
		{
			antennaSelectionSetMuGroupAntenna(groupId,defaultAntennaSelection, TRUE);
		}
	}
#endif
	antennaSelectionSetAutoReplyAntMask(defaultAntennaSelection);
}

/**********************************************************************************

antennaSelectionSetAutoReplyAntMask


Description:
------------
	Modify the the TCR0 (Holds Antenna Selection) & TCR0 (Cdd Values) 

Input: 
-----
	None
	
**********************************************************************************/
static void antennaSelectionSetAutoReplyAntMask(uint8 antennaSelectionMask)
{
	uint8 numberOfAntennas;
	
	AntennaSelectionSetBitmapInAutoReplyTcr(antennaSelectionMask);

	numberOfAntennas = AntennaSelectionCalcAntennaCount(antennaSelectionMask);
#ifndef ENET_INC_ARCH_WAVE600
	if (numberOfAntennas >= CDD_MIN_NUMBER_OF_ANTENNAS)
	{
		CddAutoReplyNumOfAntsIsChanged(numberOfAntennas);
	}
#endif
}

/**********************************************************************************

AntennaSelectionSetStaAntSelMask

Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionSetStaAntSelMask(StaId staIndex, uint8 antennaSelectionMask, bool chnageRateMask)
{
	uint8 oldNumOfAnts, newNumOfAnts;
	LinkAdaptationStaDatabase_t* pLaStaDb = &LinkAdaptationStaDatabase[staIndex];
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter,staIndex,INVALID_MU_USP_INDEX, FALSE);
	antennaSelectionMask = (uint8)(antennaSelectionMask & ActivatedAntennasBitmap);
	oldNumOfAnts = AntennaSelectionGetAntennaCount(staIndex);

 	/*Set antenna selection*/
	setAntennaCount(staIndex,antennaSelectionMask);
	pLaStaDb->laStationUnique.antennaSelectionDb.workingPointAntennaMask = antennaSelectionMask;
	
	newNumOfAnts =  AntennaSelectionGetAntennaCount(staIndex);
#ifndef ENET_INC_ARCH_WAVE600
	if (oldNumOfAnts != newNumOfAnts)
	{
		CddStaNumOfAntsIsChanged(staIndex);
	}
#endif
	if (chnageRateMask == TRUE)
	{
		if(pLaStaDb->laStaUspCommon.staTransmissionParams.staInSmpsStaticMode == FALSE)
		{
			if (pLaStaDb->laStationUnique.operatingModeNotification.staInOmn == FALSE)
			{
				/*Set rate adaptation mask according to num of antennas*/
				setMaskAccordingToMaxNumOfAntennas(&laDbDistributionParameter, newNumOfAnts); 
			}
			else
			{
				setMaskAccordingToMaxNumOfAntennas(&laDbDistributionParameter, MIN(newNumOfAnts, CONVERT_MAX_NUM_OF_NSS_TO_MAX_NUM_OF_ANTENNAS(pLaStaDb->laStationUnique.operatingModeNotification.rxNss))); 
			}
		}
		/*Adjust rate to num of antennas*/
		rateAdaptationAdjustRateToMask(&laDbDistributionParameter);
		setAntSelectionInStaTcr(staIndex, antennaSelectionMask, FALSE, LA_PACKET_TYPE_MANAGEMENT, FALSE);
	}
	/*Set antenna selection in TCR and adjust other parameters (power, BF, etc.)*/
	setAntSelectionInStaTcr(staIndex, antennaSelectionMask, FALSE, LA_PACKET_TYPE_DATA, FALSE);
}

/**********************************************************************************

AntennaSelectionSetVapAntSelMask


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionSetVapAntSelMask(uint8 vapId, uint8 antennaSelectionMask)
{
	uint8 numberOfAntennas;

	AntennaSelectionSetBitmapInVapTcr(vapId,antennaSelectionMask,LA_PACKET_TYPE_DATA_MANAGEMENT);
	CalcLimitsAndSetVapPower(vapId);
	numberOfAntennas = AntennaSelectionCalcAntennaCount(antennaSelectionMask);
	
#ifndef ENET_INC_ARCH_WAVE600
	if (numberOfAntennas >= CDD_MIN_NUMBER_OF_ANTENNAS)
	{
		CddVapNumOfAntsIsChanged(vapId,numberOfAntennas);
	}
#endif
}
/**********************************************************************************

AntennaSelectionGetDefaultBitmap


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 AntennaSelectionGetDefaultBitmap(void)
{
	uint8 antennaSelection = 0;
	uint8 i = 0;
	uint8 validAntMask = ActivatedAntennasBitmap;
	
	while (validAntMask != 0)
	{
		if ((validAntMask & 0x3) == 0x3)
		{
			/*Two antennas on port ae valid -> set lowest antenna on port*/
			antennaSelection |= (0x1<<(i*2));
		}
		else if (validAntMask & 0x3)
		{
			/*Only one antenna is valid on port*-> set the valid antenna*/
			antennaSelection |= ((validAntMask&0x3) << (i*2));
		}
		validAntMask >>= 2;
		i++;
	}

	return antennaSelection;
}

/**********************************************************************************

setAntSelectionInStaTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static bool setAntSelectionInStaTcr(StaId staIndex, uint8 antSelectionMask, bool changeProbingPoint, LaPacketType_e packetType, bool setRatePlusOne)
{

	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	LaTcrModificationStruct_t tcrModicationParams;
	bool ratePlusOneChanged;

	memset(&tcrModicationParams,0,sizeof(LaTcrModificationStruct_t));
	
	/*Fill control parameters of tcrModicationParams*/
	tcrModicationParams.controlParams.firstBwToUpdate = BANDWIDTH_TWENTY;
	tcrModicationParams.controlParams.lastBwToUpdate = LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	tcrModicationParams.controlParams.isVhtSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.vhtSta;
	tcrModicationParams.controlParams.changeProbingPoint=  changeProbingPoint;
	tcrModicationParams.controlParams.slowProbing=  changeProbingPoint;
	tcrModicationParams.controlParams.packetType = packetType;
	tcrModicationParams.controlParams.changeToRatePlusOne = setRatePlusOne;
	tcrModicationParams.tcrParams.tcrGeneralVals.tcrAntSelection = antSelectionMask;
	tcrModicationParams.controlParams.staIndex = staIndex;
	tcrModicationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX;
#ifdef ENET_INC_ARCH_WAVE600
	tcrModicationParams.controlParams.isHeGroup = FALSE;
#endif
	/*Set antenna selection and effected params in TCRs*/
	ratePlusOneChanged = modifyStaTcrsParamsReq(&tcrModicationParams,AntSelectionModificationFunc); 	
	
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_DD("setAntSelectionInStaTcr 0x%x isProbing: %d", antSelectionMask, changeProbingPoint);
#endif

	return ratePlusOneChanged;

}

/**********************************************************************************

AntSelectionModificationFunc 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void  AntSelectionModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
#ifndef ENET_INC_ARCH_WAVE600	
	AntennaSelectionDatabase_t* pAntSelDb = &(LinkAdaptationStaDatabase[pModifcationParamsIn->controlParams.staIndex].laStationUnique.antennaSelectionDb);
	bool changeProbingPoint = pModifcationParamsIn->controlParams.changeProbingPoint;
	uint8 numOfAntsPp = pAntSelDb->probingIndices.numOfAntennasIndex + 1;
	uint8 numOfAntsWp = pAntSelDb->workingPointAntennaCount;
	if ((changeProbingPoint) && (numOfAntsPp != numOfAntsWp) && (numOfAntsPp >= CDD_MIN_NUMBER_OF_ANTENNAS))
	{
		// Change CDD with Antenna Selection probing, if number of antennas is changed
#ifdef ANTENNA_SELECTION_DEBUG
		ILOG0_D("AntSelectionModificationFunc.numOfAntsPp %d, change also CDD", numOfAntsPp);
#endif
		CddfillCddValues(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals), CDD_DEFAULT_INDEX, numOfAntsPp);
	}
#endif
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrAntSelection = pModifcationParamsIn->tcrParams.tcrGeneralVals.tcrAntSelection;
}

/**********************************************************************************

AntennaSelectionSetAntennaConnected

Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void AntennaSelectionSetAntennaConnected(uint8 antennaConnectedMask)
{
	/*This parameters will affect the sta DB when sta will be connected*/
	ConnectedAntennasBitmap = antennaConnectedMask;
	setActivatedAntennas();
}

/**********************************************************************************

AntennaSelectionGetLowerAntSelection

Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 AntennaSelectionGetLowerAntSelection(uint8 numberOfAnt)
{
	uint8 antennaCount = 0;
	uint8 retAntSelection = 0;
	uint8 currentAntSelection = AntennaSelectionGetDefaultBitmap();
	uint8 iterator = 0;
	
	while ((currentAntSelection != 0) && (antennaCount < numberOfAnt))
	{
		if (currentAntSelection & 0x3)
		{
			retAntSelection |= ((currentAntSelection & 0x3)<<(iterator<<1));
			antennaCount++;
		}
		currentAntSelection >>= 2;
		iterator++;
	}
	return retAntSelection;
}

/**********************************************************************************

AntennaSelectionStaNumOfAntsIsChanged

Description:
------------
Function called when number of antenna changed by other module and need to update database without update TCR

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionStaNumOfAntsChanged(StaId staIndex, uint8 antennaSelection)
{
	UNUSED_PARAM(staIndex);
	UNUSED_PARAM(antennaSelection);
}

/**********************************************************************************

AntennaSelectionBfModeIsChanged


Description:
------------
Function called when number of antenna changed by other module and need to update database without update TCR

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionBfModeIsChanged(StaId staIndex, BeamformingMode_e newBfMode)
{
	uint8 numOfAnts = 0;
	bool limitingState = FALSE;
	
#ifdef ANTENNA_SELECTION_DEBUG
	ILOG0_D("AntennaSelectionBfModeIsChanged. newBfMode %d", newBfMode);
#endif	

	switch (newBfMode)
	{
		case BF_STATE_EXPLICIT:
		case BF_STATE_IMPLICIT:
			
			numOfAnts = ActivatedAntennasCount;
			limitingState = TRUE;
			break;
			
		case BF_STATE_STBC1X2:
		
			numOfAnts = 2;
			limitingState = TRUE;
			break;

		case BF_STATE_STBC2X4:
						
			numOfAnts = 4;
			limitingState = TRUE;
			break;
				
		case BF_STATE_NON_BF:	
			
			numOfAnts = ActivatedAntennasCount;
			limitingState = FALSE;
			break;

		default:
			break;
	}
	setAntSelParams(staIndex, numOfAnts, limitingState, TRUE);
	// Enable loop
	rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_ANT_SEL, ENABLE_LOOP);
}

/**********************************************************************************

AntennaSelectionRateIsChanged


Description:
------------
Function called when number of antenna changed by other module and need to update database without update TCR

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void AntennaSelectionRateIsChanged(StaId staIndex, uint8 newRateIndex, uint8 oldRateIndex)
{
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	
	LinkAdaptationPhyMode_e newLaPhyMode = ratesTable[newRateIndex].laPhyMode;
	LinkAdaptationPhyMode_e oldLaPhyMode = ratesTable[oldRateIndex].laPhyMode;
	uint8 newNss = ratesTable[newRateIndex].numberOfNss + 1;
	uint8 oldNss = ratesTable[oldRateIndex].numberOfNss + 1;
	bool changeWorkingPoint = TRUE;
	LinkAdaptationStaDatabase_t* pLaStaDb = &LinkAdaptationStaDatabase[staIndex];
	
#ifdef ANTENNA_SELECTION_DEBUG
//	ILOG0_D("AntennaSelectionRateIsChanged. newRateIndex %d", newRateIndex);
#endif	

	if (pLaStaDb->laStationUnique.beamFormingDb.bfWorkingPointState == BF_STATE_NON_BF)
	{
		if (newRateIndex == oldRateIndex)
		{
			/* Rate isn't changed - will happen mostly in first time. change old rate params in order to consider only the new rate */
			if (oldLaPhyMode == LA_PHY_MODE_11B)
			{
				oldLaPhyMode = LA_PHY_MODE_11AG;
			}
			else
			{
				oldLaPhyMode = LA_PHY_MODE_11B;
			}
			oldNss--;
		}
			
		if ((oldLaPhyMode != LA_PHY_MODE_11B) && (newLaPhyMode == LA_PHY_MODE_11B)) 
		{
			/* Enter to 11B mode - Antenna Selection loop is irrelevant */
			
			uint8 antBoost[4] = {0};
			
			// reset antenna boosts
			setAntBoostInStaTcr(staIndex, antBoost);
			// Disable loop
			rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_ANT_SEL, DISABLE_LOOP);
		}
		else if ((oldLaPhyMode == LA_PHY_MODE_11B) && (newLaPhyMode != LA_PHY_MODE_11B))
		{
			/* Exit from 11B mode */

			// set antenna boosts again
			setAntBoostInStaTcr(staIndex, antPowerBoost);
			// Enable loop
			rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_ANT_SEL, ENABLE_LOOP);
			
		}
		else if (oldNss != newNss)
		{
			// NSS is changed
			if (newNss <= pLaStaDb->laStationUnique.antennaSelectionDb.workingPointAntennaCount)
			{
				changeWorkingPoint = FALSE;
			}
			
			setAntSelParams(staIndex, ActivatedAntennasCount, FALSE, changeWorkingPoint);
			// Enable loop
			rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_ANT_SEL, ENABLE_LOOP);
		}	
	}
}

/**********************************************************************************

GetActivatedAntennasCount 


Description:
------------


Input: 
-----

Returns:
--------
	ActivatedAntennasCount
	
**********************************************************************************/
uint8 AntennaSelectionGetActivatedAntennasCount(void)
{
	return ActivatedAntennasCount;
}

