/***********************************************************************************
 File:		Beamforming.c 
 Module:		LinkAdaptation 
 Purpose: 	Handling Beamforming (HT / VHT) / preamble (11B) 
 Description:	This fille handling the Beamforming probing and working point update
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "HdkGlobalDefs.h"
#include "lm_StaDatabase.h"
#include "mt_sysdefs.h"
#include "linkAdaptation_api.h"
#include "LinkAdaptationPhyDriver.h"
#include "CyclicPrefix.h"
#include "Pac_Api.h"
#include "CDD.h"
#include "HwBeamforming_Api.h"
#include "Beamforming.h"
#include "RateAdaptation.h"
#include "Estimators.h"
#include "LinkAdaptation_StateMachine.h"
#include "AntennaSelection.h"
#include "ShramStationDatabase.h"
#include "CommonRamLinkAdaptation.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 2

#define STBC_1X2_TCR_VAL 1
#define STBC_2X4_TCR_VAL 2

#define STBC_1X2_NUM_OF_ANT 2
#define STBC_2X4_NUM_OF_ANT 4

#define STBC_1X2_NUM_OF_NSS 1
#define STBC_2X4_NUM_OF_NSS 2

#define EXPLICIT_BF_OVERHEAD_COMPANSATION_PERCENT 3

#define BF_TIMER_SHIFT			3
#define BF_MAX_NUM_OF_STA_FOR_TIMER 	(HW_NUM_OF_STATIONS>>BF_TIMER_SHIFT)
#define BF_TIMER_DIFF 					(EXPLICIT_BF_REQUEST_INTERVAL>>BF_TIMER_SHIFT)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
#ifdef ENET_INC_ARCH_WAVE600
typedef void (*prepareBfModeFunc)(uint16);
#else
typedef void (*prepareBfModeFunc)(uint8);
#endif //ENET_INC_ARCH_WAVE600


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void setExplicitMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static void setImplicitMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static void setStbc1x2Mode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static void setStbc2x4Mode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static BfValiationOptions_e isExplicitValid(StaId staIndex, uint8 rateIndex);
static BfValiationOptions_e isImplicitValid(StaId staIndex, uint8 rateIndex);
static void setStbcMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut,uint8 stbcNumOfAnt, uint8 stbcTcrVal, uint8 stbcNumOfNss);
static BfValiationOptions_e isStbc1x2Valid(StaId staIndex, uint8 rateIndex);
static BfValiationOptions_e isStbc2x4Valid(StaId staIndex, uint8 rateIndex);
static BfValiationOptions_e isNoBfValid(StaId staIndex, uint8 rateIndex);
static BfValiationOptions_e getNextBfState(StaId staIndex, BeamformingMode_e* pNextBfMode, StationBitmap_t* pProbeWithHigherRate);
static void handleBfPolicy(StaId staIndex);
static void setPolicyTimeout(StaId staIndex, BeamformingMode_e lastState, BeamformingMode_e newState);
static BfValiationOptions_e isStbcValid(StaId staIndex, uint8 rateIndex,uint8 stbcNumOfAnt, uint8 stbcStaSupport, uint8 stbcNumOfNss, bool activationBitmapVal);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static prepareBfModeFunc bfPrepareBfTransFuncArr[BF_NUMBER_OF_MODES] =
{
	prepareExplicitMode,
	prepareImplicitMode,
	prepareStbc1x2Mode,
	prepareStbc2x4Mode,
	prepareNoBfMode,
};
static LaTcrModificationFunc bfSetProbingPointFuncArr[BF_NUMBER_OF_MODES]=
{
	setExplicitMode,
	setImplicitMode,
	setStbc1x2Mode,
	setStbc2x4Mode,
	setNoBfMode,
};
static BfModeValidationCheckFunc isBfModeValidFuncArr[BF_NUMBER_OF_MODES]=
{
	isExplicitValid,
	isImplicitValid,
	isStbc1x2Valid,
	isStbc2x4Valid,
	isNoBfValid,
};

static BeamFormingSupportedModes_t BeamFormingActivationBitMap;
static StationBitmap_t BfProbeWithRatePlus1;
static BeamFormingCalibrationPass_t BfCalibrationPass;
static StationBitmap_t BfExplicitBfRequestTimerActive;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

BeamformingInit 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void BeamformingInit (void)
{
	memset(&bfDebugParams, 0, sizeof(LaFixedBeamformingParams_t));
	memset(&BfExplicitBfRequestTimerActive, 0, sizeof(StationBitmap_t));
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif
 
/**********************************************************************************

BeamformingAddStation 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void BeamformingAddStation(StaId staIndex, HtCapabilitiesInfo_t htCapbilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities,sTransmitBfCapInfo transmitBfCapabilities)
{
	LinkAdaptationStaDatabase_t* pLaStaDb = &LinkAdaptationStaDatabase[staIndex];
	LaBeamformingDataBase_t* pBeamformingDb = &(pLaStaDb->laStationUnique.beamFormingDb);
	LaFixedBeamformingParams_t* bflocalDebugParams;
	uint32  currentTsf;
	
	currentTsf = GET_TSF_TIMER_LOW();
	currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);

	memset(&(pLaStaDb->laStationUnique.beamFormingDb), 0, sizeof(LaBeamformingDataBase_t));


	/*Dont clear BF explicit Timer active for this index ... timer may be running for this index for previous STA*/
	//LA_CLR_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,staIndex);
	LA_CLR_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap,staIndex);

	pBeamformingDb->policy = BF_POLICY_IMPLICIT;
	pBeamformingDb->cddCounter = CDD_MAX_NUMBER_OF_PERMUTATIONS-1; //Initiate with the last value minus one so first time it will be at the first CDD permuation 
	pBeamformingDb->changeToNonExplicitStateTsf = currentTsf;

	initExplicitBfStaCapabilities(staIndex,htCapbilities,vhtCapabilities, pHeMacPhyCapabilities, transmitBfCapabilities);
#ifdef ENET_INC_ARCH_WAVE600
	if (pLaStaDb->laStaUspCommon.staTransmissionParams.heSta == TRUE) //HE station
	{
		if(pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_SU_BEAMFORMEE == TRUE)
		{
			pLaStaDb->laStaUspCommon.staTransmissionParams.staExplicitBfSupport = TRUE;
			pBeamformingDb->policy = BF_POLICY_EXPLICIT;
		}
		pLaStaDb->laStaUspCommon.staTransmissionParams.staStbcSupport = pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_STBC_RX_FOR_LESS_THAN_OR_EQUAL_80MHZ;
		pLaStaDb->laStaUspCommon.staTransmissionParams.staStbcSupport160 = pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_STBC_RX_FOR_GREATER_THAN_80MHZ;	
	}
	else
#endif
	if (pLaStaDb->laStaUspCommon.staTransmissionParams.vhtSta == TRUE) //VHT station
	{
		if(vhtCapabilities.su_beam_formee_capable == TRUE)
		{
			pLaStaDb->laStaUspCommon.staTransmissionParams.staExplicitBfSupport = TRUE;
			pBeamformingDb->policy = BF_POLICY_EXPLICIT;
		}
		pLaStaDb->laStaUspCommon.staTransmissionParams.staStbcSupport = (vhtCapabilities.rx_stbc & STA_CAPABILITIES_VHT_STBC_SUPPORT_MASK);
#ifdef ENET_INC_ARCH_WAVE600
        if(vhtCapabilities.mu_beamformee_capable == TRUE)
        {
            pLaStaDb->laStationUnique.muGroupCount = TRUE; //In wave600 we use muGroupCount for VHT MU MIMO support
        }
#endif
	}
	else //HT station
	{
		if ((transmitBfCapabilities.receive_ndp_capable == TRUE) &&
			(transmitBfCapabilities.explicit_compressed_steering_capable == TRUE))
		{
			pLaStaDb->laStaUspCommon.staTransmissionParams.staExplicitBfSupport = TRUE;
			pBeamformingDb->policy = BF_POLICY_EXPLICIT;
		}
		pLaStaDb->laStaUspCommon.staTransmissionParams.staStbcSupport = (htCapbilities.rxStbc & STA_CAPABILITIES_HT_STBC_SUPPORT_MASK); 
	}
	/*Set working point first state: explicit BF if station support it*/
	if(BeamformingValidCheck(staIndex,estimatorsGetWorkingPointRateIndexOfMainBw(&(pLaStaDb->laStaUspCommon)),BF_STATE_EXPLICIT) == BF_MODE_VALID_AT_THE_REQUIRED_RATE)
	{
		/*Station support explicit BF*/
		pBeamformingDb->bfWorkingPointState = BF_STATE_EXPLICIT;
	}
	else
	{
		/*Station not support explicit BF - start with no BF*/
		pBeamformingDb->bfWorkingPointState = BF_STATE_NON_BF;
	}
	beamFormingSetBfInTcr(staIndex, FALSE, FALSE, pBeamformingDb->bfWorkingPointState, LA_PACKET_TYPE_DATA);
	beamFormingSetBfInTcr(staIndex, FALSE, FALSE, BF_STATE_NON_BF, LA_PACKET_TYPE_MANAGEMENT);

	/*Set first probing point BF state*/
	pBeamformingDb->bfProbingPointState = pBeamformingDb->bfWorkingPointState;
	/*Function will return the first probing mode which is not WP*/
	getNextBfState(staIndex, &(pBeamformingDb->bfProbingPointState),&BfProbeWithRatePlus1);

	/*Set static BF mode if request already sent before station has been connected*/
	bflocalDebugParams = getBfDebugParams();
	/*Disable Bf loop if BF activation bitmap == 0*/
	if (BeamFormingActivationBitMap.val == 0)
	{
		rateAdaptationEnableDisableStaSlowLoop(staIndex,SLOW_PROBING_BF,DISABLE_LOOP);
	}
	/*Enable bit is high & static mode requested for all connected stations or for the specific station conected now */
	if (bflocalDebugParams->staticBfEnable == TRUE)
	{
		BeamformingSetFixedMode(bfDebugParams.stationId);
	}
	
}
/**********************************************************************************

BeamformingRemoveStation 


Description:
------------
Change to no BF state in order to stop sending BF requests
Input: 
-----

Returns:
--------
	
**********************************************************************************/
void BeamformingRemoveStation(StaId staIndex)
{
	prepareNoBfMode(staIndex);
}
/**********************************************************************************

BeamformingSetNextProbingPoint 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingSetNextProbingPointInTcr(StaId staIndex)
{
	LaBeamformingDataBase_t* pBfDatabase = &LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb;
	uint8 workingPointRateindex;
	bool probeWithRatePlus1;

	probeWithRatePlus1 = LA_GET_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	workingPointRateindex = GetStaWpRateIndexFromHwTcr(staIndex,GetDataBwLimit(staIndex,INVALID_MU_USP_INDEX, FALSE));

	beamFormingSetBfInTcr(staIndex, TRUE, TRUE, pBfDatabase->bfProbingPointState, LA_PACKET_TYPE_DATA);

	/*If we finished to probe implicit mode with rate plus one we can change policy to explicit*/
	if ((pBfDatabase->bfProbingPointState == BF_STATE_IMPLICIT) && (probeWithRatePlus1 == TRUE) && (isExplicitValid(staIndex, workingPointRateindex) == BF_MODE_VALID_AT_THE_REQUIRED_RATE))
	{
		pBfDatabase->policy = BF_POLICY_EXPLICIT;
	}
	
	
}
/**********************************************************************************

BeamformingPrepareProbingPoint 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingPrepareNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t *pDistParam, bool stayWithLastProbingPoint)
{
	LaBeamformingDataBase_t* pBfDatabase = &(pDistParam->laStationUnique->beamFormingDb);
	BfValiationOptions_e stateValid = BF_MODE_NOT_VALID;
	uint8 rateIndex;
	bool probeWithRatePlusOne; 
	UNUSED_PARAM(stayWithLastProbingPoint);
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, pDistParam->stationOrGroupIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check

	if (probeWithRatePlusOne == FALSE)
	{
		rateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(&(LinkAdaptationStaDatabase[pDistParam->stationOrGroupIndex].laStaUspCommon));
		stateValid = BeamformingValidCheck(pDistParam->stationOrGroupIndex, rateIndex, (pBfDatabase->bfProbingPointState));
	}
	else
	{
		rateIndex = estimatorsGetHigherProbingRateIndex(pDistParam->stationOrGroupIndex);
		if (rateIndex!=INVALID_RATE_INDEX_VAL)
		{
			stateValid = BeamformingValidCheck(pDistParam->stationOrGroupIndex, rateIndex, (pBfDatabase->bfProbingPointState));
		}
	}
	if (stateValid == BF_MODE_VALID_AT_THE_REQUIRED_RATE) 
	{
		/*Change state to LA_PREPARE_SLOW_PROBING_TRNASMISSION*/
		laStateMachineChangeState(pDistParam, LA_PREPARE_SLOW_PROBING_TRNASMISSION);
		setCurrentSlowProbingTaskInDb(pDistParam,SLOW_PROBING_BF);
		
		bfPrepareBfTransFuncArr[pBfDatabase->bfProbingPointState](pDistParam->stationOrGroupIndex);
	}
	else
	{
		/*Clear probe with rate plus 1 indication*/
		LA_CLR_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, pDistParam->stationOrGroupIndex);
		/*When filter is not valid (can be if all aggregation failed), change to next probing point */
		getNextBfState(pDistParam->stationOrGroupIndex, &(pBfDatabase->bfProbingPointState),&BfProbeWithRatePlus1);
	}
}

/**********************************************************************************

BeamformingSetForceBfMode


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingSetFixedMode(StaId staId)
{
	LaBeamformingDataBase_t* pBfDb;
	uint8 workingPointRateindex;
	BfValiationOptions_e isBfValid;
	StaId nextSid = (staId == DB_INVALID_SID) ? LmStaDataBase.headIndexOfStaLinkList : staId;
	bool explicitRequestTimerActivate;
	uint8 numOfStaInTimer = 0;
	uint32 timerInterval = LinkAdaptationCommonConfiguration.bfRequestInterval;
	
//	ILOG0_DDDD("BeamformingSetFixedMode, staIndex = %d, bfMode = %d, periodicReq = %d, StaticPeriodicReq = %d", pDebugParams->stationId,pDebugParams->beamformingMode, pDebugParams->periodicBfReq, bfDebugParams.periodicBfReq);
	
	while (nextSid != DB_ASYNC_SID)
	{
		DEBUG_ASSERT(nextSid < HW_NUM_OF_STATIONS);
		pBfDb = &LinkAdaptationStaDatabase[nextSid].laStationUnique.beamFormingDb;
		if (bfDebugParams.staticBfEnable == TRUE)
		{
			pBfDb->cddCounter = 0;//reset the counter

			if(bfDebugParams.maxNssNdp != 0xF)
			{
				BeamformingSetDebugMaxNdp(nextSid);
			}
			if (bfDebugParams.beamformingMode == BF_STATE_IMPLICIT)
			{
				pBfDb->policy = BF_POLICY_IMPLICIT;
			}
			else
			{
				pBfDb->policy = BF_POLICY_EXPLICIT;
			}

			workingPointRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(&(LinkAdaptationStaDatabase[nextSid].laStaUspCommon));
			isBfValid = BeamformingValidCheck(nextSid, workingPointRateindex,(BeamformingMode_e)bfDebugParams.beamformingMode);
			if (isBfValid == BF_MODE_VALID_AT_THE_REQUIRED_RATE)
			{
				explicitRequestTimerActivate = LA_GET_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,nextSid, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
				pBfDb->bfWorkingPointState = (BeamformingMode_e)bfDebugParams.beamformingMode;
				rateAdaptationEnableDisableStaSlowLoop(nextSid,SLOW_PROBING_BF,DISABLE_LOOP);
				bfPrepareBfTransFuncArr[bfDebugParams.beamformingMode](nextSid);
				beamFormingSetBfInTcr(nextSid, FALSE, FALSE, pBfDb->bfWorkingPointState, LA_PACKET_TYPE_DATA);
				if ((explicitRequestTimerActivate == FALSE) && (bfDebugParams.periodicBfReq == TRUE))
				{
					
					numOfStaInTimer ++;
					LA_SET_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,nextSid, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
					/*Set timer for explicit report request*/
					TimerUtiltyAddEvent(timerInterval,isr_LinkAdaptation_InitBfReport, nextSid);
					if (numOfStaInTimer == BF_MAX_NUM_OF_STA_FOR_TIMER)
					{
						timerInterval += BF_TIMER_DIFF;
						numOfStaInTimer = 0;
					}
				}
			}
		}
		else
		{
			rateAdaptationEnableDisableStaSlowLoop(nextSid,SLOW_PROBING_BF,ENABLE_LOOP);
			pBfDb->cddCounter = CDD_MAX_NUMBER_OF_PERMUTATIONS-1;//reset the counter
		}
		/*If stationId == INVALID_STA_ID run over all connected stations, else set only required station*/
		if (staId == DB_INVALID_SID)
		{
			nextSid = StaDbSwEntries[nextSid].nextSid;
		}
		else
		{
			nextSid = DB_ASYNC_SID;
		}
	
	}
}


/**********************************************************************************

CyclicPrefixProcessFeedback 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
		const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
		StaId staIndex;
	LinkAdaptationConfigurationParams_t *pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	LaBeamformingDataBase_t* pBfDatabase = &(laDbDistributionParameter->laStationUnique->beamFormingDb);
	Bandwidth_e currentBw = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);
	SlowProbingPointEstimators_t* pSlowProbingEst; 
	FastProbingPointEstimators_t* pWorkingPointEst;
	int8 workingPointPer;
	uint8 slowProbingPointPer;
	uint8 currentRateindex;
	uint8 slowProbingRateIndex;
	bool probeWithRatePlusOne;
	uint16 losslessTpWp;
	uint16 losslessTpPp;
	uint16 effectiveTpWp;
	uint16 effectiveTpProbing;
	ASSERT(currentBw < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G

	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_BF);

	if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
	{
		return; 
	}
	staIndex = laDbDistributionParameter->stationOrGroupIndex;
	pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);
	pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter);
	slowProbingRateIndex = estimatorsGetSlowProbingRatePlusOne(laDbDistributionParameter);
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	
	if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff) &&
		AlphaFilter_isFilterResultValid(&pWorkingPointEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
	{

		workingPointPer = AlphaFilter_GetFilterResult(&pWorkingPointEst->averagePer);
		slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);
		workingPointPer = MAX(workingPointPer - WP_PER_BIAS, 0);
		/*Get current rate index*/
		currentRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);

		if (probeWithRatePlusOne == FALSE)
		{
			/*Instead of calculate the effective TP, calculate the success ratio instead of PER, so the comparition later will be correct*/
			effectiveTpWp = HUNDRED_PERCENT - workingPointPer;
			effectiveTpProbing = HUNDRED_PERCENT - slowProbingPointPer ;


		}
		else
		{
			/*Extract lossless TP according to bandwidth*/
			losslessTpWp = ratesTable[currentRateindex].rateTableBwParmas[currentBw].shortCpPhyRate;
			losslessTpPp = ratesTable[slowProbingRateIndex].rateTableBwParmas[currentBw].shortCpPhyRate;
			/*Calc effective TP for working point and probing point*/
			effectiveTpWp = estimatorsCalcEffectiveTp(losslessTpWp,workingPointPer);
			effectiveTpProbing = estimatorsCalcEffectiveTp(losslessTpPp,slowProbingPointPer);

		}
		
		if (pBfDatabase->bfProbingPointState == BF_STATE_EXPLICIT)
		{
			effectiveTpWp = ADD_PERCENT(effectiveTpWp,EXPLICIT_BF_OVERHEAD_COMPANSATION_PERCENT);
		}
		else if(pBfDatabase->bfWorkingPointState == BF_STATE_EXPLICIT)
		{
			effectiveTpProbing = ADD_PERCENT(effectiveTpProbing,EXPLICIT_BF_OVERHEAD_COMPANSATION_PERCENT);
		}
		/*Probing PER was better than WP PER*/
		if (effectiveTpProbing > effectiveTpWp)
		{
			/*Change working point to probing point in LA BF database, HW DB should stay with the last probing point because it was better */
			//	Note: in BF probing mode,  sender does not change probig point back to working point so the HW should stay with the probing point, so only LA DB should change*/
			setPolicyTimeout(staIndex, pBfDatabase->bfWorkingPointState,pBfDatabase->bfProbingPointState);
			
			if (pBfDatabase->bfProbingPointState == BF_STATE_EXPLICIT)
			{
				pBfDatabase->policy = BF_POLICY_EXPLICIT;
			}
			else if (pBfDatabase->bfProbingPointState == BF_STATE_IMPLICIT)
			{
				pBfDatabase->policy = BF_POLICY_IMPLICIT;
			}
			/*Change working point into probing point*/
			pBfDatabase->bfWorkingPointState = pBfDatabase->bfProbingPointState;
			/*Set new working point in TCR - this change is required for CDD and antenna selection params which changed together with BF at the probing point*/
			beamFormingSetBfInTcr(staIndex, FALSE, FALSE, pBfDatabase->bfWorkingPointState, LA_PACKET_TYPE_DATA);
			/*Get the next BF probing state*/
			getNextBfState(staIndex, &(pBfDatabase->bfProbingPointState),&BfProbeWithRatePlus1);
			/*Reset thresholds*/
			rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
			/*Reset estimators*/
			estimatorsResetRatesEstimators(laDbDistributionParameter);
			/*Reset fast probing counter so fast probing can collect statistics before next fast probing event*/	
			resetFastProbingCounters(laDbDistributionParameter);
			/*State changed - stop probing in cuurent cycle*/
			disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_BF);
			switchToHigherPriority(laDbDistributionParameter, SLOW_PROBING_BF); //after change wp switch to higher priority

		}
		else
		{
			/*Probing point did not improve TP*/
			if ((probeWithRatePlusOne == FALSE) &&
				(workingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh) &&
				(slowProbingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh))
			{
				/*If WP PER is lower than threshold, try to probe with higher rate in the next probing cycle*/
				LA_SET_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
			}
			else
			{
				/*WP does not improve PER -Sender will set TCR back to working point, we need only to restore RX definitions to the working point mode*/
				bfPrepareBfTransFuncArr[pBfDatabase->bfWorkingPointState](staIndex);
				/*Clear probe with rate plus one indication*/
				LA_CLR_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex);
				/*Change probing point to next point*/
				getNextBfState(staIndex, &(pBfDatabase->bfProbingPointState),&BfProbeWithRatePlus1);
				/*If working point has not changed, set non effective loop indication */
				updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_BF);
			}
		}
		
	}
	else
	{
		/*Clear probe with rate plus 1 indication*/
		LA_CLR_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex);
		/*When filter is not valid (can be if all aggregation failed), change to next probing point */
		getNextBfState(staIndex, &(pBfDatabase->bfProbingPointState),&BfProbeWithRatePlus1);
	}	
	return;
}
/**********************************************************************************

beamFormingSetBfInTcr



Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void beamFormingSetBfInTcr(StaId staIndex, bool enableRatePlusOne, bool isBfProbing, BeamformingMode_e bfState, LaPacketType_e packetType)
{
	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	const RateObj_t* ratesTable = getRatesTable(&pLinkAdaptationDb->laStaUspCommon);
	LaTcrModificationStruct_t tcrModificationParams;
	bool	probeWithRatePlus1 = FALSE ;
	uint8 rateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(&(pLinkAdaptationDb->laStaUspCommon));
	memset(&tcrModificationParams, 0, sizeof(LaTcrModificationStruct_t));
	
	/*Fill control parameters of tcrModificationParams*/
	tcrModificationParams.controlParams.staIndex = staIndex;
	tcrModificationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX;
	tcrModificationParams.controlParams.firstBwToUpdate = BANDWIDTH_TWENTY;
	tcrModificationParams.controlParams.lastBwToUpdate= LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	tcrModificationParams.controlParams.isVhtSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.vhtSta;
	tcrModificationParams.controlParams.changeProbingPoint=  isBfProbing; //BF always change the working point
	tcrModificationParams.controlParams.slowProbing = isBfProbing;
	tcrModificationParams.controlParams.packetType = packetType;
#ifdef ENET_INC_ARCH_WAVE600
	tcrModificationParams.controlParams.isHeGroup = FALSE;
#endif

	if(enableRatePlusOne == TRUE)
	{
		probeWithRatePlus1 = LA_GET_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check

	}
	
	tcrModificationParams.controlParams.changeToRatePlusOne =  probeWithRatePlus1;
	
	if ((isBfProbing == FALSE) && (ratesTable[rateIndex].laPhyMode != PHY_MODE_11B))
	{
		AntennaSelectionBfModeIsChanged(staIndex, bfState);
	}
	/*Set Beamforming and effected params in TCRs*/
	modifyStaTcrsParamsReq(&tcrModificationParams,bfSetProbingPointFuncArr[bfState]); 	

}

/**********************************************************************************

setExplicitMode


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void setExplicitMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	uint8 defaultAntSelection;
	bool explicitRequesttimerActivate;
#ifndef ENET_INC_ARCH_WAVE600
	uint8 cddDefaultCounter = CDD_DEFAULT_INDEX;
	uint8 antennaCount;
#endif
	Bandwidth_e bandwidth;
	explicitRequesttimerActivate = LA_GET_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,pModifcationParamsIn->controlParams.staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check

	ASSERT(pModifcationParamsIn->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	for (bandwidth = pModifcationParamsIn->controlParams.firstBwToUpdate; bandwidth <= pModifcationParamsIn->controlParams.lastBwToUpdate; bandwidth++)
	{
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].tcrBfMode = TX_BF_BEAMFORMING;
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].BfSmothing = 0;
	}
	
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrStbcMode = 0;
	
	defaultAntSelection = AntennaSelectionGetDefaultBitmap();
	/*Change maximum antennas*/
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrAntSelection = defaultAntSelection;
#ifndef ENET_INC_ARCH_WAVE600
	if(BeamFormingActivationBitMap.bitFields.implicitPlusCdd == TRUE) 
	{
		/*Change CDD to fedault value, after it changed for implicit mode*/
		/*Get CDD value for probing together with BF implicit, and increment counter*/
		antennaCount = AntennaSelectionCalcAntennaCount(defaultAntSelection);
		if (antennaCount >= CDD_MIN_NUMBER_OF_ANTENNAS)
		{
			CddfillCddValues(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals), cddDefaultCounter, antennaCount);
		}
	}
#endif
	if (explicitRequesttimerActivate == FALSE) //Don't add timer event if station has already activate timer
	{
		LA_SET_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,pModifcationParamsIn->controlParams.staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check

		/*Set timer for explicit report request*/
		TimerUtiltyAddEvent(LinkAdaptationCommonConfiguration.bfRequestInterval,isr_LinkAdaptation_InitBfReport, pModifcationParamsIn->controlParams.staIndex);
	} 	
}
/**********************************************************************************

setImplicitMode

Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void setImplicitMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	uint8 defaultAntSelection = AntennaSelectionGetDefaultBitmap();
	Bandwidth_e bandwidthToUpdate = GetDataBwLimit(pModifcationParamsIn->controlParams.staIndex,INVALID_MU_USP_INDEX, FALSE);
#ifndef ENET_INC_ARCH_WAVE600
	LinkAdaptationStaDatabase_t* pLaDB = &LinkAdaptationStaDatabase[pModifcationParamsIn->controlParams.staIndex];
	uint8 antennaCount;
#endif
	Bandwidth_e bandwidth;

	/*implicit should update only the current bw*/
	pModifcationParamsOut->controlParams.firstBwToUpdate = bandwidthToUpdate;
	pModifcationParamsOut->controlParams.lastBwToUpdate = bandwidthToUpdate;

	/*Change to maximum available antennas*/
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrAntSelection = defaultAntSelection;

	/*Set beamforming parameters*/
	ASSERT(pModifcationParamsIn->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	for (bandwidth = pModifcationParamsIn->controlParams.firstBwToUpdate; bandwidth <= pModifcationParamsIn->controlParams.lastBwToUpdate; bandwidth++)
	{
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].tcrBfMode = TX_BF_BEAMFORMING;
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].BfSmothing = 0;
	}

	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrStbcMode = 0;
	
#ifndef ENET_INC_ARCH_WAVE600
	/*Set CDD if required - activate in BeamFormingActivationBitMap and not changeToRatePlusOne (rate + 1 probing should  stay at the same CDD as the previous probing)*/
	if(BeamFormingActivationBitMap.bitFields.implicitPlusCdd == TRUE) 
	{
		antennaCount = AntennaSelectionCalcAntennaCount(defaultAntSelection);	
		if (antennaCount >= CDD_MIN_NUMBER_OF_ANTENNAS)
		{
			if((pModifcationParamsIn->controlParams.changeToRatePlusOne == FALSE) && 
				(pModifcationParamsIn->controlParams.changeProbingPoint == TRUE))
			{
				/*increment Cdd counter*/
				/*Get CDD value for probing together with BF implicit, and increment counter*/
				CddIncreaseIndexAndFillValues(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals),&(pLaDB->laStationUnique.beamFormingDb.cddCounter), pModifcationParamsIn->controlParams.staIndex, antennaCount, TRUE);
			}
			else
			{
				/*Get CDD according to last CDD counter val without increment counter*/
				CddfillCddValues(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals), pLaDB->laStationUnique.beamFormingDb.cddCounter, antennaCount);
			}
		}
	}	
#endif
}
/**********************************************************************************

setStbc1x2Mode

Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/


static void setStbc1x2Mode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	setStbcMode(pModifcationParamsIn, pModifcationParamsOut,STBC_1X2_NUM_OF_ANT, STBC_1X2_TCR_VAL,STBC_1X2_NUM_OF_NSS);
	
}
/**********************************************************************************

setStbc2x4Mode

Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void setStbc2x4Mode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	setStbcMode(pModifcationParamsIn, pModifcationParamsOut,STBC_2X4_NUM_OF_ANT, STBC_2X4_TCR_VAL,STBC_2X4_NUM_OF_NSS);

}
/**********************************************************************************

setStbcMode


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void setStbcMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut,uint8 stbcNumOfAnt, uint8 stbcTcrVal, uint8 stbcNumOfNss)
{
	uint8 modifiedAntennaSelection = AntennaSelectionGetLowerAntSelection(stbcNumOfAnt);
	LinkAdaptationStaDatabase_t* pLaStaDb = &LinkAdaptationStaDatabase[pModifcationParamsIn->controlParams.staIndex];
	Bandwidth_e maxBandwidth = GetDataBwLimit(pModifcationParamsIn->controlParams.staIndex,INVALID_MU_USP_INDEX, FALSE);
	uint8 lowerBandwidth = ((uint8)maxBandwidth - 1); //We dont need to check whether it above zero because we will check it on the while
	uint8 currentRateIndex = GetStaWpRateIndexFromHwTcr(pModifcationParamsIn->controlParams.staIndex,maxBandwidth);
	RateMask_t tempRaMask;
	uint8 lowerBwRateindex;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter;
#ifndef ENET_INC_ARCH_WAVE600	
	uint8 cddDefaultCounter = CDD_DEFAULT_INDEX;
	uint8 antennaCount;
#endif
	RateMask_t* enableNssMaskArr = enableHtVhtNssRatesInMask;

	updateLaDbDistributionParam(&laDbDistributionParameter, pModifcationParamsIn->controlParams.staIndex, INVALID_MU_USP_INDEX, FALSE);
#ifdef ENET_INC_ARCH_WAVE600
		if (laDbDistributionParameter.laStaUspCommon->staTransmissionParams.heSta == TRUE)
		{
			enableNssMaskArr = enableHeNssRatesInMask;
		}
#endif

	/*
	STBC mode will be set only when NSS num at the data BW limit will be matched to STBC mode requirements, set lower BWs to NSS num that will be match to STBC mode
	When BW become below zero it will be grater than maxBandwidth (because var type is  uint) so while condition will no match
	*/
	while (lowerBandwidth < maxBandwidth) 
	{
		ASSERT(lowerBandwidth < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
		MEMCPY (&tempRaMask, &(pLaStaDb->laStaUspCommon.raIndexMask), sizeof(RateMask_t));
		/*AND between enableNssRatesInMask to RA Mask*/
		andOperator64Bit(tempRaMask.raIndexMask64bit, enableNssMaskArr[stbcNumOfNss - 1].raIndexMask64bit,tempRaMask.raIndexMask64bit);
		lowerBwRateindex = rateAdaptationGetLowerRateIndex(&laDbDistributionParameter, (Bandwidth_e)lowerBandwidth,currentRateIndex,tempRaMask,0, FALSE);
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[lowerBandwidth].BfSmothing = 1;
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[lowerBandwidth].tcrBfMode = TX_BF_FLAT;
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[lowerBandwidth].rateindex = lowerBwRateindex;
		lowerBandwidth--;
	}
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrStbcMode = stbcTcrVal;
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrAntSelection = modifiedAntennaSelection;
#ifndef ENET_INC_ARCH_WAVE600	
	if(BeamFormingActivationBitMap.bitFields.implicitPlusCdd == TRUE) 
	{
		/*Get CDD value for probing together with BF implicit, and increment counter*/
		antennaCount = AntennaSelectionCalcAntennaCount(modifiedAntennaSelection);
		if (antennaCount >= CDD_MIN_NUMBER_OF_ANTENNAS)
		{
			CddfillCddValues(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals), cddDefaultCounter, antennaCount);
		}
	}
#endif
}
/**********************************************************************************

setNoBfMode


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void setNoBfMode(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	/*If NSS grater than number of antennas, set antennas to the maximum availabe*/
	uint8 defaultAntSelection = AntennaSelectionGetDefaultBitmap();
	Bandwidth_e bandwidth;
#ifndef ENET_INC_ARCH_WAVE600
	uint8 cddTempCounter = CDD_DEFAULT_INDEX;
	uint8 antennaCount;
	if(BeamFormingActivationBitMap.bitFields.implicitPlusCdd == TRUE) 
	{
		/*Get CDD value for probing together with BF implicit, and increment counter*/
		antennaCount = AntennaSelectionCalcAntennaCount(defaultAntSelection);
		if (antennaCount >= CDD_MIN_NUMBER_OF_ANTENNAS)
		{
			CddfillCddValues(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals), cddTempCounter, antennaCount);
		}
	}
#endif
	ASSERT(pModifcationParamsIn->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	for (bandwidth = pModifcationParamsIn->controlParams.firstBwToUpdate; bandwidth <= pModifcationParamsIn->controlParams.lastBwToUpdate; bandwidth++)
	{
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].tcrBfMode = TX_BF_FLAT;
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].BfSmothing = 1;
	}
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrStbcMode = 0;

	/*Change back to maximum antennas*/
	pModifcationParamsOut->tcrParams.tcrGeneralVals.tcrAntSelection = defaultAntSelection;

}

/**********************************************************************************

BeamFormingCheckReadyForTransmission


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
bool BeamFormingCheckReadyForTransmission(StaId staIndex)
{
	LaBeamformingDataBase_t* pBfDatabase = &LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb;

	return isBeamFormingReadyForTransmit(staIndex, pBfDatabase->bfProbingPointState);
}
/**********************************************************************************

handelExplicitBfReportTimerEvent



Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void handelExplicitBfReportTimerEvent(K_MSG *psMsg)
{
	LaTimerEventsTimerEvent_t* pTimerEventParams = (LaTimerEventsTimerEvent_t*)pK_MSG_DATA(psMsg);
	StaId staIndex = pTimerEventParams->staIndex;
	LaBeamformingDataBase_t* pBfDatabase = &LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb;
	uint32 timeToGo = 0;
	uint32 tsfLow = GET_TSF_TIMER_LOW();
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_V("linkAdaptationHandleAgingTimerEvent - handelExplicitBfReportTimerEvent");
#endif

	LA_CLR_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,staIndex);

	/*We shouldn't handle timer if no station connect or state is not Explict BF*/
	if((StaDbSwEntries[staIndex].state==STA_STATE_CONNECTED) && (pBfDatabase->bfWorkingPointState==BF_STATE_EXPLICIT) 
		&& ((bfDebugParams.periodicBfReq == TRUE) || (bfDebugParams.staticBfEnable == FALSE)))
	{
		timeToGo = calculateTimerExpirationTime(LinkAdaptationCommonConfiguration.bfRequestInterval,tsfLow, pTimerEventParams->timerExpirationTsf );
		/*Send BF request*/
		sendBfRequest(staIndex);
		/*Set timer for the next report*/
		TimerUtiltyAddEvent(timeToGo,isr_LinkAdaptation_InitBfReport, staIndex);
		
		LA_SET_BIT_IN_BITMAP(BfExplicitBfRequestTimerActive.staBitMap,staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
//		ILOG0_D("handelExplicitBfReportTimerEvent, timeToGo = %d", timeToGo);
	}

}
/**********************************************************************************

setBfDebugParams




Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void setBfDebugParams(LaFixedBeamformingParams_t* pBfFixedParams)
{
	MEMCPY (&bfDebugParams, pBfFixedParams, sizeof(LaFixedBeamformingParams_t));
 //	ILOG0_DD("setBfDebugParams, debugMaxNssNdp= %d %d", bfDebugParams.maxNssNdp, bfDebugParams.maxNssNdp);
}
/**********************************************************************************

getBfDebugParams




Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

LaFixedBeamformingParams_t* getBfDebugParams(void)
{
	return &bfDebugParams;
}
/**********************************************************************************

getNextBfState

Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
static BfValiationOptions_e getNextBfState(StaId staIndex, BeamformingMode_e* pNextBfMode, StationBitmap_t* pProbeWithHigherRate)
{
	BfValiationOptions_e stateValid = BF_MODE_NOT_VALID;
	uint8 numberOfStatesChecked = 0;
	uint8 workingPointRateindex = GetStaWpRateIndexFromHwTcr(staIndex,GetDataBwLimit(staIndex,INVALID_MU_USP_INDEX, FALSE));
	LaBeamformingDataBase_t* pLaBfDb = &LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb;
	
	while (((stateValid == BF_MODE_NOT_VALID) || (*pNextBfMode == pLaBfDb->bfWorkingPointState)) && (numberOfStatesChecked < BF_NUMBER_OF_MODES))
	{
		(*pNextBfMode)++;
		if ((*pNextBfMode) > BF_LAST_STATE)
		{
			(*pNextBfMode) = BF_FIRST_STATE;
		}
		stateValid = BeamformingValidCheck(staIndex, workingPointRateindex, (*pNextBfMode));
		numberOfStatesChecked++;
	}
	if (stateValid == BF_MODE_VALID_AT_HIGER_RATE)
	{
		LA_SET_BIT_IN_BITMAP(pProbeWithHigherRate->staBitMap,staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	}
	else
	{
		LA_CLR_BIT_IN_BITMAP_WITH_CHK(pProbeWithHigherRate->staBitMap,staIndex,SIZE_OF_STATIONS_BITMAP_IN_WORDS);
	}
	if (stateValid == BF_MODE_NOT_VALID)
	{
		*pNextBfMode = BF_STATE_NON_BF;
		stateValid = BF_MODE_VALID_AT_THE_REQUIRED_RATE;
	}

	return stateValid;
}
/**********************************************************************************

isExplicitValid


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static BfValiationOptions_e isExplicitValid(StaId staIndex, uint8 rateIndex)
{
	BfValiationOptions_e retVal = BF_MODE_NOT_VALID;
	LinkAdaptationStaDatabase_t* pLaStaDb = &(LinkAdaptationStaDatabase[staIndex]);
	UNUSED_PARAM(rateIndex);
	if((BeamFormingActivationBitMap.bitFields.bfExplicit == TRUE) &&
	 			(pLaStaDb->laStaUspCommon.staTransmissionParams.staExplicitBfSupport == TRUE) &&
	 			(BfCalibrationPass.bfExplicitCalibrationPass == TRUE))
	{
		retVal = BF_MODE_VALID_AT_THE_REQUIRED_RATE;
	}

	return retVal;
}
/**********************************************************************************

isImplicitValid


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static BfValiationOptions_e isImplicitValid(StaId staIndex, uint8 rateIndex)
{
	BfValiationOptions_e retVal = BF_MODE_NOT_VALID;
	LinkAdaptationStaDatabase_t* pLaStaDb = &(LinkAdaptationStaDatabase[staIndex]);
	const RateObj_t* ratesTable = getRatesTable(&pLaStaDb->laStaUspCommon);
	uint8 wpNumberOfNss = ratesTable[rateIndex].numberOfNss;

	if(((BeamFormingActivationBitMap.bitFields.implicitBf) || (BeamFormingActivationBitMap.bitFields.implicitPlusCdd)) && 
				(pLaStaDb->laStationUnique.beamFormingDb.policy == BF_POLICY_IMPLICIT) && 
				(BfCalibrationPass.bfImplicitCalibrationPass == TRUE) &&
				(wpNumberOfNss <= (IMLICIT_MAX_NUMBER_OF_NSS-1) ))
	 {
		retVal = BF_MODE_VALID_AT_THE_REQUIRED_RATE;
	 }

	return retVal;
}
/**********************************************************************************

isStbcValid



Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static BfValiationOptions_e isStbcValid(StaId staIndex, uint8 rateIndex,uint8 stbcNumOfAnt, uint8 stbcStaSupport, uint8 stbcNumOfNss, bool activationBitmapVal)
{
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	BfValiationOptions_e retVal = BF_MODE_NOT_VALID;
	uint8 numberOfNss;
	uint8 activatedAntBitmap = AntennaSelectionGetDefaultBitmap();
	uint8 maxAntCount = AntennaSelectionCalcAntennaCount(activatedAntBitmap);
	uint8 higherRateIndex;
	uint8 staStbcSupport = LinkAdaptationStaDatabase[staIndex].laStaUspCommon.staTransmissionParams.staStbcSupport;
	
#ifdef ENET_INC_ARCH_WAVE600
	if (GetDataBwLimit(staIndex, INVALID_MU_USP_INDEX, FALSE) > BANDWIDTH_EIGHTY)
	{
		staStbcSupport = LinkAdaptationStaDatabase[staIndex].laStaUspCommon.staTransmissionParams.staStbcSupport160;
	}
#endif	
		if((activationBitmapVal == TRUE) && (staStbcSupport == stbcStaSupport) && (maxAntCount >= stbcNumOfAnt))
		{
			numberOfNss = ratesTable[rateIndex].numberOfNss;
			/*Activate only for 1 NSS*/
			if (numberOfNss == (stbcNumOfNss - 1)) 
			{
				retVal = BF_MODE_VALID_AT_THE_REQUIRED_RATE;
			}
			else
			{
				higherRateIndex = estimatorsGetHigherProbingRateIndex(staIndex);
				if (higherRateIndex != INVALID_RATE_INDEX_VAL)
				{
					numberOfNss = ratesTable[higherRateIndex].numberOfNss;
					if (numberOfNss == (stbcNumOfNss - 1)) 
					{
						retVal = BF_MODE_VALID_AT_HIGER_RATE;
					}
				}
			}
		}

	return retVal;

}

/**********************************************************************************

isStbc1x2Valid


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static BfValiationOptions_e isStbc1x2Valid(StaId staIndex, uint8 rateIndex)
{
	BfValiationOptions_e retVal;

	retVal = isStbcValid(staIndex, rateIndex,STBC_1X2_NUM_OF_ANT,STBC_1X2_TRANSMISSION,STBC_1X2_NUM_OF_NSS,BeamFormingActivationBitMap.bitFields.stbc1x2);

	return retVal;
}
/**********************************************************************************

isStbc2x4Valid


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static BfValiationOptions_e isStbc2x4Valid(StaId staIndex, uint8 rateIndex)
{
	BfValiationOptions_e retVal;

	retVal = isStbcValid(staIndex, rateIndex,STBC_2X4_NUM_OF_ANT,STBC_2X4_TRANSMISSION,STBC_2X4_NUM_OF_NSS,BeamFormingActivationBitMap.bitFields.stbc2x4);

	return retVal;
}
/**********************************************************************************

isNoBfValid


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static BfValiationOptions_e isNoBfValid(StaId staIndex, uint8 rateIndex)
{
	UNUSED_PARAM(staIndex);	
	UNUSED_PARAM(rateIndex);	
	return BF_MODE_VALID_AT_THE_REQUIRED_RATE;
}
/**********************************************************************************

BeamformingValidCheck



Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
BfValiationOptions_e BeamformingValidCheck(StaId staIndex, uint8 rateIndex, BeamformingMode_e bfRequiredState)
{
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	BfValiationOptions_e retVal = BF_MODE_NOT_VALID;
	
	/*Return validation parametr according to BF mode, if phy mode is not HT / VHT, no BF mode  is activate so return BF_MODE_NOT_VALID */
	if (bfRequiredState == BF_STATE_NON_BF)
	{
		retVal = BF_MODE_VALID_AT_THE_REQUIRED_RATE;
	}
	else if ((ratesTable[rateIndex].laPhyMode == LA_PHY_MODE_HT_VHT) || (ratesTable[rateIndex].laPhyMode == LA_PHY_MODE_HE))
	{
		retVal = isBfModeValidFuncArr[bfRequiredState](staIndex, rateIndex);
	}

	return retVal;
}

/**********************************************************************************

setPolicyTimeout




Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void setPolicyTimeout(StaId staIndex, BeamformingMode_e lastState, BeamformingMode_e newState)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	LaBeamformingDataBase_t* pBeamformingDB = &(pLaDatabase->laStationUnique.beamFormingDb);
	uint32 currentTsf;
	
	currentTsf = GET_TSF_TIMER_LOW();
	currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);

	if ((pBeamformingDB->policy == BF_POLICY_EXPLICIT)&&(lastState == BF_STATE_EXPLICIT) && (newState != BF_STATE_EXPLICIT))
	{
		/*State changed, update raEnterStateTime*/
		pBeamformingDB->changeToNonExplicitStateTsf = currentTsf;
		pBeamformingDB->numberOfMpdusSentInCurrentState = 0;
	}
	
}
/**********************************************************************************

handleBfPolicy




Description:
------------
Function check the conditions for swithcing from explicit to implicit policy, Change from explicit to implicit will be done after setting impllicit state one time
Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void handleBfPolicy(StaId staIndex)
{
	
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	LaBeamformingDataBase_t* pBeamformingDB = &(pLaDatabase->laStationUnique.beamFormingDb);
	LinkAdaptationConfigurationParams_t* pConfigurationParams = pLaDatabase->laStaGroupCommon.pLinkAdaptationConfigurationParams;
	int32 filterTimeOffset;
	uint32 currentTsf;
	

	/*Check conditions from swithcing for explicit to implicit policy. Change from explicit to implicit will be done after setting impllicit state one time*/
	if ((pBeamformingDB->policy == BF_POLICY_EXPLICIT) &&(pBeamformingDB->bfWorkingPointState!=BF_STATE_EXPLICIT)) 
	{
		currentTsf = GET_TSF_TIMER_LOW();
		currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);
		filterTimeOffset = cyclicMinusOperator(currentTsf, pBeamformingDB->changeToNonExplicitStateTsf,TRUE, TSF_RANGE_ALPHA_FILTER); 

		if ((filterTimeOffset >= pConfigurationParams->bfChangeExplicitToNonExplicitTo) && 
			(pBeamformingDB->numberOfMpdusSentInCurrentState >=pConfigurationParams->bfNumOfPacketTransInCurrentState) && 
			(BfCalibrationPass.bfImplicitCalibrationPass == TRUE))
		{
			/*Explicit  state is not the WP state more than timeout, change to BF_POLICY_IMPLICIT, in implicit mode we still try explicit so if we are in probe with rate plus one we finished the probing and than switch to implicit*/
			pBeamformingDB->policy = BF_POLICY_IMPLICIT;
		}
	}
}

/**********************************************************************************

handleBfPolicy



Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void BeamformingRateChagedHadnling(StaId staIndex)
{
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	uint8 rateIndex;
	/*Check working point rate at BW 20 MHz because if phy mode different than HT /VHT it will be effected on BW 20*/
	rateIndex = GetStaWpRateIndexFromHwTcr(staIndex,BANDWIDTH_TWENTY);

	/*Enable BF loop only when working point phy mode is HT / VHT*/
	if ((ratesTable[rateIndex].laPhyMode != LA_PHY_MODE_HT_VHT) && (ratesTable[rateIndex].laPhyMode != LA_PHY_MODE_HE))
	{
		rateAdaptationEnableDisableStaSlowLoop(staIndex,SLOW_PROBING_BF,DISABLE_LOOP);
	}
	else if (bfDebugParams.staticBfEnable != TRUE)
	{
		rateAdaptationEnableDisableStaSlowLoop(staIndex,SLOW_PROBING_BF,ENABLE_LOOP);
	}
	else if (bfDebugParams.staticBfEnable == TRUE)
	{
		BeamformingSetFixedMode(staIndex);
	}
}
/**********************************************************************************

Beamforming_AgingEvent




Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void Beamforming_AgingEvent(StaId staIndex)
{
	handleBfPolicy(staIndex);
}
/**********************************************************************************

BeamFormingUpdateCalStatus





Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamFormingUpdateCalStatus(bool bfExplicitCalPass, bool bfImplicitCalPass)
{
	BfCalibrationPass.bfExplicitCalibrationPass = (bfExplicitCalPass | (!BeamFormingActivationBitMap.bitFields.explicitBfCalibrationRequired));
	BfCalibrationPass.bfImplicitCalibrationPass = (bfImplicitCalPass | (!BeamFormingActivationBitMap.bitFields.implicitBfCalibrationRequired)); 
}
/**********************************************************************************

BeamformingSetActivationBitmap






Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingSetActivationBitmap(BeamFormingSupportedModes_t bfActivationBitmap)
{
	MEMCPY (&BeamFormingActivationBitMap, &bfActivationBitmap, sizeof(BeamFormingSupportedModes_t));

	/*Update calibration status according to relevant bit in activation bitmap*/
	BeamFormingUpdateCalStatus(!(bfActivationBitmap.bitFields.explicitBfCalibrationRequired), !(bfActivationBitmap.bitFields.implicitBfCalibrationRequired));
}
/**********************************************************************************

BeamformingStateTo



Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingStateTo(StaId staIndex)
{
	LaBeamformingDataBase_t* pBfDatabase = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb);
	uint8 workingPointRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(&(LinkAdaptationStaDatabase[staIndex].laStaUspCommon));
	/*Stop send BF request if state TO*/
	if (pBfDatabase->bfProbingPointState == BF_STATE_EXPLICIT)
	{
		stopSendBfRequest(staIndex);
	}
	/*Clear probe with rate plus 1 indication*/
	LA_CLR_BIT_IN_BITMAP(BfProbeWithRatePlus1.staBitMap, staIndex);
	/*When filter is not valid (can be if all aggregation failed), change to next probing point */
	getNextBfState(staIndex, &(pBfDatabase->bfProbingPointState),&BfProbeWithRatePlus1);
	/*Change back to exlpicit policy if valid*/
	if ((isExplicitValid(staIndex, workingPointRateindex) == BF_MODE_VALID_AT_THE_REQUIRED_RATE))
	{
		pBfDatabase->policy = BF_POLICY_EXPLICIT;
	}

}
/**********************************************************************************

BeamformingResetWorkingPointState




Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingResetWorkingPointState(StaId staIndex)
{
	LaBeamformingDataBase_t* pBfDatabase = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb);

	pBfDatabase->bfWorkingPointState = BF_STATE_NON_BF;
	BeamformingStateTo(staIndex);
}

/**********************************************************************************

BeamformingResetDb





Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void BeamformingResetDb(StaId staIndex)
{
	LaBeamformingDataBase_t* pBfDatabase = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.beamFormingDb);

	pBfDatabase->bfWorkingPointState = BF_STATE_NON_BF;
}






