/***********************************************************************************
 File:		BwAdaptation.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "ProtectionAdaptation.h"
#include "HdkGlobalDefs.h"
#include "lm_StaDatabase.h"
#include "mt_sysdefs.h"
#include "lm_StaDatabase.h"
#include "ShramStationDatabase.h"
#include "LinkAdaptation_StateMachine.h"
#include "Pac_Api.h"
#include "Estimators.h"
#include "BwAdaptation.h"
#include "RateAdaptation.h"
#include "ShramGroupDatabase.h"
#include "CommonRamLinkAdaptation.h"
#include "CyclicPrefix.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 3


//#define BW_ADAPTATION_DEBUG


#define BW_ADAPTATION_EFFECTIVE40MHZ_NEGATIVE_THRESHOLD				27
#define BW_ADAPTATION_EFFECTIVE40MHZ_POSITIVE_THRESHOLD				35
#define BW_ADAPTATION_EFFECTIVE80MHZ_NEGATIVE_THRESHOLD				60
#define BW_ADAPTATION_EFFECTIVE80MHZ_POSITIVE_THRESHOLD				67
#define BW_ADAPTATION_EFFECTIVE80MHZ_VERY_NEGATIVE_THRESHOLD		50

#define BW_ADAPTATION_EFFECTIVE160MHZ_POSITIVE_THRESHOLD			134
#define BW_ADAPTATION_EFFECTIVE160MHZ_NEGATIVE_THRESHOLD			120	
#define BW_ADAPTATION_EFFECTIVE160MHZ_VERY_NEGATIVE_THRESHOLD		100
#define BW_ADAPTATION_EFFECTIVE160MHZ_VERY_VERY_NEGATIVE_THRESHOLD	70	

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
#ifdef BW_ADAPTATION_ACTIVE
 //static wlanProtectionLimit_e wlanBwLimit;
static Bandwidth_e wlanBwLimit = BANDWIDTH_ONE_HUNDRED_SIXTY; 
static bool bwCCAAdaptationModuleEnable = FALSE;  
#endif 

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

AdaptiveAlphaFilterParams_t LinkAdaptationWlanEffectiveBW40;
uint16 WlanEffectiveBW40DeltaTAverage;
AdaptiveAlphaFilterParams_t LinkAdaptationWlanEffectiveBW80;
uint16 WlanEffectiveBW80DeltaTAverage;
#ifdef ENET_INC_ARCH_WAVE600
AdaptiveAlphaFilterParams_t LinkAdaptationWlanEffectiveBW160;
uint16 WlanEffectiveBW160DeltaTAverage;
#endif

/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

#ifdef BW_ADAPTATION_ACTIVE
static void bwAdaptationWlanSm(void); 
static void bwAdaptationStaHandler(StaId staId); 
#endif 
static Bandwidth_e setStaSupportedBW(HtCapabilitiesInfo_t htCapabilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities, LaStationUspCommonParams_t* pStaUspCommonDb);

/**********************************************************************************

BwAdaptationInit 


Description:		Initialize the module's VAP and Station SM 
------------

Input: None 
-----

Returns:	None 
--------
	
**********************************************************************************/

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void BwAdaptation_Init (void)
{

#ifdef	BW_ADAPTATION_ACTIVE
	bwCCAAdaptationModuleEnable = FALSE;
	AlphaFilter_Init(&LinkAdaptationWlanEffectiveBW40,&WlanEffectiveBW40DeltaTAverage);
	AlphaFilter_Init(&LinkAdaptationWlanEffectiveBW80,&WlanEffectiveBW80DeltaTAverage);
#ifdef ENET_INC_ARCH_WAVE600
	AlphaFilter_Init(&LinkAdaptationWlanEffectiveBW160,&WlanEffectiveBW160DeltaTAverage);
#endif

#endif 
}

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif



/**********************************************************************************

BwAdaptation_Handler 


Description: 	Called after estimators are updated and everithing is ready for VAP/STA/WLAN proteciton handling 
------------

Input: Station index and VAP index 
-----

Returns: None
--------
	
**********************************************************************************/
void BwAdaptation_Handler(StaId staIndex,uint8 uspIndex)
{
#ifdef	BW_ADAPTATION_ACTIVE
	ILOG0_DD("BwAdaptation_Handler, staIndex = %d, uspIndex= %d", staIndex, uspIndex);
	if(bwCCAAdaptationModuleEnable)
	{
		bwAdaptationWlanSm();
	}
	if(uspIndex == INVALID_MU_USP_INDEX)
	{
		bwAdaptationStaHandler(staIndex);
	}
#endif 
}



/**********************************************************************************

bwAdaptationWlanSm 


Description:	 The VAP SM entry point. Set the new state of the VAP according to the current state and the updated estimator parameters.
------------

Input:  Vap index, Forced protection method 
-----

Returns: None 
--------
	
**********************************************************************************/
#ifdef	BW_ADAPTATION_ACTIVE
static void bwAdaptationWlanSm(void)
{
	uint8 effectiveBw40 = AlphaFilter_GetFilterResult(&LinkAdaptationWlanEffectiveBW40);
	uint8 effectiveBw80 = AlphaFilter_GetFilterResult(&LinkAdaptationWlanEffectiveBW80);
	bool effectiveBw40isValid = AlphaFilter_isFilterResultValid(&LinkAdaptationWlanEffectiveBW40, FILTER_VALIDATION_TIME);
	bool effectiveBw80isValid = AlphaFilter_isFilterResultValid(&LinkAdaptationWlanEffectiveBW80, FILTER_VALIDATION_TIME);
#ifdef ENET_INC_ARCH_WAVE600
	uint8 effectiveBw160 = AlphaFilter_GetFilterResult(&LinkAdaptationWlanEffectiveBW160);
	bool effectiveBw160isValid = AlphaFilter_isFilterResultValid(&LinkAdaptationWlanEffectiveBW160, FILTER_VALIDATION_TIME);
#endif	
	switch (wlanBwLimit)  
	{
	case BANDWIDTH_TWENTY:
		if(effectiveBw40isValid && (effectiveBw40 > BW_ADAPTATION_EFFECTIVE40MHZ_POSITIVE_THRESHOLD)) /* positive threshold*/
		{
			wlanBwLimit = BANDWIDTH_FOURTY; 
		}
		break; 
	case BANDWIDTH_FOURTY:
		if(effectiveBw40isValid && (effectiveBw40 < BW_ADAPTATION_EFFECTIVE40MHZ_NEGATIVE_THRESHOLD)) /* Negative threshold*/
		{
			wlanBwLimit = BANDWIDTH_TWENTY; 
		}
		else if(effectiveBw80isValid && (effectiveBw80 > BW_ADAPTATION_EFFECTIVE80MHZ_POSITIVE_THRESHOLD))
		{
			wlanBwLimit = BANDWIDTH_EIGHTY; 
		}
		break; 
	case BANDWIDTH_EIGHTY: 
		if (effectiveBw80isValid)
		{
			if(effectiveBw80 < BW_ADAPTATION_EFFECTIVE80MHZ_VERY_NEGATIVE_THRESHOLD) /* very negative threshold*/
			{
				wlanBwLimit = BANDWIDTH_TWENTY; 
			}
			else if(effectiveBw80 < BW_ADAPTATION_EFFECTIVE80MHZ_NEGATIVE_THRESHOLD) /* negative threshold*/
			{
				wlanBwLimit = BANDWIDTH_FOURTY; 
			}
#ifdef ENET_INC_ARCH_WAVE600	
			else if(effectiveBw160 > BW_ADAPTATION_EFFECTIVE160MHZ_POSITIVE_THRESHOLD) /* positive threshold*/
			{
				wlanBwLimit = BANDWIDTH_ONE_HUNDRED_SIXTY; 
			}
#endif
		}
		break; 
#ifdef ENET_INC_ARCH_WAVE600	
	case BANDWIDTH_ONE_HUNDRED_SIXTY: 
		if (effectiveBw160isValid)
		{
			if(effectiveBw160 < BW_ADAPTATION_EFFECTIVE160MHZ_VERY_VERY_NEGATIVE_THRESHOLD) /* very very negative threshold*/
			{
				wlanBwLimit = BANDWIDTH_TWENTY; 
			}
			else if(effectiveBw160 < BW_ADAPTATION_EFFECTIVE160MHZ_VERY_NEGATIVE_THRESHOLD) /* very negative threshold*/
			{
				wlanBwLimit = BANDWIDTH_FOURTY; 
			}
			else if(effectiveBw160 < BW_ADAPTATION_EFFECTIVE160MHZ_NEGATIVE_THRESHOLD)		/* negative threshold*/
			{
				wlanBwLimit = BANDWIDTH_EIGHTY; 
			}
		}
		break; 
#endif
	default:
		DEBUG_ASSERT(0);
	}
//#ifdef BW_ADAPTATION_DEBUG
	ILOG0_D("bwAdaptationWlanSm, wlanBwLimit == %d ",wlanBwLimit); 
//#endif 
}
#endif /*BW_ADAPTATION_ACTIVE*/

/**********************************************************************************

bwAdaptationStaHandler 


Description:	 Update the current station limit according to the WLAN limit
------------

Input:  staId
-----

Returns: None 
--------
	
**********************************************************************************/
#ifdef	BW_ADAPTATION_ACTIVE
static void bwAdaptationStaHandler(StaId staId)
{
	StaDb_t* pStaDbHwEntry = NULL;
	LinkAdaptationStaDatabase_t* pStaLaDbPtr = NULL;
	PhyMode_e phyMode; 
	
	DEBUG_ASSERT(staId < HW_NUM_OF_STATIONS);

	pStaDbHwEntry = &(StaDbHwEntries[staId]);
	pStaLaDbPtr = &(LinkAdaptationStaDatabase[staId]);
#if defined(ENET_INC_ARCH_WAVE600)
	phyMode	= (PhyMode_e)pStaDbHwEntry->common.tcr0Data20MhzPhyMode;
#else
	phyMode = (PhyMode_e)pStaDbHwEntry->common.tcr020MhzPhyModeData; 
#endif  
	if(pStaDbHwEntry->common.dataBwLimit < wlanBwLimit)
	{
		if(pStaLaDbPtr->laStaGroupCommon.bwAdaptationDb.isBwDroppedDueToRA)
		{
			if((phyMode != PHY_MODE_11B)&&
				(LinkAdaptationCommonConfiguration.wlanBandwidthMax > pStaDbHwEntry->common.dataBwLimit))
			{
#ifdef BW_ADAPTATION_DEBUG
				//("bwAdaptationStaHandler, Start probing! StationBwLimit == %d wlanBwLimit == %d ",pStaDbHwEntry->dataBwLimit,wlanBwLimit); 
#endif 
				//Rate adaptation is responsible for the fact that STA BW is lower than WLAN BW - Start probing  
				rateAdaptationEnableDisableStaSlowLoop(staId,SLOW_PROBING_BW, ENABLE_LOOP);
			}
		}
		else
		{
			bwAdaptationSetStationBwLimit(staId,wlanBwLimit, TRUE);/* Set the station limit to the WLAN limit */
		}
	}
	else if(pStaDbHwEntry->common.dataBwLimit > wlanBwLimit) 
	{
#ifdef BW_ADAPTATION_DEBUG
		//("bwAdaptationStaHandler, Set Station limit to wlan!  StationBwLimit == %d wlanBwLimit == %d ",pStaDbHwEntry->dataBwLimit,wlanBwLimit);
#endif
#ifdef BW_ADAPTATION_DEBUG
		//("bwAdaptationStaHandler, StabwLimit was set to %d",pStaDbHwEntry->dataBwLimit);
#endif

		pStaLaDbPtr->laStaGroupCommon.bwAdaptationDb.isBwDroppedDueToRA = FALSE; // Mark that we BW dropped becouse of WLAN SM and not becouse of RA  
		bwAdaptationSetStationBwLimit(staId,wlanBwLimit, TRUE);/* Set the station limit to the WLAN limit */
		
	}
	/* In case stationBwLimit == wlanLimit  Do nothing */
}
#endif /*BW_ADAPTATION_ACTIVE*/

/**********************************************************************************

BwAdaptation_IsBwAdaptationActive


Description:	
------------

Input: 
-----

Returns:	None 
--------
	
**********************************************************************************/
bool BwAdaptation_IsBwAdaptationActive(void)
{
#ifdef	BW_ADAPTATION_ACTIVE
	return bwCCAAdaptationModuleEnable;
#else
	return FALSE; 
#endif 
}

/**********************************************************************************

BwSetNextProbingPoint 


Description:
------------


Input: 
-----
staIndex
stayWithLastProbingPoint - not used in this function 

Returns:
--------
	nonEffectiveLoop - if all options have been tried without change the working point return 1
	
**********************************************************************************/
void BwSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t *pDistParam, bool stayWithLastProbingPoint)
{
#ifdef	BW_ADAPTATION_ACTIVE
	StaDb_t* pStaDbHwEntry = NULL; //KW ignore (UNINIT.STACK.MIGHT) as staDbHwEntry is always initialized.
	SlowProbingPointEstimators_t* pSlowProbingEst;
	Bandwidth_e changedBw;
	bool isCurrentSlowProbingMaskEmpty;
	UNUSED_PARAM(stayWithLastProbingPoint);	
	pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(pDistParam);
	if(pDistParam->uspIndex == INVALID_MU_USP_INDEX) /* SU handling*/
	{
		pStaDbHwEntry = &(StaDbHwEntries[pDistParam->stationOrGroupIndex]);
		
		changedBw = (Bandwidth_e)pStaDbHwEntry->common.dataBwLimit;
	}
	else
	{
#ifdef ENET_INC_ARCH_WAVE600
		changedBw = (Bandwidth_e)(GroupDbHwEntries[pDistParam->stationOrGroupIndex].common.dataBwLimit);	
#else
		changedBw = (Bandwidth_e)(GroupDbHwEntries[pDistParam->stationOrGroupIndex].dataBwLimit);
#endif

	}

	/*First - Make sure that wlan limit is still higher than the Station, otherwise - probing is not needed*/
	if(changedBw < wlanBwLimit)
	{
#ifdef BW_ADAPTATION_DEBUG
		//("BwSetNextProbingPoint - Set the probing!");
#endif
		ILOG0_D("BwSetNextProbingPoint, changedBw = %d", changedBw);
		DEBUG_ASSERT(changedBw != BANDWIDTH_ONE_HUNDRED_SIXTY); //We must not get here with 160Mhz 
		if(changedBw < pDistParam->laStaGroupCommon->bwAdaptationDb.maxSupportedBw)
		{
			// When changedBw blocked with maxSupportedBw finish current probing cycle with the same BW
			// This can occur if other module changed the BW during the probing period.
			changedBw = (Bandwidth_e)(changedBw + 1);
		} 			 
		bwAdaptationSetBwLimit(pDistParam,(Bandwidth_e)(changedBw), FALSE);//20Mhz to 40Mhz and 40Mhz to 80Mhz
		laStateMachineChangeState(pDistParam,LA_WAIT_FOR_SLOW_PROBE_VALID);
		setCurrentSlowProbingTaskInDb(pDistParam,SLOW_PROBING_BW);
		/*Set BW in extimators DB - it will be used at estimatorsWorkingPointDbUpdate*/
		pSlowProbingEst->bandwidth = changedBw;
		/*Relevant only for Gen5*/
		if(pDistParam->uspIndex == INVALID_MU_USP_INDEX) /* SU handling*/
		{
			pStaDbHwEntry->common.altRateReasonSlowProbing = 1; 
		}
	}
	else
	{
		isCurrentSlowProbingMaskEmpty = disableTaskFromCurrentCycle(pDistParam, SLOW_PROBING_BW);
		if(isCurrentSlowProbingMaskEmpty == TRUE)
		{
 			//Don't use probing,   update the state of RA accodingly 
			laStateMachineChangeState(pDistParam,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
		}
	}
#else
	/*No probing should be done if BW loop is not activate*/
	FATAL("BwSetNextProbingPoint - No probing should be done if BW loop is not activate"); 
#endif

}

/**********************************************************************************

BwProcessFeedback 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void BwProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
#ifdef	BW_ADAPTATION_ACTIVE

	StaId staIndex;
	SlowProbingPointEstimators_t* pSlowProbingEst;
	FastProbingPointEstimators_t* pWorkingPointEst;
	uint8 workingPointPer;
	uint8 slowProbingPointPer;
	Bandwidth_e changedBwLimit = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);
	uint16 losslessTpWp;
	uint16 losslessTpPp;
	uint16 effectiveTpWp;
	uint16 effectiveTpProbing;
	uint8 higherBwRateIndx;
	uint8 currentBwRateIndx;
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);

#ifdef BW_ADAPTATION_DEBUG
	ILOG0_V("BwProcessFeedback");
#endif

	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_BW);

	if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
	{
		return;
	}
	staIndex = laDbDistributionParameter->stationOrGroupIndex;
	pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter); // We should compare to lower BW 
	pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);

	if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff) &&
		AlphaFilter_isFilterResultValid(&pWorkingPointEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
	{
		higherBwRateIndx = GetStaWpRateIndexFromHwTcr(staIndex,pSlowProbingEst->bandwidth);
		currentBwRateIndx = GetStaWpRateIndexFromHwTcr(staIndex,changedBwLimit);

		/*Extract lossless TP according to bandwidth*/
		losslessTpWp = getPhyRate(ratesTable, currentBwRateIndx, changedBwLimit, convertCpModeToCpType[laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp]);
		losslessTpPp = getPhyRate(ratesTable, higherBwRateIndx, pSlowProbingEst->bandwidth, convertCpModeToCpType[laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp]);

		workingPointPer = AlphaFilter_GetFilterResult(&pWorkingPointEst->averagePer);
		slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);

		/*Calc effective TP for working point and probing point*/
		effectiveTpWp = estimatorsCalcEffectiveTp(losslessTpWp,workingPointPer);
		effectiveTpProbing = estimatorsCalcEffectiveTp(losslessTpPp,slowProbingPointPer);

		if ((effectiveTpProbing >= effectiveTpWp) || (changedBwLimit == pSlowProbingEst->bandwidth))//if we probed same BW - disable loop anyway
		{
			/*Probing per was better than WP per, change to probing point BW and stop probing*/
#ifdef BW_ADAPTATION_DEBUG
			ILOG0_D("Probing per was better than WP per, change to probing point BW and stop probing bw %d", pSlowProbingEst->bandwidth);
#endif
			/*BW changed, stop probing in this cycle*/
			disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_BW);
			changedBwLimit = pSlowProbingEst->bandwidth;
			bwAdaptationSetBwLimit(laDbDistributionParameter,(Bandwidth_e)(changedBwLimit), TRUE); 
		    if((bwCCAAdaptationModuleEnable) || (changedBwLimit == laDbDistributionParameter->laStaGroupCommon->bwAdaptationDb.maxSupportedBw))
		    {
				laDbDistributionParameter->laStaGroupCommon->bwAdaptationDb.isBwDroppedDueToRA = FALSE; // Mark that we BW dropped because of WLAN SM and not because of RA  
				if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
				{
					rateAdaptationEnableDisableGroupSlowLoop(laDbDistributionParameter, SLOW_PROBING_BW, DISABLE_LOOP);
				}
				else
				{
					rateAdaptationEnableDisableStaSlowLoop(staIndex,SLOW_PROBING_BW, DISABLE_LOOP);
				}
			}
			changeTaskToDefaultPriority(laDbDistributionParameter, SLOW_PROBING_BW);
			resetNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_BW);
		}
		else
		{
			/*Probing was not improve results*/
			updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_BW);
		}
	}
	// In any case - working point mustn't be wider than wlan limit and dataBwLimit should be grater than 20 Mhz
	if (wlanBwLimit < changedBwLimit)
	{
		bwAdaptationSetBwLimit(laDbDistributionParameter, wlanBwLimit, TRUE);
		laDbDistributionParameter->laStaGroupCommon->bwAdaptationDb.isBwDroppedDueToRA = FALSE; // Mark that we BW dropped becouse of WLAN SM and not becouse of RA  
		if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
		{
			rateAdaptationEnableDisableGroupSlowLoop(laDbDistributionParameter, SLOW_PROBING_BW, DISABLE_LOOP);
		}
		else
		{
			rateAdaptationEnableDisableStaSlowLoop(staIndex,SLOW_PROBING_BW, DISABLE_LOOP);
		}
	}
	
#endif //BW_ADAPTATION_ACTIVE
	return;
}



/**********************************************************************************

bwAdaptationDisableBwDependenceOnCcA


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void BwAdaptationDisableBwDependenceOnCcA(void)
{
#ifdef BW_ADAPTATION_ACTIVE
	Pac_RxcDisableCcaIndication();
#endif /*BW_ADAPTATION_ACTIVE*/

}

 



/**********************************************************************************

BwAdaptation_AgingEvent


Description:	Handle aging event, check for each vap whether it has reach the aging limit
------------

Input: 
-----

Returns:	None 
--------
	
**********************************************************************************/

void BwAdaptation_AgingEvent(void)
{
#ifdef BW_ADAPTATION_ACTIVE

	AdaptiveAlphaFilterParams_t* pWlanEffectiveBW40 = &(LinkAdaptationWlanEffectiveBW40); 
	AdaptiveAlphaFilterParams_t* pWlanEffectiveBW80 = &(LinkAdaptationWlanEffectiveBW80); 	
#ifdef ENET_INC_ARCH_WAVE600
	AdaptiveAlphaFilterParams_t* pWlanEffectiveBW160 = &(LinkAdaptationWlanEffectiveBW160);  
#endif
	if(AlphaFilter_isFilterResultValid(pWlanEffectiveBW40,FILTER_VALIDATION_TIME) == FALSE)
	{
		AlphaFilter_Init(pWlanEffectiveBW40,&WlanEffectiveBW40DeltaTAverage);
	}
	if(AlphaFilter_isFilterResultValid(pWlanEffectiveBW80,FILTER_VALIDATION_TIME) == FALSE)
	{
		AlphaFilter_Init(pWlanEffectiveBW80,&WlanEffectiveBW80DeltaTAverage);
	}
#ifdef ENET_INC_ARCH_WAVE600	
	if(AlphaFilter_isFilterResultValid(pWlanEffectiveBW160,FILTER_VALIDATION_TIME) == FALSE)
	{
		AlphaFilter_Init(pWlanEffectiveBW160,&WlanEffectiveBW160DeltaTAverage);
	}
#endif
#endif
}

/**********************************************************************************

BwAdaptationAddSta




Description:	
------------

Input: 
-----

Returns:	None 
--------
	
**********************************************************************************/
void BwAdaptationAddSta(StaId staindex, HtCapabilitiesInfo_t htCapabilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities, LaOperationModeNotificationChannelWidth_e omnChannelWidth, uint8 initRate)
{
	BwStationDatabaseVars_t* pBwDatabase = &(LinkAdaptationStaDatabase[staindex].laStaGroupCommon.bwAdaptationDb);
	uint8 wpRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(&(LinkAdaptationStaDatabase[staindex].laStaUspCommon));
	Bandwidth_e bandwidthBeforeChange = GetDataBwLimit(staindex,INVALID_MU_USP_INDEX, FALSE);
	Bandwidth_e bandwidthAfterChange;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	uint8 newRate = initRate;
	
	updateLaDbDistributionParam(&laDbDistributionParameter,staindex,INVALID_MU_USP_INDEX, FALSE);

	pBwDatabase->isBwDroppedDueToRA = FALSE;
	pBwDatabase->staSupportedBw = setStaSupportedBW(htCapabilities, vhtCapabilities, pHeMacPhyCapabilities, laDbDistributionParameter.laStaUspCommon);
	
	/* Set BW limit according to phy mode and BW configuration*/
	initStaBwLimit(staindex,wpRateindex, omnChannelWidth, LinkAdaptationCommonConfiguration.wlanBandwidthMax); 
	ILOG0_V("BwAdaptationAddSta, disable loop");
	rateAdaptationEnableDisableStaSlowLoop(staindex,SLOW_PROBING_BW, DISABLE_LOOP);
	bandwidthAfterChange = GetDataBwLimit(staindex,INVALID_MU_USP_INDEX, FALSE);
	newRate = rateAdaptationGetLowerRateIndex(&laDbDistributionParameter, bandwidthAfterChange, initRate, laDbDistributionParameter.laStaUspCommon->raIndexMask,0, FALSE);
	if ((initRate != newRate) || (bandwidthBeforeChange != bandwidthAfterChange))
	{
		rateAdaptationChangeWorkingPoint(&laDbDistributionParameter, newRate, newRate);
	}
}

static Bandwidth_e setStaSupportedBW(HtCapabilitiesInfo_t htCapabilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities, LaStationUspCommonParams_t* pStaUspCommonDb)
{
	Bandwidth_e supportedBw = BANDWIDTH_TWENTY;
#ifdef ENET_INC_ARCH_WAVE600
	if (pStaUspCommonDb->staTransmissionParams.heSta == TRUE)
	{
		if (LinkAdaptationCommonConfiguration.band == BAND_2_4_GHZ)
		{
			if (pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_CHANNEL_WIDTH_SET_B0 == TRUE)
			{
				// Sta supports up to 40 MHz, 2.4GHz band
				supportedBw = BANDWIDTH_FOURTY;
			}
		}
		else
		{
			if (pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_CHANNEL_WIDTH_SET_B2 == TRUE)
			{
				// Sta supports up to 160 MHz, 5GHz band
				supportedBw = BANDWIDTH_ONE_HUNDRED_SIXTY;
			}
			else if (pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_CHANNEL_WIDTH_SET_B1 == TRUE)
			{
				// Sta supports up to 80 MHz, 5GHz band
				supportedBw = BANDWIDTH_EIGHTY;
			}
		}
	}
	else if (pStaUspCommonDb->staTransmissionParams.vhtSta == TRUE)
	{
		if (vhtCapabilities.supported_channel_width_set == 0)
		{
			supportedBw = BANDWIDTH_EIGHTY;
		}
		else
		{
			ASSERT(vhtCapabilities.supported_channel_width_set != 3);
			supportedBw = BANDWIDTH_ONE_HUNDRED_SIXTY;
		}
	}
	else
#endif
	{
		supportedBw = (Bandwidth_e)htCapabilities.supportedChannelWidthSet;
	}
	return supportedBw;
}

void BwAdaptationTo(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter)
{
	Bandwidth_e wpBwLimit;
	bool isHeGroup = (pLaDbDistributionParameter->laHeGroupUnique != NULL);
	
	wpBwLimit = GetWpDataBwLimit(pLaDbDistributionParameter->stationOrGroupIndex, pLaDbDistributionParameter->uspIndex, isHeGroup);
#ifdef BW_ADAPTATION_DEBUG
	ILOG0_DD("BwAdaptationTo. staIndex %d wpBwLimit %d",wpBwLimit, pLaDbDistributionParameter->stationOrGroupIndex);
#endif	
	/* Back to WP */
	bwAdaptationSetBwLimit(pLaDbDistributionParameter, wpBwLimit, TRUE);
}

/**********************************************************************************

BwAdaptationRateIsChanged 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void BwAdaptationRateIsChanged(StaId staIndex, uint8 oldRate, uint8 newRate)
{
	const RateObj_t* ratesTable;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter,staIndex,INVALID_MU_USP_INDEX, FALSE);
	
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);
	if ((ratesTable[oldRate].laPhyMode != ratesTable[newRate].laPhyMode) && (ratesTable[newRate].laPhyMode < LA_PHY_MODE_HT_VHT))
	{
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_BW, DISABLE_LOOP);
	}
}
