/***********************************************************************************
 File:			CDD.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "CDD.h"
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "HdkGlobalDefs.h"
#include "HwGlobalDefinitions.h"
#include "PSD.h"
#include "loggerAPI.h"
#include "AntennaSelection.h"
#include "RateAdaptation.h"
#include "LinkAdaptation_StateMachine.h"
#include "Estimators.h"
#include "CommonRamLinkAdaptation.h"
#include "ConfigurationManager_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 4


//#define CDD_ADAPTATION_DEBUG

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static bool setCddInStaTcr(StaId staIndex, bool changeProbingPoint, LaPacketType_e packetType, bool setRatePlusOne, CddValues_t cddVal);
static void fillCddFixedValues(CddValues_t* cddVal, uint16 cddPermutation);
static void  CddModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static uint16 getPermutation(uint8 cddIndex, uint8 antCount);
static bool isSameCddAsWorkingPoint(CddValues_t cddVal, StaId staIndex);
static void increaseCddIndex(uint8* cddIndex, StaId staIndex, uint8 antCount,bool allowNextCddSameAsWp);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static StationBitmap_t probeCddWithRatePlus1;
static CddValues_t cddValues;
static CddPermutationsDb_t cddPermutations = 
{
	{
		{0x4,	0x1,   0x2,   0x3,	 CDD_INVALID_PERMUTATION,  CDD_INVALID_PERMUTATION,  CDD_INVALID_PERMUTATION},
		{0x42,	0x21,  0x31,  0x41,  0x32, 					   0x43,					 CDD_INVALID_PERMUTATION},
		{0x321, 0x421, 0x431, 0x432, CDD_INVALID_PERMUTATION,  CDD_INVALID_PERMUTATION,  CDD_INVALID_PERMUTATION},
	}
	
};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif

void CddInit (void)
{
	CddDatabase_t* pCddDb;
	StaId staId;
	StaId firstSidInBand = ConfigurationManager_GetFirstSidInMyBand();
	uint16 numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsInMyBand();	
	

	for (staId = firstSidInBand; staId < (firstSidInBand + numOfSidsInBand); staId++)
	{
		pCddDb = &LinkAdaptationStaDatabase[staId].laStationUnique.cddDataBase;
		memset(pCddDb, 0, sizeof(CddDatabase_t));
	}
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

void CddAddSta(StaId staIndex)
{
	CddDatabase_t* pCddDb = &LinkAdaptationStaDatabase[staIndex].laStationUnique.cddDataBase;
	uint8 antCount = AntennaSelectionGetAntennaCount(staIndex);

	LA_CLR_BIT_IN_BITMAP(probeCddWithRatePlus1.staBitMap,staIndex);
	
	if (antCount >= CDD_MIN_NUMBER_OF_ANTENNAS)
	{
#ifdef CDD_ADAPTATION_DEBUG
		ILOG0_DD("CddAddSta: %d cddIndex: %d", staIndex, pCddDb->cddIndex);
#endif	
		// Fill CDD values for each antenna in CDD structure
		CddfillCddValues(&cddValues, pCddDb->cddIndex, antCount);

		// Set CDD in TCR from cddValues structure
		setCddInStaTcr(staIndex, FALSE, LA_PACKET_TYPE_DATA, FALSE, cddValues);
		
		// Increase index
		increaseCddIndex(&pCddDb->cddIndex, staIndex, antCount, FALSE);
	}	
	else
	{
		// Disable loop
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CDD, DISABLE_LOOP);
	}
}

void CddIncreaseIndexAndFillValues(CddValues_t* cddVal, uint8* cddIndex, StaId staIndex, uint8 antCount, bool allowNextCddSameAsWp)
{
	// Increase index
	increaseCddIndex(cddIndex, staIndex, antCount,allowNextCddSameAsWp);

	// Fill CDD values for each antenna in CDD structure
	CddfillCddValues(cddVal, *cddIndex, antCount);
}
void CddSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	CddDatabase_t* pCddDb = &(laDbDistributionParameter->laStationUnique->cddDataBase);
	uint8 antCount = AntennaSelectionGetAntennaCount(laDbDistributionParameter->stationOrGroupIndex);
	bool probeWithRatePlusOne;
	
	setCurrentSlowProbingTaskInDb(laDbDistributionParameter, SLOW_PROBING_CDD);

	/* Probe CDD permutation */
	
	// Fill CDD structure
	CddfillCddValues(&cddValues, pCddDb->cddIndex, antCount);
	
	// Set CDD in TCR
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(probeCddWithRatePlus1.staBitMap, laDbDistributionParameter->stationOrGroupIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	setCddInStaTcr(laDbDistributionParameter->stationOrGroupIndex, TRUE, LA_PACKET_TYPE_DATA, probeWithRatePlusOne, cddValues);
	
	laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_SLOW_PROBE_VALID);

}

void CddProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	StaId staIndex;
	LinkAdaptationConfigurationParams_t *pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	CddDatabase_t* pCddDb = &(laDbDistributionParameter->laStationUnique->cddDataBase);
	Bandwidth_e currentBw = CurrentPacketParams.bandwidth;
	SlowProbingPointEstimators_t* pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);
	FastProbingPointEstimators_t* pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter);
	int8 workingPointPer;
	uint8 slowProbingPointPer;
	bool probeWithRatePlusOne;
	uint8 slowProbingRateIndex;
	uint8 currentRateindex;
	uint8 antCount;
	uint16 losslessTpWp;
	uint16 losslessTpPp;
	uint16 effectiveTpWp;
	uint16 effectiveTpProbing;
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	
	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_CDD);

	if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
	{
		return; 
	}
	staIndex = laDbDistributionParameter->stationOrGroupIndex;
	slowProbingRateIndex = pWorkingPointEst->rateIndex;
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(probeCddWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	antCount = AntennaSelectionGetAntennaCount(staIndex);
	
	if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff) &&
		AlphaFilter_isFilterResultValid(&pWorkingPointEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
	{

		workingPointPer = AlphaFilter_GetFilterResult(&pWorkingPointEst->averagePer);
		slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);
		workingPointPer = MAX(workingPointPer - WP_PER_BIAS, 0);
		
#ifdef CDD_ADAPTATION_DEBUG	
		ILOG0_DD("CddProcessFeedback, workingPointPer = %d, slowProbingPointPer = %d", workingPointPer, slowProbingPointPer);
#endif
		/*Get current rate index*/
		currentRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);

		if (probeWithRatePlusOne == FALSE)
		{
			/*Instead of calculate the effective TP, calculate the success ratio instead of PER, so the comparition later will be correct*/
			effectiveTpWp = HUNDRED_PERCENT - workingPointPer;
			effectiveTpProbing = HUNDRED_PERCENT - slowProbingPointPer ;
		}
		else
		{
			ASSERT(currentBw < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
			/*Extract lossless TP according to bandwidth*/
			losslessTpWp = ratesTable[currentRateindex].rateTableBwParmas[currentBw].shortCpPhyRate;
			losslessTpPp = ratesTable[slowProbingRateIndex].rateTableBwParmas[currentBw].shortCpPhyRate;
			
			/*Calc effective TP for working point and probing point*/
			effectiveTpWp = estimatorsCalcEffectiveTp(losslessTpWp,workingPointPer);
			effectiveTpProbing = estimatorsCalcEffectiveTp(losslessTpPp,slowProbingPointPer);
		}
		/*probing per was better than WP per, and probing step != 0*/
		if (effectiveTpProbing > effectiveTpWp)
		{
			// PB is better than WP - change WP
#ifdef CDD_ADAPTATION_DEBUG
			ILOG0_V("PB is better than WP - change WP!");
#endif
			CddfillCddValues(&cddValues, pCddDb->cddIndex, antCount);
			setCddInStaTcr(staIndex, FALSE, LA_PACKET_TYPE_DATA, FALSE, cddValues);
			// Reset unsuccessful Probing Counter
			// Increase CDD index for the next probing
			increaseCddIndex(&pCddDb->cddIndex, staIndex, antCount, FALSE);
			/*Reset thresholds*/
			rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
			/*Reset estimators*/
			estimatorsResetRatesEstimators(laDbDistributionParameter);
			/*Reset fast probing counter so fast probing can collect statistics before next fast probing event*/	
			resetFastProbingCounters(laDbDistributionParameter);
			changeTaskToDefaultPriority(laDbDistributionParameter,SLOW_PROBING_CDD);
			resetNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_CDD);

		}
		else
		{
			/*Probing point did not improve the TP*/
			if ((probeWithRatePlusOne == FALSE) &&
				(workingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh) &&
				(slowProbingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh))
			{
				/*If WP PER is lower than threshold, try to probe with higher rate in the next probing cycle*/
				LA_SET_BIT_IN_BITMAP(probeCddWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
#ifdef CDD_ADAPTATION_DEBUG
				ILOG0_V("CddProcessFeedback - SET RATE + 1");
#endif
			}
			else
			{
				/*Clear probe with rate plus one indication*/
				LA_CLR_BIT_IN_BITMAP(probeCddWithRatePlus1.staBitMap, staIndex);
				
				// Get next CDD permutation for the next probing
				increaseCddIndex(&pCddDb->cddIndex, staIndex, antCount, FALSE);
				updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_CDD);
				
			}
		}
	}
	else
	{
		// Get next CDD permutation for the next probing
		increaseCddIndex(&pCddDb->cddIndex, staIndex,antCount, FALSE);
		
		/*Clear probe with rate plus one indication*/
		LA_CLR_BIT_IN_BITMAP(probeCddWithRatePlus1.staBitMap, staIndex);
	}
	return;
}
void CddStateIsChanged(StaId staIndex, LaTcrModificationStruct_t* tcrModicationParams)
{
	uint8 rateIndex, antennaCount, nss, stbc;
	LinkAdaptationPhyMode_e phyMode;
	Bandwidth_e bandwidth;
	bool validRateForCdd = TRUE;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);

	antennaCount = AntennaSelectionGetAntennaCount(staIndex);
	
#ifdef CDD_ADAPTATION_DEBUG
	SLOG0(0, 0, LaTcrModificationStruct_t, tcrModicationParams);
#endif

	for (bandwidth = BANDWIDTH_TWENTY; bandwidth < LA_NUM_OF_BANDWIDTH; bandwidth++)
	{
		rateIndex = tcrModicationParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex;
		phyMode = ratesTable[rateIndex].laPhyMode;
		nss = ratesTable[rateIndex].numberOfNss;
		
		if ((phyMode == LA_PHY_MODE_11B) || (nss > SPATIAL_STREAM_1))
		{
			// Invalid rate for CDD, for at least one of the BWs
			validRateForCdd = FALSE;
			break;
		}
	}
	
	stbc = tcrModicationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
	
#ifdef CDD_ADAPTATION_DEBUG
	ILOG0_DDD("validRate: %d antennaCount: %d stbc: %d", validRateForCdd, antennaCount, stbc);
#endif

	if ((validRateForCdd == TRUE) && (antennaCount >= CDD_MIN_NUMBER_OF_ANTENNAS) && (stbc == NOT_STBC_TRANSMISSION))
	{
		// All conditions for CDD are met, enable loop
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CDD, ENABLE_LOOP);
	}
	else
	{
		// Disable loop
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CDD, DISABLE_LOOP);
	}
}
void CddStaNumOfAntsIsChanged(StaId staIndex)
{
	CddDatabase_t* pCddDb = &(LinkAdaptationStaDatabase[staIndex].laStationUnique.cddDataBase);
	uint8 antCount = AntennaSelectionGetAntennaCount(staIndex);
	
#ifdef CDD_ADAPTATION_DEBUG
	ILOG0_D("CddStaNumOfAntsIsChanged: antCount %d", antCount);
#endif	
	if (antCount >= CDD_MIN_NUMBER_OF_ANTENNAS)
	{ 
		// Reset CDD index when number of active antennas is changed
		pCddDb->cddIndex = CDD_DEFAULT_INDEX;
		
		// Fill CDD values for each antenna in CDD structure
		CddfillCddValues(&cddValues, pCddDb->cddIndex, antCount);

		// Set CDD in TCR from cddValues structure
		setCddInStaTcr(staIndex, FALSE, LA_PACKET_TYPE_DATA, FALSE, cddValues);
		
		// Increase index
		increaseCddIndex(&pCddDb->cddIndex, staIndex, antCount, FALSE);
		
		// Enable loop
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CDD, ENABLE_LOOP);
	}
	else
	{
		// Disable loop
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CDD, DISABLE_LOOP);
	}
}
void CddVapNumOfAntsIsChanged(uint8 vapId, uint8 numOfAntennas)
{
	CddValues_t cddVals;
	
	CddfillCddValues(&cddVals, CDD_DEFAULT_INDEX, numOfAntennas);
	
	CddSetInVapTcr(vapId, &cddVals);
}

void CddsetFixedValuesInStaTcr(StaId staIndex, uint16 cddPermutation)
{
	// Fill CDD structure and set values in STA TCR
	fillCddFixedValues(&cddValues, cddPermutation);
	setCddInStaTcr(staIndex, FALSE, LA_PACKET_TYPE_DATA, FALSE, cddValues);
}

void CddSetDataFromPsd(CddPermutationsDb_t* cddPsdDb)
{
	memcpy(&cddPermutations, cddPsdDb, sizeof(CddPermutationsDb_t));
}

static void increaseCddIndex(uint8* cddIndex, StaId staIndex, uint8 antCount,bool allowNextCddSameAsWp)
{
	uint8 loopIndex;
	uint16 nextPermutation;
	CddValues_t cddVal;
	
	for (loopIndex = 0; loopIndex < (CDD_MAX_NUMBER_OF_PERMUTATIONS + 1); loopIndex++)
	{
		// Increase index
		(*cddIndex)++;
		
		// Get permutation from permutations DB
		nextPermutation = getPermutation(*cddIndex, antCount);
#ifdef CDD_ADAPTATION_DEBUG
		ILOG0_D("nextPermutation: 0x%x", nextPermutation);
#endif
		if (nextPermutation == CDD_INVALID_PERMUTATION)
		{
			// Last permutation in permutations group.  Reset CDD index
			*cddIndex = CDD_DEFAULT_INDEX;
		}
#ifdef CDD_ADAPTATION_DEBUG
		ILOG0_D("cddIndex: %d", *cddIndex);
#endif
		CddfillCddValues(&cddVal, *cddIndex, antCount);
		
		if ((isSameCddAsWorkingPoint(cddVal, staIndex) == FALSE) || (allowNextCddSameAsWp == TRUE))
		{
			// CDD permutation is different from the one in TCR and is valid for probing
			break;
		}
	}
}

void CddAutoReplyNumOfAntsIsChanged(uint8 numOfAntennas)
{
	CddValues_t cddVals;
	
	CddfillCddValues(&cddVals, CDD_DEFAULT_INDEX, numOfAntennas);

	CddSetInAutoReplyTcr(cddVals);
}

static bool setCddInStaTcr(StaId staIndex, bool changeProbingPoint, LaPacketType_e packetType, bool setRatePlusOne, CddValues_t cddvals)
{

	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	LaTcrModificationStruct_t tcrModicationParams;
	bool ratePlusOneChanged;

	memset(&tcrModicationParams,0,sizeof(LaTcrModificationStruct_t));
	/*Fill control parameters of tcrModicationParams*/
	tcrModicationParams.controlParams.staIndex= staIndex;
	tcrModicationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX;
	tcrModicationParams.controlParams.firstBwToUpdate = BANDWIDTH_TWENTY;
	tcrModicationParams.controlParams.lastBwToUpdate = LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	tcrModicationParams.controlParams.isVhtSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.vhtSta;
	tcrModicationParams.controlParams.changeProbingPoint=  changeProbingPoint;
	tcrModicationParams.controlParams.slowProbing=  changeProbingPoint;
	tcrModicationParams.controlParams.packetType = packetType;
	tcrModicationParams.controlParams.changeToRatePlusOne = setRatePlusOne;
	
	memcpy(&(tcrModicationParams.tcrParams.tcrGeneralVals.cddVals), &cddvals, sizeof(CddValues_t));

	
#ifdef CDD_ADAPTATION_DEBUG
	ILOG0_DDD("setCddInStaTcr. cddAnt1: %d cddAnt2: %d cddAnt3: %d", cddvals.cddAnt1, cddvals.cddAnt2, cddvals.cddAnt3);
#endif
	/*Set CDD and effected params in TCRs*/
	ratePlusOneChanged = modifyStaTcrsParamsReq(&tcrModicationParams,CddModificationFunc);	

	return ratePlusOneChanged;

}
static void  CddModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	memcpy(&(pModifcationParamsOut->tcrParams.tcrGeneralVals.cddVals), &(pModifcationParamsIn->tcrParams.tcrGeneralVals.cddVals), sizeof(CddValues_t));
}

void CddfillCddValues(CddValues_t* cddVal, uint8 cddIndex, uint8 antCount)
{
	// Get permutation from DB
	uint16 cddPermutation = getPermutation(cddIndex, antCount);

	// Fill CDD structure with values per antenna 
	cddVal->cddAnt1 = (cddPermutation & CDD_ANT1_MASK) >> CDD_ANT1_SHIFT;
	cddVal->cddAnt2 = (cddPermutation & CDD_ANT2_MASK) >> CDD_ANT2_SHIFT;
	cddVal->cddAnt3 = (cddPermutation & CDD_ANT3_MASK) >> CDD_ANT3_SHIFT;
	
#ifdef CDD_ADAPTATION_DEBUG	
	ILOG0_DDD("setCddValues. cddAnt1 %d, cddAnt2 %d cddAnt3 %d",cddVal->cddAnt1,cddVal->cddAnt2,cddVal->cddAnt3);
#endif
}

static void fillCddFixedValues(CddValues_t* cddVal, uint16 cddPermutation)
{
	// Fill CDD structure with values per antenna 
	cddVal->cddAnt1 = (cddPermutation & CDD_ANT1_MASK) >> CDD_ANT1_SHIFT;
	cddVal->cddAnt2 = (cddPermutation & CDD_ANT2_MASK) >> CDD_ANT2_SHIFT;
	cddVal->cddAnt3 = (cddPermutation & CDD_ANT3_MASK) >> CDD_ANT3_SHIFT;
	
#ifdef CDD_ADAPTATION_DEBUG		
	ILOG0_DDD("setCddFixedValues. cddAnt1 %d, cddAnt2 %d cddAnt3 %d",cddVal->cddAnt1,cddVal->cddAnt2,cddVal->cddAnt3);
#endif

}

static uint16 getPermutation(uint8 cddIndex, uint8 antCount)
{
#ifdef CDD_ADAPTATION_DEBUG		
	ILOG0_D("getPermutation. antCount%d", antCount);
#endif
	ASSERT(cddIndex <= CDD_MAX_NUMBER_OF_PERMUTATIONS);
	DEBUG_ASSERT((antCount >= CDD_MIN_NUMBER_OF_ANTENNAS) && (antCount <= CDD_MAX_NUMBER_OF_ANTENNAS));

	// Retrieve CDD permutation from DB
	return (cddPermutations.cddMatrix[antCount - CDD_MIN_NUMBER_OF_ANTENNAS].permutations[cddIndex]);
}

static bool isSameCddAsWorkingPoint(CddValues_t cddVal, StaId staIndex)
{
	uint8 tcrCddAnt1, tcrCddAnt2, tcrCddAnt3;
	bool sameAsWorkingPoint = FALSE;
	LaTcrModificationStruct_t tcrModificationParams;
	
	tcrModificationParams.controlParams.packetType = LA_PACKET_TYPE_DATA;
	tcrModificationParams.controlParams.staIndex = staIndex; 	
	tcrModificationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX; 
	// Get CDD data from TCR
	getTcrParamsFromHwDb(&tcrModificationParams);
	
	tcrCddAnt1 = tcrModificationParams.tcrParams.tcrGeneralVals.cddVals.cddAnt1;
	tcrCddAnt2 = tcrModificationParams.tcrParams.tcrGeneralVals.cddVals.cddAnt2;
	tcrCddAnt3 = tcrModificationParams.tcrParams.tcrGeneralVals.cddVals.cddAnt3;
	
#ifdef CDD_ADAPTATION_DEBUG	
	ILOG0_DDD("tcrCddAnt1: %d tcrCddAnt2: %d tcrCddAnt3: %d",tcrCddAnt1,tcrCddAnt2,tcrCddAnt3);
#endif

	// Check whether CDD values are identical to the values in TCR
	if ((cddVal.cddAnt1 == tcrCddAnt1) && (cddVal.cddAnt2 == tcrCddAnt2) && (cddVal.cddAnt3 == tcrCddAnt3))
	{
		sameAsWorkingPoint = TRUE;
	}
#ifdef CDD_ADAPTATION_DEBUG	
	ILOG0_D("sameAsWorkingPoint = %d", sameAsWorkingPoint);
#endif
	return sameAsWorkingPoint;
}


