/***********************************************************************************
 File:		CyclicPrefix.c
 Module:		LinkAdaptation 
 Purpose: 	Handling cyclic prefix (Ht / VHT) / preamble (11B) 
 Description:	This fille handling the cyclic prefix preamble probing and working point update
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "HdkGlobalDefs.h"
#include "lm_StaDatabase.h"
#include "mt_sysdefs.h"
#include "linkAdaptation_api.h"
#include "CyclicPrefix.h"
#include "Estimators.h"
#include "LinkAdaptation_StateMachine.h"
#include "RateAdaptation.h"
#include "GroupManager_API.h"
#if defined(ENET_INC_ARCH_WAVE600)
#include "PreAggregator_Api.h"
#else
#include "AggregationBuilder_Api.h"
#endif 
#include "CommonRamLinkAdaptation.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 7


//#define CYCLIC_PREFIX_LOGS 

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct CyclicPrefixBitMaskDb
{
	StationBitmap_t probeWithRatePlus1;
}CyclicPrefixBitMaskDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static bool isCpModeSupported (LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, CyclicPrefixMode_e cpMode);
static bool cyclicPrefixSetCpParams(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, CyclicPrefixMode_e* pCpModeTable, Bandwidth_e firstBwToupdate, Bandwidth_e lastBwToUpdate, bool changeProbingPoint, bool setRatePlusOne, bool isVht);
static void cyclicPrefixModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);
static void cyclicPrefixSetStaCapabilities(StaId staIndex, PhyMode_e staPhyMode, HtCapabilitiesInfo_t htCapbilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities);
static CyclicPrefixMode_e getNextSupportedCpMode(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, CyclicPrefixMode_e startIndex);
static void calcLtfFactorOnEffectiveTp(uint16* pEffectiveTp, CyclicPrefixMode_e cpMode);
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static CyclicPrefixBitMaskDb_t CyclicPrefixBitMaskDatabase;

CyclicPrefixType_e convertCpModeToCpType[CP_NUM_OF_MODES] = 
{
	CP_SHORT_HT_VHT_HE, 
	CP_LONG_HT_VHT_MED_HE,
	CP_SHORT_HT_VHT_HE,
	CP_LONG_HT_VHT_MED_HE,
	CP_SHORT_HT_VHT_HE,
	CP_LONG_HE,
};

static LtfType_e convertCpModeToLtfType[CP_NUM_OF_MODES] = 
{
	LTF_1X,
	LTF_1X,	
	LTF_2X,
	LTF_2X, 	
	LTF_4X,
	LTF_4X
};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CyclicPrefixAdaptationInit 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void CyclicPrefixAdaptationInit (void)
{
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************

CyclicPrefixAddStation 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void CyclicPrefixAddStation(StaId staIndex, HtCapabilitiesInfo_t htCapbilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	PhyMode_e staPhyMode = ExtractHighestPhyModeFromRaMask(pLaDatabase->laStaUspCommon.raIndexMask, pLaDatabase->laStaUspCommon.staTransmissionParams.heSta);
	
	updateLaDbDistributionParam(&laDbDistributionParameter, staIndex, INVALID_MU_USP_INDEX, FALSE);
	cyclicPrefixSetStaCapabilities(staIndex, staPhyMode, htCapbilities, vhtCapabilities, pHeMacPhyCapabilities);
		
	LA_CLR_BIT_IN_BITMAP(CyclicPrefixBitMaskDatabase.probeWithRatePlus1.staBitMap,staIndex);

	cyclicPrefixResetLoop(&laDbDistributionParameter, TRUE);
	
}

/**********************************************************************************

CyclicPrefixAddGroup 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void CyclicPrefixAddGroup(uint8 groupId)
{
	UNUSED_PARAM(groupId);	
}


/**********************************************************************************

CyclicPrefixRemoveGroup 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void CyclicPrefixRemoveGroup(uint8 groupId)
{
	UNUSED_PARAM(groupId);
}


/**********************************************************************************

CyclicPrefixSetNextProbingPoint 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void CyclicPrefixSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	CyclicPrefixDb_t* pCpDb = &laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb;
	Bandwidth_e bandwidth;
	bool atLeastOneBwChangeProbingPoint = FALSE;
	CyclicPrefixMode_e tcrCpTable[LA_NUM_OF_BANDWIDTH];
	bool probeWithRatePlusOne = FALSE;
	PhyMode_e staPhyMode = ExtractHighestPhyModeFromRaMask(laDbDistributionParameter->laStaUspCommon->raIndexMask, laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
	UNUSED_PARAM(stayWithLastProbingPoint);
	switch (staPhyMode)
	{
#ifdef ENET_INC_ARCH_WAVE600
		case PHY_MODE_11AX_SU:
			tcrCpTable[BANDWIDTH_TWENTY] = pCpDb->nextCpPp;
			tcrCpTable[BANDWIDTH_FOURTY] = pCpDb->nextCpPp;
			tcrCpTable[BANDWIDTH_EIGHTY] = pCpDb->nextCpPp;
			tcrCpTable[BANDWIDTH_ONE_HUNDRED_SIXTY] = pCpDb->nextCpPp;
			atLeastOneBwChangeProbingPoint = TRUE;
			break;
#endif
		case PHY_MODE_11AC:
		case PHY_MODE_11N:
			//ASSERT_IN_LOOP-FW
			ASSERT(LinkAdaptationCommonConfiguration.wlanBandwidthMax < (LA_NUM_OF_BANDWIDTH));
			for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidth++)
			{
				//ASSERT(bandwidth < (LA_NUM_OF_BANDWIDTH));
				if (isCpModeSupported(laDbDistributionParameter, bandwidth, pCpDb->nextCpPp))
				{	
					// CP mode is supported for this BW
					tcrCpTable[bandwidth] = pCpDb->nextCpPp;
					atLeastOneBwChangeProbingPoint = TRUE;
				}
				else
				{
					// CP mode isn't supported for this BW, probe same as WP
					tcrCpTable[bandwidth] = pCpDb->currentCpWp;
				}
			}
			break;
		default:
			ASSERT(0);
			break;
	}
	
	if (atLeastOneBwChangeProbingPoint == TRUE) 
	{
		if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
		{	
			probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(CyclicPrefixBitMaskDatabase.probeWithRatePlus1.staBitMap, laDbDistributionParameter->stationOrGroupIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
		}
		/*Update cyclic prefix in TCRs for specific BW*/
		cyclicPrefixSetCpParams(laDbDistributionParameter,tcrCpTable,BANDWIDTH_TWENTY,LinkAdaptationCommonConfiguration.wlanBandwidthMax,TRUE, probeWithRatePlusOne,laDbDistributionParameter->laStaUspCommon->staTransmissionParams.vhtSta);
		/*Change RA state to wait for fast probe valid state*/
		laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_SLOW_PROBE_VALID);
		setCurrentSlowProbingTaskInDb(laDbDistributionParameter,SLOW_PROBING_CP);
	}
}
/**********************************************************************************

CyclicPrefixProcessFeedback 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void CyclicPrefixProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	StaId	staIndex=0;
	LinkAdaptationConfigurationParams_t* pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	Bandwidth_e currentBw = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);
	SlowProbingPointEstimators_t* pSlowProbingEst;
	FastProbingPointEstimators_t* pWorkingPointEst;
	int8 workingPointPer;
	uint8 slowProbingPointPer;
	uint8 currentRateindex;
	uint8 slowProbingRateIndex = INVALID_RATE_INDEX_VAL;
	CyclicPrefixMode_e currentProbingPoint = laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.nextCpPp;	
	CyclicPrefixMode_e currentWorkingPoint = laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp;
	Bandwidth_e bandwidthTemp;
	CyclicPrefixMode_e tcrCpTable[LA_NUM_OF_BANDWIDTH];
	uint16 losslessTpWp;
	uint16 losslessTpPp;
	uint16 effectiveTpWp;
	uint16 effectiveTpProbing;
	bool probeWithRatePlusOne = FALSE;
	
	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_CP);

	//KW_FIX_FW_M
	ASSERT(currentProbingPoint < CP_NUM_OF_MODES);
	ASSERT(currentWorkingPoint < CP_NUM_OF_MODES);
	
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		staIndex = laDbDistributionParameter->stationOrGroupIndex;
		probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(CyclicPrefixBitMaskDatabase.probeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	}
	slowProbingRateIndex = estimatorsGetSlowProbingRatePlusOne(laDbDistributionParameter);
	pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);
	pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter);
	
	currentRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
	
	/* If CP mode in current BW is not supported (WP same as PP), return from function*/
	if (currentProbingPoint != currentWorkingPoint)
	{		
		if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff) &&
			AlphaFilter_isFilterResultValid(&pWorkingPointEst->averagePer, laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
		{
			workingPointPer = AlphaFilter_GetFilterResult(&pWorkingPointEst->averagePer);
			slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);
			workingPointPer = MAX(workingPointPer - WP_PER_BIAS, 0);
			/*Extract lossless TP according to bandwidth*/

			losslessTpWp = getPhyRate(getRatesTable(laDbDistributionParameter->laStaUspCommon), currentRateindex, currentBw, convertCpModeToCpType[currentWorkingPoint]);
			losslessTpPp = getPhyRate(getRatesTable(laDbDistributionParameter->laStaUspCommon), slowProbingRateIndex, currentBw, convertCpModeToCpType[currentProbingPoint]);
			
			/*Calc effective TP for working point and probing point*/
			effectiveTpWp = estimatorsCalcEffectiveTp(losslessTpWp,workingPointPer);
			effectiveTpProbing = estimatorsCalcEffectiveTp(losslessTpPp,slowProbingPointPer);

			calcLtfFactorOnEffectiveTp(&effectiveTpWp, currentWorkingPoint);
			calcLtfFactorOnEffectiveTp(&effectiveTpProbing, currentProbingPoint);
			
#ifdef CYCLIC_PREFIX_LOGS
			ILOG0_DDDD("CyclicPrefixProcessFeedback,workingPointPer = %d,slowProbingPointPer = %d, effectiveTpWp = %d,  effectiveTpProbing = %d",
					workingPointPer, slowProbingPointPer, effectiveTpWp, effectiveTpProbing	);
#endif
			if(effectiveTpProbing > effectiveTpWp) 
			{
				memset(tcrCpTable, currentWorkingPoint, LA_NUM_OF_BANDWIDTH);
				/*Probing point TP is better than WP - change WP*/
				laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp = laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.nextCpPp;
				//ASSERT_IN_LOOP-FW
				ASSERT(LinkAdaptationCommonConfiguration.wlanBandwidthMax < (LA_NUM_OF_BANDWIDTH));
				/*Update cyclic prefix in TCRs for all BWs, current BW must be valid, but we should check if other BWs SCP is valid*/
				for (bandwidthTemp = BANDWIDTH_TWENTY; bandwidthTemp <= LinkAdaptationCommonConfiguration.wlanBandwidthMax; bandwidthTemp++)
				{
					//ASSERT(bandwidthTemp < (LA_NUM_OF_BANDWIDTH));
					if (isCpModeSupported(laDbDistributionParameter,bandwidthTemp, currentProbingPoint))
					{
						tcrCpTable[bandwidthTemp] = currentProbingPoint;
					}
					
#ifdef CYCLIC_PREFIX_LOGS
					ILOG0_DD("CyclicPrefixProcessFeedback, Change WP, tcrCpTable[%d] = =%d",bandwidthTemp,tcrCpTable[bandwidthTemp] );
#endif
				}
				cyclicPrefixSetCpParams(laDbDistributionParameter,tcrCpTable,BANDWIDTH_TWENTY,LinkAdaptationCommonConfiguration.wlanBandwidthMax,FALSE, FALSE,laDbDistributionParameter->laStaUspCommon->staTransmissionParams.vhtSta);
	
				/*Set next probing point*/
				laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.nextCpPp = getNextSupportedCpMode(laDbDistributionParameter, currentProbingPoint);

				if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
				{
					/*Clear slowProbingWithHigherRate for the next cycle only of loow was disabled for this task*/
					LA_CLR_BIT_IN_BITMAP(CyclicPrefixBitMaskDatabase.probeWithRatePlus1.staBitMap, staIndex);
				}
				/*Reset counters thresholds*/
				rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
				/*Reset estimators*/
				estimatorsResetRatesEstimators(laDbDistributionParameter);
				/*Reset fast probing counter so fast probing can collect statistics before next fast probing event*/	
				resetFastProbingCounters(laDbDistributionParameter);
				/*CP changed state - disable loop from current cycle*/
				disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_CP);
				changeTaskToDefaultPriority(laDbDistributionParameter,SLOW_PROBING_CP);
				resetNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_CP);
			}
			else
			{
				/*Probing point has not improved the TP*/
				if ((probeWithRatePlusOne == FALSE) &&
					(workingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh)&&
					(slowProbingPointPer< pLaConfigParams->slowProbingWithHigherRateLowPerTh))
				
				{
					ASSERT(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX); /*MU flow shouldn't get here */
					/*Clear slowProbingWithHigherRate for the next cycle*/
					LA_SET_BIT_IN_BITMAP(CyclicPrefixBitMaskDatabase.probeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check

#ifdef CYCLIC_PREFIX_LOGS
					ILOG0_V("CyclicPrefix - SET RATE + 1");
#endif

				}
				else
				{
					/*Set next probing point*/
					laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.nextCpPp = getNextSupportedCpMode(laDbDistributionParameter, currentProbingPoint);
					disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_CP);
					updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_CP);

					if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
					{
						LA_CLR_BIT_IN_BITMAP(CyclicPrefixBitMaskDatabase.probeWithRatePlus1.staBitMap, staIndex);
					}
				}
			}
		}
	}
	return;
}

/**********************************************************************************

isCpModeSupported 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static bool isCpModeSupported (LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, CyclicPrefixMode_e cpMode)
{
	bool retVal = FALSE;

	CyclicPrefixDb_t* pCpDb = &laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb;

	if (GET_BIT_IN_BYTE(pCpDb->CyclicPrefixCapBitmap[bandwidth], cpMode))
	{
		retVal = TRUE;
	}

	return retVal;
}

/**********************************************************************************

CyclicPrefixSetFixedVal 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void CyclicPrefixSetFixedVal(StaId staIndex, CyclicPrefixMode_e cpMode)
{
	CyclicPrefixMode_e cpModeTable[MAX_POSSIBLE_NUM_OF_BW];
	Bandwidth_e bandwidth;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter,staIndex,INVALID_MU_USP_INDEX, FALSE);	
	for(bandwidth = BANDWIDTH_TWENTY; bandwidth < MAX_POSSIBLE_NUM_OF_BW; bandwidth++)
	{
		cpModeTable[bandwidth] = cpMode;
	}

	/*Set cyclic prefix in TCRs for specific all BWs*/
	cyclicPrefixSetCpParams(&laDbDistributionParameter, cpModeTable,BANDWIDTH_TWENTY,LinkAdaptationCommonConfiguration.wlanBandwidthMax,FALSE, FALSE,LinkAdaptationStaDatabase[staIndex].laStaUspCommon.staTransmissionParams.vhtSta);

	CyclicPrefixUpdateDbWithFixedParams(&laDbDistributionParameter, cpMode);
}

/**********************************************************************************

CyclicPrefixStoreParamsInDb 


Description:
------------
- When going to fixed rate state, store the last CP mode in DB to use it when going back to auto mode

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void CyclicPrefixUpdateDbWithFixedParams(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, CyclicPrefixMode_e currentWp)
{
	laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp = currentWp;
	laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.nextCpPp = getNextSupportedCpMode(laDbDistributionParameter, currentWp);
}

/**********************************************************************************

cyclicPrefixResetLoop 


Description:
------------

Input: 
-----
- bUseDefaultParams - TRUE for add STA etc., FALSE for restoring back from fixed mode

Returns:
--------
	
**********************************************************************************/
void cyclicPrefixResetLoop(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool bUseDefaultParams)
{
	CyclicPrefixMode_e tcrCpTable[MAX_POSSIBLE_NUM_OF_BW];
	uint8 bandwidth;
	PhyMode_e staPhyMode = ExtractHighestPhyModeFromRaMask(laDbDistributionParameter->laStaUspCommon->raIndexMask, laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
	CyclicPrefixMode_e currentWp = CP_MODE_MED_CP_SHORT_LTF; // set default WP to med CP (long for non-HE)

	/* Initialize tcrCpTable with med CP (long for non-HE) */
	memset (tcrCpTable, CP_MODE_MED_CP_SHORT_LTF, MAX_POSSIBLE_NUM_OF_BW);
	
	switch (staPhyMode)
	{
#ifdef ENET_INC_ARCH_WAVE600
		case PHY_MODE_11AX_SU:
			if(TRUE == bUseDefaultParams)
			{
				currentWp = CP_MODE_SHORT_CP_MED_LTF; // Set WP to 0.8 CP, 2x LTF (mandatory)
			}
			else
			{
				// Restore saved CP
				currentWp = laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp;
			}
			tcrCpTable[BANDWIDTH_TWENTY] = currentWp;
			tcrCpTable[BANDWIDTH_FOURTY] = currentWp;
			tcrCpTable[BANDWIDTH_EIGHTY] = currentWp;
			tcrCpTable[BANDWIDTH_ONE_HUNDRED_SIXTY] = currentWp;
			break;
#endif
		case PHY_MODE_11AC:
		case PHY_MODE_11N:
			if(TRUE == bUseDefaultParams)
			{
				currentWp = CP_MODE_SHORT_CP_SHORT_LTF;
			}
			else
			{
				// Restore saved CP
				currentWp = laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp;
			}
			for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidth++)
			{
				if (!isCpModeSupported(laDbDistributionParameter, (Bandwidth_e)bandwidth, currentWp))
				{	
					// VHT long CP
					currentWp = CP_MODE_MED_CP_SHORT_LTF;
				}
				tcrCpTable[bandwidth] = currentWp;
			}
			break;
		default:
			rateAdaptationEnableDisableStaSlowLoop(laDbDistributionParameter->stationOrGroupIndex, SLOW_PROBING_CP, DISABLE_LOOP);
			break;
	}
	
	laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.currentCpWp = currentWp;
	laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb.nextCpPp = getNextSupportedCpMode(laDbDistributionParameter, currentWp);

	/*Set cyclic prefix in TCRs for specific all BWs*/
	cyclicPrefixSetCpParams(laDbDistributionParameter,tcrCpTable,BANDWIDTH_TWENTY,LinkAdaptationCommonConfiguration.wlanBandwidthMax,FALSE, FALSE,laDbDistributionParameter->laStaUspCommon->staTransmissionParams.vhtSta);
}

/**********************************************************************************

cyclicPrefixSetCpParams 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static bool cyclicPrefixSetCpParams(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, CyclicPrefixMode_e* pCpModeTable, Bandwidth_e firstBwToupdate, Bandwidth_e lastBwToUpdate, bool changeProbingPoint, bool setRatePlusOne, bool isVht)
{
	LaTcrModificationStruct_t tcrModificationParams;
	Bandwidth_e bandwidth;
	bool ratePlusOneChanged = FALSE;

	memset(&tcrModificationParams,0,sizeof(LaTcrModificationStruct_t));

	/*Fill control parameters of tcrModificationParams*/
	tcrModificationParams.controlParams.firstBwToUpdate = firstBwToupdate;
	tcrModificationParams.controlParams.lastBwToUpdate= lastBwToUpdate;
	tcrModificationParams.controlParams.isVhtSta = isVht;
	tcrModificationParams.controlParams.changeProbingPoint = changeProbingPoint;
	tcrModificationParams.controlParams.slowProbing=  changeProbingPoint;
	tcrModificationParams.controlParams.packetType = LA_PACKET_TYPE_DATA;
	tcrModificationParams.controlParams.changeToRatePlusOne = setRatePlusOne;
	tcrModificationParams.controlParams.staIndex= laDbDistributionParameter->stationOrGroupIndex;
	tcrModificationParams.controlParams.uspIndex = laDbDistributionParameter->uspIndex;
#ifdef ENET_INC_ARCH_WAVE600
	tcrModificationParams.controlParams.isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
#endif
	for (bandwidth = firstBwToupdate; bandwidth<=lastBwToUpdate; bandwidth++)
	{
		tcrModificationParams.tcrParams.bwDependedTcrValsTable[bandwidth].cpMode = pCpModeTable[bandwidth];
	}
	/*Set cyclic prefix and effected params in TCRs*/
	ratePlusOneChanged = modifyStaTcrsParamsReq(&tcrModificationParams,cyclicPrefixModificationFunc); 	

	return ratePlusOneChanged;
}
/**********************************************************************************

cyclicPrefixModificationFunc 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
static void cyclicPrefixModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	Bandwidth_e bandwidth;

	ASSERT(pModifcationParamsIn->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	for(bandwidth = pModifcationParamsIn->controlParams.firstBwToUpdate; bandwidth <= pModifcationParamsIn->controlParams.lastBwToUpdate; bandwidth++)
	{
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].cpMode = pModifcationParamsIn->tcrParams.bwDependedTcrValsTable[bandwidth].cpMode;
	}
}
/**********************************************************************************

cyclicPrefixWorkingPointChanged 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void cyclicPrefixWorkingPointChanged(StaId staIndex, uint8 oldRate, uint8 newRate)
{
	const RateObj_t* ratesTable;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	RaEnableDisableSlowLoop_e enableDisableLoop = DISABLE_LOOP;

	updateLaDbDistributionParam(&laDbDistributionParameter,staIndex,INVALID_MU_USP_INDEX, FALSE);
	
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);
	if(ratesTable[oldRate].laPhyMode != ratesTable[newRate].laPhyMode)
	{
		if ((ratesTable[newRate].laPhyMode == LA_PHY_MODE_HT_VHT) || (ratesTable[newRate].laPhyMode == LA_PHY_MODE_HE))
		{
			// Enable loop
			enableDisableLoop = ENABLE_LOOP;
			cyclicPrefixResetLoop(&laDbDistributionParameter, TRUE);
		}
		rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CP, enableDisableLoop);
	}
}
static void cyclicPrefixSetStaCapabilities(StaId staIndex, PhyMode_e staPhyMode, HtCapabilitiesInfo_t htCapbilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	CyclicPrefixDb_t* pCpDb = &pLaDatabase->laStaGroupCommon.cyclicPrefixDb;
	
	switch (staPhyMode)
	{
#ifdef ENET_INC_ARCH_WAVE600
		case PHY_MODE_11AX_SU:
		{	
			uint8 bandwidth;
			for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidth++)
			{
				// Set HE mandatory capabilities
				TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[bandwidth], CP_MODE_SHORT_CP_MED_LTF);		// HE CP = 0.8, HE LTF = 2x 	
				TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[bandwidth], CP_MODE_MED_CP_MED_LTF);			// HE CP = 1.6, HE LTF = 2x 			
				TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[bandwidth], CP_MODE_LONG_CP_LONG_LTF);		// HE CP = 3.2, HE LTF = 4x 	

				// Set HE optional capabilities
				//SET_BIT_IN_BYTE(&pCpDb->CyclicPrefixCapBitmap[bandwidth], CP_MODE_SHORT_CP_SHORT_LTF, pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_HE_SU_PPDU_WITH_1X_HE_LTF_AND_0_8_US_GI);		// HE CP = 0.8, HE LTF = 1x 		
				SET_BIT_IN_BYTE(&pCpDb->CyclicPrefixCapBitmap[bandwidth], CP_MODE_SHORT_CP_LONG_LTF, pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_HE_SU_PPDU_AND_HE_MU_WITH_4X_HE_LTF_AND_0_8US_GI);		// HE CP = 0.8, HE LTF = 4x 
			}
#ifdef ENET_INC_ARCH_WAVE600D2
			SET_BIT_IN_BYTE(&pCpDb->HeSuExtCyclicPrefixCapBitmap, CP_MODE_SHORT_CP_LONG_LTF, pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_HE_ER_SU_PPDU_4X_HE_LTF_0_8_US_GI);		// HE CP = 0.8, HE LTF = 4x //miryam check if correct place in the switch
			SET_BIT_IN_BYTE(&pCpDb->HeSuExtCyclicPrefixCapBitmap, CP_MODE_SHORT_CP_SHORT_LTF, pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_HE_ER_SU_PPDU_1X_HE_LTF_0_8_US_GI);	// HE CP = 0.8, HE LTF = 1x 
#endif //ENET_INC_ARCH_WAVE600D2
			break;
		}
#endif
		case PHY_MODE_11AC:
			// Set VHT mandatory capabilities
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_TWENTY], CP_MODE_MED_CP_SHORT_LTF);				// HT/VHT long for BW 20		
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_FOURTY], CP_MODE_MED_CP_SHORT_LTF);				// HT/VHT long for BW 40					
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_EIGHTY], CP_MODE_MED_CP_SHORT_LTF);				// HT/VHT long for BW 80		
#ifdef ENET_INC_ARCH_WAVE600
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_ONE_HUNDRED_SIXTY], CP_MODE_MED_CP_SHORT_LTF);	// HT/VHT long for BW 160						
#endif
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_TWENTY], CP_MODE_SHORT_CP_SHORT_LTF);			// HT/VHT short for BW 20	
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_FOURTY], CP_MODE_SHORT_CP_SHORT_LTF);			// HT/VHT short for BW 40	
			
			// Set VHT optional capabilities
			SET_BIT_IN_BYTE(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_EIGHTY], CP_MODE_SHORT_CP_SHORT_LTF, vhtCapabilities.short_gi_80_mhz);								// HT/VHT short for BW 80		
#ifdef ENET_INC_ARCH_WAVE600
			SET_BIT_IN_BYTE(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_ONE_HUNDRED_SIXTY], CP_MODE_SHORT_CP_SHORT_LTF, vhtCapabilities.short_gi_160_and_80_plus_80_mhz);	// HT/VHT short for BW 160	
#endif
			break;
		case PHY_MODE_11N:
			// Set HT mandatory capabilities
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_TWENTY], CP_MODE_MED_CP_SHORT_LTF);				// HT/VHT long for BW 20		
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_FOURTY], CP_MODE_MED_CP_SHORT_LTF);				// HT/VHT long for BW 40					
			
			// Set HT optional capabilities
			SET_BIT_IN_BYTE(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_TWENTY], CP_MODE_SHORT_CP_SHORT_LTF, htCapbilities.shortGiFor20Mhz);	// HT/VHT short for BW 20	
			SET_BIT_IN_BYTE(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_FOURTY], CP_MODE_SHORT_CP_SHORT_LTF, htCapbilities.shortGiFor40Mhz);	// HT/VHT short for BW 40	
			break;
		default:		
			// Set legacy capabilities
			TURN_ON_BIT(&pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_TWENTY], CP_MODE_MED_CP_SHORT_LTF);				// HT/VHT long for BW 20		
			// Disable loop
			rateAdaptationEnableDisableStaSlowLoop(staIndex, SLOW_PROBING_CP, DISABLE_LOOP);
			break;
		}
}

static CyclicPrefixMode_e getNextSupportedCpMode(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, CyclicPrefixMode_e startIndex)
{
	CyclicPrefixDb_t* pCpDb = &laDbDistributionParameter->laStaGroupCommon->cyclicPrefixDb;
	CyclicPrefixMode_e tempCpMode = startIndex;
	uint8 allBwCapMask = (pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_TWENTY]) | (pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_FOURTY]) | (pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_EIGHTY]);
#ifdef ENET_INC_ARCH_WAVE600
	allBwCapMask |= (pCpDb->CyclicPrefixCapBitmap[BANDWIDTH_ONE_HUNDRED_SIXTY]);
#endif

	// Validate that the starting index is set in bitmask 
	ASSERT(GET_BIT_IN_BYTE(allBwCapMask,tempCpMode) != 0);

	if ((allBwCapMask & (~(1 << tempCpMode))) != 0)
	{
		// Search for the next set bit (that is not the current WP), only if there is one
		do 
		{
			tempCpMode++;
			if (tempCpMode == CP_NUM_OF_MODES)
			{
				tempCpMode = (CyclicPrefixMode_e)0;
			}
		}while ((GET_BIT_IN_BYTE(allBwCapMask,tempCpMode) == 0) || (tempCpMode == pCpDb->currentCpWp));
	}
	
	return tempCpMode;
}

static void calcLtfFactorOnEffectiveTp(uint16* pEffectiveTp, CyclicPrefixMode_e cpMode)
{
	LtfType_e ltfType = convertCpModeToLtfType[cpMode];
	// TBD  - PSDU time is hard coded, until the merge with Dynamic TxOP 
	uint16 averagedPsduTime = 125; //125 * [32 microsec] => 4 ms
	uint8 ltfDuration = (1 << ltfType); //in [4 microsec] units
	uint8 factorPercent;

	averagedPsduTime <<= 3; // convert form [32 microsec] to [4 microsec]
	factorPercent = (ltfDuration * HUNDRED_PERCENT)/(ltfDuration + averagedPsduTime);
	*pEffectiveTp = SUB_PERCENT(*pEffectiveTp, factorPercent);	
}
