/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/

#include "DynamicTxOP.h"
#include "AlphaFilter.h"
#include "LinkAdaptation.h"
#include "lm_StaDatabase.h"
#include "ShramStationDatabase.h"
#include "HwSemaphore_API.h"
#include "loggerAPI.h"
#include "CommonRamLinkAdaptation.h"
#include "ConfigurationManager_api.h"
#include "PreAggregator_Api.h"

#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 19

//#define DYNAMIC_TXOP_DEBUG


/******************************************************************************/
/***					Static Function Declaration	                                                       ***/
/******************************************************************************/

static void setTxopInHwStaDb(StaId staId, uint8 tidIndex, bool enable);
static void setTxopModeInHwStaDb(StaId staId);
static void setTxopAccordingToPsduTimeEstimation(StaId staId);
static uint16 calcMaxPsduTimePerRateAndBw(StaId staId, uint8 rateIndex, uint8 bw);
static uint16 getMaxPsduTimeFromStaDb(StaId staId);
static void updateTxopMode(StaId stationIndex, TxopMode_e mode);
static void sendTxopModeToUm(StaId stationIndex, TxopMode_e mode);

/******************************************************************************/
/***					Global Variable      	                                                             ***/
/******************************************************************************/
TxopCommonParams_t TxopCommonParams;

/******************************************************************************/
/***					Public Functions Definitions                                                       ***/
/******************************************************************************/

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
/**********************************************************************************

DynamicTxop_Init  


Description:
------------

Input:
-----
	
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void DynamicTxop_Init()
{
	StaId staId;
	StaId firstSidInBand = ConfigurationManager_GetFirstSidInMyBand();
	uint16 numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsInMyBand();	

	TxopCommonParams.txopMaxNumStas = DYNAMIC_TXOP_MAX_NUM_CONNECTED_STAS_THRESHOLD;
	TxopCommonParams.configuredTxopMode = TXOP_MODE_ENABLED;
	TxopCommonParams.currentTxopMode = TXOP_MODE_ENABLED;

	for (staId = firstSidInBand; staId < (firstSidInBand + numOfSidsInBand); staId++)
	{
		AlphaFilter_Init(&(LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb.psduTimeFilter),&(LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb.averageDtFilterResult));
		LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb.txopMode = TXOP_MODE_ENABLED;
	}
}

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************

DynamicTxop_AddStation  


Description:
------------
	For any new connected STA, check if max num connected STAs passed,
	if yes - disable TxOP
Input:
-----
	
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void DynamicTxop_AddStation(StaId stationIndex)
{
	
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DDDD("DynamicTxop_AddStation NumOfConnectedSta %d configuredTxopMode %d currentTxopMode %d txopMaxNumStas %d",LmStaDataBase.NumOfConnectedSta, TxopCommonParams.configuredTxopMode, TxopCommonParams.currentTxopMode, TxopCommonParams.txopMaxNumStas);
#endif
	if(LmStaDataBase.NumOfConnectedSta > TxopCommonParams.txopMaxNumStas)
	{
		if((TxopCommonParams.configuredTxopMode == TXOP_MODE_ENABLED) && (TxopCommonParams.currentTxopMode == TXOP_MODE_ENABLED))
		{
			TxopCommonParams.currentTxopMode = TXOP_MODE_DISABLED;
			updateTxopMode(stationIndex, TXOP_MODE_DISABLED);
		}
	}
}
/**********************************************************************************

DynamicTxop_RemoveStation  


Description:
------------
	For any disconnected STA, check if have equal or below max num connected STAs,
	if yes - enable back TxOP

Input:
-----
	
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void DynamicTxop_RemoveStation(StaId stationIndex)
{
	
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DDDD("DynamicTxop_RemoveStation NumOfConnectedSta %d configuredTxopMode %d currentTxopMode %d txopMaxNumStas %d",LmStaDataBase.NumOfConnectedSta, TxopCommonParams.configuredTxopMode, TxopCommonParams.currentTxopMode, TxopCommonParams.txopMaxNumStas);
#endif
	if(LmStaDataBase.NumOfConnectedSta <= TxopCommonParams.txopMaxNumStas)
	{
		if((TxopCommonParams.configuredTxopMode == TXOP_MODE_ENABLED) && (TxopCommonParams.currentTxopMode == TXOP_MODE_DISABLED))
		{
			TxopCommonParams.currentTxopMode = TXOP_MODE_ENABLED;
			updateTxopMode(stationIndex, TXOP_MODE_ENABLED);
		}
	}
}

/**********************************************************************************

DynamicTxop_SetTxopMode  


Description:
------------
	Function sets TxOP mode (enable/disable) unless passed max num of 
	connected STAs - than disable anyway
Input:
-----
	TxopMode_e mode - enable or disable
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void DynamicTxop_SetTxopMode(StaId stationIndex, TxopMode_e mode)
{
	TxopCommonParams.configuredTxopMode = mode;
	TxopCommonParams.currentTxopMode = mode;

	if(LmStaDataBase.NumOfConnectedSta > TxopCommonParams.txopMaxNumStas)
	{
		TxopCommonParams.currentTxopMode = TXOP_MODE_DISABLED;
		mode = TXOP_MODE_DISABLED;
	}

	updateTxopMode(stationIndex, mode);
}

/**********************************************************************************

updateTxopMode  


Description:
------------

Input:
-----
	TxopMode_e mode - enable or disable
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void updateTxopMode(StaId stationIndex, TxopMode_e mode)
{
	DynamicTxopDB_t *pDynamicTxopDb; 
	StaState_e staState;
	StaId staIdIterator;
	StaId firstSidInBand = ConfigurationManager_GetFirstSidInMyBand();
	uint16 numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsInMyBand();	

#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DDD("DynamicTxop_UpdateTxopMode. mode %d firstSidInBand %d numOfSidsInBand %d",mode, firstSidInBand, numOfSidsInBand);
#endif

	// Update STA DB (in UM) with current TxOP mode
	sendTxopModeToUm(stationIndex, mode);

	for (staIdIterator = firstSidInBand; staIdIterator < (firstSidInBand + numOfSidsInBand); staIdIterator++)
	{
		pDynamicTxopDb = &LinkAdaptationStaDatabase[staIdIterator].laStationUnique.dynamicTxopDb;
		if(pDynamicTxopDb->primaryInMuGroup)
		{
			pDynamicTxopDb->prevTxopMode = mode;
		}
		else
		{
			pDynamicTxopDb->txopMode = mode;
			staState = getStaState(staIdIterator);
			if (staState == STA_STATE_CONNECTED)
			{
				/* Set TxOP in Sta DB according to TxOP mode */
				setTxopModeInHwStaDb(staIdIterator);
			}
		}
	}
}

static void sendTxopModeToUm(StaId stationIndex, TxopMode_e mode)
{
	K_MSG* pUpdateTxopMsg = NULL;
	TxopModeMsg_t*	pTxopModeMsg;

	pUpdateTxopMsg = OSAL_GET_MESSAGE(sizeof(TxopModeMsg_t));	
	pTxopModeMsg = (TxopModeMsg_t*)pK_MSG_DATA(pUpdateTxopMsg);
	pTxopModeMsg->mode = mode;
	pTxopModeMsg->staId = stationIndex;

	OSAL_SEND_MESSAGE(TS_MANAGER_CONFIGURE_TXOP_MODE, TASK_TS_MANAGER, pUpdateTxopMsg, VAP_ID_DO_NOT_CARE);	
}



void DynamicTxop_OverwriteTxopModeForMuMember(StaId staId, uint8 mode)
{
	DynamicTxopDB_t *pDynamicTxopDb = &LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb; 
	
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DD("DynamicTxop_OverwriteTxopModeForMuMember. staId %d mode %d",staId, mode);
#endif

	// if mode == FALSE, station is primary member in  group and need to disable current txopMode
	if(mode == FALSE)
	{
		pDynamicTxopDb->primaryInMuGroup = TRUE;
		pDynamicTxopDb->prevTxopMode = pDynamicTxopDb->txopMode;
		pDynamicTxopDb->txopMode = mode;
	}
	// if mode == TRUE, return the default txopMode from prevTxopMode value
	else
	{
		pDynamicTxopDb->txopMode = pDynamicTxopDb->prevTxopMode;
		pDynamicTxopDb->primaryInMuGroup = FALSE;
	}
}

TxopMode_e DynamicTxop_GetTxopMode(void)
{
	return TxopCommonParams.configuredTxopMode;
}

void DynamicTxop_RateOrBwIsChanged(StaId staId, uint8 rateIndex, uint8 bw)
{
	DynamicTxopDB_t *pDynamicTxopDb = &LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb;
	uint8 newMaxPsduTime;
	uint8 currentMaxPsduTime = pDynamicTxopDb->maxPsduTime;
	uint16 psduTimeFromStaDb, psduTimePerRate;

	/* Get theoretical max PSDU time by the new rate and BW - in [s]*/
	psduTimePerRate = calcMaxPsduTimePerRateAndBw(staId, rateIndex, bw);	
	
	/* Get current TID0 max PSDU time from Sta DB - in [2s]*/
	psduTimeFromStaDb = getMaxPsduTimeFromStaDb(staId);
	
	/* Convert max PSDU time to [32s] resolution */
	psduTimePerRate >>= DYNAMIC_TXOP_PSDU_RESOLUTION_BITS_USEC_TO_32_USEC;
	psduTimeFromStaDb >>= (DYNAMIC_TXOP_PSDU_RESOLUTION_BITS_USEC_TO_32_USEC - 1);
	
	/* Calculate the newMaxPsduTime by the MIN between the new theoretical time and the current max time from Sta DB */ 
	newMaxPsduTime = MIN(psduTimePerRate, psduTimeFromStaDb);

	if (newMaxPsduTime != currentMaxPsduTime)
	{
		/* If the change in BW or rate caused change in the theoretical max time - initialize estimator and counter */
#ifdef DYNAMIC_TXOP_DEBUG
		ILOG0_DD("DynamicTxop_RateOrBwIsChanged. currentMaxPsduTime %d newMaxPsduTime %d",currentMaxPsduTime, newMaxPsduTime);
#endif
		pDynamicTxopDb->maxPsduTime = newMaxPsduTime;
		AlphaFilter_Init(&(pDynamicTxopDb->psduTimeFilter),&(pDynamicTxopDb->averageDtFilterResult));
		pDynamicTxopDb->psduTimeNumOfReportsCounter = 0;
	}
}

void DynamicTxop_UpdateEstimator(StaId staId, uint8 currentPsduTime, int16 lnBeta, uint8 constAlphaForDtAv)
{
	DynamicTxopDB_t *pDynamicTxopDb = &LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb;

	{
		/* Update PSDU time alpha filter */
		AlphaFilter_updateFilter(&(pDynamicTxopDb->psduTimeFilter), currentPsduTime, lnBeta, constAlphaForDtAv,&(pDynamicTxopDb->averageDtFilterResult), TRUE); 
		pDynamicTxopDb->psduTimeNumOfReportsCounter++;
#ifdef DYNAMIC_TXOP_DEBUG
		SLOG0(0, 0, DynamicTxopDB_t, pDynamicTxopDb);
		ILOG0_DD("DynamicTxop_UpdateEstimator. currentPsduTime %d counter %d",currentPsduTime, pDynamicTxopDb->psduTimeNumOfReportsCounter);
#endif
	}
}

void DynamicTxop_Handler(StaId staId)
{
	DynamicTxopDB_t *pDynamicTxopDb = &LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb;
	bool effectivePsduTimeisValid = AlphaFilter_isFilterResultValid(&(pDynamicTxopDb->psduTimeFilter), FILTER_VALIDATION_TIME);
	
#ifdef DYNAMIC_TXOP_DEBUG
		//ILOG0_D("DynamicTxop_Handler. counter %d",pDynamicTxopDb->psduTimeNumOfReportsCounter);
#endif	

	if (pDynamicTxopDb->psduTimeNumOfReportsCounter == DYNAMIC_TXOP_NUMBER_OF_BAA_REPORTS_TH)
	{
		pDynamicTxopDb->psduTimeNumOfReportsCounter = 0;
		if (effectivePsduTimeisValid == TRUE)
		{
			/* Determine TxOP policy */
			setTxopAccordingToPsduTimeEstimation(staId);
		}
	}
}


/******************************************************************************/
/***					Static Functions Definitions                                                       ***/
/******************************************************************************/

static void setTxopModeInHwStaDb(StaId staId)
{
	DynamicTxopDB_t *pDynamicTxopDb = &LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb;
	uint8 tidIndex;
	bool enableTxopInStaDb;

#ifdef WORKAROUND_FOR_TXOP_HE_A0_PRE_AGG_BUG
	bool isHeSta = LinkAdaptationStaDatabase[staId].laStaUspCommon.staTransmissionParams.heSta;
	if ((pDynamicTxopDb->txopMode == TXOP_MODE_ENABLED) && (isHeSta == FALSE))
#else
	if (pDynamicTxopDb->txopMode == TXOP_MODE_ENABLED)
#endif
	{
		enableTxopInStaDb = TRUE;
	}
	else
	{
		enableTxopInStaDb = FALSE;
	}

	for (tidIndex = 0; tidIndex < NUM_OF_TID; tidIndex++)
	{
		setTxopInHwStaDb(staId, tidIndex, enableTxopInStaDb);
	}
	
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DD("setTxopModeInHwStaDb. staId %d mode %d",staId, pDynamicTxopDb->txopMode);
#endif	
}

static void setTxopInHwStaDb(StaId staId, uint8 tidIndex, bool enable)
{
	StaDb_t* pStaDbHwEntry =  &(StaDbHwEntries[staId]);
	StaDbTid_t* pStaTidEntry = &(pStaDbHwEntry->tid[tidIndex]);
	TX_INTERRUPT_SAVE_AREA;
		
	if (pStaTidEntry->txopMultipleDataEn != enable)
	{
		/*Set TXOP only if BA enabled*/
		if (pStaTidEntry->baEnable == 0)
		{
			return;
		}
#ifdef DYNAMIC_TXOP_DEBUG
		ILOG0_DD("setTxopInHwStaDb. staId %d enable %d",staId,enable);
#endif


		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#if defined(ENET_INC_ARCH_WAVE600)
		HW_SEMAPHORE_LOCK(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
		HW_SEMAPHORE_LOCK(HW_SEMAPHORE_1_17_TID_PARAMS_WORD_1_LM);
#endif

		pStaTidEntry->txopMultipleDataEn = enable;
#if defined(ENET_INC_ARCH_WAVE600)
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_1_17_TID_PARAMS_WORD_1_LM);
#endif

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
}


static void setTxopAccordingToPsduTimeEstimation(StaId staId)
{
	DynamicTxopDB_t *pDynamicTxopDb = &LinkAdaptationStaDatabase[staId].laStationUnique.dynamicTxopDb;
	uint8 effectivePsduTime = AlphaFilter_GetFilterResult(&(pDynamicTxopDb->psduTimeFilter));
	bool enableTxopInStaDb;
#ifdef WORKAROUND_FOR_TXOP_HE_A0_PRE_AGG_BUG
	bool isHeSta = LinkAdaptationStaDatabase[staId].laStaUspCommon.staTransmissionParams.heSta;
	if ((effectivePsduTime + DYNAMIC_TXOP_PSDU_TIME_MARGIN >= pDynamicTxopDb->maxPsduTime) && (isHeSta == FALSE))
#else
	if (effectivePsduTime + DYNAMIC_TXOP_PSDU_TIME_MARGIN >= pDynamicTxopDb->maxPsduTime)
#endif
	{
		/* effectivePsduTime is equal to the theoretical time - enable TxOP */
		enableTxopInStaDb = TRUE;
	}
	else
	{
		/* effectivePsduTime is lower than the theoretical time - disable TxOP */
		enableTxopInStaDb = FALSE;
	}
	
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DDD("setTxopAccordingToPsduTimeEstimation. effectivePsduTime %d currentMaxPsduTime %d enable %d",effectivePsduTime,pDynamicTxopDb->maxPsduTime, enableTxopInStaDb);
#endif	

	/* set TxOP for TID0 */
	setTxopInHwStaDb(staId, 0, enableTxopInStaDb);
}

static uint16 calcMaxPsduTimePerRateAndBw(StaId staId, uint8 rateIndex, uint8 bw)
{
	StaDb_t* pStaDbHwEntry =  &(StaDbHwEntries[staId]);
	uint32 psduLength;  // In bits
	uint16 mpduLength = pStaDbHwEntry->common.maxMpduLengthLimit;
	uint16 phyRate, maxTime;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staId].laStaUspCommon);

	if (pStaDbHwEntry->tid[0].ppduTxMode == STA_DB_PPDU_TX_MODE_LEGACY_AND_MPDU)
	{
		// no AMSDU support
		mpduLength = MSDU_MAX_LENGTH;
	}
	mpduLength = SUB_PERCENT(mpduLength, DYNAMIC_TXOP_MPDU_LENGTH_FACTOR);
	psduLength = CONVERT_BYTES_TO_BIT_INDEX(mpduLength * (pStaDbHwEntry->common.aMpduLimit));
	phyRate = ratesTable[rateIndex].rateTableBwParmas[bw].shortCpPhyRate;
	maxTime = MIN((psduLength/phyRate), MAX_PSDU_TX_TIME_LIMIT);
	
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DDD("calcMaxPsduTimePerRateAndBw. psduLength %d phyRate %d maxTime %d",psduLength, phyRate, maxTime);
#endif

	return maxTime;
}

static uint16 getMaxPsduTimeFromStaDb(StaId staId)
{
	StaDb_t* pStaDbHwEntry =  &(StaDbHwEntries[staId]);
	uint16 maxTime = pStaDbHwEntry->tid[0].maxPsduTransmissionTimeLimit;
#ifdef DYNAMIC_TXOP_DEBUG
	ILOG0_DD("getMaxPsduTimeFromStaDb. staId %d maxTime %d ", staId, maxTime);
#endif
	return	maxTime;
}


/**********************************************************************************

DynamicTxop_GetTxopDuration  


Description:
------------
Get TxOP duration from pre aggregator

Input:
-----
			
Output:
-------
	uint16 txopDuration

Returns:
--------
	void - 
	
**********************************************************************************/
uint16 DynamicTxop_GetTxopDuration(void)
{
	uint16 txopDuration = 0;

	txopDuration = PreAggregator_GetTxopDuration();

	return txopDuration;
}
/**********************************************************************************

DynamicTxop_SetTxopDuration  


Description:
------------
Set TxOP duration in pre aggregator

Input:
-----
	uint16 txopDuration
	
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void DynamicTxop_SetTxopDuration(uint16 txopDuration)
{
	PreAggregator_SetTxopDuration(txopDuration);
}

/**********************************************************************************

DynamicTxop_GetTxopMaxNumStas  


Description:
------------

Input:
-----
			
Output:
-------
	uint16 gTxopMaxNumStas

Returns:
--------
	void - 
	
**********************************************************************************/
uint16 DynamicTxop_GetTxopMaxNumStas()
{
	return TxopCommonParams.txopMaxNumStas;
}
/**********************************************************************************

DynamicTxop_SetTxopMaxNumStas  


Description:
------------

Input:
-----
	uint16 txopMaxNumStas
	
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void DynamicTxop_SetTxopMaxNumStas(StaId stationIndex, uint16 txopMaxNumStas)
{
	TxopCommonParams.txopMaxNumStas = txopMaxNumStas;

	// When number of connected STAs is above threshold, TxOP configured to enabled and is actually enabled - disable it
	if(LmStaDataBase.NumOfConnectedSta > TxopCommonParams.txopMaxNumStas)
	{
		if((TxopCommonParams.configuredTxopMode == TXOP_MODE_ENABLED) && (TxopCommonParams.currentTxopMode == TXOP_MODE_ENABLED))
		{
			TxopCommonParams.currentTxopMode = TXOP_MODE_DISABLED;
			updateTxopMode(stationIndex, TXOP_MODE_DISABLED);
		}
	}
	else //LmStaDataBase.NumOfConnectedSta <= gTxopMaxNumStas
	{
		
		// When number of connected STAs is below or equal threshold, TxOP configured to enabled and is actually disabled - enable it
		if((TxopCommonParams.configuredTxopMode == TXOP_MODE_ENABLED) && (TxopCommonParams.currentTxopMode == TXOP_MODE_DISABLED))
		{
			TxopCommonParams.currentTxopMode = TXOP_MODE_ENABLED;
			updateTxopMode(stationIndex, TXOP_MODE_ENABLED);
		}
	}
}


