/***********************************************************************************
 File:		Estimators.h
 Module:		LinkAdaptation 
 Purpose: 	Process estimators based in the Sequencer & BAA reports
 Description:	This file contains all definitions and the structures of the estimators
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "Estimators.h"
#include "LinkAdaptation.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "ProtectedDbLock_Api.h"
#include "ShramStationDatabase.h"
#include "StaDb.h"
#include "mt_sysdefs.h"
#include "lm.h"
#include "BaAnalyzer_Api.h"
#include "BwAdaptation.h"
#include "ShramStatistics.h"
#include "RateAdaptation.h"
#include "LinkAdaptation_StateMachine.h"
#include "HwSemaphore_API.h"
#include "AntennaSelection.h"
#include "DynamicTxOP.h"
#include "ConfigurationManager_api.h"
#include "CommonRamLinkAdaptation.h"
#include "ProtectionAdaptation.h"
#include "QAMplus.h"
#include "TurboRates.h"
#include "GroupsMetricsCalculator.h"
#include "CommonRamStatistics.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 8

#define EWMA_ALPHA_NUM_OF_CONVERSION 			3
#define STATISTICS_MSDU_TIME					1	/* BA Report indicates num msdu = 0 hence we cannot calculate. Report ewma_msdu_ms = 1 */
#define STATISTICS_DURATION						256 /* BA Report indicates duration = 0 hence we cannot calculate. Use this define instead */
#define PLUME_TRACE								


#define NUMBER_OF_UPDATE_ESTIMATORS_FUNCS 5

#define FAST_DROP_BITMAP_WINDOW_SIZE 16

#define BW_20_MHZ (20)
#define BW_40_MHZ (40) 
#define BW_80_MHZ (80) 
#define BW_160_MHZ (160) 


#define MU_CANDIDATE_PS_MGMT_IN_AWINDOW_FACTOR	33
#define MU_CANDIDATE_LOW_FRAME_RATE_FACTOR		33
#define MU_CANDIDATE_GOODPUT_FACTOR				33

#define MU_CANDIDATE_PS_MGMT_WINDOW_SIZE		64

#define MU_CANDIDATE_THRESHOLD					40	// Set it to a value higher than PS MGMT as a single data transmission will set the grade to it



#define	MU_CANDIDATE_PS_MGMT_IN_AWINDOW_FACTOR_128_SCALE	MU_CANDIDATE_SCALE_FACTOR_TO_128(MU_CANDIDATE_PS_MGMT_IN_AWINDOW_FACTOR)
#define	MU_CANDIDATE_LOW_FRAME_RATE_FACTOR_128_SCALE		MU_CANDIDATE_SCALE_FACTOR_TO_128(MU_CANDIDATE_LOW_FRAME_RATE_FACTOR)
#define	MU_CANDIDATE_GOODPUT_FACTOR_128_SCALE				MU_CANDIDATE_SCALE_FACTOR_TO_128(MU_CANDIDATE_GOODPUT_FACTOR)
#define	MU_CANDIDATE_THRESHOLD_FACTOR_128_SCALE				MU_CANDIDATE_SCALE_FACTOR_TO_128(MU_CANDIDATE_THRESHOLD)

#define LT_FILTERS_COEF 7
#define COLLISION_DETECTION_MIN_MPDU_COUNT 10
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define BW_INDEX_TO_MHZ(x) ((0x2<<(x-1))*10)
#define MU_CANDIDATE_SCALE_FACTOR_TO_128(f)	((f*128)/100)
#define INPUT_SAMPLE_TO_IIR_FILTER(input,lastFilterResult,coef) PHY_SHAPED_FORMULA(((input)<<LT_FILTERS_COEF),(lastFilterResult), (coef))
#define GET_IIR_FILTER_RESULT(filter, coef) ((filter)>>coef)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum EstimatorsEvent
{
	ESTIMATORS_NORMAL_PACKET,
	ESTIMATORS_PROBING_PACKET,
	ESTIMATORS_BF_RESPONSE_PACKET,
	ESTIMATORS_NUMBER_OF_EVENTS,		
}EstimatorsEvent_e;

typedef struct FastDropParams
{
	uint8 numberOf1sInSlidingWind;
	uint16 consFailMask;
}FastDropParams_t;

typedef struct EwmaAlpha
{
	uint32	numMsdus;
	uint8	ewmaAlpha;
}EwmaAlpha_t;

EwmaAlpha_t EwmaAlpha[EWMA_ALPHA_NUM_OF_CONVERSION]=
{
	{2, 3}, /* For num msdu  = 1   use alpha = 1/8	*/
	{3, 2}, /* For num msdu  = 2   use alpha = 1/4	*/
	{5,	1}, /* For num msdu  = 3,4 use alpha = 1/2	*/
	 		/* For num msdu >= 5   use alpha = 1 	*/
};

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

static void estimatorsProcessCollisionsBaaReport(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter,uint8 vapId, BaaRateAdaptiveFullFifo_t*pBaaReport);
static void resetCurrentPacketParams(void);
static uint32 htVhtSetProbeUpPointsInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex);
static uint8 htVhtSetProbeUpMcsInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex);
static uint8 htVhtSetProbeUpNssInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex);
static uint8 htVhtSetProbeDownNssUpMcs(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex);
static uint32 htVhtSetProbeDownPointInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex);
static uint8 findRateEntryInFastProbingEstDb ( FastProbingPointEstimators_t* fastProbingEstimators, uint8 rateIndexInEstimatorsDb);
static void removeRateEntryFromEstimatorsBwDb (FastProbingPointEstimators_t* fastProbingEstimators, uint32* removedIndexesBitmap);
static void setCurrentRateIndexBitmask (FastProbingPointEstimators_t* fastProbingEstimators, uint32* currentRateBitmask);
static void addNewRateToEstimatorsBwDb (FastProbingPointEstimators_t* fastProbingEstimators, uint32* addedIndexBitmap);
static void setEstimatorDbRatesType(LaStationUspCommonParams_t* laStaUspCommon, uint8 workingPointRateIndex, uint8 lastWpRateIndex);
static void extractCurrentPacketBaaParams(StaId staIndex, BaaRateAdaptiveFullFifo_t* pBaaReport);
uint8 calculatePer( BaaRateAdaptiveFullFifo_t* pBaaReport);
#ifdef LA_CALC_REF_LEN
static uint8 calculateRefLen(BaaRateAdaptiveFullFifo_t* pBaaReport);
#endif
static uint8 calculatePerVariance(uint8 currentPer, uint8 averagePer);
static uint32 calculateGoodPut(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 currentPer,uint8 rateIndex,Bandwidth_e bandwidth,uint16 factor);

static void  estimatorsResetVapDb(uint8 vapId);


void nonHtVhtSetProbingPoints(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t RaMask, uint32* newRateIndxMask, uint8 rateIndex);
static void  fastDropEstimator (uint8 rateIndex,  LaStationUspCommonParams_t* pEstDatabase, BaaRateAdaptiveFullFifo_t*pBaaReport, bool isCurrentPacketFailed);
static bool checkCurrentPacketFailed(BaaRateAdaptiveFullFifo_t* pBaaReport);
static void getFastDropParams(uint8 rateIndex, FastDropParams_t* fastDropParams, LaStationUspCommonParams_t* pEstDatabase);
static uint32 estimatorsCalcMuCandidacyGrade(LinkAdaptationStaDatabase_t* laStationDb);
#ifdef LA_CALC_REF_LEN
static uint8 calculateRefLenFactor (uint8 refLength, uint8 refLenFactor);
#endif
void estimatorsStatisticsUpdate(BaaRateAdaptiveFullFifo_t* pBaaReport);
uint8 estimatorsStatisticsChooseAlpha(uint32 numMsdus);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
 
/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
CurrentPacketParameters_t	CurrentPacketParams;
BaaRateAdaptiveFullFifo_t 	BaaReport = {0};

uint32 wlanAckCtsCollisionBitmap = 0;
uint8 wlanCollisionTotlaCount = 0;

#ifdef ENET_INC_ARCH_WAVE600
FastDropParams_t FastDropParametersHe[HE_NUMBER_OF_MCS] = 
{
	{13,0x3F},
	{13, 0x3F},
	{10, 0xF},
	{9,0x7},
	{10,0xF},
	{9,0x7},
	{5,0x7},
	{5,0x7},
	{6,0x7},
	{5, 0x7},
	{5, 0x7},	
	{5, 0x7}
};
#endif

FastDropParams_t FastDropEnabledParametersHtVht[VHT_NUMBER_OF_MCS] = 
{
	{13,0x3F},
	{13, 0x3F},
	{10, 0xF},
	{9,0x7},
	{10,0xF},
	{9,0x7},
	{5,0x7},
	{5,0x7},
	{6,0x7},
	{5, 0x7}
};
FastDropParams_t FastDropEnabledParameters11b = 
	{13, 0x3F};

FastDropParams_t FastDropDisabledParametersHtVht[VHT_NUMBER_OF_MCS] = 
{
	{13,0xFFFF},
	{13, 0xFFFF},
	{10, 0xFFFF},
	{9,0xFFFF},
	{10,0xFFFF},
	{9,0xFFFF},
	{5,0xFFFF},
	{5,0xFFFF},
	{6,0xFFFF},
	{5, 0xFFFF}
};
FastDropParams_t FastDropDisabledParameters11b = 
	{13, 0xFFFF};

FastDropParams_t FastDropParameters11ag[NUMBER_OF_RATES_11AG]=
{
	{5, 0x3},
	{10,0xF},
	{9,0x7},
	{10,0xF},
	{9,0x7},
	{10,0xF},
	{9, 0x7},
	{5,0x3}
 };

const uint16 MaxPhyRateAcPerBwPerNss[MAX_POSSIBLE_NUM_OF_BW][SPATIAL_STREAM_NUM] = 
{
	{87,	173,	289,	347},
	{200,	400,	600,	800},
	{433,	866,	1300,	1733},	
	{866,	1733,	2340,	3466},
};

#ifdef ENET_INC_ARCH_WAVE600
const uint16 MaxPhyRateAxPerBwPerNss[MAX_POSSIBLE_NUM_OF_BW][SPATIAL_STREAM_NUM] = 
{
	{143,	286,	430,	573},
	{286,	573,	860,	1147},
	{600,	1201,	1801,	2402},	
	{1201,	2402,	3603,	4803},
};
#endif
extern	uint32 muCandidateStaBitmap[];

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************/
/**********************************************************************************

estimatorsInit 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void estimatorsInit(void)
{
	StaId stationIndex = 0x0;
	uint8 vapIndex = 0x0;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	uint8 uspIndex; 
	uint8 groupIndex; 
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();
	StaId firstSidInBand = ConfigurationManager_GetFirstSidInMyBand();
	uint16 numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsInMyBand();	

	/*Reset current packet params*/
	resetCurrentPacketParams();

	/*Run over all stations and BWs */
	for (stationIndex = firstSidInBand; stationIndex < (firstSidInBand + numOfSidsInBand); stationIndex++)
	{
		updateLaDbDistributionParam(&laDbDistributionParameter,stationIndex,INVALID_MU_USP_INDEX, FALSE);
		estimatorsResetDb(&laDbDistributionParameter);
	}
	/*Vap already been connected , so we need to go over all vaps and change antenna selection mask & max nss*/
	for (vapIndex = firstVapInBand; vapIndex < (firstVapInBand + numOfVapsInBand); vapIndex++)
	{
		estimatorsResetVapDb(vapIndex);
	}
	/* VHT MU */
	for(groupIndex = 0; groupIndex < TX_MU_GROUPS; groupIndex++)
	{
		for(uspIndex = 0; uspIndex < MAX_USP_IN_VHT_GROUP; uspIndex++)
		{
			updateLaDbDistributionParam(&laDbDistributionParameter,groupIndex,uspIndex, FALSE);
			estimatorsResetDb(&laDbDistributionParameter); 
		}
	}
#ifdef ENET_INC_ARCH_WAVE600
#if defined (ENET_INC_LMAC0)

	/* HE MU */	
	for(groupIndex = 0; groupIndex < NUM_OF_LA_HE_MU_DB_ENTRIES; groupIndex++)
	{
		for(uspIndex = 0; uspIndex < MAX_USP_IN_HE_GROUP; uspIndex++)
		{
			updateLaDbDistributionParam(&laDbDistributionParameter,groupIndex,uspIndex, TRUE);
			estimatorsResetDb(&laDbDistributionParameter); 
		}
	}
#endif
#endif
	memset32((uint32 *)(muCandidateStaBitmap), 0x0, SIZE_OF_STATIONS_BITMAP_IN_WORDS);	

	memset(&DynamicBwStatistics, 0, sizeof(DynamicBwStatistics_t));	
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************

estimatorsResetDb 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/

void estimatorsResetDb(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	//memset(&(LinkAdaptationStaDatabase[stationIndex].laStaUspCommon.stationEstimatorsDataBase),0,sizeof(stationEstimatorsDataBase_t));

	/*Reset fast probing estimators estimators DB*/
	estimatorsResetFastProbingEstimators(laDbDistributionParameter);

	/*Reset slow probing estimators estimators DB*/
	estimatorsResetSlowProbingEstimators(laDbDistributionParameter);

	/*Reset workingpoint estimators DB*/
	
	estimatorsResetWorkingPoint(laDbDistributionParameter->laStaUspCommon,laDbDistributionParameter->laStaGroupCommon);

	laDbDistributionParameter->laStaUspCommon->longTermPerFilter = 100;
	laDbDistributionParameter->laStaUspCommon->longTermVarFilter = 100;
	
	
}
/**********************************************************************************

estimatorsResetRatesEstimators 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/

void estimatorsResetRatesEstimators(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{

	/*Reset fast probing estimators estimators DB*/
	estimatorsResetFastProbingFilters(laDbDistributionParameter);

	/*Reset workingpoint estimators DB*/
	estimatorsResetWorkingPoint(laDbDistributionParameter->laStaUspCommon,laDbDistributionParameter->laStaGroupCommon);

	
	
}

/**********************************************************************************

estimatorsResetVapDb 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
static void estimatorsResetVapDb(uint8 vapId)
{	
	vapEstimatorsDataBase_t* vapLaDbEntryPtr = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase);
	memset((void*)vapLaDbEntryPtr,0,sizeof(vapEstimatorsDataBase_t));
	
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************

estimatorsResetSlowProbingEstimators 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/
void estimatorsResetSlowProbingEstimators(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	SlowProbingPointEstimators_t* pSlowEstimatorsDb = &(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators);

	AlphaFilter_Init(&(pSlowEstimatorsDb->averagePer),&(pSlowEstimatorsDb->averageDtFilterResult));
	pSlowEstimatorsDb->slowProbingRatePlusOne = INVALID_RATE_INDEX_VAL;
}
/**********************************************************************************

estimatorsResetFastProbingEstimators 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/
void estimatorsResetFastProbingEstimators(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	FastProbingPointEstimators_t* pFastEstimatorsDb;
	uint8 fastProbingIndex = 0;
	
	pFastEstimatorsDb = laDbDistributionParameter->laStaUspCommon->fastProbingEstimators;

	/*Init fastProbingEstimators DB - 4 rates*/
	for (fastProbingIndex=0; fastProbingIndex < TOTAL_NUMBER_OF_RATES_IN_DB; fastProbingIndex++)
	{
#ifdef LINK_ADAPTATION_LOGS
//				ILOG0_D("estimatorsResetDb AlphaFilter Init, fast probing index %d", fastProbingIndex);
#endif
		pFastEstimatorsDb[fastProbingIndex].rateIndex = INVALID_RATE_INDEX_VAL;
		AlphaFilter_Init(&(pFastEstimatorsDb[fastProbingIndex].averagePer),&(pFastEstimatorsDb[fastProbingIndex].averageDtFilterResult));
		AlphaFilter_Init(&(pFastEstimatorsDb[fastProbingIndex].perVariance),&(pFastEstimatorsDb[fastProbingIndex].averageDtFilterResult));
	}
	/*Init fastProbingPointEstimatorsIndexes array*/
	memset(laDbDistributionParameter->laStaUspCommon->fastProbingPointEstimatorsIndexes,INVALID_ESTIMATORS_DB_INDEX_VAL, TOTAL_NUMBER_OF_RATES_IN_DB);

}
/**********************************************************************************

estimatorsResetFastProbingFilters 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/
void estimatorsResetFastProbingFilters(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	FastProbingPointEstimators_t* pFastEstimatorsDb;
	uint8 fastProbingIndex = 0;
	
	pFastEstimatorsDb = laDbDistributionParameter->laStaUspCommon->fastProbingEstimators;

	/*Init fastProbingEstimators DB - 4 rates*/
	for (fastProbingIndex=0; fastProbingIndex < TOTAL_NUMBER_OF_RATES_IN_DB; fastProbingIndex++)
	{
#ifdef LINK_ADAPTATION_LOGS
//				("estimatorsResetDb AlphaFilter Init, fast probing index %d", fastProbingIndex);
#endif
		AlphaFilter_Init(&(pFastEstimatorsDb[fastProbingIndex].averagePer),&(pFastEstimatorsDb[fastProbingIndex].averageDtFilterResult));
		AlphaFilter_Init(&(pFastEstimatorsDb[fastProbingIndex].perVariance),&(pFastEstimatorsDb[fastProbingIndex].averageDtFilterResult));
	}
#ifdef LINK_ADAPTATION_LOGS

	//("estimatorsResetFastProbingFilters");
#endif
}

/**********************************************************************************

resetCurrentPacketParams 


Description:
------------


Input: 
-----

Returns:
--------
	
**********************************************************************************/
static void resetCurrentPacketParams(void)
{
	memset(&CurrentPacketParams,0,sizeof(CurrentPacketParameters_t));

	CurrentPacketParams.stationIndex = MAX_UINT8;
	CurrentPacketParams.bandwidth = BANDWIDTH_TWENTY;
	CurrentPacketParams.currentPer = MAX_UINT8;
}


/**********************************************************************************
nonHtVhtSetProbingpoints

Description:
------------

Input: 
-----


Returns:
--------
	
**********************************************************************************/
void nonHtVhtSetProbingPoints(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t RaMask, uint32* newRateIndxMask, uint8 rateIndex)
{
	uint8 newRateIndex;
	uint32 tempAndResultBitmap[2] = {0};
	RateMask_t rateMaskHtVht = LA_PHY_HT_VHT_RATES_MASK;
	RateMask_t raMaskAndHtVhtResult;
	StaId staIndex = laDbDistributionParameter->stationOrGroupIndex;
	
	/*set 1 rate up*/
	newRateIndex = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, bandwidth, rateIndex,RaMask,1, FALSE);
	/*set the corresponding bit in tempAndResultBitmap*/
	Utils_SetBitInBitmap(tempAndResultBitmap, newRateIndex);
	
	/*set another 2 rates up*/
	newRateIndex = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, bandwidth, rateIndex,RaMask,3, FALSE);
	/*set the corresponding bit in tempAndResultBitmap*/
	Utils_SetBitInBitmap(tempAndResultBitmap, newRateIndex);

	andOperator64Bit(RaMask.raIndexMask64bit,rateMaskHtVht.raIndexMask64bit,raMaskAndHtVhtResult.raIndexMask64bit);
	if ((raMaskAndHtVhtResult.raIndexMask64bit[0] != 0) || (raMaskAndHtVhtResult.raIndexMask64bit[1] != 0))
	{
		/*set the lowest rate after exlcuding 11A/B/G rates, in order to probe HT/VHT rate straight from 1Mbps rate */
		newRateIndex = getLowestRateinMask(staIndex, raMaskAndHtVhtResult,BANDWIDTH_TWENTY,MIN_HT_VHT_RATE_INDEX,MAX_VHT_SORT_RATE_INDEX);		
		/*set the corresponding bit in tempAndResultBitmap*/
		Utils_SetBitInBitmap(tempAndResultBitmap, newRateIndex);
	}

	/*set rate down */
	newRateIndex = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, bandwidth, rateIndex,RaMask,1, FALSE);
	/*set the corresponding bit in tempAndResultBitmap*/
	Utils_SetBitInBitmap(tempAndResultBitmap, newRateIndex);
	/*Set new rates bits in newRateIndxMask*/
	orOperator64Bit(tempAndResultBitmap,newRateIndxMask, newRateIndxMask);
	
}

/**********************************************************************************

htVhtSetProbeUpPointsInBitMask 


Description:
------------
Set two up probing point in newRateIndxMask, according to the fllowing algorithm:
Try Up Mcs
	 if not in RA mask 
		Find next phy rate index
Try up NSS (according to chosen NCS)
	if in RA mask
		done;
	else
		Try another up MCS for second point

Input: 
-----
pRaMask - RA rates mask according to station capabilities
newRateIndxMask - bitmask of the new probing rates options
nss, mcs - working point MCS & NSS


Returns:
--------
bool - 1 if new mask is different from zero, 1 otherwise
newRateIndxMask - pointer to newRateIndxMask array
	
**********************************************************************************/
static uint32 htVhtSetProbeUpPointsInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex)
{
	uint8 newRateIndexUpMcs = currentRateIndex;
	uint8 newRateIndexUpNss = currentRateIndex;
	
	/*Set up MCS index in newRateIndxMask*/
	/***************************************/
	newRateIndexUpMcs = htVhtSetProbeUpMcsInBitMask(laDbDistributionParameter, bandwidth, raMask, newRateIndxMask, currentRateIndex);
	/*Set up NSS index in newRateIndxMask*/
	/***************************************/
	newRateIndexUpNss = htVhtSetProbeUpNssInBitMask(laDbDistributionParameter, bandwidth, raMask, newRateIndxMask, currentRateIndex);

	if ((newRateIndexUpNss == currentRateIndex) || (newRateIndexUpMcs == currentRateIndex))
	{
		newRateIndexUpNss = MAX(newRateIndexUpMcs, newRateIndexUpNss);

		/*Set another Mcs up*/
		htVhtSetProbeUpMcsInBitMask(laDbDistributionParameter, bandwidth, raMask, newRateIndxMask, newRateIndexUpNss);	

	}
	/*Set down NSS up MCS index in newRateIndxMask*/
	/***************************************/
	htVhtSetProbeDownNssUpMcs(laDbDistributionParameter, bandwidth, raMask, newRateIndxMask, currentRateIndex);
	return (newRateIndxMask[0] | newRateIndxMask [1]);
}

/**********************************************************************************

htVhtSetProbeUpMcsInBitMask 


Description:
------------
Set Up Mcs bit in newRateIndxMask
Input: 
-----
pRaMask - RA rates mask according to station capabilities
newRateIndxMask - bitmask of the new probing rates options
nss, mcs - working point MCS & NSS


Returns:
--------
newRateIndexUpMcs 	
**********************************************************************************/
static uint8 htVhtSetProbeUpMcsInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex)
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	uint8 nssNum = ratesTable[currentRateIndex].numberOfNss;
	RateMask_t tempRaMask;
	uint8 newRateIndexUpMcs = currentRateIndex;
	RateMask_t* enableNssMaskArr = enableHtVhtNssRatesInMask;

#ifdef ENET_INC_ARCH_WAVE600
	if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta == TRUE)
	{
		enableNssMaskArr = enableHeNssRatesInMask;
	}
#endif

/*Set up MCS index in newRateIndxMask*/
/***************************************/
	MEMCPY (&tempRaMask, &raMask, sizeof(RateMask_t));
	/*AND between enableNssRatesInMask to RA Mask- result saved in tempRaMask*/
	andOperator64Bit(tempRaMask.raIndexMask64bit, enableNssMaskArr[nssNum].raIndexMask64bit,tempRaMask.raIndexMask64bit);
	newRateIndexUpMcs = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, bandwidth,currentRateIndex, tempRaMask,1, FALSE);
	/*set the corresponding bit in tempAndResultBitmap*/
	Utils_SetBitInBitmap(newRateIndxMask, newRateIndexUpMcs);

	/*Check if rate index is the highest in the current NSS, if it is, skip the search*/
	return newRateIndexUpMcs;
}

/**********************************************************************************

htVhtSetProbeUpNssInBitMask 


Description:
------------
Set Up Nss bit in newRateIndxMask
Input: 
-----
pRaMask - RA rates mask according to station capabilities
newRateIndxMask - bitmask of the new probing rates options
nss, mcs - working point MCS & NSS


Returns:
--------
newRateIndexUpNss 	
**********************************************************************************/
static uint8 htVhtSetProbeUpNssInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex)
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	uint8 nssNum = ratesTable[currentRateIndex].numberOfNss;
	RateMask_t tempRaMask;
	uint8 newRateIndexUpNss = currentRateIndex;
	RateMask_t* enableNssMaskArr = enableHtVhtNssRatesInMask;
	
#ifdef ENET_INC_ARCH_WAVE600
	if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta == TRUE)
	{
		enableNssMaskArr = enableHeNssRatesInMask;
	}
#endif
/*Set up NSS index in newRateIndxMask*/
/***************************************/
	/*Enable  nss +1 rates and search for closeset phy rate at the higher NSS*/
	if ((nssNum+1) < VHT_HE_NUMBER_OF_NSS)
	{
		MEMCPY (&tempRaMask, &raMask, sizeof(RateMask_t));
		/*AND between enableNssRatesInMask to RA Mask- result saved in tempRaMask*/
		andOperator64Bit(tempRaMask.raIndexMask64bit, enableNssMaskArr[nssNum+1].raIndexMask64bit,tempRaMask.raIndexMask64bit);
		/*Get higher NSS with the colsest phy rate*/
		newRateIndexUpNss = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, bandwidth,currentRateIndex, tempRaMask,1, FALSE);
		/*Set the corresponding bit in tempAndResultBitmap*/
		Utils_SetBitInBitmap(newRateIndxMask, newRateIndexUpNss);
	}
	return newRateIndexUpNss;
}
/**********************************************************************************

htVhtSetProbeDownNssUpMcs


Description:
------------
Set Up Nss bit in newRateIndxMask
Input: 
-----
pRaMask - RA rates mask according to station capabilities
newRateIndxMask - bitmask of the new probing rates options
nss, mcs - working point MCS & NSS


Returns:
--------
newRateIndexUpNss 	
**********************************************************************************/
static uint8 htVhtSetProbeDownNssUpMcs(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex)
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	int8 nssNum = ratesTable[currentRateIndex].numberOfNss - 1;
	RateMask_t tempRaMask;
	uint8 newRateIndexDownNssUpMcs = currentRateIndex;
	RateMask_t* enableNssMaskArr = enableHtVhtNssRatesInMask;

#ifdef ENET_INC_ARCH_WAVE600
	if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta == TRUE)
	{
		enableNssMaskArr = enableHeNssRatesInMask;
	}
#endif

/*Set down NSS index in newRateIndxMask*/
/***************************************/
	/*Enable  nss - 1 rates and search for closeset phy rate at the higher NSS*/
	if (nssNum >= 0)
	{
		MEMCPY (&tempRaMask, &raMask, sizeof(RateMask_t));
		/*AND between enableNssRatesInMask to RA Mask- result saved in tempRaMask*/
		andOperator64Bit(tempRaMask.raIndexMask64bit, enableNssMaskArr[nssNum].raIndexMask64bit,tempRaMask.raIndexMask64bit);
		/*Get lower NSS with the colsest phy rate*/
		newRateIndexDownNssUpMcs = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, bandwidth,currentRateIndex, tempRaMask,1, FALSE);
		/*Set the corresponding bit in tempAndResultBitmap*/
		Utils_SetBitInBitmap(newRateIndxMask, newRateIndexDownNssUpMcs);
	}
	return newRateIndexDownNssUpMcs;
}

/**********************************************************************************

htVhtSetProbeDownPointInBitMask 


Description:
------------
Set two up probing point in newRateIndxMask., according to the fllowing algorithm:
Try Down Mcs
	 if not in RA mask 
		Find next phy rate index
Try up NSS (according to chosen NCS)
	if in RA mask
		done;
	else
		Try another up MCS for second point

Input: 
-----
pRaMask - RA rates mask according to station capabilities
newRateIndxMask - bitmask of the new probing rates options
nss, mcs - working point MCS & NSS


Returns:
--------
bool - 1 if new mask is different from zero, 1 otherwise
newRateIndxMask - pointer to newRateIndxMask array
	
**********************************************************************************/
static uint32 htVhtSetProbeDownPointInBitMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, RateMask_t raMask, uint32* newRateIndxMask, uint8 currentRateIndex)
{
	RateMask_t tempRaMask;
	uint8 newRateIndexDownMcs = currentRateIndex;
	RateMask_t rateMaskZero11bRates = LA_PHY_ZERO_11B_RATES_MASK;

/*Set down MCS index in newRateIndxMask*/
/***************************************/

	/*First search down mcs without 11B rates*/
	andOperator64Bit(raMask.raIndexMask64bit, rateMaskZero11bRates.raIndexMask64bit,tempRaMask.raIndexMask64bit);
	newRateIndexDownMcs = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, bandwidth,currentRateIndex, tempRaMask,1, FALSE);

	/*Set 11B rate in down probing point, if rate was  not found at the HT/VHT rates, and BW is 20 MHz*/
	if (newRateIndexDownMcs == currentRateIndex)
	{
		if (bandwidth == BANDWIDTH_TWENTY)
		{
			newRateIndexDownMcs = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, bandwidth,currentRateIndex, raMask,1,FALSE);
		}
	}
	
	/*set the corresponding bit in tempAndResultBitmap*/
	Utils_SetBitInBitmap(newRateIndxMask, newRateIndexDownMcs);
return (newRateIndxMask[0] | newRateIndxMask[1]);
}
/**********************************************************************************

setCurrentRateIndexBitmask 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void setCurrentRateIndexBitmask (FastProbingPointEstimators_t* fastProbingEstimators, uint32* currentRateBitmask)
{
	uint8 	estimatorsDbRateIndex = 0;
	uint8 rateIndex;
	
	for(estimatorsDbRateIndex = 0; (estimatorsDbRateIndex < TOTAL_NUMBER_OF_RATES_IN_DB); estimatorsDbRateIndex++)
	{
		rateIndex = fastProbingEstimators[estimatorsDbRateIndex].rateIndex;
		if (rateIndex != INVALID_RATE_INDEX_VAL)
		{
			Utils_SetBitInBitmap(currentRateBitmask, rateIndex);
		}
	}
}
/**********************************************************************************

removeRateEntryFromEstimatorsBwDb 


Description:
------------
Remove estimators DB rates according to removedIndexesBitmap
Input: 
-----

Returns:
--------
	
**********************************************************************************/
static void removeRateEntryFromEstimatorsBwDb (FastProbingPointEstimators_t* fastProbingEstimators, uint32* removedIndexesBitmap)
{
	
	uint8 firstSetBitIndex = 0;
	uint8 estimatorsRateDbentry;
	
	while (removedIndexesBitmap[0] | removedIndexesBitmap [1])
	{
		/*Find index of the first 1 in bitmap*/
		firstSetBitIndex = Utils_CountTrailZeroBitmap(removedIndexesBitmap, firstSetBitIndex, 8);
		/*Reset bit*/
		Utils_ZeroBitInBitmap(removedIndexesBitmap, firstSetBitIndex);
		/*Find estimator DB entry of the rate index that need to be removed*/
		estimatorsRateDbentry = findRateEntryInFastProbingEstDb(fastProbingEstimators, firstSetBitIndex);
		/*Sign Entry as invald*/
		fastProbingEstimators[estimatorsRateDbentry].rateIndex = INVALID_RATE_INDEX_VAL;
		/*Clear filter before add new rate*/
		AlphaFilter_Init(&(fastProbingEstimators[estimatorsRateDbentry].averagePer),&(fastProbingEstimators[estimatorsRateDbentry].averageDtFilterResult));
		AlphaFilter_Init(&(fastProbingEstimators[estimatorsRateDbentry].perVariance),&(fastProbingEstimators[estimatorsRateDbentry].averageDtFilterResult));
#ifdef LINK_ADAPTATION_LOGS
//		ILOG0_DD("removeRateEntryFromEstimatorsBwDb, rateIndex = %d, filterResult = %d",firstSetBitIndex, pEstimatorsDbBwData->fastProbingEstimators[estimatorsRateDbentry].averagePer.filterResult);
#endif
		firstSetBitIndex++;
	}
	
}

/**********************************************************************************

addNewRateToEstimatorsBwDb 


Description:
------------
Add new rates to estimator DB according to addedIndexBitmap
Input: 
-----

Returns:
--------
	
**********************************************************************************/
static void addNewRateToEstimatorsBwDb (FastProbingPointEstimators_t* fastProbingEstimators, uint32* addedIndexBitmap)
{
	
	uint8 firstSetBitIndex = 0;
	uint8 freeEntryInEstDb = 0;
	
	while (addedIndexBitmap[0] | addedIndexBitmap [1])
	{
		/*Find index of the first 1 in bitmap*/
		firstSetBitIndex = Utils_CountTrailZeroBitmap(addedIndexBitmap, firstSetBitIndex, 8);
		/*Reset bit*/
		Utils_ZeroBitInBitmap(addedIndexBitmap, firstSetBitIndex);
		/*Get the first free entry*/
		freeEntryInEstDb =findRateEntryInFastProbingEstDb(fastProbingEstimators, INVALID_RATE_INDEX_VAL);

		DEBUG_ASSERT(freeEntryInEstDb != MAX_UINT8);

		/*Add new rate to estimators DB*/
		fastProbingEstimators[freeEntryInEstDb].rateIndex = firstSetBitIndex;
#ifdef LINK_ADAPTATION_LOGS
//		//("addNewRateToEstimatorsBwDb, rateIndex = %d, filterResult = %d",firstSetBitIndex, pEstimatorsDbBwData->fastProbingEstimators[freeEntryInEstDb].averagePer.filterResult);
#endif
		firstSetBitIndex++;

	}
}
/**********************************************************************************

findRateEntryInFastProbingEstDb 


Description:
------------
Go over all rates in fast probing dB and fnd the rates that need to be removed
Input: 
-----

Returns:
--------
	
**********************************************************************************/

static uint8 findRateEntryInFastProbingEstDb ( FastProbingPointEstimators_t* fastProbingEstimators, uint8 rateIndexInEstimatorsDb)
{
	uint8 tableIndex = 0;
	
	/*Find rate the rate that need to be removed in the fast probing estimators DB*/
	while ((tableIndex < TOTAL_NUMBER_OF_RATES_IN_DB) && (fastProbingEstimators[tableIndex].rateIndex != rateIndexInEstimatorsDb)) 
	{
		tableIndex++;
	}

	return (tableIndex < TOTAL_NUMBER_OF_RATES_IN_DB) ? tableIndex : INVALID_RATE_INDEX_VAL;
}


/**********************************************************************************

htVhtSetProbeHighUpPointInBitMask 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
/*

bool ag11bDefineProbeHighUpPoint11Bag(StaId stationIndex, uint32* raIndexMask, uint8 rate,Bandwidth_e bandwidth)
{
}
*/
/**********************************************************************************

updateRateinEstimatorsDb 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void estimatorsUpdateRatesInDb(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 newWorkingPointRateIndex, uint8 lastWpRateindex)
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	uint32	newRatesIndexMask[2] = {0};
	uint32	currentRatesIndexMask[2] = {0};
	uint32 currentNewDiffBitmask[2] = {0};
	uint32 removedIndexesBitmap[2] = {0};
	uint32 addedIndexesBitmap[2] = {0};
	RateMask_t raMask;
	/*Extract phy mode*/
	LinkAdaptationPhyMode_e laPhyMode = ratesTable[newWorkingPointRateIndex].laPhyMode;
	uint8 workingPointIndexInEstDb;
	Bandwidth_e bandwidth = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);

	getStaRateMaskPerBw(laDbDistributionParameter, bandwidth, laDbDistributionParameter->laStaUspCommon->raIndexMask, &raMask, FALSE);

	/*Set turbo rates in local raMask*/
	if( (laDbDistributionParameter->laStaUspCommon != NULL)&& (FALSE == laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta))
	{
		/*Turbo rates is not relevant for he stations or VAP*/
		TurboRatesGetBitmap(laDbDistributionParameter->stationOrGroupIndex, laDbDistributionParameter->uspIndex,  bandwidth, &raMask);
	}
#ifdef LINK_ADAPTATION_LOGS
//	ILOG0_DDDDD("estimatorsUpdateRatesInDb - START PROCESSING, stationIndex = %d, bandwidth = %d, new working point rate = %d, raMask = %X-%X", stationIndex, bandwidth,newWorkingPointRateIndex, raMask[1], raMask[0]);
	
#endif
	switch (laPhyMode)
	{
		case LA_PHY_MODE_HE:
		case LA_PHY_MODE_HT_VHT:	
			/*Define new probing points and set it in the newRatesIndexMask bitmask*/
			htVhtSetProbeUpPointsInBitMask(laDbDistributionParameter, bandwidth, raMask, newRatesIndexMask, newWorkingPointRateIndex);
			/*If last WP is not on lower NSS, set DOWN_PORBING_POINT as lower MCS */
			if ((ratesTable[newWorkingPointRateIndex].numberOfNss <= ratesTable[lastWpRateindex].numberOfNss) ||
				(isBitInMask64Bit(lastWpRateindex,raMask.raIndexMask64bit) == FALSE))
			{
				htVhtSetProbeDownPointInBitMask(laDbDistributionParameter, bandwidth, raMask, newRatesIndexMask, newWorkingPointRateIndex);
			}
			else if (isBitInMask64Bit(lastWpRateindex,raMask.raIndexMask64bit))
			{
				/*If last WP is on lower NSS and last WP is in rates mask, set DOWN_PORBING_POINT as last WP rate*/
				Utils_SetBitInBitmap(newRatesIndexMask, lastWpRateindex);
			}
			break;
		case LA_PHY_MODE_11B:
		case LA_PHY_MODE_11AG:
			nonHtVhtSetProbingPoints(laDbDistributionParameter, bandwidth, raMask, newRatesIndexMask, newWorkingPointRateIndex);
			break;
		default:
			break;
	}
	/*Set working point in newRatesBitMask*/
	Utils_SetBitInBitmap(newRatesIndexMask, newWorkingPointRateIndex);
	ASSERT(laDbDistributionParameter->laStaUspCommon != NULL) //KW_FIX_FW_G
	
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_D("[estimatorsUpdateRatesInDb], laDbDistributionParameter->stationOrGroupIndex = %X", laDbDistributionParameter->stationOrGroupIndex);
	ILOG0_D("[estimatorsUpdateRatesInDb], laDbDistributionParameter->uspIndex = %X", laDbDistributionParameter->uspIndex);

	ILOG0_DD("[estimatorsUpdateRatesInDb], newRatesIndexMask= %X-%X",newRatesIndexMask[1], newRatesIndexMask[0]); 
#endif
	/*Define current rates index bitmask*/
	setCurrentRateIndexBitmask(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators, currentRatesIndexMask);
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DD("[estimatorsUpdateRatesInDb], currentRatesIndexMask= %X-%X",currentRatesIndexMask[1], currentRatesIndexMask[0]); 
#endif
	/*Find changed rates*/
	xorOperator64Bit(currentRatesIndexMask,newRatesIndexMask,currentNewDiffBitmask);
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DD("[estimatorsUpdateRatesInDb], currentNewDiffBitmask= %X-%X",currentNewDiffBitmask[1], currentNewDiffBitmask[0]); 
#endif
	/*Find  rates that need to be added - bitwize and between diff and new rates*/
	andOperator64Bit(currentNewDiffBitmask,newRatesIndexMask,addedIndexesBitmap);
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DD("[estimatorsUpdateRatesInDb], addedIndexesBitmap= %X-%X",addedIndexesBitmap[1], addedIndexesBitmap[0]); 
#endif
	/*Find rates that need to be removed - bitwize and between diff and current rates*/
	andOperator64Bit(currentNewDiffBitmask,currentRatesIndexMask,removedIndexesBitmap);
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DD("[estimatorsUpdateRatesInDb], removedIndexesBitmap= %X-%X",removedIndexesBitmap[1], removedIndexesBitmap[0]); 
#endif
	/*Remove entry from DB*/
	removeRateEntryFromEstimatorsBwDb(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators, removedIndexesBitmap);
	/*Add new entry to DB */
	addNewRateToEstimatorsBwDb(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators, addedIndexesBitmap);

	/*Set rate type (probing type / working point)*/
	setEstimatorDbRatesType(laDbDistributionParameter->laStaUspCommon, newWorkingPointRateIndex, lastWpRateindex);
	
#ifdef LINK_ADAPTATION_LOGS
//	ILOG0_DD("estimatorsUpdateRatesInDb, Estimators DB after set new rates, staIndex = %d, bandwidth = %d", stationIndex, bandwidth);
//	SLOG0(0, 0, EstimatorsBwData_t, &pEstimatorsDbBwData);
#endif

	/*Update rate adaptation rate indexes*/
	workingPointIndexInEstDb = laDbDistributionParameter->laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	ASSERT(workingPointIndexInEstDb < TOTAL_NUMBER_OF_RATES_IN_DB);

}
/**********************************************************************************

setEstimatorDbRatesType 


Description:
----------
Function goes over estimators DB (for specific BE), sort the DB according to sortedPhyRateIndex, and define different rates type according to sorted rates
If probing rate is not available fir probing set the working point index

Input: 
-----
estimatorsBwDataDb - pointer to fastProbingEstimators for specific BW

Returns:
--------
	void - 
	
**********************************************************************************/
static void setEstimatorDbRatesType(LaStationUspCommonParams_t* laStaUspCommon, uint8 workingPointRateIndex, uint8 lastWpRateIndex)
{
	const RateObj_t* ratesTable = getRatesTable(laStaUspCommon);
	uint8 maxNumberOfRates = getMaxNumOfRatesInTable(laStaUspCommon);
	uint16 estimatorDbIndex = 0;
	uint8* pFastProbingPointEstimatorsIndexes = laStaUspCommon->fastProbingPointEstimatorsIndexes;
	uint8 currentRateIndex =0;
	uint8 maxRateindex = 0;
	UNUSED_PARAM(lastWpRateIndex);	
	/*Initiate sortedEstimatorTableIndex with  working point index - so if there is no rate to probe, the working point will be used*/

	memset(pFastProbingPointEstimatorsIndexes, INVALID_ESTIMATORS_DB_INDEX_VAL,TOTAL_NUMBER_OF_RATES_IN_DB);
	/*Go over estimators DB and define fastProbingPointEstimatorsIndexes (LOWER_RATE, UP_MCS, UP_NSS)*/
	
	for(estimatorDbIndex = 0 ; estimatorDbIndex < TOTAL_NUMBER_OF_RATES_IN_DB; estimatorDbIndex++)
	{
		currentRateIndex = laStaUspCommon->fastProbingEstimators[estimatorDbIndex].rateIndex;
		if (currentRateIndex < maxNumberOfRates)
		{
			if (currentRateIndex == workingPointRateIndex)
			{
				pFastProbingPointEstimatorsIndexes[WORKING_POINT] = estimatorDbIndex;
			}
			else if(ratesTable[currentRateIndex].sortPhyRateSameBwIndex < ratesTable[workingPointRateIndex].sortPhyRateSameBwIndex)
			{
				pFastProbingPointEstimatorsIndexes[LOWER_RATE] = estimatorDbIndex;
			}
			else if ((currentRateIndex > workingPointRateIndex) && 
						(currentRateIndex > maxRateindex))
			{
				/*If Up_NSS was not valid it will be defined as the higher rate, and UP_MCS will be the second one*/
				maxRateindex = currentRateIndex;
				pFastProbingPointEstimatorsIndexes[UP_MCS] = pFastProbingPointEstimatorsIndexes[UP_NSS];
				pFastProbingPointEstimatorsIndexes[UP_NSS] = estimatorDbIndex;
			}
			else if ((ratesTable[currentRateIndex].numberOfNss < ratesTable[workingPointRateIndex].numberOfNss) && 
						(ratesTable[currentRateIndex].vhtHeMcs > ratesTable[workingPointRateIndex].vhtHeMcs))
			{
				pFastProbingPointEstimatorsIndexes[DOWN_NSS_UP_MCS] = estimatorDbIndex;
				
			}
			else
			{
				pFastProbingPointEstimatorsIndexes[UP_MCS] = estimatorDbIndex;
			}
		}
	}
	
	DEBUG_ASSERT(pFastProbingPointEstimatorsIndexes[WORKING_POINT] < TOTAL_NUMBER_OF_RATES_IN_DB);
	
}

/**********************************************************************************

estimatorsUpdateRatesInDbForAllBws 


Description:
------------
Function gets the new working point (at a specific bw) find the equivalent rates in two other Bws and call estimatorsUpdateRatesInDb for update each BW with new working point and probing points
The equiivalent rate for other Bws is different if rate was increased or decreased

Input: 
-----
staIndex
newRateIndex 
newBandwidth - Bw with the best result
pLastWpRatesData - pointer to EstimatorsBwData_t object of the last WP parameters (for 3 Bws)

Returns:
--------
	void - 
	
**********************************************************************************/
void estimatorsUpdateRatesInDbForAllBws(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 newWorkingPointRateIndex, uint8 previousWpRateindex, Bandwidth_e mainBw)
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	Bandwidth_e bandwidth = BANDWIDTH_TWENTY;
	uint8 rateIndexToUpdateInDb = MAX_UINT8;
	Bandwidth_e maxBandwidth = LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	LinkAdaptationPhyMode_e laPhyMode = ratesTable[newWorkingPointRateIndex].laPhyMode;
	bool isPhyModeHtVhtHe = (laPhyMode == LA_PHY_MODE_HT_VHT) || (laPhyMode == LA_PHY_MODE_HE);
	
#ifdef LINK_ADAPTATION_LOGS
		//("estimatorsUpdateRatesInDbForAllBws");
#endif

	mainBw = isPhyModeHtVhtHe ? mainBw : BANDWIDTH_TWENTY;

/*Extract the the new equivalent working points in the 2 other bandwidth*/
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= maxBandwidth; bandwidth++)
	{
		/*Set rate index for the main BW otherwise override it*/
		rateIndexToUpdateInDb = newWorkingPointRateIndex;
		estimatorsUpdateRatesInDb(laDbDistributionParameter, rateIndexToUpdateInDb, previousWpRateindex);
		
	}
}
/****************************************************************************
 **
 ** NAME:        isBitInMask64Bit
 **
 ** PARAMETERS:
 ** RETURN VALUES: bool bit is in bitmask or not
 **
 ** DESCRIPTION: 
 **              
 **
 **   
 ***************************************************************************/
uint32 isBitInMask64Bit(uint8 bit, uint32* pMask)
{	
	uint32 compressedResult = 0;
	uint32 tempAndResultBitmap[2] = {0};

	Utils_SetBitInBitmap(tempAndResultBitmap, bit);
	/*Check if bit is in raIndexMask*/
	compressedResult = andOperator64Bit(tempAndResultBitmap, pMask,tempAndResultBitmap);

	return compressedResult;
}


/****************************************************************************
 **
 ** NAME:        estimatorsCalcEffectiveTp
 **
 ** PARAMETERS:  
 **
 ** RETURN VALUES: 
 **
 ** DESCRIPTION: 
 **              
 **
 **   
 ***************************************************************************/
uint16 estimatorsCalcEffectiveTp(uint16 Lossless,uint16 Retries)
{ 
	//int32 effectiveRate= ( ( Lossless * ( POLYNOMIAL_MULTIPLY_FACTOR * HUNDRED_PERCENT ) - Lossless * ( POLYNOMIAL_COEFFICIENT_A0 * ( Retries - POLYNOMIAL_POINT ) * ( Retries - POLYNOMIAL_POINT ) + POLYNOMIAL_COEFFICIENT_A1 * ( Retries - POLYNOMIAL_POINT ) + POLYNOMIAL_COEFFICIENT_A2 ) ) / ( POLYNOMIAL_MULTIPLY_FACTOR * HUNDRED_PERCENT ) );
	int32 effectiveTp;

	DEBUG_ASSERT(Retries <= HUNDRED_PERCENT);
	effectiveTp = ((Lossless * (HUNDRED_PERCENT - Retries)) / HUNDRED_PERCENT); //We must use the divided by HUNDRED_PERCENT and not using right shift!!( we need the accurate value of effective rate because relevant rate function comapre it to LossLess value.)
	
	return effectiveTp;
}

    
/***********************************************************************
* estimatorsStatisticsChooseAlpha
* 
* Description:
* ------------
* choose alpha according to the number of msuds. e.g if num msdu = 1 use alpha of 1/8.
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
uint8 estimatorsStatisticsChooseAlpha(uint32 numMsdus)
{
	/* Assume num msdus is bigger than max value in table */
	uint8	ewmaAlpha	= 0;
	uint32	i 			= 0;

	/* Loop on alpha table */
	for (i = 0; i < EWMA_ALPHA_NUM_OF_CONVERSION; i++)
	{
		if (numMsdus < EwmaAlpha[i].numMsdus)
		{
			ewmaAlpha = EwmaAlpha[i].ewmaAlpha;
			break;
		}
	}
	
	return(ewmaAlpha);	
}    
/***********************************************************************
* estimatorsStatisticsUpdate
* 
* Description:
* ------------
* PLUME statistics: 
* 
* wifi_getRadioChannelStats -> ch_utilization_busy_ext     : In case CCA reduced the BW, acuumulate the duration
*
* wifi_getApAssociatedDeviceTidStatsResult -> ewma_time_ms : Get the number of MSDUs  
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void estimatorsStatisticsUpdate(BaaRateAdaptiveFullFifo_t* pBaaReport)
{

	uint32		phyRate					= 0;
	uint32		avgMsduLen				= 0;
	uint8		ewmaAlpha				= 0;
	uint32		ewmaTimeNsec			= 0;
	uint32 		oldEwmaTimeNsec 		= 0;
	uint32 		totalSuccessMpduLength	= pBaaReport->userReport.totalSuccessMpduLength;
	uint32		totalFailMpduLength		= pBaaReport->userReport.totalFailMpduLength;
	BandId_e	bandId 					= ConfigurationManager_GetMyBand();
	bool	 	isProbing 				= (pBaaReport->userReport.altRateReasonFastProbing | pBaaReport->userReport.altRateReasonSlowProbing);
	uint32		numMpdus				= pBaaReport->userReport.successMpduCnt + pBaaReport->userReport.failedMpduCnt;
	uint32		txQStaId				= pBaaReport->userReport.txQStaId;
	/* wifi_getRadioChannelStats */
	if (pBaaReport->commonReport.bwModifycca == TRUE)
	{
		ChannelStatistics.channelUtilizationStats[bandId].chUtilizationBusyExt += pBaaReport->commonReport.duration;
	}

	/* wifi_getApAssociatedDeviceTidStatsResult: ewma_time_ms and sum_time_ms	*/

	/* Monitor only non-probing packets */
	if ((isProbing == FALSE) && (totalSuccessMpduLength + totalFailMpduLength != 0))
	{
		/* ewma_time_ms: 	AVG MSDU TIME = AVG MSDU LEN / PHY RATE
							AVG MSDU LEN = MPDU LEN / NUM MSDU				
							PHY RATE = (total_length * 64 * 8) / duration
		*/

		/* PHY RATE = (total_length * 64 * 8) / duration. The 64 is since total_length 64B quantized. The 8 is since we need the phy rate in bit/sec and total_length is in bytes. */
		if (pBaaReport->commonReport.duration != 0)
		{
			/* multiply by 64 since phy rate is given in 64B quantization */
			phyRate = ((pBaaReport->userReport.totalLength << 6) << 3)/ pBaaReport->commonReport.duration; 
		}
		else
		{
			/* Something is wrong. Use constant */
			phyRate = ((pBaaReport->userReport.totalLength << 6) << 3) / STATISTICS_DURATION;
		}

		if (phyRate == 0)			
		{
			/* Something went wrong: init phy rate and store this block ack report for debug */
			phyRate = 1;
			MEMCPY(&BaaReport, pBaaReport, sizeof(BaaRateAdaptiveFullFifo_t));
		}

		/* AVG MSDU LEN = (MPDU LEN * 64) / NUM MSDU */
		if (pBaaReport->userReport.numMsdus != 0)
		{
			/* multiply by 64 since both lengths are given in 64B quantization. Multiply by 1024 to change to Nano seconds since in small packets we get less than 1 USec */
			avgMsduLen = (((totalSuccessMpduLength + totalFailMpduLength) << 6) << 10) / pBaaReport->userReport.numMsdus;
		}
		else
		{
			/* Something is wrong. Use constant */
			avgMsduLen = STATISTICS_MSDU_TIME * phyRate;
		}

		/* AVG MSDU TIME = AVG MSDU LEN / PHY RATE. */
		ewmaTimeNsec = avgMsduLen / phyRate;

#ifdef PLUME_TRACE
		ILOG0_DDD("[PLUME] estimatorsStatisticsUpdate ewmaTimeNsec %d = avgMsduLen %d / phyRate %d ", ewmaTimeNsec, avgMsduLen, phyRate);
#endif


		/* Get alpha */
		ewmaAlpha = estimatorsStatisticsChooseAlpha(pBaaReport->userReport.numMsdus);

		/* Perform Alpha filter on the results: alpha * current + (1-alpha) * old */	
		oldEwmaTimeNsec = LinkAdaptationStatistics.ewmaTimeNsec[txQStaId];

#ifdef PLUME_TRACE
		ILOG0_DD("[PLUME] estimatorsStatisticsUpdate ewmaAlpha = %d oldEwmaTimeNsec = %d", ewmaAlpha, oldEwmaTimeNsec);
#endif

		LinkAdaptationStatistics.ewmaTimeNsec[txQStaId] = (ewmaTimeNsec >> ewmaAlpha) + (oldEwmaTimeNsec - (oldEwmaTimeNsec >> ewmaAlpha));

#ifdef PLUME_TRACE
		ILOG0_DD("[PLUME] estimatorsStatisticsUpdate ewmaTimeNsec = %d station = %d ", LinkAdaptationStatistics.ewmaTimeNsec[txQStaId], txQStaId);
#endif


		/* Total MSDU time is equal to num msdu * msdu time. FW provides the cumulative value and Driver performes the delta. */
		LinkAdaptationStatistics.sumTimeNsec[txQStaId] += (ewmaTimeNsec * pBaaReport->userReport.numMsdus) ;	

#ifdef PLUME_TRACE		
		ILOG0_D("[PLUME] estimatorsStatisticsUpdate sumTimeNsec = %d", LinkAdaptationStatistics.sumTimeNsec[txQStaId]);
#endif

		if (numMpdus > 1)
		{
			/* There is more than 1 MPDU hence it is an Aggregation */
			LinkAdaptationStatistics.transmittedAmpdu[txQStaId]++; 
		}
#ifdef PLUME_TRACE		
		ILOG0_D("[PLUME] estimatorsStatisticsUpdate transmittedAmpdu = %d", LinkAdaptationStatistics.transmittedAmpdu[txQStaId]);
#endif
		/* Store BW */
		LinkAdaptationStatistics.bw[txQStaId] = pBaaReport->commonReport.bw;

#ifdef PLUME_TRACE		
		ILOG0_D("[PLUME] estimatorsStatisticsUpdate bw = %d", LinkAdaptationStatistics.bw[txQStaId]);
#endif

		/* Count the number of MPDUs */
		LinkAdaptationStatistics.mpduInAmpdu[pBaaReport->userReport.txQStaId] = pBaaReport->userReport.successMpduCnt + pBaaReport->userReport.failedMpduCnt;

#ifdef PLUME_TRACE		
		ILOG0_D("[PLUME] estimatorsStatisticsUpdate mpduInAmpdu = %d", LinkAdaptationStatistics.mpduInAmpdu[pBaaReport->userReport.txQStaId]);
#endif

		/* Count the number of MSDUs */
		LinkAdaptationStatistics.numMsdus[txQStaId] += pBaaReport->userReport.numMsdus;

#ifdef PLUME_TRACE		
		ILOG0_D("[PLUME] estimatorsStatisticsUpdate numMsdus = %d", LinkAdaptationStatistics.numMsdus[txQStaId]);
#endif
	}
}


/**********************************************************************************

estimatorsProcessBaaReport 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void estimatorsProcessBaaReport(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, BaaRateAdaptiveFullFifo_t* pBaaReport)
{
	uint8 newRateIndex;
	bool isProbingBa = (pBaaReport->userReport.altRateReasonFastProbing | pBaaReport->userReport.altRateReasonSlowProbing);

	if((pBaaReport->commonReport.txParamOveride) && (FALSE == isProbingBa))
	{
#ifdef LINK_ADAPTATION_LOGS
		//("estimatorsProcessBaaReport, rate changed, new rate = %d", pBaaReport->rate);
#endif		
			/*Extract rate index according to phy mode*/
#ifdef ENET_INC_ARCH_WAVE600D2
			newRateIndex = convertTcr2RateIndex((PhyMode_e)pBaaReport->commonReport.txPhyMode,pBaaReport->userReport.tcrMcsNssRate);
#else
			newRateIndex = convertTcr2RateIndex((PhyMode_e)pBaaReport->commonReport.phyMode,pBaaReport->userReport.tcrMcsNssRate);
#endif //ENET_INC_ARCH_WAVE600D2
			updateOverridePendingDb(OVERRIDE_RATE_INDEX, newRateIndex, laDbDistributionParameter);
 	}

	resetCurrentPacketParams();
	extractCurrentPacketBaaParams(pBaaReport->userReport.txQStaId, pBaaReport);


	if (enableExtraLogs)
	{	

		SLOG0(0, 0, LaStationGroupCommonParams_t, laDbDistributionParameter->laStaGroupCommon);
		SLOG0(0, 0, LaStationUspCommonParams_t, laDbDistributionParameter->laStaUspCommon);

		if (laDbDistributionParameter->uspIndex == INVALID_MU_GROUP_INDEX)
		{
			SLOG0(0, 0, LaStationUniqueParams_t, laDbDistributionParameter->laStationUnique);
		}
		else
		{
			SLOG0(0, 0, LaGroupUniqueParams_t, laDbDistributionParameter->laGroupUnique);
			
			if (laDbDistributionParameter->laVhtGroupUnique != NULL)
			{
				SLOG0(0, 0, LaVhtGroupUniqueParams_t, laDbDistributionParameter->laVhtGroupUnique);
			}
			if (laDbDistributionParameter->laHeGroupUnique != NULL)
			{
				SLOG0(0, 0, LaHeGroupUniqueParams_t, laDbDistributionParameter->laHeGroupUnique);
			}
		}
	}

	estimatorsStatisticsUpdate(pBaaReport);
	
}


/**********************************************************************************

extractCurrentPacketBaaParams 


Description:
------------
extract phy mode and rate index from BAA report
Input: 
-----
staIndex
pBaaReport: pointer to BAA report in fifo

Returns:
--------
	
**********************************************************************************/

static void extractCurrentPacketBaaParams(StaId staIndex, BaaRateAdaptiveFullFifo_t* pBaaReport)
{
	
	PhyMode_e phyMode;
	uint8 rateField;
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	uint16 currentTsfMod = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);
	uint32 *statAddr;

	CurrentPacketParams.currentPacketTsf = currentTsfMod;

#ifdef ENET_INC_ARCH_WAVE600D2
	phyMode = (PhyMode_e)pBaaReport->commonReport.txPhyMode;
#else
	phyMode = (PhyMode_e)pBaaReport->commonReport.phyMode;
#endif //ENET_INC_ARCH_WAVE600D2
	rateField = pBaaReport->userReport.tcrMcsNssRate;

	/*Extract station index*/
	CurrentPacketParams.stationIndex = staIndex;

	/*Extract station BW*/
	CurrentPacketParams.bandwidth =  (Bandwidth_e)pBaaReport->commonReport.bw;
//	DEBUG_ASSERT(CurrentPacketParams.bandwidth < LA_NUM_OF_BANDWIDTH);
	CurrentPacketParams.currentPacketMpduCnt = pBaaReport->userReport.successMpduCnt + pBaaReport->userReport.failedMpduCnt;

	/*Extract rate index according to phy mode*/
	CurrentPacketParams.rateIndex = convertTcr2RateIndex(phyMode,rateField);

	/*Update BAA TX error frame counters*/
	statAddr = (uint32 *)(LinkAdaptationVapFrameErrorCountBaseAddress + pBaaReport->userReport.txQVapId * 4);
	*statAddr += pBaaReport->userReport.failedMpduCnt;
	statAddr = (uint32 *)LinkAdaptationWlanFrameErrorCountBaseAddress;
	*statAddr += pBaaReport->userReport.failedMpduCnt;
}


/**********************************************************************************

estimatorsWorkingPointDbUpdate 


Description:
------------
Update estimators working point DB
Function update the follwinf estimaotrs:
PER,  reference PER, PER variance

Input: 
-----
pLinkAdaptationDb
pBaaReport


Returns:
--------
	
**********************************************************************************/

void estimatorsWorkingPointDbUpdate(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, BaaRateAdaptiveFullFifo_t*pBaaReport)
{
	uint8 currentPer;
	uint8 perVariance;
	uint8 averagePer;
	/*Extract estimators working point statistics index in estimators DB */
	AdaptiveAlphaFilterParams_t* pFilterParams = NULL;
	uint8 workingPointStatsIndex = laDbDistributionParameter->laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	FastProbingPointEstimators_t* pWorkingPointEstimator;
	LinkAdaptationConfigurationParams_t* pConfigurtionParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	/*Extract LnBeta from link adaptation configuration params*/
	uint32 lnBeta = pConfigurtionParams->lnBeta;
	bool isCurrentPacketFailed;
	bool isBwProbingActivate;
	bool isBfStateValid;
	/*Slow probing estimators DB ptr*/
	SlowProbingPointEstimators_t* pSlowProbingEstimators = &(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators);
	uint8	currentGoodput=0; 
	uint8 bfType;

#ifdef LA_CALC_REF_LEN
	int16 refLenSuccess;
	int16 refLenFail;
	uint8 refLength;
	uint8 calculatedRefLenFactor;
#endif
	uint16 perCollisionTh;
	uint16 varCollisionTh;
	bool isHeGroup;
	bfType = pBaaReport->userReport.phyTxBf;

	isBwProbingActivate = (laDbDistributionParameter->laStaGroupCommon->linkAdaptationState == LA_WAIT_FOR_SLOW_PROBE_VALID) && 
								(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTask == SLOW_PROBING_BW);

	/* Verify that BF state is valid (only relevent if BF loop is active) */ 
	isBfStateValid = validateBfTransmission(laDbDistributionParameter, bfType);
	estimatorsProcessCollisionsBaaReport(laDbDistributionParameter, pBaaReport->userReport.txQVapId,pBaaReport);

	

	/*Increment beamforming  counter - counts how many mpdus transmit at the current BF state, if grater than maximum val stay at the max for prevant wrap arround*/
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		laDbDistributionParameter->laStationUnique->beamFormingDb.numberOfMpdusSentInCurrentState = 
						MIN(laDbDistributionParameter->laStationUnique->beamFormingDb.numberOfMpdusSentInCurrentState + CurrentPacketParams.currentPacketMpduCnt, pConfigurtionParams->bfNumOfPacketTransInCurrentState);
	}
	DEBUG_ASSERT(workingPointStatsIndex < TOTAL_NUMBER_OF_RATES_IN_DB);
	pWorkingPointEstimator = &(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[workingPointStatsIndex]);
#ifdef LINK_ADAPTATION_LOGS
//		("estimatorsWorkingPointDbUpdate,rateIndex = %d, currentPacket RateIndex = %d,bandwidth = %d,currentPacket bandwidth = %d,  weightedSuccessMpdu = %d,weightedFailMpdu = %d, packetFail = %d, packetSuccess = %d, total Size of MPDU = %d",
//		pWorkingPointEstimator->rateIndex, CurrentPacketParams.rateIndex,  
//		pWorkingPointEstimator->bandwidth, CurrentPacketParams.bandwidth, 
//		pBaaReport->weightedSuccessMpdu, pBaaReport->weightedFailMpdu, pBaaReport->failedMpdu, pBaaReport->successMpdu, pBaaReport->totalSizeOfMpdu);
#endif
	/*Validate that excepted working point is the same as received in the current packet and hasn't changed by the HW (compared to estimators DB)*/
	if (pWorkingPointEstimator->rateIndex == CurrentPacketParams.rateIndex)
	{
		currentPer = calculatePer(pBaaReport);

		currentGoodput = goodPutEstimation(laDbDistributionParameter,currentPer,pWorkingPointEstimator->rateIndex,CurrentPacketParams.bandwidth);
		if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
		{
			/*Update filter and write to estimators DB*/
			AlphaFilter_updateFilter(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut), currentGoodput,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(pWorkingPointEstimator->averageDtFilterResult), FALSE); 
			lowFrameRateEstimation(laDbDistributionParameter,pBaaReport->userReport.totalFailMpduLength+pBaaReport->userReport.totalSuccessMpduLength,laDbDistributionParameter->laStationUnique->lowFrameRateAverageDtFilterResult); 
			isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);	
			if ((pBaaReport->userReport.tidBitmap & (1 << IEEE802_1D_BE_1)) && (CurrentPacketParams.bandwidth == GetDataBwLimit(CurrentPacketParams.stationIndex, laDbDistributionParameter->uspIndex, isHeGroup)))
			{
				/*Update Dynamic TxOP estimataor - only for TID0 */
#ifndef ENET_INC_ARCH_WAVE600
				DynamicTxop_UpdateEstimator(CurrentPacketParams.stationIndex, pBaaReport->commonReport.duration, lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2);
#endif
			}
		}
		else // MU
		{
			if (laDbDistributionParameter->laHeGroupUnique == NULL)
			{
				/*Update filter and write to estimators DB*/
				ASSERT(laDbDistributionParameter->uspIndex < MAX_USP_IN_VHT_GROUP);
				AlphaFilter_updateFilter(&(laDbDistributionParameter->laVhtGroupUnique->wpAverageGoodPutPerUsp[laDbDistributionParameter->uspIndex]), currentGoodput,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(pWorkingPointEstimator->averageDtFilterResult), FALSE); 
			}
			else
			{
#ifdef DYNAMIC_MU_EFFICIENCY_CALC
				GroupsMetricsCalculator_CalcAndUpdateDlPsduEfficiency(laDbDistributionParameter, pBaaReport,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(pWorkingPointEstimator->averageDtFilterResult));
#endif
				/*Update filter and write to estimators DB*/
				ASSERT(laDbDistributionParameter->uspIndex < MAX_USP_IN_HE_GROUP);
			}
			
		}
		if  (((isBwProbingActivate == FALSE) || ((isBwProbingActivate != FALSE) && (CurrentPacketParams.bandwidth != pSlowProbingEstimators->bandwidth))) && 
			(isBfStateValid == TRUE))
		{
			/*Calculate PER */
			CurrentPacketParams.currentPer = currentPer;
			laDbDistributionParameter->laStaUspCommon->longTermPerFilter = INPUT_SAMPLE_TO_IIR_FILTER(currentPer, laDbDistributionParameter->laStaUspCommon->longTermPerFilter, LT_FILTERS_COEF);
			perVariance = calculatePerVariance(currentPer, GET_IIR_FILTER_RESULT(laDbDistributionParameter->laStaUspCommon->longTermPerFilter,LT_FILTERS_COEF));
			laDbDistributionParameter->laStaUspCommon->longTermVarFilter = INPUT_SAMPLE_TO_IIR_FILTER(perVariance,laDbDistributionParameter->laStaUspCommon->longTermVarFilter, LT_FILTERS_COEF);
			perCollisionTh = GET_IIR_FILTER_RESULT(laDbDistributionParameter->laStaUspCommon->longTermPerFilter, LT_FILTERS_COEF);
			varCollisionTh = ((GET_IIR_FILTER_RESULT(laDbDistributionParameter->laStaUspCommon->longTermVarFilter, LT_FILTERS_COEF))<<1);
			ILOG0_DDD("estimatorsWorkingPointDbUpdate, perCollisionTh = %d, varCollisionTh = %d, MaxTh = %d", perCollisionTh, varCollisionTh,pConfigurtionParams->collisionDetectionTh);
			/*If CurrentPer < (MAX(PerAverage + 2*PerVar, COLLISION_DETECTION_MIN_TH)), we will not take per into per, per var alpha filters*/
			if (((pBaaReport->userReport.successMpduCnt + pBaaReport->userReport.failedMpduCnt) < COLLISION_DETECTION_MIN_MPDU_COUNT) || (CurrentPacketParams.currentPer < (MAX((perCollisionTh + varCollisionTh), pConfigurtionParams->collisionDetectionTh))))
			{
				/******************************/
				/*Calculate working point estimators*/
				/******************************/
				/*Calculate Consequtive success/failure estimator  */
				/************************************************/
				/* Update PER filter*/
				pFilterParams = &pWorkingPointEstimator->averagePer;
				isCurrentPacketFailed = checkCurrentPacketFailed(pBaaReport);
				fastDropEstimator(pWorkingPointEstimator->rateIndex, laDbDistributionParameter->laStaUspCommon, pBaaReport,isCurrentPacketFailed);
				/*Update filter and write to estimators DB*/
				AlphaFilter_updateFilter(pFilterParams, currentPer,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(pWorkingPointEstimator->averageDtFilterResult), TRUE); 
				/*PER estimator was update => change indication in CurrentPacketParams*/
				CurrentPacketParams.estimatorsIndication.per = TRUE;
#ifdef LINK_ADAPTATION_LOGS
				{
		//			uint8 filterResult = AlphaFilter_GetFilterResult(pFilterParams);//
		//			perResults_t perDebug;
		//			perDebug.avPer = filterResult;
		//			perDebug.currentPer = currentPer;
		//			perDebug.TsfVal = pFilterParams->lastPacketTsf;
		//			SLOG0(0, 0, perResults_t, &perDebug);
		//			ILOG0_DDDD("estimatorsWorkingPointDbUpdate, AV per,RateIndex = %d,bandwidth = %d,filterinput = %d, filterResult = %d", CurrentPacketParams.rateIndex, CurrentPacketParams.bandwidth, currentPer, filterResult);
				}
#endif
				/*Calculate PER variance*/
				/***********************/
				/*Update filter and write to estimators DB*/
				averagePer = AlphaFilter_GetFilterResult(&pWorkingPointEstimator->averagePer);
				perVariance = calculatePerVariance(currentPer, averagePer);
				pFilterParams = &pWorkingPointEstimator->perVariance;
				AlphaFilter_updateFilter(pFilterParams, perVariance,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(pWorkingPointEstimator->averageDtFilterResult), FALSE); //use average dT from PER working point, because it update at the same time
			}
	//#ifdef LINK_ADAPTATION_LOGS
	//				ILOG0_DDDD("estimatorsWorkingPointDbUpdate, PER Variance,input = %d, alpha = %d,filter result = %d, last pcackt TSF = %d",perVariance, pFilterParams->alpha, pFilterParams->filterResult>>11, pFilterParams->lastPacketTsf);   
	//#endif
			/*Calculate Reference length - this parameter exist only for the working point*/
			/*************************************************************/
#ifdef LA_CALC_REF_LEN
			
			if (pBaaReport->userReport.successMpduCnt > 0)
			{
				pFilterParams = &(laDbDistributionParameter->laStaUspCommon->workingPointEstimators.refLength);
				refLength = calculateRefLen(pBaaReport);
			
				/*Update filter and write to estimators DB*/
				AlphaFilter_updateFilter(pFilterParams, refLength,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(pWorkingPointEstimator->averageDtFilterResult), FALSE); //use average dT from PER working point, because it update at the same time
			
				/*Update RefLen in DB*/
				/******************/
				refLength = AlphaFilter_GetFilterResult(pFilterParams);
				calculatedRefLenFactor = calculateRefLenFactor(refLength, pConfigurtionParams->refLenFactor);
				refLenSuccess = refLength - calculatedRefLenFactor;
				refLenFail = refLength + calculatedRefLenFactor;
	//#ifdef LINK_ADAPTATION_LOGS
	//		//("estimatorsWorkingPointDbUpdate, refLenSuccess = %d, refLenFail = %d, AvRefLen = %d", refLenSuccess, refLenFail,refLength);
	//#endif

				if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
				{
					setRefLenInStaDb(laDbDistributionParameter->stationOrGroupIndex, refLenSuccess, refLenFail);
				}
			}	
#endif
		}
	}
}

/**********************************************************************************

calculateRefLenFactor 


Description:
------------

Input: 
-----


Returns:
--------
	
**********************************************************************************/
#ifdef LA_CALC_REF_LEN

static uint8 calculateRefLenFactor (uint8 refLength, uint8 refLenFactor)
{
	int8 calcRefLenFactor;

	calcRefLenFactor = refLength - refLenFactor;
	if (calcRefLenFactor <= MINIMUM_REF_LEN_VAL)
	{
		calcRefLenFactor = refLength - MINIMUM_REF_LEN_VAL;
	}
	else
	{
		calcRefLenFactor = refLenFactor;
	}
//	DEBUG_ASSERT(calcRefLenFactor >= 0);

	return calcRefLenFactor;
}
#endif
/**********************************************************************************

estimatorsFastProbingDbUpdate 


Description:
------------
Calculate and update fast probing estimators in estimators DB
Input: 
-----

Returns:
--------
	
**********************************************************************************/

void estimatorsFastProbingDbUpdate(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, BaaRateAdaptiveFullFifo_t*pBaaReport)
{
	uint8 currentPer;
	uint8 perVariance;
	uint8 averagePer;
	LinkAdaptationConfigurationParams_t* pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	/*Extract estimator DB for specific BW*/
	FastProbingPointEstimators_t* fastProbingEstimators = laDbDistributionParameter->laStaUspCommon->fastProbingEstimators;
	/*Extract fast probing last point from RA  DB in order to decide which rate to update in estimators DB */
	RaFastProbingPoint_e lastFastProbingPoint = (RaFastProbingPoint_e)(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase.currentFastProbingPoint);
	/*Extract estimators fast probing statistics index in estimators DB according to fastProbingPoint - the options are UP_NSS, UP_MCS, LOWER_RATE*/
	uint8 estimatorsFastProbingIndex = laDbDistributionParameter->laStaUspCommon->fastProbingPointEstimatorsIndexes[lastFastProbingPoint];
	FastProbingPointEstimators_t* pFastProbingPointEstimator;
	/*Alpha filter parameters*/
	uint32 lnBeta = pLaConfigParams->lnBeta;
	//Bandwidth_e dataBwLimit = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex);


#ifdef LINK_ADAPTATION_LOGS
//	("estimatorsFastProbingDbUpdate,rateIndex = %d, currentPacket RateIndex = %d,bandwidth = %d,currentPacket bandwidth = %d,  weightedSuccessMpdu = %d,weightedFailMpdu = %d, packetFail = %d, packetSuccess = %d, lastFastProbingPoint = %d, DBindex = %d ",
//			pFastProbingPointEstimator->rateIndex, CurrentPacketParams.rateIndex,  
//			pFastProbingPointEstimator->bandwidth, CurrentPacketParams.bandwidth, 
//			pBaaReport->weightedSuccessMpdu, pBaaReport->weightedFailMpdu, pBaaReport->failedMpdu, pBaaReport->successMpdu,
//			lastFastProbingPoint, estimatorsFastProbingIndex);
#endif

	if ((estimatorsFastProbingIndex < TOTAL_NUMBER_OF_RATES_IN_DB))
	{
		pFastProbingPointEstimator = &fastProbingEstimators[estimatorsFastProbingIndex];

		if (pFastProbingPointEstimator->rateIndex == CurrentPacketParams.rateIndex) 
		{
			/*Calculate PER*/
			/*************/
			currentPer = calculatePer(pBaaReport);
			CurrentPacketParams.currentPer = currentPer;

			/*Update filter and write to estimators DB*/
			AlphaFilter_updateFilter(&pFastProbingPointEstimator->averagePer, currentPer,lnBeta,pLaConfigParams->constAlphaCoefForDtAvPower2,&(pFastProbingPointEstimator->averageDtFilterResult), TRUE);
#ifdef LINK_ADAPTATION_LOGS
			{
//				uint8 filterResult = AlphaFilter_GetFilterResult(&(pFastProbingPointEstimator->averagePer));
//				perResults_t perDebug;
//				perDebug.avPer = filterResult;
//				perDebug.currentPer = currentPer;
//				perDebug.TsfVal = pFastProbingPointEstimator->averagePer.lastPacketTsf;
//				SLOG0(0, 0, perResults_t, &perDebug);
//			ILOG0_DDDD("estimatorsWorkingPointDbUpdate, AV per,RateIndex = %d,bandwidth = %d,filterinput = %d, filterResult = %d", CurrentPacketParams.rateIndex, CurrentPacketParams.bandwidth, currentPer, filterResult);
			}
#endif

			/*Calculate PER variance*/
			/***********************/
			averagePer = AlphaFilter_GetFilterResult(&pFastProbingPointEstimator->averagePer);
			perVariance = calculatePerVariance(CurrentPacketParams.currentPer, averagePer);

			/*Update filter and write to estimators DB*/
			AlphaFilter_updateFilter(&pFastProbingPointEstimator->perVariance, perVariance,lnBeta,pLaConfigParams->constAlphaCoefForDtAvPower2,&(pFastProbingPointEstimator->averageDtFilterResult), FALSE);
//#ifdef LINK_ADAPTATION_LOGS
//				{
//					uint8 filterResult = AlphaFilter_GetFilterResult(&pFastProbingPointEstimator->perVariance);
//					ILOG0_DDDD("estimatorsFastProbingDbUpdate per variance,RateIndex = %d,bandwidth = %d,filterinput = %d, filterResult = %d", pFastProbingPointEstimator->rateIndex, pFastProbingPointEstimator->bandwidth, currentPer, filterResult<<11);
//				}
//#endif

			/*PER estimator was update => change indication in CurrentPacketParams*/
			CurrentPacketParams.estimatorsIndication.per = TRUE;
		}
			
	}
}
/**********************************************************************************

estimatorsSlowProbingDbUpdate 


Description:
------------
Calculate and update slow probing estimators in estimators DB

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void estimatorsSlowProbingDbUpdate(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, BaaRateAdaptiveFullFifo_t*pBaaReport)
{
	uint8 currentPer;
	LinkAdaptationConfigurationParams_t* pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	SlowProbingPointEstimators_t* pSlowProbingEstimators;
	/*Extract beat fro link adaptation configuration params*/
	uint32 lnBeta = pLaConfigParams->lnBeta;
	Bandwidth_e dataBwLimit;
	bool isBwProbingActivate;
	bool isBfStateValid;
	uint8 bfType;
	bool isHeGroup;

	isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);	
	dataBwLimit = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, isHeGroup);
	bfType = pBaaReport->userReport.phyTxBf;
	
	/*Slow probing estimators DB ptr*/
	pSlowProbingEstimators = &(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators);

	isBwProbingActivate = (laDbDistributionParameter->laStaGroupCommon->linkAdaptationState == LA_WAIT_FOR_SLOW_PROBE_VALID) && 
							(getCurrentSlowProbingTask(laDbDistributionParameter) == SLOW_PROBING_BW);

	/* Verify that BF state is valid (only relevent if BF loop is active) */ 
	isBfStateValid = validateBfTransmission(laDbDistributionParameter, bfType);

#ifdef LINK_ADAPTATION_LOGS
//   //("estimatorsSlowProbingDbUpdate, Calculate PER");
#endif

	
	if  (((CurrentPacketParams.bandwidth == dataBwLimit) || ((isBwProbingActivate != FALSE) && (CurrentPacketParams.bandwidth == pSlowProbingEstimators->bandwidth))) &&
		(isBfStateValid == TRUE))
	{
		/*Set BW according to current packet BW*/
		pSlowProbingEstimators->bandwidth = CurrentPacketParams.bandwidth;

		/*Calculate PER*/
		/*************/
		currentPer = calculatePer(pBaaReport);
		CurrentPacketParams.currentPer = currentPer;
		
		/*Update filter and write to estimators DB*/
		AlphaFilter_updateFilter(&pSlowProbingEstimators->averagePer, currentPer,lnBeta,pLaConfigParams->constAlphaCoefForDtAvPower2,&(pSlowProbingEstimators->averageDtFilterResult), TRUE);
#ifdef LINK_ADAPTATION_LOGS
		{
			uint8 filterResult = AlphaFilter_GetFilterResult(&pSlowProbingEstimators->averagePer);
			ILOG0_DDDD("estimatorsSlowProbingDbUpdate av per,bandwidth = %d,filterinput = %d, filterResult = %d, alpha = %d", pSlowProbingEstimators->bandwidth, currentPer, filterResult, pSlowProbingEstimators->averagePer.alpha);
		}
#endif

		/*PER estimator was update => change indication in CurrentPacketParams*/
		CurrentPacketParams.estimatorsIndication.per = TRUE;
	}
}

/**********************************************************************************

calculatePer 


Description:
------------
Function calculates PER using the following fields from BAA report:
weightedSuccessMpdu
weightedFailMpdu
PER = 100*weightedSuccessMpdu/(weightedSuccessMpdu + weightedFailMpdu)
Input: 
-----
pBaaReport - pointer to current BAA report 

Returns:
--------
	
**********************************************************************************/
 uint8 calculatePer( BaaRateAdaptiveFullFifo_t* pBaaReport)
{
	uint8 resPer;
#ifdef EXTRACT_TEST_BUS
#ifdef ENET_INC_ARCH_WAVE600D2
	if ((pBaaReport->userReport.failedMpduCnt != 0) && (pBaaReport->commonReport.txPhyMode == PHY_MODE_11AX_MU_DL))
#else
	if ((pBaaReport->userReport.failedMpduCnt != 0) && (pBaaReport->commonReport.phyMode == PHY_MODE_11AX_MU_DL))
#endif //ENET_INC_ARCH_WAVE600D2
	{
		ASSERT(0);
	}
#endif
//	if ((pBaaReport->weightedSuccessMpdu) != 0 || (pBaaReport->weightedFailMpdu != 0))
//	{
//		/*Weighted PER*/
//		resPer = (HUNDRED_PERCENT*pBaaReport->weightedFailMpdu)/(pBaaReport->weightedFailMpdu + pBaaReport->weightedSuccessMpdu);
//	}
//	else
//	{
		/*MPDU PER*/
		resPer = (HUNDRED_PERCENT*pBaaReport->userReport.failedMpduCnt)/(pBaaReport->userReport.failedMpduCnt + pBaaReport->userReport.successMpduCnt);
//	}


#ifdef LINK_ADAPTATION_LOGS
//	ILOG0_DDDDD("calculatePer, res MPDU = %d, res length = %d, res weighted = %d, weightedFail = %d, weightedSuccessMpdu = %d", resMPduNum, resLength, resWeighted, pBaaReport->weightedFailMpdu, pBaaReport->weightedSuccessMpdu);	
//	SLOG0(0, 0, BaaRateAdaptiveFifo_t, pBaaReport);
//	ILOG0_DDDD("calculatePer, weightedSuccessMpdu = %d, successMpdu = %d, weightedFailMpdu = %d, failedMpdu = %d ", weightedSuccessMpdu, pBaaReport->successMpdu, weightedFailMpdu,  pBaaReport->failedMpdu);
//	("calculatePer, totalMpdu = %d, failedMpdu = %d, successSize = %d, totalSize = %d, weightSuccess = %d, weightFail = %d",
//			pBaaReport->mpduCnt, pBaaReport->failedMpdu ,pBaaReport->totalSizeSuccessMpdu, pBaaReport->totalSizeAllMpdu, pBaaReport->weightedSuccessMpdu,pBaaReport->weightedFailMpdu); 
//	ILOG0_DDD("calculatePer, resWeighted = %d, resMPduNum = %d, resLength = %d", resWeighted, resMPduNum, resLength);

#endif
	
	DEBUG_ASSERT(resPer<=HUNDRED_PERCENT);

	return resPer;
	
}

/**********************************************************************************

calculateRefLen 


Description:
------------
Function calculates Reference aggregation length using to the following fileds from BAA report:
totalSizeOfSuccessMpdu
SuccessMpdu
RefLen = (totalSizeOfSuccessMpdu/SuccessMpdu) mod (64*N)
N is configurable factor parameter taken from LinkAdaptationConfigurationParams

Input: 
-----

Returns:
--------
	
**********************************************************************************/
#ifdef LA_CALC_REF_LEN
static uint8 calculateRefLen(BaaRateAdaptiveFullFifo_t* pBaaReport)
{
	uint16 totalSizeOfSuccessMpdu = pBaaReport->userReport.totalSuccessMpduLength;
	uint16 SuccessMpdu = pBaaReport->userReport.successMpduCnt;

//	ASSERT(SuccessMpdu > 0);

	return (totalSizeOfSuccessMpdu/SuccessMpdu);
}
#endif
/**********************************************************************************

calculatePerVariance 


Description:
------------
Function calculates PER variance using the current PER and the working point average PER (just updated)
Input: 
-----
pWorkingPointFilterParams - pointer t working point result entry in estimators DB

Returns:
--------
	
**********************************************************************************/
static uint8 calculatePerVariance(uint8 currentPer, uint8 averagePer)
{
	uint8 perVariance = ABS((int32)(currentPer - averagePer));

	DEBUG_ASSERT(perVariance <= HUNDRED_PERCENT)
	return perVariance;
}

/**********************************************************************************

calculateGoodPut 


Description:
------------

Input: 
-----
BAA report
Returns:
--------
	
**********************************************************************************/
static uint32 calculateGoodPut(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 currentPer,uint8 rateIndex,Bandwidth_e bandwidth,uint16 factor)
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	uint32 resGP;
	uint16 phyRate;

	ASSERT(factor > 0);
	phyRate = ratesTable[rateIndex].rateTableBwParmas[bandwidth].shortCpPhyRate;
	/*Multuply PER (in scale of 0:100)	by the phy rate*/
	/* Normalize by the MAX PHY rate which is 1733*/
	resGP = ((HUNDRED_PERCENT-currentPer)*phyRate)/factor;
	return resGP;
}


/**********************************************************************************

estimatorsResetWorkingPoint 


Description:
------------

Input: 
-----
staIndex

Returns:
--------
	bool wpChanged-working point changed or not 
	
**********************************************************************************/
void estimatorsResetWorkingPoint(LaStationUspCommonParams_t* laStaUspCommon,LaStationGroupCommonParams_t* laStaGroupCommon)
{
#ifdef LA_CALC_REF_LEN
	uint8 workingPointStatsIndex = laStaUspCommon->fastProbingEstimators[WORKING_POINT];
	FastProbingPointEstimators_t* pWorkingPointEstimator;

	DEBUG_ASSERT(workingPointStatsIndex < TOTAL_NUMBER_OF_RATES_IN_DB);
	pWorkingPointEstimator = &(laStaUspCommon->fastProbingEstimators[workingPointStatsIndex]);

	AlphaFilter_Init(&(laStaUspCommon->workingPointEstimators.refLength),pWorkingPointEstimator->averageDtFilterResult);
#endif

	laStaUspCommon->probingPointValidationCounter = 0;

	AlphaFilter_Init(&(laStaGroupCommon->slowProbingEstimators.averagePer),&(laStaGroupCommon->slowProbingEstimators.averageDtFilterResult));
	laStaUspCommon->windowFailedPacketBitmap = 0;
	laStaUspCommon->numberOfFailedPacketInWindow = 0; 
	laStaGroupCommon->slowProbingEstimators.slowProbingRatePlusOne = INVALID_RATE_INDEX_VAL;
	

}

/**********************************************************************************

estimatorsIsCurrentPerValid 


Description:
------------
Function check if per is valid according to the follwing criteria:
if ((Tsfcurrent < Tsflast) < diff && counter > TH )
	Per is valid;
else
	counter+= current packet MPDU cnt
	Per is not valid;
Input: 
-----
stationIndex
validationCounter pointer - stability validation counter or probing validation counter

Returns:
--------
	void - 
	
**********************************************************************************/

bool estimatorsProbingValidationCriteria(uint8* validationCounter, uint8 probingValidationTh)
{
	bool retVal;

	(*validationCounter)++;
	
	if (*validationCounter >= probingValidationTh)
	{
		*validationCounter = 0;
		retVal = TRUE;
	}
	else
	{
		retVal = FALSE;
	}
	
#ifdef LINK_ADAPTATION_LOGS
{
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	uint16 currentTsfMod = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);
	ILOG0_DDDD("estimatorsProbingValidationCriteria,currentPacketTsf = %d, probeIsValid = %d, currentTsfMod = %d, tsfDiff = %d", 
				CurrentPacketParams.currentPacketTsf, retVal, currentTsfMod, currentTsf);
	ILOG0_D("estimatorsProbingValidationCriteria,validationCounter = %d", *validationCounter);
}
#endif

	return retVal;
}
/**********************************************************************************

estimatorsAgingCheck 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void estimatorsAgingCheck(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8 	rateIndexInFastProbingDb;
	LinkAdaptationConfigurationParams_t* pLinkAdaptationParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
 	PhyMode_e highestPhyMode; 
	uint8	maxNss = SPATIAL_STREAM_1;  
#ifndef ENET_INC_ARCH_WAVE600
	K_MSG * newMsgPtr	= NULL; 
	GroupManagerGroupHasFailedMessage_t*	groupManagerMessageParameters = NULL;
#endif //ENET_INC_ARCH_WAVE600

	/*Run over all BW and reset filter if aged*/
	/*Slow probing point filters*/
	if (AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators.averagePer),pLinkAdaptationParams->maxValidFilterTsfDiff) == FALSE)
	{
		AlphaFilter_Init(&(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators.averagePer),&(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators.averageDtFilterResult));
	}
	
	/*Fast probing point filters*/
	for(rateIndexInFastProbingDb = 0; rateIndexInFastProbingDb < TOTAL_NUMBER_OF_RATES_IN_DB; rateIndexInFastProbingDb++)
	{
		if (AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[rateIndexInFastProbingDb].averagePer),pLinkAdaptationParams->maxValidFilterTsfDiff) == FALSE)
		{
			AlphaFilter_Init(&(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[rateIndexInFastProbingDb].averagePer),&(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[rateIndexInFastProbingDb].averageDtFilterResult));
			AlphaFilter_Init(&(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[rateIndexInFastProbingDb].perVariance),&(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[rateIndexInFastProbingDb].averageDtFilterResult));
	
		}
	}

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/*These two Alpha filters are updated together but they may not age together as the TSF is read for each filter
		Therefore, check both, and if one is invalid init both*/
		if ((AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laStationUnique->lowFrameRateCounter),pLinkAdaptationParams->maxValidFilterTsfDiff) == FALSE) ||
			 (AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut),pLinkAdaptationParams->maxValidFilterTsfDiff) == FALSE))
		{
			AlphaFilter_Init(&(laDbDistributionParameter->laStationUnique->lowFrameRateCounter),&(laDbDistributionParameter->laStationUnique->lowFrameRateAverageDtFilterResult));
			AlphaFilter_Init(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut),&(laDbDistributionParameter->laStaGroupCommon->averageGoodPutDtFilterResult));
			/*Clear the PS Management bitmap and Counter*/
			laDbDistributionParameter->laStationUnique->stationInPsOrManagementFramesWindow = 0;
			laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount = 0;
			ILOG0_DD("estimatorsAgingCheck, staId = %d, MU group count - %d",laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->laStationUnique->muGroupCount);
			highestPhyMode = ExtractHighestPhyModeFromRaMask(laDbDistributionParameter->laStaUspCommon->raIndexMaskOrg, 
															laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
			if (highestPhyMode >= PHY_MODE_11N)
			{
				/*Function not support 11A/G/B phy mode*/
				maxNss = LinkAdaptationGetMaxNssInMask(&(laDbDistributionParameter->laStaUspCommon->raIndexMaskOrg.raIndexMask64bit[0]),
														laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
			}

			/*If STA belongs to at least one group set the candidate bit, else clear it*/
			if ((laDbDistributionParameter->laStationUnique->muGroupCount > 0) && (maxNss != SPATIAL_STREAM_2))
            {
				Utils_SetBitInBitmap((uint32 *)(muCandidateStaBitmap), laDbDistributionParameter->stationOrGroupIndex); 	
			}
			else
			{			
				Utils_ZeroBitInBitmap((uint32 *)(muCandidateStaBitmap), laDbDistributionParameter->stationOrGroupIndex);		
			}	
		}
	}
	else /*MU*/
	{
		if (AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut),pLinkAdaptationParams->maxValidFilterTsfDiff) == FALSE)
		{
			AlphaFilter_Init(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut),&(laDbDistributionParameter->laStaGroupCommon->averageGoodPutDtFilterResult));
#ifndef ENET_INC_ARCH_WAVE600
			if((laDbDistributionParameter->laGroupUnique->groupHasFailedNotificationSent == FALSE)&&(laDbDistributionParameter->laGroupUnique->groupState == LA_ACTIVE_GROUP_STATE))
			{
				// group's mark is below the minimal threshold for group marks - Notify the group manager about it
				if (laDbDistributionParameter->laHeGroupUnique == NULL) // Don't send in case of HE MU
				{
					laDbDistributionParameter->laGroupUnique->groupHasFailedNotificationSent = TRUE; //Prevent the message to be send multiple times 
					newMsgPtr = OSAL_GET_MESSAGE(sizeof(GroupManagerGroupHasFailedMessage_t));
					groupManagerMessageParameters = ((GroupManagerGroupHasFailedMessage_t *)pK_MSG_DATA(newMsgPtr));
					groupManagerMessageParameters->groupIndex = laDbDistributionParameter->stationOrGroupIndex; 
					OSAL_SEND_MESSAGE(GROUP_MANAGER_GROUP_FAILED, TASK_GROUP_MANAGER, newMsgPtr, GET_DEFAULT_VAP_FOR_MY_BAND());
				}				
			}
#endif //ENET_INC_ARCH_WAVE600
		}
	}

	/*Reset station collision counters if vap counters was reset without reset the stations counters*/
	protectionAdaptationAgingReset(laDbDistributionParameter);

#ifdef LA_CALC_REF_LEN	
	/*Working point filters */
	if (AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[WORKING_POINT].averagePer),pLinkAdaptationParams->maxValidFilterTsfDiff) == FALSE)
	{
		AlphaFilter_Init(&(laDbDistributionParameter->laStaGroupCommon->workingPointEstimators.refLength));
	
	}
#endif


}

/**********************************************************************************

estimatorsSetSlowProbingPointRateIndex 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void estimatorsSetSlowProbingPointRateIndex(StaId stationIndex,uint8 uspIndex, uint8 val)
{
	if(uspIndex == INVALID_MU_USP_INDEX)
	{
		LinkAdaptationStaDatabase[stationIndex].laStaGroupCommon.slowProbingEstimators.slowProbingRatePlusOne = val;
	}
	else /* MU */ //TBD HE?
	{
		LinkAdaptationVhtGroupDatabase[stationIndex].laStaGroupCommon.slowProbingEstimators.slowProbingRatePlusOne = val;
	}
}
/**********************************************************************************

estimatorsGetWpPerEstimatorsPtr 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

FastProbingPointEstimators_t* estimatorsGetWpPerEstimatorsPtr(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8 fastProbingEstimatorIndex = laDbDistributionParameter->laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	FastProbingPointEstimators_t* pWorkingPointPerEst = &(laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[fastProbingEstimatorIndex]);

	return pWorkingPointPerEst;
}
/**********************************************************************************

GetSlowProbingPerEstPtr 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

SlowProbingPointEstimators_t* estimatorsGetSlowProbingPerEstPtr(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	SlowProbingPointEstimators_t* pSlowProbingEst = &(laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators);

	return pSlowProbingEst;
}
/**********************************************************************************

GetSlowProbingPerEstPtr 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 estimatorsGetSlowProbingRatePlusOne(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8 retVal;

	retVal = laDbDistributionParameter->laStaGroupCommon->slowProbingEstimators.slowProbingRatePlusOne;
	DEBUG_ASSERT(retVal!=INVALID_RATE_INDEX_VAL)

	return retVal;
}

/**********************************************************************************

updatePsWithinAwindow 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void updatePsWithinAwindow(TxSequencerReport_t* pSequencerReport, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint64* staPsWindowPtr = &(laDbDistributionParameter->laStationUnique->stationInPsOrManagementFramesWindow);
	uint8 stationOldMsb = ((*staPsWindowPtr >> 63) & 0x1); //check the oldest frames status in the window - the MSB

	*staPsWindowPtr = (*staPsWindowPtr)<<1; //slide the window 1 bit left	
	if(pSequencerReport->staInPs)
	{
		if(stationOldMsb == 0x0)
		{
		// No Ps - New LSB is 0  
			laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount++;			
		}
		*staPsWindowPtr |= 0x1; // set LSB with 1
	}
	else /* No Ps*/
	{
		if(stationOldMsb) //check the oldest frames status in the window - the MSB
		{
			/* Update counters*/
			laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount--;
		}
	}
	ASSERT(laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount <= MU_CANDIDATE_PS_MGMT_WINDOW_SIZE)
}
/**********************************************************************************

updateMgmtWithinAwindow 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void updateMgmtWithinAwindow(BaaRateAdaptiveFullFifo_t* pBaaReport, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint64* staMngtWindowPtr;
	uint8 stationOldMsb;
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX) //use SU reports only 
	{
		staMngtWindowPtr = &(laDbDistributionParameter->laStationUnique->stationInPsOrManagementFramesWindow);
		stationOldMsb = ((*staMngtWindowPtr >> 63) & 0x1); //check the oldest frames status in the window - the MSB

		*staMngtWindowPtr = (*staMngtWindowPtr)<<1; //slide the window 1 bit left	
		if((pBaaReport->commonReport.pdType == PD_TYPE_DATA)||(pBaaReport->commonReport.pdType == PD_TYPE_NDP))
		{
			if(stationOldMsb) //check the oldest frames status in the window - the MSB
			{
				/* Update counters*/
				laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount--;
			}
		}
		else /* No Mngt*/
		{
			if(stationOldMsb == 0x0)
			{
			// No Mngt - New LSB is 0  
				laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount++;			
			}
			*staMngtWindowPtr |= 0x1; // set LSB with 1
		}
		DEBUG_ASSERT(laDbDistributionParameter->laStationUnique->stationInPsOrManagementWithinWindowCount <= MU_CANDIDATE_PS_MGMT_WINDOW_SIZE);
	}
}// KW IGNORE FUNCRET.GEN

/**********************************************************************************

estimatorsProcessSequencerReport 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void estimatorsProcessSequencerReport(TxSequencerReport_t* pSequencerReport, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8	loopIterator; 
	uint8 	muCandidacyGrade = 0;
	uint32	avarageGroupGoodput = 0;  
	uint32	lnBeta; 
	uint8	normalizedAvarageGroupGoodput = 0;
	K_MSG *	newMsgPtr	= NULL; 
	LinkAdaptationConfigurationParams_t*	pConfigurtionParams;
	GroupManagerGroupHasFailedMessage_t*	groupManagerMessageParameters = NULL;
	uint8	maxNss = SPATIAL_STREAM_1; 
	PhyMode_e highestPhyMode;	
#ifdef ENET_INC_ARCH_WAVE600
	BandId_e band = ConfigurationManager_GetMyBand();
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
#endif
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		if ((pSequencerReport->txqQueueId == HW_TX_Q_TYPE_STA_TID) && (pSequencerReport->executedData))
		{

			highestPhyMode = ExtractHighestPhyModeFromRaMask(laDbDistributionParameter->laStaUspCommon->raIndexMaskOrg, 
															laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
			if (highestPhyMode >= PHY_MODE_11N)
			{
				/*Function not support 11A/G/B phy mode*/
				maxNss = LinkAdaptationGetMaxNssInMask(&(laDbDistributionParameter->laStaUspCommon->raIndexMaskOrg.raIndexMask64bit[0]),
														laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
			}

			updatePsWithinAwindow(pSequencerReport,laDbDistributionParameter);
			muCandidacyGrade = estimatorsCalcMuCandidacyGrade(&LinkAdaptationStaDatabase[laDbDistributionParameter->stationOrGroupIndex]);

			ILOG0_DDD("estimatorsProcessSequencerReport, staId = %d, Station's grade is - %d, MU group count - %d",laDbDistributionParameter->stationOrGroupIndex,muCandidacyGrade,laDbDistributionParameter->laStationUnique->muGroupCount);

            if (((muCandidacyGrade > MU_CANDIDATE_THRESHOLD_FACTOR_128_SCALE) || (laDbDistributionParameter->laStationUnique->muGroupCount > 0)) && (maxNss != SPATIAL_STREAM_2))
            {
				Utils_SetBitInBitmap((uint32 *)(muCandidateStaBitmap), laDbDistributionParameter->stationOrGroupIndex);		
			}
			else
			{			
				Utils_ZeroBitInBitmap((uint32 *)(muCandidateStaBitmap), laDbDistributionParameter->stationOrGroupIndex);		
			}
		}
	}
	else /* MU group*/ 
	{
		pConfigurtionParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
		lnBeta = pConfigurtionParams->lnBeta;
		/*Update filter of the associated  USP with a valid goodput */
		
#ifdef ENET_INC_ARCH_WAVE600
		if (isHeGroup)
		{
			for(loopIterator = 0; loopIterator < MAX_USP_IN_HE_GROUP; loopIterator++)
			{
				if(laDbDistributionParameter->laHeGroupUnique->groupStationIndexes[loopIterator] != INVALID_STA_INDEX)
				{
					if(AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laHeGroupUnique->wpAverageGoodPutPerUsp[loopIterator]), laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
					{
						avarageGroupGoodput += laDbDistributionParameter->laHeGroupUnique->wpAverageGoodPutPerUsp[loopIterator].filterResult; 
					}
				}
			}
		}
		else
#endif
		{
			for(loopIterator = 0; loopIterator < MAX_USP_IN_VHT_GROUP; loopIterator++)
			{
				if(laDbDistributionParameter->laVhtGroupUnique->groupStationIndexes[loopIterator] != INVALID_STA_INDEX)
				{
					if(AlphaFilter_isFilterResultValid(&(laDbDistributionParameter->laVhtGroupUnique->wpAverageGoodPutPerUsp[loopIterator]), laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff))
					{
						avarageGroupGoodput += laDbDistributionParameter->laVhtGroupUnique->wpAverageGoodPutPerUsp[loopIterator].filterResult; 
					}
				}
			}
		}
		normalizedAvarageGroupGoodput = (uint8)NORMELIZE_ALPHA_FILTER_RESULT(avarageGroupGoodput);
		AlphaFilter_updateFilter(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut), normalizedAvarageGroupGoodput,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(laDbDistributionParameter->laStaGroupCommon->averageGoodPutDtFilterResult), TRUE); 
		//ILOG0_DDD("Sequencer estimaror processing - goodput update, current totalgroup's avgGoodput = %d, normAvgGoodput %d, groupId %d",avarageGroupGoodput, normalizedAvarageGroupGoodput, laDbDistributionParameter->stationOrGroupIndex);
		if((laDbDistributionParameter->laGroupUnique->groupHasFailedNotificationSent == FALSE)&&(laDbDistributionParameter->laGroupUnique->groupState == LA_ACTIVE_GROUP_STATE))
		{
			/* If the group's mark is below the minimal threshold for group marks - Notify the group manager about it*/
			avarageGroupGoodput = AlphaFilter_GetFilterResult(&(laDbDistributionParameter->laStaGroupCommon->wpAverageGoodPut));
			if(avarageGroupGoodput < GROUP_MANAGER_MINIMAL_GROUP_GRADE_THRESHOLD)
			{
				if (laDbDistributionParameter->laHeGroupUnique == NULL) // Don't send in case of HE MU
				{
					laDbDistributionParameter->laGroupUnique->groupHasFailedNotificationSent = TRUE; //Prevent the message to be send multiple times 
					newMsgPtr = OSAL_GET_MESSAGE(sizeof(GroupManagerGroupHasFailedMessage_t));
					groupManagerMessageParameters = ((GroupManagerGroupHasFailedMessage_t *)pK_MSG_DATA(newMsgPtr));
					groupManagerMessageParameters->groupIndex = laDbDistributionParameter->stationOrGroupIndex; 
					OSAL_SEND_MESSAGE(GROUP_MANAGER_GROUP_FAILED, TASK_GROUP_MANAGER, newMsgPtr, GET_DEFAULT_VAP_FOR_MY_BAND());
				}
			}
		}
	}
#if defined(ENET_INC_ARCH_WAVE600)
	if(TRUE==HW_SEMAPHORE_TRY_LOCK(HW_SEMAPHORE_AOCS_INFO))
	{
		if(!(pSequencerReport->cca20PFree))
		{
			DynamicBwStatistics.dynamicBW20MHz[band]++;
		}
		if(!(pSequencerReport->cca20SFree))
		{
			DynamicBwStatistics.dynamicBW40MHz[band]++;
		}
		if(!(pSequencerReport->cca40SFree))
		{
			DynamicBwStatistics.dynamicBW80MHz[band]++;
		}	
		if(!(pSequencerReport->cca80SFree))
		{
			DynamicBwStatistics.dynamicBW160MHz[band]++;
		}
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_AOCS_INFO);
	}
#else
	if(TRUE==HW_SEMAPHORE_TRY_LOCK(HW_SEMAPHORE_2_18_AOCS_INFO))
	{
		if(!(pSequencerReport->cca20PFree))
		{
			DynamicBwStatistics.dynamicBW20MHz++;
		}
		if(!(pSequencerReport->cca20SFree))
		{
			DynamicBwStatistics.dynamicBW40MHz++;
		}
		if(!(pSequencerReport->cca40SFree))
		{
			DynamicBwStatistics.dynamicBW80MHz++;
		}	
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_2_18_AOCS_INFO);
	}
#endif
    if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
    {
		    // When Aggregation builder fails to build, since combination of transmittion time of 1 ms and zero aggregation in certions rates are invalid. 
	    if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.clientIsolationRecovery == TRUE)
	    {
	        // Set transmission time back to it value according to air time fairness new rate.
	        uint8 wpRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
	        Bandwidth_e bw = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);

	        //ILOG0_V("estimatorsProcessSequencerReport recovery");
	        
			clientIsolationRateIsChanged(laDbDistributionParameter, WORKING_POINT, wpRateIndex, wpRateIndex,bw);
			// reset bit
			LinkAdaptationStaDatabase[laDbDistributionParameter->stationOrGroupIndex].laStaUspCommon.staTransmissionParams.clientIsolationRecovery = FALSE;	    
	    }
		if ((pSequencerReport->altRateReasonFastProbing == TRUE) && (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.clientIsolationProbing == TRUE))
		{
	        // Set transmission time back to it value according to air time fairness new rate.
			uint8 wpRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon); 
			Bandwidth_e bw =  GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);
			clientIsolationRateIsChanged(laDbDistributionParameter, WORKING_POINT, wpRateIndex, wpRateIndex,bw);
			LinkAdaptationStaDatabase[laDbDistributionParameter->stationOrGroupIndex].laStaUspCommon.staTransmissionParams.clientIsolationProbing = FALSE;
		}
    }
	
	if(pSequencerReport->cca20PFree) //only if no CCA on primary - insert input to  secondary channels filter 
	{
		if((pSequencerReport->cca20SFree) == 0)
		{
			//update both BW alpha filters with 20, as the 20 secondary occupied 
			AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW40), BW_20_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW40DeltaTAverage, TRUE); 
			AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW80), BW_20_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW80DeltaTAverage, TRUE); 			
#ifdef ENET_INC_ARCH_WAVE600
			AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW160), BW_20_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW160DeltaTAverage, TRUE); 
#endif
		}
		else
		{
			if((pSequencerReport->cca40SFree) ==0)
			{
				//update 40Mhz filter with 40 and 80 Mhz with 40 also as only the 40 secondary is occupied 
				AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW40), BW_40_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW40DeltaTAverage, TRUE); 
				AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW80), BW_40_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW80DeltaTAverage, TRUE); 
#ifdef ENET_INC_ARCH_WAVE600
				AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW160), BW_40_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW160DeltaTAverage, TRUE); 
#endif
			}
			else
			{
				//Secondary channels are free, update them with the optimal BW 
				AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW40), BW_40_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW40DeltaTAverage, TRUE); 
				AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW80), BW_80_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW80DeltaTAverage, TRUE); 
#ifdef ENET_INC_ARCH_WAVE600
				AlphaFilter_updateFilter(&(LinkAdaptationWlanEffectiveBW160), BW_160_MHZ, ADAPTIVE_FILTER_LN_BETA_VAL,CONST_ALPHA_FILTER_FOR_DT_AVERAGE,&WlanEffectiveBW160DeltaTAverage, TRUE); 
#endif
			}
		}
	}
}

/**********************************************************************************

estimatorsUpdateStaAndVapProtectionEst 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void estimatorsUpdateCollisionEst(TxSequencerReport_t* pSequencerReport, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	EstimatorsStaProtection_t* staProtectionPtr = &(laDbDistributionParameter->laStaGroupCommon->staProtection);
	vapEstimatorsDataBase_t* vapEstimatorsPtr = &(LinkAdaptationVapDatabase[pSequencerReport->vapIndex].estimatorsVapDataBase);
	uint8 stationOldMsb = ((staProtectionPtr->staCollisionAckCtsBitmap >> 31) & 0x1); //check the oldest frames status in the window - the MSB
	uint8 vapOldMsb = ((vapEstimatorsPtr->vapCollisionAckCtsBitmap >> 31) & 0x1); //check the oldest frames status in the window - the MSB
	uint8 wlanOldMsb = (wlanAckCtsCollisionBitmap >> 31) & 0x1;
#ifdef ENET_INC_ARCH_WAVE600
	BandId_e myBand = ConfigurationManager_GetMyBand();
#endif //ENET_INC_ARCH_WAVE600
	
	if((pSequencerReport->protectionSent) && (pSequencerReport->ccaSetByPta == FALSE)) // Probing frames are not taking in acount or protection. In case CCA was set by PTA do nothing.  	
	{	
		staProtectionPtr->staCollisionAckCtsBitmap = (staProtectionPtr->staCollisionAckCtsBitmap)<<1; //slide the window 1 bit left	
		vapEstimatorsPtr->vapCollisionAckCtsBitmap = (vapEstimatorsPtr->vapCollisionAckCtsBitmap)<<1; //slide the window 1 bit left	
		wlanAckCtsCollisionBitmap = wlanAckCtsCollisionBitmap<<1; //slide the window 1 bit left
	
#ifdef ENET_INC_ARCH_WAVE600
		LinkAdaptationStatistics.protectionSentCounter[myBand]++; //LINK_ADAPTATION_PROTECTION_SENT_COUNTER
#else
		LinkAdaptationStatistics.protectionSentCounter++; //LINK_ADAPTATION_PROTECTION_SENT_COUNTER
#endif //ENET_INC_ARCH_WAVE600
		if(pSequencerReport->protectionSucceeded)
		{
#ifdef ENET_INC_ARCH_WAVE600
			LinkAdaptationStatistics.protectionSucceededCounter[myBand]++; //LINK_ADAPTATION_PROTECTION_SUCCEEDED_COUNTER
#else
			LinkAdaptationStatistics.protectionSucceededCounter++; //LINK_ADAPTATION_PROTECTION_SUCCEEDED_COUNTER
#endif //ENET_INC_ARCH_WAVE600
			if(stationOldMsb) //check the oldest frames status in the window - the MSB
			{
				/* Update counters*/
				staProtectionPtr->staCollisionCount--;
			}
			if(vapOldMsb)
			{
				vapEstimatorsPtr->vapCollisionTotalRatio--;
			}
			if(wlanOldMsb)
			{
				wlanCollisionTotlaCount--; 
			}
			// No collision detected - New LSB is 0   
		}
		else /* Protection failed*/
		{
			if(stationOldMsb == 0x0)
			{
				// No collision detected - New LSB is 0  
				staProtectionPtr->staCollisionCount++;			
			}
			if(vapOldMsb == 0x0)
			{
				vapEstimatorsPtr->vapCollisionTotalRatio++;
			}
			if(wlanOldMsb == 0x0)
			{
				wlanCollisionTotlaCount++; 
			}
			staProtectionPtr->staCollisionAckCtsBitmap |= 0x1; // collision detected - set LSB with 1
			vapEstimatorsPtr->vapCollisionAckCtsBitmap	|= 0x1; // collision detected - set LSB with 1
			wlanAckCtsCollisionBitmap |= 0x1;
		}
	}
}

/**********************************************************************************

goodPutEstimation 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
uint8 goodPutEstimation(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 currentPer,uint8 rateIndex,Bandwidth_e bandwidth)
{
	uint8 currentGoodPut;
	uint16 factor;

	/*Extract LnBeta from link adaptation configuration params*/
	factor = MaxPhyRateAcPerBwPerNss[bandwidth][CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(AntennaSelectionGetActivatedAntennasCount())];
#ifdef ENET_INC_ARCH_WAVE600
	if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta == TRUE)
	{
		factor = MaxPhyRateAxPerBwPerNss[bandwidth][CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(AntennaSelectionGetActivatedAntennasCount())];
	}
#endif

	currentGoodPut = calculateGoodPut(laDbDistributionParameter,currentPer,rateIndex,bandwidth,factor);

	return currentGoodPut; 
}

void estimatorsRecheckCandidacy(StaId staIndex)
{
	uint8 	muCandidacyGrade = 0;

	/*This function is called when STA is removed from all groups*/	
	muCandidacyGrade = estimatorsCalcMuCandidacyGrade(&LinkAdaptationStaDatabase[staIndex]);
	ILOG0_DD("estimatorsRecheckCandidacy, staId = %d, Station's grade is - %d", staIndex,muCandidacyGrade);
	if(muCandidacyGrade > MU_CANDIDATE_THRESHOLD_FACTOR_128_SCALE)
	{
		Utils_SetBitInBitmap((uint32 *)(muCandidateStaBitmap), staIndex); 	
	}
	else
	{			
		Utils_ZeroBitInBitmap((uint32 *)(muCandidateStaBitmap), staIndex);		
	}
}


/**********************************************************************************

calculateTsfDiff 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
uint16 calculateTsfDiff(AdaptiveAlphaFilterParams_t* pFilterParams)
{
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	uint32 adaptTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);
	uint16 diffTsf;

	if (pFilterParams->lastPacketTsf != TSF_INVALID)
	{
		diffTsf = cyclicMinusOperator (adaptTsf, pFilterParams->lastPacketTsf, TRUE, TSF_RANGE_ALPHA_FILTER); 
	}	
	else
	{
		/*When Tsf invalid, beta calc should get the maximum diff time so the beta will be zero and last sample will be taken for filter calculation*/
		diffTsf = MAX_UINT16;  
	}
	return diffTsf;
}

/**********************************************************************************

lowFrameRateEstimation 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void lowFrameRateEstimation(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint16 totalSizeOfAllMpdus,uint16 averageDtFilterResult)
{
	uint8 currentLowFrameRateCounter;
	LinkAdaptationConfigurationParams_t* pConfigurtionParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	/*Extract LnBeta from link adaptation configuration params*/
	uint32 lnBeta = pConfigurtionParams->lnBeta;
	AdaptiveAlphaFilterParams_t* pFilterParams; 
	uint16	diffTsf;
	uint8	staNssCount;
	uint16	factor;
	Bandwidth_e	bWLimit;
	uint16*	avarageDt;
	UNUSED_PARAM(averageDtFilterResult);		
	/*Update low frame rate counter filter	*/
	pFilterParams = &(laDbDistributionParameter->laStationUnique->lowFrameRateCounter);
	diffTsf = calculateTsfDiff(pFilterParams);
	diffTsf = MAX(diffTsf,1);
	avarageDt = &(laDbDistributionParameter->laStationUnique->lowFrameRateAverageDtFilterResult);
	ASSERT(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX);
	staNssCount =StaDbHwEntries[laDbDistributionParameter->stationOrGroupIndex].common.maxNssTx;
	bWLimit =GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,INVALID_MU_USP_INDEX, FALSE);

	factor = MaxPhyRateAcPerBwPerNss[bWLimit][staNssCount];
#ifdef ENET_INC_ARCH_WAVE600
	if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta == TRUE)
	{
		factor = MaxPhyRateAxPerBwPerNss[bWLimit][staNssCount];
	}
#endif
	
	/* divide the totalSizeOfMPDU in the TSF diff, then normilize by the MAX size of VHT MPDU*/
	currentLowFrameRateCounter = LOW_FRAME_RATE_CALC(totalSizeOfAllMpdus,diffTsf,factor);
	AlphaFilter_updateFilter(pFilterParams, currentLowFrameRateCounter,lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,avarageDt, FALSE); 
}


/**********************************************************************************

estimatorsProcessCollisionsBaaReport 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void estimatorsProcessCollisionsBaaReport(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter,uint8 vapId, BaaRateAdaptiveFullFifo_t*pBaaReport)
{
	
	EstimatorsStaProtection_t* staProtectionPtr = &(laDbDistributionParameter->laStaGroupCommon->staProtection);
	vapEstimatorsDataBase_t* vapEstimatorsPtr = &(LinkAdaptationVapDatabase[pBaaReport->userReport.txQVapId].estimatorsVapDataBase);
	LinkAdaptationConfigurationParams_t* pConfigurtionParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams; 
	uint8 stationOldMsb;
	uint8 vapOldMsb;
	uint8 wlanOldnMsb;
	uint8 currentBer =0;
	UNUSED_PARAM(vapId);
	/* Calculate  BER					 */
	/*1 < currentBer < totalSizeOfAllMpdus+1 */
	if((pBaaReport->userReport.totalFailMpduLength + pBaaReport->userReport.totalSuccessMpduLength) == 0)
	{
		currentBer =0x0; 
	}
	else
	{
		currentBer = (pBaaReport->userReport.totalFailMpduLength*HUNDRED_PERCENT)/(pBaaReport->userReport.totalFailMpduLength + pBaaReport->userReport.totalSuccessMpduLength); 
	}
	
	/*update the Acknowledge bitmap  */
#ifdef SEQUENCER_REPORT_NOT_AVAILABLE_PRE_BAA_REPORTS
	if(pBaaReport->commonReport.protectionSent == 0)
#else
	if(CurrentSequencerReport.protectionSent == 0) /* Protection hasn't been sent for this packet*/ 
#endif 
	{
		stationOldMsb = ((staProtectionPtr->staCollisionAckCtsBitmap >> 31) & 0x1); //check the oldest frames status in the window - the MSB
		vapOldMsb = ((vapEstimatorsPtr->vapCollisionAckCtsBitmap >> 31) & 0x1); //check the oldest frames status in the window - the MSB
		wlanOldnMsb =  (wlanAckCtsCollisionBitmap >> 31) & 0x1;
	
		staProtectionPtr->staCollisionAckCtsBitmap = (staProtectionPtr->staCollisionAckCtsBitmap)<<1; //slide station's window 1 bit left
		vapEstimatorsPtr->vapCollisionAckCtsBitmap = (vapEstimatorsPtr->vapCollisionAckCtsBitmap)<<1; //slide vap's window 1 bit left
		wlanAckCtsCollisionBitmap = wlanAckCtsCollisionBitmap<<1; //slide WLAN's window 1 bit left

		/*Protection estimators*/		
		
		/* PTA did not set the CCA - analyze the results */
		if(pBaaReport->userReport.timeout == 1)
		{
#ifndef ENET_INC_ARCH_WAVE600
			if (pBaaReport->commonReport.ccaSetByPta == FALSE)
#endif
			{
				/* Handle station's estimator*/
				if(stationOldMsb == 0x0)
				{
					staProtectionPtr->staCollisionCount++;
				}
				/* Handle vap's estimator*/
				if(vapOldMsb == 0x00)
				{
					vapEstimatorsPtr->vapCollisionTotalRatio++;
				}
				if(wlanOldnMsb == 0x0)
				{
					wlanCollisionTotlaCount++; 
				}
				staProtectionPtr->staCollisionAckCtsBitmap |= 0x1; // collision detected - set LSB with 1
				vapEstimatorsPtr->vapCollisionAckCtsBitmap	|= 0x1; // collision detected - set LSB with 1
				wlanAckCtsCollisionBitmap |= 0x1; // collision detected - set LSB with 1
				AlphaFilter_updateFilter(&(vapEstimatorsPtr->unprotectedFramesBER), currentBer,pConfigurtionParams->lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(vapEstimatorsPtr->unprotectedFramesBERDeltaTAverage), TRUE); 
			}
		}
		else // no fail detected 
		{
			/* Handle station's estimator*/
			if(stationOldMsb == 0x01)
			{
				staProtectionPtr->staCollisionCount--;
			}
			if(vapOldMsb == 0x01)
			{
				vapEstimatorsPtr->vapCollisionTotalRatio--;
			}
			if(wlanOldnMsb == 0x01)
			{
				wlanCollisionTotlaCount--; 
			}
			//LSB is set to 0 (by the sliding operation)
			AlphaFilter_updateFilter(&(vapEstimatorsPtr->unprotectedFramesBER), currentBer,pConfigurtionParams->lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(vapEstimatorsPtr->unprotectedFramesBERDeltaTAverage), TRUE); 
		}		
	}
	else
	{
		AlphaFilter_updateFilter(&(vapEstimatorsPtr->protectedFramesBER), currentBer,pConfigurtionParams->lnBeta,pConfigurtionParams->constAlphaCoefForDtAvPower2,&(vapEstimatorsPtr->protectedFramesBERDeltaTAverage),TRUE); 
	}
}	
/**********************************************************************************

consequtiveSuccessEstimators 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static void  fastDropEstimator (uint8 rateIndex, LaStationUspCommonParams_t* pEstDatabase, BaaRateAdaptiveFullFifo_t*pBaaReport, bool isCurrentPacketFailed)
{
	uint8 oldPacketInWindowFailed;
	FastDropParams_t fastDropParams;
	bool consequtiveFailsCondition;
	UNUSED_PARAM(pBaaReport);	
	getFastDropParams(rateIndex, &fastDropParams, pEstDatabase);
	
	consequtiveFailsCondition = ((pEstDatabase->windowFailedPacketBitmap) & (fastDropParams.consFailMask)) == fastDropParams.consFailMask;

	oldPacketInWindowFailed = (((pEstDatabase->windowFailedPacketBitmap) >> (FAST_DROP_BITMAP_WINDOW_SIZE - 1)) & 0x1);
	pEstDatabase->windowFailedPacketBitmap <<= 1;

	if (isCurrentPacketFailed == TRUE)
	{
		pEstDatabase->windowFailedPacketBitmap |= 0x1;
		if (oldPacketInWindowFailed == FALSE)
		{
			pEstDatabase->numberOfFailedPacketInWindow++;
		}
	}
	else
	{
		if (oldPacketInWindowFailed == TRUE)
		{
			pEstDatabase->numberOfFailedPacketInWindow--;
		}
		
	}
	if (((pEstDatabase->numberOfFailedPacketInWindow >= fastDropParams.numberOf1sInSlidingWind) && (enableFastDrop == ENABLE_FAST_DROP)) || (consequtiveFailsCondition == TRUE))
	{
		CurrentPacketParams.estimatorsIndication.consequtiveFailure = TRUE;
		CurrentPacketParams.estimatorsIndication.failsInWindow = TRUE;

	}
	else if(pEstDatabase->numberOfFailedPacketInWindow >= fastDropParams.numberOf1sInSlidingWind)
	{
		/*This bit will trigger change of the stabiliity state to High PER*/
		CurrentPacketParams.estimatorsIndication.failsInWindow = TRUE;
	}
	ASSERT(pEstDatabase->numberOfFailedPacketInWindow <= (FAST_DROP_BITMAP_WINDOW_SIZE));
#ifdef LINK_ADAPTATION_LOGS
//	ILOG0_DDDDDDD("fastDropEstimator, windowFailedPacketBitmap = %d, numberOfFailedPacketInWindow = %d,  consFailMask = %d, numbOfFailsinWind = %d, consequtiveFailsCondition = %d, FailIndication = %d,failsInWindow = %d",
	//			pEstDatabase->windowFailedPacketBitmap, pEstDatabase->numberOfFailedPacketInWindow, fastDropParams.consFailMask,fastDropParams.numberOf1sInSlidingWind, consequtiveFailsCondition, CurrentPacketParams.estimatorsIndication.consequtiveFailure,CurrentPacketParams.estimatorsIndication.failsInWindow);
	ILOG0_DD("fastDropEstimator, consequtiveFailsCondition = %d, FailIndication = %d",
			consequtiveFailsCondition, CurrentPacketParams.estimatorsIndication.consequtiveFailure);
#endif
}
/**********************************************************************************

checkCurrentPacketFailed 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static bool checkCurrentPacketFailed(BaaRateAdaptiveFullFifo_t* pBaaReport)
{
	bool isPacketFail = FALSE;

	if (pBaaReport->userReport.failedMpduCnt >= ((pBaaReport->userReport.failedMpduCnt + pBaaReport->userReport.successMpduCnt + 1) >> 1))
	{
		isPacketFail = TRUE;
	}
	return isPacketFail;
}
/**********************************************************************************

estimatorsFastProbingSearchMaxEffTp 


Description:
------------
Function find the maximum effective TP between 12 points (4 rates with 3 Bws each) 

Input: 
-----

Returns:
--------
pMaxFilterResultAddr - pointer to specific rate and BW entry with max TP in bandwidthData array
maxEffectiveTP - return valid value only if max TP is grates than WP TP + marginTp
	
**********************************************************************************/
void estimatorsFastProbingSearchMaxEffTp(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, FastProbingPointEstimators_t* fastProbingPointEstimators,uint8* fastProbingPointEstimatorsIndexes,FastProbingPointEstimators_t** pMaxTpFilterResultAddrOutput , uint32 maxValidTsfDiff, uint16 maxEffectiveTp, uint16 marginPercent) 
{
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	RaFastProbingPoint_e probingPointIndex;
	uint8 estimatorDbEntry = MAX_UINT8;
	FastProbingPointEstimators_t* pCurrentEntry = NULL;
	uint16 losslessTp;
	uint8 per;
	uint16 effectiveTp;
	Bandwidth_e bandwidth;
	estimatorDbEntry = fastProbingPointEstimatorsIndexes[WORKING_POINT];
	DEBUG_ASSERT(estimatorDbEntry < TOTAL_NUMBER_OF_RATES_IN_DB);
	bool isHeGroup;

	isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);	
	bandwidth = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, isHeGroup);
	pCurrentEntry = &(fastProbingPointEstimators[estimatorDbEntry]);
	ASSERT(bandwidth < (MAX_POSSIBLE_NUM_OF_BW));
	/*If working point filer is not valid there is no point to check the other probing point, because we dont want to change working point*/
	if (AlphaFilter_isFilterResultValid(&pCurrentEntry->averagePer, maxValidTsfDiff) == TRUE) 
	{
		/*Loop over 6 rates (5 fast probing and working point) in specific bandwidthData entry*/
		for (probingPointIndex = (RaFastProbingPoint_e) 0 ;probingPointIndex < TOTAL_NUMBER_OF_RATES_IN_DB ; probingPointIndex++)
		{
			/*Extract estimators db entry according to fast probing point */
			estimatorDbEntry = fastProbingPointEstimatorsIndexes[probingPointIndex];

			if (estimatorDbEntry < TOTAL_NUMBER_OF_RATES_IN_DB)
			{
				/*Set pointer to specific table entry*/
				pCurrentEntry = &(fastProbingPointEstimators[estimatorDbEntry]);
				//ILOG0_DD("estimatorsFastProbingSearchMaxEffTp, rateIndex = %d, bandwidth = %d", pCurrentEntry->rateIndex, pCurrentEntry->bandwidth);
#ifdef LINK_ADAPTATION_LOGS					
				ILOG0_DDD("estimatorsFastProbingSearchMaxEffTp, AvPerFilter: filetrResult = %d, alpha = %d, Last TSF = %d",(pCurrentEntry->averagePer.filterResult)>>11, pCurrentEntry->averagePer.alpha, pCurrentEntry->averagePer.lastPacketTsf);
#endif
				/*If filter result is valid (TSF check) and below the minimum value, define new min val*/
				if (AlphaFilter_isFilterResultValid(&pCurrentEntry->averagePer, maxValidTsfDiff))
				{
					/*Extract efefctive TP according to bandwidth*/
					losslessTp = ratesTable[pCurrentEntry->rateIndex].rateTableBwParmas[bandwidth].shortCpPhyRate;
					per = AlphaFilter_GetFilterResult(&pCurrentEntry->averagePer);
					effectiveTp = estimatorsCalcEffectiveTp(losslessTp,per);
					/*If working point, add margin to effective TP*/
					if (probingPointIndex == WORKING_POINT)
					{
						effectiveTp = ADD_PERCENT (effectiveTp, marginPercent);
					}
					/*If effective TP > max effective TP (not equal!) update new max*/
					if (effectiveTp > maxEffectiveTp)
					{
						maxEffectiveTp = effectiveTp;
						*pMaxTpFilterResultAddrOutput = pCurrentEntry;
					}
				}
			}
		}
	}
}
/**********************************************************************************

estimatorsGetUpMcsRateIndex 


Description:
------------
Input: 
-----

Returns:
--------
Return UP_MCS rate index for specific BW.
	
**********************************************************************************/

uint8 estimatorsGetProbingPointRateIndex(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, RaFastProbingPoint_e probingpoint)
{
	uint8 rateindex = INVALID_ESTIMATORS_DB_INDEX_VAL;
	uint8 probingPointEstimatorsIndex = laDbDistributionParameter->laStaUspCommon->fastProbingPointEstimatorsIndexes[probingpoint];

	if ((probingPointEstimatorsIndex != INVALID_ESTIMATORS_DB_INDEX_VAL) && (probingPointEstimatorsIndex < TOTAL_NUMBER_OF_RATES_IN_DB))
	{
		rateindex = laDbDistributionParameter->laStaUspCommon->fastProbingEstimators[probingPointEstimatorsIndex].rateIndex;
	}
	return rateindex;
}

/**********************************************************************************

estimatorsGetWorkingPointRateIndexOfMainBw 


Description:
------------
Input: 
-----

Returns:
--------
Return UP_MCS rate index for specific BW.
	
**********************************************************************************/

uint8 estimatorsGetWorkingPointRateIndexOfMainBw(LaStationUspCommonParams_t* laStaUspCommon)
{
	////KW_FW_FIX_M: No need for the local variable workingPointRateIndex

	uint8 workingPointEstimatorsIndex = laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];

	DEBUG_ASSERT(workingPointEstimatorsIndex != INVALID_ESTIMATORS_DB_INDEX_VAL )


	return(laStaUspCommon->fastProbingEstimators[workingPointEstimatorsIndex].rateIndex);

} //KW IGNORE FUNCRET.GEN 
/**********************************************************************************

estimatorsGetHigherProbingRateIndex


Description:
------------
Input: 
-----

Returns:
--------
Return Higher rate for specific BW
	
**********************************************************************************/
uint8 estimatorsGetHigherProbingRateIndex(StaId staIndex)
{
	const RateObj_t* ratesTable;
	uint8 rateIndex;
	uint8 wpRateIndex;
	uint8 sortWpRateIndex;
	RaFastProbingPoint_e fastProbingPoint;
	uint8 higherProbingRateIndex = INVALID_RATE_INDEX_VAL;
	uint8 sortRateIndex;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter,staIndex,INVALID_MU_USP_INDEX, FALSE);
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);

	/*Get higher rate */
	/*Find the higher closest rate to working point rate out of probing points*/
	
	higherProbingRateIndex = estimatorsGetProbingPointRateIndex(&laDbDistributionParameter, UP_MCS);
	
	/*IF Up MCS point is invalid, search for the other rate above WP out of the probing points*/
	if (higherProbingRateIndex == INVALID_RATE_INDEX_VAL)
	{
		wpRateIndex = estimatorsGetProbingPointRateIndex(&laDbDistributionParameter, WORKING_POINT);
		ASSERT(wpRateIndex !=INVALID_RATE_INDEX_VAL); 

		sortWpRateIndex = ratesTable[wpRateIndex].sortPhyRateSameBwIndex;

		for (fastProbingPoint = LOWER_RATE; fastProbingPoint < WORKING_POINT; fastProbingPoint++)
		{
			rateIndex = estimatorsGetProbingPointRateIndex(&laDbDistributionParameter, fastProbingPoint);
			if (rateIndex != INVALID_RATE_INDEX_VAL)
			{
				sortRateIndex = ratesTable[rateIndex].sortPhyRateSameBwIndex;
				if ((sortRateIndex > sortWpRateIndex) && ((int8)(ratesTable[rateIndex].vhtHeMcs - ratesTable[wpRateIndex].vhtHeMcs)<=1 ))
				{
					higherProbingRateIndex = rateIndex;
					break;
				}
			}
		}
	}
	return higherProbingRateIndex;
}
/**********************************************************************************

getFastDropParams



Description:
------------
Input: 
-----

Returns:
--------
Fast drop parameters	
**********************************************************************************/

static void getFastDropParams(uint8 rateIndex, FastDropParams_t* fastDropParams, LaStationUspCommonParams_t* pEstDatabase)
{
	const RateObj_t* ratesTable = getRatesTable(pEstDatabase);
	LinkAdaptationPhyMode_e	laPhyMode = ratesTable[rateIndex].laPhyMode;
	uint8 mcs = ratesTable[rateIndex].vhtHeMcs;

	switch (laPhyMode)
	{
#ifdef ENET_INC_ARCH_WAVE600
		case LA_PHY_MODE_HE:
			ASSERT(mcs < HE_NUMBER_OF_MCS);
			MEMCPY(fastDropParams, &FastDropParametersHe[mcs], sizeof(FastDropParams_t));
			break;
#endif
		case LA_PHY_MODE_HT_VHT:
			ASSERT(mcs<VHT_NUMBER_OF_MCS);
			if (enableFastDrop == ENABLE_FAST_DROP)
			{
				MEMCPY(fastDropParams, &FastDropEnabledParametersHtVht[mcs], sizeof(FastDropParams_t));
			}
			else
			{
				MEMCPY(fastDropParams, &FastDropDisabledParametersHtVht[mcs], sizeof(FastDropParams_t));
			}
			break;
		case LA_PHY_MODE_11B:
			if (enableFastDrop == ENABLE_FAST_DROP)
			{
				MEMCPY(fastDropParams, &FastDropEnabledParameters11b, sizeof(FastDropParams_t));
			}
			else
			{
				MEMCPY(fastDropParams, &FastDropDisabledParameters11b, sizeof(FastDropParams_t));
			}
			break;	
		case LA_PHY_MODE_11AG:
			ASSERT(mcs<8);
			MEMCPY(fastDropParams, &FastDropParameters11ag[mcs], sizeof(FastDropParams_t));
			break;	
		default:
			ASSERT(0);
	}
}


/**********************************************************************************

estimatorsCalcMuCandidacyGrade 


Description:
------------


Input: 
-----


Returns:
--------
	void - 
	
**********************************************************************************/
static uint32  estimatorsCalcMuCandidacyGrade(LinkAdaptationStaDatabase_t* laStationDb )
{
	uint32 stationGrade = HUNDRED_PERCENT_128_SCALE; 

	stationGrade = (HUNDRED_PERCENT_128_SCALE - ((laStationDb->laStationUnique.stationInPsOrManagementWithinWindowCount)*HUNDRED_PERCENT_128_SCALE/MU_CANDIDATE_PS_MGMT_WINDOW_SIZE))
					*MU_CANDIDATE_PS_MGMT_IN_AWINDOW_FACTOR_128_SCALE; 
	stationGrade += (AlphaFilter_GetFilterResult(&(laStationDb->laStationUnique.lowFrameRateCounter)) *MU_CANDIDATE_LOW_FRAME_RATE_FACTOR_128_SCALE);
	stationGrade +=	(AlphaFilter_GetFilterResult(&(laStationDb->laStaGroupCommon.wpAverageGoodPut))*MU_CANDIDATE_GOODPUT_FACTOR_128_SCALE); 
	stationGrade /= HUNDRED_PERCENT_128_SCALE;

	//ILOG0_D("linkAdaptationCalcMuCandidacyGrade - stationInPsOrManagementWithinWindowCount =%d ",(laStationDb->laStationUnique.stationInPsOrManagementWithinWindowCount));
	//ILOG0_D("linkAdaptationCalcMuCandidacyGrade - lowFrameRateCounter =%d ",AlphaFilter_GetFilterResult(&(laStationDb->laStationUnique.lowFrameRateCounter)));
	//ILOG0_D("linkAdaptationCalcMuCandidacyGrade - wpAverageGoodPut =%d ",AlphaFilter_GetFilterResult(&(laStationDb->laStaGroupCommon.wpAverageGoodPut)));

	/* In case station is already in MU - give it a grade of 100 */
	return stationGrade; 
}

