/***********************************************************************************
 File:		Estimators.h
 Module:		LinkAdaptation 
 Purpose: 	Process estimators based in the Sequencer & BAA reports
 Description:	This file contains all definitions and the structures of the estimators
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "Estimators.h"
#include "LinkAdaptation.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "ProtectedDbLock_Api.h"
#include "ShramStationDatabase.h"
#include "StaDb.h"
#include "mt_sysdefs.h"
#include "lm.h"
#include "BaAnalyzer_Api.h"
#include "BwAdaptation.h"
#include "ShramStatistics.h"
#include "RateAdaptation.h"
#include "LinkAdaptation_StateMachine.h"
#include "HwSemaphore_API.h"
#include "AntennaSelection.h"
#include "DynamicTxOP.h"
#include "ConfigurationManager_api.h"
#include "CommonRamLinkAdaptation.h"
#include "ProtectionAdaptation.h"
#include "QAMplus.h"
#include "TurboRates.h"
#include "GroupsMetricsCalculator.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 22

#define INVALID_EFFICIENCY MAX_UINT8

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct PsduEfficiencyCalcData_t
{
	uint16  sumAllUserData;
	uint8   transactionNumber;
	uint8   numberOfUsers;
	uint8   reportsCounter;
    bool    firstOccurrence;
} PsduEfficiencyCalcData_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
#ifdef DYNAMIC_MU_EFFICIENCY_CALC
void GroupsMetricsCalculator_CalcAndUpdateDlPsduEfficiency(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter,BaaRateAdaptiveFullFifo_t* pBaaReport, uint32 lnBeta, uint8 constAlphaCoefForDtAvPower2, uint16* pAvDTResult);
#endif
void MetricsCalculator_PrepareData(void);
void MetricsCalculator_ResetGroupData(uint8 groupId);
void MetricsCalculator_ResetStaData(StaId staId);
void GroupsMetricsCalculator_Init(void);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
PsduEfficiencyCalcData_t IntermediateVar;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
#ifdef DYNAMIC_MU_EFFICIENCY_CALC
void GroupsMetricsCalculator_CalcAndUpdateDlPsduEfficiency(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter,BaaRateAdaptiveFullFifo_t* pBaaReport, uint32 lnBeta, uint8 constAlphaCoefForDtAvPower2, uint16* pAvDTResult)
{
	uint8 avPsduEfficiency = 0;
	uint32 totalSuccessMpduLength = pBaaReport->userReport.totalSuccessMpduLength;
	uint32 totalFailMpduLength = pBaaReport->userReport.totalFailMpduLength;
	uint32 totalLength = pBaaReport->userReport.totalLength;
	uint8 psduAverageEfficiency = 0;

	if ((pBaaReport->commonReport.baaTransactionNumber != IntermediateVar.transactionNumber) || (IntermediateVar.firstOccurrence == TRUE))
	{
		if(IntermediateVar.firstOccurrence != TRUE)
		{
			/*New Group report*/
			/*Take the average of all users*/
			ASSERT(IntermediateVar.reportsCounter == (IntermediateVar.numberOfUsers)); // number of user reports must be equal to pBaaReport->commonReport.numOfUsers
			psduAverageEfficiency = (IntermediateVar.sumAllUserData / IntermediateVar.numberOfUsers);
			/*Input average into group alpha filter*/
			AlphaFilter_updateFilter(&(laDbDistributionParameter->laHeGroupUnique->dlPsduEfficiency), psduAverageEfficiency,lnBeta,constAlphaCoefForDtAvPower2,pAvDTResult, FALSE); for dynamic grouping debug - TBD
			
		}
		IntermediateVar.transactionNumber = pBaaReport->commonReport.baaTransactionNumber;
		IntermediateVar.numberOfUsers = pBaaReport->commonReport.numOfUsers;
		IntermediateVar.reportsCounter = 1;
        IntermediateVar.firstOccurrence = FALSE;
	}
	else 
	{
		/*Count user reports fron the same MU transmission*/
		avPsduEfficiency = HUNDRED_PERCENT_128_SCALE*(totalSuccessMpduLength + totalFailMpduLength)/totalLength;
		IntermediateVar.sumAllUserData += avPsduEfficiency;
		IntermediateVar.reportsCounter++;
	}
}

#endif
/**********************************************************************************
MetricsCalculator_PrepareData 

Description:
------------
Copy data from alpha filters metrics results to common ram for  he group manager

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void MetricsCalculator_PrepareData(void)
{
	uint8 groupId;
	uint8 filterResult;
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;

	for (groupId = 0; groupId < NUM_OF_LA_HE_MU_DB_ENTRIES; groupId++)
	{
		if(linkAdaptationIsGroupActive(groupId, TRUE))
		{
#ifdef DYNAMIC_MU_EFFICIENCY_CALC
			filterResult = AlphaFilter_GetFilterResult(&LinkAdaptationHePhaseDatabase[groupId].laHeGroupUnique.dlPsduEfficiency);
#else
			filterResult = 0;
#endif
			GroupsMetrics[groupId].dlPsduEfficiency = filterResult;
		}
	}

	while (nextSid != DB_ASYNC_SID)
	{
		linkAdaptationFindMaxRssi(nextSid, &SuMetrics[nextSid].maxRssiFromAlLAnts);
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
}


void MetricsCalculator_ResetStaData(StaId staId)
{
	SuMetrics[staId].maxRssiFromAlLAnts = QAM_PLUS_MIN_RSSI_DB;
}


void MetricsCalculator_ResetGroupData(uint8 groupId)
{
	GroupsMetrics[groupId].dlPsduEfficiency= INVALID_EFFICIENCY;
	GroupsMetrics[groupId].ulPsduEfficiency= INVALID_EFFICIENCY;
}


void GroupsMetricsCalculator_Init(void)
{
    memset(&IntermediateVar, 0, sizeof(PsduEfficiencyCalcData_t)); 
    IntermediateVar.firstOccurrence = TRUE; 
}


