/***********************************************************************************
 File:		PowerAdaptation.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "PowerAdaptation.h"
#include "HdkGlobalDefs.h"
#include "lm_StaDatabase.h"
#include "AntennaSelection.h"
#include "mt_sysdefs.h"
#include "LinkAdaptation_StateMachine.h"
#include "Pac_Api.h"
#include "RateAdaptation.h"
#include "Estimators.h"
#include "ShramStatistics.h"
#include "loggerAPI.h"
#include "Estimators.h"
#include "CommonRamLinkAdaptation.h"
#include "ConfigurationManager_api.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 14


#if (defined ENET_INC_ARCH_WAVE600 && defined DEBUG_LOGGER_ENABLE)
#define POWER_ADAPTATION_LOGS
#endif

#define MAXIMUM_POSSIBLE_VAPS 16
#define INITIAL_RATE_NUM_OF_OPTIONS_11N 8
#define INITIAL_RATE_NUM_OF_OPTIONS_11AX 12
#define INITIAL_RATE_NUM_OF_OPTIONS_11AC 10
#define INITIAL_RATE_NUM_OF_OPTIONS_11AG 8
#define INITIAL_RATE_NUM_OF_OPTIONS_11B 4
#define INITIAL_RATE_NUM_OF_OPTIONS_HT_VHT INITIAL_RATE_NUM_OF_OPTIONS_11AC
#define LA_GAIN_RESOLUTION 3

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum PowerNextProbingPoint
{
	POWER_PROBE_UP,
	POWER_PROBE_DOWN,
	NUM_OF_POWER_PROBING_OPTIONS,
	MAX_NUM_OF_POWER_PROBING_OPTIONS = 0xFF
	
}PowerNextProbingPoint_e;
typedef struct PowerGlobalParams
{
	int8 externalUpperLimit[LA_NUM_OF_BANDWIDTH];
	int8 externalLowerLimit[LA_NUM_OF_BANDWIDTH];
	uint8 refPower[LA_NUM_OF_BANDWIDTH];
}PowerGlobalParams_t;
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static RatePowerOffset_t getRatePowerOffset(bool isVht, uint8 rateindex, const RateObj_t* ratesTable);
static int8 searchMinPowerOverAntAtSpecificBw(int8  pTable[MAX_NUM_OF_ANTENNAS][LA_NUM_OF_BANDWIDTH],uint8 antennaSelectionBitmap, Bandwidth_e bandwidth);
static int8 searchMaxPowerOverActivateAnt(const int8  pTable[MAX_POSSIBLE_NUM_OF_ANTENNAS],uint8 antennaSelectiondBitmap);
static void calcInternalLowerLimit(StaId staIndex, uint8 antennaSelectiondBitmap);
static void calcExternalLowerLimit(StaId staIndex, uint8 antennaSelectionBitmap, Bandwidth_e bandwidth);
static void calcExternalUpperLimit(StaId staIndex, uint8 antennaSelectionBitmap, Bandwidth_e bandwidth, uint8 isBFExist);
static void setNextPowerStep(StaId staIndex, bool probeWasSuccess);
static void increasePowerStep(uint8* pAdditionalStep, uint8 powerStepDefault, uint8 minAbsPowerStep, uint8 configPowerAdditionalStep,uint8 powerMaxAdditionalPowerStep);
static void calcProximateStaIndication(void);
static void calcAsymmetricLinkIndication(void);
static void calcDistantStaIndication(void);
static int8 getStaAdaptaivePower(StaId staIndex, Bandwidth_e bandwidth, uint8 rateIndex, int8 accPower,bool useStepPower, bool useAdaptivePower);
static bool setPowerInStaTcr(StaId staIndex, int8* pPowerValTable, Bandwidth_e firstBwtoUpdate, Bandwidth_e lastBwToUpdate, bool changeProbingPoint, LaPacketType_e packetType, bool setRatePlusOne);
static void powerAdaptationModificationFunc(LaTcrModificationStruct_t* modifcationParamsIn, LaTcrModificationStruct_t* modifcationParamsOut);
static void SetPowerLimit(void);
static void extractInitRateTableAndLen(PhyMode_e staNetworkMode,int8** pInitialRateTable,uint8* tableLen);
static void updateInitPowerOffsetTables (PowerAdaptationTableType_e tableType);
static uint8 calcAvgAntennaBoost(uint8 antennaSelectionBitmap, uint8* antBoost);
static void extractAndSetAntBoosts(uint8* antBoost, LA_SET_TPC_DATA_t* pTpcPowerData);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*Tables from TPC*/
static int8 RegulationLimit[LA_NUM_OF_BANDWIDTH];
static int8 RegulationLimit11b;
int8 minPowerFromInternalLowerLimit[LA_NUM_OF_BANDWIDTH];
static uint8 lastPowerUpperLimitOffset;
uint8 antPowerBoost[MAX_NUM_OF_ANTENNAS];

/*Internals tables and variables*/
static int8 DefaultInternalLowerLimit;
static int8 DefaultRegulationLimitPerAnt[LA_NUM_OF_BANDWIDTH];

static int8 (*InitRatePowerOffset11AcMcs89)[VHT_HE_NUMBER_OF_NSS] = NULL;
static int8 InitRatePowerOffset11AcHighMcs89[2][VHT_HE_NUMBER_OF_NSS]; 
static int8 InitRatePowerOffset11AcLowMcs89[2][VHT_HE_NUMBER_OF_NSS];
#ifdef ENET_INC_ARCH_WAVE600
static int8 InitRatePowerOffset11Ax[4][VHT_HE_NUMBER_OF_NSS];
#endif

//5 pointers
static RatePowerOffset_t *RatePowerOffset11B = NULL;
static RatePowerOffset_t *RatePowerOffset11AG = NULL;
static RatePowerOffset_t *RatePowerOffset11N = NULL;
static RatePowerOffset_t (*RatePowerOffset11AC)[VHT_HE_NUMBER_OF_NSS] = NULL;
static RatePowerOffset_t (*RatePowerOffsetMU11AC)[VHT_HE_NUMBER_OF_NSS] =  NULL;

static RatePowerOffset_t RatePowerOffsetHigh11B = {0,0};

static RatePowerOffset_t RatePowerOffsetHigh11AG[NUMBER_OF_RATES_11AG] = 
/*
		        MCS,          	    powerOffset,      	  evmPowerBackoff,    
*/
{
	{	     /*0*/      	       0,       	       0        },
	{	     /*1*/      	       0,       	       0        },
	{	     /*2*/      	       0,       	       0        },
	{	     /*3*/      	       0,       	       0        },
	{	     /*4*/      	       0,       	       0        },
	{	     /*5*/      	       0,       	       0        },
	{	     /*6*/      	      -2,       	       0        },
	{	     /*7*/      	      -4,       	       0        }
};
static  RatePowerOffset_t RatePowerOffsetHigh11N[HT_NUMBER_OF_MCS] = 
/*
		        MCS,          	    powerOffset,      	  evmPowerBackoff,    
*/
{
	{	     /*0*/      	       0,       	       0        },
	{	     /*1*/      	       0,       	       0        },
	{	     /*2*/      	       0,       	       0        },
	{	     /*3*/      	       0,       	       0        },
	{	     /*4*/      	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0        },
	{	     /*9*/      	       0,       	       0        },
	{	     /*10*/     	       0,       	       0        },
	{	     /*11*/     	       0,       	       0        },
	{	     /*12*/     	       0,       	       0        },
	{	     /*13*/     	      -2,       	       0        },
	{	     /*14*/     	      -4,       	       0        },
	{	     /*15*/     	      -6,       	       0        },
	{	     /*16*/     	       0,       	       0        },
	{	     /*17*/     	       0,       	       0        },
	{	     /*18*/     	       0,       	       0        },
	{	     /*19*/     	       0,       	       0        },
	{	     /*20*/     	       0,       	       0        },
	{	     /*21*/     	      -2,       	       0        },
	{	     /*22*/     	      -4,       	       0        },
	{	     /*23*/     	      -6,       	       0        },
	{	     /*24*/     	       0,       	       0        },
	{	     /*25*/     	       0,       	       0        },
	{	     /*26*/     	       0,       	       0        },
	{	     /*27*/     	       0,       	       0        },
	{	     /*28*/     	       0,       	       0        },
	{	     /*29*/     	      -2,       	       0        },
	{	     /*30*/     	      -4,       	       0        },
	{	     /*31*/     	      -6,       	       0        },
	{	     /*32*/     	       0,       	       0        }
};
/*Power offset for MCS 0-7 is relative to max EVM power from TPC  while mcs 8-9 is relative to min EVM power from TPC calibration*/
static RatePowerOffset_t RatePowerOffsetHigh11AC[VHT_NUMBER_OF_MCS][VHT_HE_NUMBER_OF_NSS] = 
/*
		      MCS/NSS,        	NSS 1 - powerOffset,  	NSS 1 evmPowerBackoff, 	NSS 1 - powerOffset,  	NSS 1 - evmPowerBackoff, 	NSS 2 - powerOffset,  	NSS 2 - evmPowerBackoff, 	NSS 3 - powerOffset,  	NSS 3 - evmPowerBackoff, 
*/
{
	{	     /*0*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*1*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*2*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*3*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*4*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*9*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        }
};


static RatePowerOffset_t RatePowerOffsetLow11B = {0,0};

static RatePowerOffset_t RatePowerOffsetLow11AG[NUMBER_OF_RATES_11AG] = 
/*
		        MCS,          	    powerOffset,      	  evmPowerBackoff,    
*/
{
	{	     /*0*/      	       0,       	       0        },
	{	     /*1*/      	       0,       	       0        },
	{	     /*2*/      	       0,       	       0        },
	{	     /*3*/      	       0,       	       0        },
	{	     /*4*/      	       0,       	       0        },
	{	     /*5*/      	       0,       	       0        },
	{	     /*6*/      	      -2,       	       0        },
	{	     /*7*/      	      -4,       	       0        }
};
static  RatePowerOffset_t RatePowerOffsetLow11N[HT_NUMBER_OF_MCS] = 
/*
		        MCS,          	    powerOffset,      	  evmPowerBackoff,    
*/
{
	{	     /*0*/      	       0,       	       0        },
	{	     /*1*/      	       0,       	       0        },
	{	     /*2*/      	       0,       	       0        },
	{	     /*3*/      	       0,       	       0        },
	{	     /*4*/      	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0        },
	{	     /*9*/      	       0,       	       0        },
	{	     /*10*/     	       0,       	       0        },
	{	     /*11*/     	       0,       	       0        },
	{	     /*12*/     	       0,       	       0        },
	{	     /*13*/     	      -2,       	       0        },
	{	     /*14*/     	      -4,       	       0        },
	{	     /*15*/     	      -6,       	       0        },
	{	     /*16*/     	       0,       	       0        },
	{	     /*17*/     	       0,       	       0        },
	{	     /*18*/     	       0,       	       0        },
	{	     /*19*/     	       0,       	       0        },
	{	     /*20*/     	       0,       	       0        },
	{	     /*21*/     	      -2,       	       0        },
	{	     /*22*/     	      -4,       	       0        },
	{	     /*23*/     	      -6,       	       0        },
	{	     /*24*/     	       0,       	       0        },
	{	     /*25*/     	       0,       	       0        },
	{	     /*26*/     	       0,       	       0        },
	{	     /*27*/     	       0,       	       0        },
	{	     /*28*/     	       0,       	       0        },
	{	     /*29*/     	      -2,       	       0        },
	{	     /*30*/     	      -4,       	       0        },
	{	     /*31*/     	      -6,       	       0        },
	{	     /*32*/     	       0,       	       0        }
};
/*Power offset for MCS 0-7 is relative to max EVM power from TPC  while mcs 8-9 is relative to min EVM power from TPC calibration*/
static RatePowerOffset_t RatePowerOffsetLow11AC[VHT_NUMBER_OF_MCS][VHT_HE_NUMBER_OF_NSS] = 
/*
		      MCS/NSS,        	NSS 1 - powerOffset,  	NSS 1 evmPowerBackoff, 	NSS 1 - powerOffset,  	NSS 1 - evmPowerBackoff, 	NSS 2 - powerOffset,  	NSS 2 - evmPowerBackoff, 	NSS 3 - powerOffset,  	NSS 3 - evmPowerBackoff, 
*/
{
	{	     /*0*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*1*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*2*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*3*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*4*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*9*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        }
};

static RatePowerOffset_t RatePowerOffsetMUHigh11AC[VHT_NUMBER_OF_MCS][VHT_HE_NUMBER_OF_NSS] = 
/*
		      MCS/NSS,        	NSS 1 - powerOffset,  	NSS 1 evmPowerBackoff, 	NSS 1 - powerOffset,  	NSS 1 - evmPowerBackoff, 	NSS 2 - powerOffset,  	NSS 2 - evmPowerBackoff, 	NSS 3 - powerOffset,  	NSS 3 - evmPowerBackoff, 
*/
{
	{	     /*0*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*1*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*2*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*3*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*4*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*9*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        }
};

static RatePowerOffset_t RatePowerOffsetMULow11AC[VHT_NUMBER_OF_MCS][VHT_HE_NUMBER_OF_NSS] = 
/*
		      MCS/NSS,        	NSS 1 - powerOffset,  	NSS 1 evmPowerBackoff, 	NSS 1 - powerOffset,  	NSS 1 - evmPowerBackoff, 	NSS 2 - powerOffset,  	NSS 2 - evmPowerBackoff, 	NSS 3 - powerOffset,  	NSS 3 - evmPowerBackoff, 
*/
{
	{	     /*0*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*1*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*2*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*3*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*4*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*9*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        }
};

static PowerOffsetHighLow_t PowerOffsetHighLowfrequency[NUMBER_OF_FREQUENCIES] = 
{
	{		 /*0*/				   0,				   0		},
	{		 /*1*/				   0,				   0		},
	{		 /*2*/				   0,				   0		},
	{		 /*3*/				   0,				   0		},
	{		 /*4*/				   0,				   0		}
};


#ifdef ENET_INC_ARCH_WAVE600
/*Power offset for MCS 0-7 is relative to max EVM power from TPC  while mcs 8-11 is relative to min EVM power from TPC calibration*/
static RatePowerOffset_t RatePowerOffset11AX[HE_NUMBER_OF_MCS][VHT_HE_NUMBER_OF_NSS] = 
/*
		      MCS/NSS,        	NSS 1 - powerOffset,  	NSS 1 evmPowerBackoff, 	NSS 1 - powerOffset,  	NSS 1 - evmPowerBackoff, 	NSS 2 - powerOffset,  	NSS 2 - evmPowerBackoff, 	NSS 3 - powerOffset,  	NSS 3 - evmPowerBackoff, 
*/
{
	{	     /*0*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*1*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*2*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*3*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*4*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*5*/      	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0,       	      -2,       	       0        },
	{	     /*6*/      	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0,       	      -4,       	       0        },
	{	     /*7*/      	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0,       	      -6,       	       0        },
	{	     /*8*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*9*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },		
	{	     /*10*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        },
	{	     /*11*/      	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0,       	       0        }
};
#endif


static PowerLimit_t PowerLimits = {127,127,127,127,127,127};
int8 flagPowerOffsetHighLowfrequency = FALSE;

int8  InternalPowerLowerLimit[MAX_POSSIBLE_NUM_OF_ANTENNAS] =
/*
		      Antenna,        	       Value,         
*/
{
		     /*0*/      	       0        ,
		     /*1*/      	       0        ,
		     /*2*/      	       0        ,
		     /*3*/      	       0        
};
int8  ExternalPowerLowerLimit[MAX_POSSIBLE_NUM_OF_BW] = 
/*
		     Bandwidth,       	       Value,         
*/
{
		     /*20*/     	       0        ,
		     /*40*/     	       0        ,
		     /*80*/     	       0        ,
			 /*160*/ 			   0		,		     
};
int8 ProximateStaBackoff[MAXIMUM_POSSIBLE_VAPS]=
/*
		        VAP,          	       Value,         
*/
{
		     /*0*/      	       -12       ,
		     /*1*/      	       -12       ,
		     /*2*/      	       -12       ,
		     /*3*/      	       -12       ,
		     /*4*/      	       -12       ,
		     /*5*/      	       -12       ,
		     /*6*/      	       -12       ,
		     /*7*/      	       -12       ,
		     /*8*/      	       -12       ,
		     /*9*/      	       -12       ,
		     /*10*/     	       -12       ,
		     /*11*/     	       -12       ,
		     /*12*/     	       -12       ,
		     /*13*/     	       -12       ,
		     /*14*/     	       -12       ,
		     /*15*/     	       -12       
};
int8 DistantStaBackoff[MAXIMUM_POSSIBLE_VAPS] = 
/*
		        VAP,          	       Value,         
*/
{
	     /*0*/      	       8        ,
	     /*1*/      	       8        ,
	     /*2*/      	       8        ,
	     /*3*/      	       8        ,
	     /*4*/      	       8        ,
	     /*5*/      	       8        ,
	     /*6*/      	       8        ,
	     /*7*/      	       8        ,
	     /*8*/      	       8        ,
	     /*9*/      	       8        ,
	     /*10*/     	       8        ,
	     /*11*/     	       8        ,
	     /*12*/     	       8        ,
	     /*13*/     	       8        ,
	     /*14*/     	       8        ,
	     /*15*/     	       8        
};
uint8 const AntennaCountOffset[MAX_POSSIBLE_NUM_OF_ANTENNAS] = 
/*
		      Antenna,        	       Value,         
*/
{
	     /*0*/      	       0       ,
	     /*1*/      	       6       ,
	     /*2*/      	       9       ,
	     /*3*/      	       12      
};

#ifdef ENET_INC_ARCH_WAVE600

static const uint8 autoReply11AGMaxPowerPerBw[LA_NUM_OF_BANDWIDTH] = 
{
	60,
	56,
	56,
#ifdef ENET_INC_ARCH_WAVE600
	56 // TBD: must set correct value for bw 160
#endif //ENET_INC_ARCH_WAVE600
};

#else

static const uint8 autoReply11AGMaxPowerPerBw[MAX_POSSIBLE_NUM_OF_BW] = 
{
	60,
	56,
	56
};

#endif

#ifdef ENET_INC_ARCH_WAVE600
static int8 InitialRateAccordingToRssiHe[INITIAL_RATE_NUM_OF_OPTIONS_11AX] = 
{
/*
	RSSI TH,		Initial MCS,
*/
	   -128,		 /*0*/	
	   -84,			 /*1*/	
	   -80,			 /*2*/	
	   -76,			 /*3*/	
	   -72,			 /*4*/	
	   -68,			 /*5*/	
	   -64,			 /*6*/	
	   -60,			 /*7*/	
	   -56,			 /*8*/	
	   -50,			 /*9*/		   
	   -44,		     /*10*/  	   
	   -38,		     /*11*/  
};
#endif
static int8 InitialRateAccordingToRssiHtVht[INITIAL_RATE_NUM_OF_OPTIONS_HT_VHT] = 
{
/*
	RSSI TH,		Initial MCS,
*/
	   -128,		 /*0*/	
	   -83,			 /*1*/	
	   -82,			 /*2*/	
	   -76,			 /*3*/	
	   -73,			 /*4*/	
	   -68,			 /*5*/	
	   MAX_INT8,	 /*6*/	
	   MAX_INT8,	 /*7*/	
	   MAX_INT8,	 /*8*/	
	   MAX_INT8,	 /*9*/	

};
static int8 InitialRateAccordingToRssi11Ag[INITIAL_RATE_NUM_OF_OPTIONS_11AG] = 
{
/*
		RSSI TH,	Initial MCS,
*/
	   -128,		 /*0*/	
	   MAX_INT8,	 /*1*/	
	   MAX_INT8,	 /*2*/	
	   MAX_INT8,	 /*3*/	
	   MAX_INT8,	 /*4*/	
	   MAX_INT8,	 /*5*/	
	   MAX_INT8,	 /*6*/	
	   MAX_INT8,	 /*7*/	

};
static int8 InitialRateAccordingToRssi11b[INITIAL_RATE_NUM_OF_OPTIONS_11B] = 
{
/*
		RSSI TH,	Initial MCS,
*/
	  -128,			/*4*/
	  MAX_INT8,		/*1*/
	  MAX_INT8,		/*2*/
	  MAX_INT8,		/*3*/

};
static uint8 ConvertInitRateEntryToMcs11B[INITIAL_RATE_NUM_OF_OPTIONS_11B] = 
{
	4,	/*1Mbit/sec - long preamble*/
	1,	/*2Mbit/sec - short preamble*/
	2,	/*5.5Mbit/sec - short preamble*/
	3	/*11Mbit/sec - short preamble*/
};
static PowerAdaptationBitMaskDb_t powerAdaptationBitmaskDb;
static PowerGlobalParams_t PowerGlobalParameters;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

PowerAdaptationInit 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void PowerAdaptationInit (void)
{
	StaId staId;
	PowerAdaptationDatabase_t* pPowerAdaptationDb;
	StaId firstSidInBand = ConfigurationManager_GetFirstSidInMyBand();
	uint16 numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsInMyBand();	

	
	for (staId = firstSidInBand ; staId < (firstSidInBand + numOfSidsInBand) ; staId++)
	{
		pPowerAdaptationDb = &LinkAdaptationStaDatabase[staId].laStaGroupCommon.powerAdaptationDb;
		memset(pPowerAdaptationDb, 0, sizeof(PowerAdaptationDatabase_t));
	}

	
	memset(&powerAdaptationBitmaskDb, 0, sizeof(PowerAdaptationBitMaskDb_t));
	memset(&PowerGlobalParameters, 0, sizeof(PowerGlobalParams_t));
    memset(&FixedPowerParameters, 0, sizeof(LaFixedPowerParams_t));
        
	/*Init tables*/
	RegulationLimit11b = 0;
	memset(RegulationLimit, 0, LA_NUM_OF_BANDWIDTH);
	memset (antPowerBoost, 0, MAX_NUM_OF_ANTENNAS);

	/*Init Internals tables and variables*/
	DefaultInternalLowerLimit = 0;
	lastPowerUpperLimitOffset = 0;
	memset(DefaultRegulationLimitPerAnt,0, LA_NUM_OF_BANDWIDTH);
	
	updateInitPowerOffsetTables(POWER_OFFSET_11B_HIGH_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11AG_HIGH_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11N_HIGH_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11AC_HIGH_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11B_LOW_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11AG_LOW_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11N_LOW_TABLE);
	updateInitPowerOffsetTables(POWER_OFFSET_11AC_LOW_TABLE);
#ifdef ENET_INC_ARCH_WAVE600
	updateInitPowerOffsetTables(POWER_OFFSET_11AX_TABLE);
#endif
	UpdateRatePowerOffset(0,0);
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************

updateInitPowerOffsetTables 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
static void updateInitPowerOffsetTables (PowerAdaptationTableType_e tableType)
{
	uint8 nss;
/*Set initial power offsets table */

	switch (tableType)
	{	
		case POWER_OFFSET_11AC_HIGH_TABLE:
			for (nss = 0; nss < VHT_HE_NUMBER_OF_NSS; nss++)
			{
				InitRatePowerOffset11AcHighMcs89[0][nss] = RatePowerOffsetHigh11AC[MCS8_256QAM_3_4][nss].powerOffset;
				InitRatePowerOffset11AcHighMcs89[1][nss] = RatePowerOffsetHigh11AC[MCS9_256QAM_5_6][nss].powerOffset;
			}		
			break;
		case POWER_OFFSET_11AC_LOW_TABLE:
			for (nss = 0; nss < VHT_HE_NUMBER_OF_NSS; nss++)
			{
				InitRatePowerOffset11AcLowMcs89[0][nss] = RatePowerOffsetLow11AC[MCS8_256QAM_3_4][nss].powerOffset;
				InitRatePowerOffset11AcLowMcs89[1][nss] = RatePowerOffsetLow11AC[MCS9_256QAM_5_6][nss].powerOffset;
			}
			break;
#ifdef ENET_INC_ARCH_WAVE600
		case POWER_OFFSET_11AX_TABLE:
			for (nss = 0; nss < VHT_HE_NUMBER_OF_NSS; nss++)
			{
				InitRatePowerOffset11Ax[0][nss] = RatePowerOffset11AX[MCS8_256QAM_3_4][nss].powerOffset;
				InitRatePowerOffset11Ax[1][nss] = RatePowerOffset11AX[MCS9_256QAM_5_6][nss].powerOffset;				
				InitRatePowerOffset11Ax[2][nss] = RatePowerOffset11AX[MCS10_1024QAM_3_4][nss].powerOffset;
				InitRatePowerOffset11Ax[3][nss] = RatePowerOffset11AX[MCS11_1024QAM_5_6][nss].powerOffset;
			}
			break;
#endif
		case POWER_OFFSET_11B_HIGH_TABLE:
		case POWER_OFFSET_11AG_HIGH_TABLE:
		case POWER_OFFSET_11N_HIGH_TABLE:
		case POWER_OFFSET_11B_LOW_TABLE:
		case POWER_OFFSET_11AG_LOW_TABLE:
		case POWER_OFFSET_11N_LOW_TABLE:
		case POWER_OFFSET_11AC_MU_HIGH_TABLE:
		case POWER_OFFSET_11AC_MU_LOW_TABLE:
			break;
		default:
			DEBUG_ASSERT(0);
	}

}


/**********************************************************************************

PowerAdaptationAddStation 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
void PowerAdaptationAddStation(StaId staIndex)
{
	Bandwidth_e bandwidth;
	uint8 tcrBfMode;
	uint8 antennaSelectiondBitnap = AntennaSelectionGetAntennaSelectionBitmap(staIndex);
	LinkAdaptationStaDatabase_t* pLaStaDb = &LinkAdaptationStaDatabase[staIndex];
	PowerAdaptationDatabase_t* pPowerAdaptationDb = &(pLaStaDb->laStaGroupCommon.powerAdaptationDb);

	setAntBoostInStaTcr(staIndex, antPowerBoost);

	pPowerAdaptationDb->powerAdditionalStepUp = pLaStaDb->laStaGroupCommon.pLinkAdaptationConfigurationParams->powerConfigurationParams.powerAdditionalPowerStep;
	pPowerAdaptationDb->powerAdditionalStepDown = pLaStaDb->laStaGroupCommon.pLinkAdaptationConfigurationParams->powerConfigurationParams.powerStepDown;

	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.proximateStaIndication.staBitMap,staIndex);
	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.distantStaIndication.staBitMap,staIndex);
	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.powerModeIndication.staBitMap,staIndex);
	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap,staIndex);
	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.assymetricLinkIndication.staBitMap,staIndex);

	/*Calculate indications*/	
	calcProximateStaIndication();
	calcDistantStaIndication();
	calcAsymmetricLinkIndication();
	
	/*Start with power step up*/
	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.nextPowerStepIndication.staBitMap,staIndex);
	setNextPowerStep(staIndex, FALSE);

	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidth++)
	{
		
		tcrBfMode = FALSE;

		/*Calculate limits for specific bandwidth - set rate on 0 it will be calculated again when initial rate will be set*/
		PowerAdaptationCalcStaLimits(staIndex, bandwidth, antennaSelectiondBitnap,tcrBfMode);
	}
}

/**********************************************************************************

PowerSetNextProbingPointAndIndications 


Description:
------------
Function goes over all bandwidths and set limits and new probing power according to accomulate power and next probing power step
Function set indications:
	proximateStaIndication
	distantStaIndication
	assymetricLinkIndication

Input: 
-----
staIndex
stayWithLastProbingPoint - not used in this function 

Returns:
--------
	nonEffectiveLoop - if all options have been tried without change the working point return 1
	
**********************************************************************************/
void PowerSetNextProbingPointAndIndications(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	PowerAdaptationDatabase_t* pPoweradaptationDb = &(laDbDistributionParameter->laStaGroupCommon->powerAdaptationDb);
	Bandwidth_e bandwidth;
	uint8 rateIndex;
	int8 newProbingPower = 0;
	int8 newTcrPower;
	int8 tcrPowerTable[LA_NUM_OF_BANDWIDTH];
	int8 probingPower;
	uint8 antennaSelectiondBitmap, antennaSelectionGetCount;
	bool probeWithRatePlusOne;
	uint8 powerDownStep = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->powerConfigurationParams.powerStepDown;
	uint8 powerUpStep = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->powerConfigurationParams.powerStepUp;
	int8 wpPower;		
	int8 power11BCompansation;
	int8 upperLimit;
	LinkAdaptationPhyMode_e	laPhyMode;
	uint8 tcrBfMode  = FALSE;
	uint8 workingPointState;
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	antennaSelectiondBitmap = AntennaSelectionGetAntennaSelectionBitmap(laDbDistributionParameter->stationOrGroupIndex);
	UNUSED_PARAM(stayWithLastProbingPoint);
	
	/*Zero temp table*/
	memset(tcrPowerTable, 0, LA_NUM_OF_BANDWIDTH);
	
	/*Calculate indications*/
	calcProximateStaIndication();
	calcDistantStaIndication();
	calcAsymmetricLinkIndication();

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		workingPointState = laDbDistributionParameter->laStationUnique->beamFormingDb.bfWorkingPointState;
		if((workingPointState == BF_STATE_EXPLICIT)|| (workingPointState == BF_STATE_IMPLICIT))
		{
			tcrBfMode = TRUE;
		}
		else
		{
			tcrBfMode = FALSE;
		}
	}
	else
	{
		tcrBfMode = FALSE;
	}
	
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidth++)
	{
		ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
		/*Get current working point rate*/
		rateIndex = GetStaWpRateIndexFromHwTcr(laDbDistributionParameter->stationOrGroupIndex,bandwidth);
		laPhyMode = ratesTable[rateIndex].laPhyMode;
		if (laPhyMode == LA_PHY_MODE_11B)
		{
			upperLimit = RegulationLimit11b;
			antennaSelectionGetCount = AntennaSelectionGetAntennaCount(laDbDistributionParameter->stationOrGroupIndex) - 1;
			ASSERT(antennaSelectionGetCount < MAX_POSSIBLE_NUM_OF_ANTENNAS); //KW_FIX_FW_G
			power11BCompansation =  AntennaCountOffset[antennaSelectionGetCount];
		}
		else
		{
			upperLimit = PowerGlobalParameters.externalUpperLimit[bandwidth];
			power11BCompansation = 0;
		}
		/*Calculate limits for specific bandwidth*/
		PowerAdaptationCalcStaLimits(laDbDistributionParameter->stationOrGroupIndex, bandwidth, antennaSelectiondBitmap,tcrBfMode);//CALL THIS
		/*Calculate adaptive power*/
		probingPower = getStaAdaptaivePower(laDbDistributionParameter->stationOrGroupIndex, bandwidth, rateIndex,pPoweradaptationDb->accumulateDiff[bandwidth],TRUE, TRUE);
		/*Calculate new power taking into account the limits*/
		PowerAdaptationCalcPowerInStaLimits(laDbDistributionParameter->stationOrGroupIndex, bandwidth, rateIndex,probingPower, &newProbingPower);
		/*Prevent probing point to be with same power as working point*/
		wpPower = getStaAdaptaivePower(laDbDistributionParameter->stationOrGroupIndex, bandwidth, rateIndex,pPoweradaptationDb->accumulateDiff[bandwidth],FALSE, TRUE);
		PowerAdaptationCalcPowerInStaLimits(laDbDistributionParameter->stationOrGroupIndex, bandwidth, rateIndex,wpPower, &wpPower);
#ifdef POWER_ADAPTATION_LOGS	
		//ILOG0_DD("PowerSetNextProbingPointAndIndications,wpPower = %d, newProbingPower = %d", wpPower,newProbingPower);
#endif
		if (wpPower == newProbingPower) 
		{
			if(wpPower == upperLimit)
			{
				/*Decrease power down step from external up lmit*/
				newProbingPower -= powerDownStep;
			}
			else if((wpPower == (PowerGlobalParameters.externalLowerLimit[bandwidth] + power11BCompansation)) || (wpPower == pPoweradaptationDb->staInternalLowerLimit))
			{
				/*Increase power up step from external lower lmit*/
				newProbingPower += powerUpStep;
			}
			/*Limit power with upper / lower limits, in this pont power should be inside the limit unless (upper limit - lower limit) < power step up or power step down*/
			PowerAdaptationCalcPowerInStaLimits(laDbDistributionParameter->stationOrGroupIndex, bandwidth, rateIndex,newProbingPower, &newProbingPower);
		}
		/*Adjust power step to reflect newProbingPower*/
		pPoweradaptationDb->probingPowerStep[bandwidth]-= (probingPower - newProbingPower);
#ifdef POWER_ADAPTATION_LOGS		
		//ILOG0_DD("PowerSetNextProbingPointAndIndications,probingPower = %d, newProbingPower = %d", probingPower,newProbingPower );
#endif
		/*Convert power to product TCR power val*/
		newTcrPower = convertPower2TcrVal(bandwidth,newProbingPower);
		/*Update local table*/
		tcrPowerTable[bandwidth] = newTcrPower;
#ifdef POWER_ADAPTATION_LOGS
		ILOG0_DD("PowerSetNextProbingPointAndIndications, Bw %d, newPower = %d", bandwidth, newProbingPower);
#endif
	}
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, laDbDistributionParameter->stationOrGroupIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	/*Update power for all bandiwidths in TCR*/
	setPowerInStaTcr(laDbDistributionParameter->stationOrGroupIndex,tcrPowerTable,BANDWIDTH_TWENTY,LinkAdaptationCommonConfiguration.wlanBandwidthMax,TRUE, LA_PACKET_TYPE_DATA, probeWithRatePlusOne);
	/*Change RA state to wait for slow probe valid state*/
	laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_SLOW_PROBE_VALID);
	setCurrentSlowProbingTaskInDb(laDbDistributionParameter,SLOW_PROBING_POWER);
	
	
}
/**********************************************************************************

getStaAdaptaivePower 


Description:
------------
Input: 
-----

Returns:
--------
	
**********************************************************************************/

static int8 getStaAdaptaivePower(StaId staIndex, Bandwidth_e bandwidth, uint8 rateIndex, int8 accPower,bool useStepPower, bool useAdaptivePower)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	const RateObj_t* ratesTable = getRatesTable(&pLaDatabase->laStaUspCommon);
	PowerAdaptationDatabase_t* pPowerDatabase = &pLaDatabase->laStaGroupCommon.powerAdaptationDb;
	bool isVht = pLaDatabase->laStaUspCommon.staTransmissionParams.vhtSta;
	RatePowerOffset_t ratePowerOffsets = getRatePowerOffset(isVht, rateIndex, ratesTable);
	int8 powerStep = pPowerDatabase->probingPowerStep[bandwidth];
	int8 power;
	int8 proximateStaPowerOffset;
	int8 nonProximateStaPowerOffset;
	bool proximateStaIndication;
	uint8 vapId = StaDbSwEntries[staIndex].vapIndex;
	uint8 antennaSelectionGetCount;
	
	LinkAdaptationPhyMode_e	laPhyMode = ratesTable[rateIndex].laPhyMode;
	
	proximateStaIndication = LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.proximateStaIndication.staBitMap,staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	proximateStaPowerOffset = proximateStaIndication * ProximateStaBackoff[vapId];
	nonProximateStaPowerOffset = (1 - proximateStaIndication) * ratePowerOffsets.evmPowerBackoff;

	/*Take power offset only If power didn't changed from max EVM point, and if we are not in probing*/
	if ((useAdaptivePower == TRUE) && ((pPowerDatabase->isPowerChangedFromMaxEvm[bandwidth] == TRUE) || (useStepPower == TRUE)))
	{
		power = PowerGlobalParameters.refPower[bandwidth] + proximateStaPowerOffset + nonProximateStaPowerOffset + accPower + useStepPower*powerStep;
	}
	else
	{
		power = PowerGlobalParameters.refPower[bandwidth] + ratePowerOffsets.powerOffset + proximateStaPowerOffset + nonProximateStaPowerOffset + accPower + useStepPower*powerStep;
	}
	
	// Compensate 11b power, due to change in number of antennas
	if (laPhyMode == LA_PHY_MODE_11B)
	{
		antennaSelectionGetCount = AntennaSelectionGetAntennaCount(staIndex) - 1;
		ASSERT(antennaSelectionGetCount < MAX_POSSIBLE_NUM_OF_ANTENNAS); //KW_FIX_FW_G
		power += AntennaCountOffset[antennaSelectionGetCount];
	}

	return power;
}


    
/***********************************************************************
* PowerAdaptationAdjustAdditionalSteps
* 
* Description:
* ------------
* This function reset the Additional Steps back to their default values.
* The function is called when the working point power changes. 
*
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PowerAdaptationAdjustAdditionalSteps(StaId staIndex, PowerAdaptationDatabase_t* pPowerAdaptationDb, Bandwidth_e bandwidth, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	UNUSED_PARAM(bandwidth);
	#ifdef POWER_ADAPTATION_LOGS
	PowerNextProbingPoint_e					nextPowerProbingPoint;
	LinkAdaptationStaDatabase_t				*pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	PowerAdaptationDatabase_t				*pPowerDatabase = &pLaDatabase->laStaGroupCommon.powerAdaptationDb;
	#endif
	LinkAdaptationConfigurationParams_t		*pLaConfigurationParams = LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.pLinkAdaptationConfigurationParams;	
	UNUSED_PARAM(laDbDistributionParameter);
	
#ifdef POWER_ADAPTATION_LOGS
	nextPowerProbingPoint = (PowerNextProbingPoint_e)LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.nextPowerStepIndication.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
#endif
	/* The bit in the bitmap indicates the next probing to be done (if it indicates DOWN it means we probed UP). 
	   The probing that we performed now is held at pPowerAdaptationDb->probingPowerStep */

	/* Set the Step UP to default */
	pPowerAdaptationDb->powerAdditionalStepUp = pLaConfigurationParams->powerConfigurationParams.powerStepUpDefault; 
	/* Set the step DOWN to default */
	pPowerAdaptationDb->powerAdditionalStepDown = pLaConfigurationParams->powerConfigurationParams.powerStepDownDefault;

	#ifdef POWER_ADAPTATION_LOGS
	ILOG0_DDDD("PowerAdaptationAdjustAdditionalSteps nextPowerProbingPoint = %d probingPowerStep = %d powerAdditionalStepUp = %d powerAdditionalStepDown = %d", nextPowerProbingPoint, pPowerDatabase->probingPowerStep[bandwidth], pPowerAdaptationDb->powerAdditionalStepUp, pPowerAdaptationDb->powerAdditionalStepDown);
	#endif

}

/**********************************************************************************

PowerProcessFeedback 


Description:
------------
Function compare probing power point PER result to working point PER result for specific bandwidth and decide whethet to change power workingp point

Input: 
-----
staindex

Returns:
--------
	void - 
	
**********************************************************************************/
void PowerProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t *pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	PowerAdaptationDatabase_t* pPowerAdaptationDb = &(laDbDistributionParameter->laStaGroupCommon->powerAdaptationDb);
	Bandwidth_e mainBw;
	SlowProbingPointEstimators_t* pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);
	FastProbingPointEstimators_t* pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter);
	int8 workingPointPer;
	uint8 slowProbingPointPer;
	int8 newTcrPowerValTable[LA_NUM_OF_BANDWIDTH];
	uint8 mainBwRateIndex;
	uint8 slowProbingRateIndex;
	int8 power;
	int8 powerInlimits = 0;//KW_FW_FIX_G
	bool probeWithRatePlusOne; 
	uint16 losslessTpWp;
	uint16 losslessTpPp;
	uint16 effectiveTpWp;
	uint16 effectiveTpProbing;
	Bandwidth_e bandwidthIterator;
	int8 accumulatedDiffTemp = 0;
	uint8 rateIndexPerBw;
	bool powerBlockedBylimits;
	StaId staIndex;
	int8 wpPower;
	int8 wpPowerLimited;
	int8 ppPower;
	int8 ppPowerLimited;
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);	

	mainBw = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, isHeGroup);

	ASSERT(mainBw < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G

	if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
	{
		/* SLOW probing is not implemented for MU */
		disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_POWER);
		return; 
	}
	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_POWER);

	staIndex= laDbDistributionParameter->stationOrGroupIndex; 
	memset(newTcrPowerValTable, 0, LA_NUM_OF_BANDWIDTH);
	probeWithRatePlusOne = LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, staIndex);
	slowProbingRateIndex = estimatorsGetSlowProbingRatePlusOne(laDbDistributionParameter);
	/*Get current rate index*/
	mainBwRateIndex = GetStaWpRateIndexFromHwTcr(staIndex,mainBw);
	/*Get working point power*/
	wpPower = getStaAdaptaivePower(staIndex,mainBw,mainBwRateIndex,pPowerAdaptationDb->accumulateDiff[mainBw],FALSE, TRUE);
	/*In order to have the real wp we should limit it*/
	PowerAdaptationCalcPowerInStaLimits(staIndex, mainBw, mainBwRateIndex, wpPower, &wpPowerLimited);
	/*Get current probing point*/
	ppPower = getStaAdaptaivePower(staIndex,mainBw,mainBwRateIndex,pPowerAdaptationDb->accumulateDiff[mainBw],TRUE, TRUE);
	/*In order to have the real pp we should limit it*/
	PowerAdaptationCalcPowerInStaLimits(staIndex, mainBw, mainBwRateIndex, ppPower, &ppPowerLimited);
	
	/*if WP and PP filters not valid or working point power == probing point power there is no need to check PER result */
	if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, pLaConfigParams->maxValidFilterTsfDiff) &&
		AlphaFilter_isFilterResultValid(&pWorkingPointEst->averagePer, pLaConfigParams->maxValidFilterTsfDiff) && 
		wpPowerLimited != ppPowerLimited)
	{

		workingPointPer = AlphaFilter_GetFilterResult(&pWorkingPointEst->averagePer);
		slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);
		workingPointPer = MAX(workingPointPer - WP_PER_BIAS, 0);
#ifdef POWER_ADAPTATION_LOGS
		ILOG0_DD("PowerProcessFeedback, workingPointPer = %d, slowProbingPointPer = %d", workingPointPer, slowProbingPointPer);
#endif

		if (probeWithRatePlusOne == FALSE)
		{
			/*Instead of calculate the effective TP, calculate the success ratio instead of PER, so the comparition later will be correct*/
			effectiveTpWp = HUNDRED_PERCENT - workingPointPer;
			effectiveTpProbing = HUNDRED_PERCENT - slowProbingPointPer ;

		}
		else
		{
			/*Extract lossless TP according to bandwidth*/
			losslessTpWp = ratesTable[mainBwRateIndex].rateTableBwParmas[mainBw].shortCpPhyRate;
			losslessTpPp = ratesTable[slowProbingRateIndex].rateTableBwParmas[mainBw].shortCpPhyRate;
			/*Calc effective TP for working point and probing point*/
			effectiveTpWp = estimatorsCalcEffectiveTp(losslessTpWp,workingPointPer);
			effectiveTpProbing = estimatorsCalcEffectiveTp(losslessTpPp,slowProbingPointPer);
		}
		/*Probing per was better than WP per, and probing step != 0*/
		if (effectiveTpProbing > effectiveTpWp)
		{
			bool alreadyAdjusted = FALSE;
			for (bandwidthIterator = BANDWIDTH_TWENTY; bandwidthIterator <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidthIterator++)
			{
				ASSERT(bandwidthIterator < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
				rateIndexPerBw = GetStaWpRateIndexFromHwTcr(staIndex,bandwidthIterator);

				/*Update accomulate power for all BWs*/
				accumulatedDiffTemp = pPowerAdaptationDb->accumulateDiff[bandwidthIterator];
				accumulatedDiffTemp += pPowerAdaptationDb->probingPowerStep[bandwidthIterator];
				/*Calculate new power = accomulate power + power probing step*/
				power = getStaAdaptaivePower(staIndex, bandwidthIterator, rateIndexPerBw, pPowerAdaptationDb->accumulateDiff[bandwidthIterator], TRUE, TRUE);
				/*Calculate new power taking into account the limits*/
				powerBlockedBylimits = PowerAdaptationCalcPowerInStaLimits(staIndex, bandwidthIterator, rateIndexPerBw, power, &powerInlimits);
				/* If power was not blocked by limit: store acc diif, indicate that we moved from the MAX EVM point, reset the Additional step that got us here and adjust the opposite Additional step */
				if (powerBlockedBylimits != TRUE) 
				{
					pPowerAdaptationDb->accumulateDiff[bandwidthIterator] = accumulatedDiffTemp;
					pPowerAdaptationDb->isPowerChangedFromMaxEvm[bandwidthIterator] = TRUE;
					/* Adjust the additional steps as we have changed the working power (call it only once) */
					if (alreadyAdjusted == FALSE)
					{
						alreadyAdjusted = TRUE;
						PowerAdaptationAdjustAdditionalSteps(staIndex, pPowerAdaptationDb, mainBw, laDbDistributionParameter);
					}
				}
				/*Convert power val to TCR product val */
				newTcrPowerValTable[bandwidthIterator] = convertPower2TcrVal(bandwidthIterator,powerInlimits);
			}
			
			/*Set next probing point*/
			setNextPowerStep(staIndex, FALSE);
			/*Set working point power in TCR*/
			setPowerInStaTcr(staIndex, newTcrPowerValTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE,LA_PACKET_TYPE_DATA, FALSE);
			/*If working point changed, reset non effective loop indication */
			/*Reset thresholds*/
			rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
			/*Reset estimators*/
			estimatorsResetRatesEstimators(laDbDistributionParameter);
			/*Reset fast probing counter so fast probing can collect statistics before next fast probing event*/	
			resetFastProbingCounters(laDbDistributionParameter);
			changeTaskToDefaultPriority(laDbDistributionParameter, SLOW_PROBING_POWER);
			resetNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_POWER);

#ifdef POWER_ADAPTATION_LOGS
			ILOG0_D("PowerProcessFeedback: Change power working point newPower = %d", powerInlimits);
#endif

		}
		else
		{
			/*Probing point did not improve the TP*/
			if ((probeWithRatePlusOne == FALSE) &&
				(workingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh) &&
				(slowProbingPointPer < pLaConfigParams->slowProbingWithHigherRateLowPerTh))
			{
				/*If WP PER is lower than threshold, try to probe with higher rate in the next probing cycle*/
				LA_SET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
#ifdef POWER_ADAPTATION_LOGS
//				ILOG0_V("PowerProcessFeedback - SET RATE + 1");
#endif
			}
			else
			{
				/*If WP does not improve PER change to opposite step*/
				setNextPowerStep(staIndex, TRUE);
				/*Clear probe with rate plus one indication*/
				LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, staIndex);
				/*Clear slowProbingWithHigherRate for the next cycle*/
				LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, staIndex);
				updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_POWER);
			}
			
		}
	}
	else
	{
		/*When filter is not valid (can be if all aggregation failed), change to next probing point and clear probe with rate + 1 indication*/
		setNextPowerStep(staIndex, TRUE);
		/*Clear probe with rate plus one indication*/
		LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.probeWithRatePlus1.staBitMap, staIndex);
	}
	return;
}

/**********************************************************************************

setNextPowerStep 


Description:
------------
Change next probing state:
POWER_PROBE_UP -> POWER_PROBE_DOWN
POWER_PROBE_DOWN -> POWER_PROBE_UP
If state canged from last state (PROBING_POWER UP -> PROBING_POWER_DOWN), it meensthat we try two options without changing the working point, so set nonEffectiveLoop indication

Input: 
-----
StaIndex, bandwidth
Returns:
--------
	void - 
	
**********************************************************************************/

static void setNextPowerStep(StaId staIndex, bool increaseStep)
{
	LinkAdaptationConfigurationParams_t* pLaConfigurationParams = LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.pLinkAdaptationConfigurationParams;
	PowerAdaptationDatabase_t* pPowerAdaptationDb = &LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.powerAdaptationDb;
	PowerNextProbingPoint_e nextPowerProbingPoint;
	Bandwidth_e bandwidthIterator;
	uint8 powerAdditionalPowerStep = pLaConfigurationParams->powerConfigurationParams.powerAdditionalPowerStep;
	uint8 powerMaxAdditionalPowerStep = pLaConfigurationParams->powerConfigurationParams.powerMaxAdditionalPowerStep;
	uint8 minAbsPowerStep = MAX_UINT8;
	int8 powerStepTemp = 0;//KW_FW_FIX_G

	nextPowerProbingPoint = (PowerNextProbingPoint_e)LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.nextPowerStepIndication.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check	
	
	/*Toggle state by toggle bit in powerAdaptationBitmaskDb.nextPowerStepIndication*/
	if (nextPowerProbingPoint == POWER_PROBE_UP) /*Next probing is power up -> current is down*/
	{
		/*Set next probing step*/
		for (bandwidthIterator = BANDWIDTH_TWENTY; bandwidthIterator <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidthIterator++)
		{
			ASSERT(bandwidthIterator < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
			/*Search minimum power diff out of all BWs*/
			if (minAbsPowerStep > ABS(pPowerAdaptationDb->probingPowerStep[bandwidthIterator]))
			{
				minAbsPowerStep = ABS(pPowerAdaptationDb->probingPowerStep[bandwidthIterator]);
				powerStepTemp = pPowerAdaptationDb->probingPowerStep[bandwidthIterator];
			}
			pPowerAdaptationDb->probingPowerStep[bandwidthIterator] = pPowerAdaptationDb->powerAdditionalStepUp;
		}
		//ILOG0_D("powerStepTemp %d",powerStepTemp);
		if ((increaseStep == TRUE) && (powerStepTemp <= 0))
		{
			increasePowerStep(&pPowerAdaptationDb->powerAdditionalStepDown, pLaConfigurationParams->powerConfigurationParams.powerStepDownDefault, minAbsPowerStep,powerAdditionalPowerStep,powerMaxAdditionalPowerStep);
		}
		else
		{
			/*Go back to default power step value*/
			pPowerAdaptationDb->powerAdditionalStepDown = pLaConfigurationParams->powerConfigurationParams.powerStepDownDefault;
		}
		/*Set probing down indication in bitmap*/
		LA_SET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.nextPowerStepIndication.staBitMap, staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
	}
	else /*Next probing is probe down -> current is up*/
	{
		/*Set next probing step*/
		for (bandwidthIterator = BANDWIDTH_TWENTY; bandwidthIterator <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidthIterator++)
		{
			ASSERT(bandwidthIterator < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
			if (minAbsPowerStep > ABS(pPowerAdaptationDb->probingPowerStep[bandwidthIterator]))
			{
				minAbsPowerStep = ABS(pPowerAdaptationDb->probingPowerStep[bandwidthIterator]);
				powerStepTemp = pPowerAdaptationDb->probingPowerStep[bandwidthIterator];
			}

			pPowerAdaptationDb->probingPowerStep[bandwidthIterator] = -(pPowerAdaptationDb->powerAdditionalStepDown);

		}
		//ILOG0_D("powerStepTemp %d",powerStepTemp);
		if ((increaseStep == TRUE) && (powerStepTemp >= 0))
		{
			increasePowerStep(&pPowerAdaptationDb->powerAdditionalStepUp, pLaConfigurationParams->powerConfigurationParams.powerStepUpDefault, minAbsPowerStep,powerAdditionalPowerStep,powerMaxAdditionalPowerStep);
		}
		else
		{
			/*Go back to default power step value*/
			pPowerAdaptationDb->powerAdditionalStepUp = pLaConfigurationParams->powerConfigurationParams.powerStepUpDefault;
		}
		/*Set probing up indication in bitmap*/
		LA_CLR_BIT_IN_BITMAP(powerAdaptationBitmaskDb.nextPowerStepIndication.staBitMap, staIndex);
	
	}
	
}

static void increasePowerStep(uint8* pAdditionalStep, uint8 powerStepDefault, uint8 minAbsPowerStep, uint8 configPowerAdditionalStep,uint8 powerMaxAdditionalPowerStep)
{
	uint8 nextStepTemp;
	//ILOG0_DD("*pAdditionalStep %d minAbsPowerStep %d", *pAdditionalStep, minAbsPowerStep);
	/*If powerAdditionalStep step increased because it was below lower limit / above upper limit => adjust it to new step, otherwise increment with powerAdditionalPowerStep*/
	if (*pAdditionalStep < minAbsPowerStep)
	{
		nextStepTemp = minAbsPowerStep + configPowerAdditionalStep;
	}
	else if (*pAdditionalStep > minAbsPowerStep)
	{
		/*powerAdditionalStep crossed upper / lower limit and decreased, go back to default power step value*/
		nextStepTemp = powerMaxAdditionalPowerStep + configPowerAdditionalStep; // so it will go back to limit in next if statement
	}
	else
	{
		/*powerAdditionalStep is equal to power step*/
		nextStepTemp = *pAdditionalStep + configPowerAdditionalStep;
	}
	//ILOG0_D("nextStepTemp %d",nextStepTemp);
	if (nextStepTemp <= powerMaxAdditionalPowerStep)
	{
		
		/*Increase power probing step*/
		*pAdditionalStep = nextStepTemp;
	
	}
	else
	{
		/*Go back to default power step value*/
		*pAdditionalStep = powerStepDefault;
	}
	//ILOG0_D("*pAdditionalStep %d",*pAdditionalStep);
}
/**********************************************************************************

CalcLimitsAndGetStaTcrPowerVal


Description:
------------
Function returns the power TCR val according to rate take into account the limits

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 CalcLimitsAndGetStaTcrPowerVal(StaId staIndex, Bandwidth_e bandwidth, uint8 rateIndex, bool addAccPower, LaPacketType_e packetType, uint8 modifiedAntennaSelectionBitmap, bool useAdaptivePower,uint8 isBFExist)
{
	uint8 newTcrPowerVal = 0;
	uint8 antennaCount = AntennaSelectionGetAntennaCount(staIndex);
	int8 antennaPowerVal;
	uint8 avgAntennaBoost;
	int8 power = 0;
	int8 newPower = 0;
	PowerAdaptationDatabase_t* pPowerAdaptationDb = &(LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.powerAdaptationDb);
	
	if ((FixedPowerParameters.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) ||
			(FixedPowerParameters.changeType == packetType))
	{
		/*Fixed Power*/
		modifiedAntennaSelectionBitmap = AntennaSelectionGetAntennaSelectionBitmap(staIndex);
		avgAntennaBoost = calcAvgAntennaBoost(modifiedAntennaSelectionBitmap, antPowerBoost);
		antennaPowerVal = MAX(FixedPowerParameters.powerVal - AntennaCountOffset[antennaCount-1] - avgAntennaBoost, 0);
		newTcrPowerVal = convertPower2TcrVal(bandwidth,antennaPowerVal);
	}
	else 
	{
		/*Adaptive power*/
		/*-------------*/
		if(modifiedAntennaSelectionBitmap != AntennaSelectionGetAntennaSelectionBitmap(staIndex))
		{
			/*Antenna selection has been modified*/
			/*Calc all limits again*/
			PowerAdaptationCalcStaLimits(staIndex, bandwidth, modifiedAntennaSelectionBitmap,isBFExist);//get in this function
		}
		/*Calculate power with take into account the limits*/
		power = getStaAdaptaivePower(staIndex, bandwidth, rateIndex,(addAccPower*pPowerAdaptationDb->accumulateDiff[bandwidth]),FALSE, useAdaptivePower);
		/*Set power in limits*/
		PowerAdaptationCalcPowerInStaLimits(staIndex, bandwidth, rateIndex, power, &newPower);
		/*Convert power val to product TCR val*/
		newTcrPowerVal = convertPower2TcrVal(bandwidth,newPower);
#ifdef LINK_ADAPTATION_LOGS
		ILOG0_DDDD("CalcLimitsAndGetStaTcrPowerVal, Bw = %d, rateIndex = %d, powerVal = %d, newPower = %d", bandwidth, rateIndex, power, newPower);
		ILOG0_D("CalcLimitsAndGetStaTcrPowerVal, newTcrPowerVal = %d", newTcrPowerVal);
#endif
	}
	return newTcrPowerVal;
}
/**********************************************************************************

GetTcrDefaultPowerVal 


Description:
------------
Function returns the default power without power adaptation adaptation

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 GetTcrDefaultPowerVal(bool isVht, uint8 rateIndex, Bandwidth_e bandwidth)
{
	RatePowerOffset_t ratePowerOffsets;
	int8 power;
	int8 tcrPowerVal = 0;
	LinkAdaptationPhyMode_e LaPhyMode = htVhtRatesTable[rateIndex].laPhyMode; 
	int8 defaultRegulationLimit;
	const RateObj_t* ratesTable = NULL;

	ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH);
#ifdef ENET_INC_ARCH_WAVE600
	if (FixedRateParameters.phyMode >= PHY_MODE_11AX_SU)
	{
		ratesTable = heRatesTable;		
	}
	else
#endif
	{
			ratesTable = htVhtRatesTable;		
	}

	defaultRegulationLimit = DefaultRegulationLimitPerAnt[bandwidth];
	
	/*Get power rate offset*/
	ratePowerOffsets = getRatePowerOffset(isVht, rateIndex, ratesTable);

	
	/*Calculate default power val*/
	if (LaPhyMode == LA_PHY_MODE_11B)
	{
		uint8 txAntMask = Hdk_GetTxAntMask();
		uint8 numOfAnts = Utils_GetNumAntsFromMask(txAntMask);
		ASSERT(numOfAnts <= MAX_POSSIBLE_NUM_OF_ANTENNAS);
		
		defaultRegulationLimit = RegulationLimit11b;
		power = ratePowerOffsets.powerOffset + PowerGlobalParameters.refPower[bandwidth] + AntennaCountOffset[numOfAnts - 1];
	}
	else
	{
		power = ratePowerOffsets.powerOffset + PowerGlobalParameters.refPower[bandwidth];
	}
	power = MAX(power, DefaultInternalLowerLimit);
	power = MIN(power, defaultRegulationLimit);
	power = MAX(power, TPC_LOW_LIMIT);
	power = MIN(power, MAX_TPC_POWER_LEVELS-1);
	
	/*Convert power to product TCR power val*/
	tcrPowerVal = convertPower2TcrVal(bandwidth,power);

	return tcrPowerVal;
}

/**********************************************************************************

CalcLimitsAndSetVapPower 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void CalcLimitsAndSetVapPower(uint8 vapIndex)
{
	int8 vapTcrDataPowerValTable[LA_NUM_OF_BANDWIDTH];
	int8 vapTcrManagementPowerValTable[LA_NUM_OF_BANDWIDTH];
	uint8 vapDataRateindx;
	uint8 vapManagementRateindx;
	Bandwidth_e bandwidth;
	
	/* The next 2 function must be in this order since CalculateAutoReplyPower is using values that were calculated by CalcDefaultPowerLimits*/
	CalcDefaultPowerLimits();
	CalculateAutoReplyPower();

	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= LinkAdaptationCommonConfiguration.wlanBandwidthMax;  bandwidth++)
	{
		ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
		vapDataRateindx = GetVapRateIndex(vapIndex, LA_PACKET_TYPE_DATA, bandwidth);
		vapTcrDataPowerValTable[bandwidth] = GetTcrDefaultPowerVal(0, vapDataRateindx, bandwidth); 
		vapManagementRateindx = GetVapRateIndex(vapIndex, LA_PACKET_TYPE_MANAGEMENT, bandwidth);
		vapTcrManagementPowerValTable[bandwidth] = GetTcrDefaultPowerVal(0, vapManagementRateindx, bandwidth); 

	}

	SetPowerInVapTcr(vapIndex, vapTcrDataPowerValTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE, LA_PACKET_TYPE_DATA);
	SetPowerInVapTcr(vapIndex, vapTcrManagementPowerValTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE, LA_PACKET_TYPE_MANAGEMENT);
}

/**********************************************************************************

convertPower2TcrVal 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

int8 convertPower2TcrVal (Bandwidth_e bandwidth, int8 powerVal)
{
	UNUSED_PARAM(bandwidth);	
	return powerVal;
}

#if 0 
/**********************************************************************************

PowerAdaptationResetDiffPower 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void PowerAdaptationResetDiffPower(StaId staIndex)
{
	PowerAdaptationDatabase_t* pPowerAdaptationDb = &LinkAdaptationStaDatabase[staIndex].powerAdaptationDb;
	Bandwidth_e bandwidth; 

	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= LinkAdaptationCommonConfiguration.wlanBandwidthMax; bandwidth++)
	{
		pPowerAdaptationDb->accumulateDiff[bandwidth] = 0;
		memset(&powerAdaptationBitmaskDb.nextPowerStepIndication[bandwidth], 0, sizeof(StationBitmap_t));
	}

}

#endif 


/**********************************************************************************

setPowerInStaTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static bool setPowerInStaTcr(StaId staIndex, int8* pPowerValTable, Bandwidth_e firstBwtoUpdate, Bandwidth_e lastBwToUpdate, bool changeProbingPoint, LaPacketType_e packetType, bool setRatePlusOne)
{

	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	LaTcrModificationStruct_t tcrModificationParams;
	Bandwidth_e bandwidth;
	bool ratePlusOneChanged;
	//uint8 modifiedAntennaSelectionBitmap = GetAntennaSelectionBitmap(staIndex);
	
	memset(&tcrModificationParams,0,sizeof(LaTcrModificationStruct_t));

	/*Fill control parameters of tcrModificationParams*/
	tcrModificationParams.controlParams.firstBwToUpdate = firstBwtoUpdate;
	tcrModificationParams.controlParams.lastBwToUpdate= lastBwToUpdate;
	tcrModificationParams.controlParams.isVhtSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.vhtSta;
	tcrModificationParams.controlParams.changeProbingPoint=  changeProbingPoint;
	tcrModificationParams.controlParams.slowProbing =  changeProbingPoint;
	tcrModificationParams.controlParams.packetType = packetType;
	tcrModificationParams.controlParams.changeToRatePlusOne = setRatePlusOne;
	tcrModificationParams.controlParams.powerChanged = TRUE;
	tcrModificationParams.controlParams.staIndex = staIndex;
	tcrModificationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX;
#ifdef ENET_INC_ARCH_WAVE600
	tcrModificationParams.controlParams.isHeGroup = FALSE;
#endif
	for (bandwidth = firstBwtoUpdate; bandwidth<=lastBwToUpdate; bandwidth++)
	{
		tcrModificationParams.tcrParams.bwDependedTcrValsTable[bandwidth].tcrPower = pPowerValTable[bandwidth];
	}
	/*Set power and effected params in TCRs*/
	ratePlusOneChanged = modifyStaTcrsParamsReq(&tcrModificationParams,powerAdaptationModificationFunc); 	

	return ratePlusOneChanged;
}

/**********************************************************************************

setFixedPowerInStaTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void setFixedPowerInStaTcr(StaId staIndex)
{
	int8 antennaPowerVal;
	uint8 avgAntennaBoost;
	uint8 antennaCount;
	int8 pPowerValTable[LA_NUM_OF_BANDWIDTH];
	Bandwidth_e bandwidthIterator;
	LaPacketType_e changeType = FixedPowerParameters.changeType;

	avgAntennaBoost = calcAvgAntennaBoost(AntennaSelectionGetAntennaSelectionBitmap(staIndex), antPowerBoost);
	antennaCount = AntennaSelectionGetAntennaCount(staIndex);
	antennaPowerVal = MAX(FixedPowerParameters.powerVal - AntennaCountOffset[antennaCount-1] - avgAntennaBoost, TPC_LOW_LIMIT);
	
	for (bandwidthIterator = BANDWIDTH_TWENTY; bandwidthIterator <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax);  bandwidthIterator++)
	{
		ASSERT(bandwidthIterator < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
		pPowerValTable[bandwidthIterator] = convertPower2TcrVal(bandwidthIterator,antennaPowerVal);
	}
	
	if (changeType == LA_PACKET_TYPE_DATA_MANAGEMENT)
	{
		setPowerInStaTcr(staIndex, pPowerValTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE, LA_PACKET_TYPE_DATA, FALSE);	
		setPowerInStaTcr(staIndex, pPowerValTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE, LA_PACKET_TYPE_MANAGEMENT, FALSE);
	}
	else
	{
		setPowerInStaTcr(staIndex, pPowerValTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE, changeType, FALSE);
	}
}
/**********************************************************************************

powerAdaptationModificationFunc 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void powerAdaptationModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	Bandwidth_e bandwidth;

	ASSERT(pModifcationParamsIn->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	for(bandwidth = pModifcationParamsIn->controlParams.firstBwToUpdate; bandwidth <=pModifcationParamsIn->controlParams.lastBwToUpdate; bandwidth++)
	{
		pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].tcrPower = pModifcationParamsIn->tcrParams.bwDependedTcrValsTable[bandwidth].tcrPower;

	}
	
}



/**********************************************************************************

setPowerDataFromTpc 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
 	
**********************************************************************************/


void UpdateRatePowerOffset(uint8 freqIndex, int8 minOnLowerUpperDiffPowerTable)
{
	uint8 nss;
	
	if ( 0 == flagPowerOffsetHighLowfrequency)
	{
		RatePowerOffset11B	= &RatePowerOffsetHigh11B;
		RatePowerOffset11AG = RatePowerOffsetHigh11AG;
		RatePowerOffset11N	= RatePowerOffsetHigh11N;
		RatePowerOffset11AC = RatePowerOffsetHigh11AC;
		RatePowerOffsetMU11AC = RatePowerOffsetMUHigh11AC;
		InitRatePowerOffset11AcMcs89 = InitRatePowerOffset11AcHighMcs89;
	}
	else if (1 == flagPowerOffsetHighLowfrequency)
	{
		int i = 0;
		//PowerOffsetHighLowfrequency[i].frequency;
		while (freqIndex > PowerOffsetHighLowfrequency[i].frequency)
		{
			i++;
		}
		if (1 == PowerOffsetHighLowfrequency[i].highLow)
		{
			RatePowerOffset11B  = &RatePowerOffsetHigh11B;
			RatePowerOffset11AG = RatePowerOffsetHigh11AG;
			RatePowerOffset11N  = RatePowerOffsetHigh11N;
			RatePowerOffset11AC = RatePowerOffsetHigh11AC;
			RatePowerOffsetMU11AC = RatePowerOffsetMUHigh11AC;
			InitRatePowerOffset11AcMcs89 = InitRatePowerOffset11AcHighMcs89;
		}
		else if (0 == PowerOffsetHighLowfrequency[i].highLow)
		{
			RatePowerOffset11B  = &RatePowerOffsetLow11B;
			RatePowerOffset11AG = RatePowerOffsetLow11AG;
			RatePowerOffset11N  = RatePowerOffsetLow11N;
			RatePowerOffset11AC = RatePowerOffsetLow11AC;
			RatePowerOffsetMU11AC = RatePowerOffsetMULow11AC;
			InitRatePowerOffset11AcMcs89 = InitRatePowerOffset11AcLowMcs89;
		}
	}
	for(nss = 0; nss < VHT_HE_NUMBER_OF_NSS ; nss++)
	{
		RatePowerOffset11AC[MCS8_256QAM_3_4][nss].powerOffset = InitRatePowerOffset11AcMcs89[0][nss] + minOnLowerUpperDiffPowerTable;
		RatePowerOffset11AC[MCS9_256QAM_5_6][nss].powerOffset = InitRatePowerOffset11AcMcs89[1][nss] + minOnLowerUpperDiffPowerTable;
#ifdef ENET_INC_ARCH_WAVE600
		RatePowerOffset11AX[MCS8_256QAM_3_4][nss].powerOffset = InitRatePowerOffset11Ax[0][nss] + minOnLowerUpperDiffPowerTable;
		RatePowerOffset11AX[MCS9_256QAM_5_6][nss].powerOffset = InitRatePowerOffset11Ax[1][nss] + minOnLowerUpperDiffPowerTable;		
		RatePowerOffset11AX[MCS10_1024QAM_3_4][nss].powerOffset = InitRatePowerOffset11Ax[2][nss] + minOnLowerUpperDiffPowerTable;
		RatePowerOffset11AX[MCS11_1024QAM_5_6][nss].powerOffset = InitRatePowerOffset11Ax[3][nss] + minOnLowerUpperDiffPowerTable;
#endif
	}
}

void SetPowerDataFromTpc(LA_SET_CHANNEL_DATA_t* laSetChannelMsg)
{
	int8 minPowerFromInternalUpperLimit[LA_NUM_OF_BANDWIDTH];
	int8 LowerUpperDiffPowerVal[LA_NUM_OF_BANDWIDTH];
	int8 internalPowerUpperLimit[MAX_NUM_OF_ANTENNAS][LA_NUM_OF_BANDWIDTH];
	int8 minOnLowerUpperDiffPowerTable;
	Bandwidth_e bandwidth;
#ifdef ENET_INC_ARCH_WAVE600
	BandId_e myBand = ConfigurationManager_GetMyBand();
#endif //ENET_INC_ARCH_WAVE600
	int8 tpcInternalPowerLowerLimit[MAX_NUM_OF_ANTENNAS][LA_NUM_OF_BANDWIDTH];
	LA_SET_TPC_DATA_t* pTpcPowerData = &laSetChannelMsg->tpcInfo;

	SLOG0(0, 0, LA_SET_TPC_DATA_t, pTpcPowerData);

	MEMCPY(internalPowerUpperLimit, pTpcPowerData->internalPowerUpperLimit, MAX_NUM_OF_ANTENNAS*LA_NUM_OF_BANDWIDTH);
	MEMCPY(RegulationLimit, pTpcPowerData->regulationLimit, LA_NUM_OF_BANDWIDTH);
	MEMCPY(PowerLimits.muMimoLimit, pTpcPowerData->regulationLimitMU, LA_NUM_OF_BANDWIDTH);
	MEMCPY(PowerLimits.beamFormingLimit, pTpcPowerData->regulationLimitBF, LA_NUM_OF_BANDWIDTH);
	MEMCPY(tpcInternalPowerLowerLimit, pTpcPowerData->internalPowerLowerLimit, MAX_NUM_OF_ANTENNAS*LA_NUM_OF_BANDWIDTH);
	RegulationLimit11b = pTpcPowerData->txPower11B;
//	memset(&RatePowerOffset11B,0,sizeof(RatePowerOffset_t));
	/*Limit RegulationLimit11b only after SetPowerLimit so the power limit will take txPower11B as input */
	SetPowerLimit();
	RegulationLimit11b = MIN(RegulationLimit11b, (MAX_TPC_POWER_LEVELS-1));

	/*Update internal module arrays according to TPC data*/
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax); bandwidth++)
	{
		ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
		minPowerFromInternalUpperLimit[bandwidth] = searchMinPowerOverAntAtSpecificBw(internalPowerUpperLimit, laSetChannelMsg->antennaConnectedMask, bandwidth);
		minPowerFromInternalLowerLimit[bandwidth] = searchMinPowerOverAntAtSpecificBw(tpcInternalPowerLowerLimit, laSetChannelMsg->antennaConnectedMask, bandwidth);
		LowerUpperDiffPowerVal[bandwidth] = minPowerFromInternalLowerLimit[bandwidth] - minPowerFromInternalUpperLimit[bandwidth] ;
		DEBUG_ASSERT(LowerUpperDiffPowerVal[bandwidth] <= 0);
		PowerGlobalParameters.refPower[bandwidth] = minPowerFromInternalUpperLimit[bandwidth];

		// update min and max power internal limits in LA Statistics
#ifdef ENET_INC_ARCH_WAVE600
		LinkAdaptationStatistics.minPower[myBand][bandwidth] = minPowerFromInternalLowerLimit[bandwidth]; //LINK_ADAPTATION_MIN_POWER
		LinkAdaptationStatistics.maxPower[myBand][bandwidth] = minPowerFromInternalUpperLimit[bandwidth]; //LINK_ADAPTATION_MAX_POWER
#else
		LinkAdaptationStatistics.minPower[bandwidth] = minPowerFromInternalLowerLimit[bandwidth]; //LINK_ADAPTATION_MIN_POWER
		LinkAdaptationStatistics.maxPower[bandwidth] = minPowerFromInternalUpperLimit[bandwidth]; //LINK_ADAPTATION_MAX_POWER
#endif //ENET_INC_ARCH_WAVE600
	}

	/*256 QAM has offset refered to min power table, in order to have one offsets table we take into account the diff between tables*/
	minOnLowerUpperDiffPowerTable = searchMinOverTable(LowerUpperDiffPowerVal,(LinkAdaptationCommonConfiguration.wlanBandwidthMax + 1));
	UpdateRatePowerOffset(laSetChannelMsg->frequency_index, minOnLowerUpperDiffPowerTable);
	
	extractAndSetAntBoosts(antPowerBoost, pTpcPowerData);
}

/**********************************************************************************

powerAdaptationGetultimateEvm 


Description:
------------

Input: 
-----
BW

Returns:
--------
	Ultimate EVM per BW
	
**********************************************************************************/
int8 powerAdaptationGetultimateEvm(Bandwidth_e bandwidth)
{
	return minPowerFromInternalLowerLimit[bandwidth];
}

int8 getRatePowerOffsetForGroupMu(uint32 numberOfNss, uint8 rateindex)
{
	return RatePowerOffsetMU11AC[rateindex][numberOfNss].powerOffset;
}

int8 isMuPowerOffsetTableUsed(void)
{
	return flagPowerOffsetHighLowfrequency;
}

int8 getMuPowerLimit(Bandwidth_e bandwidth)
{
	return MIN(PowerLimits.muMimoLimit[bandwidth],DefaultRegulationLimitPerAnt[bandwidth]);
}

int8 getBfPowerLimit(Bandwidth_e bandwidth)
{
	return PowerLimits.beamFormingLimit[bandwidth];
}

int8 getRegLimitPerAnt(Bandwidth_e bandwidth)
{
	return DefaultRegulationLimitPerAnt[bandwidth];
}

/**********************************************************************************

getRateOffset 


Description:
------------
Function returns rape power offset acording to phy mode and rate link adaptation rate index

Input: 
-----
isVht = station is vht station or not
rateindex = link adaptation rate index

Returns:
--------
	RatePowerOffset_t power - struct contains power offset and additional power backoff
	
**********************************************************************************/

static RatePowerOffset_t getRatePowerOffset(bool isVht, uint8 rateindex, const RateObj_t* ratesTable)
{
	RatePowerOffset_t power = {0};
	LinkAdaptationPhyMode_e LaPhyMode = ratesTable[rateindex].laPhyMode; 

#ifdef QAM_PLUS_DEBUG
//		ILOG0_D("[getRatePowerOffset] RatesTable[rateindex].vhtHeMcs = %d",ratesTable[rateindex].vhtHeMcs);
#endif
	// When in QAMplus using rates of 256QAM in HT mode, packet will be handled as VHT when transmitted also regarding power
	if (LaPhyMode == LA_PHY_MODE_HT_VHT)
	{
		if ( (ratesTable[rateindex].vhtHeMcs == MCS8_256QAM_3_4) || (ratesTable[rateindex].vhtHeMcs == MCS9_256QAM_5_6) )
		{
			isVht = TRUE;
		}
	}

	switch (LaPhyMode)
	{
		case LA_PHY_MODE_11AG:
			MEMCPY(&power, &RatePowerOffset11AG[rateindex], sizeof(RatePowerOffset_t));
			break;
		case LA_PHY_MODE_11B:
			MEMCPY(&power, RatePowerOffset11B, sizeof(RatePowerOffset_t));
			break;
		case LA_PHY_MODE_HT_VHT:
			if (isVht)
			{
				MEMCPY(&power, &RatePowerOffset11AC[ratesTable[rateindex].vhtHeMcs][ ratesTable[rateindex].numberOfNss], sizeof(RatePowerOffset_t));
				//ILOG0_DD("mcs:%d, mss:%d",ratesTable[rateindex].vhtHeMcs,ratesTable[rateindex].numberOfNss);
			}
			else
			{
				MEMCPY(&power, &RatePowerOffset11N[ratesTable[rateindex].htMcs], sizeof(RatePowerOffset_t));
			}
			break;
#ifdef ENET_INC_ARCH_WAVE600
		case LA_PHY_MODE_HE:
			MEMCPY(&power, &RatePowerOffset11AX[ratesTable[rateindex].vhtHeMcs], sizeof(RatePowerOffset_t));
			break;
#endif			
		default:
			DEBUG_ASSERT(0);
	}
	return power;	
}

/**********************************************************************************

searchMinPowerOverAntAtSpecificBw 


Description:
------------
Function returns the minimum value out of antenna selection (acitave antenna) for a specific bw
Input: 
-----
pTable -  pointer to 2D table  [MAX_NUM_OF_ANTENNAS] [LA_NUM_OF_BANDWIDTH]
maxAntNumber - maximum anetnnas
antennaSelectionBitmap - antenna selection bitmap set in antenna selection modukle
bw - specific bandwidth

Returns:
--------
	void - 
	
**********************************************************************************/

static int8 searchMinPowerOverAntAtSpecificBw(int8  pTable[MAX_NUM_OF_ANTENNAS][LA_NUM_OF_BANDWIDTH],uint8 antennaSelectionBitmap, Bandwidth_e bandwidth)
{
	uint8 antennaNum = 0;
	int8 minVal = MAX_INT8;

//	DEBUG_ASSERT(bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax));
	DEBUG_ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH);//KW_FIX_FW_G

	while (antennaSelectionBitmap != 0) 
	{
		if (antennaSelectionBitmap & 0x3)
		{
			DEBUG_ASSERT(antennaNum < MAX_NUM_OF_ANTENNAS);
			
			if (pTable[antennaNum][bandwidth] <= minVal)
			{
				minVal = pTable[antennaNum][bandwidth];
			}
			
		}
		antennaNum++;
		antennaSelectionBitmap>>=2;
	}

	return minVal;
}
/**********************************************************************************

searchMaxOverConnetcedAntAtSpecificBw 


Description:
------------
Function returns the maximum value out of antenna selection (acitave antenna) 


Input: 
-----
pTable -  pointer to 2D table  [MAX_NUM_OF_ANTENNAS] 
maxAntNumber - maximum anetnnas
antennaSelectionBitmap - antenna selection bitmap set in antenna selection modukle

Returns:
--------
	void - 
	
**********************************************************************************/

static int8 searchMaxPowerOverActivateAnt(const int8  pTable[MAX_POSSIBLE_NUM_OF_ANTENNAS],uint8 antennaSelectiondBitmap)
{
	uint8 antennaNum = 0;
	int8 maxVal = MIN_INT8;

	while (antennaSelectiondBitmap != 0) 
	{
		if (antennaSelectiondBitmap & 0x3)
		{
			DEBUG_ASSERT(antennaNum < MAX_NUM_OF_ANTENNAS);

			if (pTable[antennaNum] >= maxVal)
			{
				maxVal = pTable[antennaNum];
			}
			
		}
		antennaNum++;
		antennaSelectiondBitmap>>=2;
	}

	return maxVal;
}

/**********************************************************************************

calcInternalLowerLimit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static void calcInternalLowerLimit(StaId staIndex, uint8 antennaSelectiondBitmap)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	PowerAdaptationDatabase_t* pPowerAdaptationDb = &pLaDatabase->laStaGroupCommon.powerAdaptationDb;
	LinkAdaptationConfigurationParams_t* pLaConfigurationParams = pLaDatabase->laStaGroupCommon.pLinkAdaptationConfigurationParams;
	uint8 vapId = StaDbSwEntries[staIndex].vapIndex;
	int8 maxOverInternalLowerLimit;
	uint8 distantStaIndication;
	int8 distantStaPowerOffset;
	bool powerModeIndiecation;
	powerModeIndiecation = LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.powerModeIndication.staBitMap,staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check		
	if (pLaConfigurationParams->powerConfigurationParams.staInternalLowerLimit == INTERNAL_LIMIT_AUTO_VAL)
	{
 		if(powerModeIndiecation == 0)
		{
			maxOverInternalLowerLimit = searchMaxPowerOverActivateAnt(InternalPowerLowerLimit, antennaSelectiondBitmap);
			distantStaIndication = LA_GET_BIT_IN_BITMAP(powerAdaptationBitmaskDb.distantStaIndication.staBitMap,staIndex, SIZE_OF_STATIONS_BITMAP_IN_WORDS); //KW_FIX_FW_G Added array bound check
			distantStaPowerOffset = distantStaIndication*DistantStaBackoff[vapId] ;

			pPowerAdaptationDb->staInternalLowerLimit = maxOverInternalLowerLimit + distantStaPowerOffset;
	 	}
	}
	else
	{
		/*Force limit*/
		pPowerAdaptationDb->staInternalLowerLimit= pLaConfigurationParams->powerConfigurationParams.staInternalLowerLimit;
	}
#ifdef POWER_ADAPTATION_LOGS
	ILOG0_D("calcInternalLowerLimit, Limit = %d", pPowerAdaptationDb->staInternalLowerLimit);
#endif
}
/**********************************************************************************

CalcExternalLowerLimit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void calcExternalLowerLimit(StaId staIndex, uint8 antennaSelectionBitmap, Bandwidth_e bandwidth)
{
	uint8 antennaCount = AntennaSelectionGetAntennaCount(staIndex);
	uint8 antennaCountoffset = AntennaCountOffset[antennaCount-1];
	uint8 avgAntennaBoost = calcAvgAntennaBoost(antennaSelectionBitmap, antPowerBoost);

	PowerGlobalParameters.externalLowerLimit[bandwidth] = MAX(TPC_LOW_LIMIT, (int8)(ExternalPowerLowerLimit[bandwidth] - antennaCountoffset - avgAntennaBoost));
	
#ifdef POWER_ADAPTATION_LOGS
		ILOG0_DD("calcExternalLowerLimit, BW %d, Limit = %d", bandwidth,PowerGlobalParameters.externalLowerLimit[bandwidth]);
#endif
}
/**********************************************************************************

CalcExternalUpperLimit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static void calcExternalUpperLimit(StaId staIndex, uint8 antennaSelectionBitmap, Bandwidth_e bandwidth, uint8 isBFExist)
{
	uint8 antennaCount = AntennaSelectionCalcAntennaCount(antennaSelectionBitmap);
	uint8 antennaCountoffset = AntennaCountOffset[antennaCount-1];
	uint8 avgAntennaBoost = calcAvgAntennaBoost(antennaSelectionBitmap, antPowerBoost);
	int8 BfPowerLimit;
	UNUSED_PARAM(staIndex);	
	if(isBFExist)
	{
		BfPowerLimit = getBfPowerLimit(bandwidth);
		BfPowerLimit = MIN(RegulationLimit[bandwidth], BfPowerLimit);
	}
	else
	{
		BfPowerLimit = RegulationLimit[bandwidth];
	}
	PowerGlobalParameters.externalUpperLimit[bandwidth] = BfPowerLimit - antennaCountoffset - avgAntennaBoost;
	PowerGlobalParameters.externalUpperLimit[bandwidth] = MIN(PowerGlobalParameters.externalUpperLimit[bandwidth], MAX_TPC_POWER_LEVELS-1);
#ifdef POWER_ADAPTATION_LOGS
	ILOG0_DD("calcExternalUpperLimit, BW %d, Limit = %d", bandwidth, PowerGlobalParameters.externalUpperLimit[bandwidth]);
	ILOG0_DDD("calcExternalUpperLimit, BfPowerLimit = %d antennaCountoffset = %d avgAntennaBoost = %d", BfPowerLimit, antennaCountoffset, avgAntennaBoost);
	ILOG0_DD("calcExternalUpperLimit, getBfPowerLimit(bandwidth) = %d RegulationLimit[bandwidth] = %d", getBfPowerLimit(bandwidth), RegulationLimit[bandwidth]);
#endif
}
/**********************************************************************************

PowerAdaptationCalcStaLimits


Description:
------------
Function calculates limits:
internal upper limit
internal lower limit
external upper limit (regulation limit)
external lower limit


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void PowerAdaptationCalcStaLimits(StaId staIndex,Bandwidth_e bandwidth, uint8 antennaSelectionBitmap,uint8 isBFExist)
{

	calcInternalLowerLimit(staIndex, antennaSelectionBitmap);
	calcExternalLowerLimit(staIndex, antennaSelectionBitmap, bandwidth);
	calcExternalUpperLimit(staIndex, antennaSelectionBitmap, bandwidth,isBFExist);
}
/**********************************************************************************

CalcDefaultPowerLimits



Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void CalcDefaultPowerLimits(void)
{
	int8 maxOverInternalLowerLimit;
	int8 maxOverExternalLowerLimit = MIN_INT8;
	uint8 activateAntennaBitmap = AntennaSelectionGetActivatedAntennasBitmap();
	uint8 avgAntennaBoost = calcAvgAntennaBoost(activateAntennaBitmap, antPowerBoost);
	uint8 antennaCount = AntennaSelectionCalcAntennaCount(activateAntennaBitmap);
	uint8 antennaCountoffset = AntennaCountOffset[antennaCount-1];
	Bandwidth_e bandwidth;
	
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax); bandwidth++)
	{
		maxOverExternalLowerLimit = MAX(ExternalPowerLowerLimit[bandwidth],maxOverExternalLowerLimit);
		DefaultRegulationLimitPerAnt[bandwidth] = RegulationLimit[bandwidth] - antennaCountoffset - avgAntennaBoost;
	}
	
	maxOverInternalLowerLimit = searchMaxPowerOverActivateAnt(InternalPowerLowerLimit, activateAntennaBitmap);
	DefaultInternalLowerLimit = MAX(maxOverInternalLowerLimit, maxOverExternalLowerLimit);

}

/**********************************************************************************

CalculateAutoReplyPower 


Description:
------------
Calculate the auto reply power values and pass them to the PAC DUR in order to 
configure the static table in the PAC DUR RAM

Input: 
-----
	
		
Output:
-------
	l

Returns:
--------
	
	
**********************************************************************************/
void CalculateAutoReplyPower(void)
{
	PacDurAutoReplyPowerParameters_t pacDurAutoReplyPowerParameters;
	Bandwidth_e bandwidth = BANDWIDTH_TWENTY;
	uint8 *autoReplyPowerValuesAddress = NULL;
	uint8 autoReplyRate;
	
	memset(&pacDurAutoReplyPowerParameters, 0, sizeof(PacDurAutoReplyPowerParameters_t));

	/* For 11B there is no need to decrease the antenna offset - it is always transmitted with one antenna */
	pacDurAutoReplyPowerParameters.autoReplyPower11B = MIN(RegulationLimit11b, MAX_TPC_POWER_LEVELS-1);

	//set values for HE Extended range
	pacDurAutoReplyPowerParameters.autoReplyPower11axSuExtDcm1Ru242 = autoReply11AGMaxPowerPerBw[BANDWIDTH_TWENTY]; //currently take the power accoding to the 11ag power for 20MHZ
	pacDurAutoReplyPowerParameters.autoReplyPower11axSuExtDcm0Ru242 = autoReply11AGMaxPowerPerBw[BANDWIDTH_TWENTY]; //currently take the power accoding to the 11ag power for 20MHZ
	pacDurAutoReplyPowerParameters.autoReplyPower11axSuExtDcm1Ru106 = autoReply11AGMaxPowerPerBw[BANDWIDTH_TWENTY]; //currently take the power accoding to the 11ag power for 20MHZ
	pacDurAutoReplyPowerParameters.autoReplyPower11axSuExtDcm0Ru106 = autoReply11AGMaxPowerPerBw[BANDWIDTH_TWENTY]; //currently take the power accoding to the 11ag power for 20MHZ
		

	for(autoReplyRate = 0; autoReplyRate < LM_PHY_11A_MAX_RATES; autoReplyRate++)
	{
		autoReplyPowerValuesAddress = &pacDurAutoReplyPowerParameters.autoReplyPower11AGBw20;
		
		for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax); bandwidth++)
		{
			*autoReplyPowerValuesAddress = MIN(DefaultRegulationLimitPerAnt[bandwidth] ,autoReply11AGMaxPowerPerBw[bandwidth]) ;
			if(autoReplyRate >= LM_PHY_11A_RATE_24)
			{
				*autoReplyPowerValuesAddress = MIN(PowerGlobalParameters.refPower[bandwidth],*autoReplyPowerValuesAddress) ;	
			}

			autoReplyPowerValuesAddress++;
		}

		Pac_DurConfigureAutoReplyPower(&pacDurAutoReplyPowerParameters,autoReplyRate);	
	}

#ifdef ENET_INC_ARCH_WAVE600D2
	// Set power for auto reply in HE SU Extended Range mode
	Pac_DurSetExtendedRangeAutoReplyPower(&pacDurAutoReplyPowerParameters);
#endif //ENET_INC_ARCH_WAVE600D2

#ifdef DYNAMIC_RTS_ISSUE_WORKAROUND
	LinkAdaptationPhyDriver_UpdatePhyGenriscScpad();
#endif //DYNAMIC_RTS_ISSUE_WORKAROUND	
}


/**********************************************************************************

PowerAdaptationCalcPowerInStaLimits 


Description:
------------
Function calculates the new power out of: 
ref power = power reference
power offset = power offset from ref power according to rate
accumulateDiff = accomulate power, adaptation diff power for specific BW
probingStepPower = probing step power

Calculate power will be blocked with the 4 limits: 2 upper and two lower


Input: 
-----
staindex, bandwidth, rateindex, probingStepPower
Returns:
--------
	*newPower (by reference)
	isPowerBlocked = indicatio whether new power was blocked by the limits
	
**********************************************************************************/
bool PowerAdaptationCalcPowerInStaLimits(StaId staIndex, Bandwidth_e bandwidth, uint8 rateIndex, int8 inputPower, int8* outputPower)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	PowerAdaptationDatabase_t* pPowerDatabase = &pLaDatabase->laStaGroupCommon.powerAdaptationDb;
	int8 tempTable[3];
	bool isPowerBlocked = FALSE;
	int8 newPower;
	int8 upperLimit = PowerGlobalParameters.externalUpperLimit[bandwidth];
	int8 lowerLimit = PowerGlobalParameters.externalLowerLimit[bandwidth];
	uint8 antennaCount = AntennaSelectionGetAntennaCount(staIndex);
	const RateObj_t* ratesTable = getRatesTable(&pLaDatabase->laStaUspCommon);
	
	if (ratesTable[rateIndex].laPhyMode == LA_PHY_MODE_11B)
	{
		// Increase power limits in 11b due to change in number of transmitting antennas
		upperLimit = RegulationLimit11b;
		lowerLimit += AntennaCountOffset[antennaCount - 1];
	}

	newPower = MIN(inputPower, upperLimit);
	newPower = MIN(newPower, MAX_TPC_POWER_LEVELS-1);

	tempTable[0] = newPower;
	tempTable[1] = lowerLimit;
	tempTable[2] = pPowerDatabase->staInternalLowerLimit;
	newPower = searchMaxOverTable(tempTable,3);

	if (newPower != inputPower)
	{
		/*Power has been blocked by limits*/
		isPowerBlocked = TRUE;
	}
	
	*outputPower = newPower;
#ifdef POWER_ADAPTATION_LOGS
	ILOG0_DD("PowerAdaptationCalcPowerInStaLimits, newPower = %d, isPowerBlocked = %d",  newPower, isPowerBlocked);
#endif
	return isPowerBlocked;
}
/**********************************************************************************

PowerAdaptationSetLastLimitOffset 


Description:
------------


Input: 
-----

Returns:
--------

	
**********************************************************************************/
void PowerAdaptationSetLastLimitOffset(K_MSG* psMsg)
{
	UMI_TX_POWER_LIMIT* powerLimitParams = (UMI_TX_POWER_LIMIT*)pK_MSG_DATA(psMsg);

	if (powerLimitParams->getSetOperation == API_GET_OPERATION)
	{
		powerLimitParams->powerLimitOffset = lastPowerUpperLimitOffset;
	}
	else
	{
		// Power Limit should not be changed on the fly
		DEBUG_ASSERT(LmStaDataBase.NumOfConnectedSta == 0);
		
		lastPowerUpperLimitOffset = powerLimitParams->powerLimitOffset;
	}
}

/**********************************************************************************

SetVapFixedPower 


Description:
------------


Input: 
-----
Returns:
--------
	
**********************************************************************************/

void SetVapFixedPower(uint8 vapindex, uint8 powerVal, LaPacketType_e changeType)
{
	int8 tcrPowerTable[LA_NUM_OF_BANDWIDTH];
	//uint8 antennaConnectedBitmap = GetActivateAntennaBitmap();
	int8 perAntennaPowerVal;
	Bandwidth_e bandwidth;
	//uint8 antennaCount = CalcAntennaCount(antennaConnectedBitmap);
	//uint8 avgAntennaBoost = GetAvgAntennaBoost(antennaConnectedBitmap);

//	perAntennaPowerVal = powerVal - AntennaCountOffset[antennaCount-1] - avgAntennaBoost;
	perAntennaPowerVal = powerVal; //value per antenna  - when antenna count will be valid it should be value according to number of antennas
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax); bandwidth++)
	{
		ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
		tcrPowerTable[bandwidth] = perAntennaPowerVal;
	}
	
	SetPowerInVapTcr(vapindex, tcrPowerTable, BANDWIDTH_TWENTY, LinkAdaptationCommonConfiguration.wlanBandwidthMax, FALSE, changeType);
	
}
/**********************************************************************************

SetPowerLimit 


Description:
------------


Input: 
-----
Returns:
--------
	
**********************************************************************************/

static void SetPowerLimit(void)
{
	Bandwidth_e bandwidth;

	/* Update power limit table */	
    for (bandwidth = BANDWIDTH_TWENTY; bandwidth < LA_NUM_OF_BANDWIDTH; bandwidth++)
	{
		RegulationLimit[bandwidth] -= (lastPowerUpperLimitOffset << 1);
    }
	RegulationLimit11b -= (lastPowerUpperLimitOffset << 1);
	RegulationLimit11b = MAX(TPC_LOW_LIMIT,RegulationLimit11b);
}

/**********************************************************************************

calcProximateStaIndication 


Description:
------------
TBD

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void calcProximateStaIndication()
{
}
/**********************************************************************************

calcProximateStaIndication 


Description:
------------
TBD

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void calcAsymmetricLinkIndication()
{
}

/**********************************************************************************

calcProximateStaIndication 


Description:
------------
TBD

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void calcDistantStaIndication()
{
}
/**********************************************************************************

SetPowerTablesFromPsd 


Description:
------------
TBD

Input: 
-----

Returns:
--------
	
**********************************************************************************/

void SetPowerTablesFromPsd(uint16 tableSizeInBytes, PowerAdaptationTableType_e tableType, int8* tableScratchpad )
{
	switch (tableType)
	{
		case POWER_OFFSET_11B_HIGH_TABLE:
			MEMCPY((void*)&RatePowerOffsetHigh11B, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11B_HIGH_TABLE);
			break;
		case POWER_OFFSET_11AG_HIGH_TABLE:
			MEMCPY((void*)RatePowerOffsetHigh11AG, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AG_HIGH_TABLE);
			break;
		case POWER_OFFSET_11N_HIGH_TABLE:
			MEMCPY((void*)RatePowerOffsetHigh11N, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11N_HIGH_TABLE);
			break;
		case POWER_OFFSET_11AC_HIGH_TABLE:
			MEMCPY((void*)RatePowerOffsetHigh11AC, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AC_HIGH_TABLE);
			break;
		case POWER_OFFSET_11B_LOW_TABLE:
			MEMCPY((void*)&RatePowerOffsetLow11B, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11B_LOW_TABLE);
			break;
		case POWER_OFFSET_11AG_LOW_TABLE:
			MEMCPY((void*)RatePowerOffsetLow11AG, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AG_LOW_TABLE);
			break;
		case POWER_OFFSET_11N_LOW_TABLE:
			MEMCPY((void*)RatePowerOffsetLow11N, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11N_LOW_TABLE);
			break;
		case POWER_OFFSET_11AC_LOW_TABLE:
			MEMCPY((void*)RatePowerOffsetLow11AC, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AC_LOW_TABLE);
			break;
		case POWER_OFFSET_11AC_MU_HIGH_TABLE:
			MEMCPY((void*)RatePowerOffsetMUHigh11AC, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AC_MU_HIGH_TABLE);
			break;
		case POWER_OFFSET_11AC_MU_LOW_TABLE:
			MEMCPY((void*)RatePowerOffsetMULow11AC, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AC_MU_LOW_TABLE);
			break;
		case POWER_OFFSET_11AX_TABLE:
#ifdef ENET_INC_ARCH_WAVE600
			MEMCPY((void*)RatePowerOffset11AX, tableScratchpad, tableSizeInBytes);
			updateInitPowerOffsetTables(POWER_OFFSET_11AX_TABLE);
			break;
#else
			ASSERT(0);
			break;
#endif
		case POWER_OFFSET_HIGH_LOW_TABLE:
			flagPowerOffsetHighLowfrequency = TRUE;
			MEMCPY((void*)PowerOffsetHighLowfrequency, tableScratchpad, tableSizeInBytes);
			//updateInitPowerOffsetTables(POWER_OFFSET_HIGH_LOW_TABLE);
			break;
		case POWER_LIMIT_TABLE:
			MEMCPY((void*)&PowerLimits, tableScratchpad, tableSizeInBytes);
			//updateInitPowerOffsetTables(POWER_LIMIT_TABLE);
			break;
		case INTERNAL_POWER_LOWER_LIMIT_TABLE:
			MEMCPY((void*)InternalPowerLowerLimit, tableScratchpad, tableSizeInBytes);
			break;
		case PROXIMATE_STA_BACKOFF_TABLE:
			MEMCPY((void*)ProximateStaBackoff, tableScratchpad, tableSizeInBytes);
			break;
		case EXTERNAL_POWER_LOWER_LIMIT_TABLE:
			MEMCPY((void*)ExternalPowerLowerLimit, tableScratchpad, tableSizeInBytes);
			break;
		case DISTANT_STA_BACKOFF_TABLE:
			MEMCPY((void*)DistantStaBackoff, tableScratchpad, tableSizeInBytes);
			break;
		case INITIAL_RATE_11AX_TABLE:
			// TBD . currently taken from the hard coded table
			break;	
		case INITIAL_RATE_11NAC_TABLE:
			MEMCPY((void*)InitialRateAccordingToRssiHtVht, tableScratchpad, tableSizeInBytes);
			break;
		case INITIAL_RATE_11AG_TABLE:
			MEMCPY((void*)InitialRateAccordingToRssi11Ag, tableScratchpad, tableSizeInBytes);
			break;
		case INITIAL_RATE_11B_TABLE:
			MEMCPY((void*)InitialRateAccordingToRssi11b, tableScratchpad, tableSizeInBytes);
			break;
	}
	
}

/**********************************************************************************

GetInitialRateAccordingToRssi 


Description:
------------
TBD

Input: 
-----

Returns:
--------
	
**********************************************************************************/
uint8 GetInitialRateAccordingToRssi(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, int8 rssiVal)
{
	RateMask_t raMask = laDbDistributionParameter->laStaUspCommon->raIndexMaskOrg;
	uint8 mcsVal = 0;
	uint8 tableEntry = 0;
	uint8 initialRateIndex;
	uint8 tableLen;
	int8* pInitialRateTable = NULL;
	PhyMode_e staHighestPhyModeSupported;
	
	staHighestPhyModeSupported = ExtractHighestPhyModeFromRaMask(raMask, laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta);
	extractInitRateTableAndLen(staHighestPhyModeSupported,&pInitialRateTable, &tableLen);
	
	while ((tableEntry < tableLen) && (rssiVal >= pInitialRateTable[tableEntry]))
	{
		tableEntry++;
	}
	tableEntry--;
	
	DEBUG_ASSERT(tableEntry < tableLen); //Check that tableEntry was not wrap arround

	if (staHighestPhyModeSupported != PHY_MODE_11B)
	{
		/*Non 11B phy mode*/
		mcsVal = tableEntry;
	}
	else
	{
		/*11B phy mode*/
		mcsVal = ConvertInitRateEntryToMcs11B[tableEntry];
	}
	/*Convert MCS to rateIndex according to phy mode - note: in AC, NSS should be zero*/
	initialRateIndex = convertTcr2RateIndex(staHighestPhyModeSupported,mcsVal);
	/*Get the closest rate in mask - stet BANDWIDTH_FOURTY because in this BW all rates are valid, if BW will be changed later the rate will be adjusted*/
	initialRateIndex = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, BANDWIDTH_FOURTY,initialRateIndex,raMask,0, TRUE);

	if (isBitInMask64Bit(initialRateIndex, raMask.raIndexMask64bit) == 0)
	{
		/*Rate is not in RA mask - search up*/
		initialRateIndex = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, BANDWIDTH_FOURTY,initialRateIndex,raMask,0, TRUE);
	}
	
	if (isBitInMask64Bit(initialRateIndex, raMask.raIndexMask64bit) == 0)
	{
		/*Still rate not in mask - required rate is not valid*/
		DEBUG_ASSERT(0);
	}
//	ILOG0_DDD("GetInitialRateAccordingToRssi,rssiVal = %d,  phyMode = %d, initialRateIndex = %d", rssiVal, staHighestPhyModeSupported, initialRateIndex);

	return initialRateIndex;
}
/**********************************************************************************

extractInitRateTableAndLen 


Description:
------------
TBD

Input: 
-----

Returns:
--------
	
**********************************************************************************/

static void extractInitRateTableAndLen(PhyMode_e phyMode,int8** pInitialRateTable,uint8* tableLen)
{
	switch (phyMode)
	{
		case (PHY_MODE_11AG):
			*pInitialRateTable = InitialRateAccordingToRssi11Ag;
			*tableLen = INITIAL_RATE_NUM_OF_OPTIONS_11AG;
			break;
		case (PHY_MODE_11B):
			*pInitialRateTable = InitialRateAccordingToRssi11b;
			*tableLen = INITIAL_RATE_NUM_OF_OPTIONS_11B;
			break;
		case (PHY_MODE_11N):
			*pInitialRateTable = InitialRateAccordingToRssiHtVht;
			*tableLen = INITIAL_RATE_NUM_OF_OPTIONS_11N;
			break;
		case (PHY_MODE_11AC):
			*pInitialRateTable = InitialRateAccordingToRssiHtVht;
			*tableLen = INITIAL_RATE_NUM_OF_OPTIONS_11AC;
			break;
#ifdef ENET_INC_ARCH_WAVE600
		case (PHY_MODE_11AX_SU):
			*pInitialRateTable = InitialRateAccordingToRssiHe;
			*tableLen = INITIAL_RATE_NUM_OF_OPTIONS_11AX;
			break;	
#endif
		default:
			ASSERT(0);			
	}
}

uint8 calcAvgAntennaBoost(uint8 antennaSelectionBitmap, uint8* antBoost)
{
	uint8 antSelBoostsSum = 0;
	uint8 avgAntBoost, isSelectedAnt, ant, antCount = 0;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		isSelectedAnt = antennaSelectionBitmap & (0x3 << (ant << 1));
		if (isSelectedAnt != 0)
		{
			antCount++;
			antSelBoostsSum += (antBoost[ant] << 1);	// convert to dB/2
		}
	}
	//KW_FIX_FW_M
	ASSERT(antCount);

	avgAntBoost = antSelBoostsSum / antCount;
	return avgAntBoost;
}
void extractAndSetAntBoosts(uint8* antBoost, LA_SET_TPC_DATA_t* pTpcPowerData)
{
	antBoost[0] = pTpcPowerData->ant0PowerBoost;
	antBoost[1] = pTpcPowerData->ant1PowerBoost;
	antBoost[2] = pTpcPowerData->ant2PowerBoost;
	antBoost[3] = pTpcPowerData->ant3PowerBoost;
	
	setAntBoostInTcrs(antBoost);
}

