/***********************************************************************************
 File:		LinkAdaptationDriver.h
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "LinkAdaptation.h"
#include "linkAdaptation_api.h"
#include "LinkAdaptationPhyDriver.h"
#include "Tcr_Descriptors.h"
#include "Rcr_Descriptors.h"
#include "PowerAdaptation.h"
#include "AntennaSelection.h"
#include "ShramVapDatabase.h"
#include "Utils_Api.h"
#include "stringLibApi.h"
#include "CommonRamLinkAdaptation.h"
#include "ulPowerControl.h"
#include "TxSender_InitApi.h"
#include "PlanManager_API.h"
#include "ShramStationDatabase.h"
#include "PreAggregator_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
/***************************************************************************/
/* 					Constants                                                  				     */
/***************************************************************************/
#define LOG_LOCAL_GID   GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 20

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void setDataPhaseTcrACommonParams(void* phasePtr, uint8 numOfValidUsers, uint8 vapId, bool isDl, bool staticGroup);
static void setDataPhaseTcrBCommonParams(void* phasePtr, uint8 vapId, bool isDl, bool staticGroup);
static void setDataPhaseCommonParams(void* phasePtr, uint8 numOfValidUsers, bool isDl, bool staticGroup);
static void setSoundingPhaseUserTcrRcrParams (SoundingPhase_t* soundingPhasePtr, uint8 uspIndex, bool staticGroup);
static void setSoundingPhaseCommonUserTcrATcrBParams(SoundingPhase_t* soundingPhasePtr, uint8 vapId, bool staticGroup);
static void setSoundingPhaseTcrACommonParams(SoundingPhase_t* soundingPhasePtr, uint8 numOfValidUsers, uint8 vapId, LaAddHeGroupParams_t *addHeGroupParameters, uint8 planIndex);
static void setSoundingPhaseTcrBCommonParams(SoundingPhase_t* soundingPhasePtr, uint8 numOfValidUsers, uint8 vapId, bool staticGroup);
static void setSoundingPhaseRcrCommonParams(SoundingPhase_t* soundingPhasePtr, uint8 numOfValidUsers, bool staticGroup);
static void setSoundingTfCommonInfoParams(SoundingPhase_t* soundingPhasePtr, bool staticGroup);
static void setDataPhaseUserTcrRcrParams(void* phasePtr, uint8 uspIndex, bool isDl, bool staticGroup, bool ldpcCapable);
static void setDataPhaseTfCommonInfoParams(void* phasePtr, bool isDl, bool staticGroup);
static void setDataPhaseCommonUserTcrATcrBParams(void* phasePtr, uint8 vapId, bool isDl, bool staticGroup);
static void setDataPhaseRcrCommonParams(void* phasePtr, uint8 numOfValidUsers, bool isDl, bool staticGroup);
static void setProtectionPhaseUserRcrParams(ProtectionPhase_t* phasePtr, uint8 uspIndex, bool staticGroup);
static void setProtectionPhaseCommonUserTcrBParams(ProtectionPhase_t* phasePtr, uint8 vapId, bool staticGroup);
static void setProtectionPhaseTcrBCommonParams(ProtectionPhase_t* phasePtr, uint8 numOfValidUsers, uint8 vapId, bool staticGroup);
static void setProtectionPhaseRcrCommonParams(ProtectionPhase_t* phasePtr, uint8 numOfValidUsers, bool staticGroup);
static void setNheLtfInTcr(DlDataPhase_t* dlDataPhase, uint8 numOfValidUsers);
static uint8 calcTfApTxPower(TcrCommon_t* pTcr);
static void overrideRateForStaticGroup(void* phasePtr, bool isDl, uint8 uspIndex);
static void overrideLdpcForGroup(void* phasePtr, bool isDl, uint8 uspIndex, bool ldpcCapable);

uint8 convertNssToNHeLtf[VHT_HE_NUMBER_OF_NSS] = {0,1,2,2};
uint8 convertMcsToDbps[MCS_5+1] = {26, 52, 78, 104, 156, 208}; // for MCS 0 - 5
/*---------------------------------------------------------------------------------
/						Functions definitions					
/----------------------------------------------------------------------------------*/

void setDlDataPhaseParams(DlDataPhase_t* dlPhasePtr, uint8 vapId, bool staticGroup)
{
    DlDataPhaseUserPhase_t* dlUserPhase = NULL;	
    StaId                   staId;
    StaDbldpcConfig_e       ldpcValue;
    bool                    ldpcCapable = TRUE;
    uint8                   numOfValidUsers;
	uint8                   uspIndex; 
	uint8                   loopIndex;	

	numOfValidUsers = getNumOfValidUspInPhase(dlPhasePtr); 
	ILOG0_D("[setDlDataPhaseParams], numOfValidUsers = %d", numOfValidUsers);

    for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
        dlUserPhase = (DlDataPhaseUserPhase_t*)&dlPhasePtr->userPhase[uspIndex];
        staId = dlUserPhase->staId;
        ldpcValue = getLdpcCapability(staId, LA_PHY_MODE_HE);
        if (ldpcValue != LDPC)
        {              
            ldpcCapable = FALSE;
#ifdef DYNAMIC_GROUPING_DEBUG
            ILOG0_D("[setDlDataPhaseParams], ldpc false, one of the users doesn't support ldpc, staId: %d", staId);
#endif
            break;
        }        
	}
	for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
		/* Set TCRs and RCR per user DL phase params */
		setDataPhaseUserTcrRcrParams(dlPhasePtr, uspIndex, TRUE, staticGroup, ldpcCapable);
	}	
	/* Set  Common user TCR B DL phase params*/
	setDataPhaseCommonUserTcrATcrBParams(dlPhasePtr, vapId, TRUE, staticGroup);
	/* Set TCR A Common DL phase params */
	setDataPhaseTcrACommonParams(dlPhasePtr, numOfValidUsers, vapId, TRUE, staticGroup);
	/* Set TCR B Common DL phase params */
	setDataPhaseTcrBCommonParams(dlPhasePtr, vapId, TRUE, staticGroup);
	
	/* Set DL Phase Common params - must be called afer TCR common setters*/
	setDataPhaseCommonParams(dlPhasePtr, numOfValidUsers, TRUE, staticGroup);
	/* Set  RCR common DL phase params */
	setDataPhaseRcrCommonParams(dlPhasePtr, numOfValidUsers, TRUE, staticGroup);
	/* Set TF Common info DL phase params - must be called afer RCR common setters*/
	setDataPhaseTfCommonInfoParams(dlPhasePtr, TRUE, staticGroup);	
#if 1
	SLOG0(0, 0, DlDataPhaseCommon_t, &(dlPhasePtr->common));
	SLOG0(0, 0, TcrCommon_t, &(dlPhasePtr->commonTcra));
	SLOG0(0, 0, TcrCommon_t, &(dlPhasePtr->commonTcrb));
	SLOG0(0, 0, RcrCommon_t, &(dlPhasePtr->commonRcr));
	SLOG0(0, 0, DlDataPhaseCommonUserTcrb_t, &(dlPhasePtr->commonUserTcrb));
	for (loopIndex = 0; loopIndex < NUM_OF_USERS_MU_HE_PHASE; loopIndex++)
	{
		SLOG0(0, 0, DlDataPhaseUserPhase_t, &(dlPhasePtr->userPhase[loopIndex]));
	}
	SLOG0(0, 0, DlDataPhaseTfParameters_t, &(dlPhasePtr->tfParameters));
	SLOG0(0, 0, DlDataPhaseTfCommonInfo_t, &(dlPhasePtr->tfCommonInfo));
	SLOG0(0, 0, DlDataPhaseTfMuBar_t, &(dlPhasePtr->tfMuBar));
	for (loopIndex = 0; loopIndex < HALF_NUM_OF_USERS_MU_HE_PHASE; loopIndex++)
	{
		SLOG0(0, 0, DlDataPhaseTfUserInfo_t, &(dlPhasePtr->tfUserInfo[loopIndex]));
	}
#endif
}

void setUlDataPhaseParams(UlDataPhase_t* ulPhasePtr, uint8 vapId, LaAddHeGroupParams_t *addHeGroupParameters)
{
    UlDataPhaseUserPhase_t* ulUserPhase = NULL;	
    StaId                   staId;
    StaDbldpcConfig_e       ldpcValue;
    bool                    ldpcCapable = TRUE;
    bool                    isStaticGroup = addHeGroupParameters->isStaticGroup;
    uint8                   uspIndex; 
	uint8                   numOfValidUsers;
	uint8                   loopIndex;
#ifdef ENET_INC_ARCH_WAVE600D2
	uint8                   rcrMcs = 0;
	uint8                   rcrNss = 0;
	uint8                   maxNssPerUsp = 0;
#endif

	numOfValidUsers = getNumOfValidUspInPhase(ulPhasePtr); 	
	ILOG0_D("[setUlDataPhaseParams], numOfValidUsers = %d", numOfValidUsers);

    for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
        ulUserPhase = (UlDataPhaseUserPhase_t*)&ulPhasePtr->userPhase[uspIndex];
        staId = ulUserPhase->staId;
        ldpcValue = getLdpcCapability(staId, LA_PHY_MODE_HE);
        if (ldpcValue != LDPC)
        {              
            ldpcCapable = FALSE;
            break;
        }        
	}
	for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
		/* Set TCRs and RCR per user UL phase params */
		setDataPhaseUserTcrRcrParams(ulPhasePtr, uspIndex, FALSE, isStaticGroup, ldpcCapable);
#ifdef ENET_INC_ARCH_WAVE600D2
		if (ulPhasePtr->common.dataMuMimo == UL_MU_MIMO_MU_MIMO) //UL MIMO phase
		{
			/* Limit NSS per user */
			DEBUG_ASSERT(uspIndex < MAX_NUM_OF_USERS_IN_MIMO_GROUP);
			maxNssPerUsp = addHeGroupParameters->maxNss[uspIndex];
			DEBUG_ASSERT(maxNssPerUsp <= SPATIAL_STREAM_2); /* We support only up to 2 nss per USP*/	
			
			rcrMcs = EXTRACT_MCS_FROM_VHT_HE_RATE(ulPhasePtr->userPhase[uspIndex].userRcrPsduRate);
			rcrNss = EXTRACT_NSS_FROM_VHT_HE_RATE(ulPhasePtr->userPhase[uspIndex].userRcrPsduRate);			
			ulPhasePtr->userPhase[uspIndex].userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(rcrMcs, MIN(rcrNss, maxNssPerUsp),NO_DCM);
		}	
#endif
	}	
	/* Set Common user TCR A&B UL phase params*/
	setDataPhaseCommonUserTcrATcrBParams(ulPhasePtr, vapId, FALSE, isStaticGroup);
	/* Set TCR A Common UL phase params */
	setDataPhaseTcrACommonParams(ulPhasePtr, numOfValidUsers, vapId, FALSE, isStaticGroup);
	/* Set TCR B Common UL phase params */
	setDataPhaseTcrBCommonParams(ulPhasePtr, vapId, FALSE, isStaticGroup);
	/* Set RCR common UL phase params */
	setDataPhaseRcrCommonParams(ulPhasePtr, numOfValidUsers, FALSE, isStaticGroup);
	/* Set TF Common info UL phase params - must be called afer RCR common setters*/
	setDataPhaseTfCommonInfoParams(ulPhasePtr, FALSE, isStaticGroup); 
#if 1
	SLOG0(0, 0, UlDataPhaseCommon_t, &(ulPhasePtr->common));
	SLOG0(0, 0, TcrCommon_t, &(ulPhasePtr->commonTcra));
	SLOG0(0, 0, TcrCommon_t, &(ulPhasePtr->commonTcrb));
	SLOG0(0, 0, RcrCommon_t, &(ulPhasePtr->commonRcr));
	SLOG0(0, 0, UlDataPhaseCommonUserTcra_t, &(ulPhasePtr->commonUserTcra));
	SLOG0(0, 0, UlDataPhaseCommonUserTcrb_t, &(ulPhasePtr->commonUserTcrb));
	for (loopIndex = 0; loopIndex < NUM_OF_USERS_MU_HE_PHASE; loopIndex++)
	{
		SLOG0(0, 0, UlDataPhaseUserPhase_t, &(ulPhasePtr->userPhase[loopIndex]));
	}
	SLOG0(0, 0, UlDataPhaseTfParameters_t, &(ulPhasePtr->tfParameters));
	SLOG0(0, 0, UlDataPhaseTfCommonInfo_t, &(ulPhasePtr->tfCommonInfo));
	for (loopIndex = 0; loopIndex < HALF_NUM_OF_USERS_MU_HE_PHASE; loopIndex++)
	{
		SLOG0(0, 0, UlDataPhaseTfUserInfo_t, &(ulPhasePtr->tfUserInfo[loopIndex]));
	}
#endif		
}

void setProtectionPhaseParams(ProtectionPhase_t* protectionPhasePtr, uint8 vapId, bool staticGroup)
{ 
	uint8 numOfValidUsers;
	uint8 uspIndex; 

	numOfValidUsers = getNumOfValidUspInPhase(protectionPhasePtr); 
	for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
		/* Set RCR per user Protection phase params */
		setProtectionPhaseUserRcrParams(protectionPhasePtr, uspIndex, staticGroup);
	}		
	/* Set Common user TCR B Protection phase params*/
	setProtectionPhaseCommonUserTcrBParams(protectionPhasePtr, vapId, staticGroup);
	/* Set TCR B Common Protection phase params */
	setProtectionPhaseTcrBCommonParams(protectionPhasePtr, numOfValidUsers, vapId, staticGroup);
	/* Set RCR common Protection phase params */
	setProtectionPhaseRcrCommonParams(protectionPhasePtr, numOfValidUsers, staticGroup);
}

/**********************************************************************************
setSoundingPhaseParams

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
void setSoundingPhaseParams(SoundingPhase_t* soundingPhasePtr, uint8 vapId, LaAddHeGroupParams_t *addHeGroupParameters, uint8 planIndex)
{
		uint8 uspIndex; 
		uint8 numOfValidUsers;
		uint8 loopIndex;
	
		numOfValidUsers = getNumOfValidUspInPhase(soundingPhasePtr);	
		ILOG0_D("[setSoundingPhaseParams], numOfValidUsers = %d", numOfValidUsers);
		for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
		{
			/* Set TCRs and RCR per user Sounding phase params */
			setSoundingPhaseUserTcrRcrParams(soundingPhasePtr, uspIndex, addHeGroupParameters->isStaticGroup);
		}	
		/* Set Common user TCR A&B Sounding phase params*/
		setSoundingPhaseCommonUserTcrATcrBParams(soundingPhasePtr, vapId, addHeGroupParameters->isStaticGroup);
		/* Set TCR A Common Sounding phase params */
		setSoundingPhaseTcrACommonParams(soundingPhasePtr, numOfValidUsers, vapId, addHeGroupParameters, planIndex);
		/* Set TCR B Common Sounding phase params */
		setSoundingPhaseTcrBCommonParams(soundingPhasePtr, numOfValidUsers, vapId, addHeGroupParameters->isStaticGroup);
		/* Set RCR common Sounding phase params */
		setSoundingPhaseRcrCommonParams(soundingPhasePtr, numOfValidUsers, addHeGroupParameters->isStaticGroup);
		/* Set TF Common info Sounding phase params - must be called after RCR common setters*/
		setSoundingTfCommonInfoParams(soundingPhasePtr, addHeGroupParameters->isStaticGroup); 
		// TODO: what about TF per user missing also in DL and UL phases??
#if 1
		SLOG0(0, 0, SoundingPhaseCommon_t, &(soundingPhasePtr->common));
		SLOG0(0, 0, SoundingPhaseCommonTcra_t, &(soundingPhasePtr->commonTcra));
		SLOG0(0, 0, SoundingPhaseCommonTcrb_t, &(soundingPhasePtr->commonTcrb));
		SLOG0(0, 0, SoundingPhaseCommonRcr_t, &(soundingPhasePtr->commonRcr));
		SLOG0(0, 0, SoundingPhaseCommonUserTcra_t, &(soundingPhasePtr->commonUserTcra));
		SLOG0(0, 0, SoundingPhaseCommonUserTcrb_t, &(soundingPhasePtr->commonUserTcrb));
		for (loopIndex = 0; loopIndex < NUM_OF_USERS_MU_HE_PHASE; loopIndex++)
		{
			SLOG0(0, 0, SoundingPhaseUserPhase_t, &(soundingPhasePtr->userPhase[loopIndex]));
		}
		SLOG0(0, 0, SoundingPhaseTfParameters_t, &(soundingPhasePtr->tfParameters));
		SLOG0(0, 0, SoundingPhaseTfCommonInfo_t, &(soundingPhasePtr->tfCommonInfo));
		for (loopIndex = 0; loopIndex < HALF_NUM_OF_USERS_MU_HE_PHASE; loopIndex++)
		{
			SLOG0(0, 0, SoundingPhaseTfUserInfo_t, &(soundingPhasePtr->tfUserInfo[loopIndex]));
		}
		SLOG0(0, 0, HeSoundingNdpaPointer_t, (HeSoundingNdpaPointer_t*)soundingPhasePtr->common.ndpaPointer);
#endif		
}

/**********************************************************************************
setSoundingPhaseUserTcrRcrParams


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void setSoundingPhaseUserTcrRcrParams(SoundingPhase_t* soundingPhasePtr, uint8 uspIndex, bool staticGroup)
{
	SoundingPhaseUserPhase_t* soundingUserPhase = NULL; 	
	StaId staId;
	uint8 maxNss = 0;
	uint8 rcrRequestedNss = 0;
	uint8 rcrRequestedMcs = 0;
	uint8 rcrMaxMcs = 0; // max MCS for the chosen RCR NSS
	uint8 rcrChosenNss = 0;
	uint8 rcrChosenMcs = 0;
	uint8 rcrRateBefore = 0;

	soundingUserPhase = (SoundingPhaseUserPhase_t*)&soundingPhasePtr->userPhase[uspIndex];
	ILOG0_D("[setSoundingPhaseUserTcrRcrParams], uspIndex = %d", uspIndex);
	
	staId = soundingUserPhase->staId;
	DEBUG_ASSERT(staId < HW_NUM_OF_STATIONS);
		
	if (staticGroup == TRUE)
	{

		rcrRateBefore = soundingUserPhase->userRcrPsduRate;
		rcrRequestedNss = EXTRACT_NSS_FROM_VHT_HE_RATE(rcrRateBefore);	
		rcrRequestedMcs = EXTRACT_MCS_FROM_VHT_HE_RATE(rcrRateBefore);	
		getHighestNssInMask(staId, &maxNss, GetDataBwLimit(staId, INVALID_MU_USP_INDEX, FALSE));

		rcrChosenNss = MIN(maxNss, rcrRequestedNss);
		getHighestMcsForNssInMask(staId, rcrChosenNss, &rcrMaxMcs, PHY_MODE_11AX_MU_DL,GetDataBwLimit(staId, INVALID_MU_USP_INDEX, FALSE));
		rcrChosenMcs = MIN(rcrMaxMcs, rcrRequestedMcs);
		// Override rate in phase RCR
		soundingUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(rcrChosenMcs, rcrChosenNss, NO_DCM);

	}
}

/**********************************************************************************
setSoundingPhaseCommonUserTcrATcrBParams


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void setSoundingPhaseCommonUserTcrATcrBParams(SoundingPhase_t* soundingPhasePtr, uint8 vapId, bool staticGroup)
{
	VapDbCommon_t* pVapDbCommon = &VapDbHwEntries[vapId].common;

	UNUSED_PARAM(staticGroup);
	/* fill TCR A (used for NDP) from VAP DB (management 20MHz) */
	soundingPhasePtr->commonUserTcra.brdcstUserTcraPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(MCS_1, pVapDbCommon->maxNssNdp, NO_DCM);

	soundingPhasePtr->commonUserTcra.brdcstUserTcraPacketExtension = 0x2; //16usec

	/* fill TCR B (used for TF) from VAP DB (management 20MHz) */
	soundingPhasePtr->commonUserTcrb.brdcstUserTcrbPsduRate = pVapDbCommon->tcr220Mhz80211PsduRate;
	soundingPhasePtr->commonUserTcrb.brdcstUserTcrbPacketExtension = 0x2; //16usec
}

/**********************************************************************************
setSoundingPhaseTcrACommonParams


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void setSoundingPhaseTcrACommonParams(SoundingPhase_t* soundingPhasePtr, uint8 numOfValidUsers, uint8 vapId, LaAddHeGroupParams_t *addHeGroupParameters, uint8 planIndex)
{
	TcrCommon_t* tcraPtr; 
	HeSoundingNdpaPointer_t* ndpaPtr;
	VapDbCommon_t* pVapDbCommon = &VapDbHwEntries[vapId].common;
	uint8 regulationLimit;
	uint8 uspIndex;
	DlMuMimo_e dlMuMimoType;
	uint16 userTcraStaAid;

	UNUSED_PARAM(planIndex);	
	tcraPtr = (TcrCommon_t*)&soundingPhasePtr->commonTcra;

	tcraPtr->antennaSelection = AntennaSelectionGetDefaultBitmap();
	tcraPtr->ant0Boost = 0;
	tcraPtr->ant1Boost = 0;
	tcraPtr->ant2Boost = 0;
	tcraPtr->ant3Boost = 0;
	tcraPtr->scp 	   = 0;
	tcraPtr->smoothing = 1; //Should be enabled when txBf = 0 and Vice Versa
	tcraPtr->stbc 	   = NOT_STBC_TRANSMISSION;
	tcraPtr->heSigRate = 0;

	dlMuMimoType = soundingPhasePtr->common.muMimoDataTransmission;

	/* nHeltf of PHY NDP should be taken from maxNssNdp (Nr) */
	tcraPtr->nHeltf = convertNssToNHeLtf[pVapDbCommon->maxNssNdp];	
	ILOG0_D("[setSoundingPhaseTcrACommonParams], tcraPtr->nHeltf = %d",tcraPtr->nHeltf);
	
	if (dlMuMimoType == DL_MU_MIMO_NO_MU_MIMO_OR_HE_MU_OFDMA) //OFDMA
	{
		regulationLimit = getRegLimitPerAnt(tcraPtr->cbw);
	}
	else // MIMO
	{
		regulationLimit = getMuPowerLimit(tcraPtr->cbw);
	}

	tcraPtr->rfPower = MIN(tcraPtr->rfPower, regulationLimit);

	// Nr = number TX antenna (AP); Nc = number RX antenna (STA); Nr x Nc report size per bin that is reported by station
	for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
		userTcraStaAid = soundingPhasePtr->userPhase[uspIndex].staId;
		ndpaPtr = (HeSoundingNdpaPointer_t*)(soundingPhasePtr->common.ndpaPointer);
		if (dlMuMimoType == DL_MU_MIMO_MU_MIMO)
		{
			ndpaPtr->sta[uspIndex].ncIndex = addHeGroupParameters->maxNss[uspIndex];
		}
		else
		{
			// TBD - ncIndex for OFDMA			
			ndpaPtr->sta[uspIndex].ncIndex = 0x0;
		}
	}
}

/**********************************************************************************
setSoundingPhaseTcrBCommonParams


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void setSoundingPhaseTcrBCommonParams(SoundingPhase_t* soundingPhasePtr, uint8 numOfValidUsers, uint8 vapId, bool staticGroup)
{
	TcrCommon_t* tcrbPtr; 
	//VapDbCommon_t* pVapDbCommon = &VapDbHwEntries[vapId].common;
	uint8 regulationLimit;
	DlMuMimo_e dlMuMimoType;

	UNUSED_PARAM(numOfValidUsers);
	UNUSED_PARAM(vapId);
	UNUSED_PARAM(staticGroup);
	tcrbPtr = (TcrCommon_t*)&soundingPhasePtr->commonTcrb;

	tcrbPtr->antennaSelection = AntennaSelectionGetDefaultBitmap();
	tcrbPtr->ant0Boost = 0;
	tcrbPtr->ant1Boost = 0;
	tcrbPtr->ant2Boost = 0;
	tcrbPtr->ant3Boost = 0;
	tcrbPtr->scp 	   = 0;
	tcrbPtr->smoothing = 1; //Should be enabled when txBf = 0 and Vice Versa
	tcrbPtr->stbc 	   = NOT_STBC_TRANSMISSION;
	tcrbPtr->heSigRate = 0;

	dlMuMimoType = soundingPhasePtr->common.muMimoDataTransmission;
	
	if (dlMuMimoType == DL_MU_MIMO_NO_MU_MIMO_OR_HE_MU_OFDMA) //OFDMA
	{
		regulationLimit = getRegLimitPerAnt(tcrbPtr->cbw);
	}
	else // MIMO
	{
		regulationLimit = getMuPowerLimit(tcrbPtr->cbw);
	}

	tcrbPtr->rfPower = MIN(tcrbPtr->rfPower, regulationLimit);
}

/**********************************************************************************
setSoundingPhaseRcrCommonParams


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void setSoundingPhaseRcrCommonParams(SoundingPhase_t* soundingPhasePtr, uint8 numOfValidUsers, bool staticGroup)
{
	SoundingPhaseUserPhase_t* soundingUserPhase = NULL; 
	RcrCommon_t* rcrCommonPtr;
	uint16 userTcraStaAid;
	uint8 uspIndex; 
	uint8 userNss;
	uint8 maxOfNssForAllUsers = 0;
	bool isOneUserLdpc = FALSE;

	UNUSED_PARAM(staticGroup);
	rcrCommonPtr = (RcrCommon_t*)&(soundingPhasePtr->commonRcr);
	for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
		soundingUserPhase = &soundingPhasePtr->userPhase[uspIndex];
		userTcraStaAid = soundingUserPhase->staId;
		userNss = EXTRACT_NSS_FROM_VHT_HE_RATE(soundingUserPhase->userRcrPsduRate);  // Assuming that psdu rate is already set
		if (userNss > maxOfNssForAllUsers)
		{
			maxOfNssForAllUsers = userNss;
		}
		if(soundingUserPhase->userRcrLdpc == 1)
		{
			isOneUserLdpc = TRUE;
		}
	}
	rcrCommonPtr->ldpcExtraSymbol = 0; 
#ifdef LDPC_EXTRA_SYMBOL_ON		
	if(isOneUserLdpc == TRUE)
	{
		rcrCommonPtr->ldpcExtraSymbol = 1; 
	}
#endif		

	rcrCommonPtr->nHeltf = convertNssToNHeLtf[maxOfNssForAllUsers]; 

}

/**********************************************************************************
setSoundingTfCommonInfoParams


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
static void setSoundingTfCommonInfoParams(SoundingPhase_t* soundingPhasePtr, bool staticGroup)
{
	RcrCommon_t* rcrCommonPtr; 
	uint8 tfApTxPower;
	TcrCommon_t* tcraPtr; 

	UNUSED_PARAM(staticGroup);
	rcrCommonPtr = (RcrCommon_t*)&(soundingPhasePtr->commonRcr);
	tcraPtr  = (TcrCommon_t*) &soundingPhasePtr->commonTcra; // TCR for TF transmission
	
	soundingPhasePtr->tfCommonInfo.tfNumberOfHeLtfSymbols = rcrCommonPtr->nHeltf;
	soundingPhasePtr->tfCommonInfo.tfPacketExtension = (rcrCommonPtr->peDisambiguty << 2);	// PE disambiguity = 1 [bit 2], pre-FEC padding factor = 0 (a=4) [bits 1:0]
	soundingPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = 0;
#ifdef LDPC_EXTRA_SYMBOL_ON						
	// TODO: in A0 bug that all users should have the same coding if if fixed at least one user with userRcrLdpc=1 should set tfLdpcExtraSymbolSegment = 1
	if (soundingPhasePtr->userPhase[0].userRcrLdpc == 1)
	{
		soundingPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = 1;
	}
#endif		
	ILOG0_D("[setSoundingTfCommonInfoParams], soundingPhasePtr->userPhase[0].userRcrLdpc = %d", soundingPhasePtr->userPhase[0].userRcrLdpc);
	ILOG0_D("[setSoundingTfCommonInfoParams], soundingPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = %d", soundingPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment);

	soundingPhasePtr->tfCommonInfo.tfStbc = NOT_STBC_TRANSMISSION;		
	tfApTxPower = calcTfApTxPower(tcraPtr);
	soundingPhasePtr->tfCommonInfo.tfApTxPower30 = (tfApTxPower & 0xF);  //Set 4 LSBs  
	soundingPhasePtr->tfCommonInfo.tfApTxPower54 = ((tfApTxPower >> 4) & 0x3); //Set 2 MSBs  
	
}

static void setDataPhaseUserTcrRcrParams(void* phasePtr, uint8 uspIndex, bool isDl, bool staticGroup, bool ldpcCapable)
{
	TcrCommon_t* tcraPtr; 
	DlDataPhase_t* dlDataPhasePtr = NULL;
	DlDataPhaseUserPhase_t* dlUserPhase = NULL;		
	UlDataPhase_t* ulDataPhasePtr = NULL;
	UlDataPhaseUserPhase_t* ulUserPhase = NULL;	
	StaId staId;
	uint8 rateIndex, tcrDlRate;
	PhyMode_e phyMode;
	uint8 ruSize;
	uint16 userTcraStaAid;
    uint8 userNss;
#ifndef DYNAMIC_GROUPING_DEBUG
	LnaType_e lnaType;
	int8 normalizedRssi;
	int8 targetRssi;
	uint8 ulMcs;
#endif

#ifdef	PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
	uint32 newMaxPsduLength = 0;
#endif
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
		tcraPtr = (TcrCommon_t*)&dlDataPhasePtr->commonTcra;
		dlUserPhase = (DlDataPhaseUserPhase_t*)&dlDataPhasePtr->userPhase[uspIndex];
		userTcraStaAid = dlUserPhase->userTcraStaAid;
		ILOG0_DDD("[setDataPhaseUserTcrRcrParams], dlUserPhase = %x, uspIndex = %d, userTcraStaAid = %d", dlUserPhase, uspIndex, userTcraStaAid);
		if (userTcraStaAid != DUMMY_AID)
		{
			staId = dlUserPhase->staId;
			DEBUG_ASSERT(staId < HW_NUM_OF_STATIONS);
			
			dlUserPhase->userTcraPacketExtension = 2; // 16usec
			dlUserPhase->ampduLimit = StaDbHwEntries[staId].common.aMpduLimit;
			dlUserPhase->currMaxMsduAtAmsdu = StaDbHwEntries[staId].common.maxMsduAtAmsduCount;
			overrideLdpcForGroup(dlDataPhasePtr, isDl, uspIndex, ldpcCapable);
#ifdef ENET_INC_ARCH_WAVE600D2
			dlUserPhase->userTcraHeSigbCompressed = LinkAdaptationStaDatabase[staId].laStationUnique.heSigBParams.heSigbCompressed;
			dlUserPhase->userTcraHeSigb16Symbols = LinkAdaptationStaDatabase[staId].laStationUnique.heSigBParams.heSigb16Symbols;
#endif
			if (staticGroup == TRUE)
			{
				overrideRateForStaticGroup(dlDataPhasePtr, isDl, uspIndex);
			}
			else
			{
				// Take DL rate from SU. In case of MIMO, NSS will be limited later (limitNssInTcrForHeMuMimo)
				rateIndex = GetStaWpRateIndexFromHwTcr(staId, tcraPtr->cbw);
#ifdef DYNAMIC_GROUPING_DEBUG
                ILOG0_D("[setDataPhaseUserTcrRcrParams], rate: %d", rateIndex);
#endif
				rateIndex = MAX(MIN_HE_RATE_INDEX, rateIndex); //Phy mode can not be 11B in HE MU group
				getTcrPhyModeAndRate(heRatesTable, FALSE, rateIndex, &phyMode, &tcrDlRate, FALSE, NO_DCM);
				ASSERT(phyMode == PHY_MODE_11AX_SU);
			// this is a private case of MU DL with BW 160 and only one user in phase
#ifdef	PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
				// During group creation, check if special case of group with one user which RU size is 160 and init rate requires limitation 
				if((PRE_AGGREGATOR_996X2_TONE_RU_SIZE ==  dlUserPhase->userTcraRuSize) && dlUserPhase->userTcraLdpc)
				{
					if(isPsduLengthLimitationNeeded(tcrDlRate, &newMaxPsduLength) == TRUE)
					{
						originalMuMaxPsduLengthLimit[staId] = dlDataPhasePtr->common.currentPsduMaxLengthLimit;
						// apply minimum in case the current limit is lower than the limitation recommended by system
						dlDataPhasePtr->common.currentPsduMaxLengthLimit = MIN(newMaxPsduLength, dlDataPhasePtr->common.currentPsduMaxLengthLimit);
						ILOG0_DDDD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: Group creation. tcrRate %d, currentPsduMaxLengthLimit %d, new limit %d, staId %d", 
							tcrDlRate, dlDataPhasePtr->common.currentPsduMaxLengthLimit, newMaxPsduLength, staId);
					}
				}
#endif
				dlUserPhase->userTcraPsduRate = tcrDlRate;
				dlUserPhase->tfPadding = getTfPaddingCapability(staId);

				ILOG0_DD("[setDataPhaseUserTcrRcrParams], userTcraStaAid = %d, dlUserPhase->userTcraLdpc = %d", userTcraStaAid, dlUserPhase->userTcraLdpc);
				ILOG0_DD("[setDataPhaseUserTcrRcrParams], userTcraStaAid = %d, dlUserPhase->userRcrLdpc = %d", userTcraStaAid, dlUserPhase->userRcrLdpc);
				
				/* Set UL power control params */
				ruSize = dlUserPhase->userRcrRuSize;
#ifdef DYNAMIC_GROUPING_DEBUG
				dlUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(4, UL_NSS, NO_DCM); // dynamic grouuping debug - TBD delete
				userNss = LinkAdaptationGetMaxNssInMask(LinkAdaptationStaDatabase[staId].laStaUspCommon.raIndexMask.raIndexMask64bit, TRUE); // dynamic grouuping debug - TBD delete
				dlUserPhase->userTcraPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(11, userNss, NO_DCM); // dynamic grouuping debug - TBD delete	             
				dlUserPhase->userRcrTargetRssi = 70; // dynamic grouuping debug - TBD delete
#else
				lnaType = LinkAdaptationStaDatabase[staId].laStationUnique.ulPcParams.lnaType;
				DEBUG_ASSERT(lnaType != LNA_TYPE_NONE);
				normalizedRssi = LinkAdaptationStaDatabase[staId].laStationUnique.ulPcParams.normalizedRssi;
				if (lnaType == LNA_TYPE_HIGH)
				{
					ULPC_calcMcsAndTargetRssiHighLna(normalizedRssi, ruSize, &ulMcs,&targetRssi);
				}
				else
				{
					ULPC_calcMcsAndTargetRssiLowLna(normalizedRssi, ruSize, &ulMcs,&targetRssi);
				}
				dlUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(ulMcs, UL_NSS, NO_DCM); 
				userNss = LinkAdaptationGetMaxNssInMask(LinkAdaptationStaDatabase[staId].laStaUspCommon.raIndexMask.raIndexMask64bit, TRUE); // dynamic grouuping debug - TBD delete
				dlUserPhase->userTcraPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(11, userNss, NO_DCM); // dynamic grouuping debug - TBD delete	             
				dlUserPhase->userRcrTargetRssi = targetRssi;
#endif
			}
		}
	}
	else // UL
	{	
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;		
		ulUserPhase = (UlDataPhaseUserPhase_t*)&ulDataPhasePtr->userPhase[uspIndex];
		staId = ulUserPhase->staId;
		overrideLdpcForGroup(ulDataPhasePtr, isDl, uspIndex, ldpcCapable);
		if (staticGroup == TRUE)
		{
			overrideRateForStaticGroup(ulDataPhasePtr, isDl, uspIndex);						
		}
		else
		{
			ulUserPhase->tfPadding = getTfPaddingCapability(staId);

			/* Set UL power control params */
			ruSize = ulUserPhase->userRcrRuSize;
#ifdef DYNAMIC_GROUPING_DEBUG
			ulUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(7, UL_NSS, NO_DCM); // dynamic grouuping debug - TBD delete
			ulUserPhase->userRcrTargetRssi = 70; // dynamic grouuping debug - TBD delete
#else
			lnaType = LinkAdaptationStaDatabase[staId].laStationUnique.ulPcParams.lnaType;
			DEBUG_ASSERT(lnaType != LNA_TYPE_NONE);
			normalizedRssi = LinkAdaptationStaDatabase[staId].laStationUnique.ulPcParams.normalizedRssi;
			if (lnaType == LNA_TYPE_HIGH)
			{
				ULPC_calcMcsAndTargetRssiHighLna(normalizedRssi, ruSize, &ulMcs,&targetRssi);
			}
			else
			{
				ULPC_calcMcsAndTargetRssiLowLna(normalizedRssi, ruSize, &ulMcs,&targetRssi);
			}
			ulUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(ulMcs, UL_NSS, NO_DCM);
			ulUserPhase->userRcrTargetRssi = targetRssi;
#endif
		}
	}
}

static void setDataPhaseCommonUserTcrATcrBParams(void* phasePtr, uint8 vapId, bool isDl, bool staticGroup)
{
	VapDbCommon_t*   pVapDbCommon = &VapDbHwEntries[vapId].common;
	DlDataPhase_t* dlDataPhasePtr = NULL;
	UlDataPhase_t* ulDataPhasePtr = NULL;
	uint8 tcrRateVal = 0;

	/* Use hardcoded MCS 0 - need to understand why basicRates mask includes only 11B rates*/
	
	//LmVapDbSwInfo_t* pLmVapDataBase = &LmVapDataBase[vapId];
	//RateMask_t 		vapLegacyBasicRateMask;
	//uint8 rateIndex;
	//PhyMode_e phyMode;

	//vapLegacyBasicRateMask.raIndexMask64bit[0] = pLmVapDataBase->basicRates & LM_PHY_11A_RATE_MSK; // Remove all 11B rates from mask, since it's invalid for TF
	//vapLegacyBasicRateMask.raIndexMask64bit[1] = 0;
	//rateIndex = getLowestRateinMask(INVALID_STA_INDEX, vapLegacyBasicRateMask,BANDWIDTH_TWENTY,0,PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX); /*Get lower rate from vap basic rates*/
	//getTcrPhyModeAndRate(heRatesTable, 0, rateIndex, &phyMode, &tcrRateVal); 
	//DEBUG_ASSERT(PHY_MODE_11AG == phyMode);
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
		/* fill TCR B (used for broadcast BAR) from VAP DB (management 20MHz) */
		dlDataPhasePtr->commonUserTcrb.brdcstUserTcrbPsduRate = tcrRateVal;
		if (staticGroup == FALSE)
		{
			dlDataPhasePtr->commonUserTcrb.brdcstUserTcrbLdpc = pVapDbCommon->tcr280211Ldpc;
		}
	}
	else
	{
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		/* fill TCR A&B (used for Basic TF & BA) from VAP DB (management 20MHz) */
		ulDataPhasePtr->commonUserTcra.brdcstUserTcraPsduRate = tcrRateVal;
		ulDataPhasePtr->commonUserTcra.brdcstUserTcraPacketExtension = 2; //16 usec

		ulDataPhasePtr->commonUserTcrb.brdcstUserTcrbPsduRate = MCS_4;
		ulDataPhasePtr->commonUserTcrb.brdcstUserTcrbPacketExtension = 2; //16 usec

		if (staticGroup == FALSE)
		{
			ulDataPhasePtr->commonUserTcra.brdcstUserTcraLdpc = pVapDbCommon->tcr280211Ldpc;
			ulDataPhasePtr->commonUserTcrb.brdcstUserTcrbLdpc = pVapDbCommon->tcr280211Ldpc;
		}
	}
}

static void setDataPhaseTcrACommonParams(void* phasePtr, uint8 numOfValidUsers, uint8 vapId, bool isDl, bool staticGroup)
{
	TcrCommon_t* tcraPtr; 
	VapDbCommon_t* pVapDbCommon = &VapDbHwEntries[vapId].common;
	DlDataPhase_t* dlDataPhasePtr = NULL;
	UlDataPhase_t* ulDataPhasePtr = NULL;
	uint8 regulationLimit;
	DlMuMimo_e dlMuMimoType;
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
#ifdef ENET_INC_ARCH_WAVE600D2
		dlMuMimoType = dlDataPhasePtr->common.dataMuMimo;
#else
		dlMuMimoType = dlDataPhasePtr->common.muMimo;
#endif	
		tcraPtr = (TcrCommon_t*)&dlDataPhasePtr->commonTcra;

		// Validate that PHY mode in TCRA is AX MU DL
#ifdef DYNAMIC_GROUPING_DEBUG
		ILOG0_D("[setDataPhaseTcrACommonParams], tcraPtr->phyMode = %d",tcraPtr->phyMode);
#endif
		ASSERT(tcraPtr->phyMode == PHY_MODE_11AX_MU_DL);

		tcraPtr->antennaSelection = AntennaSelectionGetDefaultBitmap();
		tcraPtr->ant0Boost = 0;
		tcraPtr->ant1Boost = 0;
		tcraPtr->ant2Boost = 0;
		tcraPtr->ant3Boost = 0;
		tcraPtr->scp = 0;
		tcraPtr->smoothing = 1; //Should be enabled when txBf = 0 and Vice Versa
		tcraPtr->stbc = NOT_STBC_TRANSMISSION;
		tcraPtr->heSigRate = 0;

		//set NheLtf
		setNheLtfInTcr(dlDataPhasePtr, numOfValidUsers);
		
		if (dlMuMimoType == DL_MU_MIMO_NO_MU_MIMO_OR_HE_MU_OFDMA) //OFDMA
		{
			regulationLimit = getRegLimitPerAnt(tcraPtr->cbw);
		}
		else // MIMO
		{
			regulationLimit = getMuPowerLimit(tcraPtr->cbw);
		}
		
		if (staticGroup == FALSE)
		{
			tcraPtr->heCp = HE_CP_HE_1_6_CP;
			tcraPtr->heLtf = HE_LTF_HE_2X_LTF;
			tcraPtr->rfPower = powerAdaptationGetultimateEvm(tcraPtr->cbw);
            tcraPtr->rfPower = 26; //namic grouping debug - TBD delete
		}
		tcraPtr->rfPower = MIN(tcraPtr->rfPower, regulationLimit);
	}
	else //UL
	{
		/* fill TCR A (used for Basic TF) from VAP DB (management 20MHz) */
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		tcraPtr = (TcrCommon_t*)&ulDataPhasePtr->commonTcra;
		
		regulationLimit = getRegLimitPerAnt(tcraPtr->cbw);
		tcraPtr->antennaSelection = pVapDbCommon->tcr080211AntennaSelection;
		tcraPtr->ant0Boost = pVapDbCommon->tcr080211Ant0Boost;
		tcraPtr->ant1Boost = pVapDbCommon->tcr080211Ant1Boost;
		tcraPtr->ant2Boost = pVapDbCommon->tcr080211Ant2Boost;
		tcraPtr->ant3Boost = pVapDbCommon->tcr080211Ant3Boost;
		tcraPtr->smoothing = pVapDbCommon->tcr120Mhz80211Smoothing;
		tcraPtr->stbc = pVapDbCommon->tcr120Mhz80211Stbc;
		tcraPtr->scp = pVapDbCommon->tcr120Mhz80211Scp;
		if (staticGroup == FALSE)
		{
			tcraPtr->rfPower = pVapDbCommon->tcr08021120MhzRfPower;
		}
		tcraPtr->rfPower = MIN(tcraPtr->rfPower, regulationLimit);
	}
}

static void setNheLtfInTcr(DlDataPhase_t* dlDataPhase, uint8 numOfValidUsers)
{
	TcrCommon_t* tcraPtr = (TcrCommon_t*)&dlDataPhase->commonTcra; 
#ifndef ENET_INC_ARCH_WAVE600D2
	uint8 uspIndex; 
	DlDataPhaseUserPhase_t* userPhase;
	uint16 userTcraStaAid;
	uint8 userNss = 0;
	uint8 maxOfNssForAllUsers = 0;
	uint8 sumOfNssForAllUsers = 0;

	for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
	{
		userPhase = &dlDataPhase->userPhase[uspIndex];
		userTcraStaAid = userPhase->userTcraStaAid;
		if (userTcraStaAid != DUMMY_AID)
		{
			userNss = EXTRACT_NSS_FROM_VHT_HE_RATE(userPhase->userTcraPsduRate) + 1;  // Assuming that psdu rate is already set
			sumOfNssForAllUsers += userNss;
			ILOG0_D("[setNheLtfInTcr], userNss = %d",userNss);
			if (userNss > maxOfNssForAllUsers)
			{
				maxOfNssForAllUsers = userNss;
			}
		}
	}
	ILOG0_DD("[setNheLtfInTcr], maxOfNssForAllUsers = %d, sumOfNssForAllUsers = %d", maxOfNssForAllUsers, sumOfNssForAllUsers);

	if (dlDataPhase->common.muMimo == DL_MU_MIMO_NO_MU_MIMO_OR_HE_MU_OFDMA) //OFDMA
	{
		tcraPtr->nHeltf = convertNssToNHeLtf[maxOfNssForAllUsers - 1];
	}
	else // MIMO
	{
		tcraPtr->nHeltf = convertNssToNHeLtf[sumOfNssForAllUsers - 1];					
	}
#else
	UNUSED_PARAM(numOfValidUsers);
	tcraPtr->nHeltf = 0; // Will be calculated by PHY HW
#endif
	ILOG0_D("[setNheLtfInTcr], tcraPtr->nHeltf = %d",tcraPtr->nHeltf);
}

static void setDataPhaseTcrBCommonParams(void* phasePtr, uint8 vapId, bool isDl, bool staticGroup)
{
	TcrCommon_t* tcrbPtr;
	VapDbCommon_t* pVapDbCommon = &VapDbHwEntries[vapId].common;
	DlDataPhase_t* dlDataPhasePtr = NULL;
	UlDataPhase_t* ulDataPhasePtr = NULL;
	uint8 regulationLimit;

	if (isDl)
	{
		/* fill TCR B (used for broadcast BAR) from VAP DB (management 20MHz) */
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
		tcrbPtr = (TcrCommon_t*)&dlDataPhasePtr->commonTcrb;
	}
	else
	{
		/* fill TCR B (used for BA) from VAP DB (management 20MHz) */
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		tcrbPtr = (TcrCommon_t*)&ulDataPhasePtr->commonTcrb;
	}
	regulationLimit = getRegLimitPerAnt(tcrbPtr->cbw);
	tcrbPtr->antennaSelection = pVapDbCommon->tcr080211AntennaSelection;
	tcrbPtr->ant0Boost = pVapDbCommon->tcr080211Ant0Boost;
	tcrbPtr->ant1Boost = pVapDbCommon->tcr080211Ant1Boost;
	tcrbPtr->ant2Boost = pVapDbCommon->tcr080211Ant2Boost;
	tcrbPtr->ant3Boost = pVapDbCommon->tcr080211Ant3Boost;
	tcrbPtr->smoothing = pVapDbCommon->tcr120Mhz80211Smoothing;
	tcrbPtr->stbc = pVapDbCommon->tcr120Mhz80211Stbc;		
	tcrbPtr->scp = pVapDbCommon->tcr120Mhz80211Scp;
	if (staticGroup == TRUE)
	{
		tcrbPtr->rfPower = MIN(tcrbPtr->rfPower, regulationLimit);
	}
	else
	{
		//tcrbPtr->cbw = pVapDbCommon->_80211BwLimit; // dynamic grouping debug - TBD (delete the comment)
		tcrbPtr->rfPower = pVapDbCommon->tcr08021120MhzRfPower;

		tcrbPtr->rfPower = 26;  // dynamic grouping debug - TBD delete
	}
}

static void setDataPhaseCommonParams(void* phasePtr, uint8 numOfValidUsers, bool isDl, bool staticGroup)
{
	uint8 ltfDuration, sigBduration;
	uint8 N;
	uint16 sigbBitsCount;
	TcrCommon_t* tcraPtr;
	DlDataPhase_t* dlDataPhasePtr = NULL;
	UNUSED_PARAM(staticGroup);
	DlMuMimo_e dlMuMimoType;
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
#ifdef ENET_INC_ARCH_WAVE600D2
		dlMuMimoType = dlDataPhasePtr->common.dataMuMimo;
#else
		dlMuMimoType = dlDataPhasePtr->common.muMimo;
#endif		
		tcraPtr  = (TcrCommon_t*) &dlDataPhasePtr->commonTcra;
		if (dlMuMimoType == DL_MU_MIMO_NO_MU_MIMO_OR_HE_MU_OFDMA) //OFDMA
		{
			N = (numOfValidUsers >> 1); 						  // Asumming that all RUs are 20 MHz!!!
			sigbBitsCount = (((N >> 1) * 52) + ((N % 2) * 31)); // User Specific Size
				
			if(tcraPtr->cbw == BANDWIDTH_ONE_HUNDRED_SIXTY) 	 // Assuming that cbw is already set
			{
				sigbBitsCount += 43;							 // Common Size
			}
			else if(tcraPtr->cbw == BANDWIDTH_EIGHTY)
			{
				sigbBitsCount += 27;							 // Common Size
			}
			else //20 or 40 
			{
				sigbBitsCount += 18;							 // Common Size
			}
		}
		else
		{	
			N = numOfValidUsers;
			sigbBitsCount = (((N >> 1) * 52) + ((N % 2) * 31)); 
		}
		ltfDuration = ((tcraPtr->nHeltf + 1) * (((1 << tcraPtr->heLtf) * 32) + ((tcraPtr->heCp + 1) * 8))/ 10); //ltfDuration  = (((N_HE_LTF+1)LTF*3.2)+HE_CP);
		sigBduration = CEILING(sigbBitsCount, convertMcsToDbps[tcraPtr->heSigRate]) * 4; // Assuming that heSigRate is already set
		dlDataPhasePtr->common.totalPreambleTime = ltfDuration + sigBduration + 36 + 16 ;	// 36micro+SIGB+HE LTF +packet extension (16micro) 
	}
}

static void setDataPhaseRcrCommonParams(void* phasePtr, uint8 numOfValidUsers, bool isDl, bool staticGroup)
{
	DlDataPhase_t* dlDataPhasePtr = NULL;
	UlDataPhase_t* ulDataPhasePtr = NULL;
	DlDataPhaseUserPhase_t* dlUserPhase = NULL;	
	UlDataPhaseUserPhase_t* ulUserPhase = NULL;
	RcrCommon_t* rcrCommonPtr;
	uint16 userTcraStaAid;
	uint8 uspIndex; 
	uint8 userNss;
	uint8 maxOfNssForAllUsers = 0;
	uint8 sumOfNssForAllUsers = 0;
	bool isOneUserLdpc = FALSE;
	uint8 muMimoType;

	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
#ifdef ENET_INC_ARCH_WAVE600D2
		muMimoType = dlDataPhasePtr->common.dataMuMimo;
#else
		muMimoType = dlDataPhasePtr->common.muMimo;
#endif		
		rcrCommonPtr = (RcrCommon_t*)&(dlDataPhasePtr->commonRcr);
		for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
		{
			dlUserPhase = &dlDataPhasePtr->userPhase[uspIndex];
			userTcraStaAid = dlUserPhase->userTcraStaAid;
			if (userTcraStaAid != DUMMY_AID)
			{
				userNss = EXTRACT_NSS_FROM_VHT_HE_RATE(dlUserPhase->userRcrPsduRate) + 1;  // Assuming that psdu rate is already set
				sumOfNssForAllUsers += userNss;
				if (userNss > maxOfNssForAllUsers)
				{
					maxOfNssForAllUsers = userNss;
				}
				if(dlUserPhase->userRcrLdpc == 1)
				{
					isOneUserLdpc = TRUE;
				}
			}
		}
		rcrCommonPtr->ldpcExtraSymbol = 0; 
#ifdef LDPC_EXTRA_SYMBOL_ON		
		if(isOneUserLdpc == TRUE)
		{
			rcrCommonPtr->ldpcExtraSymbol = 1; 
		}
#endif		
		if (muMimoType == DL_MU_MIMO_NO_MU_MIMO_OR_HE_MU_OFDMA) //OFDMA
		{
			rcrCommonPtr->nHeltf = convertNssToNHeLtf[maxOfNssForAllUsers - 1]; 
		}
		else
		{
			rcrCommonPtr->nHeltf = convertNssToNHeLtf[sumOfNssForAllUsers - 1];		
		}

		if (staticGroup == FALSE)
		{
			rcrCommonPtr->heCp = HE_CP_HE_1_6_CP;
			rcrCommonPtr->heLtf = HE_LTF_HE_2X_LTF;
			rcrCommonPtr->stbc = NOT_STBC_TRANSMISSION;
		}
	}
	else
	{
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		rcrCommonPtr = (RcrCommon_t*)&(ulDataPhasePtr->commonRcr);
		for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
		{
			ulUserPhase = &ulDataPhasePtr->userPhase[uspIndex];
			userTcraStaAid = ulUserPhase->userTcraStaAid;
			if (userTcraStaAid != DUMMY_AID)
			{
				userNss = EXTRACT_NSS_FROM_VHT_HE_RATE(ulUserPhase->userRcrPsduRate) + 1;  // Assuming that psdu rate is already set
				sumOfNssForAllUsers += userNss;
				if (userNss > maxOfNssForAllUsers)
				{
					maxOfNssForAllUsers = userNss;
				}
				if(ulUserPhase->userRcrLdpc == 1)
				{
					isOneUserLdpc = TRUE;
				}
			}
		}
		rcrCommonPtr->ldpcExtraSymbol = 0; 
#ifdef LDPC_EXTRA_SYMBOL_ON		
		if(isOneUserLdpc == TRUE)
		{
			rcrCommonPtr->ldpcExtraSymbol = 1; 
		}
#endif		
#ifdef ENET_INC_ARCH_WAVE600D2
		muMimoType = ulDataPhasePtr->common.dataMuMimo;
		if (muMimoType == UL_MU_MIMO_MU_MIMO) //UL MIMO
		{
			rcrCommonPtr->nHeltf = convertNssToNHeLtf[sumOfNssForAllUsers - 1]; 
		}
		else
#endif
		{
			rcrCommonPtr->nHeltf = convertNssToNHeLtf[maxOfNssForAllUsers - 1];		
		}

		if (staticGroup == FALSE)
		{
			rcrCommonPtr->heCp = HE_CP_HE_1_6_CP;
			rcrCommonPtr->heLtf = HE_LTF_HE_2X_LTF;
			rcrCommonPtr->stbc = NOT_STBC_TRANSMISSION;
		}
	}
}

static void setDataPhaseTfCommonInfoParams(void* phasePtr, bool isDl, bool staticGroup)
{
	RcrCommon_t* rcrCommonPtr; 
	DlDataPhase_t* dlDataPhasePtr = NULL;
	UlDataPhase_t* ulDataPhasePtr = NULL;
	uint8 tfApTxPower;
	TcrCommon_t* tcraPtr; 
	TcrCommon_t* tcrbPtr; 
	UNUSED_PARAM(staticGroup);
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
		rcrCommonPtr = (RcrCommon_t*)&(dlDataPhasePtr->commonRcr);
		tcrbPtr  = (TcrCommon_t*) &dlDataPhasePtr->commonTcrb; // TCR for TF transmission
		
		dlDataPhasePtr->tfCommonInfo.tfNumberOfHeLtfSymbols = rcrCommonPtr->nHeltf;
		dlDataPhasePtr->tfCommonInfo.tfPacketExtension = (rcrCommonPtr->peDisambiguty << 2);	// PE disambiguity = 1 [bit 2], pre-FEC padding factor = 0 (a=4) [bits 1:0]
		dlDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = 0;
#ifdef LDPC_EXTRA_SYMBOL_ON				
		// TODO: in A0 bug that all users should have the same coding if if fixed at least one user with userRcrLdpc=1 should set tfLdpcExtraSymbolSegment = 1
		if (dlDataPhasePtr->userPhase[0].userRcrLdpc == 1)
		{
			dlDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = 1;
		}
#endif		
		dlDataPhasePtr->tfCommonInfo.tfStbc = NOT_STBC_TRANSMISSION;
		tfApTxPower = calcTfApTxPower(tcrbPtr);
		dlDataPhasePtr->tfCommonInfo.tfApTxPower30 = (tfApTxPower & 0xF); //Set 4 LSBs  
		dlDataPhasePtr->tfCommonInfo.tfApTxPower54 = ((tfApTxPower >> 4) & 0x3); //Set 2 MSBs  
	}
	else
	{ //UL phase including BSRP phase
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		rcrCommonPtr = (RcrCommon_t*)&(ulDataPhasePtr->commonRcr);
		tcraPtr  = (TcrCommon_t*) &ulDataPhasePtr->commonTcra; // TCR for TF transmission
		
		ulDataPhasePtr->tfCommonInfo.tfNumberOfHeLtfSymbols = rcrCommonPtr->nHeltf;
		ulDataPhasePtr->tfCommonInfo.tfPacketExtension = (rcrCommonPtr->peDisambiguty << 2);	// PE disambiguity = 1 [bit 2], pre-FEC padding factor = 0 (a=4) [bits 1:0]
		ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = 0;
		
		ILOG0_D("[setDataPhaseTfCommonInfoParams]1, ulDataPhasePtr->userPhase[0].userRcrLdpc = %d", ulDataPhasePtr->userPhase[0].userRcrLdpc);
#ifdef LDPC_EXTRA_SYMBOL_ON						
		// TODO: in A0 bug that all users should have the same coding if if fixed at least one user with userRcrLdpc=1 should set tfLdpcExtraSymbolSegment = 1
		
		ILOG0_D("[setDataPhaseTfCommonInfoParams]2, ulDataPhasePtr->userPhase[0].userRcrLdpc = %d", ulDataPhasePtr->userPhase[0].userRcrLdpc);
		if (ulDataPhasePtr->userPhase[0].userRcrLdpc == 1)
		{			
			ILOG0_D("[setDataPhaseTfCommonInfoParams3], ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = %d", ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment);
			ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = 1;			
			ILOG0_D("[setDataPhaseTfCommonInfoParams4], ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = %d", ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment);
		}
#endif		
		ILOG0_D("[setDataPhaseTfCommonInfoParams], ulDataPhasePtr->userPhase[0].userRcrLdpc = %d", ulDataPhasePtr->userPhase[0].userRcrLdpc);
		ILOG0_D("[setDataPhaseTfCommonInfoParams], ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment = %d", ulDataPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment);

		ulDataPhasePtr->tfCommonInfo.tfStbc = NOT_STBC_TRANSMISSION;		
		tfApTxPower = calcTfApTxPower(tcraPtr);
		ulDataPhasePtr->tfCommonInfo.tfApTxPower30 = (tfApTxPower & 0xF);  //Set 4 LSBs  
		ulDataPhasePtr->tfCommonInfo.tfApTxPower54 = ((tfApTxPower >> 4) & 0x3); //Set 2 MSBs  
	}
}

static void setProtectionPhaseUserRcrParams(ProtectionPhase_t* phasePtr, uint8 uspIndex, bool staticGroup)
{
	ProtectionPhaseUserPhase_t* protectionUserPhase = NULL;		
	StaId staId;
	LnaType_e lnaType;
	int8 normalizedRssi;
	uint8 ruSize;
	uint8 ulMcs;
	int8 targetRssi;

	if (staticGroup == FALSE)
	{
		protectionUserPhase = (ProtectionPhaseUserPhase_t*)&phasePtr->userPhase[uspIndex];
		staId = protectionUserPhase->staId;
		
		protectionUserPhase->tfPadding = getTfPaddingCapability(staId);
		
		/* Set UL power control params */
		lnaType = LinkAdaptationStaDatabase[staId].laStationUnique.ulPcParams.lnaType;
		normalizedRssi = LinkAdaptationStaDatabase[staId].laStationUnique.ulPcParams.normalizedRssi;
		ruSize = protectionUserPhase->userRcrRuSize;
		DEBUG_ASSERT(lnaType != LNA_TYPE_NONE);
		if (lnaType == LNA_TYPE_HIGH)
		{
			ULPC_calcMcsAndTargetRssiHighLna(normalizedRssi, ruSize, &ulMcs,&targetRssi);
		}
		else
		{
			ULPC_calcMcsAndTargetRssiLowLna(normalizedRssi, ruSize, &ulMcs,&targetRssi);
		}
		protectionUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(ulMcs, UL_NSS, NO_DCM);
		protectionUserPhase->userRcrTargetRssi = targetRssi;
	}
}

static void setProtectionPhaseCommonUserTcrBParams(ProtectionPhase_t* phasePtr, uint8 vapId, bool staticGroup)
{
	UNUSED_PARAM(vapId);
	UNUSED_PARAM(staticGroup);

	phasePtr->commonUserTcrb.brdcstUserTcrbPsduRate = 0x0;
	phasePtr->commonUserTcrb.brdcstUserTcrbPacketExtension = 0x0; 
	phasePtr->commonUserTcrb.brdcstUserTcrbLdpc = 0x0;
	phasePtr->commonUserTcrb.brdcstUserTcrbTxbf = 0x0;
}

static void setProtectionPhaseTcrBCommonParams(ProtectionPhase_t* phasePtr, uint8 numOfValidUsers, uint8 vapId, bool staticGroup)
{
	VapDb_t*				pVapDbHwEntry = &VapDbHwEntries[vapId];
	TcrCommon_t*			tcrCommonTemplate = (TcrCommon_t*)pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word0;
	TcrCommon_t*			tcrbPtr	= (TcrCommon_t*)&phasePtr->commonTcrb;
	int8 regulationLimit;

	UNUSED_PARAM(numOfValidUsers);
	/* fill TCR B (used for MU-RTS) */
	tcrbPtr->nHeltf = 0; // N/A
	tcrbPtr->heCp = 0;
	tcrbPtr->heLtf = 0;
	tcrbPtr->heSigRate = 0; //N/A

	regulationLimit = getRegLimitPerAnt(tcrbPtr->cbw);
	tcrbPtr->antennaSelection = pVapDbHwEntry->common.tcr080211AntennaSelection;
	tcrbPtr->ant0Boost = pVapDbHwEntry->common.tcr080211Ant0Boost;
	tcrbPtr->ant1Boost = pVapDbHwEntry->common.tcr080211Ant1Boost;
	tcrbPtr->ant2Boost = pVapDbHwEntry->common.tcr080211Ant2Boost;
	tcrbPtr->ant3Boost = pVapDbHwEntry->common.tcr080211Ant3Boost;
	tcrbPtr->smoothing = pVapDbHwEntry->common.tcr120Mhz80211Smoothing;
	tcrbPtr->stbc = 0x0;		
	tcrbPtr->scp = 0x0;
	if (staticGroup == TRUE)
	{
		tcrbPtr->rfPower = MIN(tcrbPtr->rfPower, regulationLimit);
	}
	else
	{
		tcrbPtr->txLoopMode = tcrCommonTemplate->txLoopMode;
		tcrbPtr->phyMode = tcrCommonTemplate->phyMode;
		tcrbPtr->rfPower = pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].rfPower20MhzTemplate;
	}
}

static void setProtectionPhaseRcrCommonParams(ProtectionPhase_t* phasePtr, uint8 numOfValidUsers, bool staticGroup)
{
	RcrCommon_t* rcrCommonPtr = (RcrCommon_t*)&phasePtr->commonRcr;

	UNUSED_PARAM(numOfValidUsers);
	UNUSED_PARAM(staticGroup);		
	rcrCommonPtr->ldpcExtraSymbol = 0; 
	rcrCommonPtr->nHeltf = 0;
}
void getHeMuTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams)
{
	DlDataPhase_t* phasePtr = getWpPhaseAddress(tcrModificationParams->controlParams.staIndex);
	TcrCommon_t* tcraPtr = (TcrCommon_t*)&phasePtr->commonTcra; 
	DlDataPhaseUserPhase_t* dlUserPhase = (DlDataPhaseUserPhase_t*)&phasePtr->userPhase[tcrModificationParams->controlParams.uspIndex];
	Bw_e bw = tcraPtr->cbw;
		
	ASSERT(tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA);

	/* General */
	tcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection = tcraPtr->antennaSelection;
	tcrModificationParams->tcrParams.tcrGeneralVals.ldpcMode = dlUserPhase->userTcraLdpc;

	ILOG0_DD("[getHeMuTcrParamsFromHwDb], tcrModificationParams->controlParams.uspIndex = %d, dlUserPhase->userTcraLdpc = %d", tcrModificationParams->controlParams.uspIndex, dlUserPhase->userTcraLdpc);
	
	/* Per BW */
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].rateindex = convertTcr2RateIndex(PHY_MODE_11AX_SU, dlUserPhase->userTcraPsduRate);
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].tcrPower = tcraPtr->rfPower; 
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].cpMode = convertHeCpLtfTcrValToCpMode(tcraPtr->heCp, tcraPtr->heLtf); 
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].tcrBfMode = dlUserPhase->userTcraTxbf;	
}

void setTcrsInPhaseDb(LaTcrModificationStruct_t* pTcrModificationParams)
{
	DlDataPhase_t* phasePtr = getWpPhaseAddress(pTcrModificationParams->controlParams.staIndex);
	TcrCommon_t* tcraPtr; 
	DlDataPhaseUserPhase_t* dlUserPhase;
	uint8 tcrPhyMode;
	uint8 rateIndex;
	uint8 tcrRate;
	uint8 uspIndex = pTcrModificationParams->controlParams.uspIndex;
	const RateObj_t* ratesTable;
	uint8 maxNumOfRatesInTable;
	HeCp_e tcrCp;
	HeLtf_e tcrLtf;
	Bw_e bw;
	uint8 numOfValidUsers = getNumOfValidUspInPhase(phasePtr);
	bool isDummyUser = FALSE;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
	uint32 newMaxPsduLength = 0;
	uint32 tmpMaxPsduLength = 0;
#endif 	

	ASSERT(uspIndex != INVALID_MU_USP_INDEX); 
	ASSERT(pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA); 
	ASSERT(uspIndex < MAX_USP_IN_HE_GROUP);
	
	updateLaDbDistributionParam(&laDbDistributionParameter, pTcrModificationParams->controlParams.staIndex, uspIndex, pTcrModificationParams->controlParams.isHeGroup);
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);		
	maxNumOfRatesInTable = getMaxNumOfRatesInTable(laDbDistributionParameter.laStaUspCommon);

	isDummyUser = (phasePtr->userPhase[uspIndex].userTcraStaAid == DUMMY_AID) ? TRUE : FALSE;

	if (isDummyUser == FALSE)
	{
		if (laDbDistributionParameter.laGroupUnique->groupState != LA_INACTIVE_GROUP_STATE)
		{
			/* Change TCRs on WP phase only at init (linkAdaptationAddHeDlDataPhase). When group is active - set TCRs on the mirror phase */
			phasePtr = (DlDataPhase_t*)phasePtr->common.nextOrMirrorPhasePointer;
		}
		tcraPtr = (TcrCommon_t*)&phasePtr->commonTcra; 
		dlUserPhase = (DlDataPhaseUserPhase_t*)&phasePtr->userPhase[uspIndex];
		bw = tcraPtr->cbw;

		tcrCp = cpModeToHeCpTcrVal[pTcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].cpMode];
		tcrLtf = cpModeToHeLtfTcrVal[pTcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].cpMode];
		rateIndex = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[bw].rateindex;
		DEBUG_ASSERT(rateIndex < maxNumOfRatesInTable);
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta, rateIndex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
	
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400	
	// case of group with only one user which RU size is 160 and ldpc is on. 
	if((PRE_AGGREGATOR_996X2_TONE_RU_SIZE ==  dlUserPhase->userTcraRuSize) && dlUserPhase->userTcraLdpc)
	{
		// if new rate requires limitation of PsduMaxLength
		if(isPsduLengthLimitationNeeded(tcrRate, &newMaxPsduLength) == TRUE)
		{
			// check if current tcrRate required limitation in currentPsduMaxLengthLimit
			if(isPsduLengthLimitationNeeded(dlUserPhase->userTcraPsduRate, &tmpMaxPsduLength) == TRUE)
			{
				// no need to save the original value of this field because it was save already when current tcrRate was set to problematic rate
				//NOTE: in case we move from one problematic rate to another one and original rate was lower than the limitation recommended by system in both problematic rates
				// we might set here a higher limit than it was calculated initially for this station. Need to check how to avoid this!
				phasePtr->common.currentPsduMaxLengthLimit = newMaxPsduLength;
				ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setTcrsInPhaseDb- current rate already limited: new Rate %d, currentRate %d",tcrRate, dlUserPhase->userTcraPsduRate);
			}
			// if moving from rate which doesn't require limitation in PsduMaxLengthLimit to the one that does
			else
			{
				// save original value and update to the new limit
				originalMuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex] = phasePtr->common.currentPsduMaxLengthLimit;
				// apply minimum in case original maxLimit is lower than the limit recommended by system
				phasePtr->common.currentPsduMaxLengthLimit = MIN(newMaxPsduLength, phasePtr->common.currentPsduMaxLengthLimit);
				ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setTcrsInPhaseDb - current rate is good %d, moving to limit rate%d", dlUserPhase->userTcraPsduRate, tcrRate);
				
			}
		}
		else
		{
			// check if moving to the rate that doesn't require limitation in PsduMaxLengthLimit - restore the original value
			if(isPsduLengthLimitationNeeded(dlUserPhase->userTcraPsduRate, &newMaxPsduLength) == TRUE)
			{
				phasePtr->common.currentPsduMaxLengthLimit = originalMuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex];
				ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setTcrsInPhaseDb - new rate is good %d, current rate %d must be limited", tcrRate, dlUserPhase->userTcraPsduRate);
			}
		}
			
	}
	
#endif

		dlUserPhase->userTcraPsduRate = tcrRate;

		//set tcraPtr->nHeLtf according to NSS 
		setNheLtfInTcr(phasePtr, numOfValidUsers);
		
		tcraPtr->scp = 0;
		tcraPtr->heCp = tcrCp;
		tcraPtr->heLtf = tcrLtf;
	}
}

DlDataPhase_t* getWpPhaseAddress(uint8 laPhaseId)
{
	return LinkAdaptationHePhaseDatabase[laPhaseId].laHeGroupUnique.wpPhasePtr;
}

void setWpPhaseAddress(DlDataPhase_t* phaseAddr, uint8 laPhaseId)
{
	LinkAdaptationHePhaseDatabase[laPhaseId].laHeGroupUnique.wpPhasePtr = phaseAddr;
}

uint8 getLaPhaseIdFromPlanIndex(uint8 planIndex, uint8 phaseOffset)
{
	uint32* phase0Addr;
	CommonPlan_t* planPtr = &heMuPlansArray[planIndex];
	DlDataPhase_t* phasePtr;
	uint8 laPhaseId;
	
	phase0Addr = &planPtr->word2;
	phasePtr = (DlDataPhase_t*)(phase0Addr[phaseOffset]); 
	laPhaseId = phasePtr->common.dlPhaseId;
	ASSERT(laPhaseId  < NUM_OF_LA_HE_MU_DB_ENTRIES);
	
	return laPhaseId;
}

PhaseType_e getPhaseTypeFromPlanIndex(uint8 planIndex, uint8 phaseOffset)
{
	uint32* phase0Addr;
	CommonPlan_t* planPtr = &heMuPlansArray[planIndex];
	DlDataPhase_t* phasePtr;
	
	phase0Addr = &planPtr->word2;
	phasePtr = (DlDataPhase_t*)(phase0Addr[phaseOffset]); 
	
	return phasePtr->common.phaseType;
}
void replacePtrInPlanToMirrorPhase(uint8 laPhaseId, uint8 phaseOffsetInPlan)
{
	DlDataPhase_t* phasePtr = getWpPhaseAddress(laPhaseId);
	CommonPlan_t* planPtr = &heMuPlansArray[LinkAdaptationHePhaseDatabase[laPhaseId].laHeGroupUnique.planIndex];
	uint32* phase0Addr = &planPtr->word2;

	/* Change pointer in plan to point to the mirror phase  */
	phase0Addr[phaseOffsetInPlan] = (uint32)phasePtr->common.nextOrMirrorPhasePointer;

}

void heMuWorkingPointIsChanged(uint8 laPhaseId)
{
	DlDataPhase_t* phasePtr = getWpPhaseAddress(laPhaseId);
	DlDataPhase_t* mirrorPhasePtr = (DlDataPhase_t*)phasePtr->common.nextOrMirrorPhasePointer;
	uint16 phaseEntriesBitmap = LinkAdaptationHePhaseDatabase[laPhaseId].laHeGroupUnique.phaseEntriesBitmap;
	uint8 phaseIterator;

	ILOG0_D("[heMuWorkingPointIsChanged] laPhaseId %d", laPhaseId);
	/* Prepare mirror phase to become WP phase - clear one shot bit */
	mirrorPhasePtr->common.oneShotPhase = 0;
	/* change all relevant pointers in plan to point to the mirror phase */
	for (phaseIterator = 0; phaseIterator < MAX_NUM_OF_PHASES_IN_A_PLAN; phaseIterator++)
	{
		if (GET_BIT_IN_BYTE(phaseEntriesBitmap, phaseIterator) != 0)
		{
			replacePtrInPlanToMirrorPhase(laPhaseId, phaseIterator);
		}
	}
	/* Prepare WP phase to become mirror phase - set one shot bit */
	phasePtr->common.oneShotPhase = 1;	
	setWpPhaseAddress(mirrorPhasePtr, laPhaseId);
}
uint8 getNumOfValidUspInPhase(void* phasePtr)
{
	DlDataPhase_t* dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
	uint8 numOfValidUsp = 0;

	if (dlDataPhasePtr->common.validUsers310 != 0)
	{
		numOfValidUsp = Utils_FindFirstSet(dlDataPhasePtr->common.validUsers310) + 1;
	}
	if (dlDataPhasePtr->common.validUsers3532 != 0)
	{
		numOfValidUsp += Utils_FindFirstSet(dlDataPhasePtr->common.validUsers3532) + 1;
	}
	ASSERT(numOfValidUsp <= MAX_USP_IN_HE_GROUP);
	
	return numOfValidUsp;
}

static uint8 calcTfApTxPower(TcrCommon_t* pTcr)
{
	uint8 tfApTxPower;
	int8 rfPower = pTcr->rfPower;
	Bw_e bw = pTcr->cbw;
	uint8 antennaCount = AntennaSelectionCalcAntennaCount(pTcr->antennaSelection);

	tfApTxPower = ((rfPower >> 1) + POWER_OFFSET_IN_DB_FROM_TCR_VALUE) - (bw * 3) + (AntennaCountOffset[antennaCount-1] >> 1); // convert to 1 dB resolution, + offset from TCR val, normalized to 20 MHz BW, +3dB per antenna ( 0 to 60 indicates -20 to 40 dBm)

	return tfApTxPower;
}

static void overrideRateForStaticGroup(void* phasePtr, bool isDl, uint8 uspIndex)
{
	DlDataPhase_t* dlDataPhasePtr = NULL;
	DlDataPhaseUserPhase_t* dlUserPhase = NULL; 	
	UlDataPhase_t* ulDataPhasePtr = NULL;
	UlDataPhaseUserPhase_t* ulUserPhase = NULL;
	StaId staId;
	uint8 tcrRequestedNss = 0;
	uint8 tcrRequestedMcs = 0;
	uint8 rcrRequestedNss = 0;
	uint8 rcrRequestedMcs = 0;
	uint8 maxNss = 0;
	uint8 tcrMaxMcs = 0; // max MCS for the chosen TCR NSS
	uint8 rcrMaxMcs = 0; // max MCS for the chosen RCR NSS
	uint8 tcrChosenNss = 0;
	uint8 tcrChosenMcs = 0;
	uint8 rcrChosenNss = 0;
	uint8 rcrChosenMcs = 0;
	uint8 tcrRateBefore = 0;
	uint8 rcrRateBefore = 0;
	TcrCommon_t* commonTcraPtr;
	RcrCommon_t* commonRcraPtr;
	Bandwidth_e requiredBw = BANDWIDTH_TWENTY;
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
		dlUserPhase = (DlDataPhaseUserPhase_t*)&dlDataPhasePtr->userPhase[uspIndex];
		staId = dlUserPhase->staId;
		tcrRateBefore = dlUserPhase->userTcraPsduRate;				
		rcrRateBefore = dlUserPhase->userRcrPsduRate;
		tcrRequestedNss = EXTRACT_NSS_FROM_VHT_HE_RATE(tcrRateBefore);			
		tcrRequestedMcs = EXTRACT_MCS_FROM_VHT_HE_RATE(tcrRateBefore);	
		commonTcraPtr  = (TcrCommon_t*) &dlDataPhasePtr->commonTcra;
		requiredBw = (Bandwidth_e)commonTcraPtr->cbw;
	}
	else
	{
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		ulUserPhase = (UlDataPhaseUserPhase_t*)&ulDataPhasePtr->userPhase[uspIndex];
		staId = ulUserPhase->staId;
		rcrRateBefore = ulUserPhase->userRcrPsduRate;
		commonRcraPtr  = (RcrCommon_t*) &ulDataPhasePtr->commonTcra;
		requiredBw = (Bandwidth_e)commonRcraPtr->cbw;

	}
	
	rcrRequestedNss = EXTRACT_NSS_FROM_VHT_HE_RATE(rcrRateBefore);	
	rcrRequestedMcs = EXTRACT_MCS_FROM_VHT_HE_RATE(rcrRateBefore);	
	getHighestNssInMask(staId, &maxNss, requiredBw);
	
	if (isDl)
	{
		tcrChosenNss = MIN(maxNss, tcrRequestedNss);		
		getHighestMcsForNssInMask(staId, tcrChosenNss, &tcrMaxMcs, PHY_MODE_11AX_MU_DL,requiredBw);		
		tcrChosenMcs = MIN(tcrMaxMcs, tcrRequestedMcs);
		ILOG0_DD("[overrideRateForStaticGroup], tcrRequestedNss %d, tcrRequestedMcs %d", tcrRequestedNss, tcrRequestedMcs);		
		ILOG0_DD("[overrideRateForStaticGroup], tcrChosenNss %d, tcrChosenMcs %d", tcrChosenNss, tcrChosenMcs);
	}
	rcrChosenNss = MIN(maxNss, rcrRequestedNss);
	getHighestMcsForNssInMask(staId, rcrChosenNss, &rcrMaxMcs, PHY_MODE_11AX_MU_DL,requiredBw);
	rcrChosenMcs = MIN(rcrMaxMcs, rcrRequestedMcs);
	
	ILOG0_DD("[overrideRateForStaticGroup], maxNss %d, isDl = %d", maxNss, isDl);
	ILOG0_DD("[overrideRateForStaticGroup], rcrRequestedNss %d, rcrRequestedMcs %d", rcrRequestedNss, rcrRequestedMcs);
	ILOG0_DD("[overrideRateForStaticGroup], rcrChosenNss %d, rcrChosenMcs %d", rcrChosenNss, rcrChosenMcs);
	
	if (isDl)
	{
		// Override rate in phase TCR
		dlUserPhase->userTcraPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(tcrChosenMcs, tcrChosenNss, NO_DCM);		
		dlUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(rcrChosenMcs, rcrChosenNss, NO_DCM);
	}
	else
	{
		// Override rate in phase RCR
		ulUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(rcrChosenMcs, rcrChosenNss, NO_DCM);
	}
}
static void overrideLdpcForGroup(void* phasePtr, bool isDl, uint8 uspIndex, bool ldpcCapable)
{
	DlDataPhase_t* dlDataPhasePtr = NULL;
	DlDataPhaseUserPhase_t* dlUserPhase = NULL; 	
	UlDataPhase_t* ulDataPhasePtr = NULL;
	UlDataPhaseUserPhase_t* ulUserPhase = NULL;
	
	if (isDl)
	{
		dlDataPhasePtr = (DlDataPhase_t*)phasePtr;
		dlUserPhase = (DlDataPhaseUserPhase_t*)&dlDataPhasePtr->userPhase[uspIndex];	
		/* If LDPC is set in plan, check if STA is LDPC capbale, and override TCR, RCR and TF to BCC otherwise */
		if (dlUserPhase->userTcraLdpc == LDPC)
		{
			dlUserPhase->userTcraLdpc = ldpcCapable;
		}
		if (dlUserPhase->userRcrLdpc == LDPC)
		{
			dlUserPhase->userRcrLdpc = ldpcCapable;
			if (uspIndex % 2 == 0)
			{
#ifdef ENET_INC_ARCH_WAVE600D2				
				dlDataPhasePtr->tfUserInfo[uspIndex/2].tfFecCodingType0 = ldpcCapable;
#else
				dlDataPhasePtr->tfUserInfo[uspIndex/2].tfCodingType0 = ldpcCapable;
#endif
			}
			else
			{
#ifdef ENET_INC_ARCH_WAVE600D2				
				dlDataPhasePtr->tfUserInfo[uspIndex/2].tfFecCodingType1 = ldpcCapable;
#else
				dlDataPhasePtr->tfUserInfo[uspIndex/2].tfCodingType1 = ldpcCapable;
#endif
			}
			
		}
	}
	else
	{
		ulDataPhasePtr = (UlDataPhase_t*)phasePtr;
		ulUserPhase = (UlDataPhaseUserPhase_t*)&ulDataPhasePtr->userPhase[uspIndex];
		/* If LDPC is set in plan, check if STA is LDPC capbale, and override TCR, RCR and TF to BCC otherwise */
		if (ulUserPhase->userTcraLdpc == LDPC)
		{
			ulUserPhase->userTcraLdpc = ldpcCapable;
		}
		if (ulUserPhase->userRcrLdpc == LDPC)
		{
			ulUserPhase->userRcrLdpc = ldpcCapable;
			if (uspIndex % 2 == 0)
			{
#ifdef ENET_INC_ARCH_WAVE600D2				
			    ulDataPhasePtr->tfUserInfo[uspIndex/2].tfFecCodingType0 = ldpcCapable;
#else
				ulDataPhasePtr->tfUserInfo[uspIndex/2].tfCodingType0 = ldpcCapable;
#endif
			}
			else
			{
#ifdef ENET_INC_ARCH_WAVE600D2				
				ulDataPhasePtr->tfUserInfo[uspIndex/2].tfFecCodingType1 = ldpcCapable;
#else
				ulDataPhasePtr->tfUserInfo[uspIndex/2].tfCodingType1 = ldpcCapable;
#endif
			}
		}
	}
}

void overrideUlPhaseAfterOmiRx(LaUlprReport_t * ulprReport, uint32 planIndex, uint8 hePhaseIndex)
{
	CommonPlan_t* planPtr = &heMuPlansArray[planIndex];
	uint32* phase0Addr;
	UlDataPhase_t* ulPhasePtr;
	RcrCommon_t* rcrCommonPtr;
#ifdef OMI_BW_FOR_ONE_USER_WORKAROUND
	TcrCommon_t* phaseCommonDataTcrPtr;
#endif
	UlDataPhaseTfCommonInfo_t* tfCommonInfo;
	uint8 numOfValidUsers;
	uint8 uspIndex;
	UlDataPhaseUserPhase_t* ulUserPhase;
	uint16 userTcraStaAid;
	uint8 userNss;
	uint8 maxOfNssForAllUsers = 0;
	uint8 mcs;

	if (GET_BIT_IN_BYTE(planPtr->phaseValid, DATA_PHASE_BIT_INDEX) != 0)
	{		
		phase0Addr = &planPtr->word2;
		ulPhasePtr = (UlDataPhase_t*)(phase0Addr[hePhaseIndex]);		
		if (ulPhasePtr->common.phaseType == PHASE_TYPE_UL_DATA_PHASE)
		{
			rcrCommonPtr = (RcrCommon_t*)&(ulPhasePtr->commonRcr); 
#ifdef OMI_BW_FOR_ONE_USER_WORKAROUND
			phaseCommonDataTcrPtr = (TcrCommon_t*)&(ulPhasePtr->commonTcra);
#endif
			tfCommonInfo = &(ulPhasePtr->tfCommonInfo);
			numOfValidUsers = getNumOfValidUspInPhase(ulPhasePtr);
			ulUserPhase = &ulPhasePtr->userPhase[ulprReport->ulprUserReport.rxUserId];
#ifdef OMI_BW_FOR_ONE_USER_WORKAROUND
			/* Override BW and ru Size according to OMI BW*/	
			if (numOfValidUsers == 1)
			{
				planPtr->startBw = ulprReport->omiControl.bitFields.channelWidth;
				phaseCommonDataTcrPtr->cbw = ulprReport->omiControl.bitFields.channelWidth;
				rcrCommonPtr->cbw = ulprReport->omiControl.bitFields.channelWidth;
				ulUserPhase->userRcrRuSize = CONVERT_BW_TO_RU_SIZE(rcrCommonPtr->cbw);		
				ILOG0_DD("[overrideUlPhaseAfterOmiRx] plan BW %d ruSize %d", planPtr->startBw, ulUserPhase->userRcrRuSize);
				ILOG0_DD("[overrideUlPhaseAfterOmiRx] planIndex %d hePhaseIndex %d", planIndex, hePhaseIndex);
			}
#endif
	
			/* Override UL nss and nHeLtf according to OMI NSS*/	
			mcs = EXTRACT_MCS_FROM_VHT_HE_RATE(ulUserPhase->userRcrPsduRate);
			ulUserPhase->userRcrPsduRate = RATE_PARAMS_TO_TCR_VAL_VHT_HE(mcs,ulprReport->omiControl.bitFields.txNsts,NO_DCM); // txNsts refer for STA Tx (UL)
			ILOG0_D("[overrideUlPhaseAfterOmiRx] userRcrPsduRate %d", ulUserPhase->userRcrPsduRate);
			
			for(uspIndex = 0; uspIndex < numOfValidUsers; uspIndex++)
			{
				ulUserPhase = &ulPhasePtr->userPhase[uspIndex];
				userTcraStaAid = ulUserPhase->userTcraStaAid;
				if (userTcraStaAid != DUMMY_AID)
				{
					userNss = EXTRACT_NSS_FROM_VHT_HE_RATE(ulUserPhase->userRcrPsduRate); 
					if (userNss > maxOfNssForAllUsers)
					{
						maxOfNssForAllUsers = userNss;
					}
				}
			}
			ILOG0_D("[overrideUlPhaseAfterOmiRx] maxOfNssForAllUsers %d", maxOfNssForAllUsers);
			rcrCommonPtr->nHeltf = convertNssToNHeLtf[maxOfNssForAllUsers]; // must be OFDMA
			tfCommonInfo->tfNumberOfHeLtfSymbols = rcrCommonPtr->nHeltf;	
			ILOG0_D("[overrideUlPhaseAfterOmiRx] nHeltf %d", rcrCommonPtr->nHeltf);
		}
 	}
}
