/***********************************************************************************
 File:		LinkAdaptationDriver.h
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "ProtectedDbLock_Api.h"
#include "ShramStationDatabase.h"
#include "LinkAdaptation.h"
#include "StaDatabase_Api.h"
#include "LinkAdaptationPhyDriver.h"
#include "VapDb.h"
#include "ShramVapDatabase.h"
#include "lm_VapDatabase.h"
#include "lm_StaDatabase.h"
#include "VapDatabase_Api.h"
#include "PowerAdaptation.h"
#include "mt_sysdefs.h"
#include "CyclicPrefix.h"
#include "TxSelector_Api.h"
#include "stringLibApi.h"
#include "ShramNewDeliaRecipes.h"
#include "ShramStatistics.h"
#include "HwSemaphore_API.h"
#include "RateAdaptation.h"
#include "Estimators.h"
#include "Beamforming.h"
#include "GroupDb.h"
#include "frame.h"
#include "DeliaRecipes_Api.h"
#include "AntennaSelection.h"
#include "ShramGroupDatabase.h"
#include "Tcr_Descriptors.h"
#include "TxSequencer_Api.h"
#include "PreAggregator_Api.h"
#include "LaStaProtectedZone_Descriptors.h"
#include "LaVapProtectedZone_Descriptors.h"
#include "LaMuGrpProtectedZone_Descriptors.h"
#include "TxSender_InitApi.h"
#include "CommonRamLinkAdaptation.h"
#include "TurboRates.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
/***************************************************************************/
/* 					Constants                                                  				     */
/***************************************************************************/
#define LOG_LOCAL_GID   GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 16

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
#define HW_RX_IMLICIT_BF_MODE 0x100
#define TX_SELECTOR_ACTION_SEND_BF_WITHOUT_DATA 0
#define TX_SELECTOR_ACTION_SEND_BF_WITH_DATA 1
#define TX_SELECTOR_ACTION_SEND_BF_REQUEST 1
#define TX_SELECTOR_ACTION_STOP_BF_REQUEST 0

//Sequencer report definitions 
#define USER_POS_0_MASK 0x1 

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400 // work arround for issue https://jira-chd.intel.com/browse/WLANVLSIIP-3003

#define MCS11_NSS4_PSDU_LEN_LIMIT	2278432
#define MCS9_NSS4_PSDU_LEN_LIMIT	1909312
#define MCS9_NSS2_PSDU_LEN_LIMIT	991392

#endif
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
#ifdef SDL_IGNORE_UNUSED_FUNCTION
static void setRefLenModificationFunc(uint32* scratchPadAddress, void* functionParams);
static void setCddnVapDbModificationFunc(uint32* scratchPadAddress, void* functionParams);
#endif
static void setStaFixedRateInTcrsModificationFunc(uint32* scratchPadAddress, void* functionParams);
static void setVapFixedRateInTcrsModificationFunc(uint32* scratchPadAddress, void* functionParams);
static void setVapFixedBwLimit(Bandwidth_e bandwidth, VapDb_t* pVapDbHwEntries);
static void setStaFixedBwLimit(StaId staIndex);
static void setPowerInVapDbModificationFunc(uint32* scratchPadAddress, void* functionParams);
static void setAntennaSelectionInVapDbModificationFunc(uint32* scratchPadAddress, void* functionParams);
static void setTcrsInHwStaDb(uint32* scratchPadAddress, void* functionParams);
static void setTcrsInHwGroupDb(uint32* scratchPadAddress, void* functionParams);
static void getSuTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams);
static	void getVhtMuTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams);
static	void setGroupDataBwLimit(uint8 groupIndex,Bandwidth_e bandwidth, bool changeWorkingPoint);
static void setHeMaxNssNdp(StaId staId, Bandwidth_e bw, uint8 activatedAntennasCount);


uint8 laHtMsduInAmsdu = 7;
uint8 laVhtMsduInAmsdu = 5;
uint8 laHeMsduInAmsdu = 7;
uint8 StaDb_SuNumOfMpdusInAmpdu = SU_NUM_OF_MPDUS_IN_A_MPDU;
#ifdef ENET_INC_ARCH_WAVE600
uint8 StaDb_SuHeNumOfMpdusInAmpdu = SU_HE_NUM_OF_MPDUS_IN_A_MPDU;

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
uint32 originalMuMaxPsduLengthLimit[HW_NUM_OF_STATIONS];
uint32 originalSuMaxPsduLengthLimit[HW_NUM_OF_STATIONS];
#endif

#endif

uint8 laPpduTxMode = STA_DB_PPDU_TX_MODE_LEGACY_ONLY;

static LinkAdaptationRtsMode_t 	LinkAdaptationRtsMode;
#ifdef TRAINING_FIRST_PHASE_POOL
uint8 numberOfStationInBw[HW_NUM_OF_VAPS][MAX_POSSIBLE_NUM_OF_BW] = {{0}}; 
#endif //TRAINING_FIRST_PHASE_POOL


uint32 MuScratchPad[PROTECTED_GROUP_DB_SIZE_WORDS];

Scp_e cpModeToHtVhtCpTcrVal[CP_NUM_OF_MODES_HT_VHT] = 
{
	SCP_SHORT_CP,
	SCP_LONG_CP,
};

HeCp_e cpModeToHeCpTcrVal[CP_NUM_OF_MODES] = 
{
	HE_CP_HE_0_8_CP,
	HE_CP_HE_1_6_CP,
	HE_CP_HE_0_8_CP,
	HE_CP_HE_1_6_CP,
	HE_CP_HE_0_8_CP,
	HE_CP_HE_3_2_CP,
};

HeLtf_e cpModeToHeLtfTcrVal[CP_NUM_OF_MODES] = 
{
	HE_LTF_HE_1X_LTF,
	HE_LTF_HE_1X_LTF,	
	HE_LTF_HE_2X_LTF,
	HE_LTF_HE_2X_LTF, 	
	HE_LTF_HE_4X_LTF,
	HE_LTF_HE_4X_LTF
};

CyclicPrefixMode_e heCpLtfTcrValToCpMode[HE_CP_LAST][HE_LTF_LAST] = 
/*
	  CP/LTF,				   1X, 						  2X, 					  4X,    
*/
{
	/*0.8*/		{CP_MODE_SHORT_CP_SHORT_LTF, 	CP_MODE_SHORT_CP_MED_LTF, 	CP_MODE_SHORT_CP_LONG_LTF},
	/*1.6*/		{CP_MODE_MED_CP_SHORT_LTF,		CP_MODE_MED_CP_MED_LTF,	  	CP_MODE_INVALID},
	/*3.2*/		{CP_MODE_INVALID,	 			CP_MODE_INVALID,	      	CP_MODE_LONG_CP_LONG_LTF},
};

CyclicPrefixMode_e htVhtCpTcrValToCpMode[SCP_LAST] = 
{
	CP_MODE_MED_CP_SHORT_LTF,
	CP_MODE_SHORT_CP_SHORT_LTF
};

/*---------------------------------------------------------------------------------
/						Functions definitions					
/----------------------------------------------------------------------------------*/


/**********************************************************************************

decreaseStaBwLimit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void decreaseStaBwLimit(StaId staIndex)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	Bandwidth_e changedBw = GetDataBwLimit(staIndex,INVALID_MU_USP_INDEX, FALSE);

	if (changedBw != BANDWIDTH_TWENTY)
	{
		LinkAdaptationSetStaDataBwLimit(staIndex,(Bandwidth_e)((uint8)changedBw-1),TRUE);
		pLaDatabase->laStaGroupCommon.bwAdaptationDb.isBwDroppedDueToRA = TRUE;
	}

}

/**********************************************************************************

initStaBwLimit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void initStaBwLimit(StaId staIndex,uint8 rateIndex, LaOperationModeNotificationChannelWidth_e omnChannelWidth, Bandwidth_e maxBW)
{
	LinkAdaptationStaDatabase_t* pLaDatabase = &LinkAdaptationStaDatabase[staIndex];
	Bandwidth_e wlanBandwidthMax = maxBW;
	Bandwidth_e changedBwLimit;
	Bandwidth_e tempBw = BANDWIDTH_TWENTY;
	const RateObj_t* ratesTable = getRatesTable(&pLaDatabase->laStaUspCommon);
	
	if ((ratesTable[rateIndex].laPhyMode != LA_PHY_MODE_HT_VHT) && (ratesTable[rateIndex].laPhyMode != LA_PHY_MODE_HE))
	{
		/*If rate is non HT /VHT rate set BW to 20*/
		changedBwLimit = BANDWIDTH_TWENTY;
		pLaDatabase->laStaGroupCommon.bwAdaptationDb.maxSupportedBw = BANDWIDTH_TWENTY;
 		pLaDatabase->laStaGroupCommon.bwAdaptationDb.isBwDroppedDueToRA = FALSE; 
	}
	else 
	{
		/*Set bw limit accordint to sta - HT /VHT / HE */
		/* Operating Mode Notification BW limitation */
		tempBw = (omnChannelWidth != OMN_BW_INVALID) ? (Bandwidth_e)omnChannelWidth : pLaDatabase->laStaGroupCommon.bwAdaptationDb.staSupportedBw;
		pLaDatabase->laStaGroupCommon.bwAdaptationDb.maxSupportedBw = MIN(wlanBandwidthMax, tempBw);
		changedBwLimit = pLaDatabase->laStaGroupCommon.bwAdaptationDb.maxSupportedBw;  // To be discussed - what should be the initial limit?
	}
	LinkAdaptationSetStaDataBwLimit(staIndex,changedBwLimit,TRUE);
	SetStaManagementBwLimit(staIndex,BANDWIDTH_TWENTY);
}
/**********************************************************************************

setRateinTcr 


Description:
------------
Write HW DB using protected DB API

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void setAllBwsTcrsParams(LaTcrModificationStruct_t* pTcrParams )
{
	uint32 protectedDBscratchPad[MAX(PROTECTED_STA_DB_SIZE_WORDS,PROTECTED_GROUP_DB_SIZE_WORDS)];

	if(pTcrParams->controlParams.uspIndex == INVALID_MU_USP_INDEX)
	{
		/*Update new working point in STA DB using protected DB*/
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
		ProtectedDbLock_ReadModifyWriteReq((uint32*)&(StaDbHwEntries[pTcrParams->controlParams.staIndex].common.word10 ), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setTcrsInHwStaDb,(void*)pTcrParams);
#else
		ProtectedDbLock_ReadModifyWriteReq((uint32*)&(StaDbHwEntries[pTcrParams->controlParams.staIndex].common.word12/*tcr020MhzData*/), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setTcrsInHwStaDb,(void*)pTcrParams);
#endif
	}
	else
	{
		if (pTcrParams->controlParams.isHeGroup)
		{
			// HE MU
			setTcrsInPhaseDb(pTcrParams);
		}
		else
		{
			// VHT MU
			if (pTcrParams->controlParams.fastProbing == TRUE)
			{
				/*Update Global Scratchpad*/
				setTcrsInHwGroupDb(MuScratchPad,(void *)pTcrParams);
			}
			else
			{
				/*Update new working point in Group DB using protected DB*/
				ProtectedDbLock_ReadModifyWriteReq((uint32*)&(GroupDbHwEntries[pTcrParams->controlParams.staIndex].common.word6), (PROTECTED_GROUP_DB_SIZE_WORDS), protectedDBscratchPad, setTcrsInHwGroupDb,(void*)pTcrParams); 
			}
		}
	}
}

/**********************************************************************************

updateTransTimeLimit 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void updateTransTimeLimit(StaId staIndex)
{
	LaFixedRateStationParams_t *laFixedRateStationParams_p = &LinkAdaptationStaDatabase[staIndex].laStationUnique.fixedRateStationParams;
	uint8 rateIndex = convertTcr2RateIndex(laFixedRateStationParams_p->phyMode, laFixedRateStationParams_p->rate);
	uint16 transmissionTime;
	uint16 newPhyRate;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	
	if (AggRateLimitMode == TRUE)
	{
		newPhyRate = ratesTable[rateIndex].rateTableBwParmas[GetDataBwLimit(staIndex,INVALID_MU_USP_INDEX, FALSE)].shortCpPhyRate;

		updateTransmissionTimeParams(staIndex, rateIndex);

		/* Set time limit */
		if (LinkAdaptationStaDatabase[staIndex].laStaUspCommon.staTransmissionParams.clientIsolationMode == FALSE)
		{
			transmissionTime = MAX_TRANSMISSION_TIME;
		}
		else
		{
			transmissionTime = (newPhyRate < LA_TIME_LIMIT_RATE_TH) ? MED_TRANSMISSION_TIME : MIN_TRANSMISSION_TIME;			
		}

		linkAdaptationSetTransmissionTimeLimit(staIndex, transmissionTime);

	}
}

/**********************************************************************************

setStaFixedRatesInHwDb 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void setStaFixedRatesInHwDb(StaId staIndex)
{
	
	uint32 protectedDBscratchPad[PROTECTED_STA_DB_SIZE_WORDS];
	StaDb_t* pStaDbHwEntries = &StaDbHwEntries[staIndex];
	LaFixedRateStationProtectedDbParams_t	protectedDbParams;
	LaStationUspCommonParams_t* pLaStaUspCommon = &LinkAdaptationStaDatabase[staIndex].laStaUspCommon;
	//Bandwidth_e bw = GetDataBwLimit(staIndex,INVALID_MU_USP_INDEX);
	uint8 workingPointEstimatorsIndex;
	uint8 oldRate, rateIndex;
	bool isRateInc = FALSE;
#ifdef LINK_ADAPTATION_LOGS
		//("setStaFixedRatesInHwDb");
#endif
	MEMCPY (&protectedDbParams.fixedRateStationParams, &LinkAdaptationStaDatabase[staIndex].laStationUnique.fixedRateStationParams, sizeof(LaFixedRateStationParams_t));
	protectedDbParams.staIndex = staIndex;

	workingPointEstimatorsIndex = pLaStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	oldRate = pLaStaUspCommon->fastProbingEstimators[workingPointEstimatorsIndex].rateIndex;	
	rateIndex = convertTcr2RateIndex(protectedDbParams.fixedRateStationParams.phyMode, protectedDbParams.fixedRateStationParams.rate);
	isRateInc = rateAdaptation_RateIsIncreased(rateIndex, oldRate, getRatesTable(pLaStaUspCommon));	


#ifdef ENET_INC_ARCH_WAVE600D2

	//if HE ER SU or HE SU with DCM - disable MGMT aggregations & Multi TID
	if ((protectedDbParams.fixedRateStationParams.phyMode == PHY_MODE_11AX_SU_EXT) || (EXTRACT_DCM_FROM_VHT_HE_RATE(protectedDbParams.fixedRateStationParams.rate)))
	{
		StaDB_SetHeMngAgg(staIndex, MULTI_TID_MNG_AGGREGATION_OFF);
		StaDb_SetMaxStationTidLimitInAggregation(staIndex, 0x0);
	}
	else
	{
		bool returnToInitial = TRUE;

		/*if changeType is not both DATA & MMGMT - need to validate that the current fixed rate PHY MODE value 
		of the other type (the one that is not chosen) is not PHY_MODE_11AX_SU_EXT.
		If one of the TCRs left with PHY_MODE_11AX_SU_EXT, we will not return to the initial configuration
		*/
		if(protectedDbParams.fixedRateStationParams.changeType != LA_PACKET_TYPE_DATA_MANAGEMENT)
		{
			switch(protectedDbParams.fixedRateStationParams.changeType)
			{
				case LA_PACKET_TYPE_DATA:
				{
					returnToInitial = pStaDbHwEntries->common.tcr08021120MhzPhyMode == PHY_MODE_11AX_SU_EXT? FALSE:TRUE;
					
					break;
				}
				case LA_PACKET_TYPE_MANAGEMENT:
				{
					returnToInitial = pStaDbHwEntries->common.tcr0Data20MhzPhyMode == PHY_MODE_11AX_SU_EXT? FALSE:TRUE;
					
					break;
				}
				default:
					FATAL("setStaFixedRatesInHwDb");
			}		
		}
		
		//return back to initial configuration

		if(returnToInitial)
		{
		
#ifdef HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED
			// According to current System requirements - by default, MNG aggregation is disabled, remove this if decided differently
			StaDB_SetHeMngAgg(staIndex, MULTI_TID_MNG_AGGREGATION_OFF);
#else
			if(StaDb_GetStationAckEnabledSupport(staIndex) == MULTI_TID_ACK_ENABLED_SUPPORTED)
			{
				StaDB_SetHeMngAgg(staIndex, MULTI_TID_MNG_AGGREGATION_ON);
			}
			else
			{
				StaDB_SetHeMngAgg(staIndex, MULTI_TID_MNG_AGGREGATION_OFF);
			}
#endif //HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED

			StaDb_SetMaxStationTidLimitInAggregation(staIndex, StaDb_GetMaxStationTidLimitInAggregation(staIndex));
		}

	}
#endif //#ifdef ENET_INC_ARCH_WAVE600D2

	/*If Change from non 11B to 11B, first set BW and after it set rate*/
	if (protectedDbParams.fixedRateStationParams.phyMode != PHY_MODE_11B)
	{
		/*Set bandwidth*/
		setStaFixedBwLimit(staIndex);
		ILOG0_DDD("setStaFixedRatesInHwDb 1 isRateInc = %d newRateindex= %d oldRate= %d ",isRateInc,rateIndex, oldRate);

		if (isRateInc == TRUE)
		{
			/*Set rate*/
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
			ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pStaDbHwEntries->common.word10), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setStaFixedRateInTcrsModificationFunc,(void*)(&protectedDbParams)); 
#else
			ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pStaDbHwEntries->common.word12/*tcr020MhzData*/), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setStaFixedRateInTcrsModificationFunc,(void*)(&protectedDbParams)); 
#endif
			updateTransTimeLimit(staIndex);
		}
		else
		{			
			updateTransTimeLimit(staIndex);
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
			ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pStaDbHwEntries->common.word10), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setStaFixedRateInTcrsModificationFunc,(void*)(&protectedDbParams)); 
#else
			ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pStaDbHwEntries->common.word12/*tcr020MhzData*/), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setStaFixedRateInTcrsModificationFunc,(void*)(&protectedDbParams)); 
#endif
		}
	}
	else
	{
		ILOG0_DDD("setStaFixedRatesInHwDb 2 isRateInc = %d newRateindex= %d oldRate= %d ",isRateInc,rateIndex, oldRate);
		if (isRateInc == TRUE)
		{
			/*Set rate*/
			ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pStaDbHwEntries->common.word12/*tcr020MhzData*/), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setStaFixedRateInTcrsModificationFunc,(void*)(&protectedDbParams)); 
			updateTransTimeLimit(staIndex);
		}
		else
		{			
			updateTransTimeLimit(staIndex);
			ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pStaDbHwEntries->common.word12/*tcr020MhzData*/), (PROTECTED_STA_DB_SIZE_WORDS), protectedDBscratchPad, setStaFixedRateInTcrsModificationFunc,(void*)(&protectedDbParams)); 
		}
		/*Set bandwidth*/
		setStaFixedBwLimit(staIndex);
	}
	
}
/**********************************************************************************

setNewRateInStationDb 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setStaFixedRateInTcrsModificationFunc(uint32* scratchPadAddress, void* functionParams)
{
	LaFixedRateStationProtectedDbParams_t* pFixedRateParams = (LaFixedRateStationProtectedDbParams_t*) functionParams;
	LaStaProtectedZone_t* laProtectedStruct = (LaStaProtectedZone_t*) scratchPadAddress;
	uint8 powerTcrDataVal;
	uint8 powerTcrManagemnentVal;
	uint8 rateIndex = convertTcr2RateIndex(pFixedRateParams->fixedRateStationParams.phyMode, pFixedRateParams->fixedRateStationParams.rate);
	uint8 updatedRateIndex;
	PhyMode_e phyModePerBw = pFixedRateParams->fixedRateStationParams.phyMode;
	uint8 ratePerBw = pFixedRateParams->fixedRateStationParams.rate;	
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	uint8 bfWorkingPointState;
	uint8 tcrBfMode = FALSE;
	RateMask_t allowdRatesMask;	
	bool isVht = (FixedRateParameters.phyMode == PHY_MODE_11AC) ? TRUE : FALSE;
	bool isHeExt = (FixedRateParameters.phyMode == PHY_MODE_11AX_SU_EXT) ? TRUE : FALSE;
	const RateObj_t* ratesTable;
	CyclicPrefixMode_e cpMode = pFixedRateParams->fixedRateStationParams.cpMode;
	HeCp_e heCp;// = cpModeToHeCpTcrVal[cpMode];
	HeLtf_e heLtf;// = cpModeToHeLtfTcrVal[cpMode];	
	Dcm_e dcm = EXTRACT_DCM_FROM_VHT_HE_RATE(pFixedRateParams->fixedRateStationParams.rate);

	//KW_FIX_FW_M
	ASSERT(cpMode < CP_NUM_OF_MODES);
	heCp = cpModeToHeCpTcrVal[cpMode];
	heLtf = cpModeToHeLtfTcrVal[cpMode];
	
	updateLaDbDistributionParam(&laDbDistributionParameter,pFixedRateParams->staIndex,INVALID_MU_USP_INDEX, FALSE);
	ratesTable = getRatesTableFixRateConfig(&(pFixedRateParams->fixedRateStationParams));

	/* Set 20 MHz BW TCRs */
	updatedRateIndex = rateIndex;
	
	if (((pFixedRateParams->fixedRateStationParams.phyMode != PHY_MODE_11AX_SU) || (pFixedRateParams->fixedRateStationParams.phyMode != PHY_MODE_11AX_SU_EXT)) && (pFixedRateParams->fixedRateStationParams.bandwidth != BANDWIDTH_TWENTY))
	{
		/*Clear allowdRatesMask before or operation*/
		memset(&allowdRatesMask, 0, sizeof(RateMask_t));
		/*Set AllowedRatesMask in local variable*/
		orOperator64Bit(allowdRatesMask.raIndexMask64bit, (uint32*)AllowedRatesMask[BANDWIDTH_TWENTY].raIndexMask64bit, allowdRatesMask.raIndexMask64bit);
		/*Set turbo rates in local raMask*/
		if( (laDbDistributionParameter.laStaUspCommon != NULL)&& (FALSE == laDbDistributionParameter.laStaUspCommon->staTransmissionParams.heSta))
		{

			/*Turbo rates is not relevant for he stations or VAP*/
			TurboRatesGetBitmap(pFixedRateParams->staIndex, INVALID_MU_USP_INDEX, BANDWIDTH_TWENTY, &allowdRatesMask);
		}
		SLOG0(0, 0, RateMask_t, &allowdRatesMask);
		
		/*If rate is not in BW mask decrease to rate -1*/
		if (((ratesTable[rateIndex].vhtHeMcs) != 0)&&
			(rateIndex !=0) && 
			(isBitInMask64Bit(rateIndex, (uint32*)allowdRatesMask.raIndexMask64bit)==0))
		{
			updatedRateIndex--;
		}
	}
	getTcrPhyModeAndRate(ratesTable, isVht, updatedRateIndex, &phyModePerBw, &ratePerBw, isHeExt, dcm);
	
	if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA))
	{
		bfWorkingPointState = LinkAdaptationStaDatabase[pFixedRateParams->staIndex].laStationUnique.beamFormingDb.bfWorkingPointState;
		if((bfWorkingPointState == BF_STATE_EXPLICIT)|| (bfWorkingPointState == BF_STATE_IMPLICIT))
		{
			tcrBfMode = TRUE;
		}
		powerTcrDataVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_DATA,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex), FALSE,tcrBfMode);

		/*Change transmitting fields in STA DB in both cases - probing and worKing point modification*/
		/*Change main TCR*/
		laProtectedStruct->common.tcr0Data20MhzPhyMode = phyModePerBw;
		laProtectedStruct->common.tcr220MhzDataPsduRate = ratePerBw;
		laProtectedStruct->common.tcr0Data20MhzRfPower = powerTcrDataVal;
		laProtectedStruct->common.tcr2DataLdpc = getLdpcCapability(pFixedRateParams->staIndex,ratesTable[updatedRateIndex].laPhyMode);

#ifdef ENET_INC_ARCH_WAVE600D2
		laProtectedStruct->common.heSuExtDataRuSize = pFixedRateParams->fixedRateStationParams.heExtRuSizeData;
#endif //ENET_INC_ARCH_WAVE600D2
		laProtectedStruct->common.tcr120MhzDataHeCp = 0;
		laProtectedStruct->wp.tcr120MhzDataHeCp = 0;
		laProtectedStruct->common.tcr120MhzDataHeLtf= 0;
		laProtectedStruct->wp.tcr120MhzDataHeLtf= 0;
		laProtectedStruct->common.tcr120MhzDataScp = 0;
		laProtectedStruct->wp.tcr120MhzDataScp = 0;	

		if ((phyModePerBw == PHY_MODE_11AX_SU) || (phyModePerBw == PHY_MODE_11AX_SU_EXT))
		{
			laProtectedStruct->common.tcr120MhzDataHeCp = heCp;
			laProtectedStruct->wp.tcr120MhzDataHeCp = heCp;

			laProtectedStruct->common.tcr120MhzDataHeLtf= heLtf;
			laProtectedStruct->wp.tcr120MhzDataHeLtf= heLtf;
		}
		else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
		{
			laProtectedStruct->common.tcr120MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			laProtectedStruct->wp.tcr120MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
		}
		
		/*Change working point fields in STA DB*/
		laProtectedStruct->wp.tcr0Data20MhzPhyMode = phyModePerBw;
		laProtectedStruct->wp.tcr220MhzDataPsduRate = ratePerBw;
		laProtectedStruct->wp.tcr0Data20MhzRfPower = powerTcrDataVal;
	}
	if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_MANAGEMENT))
	{
		tcrBfMode = FALSE;
		powerTcrManagemnentVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_MANAGEMENT,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex),FALSE,tcrBfMode);

		/*Change transmitting fields in STA DB in both cases - probing and woring point modifcation*/
		/*Change main TCR*/
		laProtectedStruct->common.tcr08021120MhzPhyMode = phyModePerBw;
		laProtectedStruct->common.tcr220Mhz80211PsduRate = ratePerBw;
		laProtectedStruct->common.tcr08021120MhzRfPower = powerTcrManagemnentVal;		
#ifdef ENET_INC_ARCH_WAVE600D2
		laProtectedStruct->common.heSuExtMngRuSize= pFixedRateParams->fixedRateStationParams.heExtRuSizeMng;
#endif //ENET_INC_ARCH_WAVE600D2
		laProtectedStruct->common.tcr120Mhz80211HeCp = 0;
		laProtectedStruct->common.tcr120Mhz80211HeLtf= 0;
		laProtectedStruct->common.tcr120Mhz80211Scp = 0;

		if ((phyModePerBw == PHY_MODE_11AX_SU) || (phyModePerBw == PHY_MODE_11AX_SU_EXT))
		{
			laProtectedStruct->common.tcr120Mhz80211HeCp = heCp;
			laProtectedStruct->common.tcr120Mhz80211HeLtf= heLtf;
		}
		else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
		{
			laProtectedStruct->common.tcr120Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
		}

	}
	if (pFixedRateParams->fixedRateStationParams.bandwidth >= BANDWIDTH_FOURTY)
	{
		/* Set 40 MHz BW TCRs */
		updatedRateIndex = rateIndex;
		
		if ((pFixedRateParams->fixedRateStationParams.phyMode != PHY_MODE_11AX_SU) && (pFixedRateParams->fixedRateStationParams.bandwidth != BANDWIDTH_FOURTY))
		{
			/*Clear allowdRatesMask before or operation*/
			memset(&allowdRatesMask, 0, sizeof(RateMask_t));
			/*Set AllowedRatesMask in local variable*/
			orOperator64Bit(allowdRatesMask.raIndexMask64bit, (uint32*)AllowedRatesMask[BANDWIDTH_FOURTY].raIndexMask64bit, allowdRatesMask.raIndexMask64bit);
			/*Set turbo rates in local raMask*/
			if( (laDbDistributionParameter.laStaUspCommon != NULL)&& (FALSE == laDbDistributionParameter.laStaUspCommon->staTransmissionParams.heSta))
			{
				/*Turbo rates is not relevant for he stations or VAP*/
				TurboRatesGetBitmap(pFixedRateParams->staIndex, INVALID_MU_USP_INDEX, BANDWIDTH_FOURTY, &allowdRatesMask);
			}
			SLOG0(0, 0, RateMask_t, &allowdRatesMask);
			
			/*If rate is not in BW mask decrease to rate -1*/
			if (((ratesTable[rateIndex].vhtHeMcs) != 0)&&
				(rateIndex !=0) && 
				(isBitInMask64Bit(rateIndex, (uint32*)allowdRatesMask.raIndexMask64bit)==0))
			{
				updatedRateIndex--;
			}
		}
		getTcrPhyModeAndRate(ratesTable, isVht, updatedRateIndex, &phyModePerBw, &ratePerBw, FALSE, dcm);
		
		if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA))
		{
			bfWorkingPointState = LinkAdaptationStaDatabase[pFixedRateParams->staIndex].laStationUnique.beamFormingDb.bfWorkingPointState;
			if((bfWorkingPointState == BF_STATE_EXPLICIT)|| (bfWorkingPointState == BF_STATE_IMPLICIT))
			{
				tcrBfMode = TRUE;
			}
			powerTcrDataVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_DATA,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex), FALSE,tcrBfMode);
	
			/*Change transmitting fields in STA DB in both cases - probing and woring point modification*/
			/*Change main TCR*/
			laProtectedStruct->common.tcr0Data40MhzPhyMode = phyModePerBw;
			laProtectedStruct->common.tcr240MhzDataPsduRate = ratePerBw;
			laProtectedStruct->common.tcr0Data40MhzRfPower = powerTcrDataVal;
	
			laProtectedStruct->common.tcr140MhzDataHeCp = 0;
			laProtectedStruct->wp.tcr140MhzDataHeCp = 0;
			laProtectedStruct->common.tcr140MhzDataHeLtf= 0;
			laProtectedStruct->wp.tcr140MhzDataHeLtf= 0;
			laProtectedStruct->common.tcr140MhzDataScp = 0;
			laProtectedStruct->wp.tcr140MhzDataScp = 0; 
			
			if (phyModePerBw == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr140MhzDataHeCp = heCp;
				laProtectedStruct->wp.tcr140MhzDataHeCp = heCp;
			
				laProtectedStruct->common.tcr140MhzDataHeLtf= heLtf;
				laProtectedStruct->wp.tcr140MhzDataHeLtf= heLtf;
			}
			else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr140MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				laProtectedStruct->wp.tcr140MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}
	
			/*Change working point fields in STA DB*/
			
			laProtectedStruct->wp.tcr0Data40MhzPhyMode = phyModePerBw;
			laProtectedStruct->wp.tcr240MhzDataPsduRate = ratePerBw;
			laProtectedStruct->wp.tcr0Data40MhzRfPower = powerTcrDataVal;
		}
		if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_MANAGEMENT))
		{
			tcrBfMode = FALSE;
			powerTcrManagemnentVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_MANAGEMENT,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex),FALSE,tcrBfMode);
	
			/*Change transmitting fields in STA DB in both cases - probing and woring point modifcation*/
			/*Change main TCR*/
			laProtectedStruct->common.tcr08021140MhzPhyMode = phyModePerBw;
			laProtectedStruct->common.tcr240Mhz80211PsduRate = ratePerBw;
			laProtectedStruct->common.tcr08021140MhzRfPower = powerTcrManagemnentVal;
	
			laProtectedStruct->common.tcr140Mhz80211HeCp = 0;
			laProtectedStruct->common.tcr140Mhz80211HeLtf= 0;
			laProtectedStruct->common.tcr140Mhz80211Scp = 0;
			
			if (phyModePerBw == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr140Mhz80211HeCp = heCp;
				laProtectedStruct->common.tcr140Mhz80211HeLtf= heLtf;
			}
			else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr140Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}
		}
	}
	if (pFixedRateParams->fixedRateStationParams.bandwidth >= BANDWIDTH_EIGHTY)
	{
		/* Set 80 MHz BW TCRs */
		updatedRateIndex = rateIndex;
		
		if ((pFixedRateParams->fixedRateStationParams.phyMode != PHY_MODE_11AX_SU) && (pFixedRateParams->fixedRateStationParams.bandwidth != BANDWIDTH_EIGHTY))
		{
			/*Clear allowdRatesMask before or operation*/
			memset(&allowdRatesMask, 0, sizeof(RateMask_t));
			/*Set AllowedRatesMask in local variable*/
			orOperator64Bit(allowdRatesMask.raIndexMask64bit, (uint32*)AllowedRatesMask[BANDWIDTH_EIGHTY].raIndexMask64bit, allowdRatesMask.raIndexMask64bit);
			/*Set turbo rates in local raMask*/
			if((laDbDistributionParameter.laStaUspCommon != NULL)&& (FALSE == laDbDistributionParameter.laStaUspCommon->staTransmissionParams.heSta))
			{
				/*Turbo rates is not relevant for he stations or VAP*/
				TurboRatesGetBitmap(pFixedRateParams->staIndex, INVALID_MU_USP_INDEX, BANDWIDTH_EIGHTY, &allowdRatesMask);
			}
			SLOG0(0, 0, RateMask_t, &allowdRatesMask);
			
			/*If rate is not in BW mask decrease to rate -1*/
			if (((ratesTable[rateIndex].vhtHeMcs) != 0)&&
				(rateIndex !=0) && 
				(isBitInMask64Bit(rateIndex, (uint32*)allowdRatesMask.raIndexMask64bit)==0))
			{
				updatedRateIndex--;
			}
		}
		getTcrPhyModeAndRate(ratesTable, isVht, updatedRateIndex, &phyModePerBw, &ratePerBw, FALSE, dcm);
		
		if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA))
		{
			bfWorkingPointState = LinkAdaptationStaDatabase[pFixedRateParams->staIndex].laStationUnique.beamFormingDb.bfWorkingPointState;
			if((bfWorkingPointState == BF_STATE_EXPLICIT)|| (bfWorkingPointState == BF_STATE_IMPLICIT))
			{
				tcrBfMode = TRUE;
			}
			powerTcrDataVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_DATA,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex), FALSE,tcrBfMode);
	
			/*Change transmitting fields in STA DB in both cases - probing and woring point modification*/
			/*Change main TCR*/
			laProtectedStruct->common.tcr0Data80MhzPhyMode = phyModePerBw;
			laProtectedStruct->common.tcr280MhzDataPsduRate = ratePerBw;
			laProtectedStruct->common.tcr0Data80MhzRfPower = powerTcrDataVal;
	
			laProtectedStruct->common.tcr180MhzDataHeCp = 0;
			laProtectedStruct->wp.tcr180MhzDataHeCp = 0;
			laProtectedStruct->common.tcr180MhzDataHeLtf= 0;
			laProtectedStruct->wp.tcr180MhzDataHeLtf= 0;
			laProtectedStruct->common.tcr180MhzDataScp = 0;
			laProtectedStruct->wp.tcr180MhzDataScp = 0; 
			
			if (phyModePerBw == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr180MhzDataHeCp = heCp;
				laProtectedStruct->wp.tcr180MhzDataHeCp = heCp;
			
				laProtectedStruct->common.tcr180MhzDataHeLtf= heLtf;
				laProtectedStruct->wp.tcr180MhzDataHeLtf= heLtf;
			}
			else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr180MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				laProtectedStruct->wp.tcr180MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}
	
			/*Change working point fields in STA DB*/
			
			laProtectedStruct->wp.tcr0Data80MhzPhyMode = phyModePerBw;
			laProtectedStruct->wp.tcr280MhzDataPsduRate = ratePerBw;
			laProtectedStruct->wp.tcr0Data80MhzRfPower = powerTcrDataVal;
		}
		if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_MANAGEMENT))
		{
			tcrBfMode = FALSE;
			powerTcrManagemnentVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_MANAGEMENT,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex),FALSE,tcrBfMode);
	
			/*Change transmitting fields in STA DB in both cases - probing and woring point modifcation*/
			/*Change main TCR*/
			laProtectedStruct->common.tcr08021180MhzPhyMode = phyModePerBw;
			laProtectedStruct->common.tcr280Mhz80211PsduRate = ratePerBw;
			laProtectedStruct->common.tcr08021180MhzRfPower = powerTcrManagemnentVal;
	
			laProtectedStruct->common.tcr180Mhz80211HeCp = 0;
			laProtectedStruct->common.tcr180Mhz80211HeLtf= 0;
			laProtectedStruct->common.tcr180Mhz80211Scp = 0;
			
			if (phyModePerBw == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr180Mhz80211HeCp = heCp;
				laProtectedStruct->common.tcr180Mhz80211HeLtf= heLtf;
			}
			else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr180Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}
		}
	}
	if (pFixedRateParams->fixedRateStationParams.bandwidth == BANDWIDTH_ONE_HUNDRED_SIXTY)
	{

		/* Set 160 MHz BW TCRs */
		updatedRateIndex = rateIndex;
		
		getTcrPhyModeAndRate(ratesTable, isVht, updatedRateIndex, &phyModePerBw, &ratePerBw, FALSE, dcm);
		
		if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA))
		{

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
			uint32 newMaxPsduLength = 0;
			uint32 tmpPsduLength = 0;
#endif
			bfWorkingPointState = LinkAdaptationStaDatabase[pFixedRateParams->staIndex].laStationUnique.beamFormingDb.bfWorkingPointState;
			if((bfWorkingPointState == BF_STATE_EXPLICIT)|| (bfWorkingPointState == BF_STATE_IMPLICIT))
			{
				tcrBfMode = TRUE;
			}
			powerTcrDataVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_DATA,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex), FALSE,tcrBfMode);
	
			/*Change transmitting fields in STA DB in both cases - probing and woring point modification*/
			/*Change main TCR*/
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
			if((laProtectedStruct->common.tcr2DataLdpc == 1) && (phyModePerBw == PHY_MODE_11AX_SU))
			{
				// if new rate to be used requires limitation
				if(isPsduLengthLimitationNeeded(ratePerBw, &newMaxPsduLength) == TRUE)
				{
					ILOG0_D("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setStaFixedRateInTcrsModificationFunc - new rate requires limitation: new Rate %d",ratePerBw);
					
					// if current rate is not problematic
					if(isPsduLengthLimitationNeeded(laProtectedStruct->common.tcr2160MhzDataPsduRate, &tmpPsduLength) == FALSE)
					{
				
				 		// store the original maxPsduLengthLimit 
						originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex] = laProtectedStruct->wp.maxPsduLengthLimit;
						ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setStaFixedRateInTcrsModificationFunc store limit- current rate is good: current Rate %d, origLimit %d",laProtectedStruct->common.tcr2160MhzDataPsduRate, originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex]);
					}
				
					// if current TcrRate is also problematic - required limitation in maxPsduLengthLimit
					else
					{
						// no need to store the current maxPsduLengthLimit - because was previously stored
						ILOG0_D("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setStaFixedRateInTcrsModificationFunc, original limit was stored previously - current rate is problematic: current Rate %d",laProtectedStruct->common.tcr2160MhzDataPsduRate);
					}
					// overwrite common and WP values , apply minimum in case original maxLimit is lower than the limit recommended by system
					ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setStaFixedRateInTcrsModificationFunc overwrite - newLimit %d, origLimit %d", newMaxPsduLength, originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex]);
					laProtectedStruct->common.maxPsduLengthLimit = MIN(newMaxPsduLength, originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex]);
					laProtectedStruct->wp.maxPsduLengthLimit = MIN(newMaxPsduLength, originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex]);
				}
				// if new TcrRate to be written doesn't require limitation in maxPsduLengthLimit
				else
				{
					ILOG0_D("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setStaFixedRateInTcrsModificationFunc - new rate is good %d",ratePerBw);
					// check if rollback must be done to original maxPsduLengthLimit - if current tcrRate is problematic and limitation is applied on maxPsduLengthLimit
					if(isPsduLengthLimitationNeeded(laProtectedStruct->common.tcr2160MhzDataPsduRate, &tmpPsduLength) == TRUE)
					{
						ILOG0_DDDD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400:setStaFixedRateInTcrsModificationFunc - rollback to original limit: common limit %d, wp limit %d, original limit %d staId %d",
							laProtectedStruct->common.maxPsduLengthLimit, laProtectedStruct->wp.maxPsduLengthLimit, originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex], pFixedRateParams->staIndex);
						laProtectedStruct->common.maxPsduLengthLimit = originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex];
						laProtectedStruct->wp.maxPsduLengthLimit = originalSuMaxPsduLengthLimit[pFixedRateParams->staIndex];
					}
				}			
			}

#endif
			laProtectedStruct->common.tcr0Data160MhzPhyMode = phyModePerBw;
			laProtectedStruct->common.tcr2160MhzDataPsduRate = ratePerBw;
			laProtectedStruct->common.tcr0Data160MhzRfPower = powerTcrDataVal;
	
			laProtectedStruct->common.tcr1160MhzDataHeCp = 0;
			laProtectedStruct->wp.tcr1160MhzDataHeCp = 0;
			laProtectedStruct->common.tcr1160MhzDataHeLtf= 0;
			laProtectedStruct->wp.tcr1160MhzDataHeLtf= 0;
			laProtectedStruct->common.tcr1160MhzDataScp = 0;
			laProtectedStruct->wp.tcr1160MhzDataScp = 0; 
			
			if (phyModePerBw == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr1160MhzDataHeCp = heCp;
				laProtectedStruct->wp.tcr1160MhzDataHeCp = heCp;
			
				laProtectedStruct->common.tcr1160MhzDataHeLtf= heLtf;
				laProtectedStruct->wp.tcr1160MhzDataHeLtf= heLtf;
			}
			else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr1160MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				laProtectedStruct->wp.tcr1160MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}

			/*Change working point fields in STA DB*/
			
			laProtectedStruct->wp.tcr0Data160MhzPhyMode = phyModePerBw;
			laProtectedStruct->wp.tcr2160MhzDataPsduRate = ratePerBw;
			laProtectedStruct->wp.tcr0Data160MhzRfPower = powerTcrDataVal;
		}
		if((pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (pFixedRateParams->fixedRateStationParams.changeType == LA_PACKET_TYPE_MANAGEMENT))
		{
			tcrBfMode = FALSE;
			powerTcrManagemnentVal = CalcLimitsAndGetStaTcrPowerVal(pFixedRateParams->staIndex, pFixedRateParams->fixedRateStationParams.bandwidth,rateIndex, FALSE, LA_PACKET_TYPE_MANAGEMENT,AntennaSelectionGetAntennaSelectionBitmap(pFixedRateParams->staIndex),FALSE,tcrBfMode);
	
			/*Change transmitting fields in STA DB in both cases - probing and woring point modifcation*/
			/*Change main TCR*/
			laProtectedStruct->common.tcr080211160MhzPhyMode = phyModePerBw;
			laProtectedStruct->common.tcr2160Mhz80211PsduRate = ratePerBw;
			laProtectedStruct->common.tcr080211160MhzRfPower = powerTcrManagemnentVal;
	
			laProtectedStruct->common.tcr1160Mhz80211HeCp = 0;
			laProtectedStruct->common.tcr1160Mhz80211HeLtf= 0;
			laProtectedStruct->common.tcr1160Mhz80211Scp = 0;
			
			if (phyModePerBw == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr1160Mhz80211HeCp = heCp;
				laProtectedStruct->common.tcr1160Mhz80211HeLtf= heLtf;
			}
			else if ((phyModePerBw == PHY_MODE_11N) || (phyModePerBw == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr1160Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}
		}
	}
}
#ifdef SDL_IGNORE_UNUSED_FUNCTION
/**********************************************************************************

setRefLenModificationFunc 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static void setRefLenModificationFunc(uint32* scratchPadAddress, void* functionParams)
{

}
#endif
/**********************************************************************************

setRefLenInStaDb 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void setRefLenInStaDb(StaId staIndex, uint8 refLenSuccess, uint8 refLenFail)
{

	UNUSED_PARAM(staIndex);	
	UNUSED_PARAM(refLenSuccess);	
	UNUSED_PARAM(refLenFail);	
}

/********************************************************************************
updateTcrCommonTemplate1Rate

Description: updates template1's phy mode & rate
----------

Input: valid rates to mask wanted rates from basic rate set
-----

Output:
-------

Returns:
--------

********************************************************************************/
void updateTcrCommonTemplate1Rate(uint32 validRateSet)
{
	VapDb_t* 			vapDbEntry;
	LmVapDbSwInfo_t*	vapSwDb;
	TcrCommon_t* 		tcrCommonTemplate;
	TcrUser_t*			tcrUserTemplate;
	RateMask_t			rateSet;
	uint8 				vapId;
	uint8 				tcrRateVal;
	uint8 				rateIndex;
	PhyMode_e 			phymode;
	uint8				firstvap;
	uint8				numvaps;

	firstvap = ConfigurationManager_GetFirstVapForMyBand();
	numvaps = ConfigurationManager_GetNumOfVapsInMyBand();

	TX_INTERRUPT_SAVE_AREA;
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	rateSet.raIndexMask64bit[1] = 0;
	for (vapId = firstvap; vapId < (firstvap + numvaps); vapId++)
	{
		vapDbEntry = &VapDbHwEntries[vapId];
		vapSwDb = &LmVapDataBase[vapId];
		rateSet.raIndexMask64bit[0] = (vapSwDb->basicRates & validRateSet) | LM_PHY_11A_RATE_MSK;
		ASSERT(rateSet.raIndexMask64bit[0]);
		rateIndex = getLowestRateinMask(INVALID_STA_INDEX, rateSet, BANDWIDTH_TWENTY, 0, PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX);
		getTcrPhyModeAndRate(htVhtRatesTable, 0, rateIndex, &phymode, &tcrRateVal, FALSE, NO_DCM);  
		tcrCommonTemplate = (TcrCommon_t*)(&(vapDbEntry->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word0));
		tcrCommonTemplate->phyMode = phymode; 
		tcrUserTemplate = (TcrUser_t*)(&(vapDbEntry->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word8));
		tcrUserTemplate->rate = tcrRateVal;
	}

	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}


/********************************************************************************
fillVapHwDb




Description:
----------

Input:
-----

Output:
-------

Returns:
--------

********************************************************************************/
RateMask_t 				vapBasicRateMask;

void fillVapHwDb (uint8 vapIndex, uint8 fixedMcsVapManagement)
{
	VapDb_t*				pVapDbHwEntry = &VapDbHwEntries[vapIndex];
	LmVapDbSwInfo_t* 		pLmVapDataBase = &LmVapDataBase[vapIndex];
	uint8 					rateIndex;
	PhyMode_e 				phyMode;
	uint8 					tcrRateVal;
//	RateMask_t 				vapBasicRateMask;
	TcrCommon_t*			tcrCommonTemplate;
	TcrUser_t*				tcrUserTemplate; 
	uint8					tcrIterator; 
	uint8 					rateIndexTcr1;
	PhyMode_e 				phyModeTcr1;
	uint8 					tcrRateValTcr1;	
	uint8					highestRateIndexInMask;
	uint8					newRateIndex;

	TX_INTERRUPT_SAVE_AREA;
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	vapBasicRateMask.raIndexMask64bit[0] = pLmVapDataBase->basicRates;
	ASSERT(vapBasicRateMask.raIndexMask64bit[0]);
	vapBasicRateMask.raIndexMask64bit[1] = 0;

	rateIndex = getLowestRateinMask(INVALID_STA_INDEX, vapBasicRateMask,BANDWIDTH_TWENTY,0,PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX); /*Get lower rate from vap basic rates*/

	if (fixedMcsVapManagement !=FIXED_MCS_VAP_MANAGEMENT_IS_NOT_VALID)
	{
		newRateIndex = convertTcr2RateIndex(PHY_MODE_11AG, fixedMcsVapManagement);
		if(isBitInMask64Bit(newRateIndex, vapBasicRateMask.raIndexMask64bit))
		{
			rateIndex = newRateIndex;
		}
		else
		{
			FATAL("The FW got wrong MCS for Vap Management ");
			highestRateIndexInMask = getHighestRateinMask(INVALID_STA_INDEX, vapBasicRateMask,BANDWIDTH_TWENTY,0,PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX); /*Get highest rate from vap basic rates*/

			ASSERT(highestRateIndexInMask != INVALID_RATE_INDEX_VAL);

			if (htVhtRatesTable[highestRateIndexInMask].rateTableBwParmas[BANDWIDTH_TWENTY].shortCpPhyRate < htVhtRatesTable[newRateIndex].rateTableBwParmas[BANDWIDTH_TWENTY].shortCpPhyRate )
			{
				rateIndex = highestRateIndexInMask;
			}
		}
	}
	
	/*Init the Multicast Rate Index*/
	pLmVapDataBase->mcRateIndex = rateIndex;
	getTcrPhyModeAndRate(htVhtRatesTable, 0, rateIndex, &phyMode, &tcrRateVal, FALSE, NO_DCM);  

	pVapDbHwEntry->common.maxMpduLengthLimit = getMpduMaxLengthLimit(FALSE); // Should be Initialized from the driver. Hard coded for release 0.5
	pVapDbHwEntry->common.bfReportResponseFormat = DISABLED; // SAS :set to 00: selection of no BF sequence // [15:14] 
	pVapDbHwEntry->common.qosEnableCapability = DISABLED;
	pVapDbHwEntry->common.negotiatedWds = DISABLED;		  // SAS :shopuld be set to 0					   [17]  
	pVapDbHwEntry->common.dataBwLimit = BANDWIDTH_TWENTY;	// Initiate by Link adaptation
	pVapDbHwEntry->common._80211BwLimit = BANDWIDTH_TWENTY; // Initiate by Link adaptation
	pVapDbHwEntry->common.dynamicBwEnable = DISABLED;		  // SAS :set to 00:No Dynamic BW transmission (legacy/static)	 [22]
	pVapDbHwEntry->common.staticBwEnable  = DISABLED;		  // SAS :set to 00:No Static BW transmission (legacy/static)		[23]
	pVapDbHwEntry->common.maxMsduAtAmsduCount = 1; 		  // SAS : set to 1:No A-MSDU							 [31:24]
	
	pVapDbHwEntry->common.maxNssNdp = CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(AntennaSelectionGetActivatedAntennasCount());
#ifndef ENET_INC_ARCH_WAVE600B
	pVapDbHwEntry->common.bfRxMode = BF_RX_MODE_DISABLED;
#else 
    pVapDbHwEntry->common.bfImplicitSupport = NOT_SUPPORTED;
    pVapDbHwEntry->common.bfBfmeeSupport = NOT_SUPPORTED;
#endif
	pVapDbHwEntry->common.tcr0DataAntennaSelection = ANT_SELECTION_ALL;   //  choose antena to transmit from; 
	pVapDbHwEntry->common.dynamicBwEnable = DISABLED; // specified in SAS to deafult value 0.
	pVapDbHwEntry->common.tcr2DataLdpc = CONVOLUTIONAL_CODING; 
	pVapDbHwEntry->common.tcr280211Ldpc = CONVOLUTIONAL_CODING; 	
	pVapDbHwEntry->common.tcr1BssColor = pLmVapDataBase->bssColor;
	pVapDbHwEntry->common.tcr1TxopPsNotAllowed =FALSE;//vapDbGetTxopPsAllowed(FALSE);  // assocParams->htCapabilityInfo.lSigTxop_protection_support;
	pVapDbHwEntry->common.tcr1GroupId = DEFAULT_MIMO_GROUP_ID;// Set to 0 for non VHT
	pVapDbHwEntry->common.tcr1BwChange = DISABLED; 
	pVapDbHwEntry->common.tcr0DataAnt0Boost = 0x0;
	pVapDbHwEntry->common.tcr0DataAnt1Boost = 0x0;
	pVapDbHwEntry->common.tcr0DataAnt2Boost = 0x0;
	pVapDbHwEntry->common.tcr0DataAnt3Boost = 0x0;

	/* 20MHZ DATA TCR*/
	pVapDbHwEntry->common.tcr0Data20MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr0Data20MhzCbw = BANDWIDTH_TWENTY;
	pVapDbHwEntry->common.tcr0Data20MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr0Data20MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_TWENTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr120MhzDataNotSounding = ENABLED;
	pVapDbHwEntry->common.tcr120MhzDataSmoothing = ENABLED;
	pVapDbHwEntry->common.tcr220MhzDataPsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr120MhzDataScp = DISABLED; 
	pVapDbHwEntry->common.tcr120MhzDataStbc = NOT_STBC_TRANSMISSION; 
	/*20MHZ MGMT TCR*/
	pVapDbHwEntry->common.tcr08021120MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr08021120MhzCbw = BANDWIDTH_TWENTY;
	pVapDbHwEntry->common.tcr08021120MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr08021120MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_TWENTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr120Mhz80211NotSounding = ENABLED;
	pVapDbHwEntry->common.tcr120Mhz80211Smoothing = ENABLED;
	pVapDbHwEntry->common.tcr220Mhz80211PsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr120Mhz80211Scp = DISABLED; 
	pVapDbHwEntry->common.tcr120Mhz80211Stbc = NOT_STBC_TRANSMISSION; 
	

	/* 40MHz DATA TCR*/
	pVapDbHwEntry->common.tcr0Data40MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr0Data40MhzCbw = BANDWIDTH_FOURTY;
	pVapDbHwEntry->common.tcr0Data40MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr0Data40MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_FOURTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr140MhzDataNotSounding = ENABLED;
	pVapDbHwEntry->common.tcr140MhzDataSmoothing = ENABLED;
	pVapDbHwEntry->common.tcr240MhzDataPsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr140MhzDataScp = DISABLED; 
	pVapDbHwEntry->common.tcr140MhzDataStbc = NOT_STBC_TRANSMISSION; 
	/* 40MHz MGMT TCR*/
	pVapDbHwEntry->common.tcr08021140MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr08021140MhzCbw = BANDWIDTH_FOURTY;
	pVapDbHwEntry->common.tcr08021140MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr08021140MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_FOURTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr140Mhz80211NotSounding = ENABLED;
	pVapDbHwEntry->common.tcr140Mhz80211Smoothing = ENABLED;
	pVapDbHwEntry->common.tcr240Mhz80211PsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr140Mhz80211Scp = DISABLED; 
	pVapDbHwEntry->common.tcr140Mhz80211Stbc = NOT_STBC_TRANSMISSION; 

	/* 80MHz DATA TCR*/
	pVapDbHwEntry->common.tcr0Data80MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr0Data80MhzCbw = BANDWIDTH_EIGHTY;
	pVapDbHwEntry->common.tcr0Data80MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr0Data80MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_EIGHTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr180MhzDataNotSounding = ENABLED;
	pVapDbHwEntry->common.tcr180MhzDataSmoothing = ENABLED;
	pVapDbHwEntry->common.tcr280MhzDataPsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr180MhzDataScp = DISABLED;
	pVapDbHwEntry->common.tcr180MhzDataStbc = NOT_STBC_TRANSMISSION;

	/*80MHz MGMT TCR*/
	pVapDbHwEntry->common.tcr08021180MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr08021180MhzCbw = BANDWIDTH_EIGHTY;
	pVapDbHwEntry->common.tcr08021180MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr08021180MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_EIGHTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr180Mhz80211NotSounding = ENABLED;
	pVapDbHwEntry->common.tcr180Mhz80211Smoothing = ENABLED;
	pVapDbHwEntry->common.tcr280Mhz80211PsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr180Mhz80211Scp = DISABLED; 
	pVapDbHwEntry->common.tcr180Mhz80211Stbc = NOT_STBC_TRANSMISSION; 

	/* 160MHz DATA TCR*/
	pVapDbHwEntry->common.tcr0Data160MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr0Data160MhzCbw = BANDWIDTH_ONE_HUNDRED_SIXTY;
	pVapDbHwEntry->common.tcr0Data160MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr0Data160MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_ONE_HUNDRED_SIXTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr1160MhzDataNotSounding = ENABLED;
	pVapDbHwEntry->common.tcr1160MhzDataSmoothing = ENABLED;
	pVapDbHwEntry->common.tcr2160MhzDataPsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr1160MhzDataScp = DISABLED; 
	pVapDbHwEntry->common.tcr1160MhzDataStbc = NOT_STBC_TRANSMISSION;
	
	/*160MHz MGMT TCR*/
	pVapDbHwEntry->common.tcr080211160MhzPhyMode = phyMode;
	pVapDbHwEntry->common.tcr080211160MhzCbw = BANDWIDTH_ONE_HUNDRED_SIXTY;
	pVapDbHwEntry->common.tcr080211160MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pVapDbHwEntry->common.tcr080211160MhzRfPower = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_ONE_HUNDRED_SIXTY); // transmission power  0x3F max power
	pVapDbHwEntry->common.tcr1160Mhz80211NotSounding = ENABLED;
	pVapDbHwEntry->common.tcr1160Mhz80211Smoothing = ENABLED;
	pVapDbHwEntry->common.tcr2160Mhz80211PsduRate = tcrRateVal; 
	pVapDbHwEntry->common.tcr1160Mhz80211Scp = DISABLED; 
	pVapDbHwEntry->common.tcr1160Mhz80211Stbc = NOT_STBC_TRANSMISSION; 

	PreAggregator_FillConfigurationDb(vapIndex,tcrRateVal,phyMode,tcrRateVal,phyMode);

	memset32(pVapDbHwEntry->tcrTemplates,0x0, CONVERT_BYTES_TO_WORDS(sizeof(pVapDbHwEntry->tcrTemplates))); 

	/* Common parameters for all templates */
	for(tcrIterator=0; tcrIterator < NUM_OF_VAP_TCR_TEMPLATES; tcrIterator++)
	{
		tcrCommonTemplate = (TcrCommon_t*)(&(pVapDbHwEntry->tcrTemplates[tcrIterator].word0)); 
		tcrCommonTemplate->antennaSelection = 0x0; 
		tcrCommonTemplate->ant0Boost = 0x0; 
		tcrCommonTemplate->ant1Boost = 0x0; 
		tcrCommonTemplate->ant2Boost = 0x0; 
		tcrCommonTemplate->ant3Boost = 0x0; 
		tcrCommonTemplate->cbw = BANDWIDTH_TWENTY; 
		tcrCommonTemplate->rfPower = 0x0; 
		tcrCommonTemplate->stbc = DISABLED; 
		tcrCommonTemplate->groupId = DEFAULT_MIMO_GROUP_ID;// group ID default value is 0 for VAP
		tcrCommonTemplate->muTraining = 0x0; 
		tcrCommonTemplate->muPhyNdp= 0x0; 
		tcrCommonTemplate->bwChange = DISABLED; 
		tcrCommonTemplate->aggregate = 0x0; 
		tcrCommonTemplate->notSounding= 0x1; 
		tcrCommonTemplate->smoothing = ENABLED; 
		tcrCommonTemplate->dynamicBw = 0x0; 
		tcrCommonTemplate->scp = DISABLED; 
		tcrCommonTemplate->hePuncturingMap = 0xFF;
		tcrCommonTemplate->txLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
			
		/* RF Power template*/
		pVapDbHwEntry->tcrTemplates[tcrIterator].rfPower20MhzTemplate = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_TWENTY);
		pVapDbHwEntry->tcrTemplates[tcrIterator].rfPower40MhzTemplate = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_FOURTY);
		pVapDbHwEntry->tcrTemplates[tcrIterator].rfPower80MhzTemplate = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_EIGHTY);
		pVapDbHwEntry->tcrTemplates[tcrIterator].rfPower160MhzTemplate = GetTcrDefaultPowerVal(0, rateIndex, BANDWIDTH_ONE_HUNDRED_SIXTY);
	}

	/* Template 1 specific parameters */
	// Remove 11b rates and add 11g rates
	vapBasicRateMask.raIndexMask64bit[0] = (pLmVapDataBase->basicRates & (~LM_PHY_11B_RATE_MSK)) | LM_PHY_11A_RATE_MSK;
	ASSERT(vapBasicRateMask.raIndexMask64bit[0]);
	vapBasicRateMask.raIndexMask64bit[1] = 0;
	rateIndexTcr1 = getLowestRateinMask(INVALID_STA_INDEX, vapBasicRateMask,BANDWIDTH_TWENTY,0,PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX); /*Get lower rate from vap basic rates*/
	getTcrPhyModeAndRate(htVhtRatesTable, 0, rateIndexTcr1, &phyModeTcr1, &tcrRateValTcr1, FALSE, NO_DCM);  
	
	tcrCommonTemplate = (TcrCommon_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word0)); 
	tcrCommonTemplate->phyMode = phyModeTcr1; 
	tcrUserTemplate = (TcrUser_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word8));
	tcrUserTemplate->rate = tcrRateValTcr1;

	/* Template 2 specific parameters */
	tcrCommonTemplate = (TcrCommon_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_MAC_HT_NDP_ANNOUNCEMENT].word0)); 
	tcrCommonTemplate->phyMode = PHY_MODE_11N; 
	tcrUserTemplate = (TcrUser_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_MAC_HT_NDP_ANNOUNCEMENT].word8));
	tcrUserTemplate->rate = MCS_0;

	/* Template 3 specific parameters */
	tcrCommonTemplate = (TcrCommon_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_HT_PHY_NDP].word0)); 
	tcrCommonTemplate->phyMode = PHY_MODE_11N; 
	tcrCommonTemplate->notSounding= 0; 
	tcrUserTemplate = (TcrUser_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_HT_PHY_NDP].word8));
	tcrUserTemplate->rate = MCS_1;

	/* Template 4 specific parameters */
	tcrCommonTemplate = (TcrCommon_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_VHT_PHY_NDP].word0)); 
	tcrCommonTemplate->phyMode = PHY_MODE_11AC; 
	tcrCommonTemplate->smoothing = 0;   // N/A in gen6 VHT
	tcrCommonTemplate->notSounding = 0; // N/A in gen6 VHT
	tcrUserTemplate = (TcrUser_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_VHT_PHY_NDP].word8));
	tcrUserTemplate->rate = MCS_1;
	tcrUserTemplate->stationAid = (pVapDbHwEntry->common.vapMode == VAP_MODE_AP)?((((pVapDbHwEntry->bssid.macAddress45 & 0xF000) >> 0xC) ^ ((pVapDbHwEntry->bssid.macAddress45 & 0x0F00) >> 0x8)) << 0x5):0; //according to SAS

	/* Template 5 specific parameters */
	tcrCommonTemplate = (TcrCommon_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_HE_PHY_NDP].word0)); 
	tcrCommonTemplate->phyMode = PHY_MODE_11AX_SU; 
	tcrCommonTemplate->smoothing = 0;   
	tcrCommonTemplate->notSounding = 0; 
	tcrCommonTemplate->txLoopMode = TX_LOOP_MODE_OPEN_LOOP;
	tcrCommonTemplate->heCp = HE_CP_HE_1_6_CP; //for he_ltf 2x he_cp should be 1.6uS; for he_ltf 4x he_cp should be 3.2uS
	tcrCommonTemplate->heLtf = HE_LTF_HE_2X_LTF; 
	tcrCommonTemplate->bssColor = pLmVapDataBase->bssColor;
	tcrUserTemplate = (TcrUser_t*)(&(pVapDbHwEntry->tcrTemplates[TCR_TEMPLATE_ID_HE_PHY_NDP].word8));
	tcrUserTemplate->rate = MCS_1;


	pVapDbHwEntry->tid.baEnable = DISABLED;							  /* BA transmission enabled				     [0] 	   */
	pVapDbHwEntry->tid.ppduTxMode = VAP_DB_PPDU_TX_MODE_LEGACY_ONLY;  /* 0-legacy only 1-legacy+MPDU 2-two level aggregation  [2:1] 	   */
	pVapDbHwEntry->tid.maxPsduTransmissionTimeLimit = (MAX_PSDU_TX_TIME_LIMIT>>1); // max time resolution is 2microSec
	pVapDbHwEntry->tid.currentBaWindowLimit = 0; 			  /* no limit on BA window							     [21:16] */

	if (phyMode == PHY_MODE_11B)
	{
		/* reset antenna boosts */
		uint8 antBoost[4] = {0};
		setAntBoostInVapTcr(vapIndex, antBoost);
	}

	OSAL_ENABLE_INTERRUPTS(interrupt_save);

}

/********************************************************************************
resetSequenceNumderInVapHwDb





Description:
----------

Input:
-----

Output:
-------

Returns:
--------

********************************************************************************/
void resetSequenceNumderInVapHwDb (uint8 vapIndex)
{
	VapDb_t*				pVapDbHwEntry = &VapDbHwEntries[vapIndex];

	pVapDbHwEntry->tid.currentSequenceNumber = 0; /* [0:11] */	
	pVapDbHwEntry->tid.lowSequenceNumber = 0;	 /* [27:16] */
}

/********************************************************************************
fillStaHwDb




Description:
----------

Input:
-----

Output:
-------

Returns:
--------

********************************************************************************/

void fillStaHwDb(UMI_STA_ADD *pAddSta, uint8 initialRate)
{
	uint8 		stationIndex = pAddSta->u16SID;
	StaDb_t*    pStaDbHwEntry =  &StaDbHwEntries[stationIndex];
	uint8 tidIndex;
	AmpduParameters_t*		pAmpduParameters = (AmpduParameters_t*)&pAddSta->u8AMPDU_Param;
	uint8 DummytcrRateVal;
	bool isStaVht = LinkAdaptationStaDatabase[stationIndex].laStaUspCommon.staTransmissionParams.vhtSta;
	PhyMode_e phyMode;
	StaDbTid_t* 	  	pStaTidEntry = NULL;	
	HtCapabilitiesInfo_t 	htCapabilities;
	sVHT_CAPABILITIES_INFO vhtCapabilities;
	HE_MAC_PHY_CAPABILITIES_INFO* pHeCapInfo = (HE_MAC_PHY_CAPABILITIES_INFO*)&pAddSta->u8HE_Mac_Phy_Cap_Info;
	LinkAdaptationPhyMode_e laPhyMode;
	uint8 isXFilterOpen = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[stationIndex].laStaUspCommon);
	uint8 maximumAmpdulengthExponent = 0;
#ifdef ENET_INC_ARCH_WAVE600D2	
	uint8 indexTid;
#endif
	MEMCPY(&htCapabilities, &(pAddSta->u16HT_Cap_Info) , sizeof(sHT_CAPABILITIES_INFO) );
	MEMCPY(&vhtCapabilities, &(pAddSta->u32VHT_Cap_Info) , sizeof(sVHT_CAPABILITIES_INFO) );
	
	getTcrPhyModeAndRate(ratesTable, isStaVht,initialRate, &phyMode, &DummytcrRateVal, FALSE, NO_DCM); 

	laPhyMode = ConvertTcrPhyMode2LaPhyMode[phyMode];

	pStaDbHwEntry->common.bfReportResponseFormat = BF_RE_FORMAT_IMMEDIATE_REPORT_EXPECTED; // StaDb_GetBfReportResponseFormat(assocParams->txBfCapabilities); // [15:14] 
	pStaDbHwEntry->common.qosEnableCapability = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_WME); 		// [16] 	
	if ((isXFilterOpen == FALSE) && (pAddSta->u8WDS_client_type == FOUR_ADDRESSES_STATION))
	{
		pStaDbHwEntry->common.negotiatedWds = FALSE;	/*If filter is closed and 4 address STA, set WDS = 0 so EAPOL are sent in 3 address*/
	}
	else
	{
		pStaDbHwEntry->common.negotiatedWds = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_WDS);			// [17]   should be 0 -no WDS
	}
	pStaDbHwEntry->common.tcr1BssColor = LmVapDataBase[pAddSta->u8VapIndex].bssColor;
	pStaDbHwEntry->common.tcr1CustomBf = 0;
	pStaDbHwEntry->common.dynamicBwEnable = DISABLED;								// [22]
	pStaDbHwEntry->common.staticBwEnable  = DISABLED;								// [23] 	 
	pStaDbHwEntry->common.wdsClientType = pAddSta->u8WDS_client_type;
	pStaDbHwEntry->common.psForceOneNss = 1;

	if (MTLK_BFIELD_GET(pAddSta->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE) == TRUE)
	{
		pStaDbHwEntry->common.psForceOneNss = 0;
#ifdef ENET_INC_ARCH_WAVE600D2
		for (indexTid = 0; indexTid < NUM_OF_TID; indexTid++)
		{
			pStaDbHwEntry->tid[indexTid].fragLevel = pHeCapInfo->heMacCapInfo.HE_MAC_FRAGMENTATION_SUPPORT; // TODO: should be given per TID so there is a different name fragLevel; minimumlength  - same for all TIDs
		}
#else
		pStaDbHwEntry->common.fragLevel = pHeCapInfo->heMacCapInfo.HE_MAC_FRAGMENTATION_SUPPORT; 
#endif
		// TODO: need to initalize fragment counter for the added station: REG_OTFA_SW_UPDATE_FRAG_DB: 	swUpdateFragDbStaId  = x;	swUpdateFragDbValue  = 0;	swUpdateFragDbAction  = 1;  swUpdateFragDbDone	= polling till done;
		pStaDbHwEntry->common.dynamicBwEnable = LinkAdaptationRtsMode.dynamicBw;				
		pStaDbHwEntry->common.staticBwEnable = LinkAdaptationRtsMode.staticBw;
		pStaDbHwEntry->common.maxMsduAtAmsduCount = laHeMsduInAmsdu;
#ifdef ENET_INC_ARCH_WAVE600D2
		pStaDbHwEntry->common.heSuErRu106Support = pHeCapInfo->hePhyCapInfo.HE_PHY_PARTIAL_BANDWIDTH_EXTENDED_RANGE;
		pStaDbHwEntry->common.heDcmSupport = pHeCapInfo->hePhyCapInfo.HE_PHY_DCM_MAX_CONSTELLATION_RX >= DCM_MAX_CONSTELLATION_FIRST_SUPPORTED? TRUE: FALSE;
		pStaDbHwEntry->common.heLdpcSupport = pHeCapInfo->hePhyCapInfo.HE_PHY_LDPC_CODING_IN_PAYLOAD;
#endif //ENET_INC_ARCH_WAVE600D2
		
		if (MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_VHT) == TRUE)
		{
            pStaDbHwEntry->common.maxMpduLengthLimit = calculateMpduMaxLengthLimitVht(vhtCapabilities.max_mpdu_len); //the he max mpdu length capability is same for HT & VHT one (no special one for the HE)
			switch (pHeCapInfo->heMacCapInfo.HE_MAC_MAXIMUM_A_MPDU_LENGTH_EXPONENT)
			{
				case 0:
					maximumAmpdulengthExponent = vhtCapabilities.maximum_ampdu_length_exponent;
					break;
				case 1:
					maximumAmpdulengthExponent = VHT_MAX_AMPDU_LENGTH_EXPONENT + 1;
					break;
				case 2:
					maximumAmpdulengthExponent = VHT_MAX_AMPDU_LENGTH_EXPONENT + 2;
					break;
				case 3:
					maximumAmpdulengthExponent = VHT_MAX_AMPDU_LENGTH_EXPONENT + 3;
			}
		}
		else //HE+HT
		{
			if (QAMplus_get() == TRUE)
			{
				/* if QAM+ enabled, use VHT max capabilities for HE 2.4G (same as in QAM PLUS for HT 2.4G) */
                pStaDbHwEntry->common.maxMpduLengthLimit = calculateMpduMaxLengthLimitVht(2);
				maximumAmpdulengthExponent = VHT_MAX_AMPDU_LENGTH_EXPONENT;
			}
			else
			{
                pStaDbHwEntry->common.maxMpduLengthLimit = calculateMpduMaxLengthLimitHt(htCapabilities.maximalAmsduSize); //the he max mpdu length capability is same for HT & VHT one (no special one for the HE)
				switch (pHeCapInfo->heMacCapInfo.HE_MAC_MAXIMUM_A_MPDU_LENGTH_EXPONENT)
				{
					case 0:
						maximumAmpdulengthExponent = pAmpduParameters->maximumAmpdulengthExponent;
						break;
					case 1:
						maximumAmpdulengthExponent = HT_MAX_AMPDU_LENGTH_EXPONENT + 1;
						break;
					case 2:
						maximumAmpdulengthExponent = HT_MAX_AMPDU_LENGTH_EXPONENT + 2;
						break;
					case 3:
						FATAL("bad param - station assoc request must be rejected in host apd");
				}
			}
		}
	}
	else if (MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_VHT) == TRUE)
	{
		maximumAmpdulengthExponent = vhtCapabilities.maximum_ampdu_length_exponent;

		/* calculate ( 2 power (13 + maxMpduLengthLimit)) -1 */
		pStaDbHwEntry->common.maxMpduLengthLimit = calculateMpduMaxLengthLimitVht(vhtCapabilities.max_mpdu_len); 
		
		/* Is this should be taken from 8 -123(supported channel width set in HT capabilities table
			or from 8-129 ( HT operation element STA channel width. Also only 20/40 is supported  */
		pStaDbHwEntry->common.maxMsduAtAmsduCount = laVhtMsduInAmsdu;		
		if ((vhtCapabilities.rx_stbc & STA_CAPABILITIES_VHT_STBC_SUPPORT_MASK) >= STBC_CAPABILITIES_SUPPORT_1_2_NSS)
		{
			//In order to prevent forbiden case when STBC 2x4 will be transmitted with 1NSS after aggregation builder will set 1NSS if sta in PS, we set this bit on zero if station support STBC 2X4
			pStaDbHwEntry->common.psForceOneNss = 0;
		}

		pStaDbHwEntry->common.dynamicBwEnable = LinkAdaptationRtsMode.dynamicBw;								// [22]
		pStaDbHwEntry->common.staticBwEnable  = LinkAdaptationRtsMode.staticBw;								// [23] 	 

	}
	else if (MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_HT) == TRUE)
	{
		
		maximumAmpdulengthExponent = pAmpduParameters->maximumAmpdulengthExponent;
		pStaDbHwEntry->common.maxMpduLengthLimit =  calculateMpduMaxLengthLimitHt(htCapabilities.maximalAmsduSize);
		pStaDbHwEntry->common.maxMsduAtAmsduCount = laHtMsduInAmsdu;
		if ((htCapabilities.rxStbc & STA_CAPABILITIES_HT_STBC_SUPPORT_MASK) >= STBC_CAPABILITIES_SUPPORT_1_2_NSS)
		{
			//In order to prevent forbiden case when STBC 2x4 will be transmitted with 1NSS after aggregation builder will set 1NSS if sta in PS, we set this bit on zero if station support STBC 2X4
			pStaDbHwEntry->common.psForceOneNss = 0; 
		}

	}
	else
	{
		/* This is the default value for 11A/G, for 11B the field should not be used by the aggregation builder */
		pStaDbHwEntry->common.maxMpduLengthLimit = MAX_HT_MPDU_IN_A_MPDU_LENGTH_TX;
	}
	pStaDbHwEntry->common.maxPsduLengthLimit = getPsduMaxLengthLimit(maximumAmpdulengthExponent); /* [19:0]  */
	pStaDbHwEntry->common.durationUpdateMode = DURATION_UPDATE_MODE_FULL_TXOP; //working with full txop mode
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
	pStaDbHwEntry->wp.maxPsduLengthLimit = pStaDbHwEntry->common.maxPsduLengthLimit;
	pStaDbHwEntry->wp.durationUpdateMode = pStaDbHwEntry->common.durationUpdateMode;
	// NOTE: this storage must be done before function rateAdaptationChangeWorkingPoint is called (from linkAdaptationAddStationReq). DON"T CHANGE THE ORDER!
	originalSuMaxPsduLengthLimit[stationIndex] = pStaDbHwEntry->common.maxPsduLengthLimit;
#endif
	pStaDbHwEntry->common.staPhyCapabilities = phyMode;
	pStaDbHwEntry->common.noAggInPs = 1;
	pStaDbHwEntry->common.sppAmsduCapable = DISABLED;
	pStaDbHwEntry->common.maxSpLength = CALC_MAX_SP_LENGTH(pAddSta->u8Max_SP);   /* maximal service period  [10:3] 	*/
	pStaDbHwEntry->common.txPostpone = TX_POSTPONE_CTS2SELF;
	pStaDbHwEntry->common.currentPacketDensity = pAmpduParameters->minimumMpduStartSpacing; /* [27:25] */
	pStaDbHwEntry->common.rtsCtsTxMethod = RTS_CTS_TX_METHOD_NO_RTS_CTS;		  /* [29:28] */
	pStaDbHwEntry->common.ctsToSelfTxMethod = CTS_TO_SELF_TX_METHOD_NO_CTS2SELF;	  /* [30:31] */




	/*****************************/
	/*			ZONE B (TCRs)		*/
	/*****************************/

	/* 20MHz Data TCR*/

	pStaDbHwEntry->common.tcr0Data20MhzCbw = BANDWIDTH_TWENTY;
	pStaDbHwEntry->common.tcr0Data20MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr120MhzDataNotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr120MhzDataSmoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr120MhzDataScp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr120MhzDataHeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr120MhzDataHeLtf = HE_LTF_HE_2X_LTF;

	/* 20MHz Management TCR*/

	pStaDbHwEntry->common.tcr08021120MhzCbw = BANDWIDTH_TWENTY;
	pStaDbHwEntry->common.tcr08021120MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr120Mhz80211NotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr120Mhz80211Smoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr120Mhz80211Scp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr120Mhz80211Stbc = NOT_STBC_TRANSMISSION;
	pStaDbHwEntry->common.tcr120Mhz80211HeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr120Mhz80211HeLtf = HE_LTF_HE_2X_LTF;

	/* 40MHz Data TCR*/

	pStaDbHwEntry->common.tcr0Data40MhzCbw = BANDWIDTH_FOURTY;
	pStaDbHwEntry->common.tcr0Data40MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr140MhzDataNotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr140MhzDataSmoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr140MhzDataScp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr140MhzDataHeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr140MhzDataHeLtf = HE_LTF_HE_2X_LTF;

	/* 40MHz Management TCR*/

	pStaDbHwEntry->common.tcr08021140MhzCbw = BANDWIDTH_FOURTY;
	pStaDbHwEntry->common.tcr08021120MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr140Mhz80211NotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr140Mhz80211Smoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr140Mhz80211Scp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr140Mhz80211Stbc = NOT_STBC_TRANSMISSION;
	pStaDbHwEntry->common.tcr140Mhz80211HeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr140Mhz80211HeLtf = HE_LTF_HE_2X_LTF;
	
	/* 80MHz Data TCR*/

	pStaDbHwEntry->common.tcr0Data80MhzCbw = BANDWIDTH_EIGHTY;
	pStaDbHwEntry->common.tcr0Data80MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr180MhzDataNotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr180MhzDataSmoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr180MhzDataScp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr180MhzDataHeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr180MhzDataHeLtf = HE_LTF_HE_2X_LTF;

	/* 80MHz Management TCR*/

	pStaDbHwEntry->common.tcr08021180MhzCbw = BANDWIDTH_EIGHTY;
	pStaDbHwEntry->common.tcr08021180MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr180Mhz80211NotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr180Mhz80211Smoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr180Mhz80211Scp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr180Mhz80211Stbc = NOT_STBC_TRANSMISSION;
	pStaDbHwEntry->common.tcr180Mhz80211HeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr180Mhz80211HeLtf = HE_LTF_HE_2X_LTF;

	/* 160MHz Data TCR*/

	pStaDbHwEntry->common.tcr0Data160MhzCbw = BANDWIDTH_ONE_HUNDRED_SIXTY;
	pStaDbHwEntry->common.tcr0Data160MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr1160MhzDataNotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr1160MhzDataSmoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr1160MhzDataScp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr1160MhzDataHeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr1160MhzDataHeLtf = HE_LTF_HE_2X_LTF;


	/* 160MHz Management TCR*/

	pStaDbHwEntry->common.tcr080211160MhzCbw = BANDWIDTH_ONE_HUNDRED_SIXTY;
	pStaDbHwEntry->common.tcr080211160MhzTxLoopMode = TX_LOOP_MODE_CLOSED_LOOP;
	pStaDbHwEntry->common.tcr1160Mhz80211NotSounding = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr1160Mhz80211Smoothing = ENABLED; // specified in SAS to deafult value 0.
	pStaDbHwEntry->common.tcr1160Mhz80211Scp = 0; //will be set later by dedicated function
	pStaDbHwEntry->common.tcr1160Mhz80211Stbc = NOT_STBC_TRANSMISSION;
	pStaDbHwEntry->common.tcr1160Mhz80211HeCp = HE_CP_HE_0_8_CP;
	pStaDbHwEntry->common.tcr1160Mhz80211HeLtf = HE_LTF_HE_2X_LTF;

	/*Those fields are relevant for all BWs*/
	pStaDbHwEntry->common.tcr2PacketExtension = 2; // In Z0 - set the maximum (2 means 16usec) and don't take it from sta capabilities
	pStaDbHwEntry->common.tcr2DataLdpc = getLdpcCapability(stationIndex,laPhyMode);	
	pStaDbHwEntry->common.tcr2VhtSuPartialAid = (pAddSta->u16AID + ((((VapDbHwEntries[pAddSta->u8VapIndex].bssid.macAddress45 & 0xF000) >> 0xC) ^ ((VapDbHwEntries[pAddSta->u8VapIndex].bssid.macAddress45 & 0x0F00) >> 0x8)) << 0x5)) & 0x01FF; //according to SAS
	pStaDbHwEntry->common.tcr1TxopPsNotAllowed = TRUE;
	pStaDbHwEntry->common.tcr1GroupId = DEFAULT_MIMO_GROUP_ID;
	pStaDbHwEntry->common.tcr280211Ldpc = CONVOLUTIONAL_CODING; 	
	pStaDbHwEntry->common.altRateReasonFastProbing = 0;//will be changed later by dedicated function
	pStaDbHwEntry->common.altRateReasonSlowProbing = 0;//will be changed later by dedicated function

	// Is this taken from ASEL Capability field in Capabilities info IE 
	pStaDbHwEntry->common.tcr1BwChange = DISABLED;
	pStaDbHwEntry->common.tcr0DataAnt0Boost = 0;
	pStaDbHwEntry->common.tcr0DataAnt1Boost = 0;
	pStaDbHwEntry->common.tcr0DataAnt2Boost = 0;
	pStaDbHwEntry->common.tcr0DataAnt3Boost = 0;	
	pStaDbHwEntry->common.tcr0DataAnt4Boost = 0;
	pStaDbHwEntry->common.tcr0DataAnt5Boost = 0;
	pStaDbHwEntry->common.tcr0DataAnt6Boost = 0;
	pStaDbHwEntry->common.tcr0DataAnt7Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant0Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant1Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant2Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant3Boost = 0;	
	pStaDbHwEntry->common.tcr080211Ant4Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant5Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant6Boost = 0;
	pStaDbHwEntry->common.tcr080211Ant7Boost = 0;

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400

	memcpy32(&pStaDbHwEntry->wp.word2, &pStaDbHwEntry->common.word12, 1);
	memcpy32(&pStaDbHwEntry->wp.word3, &pStaDbHwEntry->common.word13, 1);
	memcpy32(&pStaDbHwEntry->wp.word4, &pStaDbHwEntry->common.word14, 1);
	memcpy32(&pStaDbHwEntry->wp.word5, &pStaDbHwEntry->common.word21, 1);
	memcpy32(&pStaDbHwEntry->wp.word6, &pStaDbHwEntry->common.word24, 1);	
	memcpy32(&pStaDbHwEntry->wp.word7, &pStaDbHwEntry->common.word26, 1);
#else
	memcpy32(&pStaDbHwEntry->wp.word0, &pStaDbHwEntry->common.word12, 1);
	memcpy32(&pStaDbHwEntry->wp.word1, &pStaDbHwEntry->common.word13, 1);
	memcpy32(&pStaDbHwEntry->wp.word2, &pStaDbHwEntry->common.word14, 1);
	memcpy32(&pStaDbHwEntry->wp.word3, &pStaDbHwEntry->common.word21, 1);
	memcpy32(&pStaDbHwEntry->wp.word4, &pStaDbHwEntry->common.word24, 1);	
	memcpy32(&pStaDbHwEntry->wp.word5, &pStaDbHwEntry->common.word26, 1);
#endif
	
	/* INITIALIZE the TID info for all TIDs (x8) */
	for ( tidIndex = 0 ; tidIndex < NUM_OF_TOTAL_TIDS ; tidIndex++)
	{
		pStaTidEntry = (StaDbTid_t*)&(pStaDbHwEntry->tid[tidIndex]);
		/* DW 0 */
		pStaTidEntry->currentSequenceNumber = 0; /* [0:11]  */	
		pStaTidEntry->lowSequenceNumber = 0;     /* [27:16] */
		/* DW 1 */
		pStaTidEntry->baEnable = DISABLED;							  /* BA transmission enabled							[0] 	   */		
		pStaTidEntry->ppduTxMode = laPpduTxMode;  // values can be seen in PpduTxMode_e		
		pStaTidEntry->currentBaWindowLimit = 0; 		      /* no limit on BA window						          [21:16]    */
		pStaTidEntry->txopMultipleDataEn = DISABLED;
		/*Relevant only for 11N (by configuration)*/
		pStaTidEntry->maxPsduTransmissionTimeLimit = (MAX_PSDU_TX_TIME_LIMIT>>1); // max time resolution is 2microSec;
		
#ifdef ENET_INC_ARCH_WAVE600D2

		if (MTLK_BFIELD_GET(pAddSta->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE) == TRUE)
		{
			pStaTidEntry->fragLevel = pHeCapInfo->heMacCapInfo.HE_MAC_FRAGMENTATION_SUPPORT; 
		}
#endif //ENET_INC_ARCH_WAVE600D2
	}
}
/**********************************************************************************

setVapFixedRatesInHwDb 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void setVapFixedRatesInHwDb(void)
{
	uint32 protectedDBscratchPad[PROTECTED_VAP_DB_SIZE_WORDS];
	uint8	vapid= FixedRateParameters.vapId;
	VapDb_t* pVapDbHwEntries = &VapDbHwEntries[vapid];
	uint8*	 pRfPowerTemplate = NULL;
	bool isVht = (FixedRateParameters.phyMode == PHY_MODE_11AC) ? TRUE : FALSE;
	uint8 rateIndex;
	
	/*If Change from non 11B to 11B, first set BW and after it set rate*/
	if (FixedRateParameters.phyMode != PHY_MODE_11B)
	{
		/*Set bw limit*/
		setVapFixedBwLimit(FixedRateParameters.bandwidth, pVapDbHwEntries);
		/*Set rate*/
		ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pVapDbHwEntries->common.word12), (PROTECTED_VAP_DB_SIZE_WORDS), protectedDBscratchPad, setVapFixedRateInTcrsModificationFunc,(void*)&FixedRateParameters); 
		
	}
	else
	{
		/*Set rate*/
		ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pVapDbHwEntries->common.word12), (PROTECTED_VAP_DB_SIZE_WORDS), protectedDBscratchPad, setVapFixedRateInTcrsModificationFunc,(void*)&FixedRateParameters); 
		/*Set bw limit*/
		setVapFixedBwLimit(FixedRateParameters.bandwidth, pVapDbHwEntries);

	}

	// Set RTS Power template for all BWs (template 1 in RF Power Template)
	
	rateIndex = convertTcr2RateIndex(FixedRateParameters.phyMode, FixedRateParameters.rate);
	pRfPowerTemplate = (uint8*)(&(pVapDbHwEntries->tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word12)); // point to rfPowerTemplates - for reference as an aray per BW 
	pRfPowerTemplate = &(pRfPowerTemplate[FixedRateParameters.bandwidth]); 

	*pRfPowerTemplate = GetTcrDefaultPowerVal(isVht, rateIndex, FixedRateParameters.bandwidth);
}
/**********************************************************************************

setVapFixedRateInTcrsModificationFunc 


Description: tx mngmnt legacy rate.  DUT uses mngmnt to tx data for all 802.11
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setVapFixedRateInTcrsModificationFunc(uint32* scratchPadAddress, void* functionParams)
{
	LaFixedRateParams_t* fixedRateParams = (LaFixedRateParams_t*) functionParams; 
	LaVapProtectedZone_t* laProtectedStruct = (LaVapProtectedZone_t*) scratchPadAddress;
	uint8 rateIndex = convertTcr2RateIndex(fixedRateParams->phyMode, fixedRateParams->rate);
	bool isVht = (fixedRateParams->phyMode == PHY_MODE_11AC) ? TRUE : FALSE;
	uint8 powerTcrVal;
	CyclicPrefixMode_e cpMode = fixedRateParams->cpMode;
	
	HeCp_e heCp;// = cpModeToHeCpTcrVal[cpMode];
	HeLtf_e heLtf;// = cpModeToHeLtfTcrVal[cpMode];	

	//KW_FIX_FW_M
	ASSERT(cpMode < CP_NUM_OF_MODES);
	heCp = cpModeToHeCpTcrVal[cpMode];
	heLtf = cpModeToHeLtfTcrVal[cpMode];

	powerTcrVal = GetTcrDefaultPowerVal(isVht, rateIndex, fixedRateParams->bandwidth);

	if((fixedRateParams->changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (fixedRateParams->changeType == LA_PACKET_TYPE_DATA))
	{	
		if (fixedRateParams->phyMode >= PHY_MODE_11AX_SU)
		{
			laProtectedStruct->tcr2DataLdpc = LDPC;			
		}
		else
		{
			laProtectedStruct->tcr2DataLdpc = CONVOLUTIONAL_CODING;			
		}
		switch (fixedRateParams->bandwidth)
		{
			case BANDWIDTH_TWENTY: 	
				laProtectedStruct->tcr0Data20MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr220MhzDataPsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr0Data20MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr120MhzDataHeCp = heCp;				
					laProtectedStruct->tcr120MhzDataHeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr120MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 
			case BANDWIDTH_FOURTY:
				laProtectedStruct->tcr0Data40MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr240MhzDataPsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr0Data40MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr140MhzDataHeCp = heCp;				
					laProtectedStruct->tcr140MhzDataHeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr140MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 
			case BANDWIDTH_EIGHTY: 
				laProtectedStruct->tcr0Data80MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr280MhzDataPsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr0Data80MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr180MhzDataHeCp = heCp;				
					laProtectedStruct->tcr180MhzDataHeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr180MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 	
			case BANDWIDTH_ONE_HUNDRED_SIXTY: 
				laProtectedStruct->tcr0Data160MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr2160MhzDataPsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr0Data160MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr1160MhzDataHeCp = heCp;				
					laProtectedStruct->tcr1160MhzDataHeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr1160MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 	
			default:
				ASSERT(0);
				break;
		}
	}
	if((fixedRateParams->changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (fixedRateParams->changeType == LA_PACKET_TYPE_MANAGEMENT))
	{
		if (fixedRateParams->phyMode >= PHY_MODE_11AX_SU)
		{
			laProtectedStruct->tcr280211Ldpc = LDPC;			
		}
		else
		{
			laProtectedStruct->tcr280211Ldpc = CONVOLUTIONAL_CODING;			
		}
		switch (fixedRateParams->bandwidth)
		{
			case BANDWIDTH_TWENTY: 	
				laProtectedStruct->tcr08021120MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr220Mhz80211PsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr08021120MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr120Mhz80211HeCp = heCp;				
					laProtectedStruct->tcr120Mhz80211HeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr120Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 
			case BANDWIDTH_FOURTY:
				laProtectedStruct->tcr08021140MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr240Mhz80211PsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr08021140MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr140Mhz80211HeCp = heCp;				
					laProtectedStruct->tcr140Mhz80211HeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr140Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 
			case BANDWIDTH_EIGHTY: 
				laProtectedStruct->tcr08021180MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr280Mhz80211PsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr08021180MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr180Mhz80211HeCp = heCp;				
					laProtectedStruct->tcr180Mhz80211HeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr180Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 	
			case BANDWIDTH_ONE_HUNDRED_SIXTY: 
				laProtectedStruct->tcr080211160MhzPhyMode = fixedRateParams->phyMode;				
				laProtectedStruct->tcr2160Mhz80211PsduRate = fixedRateParams->rate;
				laProtectedStruct->tcr080211160MhzRfPower = powerTcrVal;
				if (fixedRateParams->phyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->tcr1160Mhz80211HeCp = heCp;				
					laProtectedStruct->tcr1160Mhz80211HeLtf= heLtf;
				}
				else if ((fixedRateParams->phyMode == PHY_MODE_11N) || (fixedRateParams->phyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->tcr1160Mhz80211Scp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
				break; 	
			default:
				ASSERT(0);
				break;
		}
	}
}
	
/********************************************************************************
setStaFixedBwLimit




Description:
----------

Input:
-----

Output:
-------

Returns:
--------

********************************************************************************/

static void setStaFixedBwLimit(StaId staIndex)
{
	LaFixedRateStationParams_t *laFixedRateStationParams_p = &LinkAdaptationStaDatabase[staIndex].laStationUnique.fixedRateStationParams;
	
	if(laFixedRateStationParams_p->changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) 
	{
		SetStaDataBwLimitInHwDb(staIndex,laFixedRateStationParams_p->bandwidth,TRUE);
		SetStaManagementBwLimit(staIndex,laFixedRateStationParams_p->bandwidth);
	}
	else if (laFixedRateStationParams_p->changeType == LA_PACKET_TYPE_DATA)
	{
		SetStaDataBwLimitInHwDb(staIndex,laFixedRateStationParams_p->bandwidth,TRUE);
	}
	else if (laFixedRateStationParams_p->changeType == LA_PACKET_TYPE_MANAGEMENT)
	{
		SetStaManagementBwLimit(staIndex,laFixedRateStationParams_p->bandwidth);
	}
}

/********************************************************************************
setVapFixedBwLimit




Description:
----------

Input:
-----

Output:
-------

Returns:
--------

********************************************************************************/

static void setVapFixedBwLimit(Bandwidth_e bandwidth, VapDb_t* pVapDbHwEntries)
{
	UNUSED_PARAM(bandwidth);	
	
	if(FixedRateParameters.changeType == LA_PACKET_TYPE_DATA_MANAGEMENT) 
	{
		pVapDbHwEntries->common.dataBwLimit = FixedRateParameters.bandwidth;
		pVapDbHwEntries->common._80211BwLimit = FixedRateParameters.bandwidth;

	}
	else if (FixedRateParameters.changeType == LA_PACKET_TYPE_DATA)
	{
		pVapDbHwEntries->common.dataBwLimit = FixedRateParameters.bandwidth;

	}
	else if (FixedRateParameters.changeType == LA_PACKET_TYPE_MANAGEMENT)
	{
		pVapDbHwEntries->common._80211BwLimit = FixedRateParameters.bandwidth;
	}
}
/**********************************************************************************

setPowerInStaTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void SetPowerInVapTcr(uint8 vapIndex, int8* pPowerValTable, Bandwidth_e bandwidth, Bandwidth_e lastBwToUpdate, bool changeProbingPoint, LaPacketType_e packetType)
{

	ProtectedDbModFuncPowerParams_t protectedDbfuncParams;
	uint32 protectedDBscratchPad[PROTECTED_VAP_DB_SIZE_WORDS];
	uint8*	VapDbRfPowerTemplate0Tcr = (uint8 *)&(VapDbHwEntries[vapIndex].tcrTemplates[TCR_TEMPLATE_ID_NOT_SOUNDING_COMMON].word12); // template0
	uint8*	VapDbRfPowerTemplate1Tcr = (uint8 *)&(VapDbHwEntries[vapIndex].tcrTemplates[TCR_TEMPLATE_ID_MAC_HT_NDP_ANNOUNCEMENT].word12); // template1
	uint8*	VapDbRfPowerTemplate2Tcr = (uint8 *)&(VapDbHwEntries[vapIndex].tcrTemplates[TCR_TEMPLATE_ID_HT_PHY_NDP].word12); // template2
	uint8*	VapDbRfPowerTemplate3Tcr = (uint8 *)&(VapDbHwEntries[vapIndex].tcrTemplates[TCR_TEMPLATE_ID_VHT_PHY_NDP].word12); // template3
	Bandwidth_e bwIndex;

	DEBUG_ASSERT((bandwidth <= LinkAdaptationCommonConfiguration.wlanBandwidthMax) && (lastBwToUpdate <= LinkAdaptationCommonConfiguration.wlanBandwidthMax));
	/*Update new working point in STA DB using protected DB*/
	protectedDbfuncParams.changeProbingPoint = changeProbingPoint;
	protectedDbfuncParams.bandwidth = bandwidth;
	protectedDbfuncParams.lastBwToUpdate = lastBwToUpdate;
	protectedDbfuncParams.pTcrDataPowerValTable = pPowerValTable;
	protectedDbfuncParams.packetType = packetType;

	ProtectedDbLock_ReadModifyWriteReq((uint32*)&(VapDbHwEntries[vapIndex].common.word12), (PROTECTED_VAP_DB_SIZE_WORDS), protectedDBscratchPad, setPowerInVapDbModificationFunc,(void*)&protectedDbfuncParams); 

	// Set RTS Power template for all BWs (template 1 in RF Power Template)

	if ((packetType == LA_PACKET_TYPE_DATA) || (packetType == LA_PACKET_TYPE_DATA_MANAGEMENT) || (packetType == LA_PACKET_TYPE_MU_DATA))
	{
		for (bwIndex = bandwidth; bwIndex <= lastBwToUpdate; bwIndex++)
		{
			VapDbRfPowerTemplate0Tcr[bwIndex] = pPowerValTable[bwIndex];
			VapDbRfPowerTemplate1Tcr[bwIndex] = pPowerValTable[bwIndex];
			VapDbRfPowerTemplate2Tcr[bwIndex] = pPowerValTable[bwIndex];
			VapDbRfPowerTemplate3Tcr[bwIndex] = pPowerValTable[bwIndex];
		}
	}
}

/**********************************************************************************

setPowerInStaDbModificationFunc 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setPowerInVapDbModificationFunc(uint32* scratchPadAddress, void* functionParams)
{
	ProtectedDbModFuncPowerParams_t* powerParams = (ProtectedDbModFuncPowerParams_t*) functionParams;
	LaVapProtectedZone_t* laProtectedStruct = (LaVapProtectedZone_t*) scratchPadAddress;
	
	//KW_IGNORE: Comparison of unsigned variable lastBwtoUpdate against 0 is always true.. Should we keep this code for better readability?
	if ((BANDWIDTH_TWENTY >= powerParams->bandwidth) && (BANDWIDTH_TWENTY <= powerParams->lastBwToUpdate))
	{
		// BW 20 MHz
		if ((powerParams->packetType == LA_PACKET_TYPE_DATA) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr0Data20MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_TWENTY];
		}
		if ((powerParams->packetType == LA_PACKET_TYPE_MANAGEMENT) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr08021120MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_TWENTY];
		}
	}

	if ((BANDWIDTH_FOURTY >= powerParams->bandwidth) && (BANDWIDTH_FOURTY <= powerParams->lastBwToUpdate))
	{	
		// BW 40 MHz
		if ((powerParams->packetType == LA_PACKET_TYPE_DATA) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr0Data40MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_FOURTY];
		}
		if ((powerParams->packetType == LA_PACKET_TYPE_MANAGEMENT) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr08021140MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_FOURTY];
		}
	}

	if ((BANDWIDTH_EIGHTY >= powerParams->bandwidth) && (BANDWIDTH_EIGHTY <= powerParams->lastBwToUpdate))
	{		
		// BW 80 MHz
		if ((powerParams->packetType == LA_PACKET_TYPE_DATA) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr0Data80MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_EIGHTY];
		}
		if ((powerParams->packetType == LA_PACKET_TYPE_MANAGEMENT) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr08021180MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_EIGHTY];
		}
	}
	
	if ((BANDWIDTH_ONE_HUNDRED_SIXTY >= powerParams->bandwidth) && (BANDWIDTH_ONE_HUNDRED_SIXTY <= powerParams->lastBwToUpdate))
	{	
		// BW 160 MHz
		if ((powerParams->packetType == LA_PACKET_TYPE_DATA) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr0Data160MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_ONE_HUNDRED_SIXTY];
		}
		if ((powerParams->packetType == LA_PACKET_TYPE_MANAGEMENT) || (powerParams->packetType == LA_PACKET_TYPE_DATA_MANAGEMENT))
		{
			laProtectedStruct->tcr080211160MhzRfPower = powerParams->pTcrDataPowerValTable[BANDWIDTH_ONE_HUNDRED_SIXTY];
		}
	}
}



/**********************************************************************************

SetAntSelectionInVapTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void AntennaSelectionSetBitmapInVapTcr(uint8 vapId, uint8 antSelectionMask, LaPacketType_e packetType)
{

	ProtectedDbModFuncAntSelectionParams_t protectedDbfuncParams;
	uint32 protectedDBscratchPad[PROTECTED_VAP_DB_SIZE_WORDS];

	/*Update new working point in STA DB using protected DB*/
	protectedDbfuncParams.changeProbingPoint = FALSE;
	protectedDbfuncParams.tcrAntennaSelctionMask = antSelectionMask;
	protectedDbfuncParams.packetType = packetType;

	ProtectedDbLock_ReadModifyWriteReq((uint32*)&(VapDbHwEntries[vapId].common.word12), (PROTECTED_VAP_DB_SIZE_WORDS), protectedDBscratchPad, setAntennaSelectionInVapDbModificationFunc,(void*)&protectedDbfuncParams); 
}

/**********************************************************************************

SetAntSelectionInAutoReplyTcr 


Description:
------------
	Set the antenna selection field of TCR 0 for auto response frames

Input: 
-----
	antSelectionMask - the Value of anrtenna selection field  in the TCR
	
**********************************************************************************/
void AntennaSelectionSetBitmapInAutoReplyTcr(uint8 antSelectionMask)
{	
 	pDeliaAutoResponseTcrs->common.antennaSelection = antSelectionMask;
}

/**********************************************************************************


CyclicPrefixSetModeInStaTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setAntennaSelectionInVapDbModificationFunc(uint32* scratchPadAddress, void* functionParams)
{
	ProtectedDbModFuncAntSelectionParams_t* antSelectionParams = (ProtectedDbModFuncAntSelectionParams_t*) functionParams;
	LaVapProtectedZone_t* laProtectedStruct = (LaVapProtectedZone_t*) scratchPadAddress;

	DEBUG_ASSERT(antSelectionParams->tcrAntennaSelctionMask > 0);

	switch (antSelectionParams->packetType)
	{
		case(LA_PACKET_TYPE_DATA):
		{
			laProtectedStruct->tcr0DataAntennaSelection = antSelectionParams->tcrAntennaSelctionMask;
			break;
		}
		case(LA_PACKET_TYPE_MANAGEMENT):
		{
			laProtectedStruct->tcr080211AntennaSelection = antSelectionParams->tcrAntennaSelctionMask;
			break;
		}
		case(LA_PACKET_TYPE_DATA_MANAGEMENT):
		{
			laProtectedStruct->tcr0DataAntennaSelection = antSelectionParams->tcrAntennaSelctionMask;
			laProtectedStruct->tcr080211AntennaSelection = antSelectionParams->tcrAntennaSelctionMask;
			break;
		}
		default:
		{
			DEBUG_ASSERT(0);
			break;
		}
	}
}

/***************************************************************************
**
** NAME         antennaSelectionSetMuGroupAntenna
**
** PARAMETERS   
**
** DESCRIPTION  

****************************************************************************/
void  antennaSelectionSetMuGroupAntenna(uint8 groupId, uint8 antennaSelection, bool isHeGroup)
{
	uint8 uspIndex;
	LinkAdaptationDatabaseDistributionPack_t distributionParams; 
	MuGrpDbCommon_t* pGroupDbCommon = NULL;
	uint8 numOfAntennas = AntennaSelectionCalcAntennaCount(antennaSelection);

	if (isHeGroup)
	{
		for(uspIndex = 0; uspIndex < MAX_USP_IN_HE_GROUP; uspIndex++)
		{
			if(LinkAdaptationHePhaseDatabase[groupId].laHeGroupUnique.groupStationIndexes[uspIndex] != INVALID_STA_INDEX)
			{
				updateLaDbDistributionParam(&distributionParams,groupId,uspIndex, isHeGroup);
				/*Set rate adaptation mask according to num of antennas*/
				setMaskAccordingToMaxNumOfAntennas(&distributionParams, numOfAntennas); 
				/*Adjust rate to num of antennas*/
				rateAdaptationAdjustRateToMask(&distributionParams);
			}
		}
		// TBD 
	}
	else
	{
		for(uspIndex = 0; uspIndex < MAX_USP_IN_VHT_GROUP; uspIndex++)
		{
			if(LinkAdaptationVhtGroupDatabase[groupId].laVhtGroupUnique.groupStationIndexes[uspIndex] != INVALID_STA_INDEX)
			{
				updateLaDbDistributionParam(&distributionParams,groupId,uspIndex, isHeGroup);
				/*Set rate adaptation mask according to num of antennas*/
				setMaskAccordingToMaxNumOfAntennas(&distributionParams, numOfAntennas); 
				/*Adjust rate to num of antennas*/
				rateAdaptationAdjustRateToMask(&distributionParams);
			}
		}
		pGroupDbCommon =  &(GroupDbHwEntries[groupId].common);
		pGroupDbCommon->tcr0DataAntennaSelection = antennaSelection;
	}
}

/**********************************************************************************

SetCDDInVapTcr 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void CddSetInVapTcr(uint8 vapId, CddValues_t *cddVals)
{
	UNUSED_PARAM(vapId);
	UNUSED_PARAM(cddVals);
}

/**********************************************************************************

SetCddInAutoReplyTcr 


Description:
------------
	Set CDD vals of the Auto Response TCR 

Input: 
-----
	cddVals- Cdd default Values for the number of antennas configured

Returns:
--------
	void - 
	
**********************************************************************************/
void CddSetInAutoReplyTcr(CddValues_t cddVals)
{	
	UNUSED_PARAM(cddVals);
}
#ifdef SDL_IGNORE_UNUSED_FUNCTION	
/**********************************************************************************


setCddnVapDbModificationFunc 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setCddnVapDbModificationFunc(uint32* scratchPadAddress, void* functionParams)
{

}
#endif
/**********************************************************************************

GetVapRateIndex


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 GetVapRateIndex(uint8 vapId , LaPacketType_e packetType, Bandwidth_e bandwidth)
{
	VapDb_t* vapDb = &VapDbHwEntries[vapId];
	PhyMode_e wpPhyMode = PHY_MODE_LAST;
	uint8 wpTcrRate = 0;
	uint8 rateIndex;

	if (packetType == LA_PACKET_TYPE_DATA)
	{
		switch (bandwidth)
		{
			case BANDWIDTH_TWENTY: 	
				wpTcrRate = vapDb->common.tcr220MhzDataPsduRate;
				wpPhyMode = vapDb->common.tcr0Data20MhzPhyMode;
				break; 
			case BANDWIDTH_FOURTY:
				wpTcrRate = vapDb->common.tcr240MhzDataPsduRate;
				wpPhyMode = vapDb->common.tcr0Data40MhzPhyMode;
				break; 
			case BANDWIDTH_EIGHTY: 
				wpTcrRate = vapDb->common.tcr280MhzDataPsduRate;
				wpPhyMode = vapDb->common.tcr0Data80MhzPhyMode;
				break; 	
			case BANDWIDTH_ONE_HUNDRED_SIXTY: 
				wpTcrRate = vapDb->common.tcr2160MhzDataPsduRate;
				wpPhyMode = vapDb->common.tcr0Data160MhzPhyMode;
				break; 	
			default:
				ASSERT(0);
				break;
		}
	}
	else if (packetType == LA_PACKET_TYPE_MANAGEMENT)
	{
		switch (bandwidth)
		{
			case BANDWIDTH_TWENTY: 	
				wpTcrRate = vapDb->common.tcr220Mhz80211PsduRate;
				wpPhyMode = vapDb->common.tcr08021120MhzPhyMode;
				break; 
			case BANDWIDTH_FOURTY:
				wpTcrRate = vapDb->common.tcr240Mhz80211PsduRate;
				wpPhyMode = vapDb->common.tcr08021140MhzPhyMode;
				break; 
			case BANDWIDTH_EIGHTY: 
				wpTcrRate = vapDb->common.tcr280Mhz80211PsduRate;
				wpPhyMode = vapDb->common.tcr08021180MhzPhyMode;
				break; 	
			case BANDWIDTH_ONE_HUNDRED_SIXTY: 
				wpTcrRate = vapDb->common.tcr2160Mhz80211PsduRate;
				wpPhyMode = vapDb->common.tcr080211160MhzPhyMode;
				break; 	
			default:
				DEBUG_ASSERT(0)
				break;
		}
	}
	else
	{
		ASSERT(0);
	}
	rateIndex = convertTcr2RateIndex(wpPhyMode,wpTcrRate);

	return rateIndex;
}

/**********************************************************************************

setTcrinHwDb


Description:
------------
Function called by protected DB API and write all TCR params to HW DB

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setTcrsInHwStaDb(uint32* scratchPadAddress, void* functionParams)
{	
	LaTcrModificationStruct_t* pTcrModificationParams = (LaTcrModificationStruct_t*) functionParams;
	LaStaProtectedZone_t* laProtectedStruct = (LaStaProtectedZone_t*) scratchPadAddress;

	uint8 tcrRate;
	uint8 tcrPhyMode;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[pTcrModificationParams->controlParams.staIndex].laStaUspCommon);
	uint8 maxNumberOfRatesInTable = getMaxNumOfRatesInTable(&LinkAdaptationStaDatabase[pTcrModificationParams->controlParams.staIndex].laStaUspCommon);
	CyclicPrefixMode_e cpMode;

	HeCp_e heCp;
	HeLtf_e heLtf;	

	DEBUG_ASSERT(pTcrModificationParams->controlParams.uspIndex == INVALID_MU_USP_INDEX);
	DEBUG_ASSERT(pTcrModificationParams->controlParams.staIndex < HW_NUM_OF_STATIONS);
	DEBUG_ASSERT(pTcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection > 0);

	if (pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
	{
		laProtectedStruct->common.tcr2DataLdpc = pTcrModificationParams->tcrParams.tcrGeneralVals.ldpcMode;
		laProtectedStruct->common.tcr0DataAntennaSelection = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection;
		if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
		{
			laProtectedStruct->wp.tcr0DataAntennaSelection = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection;
			/*Clear probing bits - we may be moving to working point while waiting for probing*/
			laProtectedStruct->common.altRateReasonFastProbing = 0;
			laProtectedStruct->common.altRateReasonSlowProbing = 0;
		}
		else
		{
			/*Set probing bit*/
			laProtectedStruct->common.altRateReasonFastProbing = 1;
			laProtectedStruct->common.altRateReasonSlowProbing = 1;
		}
	}
	else if (pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_MANAGEMENT)
	{
		laProtectedStruct->common.tcr080211AntennaSelection = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection;
	}
	else
	{
		DEBUG_ASSERT(0);
	}
	
	/* 20MHz TCR*/
	if ((BANDWIDTH_TWENTY >= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_TWENTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		DEBUG_ASSERT(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].rateindex < maxNumberOfRatesInTable);
		
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta,pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].rateindex,&tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		cpMode = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode;
		ASSERT(cpMode < CP_NUM_OF_MODES);
		heCp = cpModeToHeCpTcrVal[cpMode];
		heLtf = cpModeToHeLtfTcrVal[cpMode];
		
		if (pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
			laProtectedStruct->common.tcr0Data20MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr220MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrBfMode;
			laProtectedStruct->common.tcr0Data20MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrPower;
			laProtectedStruct->common.tcr220MhzDataPsduRate = tcrRate;
			laProtectedStruct->common.tcr120MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
			laProtectedStruct->common.tcr120MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].BfSmothing;

			laProtectedStruct->common.tcr120MhzDataHeCp = 0;
			laProtectedStruct->common.tcr120MhzDataHeLtf = 0;
			laProtectedStruct->common.tcr120MhzDataScp = 0;

			if (tcrPhyMode == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr120MhzDataHeCp = heCp;
				laProtectedStruct->common.tcr120MhzDataHeLtf = heLtf;
			}
			else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr120MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}
			
			if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
			{
				laProtectedStruct->wp.tcr0Data20MhzPhyMode = tcrPhyMode;
				laProtectedStruct->wp.tcr220MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrBfMode;
				laProtectedStruct->wp.tcr0Data20MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrPower;
				laProtectedStruct->wp.tcr220MhzDataPsduRate = tcrRate;
				laProtectedStruct->wp.tcr120MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
				laProtectedStruct->wp.tcr120MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].BfSmothing;

				laProtectedStruct->wp.tcr120MhzDataHeCp = 0;
				laProtectedStruct->wp.tcr120MhzDataHeLtf = 0;
				laProtectedStruct->wp.tcr120MhzDataScp = 0;

				if (tcrPhyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->wp.tcr120MhzDataHeCp = heCp;
					laProtectedStruct->wp.tcr120MhzDataHeLtf = heLtf;
				}
				else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->wp.tcr120MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
			}	
		}
		else
		{
			laProtectedStruct->common.tcr08021120MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr220Mhz80211Txbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrBfMode;
			laProtectedStruct->common.tcr08021120MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrPower;
			laProtectedStruct->common.tcr220Mhz80211PsduRate = tcrRate;
			laProtectedStruct->common.tcr120Mhz80211Smoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].BfSmothing;		
			laProtectedStruct->common.tcr120Mhz80211Stbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
		}
	}

	/* 40MHz TCR*/
	if ((BANDWIDTH_FOURTY >= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_FOURTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		DEBUG_ASSERT(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].rateindex < maxNumberOfRatesInTable);
		
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta,pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].rateindex,&tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		cpMode = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode;
		
		//KW_FIX_FW_M
		ASSERT(cpMode < CP_NUM_OF_MODES);
		heCp = cpModeToHeCpTcrVal[cpMode];
		heLtf = cpModeToHeLtfTcrVal[cpMode];

		if (pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
			laProtectedStruct->common.tcr0Data40MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr240MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrBfMode;
			laProtectedStruct->common.tcr0Data40MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrPower;
			laProtectedStruct->common.tcr240MhzDataPsduRate = tcrRate;
			laProtectedStruct->common.tcr140MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
			laProtectedStruct->common.tcr140MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].BfSmothing;

			laProtectedStruct->common.tcr140MhzDataHeCp = 0;
			laProtectedStruct->common.tcr140MhzDataHeLtf = 0;			
			laProtectedStruct->common.tcr140MhzDataScp = 0;
			
			if (tcrPhyMode == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr140MhzDataHeCp = heCp;
				laProtectedStruct->common.tcr140MhzDataHeLtf = heLtf;
			}
			else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr140MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}

			if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
			{
				laProtectedStruct->wp.tcr0Data40MhzPhyMode = tcrPhyMode;
				laProtectedStruct->wp.tcr240MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrBfMode;
				laProtectedStruct->wp.tcr0Data40MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrPower;
				laProtectedStruct->wp.tcr240MhzDataPsduRate = tcrRate;
				laProtectedStruct->wp.tcr140MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
				laProtectedStruct->wp.tcr140MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].BfSmothing;

				laProtectedStruct->wp.tcr140MhzDataHeCp = 0;
				laProtectedStruct->wp.tcr140MhzDataHeLtf = 0;
				laProtectedStruct->wp.tcr140MhzDataScp = 0;

				if (tcrPhyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->wp.tcr140MhzDataHeCp = heCp;
					laProtectedStruct->wp.tcr140MhzDataHeLtf = heLtf;
				}
				else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->wp.tcr140MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
			}			
		}
		else
		{
			laProtectedStruct->common.tcr08021140MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr240Mhz80211Txbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrBfMode;
			laProtectedStruct->common.tcr08021140MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrPower;
			laProtectedStruct->common.tcr240Mhz80211PsduRate = tcrRate;
			laProtectedStruct->common.tcr140Mhz80211Smoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].BfSmothing;		
			laProtectedStruct->common.tcr140Mhz80211Stbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
		}
	}

	/* 80MHz TCR*/
	if ((BANDWIDTH_EIGHTY >= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_EIGHTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		DEBUG_ASSERT(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex < maxNumberOfRatesInTable);
		
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta,pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex,&tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		cpMode = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode;

		//KW_FIX_FW_M
		ASSERT(cpMode < CP_NUM_OF_MODES);
		heCp = cpModeToHeCpTcrVal[cpMode];
		heLtf = cpModeToHeLtfTcrVal[cpMode];

		if (pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
			laProtectedStruct->common.tcr0Data80MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr280MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrBfMode;
			laProtectedStruct->common.tcr0Data80MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrPower;
			laProtectedStruct->common.tcr280MhzDataPsduRate = tcrRate;
			laProtectedStruct->common.tcr180MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
			laProtectedStruct->common.tcr180MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].BfSmothing;
			
			laProtectedStruct->common.tcr180MhzDataHeCp = 0;
			laProtectedStruct->common.tcr180MhzDataHeLtf = 0;		
			laProtectedStruct->common.tcr180MhzDataScp = 0;
			
			if (tcrPhyMode == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr180MhzDataHeCp = heCp;
				laProtectedStruct->common.tcr180MhzDataHeLtf = heLtf;
			}
			else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr180MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}

			if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
			{
				laProtectedStruct->wp.tcr0Data80MhzPhyMode = tcrPhyMode;
				laProtectedStruct->wp.tcr280MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrBfMode;
				laProtectedStruct->wp.tcr0Data80MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrPower;
				laProtectedStruct->wp.tcr280MhzDataPsduRate = tcrRate;
				laProtectedStruct->wp.tcr180MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
				laProtectedStruct->wp.tcr180MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].BfSmothing;
	
				laProtectedStruct->wp.tcr180MhzDataHeCp = 0;
				laProtectedStruct->wp.tcr180MhzDataHeLtf = 0;
				laProtectedStruct->wp.tcr180MhzDataScp = 0;
				
				if (tcrPhyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->wp.tcr180MhzDataHeCp = heCp;
					laProtectedStruct->wp.tcr180MhzDataHeLtf = heLtf;
				}
				else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->wp.tcr180MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
			}			
		}
		else
		{
			laProtectedStruct->common.tcr08021180MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr280Mhz80211Txbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrBfMode;
			laProtectedStruct->common.tcr08021180MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrPower;
			laProtectedStruct->common.tcr280Mhz80211PsduRate = tcrRate;
			laProtectedStruct->common.tcr180Mhz80211Smoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].BfSmothing;		
			laProtectedStruct->common.tcr180Mhz80211Stbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
		}
	}
	/* 160MHz TCR*/
	if ((BANDWIDTH_ONE_HUNDRED_SIXTY >= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_ONE_HUNDRED_SIXTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		DEBUG_ASSERT(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].rateindex < maxNumberOfRatesInTable);
		
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta,pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].rateindex,&tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		cpMode = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode;

		//KW_FIX_FW_M
		ASSERT(cpMode < CP_NUM_OF_MODES);
		heCp = cpModeToHeCpTcrVal[cpMode];
		heLtf = cpModeToHeLtfTcrVal[cpMode];		

		if (pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
			overwriteMaxPsduLengthLimitInStaDb(laProtectedStruct, pTcrModificationParams, tcrPhyMode, tcrRate);
#endif 
			laProtectedStruct->common.tcr0Data160MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr2160MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrBfMode;
			laProtectedStruct->common.tcr0Data160MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrPower;
			laProtectedStruct->common.tcr2160MhzDataPsduRate = tcrRate;
			laProtectedStruct->common.tcr1160MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
			laProtectedStruct->common.tcr1160MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].BfSmothing;
			
			laProtectedStruct->common.tcr1160MhzDataHeCp = 0;
			laProtectedStruct->common.tcr1160MhzDataHeLtf = 0;
			laProtectedStruct->common.tcr1160MhzDataScp = 0;
			
			if (tcrPhyMode == PHY_MODE_11AX_SU)
			{
				laProtectedStruct->common.tcr1160MhzDataHeCp = heCp;
				laProtectedStruct->common.tcr1160MhzDataHeLtf = heLtf;
			}
			else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
			{
				laProtectedStruct->common.tcr1160MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
			}

			if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
			{
				laProtectedStruct->wp.tcr0Data160MhzPhyMode = tcrPhyMode;
				laProtectedStruct->wp.tcr2160MhzDataTxbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrBfMode;
				laProtectedStruct->wp.tcr0Data160MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrPower;
				laProtectedStruct->wp.tcr2160MhzDataPsduRate = tcrRate;
				laProtectedStruct->wp.tcr1160MhzDataStbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
				laProtectedStruct->wp.tcr1160MhzDataSmoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].BfSmothing;
				
				laProtectedStruct->wp.tcr1160MhzDataHeCp = 0;
				laProtectedStruct->wp.tcr1160MhzDataHeLtf = 0;
				laProtectedStruct->wp.tcr1160MhzDataScp = 0;
				
				if (tcrPhyMode == PHY_MODE_11AX_SU)
				{
					laProtectedStruct->wp.tcr1160MhzDataHeCp = heCp;
					laProtectedStruct->wp.tcr1160MhzDataHeLtf = heLtf;
				}
				else if ((tcrPhyMode == PHY_MODE_11N) || (tcrPhyMode == PHY_MODE_11AC))
				{
					laProtectedStruct->wp.tcr1160MhzDataScp = convertCpModeToHtVhtCpTcrVal(cpMode);
				}
			}			
		}
		else
		{
			laProtectedStruct->common.tcr080211160MhzPhyMode = tcrPhyMode;
			laProtectedStruct->common.tcr2160Mhz80211Txbf = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrBfMode;
			laProtectedStruct->common.tcr080211160MhzRfPower = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrPower;
			laProtectedStruct->common.tcr2160Mhz80211PsduRate = tcrRate;
			laProtectedStruct->common.tcr1160Mhz80211Smoothing = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].BfSmothing;		
			laProtectedStruct->common.tcr1160Mhz80211Stbc = pTcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode;
		}

	}
}
/**********************************************************************************

setTcrsInHwGroupDb



Description:
------------
Function called by protected DB API and write all TCR params to HW DB

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setTcrsInHwGroupDb(uint32* scratchPadAddress, void* functionParams)
{
	LaTcrModificationStruct_t* pTcrModificationParams = (LaTcrModificationStruct_t*) functionParams;
	LaMuGrpProtectedZone_t* laMuGrpProtectedStruct = (LaMuGrpProtectedZone_t*) scratchPadAddress;
	uint8 tcrPhyMode;
	uint8 rateIndex;
	uint8 tcrRate;
	const RateObj_t* ratesTable =  getRatesTable(&LinkAdaptationVhtGroupDatabase[pTcrModificationParams->controlParams.staIndex].laStaUspCommon[pTcrModificationParams->controlParams.uspIndex]);
	uint8 maxNumOfRatesInTable = getMaxNumOfRatesInTable(&LinkAdaptationVhtGroupDatabase[pTcrModificationParams->controlParams.staIndex].laStaUspCommon[pTcrModificationParams->controlParams.uspIndex]);
	uint8 tcrCp;

	DEBUG_ASSERT(pTcrModificationParams->controlParams.uspIndex != INVALID_MU_USP_INDEX); 
	DEBUG_ASSERT(pTcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA); 
	ASSERT(pTcrModificationParams->controlParams.uspIndex < MAX_USP_IN_VHT_GROUP);

	/* 20MHz TCR*/
	if ((BANDWIDTH_TWENTY >= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_TWENTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		tcrCp = convertCpModeToHtVhtCpTcrVal(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode);
		rateIndex = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].rateindex;
		DEBUG_ASSERT(rateIndex < maxNumOfRatesInTable);
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta, rateIndex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		switch (pTcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				laMuGrpProtectedStruct->common.user0Tcr220MhzDataPsduRate = tcrRate;
				break;	
			case USP_USP_1:
				laMuGrpProtectedStruct->common.user1Tcr220MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_2:
				laMuGrpProtectedStruct->common.user2Tcr220MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_3:
				laMuGrpProtectedStruct->common.user3Tcr220MhzDataPsduRate = tcrRate;
				break;	
		};
		laMuGrpProtectedStruct->common.tcr120MhzDataScp = tcrCp;
		if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
		{
			switch (pTcrModificationParams->controlParams.uspIndex)
			{
				case USP_USP_0:
					laMuGrpProtectedStruct->wp.user0Tcr220MhzDataPsduRate = tcrRate;
					break;	
				case USP_USP_1:
					laMuGrpProtectedStruct->wp.user1Tcr220MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_2:
					laMuGrpProtectedStruct->wp.user2Tcr220MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_3:
					laMuGrpProtectedStruct->wp.user3Tcr220MhzDataPsduRate = tcrRate;
					break;	
			};
			laMuGrpProtectedStruct->wp.tcr120MhzDataScp = tcrCp;
		}
	}
	/* 40MHz TCR*/
	if ((BANDWIDTH_FOURTY>= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_FOURTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		tcrCp = convertCpModeToHtVhtCpTcrVal(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode);
		rateIndex = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].rateindex;
		DEBUG_ASSERT(rateIndex < maxNumOfRatesInTable);
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta, rateIndex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		switch (pTcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				laMuGrpProtectedStruct->common.user0Tcr240MhzDataPsduRate = tcrRate;
				break;	
			case USP_USP_1:
				laMuGrpProtectedStruct->common.user1Tcr240MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_2:
				laMuGrpProtectedStruct->common.user2Tcr240MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_3:
				laMuGrpProtectedStruct->common.user3Tcr240MhzDataPsduRate = tcrRate;
				break;	
		};
		laMuGrpProtectedStruct->common.tcr140MhzDataScp = tcrCp;
		if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
		{
			switch (pTcrModificationParams->controlParams.uspIndex)
			{
				case USP_USP_0:
					laMuGrpProtectedStruct->wp.user0Tcr240MhzDataPsduRate = tcrRate;
					break;	
				case USP_USP_1:
					laMuGrpProtectedStruct->wp.user1Tcr240MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_2:
					laMuGrpProtectedStruct->wp.user2Tcr240MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_3:
					laMuGrpProtectedStruct->wp.user3Tcr240MhzDataPsduRate = tcrRate;
					break;	
			};
			laMuGrpProtectedStruct->wp.tcr140MhzDataScp = tcrCp;
		}
	}
	/* 80MHz TCR*/
	if ((BANDWIDTH_EIGHTY>= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_EIGHTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		tcrCp = convertCpModeToHtVhtCpTcrVal(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode);
		rateIndex = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex;
		DEBUG_ASSERT(rateIndex < maxNumOfRatesInTable);
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta, rateIndex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		switch (pTcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				laMuGrpProtectedStruct->common.user0Tcr280MhzDataPsduRate = tcrRate;
				break;	
			case USP_USP_1:
				laMuGrpProtectedStruct->common.user1Tcr280MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_2:
				laMuGrpProtectedStruct->common.user2Tcr280MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_3:
				laMuGrpProtectedStruct->common.user3Tcr280MhzDataPsduRate = tcrRate;
				break;	
		};
		laMuGrpProtectedStruct->common.tcr180MhzDataScp = tcrCp;
		if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
		{
			switch (pTcrModificationParams->controlParams.uspIndex)
			{
				case USP_USP_0:
					laMuGrpProtectedStruct->wp.user0Tcr280MhzDataPsduRate = tcrRate;
					break;	
				case USP_USP_1:
					laMuGrpProtectedStruct->wp.user1Tcr280MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_2:
					laMuGrpProtectedStruct->wp.user2Tcr280MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_3:
					laMuGrpProtectedStruct->wp.user3Tcr280MhzDataPsduRate = tcrRate;
					break;	
			};
			laMuGrpProtectedStruct->wp.tcr180MhzDataScp = tcrCp;
		}
	}
	/* 160MHz TCR*/
	if ((BANDWIDTH_ONE_HUNDRED_SIXTY>= pTcrModificationParams->controlParams.firstBwToUpdate) && (BANDWIDTH_ONE_HUNDRED_SIXTY <= pTcrModificationParams->controlParams.lastBwToUpdate))
	{
		tcrCp = convertCpModeToHtVhtCpTcrVal(pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode);
		rateIndex = pTcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].rateindex;
		DEBUG_ASSERT(rateIndex < maxNumOfRatesInTable);
		/*Get phymode and rate according to rate index*/
		getTcrPhyModeAndRate(ratesTable, pTcrModificationParams->controlParams.isVhtSta, rateIndex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
		switch (pTcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				laMuGrpProtectedStruct->common.user0Tcr2160MhzDataPsduRate = tcrRate;
				break;	
			case USP_USP_1:
				laMuGrpProtectedStruct->common.user1Tcr2160MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_2:
				laMuGrpProtectedStruct->common.user2Tcr2160MhzDataPsduRate = tcrRate;
				break;
			case USP_USP_3:
				laMuGrpProtectedStruct->common.user3Tcr2160MhzDataPsduRate = tcrRate;
				break;	
		};
		laMuGrpProtectedStruct->common.tcr1160MhzDataScp = tcrCp;
		if (pTcrModificationParams->controlParams.changeProbingPoint == FALSE)
		{
			switch (pTcrModificationParams->controlParams.uspIndex)
			{
				case USP_USP_0:
					laMuGrpProtectedStruct->wp.user0Tcr2160MhzDataPsduRate = tcrRate;
					break;	
				case USP_USP_1:
					laMuGrpProtectedStruct->wp.user1Tcr2160MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_2:
					laMuGrpProtectedStruct->wp.user2Tcr2160MhzDataPsduRate = tcrRate;
					break;
				case USP_USP_3:
					laMuGrpProtectedStruct->wp.user3Tcr2160MhzDataPsduRate = tcrRate;
					break;	
			};
			laMuGrpProtectedStruct->wp.tcr1160MhzDataScp = tcrCp;
		}
	}
	
	if (pTcrModificationParams->controlParams.fastProbing == TRUE)
	{
		laMuGrpProtectedStruct->common.altRateReasonFastProbing |= (1 << pTcrModificationParams->controlParams.uspIndex);
	}
	else if (pTcrModificationParams->controlParams.slowProbing == TRUE)
	{
		laMuGrpProtectedStruct->common.altRateReasonSlowProbing |= (1 << pTcrModificationParams->controlParams.uspIndex);
	}
	else
	{
		laMuGrpProtectedStruct->common.altRateReasonFastProbing &= (~(1 << pTcrModificationParams->controlParams.uspIndex));
		laMuGrpProtectedStruct->common.altRateReasonSlowProbing &= (~(1 << pTcrModificationParams->controlParams.uspIndex));
	}
//	SLOG0(0, 0, LaMuGrpProtectedZone_t, laMuGrpProtectedStruct);
}


/**********************************************************************************

getMuTcrParamsFromHwDb



Description:
------------
Functino fills tcrModificationParams accordig to HW DB

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void getVhtMuTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams)
{
	MuGrpDb_t*  pGroupDbHwEntry = &(GroupDbHwEntries[tcrModificationParams->controlParams.staIndex]); // [NUM_GROUP_DBASE_ENTRIES];
	uint8 rateIndex;
	uint8 tcrRate=0;
	uint8 ldpcMode=0;//KW_FIX_FW_G
	
	/* Fields are taken from WP section. If they aren't exist there, take from common */
	
	ASSERT(tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA);

	/* General */
	tcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection = pGroupDbHwEntry->common.tcr0DataAntennaSelection;
	switch (tcrModificationParams->controlParams.uspIndex)
	{
		case USP_USP_0:
			ldpcMode = pGroupDbHwEntry->common.user0Tcr2DataLdpc;
			break;	
		case USP_USP_1:
			ldpcMode = pGroupDbHwEntry->common.user1Tcr2DataLdpc;
			break;
		case USP_USP_2:
			ldpcMode = pGroupDbHwEntry->common.user2Tcr2DataLdpc;
			break;
		case USP_USP_3:
			ldpcMode = pGroupDbHwEntry->common.user3Tcr2DataLdpc;
			break;	
	};
	tcrModificationParams->tcrParams.tcrGeneralVals.ldpcMode = ldpcMode;
	
	/* Per BW */
	
	/* 20 Mhz */
	switch (tcrModificationParams->controlParams.uspIndex)
	{
		case USP_USP_0:
			tcrRate = pGroupDbHwEntry->wp.user0Tcr220MhzDataPsduRate;
			break;	
		case USP_USP_1:
			tcrRate = pGroupDbHwEntry->wp.user1Tcr220MhzDataPsduRate;
			break;
		case USP_USP_2:
			tcrRate = pGroupDbHwEntry->wp.user2Tcr220MhzDataPsduRate;
			break;
		case USP_USP_3:
			tcrRate = pGroupDbHwEntry->wp.user3Tcr220MhzDataPsduRate;
			break;	
	};
	rateIndex = convertTcr2RateIndex(PHY_MODE_11AC, tcrRate);
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].rateindex = rateIndex;
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrPower = pGroupDbHwEntry->wp.tcr0Data20MhzRfPower; 
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode = htVhtCpTcrValToCpMode[pGroupDbHwEntry->wp.tcr120MhzDataScp]; 
	tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrBfMode = pGroupDbHwEntry->common.tcr2DataTxbf;	

	/* 40 Mhz */
	if (BANDWIDTH_FOURTY <= LinkAdaptationCommonConfiguration.wlanBandwidthMax)
	{
		switch (tcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				tcrRate = pGroupDbHwEntry->wp.user0Tcr240MhzDataPsduRate;
				break;	
			case USP_USP_1:
				tcrRate = pGroupDbHwEntry->wp.user1Tcr240MhzDataPsduRate;
				break;
			case USP_USP_2:
				tcrRate = pGroupDbHwEntry->wp.user2Tcr240MhzDataPsduRate;
				break;
			case USP_USP_3:
				tcrRate = pGroupDbHwEntry->wp.user3Tcr240MhzDataPsduRate;
				break;	
		};
		rateIndex = convertTcr2RateIndex(PHY_MODE_11AC, tcrRate);
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].rateindex = rateIndex;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrPower = pGroupDbHwEntry->wp.tcr0Data40MhzRfPower; 
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode = htVhtCpTcrValToCpMode[pGroupDbHwEntry->wp.tcr140MhzDataScp]; 
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrBfMode = pGroupDbHwEntry->common.tcr2DataTxbf;	
	}
	
	/* 80 Mhz */
	if (BANDWIDTH_EIGHTY <= LinkAdaptationCommonConfiguration.wlanBandwidthMax)
	{
		switch (tcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				tcrRate = pGroupDbHwEntry->wp.user0Tcr280MhzDataPsduRate;
				break;	
			case USP_USP_1:
				tcrRate = pGroupDbHwEntry->wp.user1Tcr280MhzDataPsduRate;
				break;
			case USP_USP_2:
				tcrRate = pGroupDbHwEntry->wp.user2Tcr280MhzDataPsduRate;
				break;
			case USP_USP_3:
				tcrRate = pGroupDbHwEntry->wp.user3Tcr280MhzDataPsduRate;
				break;	
		};
		rateIndex = convertTcr2RateIndex(PHY_MODE_11AC, tcrRate);
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex = rateIndex;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrPower = pGroupDbHwEntry->wp.tcr0Data80MhzRfPower; 
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode = htVhtCpTcrValToCpMode[pGroupDbHwEntry->wp.tcr180MhzDataScp]; 
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrBfMode = pGroupDbHwEntry->common.tcr2DataTxbf;	
	}
	
	/* 160 Mhz */	
	if (BANDWIDTH_ONE_HUNDRED_SIXTY <= LinkAdaptationCommonConfiguration.wlanBandwidthMax)
	{
		switch (tcrModificationParams->controlParams.uspIndex)
		{
			case USP_USP_0:
				tcrRate = pGroupDbHwEntry->wp.user0Tcr2160MhzDataPsduRate;
				break;	
			case USP_USP_1:
				tcrRate = pGroupDbHwEntry->wp.user1Tcr2160MhzDataPsduRate;
				break;
			case USP_USP_2:
				tcrRate = pGroupDbHwEntry->wp.user2Tcr2160MhzDataPsduRate;
				break;
			case USP_USP_3:
				tcrRate = pGroupDbHwEntry->wp.user3Tcr2160MhzDataPsduRate;
				break;	
		};
		rateIndex = convertTcr2RateIndex(PHY_MODE_11AC, tcrRate);
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].rateindex = rateIndex;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrPower = pGroupDbHwEntry->wp.tcr0Data160MhzRfPower; 
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode = htVhtCpTcrValToCpMode[pGroupDbHwEntry->wp.tcr1160MhzDataScp]; 
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrBfMode = pGroupDbHwEntry->common.tcr2DataTxbf;	
	}
}

/**********************************************************************************

getSuTcrParamsFromHwDb



Description:
------------
Functino fills tcrModificationParams accordig to HW DB

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void getSuTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams)
{
	StaDb_t* pStaDbHwEntry =  &StaDbHwEntries[tcrModificationParams->controlParams.staIndex];
	StaDbWp_t*	pWorkingPointParams;
	uint8 rateIndex;
	PhyMode_e phyMode;

	pWorkingPointParams = &(pStaDbHwEntry->wp);

	if (tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
	{	
		tcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection =  pWorkingPointParams->tcr0DataAntennaSelection;
		tcrModificationParams->tcrParams.tcrGeneralVals.ldpcMode = pStaDbHwEntry->common.tcr2DataLdpc;
	}
	else if (tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_MANAGEMENT)
	{
		tcrModificationParams->tcrParams.tcrGeneralVals.tcrAntSelection =  pStaDbHwEntry->common.tcr080211AntennaSelection;
		tcrModificationParams->tcrParams.tcrGeneralVals.ldpcMode = pStaDbHwEntry->common.tcr280211Ldpc;
	}	
	else
	{
		DEBUG_ASSERT(0);
	}

	/* 20MHz TCR*/
	if (tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
	{
		phyMode = (PhyMode_e)pWorkingPointParams->tcr0Data20MhzPhyMode;
		rateIndex = convertTcr2RateIndex(phyMode, pWorkingPointParams->tcr220MhzDataPsduRate);

		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrPower = pWorkingPointParams->tcr0Data20MhzRfPower; 
		
		if ((phyMode == PHY_MODE_11AX_SU) || (phyMode == PHY_MODE_11AX_SU_EXT))
		{
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode = convertHeCpLtfTcrValToCpMode(pWorkingPointParams->tcr120MhzDataHeCp, pWorkingPointParams->tcr120MhzDataHeLtf);
		}
		else if ((phyMode == PHY_MODE_11N) || (phyMode == PHY_MODE_11AC))
		{
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode = htVhtCpTcrValToCpMode[pWorkingPointParams->tcr120MhzDataScp];
		}
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].rateindex = rateIndex;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrBfMode = pWorkingPointParams->tcr220MhzDataTxbf;
		tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pWorkingPointParams->tcr120MhzDataStbc;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].BfSmothing = pWorkingPointParams->tcr120MhzDataSmoothing;
	}
	else
	{
		phyMode = (PhyMode_e)pStaDbHwEntry->common.tcr08021120MhzPhyMode;
		rateIndex = convertTcr2RateIndex(phyMode, pStaDbHwEntry->common.tcr220Mhz80211PsduRate);

		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrPower = pStaDbHwEntry->common.tcr08021120MhzRfPower;
		if ((phyMode == PHY_MODE_11AX_SU) || (phyMode == PHY_MODE_11AX_SU_EXT))
		{
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode = convertHeCpLtfTcrValToCpMode(pStaDbHwEntry->common.tcr120Mhz80211HeCp, pStaDbHwEntry->common.tcr120Mhz80211HeLtf);
		}
		else if ((phyMode == PHY_MODE_11N) || (phyMode == PHY_MODE_11AC))
		{
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].cpMode = htVhtCpTcrValToCpMode[pStaDbHwEntry->common.tcr120Mhz80211Scp];
		}
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].rateindex = rateIndex;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].tcrBfMode = pStaDbHwEntry->common.tcr220Mhz80211Txbf;
		tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pStaDbHwEntry->common.tcr120Mhz80211Stbc;
		tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_TWENTY].BfSmothing = pStaDbHwEntry->common.tcr120Mhz80211Smoothing;
	}

	/* 40MHz TCR*/
	if (BANDWIDTH_FOURTY <= LinkAdaptationCommonConfiguration.wlanBandwidthMax)
	{
		if (tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
			phyMode = (PhyMode_e)pWorkingPointParams->tcr0Data40MhzPhyMode;
			rateIndex = convertTcr2RateIndex(phyMode, pWorkingPointParams->tcr240MhzDataPsduRate);
	
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrPower = pWorkingPointParams->tcr0Data40MhzRfPower; 
			if (phyMode == PHY_MODE_11AX_SU)
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode = convertHeCpLtfTcrValToCpMode(pWorkingPointParams->tcr140MhzDataHeCp, pWorkingPointParams->tcr140MhzDataHeLtf);
			}
			else
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode = htVhtCpTcrValToCpMode[pWorkingPointParams->tcr140MhzDataScp];
			}
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].rateindex = rateIndex;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrBfMode = pWorkingPointParams->tcr240MhzDataTxbf;
			tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pWorkingPointParams->tcr140MhzDataStbc;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].BfSmothing = pWorkingPointParams->tcr140MhzDataSmoothing;
		}
		else
		{
			phyMode = (PhyMode_e)pStaDbHwEntry->common.tcr08021140MhzPhyMode;
			rateIndex = convertTcr2RateIndex(phyMode, pStaDbHwEntry->common.tcr240Mhz80211PsduRate);
	
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrPower = pStaDbHwEntry->common.tcr08021140MhzRfPower;
			if (phyMode == PHY_MODE_11AX_SU)
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode = convertHeCpLtfTcrValToCpMode(pStaDbHwEntry->common.tcr140Mhz80211HeCp, pStaDbHwEntry->common.tcr140Mhz80211HeLtf);
			}
			else
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].cpMode = htVhtCpTcrValToCpMode[pStaDbHwEntry->common.tcr140Mhz80211Scp];
			}
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].rateindex = rateIndex;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].tcrBfMode = pStaDbHwEntry->common.tcr240Mhz80211Txbf;
			tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pStaDbHwEntry->common.tcr140Mhz80211Stbc;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_FOURTY].BfSmothing = pStaDbHwEntry->common.tcr140Mhz80211Smoothing;
		}
	}
	
	/* 80MHz TCR*/
	if (BANDWIDTH_EIGHTY <= LinkAdaptationCommonConfiguration.wlanBandwidthMax)
	{
		if (tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
			phyMode = (PhyMode_e)pWorkingPointParams->tcr0Data80MhzPhyMode;
			rateIndex = convertTcr2RateIndex(phyMode, pWorkingPointParams->tcr280MhzDataPsduRate);
	
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrPower = pWorkingPointParams->tcr0Data80MhzRfPower; 
			if (phyMode == PHY_MODE_11AX_SU)
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode = convertHeCpLtfTcrValToCpMode(pWorkingPointParams->tcr180MhzDataHeCp, pWorkingPointParams->tcr180MhzDataHeLtf);
			}
			else
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode = htVhtCpTcrValToCpMode[pWorkingPointParams->tcr180MhzDataScp];
			}
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex = rateIndex;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrBfMode = pWorkingPointParams->tcr280MhzDataTxbf;
			tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pWorkingPointParams->tcr180MhzDataStbc;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].BfSmothing = pWorkingPointParams->tcr180MhzDataSmoothing;
		}
		else
		{
			phyMode = (PhyMode_e)pStaDbHwEntry->common.tcr08021180MhzPhyMode;
			rateIndex = convertTcr2RateIndex(phyMode, pStaDbHwEntry->common.tcr280Mhz80211PsduRate);
	
			if (phyMode == PHY_MODE_11AX_SU)
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode = convertHeCpLtfTcrValToCpMode(pStaDbHwEntry->common.tcr180Mhz80211HeCp, pStaDbHwEntry->common.tcr180Mhz80211HeLtf);
			}
			else
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].cpMode = htVhtCpTcrValToCpMode[pStaDbHwEntry->common.tcr180Mhz80211Scp];
			}
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrPower = pStaDbHwEntry->common.tcr08021180MhzRfPower;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex = rateIndex;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].tcrBfMode = pStaDbHwEntry->common.tcr280Mhz80211Txbf;
			tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pStaDbHwEntry->common.tcr180Mhz80211Stbc;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].BfSmothing = pStaDbHwEntry->common.tcr180Mhz80211Smoothing;
		}
	}
	/* 160MHz TCR*/
	if (BANDWIDTH_ONE_HUNDRED_SIXTY == LinkAdaptationCommonConfiguration.wlanBandwidthMax)
	{
		if (tcrModificationParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
		{
			phyMode = (PhyMode_e)pWorkingPointParams->tcr0Data160MhzPhyMode;
			rateIndex = convertTcr2RateIndex(phyMode, pWorkingPointParams->tcr2160MhzDataPsduRate);
	
			if (phyMode == PHY_MODE_11AX_SU)
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode = convertHeCpLtfTcrValToCpMode(pWorkingPointParams->tcr1160MhzDataHeCp, pWorkingPointParams->tcr1160MhzDataHeLtf);
			}
			else
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode = htVhtCpTcrValToCpMode[pWorkingPointParams->tcr1160MhzDataScp];
			}
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrPower = pWorkingPointParams->tcr0Data160MhzRfPower; 
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].rateindex = rateIndex;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrBfMode = pWorkingPointParams->tcr2160MhzDataTxbf;
			tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pWorkingPointParams->tcr1160MhzDataStbc;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].BfSmothing = pWorkingPointParams->tcr1160MhzDataSmoothing;
		}
		else
		{
			phyMode = (PhyMode_e)pStaDbHwEntry->common.tcr080211160MhzPhyMode;
			rateIndex = convertTcr2RateIndex(phyMode, pStaDbHwEntry->common.tcr2160Mhz80211PsduRate);
	
			if (phyMode == PHY_MODE_11AX_SU)
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode = convertHeCpLtfTcrValToCpMode(pStaDbHwEntry->common.tcr1160Mhz80211HeCp, pStaDbHwEntry->common.tcr1160Mhz80211HeLtf);
			}
			else
			{
				tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].cpMode = htVhtCpTcrValToCpMode[pStaDbHwEntry->common.tcr1160Mhz80211Scp];
			}
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrPower = pStaDbHwEntry->common.tcr080211160MhzRfPower;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].rateindex = rateIndex;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].tcrBfMode = pStaDbHwEntry->common.tcr2160Mhz80211Txbf;
			tcrModificationParams->tcrParams.tcrGeneralVals.tcrStbcMode = pStaDbHwEntry->common.tcr1160Mhz80211Stbc;
			tcrModificationParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_ONE_HUNDRED_SIXTY].BfSmothing = pStaDbHwEntry->common.tcr1160Mhz80211Smoothing;
		}
	}
}

/**********************************************************************************

getSuTcrParamsFromHwDb



Description:
------------
Functino fill tcrModificationParams accordig to HW DB

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void getTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams)
{
	if(tcrModificationParams->controlParams.uspIndex == INVALID_MU_USP_INDEX)
	{
		getSuTcrParamsFromHwDb(tcrModificationParams);
	}
	else /* MU*/
	{
		if (tcrModificationParams->controlParams.isHeGroup)
		{
			getHeMuTcrParamsFromHwDb(tcrModificationParams);
		}
		else
		{
			getVhtMuTcrParamsFromHwDb(tcrModificationParams);
		}
	}
}
/**********************************************************************************

GetDataBwLimit

Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

Bandwidth_e GetDataBwLimit(StaId staOrGroupIndex,uint8 uspIndex, bool isHeGroup)
{
	DlDataPhase_t* phasePtr = NULL; 
	TcrCommon_t* phaseCommonDataTcrPtr = NULL;

	if(uspIndex == INVALID_MU_USP_INDEX)
	{
		return (Bandwidth_e)StaDbHwEntries[staOrGroupIndex].common.dataBwLimit;
	}
	else if (isHeGroup == TRUE)
	{
		phasePtr = getWpPhaseAddress(staOrGroupIndex);
		phaseCommonDataTcrPtr = (TcrCommon_t*)&(phasePtr->commonTcra);
		return (Bandwidth_e)phaseCommonDataTcrPtr->cbw;
	}
	else /* MU*/
	{
		
		return (Bandwidth_e)(GroupDbHwEntries[staOrGroupIndex].common.dataBwLimit); 
	}
}

/**********************************************************************************

GetWpDataBwLimit

Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

Bandwidth_e GetWpDataBwLimit(StaId staOrGroupIndex,uint8 uspIndex, bool isHeGroup)
{
	DlDataPhase_t* phasePtr = NULL; 
	TcrCommon_t* phaseCommonDataTcrPtr = NULL;

	if(uspIndex == INVALID_MU_USP_INDEX)
	{
		return (Bandwidth_e)StaDbHwEntries[staOrGroupIndex].wp.dataBwLimit;
	}
	else if (isHeGroup == TRUE)
	{
		phasePtr = getWpPhaseAddress(staOrGroupIndex);
		phaseCommonDataTcrPtr = (TcrCommon_t*)&(phasePtr->commonTcra);
		return (Bandwidth_e)phaseCommonDataTcrPtr->cbw;
	}
	else /* MU*/
	{
		
		return (Bandwidth_e)(GroupDbHwEntries[staOrGroupIndex].wp.dataBwLimit); 
	}
}

/**********************************************************************************

LinkAdaptationSetStaDataBwLimit


Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void SetStaDataBwLimitInHwDb(StaId staIndex,Bandwidth_e bandwidth, bool changeWorkingPoint)
{

	uint8 activatedAntennasCount;
	
	if (changeWorkingPoint == TRUE)
	{
#ifdef TRAINING_FIRST_PHASE_POOL 
		numberOfStationInBw[LinkAdaptationStaDatabase[staIndex].laStationUnique.vapId][bandwidth]++; 
		if(LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.linkAdaptationState != LA_STA_NOT_CONNECT_RA_BYPASS)
		{
			/* In case this function isn't being called from linkAdaptationAddStationReq context, 
			when station is being set with BW at the first time - we have no BW entry to substruct from*/
			numberOfStationInBw[LinkAdaptationStaDatabase[staIndex].laStationUnique.vapId][StaDbHwEntries[staIndex].common.dataBwLimit]--; 
		}
#endif //TRAINING_FIRST_PHASE_POOL
		StaDbHwEntries[staIndex].wp.dataBwLimit = bandwidth;
		LinkAdaptationStatistics.dataBwLimit[staIndex] = bandwidth;

	}
	StaDbHwEntries[staIndex].common.dataBwLimit = bandwidth;

	if (LinkAdaptationStaDatabase[staIndex].laStaUspCommon.staTransmissionParams.heSta == TRUE)
	{
		activatedAntennasCount = AntennaSelectionGetActivatedAntennasCount();
		
		// If HE station, update max NSS NDP according to new BW
		setHeMaxNssNdp(staIndex, bandwidth, activatedAntennasCount);
	}
}


/**********************************************************************************

LinkAdaptationSetGroupDataBwLimit



Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void setGroupDataBwLimit(uint8 groupIndex,Bandwidth_e bandwidth, bool changeWorkingPoint)
{
	if (changeWorkingPoint == TRUE)
	{
		GroupDbHwEntries[groupIndex].wp.dataBwLimit = bandwidth;
	}
	LinkAdaptationMuStatistics.dataBwLimit[groupIndex] = bandwidth;
	GroupDbHwEntries[groupIndex].common.dataBwLimit = bandwidth; 
}

/**********************************************************************************

SetStaManagementBwLimit



Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void SetStaManagementBwLimit(StaId staIndex,Bandwidth_e bandwidth)
{
	StaDbHwEntries[staIndex].common._80211BwLimit= bandwidth;
}

/**********************************************************************************

bwAdaptationSetStationBwLimit 


Description:
------------
At each station's BW limit set, we update the number of stations working on a BW
This way we can cal

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void bwAdaptationSetStationBwLimit(StaId staId, Bandwidth_e bwLimit, bool changeWorkingPoint)
{
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staId]);
	LinkAdaptationStaDatabase_t* laDatabase = &LinkAdaptationStaDatabase[staId];
	Bandwidth_e changedDataBw;
	Bandwidth_e currentBwLimit = GetDataBwLimit(staId,INVALID_MU_USP_INDEX, FALSE);

	if((pStaDbHwEntry->common.tcr0Data20MhzPhyMode == PHY_MODE_11B) || (pStaDbHwEntry->common.tcr0Data20MhzPhyMode == PHY_MODE_11B))
	{
		changedDataBw = BANDWIDTH_TWENTY; 
	}
	else
	{
		changedDataBw = MIN(laDatabase->laStaGroupCommon.bwAdaptationDb.maxSupportedBw, bwLimit); 
	}

	if (changedDataBw != currentBwLimit)
	{
		LinkAdaptationSetStaDataBwLimit(staId,changedDataBw,changeWorkingPoint);
	}
}

/**********************************************************************************

bwAdaptationSetBwLimit 


Description:
------------
At each station's BW limit set, we update the number of stations working on a BW
This way we can cal

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void bwAdaptationSetBwLimit(LinkAdaptationDatabaseDistributionPack_t *pDistParam, Bandwidth_e bwLimit, bool changeWorkingPoint)
{
	Bandwidth_e changedDataBw;
	Bandwidth_e currentBwLimit = GetDataBwLimit(pDistParam->stationOrGroupIndex,pDistParam->uspIndex, FALSE);

	if(pDistParam->uspIndex == INVALID_MU_USP_INDEX)
	{
		StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[pDistParam->stationOrGroupIndex]);
		if(pStaDbHwEntry->common.tcr0Data20MhzPhyMode == PHY_MODE_11B) /*|| (pStaDbHwEntry->common._20MhzWpPhyModeData == PHY_MODE_11B))*/
		{
			changedDataBw = BANDWIDTH_TWENTY; 
		}
		else
		{
			changedDataBw = MIN(pDistParam->laStaGroupCommon->bwAdaptationDb.maxSupportedBw, bwLimit); 
		}
		if (changedDataBw != currentBwLimit)
		{
			LinkAdaptationSetStaDataBwLimit(pDistParam->stationOrGroupIndex,changedDataBw,changeWorkingPoint);
		}
	}
	else /*MU*/
	{
		changedDataBw = MIN(pDistParam->laStaGroupCommon->bwAdaptationDb.maxSupportedBw, bwLimit); 
		if (changedDataBw != currentBwLimit)
		{
		
			setGroupDataBwLimit(pDistParam->stationOrGroupIndex,changedDataBw,changeWorkingPoint);
		}
	}

	ILOG0_DD("bwAdaptationSetBwLimit, currentBwLimit %d changedDataBw = %d", currentBwLimit, changedDataBw);
	
}


#ifdef TRAINING_FIRST_PHASE_POOL
/**********************************************************************************

getMaximalWorkingBW 


Description:
------------
return the current maximal BW which at least one station is working with

Input: 
-----

Returns:
--------
	Bandwidth_e -  Bw index
	
**********************************************************************************/
void getMaximalWorkingBW(Bandwidth_e* maximalWorkingBw)
{
	uint8 vapId;
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();

	for (vapId = firstVapInBand; vapId < (firstVapInBand + numOfVapsInBand); vapId++)
	{
		if(numberOfStationInBw[vapId][BANDWIDTH_EIGHTY])
		{
			maximalWorkingBw[vapId] = BANDWIDTH_EIGHTY;
		}
		else if(numberOfStationInBw[vapId][BANDWIDTH_FOURTY])
		{
			maximalWorkingBw[vapId] = BANDWIDTH_FOURTY;
		}
		else
		{
			maximalWorkingBw[vapId] = BANDWIDTH_TWENTY;
		}
	}
}


/**********************************************************************************

decreaseMaxBwCount 


Description:
------------
reduce the current maximal BW which a station is not working with anymore

Input: 
-----

Returns:
--------
	
	
**********************************************************************************/
void decreaseMaxBwCount(StaId staIndex, Bandwidth_e bw)
{
	uint8 vapId = LinkAdaptationStaDatabase[staIndex].laStationUnique.vapId; 
	numberOfStationInBw[vapId][bw]--;
	return; 
}

#endif //TRAINING_FIRST_PHASE_POOL

/**********************************************************************************

explicitBfHandler 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void prepareExplicitMode(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staId]);
    
#ifndef ENET_INC_ARCH_WAVE600B
	pStaDbHwEntry->common.bfRxMode = BF_RX_MODE_EXPLICIT_BEAMFORMER_AND_BEAMFORMEE;
#else
    pStaDbHwEntry->common.bfImplicitSupport = NOT_SUPPORTED;
#endif	
	/*Configure Tx selector to send BF report*/
	if ((LinkAdaptationStaDatabase[staId].laStaUspCommon.probingPointValidationCounter == 0)||
		(LinkAdaptationStaDatabase[staId].laStaUspCommon.probingPointValidationCounter == 4)||
		(LinkAdaptationStaDatabase[staId].laStaUspCommon.probingPointValidationCounter == 8))
	{
		sendBfRequest(staId);	
	}	
}
	

/**********************************************************************************

sendBfRequest 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void sendBfRequest(StaId staId)
{
	TxSelectorActionParams_t txSelectorActionparams;

	/*Configure TX selectior with BF sequence type*/

	txSelectorActionparams.queueType = HW_TX_Q_TYPE_STA_TID;
	txSelectorActionparams.stationOrVapNum = staId;
#ifdef SEND_BF_REQUEST_WITH_DATA
	txSelectorActionparams.action = TX_SELECTOR_ACTION_SEND_BF_WITH_DATA;
#else
	txSelectorActionparams.action = TX_SELECTOR_ACTION_SEND_BF_WITHOUT_DATA;
#endif //SEND_BF_REQUEST_WITH_DATA

	TxSelector_SetBfSequenceType(&txSelectorActionparams);

	/*Configure tx selector to send BF request*/
	txSelectorActionparams.action = TX_SELECTOR_ACTION_SEND_BF_REQUEST;

	TxSelector_SetBfSequenceRequest(&txSelectorActionparams);

}
/**********************************************************************************

stopSendBfRequest 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void stopSendBfRequest(StaId staId)
{
	TxSelectorActionParams_t txSelectorActionparams;

	/*Configure TX selectior with BF sequence type*/
	txSelectorActionparams.queueType = HW_TX_Q_TYPE_STA_TID;
	txSelectorActionparams.stationOrVapNum = staId;
#ifdef SEND_BF_REQUEST_WITH_DATA
	txSelectorActionparams.action = TX_SELECTOR_ACTION_SEND_BF_WITH_DATA;
#else
	txSelectorActionparams.action = TX_SELECTOR_ACTION_SEND_BF_WITHOUT_DATA;
#endif //SEND_BF_REQUEST_WITH_DATA
	
	TxSelector_SetBfSequenceType(&txSelectorActionparams);

	/*Configure tx selector to stop sending BF request*/
	txSelectorActionparams.action = TX_SELECTOR_ACTION_STOP_BF_REQUEST;

	TxSelector_SetBfSequenceRequest(&txSelectorActionparams);

}

/**********************************************************************************

prepareImplicitMode 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void prepareImplicitMode(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staId]);

#ifndef ENET_INC_ARCH_WAVE600B
	pStaDbHwEntry->common.bfRxMode = BF_RX_MODE_EXPLICIT_BEAMFORMEE;
#else
    pStaDbHwEntry->common.bfImplicitSupport = SUPPORTED;
#endif
}
/**********************************************************************************

prepareStbc1x2Mode 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void prepareStbc1x2Mode(StaId staId)
{
	stopSendBfRequest(staId);
}
/**********************************************************************************

prepareStbc2x4Mode 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void prepareStbc2x4Mode(StaId staId)
{
	stopSendBfRequest(staId);
}
/**********************************************************************************

prepareNoBfMode 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void prepareNoBfMode(StaId staId)
{
	stopSendBfRequest(staId);
}

/**********************************************************************************

explicitBfHandler 


Description:
------------
Check BF DB valid if EXPLICIT or IMPLICIT state

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

bool isBeamFormingReadyForTransmit(StaId staIndex, BeamformingMode_e bfState)
{
	bool retVal = FALSE;
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staIndex]);
	if ((bfState==BF_STATE_EXPLICIT)||(bfState==BF_STATE_IMPLICIT))
	{
		retVal = (pStaDbHwEntry->common.bferStatus == TRUE);
	}
	else
	{
		retVal = TRUE;
	}
	return retVal;
}
/**********************************************************************************

initExplicitBfStaCapabilities 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void initExplicitBfStaCapabilities(StaId staIndex, HtCapabilitiesInfo_t htCapbilities, sVHT_CAPABILITIES_INFO vhtCapabilities, HE_MAC_PHY_CAPABILITIES_INFO* pHeMacPhyCapabilities,sTransmitBfCapInfo transmitBfCapabilities)
{
	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	StaDb_t* pStaDbHwEntry =  &StaDbHwEntries[staIndex];
	uint8 highestNss;
	//LaFixedBeamformingParams_t* pLocalBfDebugParams = getBfDebugParams();
	bool isVhtSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.vhtSta;
	bool isHe = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.heSta;
	uint8 activatedAntennasCount = AntennaSelectionGetActivatedAntennasCount();
	UNUSED_PARAM(htCapbilities);
	
	getHighestNssInMask(staIndex, &highestNss, GetDataBwLimit(staIndex, INVALID_MU_USP_INDEX, FALSE)); // phy mode is don't care, as only NSS is used here
	/* Beam forming parameters */
#ifndef ENET_INC_ARCH_WAVE600B
	pStaDbHwEntry->common.bfRxHwFwMode = BF_DB_UPDATE_MODE_HW;
    pStaDbHwEntry->common.bfRxMode = BF_RX_MODE_EXPLICIT_BEAMFORMER_AND_BEAMFORMEE;
#else
    pStaDbHwEntry->common.bfImplicitSupport = NOT_SUPPORTED;
    pStaDbHwEntry->common.bfBfmeeSupport = SUPPORTED;
    pStaDbHwEntry->common.reserved93 = 0;
    pStaDbHwEntry->common.bfImplicitFiller = 0;
    pStaDbHwEntry->common.htcFieldCapable = 0;
	pStaDbHwEntry->common.implicitBfRulesTableIndex = 0;
#endif
	pStaDbHwEntry->common.bfMinimalGrouping = MINIMAL_GROUPING_1_2_4; 
	pStaDbHwEntry->common.maxNssTx = highestNss;  // maximum number of Spatial streams supported by Station.		  			     [9:8] 
	pStaDbHwEntry->common.bfMfbUpdating = HW_MCS_FEEDBACK_UPDATING_ENABLED; 
	pStaDbHwEntry->common.bferStatus = BF_DB_NOT_VALID; 
	pStaDbHwEntry->common.mfbValid = MCS_FEEDBACK_NOT_VALID; 

	if (isHe)
	{
		pLinkAdaptationDb->laStationUnique.beamFormingDb.maxNssNdpLessOrEqual80Mhz = pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_BEAMFORMEE_STS_FOR_LESS_THAN_OR_EQUAL_80MHz;
		pLinkAdaptationDb->laStationUnique.beamFormingDb.maxNssNdpGreaterThan80Mhz = pHeMacPhyCapabilities->hePhyCapInfo.HE_PHY_BEAMFORMEE_STS_FOR_GREATER_THAN_80MHz;

		setHeMaxNssNdp(staIndex, GetDataBwLimit(staIndex, INVALID_MU_USP_INDEX, FALSE), activatedAntennasCount);
	}
	else if (isVhtSta)
	{
		pStaDbHwEntry->common.maxNssNdpStaCapability = vhtCapabilities.beamformee_sts_capability;
		pStaDbHwEntry->common.maxNssNdp = MIN(pStaDbHwEntry->common.maxNssNdpStaCapability, CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(activatedAntennasCount));
#ifndef ENET_INC_ARCH_WAVE600B		
		pStaDbHwEntry->common.nonStandardImplicitBfRulesTableIndex = MAC_BF_BF_NONSTANDARD_IMP_RULES_INDX_VHT_STA;
#endif
	}
	else
	{
		pStaDbHwEntry->common.maxNssNdpStaCapability = transmitBfCapabilities.compressed_steering_number_of_beamformer_antennas_supported;
		pStaDbHwEntry->common.maxNssNdp = MIN(pStaDbHwEntry->common.maxNssNdpStaCapability, CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(activatedAntennasCount));
#ifndef ENET_INC_ARCH_WAVE600B		
		pStaDbHwEntry->common.nonStandardImplicitBfRulesTableIndex = MAC_BF_BF_NONSTANDARD_IMP_RULES_INDX_HT_STA;
#endif
	}
}

/**********************************************************************************

LinkAdaptationPhyDriver_Init 


Description:
------------

	
Input: 
-----
	None
	
**********************************************************************************/
void LinkAdaptationPhyDriver_Init(void)
{
	uint8 userId;
	memset(pDeliaAutoResponseTcrs, 0, sizeof(Tcr_t));

    //This is a patch till a static table of RF power will be configured in the shram
    pDeliaAutoResponseTcrs->common.rfPower = 32;

	// 2.4G COEX: set "force tx" bit as TRUE so that PHY shall transmit ACK\BA when we are at "TX DISABLE".
	pDeliaAutoResponseTcrs->common.forceTx = 1;

	for(userId = 0; userId < NUM_OF_USERS_MU_HE; userId++)
	{
		pDeliaAutoResponseTcrs->user[userId].txbf = TX_BF_FLAT;
	}
	
	//init RtsMode global parameter which is used each ADD_STATION to configure staDBentry
	LinkAdaptationRtsMode.dynamicBw = FALSE;
	LinkAdaptationRtsMode.staticBw = FALSE;
	
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
	memset32(originalSuMaxPsduLengthLimit, 0, HW_NUM_OF_STATIONS);
	memset32(originalMuMaxPsduLengthLimit, 0, HW_NUM_OF_STATIONS);

#endif

}

/**********************************************************************************

setTransmissionTimeParams 


Description:
------------

	
Input: 
-----
	None
	
**********************************************************************************/
void setTransmissionTimeParams()
{
	uint8 rateIndex;
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	
	if (AggRateLimitMode == TRUE)
	{
		// Client Isolation is enabled
		while (nextSid != DB_ASYNC_SID)
		{
			DEBUG_ASSERT(nextSid < HW_NUM_OF_STATIONS);
			
			rateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(&(LinkAdaptationStaDatabase[nextSid].laStaUspCommon));
			updateTransmissionTimeParams(nextSid, rateIndex);
			
			nextSid = StaDbSwEntries[nextSid].nextSid;
		}
	}
}

/**********************************************************************************

convertTcrVal2Power 


Description:
------------

Input: 
-----
Returns:
--------
	
**********************************************************************************/

uint8 convertTcrVal2Power(Bandwidth_e bandwidth, uint8 tcrVal)
{
	UNUSED_PARAM(bandwidth);	
	return tcrVal;
}
/**********************************************************************************

BeamformingSetDebugMaxNdp 


Description:
------------

	
Input: 
-----
	None
	
**********************************************************************************/

void BeamformingSetDebugMaxNdp(StaId staIndex)
{
	StaDb_t* pStaDbHwEntry =  &StaDbHwEntries[staIndex];
	LaFixedBeamformingParams_t* pLocalBfDebugParams = getBfDebugParams();

	pStaDbHwEntry->common.maxNssNdp = pLocalBfDebugParams->maxNssNdp;
}
/**********************************************************************************

GetStaWpRateIndexFromHwTcr 


Description:
------------

	
Input: 
-----
	None
	
**********************************************************************************/

uint8 GetStaWpRateIndexFromHwTcr(StaId staIndex, Bandwidth_e bandwidth)
{
	StaDb_t* pStaDbHwEntry =  &StaDbHwEntries[staIndex];
	StaDbWp_t*	pWorkingPointParams;
	uint8 rateIndex;
	PhyMode_e wpPhyMode = PHY_MODE_LAST;
	uint8 wpTcrRate = 0;
	pWorkingPointParams = &(pStaDbHwEntry->wp);

	switch (bandwidth)
	{
		case BANDWIDTH_TWENTY: 	
			wpTcrRate = pWorkingPointParams->tcr220MhzDataPsduRate;
			wpPhyMode = pWorkingPointParams->tcr0Data20MhzPhyMode;
			break; 
		case BANDWIDTH_FOURTY:
			wpTcrRate = pWorkingPointParams->tcr240MhzDataPsduRate;
			wpPhyMode = pWorkingPointParams->tcr0Data40MhzPhyMode;
			break; 
		case BANDWIDTH_EIGHTY: 
			wpTcrRate = pWorkingPointParams->tcr280MhzDataPsduRate;
			wpPhyMode = pWorkingPointParams->tcr0Data80MhzPhyMode;
			break; 	
		case BANDWIDTH_ONE_HUNDRED_SIXTY: 
			wpTcrRate = pWorkingPointParams->tcr2160MhzDataPsduRate;
			wpPhyMode = pWorkingPointParams->tcr0Data160MhzPhyMode;
			break; 	
		default:
			ASSERT(0);
			break;
	}
	rateIndex = convertTcr2RateIndex(wpPhyMode, wpTcrRate);

	return rateIndex;
}

/**********************************************************************************

isBfReportNotReceivedAfterRequest 


Description:
------------
if BF request has sent without response, return TRUE otherwize return FALSE

	
Input: 
-----
	None
	
**********************************************************************************/

bool isBfReportNotReceivedAfterRequest(StaId staIndex)
{
	bool retVal = FALSE;
	BandId_e myBand = ConfigurationManager_GetMyBand();
	UNUSED_PARAM(staIndex);	
	//if BF request has sent without response, return TRUE otherwize return FALSE
#ifdef SEND_BF_REQUEST_WITH_DATA
	/*BF request sent without PHY NDP or with data after it  => BF sequcnce failed*/
	if ((CurrentSequencerReport.executedBfReq) &&
		(((CurrentSequencerReport.executedPhyNdp) == 0) ||
		((CurrentSequencerReport.executedData) == 0)))
	{
		retVal = TRUE;
	}
	else if ((CurrentSequencerReport.executedBfReq) &&
			((CurrentSequencerReport.executedPhyNdp)) &&
			((CurrentSequencerReport.executedData)))
	{
		LinkAdaptationStatistics.bfReportReceivedCounter[myBand]++; //LINK_ADAPTATION_BF_REPORT_RECEIVED_COUNTER;
	}
#else // SEND_BF_REQUEST_WITHOUT_DATA
	/*BF request NDPA sent but without PHY NDP or there was  timout or other RX after it  => BF sequcnce failed*/
	if ((CurrentSequencerReport.executedBfReq) &&
		(((CurrentSequencerReport.executedPhyNdp)==0) ||
		(((CurrentSequencerReport.timeout & USER_POS_0_MASK) == USER_POS_0_MASK) && (CurrentSequencerReport.bfIteration == TX_SEQUENCER_MAX_NUM_OF_BF_ITER)) ||
		((CurrentSequencerReport.otherRx & USER_POS_0_MASK) == USER_POS_0_MASK) ||
		(((CurrentSequencerReport.timeout & USER_POS_0_MASK) == USER_POS_0_MASK) && (!(CurrentSequencerReport.cca20PFree)))))
	{
		retVal = TRUE;
	}
	else if	((CurrentSequencerReport.executedBfReq) &&
			((CurrentSequencerReport.executedPhyNdp)) &&
			(((CurrentSequencerReport.timeout & USER_POS_0_MASK) == 0) || (CurrentSequencerReport.bfIteration < TX_SEQUENCER_MAX_NUM_OF_BF_ITER)) &&
			((CurrentSequencerReport.otherRx & USER_POS_0_MASK) == 0))
	{
		LinkAdaptationStatistics.bfReportReceivedCounter[myBand]++; //LINK_ADAPTATION_BF_REPORT_RECEIVED_COUNTER;
	}
#endif

	return retVal;
}

void linkAdaptationUpdateCalStatistics(K_MSG *psMsg)
{
	LinkAdaptationCalibrationStatus_t* pCalibrationStatistics = (LinkAdaptationCalibrationStatus_t*)psMsg->abData;
	uint8 antMask, numOfAnts;	
	uint8 ant;
    // Get the enabled antennas
	HDK_GetMaxActiveAntennasNumAndMask(&numOfAnts, &antMask);
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (antMask & (1 << ant))
		{
			LinkAdaptationStatistics.rxDc[ant]= pCalibrationStatistics->rxDc[ant];
			LinkAdaptationStatistics.txLo[ant]= pCalibrationStatistics->txLo[ant];
			LinkAdaptationStatistics.rxIq[ant]= pCalibrationStatistics->rxIq[ant];
			LinkAdaptationStatistics.txIq[ant]= pCalibrationStatistics->txIq[ant];
#ifdef WRX600_BU_LOGS
			ILOG0_DDDD("Statistics rxdc:%d, txlo:%d, rxiq:%d, txiq:%d",LinkAdaptationStatistics.rxDc[ant],LinkAdaptationStatistics.txLo[ant],LinkAdaptationStatistics.rxIq[ant],LinkAdaptationStatistics.txIq[ant]);
#endif
		}
	}
}

/**********************************************************************************
updateLaStatistics


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
void updateLaStatistics(LaTcrModificationStruct_t* pTransmissionParams)
{
	Bandwidth_e bandwidth;
	uint8 tcrRate;
	uint8 tcrPhyMode;
	uint8 tempMcs;
	uint8 tempNss;
	const RateObj_t* ratesTable;
	uint8 maxNumOfRatesInTable;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter, pTransmissionParams->controlParams.staIndex, pTransmissionParams->controlParams.uspIndex, pTransmissionParams->controlParams.isHeGroup);
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);		

	ASSERT(pTransmissionParams->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	if(pTransmissionParams->controlParams.uspIndex != INVALID_MU_USP_INDEX) /*MU*/
	{
		for (bandwidth = pTransmissionParams->controlParams.firstBwToUpdate; bandwidth <= pTransmissionParams->controlParams.lastBwToUpdate; bandwidth++)
		{

			/*Get phymode and rate according to rate index*/
#ifdef ENET_INC_ARCH_WAVE600D2
			getTcrPhyModeAndRate(ratesTable, pTransmissionParams->controlParams.isVhtSta, pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex, &tcrPhyMode, &tcrRate, pTransmissionParams->controlParams.isHeExt, /*D/C in this case no usage for DCM parameter*/ FALSE);
#else
			getTcrPhyModeAndRate(ratesTable, pTransmissionParams->controlParams.isVhtSta, pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
#endif //ENET_INC_ARCH_WAVE600D2
			
			if (tcrPhyMode <= PHY_MODE_11AC)
			{
				LinkAdaptationMuStatistics.scpData[pTransmissionParams->controlParams.staIndex][bandwidth] = convertCpModeToHtVhtCpTcrVal(pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].cpMode); //LINK_ADAPTATION_SCP_DATA
			}
			else
			{
				LinkAdaptationMuStatistics.scpData[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].cpMode; //LINK_ADAPTATION_SCP_DATA
			}
			LinkAdaptationMuStatistics.powerData[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].tcrPower; //LINK_ADAPTATION_POWER_DATA
		}
		tempMcs = ratesTable[pTransmissionParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex].vhtHeMcs;	
		tempNss = ratesTable[pTransmissionParams->tcrParams.bwDependedTcrValsTable[BANDWIDTH_EIGHTY].rateindex].numberOfNss;
		
		LinkAdaptationMuStatistics.nssData[pTransmissionParams->controlParams.staIndex][pTransmissionParams->controlParams.uspIndex] = tempNss+1; //LINK_ADAPTATION_NSS_DATA (start from 1)
		LinkAdaptationMuStatistics.mcsData[pTransmissionParams->controlParams.staIndex][pTransmissionParams->controlParams.uspIndex] = tempMcs; //LINK_ADAPTATION_MCS_DATA
		LinkAdaptationMuStatistics.dataBwLimit[pTransmissionParams->controlParams.staIndex] = GroupDbHwEntries[pTransmissionParams->controlParams.staIndex].common.dataBwLimit;
	}
	else /*SU*/
	{

		DEBUG_ASSERT(pTransmissionParams->controlParams.staIndex < HW_NUM_OF_STATIONS);
		maxNumOfRatesInTable = getMaxNumOfRatesInTable(laDbDistributionParameter.laStaUspCommon);

		for (bandwidth = pTransmissionParams->controlParams.firstBwToUpdate; bandwidth <= pTransmissionParams->controlParams.lastBwToUpdate; bandwidth++)
		{
			DEBUG_ASSERT(pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex < maxNumOfRatesInTable);
			
			/*Get phymode and rate according to rate index*/
#ifdef ENET_INC_ARCH_WAVE600D2
			getTcrPhyModeAndRate(ratesTable, pTransmissionParams->controlParams.isVhtSta, pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex, &tcrPhyMode, &tcrRate, pTransmissionParams->controlParams.isHeExt, /*D/C in this case no usage for DCM parameter*/ FALSE);
#else
			getTcrPhyModeAndRate(ratesTable, pTransmissionParams->controlParams.isVhtSta, pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex, &tcrPhyMode, &tcrRate, FALSE, NO_DCM);
#endif //ENET_INC_ARCH_WAVE600D2

			if ((tcrPhyMode == PHY_MODE_11AC) || (tcrPhyMode == PHY_MODE_11AX_SU) || (tcrPhyMode == PHY_MODE_11AX_SU_EXT))
			{
				tempMcs = ratesTable[pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex].vhtHeMcs;		
			}
			else
			{
				tempMcs = ratesTable[pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex].htMcs;
			}
			tempNss = ratesTable[pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex].numberOfNss;

			if (pTransmissionParams->controlParams.packetType == LA_PACKET_TYPE_DATA)
			{
				/*Set Statistics Data TCRs for all BWs*/
				LinkAdaptationStatistics.DataPhyMode[pTransmissionParams->controlParams.staIndex][bandwidth] = tcrPhyMode; //LINK_ADAPTATION_DATA_PHY_MODE
				if (tcrPhyMode <= PHY_MODE_11B)
				{
					/* For 11AG\B use LONG CP (WLANRTSYS-8448) */
					LinkAdaptationStatistics.scpData[pTransmissionParams->controlParams.staIndex][bandwidth]  = SCP_LONG_CP;
				}
				else if (tcrPhyMode <= PHY_MODE_11AC)
				{
					LinkAdaptationStatistics.scpData[pTransmissionParams->controlParams.staIndex][bandwidth]  = convertCpModeToHtVhtCpTcrVal(pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].cpMode);
				}
				else
				{
					LinkAdaptationStatistics.scpData[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].cpMode; //LINK_ADAPTATION_SCP_DATA
				}
					
				LinkAdaptationStatistics.stbcModeData[pTransmissionParams->controlParams.staIndex] = pTransmissionParams->tcrParams.tcrGeneralVals.tcrStbcMode; //LINK_ADAPTATION_STBC_MODE_DATA 
				LinkAdaptationStatistics.bfModeData[pTransmissionParams->controlParams.staIndex] = pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].tcrBfMode; //LINK_ADAPTATION_BF_MODE_DATA
				LinkAdaptationStatistics.powerData[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].tcrPower; //LINK_ADAPTATION_POWER_DATA
				LinkAdaptationStatistics.antennaSelectionData[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.tcrGeneralVals.tcrAntSelection; //LINK_ADAPTATION_ANTENNA_SELECTION_DATA
				LinkAdaptationStatistics.nssData[pTransmissionParams->controlParams.staIndex][bandwidth] = tempNss+1; //LINK_ADAPTATION_NSS_DATA (start from 1)
				LinkAdaptationStatistics.mcsData[pTransmissionParams->controlParams.staIndex][bandwidth] = tempMcs; //LINK_ADAPTATION_MCS_DATA
				LinkAdaptationStatistics.dataBwLimit[pTransmissionParams->controlParams.staIndex] = StaDbHwEntries[pTransmissionParams->controlParams.staIndex].common.dataBwLimit; //LINK_ADAPTATION_DATA_BW_LIMIT
			}
			else if (pTransmissionParams->controlParams.packetType == LA_PACKET_TYPE_MANAGEMENT)
			{
				
				/*Set Statistics Management TCRs for all BWs*/
				LinkAdaptationStatistics.ManagementPhyMode[pTransmissionParams->controlParams.staIndex][bandwidth] = tcrPhyMode; //LINK_ADAPTATION_MANAGEMENT_PHY_MODE
				LinkAdaptationStatistics.powerManagement[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.bwDependedTcrValsTable[bandwidth].tcrPower; //LINK_ADAPTATION_POWER_MANAGEMENT
				LinkAdaptationStatistics.antennaSelectionManagement[pTransmissionParams->controlParams.staIndex][bandwidth] = pTransmissionParams->tcrParams.tcrGeneralVals.tcrAntSelection; //LINK_ADAPTATION_ANTENNA_SELECTION_MANAGEMENT
				LinkAdaptationStatistics.nssManagement[pTransmissionParams->controlParams.staIndex][bandwidth] = tempNss; //LINK_ADAPTATION_NSS_MANAGEMENT
				LinkAdaptationStatistics.mcsManagement[pTransmissionParams->controlParams.staIndex][bandwidth] = tempMcs; //LINK_ADAPTATION_MCS_MANAGEMENT
			}
			else
			{
				DEBUG_ASSERT(0);
			}
		}
	
	}
}

void UpdateStationMaxSupportedNss(StaId staIndex, SpatialStreamNum_e maxNss)
{
	StaDb_t* pStaDbHwEntry =  NULL;
	uint8 activatedAntennasCount = AntennaSelectionGetActivatedAntennasCount();

	DEBUG_ASSERT(staIndex < HW_NUM_OF_STATIONS);
	
	pStaDbHwEntry = &StaDbHwEntries[staIndex];
	
	/* Training Manager MAX NSS */
	pStaDbHwEntry->common.maxSupportedNss = maxNss;

	/* Update Max NSS supported by STA */
	pStaDbHwEntry->common.maxNssTx = maxNss;

	/* Update Max NSS NDP: */
	/* Transmit NDP cannot use more antennas than AP uses (phy gets stuck): */
	pStaDbHwEntry->common.maxNssNdp = MIN(pStaDbHwEntry->common.maxNssNdpStaCapability, CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(activatedAntennasCount));	
}

void UpdateVapMaxNssNdp(uint8 vapIdx, SpatialStreamNum_e maxNss)
{
	VapDb_t* pVapDbHwEntry =  NULL;
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();

	DEBUG_ASSERT((vapIdx >= firstVapInBand) && (vapIdx < firstVapInBand + numOfVapsInBand));
	
	pVapDbHwEntry = &VapDbHwEntries[vapIdx];
	pVapDbHwEntry->common.maxNssNdp = maxNss;
}

void linkAdaptationSetMsduInAmsdu(K_MSG* psMsg)
{
	UMI_MSDU_IN_AMSDU_CONFIG *pLaMsduInAmsduConfigMsg = (UMI_MSDU_IN_AMSDU_CONFIG*)pK_MSG_DATA(psMsg);
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	uint8 msduInAmsdu;
	
	SLOG0(0, 0, UMI_MSDU_IN_AMSDU_CONFIG, pLaMsduInAmsduConfigMsg);
	if (pLaMsduInAmsduConfigMsg->getSetOperation == API_GET_OPERATION)
	{
		pLaMsduInAmsduConfigMsg->htMsduInAmsdu = laHtMsduInAmsdu;
		pLaMsduInAmsduConfigMsg->vhtMsduInAmsdu = laVhtMsduInAmsdu;
		pLaMsduInAmsduConfigMsg->heMsduInAmsdu= laHeMsduInAmsdu;

	}
	else
	{
		laHtMsduInAmsdu = pLaMsduInAmsduConfigMsg->htMsduInAmsdu;
		laVhtMsduInAmsdu = pLaMsduInAmsduConfigMsg->vhtMsduInAmsdu;
		laHeMsduInAmsdu = pLaMsduInAmsduConfigMsg->heMsduInAmsdu;
	}
	
	while (nextSid != DB_ASYNC_SID)
	{
		ASSERT(nextSid < HW_NUM_OF_STATIONS); //KW_FIX_FW_G
		if (LinkAdaptationStaDatabase[nextSid].laStaUspCommon.staTransmissionParams.heSta == TRUE)
		{
			msduInAmsdu = 	laHeMsduInAmsdu;
		}
		else if (LinkAdaptationStaDatabase[nextSid].laStaUspCommon.staTransmissionParams.vhtSta == TRUE)
		{
			msduInAmsdu = 	laVhtMsduInAmsdu;
		}
		else
		{
			msduInAmsdu = 	laHtMsduInAmsdu;
		}
		StaDbHwEntries[nextSid].common.maxMsduAtAmsduCount = msduInAmsdu;
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
	
}

void linkAdaptationSetPpduTxMode(uint8 ppduTxMode)
{
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	uint8 tidIndex = FIRST_TID;
	StaDbTid_t* pStaDbTid = NULL;
	laPpduTxMode = ppduTxMode;
	while (nextSid != DB_ASYNC_SID)
	{
		for (tidIndex = FIRST_TID; tidIndex < NUM_OF_TIDS; tidIndex++)
		{
			pStaDbTid =  &(StaDbHwEntries[nextSid].tid[tidIndex]);
			pStaDbTid->ppduTxMode = ppduTxMode;
		}
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
}



void  linkAdaptationSetRtsMode(UMI_RTS_MODE_CONFIG *pRtsModeConfigMsg)
{
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	StaDb_t* pStaDbHwEntry = NULL;
	
	/*First set global flag - all new STAs will use this value*/
	LinkAdaptationRtsMode.dynamicBw = pRtsModeConfigMsg->dynamicBw;
	LinkAdaptationRtsMode.staticBw= pRtsModeConfigMsg->staticBw;
	
	/*Now loop thru connected STAs and change the value*/

	while (nextSid != DB_ASYNC_SID)
	{
		//ASSERT_IN_LOOP-FW : Unless StaDbSwEntries has wrong Sid, we can ignore this assert.. Cannot be moved out of loop
		//check if station supports VHT and update DB if so
		ASSERT(nextSid < HW_NUM_OF_STATIONS);//KW_FIX_FW_G
		if((LinkAdaptationStaDatabase[nextSid].laStaUspCommon.staTransmissionParams.vhtSta) || (LinkAdaptationStaDatabase[nextSid].laStaUspCommon.staTransmissionParams.heSta))
		{			
			pStaDbHwEntry =  &(StaDbHwEntries[nextSid]);
			pStaDbHwEntry->common.dynamicBwEnable = pRtsModeConfigMsg->dynamicBw;
			pStaDbHwEntry->common.staticBwEnable= pRtsModeConfigMsg->staticBw;	
		}
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
}

void setAntBoostInStaTcr(StaId staId, uint8* antBoostTable)
{
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staId]);

	/* Data */
	pStaDbHwEntry->common.tcr0DataAnt0Boost = antBoostTable[0];
	pStaDbHwEntry->common.tcr0DataAnt1Boost = antBoostTable[1];
	pStaDbHwEntry->common.tcr0DataAnt2Boost = antBoostTable[2];
	pStaDbHwEntry->common.tcr0DataAnt3Boost = antBoostTable[3];
	
	/* Management */
	pStaDbHwEntry->common.tcr080211Ant0Boost = antBoostTable[0];
	pStaDbHwEntry->common.tcr080211Ant1Boost = antBoostTable[1];
	pStaDbHwEntry->common.tcr080211Ant2Boost = antBoostTable[2];
	pStaDbHwEntry->common.tcr080211Ant3Boost = antBoostTable[3];
}

void setAntBoostInVapTcr(uint8 vapId, uint8* antBoostTable)
{
	TcrCommon_t* tcrTemplate;
	uint8 templateId;
	
	VapDb_t* pVapDb = &(VapDbHwEntries[vapId]);

	/* Data */
	pVapDb->common.tcr0DataAnt0Boost = antBoostTable[0];
	pVapDb->common.tcr0DataAnt1Boost = antBoostTable[1];
	pVapDb->common.tcr0DataAnt2Boost = antBoostTable[2];
	pVapDb->common.tcr0DataAnt3Boost = antBoostTable[3];

	/* Management */
	pVapDb->common.tcr080211Ant0Boost = antBoostTable[0];
	pVapDb->common.tcr080211Ant1Boost = antBoostTable[1];
	pVapDb->common.tcr080211Ant2Boost = antBoostTable[2];
	pVapDb->common.tcr080211Ant3Boost = antBoostTable[3];

	/* Templates */
	for (templateId = 0; templateId < NUM_OF_VAP_TCR_TEMPLATES; templateId++)
	{
		tcrTemplate = (TcrCommon_t*)(&(pVapDb->tcrTemplates[templateId].word0)); 
		tcrTemplate->ant0Boost = antBoostTable[0];
		tcrTemplate->ant1Boost = antBoostTable[1];
		tcrTemplate->ant2Boost = antBoostTable[2];
		tcrTemplate->ant3Boost = antBoostTable[3];
	}
}

void setAntBoostInTcrs(uint8* antBoostTable)
{
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	uint8 vapId;
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();

	/*loop over connected STAs */
	while (nextSid != DB_ASYNC_SID)
	{
		setAntBoostInStaTcr(nextSid, antBoostTable);
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}

	/*loop over Vaps */
	for (vapId = firstVapInBand; vapId < (firstVapInBand + numOfVapsInBand); vapId++)
	{
		setAntBoostInVapTcr(vapId, antBoostTable);
	}
}

void linkAdaptationSetFreqJumpMode(uint8 freqJumpMode)
{
#ifdef ENET_INC_ARCH_WAVE600
	UNUSED_PARAM(freqJumpMode);
	FATAL("Freq jump not_supported_yet");
#else
	DeliaAutoResponseCtsTcr2.bwChange = freqJumpMode; //for frequency jump usage 
#endif
}


void linkAdaptationSetMcastRateInVapDb(uint8 vapId,uint8 rate,PhyMode_e phyMode )
{
    LaFixedRateParams_t protectedDBParams;
    VapDb_t* pVapDbHwEntries = &VapDbHwEntries[vapId];
    uint32 protectedDBscratchPad[PROTECTED_VAP_DB_SIZE_WORDS];

    protectedDBParams.vapId = vapId;
    protectedDBParams.bandwidth = BANDWIDTH_TWENTY;
    protectedDBParams.changeType = LA_PACKET_TYPE_DATA;
    protectedDBParams.cpMode = CP_MODE_MED_CP_SHORT_LTF;
	protectedDBParams.rate = rate;
	protectedDBParams.phyMode = phyMode;
	ProtectedDbLock_ReadModifyWriteReq((uint32*)&(pVapDbHwEntries->common.word12), (PROTECTED_VAP_DB_SIZE_WORDS), protectedDBscratchPad, setVapFixedRateInTcrsModificationFunc,(void*)&protectedDBParams); 
}
void getStaRateMaskPerBw(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bw, RateMask_t inputMask, RateMask_t* outputMask, bool forceAddLegacyRatesToAllowedMask)
{
	RateMask_t allowdRateMask;
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
	bool isWds = getIsWdsWep(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex,isHeGroup);
	uint8 isHeSta = FALSE;

	ASSERT(bw< MAX_POSSIBLE_NUM_OF_BW);
	
	if (laDbDistributionParameter->stationOrGroupIndex != INVALID_STA_INDEX)
	{
		isHeSta = laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta;
	}
	
	if (isHeSta == TRUE)
	{
		/*Copy allowd bitmask to local var*/	
		memcpy32(allowdRateMask.raIndexMask64bit, AllowedHeRatesMask.raIndexMask64bit, RATES_BIT_MASK_SIZE);
	}
	else
	{
		if (isWds == FALSE)
		{
			/*Copy allowd bitmask to local var*/	
			memcpy32(allowdRateMask.raIndexMask64bit, AllowedRatesMask[bw].raIndexMask64bit, RATES_BIT_MASK_SIZE);
		}
		else
		{//The STA is WDS WEP (WEP WDS can not support more then 460 Mbitsec. Need to reduce the rate).  
			/*Copy allowd bitmask to local var*/	
			memcpy32(allowdRateMask.raIndexMask64bit, AllowedWdsWepRatesMask[bw].raIndexMask64bit, RATES_BIT_MASK_SIZE);
		}
		if (forceAddLegacyRatesToAllowedMask == TRUE)
		{
			allowdRateMask.raIndexMask64bit[0] |= LM_PHY_11G_RATE_MSK;
		}
	}
	
	/*Clear not allowed rates of the specific BW*/
	andOperator64Bit(inputMask.raIndexMask64bit,allowdRateMask.raIndexMask64bit,outputMask->raIndexMask64bit); 
}
void getHighestNssInMask(StaId staIndex, uint8* highestNss, Bandwidth_e bw)
{
	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	uint8 highestRateindex;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	bool isHeSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.heSta;

	highestRateindex = getHighestRateinMask(staIndex, pLinkAdaptationDb->laStaUspCommon.raIndexMask, bw, 0, getMaxSortedTableRateIndex(isHeSta));
	/* Take the max NSS and MCS (for this NSS) */
	*highestNss = ratesTable[highestRateindex].numberOfNss;
	DEBUG_ASSERT(*highestNss < VHT_HE_NUMBER_OF_NSS);

	ILOG0_D("[getHighestNssInMask], highestNss %d", *highestNss);
}

void getHighestMcsForNssInMask(StaId staIndex, uint8 nss, uint8* highestMcs, PhyMode_e phyMode,Bandwidth_e bw)
{
	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	uint8 highestRateindex;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	bool isHeSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.heSta;
	RateMask_t* enableNssMaskArr = enableHtVhtNssRatesInMask;
	RateMask_t rateMaskForChosenNss;
		
	if (isHeSta == TRUE)
	{
		enableNssMaskArr = enableHeNssRatesInMask;
	}
	/*AND between enableNssRatesInMask to RA Mask- result saved in tempRaMask*/
	andOperator64Bit(pLinkAdaptationDb->laStaUspCommon.raIndexMask.raIndexMask64bit, enableNssMaskArr[nss].raIndexMask64bit,rateMaskForChosenNss.raIndexMask64bit);
	
	highestRateindex = getHighestRateinMask(staIndex, rateMaskForChosenNss, bw, 0, getMaxSortedTableRateIndex(isHeSta));

	if (phyMode >= PHY_MODE_11AC)
	{
		*highestMcs = ratesTable[highestRateindex].vhtHeMcs;
		DEBUG_ASSERT(*highestMcs < HE_NUMBER_OF_MCS);
	}
	else
	{
		*highestMcs = ratesTable[highestRateindex].htMcs;
	}	

	ILOG0_DDD("[getHighestMcsForNssInMask], nss %d, highestMcs %d, for phyMode %d", nss, *highestMcs, phyMode);
}
void getHighestMcsInMask(StaId staIndex,uint8* highestMcs, PhyMode_e phyMode, Bandwidth_e bw)
{
	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	uint8 highestRateindex;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	bool isHeSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.heSta;
	
	highestRateindex = getHighestRateinMask(staIndex, pLinkAdaptationDb->laStaUspCommon.raIndexMask, bw, 0, getMaxSortedTableRateIndex(isHeSta));

	if (phyMode >= PHY_MODE_11AC)
	{
		*highestMcs = ratesTable[highestRateindex].vhtHeMcs;
		DEBUG_ASSERT(*highestMcs < HE_NUMBER_OF_MCS);
	}
	else
	{
		*highestMcs = ratesTable[highestRateindex].htMcs;
	}	

	ILOG0_DD("[getHighestMcsInMask], highestMcs %d, for phyMode %d", *highestMcs, phyMode);
}

Scp_e convertCpModeToHtVhtCpTcrVal(CyclicPrefixMode_e cpMode)
{	
	/* WLANRTSYS-9260: we got here while connected to a VHT station with fixed rate of CP = CP_MODE_SHORT_CP_MED_LTF */
	if (cpMode >= CP_NUM_OF_MODES_HT_VHT)
	{
		/* Convert from CP_MODE = 2(which is valid only in HE mode) to CP_MODE =1 (which is valid in VHT mode), 
		   so we will keep with the same CP mode (0.8) but with short LTF which is valid in VHT mode */
		cpMode = CP_MODE_MED_CP_SHORT_LTF;
	}
	return cpModeToHtVhtCpTcrVal[cpMode];
}


CyclicPrefixMode_e convertHeCpLtfTcrValToCpMode(HeCp_e heCp, HeLtf_e heLtf)
{
	CyclicPrefixMode_e cpMode = heCpLtfTcrValToCpMode[heCp][heLtf];
	ASSERT(cpMode != CP_MODE_INVALID);
	
	return cpMode;
}

static void setHeMaxNssNdp(StaId staId, Bandwidth_e bw, uint8 activatedAntennasCount)
{
	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staId];
	StaDb_t* pStaDbHwEntry =  &StaDbHwEntries[staId];

	if (bw <= BANDWIDTH_EIGHTY)
	{
		pStaDbHwEntry->common.maxNssNdpStaCapability = pLinkAdaptationDb->laStationUnique.beamFormingDb.maxNssNdpLessOrEqual80Mhz;
	}
	else
	{
		pStaDbHwEntry->common.maxNssNdpStaCapability = pLinkAdaptationDb->laStationUnique.beamFormingDb.maxNssNdpGreaterThan80Mhz;
	}
	pStaDbHwEntry->common.maxNssNdp = MIN(pStaDbHwEntry->common.maxNssNdpStaCapability, CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(activatedAntennasCount)); 
}

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
void overwriteMaxPsduLengthLimitInStaDb(LaStaProtectedZone_t *pLaProtectedStruct, LaTcrModificationStruct_t *pTcrModificationParams, uint8 tcrPhyMode, uint8 tcrRate)
{
	if((pTcrModificationParams->tcrParams.tcrGeneralVals.ldpcMode == 1) && (tcrPhyMode == PHY_MODE_11AX_SU))
	{
		ILOG0_V("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: overwriteMaxPsduLengthLimitInStaDb");
		uint32 newMaxPsduLength = 0;
		uint32 tmpPsduLength = 0;
		bool limitPsduLength = isPsduLengthLimitationNeeded(tcrRate, &newMaxPsduLength);
		// if new tcrRate to be written requires limitation in maxPsduLengthLimit
		if(limitPsduLength == TRUE) 
		{
			ILOG0_V("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: overwriteMaxPsduLengthLimitInStaDb - new rate requires limit");
			// if current TcrRate is good - doesn't require limitation in maxPsduLengthLimit
			if(isPsduLengthLimitationNeeded(pLaProtectedStruct->common.tcr2160MhzDataPsduRate, &tmpPsduLength) == FALSE)
			{
				// changing tcrRate only for probing
				if(pTcrModificationParams->controlParams.changeProbingPoint == TRUE)
				{
					// in case of problematic tcrRate in probing only - overwrite common.maxPsduLengthLimit for this station with minimal value between the two
					pLaProtectedStruct->common.maxPsduLengthLimit = MIN(originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], newMaxPsduLength);
					// no need to store the original maxPsduLengthLimit because after the Tx of probing sender will restore maxPsduLengthLimit in common section with original value written in WP section.
					ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: probing case. don't store origLimit. overwriteCommon.  common Limit %d, new Limit %d ",pLaProtectedStruct->common.maxPsduLengthLimit, newMaxPsduLength);
				}
				// changing tcrRate for WP
				else
				{
				 // store the original maxPsduLengthLimit 
					originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex] = pLaProtectedStruct->wp.maxPsduLengthLimit;
				 	ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: overwriteMaxPsduLengthLimitInStaDb in SU. stored origLimit %d, staId %d",originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], pTcrModificationParams->controlParams.staIndex);
				 // apply minimum in case original maxLimit is lower than the limit recommended by system
				 	pLaProtectedStruct->common.maxPsduLengthLimit = MIN(pLaProtectedStruct->common.maxPsduLengthLimit, newMaxPsduLength);
					pLaProtectedStruct->wp.maxPsduLengthLimit = MIN(pLaProtectedStruct->wp.maxPsduLengthLimit, newMaxPsduLength);
					ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: overwrite wp and common. new wp %d, new common %d ",pLaProtectedStruct->wp.maxPsduLengthLimit, pLaProtectedStruct->common.maxPsduLengthLimit);
				}
			}
			// if current TcrRate was also problematic - required limitation in maxPsduLengthLimit
			else
			{
				
				// changing tcrRate only for probing
				if(pTcrModificationParams->controlParams.changeProbingPoint == TRUE)
				{
					
					pLaProtectedStruct->common.maxPsduLengthLimit = MIN(originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], newMaxPsduLength);
					ILOG0_DDD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: current rate problematic. Probing case. origLimit %d, newLimit %d, staId %d",originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], newMaxPsduLength, pTcrModificationParams->controlParams.staIndex);
				}
				// changing tcrRate for WP
				else
				{
					// apply minimum in case original maxLimit is lower than the limit recommended by system
					pLaProtectedStruct->common.maxPsduLengthLimit = MIN(originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], newMaxPsduLength);
					pLaProtectedStruct->wp.maxPsduLengthLimit = MIN(originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], newMaxPsduLength);
					// no need to not store the current maxPsduLengthLimit - because was previously stored
					ILOG0_DDD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: current rate problematic. Change WP case.  origLimit %d, newLimit %d, staId %d",originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], newMaxPsduLength, pTcrModificationParams->controlParams.staIndex);
				}
			}
		}
		// if tcrRate to be written doesn't require limitation in maxPsduLengthLimit
		else
		{
			// check if rollback must be done to original maxPsduLengthLimit - if current tcrRate is problematic and limitation is applied on maxPsduLengthLimit
			if(isPsduLengthLimitationNeeded(pLaProtectedStruct->common.tcr2160MhzDataPsduRate, &tmpPsduLength) == TRUE)
			{
				// if moving to "good" tcrRate only for probing - rollback maxPsduLengthLimit only in common part to it's original value
				if(pTcrModificationParams->controlParams.changeProbingPoint == TRUE)
				{
					pLaProtectedStruct->common.maxPsduLengthLimit = originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex];	
					ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: overwriteMaxPsduLengthLimitInStaDb - rollback in case of probing. origLimit %d, staId %d", originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], pTcrModificationParams->controlParams.staIndex);
				}
				// if moving to good tcrRate in WP as well - rollback maxPsduLengthLimit in commong and WP sections
				else
				{
					pLaProtectedStruct->common.maxPsduLengthLimit = originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex];
					pLaProtectedStruct->wp.maxPsduLengthLimit = originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex];
					ILOG0_DD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: overwriteMaxPsduLengthLimitInStaDb - rollback in case of wp. origLimit %d, staId %d", originalSuMaxPsduLengthLimit[pTcrModificationParams->controlParams.staIndex], pTcrModificationParams->controlParams.staIndex);
				}
			}
		}
	}
}

bool isPsduLengthLimitationNeeded(uint8 tcrRate, uint32 *newMaxPsduLength)
{
	bool limitPsduLength = FALSE;
	uint8 nss = EXTRACT_NSS_FROM_VHT_HE_RATE(tcrRate);
	uint8 mcs = EXTRACT_MCS_FROM_VHT_HE_RATE(tcrRate); 	

	if((MCS11_1024QAM_5_6 == mcs) && (LA_NSS_4 == nss))
	{
		*newMaxPsduLength = MCS11_NSS4_PSDU_LEN_LIMIT;
		limitPsduLength = TRUE;
	}
	else if(( MCS9_256QAM_5_6 == mcs) && (LA_NSS_4 == nss))
	{
		*newMaxPsduLength = MCS9_NSS4_PSDU_LEN_LIMIT;
		limitPsduLength = TRUE;
	}
	else if((MCS9_256QAM_5_6 == mcs) && (LA_NSS_2 == nss))
	{
		*newMaxPsduLength = MCS9_NSS2_PSDU_LEN_LIMIT;
		limitPsduLength = TRUE;
	}
	else 
	{
		*newMaxPsduLength = 0;
	}
	ILOG0_DDDD("PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400: isPsduLengthLimitationNeeded - limit required %d,nss %d, mcs %d new limit %d", limitPsduLength, nss, mcs,  *newMaxPsduLength);
	return limitPsduLength;
}

#endif
void setMpduInAmpduLimit(K_MSG *psMsg)
{
	LaMpduInAmpduConfigMsg_t* pMpdusInAmpdu = ((LaMpduInAmpduConfigMsg_t *)pK_MSG_DATA(psMsg));
	StaId nextSid;

	for (nextSid = 0; nextSid< HW_NUM_OF_STATIONS; nextSid++)
	{
		StaDbHwEntries[nextSid].common.aMpduLimit = pMpdusInAmpdu->suMpduInAmpdu;
	}

}

