/***********************************************************************************
 File:		ProtectionAdaptation.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "ProtectionAdaptation.h"
#include "HdkGlobalDefs.h"
#include "lm_StaDatabase.h"
#include "mt_sysdefs.h"
#include "Pac_Api.h"
#include "lm_StaDatabase.h"
#include "ShramStationDatabase.h"
#include "ShramVapDatabase.h"
#include "PacketDescriptor.h"
#include "ShramGroupDatabase.h"
#if defined(ENET_INC_ARCH_WAVE600)
#include "PreAggregator_Api.h"
#else
#include "AggregationBuilder_Api.h"
#endif 
#include "ConfigurationManager_api.h"
#include "CommonRamLinkAdaptation.h"
#include "ShramPreAggregator.h"
#include "PreAggRegs.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 15


#define PROTECTION_ADAPTATION_DEBUG
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

typedef enum 
{
	STA_PROTECTION_DISABLE_STATE,
	STA_PROTECTION_NONE_STATE,
	STA_PROTECTION_THRESHOLD_STATE,
	STA_PROTECTION_FORCE_STATE,
	STA_PROTECTION_INVALID_STATE,
	TOTAL_STA_PROTECTION_STATES = STA_PROTECTION_INVALID_STATE,
	STA_PROTECTION_LAST
} stationProtectionStates_e;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static vapProtectionStates_e vapProtectionSm[HW_NUM_OF_VAPS]; 
static stationProtectionStates_e staProtectionSm[HW_NUM_OF_STATIONS];
static ProtectionVapConfigParams_t ProtectionVapConfiguration;
static uint32 protectionNumberOfStations = 0; 

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

extern LinkAdaptationConfigurationParams_t LinkAdaptationConfigParamsSteadyState;
extern LinkAdaptationConfigurationParams_t LinkAdaptationConfigParamsFastInit;


/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

static void protectionAdaptationStaSm(StaId staId,uint8 vapId); 
static void protectionAdaptationSetStaProtectionMethod(StaId staId, RtsCtsTxMethod_e protectionMethod);
static void protectionAdaptationSetGroupProtectionMethod(uint8 groupId, RtsCtsTxMethod_e protectionMethod);
static void protectionAdaptationGroupHandler(uint8 groupId,uint8 vapId);

static void protectionAdaptationResetCollisionsCounters(uint8 vapId);
/**********************************************************************************

ProtectionAdaptationInit 


Description:		Initialize the module's VAP and Station SM 
------------

Input: None 
-----

Returns:	None 
--------
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif

void ProtectionAdaptation_Init (void)
{
	uint32 loopIterator; 
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();

	ProtectionVapConfiguration.vapBerGapPositiveThreshold = DEFAULT_PROTECTION_VAP_BER_GAP_POSITIVE_THRESHOLD;
	ProtectionVapConfiguration.vapStaCollisionTotalRatioPositiveThreshold = DEFAULT_PROTECTION_VAP_STACOLLISION_POSITIVE_THRESHOLD;
	ProtectionVapConfiguration.vapCollisionNonImprovementThreshold = DEFAULT_PROTECTION_VAP_COLLISION_NO_IMPROVEMENT_TH;
	ProtectionVapConfiguration.vapStaCollisionTotalRatioNegativeThreshold = DEFAULT_PROTECTION_VAP_STA_COLLISION_NEGATIVE_THRESHOLD;
	ProtectionVapConfiguration.wlanTimerIntervals = WALN_PROTECTION_TIMER_EVENT_VAL_USEC;
	ProtectionVapConfiguration.maxValidFilterTsfDiff = FILTER_VALIDATION_TIME; 

	for (loopIterator = firstVapInBand; loopIterator < (firstVapInBand + numOfVapsInBand); loopIterator++)
	{ 
		vapProtectionSm[loopIterator] = VAP_PROTECTION_INVALID_STATE;
#if defined (ENET_INC_ARCH_WAVE600)
		PreAggregator_SetForceProtection(loopIterator,PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS, FALSE); // This will clear both protection modes
#else
		AggregationBuilder_SetForceProtection(loopIterator,AGGR_BUILDER_FORCE_PROTECTION_MODE_RTS, FALSE); // This will clear both protection modes
#endif //ENET_INC_ARCH_WAVE600
		//AggregationBuilder_SetForceProtection(loopIterator,AGGR_BUILDER_FORCE_PROTECTION_MODE_CTS2SELF, FALSE);
		AlphaFilter_Init(&(LinkAdaptationVapDatabase[loopIterator].estimatorsVapDataBase.protectedFramesBER),&(LinkAdaptationVapDatabase[loopIterator].estimatorsVapDataBase.protectedFramesBERDeltaTAverage));
		AlphaFilter_Init(&(LinkAdaptationVapDatabase[loopIterator].estimatorsVapDataBase.unprotectedFramesBER),&(LinkAdaptationVapDatabase[loopIterator].estimatorsVapDataBase.unprotectedFramesBERDeltaTAverage));

	}
	for(loopIterator = 0; loopIterator < HW_NUM_OF_STATIONS; loopIterator++)
	{
		staProtectionSm[loopIterator] = STA_PROTECTION_INVALID_STATE;
	}
		

}

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default 
#endif




/**********************************************************************************

ProtectionAdaptation_SetConfiguration 


Description:		Set the protection configuration from external module

------------

Input: Vap index 
-----

Returns:	none 
--------
	
**********************************************************************************/

void ProtectionAdaptation_SetConfiguration(ProtectionConfigurationIndex configurationIndex, uint8 value)
{
	switch(configurationIndex)
	{
		case PROTECTION_VAP_BER_GAP_POSITIVE_THRESHOLD_INDX:
			ProtectionVapConfiguration.vapBerGapPositiveThreshold = value;
			break;
		case PROTECTION_VAP_COLLISION_POSITIVE_THRESHOLD_INDX:
			ProtectionVapConfiguration.vapStaCollisionTotalRatioPositiveThreshold = value;
			break;
		case PROTECTION_VAP_COLLISION_NEGATIVE_THRESHOLD_INDX:
			ProtectionVapConfiguration.vapStaCollisionTotalRatioNegativeThreshold = value;
			break;
		case PROTECTION_STA_NONE_MODE_POSITIVE_THRESHOLD_INDX:
			LinkAdaptationConfigParamsSteadyState.protectionConf.staCollisionNoProtPositiveThreshold = value;
			LinkAdaptationConfigParamsFastInit.protectionConf.staCollisionNoProtPositiveThreshold = value;
			break;
		case PROTECTION_STA_THRESHOLD_MODE_POSITIVE_THRESHOLD_INDX:
			LinkAdaptationConfigParamsSteadyState.protectionConf.staCollisionThresholdModePsotiveThreshold = value;
			LinkAdaptationConfigParamsFastInit.protectionConf.staCollisionThresholdModePsotiveThreshold = value;
			break;
		case PROTECTION_STA_THRESHOLD_MODE_NGATIVE_THRESHOLD_INDX:
			LinkAdaptationConfigParamsSteadyState.protectionConf.staCollisionThresholdModeNegativeThreshold = value;
			LinkAdaptationConfigParamsFastInit.protectionConf.staCollisionThresholdModeNegativeThreshold = value;

			break;
		case PROTECTION_STA_FORCE_MODE_NEGATIVE_THRESHOLD_INDX:
			LinkAdaptationConfigParamsSteadyState.protectionConf.staCollisionForceModeNegativeThreshold = value;
			LinkAdaptationConfigParamsFastInit.protectionConf.staCollisionForceModeNegativeThreshold = value;

			break;
	}
}


/**********************************************************************************

ProtectionAdaptationAddVap 


Description:		Following "Add vap" event, the Vap SM warms up and its at the initial state of  
				VAP_PROTECTION_NONE_STATE
------------

Input: Vap index 
-----

Returns:	none 
--------
	
**********************************************************************************/
void ProtectionAdaptation_AddVap(uint8 vapId)
{

	DEBUG_ASSERT(vapProtectionSm[vapId] == VAP_PROTECTION_INVALID_STATE);
	memset(&(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase),0,sizeof(vapEstimatorsDataBase_t));
	AlphaFilter_Init(&(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.protectedFramesBER),&(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.protectedFramesBERDeltaTAverage));
	AlphaFilter_Init(&(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.unprotectedFramesBER),&(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.unprotectedFramesBERDeltaTAverage));
	vapProtectionSm[vapId] = VAP_PROTECTION_NONE_STATE;
}

/**********************************************************************************

ProtectionAdaptationRemoveVap 


Description:		Following "Remove vap" event, the Vap SM goes to inactive mode (invalid protection state)
------------

Input: Vap index 
-----

Returns:	none 
--------
	
**********************************************************************************/

void ProtectionAdaptation_RemoveVap(uint8 vapId)
{
	DEBUG_ASSERT(vapProtectionSm[vapId] != VAP_PROTECTION_INVALID_STATE);
	vapProtectionSm[vapId] = VAP_PROTECTION_INVALID_STATE;
}

/**********************************************************************************

ProtectionAdaptation_VapSetBss 


Description:		Following "Add vap" event, the Vap SM warms up and its at the initial state of  
				VAP_PROTECTION_NONE_STATE
------------

Input: Vap index 
-----

Returns:	none 
--------
	
**********************************************************************************/

void ProtectionAdaptation_VapSetBss(UMI_SET_BSS* pVapSetBss)
{
	SLOG0(0, 0, UMI_SET_BSS, pVapSetBss);
	if(pVapSetBss->protectionMode == PROTECTION_MODE_NO_PROTECTION)
	{
		ProtectionAdaptation_VapSm(pVapSetBss->vapId, VAP_DISABLE_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS); // Protection mode is don't care in this case		
	}
	else if(pVapSetBss->protectionMode == PROTECTION_MODE_NO_FORCE_PROTECTION)
	{
		ProtectionAdaptation_VapSm(pVapSetBss->vapId, VAP_RELEASE_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS); // Protection mode is don't care in this case		
		ProtectionAdaptation_CtsEnable(pVapSetBss->vapId, PROTECTION_STA_RTS_METHOD); //cts protection disabled
	}	
	else if (pVapSetBss->protectionMode == PROTECTION_MODE_FORCE_RTS_CTS)
	{
		ProtectionAdaptation_VapSm(pVapSetBss->vapId, VAP_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS);
	}
	else if (pVapSetBss->protectionMode == PROTECTION_MODE_FORCE_CTS_TO_SELF)
	{
		ProtectionAdaptation_VapSm(pVapSetBss->vapId, VAP_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_CTS2SELF);
	}
	else // pVapSetBss->protectionMode == PROTECTION_MODE_CTS_TO_SELF_ENABLE
	{
		ProtectionAdaptation_CtsEnable(pVapSetBss->vapId, PROTECTION_STA_CTS2SELF_METHOD);
		ProtectionAdaptation_VapSm(pVapSetBss->vapId, VAP_RELEASE_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_CTS2SELF);
	}
}


void ProtectionAdaptation_SetFixedMode(K_MSG *psMsg)
{
	ProtectionFixed_t *pProtectionFixed = (ProtectionFixed_t *)pK_MSG_DATA(psMsg);


	if(pProtectionFixed->event == PROTECTION_MODE_NO_PROTECTION)
	{
	
		ProtectionAdaptation_VapSm(pProtectionFixed->vapId, VAP_DISABLE_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS); // Protection mode is don't care in this case
		
	}
	else if (pProtectionFixed->event == PROTECTION_MODE_NO_FORCE_PROTECTION)
	{
		ProtectionAdaptation_VapSm(pProtectionFixed->vapId, VAP_RELEASE_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS); // Protection mode is don't care in this case
		
	}
	else if (pProtectionFixed->event == PROTECTION_MODE_FORCE_RTS_CTS)
	{
		ProtectionAdaptation_VapSm(pProtectionFixed->vapId, VAP_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS);
	}
	else // pVapSetBss->protectionMode == PROTECTION_MODE_FORCE_CTS_TO_SELF
	{
		ProtectionAdaptation_VapSm(pProtectionFixed->vapId, VAP_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_CTS2SELF);
	}
}

/**********************************************************************************

ProtectionAdaptationAddSta


Description:	Following "Add Station" event, the STA SM warms up and its at the initial state of  
			STA_PROTECTION_NONE_STATE
------------

Input: Station index
-----

Returns: None
--------
	
**********************************************************************************/

void ProtectionAdaptation_AddSta(StaId staId)
{
	staProtectionSm[staId] = STA_PROTECTION_INVALID_STATE;
//	memset(&(LinkAdaptationStaDatabase[staId].stationEstimatorsDataBase.staProtection),0,sizeof(EstimatorsStaProtection_t));
	staProtectionSm[staId] = STA_PROTECTION_NONE_STATE;
	if(RTS_THRESHOLD_PPDU_SIZE_ON == LinkAdaptationVapDatabase[StaDbSwEntries[staId].vapIndex].estimatorsVapDataBase.vapRtsThresholdPpduSizeEnable) //rts threshold by ppdu size is activated
	{
		protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_RTS_CTS_BASED_THRESHOLD);
	}
	else
	{
		protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_NO_RTS_CTS);
	}
	DEBUG_ASSERT(protectionNumberOfStations < HW_NUM_OF_STATIONS);
	protectionNumberOfStations++;
	if (protectionNumberOfStations == 1)
	{
		linkAdaptaionActivateTimersIds[PROTECTION_ADAPTATION_WLAN_TIMER] = TimerUtiltyAddEvent(ProtectionVapConfiguration.wlanTimerIntervals,ProtectionAdaptation_WlanTimerISR, INVALID_STA_INDEX);
	}
	
}

/**********************************************************************************

ProtectionAdaptationRemoveSta


Description:	Following "Remove Station" event, the STA SM goes to inactive mode (invalid protection state)
------------

Input: Station index
-----

Returns: None
--------
	
**********************************************************************************/

void ProtectionAdaptation_RemoveSta(StaId staId)
{
	DEBUG_ASSERT(staProtectionSm[staId] != STA_PROTECTION_INVALID_STATE);
	staProtectionSm[staId] = STA_PROTECTION_INVALID_STATE;
	DEBUG_ASSERT(protectionNumberOfStations > 0);
	protectionNumberOfStations --;
	if(protectionNumberOfStations == 0)
	{
		TimerUtiltyRemoveEvent(&linkAdaptaionActivateTimersIds[PROTECTION_ADAPTATION_WLAN_TIMER]);
	}
}


/**********************************************************************************

ProtectionAdaptationProcessBaaReport 


Description: 	Called after estimators are updated and everithing is ready for VAP/STA/WLAN proteciton handling 
------------

Input: Station index and VAP index 
-----

Returns: None
--------
	
**********************************************************************************/
void ProtectionAdaptation_Handler(StaId staId, uint8 vapId, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	//when rts threshold by ppdu size is activeated, adaptation is in wlan level ProtectionAdaptation_WlanTimerISR
	if(RTS_THRESHOLD_PPDU_SIZE_OFF == LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.vapRtsThresholdPpduSizeEnable)
	{
#if defined (ENET_INC_ARCH_WAVE600)
		ProtectionAdaptation_VapSm(vapId, VAP_NONE_FORCED_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS); // Protection mode is don't care in this case
#else
		ProtectionAdaptation_VapSm(vapId, VAP_NONE_FORCED_PROTECTION, AGGR_BUILDER_FORCE_PROTECTION_MODE_RTS); // Protection mode is don't care in this case
#endif //ENET_INC_ARCH_WAVE600

		if(laDbDistributionParameter->uspIndex==  INVALID_MU_USP_INDEX)
		{
			if ((vapProtectionSm[vapId] != VAP_PROTECTION_FORCED_STATE) && (vapProtectionSm[vapId] != VAP_DISABLE_PROTECTION_STATE))
			{
				protectionAdaptationStaSm(staId,vapId);
			}
		}
		else /*MU*/
		{
			protectionAdaptationGroupHandler(laDbDistributionParameter->stationOrGroupIndex, vapId);
		}
	}
}

/**********************************************************************************

ProtectionAdaptation_VapSm 


Description:	 The VAP SM entry point. Set the new state of the VAP according to the current state and the updated estimator parameters.
------------

Input:  Vap index, Forced protection method 
-----

Returns: None 
--------
	
**********************************************************************************/
#if defined (ENET_INC_ARCH_WAVE600)
void ProtectionAdaptation_VapSm(uint8 vapId, vapForcedProtectionConf_e forcedProtectionConf, PreAggForceProtectionMode_e protectionMethod)
#else
void ProtectionAdaptation_VapSm(uint8 vapId, vapForcedProtectionConf_e forcedProtectionConf, AggrBuilderForceProtectionMode_e protectionMethod)
#endif //ENET_INC_ARCH_WAVE600
{
	vapEstimatorsDataBase_t* pVapEstimatorsPtr = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase);
	uint32	berGap = 0;


	SLOG0(0, 0, vapEstimatorsDataBase_t, pVapEstimatorsPtr);

#ifdef PROTECTION_ADAPTATION_DEBUG
//	if(pVapEstimatorsPtr->vapCollisionTotalRatio > 60)
//		ILOG0_DDD("ProtectionAdaptation_VapSm - vapCollisionTotalRatio is very high : %d, Ackbitmap:%d, Ctsbitmap:%d",pVapEstimatorsPtr->vapCollisionTotalRatio,pVapEstimatorsPtr->vapCollisionAckBitmap,pVapEstimatorsPtr->vapCollisionAckCtsBitmap);
#endif
	/*********************/
	/* Vap protection SM	 */
	/********************/
	switch (vapProtectionSm[vapId])  
	{
	case VAP_PROTECTION_INVALID_STATE:
#ifdef PROTECTION_ADAPTATION_DEBUG
		ILOG0_V("ProtectionAdaptation_VapSm -got indication for invalid VAP ");
#endif 
		break; /* In case where vap isn't active anymore - ignore reports of previouse transmissions*/
		
	case VAP_PROTECTION_NONE_STATE:
		switch (forcedProtectionConf) 
		{
			case VAP_FORCED_PROTECTION:
				vapProtectionSm[vapId] = VAP_PROTECTION_FORCED_STATE;
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,protectionMethod, TRUE);
#else
				AggregationBuilder_SetForceProtection(vapId,protectionMethod, TRUE);
#endif //ENET_INC_ARCH_WAVE600
#ifdef PROTECTION_ADAPTATION_DEBUG
				ILOG0_D("ProtectionAdaptation_VapSm -VAP state:NONE-VAP_PROTECTION_FORCE_STATE : %d",protectionMethod);
#endif

				break; 
			case VAP_NONE_FORCED_PROTECTION:
#ifdef PROTECTION_ADAPTATION_DEBUG
				ILOG0_D("ProtectionAdaptation_VapSm -VAP in PROTECTION_NONE_STATE collsion Count : %d",pVapEstimatorsPtr->vapCollisionTotalRatio);
#endif
				if(pVapEstimatorsPtr->vapCollisionTotalRatio > ProtectionVapConfiguration.vapStaCollisionTotalRatioPositiveThreshold)
				{
#ifdef PROTECTION_ADAPTATION_DEBUG
					ILOG0_D("ProtectionAdaptation_VapSm -Transition to threshold - collsion Count : %d",pVapEstimatorsPtr->vapCollisionTotalRatio);
#endif
					vapProtectionSm[vapId] = VAP_PROTECTION_THRESHOLD_STATE; 
				}
				break; 
			case VAP_RELEASE_FORCED_PROTECTION:
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#else				
				AggregationBuilder_SetForceProtection(vapId,AGGR_BUILDER_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#endif //ENET_INC_ARCH_WAVE600
				break; 
			case VAP_DISABLE_PROTECTION:
				vapProtectionSm[vapId] = VAP_DISABLE_PROTECTION_STATE; 		
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#else				
				AggregationBuilder_SetForceProtection(vapId,AGGR_BUILDER_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#endif //ENET_INC_ARCH_WAVE600				
				ProtectionAdaptation_DisableProtectionForAllStationsInVap(vapId);
				break; 				
			default:
				DEBUG_ASSERT(0); //klocwork ignore, UNREACH.GEN..... All the cases in the ENUM is handled, klockwork sees this assert as unreachable code.
		}
		break;
	case VAP_DISABLE_PROTECTION_STATE: // 11n protection Disabled 
		switch (forcedProtectionConf) 
		{
			case VAP_FORCED_PROTECTION:
				vapProtectionSm[vapId] = VAP_PROTECTION_FORCED_STATE;
				
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,protectionMethod, TRUE);
#else				
				AggregationBuilder_SetForceProtection(vapId,protectionMethod, TRUE);
#endif //ENET_INC_ARCH_WAVE600
				
				break; 
			case VAP_NONE_FORCED_PROTECTION:
				break; 
			case VAP_RELEASE_FORCED_PROTECTION:
				vapProtectionSm[vapId] = VAP_PROTECTION_NONE_STATE;
				ProtectionAdaptation_ResetProtectiontoAllStationsInVap(vapId);
				ProtectionAdaptation_ResetVapEstimators(vapId);
				break; 
			case VAP_DISABLE_PROTECTION:
				break; 
			default:
				DEBUG_ASSERT(0); //klocwork ignore, UNREACH.GEN..... All the cases in the ENUM is handled, klockwork sees this assert as unreachable code.
		}
		break;
	case VAP_PROTECTION_THRESHOLD_STATE: 
		switch (forcedProtectionConf) 
		{
			case VAP_FORCED_PROTECTION:
				vapProtectionSm[vapId] = VAP_PROTECTION_FORCED_STATE;
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,protectionMethod, TRUE);
#else	
				AggregationBuilder_SetForceProtection(vapId,protectionMethod, TRUE);
#endif //ENET_INC_ARCH_WAVE600
				break; 
			case VAP_NONE_FORCED_PROTECTION:
#ifdef PROTECTION_ADAPTATION_DEBUG
				ILOG0_V("ProtectionAdaptation_VapSm - VAP state:  VAP_PROTECTION_THRESHOLD_STATE");
#endif

				if((AlphaFilter_isFilterResultValid(&pVapEstimatorsPtr->protectedFramesBER, ProtectionVapConfiguration.maxValidFilterTsfDiff) == FALSE) || 
				(AlphaFilter_isFilterResultValid(&pVapEstimatorsPtr->unprotectedFramesBER, ProtectionVapConfiguration.maxValidFilterTsfDiff) == FALSE))
				{
					berGap = 0; 
				}
				else
				{
					berGap = (uint32)(MAX(0,(int32)(AlphaFilter_GetFilterResult(&(pVapEstimatorsPtr->protectedFramesBER)) - AlphaFilter_GetFilterResult(&(pVapEstimatorsPtr->unprotectedFramesBER)))));
				}
				ILOG0_D("ProtectionAdaptation_VapSm - berGap = %d", berGap);
				if((pVapEstimatorsPtr->vapCollisionTotalRatio < ProtectionVapConfiguration.vapStaCollisionTotalRatioNegativeThreshold)||
				(berGap > ProtectionVapConfiguration.vapBerGapPositiveThreshold)) 		
				{
#ifdef PROTECTION_ADAPTATION_DEBUG
					ILOG0_V("ProtectionAdaptation_VapSm - Transition to  VAP_PROTECTION_NONE_STATE");
#endif
					vapProtectionSm[vapId] = VAP_PROTECTION_NONE_STATE;
					pVapEstimatorsPtr->vapThresholdModeCounter = 0; 
					protectionAdaptationResetCollisionsCounters(vapId);
				}
				else
				{
					pVapEstimatorsPtr->vapThresholdModeCounter++; 
					/*If no imporevent achived with protection after a while - don't use protection*/
					if(pVapEstimatorsPtr->vapThresholdModeCounter > VAP_THRES_MODE_ITERATIONS_NO_IMPROVEMENT)
					{
						if(pVapEstimatorsPtr->vapCollisionTotalRatio > ProtectionVapConfiguration.vapCollisionNonImprovementThreshold)
						{
							vapProtectionSm[vapId] = VAP_PROTECTION_NONE_STATE;
							pVapEstimatorsPtr->vapThresholdModeCounter = 0; 
							/*Start from zero - as if the Vap has just been added (BER would stay with its value)*/
							/*Reset collisions counters*/
							protectionAdaptationResetCollisionsCounters(vapId);
#ifdef PROTECTION_ADAPTATION_DEBUG
							ILOG0_V("ProtectionAdaptation_VapSm - Transition to  VAP_PROTECTION_NONE_STATE - protection doen't improve");
#endif
						}
						else
						{
							/*we should wait and see if the next packet would make us pass the threshold*/
							pVapEstimatorsPtr->vapThresholdModeCounter--; 	
						}
					}
				}
				break; 
			case VAP_RELEASE_FORCED_PROTECTION:
				break; 
			case VAP_DISABLE_PROTECTION:
#ifdef PROTECTION_ADAPTATION_DEBUG
				ILOG0_V("ProtectionAdaptation_VapSm -VAP state:TH- VAP_PROTECTION_DISABLE_STATE");
#endif
				vapProtectionSm[vapId] = VAP_DISABLE_PROTECTION_STATE;
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#else	
				AggregationBuilder_SetForceProtection(vapId,AGGR_BUILDER_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#endif //ENET_INC_ARCH_WAVE600
				ProtectionAdaptation_DisableProtectionForAllStationsInVap(vapId);
				break;
			default:
				DEBUG_ASSERT(0); //klocwork ignore, UNREACH.GEN..... All the cases in the ENUM is handled, klockwork sees this assert as unreachable code.
		}
		break; 
	case VAP_PROTECTION_FORCED_STATE: 
#ifdef PROTECTION_ADAPTATION_DEBUG
		ILOG0_V("ProtectionAdaptation_VapSm -VAP state: VAP_PROTECTION_FORCED_STATE");
#endif
		switch (forcedProtectionConf) 
		{
			case VAP_RELEASE_FORCED_PROTECTION:
#ifdef PROTECTION_ADAPTATION_DEBUG
			ILOG0_V("ProtectionAdaptation_VapSm -transition to VAP_PROTECTION_NONE_STATE");
#endif		
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#else	
				AggregationBuilder_SetForceProtection(vapId,AGGR_BUILDER_FORCE_PROTECTION_MODE_RTS, FALSE); // ProtectionMode is DontCare when we turn it off.
#endif //ENET_INC_ARCH_WAVE600

				vapProtectionSm[vapId] = VAP_PROTECTION_NONE_STATE; 
				break;
			case VAP_FORCED_PROTECTION:	
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,protectionMethod, TRUE);
#else	
				AggregationBuilder_SetForceProtection(vapId,protectionMethod, TRUE);
#endif //ENET_INC_ARCH_WAVE600
				
				break; 
			case VAP_DISABLE_PROTECTION:
#ifdef PROTECTION_ADAPTATION_DEBUG
				ILOG0_V("ProtectionAdaptation_VapSm -VAP state:Force- VAP_PROTECTION_DISABLE_STATE");
#endif
				vapProtectionSm[vapId] = VAP_DISABLE_PROTECTION_STATE;
#if defined (ENET_INC_ARCH_WAVE600)
				PreAggregator_SetForceProtection(vapId,protectionMethod, FALSE);
#else	
				AggregationBuilder_SetForceProtection(vapId,protectionMethod, FALSE);
#endif //ENET_INC_ARCH_WAVE600				
				ProtectionAdaptation_DisableProtectionForAllStationsInVap(vapId);
				break;
			case VAP_NONE_FORCED_PROTECTION:
				break;
			default:
				DEBUG_ASSERT(0); //klocwork ignore, UNREACH.GEN..... All the cases in the ENUM is handled, klockwork sees this assert as unreachable code.
		}
		break; 
	default:
		DEBUG_ASSERT(0);
	}
}


/**********************************************************************************

protectionAdaptationStaSm 


Description:	The Station SM entry point. Set the new state of the Station according to the current state and the updated estimator parameters.
------------

Input: Station index. Vap index
-----

Returns:	None 
--------
	
**********************************************************************************/

static void protectionAdaptationStaSm(StaId staId,uint8 vapId)
{
	ProtectionStaConfigParams_t* pStaProtectionConf = &(LinkAdaptationStaDatabase[staId].laStaGroupCommon.pLinkAdaptationConfigurationParams->protectionConf);
	EstimatorsStaProtection_t* pStaProtectionEstimatorPtr = &(LinkAdaptationStaDatabase[staId].laStaGroupCommon.staProtection);
	vapEstimatorsDataBase_t* pVapEstimatorsPtr = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase);
	int8 vapStaCollisionDelta;
	
#ifdef PROTECTION_ADAPTATION_DEBUG
	ILOG0_DDD("protectionAdaptationStaSm - staCollisionCount : %d, Ackbitmap:%d, Ctsbitmap:%d",pStaProtectionEstimatorPtr->staCollisionCount,pStaProtectionEstimatorPtr->staCollisionAckCtsBitmap,pStaProtectionEstimatorPtr->staCollisionAckCtsBitmap);
	if(pStaProtectionEstimatorPtr->staCollisionCount> 60)
		ILOG0_V("protectionAdaptationStaSm - staCollisionCount is very high"); 
#endif
	ILOG0_DD("protectionAdaptationStaSm, vapCollisionResetCounter = %d, staCollisionResetCounter = %d", pVapEstimatorsPtr->vapCollisionResetCounter, pStaProtectionEstimatorPtr->staCollisionResetCounter);
	if (pVapEstimatorsPtr->vapCollisionResetCounter != pStaProtectionEstimatorPtr->staCollisionResetCounter)
	{
		pStaProtectionEstimatorPtr->staCollisionCount = 0;
		pStaProtectionEstimatorPtr->staCollisionAckCtsBitmap = 0; 
		pStaProtectionEstimatorPtr->staCollisionResetCounter = pVapEstimatorsPtr->vapCollisionResetCounter;
		ILOG0_D("protectionAdaptationStaSm, pStaProtectionEstimatorPtr->staCollisionResetCounter = %d", pStaProtectionEstimatorPtr->staCollisionResetCounter);
	}
	vapStaCollisionDelta = pStaProtectionEstimatorPtr->staCollisionCount- pVapEstimatorsPtr->vapCollisionTotalRatio; 
	/*********************/
	/* Station protection SM*/
	/********************/
	switch (staProtectionSm[staId])  
	{
	case STA_PROTECTION_DISABLE_STATE:
		break;
	case STA_PROTECTION_NONE_STATE:
#ifdef PROTECTION_ADAPTATION_DEBUG
		ILOG0_V("protectionAdaptationStaSm -statino state: STA_PROTECTION_NONE_STATE");
#endif
		if((pStaProtectionEstimatorPtr->staCollisionCount > ProtectionVapConfiguration.vapStaCollisionTotalRatioPositiveThreshold) ||
			(vapStaCollisionDelta > (int)(pStaProtectionConf->staCollisionNoProtPositiveThreshold))||
			((vapProtectionSm[vapId] == VAP_PROTECTION_THRESHOLD_STATE)&&(vapStaCollisionDelta > (pStaProtectionConf->staCollisionBetterThanVapThreshold))))
		{
#ifdef PROTECTION_ADAPTATION_DEBUG
			ILOG0_D("protectionAdaptationStaSm -transition to STA_THRESHOLD_MODE vapStaCollisionDelta = %d",vapStaCollisionDelta);
#endif
			staProtectionSm[staId] = STA_PROTECTION_THRESHOLD_STATE;
			/*configure DB*/
			protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_RTS_CTS_BASED_THRESHOLD);
		}
		break;	
	case STA_PROTECTION_THRESHOLD_STATE: 
#ifdef PROTECTION_ADAPTATION_DEBUG
		ILOG0_D("protectionAdaptationStaSm -statino state: STA_PROTECTION_THRESHOLD_STATE  vapStaCollisionDelta = %d",vapStaCollisionDelta);
#endif
		if(vapStaCollisionDelta > (int)(pStaProtectionConf->staCollisionThresholdModePsotiveThreshold))
		{
#ifdef PROTECTION_ADAPTATION_DEBUG
			ILOG0_D("protectionAdaptationStaSm -transition to STA_FORCE_PROTECTION_MODE vapStaCollisionDelta = %d",vapStaCollisionDelta);
#endif
			staProtectionSm[staId] = STA_PROTECTION_FORCE_STATE;
			/*configure DB*/
			protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_FORCE_RTS_CTS);
		}
		else if((vapStaCollisionDelta < (int)(pStaProtectionConf->staCollisionThresholdModeNegativeThreshold))
				&& (((vapProtectionSm[vapId] == VAP_PROTECTION_NONE_STATE) || (vapProtectionSm[vapId] == VAP_DISABLE_PROTECTION_STATE)) && (pStaProtectionEstimatorPtr->staCollisionCount < ProtectionVapConfiguration.vapStaCollisionTotalRatioNegativeThreshold)))
		{
#ifdef PROTECTION_ADAPTATION_DEBUG
			ILOG0_D("protectionAdaptationStaSm -transition to STA_NO_PROTECTION_MODE vapStaCollisionDelta = %d",vapStaCollisionDelta);
#endif
			staProtectionSm[staId] = STA_PROTECTION_NONE_STATE;
			/*configure DB*/
			protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_NO_RTS_CTS);
		}

		break; 
	case STA_PROTECTION_FORCE_STATE: 
#ifdef PROTECTION_ADAPTATION_DEBUG
		ILOG0_D("protectionAdaptationStaSm -statino state: STA_PROTECTION_FORCE_STATE vapStaCollisionDelta = %d",vapStaCollisionDelta);
#endif
		if(vapStaCollisionDelta < (int)(pStaProtectionConf->staCollisionForceModeNegativeThreshold))
		{
#ifdef PROTECTION_ADAPTATION_DEBUG
			ILOG0_D("protectionAdaptationStaSm -transition to STA_THRESHOLD_MODE - delta exceed threshold, delta = %d", vapStaCollisionDelta);
#endif
			staProtectionSm[staId] = STA_PROTECTION_THRESHOLD_STATE;
			/*configure DB*/
			protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_RTS_CTS_BASED_THRESHOLD);
		}
		else
		{
			pStaProtectionEstimatorPtr->staForceProtectionIterationsCount++; 
			/*If no imporevent achived with protection after a while - don't use protection*/
			if(pStaProtectionEstimatorPtr->staForceProtectionIterationsCount > STA_FORCE_MODE_NO_IMPROVEMENT_ITERATIONS)
			{
				if((pStaProtectionEstimatorPtr->staCollisionCount > ProtectionVapConfiguration.vapStaCollisionTotalRatioPositiveThreshold) 
					&&(vapStaCollisionDelta > (int)(pStaProtectionConf->staCollisionThresholdModePsotiveThreshold)))
				{
#ifdef PROTECTION_ADAPTATION_DEBUG
					ILOG0_D("protectionAdaptationStaSm -transition to STA_NO_PROTECTION_MODE - No improvement with protection vapStaCollisionDelta = %d",vapStaCollisionDelta);
#endif
					staProtectionSm[staId] = STA_PROTECTION_NONE_STATE;
					/*configure DB*/
					protectionAdaptationSetStaProtectionMethod(staId, RTS_CTS_TX_METHOD_NO_RTS_CTS);
					pStaProtectionEstimatorPtr->staForceProtectionIterationsCount = 0; 
					/* Start from zero - as if the station has just connected*/
					pStaProtectionEstimatorPtr->staCollisionCount = 0;
					pStaProtectionEstimatorPtr->staCollisionAckCtsBitmap = 0;
				}
				else
				{
					/*we should wait and see if the next packet would make us pass the threshold*/
					pStaProtectionEstimatorPtr->staForceProtectionIterationsCount--; 	
				}
			}
		}
		break; 
	case STA_PROTECTION_INVALID_STATE:
	/* There might be a case where a station has been removed and then a priviouse BA report arrived*/
#ifdef PROTECTION_ADAPTATION_DEBUG
		ILOG0_V("protectionAdaptationStaSm -BAA report arrived to a non-valid station!");
#endif	
		break;
	default:
		DEBUG_ASSERT(0);
		break;
	}
}


static void protectionAdaptationGroupHandler(uint8 groupId,uint8 vapId)
{
	/*take the vap policy as the group's protection policy*/
	switch (vapProtectionSm[vapId])	
	{
	case VAP_PROTECTION_INVALID_STATE:
		//FATAL("ProtectionAdaptation_Handler - BAA report associated with inactive VAP recieved ");
		break;
	case VAP_PROTECTION_NONE_STATE: 
	case VAP_DISABLE_PROTECTION_STATE:
		protectionAdaptationSetGroupProtectionMethod(groupId,RTS_CTS_TX_METHOD_NO_RTS_CTS);
		break; 
	case VAP_PROTECTION_THRESHOLD_STATE: 
		protectionAdaptationSetGroupProtectionMethod(groupId,RTS_CTS_TX_METHOD_RTS_CTS_BASED_THRESHOLD);
		break; 
	case VAP_PROTECTION_FORCED_STATE: 
		protectionAdaptationSetGroupProtectionMethod(groupId,RTS_CTS_TX_METHOD_FORCE_RTS_CTS);
		break; 
	default:
		DEBUG_ASSERT(0);
	}
}


/**********************************************************************************

protectionAdaptationWlanHandler 


Description:	Handle the WLAN protection, which means set a duration ratio threshlod
------------

	This function should called periodically by timer 

Input: 
-----

Returns:	None 
--------
	
**********************************************************************************/
void ProtectionAdaptation_WlanTimerEventHandler(K_MSG *psMsg)
{
	LaTimerEventsTimerEvent_t* pTimerExpirationTsf = (LaTimerEventsTimerEvent_t*)pK_MSG_DATA(psMsg);
	uint32 timeToGo = 0;
	uint32 tsfLow = GET_TSF_TIMER_LOW();
	uint8 collisionRatio =	(NUMBER_OF_FRAMES_IN_COLLISION_BITMAPS - wlanCollisionTotlaCount)>>DURATION_RATIO_FACTOR_SHIFT;
	
	timeToGo = calculateTimerExpirationTime(ProtectionVapConfiguration.wlanTimerIntervals,tsfLow, pTimerExpirationTsf->timerExpirationTsf );
	/*Periodic timer - set timer after the previous has expired - in case where at least one VAP is up */
	if(protectionNumberOfStations > 0)
	{
		linkAdaptaionActivateTimersIds[PROTECTION_ADAPTATION_WLAN_TIMER] = TimerUtiltyAddEvent(timeToGo,ProtectionAdaptation_WlanTimerISR, INVALID_STA_INDEX);
	}
	/* Write the updated threshold configuration to aggregation builder/pre aggregator*/
#if defined (ENET_INC_ARCH_WAVE600)
	PreAggregator_SetDurationRatioThreshold(MIN(PRE_AGGREGATOR_MAX_DURATION_THRESHOLD_CONFIGURATION,collisionRatio)); //Values in 5 bit resoulsion
#else	
	AggregationBuilder_SetDurationRatioThreshold(MIN(AGGR_BUILDER_MAX_DURATION_THRESHOLD_CONFIGURATION,collisionRatio)); //Values in 5 bit resoulsion
#endif //ENET_INC_ARCH_WAVE600

#ifdef PROTECTION_ADAPTATION_DEBUG
	ILOG0_D("ProtectionAdaptation_WlanTimerEventHandler -collosionRatio == %d",collisionRatio);
#endif
}



/**********************************************************************************

ProtectionAdaptation_WlanTimerISR 


Description:	Handle the WLAN protection timer ISR,  send message to the handler 
------------

Input: 
-----

Returns:	None 
--------
	
**********************************************************************************/
void ProtectionAdaptation_WlanTimerISR(uint32 param)
{
	K_MSG* pWlanProtectiontimerEventMsg = NULL;
	LaTimerEventsTimerEvent_t*	pTimerExpirationTsf;
	UNUSED_PARAM(param);	
	
	pWlanProtectiontimerEventMsg = OSAL_GET_MESSAGE(sizeof(LaTimerEventsTimerEvent_t));	
	pTimerExpirationTsf = (LaTimerEventsTimerEvent_t*)pK_MSG_DATA(pWlanProtectiontimerEventMsg);
	pTimerExpirationTsf->timerExpirationTsf = GET_TSF_TIMER_LOW();
	linkAdaptaionActivateTimersIds[PROTECTION_ADAPTATION_WLAN_TIMER] = MAX_UINT16;
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_WLAN_PROTECTION_TIMER_EVENT, TASK_LINK_ADAPTATION, pWlanProtectiontimerEventMsg, VAP_ID_DO_NOT_CARE);	
}


/**********************************************************************************

ProtectionAdaptation_AgingEvent


Description:	Handle aging event, check for each vap whether it has reach the aging limit
------------

Input: 
-----

Returns:	None 
--------
	
**********************************************************************************/

void ProtectionAdaptation_AgingEvent(void)
{
	uint8 vapId = 0; 
	AdaptiveAlphaFilterParams_t* pProtectedFramesBER;
	AdaptiveAlphaFilterParams_t* pUnprotectedFramesBER;
	uint16* pProtectedFramesBERAverageDt;
	uint16* pUnProtectedFramesBERAverageDt;
	uint16 maxValidTsfDiff;
	uint8 firstVapInBand = ConfigurationManager_GetFirstVapForMyBand();
	uint8 numOfVapsInBand = ConfigurationManager_GetNumOfVapsInMyBand();

	for (vapId = firstVapInBand; vapId < (firstVapInBand + numOfVapsInBand); vapId++)
	{
		pProtectedFramesBER = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.protectedFramesBER); 
		pUnprotectedFramesBER = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.unprotectedFramesBER); 	
		pProtectedFramesBERAverageDt = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.protectedFramesBERDeltaTAverage);
		pUnProtectedFramesBERAverageDt = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.unprotectedFramesBERDeltaTAverage);

		maxValidTsfDiff  = ProtectionVapConfiguration.maxValidFilterTsfDiff; 
		if((AlphaFilter_isFilterResultValid(pProtectedFramesBER,maxValidTsfDiff) == FALSE) ||
		   (AlphaFilter_isFilterResultValid(pUnprotectedFramesBER,maxValidTsfDiff) == FALSE))

		{
			AlphaFilter_Init(pProtectedFramesBER,pProtectedFramesBERAverageDt);
			AlphaFilter_Init(pUnprotectedFramesBER,pUnProtectedFramesBERAverageDt);
		}
		LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.vapCollisionResetCounter = 0; //Reset vap counter after reset stations through RA_TIMER_AGER_EVENT
	}
	
}




/****************************************************************************
**
** NAME:        protectionAdaptationSetStaPortectionMethod
**
** Parameters:  
**
**
** Returns:     None
**
** Description: Set the protection method to Station DB 
**
****************************************************************************/

static void protectionAdaptationSetStaProtectionMethod(StaId staId, RtsCtsTxMethod_e protectionMethod)
{
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staId]);

	pStaDbHwEntry->common.rtsCtsTxMethod = protectionMethod;
	pStaDbHwEntry->common.ctsToSelfTxMethod =	protectionMethod;
	if(PROTECTION_STA_CTS2SELF_METHOD == LinkAdaptationVapDatabase[StaDbSwEntries[staId].vapIndex].estimatorsVapDataBase.vapCtsProtectionEnable)
	{
		//HW Pre Agg behavior is to select CtsToSelf Protection method only if Rts is not selected.
		//Rts protection is selected if (Q_ID == STA AND force_cts2self_vap[VAP ID] is set) OR according to rtsCtsTxMethod.
		pStaDbHwEntry->common.rtsCtsTxMethod = RTS_CTS_TX_METHOD_NO_RTS_CTS;
	}

}

/****************************************************************************
**
** NAME:        protectionAdaptationSetGroupProtectionMethod
**
** Parameters:  
**
**
** Returns:     None
**
** Description: Set the protection method to Group DB 
**
****************************************************************************/
static void protectionAdaptationSetGroupProtectionMethod(uint8 groupId, RtsCtsTxMethod_e protectionMethod)
{
	MuGrpDb_t* pGroupDbHwEntry = &(GroupDbHwEntries[groupId]);
	
#ifdef ENET_INC_ARCH_WAVE600
	pGroupDbHwEntry->common.rtsCtsTxMethod = protectionMethod; 
	pGroupDbHwEntry->common.ctsToSelfTxMethod = protectionMethod;
#else
	pGroupDbHwEntry->rtsCtsTxMethod = protectionMethod; 
	pGroupDbHwEntry->ctsToSelfTxMethod = protectionMethod;
#endif
}
/***********************************************************************
* linkAdaptationSetRtsThreshold
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void linkAdaptationSetRtsThreshold(K_MSG *psMsg)
{
	UMI_SET_RTS_THRESHOLD *rtsThresholdMsgParams = (UMI_SET_RTS_THRESHOLD*)pK_MSG_DATA(psMsg);
	uint8 firstVapIdInBand = rtsThresholdMsgParams->vapId;
	uint8 lastVapInBand = ConfigurationManager_GetLastVapForMyBand();
	uint8 vapIdCounter;
	AggrDb_t* PreAggConfDbPerVap_p = &PreAggregatorConfigurationDb[firstVapIdInBand];
	AggrDb_t rtsThreshold;

	rtsThresholdMsgParams->Status = UMI_OK;
	//ILOG0_DDDD("[linkAdaptationSetRtsThreshold] - first vap in band: %d, rtsThreshold: %x,isDisabled: %d, operation: %d", rtsThresholdMsgParams->vapId, rtsThresholdMsgParams->newRtsThreshold, rtsThresholdMsgParams->isDisabled, rtsThresholdMsgParams->getSetOperation);
	
	if(API_GET_OPERATION == rtsThresholdMsgParams->getSetOperation)
	{
		memcpy32(&rtsThreshold, PreAggConfDbPerVap_p, sizeof(AggrDb_t) >> 2);
		rtsThresholdMsgParams->newRtsThreshold = rtsThreshold.rtsCtsByteLengthThreshold;
	}
	else //set
	{			
		for(vapIdCounter = firstVapIdInBand; vapIdCounter <= lastVapInBand; vapIdCounter++)
		{	
			PreAggConfDbPerVap_p = &PreAggregatorConfigurationDb[vapIdCounter];
			memcpy32(&rtsThreshold, PreAggConfDbPerVap_p, sizeof(AggrDb_t) >> 2);
			
			if(FALSE == rtsThresholdMsgParams->isDisabled)
			{
				//protection is enabled
				LinkAdaptationVapDatabase[vapIdCounter].estimatorsVapDataBase.vapRtsThresholdPpduSizeEnable = RTS_THRESHOLD_PPDU_SIZE_ON;
				ProtectionAdaptation_EnableProtectionForAllStationsInVap(vapIdCounter);
				rtsThreshold.rtsCtsByteLengthThreshold = rtsThresholdMsgParams->newRtsThreshold;
				rtsThreshold.cts2SelfByteLengthThreshold = rtsThresholdMsgParams->newRtsThreshold;
				if(VAP_PROTECTION_INVALID_STATE != vapProtectionSm[vapIdCounter])
				{
					vapProtectionSm[vapIdCounter] = VAP_PROTECTION_THRESHOLD_STATE;
				}
			}
			else 
			{
				//protection is disabled. please note default state requires recreation of vap
				LinkAdaptationVapDatabase[vapIdCounter].estimatorsVapDataBase.vapRtsThresholdPpduSizeEnable = RTS_THRESHOLD_PPDU_SIZE_OFF;
				ProtectionAdaptation_VapSm(vapIdCounter, VAP_DISABLE_PROTECTION, PRE_AGGREGATOR_FORCE_PROTECTION_MODE_RTS);
			}
			memcpy32(PreAggConfDbPerVap_p, &rtsThreshold, sizeof(AggrDb_t) >> 2);
			//ILOG0_DD("[linkAdaptationSetRtsThreshold] -VAP %d - rtsCtsByteLengthThreshold = %x", vapIdCounter, PreAggregatorConfigurationDb[vapIdCounter].rtsCtsByteLengthThreshold);
		}
	}	
	OSAL_SEND_MESSAGE(UMI_MC_SET_RTS_THRESHOLD_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
}

static void protectionAdaptationResetCollisionsCounters(uint8 vapId)
{
	vapEstimatorsDataBase_t* pVapEstimatorsPtr = &LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase;
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	EstimatorsStaProtection_t* pStaProtectionEstimatorPtr;

	pVapEstimatorsPtr->vapCollisionAckCtsBitmap = 0;
	pVapEstimatorsPtr->vapCollisionTotalRatio = 0;

	if (pVapEstimatorsPtr->vapCollisionResetCounter < MAX_UINT8)
	{
		pVapEstimatorsPtr->vapCollisionResetCounter++;
	}
	else
	{
		pVapEstimatorsPtr->vapCollisionResetCounter = 0;
		while (nextSid != DB_ASYNC_SID)
		{
			if (StaDbSwEntries[nextSid].vapIndex == vapId)
			{
				ASSERT(nextSid<HW_NUM_OF_STATIONS);				
				pStaProtectionEstimatorPtr = &(LinkAdaptationStaDatabase[nextSid].laStaGroupCommon.staProtection);
				pStaProtectionEstimatorPtr->staCollisionCount = 0;
				pStaProtectionEstimatorPtr->staCollisionAckCtsBitmap = 0;
				pStaProtectionEstimatorPtr->staCollisionResetCounter = 0;
				ILOG0_DD("protectionAdaptationResetCollisionsCounters, vapCollisionResetCounter = %d, staCollisionResetCounter = %d", pVapEstimatorsPtr->vapCollisionResetCounter, pStaProtectionEstimatorPtr->staCollisionResetCounter);
			}
			nextSid = StaDbSwEntries[nextSid].nextSid;
		}
	}

}
void protectionAdaptationAgingReset(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8 vapId;

	vapId = StaDbSwEntries[laDbDistributionParameter->stationOrGroupIndex].vapIndex;
	ILOG0_DD("protectionAdaptationAgingReset, staCollisionResetCounter = %d, vapCollisionResetCounter = %d", laDbDistributionParameter->laStaGroupCommon->staProtection.staCollisionResetCounter, LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.vapCollisionResetCounter);
	if (laDbDistributionParameter->laStaGroupCommon->staProtection.staCollisionResetCounter != LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.vapCollisionResetCounter)
	{
		laDbDistributionParameter->laStaGroupCommon->staProtection.staCollisionCount = 0;
		laDbDistributionParameter->laStaGroupCommon->staProtection.staCollisionAckCtsBitmap = 0;
		laDbDistributionParameter->laStaGroupCommon->staProtection.staCollisionResetCounter = 0;
	}

}
void ProtectionAdaptation_DisableProtectionForAllStationsInVap(uint8 vapId)
{
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	
	while (nextSid != DB_ASYNC_SID)
	{
		if (StaDbSwEntries[nextSid].vapIndex == vapId)
		{
			staProtectionSm[nextSid] = STA_PROTECTION_DISABLE_STATE;
			/*configure DB*/
			protectionAdaptationSetStaProtectionMethod(nextSid, RTS_CTS_TX_METHOD_NO_RTS_CTS);
		}
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
}
void ProtectionAdaptation_EnableProtectionForAllStationsInVap(uint8 vapId)
{
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;
	
	while (nextSid != DB_ASYNC_SID)
	{
		if (StaDbSwEntries[nextSid].vapIndex == vapId)
		{
			protectionAdaptationSetStaProtectionMethod(nextSid, RTS_CTS_TX_METHOD_RTS_CTS_BASED_THRESHOLD);
		}
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
}

void ProtectionAdaptation_ResetProtectiontoAllStationsInVap(uint8 vapId)
{
	StaId nextSid = LmStaDataBase.headIndexOfStaLinkList;

	while (nextSid != DB_ASYNC_SID)
	{
		if (StaDbSwEntries[nextSid].vapIndex == vapId)
		{
			staProtectionSm[nextSid] = STA_PROTECTION_NONE_STATE;
			ProtectionAdaptation_ResetStaEstimators(nextSid);
		}
		nextSid = StaDbSwEntries[nextSid].nextSid;
	}
}
void ProtectionAdaptation_ResetVapEstimators(uint8 vapId)
{
	vapEstimatorsDataBase_t* pVapEstimatorsPtr = &(LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase);

	pVapEstimatorsPtr->vapCollisionAckCtsBitmap = 0;
	pVapEstimatorsPtr->vapCollisionTotalRatio = 0;
	pVapEstimatorsPtr->vapThresholdModeCounter = 0;
	pVapEstimatorsPtr->vapCollisionResetCounter = 0;
	AlphaFilter_Init(&pVapEstimatorsPtr->protectedFramesBER,&pVapEstimatorsPtr->protectedFramesBERDeltaTAverage);
	AlphaFilter_Init(&pVapEstimatorsPtr->unprotectedFramesBER,&pVapEstimatorsPtr->unprotectedFramesBERDeltaTAverage);
}
void ProtectionAdaptation_ResetStaEstimators(StaId staIndex)
{
	EstimatorsStaProtection_t* pStaProtectionEstimatorPtr = NULL;
	
	pStaProtectionEstimatorPtr = &(LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.staProtection);
	/*configure DB*/
	pStaProtectionEstimatorPtr->staForceProtectionIterationsCount = 0; 
	/* Start from zero - as if the station has just connected*/
	pStaProtectionEstimatorPtr->staCollisionCount = 0;
	pStaProtectionEstimatorPtr->staCollisionAckCtsBitmap = 0;
	pStaProtectionEstimatorPtr->staCollisionResetCounter = 0;
}

vapProtectionStates_e protectionAdaptationGetVapState(uint8 vapId)
{
	return vapProtectionSm[vapId];
}

void ProtectionAdaptation_CtsEnable(uint8 vapId, ProtectionStaDbTxMethod_e ProtectionType)
{
	LinkAdaptationVapDatabase[vapId].estimatorsVapDataBase.vapCtsProtectionEnable = (uint8)ProtectionType;
	ProtectionAdaptation_EnableProtectionForAllStationsInVap(vapId);
}



