/***********************************************************************************
 File:			QAMplus.c
 Module:		LinkAdaptation 
 Purpose:	    Allow LinkAdaptation to work in QAMplus when possible
 Description:   This file contains all definitions and the structures of the QAMplus
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "QAMplus.h" 
#include "shram.h"
#include "Utils_Api.h"
#include "linkAdaptation_api.h"
#include "RateAdaptation.h"
#include "CommonRamLinkAdaptation.h"
#include "Statistics_Descriptors.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 5

#define QAM_PLUS_MAX_POSSIBLE_RATE_MASK (3)
#define QAM_PLUS_TRIAL_THRESH			(16)
#define QAM_PLUS_INACTIVE_TRIAL_THRESH	(255)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum QAMplusCodeRateMode
{
	QAM_PLUS_MCS8_256QAM_ENABLE,
	QAM_PLUS_MCS89_256QAM_ENABLE,
	QAM_PLUS_256QAM_DISABLE,
	QAM_PLUS_CODERATE_NOT_VALID
}QAMplusCodeRateMode_e;

typedef enum QAMplusTrialCounterVal
{
	QAM_PLUS_TRIAL_COUNTER_VAL_LARGER_0,
	QAM_PLUS_TRIAL_COUNTER_VAL_LARGER_2 = 2,
	QAM_PLUS_TRIAL_COUNTER_VAL_LARGER_5 = 5,
	QAM_PLUS_TRIAL_COUNTER_VAL_NOT_VALID
}QAMplusTrialCounterVal_e;

typedef enum QAMplusSuccessThreshVal
{
	QAM_PLUS_SUCCESS_THRESH_VAL_2 = 2,
	QAM_PLUS_SUCCESS_THRESH_VAL_10 = 10,
	QAM_PLUS_SUCCESS_THRESH_VAL_20 = 20,
	QAM_PLUS_SUCCESS_THRESH_NOT_VALID
}QAMplusSuccessThreshVal_e;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void QAMplus_SupportedState(BaaRateAdaptiveFullFifo_t* pBaaReport, StaId stationId, bool isWorkingPoint, uint8 highestRateIndex, uint8 rateIndexFromCurrentBA);
static void QAMplus_TrialState(BaaRateAdaptiveFullFifo_t* pBaaReport, StaId stationId, uint8 rateIndexFromCurrentBA);
static void QAMplus_SetSupportedThresh(StaId stationId);
static void QAMplus_UpdateLaRateMask(StaId stationId, QAMplusCodeRateMode_e codeRateMode);
static void QAMplus_ResetCounters(StaId stationId);
static void QAMplus_ResetParams(StaId stationId);

//DEBUG
#ifdef QAM_PLUS_DEBUG	
static void QAMplus_GetCounters(StaId stationId); 
#endif

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
extern LinkAdaptationStaDatabase_t 	LinkAdaptationStaDatabase[HW_NUM_OF_STATIONS];
extern CurrentPacketParameters_t	CurrentPacketParams;
extern wholePhyRxStatusDb_t 		phyRxStatusDb;

QAMplusConfigurationParams_t 		QAMplusConfigParams;
bool								QAMplusEnable;
bool								QAMplusEnableFromDriver;


//32 LSB, 32MSB - only 256QAM rates 
//256QAM only code-rate 3/4 enabled 
//256QAM both code-rate 3/4+5/6 enabled
//256QAM both code-rate 3/4+5/6 disabled
RateMask_t 							AllowedQAMplusRatesMask[QAM_PLUS_MAX_POSSIBLE_RATE_MASK] = 
										{
											{0x00800000, 0x00200802},
											{0x01800000, 0x00601806},
											{0xFE7FFFFF, 0xFF9FE7E9},
										};

/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

QAMplus_ResetCounters 


Description: 
------------
			Initalize counters for a given station

Input: 
-----
			station Index

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_ResetCounters(StaId stationId)
{	
	QAMplusStaDataBase_t* pQAMplusDatabase = &(LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb);

	pQAMplusDatabase->failureCounter = 0;
	pQAMplusDatabase->numberTrialsCounter = 0;
	pQAMplusDatabase->successHighestRateWPCounter = 0;
	pQAMplusDatabase->trialProbingCounter = 0;
	
#ifdef QAM_PLUS_DEBUG
	ILOG0_V("[QAMplus_ResetCounters]"); 
#endif

}

/**********************************************************************************

QAMplus_ResetParams 


Description: 
------------
			Initalize parameters for a given station

Input: 
-----
			station Index

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_ResetParams(StaId stationId)
{	
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;

	// reset all counters
	QAMplus_ResetCounters(stationId);

	// reset parameters to default values
	pQAMplusDatabase->lastRSSIdB = QAM_PLUS_MIN_RSSI_DB;
	pQAMplusDatabase->laSteadyStateEnable = FALSE;

	//set thresholds of counters
	QAMplus_SetThresh(QAM_PLUS_TRIAL_THRESH, QAM_PLUS_INACTIVE_TRIAL_THRESH);
	//update supportedThresh 
	QAMplus_SetSupportedThresh(stationId);		
	
#ifdef QAM_PLUS_DEBUG
	ILOG0_V("[QAMplus_ResetParams]");	
#endif

}

/**********************************************************************************

QAMplus_SupportedState 


Description: 
------------
			Handle operations when QAMplus is in SUPPORTED state

Input: 
-----
			phyMode
			rateIndexFromCurrentBA

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_SupportedState(BaaRateAdaptiveFullFifo_t* pBaaReport, StaId stationId, bool isWorkingPoint, uint8 highestRateIndex, uint8 rateIndexFromCurrentBA)
{
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter;
	QAMplusConfigurationParams_t* pQAMplusConfigParams = &QAMplusConfigParams;
	const RateObj_t* ratesTable;

	updateLaDbDistributionParam(&laDbDistributionParameter,stationId,INVALID_MU_USP_INDEX, FALSE);	
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);
	// Process only BA reports that relate to WP with highest possible HT mode			
	if ( isWorkingPoint && 
		 (ratesTable[rateIndexFromCurrentBA].laPhyMode == LA_PHY_MODE_HT_VHT) && 
		 (ratesTable[rateIndexFromCurrentBA].vhtHeMcs == ratesTable[highestRateIndex].vhtHeMcs) ) 
	{
		// increase successHighestRateWPCounter counter and clip in case of overlap
		pQAMplusDatabase->successHighestRateWPCounter = MIN(pQAMplusDatabase->supportedThresh, (pQAMplusDatabase->successHighestRateWPCounter + pBaaReport->userReport.successMpduCnt) );
	}

	// LA is in steady state mode
	if (pQAMplusDatabase->laSteadyStateEnable == TRUE)
	{
#ifdef	QAM_PLUS_DEBUG
		ILOG0_V("[QAMplus_SupportedState], LA in steadyState");
#endif
		// successHighestRateWPCounter is larger than supportedThresh 
		if (pQAMplusDatabase->successHighestRateWPCounter >= pQAMplusDatabase->supportedThresh)
		{
			// transit to TRIAL STATE 
			QAMplus_SetState(stationId,QAM_PLUS_STATE_TRIAL);

			// reset failureCounter 
			pQAMplusDatabase->failureCounter = 0;

			// reset probingCounter
			pQAMplusDatabase->trialProbingCounter = 0;

			// increase numberTrialsCounter counter and clip in case of overlap
			pQAMplusDatabase->numberTrialsCounter = MIN( pQAMplusConfigParams->inactiveTrialThresh, (pQAMplusDatabase->numberTrialsCounter + 1) );

			// update rate mask of LA to enable lowest code rate of 256QAM rates				
			QAMplus_UpdateLaRateMask(stationId, QAM_PLUS_MCS8_256QAM_ENABLE);

			//synchronize rate mask change with LA module: value 0 since it is not relevant in this case
			updateOverridePendingDb(ADJUST_RATE_TO_MASK, 0, &laDbDistributionParameter);
		}
	}


}

/**********************************************************************************

QAMplus_TrialState 


Description: 
------------
			Handle operations when QAMplus is in TRIAL state

Input: 
-----
			phyMode
			rateIndexFromCurrentBA

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_TrialState(BaaRateAdaptiveFullFifo_t* pBaaReport, StaId stationId, uint8 rateIndexFromCurrentBA)
{
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;
	LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigParams = LinkAdaptationStaDatabase[stationId].laStaGroupCommon.pLinkAdaptationConfigurationParams;
	QAMplusConfigurationParams_t* pQAMplusConfigParams = &QAMplusConfigParams;
	int8 maxCurrentRssidB = QAM_PLUS_MIN_RSSI_DB;
	const RateObj_t* ratesTable;
	uint8 currentRateIndex;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter;
	PhyMode_e phyMode;

#ifdef ENET_INC_ARCH_WAVE600D2	
	phyMode = (PhyMode_e) pBaaReport->commonReport.txPhyMode;
#else
	phyMode = (PhyMode_e) pBaaReport->commonReport.phyMode;
#endif
	updateLaDbDistributionParam(&laDbDistributionParameter,stationId,INVALID_MU_USP_INDEX, FALSE);	
	ratesTable = getRatesTable(laDbDistributionParameter.laStaUspCommon);
	// in trial mode process only BA reports of VHT stations with MCS == 8/9 that relate to 256QAM (in VHT)
	if ( (phyMode == PHY_MODE_11AC) && ( (ratesTable[rateIndexFromCurrentBA].vhtHeMcs == MCS8_256QAM_3_4) || (ratesTable[rateIndexFromCurrentBA].vhtHeMcs == MCS9_256QAM_5_6) ) )
	{
#ifdef QAM_PLUS_DEBUG
		ILOG0_DD("[QAMplus_TrialState] pBaaReport->successMpdu = %d, pBaaReport->mpduCnt = %d", pBaaReport->userReport.successMpduCnt, pBaaReport->userReport.successMpduCnt+pBaaReport->userReport.failedMpduCnt);
#endif
		// if at least one success TX 
		if (pBaaReport->userReport.successMpduCnt > 0)
		{
			// transit to ACTIVE state and
			QAMplus_SetState(stationId, QAM_PLUS_STATE_ACTIVE);

//#ifdef QAM_PLUS_ALWAYS
//			QAMplus_ChangeTCRparamsToVHT(stationId);
//#endif
			// update LA to enable all 256QAM rates
			QAMplus_UpdateLaRateMask(stationId, QAM_PLUS_MCS89_256QAM_ENABLE);

			currentRateIndex = GetStaWpRateIndexFromHwTcr(stationId, GetDataBwLimit(stationId,INVALID_MU_USP_INDEX, FALSE));

			// make sure LA is not already in WP 256QAM
			ASSERT((ratesTable[currentRateIndex].vhtHeMcs != MCS8_256QAM_3_4)&&(ratesTable[currentRateIndex].vhtHeMcs != MCS9_256QAM_5_6));
			
			//synchronize rate mask change with LA module: value 0 since it is not relevant in this case
            updateOverridePendingDb(OVERRIDE_VHT_MPDU, 0, &laDbDistributionParameter);	// ADJUST_RATE_TO_MASK also called in OVERRIDE_VHT_MPDU
		}
		else //if no success, then updated failureCounter accordingly
		{
			// increase failure counter and clip in case of overlap
			pQAMplusDatabase->failureCounter  =  MIN(pQAMplusConfigParams->trialThresh, (pQAMplusDatabase->failureCounter + pBaaReport->userReport.successMpduCnt + pBaaReport->userReport.failedMpduCnt));
			pQAMplusDatabase->trialProbingCounter++;
		}
	}

	
	// failureCounter is larger than threshold or probing counter equal to LA probing validation counter
	if ((pQAMplusDatabase->failureCounter >= pQAMplusConfigParams->trialThresh) || 
		(pQAMplusDatabase->trialProbingCounter >= (pLinkAdaptationConfigParams->fastProbingValidationTh-1)))
	{
		// trialcounter is smaller than inactive trial threshold
		if (pQAMplusDatabase->numberTrialsCounter < pQAMplusConfigParams->inactiveTrialThresh)
		{
			// transit to SUPPORTED state 
			QAMplus_SetState(stationId,QAM_PLUS_STATE_SUPPORTED);

			// reset successHighestRateWPCounter
			pQAMplusDatabase->successHighestRateWPCounter = 0;	

			//update supportedThresh 
			QAMplus_SetSupportedThresh(stationId);
		}
		else //// trialcounter excceeds inactiveTrialThresh
		{
			// transit to INACTIVE state 
			QAMplus_SetState(stationId,QAM_PLUS_STATE_INACTIVE);

			// reset counters
			QAMplus_ResetCounters(stationId);

			// update lastRSSI with current max value
			linkAdaptationFindMaxRssi(stationId, &maxCurrentRssidB);

			pQAMplusDatabase->lastRSSIdB = maxCurrentRssidB;	

#ifdef QAM_PLUS_DEBUG
			ILOG0_D("[QAMplus_TrialState] pQAMplusDatabase->lastRSSIdB = %d", pQAMplusDatabase->lastRSSIdB);
#endif
		}
	
		// update LA mask to disable all 256QAM rates 
		QAMplus_UpdateLaRateMask(stationId, QAM_PLUS_256QAM_DISABLE);

		//synchronize rate mask change with LA module: value 0 since it is not relevant in this case
		updateOverridePendingDb(ADJUST_RATE_TO_MASK, 0, &laDbDistributionParameter);
	}

}

/**********************************************************************************

QAMplus_NewBAReport 


Description: 
------------
			Process new BA report to determine next operation in QAMplus SM
			NOTE: this function must be called only after function estimatorsProcessBaaReport!!!!

Input: 
-----
			Station Index
			Pointer to BA report

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_NewBAReport(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, BaaRateAdaptiveFullFifo_t* pBaaReport)
{
	QAMplusStaDataBase_t* pQAMplusDatabase;
	QAMplusState_e QAMplusStateLocal;
	bool isWorkingPoint;
	uint8 highestRateIndex;
	uint8 rateIndexFromCurrentBA;
	StaId stationId;
#ifdef QAM_PLUS_DEBUG
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
#endif
	ASSERT(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX); /*Must not happen due to the QUAM+ intenal SM validation*/
	stationId = laDbDistributionParameter->stationOrGroupIndex;
	pQAMplusDatabase = &(LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb);
	QAMplusStateLocal =  pQAMplusDatabase->QAMplusState;
	if (QAMplusStateLocal != QAM_PLUS_STATE_NOT_ALLOWED)
	{
		isWorkingPoint = (!(pBaaReport->userReport.altRateReasonFastProbing));

#ifdef QAM_PLUS_DEBUG
	ILOG0_V("[QAMplus_NewBAReport] start");
	SLOG0(0, 0, RateMask_t, &(laDbDistributionParameter->laStaUspCommon->raIndexMask));
#endif
		highestRateIndex = getHighestRateinMask(laDbDistributionParameter->stationOrGroupIndex,(laDbDistributionParameter->laStaUspCommon->raIndexMask), GetDataBwLimit(stationId,INVALID_MU_USP_INDEX, FALSE) ,0, MAX_VHT_SORT_RATE_INDEX);

		//rate index to ratesTable.vhtHeMcs calculated in convertTcr2RateIndex()
		rateIndexFromCurrentBA = CurrentPacketParams.rateIndex;

		// Process only BA reports for stations which are configured in QAMplus DB as SUPPORTED, INACTIVE or TRIAL states
		switch(QAMplusStateLocal)
		{
			case QAM_PLUS_STATE_INACTIVE:

				// current RSSI is larger then last RSSI 
				if (linkAdaptationCompareRSSI(laDbDistributionParameter->stationOrGroupIndex, pQAMplusDatabase->lastRSSIdB,DELTA_RSSI_DB))
				{
					// change state to SUPPORTED
					QAMplus_SetState(stationId, QAM_PLUS_STATE_SUPPORTED);				

					//update supportedThresh 
					QAMplus_SetSupportedThresh(stationId);
				}

				break;

			case QAM_PLUS_STATE_SUPPORTED:

#ifdef QAM_PLUS_DEBUG
				ILOG0_DD("[QAMplus_NewBAReport] start, isWorkingPoint = %d, highestRateIndex = %d", isWorkingPoint, highestRateIndex);
				ILOG0_D("[QAMplus_NewBAReport] start, rateIndexFromCurrentBA = %d", rateIndexFromCurrentBA);
				ILOG0_D("[QAMplus_NewBAReport] QAM_PLUS_STATE_SUPPORTED start, pQAMplusDatabase->successHighestRateWPCounter = %d", pQAMplusDatabase->successHighestRateWPCounter);
				ILOG0_D("[QAMplus_NewBAReport] QAM_PLUS_STATE_SUPPORTED start, pBaaReport->successMpdu = %d", pBaaReport->userReport.successMpduCnt);
#endif
				QAMplus_SupportedState(pBaaReport, stationId, isWorkingPoint, highestRateIndex, rateIndexFromCurrentBA);
				
#ifdef QAM_PLUS_DEBUG
				ILOG0_D("[QAMplus_NewBAReport] QAM_PLUS_STATE_SUPPORTED end, pQAMplusDatabase->successHighestRateWPCounter = %d", pQAMplusDatabase->successHighestRateWPCounter);
#endif			
		
				break; 
				
			case QAM_PLUS_STATE_TRIAL:

#ifdef QAM_PLUS_DEBUG
				ILOG0_DDD("[QAMplus_NewBAReport] QAM_PLUS_STATE_TRIAL start, pBaaReport->phyMode = %d, rateIndexFromCurrentBA = %d, RatesTable[rateIndexFromCurrentBA].vhtHeMcs = %d", pBaaReport->commonReport.phyMode, rateIndexFromCurrentBA, ratesTable[rateIndexFromCurrentBA].vhtHeMcs);
#endif
				QAMplus_TrialState(pBaaReport, stationId, rateIndexFromCurrentBA);		
				
				break;

			default:

				// do nothing
				break;	
		}
		
#ifdef QAM_PLUS_DEBUG
		QAMplus_GetCounters(stationId);
		ILOG0_V("[QAMplus_NewBAReport] end");
#endif	
	}
	
}

/**********************************************************************************

QAMplus_AddStationReq 


Description: 
------------
			This function is called from LA context when new HT station connects
			It moves QAMplus from INACTIVE STATE to SUPPORTED STATE

Input: 
-----
			station ID

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_AddStationReq(StaId stationId, PhyMode_e staPhyMode)
{
	
#ifdef QAM_PLUS_DEBUG
	ILOG0_DD("[QAMplus_AddStationReq]  QAMplusEnableFromDriver = %d, QAMplusEnable = %d", QAMplusEnableFromDriver, QAMplusEnable);
#endif	

	// init state to NOT_ALLOWED state for the given station before adding station
	QAMplus_SetState(stationId, QAM_PLUS_STATE_NOT_ALLOWED);

	if (QAMplusEnableFromDriver)
	{
#ifdef QAM_PLUS_DEBUG
		ILOG0_D("[QAMplus_AddStationReq]  QAMplus_GetState = %d", QAMplus_GetState(stationId));
#endif	
		//QAMplus will be activated and set to SUPPORTED mode if platform is 2.4GHz and station added is in HT mode
		if ((QAMplusEnable) && (staPhyMode == PHY_MODE_11N))
		{
			// change state to SUPPORTED state for the given station (since it is in HT mode)
			QAMplus_SetState(stationId, QAM_PLUS_STATE_SUPPORTED);
			
			// reset parameters
			QAMplus_ResetParams(stationId);	
				
			ILOG0_D("[QAMplus_AddStationReq] stationId = %d", stationId);
		}
	}
}



/**********************************************************************************

QAMplus_SetState 


Description: 
------------
			Change state in Turbo QAM state machine

Input: 
-----
			staId - station Id
			newQAMplusState - new Turbo QAM state

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_SetState(StaId stationId , QAMplusState_e newQAMplusState)
{
	LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb.QAMplusState = newQAMplusState;
#ifdef QAM_PLUS_DEBUG
	ILOG0_D("[QAMplus_SetState] QAMplus_GetState = %d", QAMplus_GetState(stationId));
#endif
}

/**********************************************************************************

QAMplus_GetState 


Description: 
------------
			Extract state of Turbo QAM state machine

Input: 
-----
			staId - station Id
			currentQAMplusState - new Turbo QAM state

Returns:
--------
			None

	
**********************************************************************************/
QAMplusState_e QAMplus_GetState(StaId stationId)
{
	QAMplusState_e currentQAMplusState;
	
	currentQAMplusState = LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb.QAMplusState;
	
	return(currentQAMplusState);
}

/**********************************************************************************

QAMplus_SetSupportedThresh 


Description: 
------------
			Set supportedThresh according to numberTrialsCounter

Input: 
-----
			None


Returns:
--------
			indexThresh

	
**********************************************************************************/
void QAMplus_SetSupportedThresh(StaId stationId)
{
	// TODO: currently only 4 steps out of 6 are definded, need to calibrate in the future
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;
	
	pQAMplusDatabase->supportedThresh = QAM_PLUS_SUCCESS_THRESH_VAL_2;

	if(pQAMplusDatabase->numberTrialsCounter > QAM_PLUS_TRIAL_COUNTER_VAL_LARGER_5)	
	{
		pQAMplusDatabase->supportedThresh = QAM_PLUS_SUCCESS_THRESH_VAL_20;
	}
	else if (pQAMplusDatabase->numberTrialsCounter > QAM_PLUS_TRIAL_COUNTER_VAL_LARGER_2)	
	{
		pQAMplusDatabase->supportedThresh = QAM_PLUS_SUCCESS_THRESH_VAL_20;
	}
	else if (pQAMplusDatabase->numberTrialsCounter > QAM_PLUS_TRIAL_COUNTER_VAL_LARGER_0)	
	{
		pQAMplusDatabase->supportedThresh = QAM_PLUS_SUCCESS_THRESH_VAL_10;
	}

#ifdef QAM_PLUS_DEBUG
	ILOG0_D("[QAMplus_SetSupportedThresh], pQAMplusDatabase->supportedThresh = %d", pQAMplusDatabase->supportedThresh);	
#endif
}

/**********************************************************************************

QAMplus_UpdateLaRateMask 


Description: 
------------
			Update LA rate mask to enable/disable 256QAM rates
			

Input: 
-----
			stationId
			codeRateMode
			

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_UpdateLaRateMask(StaId stationId, QAMplusCodeRateMode_e codeRateMode)
{
	uint32* pRaIndexMask64bit = LinkAdaptationStaDatabase[stationId].laStaUspCommon.raIndexMask.raIndexMask64bit;
	uint32* pRaIndexMaskOrg64bit = LinkAdaptationStaDatabase[stationId].laStaUspCommon.raIndexMaskOrg.raIndexMask64bit;
	uint8 	maxNss;                                                                                          
	uint8 	maxNssOrg; 
	uint8   currentRateIndex;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[stationId].laStaUspCommon);
	
#ifdef QAM_PLUS_DEBUG
	ILOG0_V("[QAMplus_UpdateLaRateMask] before update");
	SLOG0(0, 0, RateMask_t, &(LinkAdaptationStaDatabase[stationId].laStaUspCommon.raIndexMask));
	SLOG0(0, 0, RateMask_t, &LinkAdaptationStaDatabase[stationId].laStaUspCommon.raIndexMaskOrg);
#endif

	if (codeRateMode == QAM_PLUS_256QAM_DISABLE)
	{		
		currentRateIndex = GetStaWpRateIndexFromHwTcr(stationId, GetDataBwLimit(stationId,INVALID_MU_USP_INDEX, FALSE));

		// when deciding to move to SUPPORTED state or INACTIVE state make sure LA is not already in WP 256QAM
		ASSERT((ratesTable[currentRateIndex].vhtHeMcs != MCS8_256QAM_3_4)&&(ratesTable[currentRateIndex].vhtHeMcs != MCS9_256QAM_5_6));

		//disable both code rate 3/4 and 5/6 in 256QAM
		andOperator64Bit(pRaIndexMask64bit, AllowedQAMplusRatesMask[codeRateMode].raIndexMask64bit, pRaIndexMask64bit);
		andOperator64Bit(pRaIndexMaskOrg64bit, AllowedQAMplusRatesMask[codeRateMode].raIndexMask64bit, pRaIndexMaskOrg64bit);
	}
	else
	{
		// find maximum NSS from current bit mask
		maxNss = LinkAdaptationGetMaxNssInMask(pRaIndexMask64bit, LinkAdaptationStaDatabase[stationId].laStaUspCommon.staTransmissionParams.heSta);
		// find maximum NSS from current bit mask
		maxNssOrg = LinkAdaptationGetMaxNssInMask(pRaIndexMaskOrg64bit, LinkAdaptationStaDatabase[stationId].laStaUspCommon.staTransmissionParams.heSta);

#ifdef QAM_PLUS_DEBUG
		ILOG0_D("[QAMplus_UpdateLaRateMask], maxNss according to raIndexMask= %d", maxNss);
		ILOG0_D("[QAMplus_UpdateLaRateMask], maxNss according to raIndexMaskOrg= %d", maxNssOrg);
		SLOG0(0, 0, RateMask_t, &AllowedQAMplusRatesMask[codeRateMode].raIndexMask64bit);
#endif
	
		ASSERT(maxNss < VHT_HE_NUMBER_OF_NSS);
		ASSERT(maxNssOrg < VHT_HE_NUMBER_OF_NSS);

		//enable either only rate 3/4 or both code rate 3/4 and 5/6 in 256QAM, according to codeRateMode input
		orOperator64Bit(pRaIndexMask64bit, AllowedQAMplusRatesMask[codeRateMode].raIndexMask64bit, pRaIndexMask64bit);
		orOperator64Bit(pRaIndexMaskOrg64bit, AllowedQAMplusRatesMask[codeRateMode].raIndexMask64bit, pRaIndexMaskOrg64bit);

		
		// update AllowedQAMplusRatesMask according to maximum NSS
		andOperator64Bit(pRaIndexMask64bit, disableHtVhtNssMaskAccordingtoAntNum[maxNss].raIndexMask64bit, pRaIndexMask64bit);
		andOperator64Bit(pRaIndexMaskOrg64bit, disableHtVhtNssMaskAccordingtoAntNum[maxNssOrg].raIndexMask64bit, pRaIndexMaskOrg64bit);
					
	}
		
#ifdef QAM_PLUS_DEBUG
	ILOG0_V("[QAMplus_UpdateLaRateMask] after update");
	SLOG0(0, 0, RateMask_t, &(LinkAdaptationStaDatabase[stationId].laStaUspCommon.raIndexMask));
	SLOG0(0, 0, RateMask_t, &LinkAdaptationStaDatabase[stationId].laStaUspCommon.raIndexMaskOrg);
#endif

}

/**********************************************************************************

QAMplus_SetThresh 


Description: 
------------
			Function that sets QAMplus's thresholds
			Used also for DEBUG as CLI function
			

Input: 
-----
		counterVal1
		counterVal2

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_SetThresh(uint16 counterVal1, uint16 counterVal2)
{
	QAMplusConfigurationParams_t* pQAMplusConfigParams = &QAMplusConfigParams;

	pQAMplusConfigParams->trialThresh = counterVal1;
	pQAMplusConfigParams->inactiveTrialThresh = counterVal2;
	
#ifdef QAM_PLUS_DEBUG	
	ILOG0_V("[QAMplus_SetThresh]");
#endif
}

/**********************************************************************************

QAMplus_Enable

Description: 
------------
			Function that Enable/Dsiable QAMplus
			Used also for DEBUG as CLI function
			

Input: 
-----
			enableFlag


Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_Enable(bool enableFlag)
{
	QAMplusEnableFromDriver = enableFlag;

	ILOG0_D("[QAMplus_Enable], QAMplusEnableFromDriver = %d", QAMplusEnableFromDriver); 
}
/**********************************************************************************

QAMplus_get

Description: 
------------
			returns QAM plus state
			

Input: 
-----
			None


Returns:
--------
			QAM plus state

	
**********************************************************************************/
bool QAMplus_get(void)
{
	if(LinkAdaptationCommonConfiguration.band == BAND_2_4_GHZ)
	{
		return QAMplusEnableFromDriver;
	}
	else
	{
		return QAMplusEnable;
	}
}
/**********************************************************************************

QAMplus_SetIndexThresh 


Description: 
------------
			Function that sets QAMplus's index threshold for supportedThresh
			Used also for DEBUG as CLI function
			

Input: 
-----
		counterVal1
		counterVal2

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_DebugSetSupportedThresh(StaId stationId, uint16 supportedThreshVal)
{
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;

	pQAMplusDatabase->supportedThresh = supportedThreshVal;
	
#ifdef QAM_PLUS_DEBUG
	ILOG0_V("[QAMplus_SetSupportedThresh]");
#endif
}

/**********************************************************************************

QAMplus_GetThresh 


Description: 
------------
			Function that sets QAMplus's thresholds
			Used also for DEBUG as CLI function
			

Input: 
-----
		counterVal1
		counterVal2

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_GetThresh(StaId stationId)
{
#ifdef QAM_PLUS_DEBUG	
	QAMplusConfigurationParams_t* pQAMplusConfigParams = &QAMplusConfigParams;
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;

	ILOG0_DDD("[QAMplus_GetThresh], pQAMplusConfigParams->trialThresh=%d, pQAMplusConfigParams->inactiveTrialThresh = %d, pQAMplusDatabase->supportedThresh = %d", pQAMplusConfigParams->trialThresh, pQAMplusConfigParams->inactiveTrialThresh, pQAMplusDatabase->supportedThresh);
#else
	UNUSED_PARAM(stationId);
#endif
}


#ifdef QAM_PLUS_DEBUG	
/**********************************************************************************

QAMplus_GetCounters 


Description: 
------------
			DEBUG
			

Input: 
-----
			

Returns:
--------
			None

	
**********************************************************************************/
void QAMplus_GetCounters(StaId stationId)
{
	QAMplusStaDataBase_t* pQAMplusDatabase = &LinkAdaptationStaDatabase[stationId].laStationUnique.QAMplusStaDb;
	UNUSED_PARAM(pQAMplusDatabase);
	ILOG0_DDDD("[QAMplus_GetCounters] stationId = %d: failureCounter = %d, numberTrialsCounter = %d, successHighestRateWPCounter = %d", stationId, pQAMplusDatabase->failureCounter, pQAMplusDatabase->numberTrialsCounter, pQAMplusDatabase->successHighestRateWPCounter);
}
#endif

/**********************************************************************************

QAMplus_ResetCounters 


Description: 
------------
			DEBUG
			

Input: 
-----
			

Returns:
--------
			None

	
**********************************************************************************/
//void QAMplus_ResetCounters(StaId stationId, uint8 counterType)
//{
//}

/**********************************************************************************

QAMplus_SetThreshCounters 


Description: 
------------
			DEBUG
			

Input: 
-----
			

Returns:
--------
			None

	
**********************************************************************************/
//void QAMplus_SetThreshCounters(StaId stationId, int16 newThreshVal, uint8 counterType)
//{
//}









