/***********************************************************************************
 File:		RateAdaptation.c
 Module:		LinkAdaptation 
 Purpose: 	find the best transmission parameters for achieve best rate
 Description:	This file contains all definitions and the structures of the RateAdaptation
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "RateAdaptation.h"
#include "Pac_Api.h"
#include "stringLibApi.h"
#include "Estimators.h"
#include "mt_sysdefs.h"
#include "StaDatabase_Api.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "lm_StaDatabase.h"
#include "lm.h"
#include "CyclicPrefix.h"
#include "BwAdaptation.h"
#include "Beamforming.h"
#include "ShramStatistics.h"
#include "LinkAdaptation_StateMachine.h"
#include "PowerAdaptation.h"
#include "CDD.h"
#include "AntennaSelection.h"
#include "QAMplus.h"
#include "DynamicTxOP.h"
#include "ConfigurationManager_api.h"
#include "GroupManager_API.h"
#include "GroupDb.h"
#include "LinkAdaptationPhyDriver.h"
#include "ShramGroupDatabase.h"
#include "ProtectedDbLock_Api.h"
#include "CommonRamLinkAdaptation.h"
#include "lm_VapDatabase.h"
#include "TurboRates.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 6

/*Fast probing dfinition*/
#define NUMBER_OF_PROBE_UP_OPTIONS 4
#define	NUMBER_OF_PROBE_DOWN_OPTIONS 2
#define NUMBER_OF_PROBE_RR_OPTIONS 4

/*Slow probing definitions*/
#define SLOW_PROBING_ENABLE_ALL_LOOPS_MASK 0xF
#define NUMBER_OF_SLOW_PROBING_POWER_OPTIONS 2
#define NUMBER_OF_SLOW_PROBING_ANTENNA_SELECTION_OPTIONS 2
#define NUMBER_OF_SLOW_PROBING_BW_OPTIONS 2
#define NUMBER_OF_SLOW_PROBING_CHANNEL_OPTIONS 2

#define RATE_TO_TCR_OFFSET_11B 8
#define FAST_DROP_RATES_DECREASE 1

#define	GET_SIX_LS_BITS_MU(val) 					(val & 0x3f) 

#ifdef ENET_INC_ARCH_WAVE600
#define SLOW_PROBING_PRIORITY0_ENABLED_BITMAP 	((1<<SLOW_PROBING_POWER) | \
												(1<<SLOW_PROBING_BW) | \
												(1<<SLOW_PROBING_CP) | \
												(1<<SLOW_PROBING_BF))
#else
#define SLOW_PROBING_PRIORITY0_ENABLED_BITMAP 	((1<<SLOW_PROBING_POWER) | \
												(1<<SLOW_PROBING_BW) | \
												(1<<SLOW_PROBING_CP) | \
												(1<<SLOW_PROBING_BF) | \
												(1<<SLOW_PROBING_TURBO_RATES))
#endif //ENET_INC_ARCH_WAVE600

#ifdef ENET_INC_ARCH_WAVE600
#define SLOW_PROBING_PRIORITY1_ENABLED_BITMAP 	(1<<SLOW_PROBING_ANT_SEL)
#else
#define SLOW_PROBING_PRIORITY1_ENABLED_BITMAP 	((1<<SLOW_PROBING_CDD) | \
												(1<<SLOW_PROBING_ANT_SEL))

#endif

												
/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

typedef struct SlowProbingParams SlowProbingParams_t;

typedef void (*RaSlowProbingHandlerSetNextProbingPoint)(LinkAdaptationDatabaseDistributionPack_t*,bool);
typedef void (*RaSlowProbingHandlerProcessFeedback)(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);

typedef struct FastProbingStateParams
{
	RaFastProbingPoint_e* nextProbingPointTable;
	uint8 maxNumberOfProbingOptions;
}FastProbingStateParams_t;


struct SlowProbingParams
{
	
	RaSlowProbingHandlerSetNextProbingPoint setNextProbingPointFunc;
	RaSlowProbingHandlerProcessFeedback  slowProbingProcessFeedbackFunc;
	uint8 maxNumberOfIterationsInCycle;
};


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void probingHandlerConfigInit(void);
static void setSlowProbingEventIndication(LinkAdaptationEventHandler_t* pEventHandlerObj, RaProbingIndication_e* pNextProbeIndication, int8 slowProbingType, RaProbingIndication_e probingType);
static void setFastProbingEventIndication(LinkAdaptationEventHandler_t* pEventHandlerObj, RaProbingIndication_e* pNextProbeIndication);
static void resetCounters(LinkAdaptationEventHandler_t* pEventHandler , LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams);
static RaStabilityState_e searchPerDeviation(LaStationUspCommonParams_t* laStaUspCommon, LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams, uint8 currentPer);
static RaStabilityState_e updatePerStability(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void slowProbingInit(void);
static bool fastProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint );
#ifdef ENET_INC_ARCH_WAVE600
static void heMuFastProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint);
#endif
static void vhtMuFastProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint);
static void slowProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint);
static void incrementFastProbingKTh(LinkAdaptationEventHandler_t* pEventHandlerObj, LinkAdaptationConfigurationParams_t* pLinkAdaptationParams);
static void setManagementRatesInRaDB(StaId staIndex, uint8* pWpManagementRatesTable);
static RaStabilityState_e checkTargetPerStability(LaStationUspCommonParams_t* laStaUspCommon, LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams);
static bool prepareStaTcrDataRate(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter, LaTcrModificationStruct_t* pTcrParams, RaFastProbingPoint_e newFastProbingPoint);
static bool setStaDataRate(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, RaFastProbingPoint_e newFastProbingPoint, uint8 oldRate, Bandwidth_e bw);
static void setStaManagementRate(StaId staIndex, LaStationUspCommonParams_t* pLaStaUspCommon);
static void findMaxRateInMuGroup(StaId staIndex, uint32 maxRatePerBW[3], uint32 numberOfNss[3]);
uint8 GetMuPowerValInRegulationLimit(Bandwidth_e bandwidth,uint8 rate,uint32 numberOfNss,StaId staIndex);

static void rateModificationFunc(LaTcrModificationStruct_t* modifcationParamsIn, LaTcrModificationStruct_t* modifcationParamsOut);
extern uint32 MuScratchPad[PROTECTED_GROUP_DB_SIZE_WORDS];
#if 0
static bool isHighestOrLowestRate (Bandwidth_e bandwidth, RateMask_t raMask, uint8 rateIndex);
#endif
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*Fast probing static variables*/
static 	RaFastProbingPoint_e 		FastProbingUpNextProbingPointTable [NUMBER_OF_PROBE_UP_OPTIONS]= {UP_MCS, UP_NSS, DOWN_NSS_UP_MCS,LOWER_RATE};
static	RaFastProbingPoint_e 		FastProbingDownNextProbingPointTable [NUMBER_OF_PROBE_DOWN_OPTIONS] = {LOWER_RATE, UP_MCS};
static	RaFastProbingPoint_e 		FastProbingRRNextProbingPointTable [NUMBER_OF_PROBE_RR_OPTIONS] = {UP_MCS, UP_NSS,DOWN_NSS_UP_MCS, LOWER_RATE};
static	FastProbingStateParams_t 	arrFastProbingStateParams[NUMBER_OF_FAST_PROBING_STATES];
static 	SlowProbingParams_t 		arrSlowProbingTaskParams[SLOW_PROBING_TASKS_NUM];
static 	uint8						slowProbingEnabledLoopdBitmap = SLOW_PROBING_ENABLED_BITMAP;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
rateAdaptationInit


Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void rateAdaptationInit(void)
{	
	StaId staIndex = 0x0;
	uint8 groupIndex = 0x0;
	uint8 uspIndex = 0x0;
	StaId firstSidInBand = ConfigurationManager_GetFirstSidInMyBand();
	uint16 numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsInMyBand();	
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	

	for (staIndex = firstSidInBand; staIndex < (firstSidInBand + numOfSidsInBand); staIndex++)
	{
		laStateMachineInit(staIndex,INVALID_MU_USP_INDEX, FALSE);
		/*Reset RA DB*/
		rateAdaptationResetDb(staIndex,INVALID_MU_USP_INDEX, FALSE);
		updateLaDbDistributionParam(&laDbDistributionParameter,staIndex,INVALID_MU_USP_INDEX, FALSE);
		
		resetSlowProbingPriorityBitmaps(&laDbDistributionParameter);
		resetSlowProbingHandlerCounters(&laDbDistributionParameter);
	}
	for (groupIndex = 0x0; groupIndex < TX_MU_GROUPS; groupIndex++)
	{
		for(uspIndex= 0; uspIndex<MAX_USP_IN_VHT_GROUP; uspIndex++)
		{
			laStateMachineInit(groupIndex,uspIndex, FALSE);
			/*Reset RA DB*/
			rateAdaptationResetDb(groupIndex,uspIndex, FALSE);
			updateLaDbDistributionParam(&laDbDistributionParameter,groupIndex,uspIndex, FALSE);
			resetSlowProbingPriorityBitmaps(&laDbDistributionParameter);
			resetSlowProbingHandlerCounters(&laDbDistributionParameter);
		}
	}
#ifdef ENET_INC_ARCH_WAVE600
#if defined (ENET_INC_LMAC0)
	for (groupIndex = 0x0; groupIndex < NUM_OF_LA_HE_MU_DB_ENTRIES; groupIndex++)
	{
		for(uspIndex= 0; uspIndex<MAX_USP_IN_HE_GROUP; uspIndex++)
		{
			laStateMachineInit(groupIndex,uspIndex, TRUE);
			/*Reset RA DB*/
			rateAdaptationResetDb(groupIndex,uspIndex, TRUE);
		}
	}
#endif
#endif
	probingHandlerConfigInit();
	
}



/**********************************************************************************

slowProbingInit



Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void slowProbingInit(void)
{
	arrSlowProbingTaskParams[SLOW_PROBING_POWER].setNextProbingPointFunc = PowerSetNextProbingPointAndIndications;
	arrSlowProbingTaskParams[SLOW_PROBING_POWER].slowProbingProcessFeedbackFunc = PowerProcessFeedback;

	arrSlowProbingTaskParams[SLOW_PROBING_CP].setNextProbingPointFunc = CyclicPrefixSetNextProbingPoint;
	arrSlowProbingTaskParams[SLOW_PROBING_CP].slowProbingProcessFeedbackFunc = CyclicPrefixProcessFeedback;

#ifndef ENET_INC_ARCH_WAVE600
	arrSlowProbingTaskParams[SLOW_PROBING_CDD].setNextProbingPointFunc = CddSetNextProbingPoint;
	arrSlowProbingTaskParams[SLOW_PROBING_CDD].slowProbingProcessFeedbackFunc = CddProcessFeedback;
#endif

	arrSlowProbingTaskParams[SLOW_PROBING_ANT_SEL].setNextProbingPointFunc = AntennaSelectionSetNextProbingPoint;
	arrSlowProbingTaskParams[SLOW_PROBING_ANT_SEL].slowProbingProcessFeedbackFunc = AntennaSelectionProcessFeedback;
	
	arrSlowProbingTaskParams[SLOW_PROBING_BW].setNextProbingPointFunc = BwSetNextProbingPoint;
	arrSlowProbingTaskParams[SLOW_PROBING_BW].slowProbingProcessFeedbackFunc = BwProcessFeedback;

	arrSlowProbingTaskParams[SLOW_PROBING_BF].setNextProbingPointFunc = BeamformingPrepareNextProbingPoint;
	arrSlowProbingTaskParams[SLOW_PROBING_BF].slowProbingProcessFeedbackFunc = BeamformingProcessFeedback;

	arrSlowProbingTaskParams[SLOW_PROBING_TURBO_RATES].setNextProbingPointFunc = TurboRatesSetNextProbingPoint;
	arrSlowProbingTaskParams[SLOW_PROBING_TURBO_RATES].slowProbingProcessFeedbackFunc = TurboRatesProcessFeedback;

}

/**********************************************************************************

fastProbingInit



Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static void fastProbingInit(void)
{
	arrFastProbingStateParams[PROBE_UP].maxNumberOfProbingOptions = NUMBER_OF_PROBE_UP_OPTIONS;
	arrFastProbingStateParams[PROBE_UP].nextProbingPointTable = FastProbingUpNextProbingPointTable;

	arrFastProbingStateParams[PROBE_DOWN].maxNumberOfProbingOptions = NUMBER_OF_PROBE_DOWN_OPTIONS;
	arrFastProbingStateParams[PROBE_DOWN].nextProbingPointTable = FastProbingDownNextProbingPointTable;

	arrFastProbingStateParams[PROBE_RR].maxNumberOfProbingOptions = NUMBER_OF_PROBE_RR_OPTIONS;
	arrFastProbingStateParams[PROBE_RR].nextProbingPointTable = FastProbingRRNextProbingPointTable;
}

/**********************************************************************************

probingHnalderInit




Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void probingHandlerConfigInit(void)
{
	fastProbingInit();
	slowProbingInit();
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/**********************************************************************************
resetProbingStates



Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/

void rateAdaptationResetDb(StaId stationIndex,uint8 uspIndex, bool isHeGroup)
{
	RateAdaptationDataBase_t* pRaDbEntry;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter,stationIndex,uspIndex, isHeGroup);
	pRaDbEntry =  &(laDbDistributionParameter.laStaUspCommon->rateAdaptationDataBase);
	pRaDbEntry->fastProbeState = PROBE_UP;
	pRaDbEntry->stabilityState = laDbDistributionParameter.laStaGroupCommon->pLinkAdaptationConfigurationParams->defaultRaStabilityState;
	pRaDbEntry->fastProbingPointCounters.nextProbingPointCounter = NUMBER_OF_FAST_PROBING_OPTIONS;
	pRaDbEntry->fastProbingPointCounters.notEffectveProbeCounter = MAX_UINT8;
	laDbDistributionParameter.laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTask = SLOW_PROBING_TASKS_NUM;
	laDbDistributionParameter.laStaGroupCommon->probingIndication = NO_PROBING_INDICATION;
	if(laDbDistributionParameter.uspIndex != INVALID_MU_USP_INDEX) 
	{
		if (laDbDistributionParameter.laHeGroupUnique != NULL) //HE MU
		{
			/*MU - set probing validation bit so probing event will not activate in next probe cycle for this USP*/
			LA_SET_BIT_IN_BITMAP(laDbDistributionParameter.laHeGroupUnique->probingValidationBitmap.heUspBitmap, laDbDistributionParameter.uspIndex, BITMAP_ARRAY_MAX_SIZE_2); //KW_FIX_FW_G Added array bound check
		}
		laDbDistributionParameter.laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = FALSE;
	}
}


/**********************************************************************************
rateAdaptationCountEvent


Description:
------------
Countes MPDU event and probing timer events and initiate fast/slow probing event

Input: 
-----
pEventObj: Specfific evnet object (MPDU event or timer event_
pEventHandlerObj: pointer to teso objects: counters and timers 

Returns:
--------
	nextProbeIndication
	
**********************************************************************************/
RaProbingIndication_e  rateAdaptationCountEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool onlyFastIndications,RaEventObj_t* pEventObj, uint8 slowProbingKTh)
{
	LinkAdaptationEventHandler_t* pEventHandlerObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler);
	uint16*						pFastProbingCounter = &(pEventObj->raEventsCounters.fastProbingCounter);
	uint16*						pSlowProbingCounter = &(pEventObj->raEventsCounters.slowProbingCounter);
	uint16*						pSlowProbingPrioirtyCounter = pEventObj->raEventsCounters.slowProbingPrCounterArr;
	uint8* 						pSlowProbingPriorityKTh = pEventObj->raEventsThresholds.slowProbingPriorityKThArr;
	RaProbingIndication_e 		nextProbeIndication;
	int8						slowProbingType;
	RaProbingIndication_e		probingType;

//#ifdef LINK_ADAPTATION_LOGS
//	ILOG0_V("rateAdaptationCountEvent");
//	SLOG0(0, 0, LinkAdaptationEventHandler_t, pEventHandlerObj);
//	SLOG0("rateAdaptationCountEvent, FastProbingCounter = %d, SlowProbingCounter = %d, pr0 = %d, pr1 = %d, pr2 = %d, pr3 = %d, nextProbeIndication = %d",
//					(pEventHandlerObj->mpduEvent.raEventsCounters.fastProbingCounter,pEventHandlerObj->mpduEvent.raEventsCounters.slowProbingPrCounterArr[0], pEventHandlerObj->mpduEvent.raEventsCounters.slowProbingPrCounterArr[1], pEventHandlerObj->mpduEvent.raEventsCounters.slowProbingPrCounterArr[2], pEventHandlerObj->mpduEvent.raEventsCounters.slowProbingPrCounterArr[3], nextProbeIndication);
//#endif
	nextProbeIndication = NO_PROBING_INDICATION;	
	ILOG0_D("rateAdaptationCountEvent, seqTransactionNumber = %d", CurrentBaaReport.commonReport.seqTransactionNumber);
	if((laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX) || 
		(CurrentBaaReport.commonReport.seqTransactionNumber!= LastBaaReport.commonReport.seqTransactionNumber))
	{        
		(*pFastProbingCounter)++;
		(*pSlowProbingCounter)++;

		/*Probing event*/
		if (onlyFastIndications == FALSE)
		{
			/*Count either fast and slow events */				
			if ((*pSlowProbingCounter) >= EVENT_HANDLER_CALCULATE_TH(slowProbingKTh))
			{
				/*Slow probing >=TH	
					Initiate slow probing event

					zero MPDU general counters and timers  of slow probing  and fast probing					
					if slow probign priority 3 >= Th
						Initiate slow probing priority 3 event
						Zero MPDU counters and timers counter of priority 3 and below
					if slow probign priority 2>=Th => initiate slow probing priority 2 event
						Initiate slow probing priority 2 event
						Zero MPDU counters and timers counter of priority 2 and below
					if slow probign priority 1>=Th => initiate slow probing priority 1 event
						Initiate slow probing priority 1 event
						Zero MPDU counters and timers counter of priority 1 and below
					if slow probign priority 2>=Th => initiate slow probing priority 0 event
						Initiate slow probing priority 0 event
						Zero MPDU counters and timers counter of priority 0 
				*/
				/*Reset MPDU and timer fast probing and slow probing general counters*/		
				pEventHandlerObj->mpduEvent.raEventsCounters.fastProbingCounter = 0;
				pEventHandlerObj->mpduEvent.raEventsCounters.slowProbingCounter = 0;
				pEventHandlerObj->timerEvent.raEventsCounters.fastProbingCounter = 0;
				pEventHandlerObj->timerEvent.raEventsCounters.slowProbingCounter = 0;
				
				/*Increment all slow probing priority counters*/
				for (slowProbingType = SLOW_PROBING_PRIORITY0; slowProbingType <= MAX_SLOW_PRIORITY; slowProbingType++)
				{
					pSlowProbingPrioirtyCounter[slowProbingType]++;
				}
				/*Scan from the highest slow probing priority to the lower, which counter reached threshold. if so, set probe reset relevant counters and break loop*/
				for (slowProbingType = MAX_SLOW_PRIORITY, probingType = MAX_SLOW_PROBING_PRIORITY_INDICATION; slowProbingType >= SLOW_PROBING_PRIORITY0; slowProbingType--, probingType--)
				{
			
					if (pSlowProbingPrioirtyCounter[slowProbingType] >= EVENT_HANDLER_CALCULATE_TH(pSlowProbingPriorityKTh[slowProbingType]))
					{
						if ((laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] != 0) ||
							(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] != 0))
						{
							setSlowProbingEventIndication(pEventHandlerObj,&nextProbeIndication,slowProbingType,probingType);
							break;	
						}
						else /* In case this module disabled - set it always to threshold so it would be expire as soon as it will be enable */
						{
							pSlowProbingPrioirtyCounter[slowProbingType] = EVENT_HANDLER_CALCULATE_TH(pSlowProbingPriorityKTh[slowProbingType]); 
						}
					}
				}
			}
			else if ((*pFastProbingCounter) >= EVENT_HANDLER_CALCULATE_TH(pEventObj->raEventsThresholds.fastProbingThK))
			{
				/*Slow probing <TH && fast probing >=Th
					Initiate fast probing event
					Zero MPDU counters and timers counter of priority 2 and below*/

				/*Fast probing event*/
				setFastProbingEventIndication(pEventHandlerObj, &nextProbeIndication);
			}
		}
		else if ((*pFastProbingCounter) >= EVENT_HANDLER_CALCULATE_TH(pEventObj->raEventsThresholds.fastProbingThK))
		{
			/*Initiate only fast probing events
				Initiate another fast probe in the next counter event without slow probing
			Fast probing event*/
			setFastProbingEventIndication(pEventHandlerObj, &nextProbeIndication);
		}
	}		/*No probing event*/
	return nextProbeIndication;
}

/**********************************************************************************

setFastProbingEventIndication 


Description:
------------
Set fast probing event indication and reset relevant counters


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setFastProbingEventIndication(LinkAdaptationEventHandler_t* pEventHandlerObj, RaProbingIndication_e* pNextProbeIndication)
{
	pEventHandlerObj->mpduEvent.raEventsCounters.fastProbingCounter = 0;
	pEventHandlerObj->timerEvent.raEventsCounters.fastProbingCounter = 0;
	
	*pNextProbeIndication = FAST_PROBING_INDICATION;
}
/**********************************************************************************

setSlowProbingEventIndication 


Description:
------------
Set slow probing event  indication and reset  current priority and lower priorities MPDU and timers counters

Input: 
-----
Returns:
--------
	void - 
	
**********************************************************************************/
static void setSlowProbingEventIndication(LinkAdaptationEventHandler_t* pEventHandlerObj, RaProbingIndication_e* pNextProbeIndication, int8 slowProbingType, RaProbingIndication_e probingType)
{
	
	for(;slowProbingType >= 0; slowProbingType--)
	{
		pEventHandlerObj->mpduEvent.raEventsCounters.slowProbingPrCounterArr[slowProbingType] = 0;
		pEventHandlerObj->timerEvent.raEventsCounters.slowProbingPrCounterArr[slowProbingType] = 0;
	}
	
	/*Set relevant probing event in bitmap*/
	*pNextProbeIndication = probingType;
}

/**********************************************************************************

rateAdaptationProbingHandler 


Description:
------------
Handle fast/slow probing initiate

Input: 
-----
sta index, stayWithLastProbingPoint indication, 
Extract frtom RateAdaptationLocalParams: RA stability state, nextProbeIndication
Returns:
--------
	Update 
	
**********************************************************************************/

void rateAdaptationProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	LinkAdaptationConfigurationParams_t* pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	FastProbingPointEstimators_t* pWorkingPointEst = estimatorsGetWpPerEstimatorsPtr(laDbDistributionParameter);
	bool isWorkingPointValid = AlphaFilter_isFilterResultValid(&(pWorkingPointEst->averagePer),pLaConfigParams->maxValidFilterTsfDiff);
#ifdef ENET_INC_ARCH_WAVE600
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);	
#endif
#ifdef LINK_ADAPTATION_LOGS
//	//("rateAdaptationProbingHandler, staIndex = %d, pRaDbEntry->probingIndication = %d, Stability = %d, stayWithLastProbingPoint = %d",stationindex,  pRaDbEntry->probingIndication, pRaDbEntry->stabilityState, stayWithLastProbingPoint);
#endif	
	/*Check if we already have probing indication and if the WP is valid */
#ifdef LA_HE_MU_DEBUG
	ILOG0_DD("[rateAdaptationProbingHandler], laDbDistributionParameter->laStaGroupCommon->probingIndication = %d, isWorkingPointValid = %d", laDbDistributionParameter->laStaGroupCommon->probingIndication, isWorkingPointValid);
#endif
	if ((laDbDistributionParameter->laStaGroupCommon->probingIndication != NO_PROBING_INDICATION) && (isWorkingPointValid == TRUE))
	{	
#ifdef LA_HE_MU_DEBUG
		ILOG0_D("[rateAdaptationProbingHandler], laDbDistributionParameter->laStaGroupCommon->probingIndication = %d", laDbDistributionParameter->laStaGroupCommon->probingIndication);
#endif
		if(laDbDistributionParameter->laStaGroupCommon->probingIndication == FAST_PROBING_INDICATION)
		{	
#ifdef LA_HE_MU_DEBUG
			ILOG0_D("[rateAdaptationProbingHandler], laDbDistributionParameter->stationOrGroupIndex = %d", laDbDistributionParameter->stationOrGroupIndex);
			ILOG0_D("[rateAdaptationProbingHandler], laDbDistributionParameter->uspIndex = %d", laDbDistributionParameter->uspIndex);
#endif
			if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
			{
				fastProbingHandler(laDbDistributionParameter, stayWithLastProbingPoint);
			}
			else /* MU */
			{
#ifdef LA_HE_MU_DEBUG
				ILOG0_D("[rateAdaptationProbingHandler], laDbDistributionParameter->uspIndex = %d", laDbDistributionParameter->uspIndex);
#endif

#ifdef LA_HE_MU_DEBUG
				ILOG0_D("[rateAdaptationProbingHandler], isHeGroup = %d", isHeGroup);
#endif
				
#ifdef ENET_INC_ARCH_WAVE600
    			if (isHeGroup)
    			{
    				heMuFastProbingHandler(laDbDistributionParameter, stayWithLastProbingPoint);
    			}
    			else
#endif
    			{
    				vhtMuFastProbingHandler(laDbDistributionParameter, stayWithLastProbingPoint);
    			}
			}
		}
		else /*stabilityState == RA_STABLE && Slow probing indication*/
		{
			laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.probingHandlerInterleavingCounter++;

			if (laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.probingHandlerInterleavingCounter >= (1 << pLaConfigParams->slowProbingInterleavingLog2Val))
			{
				laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.probingHandlerInterleavingCounter = 0;
				slowProbingHandler(laDbDistributionParameter, stayWithLastProbingPoint);
			}
		}
	}
}
#ifdef ENET_INC_ARCH_WAVE600
static void heMuFastProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionVarPerUsp; 
	uint8	uspCount;
	bool probingPointChanged = FALSE;
	
	for(uspCount = 0; uspCount < MAX_USP_IN_HE_GROUP; uspCount++)
	{
#ifdef LA_HE_MU_DEBUG
		ILOG0_DD("[heMuFastProbingHandler], laDbDistributionParameter->laHeGroupUnique->groupStationIndexes[%d] = %d", uspCount, laDbDistributionParameter->laHeGroupUnique->groupStationIndexes[uspCount]);
#endif
		if(laDbDistributionParameter->laHeGroupUnique->groupStationIndexes[uspCount] != INVALID_STA_INDEX)
		{
			updateLaDbDistributionParam(&laDbDistributionVarPerUsp,laDbDistributionParameter->stationOrGroupIndex,uspCount, TRUE); 
#ifdef LA_HE_MU_DEBUG
			ILOG0_DD("[heMuFastProbingHandler], laDbDistributionVarPerUsp.laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = %d, LA_GET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap, uspCount, BITMAP_ARRAY_MAX_SIZE_2) = %d", laDbDistributionVarPerUsp.laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse, LA_GET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap, uspCount, BITMAP_ARRAY_MAX_SIZE_2));
#endif
			if((laDbDistributionVarPerUsp.laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse == FALSE)
				&&(LA_GET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap, uspCount, BITMAP_ARRAY_MAX_SIZE_2) == 0x0)) //KW_FIX_FW_G Added array bound check
			{
				probingPointChanged = fastProbingHandler(&laDbDistributionVarPerUsp, stayWithLastProbingPoint);
#ifdef LA_HE_MU_DEBUG
				ILOG0_D("[heMuFastProbingHandler], probingPointChanged = %d", probingPointChanged);
#endif
				// In case probing point haven't changed because the required type of probing is not valid - update the bitmap anyway to indicate probing done
				if(probingPointChanged == FALSE) 
				{
					LA_SET_BIT_IN_BITMAP(laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap, laDbDistributionVarPerUsp.uspIndex, BITMAP_ARRAY_MAX_SIZE_2); //KW_FIX_FW_G Added array bound check
#ifdef LA_HE_MU_DEBUG
					ILOG0_DDD("[heMuFastProbingHandler], laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] = %d, laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] = %d, laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationMask.heUspBitmap[1] = %d", laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[0], laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[1], laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationMask.heUspBitmap[1]);
#endif
					if((laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] == laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationMask.heUspBitmap[0]) && (laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] == laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationMask.heUspBitmap[1]))
					{
						laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] = 0x0; /*reset probing validation bitmap*/
						laDbDistributionVarPerUsp.laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] = 0x0; 
						laStateMachineChangeState(&laDbDistributionVarPerUsp,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
						break; // Stop probing triggering for the rest of the group
					}	
				}					
			}
		}
	}
	/* Replace the first occurance pointer to this phase in plan to point to the mirror phase */
	replacePtrInPlanToMirrorPhase(laDbDistributionParameter->stationOrGroupIndex, Utils_FindFirstSet(laDbDistributionParameter->laHeGroupUnique->phaseEntriesBitmap));
}
#endif

static void vhtMuFastProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionVarPerUsp; 
	uint8	uspCount;
	bool probingPointChanged = FALSE;
	
	/*Read the Group DB area to a Global Scratchpad*/
#ifdef ENET_INC_ARCH_WAVE600
	ProtectedDbLock_ReadReq((uint32*)&(GroupDbHwEntries[laDbDistributionParameter->stationOrGroupIndex].common.word6), PROTECTED_GROUP_DB_SIZE_WORDS, MuScratchPad);
#else
	ProtectedDbLock_ReadReq((uint32*)&(GroupDbHwEntries[laDbDistributionParameter->stationOrGroupIndex].word19), PROTECTED_GROUP_DB_SIZE_WORDS, MuScratchPad);
#endif			
	for(uspCount = 0; uspCount <  MAX_USP_IN_VHT_GROUP; uspCount++)
	{
		if(laDbDistributionParameter->laVhtGroupUnique->groupStationIndexes[uspCount] != INVALID_STA_INDEX)
		{
			updateLaDbDistributionParam(&laDbDistributionVarPerUsp,laDbDistributionParameter->stationOrGroupIndex,uspCount, FALSE); 
			if((laDbDistributionVarPerUsp.laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse == FALSE)
				&&((laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap&(0x1<<uspCount)) == 0x0))
			{
				//ILOG0_V("rateAdaptationProbingHandler MU - calling fastProbingHandler ");
				probingPointChanged = fastProbingHandler(&laDbDistributionVarPerUsp, stayWithLastProbingPoint);
				if(probingPointChanged == FALSE) /* In case probing point changed - update the validation bitmap accordingly*/ 
				{
					laDbDistributionVarPerUsp.laVhtGroupUnique->probingValidationBitmap |= (0x1<<laDbDistributionVarPerUsp.uspIndex);	
					if(laDbDistributionVarPerUsp.laVhtGroupUnique->probingValidationBitmap == laDbDistributionVarPerUsp.laVhtGroupUnique->probingValidationMask)
					{
						laDbDistributionVarPerUsp.laVhtGroupUnique->probingValidationBitmap = 0x0; /*reset probing validation bitmap*/
						laStateMachineChangeState(&laDbDistributionVarPerUsp,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
						break; // Stop probing triggering for the rest of the group
					}	
				}					
			}
		}
	}
	/*Write back the Global Scratchpad to the Group DB area*/					
#ifdef ENET_INC_ARCH_WAVE600
	ProtectedDbLock_WriteReq((uint32*)&(GroupDbHwEntries[laDbDistributionParameter->stationOrGroupIndex].common.word6), MuScratchPad, PROTECTED_GROUP_DB_SIZE_WORDS);
#else
	ProtectedDbLock_WriteReq((uint32*)&(GroupDbHwEntries[laDbDistributionParameter->stationOrGroupIndex].word19), MuScratchPad, PROTECTED_GROUP_DB_SIZE_WORDS);
#endif
}

/**********************************************************************************

slowProbingHandler 


Description:
------------

Input: 
-----
Returns:
--------
	Update 
	
**********************************************************************************/
static void slowProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	SlowProbingTasks_e currentSlowProbingTask;
	SlowProbingParams_t* pSlowProbeParams ;
	uint8 slowProbingIter = 0;

	do 
	{
		/*Get next slow probing task according to slowProbingCurrentTaskBitmap*/
		currentSlowProbingTask = getSlowProbingTaskFromBitmap(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTaskBitmap);
		if (currentSlowProbingTask < SLOW_PROBING_TASKS_NUM)
		{
			/*Call next probing point function according to task*/
			pSlowProbeParams = &arrSlowProbingTaskParams[currentSlowProbingTask];
			pSlowProbeParams->setNextProbingPointFunc(laDbDistributionParameter, stayWithLastProbingPoint);
		}
		else 
		{
			/*If there are no other tasks in slowProbingCurrentTaskBitmap, update slowProbingCurrentTaskBitmap and get first task in slowProbingCurrentTaskBitmap*/
			laStateMachineUpdateCurrentSlowProbingBitMap(laDbDistributionParameter->laStaGroupCommon->probingIndication, &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData));
			currentSlowProbingTask = getSlowProbingTaskFromBitmap(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTaskBitmap);
			if(currentSlowProbingTask < SLOW_PROBING_TASKS_NUM)
			{
				/*Call next probing point function according to task*/
				pSlowProbeParams = &arrSlowProbingTaskParams[currentSlowProbingTask];
				pSlowProbeParams->setNextProbingPointFunc(laDbDistributionParameter, stayWithLastProbingPoint);
			}
			else
			{
				/*No task to run (all tasks are disabled), change state to LA_WAIT_FOR_NEXT_PROBE_CYCLE and break the loop*/
				laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
				break;
			}
		}
		slowProbingIter++;
	}
	/*If loop was not triggered slow probing search for next slow probing task*/
	while ((laDbDistributionParameter->laStaGroupCommon->linkAdaptationState != LA_WAIT_FOR_SLOW_PROBE_VALID) 
			&& (laDbDistributionParameter->laStaGroupCommon->linkAdaptationState != LA_PREPARE_SLOW_PROBING_TRNASMISSION)
			&& (slowProbingIter < SLOW_PROBING_TASKS_NUM)); 	

}
/**********************************************************************************

rateAdaptationProbingHandler 


Description:
------------
Increment nesxt probing conter
Input: 
-----
Returns:
--------
	Update 
	
**********************************************************************************/

static void handleNextProbingPointCounter(RaFastProbingPointCounters_t* probingPointCounters, uint8 maxNumberOfProbingOptions)
{

	/*Increment fast probing option counter in order to probe the next fast points according to state*/
	probingPointCounters->nextProbingPointCounter++;
	
	/*If new rate grater or equal to last option of the state, change to the first probing point option*/
	if (probingPointCounters->nextProbingPointCounter >= maxNumberOfProbingOptions)
	{
		/*newProbingPoint >= maxProbingOptions indicates that all probing option were tried */

		/*Increment counter indicates how many times sequently current state  was tried */
		probingPointCounters->notEffectveProbeCounter++;
		
		/*Change to first probing option of the current state*/
		probingPointCounters->nextProbingPointCounter = 0;
		
	}
}

/**********************************************************************************

rateAdaptationProbingHandler 


Description:
------------
Handle fast/slow probing initiate

Input: 
-----
sta index, stayWithLastProbingPoint indication, 
Extract frtom RateAdaptationLocalParams: RA stability state, nextProbeIndication
Returns:
--------
	Update 
	
**********************************************************************************/

void handleProbingPoint(StaId staIndex, bool stayWithLastProbingPoint)
{
	UNUSED_PARAM(staIndex);	
	UNUSED_PARAM(stayWithLastProbingPoint);	
}

/**********************************************************************************

raFastProbingProcessFeedback 


Description:
------------
Function decide whteher working point need to be changes according to probing point filter resukts 
compare 3 working point to 9 probing points and if one of the probing points  is grater of one of the working points (no matter which Bw), change working point for 3 Bws and update probing points for 3 Bws

Input: 
-----
staIndex

Returns:
--------
	bool wpChanged-working point changed or not 
	
**********************************************************************************/
void raFastProbingProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LaStationUspCommonParams_t* pLaStaUspCommon = laDbDistributionParameter->laStaUspCommon;
	LinkAdaptationConfigurationParams_t* pLinkAdaptationParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	uint8 previousWpRateIndex = MAX_UINT8;
	uint8 previousWpPointIndexInEstimatorsDb = MAX_UINT8;
	FastProbingPointEstimators_t* pMaxEffectiveTpParams = NULL;
	FastProbingPointEstimators_t* previousWpEstAddr = NULL;
	Bandwidth_e bw;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 uspIndex;
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionVarPerUsp; 
#endif	
	bw = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex,isHeGroup);

	/*Search maximum TP over all probing points and working point*/
	estimatorsFastProbingSearchMaxEffTp(laDbDistributionParameter, pLaStaUspCommon->fastProbingEstimators,pLaStaUspCommon->fastProbingPointEstimatorsIndexes, &pMaxEffectiveTpParams, pLinkAdaptationParams->maxValidFilterTsfDiff, 0, pLinkAdaptationParams->workingPointChangeTpPercentMargin);

	/*Extract the last working point index*/
	previousWpPointIndexInEstimatorsDb = pLaStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	DEBUG_ASSERT(previousWpPointIndexInEstimatorsDb < TOTAL_NUMBER_OF_RATES_IN_DB);
	/*Extract the last rate index in the new BW*/
	previousWpEstAddr = &(pLaStaUspCommon->fastProbingEstimators[previousWpPointIndexInEstimatorsDb]);

#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DDD("[raFastProbingProcessFeedback], pMaxEffectiveTpParams = %d, previousWpEstAddr = %d, pMaxEffectiveTpParams = %d", pMaxEffectiveTpParams, previousWpEstAddr, pMaxEffectiveTpParams);
#endif

	// changing rate even if rate stays the same so that in case of HE-MU mirror phase is allways up to date
	if (pMaxEffectiveTpParams != NULL) 
	{
		/*Extract the last rate index in the new BW*/
		previousWpRateIndex = previousWpEstAddr->rateIndex;	
		if (previousWpEstAddr != pMaxEffectiveTpParams)
		{
#ifdef LINK_ADAPTATION_LOGS
			ILOG0_DDD("raFastProbingProcessFeedback, WORKING POINT CHANGED, Last rate = %d, new Rate = %d, newPER = %d", previousWpRateIndex,  pMaxEffectiveTpParams->rateIndex, pMaxEffectiveTpParams->averagePer.filterResult);
#endif
			/*Change working point*/
			rateAdaptationChangeWorkingPoint(laDbDistributionParameter, pMaxEffectiveTpParams->rateIndex, previousWpRateIndex);

			ILOG0_V("[raFastProbingProcessFeedback]");
#ifdef ENET_INC_ARCH_WAVE600
			if (isHeGroup) //rates are the same and HE-MU group then need to update rates although they haven't changed since mirror phase will be used and should be udpated for all users
			{
				for(uspIndex = 0; uspIndex < MAX_USP_IN_HE_GROUP; uspIndex++)
				{
					if((uspIndex != laDbDistributionParameter->uspIndex) && (laDbDistributionParameter->laHeGroupUnique->groupStationIndexes[uspIndex] != INVALID_STA_INDEX))
					{
						updateLaDbDistributionParam(&laDbDistributionVarPerUsp, laDbDistributionParameter->stationOrGroupIndex, uspIndex, TRUE); 
						setStaDataRate(&laDbDistributionVarPerUsp, WORKING_POINT, estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionVarPerUsp.laStaUspCommon), bw); 	
						ILOG0_DD("[raFastProbingProcessFeedback] for HE group. uspIndex = %d rate %d", uspIndex, estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionVarPerUsp.laStaUspCommon));
					}
				}
				heMuWorkingPointIsChanged(laDbDistributionParameter->stationOrGroupIndex);
			}
#endif
		}
	}
}

/**********************************************************************************

raFastProbingProcessFeedback 


Description:
------------
Function decide whteher working point change is required according to probing point filter results 
compare 3 working point to 9 probing points and if one of the probing points  is grater of one of the working points (no matter which Bw), change working point for 3 Bws and update probing points for 3 Bws

Input: 
-----
staIndex

Returns:
--------
	bool wpChanged-working point changed or not 
	
**********************************************************************************/
bool rateAdaptationChangeWorkingPoint(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter, uint8 newRateIndex,uint8 previousWpRateIndex)
{
	LaStationUspCommonParams_t* pLaStaUspCommon = pLaDbDistributionParameter->laStaUspCommon;
	uint8 workingPointEstimatorsIndex;
    uint8 newRate, oldRate = INVALID_RATE_INDEX_VAL;
	Bandwidth_e bw;
	bool isHeGroup = (pLaDbDistributionParameter->laHeGroupUnique != NULL);

#ifdef LINK_ADAPTATION_REGISTRATION
	RateAdaptationDataBase_t* pRaDb = &(pLaStaUspCommon->rateAdaptationDataBase);
	LaChangeWpCbParams_t registrationCbParams;
#endif

	bw = GetDataBwLimit(pLaDbDistributionParameter->stationOrGroupIndex,pLaDbDistributionParameter->uspIndex,isHeGroup);

	/* Got WD from phy due LDPC and phy mode is 11A (https://jira-chd.intel.com/browse/WLANRTSYS-7513?filter=-1) */
	/* check that new rate is in mask: */
#ifdef DYNAMIC_GROUPING_DEBUG
    SLOG0(0, 0, RateMask_t, &(pLaDbDistributionParameter->laStaUspCommon->raIndexMask));
#endif
	if (LA_GET_BIT_IN_BITMAP(pLaDbDistributionParameter->laStaUspCommon->raIndexMask.raIndexMask64bit, newRateIndex, RATES_BIT_MASK_SIZE) == FALSE) //KW_FIX_FW_G Added array bound check
	{
		/* Rate is not in mask */
		ASSERT(0);
	}

	workingPointEstimatorsIndex = pLaStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	if (workingPointEstimatorsIndex != INVALID_ESTIMATORS_DB_INDEX_VAL)
	{
		// protect from invalid access to fastProbingEstimators array when function is called from add STA at first time
		ASSERT(workingPointEstimatorsIndex < TOTAL_NUMBER_OF_RATES_IN_DB);//KW_FIX_FW_G
		oldRate = pLaStaUspCommon->fastProbingEstimators[workingPointEstimatorsIndex].rateIndex;
	}

	/*********************************************************/
	/*Update new working point and probing points in estimators DB for 3 Bws*/
	/********************************************************/
	estimatorsUpdateRatesInDb(pLaDbDistributionParameter, newRateIndex, previousWpRateIndex);

	newRate = estimatorsGetWorkingPointRateIndexOfMainBw(pLaDbDistributionParameter->laStaUspCommon);

	/*When sta connect oldRate = INVALID_RATE_INDEX_VAL.  In order to have old rate valid it will be replaced with newRate*/
	oldRate = (oldRate == INVALID_RATE_INDEX_VAL)? newRate : oldRate;

	/******************/
	/*	Set data rate		*/
	/******************/
	setStaDataRate(pLaDbDistributionParameter, WORKING_POINT, oldRate, bw);

	/*********************/
	/*reset RA dB                    */
	/********************/
	rateAdaptationResetDb(pLaDbDistributionParameter->stationOrGroupIndex,pLaDbDistributionParameter->uspIndex, isHeGroup);

	if(pLaDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/*********************/
		/*	Set management rates */
		/********************/
		setStaManagementRate(pLaDbDistributionParameter->stationOrGroupIndex, pLaStaUspCommon);
		
		/*Update beamforming loop according to working point rate*/
		/*********************************************/
		BeamformingRateChagedHadnling(pLaDbDistributionParameter->stationOrGroupIndex);
		/*Rate chage reset cyclic prefix DB*/
		cyclicPrefixWorkingPointChanged(pLaDbDistributionParameter->stationOrGroupIndex, oldRate, newRate);
		/* Disable BW loop if new rate is 11b/legacy */
		BwAdaptationRateIsChanged(pLaDbDistributionParameter->stationOrGroupIndex, oldRate, newRate);
		/*Update antenna selection loop according to working point rate*/
		/*********************************************/
		AntennaSelectionRateIsChanged(pLaDbDistributionParameter->stationOrGroupIndex, newRate, previousWpRateIndex);
		DynamicTxop_RateOrBwIsChanged(pLaDbDistributionParameter->stationOrGroupIndex, newRate, bw);
#ifdef LINK_ADAPTATION_REGISTRATION
		/*Send Wp change indication to registrated modules*/
		registrationCbParams.StationIndex = pLaDbDistributionParameter->stationOrGroupIndex;
		registrationCbParams.bwRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(pLaDbDistributionParameter->stationOrGroupIndex);
		linkAdaptationRegistrationSendEventInd(WORKING_POINT_CHANGED,(LaRgistrationGeneralCbParams_t*)&registrationCbParams);
#endif
	}

	rateAdaptationResetThresholds(pLaDbDistributionParameter->laStaGroupCommon);
	/**************************/
	/*Reset working point estimators*/
	/*************************/
	estimatorsResetWorkingPoint(pLaDbDistributionParameter->laStaUspCommon,pLaDbDistributionParameter->laStaGroupCommon);

    if (newRateIndex != previousWpRateIndex) // tHERE ARE MANY CASES BESIDE ADD_STATION THAT THOSE ARE EQUAL, WE HANDLE ADD STATION SEPARATELY
    {
        linkAdaptationDmrStationEventHandler(pLaDbDistributionParameter->stationOrGroupIndex,
                                             newRateIndex,
                                             previousWpRateIndex,
                                             DMR_STATION_RATE_CHANGED);
    }
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DDDD("[rateAdaptationChangeWorkingPoint], newRateIndex = %d, previousWpRateIndex = %d, pLaDbDistributionParameter->uspIndex = %d, pLaDbDistributionParameter->stationOrGroupIndex=%d", newRateIndex, previousWpRateIndex, pLaDbDistributionParameter->uspIndex, pLaDbDistributionParameter->stationOrGroupIndex);
#endif
	return TRUE;
}

/**********************************************************************************

raSlowProbingProcessFeedback 


Description:
------------
Input: 
-----
staIndex

Returns:
--------
**********************************************************************************/
void raSlowProbingProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	SlowProbingTasks_e currentSlowProbingTask = getSlowProbingTaskFromBitmap(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTaskBitmap);
	
	ASSERT(currentSlowProbingTask < SLOW_PROBING_TASKS_NUM);
	/*Function call*/
	arrSlowProbingTaskParams[currentSlowProbingTask].slowProbingProcessFeedbackFunc(laDbDistributionParameter);
	/*Reset slow probing estimators DB after process feedback*/
	estimatorsResetSlowProbingEstimators(laDbDistributionParameter);

}

/**********************************************************************************

rateAdaptationUpdateStabilityState 


Description:
------------
Set stability according to required estimator

Input: 
-----
stationIndex

Returns:
--------
	void - 
	
**********************************************************************************/
void rateAdaptationUpdateStabilityState(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	RateAdaptationDataBase_t* pRaDataBase = &(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
	RaStabilityState_e* pStabilityState = &pRaDataBase->stabilityState;
	EstimtorsIndication_t estimatorsIndication = CurrentPacketParams.estimatorsIndication;
#ifdef LINK_ADAPTATION_LOGS
	RaStabilityState_e stabilityStateBeforeChange = *pStabilityState;
#endif

	if ((estimatorsIndication.consequtiveFailure == TRUE) ||(estimatorsIndication.failsInWindow == TRUE))
	{
		*pStabilityState = RA_HIGH_PER;
		resetSlowProbingPriorityBitmaps(laDbDistributionParameter);
	}
	else if (estimatorsIndication.per == TRUE)
	{
		*pStabilityState = updatePerStability(laDbDistributionParameter);
	}

	/*If state changed from stable to un stable, reset counters and thresholds*/
	if (*pStabilityState != RA_STABLE)
	{
		rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
	}

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX) /* update statistic for SU only */
	{
		LinkAdaptationStatistics.raStability[laDbDistributionParameter->stationOrGroupIndex] = *pStabilityState; //LINK_ADAPTATION_RA_STABILITY
	}
	else
	{
		LinkAdaptationMuStatistics.raStability[laDbDistributionParameter->stationOrGroupIndex][laDbDistributionParameter->uspIndex]  = *pStabilityState;
	}
	
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DD("updatePerStability, stabilityStateBeforeChange = %d, pStabilityState = %d", stabilityStateBeforeChange, *pStabilityState);
#endif
}
/**********************************************************************************

updatePerStability 


Description:
------------
Set stability state:
currentPer < Target PER - margin => LOW_PER
currentPer > targetPer + Margin => high per
else, 
check whether PER change compared to average PER (using function searchPerDeviation)

Input: 
-----
pEstimatorsBwDb
pLinkAdaptationConfigurationParamst - use for extract targetPer, targetPerMargin
currentPer

Returns:
--------
	void - 
	
**********************************************************************************/
static RaStabilityState_e updatePerStability(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	LaStationUspCommonParams_t* laStaUspCommon = laDbDistributionParameter->laStaUspCommon;
	RaStabilityState_e perDeviationStability;
	RaStabilityState_e targetPerStability;
	RaStabilityState_e resultPerStability;
//	uint8 currentRateIndex = pLinkAdaptationDb->rateAdaptationDataBase.wpDataRateIndex[CurrentPacketParams.bandwidth];

	perDeviationStability = searchPerDeviation(laStaUspCommon,  pLinkAdaptationConfigurationParams, CurrentPacketParams.currentPer);
	targetPerStability = checkTargetPerStability(laStaUspCommon,  pLinkAdaptationConfigurationParams);
	
	if (perDeviationStability == RA_STABLE)
	{
	//	if ((targetPerStability == RA_STABLE) ||(isHighestOrLowestRate(pLinkAdaptationDb->raIndexMask,currentRateIndex) == TRUE))
	//	{
			resultPerStability = RA_STABLE;
	//	}
	//	else
	//	{
	//		resultPerStability = targetPerStability;
	//	}
		
	}
	else if (targetPerStability != RA_STABLE)
	{
		resultPerStability = targetPerStability;
	}
	else
	{
		resultPerStability = perDeviationStability;
	}

	
#ifdef LINK_ADPATATION_LOGS
	ILOG0_DDD("updatePerStability, perDeviationStability = %d,targetPerStability = %d, resultPerStability = %d", 
			perDeviationStability, targetPerStability, resultPerStability);
#endif
	return resultPerStability;
}
/**********************************************************************************

isHighestOrLowestRate 


Description:
------------

Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
#if 0
static bool isHighestOrLowestRate (Bandwidth_e bandwidth, RateMask_t raMask, uint8 rateIndex)
{
	
	uint8 tempRateIndex;
	bool boundaryRate = FALSE;

	tempRateIndex = rateAdaptationGetHigherRateIndex(bandwidth, rateIndex,raMask,1);

	if (rateIndex == tempRateIndex)
	{
		boundaryRate = TRUE;
	}
	else
	{
		tempRateIndex = rateAdaptationGetLowerRateIndex(bandwidth,rateIndex,raMask,1);
		if (tempRateIndex == rateIndex)
		{
			boundaryRate = TRUE;
		}
	}

	return boundaryRate;
	

}
#endif
/**********************************************************************************

checkTargetPerStability 


Description:
------------
Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static RaStabilityState_e checkTargetPerStability(LaStationUspCommonParams_t* laStaUspCommon, LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams)
{

	FastProbingPointEstimators_t* pfastProbingPointEstimatorsDb;
	uint8 avPerResult;
	uint32 maxValidTsfDiff = pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff;
	RaStabilityState_e raPerStability = RA_STABLE;
	uint8 fastProbingEstIndex;
	uint8 targetPer = pLinkAdaptationConfigurationParams->targetPer;
	uint8 targetPerMargin = pLinkAdaptationConfigurationParams->targetPerMargin;
	
	/*Extract pointer to working point filters in estimators DB*/
	fastProbingEstIndex = laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	DEBUG_ASSERT(fastProbingEstIndex < TOTAL_NUMBER_OF_RATES_IN_DB);
	pfastProbingPointEstimatorsDb = &(laStaUspCommon->fastProbingEstimators[fastProbingEstIndex]);
	/*Check if PER and PER variance are valid*/
	if (AlphaFilter_isFilterResultValid(&pfastProbingPointEstimatorsDb->averagePer, maxValidTsfDiff))
	{
		/*Convert filter result to PER (percent untis)*/
		avPerResult = AlphaFilter_GetFilterResult(&pfastProbingPointEstimatorsDb->averagePer);

		/*Convert filter result to PER variance */

		if (avPerResult < (targetPer - targetPerMargin))
		{
			/*currentPer < Target PER - margin => LOW_PER*/
			raPerStability = RA_LOW_PER;
		}
		else if (avPerResult > (targetPer + targetPerMargin))
		{
			/*currentPer > targetPer + Margin => high per*/
			raPerStability = RA_HIGH_PER;
		}
		else
		{
			raPerStability = RA_STABLE;
		}
		
	}
#ifdef LINK_ADAPTATION_LOGS	
	{

		pfastProbingPointEstimatorsDb = &(laStaUspCommon->fastProbingEstimators[laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT]]);
		avPerResult = AlphaFilter_GetFilterResult(&pfastProbingPointEstimatorsDb->perVariance);

//		ILOG0_DD("checkTargetPerStability, avPerResult = %d, targetPer - targetPerMargin = %d", avPerResult, targetPer - targetPerMargin);
	}
#endif
	return raPerStability;

}

/**********************************************************************************

searchPerDeviation 


Description:
------------
Compare current PER with 3 BWs average per, when one of te condition is fullfilled with one of the Bws set stability state - start with 80 MHz

PER < AvPer - 2Sigma => LOW PER
PER > AvPer + 2Sigma => HIgh PER
AvPer - 2Sigma <=currentPER <= AvPer + 2Sigma => stable

Input: 
-----
pEstimatorsBwDb - 3 BWs statistics array
pLinkAdaptationConfigurationParamst - configration paramters - using for extract 'maxValidTsfDiff
currentPer - current BA PER

Returns:
--------
	void - 
	
**********************************************************************************/
static RaStabilityState_e searchPerDeviation(LaStationUspCommonParams_t* laStaUspCommon, LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams, uint8 currentPer)
{
	FastProbingPointEstimators_t* pfastProbingPointEstimatorsDb;
	uint8 avPerResult;
	uint8 perVarianceResult;
	uint32 maxValidTsfDiff = pLinkAdaptationConfigurationParams->maxValidFilterTsfDiff;
	RaStabilityState_e raPerStability = RA_STABLE;
	uint8 fastProbingEstIndex;
	
	/*Extract pointer to working point filters in estimators DB*/
	fastProbingEstIndex = laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT];
	DEBUG_ASSERT(fastProbingEstIndex < TOTAL_NUMBER_OF_RATES_IN_DB);
	pfastProbingPointEstimatorsDb = &(laStaUspCommon->fastProbingEstimators[fastProbingEstIndex]);
	/*Check if PER and PER variance are valid*/
	if (AlphaFilter_isFilterResultValid(&pfastProbingPointEstimatorsDb->averagePer, maxValidTsfDiff) 
			&&(AlphaFilter_isFilterResultValid(&pfastProbingPointEstimatorsDb->perVariance, maxValidTsfDiff))) 
	{
		/*Convert filter result to PER (percent untis)*/
		avPerResult = AlphaFilter_GetFilterResult(&pfastProbingPointEstimatorsDb->averagePer);
		/*Convert filter result to PER variance */
		perVarianceResult = AlphaFilter_GetFilterResult(&pfastProbingPointEstimatorsDb->perVariance);
#ifdef LINK_ADAPTAION_LOGS
		//("searchPerDeviation,per variance,  RateIndex = %d, bandwidth = %d, CurrentPER = %d, filter result = %d, alpha = %d, lastPcacketTsf = %d",
		//		pfastProbingPointEstimatorsDb->rateIndex, pfastProbingPointEstimatorsDb->bandwidth, CurrentPacketParams.currentPer, pfastProbingPointEstimatorsDb->perVariance.filterResult>>11, pfastProbingPointEstimatorsDb->perVariance.alpha, pfastProbingPointEstimatorsDb->perVariance.lastPacketTsf);
#endif
		if (currentPer < ((int32)(avPerResult - ((perVarianceResult+2) << 1)))) //add 2 so of perVarianceResultis arroud zero it will not chnage stability
		{
			/*PER < AvPer - 2Sigma => LOW PER*/
			raPerStability = RA_LOW_PER;
		}
		else if (currentPer > (avPerResult + ((perVarianceResult+2) << 1))) //add 2 so of perVarianceResultis arroud zero it will not chnage stability
		{
			/*PER > AvPer + 2Sigma => HIgh PER*/
			raPerStability = RA_HIGH_PER;
		}
		else
		{
			/*(AvPer - 2Sigma )<=currentPER <= (AvPer + 2Sigma) => RA_STABLE*/
			raPerStability = RA_STABLE;
		}
		
	}
#ifdef LINK_ADAPTATION_LOGS	
	{

		pfastProbingPointEstimatorsDb = &(laStaUspCommon->fastProbingEstimators[laStaUspCommon->fastProbingPointEstimatorsIndexes[WORKING_POINT]]);
		perVarianceResult = AlphaFilter_GetFilterResult(&pfastProbingPointEstimatorsDb->perVariance);

//		ILOG0_DDD("searchPerDeviation, avPerResult = %d, currentPer = %d, perVarianceResult = %d", avPerResult, currentPer, perVarianceResult);
	}
#endif
	return raPerStability;
}


/**********************************************************************************

probingStateMachineEnterEvent 


Description:
------------
This function configure new probing point according to fast probing state:
Probe up state: round robin between 2 rates:
				up NSS, up MCS
Probe down state: try one option: down MCS
Probe RR : round robin betwen 3 options:
		up NSS, up MCS, down MCS.
The new rate is chosen according to counter in the state parmeters
Number of iteration in the same state is counted and if grater than threshold, Kfast is increment in order to decrease the fast probing frequency

Input: 
-----
stationIndex, lastFastProbingState, newFastProbingState

Returns:
--------
	void - TURE if rate was changed, FALSE - rate canot being changing
	
**********************************************************************************/
static bool fastProbingHandler(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter, bool stayWithLastProbingPoint )
{
	LinkAdaptationConfigurationParams_t* pLinkAdaptationParams = pLaDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	RateAdaptationDataBase_t* pRaDbEntry = &(pLaDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
	RaFastProbingPointCounters_t* pFastProbingCounters = &(pRaDbEntry->fastProbingPointCounters);
	RaFastProbingPoint_e newProbingPoint = (RaFastProbingPoint_e)0;
	RaFastProbingState_e fastProbingLastState = (RaFastProbingState_e)(pRaDbEntry->fastProbeState);
	FastProbingStateParams_t* pNewFastProbingStateParams = NULL;
	bool probingPointChanged = FALSE;
	uint8 workingPointRateIndex;
	Bandwidth_e bandwidth;
	uint8 probingPointArrIndex = 0;
	bool isHeGroup = (pLaDbDistributionParameter->laHeGroupUnique != NULL);

	bandwidth = GetDataBwLimit(pLaDbDistributionParameter->stationOrGroupIndex,pLaDbDistributionParameter->uspIndex,isHeGroup);
	workingPointRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(pLaDbDistributionParameter->laStaUspCommon);
	
	if(stayWithLastProbingPoint != TRUE)
	{
		/*Fast probing state define according to RA stability state:
			RA_STABLE => PROBE_RR	
			LOW_PER =>	PROBE_UP
			HIGH_PER => PROBE_DOWN
		*/
		pRaDbEntry->fastProbeState =  (RaFastProbingState_e)pRaDbEntry->stabilityState; 
		DEBUG_ASSERT(pRaDbEntry->fastProbeState < NUMBER_OF_FAST_PROBING_STATES);
		
		/*Extract fast probe parameters according to new fast probing state*/
		pNewFastProbingStateParams = &arrFastProbingStateParams[pRaDbEntry->fastProbeState];

		/*if fast probing state changed from the last state, start from first probing point in new state*/
		if (fastProbingLastState != pRaDbEntry->fastProbeState)
		{
			pFastProbingCounters->nextProbingPointCounter = 0;
		}
		do 
		{
			/*Increment next probing point couner*/
			handleNextProbingPointCounter(pFastProbingCounters, pNewFastProbingStateParams->maxNumberOfProbingOptions);
			DEBUG_ASSERT(pFastProbingCounters->nextProbingPointCounter < pNewFastProbingStateParams->maxNumberOfProbingOptions);			
			
			/*Extract new rate option from probing optionsTable according to probing counter*/
			newProbingPoint = pNewFastProbingStateParams->nextProbingPointTable[pFastProbingCounters->nextProbingPointCounter];
			probingPointArrIndex++;

		}while ((probingPointArrIndex < pNewFastProbingStateParams->maxNumberOfProbingOptions) &&
													 (estimatorsGetProbingPointRateIndex(pLaDbDistributionParameter,newProbingPoint) == INVALID_ESTIMATORS_DB_INDEX_VAL));
	}
	else
	{
		/*Extract fast probe parameters according to new fast probing state*/
		pNewFastProbingStateParams = &arrFastProbingStateParams[pRaDbEntry->fastProbeState];
		
		DEBUG_ASSERT(pFastProbingCounters->nextProbingPointCounter < pNewFastProbingStateParams->maxNumberOfProbingOptions);
		/*Extract new rate option from probing optionsTable according to probing counter*/
		newProbingPoint = pNewFastProbingStateParams->nextProbingPointTable[pFastProbingCounters->nextProbingPointCounter];
	}



#ifdef LINK_ADAPTATION_LOGS
//		("fastProbingHandler,staindex = %d, stayWithLastPoint = %d, fastProbingLastState = %d, fastProbingNewState = %d,StabilityState = %d, newProbingPoint=%d, notEffectveProbeCounter = %d, fastProbingThreshold = %d", 
//				staindex, stayWithLastProbingPoint, fastProbingLastState, pRaDbEntry->fastProbeState, pRaDbEntry->stabilityState, newProbingPoint, pFastProbingCounters->notEffectveProbeCounter, pRaDbEntry->laEventsHandler.mpduEvent.raEventsThresholds.fastProbingThK);
#endif

	
	/*If fast probing state (UP_NSS, UPMCS, LOWER_RATE) was changed, reset same state counter*/
	if (fastProbingLastState != pRaDbEntry->fastProbeState)
	{
		pFastProbingCounters->notEffectveProbeCounter = 0;
	}
	else /*fast probing state was not changed*/
	{
		if (pFastProbingCounters->notEffectveProbeCounter >= pLinkAdaptationParams->fastProbingTriesBeforeChangeKTh)
		{
			/*Same state has been tried for several times without changing the working point increment thresholds in order to decrease probing frequency*/
			pFastProbingCounters->notEffectveProbeCounter = 0;
			/*Increlent fast probing k Th and slow probing k th accordingly*/
			/*For HE group we should not increment for now - Additional implementation need to be done */
			if (pLaDbDistributionParameter->laHeGroupUnique==NULL)
			{
				incrementFastProbingKTh(&(pLaDbDistributionParameter->laStaGroupCommon->laEventsHandler), pLinkAdaptationParams);
			}
			/*Change to steady state configuration parameters*/
			ChangeStaToSteadyStateConfigurationParams(pLaDbDistributionParameter, TRUE);
		}
	}
	pRaDbEntry->currentFastProbingPoint = newProbingPoint;
	/*Call function for change rate according to new rate probing point*/

	probingPointChanged = setStaDataRate(pLaDbDistributionParameter, newProbingPoint,workingPointRateIndex, bandwidth);
	
	if (probingPointChanged == TRUE)
	{
		/*Change RA state to wait for fast probe valid state*/
		laStateMachineChangeState(pLaDbDistributionParameter,LA_WAIT_FOR_FAST_PROBE_VALID);
	
	}
	return probingPointChanged;
}
/**********************************************************************************

incrementFastProbingKTh 


Description:
------------

Input: 
-----

Returns:
--------

	
**********************************************************************************/
static void incrementFastProbingKTh(LinkAdaptationEventHandler_t* pEventHandlerObj, LinkAdaptationConfigurationParams_t* pLinkAdaptationParams)
{
	uint8* mpduSlowProbingTh = &pLinkAdaptationParams->mpduEventSlowProbingKThInit;
	uint8* timerSlowProbingTh = &pLinkAdaptationParams->timerEventSlowProbingKThInit;
	uint8* mpduFastProbingTh = &pEventHandlerObj->mpduEvent.raEventsThresholds.fastProbingThK;
	uint8* timerFastProbingTh = &pEventHandlerObj->timerEvent.raEventsThresholds.fastProbingThK;
	
	/*Increment fast probing k th - maximum value is slow probing K th -1*/
	
	*mpduFastProbingTh = MIN((*mpduFastProbingTh + 1),(*mpduSlowProbingTh-1));
	*timerFastProbingTh = MIN((*timerFastProbingTh + 1),(*timerSlowProbingTh-1));
	
#ifdef LINK_ADAPTATION_LOGS
	//("incrementFastProbingKTh,mpduSlowProbingTh = %d,  timerSlowProbingTh = %d, mpduFastProbingTh = %d, timerFastProbingTh = %d",
//			*mpduSlowProbingTh, *timerSlowProbingTh, *mpduFastProbingTh, *timerFastProbingTh);
#endif

}

/****************************************************************************
 **
 ** NAME:        rateAdaptationResetCountersAndThresholds
 **
 ** PARAMETERS:  stationIndex
 **             
 **
 ** RETURN VALUES:  
 **
 ** DESCRIPTION:  
 **              
 **
 **   
 ***************************************************************************/
void rateAdaptationResetCountersAndThresholds(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams;
	LinkAdaptationEventHandler_t* pEventHandler;  

	pEventHandler = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler);
	pLinkAdaptationConfigurationParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	
#ifdef LINK_ADAPTATION_LOGS
//		//("rateAdaptationResetCountersAndThresholds");
#endif
	laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.probingHandlerInterleavingCounter= 0;

	resetCounters(pEventHandler , pLinkAdaptationConfigurationParams);
	rateAdaptationResetThresholds(laDbDistributionParameter->laStaGroupCommon);
}
/****************************************************************************
 **
 ** NAME:        resetFastProbingCounters
 **
 ** PARAMETERS:  resetFastProbingCounters
 **             
 **
 ** RETURN VALUES:  
 **
 ** DESCRIPTION:  
 **              
 **
 **   
 ***************************************************************************/

void resetFastProbingCounters(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationEventHandler_t* pEventHandler;  

	pEventHandler = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler);

	pEventHandler->mpduEvent.raEventsCounters.fastProbingCounter = 0;
	pEventHandler->timerEvent.raEventsCounters.fastProbingCounter = 0;

}

/****************************************************************************
 **
 ** NAME:        resetCounters
 **
 ** PARAMETERS:  stationIndex
 **             
 **
 ** RETURN VALUES:  
 **
 ** DESCRIPTION:  
 **              
 **
 **   
 ***************************************************************************/

static void resetCounters(LinkAdaptationEventHandler_t* pEventHandler , LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams)
{
	UNUSED_PARAM(pLinkAdaptationConfigurationParams);	

	memset(&pEventHandler->mpduEvent.raEventsCounters, 0, sizeof(RaEventsCounters_t));
	memset(&pEventHandler->timerEvent.raEventsCounters, 0, sizeof(RaEventsCounters_t));
}
/****************************************************************************
 **
 ** NAME:        resetCountersAndThresholds
 **
 ** PARAMETERS:  stationIndex
 **             
 **
 ** RETURN VALUES:  
 **
 ** DESCRIPTION:  
 **              
 **
 **   
 ***************************************************************************/

void rateAdaptationResetThresholds(LaStationGroupCommonParams_t* laStationGroupCommonParameter)
{
	LinkAdaptationEventHandler_t* pEventHandler = &(laStationGroupCommonParameter->laEventsHandler);
	LinkAdaptationConfigurationParams_t* pLinkAdaptationConfigurationParams = laStationGroupCommonParameter->pLinkAdaptationConfigurationParams;
	RaEventThresholds_t* pCountersThConfiguration = &pLinkAdaptationConfigurationParams->mpduCountersThresholds;
	RaEventThresholds_t* pTimerEventThConfiguration = &pLinkAdaptationConfigurationParams->timerEventCountersThresholds;
	

	/*Reset MPDU events threshold*/
	MEMCPY(&(pEventHandler->mpduEvent.raEventsThresholds), pCountersThConfiguration, sizeof(RaEventThresholds_t));

	/*Reset timer event thresholds*/
	MEMCPY(&(pEventHandler->timerEvent.raEventsThresholds), pTimerEventThConfiguration, sizeof(RaEventThresholds_t));
}

/****************************************************************************
 **
 ** NAME:        rateAdaptationGetHigherRateIndex
 **
 ** PARAMETERS:  
 **rateIndex - current rate index
 ** pRaMask - rate adaptation index mask
 **
 ** RETURN VALUES: 
 ** newRateIndex - rate index of higher phy rate
 **
 ** DESCRIPTION: 
 ** Find the index of the next phy rate
 **              
 **
 **   
 ***************************************************************************/
uint8 rateAdaptationGetHigherRateIndex(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, uint8 rateIndex, RateMask_t rateMask, int8 numberOfRateLevelToSkip, bool forceAddLegacyRatesToAllowedMask)
{
	
	uint8 newRateIndx = rateIndex;
	uint8 tempRateIndex = rateIndex;
	uint32 indexInRaMask = 0;
	RateMask_t bandwidthRaMask;	
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	const uint8* sortedRatesTable = getSortedRatesTable(laDbDistributionParameter->laStaUspCommon);
	uint8 maxRatesInTable = getMaxNumOfRatesInTable(laDbDistributionParameter->laStaUspCommon);
	uint8 sortedPhyRateSameBwIndex;


	getStaRateMaskPerBw(laDbDistributionParameter, bandwidth, rateMask, &bandwidthRaMask, forceAddLegacyRatesToAllowedMask);
	/*Set turbo rates in local raMask*/
	if( (laDbDistributionParameter->laStaUspCommon != NULL)&& (FALSE == laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta))
	{

		/*Turbo rates is not relevant for he stations or VAP*/
		TurboRatesGetBitmap(laDbDistributionParameter->stationOrGroupIndex, laDbDistributionParameter->uspIndex,  bandwidth,&bandwidthRaMask);
	}
	DEBUG_ASSERT(rateIndex < maxRatesInTable);
		
	sortedPhyRateSameBwIndex = ratesTable[rateIndex].sortPhyRateSameBwIndex;

	/*Check if new rate is in RA mask, if not increment by one*/
	do
	{
		tempRateIndex = sortedRatesTable[sortedPhyRateSameBwIndex];	
		/*Check if new rate index is in RA mask and phy rate is greater*/
		indexInRaMask = isBitInMask64Bit(tempRateIndex,bandwidthRaMask.raIndexMask64bit);
		if ((indexInRaMask != 0) && (ratesTable[tempRateIndex].rateTableBwParmas[bandwidth].shortCpPhyRate > ratesTable[rateIndex].rateTableBwParmas[bandwidth].shortCpPhyRate))
		{
			/*new index is in mask and new nss number is  lower than the original rate nss number*/
			newRateIndx = tempRateIndex;
			numberOfRateLevelToSkip--;
		}
		
		sortedPhyRateSameBwIndex++;
	}
	while ((sortedPhyRateSameBwIndex < maxRatesInTable) && (numberOfRateLevelToSkip > 0));
#ifdef LINK_ADAPTATION_LOGS
//	//("rateAdaptationGetHigherRateIndex, tempAndResultBitmap = %X-%X", tempAndResultBitmap[1], tempAndResultBitmap[0]);
#endif
	return newRateIndx;
}
/****************************************************************************
 **
 ** NAME:        rateAdaptationGetLowerRateIndex
 **
 ** PARAMETERS:  
 **rateIndex - current rate index
 ** pRaMask - rate adaptation index mask
 **
 ** RETURN VALUES: 
 ** newRateIndex - rate index of lower phy rate
 **
 ** DESCRIPTION: 
 ** Find the index of the lower phy rate
 **              
 **
 **   
 ***************************************************************************/
uint8 rateAdaptationGetLowerRateIndex(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bandwidth, uint8 rateIndex, RateMask_t rateMask, int8 numberOfRateLevelToSkip, bool forceAddLegacyRatesToAllowedMask)
{
	int8 sortedPhyRateSameBwIndex=0;
	uint8 newRateIndx = rateIndex;
	uint8 tempRateIndex = rateIndex;
	uint32 indexInRaMask = 0;
	RateMask_t bandwidthRaMask;	
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	const uint8* sortedRatesTable = getSortedRatesTable(laDbDistributionParameter->laStaUspCommon);
	uint8 maxRatesInTable = getMaxNumOfRatesInTable(laDbDistributionParameter->laStaUspCommon);
	uint8 nssNum = ratesTable[rateIndex].numberOfNss;
	uint8 tempNssNum = nssNum;
	uint8 orgMcsNum = ratesTable[rateIndex].vhtHeMcs;
	uint8 tempMcsNum = orgMcsNum;
	bool inputRateNotInMask = FALSE;
	LinkAdaptationPhyMode_e laPhyMode;

	
	getStaRateMaskPerBw(laDbDistributionParameter, bandwidth, rateMask, &bandwidthRaMask, forceAddLegacyRatesToAllowedMask);
	/*Set turbo rates in local raMask*/
	if( (laDbDistributionParameter->laStaUspCommon != NULL)&& (FALSE == laDbDistributionParameter->laStaUspCommon->staTransmissionParams.heSta))
	{
		/*Turbo rates is not relevant for he stations or VAP*/
		TurboRatesGetBitmap(laDbDistributionParameter->stationOrGroupIndex, laDbDistributionParameter->uspIndex,  bandwidth,&bandwidthRaMask);
	}	
	/*Check if input rate NSS is in mask*/
	if(isBitInMask64Bit(rateIndex,bandwidthRaMask.raIndexMask64bit) == 0)
	{
		inputRateNotInMask = TRUE;
	}

	DEBUG_ASSERT(rateIndex < maxRatesInTable)

	sortedPhyRateSameBwIndex = (int8)(ratesTable[rateIndex].sortPhyRateSameBwIndex ); //KW_FIX_FW_G (explicit typecast from uint to int needed)

	do
	{
		tempRateIndex = sortedRatesTable[sortedPhyRateSameBwIndex];	
		tempNssNum = ratesTable[tempRateIndex].numberOfNss;
		tempMcsNum = ratesTable[tempRateIndex].vhtHeMcs;
		laPhyMode = ratesTable[tempRateIndex].laPhyMode;
		/*Check if new rate index is in RA mask*/
		indexInRaMask = isBitInMask64Bit(tempRateIndex,bandwidthRaMask.raIndexMask64bit);
		if ((indexInRaMask != 0) && ((inputRateNotInMask)||(tempNssNum == nssNum)) && ((tempMcsNum <= orgMcsNum) ||(laPhyMode < LA_PHY_MODE_HT_VHT)))
		{
			/*new index is in mask and new nss number is  lower than the original rate nss number*/
			newRateIndx = tempRateIndex;
			numberOfRateLevelToSkip--;
			
			/*If we reached MCS, search for lower NSS with MCS 0 or lower PHY mode (according to mask)*/
			if ((tempMcsNum == 0) && (nssNum > 0))
			{
				nssNum--;
			}
		}
		sortedPhyRateSameBwIndex--;
	}
	while ((sortedPhyRateSameBwIndex >= 0) && (numberOfRateLevelToSkip >= 0 ));

	return newRateIndx;
}

/****************************************************************************
 **
 ** NAME:        rateAdaptationFastDrop
 **
 ** PARAMETERS:  
 **stationIndex
 **
 ** RETURN VALUES: 
 **
 **
 ** DESCRIPTION: 
 ** Drop one rate down and change state
 **              
 **
 **   
 ***************************************************************************/

bool rateAdaptationFastDrop(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	Bandwidth_e bandwidth;
	uint8 currentRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
	uint8 newRateindex;
	RateMask_t tempRaMask;
	RateMask_t rateMaskZero11bRates = LA_PHY_ZERO_11B_RATES_MASK;
	bool changeState = TRUE; 
	const RateObj_t* ratesTable = getRatesTable(laDbDistributionParameter->laStaUspCommon);
	bool fastDropInLowestRate = FALSE;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 uspCount;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionVarPerUsp; 
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);	
#endif	
	MEMCPY(&tempRaMask, &(laDbDistributionParameter->laStaUspCommon->raIndexMask), sizeof(RateMask_t));
	bandwidth = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, isHeGroup);

	if (laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
	{
		// if Mu Group Fixed Rate mode - don't perform the fast drop. 
	 	if (MuGroupFixedRateRequested == FALSE)
	 	{
			changeState = FALSE;
#ifdef ENET_INC_ARCH_WAVE600
			if (isHeGroup)
			{
				LA_SET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap, laDbDistributionParameter->uspIndex, BITMAP_ARRAY_MAX_SIZE_2); //KW_FIX_FW_G Added array bound check
				if((laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] == laDbDistributionParameter->laHeGroupUnique->probingValidationMask.heUspBitmap[0]) && (laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] == laDbDistributionParameter->laHeGroupUnique->probingValidationMask.heUspBitmap[1]))
				{
					laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] = 0x0; /*reset probing validation bitmap*/
					laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] = 0x0;
					changeState = TRUE; /* only when all USPs probe is valid - change the group's state*/
				}
			}
			else
#endif
			{
				laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap |= (0x1<<laDbDistributionParameter->uspIndex);  
				if(laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap == laDbDistributionParameter->laVhtGroupUnique->probingValidationMask)
				{
					laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap = 0x0; /*reset probing validation bitmap*/
					changeState = TRUE; /* only when all USPs probe is valid - change the group's state*/
				}
			}
	 	}
		else
		{
			return fastDropInLowestRate; // if Mu Group Fixed Rate mode - don't perform the fast drop. 
		}
		
	}
	
	/*If not 11B and BW != 20 MHz don't drop down to 11B rates */
	if ((ratesTable[currentRateIndex].laPhyMode != LA_PHY_MODE_11B)  &&  
		(bandwidth != BANDWIDTH_TWENTY) && 
		(FALSE == isHeGroup))
	{
		andOperator64Bit(tempRaMask.raIndexMask64bit,rateMaskZero11bRates.raIndexMask64bit,tempRaMask.raIndexMask64bit);
	}
	
	/*Get lower rate*/
	newRateindex = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, bandwidth, currentRateIndex,tempRaMask, FAST_DROP_RATES_DECREASE, FALSE);

	if (newRateindex != currentRateIndex)
	{
		/*Change rate*/
		rateAdaptationChangeWorkingPoint(laDbDistributionParameter,newRateindex,currentRateIndex);
#ifdef ENET_INC_ARCH_WAVE600
		if (isHeGroup)
		{
			ILOG0_DD("[rateAdaptationFastDrop] for HE group. uspIndex %d executing fast drop to rateIndex %d", laDbDistributionParameter->uspIndex, newRateindex);
			for(uspCount = 0; uspCount < MAX_USP_IN_HE_GROUP; uspCount++)
			{
				if((uspCount != laDbDistributionParameter->uspIndex) && (laDbDistributionParameter->laHeGroupUnique->groupStationIndexes[uspCount] != INVALID_STA_INDEX))
				{
					updateLaDbDistributionParam(&laDbDistributionVarPerUsp,laDbDistributionParameter->stationOrGroupIndex,uspCount, TRUE); 
					setStaDataRate(&laDbDistributionVarPerUsp, WORKING_POINT, estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionVarPerUsp.laStaUspCommon), bandwidth);		
					ILOG0_DD("[rateAdaptationFastDrop] for HE group. uspIndex %d stays at rateIndex %d", uspCount, estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionVarPerUsp.laStaUspCommon));
				}
			}
			heMuWorkingPointIsChanged(laDbDistributionParameter->stationOrGroupIndex);
		}
#endif
		/*Change stat only if don't reach the lowest rate other wize don't change so probe can take us out from the problematic point*/
		if (changeState) 
		{
			laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
		}
	}
	else
	{
		if (((ratesTable[currentRateIndex].laPhyMode == LA_PHY_MODE_11B) || (bandwidth == BANDWIDTH_TWENTY)) 
			&&(FALSE == isHeGroup))
		{
			fastDropInLowestRate = TRUE;
		}
		else
		{
			if (changeState) 
			{
				laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
			}
		}
		if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
		{
			decreaseStaBwLimit(laDbDistributionParameter->stationOrGroupIndex);
		}
	}
	
	/*Change to steady state configuration params*/
	ChangeStaToSteadyStateConfigurationParams(laDbDistributionParameter, FALSE);
//	PowerAdaptationResetDiffPower(stationIndex);
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_DDDD("FAST DROP, StaINdex = %d, BW = %d, currentRateIndex = %d, newRateindex = %d", laDbDistributionParameter->stationOrGroupIndex, bandwidth, currentRateIndex, newRateindex);
#endif
	return fastDropInLowestRate;
}
/****************************************************************************
 **
 ** NAME:        rateAdaptationProbingTimerEvent
 **
 ** PARAMETERS:  
 **stationIndex
 **
 ** RETURN VALUES: 
 **
 **
 ** DESCRIPTION: 
 ** Check estimators indication and trigger Rate Adaptation accordingly
 ** This function runs on interrupt context
 **
 **   
 ***************************************************************************/

void rateAdaptationProbingTimerEvent(StaId stationGroupIndex, uint8 uspIndex, bool isHeGroup)
{
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	
	updateLaDbDistributionParam(&laDbDistributionParameter,stationGroupIndex,uspIndex, isHeGroup);
	ILOG2_D("rateAdaptationProbingTimerEvent , event %d ", PROBING_TIMER_EVENT);
	laStateMachineEnterEvent(&laDbDistributionParameter,PROBING_TIMER_EVENT);
}
/****************************************************************************
 **
 ** NAME:        rateAdaptationCheckStateTimeout
 **
 ** PARAMETERS:  
 **stationIndex
 **
 ** RETURN VALUES: 
 **
 **
 ** DESCRIPTION: 
 **
 **   
 ***************************************************************************/
void rateAdaptationCheckStateTimeout(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	LinkAdaptationConfigurationParams_t* pConfigurationParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	RateAdaptationDataBase_t* pRaDatabase = &(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
	LaStationUspCommonParams_t* pLaStaUspCommon = laDbDistributionParameter->laStaUspCommon;
	int32 filterTimeOffset;
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
	
#ifdef LINK_ADAPTATION_LOGS		
	ILOG0_D("rateAdaptationCheckStateTimeout staId %d",laDbDistributionParameter->stationOrGroupIndex);
#endif
	currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);
	filterTimeOffset = cyclicMinusOperator(currentTsf, laDbDistributionParameter->laStaGroupCommon->raEnterStateTime,TRUE, TSF_RANGE_ALPHA_FILTER); 
	if ((filterTimeOffset >= pConfigurationParams->linkAdaptationStateTo) && ((laDbDistributionParameter->laStaGroupCommon->linkAdaptationState) !=LA_WAIT_FOR_NEXT_PROBE_CYCLE))
	{
		/*Reset RA DB*/
		rateAdaptationResetDb(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, isHeGroup);
		rateAdaptationResetCountersAndThresholds(laDbDistributionParameter);
		resetSlowProbingPriorityBitmaps(laDbDistributionParameter);
		resetSlowProbingHandlerCounters(laDbDistributionParameter);
		pRaDatabase->waitForProbeResponse = FALSE;
		pLaStaUspCommon->probingPointValidationCounter = 0;
		if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
		{
			/*Change to state LA_WAIT_FOR_NEXT_PROBE_CYCLE*/
			laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
			BwAdaptationTo(laDbDistributionParameter);
			BeamformingStateTo(laDbDistributionParameter->stationOrGroupIndex);
			updateOverridePendingDb(ADJUST_RATE_TO_MASK, 0, laDbDistributionParameter); //Call this in order to remove probing bit from TCR
			
		}
		else /*MU*/
		{
#ifdef ENET_INC_ARCH_WAVE600
			if (isHeGroup)
			{
				LA_SET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap, laDbDistributionParameter->uspIndex, BITMAP_ARRAY_MAX_SIZE_2); //KW_FIX_FW_G Added array bound check
				if((laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] == laDbDistributionParameter->laHeGroupUnique->probingValidationMask.heUspBitmap[0]) && (laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] == laDbDistributionParameter->laHeGroupUnique->probingValidationMask.heUspBitmap[1]))
				{
					laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] = 0x0; /*reset probing validation bitmap*/
					laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] = 0x0;
					/*Change to state LA_WAIT_FOR_NEXT_PROBE_CYCLE*/
					laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
					updateOverridePendingDb(ADJUST_RATE_TO_MASK, 0, laDbDistributionParameter); //Call this in order to remove probing bit from TCR
				}
			}
			else
#endif
			{
				laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap |= (0x1<<laDbDistributionParameter->uspIndex);
				if(laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap == laDbDistributionParameter->laVhtGroupUnique->probingValidationMask)
				{
					laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap = 0x0; /*reset probing validation bitmap*/
					/*Change to state LA_WAIT_FOR_NEXT_PROBE_CYCLE*/
					laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
					updateOverridePendingDb(ADJUST_RATE_TO_MASK, 0, laDbDistributionParameter); //Call this in order to remove probing bit from TCR
				}
			}
		}
	}
}

/********************************************************************************
rateAdaptationChangeStaToFixedRate





Description:
------------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
void rateAdaptationChangeStaToFixedRate(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	/*Change to fixed rate state*/
	laStateMachineChangeState(laDbDistributionParameter,LA_SET_FIXED_RATE) ;
	
}
/********************************************************************************
rateAdaptationChangeToAutoRate






Description:
------------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/

void rateAdaptationChangeToAutoRate(StaId stationIndex)
{
	Bandwidth_e staBandwidthMax = GetDataBwLimit(stationIndex, INVALID_MU_USP_INDEX, FALSE);
	uint8 workingPointRateindexBw20;
	uint8 workingPointRateindexBwMax;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 
	PhyMode_e staPhyMode;				
    bool isFixedRateRequested = LinkAdaptationStaDatabase[stationIndex].laStationUnique.fixedRateStationParams.FixedRateRequested;

	if (isFixedRateRequested)
	{
		LinkAdaptationStaDatabase[stationIndex].laStationUnique.fixedRateStationParams.FixedRateRequested = FALSE;
		updateLaDbDistributionParam(&laDbDistributionParameter,stationIndex,INVALID_MU_USP_INDEX, FALSE);
		
		if (getStaState(stationIndex) == STA_STATE_CONNECTED)
		{
			workingPointRateindexBw20 = GetStaWpRateIndexFromHwTcr(stationIndex,BANDWIDTH_TWENTY);
			workingPointRateindexBwMax = GetStaWpRateIndexFromHwTcr(stationIndex,staBandwidthMax);
			/*reset Db before go back to auto rate*/
			linkAdaptationResetStaDb(&laDbDistributionParameter);
			/*Set initial rate for all BWs according to MAX available bandwidth*/
			rateAdaptationChangeWorkingPoint(&laDbDistributionParameter, workingPointRateindexBwMax, workingPointRateindexBwMax);
			initStaBwLimit(stationIndex, workingPointRateindexBw20, OMN_BW_INVALID, LinkAdaptationCommonConfiguration.wlanBandwidthMax);

			cyclicPrefixResetLoop(&laDbDistributionParameter, FALSE);

#ifdef ENET_INC_ARCH_WAVE600D2

	//u0pdate station TID Limit
	StaDb_SetMaxStationTidLimitInAggregation(stationIndex, StaDb_GetMaxStationTidLimitInAggregation(stationIndex));
			
	//change Mgmt aggregations back to its initial value				
#ifdef HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED
						// According to current System requirements - by default, MNG aggregation is disabled, remove this if decided differently
						StaDB_SetHeMngAgg(stationIndex, MULTI_TID_MNG_AGGREGATION_OFF);
#else
						if(StaDb_GetStationAckEnabledSupport(stationIndex) == MULTI_TID_ACK_ENABLED_SUPPORTED)
						{
							StaDB_SetHeMngAgg(stationIndex, MULTI_TID_MNG_AGGREGATION_ON);
						}
						else
						{
							StaDB_SetHeMngAgg(stationIndex, MULTI_TID_MNG_AGGREGATION_OFF);
						}
#endif //HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED
#endif //ENET_INC_ARCH_WAVE600D2
			
			laStateMachineChangeState(&laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE) ;
			//call QAMplus to indicate add HT station
			staPhyMode = ExtractHighestPhyModeFromRaMask(laDbDistributionParameter.laStaUspCommon->raIndexMask, laDbDistributionParameter.laStaUspCommon->staTransmissionParams.heSta);
			QAMplus_AddStationReq(stationIndex,staPhyMode);
#ifndef BW_ADAPTATION_ACTIVE
			rateAdaptationEnableDisableStaSlowLoop(stationIndex,SLOW_PROBING_BW, DISABLE_LOOP);
#endif
		}
		else
		{
			laStateMachineChangeState(&laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE) ;
		}
	}
}


/********************************************************************************
setManagementRatesInRaDB


Description:
----------
Find the management rate according to the data rate
The management rate derived from the data rate using managementRateOffset param 

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/

static void setManagementRatesInRaDB(StaId staIndex, uint8* pWpManagementRatesTable)
{
	uint8 wpDataRatesIndex;
	int8 tempManagementSortRateindex;
//	uint8* wpManagementRateIndex = LinkAdaptationStaDatabase[staIndex].rateAdaptationDataBase.wpManagementRateIndex;
	uint8 highestFromBasicRates;
	uint8 lowestFromBasicRates;
	int8 mangementRateOffset = (int8)LinkAdaptationStaDatabase[staIndex].laStaGroupCommon.pLinkAdaptationConfigurationParams->managementRateOffset;
	Bandwidth_e bandwidth;
	RateMask_t basicRatesMask;
	const RateObj_t* ratesTable = getRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	const uint8* sortedRatesTable = getSortedRatesTable(&LinkAdaptationStaDatabase[staIndex].laStaUspCommon);
	
	basicRatesMask.raIndexMask64bit[0] = StaDbSwEntries[staIndex].basicRates;
	basicRatesMask.raIndexMask64bit[1] = 0;
	
	highestFromBasicRates = getHighestRateinMask(staIndex, basicRatesMask,BANDWIDTH_TWENTY,0,PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX);
	lowestFromBasicRates = getLowestRateinMask(staIndex, basicRatesMask,BANDWIDTH_TWENTY,0,PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX);

	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax); bandwidth++)
	{
		wpDataRatesIndex = GetStaWpRateIndexFromHwTcr(staIndex,bandwidth);

		tempManagementSortRateindex = ratesTable[wpDataRatesIndex].sortPhyRateSameBwIndex - mangementRateOffset;
		if (tempManagementSortRateindex > ratesTable[highestFromBasicRates].sortPhyRateSameBwIndex)
		{
			pWpManagementRatesTable[bandwidth] = highestFromBasicRates;
		}
		else
		{
			pWpManagementRatesTable[bandwidth] = sortedRatesTable[MAX(tempManagementSortRateindex, ratesTable[lowestFromBasicRates].sortPhyRateSameBwIndex)];
		}
	}
}
/********************************************************************************
rateAdaptationAdjustRateToMask



Description:
----------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
void rateAdaptationAdjustRateToMask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
//	RateAdaptationDataBase_t* pRaDatabase = &(pLaDatabase->rateAdaptationDataBase);
	uint8 rateIndex;
	uint8 newRateindex;
					
	/*Get rate from the highest BW*/
	rateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
	//("[OMN][rateAdaptationAdjustRateToMask] rateIndex = %d",rateIndex);
	laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE) ;
	/*Highest BW rate is not in mask, change to the closeset rate in mask*/
	newRateindex = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, LinkAdaptationCommonConfiguration.wlanBandwidthMax, rateIndex, laDbDistributionParameter->laStaUspCommon->raIndexMask, 0, FALSE);
	//("[OMN][rateAdaptationAdjustRateToMask] newRateindex = %d",newRateindex);
	/*Update rate adaptation DB with the new rate*/
	rateAdaptationChangeWorkingPoint(laDbDistributionParameter,newRateindex,rateIndex);
}

#ifdef DEBUG_UM_INTERFACE
/********************************************************************************



Description:
----------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/

void rateAdaptationEnDisSlowProbingLoops(uint8 slowProbingMask)
{
	slowProbingEnabledLoopdBitmap = slowProbingMask;
}
#endif

/**********************************************************************************

prepareStaTcrDataRate 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static bool prepareStaTcrDataRate(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter, LaTcrModificationStruct_t* pTcrParams, RaFastProbingPoint_e newFastProbingPoint)
{
	LaStationUspCommonParams_t* laStaUspCommon = pLaDbDistributionParameter->laStaUspCommon;
	uint8 nextProbingPointIndex;
	Bandwidth_e bandwidth;
	Bandwidth_e mainBw; 
	uint8 mainBwRateIndex;
	bool isFastProbingChanged = FALSE;
	uint8 higherBwRateOffset =  pLaDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->higherBwOffsetRate;
	bool isHeGroup = (pLaDbDistributionParameter->laHeGroupUnique != NULL);	

	mainBw = GetDataBwLimit(pLaDbDistributionParameter->stationOrGroupIndex,pLaDbDistributionParameter->uspIndex, isHeGroup);
	nextProbingPointIndex = laStaUspCommon->fastProbingPointEstimatorsIndexes[newFastProbingPoint];
	/*Set main BW rate*/
	if (nextProbingPointIndex  < TOTAL_NUMBER_OF_RATES_IN_DB)
	{
		mainBwRateIndex = laStaUspCommon->fastProbingEstimators[nextProbingPointIndex].rateIndex;
		isFastProbingChanged = TRUE; // ask Eran gonen if this field is TRUE when AddSta from link adaptation is invoked. 
	}
	else
	{
		mainBwRateIndex = INVALID_RATE_INDEX_VAL;
	}
	ASSERT(laStaUspCommon->pRatesTablesDb != NULL);

	/*Set rate for all BWs*/
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= (LinkAdaptationCommonConfiguration.wlanBandwidthMax); bandwidth++)
	{
		ASSERT(bandwidth < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
		if ((mainBwRateIndex != INVALID_RATE_INDEX_VAL) && (bandwidth!=mainBw))
		{
			if (bandwidth < mainBw)
			{
				/*Set the same rate for lower BW than the data BW limit*/
				pTcrParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex = rateAdaptationGetLowerRateIndex(pLaDbDistributionParameter, bandwidth,mainBwRateIndex,pLaDbDistributionParameter->laStaUspCommon->raIndexMask,0, FALSE);
			}
			else if(bandwidth > mainBw)
			{
				/*Set lower rate to higher than data BW limit*/
				pTcrParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex = rateAdaptationGetLowerRateIndex(pLaDbDistributionParameter, bandwidth,mainBwRateIndex,pLaDbDistributionParameter->laStaUspCommon->raIndexMask,higherBwRateOffset, FALSE);
			}
		}
		else
		{
			/*Main BW*/
			pTcrParams->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex = mainBwRateIndex;
		}
	}
	return isFastProbingChanged;
}
/**********************************************************************************

rateAdaptation_RateIsIncreased 


Description:
------------
This function checks if the new rate is higher than the old rate  


Input: 
-----

Returns:
--------
	bool - TRUE: rate is increased 
	
**********************************************************************************/
bool rateAdaptation_RateIsIncreased(uint8 newRate, uint8 oldRate, const RateObj_t* ratesTable)
{
	bool rateIncreasd = FALSE;

	if (ratesTable[newRate].sortPhyRateSameBwIndex > ratesTable[oldRate].sortPhyRateSameBwIndex)
	{
		rateIncreasd = TRUE;
	}
	
	return rateIncreasd;
}

/**********************************************************************************

setStaDataRate 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

static bool setStaDataRate(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter, RaFastProbingPoint_e newFastProbingPoint, uint8 oldRate, Bandwidth_e bw)
{
	bool probingPointChanged;
	LaTcrModificationStruct_t tcrModificationParams;

	uint8 newRateindex = INVALID_ESTIMATORS_DB_INDEX_VAL;
	bool isRateInc = TRUE;	
	uint32 maxRatePerBwPerGroup[3] = {0,0,0};
	uint32 numberOfNss[3] = {0,0,0};
	int8 isMuPowerOffsetTable = FALSE;

	memset(&tcrModificationParams,0,sizeof(LaTcrModificationStruct_t));

	/*Fill control parameters of hwTcrData*/
	tcrModificationParams.controlParams.firstBwToUpdate = BANDWIDTH_TWENTY;
	tcrModificationParams.controlParams.lastBwToUpdate= LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	tcrModificationParams.controlParams.isVhtSta = pLaDbDistributionParameter->laStaUspCommon->staTransmissionParams.vhtSta;
	tcrModificationParams.controlParams.changeProbingPoint = (newFastProbingPoint == WORKING_POINT ? FALSE : TRUE);
	tcrModificationParams.controlParams.fastProbing = tcrModificationParams.controlParams.changeProbingPoint;
	tcrModificationParams.controlParams.packetType = LA_PACKET_TYPE_DATA;
	tcrModificationParams.controlParams.changeToRatePlusOne = FALSE;
	tcrModificationParams.controlParams.staIndex= pLaDbDistributionParameter->stationOrGroupIndex;
	tcrModificationParams.controlParams.uspIndex = pLaDbDistributionParameter->uspIndex; 
#ifdef ENET_INC_ARCH_WAVE600
	tcrModificationParams.controlParams.isHeGroup = (pLaDbDistributionParameter->laHeGroupUnique != NULL);
#endif
	probingPointChanged = prepareStaTcrDataRate(pLaDbDistributionParameter, &tcrModificationParams,newFastProbingPoint);

	isMuPowerOffsetTable = isMuPowerOffsetTableUsed();
	//ILOG0_D("setStaDataRate isRateInc = %d",isMuPowerOffsetTable);

	if(isMuPowerOffsetTable == TRUE)
	{
	//	ILOG0_D("setStaDataRate pLaDbDistributionParameter->uspIndex = %d",pLaDbDistributionParameter->uspIndex);
		if(pLaDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX)
		{
			findMaxRateInMuGroup(tcrModificationParams.controlParams.staIndex,&maxRatePerBwPerGroup[0],&numberOfNss[0]);
			GetMuPowerValInRegulationLimit(BANDWIDTH_TWENTY,maxRatePerBwPerGroup[0],numberOfNss[0],tcrModificationParams.controlParams.staIndex);
			GetMuPowerValInRegulationLimit(BANDWIDTH_FOURTY,maxRatePerBwPerGroup[1],numberOfNss[1],tcrModificationParams.controlParams.staIndex);
			GetMuPowerValInRegulationLimit(BANDWIDTH_EIGHTY,maxRatePerBwPerGroup[2],numberOfNss[2],tcrModificationParams.controlParams.staIndex);
		}
	}

	if (probingPointChanged == TRUE)
	{
		if(pLaDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
		{
			newRateindex = tcrModificationParams.tcrParams.bwDependedTcrValsTable[bw].rateindex;
			
			/* Check if new rate is higher than old rate */
			isRateInc = rateAdaptation_RateIsIncreased(newRateindex, oldRate, getRatesTable(pLaDbDistributionParameter->laStaUspCommon));
			ILOG2_DDD("setStaDataRate isRateInc = %d newRateindex = %d oldRate = %d ",isRateInc,newRateindex, oldRate);
			/* Change rate and time limit in correct order: if rate is increased then first change the rate (and vice versa) */  
			if (isRateInc == TRUE)
			{
				/*Fill TCR params in HW DB*/
				modifyStaTcrsParamsReq(&tcrModificationParams,rateModificationFunc);
				clientIsolationRateIsChanged(pLaDbDistributionParameter, newFastProbingPoint,oldRate, newRateindex, bw);
			}
			else
			{
				clientIsolationRateIsChanged(pLaDbDistributionParameter,newFastProbingPoint, oldRate, newRateindex, bw);

				/*Fill TCR params in HW DB*/
				modifyStaTcrsParamsReq(&tcrModificationParams,rateModificationFunc);
			}
		}
		else
		{
			/*Fill TCR params in HW DB*/
			modifyStaTcrsParamsReq(&tcrModificationParams,rateModificationFunc);
		}
	}
	
	return probingPointChanged;
}

static void findMaxRateInMuGroup(StaId staIndex,uint32 maxRatePerBW[3],uint32 numberOfNss[3])
{
	uint8 dataTcrTempValue;
	uint32* dataTcr32Bit;
	uint8* dataTcr40Mhz;
	uint8* dataTcr80Mhz;
	uint32 stationIndex;
	uint32 uspIndex;
	uint32 rateIndex;
	uint32 vhtMcs;
	const RateObj_t* ratesTable; 
#ifdef ENET_INC_ARCH_WAVE600
	dataTcr32Bit = (((uint32*)&(GroupDbHwEntries[staIndex].common.word12))); 
#else
	dataTcr32Bit = (((uint32*)&(GroupDbHwEntries[staIndex].word20))); 
#endif
	
	for(uspIndex = 0; uspIndex < GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; uspIndex++)
	{
		dataTcr40Mhz = (uint8*)dataTcr32Bit; 
		dataTcr40Mhz++;
		dataTcr80Mhz = dataTcr40Mhz;
		dataTcr80Mhz++;
		
		stationIndex = LinkAdaptationVhtGroupDatabase[staIndex].laVhtGroupUnique.groupStationIndexes[uspIndex];
		if(stationIndex != INVALID_STA_INDEX)
		{
			ratesTable = getRatesTable(&LinkAdaptationVhtGroupDatabase[staIndex].laStaUspCommon[uspIndex]);
			dataTcrTempValue = GET_SIX_LS_BITS_MU(*dataTcr32Bit);
			rateIndex = convertTcr2RateIndex(PHY_MODE_11AC,dataTcrTempValue);
			vhtMcs = ratesTable[rateIndex].vhtHeMcs;
			if(vhtMcs > maxRatePerBW[0])
			{
				maxRatePerBW[0] = vhtMcs;
				numberOfNss[0] = ratesTable[rateIndex].numberOfNss;
				//ILOG0_DDD("eeeeeeeeeee stationIndex = %d maxRatePerBW[0] = %d numberOfNss[0] = %d ",stationIndex,maxRatePerBW[0], numberOfNss[0]);
			}

			
			dataTcrTempValue = GET_SIX_LS_BITS_MU(*dataTcr40Mhz);
			rateIndex = convertTcr2RateIndex(PHY_MODE_11AC,dataTcrTempValue);
			vhtMcs = ratesTable[rateIndex].vhtHeMcs;
			if(vhtMcs > maxRatePerBW[1])
			{
				maxRatePerBW[1] = vhtMcs;
				numberOfNss[1] = ratesTable[rateIndex].numberOfNss;
				//ILOG0_DDD("eeeeeeeeeee stationIndex = %d maxRatePerBW[1] = %d numberOfNss[1] = %d ",stationIndex,maxRatePerBW[1], numberOfNss[1]);
			}
			
			dataTcrTempValue = GET_SIX_LS_BITS_MU(*dataTcr80Mhz);
			rateIndex = convertTcr2RateIndex(PHY_MODE_11AC,dataTcrTempValue);
			vhtMcs = ratesTable[rateIndex].vhtHeMcs;
			if(vhtMcs > maxRatePerBW[2])
			{
				maxRatePerBW[2] = vhtMcs;
				numberOfNss[2] = ratesTable[rateIndex].numberOfNss;
				//ILOG0_DDD("eeeeeeeeeee stationIndex = %d maxRatePerBW[2] = %d numberOfNss[2] = %d ",stationIndex,maxRatePerBW[2], numberOfNss[2]);
			}
			
		}
		else
		{
			dataTcrTempValue = 0;
		}
		dataTcr32Bit++;
		
	}
}


uint8 GetMuPowerValInRegulationLimit(Bandwidth_e bandwidth,uint8 rate,uint32 numberOfNss,StaId staIndex)
{
#ifdef ENET_INC_ARCH_WAVE600
	UNUSED_PARAM(bandwidth);
	UNUSED_PARAM(rate);
	UNUSED_PARAM(numberOfNss);	
	UNUSED_PARAM(staIndex);	
	FATAL("GetMuPowerValInRegulationLimit");
	return FALSE;
#else
	uint8 power;
	uint8 tcrPowerVal = 0;
	Tcr0_t* pTcr0Data = NULL;

	//uint8 activateAntennaBitmap = AntennaSelectionGetActivatedAntennasBitmap();
	int8 muMimoPowerLimit	= getMuPowerLimit(bandwidth);

	/*Calculate default power val*/
	power =  powerAdaptationGetultimateEvm(bandwidth) + getRatePowerOffsetForGroupMu(numberOfNss,rate);
	//ILOG0_DDD("eeeeeee powerAdaptationGetultimateEvm = %d getRatePowerOffsetForGroupMu = %d power = %d ",powerAdaptationGetultimateEvm(bandwidth),getRatePowerOffsetForGroupMu(numberOfNss,rate), power);
	power = MIN(muMimoPowerLimit,power);

	/*Convert power to product TCR power val*/
	//tcrPowerVal = convertPower2TcrVal(bandwidth,power);
	pTcr0Data = (((Tcr0_t*)&(GroupDbHwEntries[staIndex].word7))); 
	if (bandwidth == BANDWIDTH_FOURTY)
	{
		pTcr0Data++;
	}
	if (bandwidth == BANDWIDTH_EIGHTY)
	{
		pTcr0Data++;
		pTcr0Data++;
	}

	//ILOG0_DDD("eeeeeee  pTcr0Data->rfPower = %d pTcr0Data->rfPower = %d bfMode = %d ", pTcr0Data->rfPower,pTcr0Data->bw, pTcr0Data->bfMode);
	pTcr0Data->rfPower = power;
	//ILOG0_DDD("eeeeeee  pTcr0Data->rfPower = %d pTcr0Data->rfPower = %d bfMode = %d ", pTcr0Data->rfPower,pTcr0Data->bw, pTcr0Data->bfMode);


	return tcrPowerVal;
#endif //ENET_INC_ARCH_WAVE600

}


/**********************************************************************************

setStaManagementRate 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/
static void setStaManagementRate(StaId staIndex, LaStationUspCommonParams_t* pLaStaUspCommon)
{
	Bandwidth_e bandwidth;
	LaTcrModificationStruct_t tcrModificationParams;
	uint8 pWpManagementRatesTable[LA_NUM_OF_BANDWIDTH];
	
	memset(&tcrModificationParams,0,sizeof(LaTcrModificationStruct_t));

	setManagementRatesInRaDB(staIndex, pWpManagementRatesTable);
	
	/*Fill control parameters of hwTcrData*/
	tcrModificationParams.controlParams.firstBwToUpdate = BANDWIDTH_TWENTY;
	tcrModificationParams.controlParams.lastBwToUpdate= LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	tcrModificationParams.controlParams.isVhtSta = pLaStaUspCommon->staTransmissionParams.vhtSta;
	tcrModificationParams.controlParams.changeProbingPoint=  FALSE;
	tcrModificationParams.controlParams.packetType = LA_PACKET_TYPE_MANAGEMENT;
	tcrModificationParams.controlParams.changeToRatePlusOne = FALSE;
	tcrModificationParams.controlParams.staIndex= staIndex;
	tcrModificationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX;
#ifdef ENET_INC_ARCH_WAVE600
	tcrModificationParams.controlParams.isHeGroup = FALSE;
#endif
	for (bandwidth = BANDWIDTH_TWENTY; bandwidth <= LinkAdaptationCommonConfiguration.wlanBandwidthMax; bandwidth++)
	{
		ASSERT(bandwidth < LA_NUM_OF_BANDWIDTH); //KW_FIX_FW_G
		tcrModificationParams.tcrParams.bwDependedTcrValsTable[bandwidth].rateindex = pWpManagementRatesTable[bandwidth];
	}
	
	/*Fill hwTcrData from HW tcrs*/
	modifyStaTcrsParamsReq(&tcrModificationParams,rateModificationFunc);
}

uint8 AddManagementRateToMask(StaId stationIndex, LinkAdaptationStaDatabase_t *pLinkAdaptationDb, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8 		pWpManagementRatesTable[LA_NUM_OF_BANDWIDTH];
	uint8 		rate = 0;
	RateMask_t	raMask = laDbDistributionParameter->laStaUspCommon->raIndexMaskOrg;

	setManagementRatesInRaDB(stationIndex, pWpManagementRatesTable);

	rate = pWpManagementRatesTable[BANDWIDTH_TWENTY];

	/* Check if rate exists in Org mask: in case it exists it will not be removed from mask when set filter command is sent (linkAdaptationSetFilter) */
	if(isBitInMask64Bit(rate, raMask.raIndexMask64bit) == 0)
	{
		/* Rate does not exist in Org: set the flag as true */
		LinkAdaptationStaDatabase[stationIndex].laStationUnique.isSnifferEapolWaActive = TRUE;
		/* Add the rate to both masks */
		LA_SET_BIT_IN_BITMAP(pLinkAdaptationDb->laStaUspCommon.raIndexMask.raIndexMask64bit, rate, RATES_BIT_MASK_SIZE); //KW_FIX_FW_G Added array bound check
		LA_SET_BIT_IN_BITMAP(pLinkAdaptationDb->laStaUspCommon.raIndexMaskOrg.raIndexMask64bit, rate, RATES_BIT_MASK_SIZE); //KW_FIX_FW_G Added array bound check
	}

	return(rate);
}

/**********************************************************************************

cyclicPrefixSetCpParams 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
static void rateModificationFunc(LaTcrModificationStruct_t* tcrModifcationParamsIn, LaTcrModificationStruct_t* tcrModifcationParamsOut)
{
	Bandwidth_e bandwidth;
	ASSERT(tcrModifcationParamsIn->controlParams.lastBwToUpdate < MAX_POSSIBLE_NUM_OF_BW); //KW_FIX_FW_G
	for(bandwidth = tcrModifcationParamsIn->controlParams.firstBwToUpdate; bandwidth <=tcrModifcationParamsIn->controlParams.lastBwToUpdate; bandwidth++)
	{
		if (tcrModifcationParamsIn->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex != INVALID_RATE_INDEX_VAL)
		{
			tcrModifcationParamsOut->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex = tcrModifcationParamsIn->tcrParams.bwDependedTcrValsTable[bandwidth].rateindex;
		}
	}
}

/********************************************************************************
rateAdaptationEnableDisableStaSlowLoop




Description:
----------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
void rateAdaptationEnableDisableStaSlowLoop(StaId staId, SlowProbingTasks_e loopType, RaEnableDisableSlowLoop_e enableDisable)
{
	LaStationGroupCommonParams_t* pLaStaGroupCommon = &(LinkAdaptationStaDatabase[staId].laStaGroupCommon);
	if (enableDisable == ENABLE_LOOP)
	{
		/*Clear bit from two priorities masks*/
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] &= ~(1<<loopType);
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] &= ~(1<<loopType);
		/*Set bit in two priorities masks*/
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] |= (SLOW_PROBING_PRIORITY0_ENABLED_BITMAP & (1<<loopType));
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] |= (SLOW_PROBING_PRIORITY1_ENABLED_BITMAP & (1<<loopType));
	}
	else
	{
	
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] &= ~(1<<loopType);
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] &= ~(1<<loopType);
	}
	
	ILOG0_DDDD("rateAdaptationEnableDisableStaSlowLoop, enableDisable = %d,loopType = %d, probingTaskBitmap[Pr0] = %d, slowProbingTaskBitmap[Pr1] = %d",
			enableDisable,loopType,
			pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0],
			pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1]);
}

/********************************************************************************
rateAdaptationEnableDisableGroupSlowLoop





Description:
----------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
void rateAdaptationEnableDisableGroupSlowLoop(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e loopType, RaEnableDisableSlowLoop_e ebanleDisable)
{
	LaStationGroupCommonParams_t* pLaStaGroupCommon = laDbDistributionParameter->laStaGroupCommon;

	if (ebanleDisable == ENABLE_LOOP)
	{
		/*Clear bit from two priorities masks*/
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] &= ~ (1<<loopType);
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] &= ~ (1<<loopType);
		/*Set bit from two priorities masks*/
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] |= (SLOW_PROBING_PRIORITY0_ENABLED_BITMAP & (1<<loopType));
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] |= (SLOW_PROBING_PRIORITY1_ENABLED_BITMAP & (1<<loopType));
	}
	else
	{
	
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] &= ~(1<<loopType);
		pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] &= ~(1<<loopType);
	}
}

/********************************************************************************
setCurrentSlowProbingTaskInDb





Description:
----------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
void setCurrentSlowProbingTaskInDb(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e currentSlowProbingTask)
{
	laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTask = currentSlowProbingTask;
}

/********************************************************************************
getSlowProbingTaskFromBitmap






Description:
----------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
SlowProbingTasks_e getSlowProbingTaskFromBitmap(uint8 slowProbingCurrentTaskBitmap)
{
	SlowProbingTasks_e nextTask = SLOW_PROBING_FIRST_TASK;
	uint8 nextTaskFound = 0;
	
	while ((nextTaskFound == 0) && (slowProbingCurrentTaskBitmap != 0))
	{
		nextTaskFound = (slowProbingCurrentTaskBitmap & 0x1);
		slowProbingCurrentTaskBitmap>>=1;
		nextTask++;
	}
	nextTask--;

	return nextTask;
}

void handleSlowProbingIterationCounter(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e slowProbingTask)
{
	uint8* iterationConuter;
	uint8* slowProbingMaxIter;

	ASSERT(slowProbingTask < SLOW_PROBING_TASKS_NUM); //KW_FIX_FW_G

	iterationConuter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.iterationCounter[slowProbingTask]);
	slowProbingMaxIter = &(laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->slowProbingMaxNumOfIterations[slowProbingTask]);

	(*iterationConuter)++;			//Count iteration per slow loop
	
	/*If iterations per loop reached max clear bit from bitmap and continue with next task*/
	if (*iterationConuter >= ((*slowProbingMaxIter)))
	{
		disableTaskFromCurrentCycle(laDbDistributionParameter, slowProbingTask);
	}

}

bool checkMaxSlowProbingIterations(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8* totalIterationsCounter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.totalIterationsPerCycle);
	uint8* maxSlowProbingIterationsPerCycle;
	bool lastIteration = FALSE;
	RaProbingIndication_e probingIndication = laDbDistributionParameter->laStaGroupCommon->probingIndication;
	RaSlowProbingPriority_e nextSlowProbePriority = (RaSlowProbingPriority_e)(probingIndication - SLOW_PROBING_PRIORITY0_INDICATION);

	(*totalIterationsCounter)++;
	if (nextSlowProbePriority == SLOW_PROBING_PRIORITY1)
	{
		maxSlowProbingIterationsPerCycle =  &(laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxSlowProbingIterationsPerCyclePr1);
	}
	else
	{
		maxSlowProbingIterationsPerCycle =  &(laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->maxSlowProbingIterationsPerCyclePr0);
	}

	if ((*totalIterationsCounter) >= ((*maxSlowProbingIterationsPerCycle)))
	{
		*totalIterationsCounter = 0;
		laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
		lastIteration = TRUE;
	}
	return lastIteration;
}

bool disableTaskFromCurrentCycle(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e slowProbingTask)
{
	uint8* iterationConuter;
	uint8* slowProbingCurrentTaskBitmap;
	bool isCurrentTaskBitMapEmpty;

	ASSERT(slowProbingTask < SLOW_PROBING_TASKS_NUM); //KW_FIX_FW_G

	iterationConuter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.iterationCounter[slowProbingTask]);
	slowProbingCurrentTaskBitmap = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTaskBitmap);


	
	(*slowProbingCurrentTaskBitmap) &= (~(1<<(slowProbingTask)));
	*iterationConuter = 0;

	if ((*slowProbingCurrentTaskBitmap) != 0)
	{
		isCurrentTaskBitMapEmpty = FALSE;
	}
	else
	{
		isCurrentTaskBitMapEmpty = TRUE;
	}
	
	return isCurrentTaskBitMapEmpty;
}

bool updateNonEffectiveLoopCounter(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e slowProbingTask)
{
	uint8* pNonEffectiveLoopCounter;
	uint8* pSlowProbingNonEffectiveLoopTh;
	uint8* slowProbingTaskBitmapPr0;
	uint8* slowProbingTaskBitmapPr1;
	bool nonEffectiveLoop = FALSE;

	ASSERT(slowProbingTask < SLOW_PROBING_TASKS_NUM);//KW_FIX_FW_G

	pNonEffectiveLoopCounter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingNotEffectveLoopCounter[slowProbingTask]);
	pSlowProbingNonEffectiveLoopTh = &(laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->slowProbingNonEffectiveLoopTh[slowProbingTask]);
	slowProbingTaskBitmapPr0 = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0]);
	slowProbingTaskBitmapPr1 = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1]);
	(*pNonEffectiveLoopCounter)++;
	
	/*If loop is not effective change its priority to lower priority*/
	if ((*pNonEffectiveLoopCounter) >= ((*pSlowProbingNonEffectiveLoopTh)))
	{
		nonEffectiveLoop = TRUE;
		(*pNonEffectiveLoopCounter) = 0;
		(*slowProbingTaskBitmapPr0) &= (~(1<< slowProbingTask)); //remove task from priority 0 bitmap
		(*slowProbingTaskBitmapPr1) |= (1<< slowProbingTask); //set task in priority 1 bitmap

	}
	
	return nonEffectiveLoop;
}


void switchToHigherPriority (LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e slowProbingTask)
{
	uint8* pNonEffectiveLoopCounter;
	uint8* slowProbingTaskBitmapPr0;
	uint8* slowProbingTaskBitmapPr1;

	ASSERT(slowProbingTask < SLOW_PROBING_TASKS_NUM);//KW_FIX_FW_G 

	pNonEffectiveLoopCounter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingNotEffectveLoopCounter[slowProbingTask]);
	slowProbingTaskBitmapPr0 = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0]);
	slowProbingTaskBitmapPr1 = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1]);

	(*pNonEffectiveLoopCounter) = 0;
	(*slowProbingTaskBitmapPr0) &= (~(1<< slowProbingTask)); //remove task from priority 0 bitmap
	(*slowProbingTaskBitmapPr1) |= (1<< slowProbingTask); //set task in priority 1 bitmap

}

void resetNonEffectiveLoopCounter(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e slowProbingTask)
{
	uint8* pNonEffectiveLoopCounter;

	ASSERT(slowProbingTask < SLOW_PROBING_TASKS_NUM);//KW_FIX_FW_G

	pNonEffectiveLoopCounter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingNotEffectveLoopCounter[slowProbingTask]);

	*pNonEffectiveLoopCounter = 0;
}

void resetSlowProbingPriorityBitmaps(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	SlowProbingTasks_e slowProbingTask;

	for (slowProbingTask = SLOW_PROBING_FIRST_TASK;slowProbingTask <= SLOW_PROBING_LAST_TASK; slowProbingTask++)
	{
		changeTaskToDefaultPriority(laDbDistributionParameter, slowProbingTask);
	}
}

void changeTaskToDefaultPriority(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, SlowProbingTasks_e slowProbingTask)
{
	LaStationGroupCommonParams_t* pLaStaGroupCommon = laDbDistributionParameter->laStaGroupCommon;
	uint8* pslowProbingTaskBitmap = pLaStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap;
	/*Set bit in default priority only if it was set in one of the 2 priorities - AND with the other priority*/
	if ((pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] & (1<<slowProbingTask)) || (pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] & (1<<slowProbingTask)))
	{
		pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] &=(~(1<<slowProbingTask));
		pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] &=(~(1<<slowProbingTask));

		pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] |= (SLOW_PROBING_PRIORITY0_ENABLED_BITMAP & (1<<slowProbingTask));
		pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] |= (SLOW_PROBING_PRIORITY1_ENABLED_BITMAP & (1<<slowProbingTask));

			
	} 

	ILOG0_DD("changeTaskToDefaultPriority, pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] = %X, pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY1] = %X",
			pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY0], pslowProbingTaskBitmap[SLOW_PROBING_PRIORITY1]);
}

void resetSlowProbingHandlerCounters(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	SlowProbingTasks_e slowProbingTask;
	laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.totalIterationsPerCycle = 0;

	for (slowProbingTask = SLOW_PROBING_FIRST_TASK;slowProbingTask <= SLOW_PROBING_LAST_TASK; slowProbingTask++)
	{
		laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingNotEffectveLoopCounter[slowProbingTask] = 0;
		laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.iterationCounter[slowProbingTask] = 0;
	}
}

uint8 rateAdaptationGetSlowProbingEnBitmap(void)
{
	return slowProbingEnabledLoopdBitmap;
}

