/***********************************************************************************
 File:		TurboRates.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "stringLibApi.h"
#include "LinkAdaptation.h"
#include "PowerAdaptation.h"
#include "HdkGlobalDefs.h"
#include "lm_StaDatabase.h"
#include "AntennaSelection.h"
#include "mt_sysdefs.h"
#include "LinkAdaptation_StateMachine.h"
#include "Pac_Api.h"
#include "RateAdaptation.h"
#include "Estimators.h"
#include "ShramStatistics.h"
#include "loggerAPI.h"
#include "Estimators.h"
#include "CommonRamLinkAdaptation.h"
#include "TurboRates.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 21


#define TURBO_RATES_NUMBER_OF_RATES 4
#define DELTA_RSSI_TURBO_RATES 3
#define TURBO_RATES_SUCCESS_PER 95 //%
#define TURBO_RATES_INVALID_INDEX MAX_UINT8
#define TURBO_RATES_SET_ALL_RATES_BITMASK 0xF
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct TurboRatesParameters
{
	uint8 rateIndex;
	uint8 bw;
}TurboRatesParameters_t;
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static uint8 isRateInturboRatesValidBitmapAndNotValid(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 rateIndex, Bandwidth_e bandwidth);
static bool setTurboRatesInStaTcr(uint8 staIndex, uint8 uspIndex, uint8 rateIndex, Bandwidth_e bandwidth, bool changeProbingPoint);
static void turboRatesModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
TurboRatesParameters_t turboRatesParams[TURBO_RATES_NUMBER_OF_RATES] = 
{
	/*bitmapIndex, RateIndex, BW*/
	{24, BANDWIDTH_TWENTY},	/*20M 1 NSS MCS 9*/
	{34, BANDWIDTH_TWENTY},	/*20M 2 NSS MCS 9*/
	{54, BANDWIDTH_TWENTY},	/*20M 4 NSS MCS 9*/
	{41, BANDWIDTH_EIGHTY}	/*80M 3 NSS MCS 6*/
 };

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

PowerAdaptationInit 


Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
#if (defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void TurboRatesInit (LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	laDbDistributionParameter->laStaUspCommon->turboRatesDb.probingTurboRateIndex = TURBO_RATES_INVALID_INDEX;
	laDbDistributionParameter->laStaUspCommon->turboRatesDb.lastRssi= RX_THRESHOLD_DEFAULT_VALUE;
	laDbDistributionParameter->laStaUspCommon->turboRatesDb.state = TURBO_RATES_PROBING_ENABLE;
	laDbDistributionParameter->laStaUspCommon->turboRatesDb.turboRatesValidBitmap = 0;
}

#if (defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

void TurboRatesSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint)
{
	TurboRatesDataBase_t* pTurboRatesDb = &(laDbDistributionParameter->laStaUspCommon->turboRatesDb);
	Bandwidth_e bandwidth = GetDataBwLimit(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex, FALSE);
	uint8 currentRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
	uint8 newRateIndex;
	bool removeTaskFromCurrentLoopCycle = TRUE;
	uint8 turboRatesValidBitmap;
	uint8 turboRateProbingIndex;
	ASSERT (laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX); /*Loop is not active in MU*/

	UNUSED_PARAM(stayWithLastProbingPoint);
	ILOG0_V("TurboRatesSetNextProbingPoint, START_PROBING");
	SLOG0(0, 0, TurboRatesDataBase_t, pTurboRatesDb);
	SLOG0(0, 0, LinkAdaptationDatabaseDistributionPack_t, laDbDistributionParameter);
	if (laDbDistributionParameter->laStaUspCommon->staTransmissionParams.vhtSta == TRUE)
	{
		/*Store turboRatesValidBitmap value*/
		turboRatesValidBitmap = pTurboRatesDb->turboRatesValidBitmap;
		/*Set all turbo rates temporary in order to enable them for probing  (use by rateAdaptationGetHigherRateIndex and rateAdaptationGetLowerRateIndex functions)*/
		pTurboRatesDb->turboRatesValidBitmap = TURBO_RATES_SET_ALL_RATES_BITMASK;
		newRateIndex = rateAdaptationGetHigherRateIndex(laDbDistributionParameter, bandwidth, currentRateIndex,laDbDistributionParameter->laStaUspCommon->raIndexMask,1,FALSE);
		ILOG0_D("TurboRatesSetNextProbingPoint, HigherRateIndex = %d", newRateIndex);

		if (newRateIndex == currentRateIndex)
		{
			newRateIndex = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, bandwidth, currentRateIndex,laDbDistributionParameter->laStaUspCommon->raIndexMask,1,FALSE);
			ILOG0_D("TurboRatesSetNextProbingPoint, lowerRateIndex = %d", newRateIndex);
		}
		/*Restore turboRatesValidBitmap value*/
		pTurboRatesDb->turboRatesValidBitmap = turboRatesValidBitmap;

		if (newRateIndex != currentRateIndex)
		{
			turboRateProbingIndex = isRateInturboRatesValidBitmapAndNotValid(laDbDistributionParameter, newRateIndex, bandwidth);
			if ((turboRateProbingIndex != TURBO_RATES_INVALID_INDEX) && 
				(isBitInMask64Bit(newRateIndex,laDbDistributionParameter->laStaUspCommon->raIndexMask.raIndexMask64bit)))
			{
				if ((pTurboRatesDb->state == TURBO_RATES_PROBING_ENABLE)||
					(linkAdaptationCompareRSSI(laDbDistributionParameter->stationOrGroupIndex,pTurboRatesDb->lastRssi,DELTA_RSSI_TURBO_RATES)))
				{

					ILOG0_D("TurboRatesSetNextProbingPoint, Set probing point, rateIndex = %d", newRateIndex);
					/*Change RA state to wait for slow probe valid state*/
					laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_SLOW_PROBE_VALID);
					setCurrentSlowProbingTaskInDb(laDbDistributionParameter,SLOW_PROBING_TURBO_RATES);
					pTurboRatesDb->probingTurboRateIndex = turboRateProbingIndex;
					setTurboRatesInStaTcr(laDbDistributionParameter->stationOrGroupIndex, INVALID_MU_USP_INDEX, newRateIndex, bandwidth, TRUE);
					removeTaskFromCurrentLoopCycle = FALSE;
				}
			}
		}
	}
	
	if (removeTaskFromCurrentLoopCycle == TRUE)
	{
		/*	Remove loop from current slow probing bitmap*/
		
		ILOG0_V("TurboRatesSetNextProbingPoint, removeTaskFromCurrentLoopCycle");
		disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_TURBO_RATES);
		laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);

	}
}
void TurboRatesProcessFeedback(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t *pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	SlowProbingPointEstimators_t* pSlowProbingEst = estimatorsGetSlowProbingPerEstPtr(laDbDistributionParameter);
	TurboRatesDataBase_t* pTurboRatesDb = &(laDbDistributionParameter->laStaUspCommon->turboRatesDb);
	int8 slowProbingPointPer;
	bool nonEffectiveLoop;
	uint8 currentRateindex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
	
	ILOG0_V("TurboRatesProcessFeedback - PROCESS_FEEDBACK");
	SLOG0(0, 0, TurboRatesDataBase_t, pTurboRatesDb);
	handleSlowProbingIterationCounter(laDbDistributionParameter, SLOW_PROBING_TURBO_RATES);
	
	if (AlphaFilter_isFilterResultValid(&pSlowProbingEst->averagePer, pLaConfigParams->maxValidFilterTsfDiff)) 
	{
		slowProbingPointPer = AlphaFilter_GetFilterResult(&pSlowProbingEst->averagePer);
        
		if (slowProbingPointPer < TURBO_RATES_SUCCESS_PER)
		{
			ILOG0_D("TurboRatesProcessFeedback - PROCESS_FEEDBACK, set rate in turbo rates DB, rateIndex = %d", pTurboRatesDb->probingTurboRateIndex);

			ASSERT(pTurboRatesDb->probingTurboRateIndex < TURBO_RATES_INVALID_INDEX);
            
			LA_SET_BIT_IN_BITMAP(&(pTurboRatesDb->turboRatesValidBitmap), pTurboRatesDb->probingTurboRateIndex, BITMAP_ARRAY_MAX_SIZE_1);
            
			/*Trigger working point change to the current ratein order to update probing point according to new valid rate*/
			rateAdaptationChangeWorkingPoint(laDbDistributionParameter,currentRateindex,currentRateindex);
		}
		else
		{
			ILOG0_V("TurboRatesProcessFeedback - PROCESS_FEEDBACK, Rate is not valid");

			nonEffectiveLoop = updateNonEffectiveLoopCounter(laDbDistributionParameter, SLOW_PROBING_TURBO_RATES);
            
			/*If loop is not effective and is in priority 1 (max priority) change state to disable until RSSI will be changed*/
			if ((nonEffectiveLoop == TRUE) &&
				((laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingTaskBitmap[MAX_SLOW_PRIORITY]) & 
				(1<<SLOW_PROBING_TURBO_RATES)))
				{
					pTurboRatesDb->state = TURBO_RATES_PROBING_DISABLE;
					ILOG0_V("TurboRatesProcessFeedback - PROCESS_FEEDBACK, Non effective loop");
					
				}
			
		}
	}
}
static uint8 isRateInturboRatesValidBitmapAndNotValid(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 rateIndex, Bandwidth_e bandwidth)
{
	uint8 entry = 0;
	uint8 turboRateProbintEntry = TURBO_RATES_INVALID_INDEX;

	UNUSED_PARAM(laDbDistributionParameter);
	while (entry < TURBO_RATES_NUMBER_OF_RATES)
	{
		if ((turboRatesParams[entry].rateIndex == rateIndex)&&(turboRatesParams[entry].bw == bandwidth))
		{
			turboRateProbintEntry = entry;
			break;
		}
		entry++;
	}
	return turboRateProbintEntry;
}
static bool setTurboRatesInStaTcr(uint8 staIndex, uint8 uspIndex, uint8 rateIndex, Bandwidth_e bandwidth, bool changeProbingPoint)
{

	LinkAdaptationStaDatabase_t* pLinkAdaptationDb = &LinkAdaptationStaDatabase[staIndex];
	LaTcrModificationStruct_t tcrModificationParams;
	bool ratePlusOneChanged;

	UNUSED_PARAM(uspIndex);
	memset(&tcrModificationParams,0,sizeof(LaTcrModificationStruct_t));

	/*Fill control parameters of tcrModificationParams*/
	tcrModificationParams.controlParams.firstBwToUpdate = bandwidth;
	tcrModificationParams.controlParams.lastBwToUpdate= bandwidth;
	tcrModificationParams.controlParams.isVhtSta = pLinkAdaptationDb->laStaUspCommon.staTransmissionParams.vhtSta;
	tcrModificationParams.controlParams.changeProbingPoint=  changeProbingPoint;
	tcrModificationParams.controlParams.slowProbing =  changeProbingPoint;
	tcrModificationParams.controlParams.packetType = LA_PACKET_TYPE_DATA;
	tcrModificationParams.controlParams.changeToRatePlusOne = FALSE;
	tcrModificationParams.controlParams.powerChanged = TRUE;
	tcrModificationParams.controlParams.staIndex = staIndex;
	tcrModificationParams.controlParams.uspIndex = INVALID_MU_USP_INDEX;
#ifdef ENET_INC_ARCH_WAVE600
	tcrModificationParams.controlParams.isHeGroup = FALSE;
#endif
	
	
	tcrModificationParams.tcrParams.bwDependedTcrValsTable[bandwidth].rateindex = rateIndex;
	/*Set power and effected params in TCRs*/
	ratePlusOneChanged = modifyStaTcrsParamsReq(&tcrModificationParams,turboRatesModificationFunc); 	

	return ratePlusOneChanged;
}
static void turboRatesModificationFunc(LaTcrModificationStruct_t* pModifcationParamsIn, LaTcrModificationStruct_t* pModifcationParamsOut)
{
	pModifcationParamsOut->tcrParams.bwDependedTcrValsTable[pModifcationParamsIn->controlParams.firstBwToUpdate].rateindex = pModifcationParamsIn->tcrParams.bwDependedTcrValsTable[pModifcationParamsIn->controlParams.firstBwToUpdate].rateindex;

}
void TurboRatesGetBitmap(uint16 stationOrGroupIndex,uint8 uspIndex,Bandwidth_e bw, RateMask_t* raMask)
{
	TurboRatesDataBase_t* pTurboRatesDb;
	uint8 bitmap;
	uint8 turboRatesEntry = 0;
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter;
	
	if (stationOrGroupIndex != INVALID_STA_INDEX) /*Function uses also for VAP initiation - In this case stay without TurboRates, Turbo rates can be enabled only per station*/
	{
		updateLaDbDistributionParam(&laDbDistributionParameter, stationOrGroupIndex,uspIndex, FALSE);

		pTurboRatesDb = &(laDbDistributionParameter.laStaUspCommon->turboRatesDb);
		bitmap = pTurboRatesDb->turboRatesValidBitmap;

		while (bitmap != 0)
		{
			if ((bitmap & 0x1)!= 0)
			{
				if (turboRatesParams[turboRatesEntry].bw==bw)
				{
					Utils_SetBitInBitmap(raMask->raIndexMask64bit, turboRatesParams[turboRatesEntry].rateIndex);
				}
			}
			turboRatesEntry++;
			bitmap>>=1;
		}
	}
}


