/***********************************************************************************
 File:		ulPowerControl.c
 Module:		LinkAdaptation 
 Purpose: 	
 Description:	
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "LinkAdaptation.h"
#include "ulPowerControl.h"
#include "Utils_Api.h"
#include "loggerAPI.h"
#include "PreAggregator_Api.h"
#include "AntennaSelection.h"
#include "ContinuousInterfererDetection.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 18

#define LNA_NOISE_FIGURE			  (5)
#define THERMAL_NOISE_1M			  ((-174) + 60)
#define LNA_NOISE_FLOOR_1M  		  (THERMAL_NOISE_1M + LNA_NOISE_FIGURE) // dBm per 1MHz
#define DELTA_SNR_NOISES_HIGH_LNA	  (3)
#define DELTA_SNR_NOISES_LOW_LNA	  (4)
#define ROBUST_PA_NOISE_DELTA		  (6)
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static int8 calcNinTxNoise(int8 normalizedRssi);
static int8 getNormalizedBackgroundNoise(void);
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
uint8 evmPerMcs[HE_NUMBER_OF_MCS]	= {13,13,13,16,19,22,25,27,30,32,35,35};//dB
uint8 snrPerMcs1Nss[HE_NUMBER_OF_MCS] = {1,3, 6, 9, 12,15,18,21,24,27,30,33}; //dB
int8 rssiMaxPowerDeltaPerMcs_1M[HE_NUMBER_OF_MCS] = {2, 2, 2, 2, 2, 1, 1, 0, -1, -2, -3, -3};
uint8 ruSize10Log10[PRE_AGGREGATOR_NUM_OF_RU_SIZES] = {3, 7, 10, 13, 16, 19, 22}; //10*log(BW [1MHz])
int8 antsRatio10Log10[MAX_NUM_OF_ANTENNAS] = {0, -3, -5, -6}; // 10*log(NSS/Nrx) - assuming NSS =1 (Phase 0)
/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Functions Definitions									
/----------------------------------------------------------------------------------*/

int8 ULPC_calcNormalizedRssi(int8 rssi, uint8 mcs)
{
	int8 normalizedRssi;
	
//	ILOG0_DD("PC_calcNormalizedRssi. inputs: rssi %d mcs %d ",rssi,mcs);

	/* Normalize RSSI to MCS 7 */
	normalizedRssi = rssi + rssiMaxPowerDeltaPerMcs_1M[mcs] - rssiMaxPowerDeltaPerMcs_1M[MCS_7] ;  

//	ILOG0_D("PC_calcNormalizedRssi. output: normalizedRssi %d",normalizedRssi);

	return normalizedRssi;
}

LnaType_e ULPC_calcLnaType(int8 rssi, uint8 mcs, uint8 ruSize)
{
	LnaType_e lnaClass = LNA_TYPE_LOW;
	uint8 maxAttenuation; 
	uint8 bwLog; 
	int8 requiredSnrMcs7; 
	int16 sigMinPower; 
	int8 noisePowerRU;
	int8 normalizedRssi;
	int8 normalizedBgNoise = getNormalizedBackgroundNoise();
	
//	ILOG0_DDD("PC_calcLnaType. inputs: rssi %d mcs %d ruSize %d",rssi,mcs,ruSize);

	bwLog = ruSize10Log10[ruSize];
	maxAttenuation = 32 - (rssiMaxPowerDeltaPerMcs_1M[MCS_0] - rssiMaxPowerDeltaPerMcs_1M[MCS_7]);
	requiredSnrMcs7 = evmPerMcs[MCS_7] - ROBUST_PA_NOISE_DELTA;
	normalizedRssi = ULPC_calcNormalizedRssi(rssi, mcs);

	//ILOG0_DDDD("PC_calcLnaType. bwLog %d maxAttenuation %d requiredSnrMcs7 %d normalizedRssi %d",bwLog,maxAttenuation,requiredSnrMcs7, normalizedRssi);

	sigMinPower = normalizedRssi - maxAttenuation;
	noisePowerRU = MAX(normalizedBgNoise, LNA_NOISE_FLOOR_1M) + bwLog;	

	//ILOG0_DD("PC_calcLnaType. signal %d noise %d",sigMinPower, noisePowerRU);

	if (sigMinPower < (noisePowerRU + requiredSnrMcs7))
	{
		lnaClass = LNA_TYPE_HIGH;
	}

//	ILOG0_D("PC_calcLnaType. output: lnaClass %d",lnaClass);

	return lnaClass;
}
void ULPC_calcMcsAndTargetRssiHighLna(int8 normalizedRssi, uint8 ruSize, uint8* outMcs, int8* outTargetRssi)
{
	uint8 selectedMcs;
	uint8 nss = UL_NSS; // fixed value for Phase 0
	int8 maxRssi;
	int8 maxSnr;
	int8 requiredSnr;
	uint8 activatedRxAntennasNum = AntennaSelectionGetActivatedAntennasCount();
	int8 antsRatioLog = antsRatio10Log10[activatedRxAntennasNum - 1];
	int8 normalizedBgNoise = getNormalizedBackgroundNoise();
	int8 noiseRu;
	uint8 evm;
	int8 rssi;	
	uint8 deltaPowerFromMax;
	int8 headroomFromMcs0;
	uint8 snrRu;

//	ILOG0_DDD("ULPC_calcMcsAndTargetRssiHighLna. activatedRxAntennasNum %d normalizedRssi %d ruSize %d",activatedRxAntennasNum, normalizedRssi, ruSize);

	*outMcs = MCS_0;
	*outTargetRssi = MIN_INT8;
	for(selectedMcs = MCS11_1024QAM_5_6; selectedMcs > MCS_0; selectedMcs--)
	{
		//ILOG0_D("ULPC_calcMcsAndTargetRssiHighLna. selectedMcs %d",selectedMcs);

		maxRssi = normalizedRssi + rssiMaxPowerDeltaPerMcs_1M[selectedMcs] - rssiMaxPowerDeltaPerMcs_1M[MCS_7];
		noiseRu = MAX(normalizedBgNoise, LNA_NOISE_FLOOR_1M) + ruSize10Log10[ruSize];
		maxSnr = maxRssi - noiseRu;
		evm = evmPerMcs[selectedMcs];		
		requiredSnr = snrPerMcs1Nss[selectedMcs] + (3 * nss) + DELTA_SNR_NOISES_HIGH_LNA + antsRatioLog;
		//ILOG0_DDD("ULPC_calcMcsAndTargetRssiHighLna. maxRssi %d noiseRu %d maxSnr %d",maxRssi,noiseRu,maxSnr);
		//ILOG0_DD("ULPC_calcMcsAndTargetRssiHighLna. evm %d requiredSnr %d", evm, requiredSnr);
		if (maxSnr < requiredSnr)
		{
			//ILOG0_V("Not enough SNR, try lower MCS");
			continue; // Not enough SNR, try lower MCS
		}
		
		if ((selectedMcs < MCS_7) && (maxRssi <= normalizedRssi))
		{			
			evm = evmPerMcs[MCS_7]; // klocwork ignore, UNREACH.GEN, this code is not unreachable as we do SelectedMcs-- in the forloop update statement.
			//ILOG0_V("ULPC_calcMcsAndTargetRssiHighLna. ((ulMcs < MCS_7) && (maxRssi <= normalizedRssi)) == TRUE. ");
		}
		else
		{
			//ILOG0_V("ULPC_calcMcsAndTargetRssiHighLna. ((ulMcs < MCS_7) && (maxRssi <= normalizedRssi)) != TRUE");
			rssi = noiseRu - ROBUST_PA_NOISE_DELTA + evm;			
			deltaPowerFromMax = maxRssi - rssi;
			headroomFromMcs0 = deltaPowerFromMax + (rssiMaxPowerDeltaPerMcs_1M[MCS_0] - rssiMaxPowerDeltaPerMcs_1M[selectedMcs]);
			headroomFromMcs0 = MAX(0, headroomFromMcs0);
			headroomFromMcs0 = MIN(headroomFromMcs0, 31);
			//ILOG0_DD("ULPC_calcMcsAndTargetRssiHighLna. HIGH LNA. rssi %d headRoomFromMcs0 %d",rssi,headroomFromMcs0);
			
			evm += MIN(22, (3 * headroomFromMcs0));		// evm += DeltaEvmPerDeltaPowerFromMaxMcs0[headroomFromMcs0]
			evm = MIN(evm, evmPerMcs[MCS11_1024QAM_5_6]); 	
		}
		//ILOG0_D("evm %d", evm);
		*outTargetRssi = noiseRu - ROBUST_PA_NOISE_DELTA + evm;
		*outTargetRssi = MIN(*outTargetRssi, maxRssi);
		
		snrRu = *outTargetRssi - noiseRu;
		//ILOG0_D("snrRu %d", snrRu);
		if (snrRu >= requiredSnr)
		{
			//ILOG0_D("MCS found. %d", selectedMcs);
			*outMcs = selectedMcs;
			break; // klocwork ignore, UNREACH.GEN
		}	
		//ILOG0_V("Not enough SNR, try lower MCS");
	}
//	ILOG0_DD("ULPC_calcMcsAndTargetRssiHighLna, MCS %d targetRssi %d",*outMcs, *outTargetRssi);
}

void ULPC_calcMcsAndTargetRssiLowLna(int8 normalizedRssi, uint8 ruSize, uint8* outMcs, int8* outTargetRssi)
{
	uint8 selectedMcs;
	uint8 nss = UL_NSS; // fixed value for Phase 0
	int8 maxRssi;
	int8 maxSnr;
	int8 requiredSnr;
	uint8 activatedRxAntennasNum = AntennaSelectionGetActivatedAntennasCount();
	int8 antsRatioLog = antsRatio10Log10[activatedRxAntennasNum - 1];
	int8 normalizedBgNoise = getNormalizedBackgroundNoise();
	int8 noiseRu;
	uint8 evm;
	int8 rssi;	
	uint8 deltaPowerFromMax;
	int8 headroomFromMcs0;
	uint8 snrRu;
	uint8 MinTxNoise = calcNinTxNoise(normalizedRssi);
	//uint8 expectedHeadroom;

//	ILOG0_DDD("ULPC_calcMcsAndTargetRssiLowLna. activatedRxAntennasNum %d normalizedRssi %d ruSize %d",activatedRxAntennasNum, normalizedRssi, ruSize);

	*outMcs = MCS_0;
	*outTargetRssi = MIN_INT8;
	for(selectedMcs = MCS11_1024QAM_5_6; selectedMcs > MCS_0; selectedMcs--)
	{
		//ILOG0_D("ULPC_calcMcsAndTargetRssiLowLna. selectedMcs %d",selectedMcs);

		maxRssi = normalizedRssi + rssiMaxPowerDeltaPerMcs_1M[selectedMcs] - rssiMaxPowerDeltaPerMcs_1M[MCS_7];
		noiseRu = MAX(normalizedBgNoise, MinTxNoise) + ruSize10Log10[ruSize];
		evm = evmPerMcs[selectedMcs];			
		maxSnr = MIN(maxRssi - noiseRu, evm);
		requiredSnr = snrPerMcs1Nss[selectedMcs] + (3 * nss) + DELTA_SNR_NOISES_LOW_LNA + antsRatioLog;
		
		//ILOG0_DDD("ULPC_calcMcsAndTargetRssiLowLna. maxRssi %d noiseRu %d maxSnr %d",maxRssi,noiseRu,maxSnr);
		if (selectedMcs >= MCS10_1024QAM_3_4)
		{
			requiredSnr += 3;
		}
		//ILOG0_DD("ULPC_calcMcsAndTargetRssiLowLna. evm %d requiredSnr %d", evm, requiredSnr);

		if (maxSnr < requiredSnr)
		{			
			//ILOG0_V("Not enough SNR, try lower MCS");
			continue; // Not enough SNR, try lower MCS
		}
		
		if ((selectedMcs < MCS_7) && (maxRssi <= normalizedRssi))
		{			
			evm = evmPerMcs[MCS_7]; // klocwork ignore, UNREACH.GEN, this code is not unreachable as we do SelectedMcs-- in the forloop update statement.
			//ILOG0_D("ULPC_calcMcsAndTargetRssiLowLna. ((ulMcs < MCS_7) && (maxRssi <= normalizedRssi)) == TRUE. evm %d",evm);
		}
		else
		{
			//ILOG0_V("ULPC_calcMcsAndTargetRssiLowLna. ((ulMcs < MCS_7) && (maxRssi <= normalizedRssi)) != TRUE");
	
			rssi = noiseRu + requiredSnr;			
			deltaPowerFromMax = maxRssi - rssi;
			headroomFromMcs0 = deltaPowerFromMax + (rssiMaxPowerDeltaPerMcs_1M[MCS_0] - rssiMaxPowerDeltaPerMcs_1M[selectedMcs]);
			headroomFromMcs0 = MAX(0, headroomFromMcs0);
			headroomFromMcs0 = MIN(31, headroomFromMcs0); 		
			
			//ILOG0_DD("ULPC_calcMcsAndTargetRssiLowLna. rssi %d headRoomFromMcs0 %d",rssi,headroomFromMcs0);
			
			evm += MIN(22, (3 * headroomFromMcs0));		// evm += DeltaEvmPerDeltaPowerFromMaxMcs0[headroomFromMcs0]	
			evm = MIN(evm, evmPerMcs[MCS11_1024QAM_5_6]); 	
		}
		
		//ILOG0_D("evm %d", evm);
		*outTargetRssi = noiseRu + evm;
		*outTargetRssi = MIN(*outTargetRssi, maxRssi);

		snrRu = *outTargetRssi - noiseRu;		
		//ILOG0_D("snrRu %d", snrRu);
		if (snrRu >= requiredSnr)
		{
			//ILOG0_D("MCS found. %d", selectedMcs);
			*outMcs = selectedMcs;
			break;
		}	
		//ILOG0_V("Not enough SNR, try lower MCS");
	}	
//	ILOG0_DD("ULPC_calcMcsAndTargetRssiLowLna, MCS %d targetRssi %d",*outMcs, *outTargetRssi);
 }

static int8 calcNinTxNoise(int8 normalizedRssi)
{
	int8 maxRssiMcs11;
	int8 noise;
	int8 minTxNoise;

	maxRssiMcs11 = normalizedRssi + rssiMaxPowerDeltaPerMcs_1M[MCS11_1024QAM_5_6] - rssiMaxPowerDeltaPerMcs_1M[MCS_7];
	noise = maxRssiMcs11 - evmPerMcs[MCS11_1024QAM_5_6];
	minTxNoise = noise - (DELTA_SNR_NOISES_LOW_LNA + 3);

//	ILOG0_DD("PC_calcCommonEvmLevel. maxRssiMcs11 %d noise %d",maxRssiMcs11, noise);

//	ILOG0_D("PC_calcCommonEvmLevel. output: minTxNoise %d",minTxNoise);

	return minTxNoise;
}

static int8 getNormalizedBackgroundNoise(void)
{
	Bandwidth_e bw = LinkAdaptationCommonConfiguration.wlanBandwidthMax;
	int16 bgNoise;
	
	/* Get BG noise from Continuous Interferer module and normalize it to 1MHz BW */
	bgNoise = (int16)ContinuousInterfererDetection_GetMaxMeasuremenet();
	bgNoise -= ruSize10Log10[CONVERT_BW_TO_RU_SIZE(bw)];
	bgNoise = MAX(bgNoise, MIN_INT8);
	
	return bgNoise;
}
