/***********************************************************************************
 File:		LinkAdaptation.h
 Module:		LinkAdaptation
 Purpose: 	Handle Link Adaptation
 Description:	This file contains all definitions and the structures of the LinkAdaptation  
 				
************************************************************************************/
#ifndef LINK_ADAPTATION_H_
#define LINK_ADAPTATION_H_
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "HwGlobalDefinitions.h"
#include "System_GlobalDefinitions.h"
#include "linkAdaptation_api.h"
#include "shram.h"
#include "SharedDbTypes.h"
#include "ShramLinkAdaptation.h"
#include "LinkAdaptationCommon.h"
#include "lmi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "lmtimerUtility.h"
#include "TxSequencer_Api.h"
#include "HwBeamforming_Api.h"
#include "lm_StaDatabase.h"
#include "Utils_Api.h"
#ifdef ENET_INC_ARCH_WAVE600
#include "ShramPlannerResources.h"
#include "PlanManager_API.h"
#endif
//---------------------------------------------------------------------------------
//						CompliationFlags						
//---------------------------------------------------------------------------------
#define SEND_BF_REQUEST_WITH_DATA //Default is SEND_BF_REQUEST_WITHOUT_DATA
//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------


/*Configuration parameters default values*/
#define FILTER_VALIDATION_TIME 6000 //in apha filter resolution units - 512 usec -> 3 sec
#define FAST_PROBING_TRIES_BEFORE_CHANGE_K 1

#define MPDU_COUNTERS_SLOW_PROBING_INIT_K_TH 7  				//after 128 MPDU events (2^7), set slow probing
#define MPDU_COUNTERS_FAST_PROBING_INIT_K_TH 3					//after 8 MPDU events (2^3), set fast probing
#define MPDU_COUNTERS_FAST_PROBING_INIT_K_TH_FAST_INIT_STAGE 1

#define MPDU_COUNTERS_SLOW_PROBING_PRIORITY0_INIT_K_TH 0
#define MPDU_COUNTERS_SLOW_PROBING_PRIORITY1_INIT_K_TH 7

#define SLOW_PROBING_POWER_ITERATION_TH 2
#define SLOW_PROBING_BW_ITERATION_TH 1
#define SLOW_PROBING_BF_ITERATION_TH 1
#define SLOW_PROBING_CP_ITERATION_TH 1
#define SLOW_PROBING_ANT_SEL_ITERATION_TH 1
#define SLOW_PROBING_CDD_ITERATION_TH 1
#define SLOW_PROBING_TURBO_RATES_ITERATION_TH 1


#define MAX_SLOW_PROBING_ITERATIONS_PER_CYCLE_PR0 1 //Max number of slow probing tasks for one slow probing cycle
#define MAX_SLOW_PROBING_ITERATIONS_PER_CYCLE_PR1 1 //Max number of slow probing tasks for one slow probing cycle

#define SLOW_PROBING_NON_EFFECTIVE_POWER_ITERATION_TH 4
#define SLOW_PROBING_NON_EFFECTIVE_BW_ITERATION_TH 8
#define SLOW_PROBING_NON_EFFECTIVE_BF_ITERATION_TH 4
#define SLOW_PROBING_NON_EFFECTIVE_CP_ITERATION_TH 2
#define SLOW_PROBING_NON_EFFECTIVE_ANT_SEL_ITERATION_TH 2
#define SLOW_PROBING_NON_EFFECTIVE_CDD_ITERATION_TH 2
#define SLOW_PROBING_NON_EFFECTIVE_TURBO_RATES_ITERATION_TH 1


#define FAST_PROBING_TIMER_EVENT_VAL_USEC 80000	//80 msec

#define TIMER_EVENT_COUNTERS_SLOW_PROBING_INIT_K_TH 5			//after 32 timer expiration events (2^5, 2560 msec), set slow probing
#define TIMER_EVENT_COUNTERS_FAST_PROBING_INIT_K_TH 1 			//after two timer expiration events (2^1, 160 msec), set fast probing

#define TIMER_EVENT_COUNTERS_SLOW_PROBING_INIT_K_TH_FAST_INIT_STAGE 9			//after 32 timer expiration events (2^5, 2560 msec), set slow probing
#define TIMER_EVENT_COUNTERS_FAST_PROBING_INIT_K_TH_FAST_INIT_STAGE 7 			//after two timer expiration events (2^1, 160 msec), set fast probing

#define TIMER_EVENT_COUNTERS_SLOW_PROBING_PRIORITY0_INIT_K_TH 0
#define TIMER_EVENT_COUNTERS_SLOW_PROBING_PRIORITY1_INIT_K_TH 7

#define LINK_ADAPTATION_FILTERS_AGING_TIME_USEC 3000000			//  3sec

#define TARGET_PER 10			//percent units
#define TARGET_PER_MARGIN 2 	//percent units
#define ADAPTIVE_FILTER_LN_BETA_VAL  0x100// beta = 0.88 => dec2hex(round(-ln(0.88)*2^11)), 16 bit representation, 11 bit fraction, in beta calc the negative sign of Ln beta will be take into calculation
#define REF_LEN_FACTOR 40
#define WORKING_POINT_CHANGE_TP_PERCENT_MARGIN 3 
#define FAST_PROBING_VALIDATION_TH 6
#define SLOW_PROBING_VALIDATION_TH 10
#define FAST_PROBING_VALIDATION_TH_FAST_INIT_STAGE 1

#define DEFAULT_RA_STABILITY_STATE RA_LOW_PER 
#define MINIMUM_REF_LEN_VAL 0
#define REF_LEN_FAIL_DEFAULT_VAL 0
#define SLOW_PROBING_WITH_HIGHER_RATE_LOW_PER_TH 5 //5%

#define RATE_ADAPTATION_STATE_TO 4000 //in apha filter resolution units - 512 usec -> 2 sec
/*Explicit BF configurations*/
#define BF_CHANGE_EXPLICIT_TO_NON_EXPLICIT_TO 6000 //in apha filter resolution units - 512 usec -> 3 sec
#define BF_NUMBER_OF_PACKET_SENT_IN_NON_EXPL_STATE 31 //in apha filter resolution units - 512 usec -> 3 sec

#define	POWER_ADAPTATION_LOWER_POWER_PER_MARGIN	3 //3%
#define SLOW_PROBING_INTERLEAVING_LOG2_VAL 2


#define HIGHER_BW_OFFSET_RATE 0

#define SU_PROBING_THR	6
#define MU_PRELIMINARY_PROBING_THR	64

#define SU_STABILITY_FACTOR			33

#define SU_STABILITY_WEIGHT			33
#define SU_LOW_FRAME_RATE_WEIGHT	33
#define MU_USP_GOODPUT_WEIGHT		(HUNDRED_PERCENT-SU_STABILITY_WEIGHT-SU_LOW_FRAME_RATE_WEIGHT)

#define	MU_STA_MARK_DENOMINATOR		(SU_STABILITY_WEIGHT+SU_LOW_FRAME_RATE_WEIGHT+MU_USP_GOODPUT_WEIGHT)
#define CONVERT_SU_STABILITY_STATE_TO_GRADE(x) (3-x)


/*Bw coupling*/
#define RATE_INCREASE_EQ_BW_PHY_LEVELS_OFFSET 1
#define RATE_DECREASE_EQ_BW_PHY_LEVELS_OFFSET 2

/*Tables definitions*/
#define NUMBER_OF_ALL_RATES_HT_VHT (NUMBER_OF_RATES_11B + NUMBER_OF_RATES_11AG + NUMBER_OF_VHT_RATES + 1)
#define NUMBER_OF_ALL_RATES_HE (NUMBER_OF_RATES_11B + NUMBER_OF_RATES_11AG + NUMBER_OF_HE_RATES)
#define NUMBER_OF_RATES_11B 7
#define NUMBER_OF_RATES_11AG 8
#define NUMBER_OF_VHT_RATES (VHT_HE_NUMBER_OF_NSS * VHT_NUMBER_OF_MCS)
#define NUMBER_OF_HE_RATES (VHT_HE_NUMBER_OF_NSS * HE_NUMBER_OF_MCS)
#define VHT_HE_NUMBER_OF_NSS 4
#define VHT_NUMBER_OF_MCS 10
#define HE_NUMBER_OF_MCS 12
#define HT_NUMBER_OF_MCS 33
#define CONVERT_OFFSET_RATE_INDEX_TO_TCR_INDEX_11B 8
#define NUMBER_OF_FREQUENCIES 5

#define MCS8_256QAM_3_4 8
#define MCS9_256QAM_5_6 9
#define MCS10_1024QAM_3_4 10
#define MCS11_1024QAM_5_6 11

/*Explicit BF period*/
#define EXPLICIT_BF_REQUEST_INTERVAL 100000	//100 msec

/*General definitions*/
#ifndef HUNDRED_PERCENT_128_SCALE		
#define		HUNDRED_PERCENT_128_SCALE	128
#endif

/*Sequencer Report field parser*/
#define TXQ_STA_ID_SHIFT 3
#define TXQ_STA_ID_MASK 0xFF

#define BASICRATE_HIGHEST_SORT_RATE_INDEX (PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX + 1)
#define PHY_MODE_11N_FIRST_RATE_INDEX 15
#define PHY_MODE_11G_HIGHEST_SORT_RATE_INDEX 14
#define PHY_MODE_11G_HIGHEST_RATE_INDEX 14
#define HIGHEST_RATE (NUMBER_OF_ALL_RATES_HE-1)
#define HT_INVALID_MCS 0x3F
#define PHY11B_TCR_TO_RATE_INDEX_OFFSET 7
#define MANGEMENT_RATE_OFFSET HIGHEST_RATE //set minimum rate
#define MAXIMUM_BITMAP_SIZE_2BYTES 16

#define LA_PHY_11B_SHORT_PREAMBLE_RATE_MSK     0x3
#define LA_PHY_11B_LONG_PREAMBLE_BIT_OFFSET    2
#define PHY_11B_1M_MCS_INDEX 4
#define PHY_11B_1M_LA_RATE_INDEX 11
#define LA_PHY_11B_LONG_RATE_MSK     (LM_PHY_11B_RATE_MSK & (~LM_PHY_11B_SHORT_RATE_MSK))
#define LA_PHY_11B_RATES_MASK        {LM_PHY_11B_RATE_MSK, 0}
#define LA_PHY_11AG_RATES_MASK       {0x000000FF, 0}
#define LA_PHY_ZERO_11B_RATES_MASK   {~LM_PHY_11B_RATE_MSK, 0xFFFFFFFF}
#define LA_PHY_11N_RATES_MASK  		 {0xfe7f8000,0x1fe7f9}
#define LA_PHY_11AC_RATES_MASK  	 {0x1800000,0x601806}
#define LA_PHY_11AX_RATES_MASK  	 {0xFFFF8000, 0x7FFFFFFF}
#define LA_PHY_HT_VHT_RATES_MASK     {0xFFFF8000, 0x7FFFFF}


#define ANT_SELECTION_ALL 0xFF
#define ANT_SELECTION_ANT0_WP_ALL 0x3
#define ANT_SELECTION_ANT1_WP_ALL 0x3
#define ANT_SELECTION_ANT2_WP_ALL 0x3
#define ANT_SELECTION_ANT3_WP_ALL 0x3
#define ANT_SELECTION_WP_DEFAULT  0x3
#define ANT_SELECTION_ONE_ANT_MASK  0x3


#define ENABLE_ALL_NSS_MASK_HT_VHT {0xFFFFFFFF, 0x7FFFFF}
#define DISBALE_NSS_4_MASK_HT_VHT 	{0xFFFFFFFF, 0x001FFF}
#define DISBALE_NSS_3_4_MASK_HT_VHT {0xFFFFFFFF, 0x000007}
#define DISBALE_NSS_2_3_4_MASK_HT_VHT {0x01FFFFFF, 0x000000}

#define ENABLE_NSS_1_MASK_HT_VHT {0x01FF8000, 0x0}
#define ENABLE_NSS_2_MASK_HT_VHT 	{0xFE000000, 0x000007}
#define ENABLE_NSS_3_MASK_HT_VHT {0x0, 0x001FF8}
#define ENABLE_NSS_4_MASK_HT_VHT {0x0, 0x7FE000}

#ifdef ENET_INC_ARCH_WAVE600
#define ENABLE_ALL_NSS_MASK_HE {0xFFFFFFFF, 0x7FFFFFFF}
#define DISBALE_NSS_4_MASK_HE 	{0xFFFFFFFF, 0x0007FFFF}
#define DISBALE_NSS_3_4_MASK_HE {0xFFFFFFFF, 0x0000007F}
#define DISBALE_NSS_2_3_4_MASK_HE {0x07FFFFFF, 0x00000000}

#define ENABLE_NSS_1_MASK_HE {0x07FF8000, 0x0}
#define ENABLE_NSS_2_MASK_HE 	{0xF8000000, 0x00007F}
#define ENABLE_NSS_3_MASK_HE {0x0, 0x07FF80}
#define ENABLE_NSS_4_MASK_HE {0x0, 0x7FF80000}

#ifdef WORKAROUND_FOR_PHY_BUG_IN_LDPC
//Due to phy bug in A0 there are some non valid combinations of RU size, MCS, NSS and psdu length in LDPC transmission - WLANVLSIIP-3823
#define PHY_VALID_RATES_FOR_RU_SIZE_26_WITH_LDPC_MASK       {0xF7FF7FFF, 0x7FF7FF7F}
#define PHY_VALID_RATES_FOR_RU_SIZE_52_WITH_LDPC_MASK       {0xF7FF7FFF, 0x7FF7FF7F}
#define PHY_VALID_RATES_FOR_RU_SIZE_106_WITH_LDPC_MASK      {0xF7FF7FFF, 0x7FF7FF7F}
#define PHY_VALID_RATES_FOR_RU_SIZE_242_WITH_LDPC_MASK      {0xF7FF7FFF, 0x7FF7FF3F}
#define PHY_VALID_RATES_FOR_RU_SIZE_484_WITH_LDPC_MASK      {0xF7FF7FFF, 0x7FF7FF2E}
#define PHY_VALID_RATES_FOR_RU_SIZE_996_WITH_LDPC_MASK      {0x77FF7FFF, 0x7FF7FF10}
#define PHY_VALID_RATES_FOR_RU_SIZE_996X2_WITH_LDPC_MASK    {0xF7FF7FFF, 0x7FF7FF7F}
#define ENABLE_ALL_HE_RATES_MASK                            {0xFFFFFFFF, 0x7FFFFFFF}
#endif
#endif

#define PROTECTION_FAIL_MPDU_THRESHOLD 2 

#define IE_MAX_VHT_MCS_TO_MCS_VALUE_OFFSET 7 
#define IE_MAX_HE_MCS_TO_MCS_VALUE_OFFSET 7 
#define MIN_RATE_INDEX 0
#define MIN_HT_VHT_RATE_INDEX 15
#define MAX_VHT_SORT_RATE_INDEX 54
#define MAX_HE_SORT_RATE_INDEX 62
#define MAX_HT_RATE_INDEX 52
#define MIN_HE_RATE_INDEX 15

#define MAX_USP_IN_HE_GROUP (36)
#define NUM_OF_LA_HE_MU_DB_ENTRIES (16)

#define STA_CAPABILITIES_VHT_STBC_SUPPORT_MASK 0x7
#define STA_CAPABILITIES_HT_STBC_SUPPORT_MASK 0x3

#define	GROUP_DB_RATE_MASK	0x3f
#define GROUP_DB_TCR3_BW_SHIFT_SIZE	8


/* Data rate threshold in MBPS under which we cannot set 1000 miliseconds as time limit (as it would cause AGG to try and build 6M with 1000 microsecond */
#define LA_TIME_LIMIT_RATE_TH	12


#define	SIZE_OF_STATIONS_BITMAP_IN_WORDS	(HW_NUM_OF_STATIONS >> 5)
#define POWER_OFFSET_IN_DB_FROM_TCR_VALUE   (20)
#define COLLISION_DETECTION_MIN_TH 80
#define WP_PER_BIAS 5

#define BITMAP_ARRAY_MAX_SIZE_1		0x1
#define BITMAP_ARRAY_MAX_SIZE_2		0x2
#define BITMAP_ARRAY_MAX_SIZE_3		0x3
#define BITMAP_ARRAY_MAX_SIZE_4		0x4
#define BITMAP_ARRAY_MAX_SIZE_5		0x5

#ifdef ENET_INC_ARCH_WAVE600
#define NUMBER_OF_WORDS_IN_COMMONPLAN_DESC		(sizeof(CommonPlan_t)/4)	//KW_FIX_FW_G
#define COMMON_PLAN_PHASE0_POINTER_LOCATION		2
#endif

#define RTS_THRESHOLD_PPDU_SIZE_OFF 0
#define RTS_THRESHOLD_PPDU_SIZE_ON 1

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------
#define LA_CLR_BIT_IN_BITMAP(bitmap,offset)  (bitmap)[(offset >> 5)] &= ~(1 << ((offset) & 0x1F))
#define LA_CLR_BIT_IN_BITMAP_WITH_CHK(bitmap,offset,check)  if((offset>>5)<check){(bitmap)[(offset >> 5)] &= ~(1 << ((offset) & 0x1F));}else{ASSERT(0);}

#define LA_SET_BIT_IN_BITMAP(bitmap,offset, check)  if ((offset >> 5) < check){(bitmap)[(offset >> 5)] |= (1 << ((offset) & 0x1F));}else{ASSERT(0);} //KW_FIX_FW_G Added array bound check
#define LA_GET_BIT_IN_BITMAP(bitmap,offset, check)  ((((bitmap)[(offset >> 5)%check] & (1 << ((offset) & 0x1F)))) >> ((offset) & 0x1F)) //KW_FIX_FW_G Added array bound check
#define LA_TOGGLE_BIT_IN_BITMAP(bitmap,offset) (bitmap)[(offset >> 5)] ^= (1 << ((offset) & 0x1F))
#define LA_MODIFY_BITS_IN_BITMAP(bitmap,mask,val) ((bitmap & (~mask))|val)

#define	SET_SIX_LS_BITS(target, val) 			(*target = ((val & 0x3f)|((*target)&(0xC0)))) /*target parameter shall be a pointer*/
#define	GET_SIX_LS_BITS(val) 					(val & 0x3f) 
#define	SET_TWO_MS_BITS(target, val) 			(*target = ((val <<6)|((*target)&(0x3f)))) /* Val must be in range of (0x0:0x3), target parameter shall be a pointer*/

#define	TURN_ON_BIT(target, bitIndex) 			(*target |= (0x1 << (bitIndex))) /*set 1 in the correspond bit*/
#define	TURN_OFF_BIT(target,bitIndex) 			(*target &= (~(0x1 << (bitIndex)))) /*set 0 in the correspond bit*/
#define	SET_BIT_IN_BYTE(target,bitIndex,val) 	(val?	TURN_ON_BIT(target,(bitIndex)) : TURN_OFF_BIT(target,(bitIndex)))
#define GET_BIT_IN_BYTE(bitmap,bitIndex) 		(bitmap & (1 << (bitIndex)))

#define ADD_PERCENT(val, percent) (val*(100+percent)/100);
#define SUB_PERCENT(val, percent) (val*(100-percent)/100);

#define CONVERT_MAX_NUM_OF_NSS_TO_MAX_NUM_OF_ANTENNAS(x) (x+1)
#define CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(x) (x-1)

#define QAM_PLUS_MIN_RSSI_DB			(RX_THRESHOLD_DEFAULT_VALUE) 


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

typedef struct StationBitmap
{
	uint32 staBitMap[SIZE_OF_STATIONS_BITMAP_IN_WORDS];
}StationBitmap_t;

typedef struct _MuGroupBitmap_t
{
	uint16 muGroupBitMap[1];
} MuGroupBitmap_t;

typedef struct HeUspBitmap
{
	uint32 heUspBitmap[BITMAP_ARRAY_MAX_SIZE_2];
}HeUspBitmap_t;

typedef struct PowerConfigParams
{
	uint8	staInternalLowerLimit;
	uint8 	powerStepUp;
	uint8	powerStepDown;
	uint8  	powerStepUpDefault;
	uint8  	powerStepDownDefault;
	uint8 	powerAdditionalPowerStep;
	uint8 	powerMaxAdditionalPowerStep;
}PowerConfigParams_t;

typedef struct ProtectionStaConfigParams
{
	uint8 staCollisionNoProtPositiveThreshold;
	uint8 staCollisionThresholdModePsotiveThreshold;
	uint8 staCollisionThresholdModeNegativeThreshold; 
	uint8 staCollisionForceModeNegativeThreshold; 
	int8 staCollisionBetterThanVapThreshold; 
}ProtectionStaConfigParams_t;

typedef struct ProtectionVapConfigParams
{
	uint32 wlanTimerIntervals;
	uint16 maxValidFilterTsfDiff;
	uint8 vapStaCollisionTotalRatioPositiveThreshold; 
	uint8 vapCollisionNonImprovementThreshold; 
	uint8 vapStaCollisionTotalRatioNegativeThreshold; 
	uint8 vapBerGapPositiveThreshold; 
}ProtectionVapConfigParams_t;

typedef struct LinkAdaptationConfigurationParams
{
	uint16 maxValidFilterTsfDiff;
	uint8 fastProbingTriesBeforeChangeKTh;
	uint8 slowProbingTriesBeforeChangeKTh;
	
	RaEventThresholds_t mpduCountersThresholds;
	RaEventThresholds_t timerEventCountersThresholds;
	
	uint8 mpduEventSlowProbingKThInit;
	uint8 timerEventSlowProbingKThInit;
	uint8 slowProbingMaxNumOfIterations[SLOW_PROBING_TASKS_NUM];
	uint8 slowProbingNonEffectiveLoopTh[SLOW_PROBING_TASKS_NUM];
	uint8 maxSlowProbingIterationsPerCyclePr0;
	uint8 maxSlowProbingIterationsPerCyclePr1;
	uint8 mpduEventSlowProbingKThMax;
	uint8 timerEventSlowProbingKThMax;
	uint8 targetPer;
	uint8 targetPerMargin;
	uint16 lnBeta;
	uint8 constAlphaCoefForDtAvPower2;
	uint8 workingPointChangeTpPercentMargin;
	uint8 fastProbingValidationTh;
	uint8 slowProbingValidationTh;
	RaStabilityState_e defaultRaStabilityState;
	uint8 refLenFactor;
	uint8 managementRateOffset;
	uint8 fastDropEstimatorsTh;
	uint8 fastDropThreshold;
	uint8 averageDeltaTWorkingPoint;
	PowerConfigParams_t powerConfigurationParams;
	ProtectionStaConfigParams_t protectionConf;
	uint8 slowProbingWithHigherRateLowPerTh;
	uint8 currentPacketFailedDiffFromAvTh;
	uint16 linkAdaptationStateTo;
	uint16 bfChangeExplicitToNonExplicitTo;
	uint8 bfNumOfPacketTransInCurrentState;
	uint8 slowProbingInterleavingLog2Val;
	uint8 powerAdaptationLowerPowerPerMargin;
	uint8 higherBwOffsetRate;
	uint8 suProbingValidationTh; 
	uint8 muPreliminaryProbingTh; 
	uint8 collisionDetectionTh;
}LinkAdaptationConfigurationParams_t;
typedef enum LinkAdaptationPhyMode
{
	LA_PHY_MODE_11AG,
	LA_PHY_MODE_11B,
	LA_PHY_MODE_HT_VHT,
	LA_PHY_MODE_HE,
	NUM_OF_LA_PHY_MODE
}LinkAdaptationPhyMode_e;

typedef struct _LinkAdaptationCommonConfigParams_t
{
	uint32 fastProbingTimerEventIntervalUsec; 
	uint32 estimatorsAgingTimeUsec;
	uint32 bfRequestInterval;
	Bandwidth_e wlanBandwidthMax;
	uint8 band;
} LinkAdaptationCommonConfigParams_t;

typedef struct _TransmissionParams_t
{
	uint16 heSta						:1;
	uint16 vhtSta						:1;
	uint16 staInSmpsStaticMode 			:1;
	uint16 smpsRecievedBeforeAddStaFlag	:1;
	uint16 smpsRecievedBeforeAddStaVal 	:1;
	uint16 clientIsolationMode 			:1;
	uint16 clientIsolationProbing		:1;
   	uint16 clientIsolationRecovery		:1;
	uint16 staExplicitBfSupport			:1;
	uint16 staStbcSupport				:2; 
	uint16 staLdpcSupport				:1;
	uint16 isWdsWep						:1;
#ifdef ENET_INC_ARCH_WAVE600
	uint16 tfPadding					:2;
	uint16 staStbcSupport160			:1;
#else
	uint16 reserved 					:3;
#endif
} TransmissionParams_t; 

typedef enum LaOperatingModeNotificationRxNssVal
{
	OMN_RX_NSS_1,
	OMN_RX_NSS_2,
	OMN_RX_NSS_3,
	OMN_RX_NSS_4,
	OMN_RX_NSS_INVALID,
}LaOperatingModeNotificationRxNssVal_e;

typedef enum LaNssNum_e
{
	LA_NSS_1,
	LA_NSS_2,
	LA_NSS_3,
	LA_NSS_4,
	LA_NSS_INVALID,
}LaNssNum_e;

typedef struct LaOperatingModeNotification
{
	uint8	channelWidth				:2;
	uint8	rxNss						:2;
	uint8	isFrameReceivedBeforeAddSta :1;		
	uint8	staInOmn					:1;
	uint8	reserved					:2;
} LaOperatingModeNotification_t;

typedef union overridePendingBitmap
{
	uint8 val;
	struct _bitfields
	{
		uint8 rateIndexBit		: 1;
		uint8 nssBit			: 1;
		uint8 antennaMaskBit	: 1;
		uint8 bwBit				: 1;
        uint8 vhtMpdu			: 1;
		uint8 adjustRate2Mask 	: 1;
		uint8 reserved1			: 2;	
	} bitFields;
} overridePendingBitmap_u;

typedef struct overridePendingVals
{
    uint8 rateIndexVal;
    uint8 antennaMaskVal; 
    uint8 bwVal			: 2;
	uint8 nssVal		: 2; //antennaCount
	uint8 reserved2     : 4;
} overridePendingVals_t;

typedef struct overridePendingDb
{
	overridePendingVals_t 	overrideVals;
	overridePendingBitmap_u overrideBitmap;
} overridePendingDb_t;

typedef enum overridePending
{
	OVERRIDE_RATE_INDEX,
	OVERRIDE_NSS,
	OVERRIDE_ANTENNA_SELECTION,
	OVERRIDE_BW,
	OVERRIDE_VHT_MPDU,
	ADJUST_RATE_TO_MASK
} overridePending_e;

typedef struct LinkAdaptationVapDatabase
{
	vapEstimatorsDataBase_t 	estimatorsVapDataBase;

}LinkAdaptationVapDatabase_t;

typedef struct RateTableBwParmas
{
	uint16 shortCpPhyRate;
	uint16 mediumCpPhyRate;
	uint16 longCpPhyRate;
}RateTableBwParmas_t;

typedef struct RateObj
{
	LinkAdaptationPhyMode_e laPhyMode;
 	uint32 numberOfNss 				:2;
	uint32 htMcs					:6;
	uint32 vhtHeMcs					:4;
	uint32 sortPhyRateSameBwIndex	:7;
	RateTableBwParmas_t  rateTableBwParmas[MAX_POSSIBLE_NUM_OF_BW];
}RateObj_t;

typedef struct RatesTablesDb
{
	const RateObj_t* ratesTable;
	const uint8* sortedByPhyRateTable;
	uint8 maxRatesInTable;
} RatesTablesDb_t;

typedef enum StaDbStbcConfig
{
	NOT_STBC_TRANSMISSION = 0,
	STBC_1X2_TRANSMISSION,
	STBC_2X4_TRANSMISSION
} StaDbStbcConfig_e;

typedef enum _GiAndLtfType_e
{
	GI_AND_LTF_TYPE_1X_LTF_1_6_GI = 0,
	GI_AND_LTF_TYPE_2X_LTF_1_6_GI = 1,
	GI_AND_LTF_TYPE_4X_LTF_3_2_GI = 2,
	GI_AND_LTF_TYPE_RESERVED = 3,
} GiAndLtfType_e;

typedef struct LaTimerEventsTimeExpiration
{
	StaId staIndex;
	uint32 timerExpirationTsf;
} LaTimerEventsTimerEvent_t;

typedef struct
{
	CyclicPrefixMode_e currentCpWp;
	CyclicPrefixMode_e nextCpPp;
	uint8 CyclicPrefixCapBitmap[LA_NUM_OF_BANDWIDTH];  // bit per CP mode
#ifdef ENET_INC_ARCH_WAVE600D2	
	uint8 HeSuExtCyclicPrefixCapBitmap;  // bit per CP mode, 20MHz only
#endif
}CyclicPrefixDb_t;

typedef struct LaSlowProbingHandlingParams
{
	uint8					slowProbingTaskBitmap[SLOW_PROBING_NUMBER_OF_PRIORITIES];
	uint8					slowProbingNotEffectveLoopCounter[SLOW_PROBING_TASKS_NUM];
	uint8 					iterationCounter[SLOW_PROBING_TASKS_NUM];
	uint8					totalIterationsPerCycle;
	uint8 					slowProbingCurrentTaskBitmap;
	SlowProbingTasks_e		slowProbingCurrentTask;
	uint8 					probingHandlerInterleavingCounter;	
}LaSlowProbingHandlerParams_t;

typedef struct LaStationUniqueParams
{
	DynamicTxopDB_t					dynamicTxopDb;
	QAMplusStaDataBase_t			QAMplusStaDb;
	AdaptiveAlphaFilterParams_t 	lowFrameRateCounter;/*500B only*/
	uint16 							lowFrameRateAverageDtFilterResult;/*500B only*/
	uint64							stationInPsOrManagementFramesWindow; /*500B only*/
//	uint32							stationInPsWindow; /*500B only*/
	uint32							uspIndexPerVhtGroup; /*500B only*/ /*Each 2 bits represent the USP index in a group, group ID corresponds the offset*/
	LaBeamformingDataBase_t			beamFormingDb;
	AntennaSelectionDatabase_t		antennaSelectionDb;
#ifndef ENET_INC_ARCH_WAVE600
	CddDatabase_t					cddDataBase;
#endif
	LaOperatingModeNotification_t	operatingModeNotification;
	
#ifdef TRAINING_FIRST_PHASE_POOL
	uint8							vapId; 
#endif // TRAINING_FIRST_PHASE_POOL	

	uint8							stationInPsOrManagementWithinWindowCount; /*500B only*/
	uint8							singleUserProbingCount	:6 ; /*500B only*/
	uint8							singleUserProbingState	:2; /*500B only*/
	uint8							muGroupCount; /*Number of MU group which the station is a member*/ /*500B only*/ /*In wave600 we use muGroupCount for VHT MU MIMO upport*/
	LaFixedRateStationParams_t 		fixedRateStationParams;
#ifdef ENET_INC_ARCH_WAVE600
	LaHeUlPowerControlParams_t		ulPcParams;
#endif
	uint8							initialRateIndex;					/* Holds the initial data rate. We override it with Management rate so that the Sniffer can record EAPOL (WLANRTSYS-6070)*/
	bool							isSnifferEapolWaActive;				/* TRUE: Management rate has been added to the Org mask (WLANRTSYS-7513) */
	uint8							snifferEapolWaRate;					/* The management rate used for the EAPOL WA (WLANRTSYS-7513) */
#ifdef ENET_INC_ARCH_WAVE600D2
	LaHeExtSuParams_t				heExtSuParams;
	LaHeSigBParams_t				heSigBParams;
#endif
}LaStationUniqueParams_t;

typedef struct LaGroupUniqueParams
{
	uint8						groupPreliminaryProbingCount;
	LinkAdaptationGroupState_e	groupState; 
	bool						groupHasFailedNotificationSent;
}LaGroupUniqueParams_t;

typedef struct LaVhtGroupUniqueParams
{
	AdaptiveAlphaFilterParams_t wpAverageGoodPutPerUsp[MAX_USP_IN_VHT_GROUP];
	uint8						probingValidationBitmap;
	uint8						probingValidationMask;
	StaId 						groupStationIndexes[MAX_USP_IN_VHT_GROUP];
}LaVhtGroupUniqueParams_t;

typedef struct LaHeGroupUniqueParams
{
	AdaptiveAlphaFilterParams_t wpAverageGoodPutPerUsp[MAX_USP_IN_HE_GROUP];
	StaId 						groupStationIndexes[MAX_USP_IN_HE_GROUP];
	HeUspBitmap_t				probingValidationBitmap;
	HeUspBitmap_t				probingValidationMask;
#ifdef ENET_INC_ARCH_WAVE600
	DlDataPhase_t*				wpPhasePtr;
#endif
	uint16 						phaseEntriesBitmap;
	uint8						planIndex;
	AdaptiveAlphaFilterParams_t dlPsduEfficiency;
	AdaptiveAlphaFilterParams_t ulPsduEfficiency;
	
}LaHeGroupUniqueParams_t;

typedef struct LaStationGroupCommonParams
{
	LinkAdaptationEventHandler_t			laEventsHandler;
	SlowProbingPointEstimators_t			slowProbingEstimators;
	CyclicPrefixDb_t						cyclicPrefixDb;
	EstimatorsStaProtection_t 				staProtection;
	AdaptiveAlphaFilterParams_t 			wpAverageGoodPut;
	uint16 									averageGoodPutDtFilterResult;
	LinkAdaptationConfigurationParams_t* 	pLinkAdaptationConfigurationParams;
	PowerAdaptationDatabase_t				powerAdaptationDb;
	LaSlowProbingHandlerParams_t 			slowProbingHandlerData;
	uint16									raEnterStateTime;
	LinkAdaptationState_e					linkAdaptationState;
	RaProbingIndication_e					probingIndication;
	BwStationDatabaseVars_t					bwAdaptationDb;
	
#ifdef LA_CALC_REF_LEN
	WorkingPointEstimators_t 				workingPointEstimators;
#endif 
}LaStationGroupCommonParams_t;

typedef struct LaStationUspCommonParams
{
	RateAdaptationDataBase_t 		rateAdaptationDataBase;
	FastProbingPointEstimators_t 	fastProbingEstimators[TOTAL_NUMBER_OF_RATES_IN_DB];
	RateMask_t						raIndexMask;
	RateMask_t						raIndexMaskOrg;
	overridePendingDb_t				overridePending;
#ifdef LA_CALC_REF_LEN
	WorkingPointEstimators_t 		workingPointEstimators;
#endif 
	uint8							probingPointValidationCounter;
	uint16 							windowFailedPacketBitmap;
	TransmissionParams_t 			staTransmissionParams;
	uint8 							fastProbingPointEstimatorsIndexes [TOTAL_NUMBER_OF_RATES_IN_DB];
	uint8 							numberOfFailedPacketInWindow;
	RatesTablesDb_t*				pRatesTablesDb;
	uint16 							longTermPerFilter;
	uint16 							longTermVarFilter;	
	TurboRatesDataBase_t			turboRatesDb;
}LaStationUspCommonParams_t;

typedef struct LinkAdaptationStaDatabase
{ 
	LaStationUspCommonParams_t		laStaUspCommon;
	LaStationGroupCommonParams_t 	laStaGroupCommon;
	LaStationUniqueParams_t			laStationUnique;
}LinkAdaptationStaDatabase_t;

typedef struct LinkAdaptationVhtGroupDatabase
{
	LaStationUspCommonParams_t 		laStaUspCommon[MAX_USP_IN_VHT_GROUP];
	LaStationGroupCommonParams_t 	laStaGroupCommon;
	LaGroupUniqueParams_t			laGroupUnique; 
	LaVhtGroupUniqueParams_t		laVhtGroupUnique;
}LinkAdaptationVhtGroupDatabase_t;

typedef struct LinkAdaptationHePhaseDatabase
{
	LaStationUspCommonParams_t 		laStaUspCommon[MAX_USP_IN_HE_GROUP];
	LaStationGroupCommonParams_t 	laStaGroupCommon;
	LaGroupUniqueParams_t			laGroupUnique; 
	LaHeGroupUniqueParams_t			laHeGroupUnique; 	

}LinkAdaptationHePhaseDatabase_t;

typedef struct LinkAdaptationDatabaseDistributionPack
{
	LaStationGroupCommonParams_t* 	laStaGroupCommon;
	LaStationUspCommonParams_t* 	laStaUspCommon;
	LaStationUniqueParams_t*		laStationUnique;
	LaGroupUniqueParams_t*			laGroupUnique; 	
	LaVhtGroupUniqueParams_t*		laVhtGroupUnique;
	LaHeGroupUniqueParams_t*		laHeGroupUnique; 
	uint16 							stationOrGroupIndex; 
	uint8 							uspIndex;
}LinkAdaptationDatabaseDistributionPack_t;


typedef void (*LaTcrModificationFunc)(LaTcrModificationStruct_t*, LaTcrModificationStruct_t* );

typedef enum DmrEvent
{
	DMR_STATION_RATE_CHANGED,
	DMR_ADD_STATION,
	DMR_REMOVE_STATION,
}DmrEvent_e;
typedef struct LaFixedRateStationProtectedDbParams
{
	LaFixedRateStationParams_t fixedRateStationParams;
	StaId staIndex;
}LaFixedRateStationProtectedDbParams_t;

#ifdef ENET_INC_ARCH_WAVE600

//KW_FIX_FW_G.
/*
In LinkAdaptation.c (uint32 *) is used to iterate over the phase pointers, klockwork sees it as a violation
and possible out of bound access. Since we know we have "NUMBER_OF_WORDS_IN_COMMONPLAN_DESC (19)" uint32 in
the common plan structure, I am explicitly declaring a uint32 PhasePointers[NUMBER_OF_WORDS_IN_COMMONPLAN_DESC]
and tCommonPlan in an union. We will use PhasePointer to access different words in CommonPlan
*/
typedef union CommonPlan
{
	uint32 PhasePointers[NUMBER_OF_WORDS_IN_COMMONPLAN_DESC];
	CommonPlan_t tCommonPlan;
}CommonPlan_u;
typedef struct GroupsMetricsCalc
{
	uint8 dlPsduEfficiency;
	uint8 ulPsduEfficiency;
}HegroupManagerGroupsMetrics_t;
typedef struct SuGroupsMetricsCalc
{
	int8 maxRssiFromAlLAnts;
}HegroupManagersuMetrics_t;

#endif //#ifdef ENET_INC_ARCH_WAVE600

//---------------------------------------------------------------------------------
//						Public Function Declaration				
//---------------------------------------------------------------------------------
uint32 andOperator64Bit (uint32* inputA, uint32* inputB, uint32* result);
void xorOperator64Bit(uint32* inputA, uint32* inputB, uint32* result);
uint32 isBitInMask64Bit(uint8 bit, uint32* pMask);
uint8 getHighestRateinMask(StaId stationIndex, RateMask_t rateMask, Bandwidth_e bandwidth, uint8 minRateSortIndex, uint8 maxRateSortIndex);
void getTcrPhyModeAndRate(const RateObj_t* ratesTable, bool isVhtSta, uint8 rateIndex, PhyMode_e* outPhyMode, uint8* tcrRateVal, bool isHeExt, bool dcm);
uint8 getLowestRateinMask(StaId stationIndex, RateMask_t rateMask, Bandwidth_e bandwidth, uint8 minRateSortIndex, uint8 maxRateSortIndex);
void fillStaHwDb (UMI_STA_ADD *pAddSta, uint8 initialRate);
void fillVapHwDb (uint8 vapIndex, uint8 fixedMcsVapManagement);
#ifdef ENET_INC_ARCH_WAVE600
void resetSequenceNumderInVapHwDb (uint8 vapIndex);
#endif

uint32 getPsduMaxLengthLimit(uint8 maximumAmpdulengthExponent);
uint16 calculateMpduMaxLengthLimitVht(uint8 maxMpduSize);
uint16 calculateMpduMaxLengthLimitHt(uint8 maxMpduSize);
uint32 getMpduMaxLengthLimit(bool vhtVap);
void setLdpcCapability(UMI_STA_ADD *pAddSta,LinkAdaptationStaDatabase_t* pLinkAdaptationDb);
void setIsWdsWep(UMI_STA_ADD *pAddSta,LinkAdaptationStaDatabase_t* pLinkAdaptationDb);
bool getIsWdsWep(StaId stationIndex,uint8 uspIndex, bool isHeGroup);
StaDbldpcConfig_e getLdpcCapability(StaId staIndex, LinkAdaptationPhyMode_e laPhyMode);
StaDbStbcConfig_e getStbcCapability(bool StbcCapability, PhyMode_e phyMode);
uint8 convertTcr2RateIndex(PhyMode_e phyMode, uint8 tcrVal);
int8 searchMaxOverTable (int8* pTable, uint8 tableLen);
int8 searchMinOverTable (int8* pTable, uint8 tableLen);
void setMaskAccordingToMaxNumOfAntennas(LinkAdaptationDatabaseDistributionPack_t* distributionParams, uint8 antennaCount);
uint32 calculateTimerExpirationTime(uint32 timeToGo, uint32 currentTsf, uint32 timerExpirationTsf);
bool modifyStaTcrsParamsReq(LaTcrModificationStruct_t* modifyParamsIn, LaTcrModificationFunc modificationFunc);
void setAllBwsTcrsParams(LaTcrModificationStruct_t* pTcrParams);
void getTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams);
void LinkAdaptationSetStaDataBwLimit(StaId staIndex,Bandwidth_e bandwidth, bool changeWorkingPoint);
void SetStaDataBwLimitInHwDb(StaId staIndex,Bandwidth_e bandwidth,uint8 changeWorkingPoint);
void SetStaManagementBwLimit(StaId staIndex,Bandwidth_e bandwidth);
void bwAdaptationSetStationBwLimit(StaId staId, Bandwidth_e bwLimit, bool changeWorkingPoint);
void bwAdaptationSetBwLimit(LinkAdaptationDatabaseDistributionPack_t *pDistParam, Bandwidth_e bwLimit, bool changeWorkingPoint);
Bandwidth_e GetDataBwLimit(StaId staOrGroupIndex,uint8 uspIndex, bool isHeGroup);
Bandwidth_e GetWpDataBwLimit(StaId staOrGroupIndex,uint8 uspIndex, bool isHeGroup);
void clientIsolationRateIsChanged(LinkAdaptationDatabaseDistributionPack_t* pLaDbDistributionParameter, RaFastProbingPoint_e newFastProbingPoint, uint8 oldRateIndex, uint8 newRateIndex, Bandwidth_e bw);
#ifdef DYNAMIC_RTS_ISSUE_WORKAROUND
void LinkAdaptationPhyDriver_UpdatePhyGenriscScpad(void);
#endif
void updateTransmissionTimeParams(StaId staIndex, uint8 rateIndex);
void setTransmissionTimeParams(void);
void linkAdaptationResetStaDb(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
void linkAdaptationResetGroupDb(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
void linkAdaptationUpdateCalStatistics(K_MSG *psMsg);
void updateLaStatistics(LaTcrModificationStruct_t* pTransmissionParams);
uint8 GetStaWpRateIndexFromHwTcr(StaId staIndex, Bandwidth_e bandwidth);
PhyMode_e ExtractHighestPhyModeFromRaMask(RateMask_t raMask, bool isHe);
void ChangeStaToSteadyStateConfigurationParams (LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool resetCounters);
void resetFastProbingCounters(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
void linkAdaptationSetRtsMode(UMI_RTS_MODE_CONFIG *pRtsModeConfigMsg);
void linkAdaptationSetRecoveryTimeLimit(K_MSG *psMsg );
void UpdateStationMaxSupportedNss(StaId staIndex, SpatialStreamNum_e maxNss);
void UpdateVapMaxNssNdp(uint8 vapIdx, SpatialStreamNum_e maxNss);
void linkAdaptationSetFreqJumpMode(uint8 freqJumpMode);
void linkAdaptationSetMcastRateInVapDb(uint8 vapId,uint8 rate,PhyMode_e phyMode );
void AntennaSelectionBfModeIsChanged(StaId staIndex, BeamformingMode_e newBfMode);
void updateLaDbDistributionParam (LinkAdaptationDatabaseDistributionPack_t *pDistParam, StaId stationOrGroupIndex, uint8 uspIndex, bool isHeGroup);
void CddSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint);
void AntennaSelectionSetNextProbingPoint(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, bool stayWithLastProbingPoint);
void updateOverridePendingDb(overridePending_e overridePending, uint8 value, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
bool executeParamOverride(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
void linkAdaptationSuProbingHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
void linkAdaptationMuPreliminaryHandler(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
bool	validateBfTransmission(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, uint8 bfType);
SlowProbingTasks_e	getCurrentSlowProbingTask(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
void setAntBoostInStaTcr(StaId staId, uint8* antBoostTable);
void setAntBoostInVapTcr(uint8 vapId, uint8* antBoostTable);
void setAntBoostInTcrs(uint8* antBoostTable);
void linkAdaptationSetFreqJumpModeInCtsAutoReply(K_MSG *psMsg);
void setUspRatesTables(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, RatesTablesDb_t* pRatesTablesDb);
const RateObj_t* getRatesTable(LaStationUspCommonParams_t* pStaUspDb);
const RateObj_t* getRatesTableFixRateConfig(LaFixedRateStationParams_t* params);
const uint8* getSortedRatesTable(LaStationUspCommonParams_t* pStaUspDb);
uint8 getMaxNumOfRatesInTable(LaStationUspCommonParams_t* pStaUspDb);
#ifdef ENET_INC_ARCH_WAVE600
uint8 getMaxSortedTableRateIndex(bool isHe);
#endif
void getStaRateMaskPerBw(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, Bandwidth_e bw, RateMask_t inputMask, RateMask_t* outputMask, bool forceAddLegacyRatesToAllowedMask);
void  antennaSelectionSetMuGroupAntenna(uint8 groupId, uint8 antennaSelection, bool isHeGroup);

#ifdef ENET_INC_ARCH_WAVE600
uint8 getTfPaddingCapability(StaId staId);
void setDlDataPhaseParams(DlDataPhase_t* dlPhasePtr, uint8 vapId, bool staticGroup);
void setUlDataPhaseParams(UlDataPhase_t* ulPhasePtr, uint8 vapId, LaAddHeGroupParams_t *addHeGroupParameters);
void setProtectionPhaseParams(ProtectionPhase_t* protectionPhasePtr, uint8 vapId, bool staticGroup);
void setSoundingPhaseParams(SoundingPhase_t* soundingPhasePtr, uint8 vapId, LaAddHeGroupParams_t *addHeGroupParameters, uint8 planIndex);
DlDataPhase_t* getWpPhaseAddress(uint8 phaseIndex);
void setWpPhaseAddress(DlDataPhase_t* phaseAddr, uint8 laPhaseId);
uint8 getLaPhaseIdFromPlanIndex(uint8 planIndex, uint8 phaseOffset);
PhaseType_e getPhaseTypeFromPlanIndex(uint8 planIndex, uint8 phaseOffset);
void getHeMuTcrParamsFromHwDb(LaTcrModificationStruct_t* tcrModificationParams);
void setTcrsInPhaseDb(LaTcrModificationStruct_t* pTcrModificationParams);
void heMuWorkingPointIsChanged(uint8 laPhaseId);
void replacePtrInPlanToMirrorPhase(uint8 laPhaseId, uint8 phaseOffsetInPlan);
uint8 getNumOfValidUspInPhase(void* phasePtr);
void overrideUlPhaseAfterOmiRx(LaUlprReport_t * ulprReport, uint32 planIndex, uint8 hePhaseIndex);

#endif //ENET_INC_ARCH_WAVE600
void linkAdaptationSetMsduInAmsdu(K_MSG* psMsg);
void linkAdaptationSetPpduTxMode(uint8 ppduTxMode);


#ifdef TRAINING_FIRST_PHASE_POOL
void decreaseMaxBwCount(StaId staIndex, Bandwidth_e bw);
void getMaximalWorkingBW(Bandwidth_e* maximalWorkingBw);
#endif //TRAINING_FIRST_PHASE_POOL
#ifdef LINK_ADAPTATION_REGISTRATION
void linkAdaptationRegistrationSendEventInd(LaRegistrationEvent_e registrationEvent, LaRgistrationGeneralCbParams_t* pGenCbParams);
#endif
uint16 getPhyRate(const RateObj_t*  ratesTable, uint8 rateIndex, Bandwidth_e bandwidth, uint8 cpMode);

void linkAdaptationDmrStationEventHandler(StaId stationId, uint8 newRateIndex, uint8 oldRateIndex, DmrEvent_e event);
uint8 LinkAdaptationGetMaxNssInMask(uint32* rateMask, bool isHeSta);
bool linkAdaptationCompareRSSI(StaId stationId, int8 lastRssi, uint8 deltaRssi);
void linkAdaptationFindMaxRssi(StaId stationId, int8* maxCurrentRssidB);
void setMpduInAmpduLimit(K_MSG *psMsg);
void linkAdaptationSetRtsThreshold(K_MSG* pkMsg);

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

extern uint8 wlanCollisionTotlaCount;
extern TxSequencerReport_t CurrentSequencerReport;
extern BaaRateAdaptiveFullFifo_t CurrentBaaReport;
extern BaaRateAdaptiveFullFifo_t LastBaaReport;
extern const RateObj_t htVhtRatesTable[NUMBER_OF_ALL_RATES_HT_VHT];
extern const uint8  vhtTcr2RateIndex[VHT_NUMBER_OF_MCS][VHT_HE_NUMBER_OF_NSS];
extern const uint8 htTcr2RateIndex[HT_NUMBER_OF_MCS];
#ifdef ENET_INC_ARCH_WAVE600
extern const RateMask_t AllowedHeRatesMask;
extern const RateObj_t heRatesTable[NUMBER_OF_ALL_RATES_HE];

#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
extern uint32 originalSuMaxPsduLengthLimit[HW_NUM_OF_STATIONS];
extern uint32 originalMuMaxPsduLengthLimit[HW_NUM_OF_STATIONS];
#endif

#endif
extern const RateMask_t AllowedRatesMask[MAX_POSSIBLE_NUM_OF_BW];
extern const RateMask_t AllowedWdsWepRatesMask[MAX_POSSIBLE_NUM_OF_BW];
extern LaFixedRateParams_t FixedRateParameters;
extern LaFixedPowerParams_t FixedPowerParameters;
extern LinkAdaptationCommonConfigParams_t LinkAdaptationCommonConfiguration;
extern uint16 linkAdaptaionActivateTimersIds[NUMBER_OF_LINK_ADAPTATION_TIMERS];
extern RateMask_t disableHtVhtNssMaskAccordingtoAntNum[MAX_POSSIBLE_NUM_OF_ANTENNAS];
extern RateMask_t enableHtVhtNssRatesInMask[VHT_HE_NUMBER_OF_NSS]; 
extern RateMask_t disableHeNssMaskAccordingtoAntNum[MAX_POSSIBLE_NUM_OF_ANTENNAS];
extern RateMask_t enableHeNssRatesInMask[VHT_HE_NUMBER_OF_NSS]; 
extern uint8 AggRateLimitMode;
extern uint16 AggRateLimitMaxRate;
extern int8 minPowerFromInternalLowerLimit[LA_NUM_OF_BANDWIDTH];
extern LinkAdaptationPhyMode_e ConvertTcrPhyMode2LaPhyMode[PHY_MODE_LAST]; 
extern const uint8 AntennaCountOffset[MAX_POSSIBLE_NUM_OF_ANTENNAS];
extern uint8 antPowerBoost[MAX_NUM_OF_ANTENNAS];
extern bool MuGroupFixedRateRequested;
extern uint8 enableExtraLogs;

extern LaFixedBeamformingParams_t bfDebugParams;
extern bool enableFastDrop;
extern HeUspBitmap_t BaReportReceivedForUserIndication;

#endif /* LINK_ADAPTATION_H_ */

