/***********************************************************************************
 File:		LinkAdaptation_StateMachine.c
 Module:		LinkAdaptation 
 Purpose: 	find the best transmission parameters for achieve best rate
 Description:	This file contains all definitions and the structures of the RateAdaptation
 				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
//#include "LinkAdaptation.h"
#include "LinkAdaptation_StateMachine.h"
#include "Estimators.h"
#include "Pac_Api.h"
#include "Beamforming.h"
#include "ShramStatistics.h"
#include "stringLibApi.h"
#include "ProtectionAdaptation.h"
#include "LinkAdaptationCommon.h"
#include "RateAdaptation.h"
#include "BwAdaptation.h"
#include "QAMplus.h"
#include "AntennaSelection.h"
#include "ShramStationDatabase.h"
#include "DynamicTxOP.h"
#include "CommonRamLinkAdaptation.h"
#include "CyclicPrefix.h"

#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 12

#define WAIT_DOR_NEX_PROBE_CYCLE_STATE_NUMBER_OF_EVENTS 6
#define WAIT_FOR_NEXT_FAST_PROBE_CYCLE_STATE_NUMBER_OF_EVENTS 5
#define WAIT_FOR_NEXT_SLOW_PROBE_CYCLE_NUMBER_OF_EVENTS 5

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef void (*LaStateMachineFunctionEntry_t)(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void linkAdaptationStateMachine(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, LinkAdaptationEvent_e inputEvent);
static void bypassLinkAdaptation(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForNextProbeCycleStateBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForNextProbeCycleStateProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForFastProbeValidStateBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForFastProbeValidStateBaProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForFastProbeValidStateProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForSlowProbeValidStateBaProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForSlowProbeValidStateBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void waitForSlowProbeValidStateProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void nonValidEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void prepareSlowProbingTransmissionProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void prepareSlowProbingTransmissionSeqReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void prepareSlowProbingTransmissionOverrideParamEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void prepareSlowProbingTransmissionBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void probeResponseIgnore(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void setFixedRateStateTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void setFixedRateStateProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void timerAgerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void processSequencerReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void processSequencerProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void laStaNotConnectRaBypassStateSequencerReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);
static void laSmPerformExecuteParamsOveride(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const LaStateMachineFunctionEntry_t afpLaStateMachineFunc[LA_NUMBER_OF_STATES][LA_NUMBER_OF_EVENTS] = 
{
	/*STATE - LA_STA_NOT_CONNECT_RA_BYPASS  */
	{
		bypassLinkAdaptation,							/*EVENT - BA_TX_REPORT_RECEIVED */
		bypassLinkAdaptation,							/*EVENT - BA_PROBE_RESPONSE_RECEIVED*/
		bypassLinkAdaptation, 							/*EVENT - PROBING_TIMER_EVENT*/
		laStaNotConnectRaBypassStateSequencerReportEvent,/*EVENT - SEQ_TX_REPORT_RECEIVED*/
		laStaNotConnectRaBypassStateSequencerReportEvent,/*EVENT - SEQ_PROBE_RES_RECEIVED*/
		bypassLinkAdaptation,							/*EVENT - RA_TIMER_AGER_EVENT*/
		laSmPerformExecuteParamsOveride,				/*EVENT - RA_OVERRIDE_PARAM_EVENT*/ 
		bypassLinkAdaptation,							/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ 
		
	},

	/*STATE - LA_WAIT_FOR_FILTER_OPEN	*/
	{
		bypassLinkAdaptation,								/*EVENT - BA_TX_REPORT_RECEIVED 	 */
		bypassLinkAdaptation,								/*EVENT - BA_PROBE_RESPONSE_RECEIVED */
		bypassLinkAdaptation,							  	/*EVENT - PROBING_TIMER_EVENT		 */
		laStaNotConnectRaBypassStateSequencerReportEvent, 	/*EVENT - SEQ_TX_REPORT_RECEIVED	 */
		laStaNotConnectRaBypassStateSequencerReportEvent, 	/*EVENT - SEQ_PROBE_RES_RECEIVED	 */
		bypassLinkAdaptation,							  	/*EVENT - RA_TIMER_AGER_EVENT		 */
		bypassLinkAdaptation,							  	/*EVENT - RA_OVERRIDE_PARAM_EVENT	 */ 
		bypassLinkAdaptation,								/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ 
	},

	/*STATE - LA_WAIT_FOR_NEXT_PROBE_CYCLE  */
	{
		waitForNextProbeCycleStateBaTxReportEvent,		/*EVENT - BA_TX_REPORT_RECEIVED */
		probeResponseIgnore,							/*EVENT - BA_PROBE_RESPONSE_RECEIVED*/
		waitForNextProbeCycleStateProbingTimerEvent, 	/*EVENT - PROBING_TIMER_EVENT*/
		processSequencerReportEvent,					/*EVENT - SEQ_TX_REPORT_RECEIVED*/
		processSequencerProbeResponseEvent	,			/*EVENT - SEQ_PROBE_RES_RECEIVED*/
		timerAgerEvent,									/*EVENT - RA_TIMER_AGER_EVENT*/
		laSmPerformExecuteParamsOveride,				/*EVENT - RA_OVERRIDE_PARAM_EVENT*/ 
		bypassLinkAdaptation,									/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ 
	},
	/*STATE - LA_WAIT_FOR_FAST_PROBE_VALID  */
	{
		waitForFastProbeValidStateBaTxReportEvent,		/*EVENT - BA_TX_REPORT_RECEIVED */
		waitForFastProbeValidStateBaProbeResponseEvent,	/*EVENT - BA_PROBE_RESPONSE_RECEIVED*/
		waitForFastProbeValidStateProbingTimerEvent, 	/*EVENT - PROBING_TIMER_EVENT*/
		processSequencerReportEvent,					/*EVENT - SEQ_TX_REPORT_RECEIVED*/
		processSequencerProbeResponseEvent,				/*EVENT - SEQ_PROBE_RES_RECEIVED*/
		timerAgerEvent,									/*EVENT - RA_TIMER_AGER_EVENT*/
		bypassLinkAdaptation,							/*EVENT - RA_OVERRIDE_PARAM_EVENT*/ 
		raFastProbingProcessFeedback,					/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ 
	},
	/*STATE - LA_WAIT_FOR_SLOW_PROBE_VALID  */
	{
		waitForSlowProbeValidStateBaTxReportEvent,		/*EVENT - BA_TX_REPORT_RECEIVED */
		waitForSlowProbeValidStateBaProbeResponseEvent,	/*EVENT - BA_PROBE_RESPONSE_RECEIVED*/
		waitForSlowProbeValidStateProbingTimerEvent, 	/*EVENT - PROBING_TIMER_EVENT*/
		processSequencerReportEvent,					/*EVENT - SEQ_TX_REPORT_RECEIVED*/
		processSequencerProbeResponseEvent,				/*EVENT - SEQ_PROBE_RES_RECEIVED*/
		timerAgerEvent,									/*EVENT - RA_TIMER_AGER_EVENT*/
		bypassLinkAdaptation,							/*EVENT - RA_OVERRIDE_PARAM_EVENT*/ 
		bypassLinkAdaptation,							/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ /*Currently this event is use only for MU but slow probing is not activate for MU*/
	},
	/*STATE - LA_PREPARE_SLOW_PROBING_TRNASMISSION  */
	{
		prepareSlowProbingTransmissionBaTxReportEvent,		/*EVENT - BA_TX_REPORT_RECEIVED */
		nonValidEvent,										/*EVENT - BA_PROBE_RESPONSE_RECEIVED*/
		prepareSlowProbingTransmissionProbingTimerEvent,	/*EVENT - PROBING_TIMER_EVENT*/
		prepareSlowProbingTransmissionSeqReportEvent,		/*EVENT - SEQ_TX_REPORT_RECEIVED*/
		prepareSlowProbingTransmissionSeqReportEvent,		/*EVENT - SEQ_PROBE_RES_RECEIVED*/
		timerAgerEvent,										/*EVENT - RA_TIMER_AGER_EVENT*/
		prepareSlowProbingTransmissionOverrideParamEvent,	/*EVENT - RA_OVERRIDE_PARAM_EVENT*/
		bypassLinkAdaptation,								/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ 
	},
	/*STATE - LA_SET_FIXED_RATE  */
	{
		setFixedRateStateTxReportEvent,					/*EVENT - BA_TX_REPORT_RECEIVED */
		setFixedRateStateProbeResponseEvent,			/*EVENT - BA_PROBE_RESPONSE_RECEIVED*/
		setFixedRateStateProbeResponseEvent,			/*EVENT - PROBING_TIMER_EVENT*/
		bypassLinkAdaptation,							/*EVENT - SEQ_TX_REPORT_RECEIVED*/
		bypassLinkAdaptation,							/*EVENT - SEQ_PROBE_RES_RECEIVED*/
		bypassLinkAdaptation,							/*EVENT - RA_TIMER_AGER_EVENT*/
		bypassLinkAdaptation,							/*EVENT - RA_OVERRIDE_PARAM_EVENT*/ 
		bypassLinkAdaptation,							/*EVENT - LA_PROCESS_FEEDBACK_OUT_OF_SM*/ 
	}
};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						function definitions									
/----------------------------------------------------------------------------------*/

static void linkAdaptationStateMachine(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, LinkAdaptationEvent_e inputEvent)
{	
	ILOG2_DD("linkAdaptationStateMachine: la_state %d, event %d", laDbDistributionParameter->laStaGroupCommon->linkAdaptationState, inputEvent);
	/*Run corresponding function*/
	afpLaStateMachineFunc[laDbDistributionParameter->laStaGroupCommon->linkAdaptationState][inputEvent](laDbDistributionParameter);

}

/********************************************************************************
bypassLinkAdaptation




Description:
------------

State: bypassLinkAdaptation 
Event: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
static void bypassLinkAdaptation(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	UNUSED_PARAM(laDbDistributionParameter);
	ILOG0_D("[LA STATE MACHINE - BY PASS EVENT, current state = %d", laDbDistributionParameter->laStaGroupCommon->linkAdaptationState);
}
/********************************************************************************
bypassLinkAdaptation




Description:
------------

State: bypassLinkAdaptation 
Event: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
static void timerAgerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
#ifdef LINK_ADAPTATION_LOGS		
	ILOG0_D("timerAgerEvent. staIndex %d", laDbDistributionParameter->stationOrGroupIndex);
#endif
	/*Check estimaotrs aging and reset if needed*/
	estimatorsAgingCheck(laDbDistributionParameter);
	
	/*Check Rate adaptation state TO and reset state if needed*/
	rateAdaptationCheckStateTimeout(laDbDistributionParameter);
}

/********************************************************************************
setFixedRateStateTxReportEvent



Description:
------------
Change to fixed rate, if we are not waiting to probe reponse
State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
static void setFixedRateStateTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LaTcrModificationStruct_t tcrParams;
	LaFixedRateStationParams_t *laFixedRateStationParams_p; 
//	ILOG0_V("setFixedRateStateTxReportEvent");

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{

		/*If we are not waitning for probe change rate and change state to LA_STA_NOT_CONNECT_RA_BYPASS*/
		if (laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse == FALSE)
		{
			laStateMachineChangeState(laDbDistributionParameter,LA_STA_NOT_CONNECT_RA_BYPASS) ;
			setStaFixedRatesInHwDb(laDbDistributionParameter->stationOrGroupIndex);
			laFixedRateStationParams_p = &LinkAdaptationStaDatabase[laDbDistributionParameter->stationOrGroupIndex].laStationUnique.fixedRateStationParams;
			CyclicPrefixUpdateDbWithFixedParams(laDbDistributionParameter, laFixedRateStationParams_p->cpMode);

			/*Update statistics*/
			memset (&tcrParams, 0, sizeof(LaTcrModificationStruct_t));
			tcrParams.controlParams.staIndex = laDbDistributionParameter->stationOrGroupIndex;
			tcrParams.controlParams.isVhtSta = laFixedRateStationParams_p->phyMode == PHY_MODE_11AC ? 1 : 0;
			tcrParams.controlParams.firstBwToUpdate = laFixedRateStationParams_p->bandwidth;
			tcrParams.controlParams.lastBwToUpdate = laFixedRateStationParams_p->bandwidth;
			tcrParams.controlParams.packetType = LA_PACKET_TYPE_DATA;
			tcrParams.controlParams.uspIndex = laDbDistributionParameter->uspIndex;
#ifdef ENET_INC_ARCH_WAVE600
			tcrParams.controlParams.isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
#endif
#ifdef ENET_INC_ARCH_WAVE600D2
			tcrParams.controlParams.isHeExt= laFixedRateStationParams_p->phyMode == PHY_MODE_11AX_SU_EXT? TRUE : FALSE;
#endif

			getTcrParamsFromHwDb(&tcrParams);
			updateLaStatistics(&tcrParams);
            
			/*Update statistics*/
			tcrParams.controlParams.packetType = LA_PACKET_TYPE_MANAGEMENT;
			getTcrParamsFromHwDb(&tcrParams);
			updateLaStatistics(&tcrParams);
            
		}
	}
	else /* MU*/ 
	{
		/* Calc group fixed rate!*/ 
		estimatorsWorkingPointDbUpdate(laDbDistributionParameter,&CurrentBaaReport); 
	}	
}
/********************************************************************************
setFixedRateStateProbeResponseEvent







Description:
------------

State: 

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/

static void setFixedRateStateProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t *pDistParam)
{
	LaTcrModificationStruct_t tcrParams;
	LaFixedRateStationParams_t *laFixedRateStationParams_p;

#ifdef LINK_ADAPTATION_LOGS
	//("setFixedRateStateProbeResponseEvent");
#endif
	pDistParam->laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = FALSE;
	/*When we get probe reposne we can change rate in sta dB, and change state to LA_STA_NOT_CONNECT_RA_BYPASS*/
	laStateMachineChangeState(pDistParam,LA_STA_NOT_CONNECT_RA_BYPASS) ;
	setStaFixedRatesInHwDb(pDistParam->stationOrGroupIndex);
	
	laFixedRateStationParams_p = &LinkAdaptationStaDatabase[pDistParam->stationOrGroupIndex].laStationUnique.fixedRateStationParams;

	CyclicPrefixUpdateDbWithFixedParams(pDistParam, laFixedRateStationParams_p->cpMode);
	
	/*Update statistics*/
	memset (&tcrParams, 0, sizeof(LaTcrModificationStruct_t));
	tcrParams.controlParams.staIndex = pDistParam->stationOrGroupIndex;
	tcrParams.controlParams.isVhtSta = laFixedRateStationParams_p->phyMode == PHY_MODE_11AC ? 1 : 0;
	tcrParams.controlParams.firstBwToUpdate = laFixedRateStationParams_p->bandwidth;
	tcrParams.controlParams.lastBwToUpdate = laFixedRateStationParams_p->bandwidth;
	tcrParams.controlParams.packetType = FixedRateParameters.changeType;
	tcrParams.controlParams.uspIndex = pDistParam->uspIndex; 
#ifdef ENET_INC_ARCH_WAVE600
	tcrParams.controlParams.isHeGroup = (pDistParam->laHeGroupUnique != NULL);
#endif
#ifdef ENET_INC_ARCH_WAVE600D2
	tcrParams.controlParams.isHeExt= laFixedRateStationParams_p->phyMode == PHY_MODE_11AX_SU_EXT? TRUE : FALSE;
#endif

	getTcrParamsFromHwDb(&tcrParams);
	updateLaStatistics(&tcrParams);

    tcrParams.controlParams.packetType = LA_PACKET_TYPE_MANAGEMENT;
	getTcrParamsFromHwDb(&tcrParams);
	updateLaStatistics(&tcrParams);
}

/********************************************************************************
waitForNextProbeCycleStateBaTxReportEvent




Description:
------------

State: waitForNextProbeCycle - no probe event was initiate
Event: TxReport

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
static void waitForNextProbeCycleStateBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{

	RaEventObj_t* pEventObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler.mpduEvent);
	uint8	slowProbingKth = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->mpduEventSlowProbingKThInit;
	bool countEventAndHandleProbe = TRUE;

	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}

	/*Update estimators working point*/
	estimatorsWorkingPointDbUpdate(laDbDistributionParameter, &CurrentBaaReport);
	
	/*Protection handler*/
	ProtectionAdaptation_Handler(CurrentBaaReport.userReport.txQStaId ,CurrentBaaReport.userReport.txQVapId, laDbDistributionParameter);
	
	/*BW adaptation Handler*/
	BwAdaptation_Handler(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/* Dynamic TxOP handler */
		DynamicTxop_Handler(laDbDistributionParameter->stationOrGroupIndex);
	}

	/*Set stability*/
	rateAdaptationUpdateStabilityState(laDbDistributionParameter); 

	if (CurrentPacketParams.estimatorsIndication.consequtiveFailure == TRUE)
	{
		countEventAndHandleProbe = rateAdaptationFastDrop(laDbDistributionParameter);
	}
	if (countEventAndHandleProbe == TRUE)
	{
		/*Count Tx event*/ 
		laDbDistributionParameter->laStaGroupCommon->probingIndication = rateAdaptationCountEvent(laDbDistributionParameter,FALSE,pEventObj, slowProbingKth);
		laStateMachineUpdateCurrentSlowProbingBitMap(laDbDistributionParameter->laStaGroupCommon->probingIndication,&(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData));
		rateAdaptationProbingHandler(laDbDistributionParameter, FALSE);
	}
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/*check if Single user probing valid*/
		linkAdaptationSuProbingHandler(laDbDistributionParameter);
	}
	else /*MU*/
	{
		/*check if MU group grade is valid*/
		linkAdaptationMuPreliminaryHandler(laDbDistributionParameter);
	}
		
#ifdef LINK_ADAPTATION_LOGS	
	//	("waitForNextProbeCycleStateBaTxReportEvent, FastProbingCounter = %d, SlowProbingCounter = %d, pr0 = %d, pr1 = %d, pr2 = %d, pr3 = %d, nextProbeIndication = %d",
	//				pEventObj->raEventsCounters.fastProbingCounter,pEventObj->raEventsCounters.slowProbingCounter, 
	//				pEventObj->raEventsCounters.slowProbingPrCounterArr[0], pEventObj->raEventsCounters.slowProbingPrCounterArr[1], pEventObj->raEventsCounters.slowProbingPrCounterArr[2], pEventObj->raEventsCounters.slowProbingPrCounterArr[3], 
	//				pLinkAdaptationDb->rateAdaptationDataBase.probingIndication);
#endif
}
/********************************************************************************
waitForNextProbeCycleStateProbingTimerEvent



Description:
------------
State: waitForNextProbeCycle - no probe event was initiate
Event: Timer event

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/

static void waitForNextProbeCycleStateProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	RaEventObj_t* pEventObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler.timerEvent);
	uint8	slowProbingKth = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->timerEventSlowProbingKThInit;
	/*Count Tx event*/
	laDbDistributionParameter->laStaGroupCommon->probingIndication = rateAdaptationCountEvent(laDbDistributionParameter, FALSE,pEventObj, slowProbingKth);
	/*Initiate probe according to rateAdaptationCountEvent probingIndication*/
	laStateMachineUpdateCurrentSlowProbingBitMap(laDbDistributionParameter->laStaGroupCommon->probingIndication,&(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData));
	rateAdaptationProbingHandler(laDbDistributionParameter, FALSE);
#ifdef LINK_ADAPTATION_LOGS	
//	("waitForNextProbeCycleStateProbingTimerEvent, FastProbingCounter = %d, SlowProbingCounter = %d, pr0 = %d, pr1 = %d, pr2 = %d, pr3 = %d, nextProbeIndication = %d",
//				pEventObj->raEventsCounters.fastProbingCounter,pEventObj->raEventsCounters.slowProbingCounter, 
//				pEventObj->raEventsCounters.slowProbingPrCounterArr[0], pEventObj->raEventsCounters.slowProbingPrCounterArr[1], pEventObj->raEventsCounters.slowProbingPrCounterArr[2], pEventObj->raEventsCounters.slowProbingPrCounterArr[3], 
//				pLinkAdaptationDb->rateAdaptationDataBase.probingIndication);
#endif
}
/********************************************************************************
processSequencerReportEvent



Description:
------------
State: waitForNextProbeCycle - no probe event was initiate
Event: Timer event

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void processSequencerReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	StaId stationIndex = laDbDistributionParameter->stationOrGroupIndex;	
	
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		if (isBfReportNotReceivedAfterRequest(stationIndex) == TRUE)
		{
				/*Stop sending BF request if not received BF report in response*/
				stopSendBfRequest(stationIndex);
		}
	}
	/*Call estimators module for proccesing report*/
	estimatorsProcessSequencerReport(&CurrentSequencerReport, laDbDistributionParameter);
	estimatorsUpdateCollisionEst(&CurrentSequencerReport, laDbDistributionParameter);
	/*Protection handler*/
	if((CurrentSequencerReport.protectionSent == TRUE) && (CurrentSequencerReport.protectionSucceeded != TRUE))
	{
		ILOG0_V("processSequencerReportEvent, CALL ProtectionAdaptation_Handler!!");
		/*In this case data will not send and protectionhandler should be process after sequencer report*/
		ProtectionAdaptation_Handler(stationIndex, CurrentSequencerReport.vapIndex, laDbDistributionParameter);
	}

}
/********************************************************************************
processSequencerProbeResponseEvent




Description:
------------
State: waitForNextProbeCycle - no probe event was initiate
Event: Timer event

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/

static void processSequencerProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	StaId stationIndex = laDbDistributionParameter->stationOrGroupIndex;
	
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		if (isBfReportNotReceivedAfterRequest(stationIndex) == TRUE)
		{
				/*Stop sending BF request if not received BF report in response*/
				stopSendBfRequest(stationIndex);
		}
	}
	estimatorsProcessSequencerReport(&CurrentSequencerReport, laDbDistributionParameter);
	estimatorsUpdateCollisionEst(&CurrentSequencerReport, laDbDistributionParameter);

	/*Protection handler*/
	if((CurrentSequencerReport.protectionSent == TRUE) && (CurrentSequencerReport.protectionSucceeded != TRUE))
	{
		ILOG0_V("processSequencerReportEvent, CALL ProtectionAdaptation_Handler!!");

		/*In this case data will not send and protectionhandler should be process after sequencer report*/
		ProtectionAdaptation_Handler(stationIndex, CurrentSequencerReport.vapIndex, laDbDistributionParameter);
	}
}
/********************************************************************************
laStaNotConnectRaBypassStateSequencerReportEvent






Description:
------------
State: laStaNotConnectRaBypassState
Event: Timer event

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/

static void laStaNotConnectRaBypassStateSequencerReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	StaId stationIndex;
	
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		stationIndex = laDbDistributionParameter->stationOrGroupIndex;	
		if (isBfReportNotReceivedAfterRequest(stationIndex) == TRUE)
		{
				/*Stop sending BF request if not received BF report in response*/
				stopSendBfRequest(stationIndex);
		}
	}
}


/********************************************************************************
waitForFastProbeValidStateBaTxReportEvent





Description:
------------
State: waitForFastProbeValid - Fast probe was initiated, wait for probe response will be received (in BAA FIFO) and valid
Event: TxReportEvent - Tx report has received

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
static void waitForFastProbeValidStateBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	RateAdaptationDataBase_t* raDbEntry =  &(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
	RaEventObj_t* pEventObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler.mpduEvent);
	uint8 slowProbingKth = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->mpduEventSlowProbingKThInit;
	bool countEventAndHandleProbe = TRUE;
	bool workingPointChanged = FALSE;
	
	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}
	else
	{
		/*check if MU group grade is valid*/
		linkAdaptationMuPreliminaryHandler(laDbDistributionParameter);
	}

	/*Update estimators working point*/
	estimatorsWorkingPointDbUpdate(laDbDistributionParameter, &CurrentBaaReport);
	
	/*Protection handler*/
	ProtectionAdaptation_Handler(CurrentBaaReport.userReport.txQStaId ,CurrentBaaReport.userReport.txQVapId,  laDbDistributionParameter);

	/*BW adaptation Handler*/
	BwAdaptation_Handler(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/* Dynamic TxOP handler */
		DynamicTxop_Handler(laDbDistributionParameter->stationOrGroupIndex);
	}

	/*Count event*/
	laDbDistributionParameter->laStaGroupCommon->probingIndication = rateAdaptationCountEvent(laDbDistributionParameter, TRUE,pEventObj, slowProbingKth);
	

	/*Probe can be already sent, wait for response before call rateAdaptationProbingHandler for initiate another porbe event*/
	if (raDbEntry->waitForProbeResponse == FALSE)
	{
		workingPointChanged = executeParamOverride(laDbDistributionParameter);
		if  (TRUE == workingPointChanged)
		{
			/*When working point changing, LA state change in addition,  RateAdaptation and estimators DB are also called - we cannot continue with this state handler - WLANRTSYS-12548*/
			return;
		}
		if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
		{
			if(CurrentPacketParams.estimatorsIndication.consequtiveFailure == TRUE)
			{
				countEventAndHandleProbe = rateAdaptationFastDrop(laDbDistributionParameter);
			}
			if ((countEventAndHandleProbe == TRUE) && (laDbDistributionParameter->laStaGroupCommon->linkAdaptationState == LA_WAIT_FOR_FAST_PROBE_VALID))
			{
				rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
			}
		}
		else /* MU group*/
		{
			if (CurrentPacketParams.estimatorsIndication.consequtiveFailure == TRUE) 
			{
				rateAdaptationFastDrop(laDbDistributionParameter);
			}
			if(laDbDistributionParameter->laStaGroupCommon->linkAdaptationState == LA_WAIT_FOR_FAST_PROBE_VALID)
			{
				/*If state changed as a result of fast drop dont call probing handler with same probing point*/
				rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
			}
		}
		
	}
#ifdef LINK_ADAPTATION_LOGS	
//	("waitForFastProbeValidStateBaTxReportEvent, FastProbingCounter = %d, SlowProbingCounter = %d, pr0 = %d, pr1 = %d, pr2 = %d, pr3 = %d, nextProbeIndication = %d",
//				pEventObj->raEventsCounters.fastProbingCounter,pEventObj->raEventsCounters.slowProbingCounter, 
//				pEventObj->raEventsCounters.slowProbingPrCounterArr[0], pEventObj->raEventsCounters.slowProbingPrCounterArr[1], pEventObj->raEventsCounters.slowProbingPrCounterArr[2], pEventObj->raEventsCounters.slowProbingPrCounterArr[3], 
//				pLinkAdaptationDb->rateAdaptationDataBase.probingIndication);
#endif
}

/********************************************************************************
waitForFastProbeValidStateBaProbeResponseEvent



Description:
------------
State: waitForFastProbeValid - Fast probe was initiated, wait for probe response will be received (in BAA FIFO) and valid
Event: ProbeResponse - Probe response has received , if valid - change to LA_WAIT_FOR_NEXT_PROBE_CYCLE state and check if WP has changed

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/

static void waitForFastProbeValidStateBaProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t* pLinkadaptaionConfigParam = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	LaStationUspCommonParams_t* pLaStaUspCommon = laDbDistributionParameter->laStaUspCommon;
	bool probeIsValid = FALSE;
	bool changeState = FALSE;
	RaEventObj_t* pEventObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler.mpduEvent);
	uint8 slowProbingKth = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->mpduEventSlowProbingKThInit;
	bool workingPointChanged = FALSE;

#ifdef ENET_INC_ARCH_WAVE600
	bool isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
#endif	
	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);
	
	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}

	/*Update fast probing estimators*/
	estimatorsFastProbingDbUpdate(laDbDistributionParameter,&CurrentBaaReport);

	/*Probe response arrived, clear bit*/
	laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = FALSE;
	/*Count event*/
	laDbDistributionParameter->laStaGroupCommon->probingIndication = rateAdaptationCountEvent(laDbDistributionParameter, TRUE,pEventObj, slowProbingKth);

	/*executeParamOverride func should be done before another probe is initiated ( rateAdaptationProbingHandler)*/
	workingPointChanged = executeParamOverride(laDbDistributionParameter);
	if	(TRUE == workingPointChanged)
	{
		/*When working point changing, LA state change in addition,  RateAdaptation and estimators DB are also called - we cannot continue with this state handler - WLANRTSYS-12548*/
		return;
	}

	/*Check validation criteria*/
	/*For HE group - Because of dummy user handling - validatio counter should not increment if user probing is alreay valid - it will be increement again after all users will be valid*/
	if ((!isHeGroup)|| (LA_GET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap,laDbDistributionParameter->uspIndex, BITMAP_ARRAY_MAX_SIZE_2) != TRUE))
	{
		probeIsValid = estimatorsProbingValidationCriteria(&(pLaStaUspCommon->probingPointValidationCounter),pLinkadaptaionConfigParam->fastProbingValidationTh);
	}
	if (probeIsValid == TRUE)
	{
		/*Probe received and valid*/
		/*Updae WP if needed*/
		raFastProbingProcessFeedback(laDbDistributionParameter);

		if(laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX) /* in case of MU*/ 
		{
#ifdef ENET_INC_ARCH_WAVE600
			if (isHeGroup)
			{
				LA_SET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap,laDbDistributionParameter->uspIndex, BITMAP_ARRAY_MAX_SIZE_2); //KW_FIX_FW_G Added array bound check
				if((laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] == laDbDistributionParameter->laHeGroupUnique->probingValidationMask.heUspBitmap[0]) && (laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] == laDbDistributionParameter->laHeGroupUnique->probingValidationMask.heUspBitmap[1]))
				{
					laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] = 0x0; /*reset probing validation bitmap*/
					laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] = 0x0; 
					changeState = TRUE; /* only when all USPs probe is valid - change the group's state*/
				}
			}
			else
#endif
			{
				laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap |= (0x1<<laDbDistributionParameter->uspIndex);  
				if(laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap == laDbDistributionParameter->laVhtGroupUnique->probingValidationMask)
				{
					laDbDistributionParameter->laVhtGroupUnique->probingValidationBitmap = 0x0; /*reset probing validation bitmap*/
					changeState = TRUE; /* only when all USPs probe is valid - change the group's state*/
				}
			}
		}
		else /* SU */
		{
			changeState = TRUE; 
		}

		if(changeState == TRUE)
		{
			/*Change state*/
			laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
			/*Initiate another porbe according to probingIndication*/
			rateAdaptationProbingHandler(laDbDistributionParameter, FALSE);
		}
	}
	else /* probe isn't valid*/
	{
		/*Probe received but not valid yet - wait for another probe*/
		/*Initiate another fast probe according to probingIndication*/
		rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
	}
}


/********************************************************************************
waitForFastProbeValidStateProbingTimerEvent



Description:
------------
State: waitForFastProbeValid - Fast probe was initiated, wait for probe response will be received (in BAA FIFO) and valid
Event: ProbingTimerEvent - Timer has expired, increment event handler counters and initiate another probe if needed

Input:
-----
stationIndex


Output:
-------
Returns:
--------

********************************************************************************/
static void waitForFastProbeValidStateProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	RateAdaptationDataBase_t* raDbEntry = &(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
	RaEventObj_t* pEventObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler.timerEvent);
	uint8 slowProbingKth = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->timerEventSlowProbingKThInit;

	/*Count event and produce probingIndication*/
	laDbDistributionParameter->laStaGroupCommon->probingIndication= rateAdaptationCountEvent(laDbDistributionParameter, TRUE, pEventObj, slowProbingKth);
	if (raDbEntry->waitForProbeResponse == FALSE)
	{
		/*If previous probe already receieved, call probing handler and initiate another probe if needed according to probingIndication*/
		rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
	}
#ifdef LINK_ADAPTATION_LOGS	
//	("waitForFastProbeValidStateProbingTimerEvent, FastProbingCounter = %d, SlowProbingCounter = %d, pr0 = %d, pr1 = %d, pr2 = %d, pr3 = %d, nextProbeIndication = %d",
//				pEventObj->raEventsCounters.fastProbingCounter,pEventObj->raEventsCounters.slowProbingCounter, 
//				pEventObj->raEventsCounters.slowProbingPrCounterArr[0], pEventObj->raEventsCounters.slowProbingPrCounterArr[1], pEventObj->raEventsCounters.slowProbingPrCounterArr[2], pEventObj->raEventsCounters.slowProbingPrCounterArr[3], 
//				pLinkAdaptationDb->rateAdaptationDataBase.probingIndication);
#endif
}


/********************************************************************************
waitForSlowProbeValidStateBaProbeResponseEvent



Description:
------------
State:  waitForSlowProbeValid - Slow probe was initiated, wait for probe response will be received (in BAA FIFO) and valid
Event: ProbeResponse - Probe response has received, if valid - change to LA_WAIT_FOR_NEXT_PROBE_CYCLE state and check if WP has changed

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void waitForSlowProbeValidStateBaProbeResponseEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	LinkAdaptationConfigurationParams_t* pLinkadaptaionConfigParam = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	LaStationUspCommonParams_t* pLaStaUspCommon = laDbDistributionParameter->laStaUspCommon;
	bool probeIsValid = FALSE;
	RaEventObj_t* pEventObj = &(laDbDistributionParameter->laStaGroupCommon->laEventsHandler.mpduEvent);
	uint8 slowProbingKth = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams->mpduEventSlowProbingKThInit;
	uint8* slowProbingCurrentTaskBitmap = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.slowProbingCurrentTaskBitmap);
	bool lastIteration = FALSE;
	bool workingPointChanged = FALSE;

	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}

	estimatorsSlowProbingDbUpdate(laDbDistributionParameter,&CurrentBaaReport);

	/*Probe response arrived, clear bit*/
	laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = FALSE;

	/*executeParamOverride func should be done before another probe is initiated ( rateAdaptationProbingHandler)*/
	workingPointChanged = executeParamOverride(laDbDistributionParameter);
	if	(TRUE == workingPointChanged)
	{
		/*When working point changing, LA state change in addition,  RateAdaptation and estimators DB are also called - we cannot continue with this state handler - WLANRTSYS-12548*/
		return;
	}

	/*Check validation criteria*/
	probeIsValid = estimatorsProbingValidationCriteria(&(pLaStaUspCommon->probingPointValidationCounter),pLinkadaptaionConfigParam->slowProbingValidationTh);

	if (probeIsValid == TRUE)
	{
		/*Probe received and valid*/
		/*Update slow WP if needed according to probe filter result*/
		/*Call process feedback evenn if per was not update, process feedback function will check if filter is valid and of not cit will change next probing point*/
		raSlowProbingProcessFeedback(laDbDistributionParameter);
		lastIteration = checkMaxSlowProbingIterations(laDbDistributionParameter);

		/*Reset current slow probing task after task finished, it will be update again at the next slow probing task*/
		setCurrentSlowProbingTaskInDb(laDbDistributionParameter,SLOW_PROBING_INVALID_TASK);
		if ((lastIteration != TRUE) && ((*slowProbingCurrentTaskBitmap) != 0))
		{
			/*Continue with next slow porbing loop*/
			rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
		}
		else
		{
			/*Slow probing loops for current cycle are done, change state*/
			/*Count event and update probing indication*/
			laDbDistributionParameter->laStaGroupCommon->probingIndication = rateAdaptationCountEvent(laDbDistributionParameter, FALSE, pEventObj, slowProbingKth);
			/*Change state to LA_WAIT_FOR_NEXT_PROBE_CYCLE*/
			laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
		}
	}
	else
	{
		/*Stay with last slow probing point - initiate another slow probing at the next transmission*/
		rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
	}
}


/********************************************************************************
waitForSlowProbeValidStateBaTxReportEvent



Description:
------------
State: waitForSlowProbeValid - Slow probe was initiated, wait for probe response will be received (in BAA FIFO) and valid
event: TxRepor - Tx report (not probe response) arrived

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void waitForSlowProbeValidStateBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	RateAdaptationDataBase_t* raDbEntry =  &(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
 	bool countEventAndHandleProbe = TRUE;
	bool workingPointChanged = FALSE;

	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}

	/*Update estimators working point*/
	estimatorsWorkingPointDbUpdate(laDbDistributionParameter, &CurrentBaaReport);

	/*Protection handler*/
	ProtectionAdaptation_Handler(CurrentBaaReport.userReport.txQStaId ,CurrentBaaReport.userReport.txQVapId,  laDbDistributionParameter);

	/*BW adaptation Handler*/
	BwAdaptation_Handler(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/* Dynamic TxOP handler */
		DynamicTxop_Handler(laDbDistributionParameter->stationOrGroupIndex);
	}

	/*Probe can be already sent, wait for response before call rateAdaptationProbingHandler for initiate another porbe event*/
	if (raDbEntry->waitForProbeResponse == FALSE)
	{
		/*executeParamOverride func should be done before another probe is initiated ( rateAdaptationProbingHandler)*/
		workingPointChanged = executeParamOverride(laDbDistributionParameter);
		if	(TRUE == workingPointChanged)
		{
			/*When working point changing, LA state change in addition,  RateAdaptation and estimators DB are also called - we cannot continue with this state handler - WLANRTSYS-12548*/
			return;
		}

		if (CurrentPacketParams.estimatorsIndication.consequtiveFailure == TRUE) 
		{
			countEventAndHandleProbe = rateAdaptationFastDrop(laDbDistributionParameter);

		}
		if (countEventAndHandleProbe == TRUE)
		{
			/*If state changed as a result of fast drop dont call probing handler with same probing point*/
			rateAdaptationProbingHandler(laDbDistributionParameter, TRUE);
		}
	}
}


/********************************************************************************
prepareSlowProbingTransmissionBaTxReportEvent



Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void prepareSlowProbingTransmissionBaTxReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{

	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}

	/*Update estimators working point*/
	estimatorsWorkingPointDbUpdate(laDbDistributionParameter, &CurrentBaaReport);

	ProtectionAdaptation_Handler(CurrentBaaReport.userReport.txQStaId ,CurrentBaaReport.userReport.txQVapId,  laDbDistributionParameter);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		/* Dynamic TxOP handler */
		DynamicTxop_Handler(laDbDistributionParameter->stationOrGroupIndex);
	}

	/*BW adaptation Handler*/
	BwAdaptation_Handler(laDbDistributionParameter->stationOrGroupIndex,laDbDistributionParameter->uspIndex);
}


/********************************************************************************
waitForSlowProbeValidStateProbingTimerEvent




Description:
------------
State:waitForSlowProbeValid - Slow probe was initiated, wait for probe response will be received (in BAA FIFO) and valid
Event: ProbingTimer - Timer has expired, increment event handler counters but without initiate another probe event - when next probe response will be arrived another probe will be initiated

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void waitForSlowProbeValidStateProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	UNUSED_PARAM(laDbDistributionParameter);	
	/*Nothing should be done in this atate , we wait to probe to be valid*/

}


/********************************************************************************
prepareSlowProbingTransmissionSeqReportEvent



Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void prepareSlowProbingTransmissionSeqReportEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	bool isCurrentSlowProbingBitmapEmpty = FALSE;

	/*Call estimators module for proccesing report*/
	estimatorsProcessSequencerReport(&CurrentSequencerReport, laDbDistributionParameter);
	estimatorsUpdateCollisionEst(&CurrentSequencerReport, laDbDistributionParameter);
	
	/*For now this state is used only for BF, it could be expanded to other slow probing modules if required by call the corresponding slow probing funtion according to slow probing indication - TBD*/
	if (laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		if (isBfReportNotReceivedAfterRequest(laDbDistributionParameter->stationOrGroupIndex) == TRUE)
		{
				/*If BF requset (NDPA / NDP) sent and BF report has not received for prepareSlowProbingNumOfIterationsThreshold tries, stop sending NDAP / NDP, and change to RA_WAIT_FOR_NEXT_PROBE_CYCLE state*/
			laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_SLOW_PROBE_VALID);
			BeamformingStateTo(laDbDistributionParameter->stationOrGroupIndex);
			isCurrentSlowProbingBitmapEmpty = disableTaskFromCurrentCycle(laDbDistributionParameter, SLOW_PROBING_BF);
			if (isCurrentSlowProbingBitmapEmpty == TRUE)
			{
				/*No slow probing task in current cycle, change state*/
				laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);
			}
		}
		else if(BeamFormingCheckReadyForTransmission(laDbDistributionParameter->stationOrGroupIndex))
		{
			/*Wait for BF DB valid and than change to LA_WAIT_FOR_SLOW_PROBE_VALID*/
			/*Change state to LA_WAIT_FOR_SLOW_PROBE_VALID*/
			laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_SLOW_PROBE_VALID);
			BeamformingSetNextProbingPointInTcr(laDbDistributionParameter->stationOrGroupIndex);
		}
	}
}


/********************************************************************************
prepareSlowProbingTransmissionOverrideParamEvent


Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void prepareSlowProbingTransmissionOverrideParamEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	uint8* pProbingValidationCounter = &(laDbDistributionParameter->laStaUspCommon->probingPointValidationCounter);

	executeParamOverride(laDbDistributionParameter);
	BeamformingStateTo(laDbDistributionParameter->stationOrGroupIndex);
	*pProbingValidationCounter = 0;
	laStateMachineChangeState(laDbDistributionParameter, LA_WAIT_FOR_NEXT_PROBE_CYCLE);
}

/********************************************************************************
nonValidEvent




Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/

static void nonValidEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	UNUSED_PARAM(laDbDistributionParameter);	
	ASSERT(0);
}
/********************************************************************************
prepareSlowProbingTransmissionProbingTimerEvent




Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
static void prepareSlowProbingTransmissionProbingTimerEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	UNUSED_PARAM(laDbDistributionParameter);
	/*Nothing should be done in this atate because anyway new slow probing will be transmited*/

}

/********************************************************************************
probeResponseIgnore




Description:
------------
Ignore probe response when state is "wait for next probe cycle"
Probe reponse in this state could be only after reset state
Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/

static void probeResponseIgnore(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
#ifdef LINK_ADAPTATION_LOGS
	ILOG0_V("Probe response ignored after state aging");
#endif

	/*Probe response arrived, clear bit*/
	laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = FALSE;

	estimatorsProcessBaaReport(laDbDistributionParameter,&CurrentBaaReport);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		//Call QAMplus must be after estimatorsProcessBaaReport
		QAMplus_NewBAReport(laDbDistributionParameter, &CurrentBaaReport);
	}
}

void updateOverridePendingDb(overridePending_e overridePending, uint8 value, LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	overridePendingDb_t* overridePendingPtr = &(laDbDistributionParameter->laStaUspCommon->overridePending);

	if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
	{
		switch (overridePending)
		{
			case OVERRIDE_RATE_INDEX:
				overridePendingPtr->overrideBitmap.bitFields.rateIndexBit = TRUE;
				overridePendingPtr->overrideVals.rateIndexVal = value;
			break;

			case OVERRIDE_NSS:
				overridePendingPtr->overrideBitmap.bitFields.nssBit = TRUE;
				DEBUG_ASSERT(overridePendingPtr->overrideVals.nssVal < LA_NSS_INVALID);
				overridePendingPtr->overrideVals.nssVal = value; // nssVal range is 0->3, according to number of NSS -1
			break;
			
			case OVERRIDE_ANTENNA_SELECTION:
				overridePendingPtr->overrideBitmap.bitFields.antennaMaskBit = TRUE;
				overridePendingPtr->overrideVals.antennaMaskVal = value;
			break;
			
			case OVERRIDE_BW:
				overridePendingPtr->overrideBitmap.bitFields.bwBit = TRUE;
				overridePendingPtr->overrideVals.bwVal = value;
			break;
			
			case OVERRIDE_VHT_MPDU:
	   			overridePendingPtr->overrideBitmap.bitFields.vhtMpdu = TRUE;
			break;
			
			case ADJUST_RATE_TO_MASK:
				overridePendingPtr->overrideBitmap.bitFields.adjustRate2Mask = TRUE;
			break;
			
			
		}
		laStateMachineEnterEvent(laDbDistributionParameter, LA_OVERRIDE_PARAM_EVENT);
	
		ILOG2_DD("updateOverridePendingDb. override %d value %d", overridePending, value);
	}
	else /* MU */  
	{
		switch (overridePending)
		{
			case OVERRIDE_RATE_INDEX:
				//overridePendingPtr->overrideBitmap.bitFields.rateIndexBit = TRUE;
				//overridePendingPtr->overrideVals.rateIndexVal = value;
			break;

			case OVERRIDE_BW:
				//TBD in executeParamOverride func.
			break;
			
			case ADJUST_RATE_TO_MASK:
				overridePendingPtr->overrideBitmap.bitFields.adjustRate2Mask = TRUE;
			break;

			default:
				DEBUG_ASSERT(0);
			break;
			
		}
		
		laStateMachineEnterEvent(laDbDistributionParameter, LA_OVERRIDE_PARAM_EVENT);

	}
//	SLOG0(0, 0, overridePendingDb_t, overridePendingPtr);
}


bool executeParamOverride(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	overridePendingDb_t* overridePendingPtr = &(laDbDistributionParameter->laStaUspCommon->overridePending);
	bool workingPointChanged = FALSE;
	
	if (overridePendingPtr->overrideBitmap.val != 0)
	{
		uint8 rateIndex = overridePendingPtr->overrideVals.rateIndexVal;
		uint8 antSel = overridePendingPtr->overrideVals.antennaMaskVal;
		Bandwidth_e bw = (Bandwidth_e)overridePendingPtr->overrideVals.bwVal;
		uint8 activatedAntCount = AntennaSelectionGetActivatedAntennasCount();
		uint8 antennaCount = MIN(activatedAntCount, CONVERT_MAX_NUM_OF_NSS_TO_MAX_NUM_OF_ANTENNAS(overridePendingPtr->overrideVals.nssVal));// NSS range is 0->3, antenna count range is from 1 to 4
		uint8 maxNssAccordingToAnt = CONVERT_MAX_NUM_OF_ANTENNAS_TO_MAX_NUM_OF_NSS(antennaCount);
		StaId stationIndex = laDbDistributionParameter->stationOrGroupIndex; 
		SLOG0(0, 0, overridePendingDb_t, overridePendingPtr);
	
		if (overridePendingPtr->overrideBitmap.bitFields.bwBit == TRUE)
		{
			SetStaDataBwLimitInHwDb(stationIndex, bw, TRUE);
			overridePendingPtr->overrideBitmap.bitFields.bwBit = FALSE;
		}
		
		if (overridePendingPtr->overrideBitmap.bitFields.antennaMaskBit == TRUE)
		{
			AntennaSelectionSetStaAntSelMask(stationIndex, antSel, TRUE);
			overridePendingPtr->overrideBitmap.bitFields.antennaMaskBit = FALSE;
			workingPointChanged = TRUE;
		}
				
		if (overridePendingPtr->overrideBitmap.bitFields.rateIndexBit == TRUE)
		{
			uint8 currentRateIndex = estimatorsGetWorkingPointRateIndexOfMainBw(laDbDistributionParameter->laStaUspCommon);
			laStateMachineChangeState(laDbDistributionParameter,LA_WAIT_FOR_NEXT_PROBE_CYCLE);

			/* https://jira-chd.intel.com/browse/WLANRTSYS-15356 */
			rateIndex = rateAdaptationGetLowerRateIndex(laDbDistributionParameter, LinkAdaptationCommonConfiguration.wlanBandwidthMax, rateIndex, laDbDistributionParameter->laStaUspCommon->raIndexMask, 0, FALSE);
			/*Update rate adaptation DB with the new rate*/
			rateAdaptationChangeWorkingPoint(laDbDistributionParameter,rateIndex,currentRateIndex);
			
			overridePendingPtr->overrideBitmap.bitFields.rateIndexBit = FALSE;
			workingPointChanged = TRUE;
		}

		if (overridePendingPtr->overrideBitmap.bitFields.adjustRate2Mask == TRUE)
		{
			rateAdaptationAdjustRateToMask(laDbDistributionParameter);
			overridePendingPtr->overrideBitmap.bitFields.adjustRate2Mask = FALSE;
			workingPointChanged = TRUE;
		}

		if (overridePendingPtr->overrideBitmap.bitFields.nssBit == TRUE)
		{
			setMaskAccordingToMaxNumOfAntennas(laDbDistributionParameter, antennaCount);
			/*Adjust rate to num of antennas*/
			rateAdaptationAdjustRateToMask(laDbDistributionParameter);
			overridePendingPtr->overrideBitmap.bitFields.nssBit = FALSE;
			UpdateStationMaxSupportedNss(stationIndex, (SpatialStreamNum_e)maxNssAccordingToAnt);			
			workingPointChanged = TRUE;
		}

		if (overridePendingPtr->overrideBitmap.bitFields.vhtMpdu == TRUE)
		{
			StaDbHwEntries[stationIndex].common.maxMpduLengthLimit = calculateMpduMaxLengthLimitVht(2);
            StaDbHwEntries[stationIndex].common.maxPsduLengthLimit = VHT_MAX_A_MPDU_SIZE;
#ifdef PHY_STUCK_LDPC_IS_SET_SYMBOLS_ABOVE_400
            StaDbHwEntries[stationIndex].wp.maxPsduLengthLimit = VHT_MAX_A_MPDU_SIZE;
#endif
			LinkAdaptationStaDatabase[stationIndex].laStaUspCommon.staTransmissionParams.vhtSta = TRUE;
			rateAdaptationAdjustRateToMask(laDbDistributionParameter);
			overridePendingPtr->overrideBitmap.bitFields.vhtMpdu = FALSE;
			workingPointChanged = TRUE;
            ILOG0_D("[executeParamOverride], staIdx: %d", stationIndex);
            //SLOG0(0, 0, overridePendingDb_t, overridePendingPtr);
		}
//		SLOG0(0, 0, overridePendingDb_t, overridePendingPtr);
	}
	return workingPointChanged;
}

static void laSmPerformExecuteParamsOveride(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter)
{
	executeParamOverride(laDbDistributionParameter);
}
/********************************************************************************
laStateMachineChangeState


Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
void laStateMachineChangeState(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, LinkAdaptationState_e newState)
{	
	LinkAdaptationConfigurationParams_t* pLaConfigParams = laDbDistributionParameter->laStaGroupCommon->pLinkAdaptationConfigurationParams;
	RateAdaptationDataBase_t* pRaDataBase = &(laDbDistributionParameter->laStaUspCommon->rateAdaptationDataBase);
	uint32 currentTsf = 0;
	uint8* totalIterationsCounter = &(laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.totalIterationsPerCycle);

	if ((laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX) && 
		(laDbDistributionParameter->laStationUnique->isSnifferEapolWaActive == TRUE))
	{
		ILOG0_V("[EAPOL] laStateMachineChangeState skip change state");
		return;
	}

	ILOG0_DD("[EAPOL] laStateMachineChangeState current state = %d new state = %d", laDbDistributionParameter->laStaGroupCommon->linkAdaptationState, newState);

	if (!((laDbDistributionParameter->uspIndex != INVALID_MU_USP_INDEX) && (MuGroupFixedRateRequested == TRUE))) /*Ignore if MU-Fixed Rate*/
	{
		if (newState == LA_WAIT_FOR_NEXT_PROBE_CYCLE)
		{
			laDbDistributionParameter->laStaGroupCommon->slowProbingHandlerData.probingHandlerInterleavingCounter = (1 << (pLaConfigParams->slowProbingInterleavingLog2Val));
			*totalIterationsCounter = 0;
			estimatorsResetSlowProbingEstimators(laDbDistributionParameter);// reset slow probing estimators before skip to next loop
			if (laDbDistributionParameter->laHeGroupUnique != NULL) //HE MU
			{
				laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[0] = 0x0; /*reset probing validation bitmap*/
				laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap[1] = 0x0;
			}
		}
		else if ((newState == LA_WAIT_FOR_FAST_PROBE_VALID) || (newState == LA_WAIT_FOR_SLOW_PROBE_VALID))
		{
			pRaDataBase->waitForProbeResponse = TRUE;
		}

		if (laDbDistributionParameter->laStaGroupCommon->linkAdaptationState != newState)
		{

#ifdef LINK_ADAPTATION_LOGS
			//("laStateMachineChangeState, currentState = %d, newState = %d", pRaDataBase->linkAdaptationState, newState);
#endif
			/*State changed, update raEnterStateTime*/
			currentTsf = GET_TSF_TIMER_LOW();
			currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);

			laDbDistributionParameter->laStaGroupCommon->raEnterStateTime = currentTsf;

			laDbDistributionParameter->laStaGroupCommon->linkAdaptationState = newState;
			if(laDbDistributionParameter->uspIndex == INVALID_MU_USP_INDEX)
			{
				LinkAdaptationStatistics.raState[laDbDistributionParameter->stationOrGroupIndex] = newState; //LINK_ADAPTATION_RA_STATE
			}
			else /*MU*/
			{
				LinkAdaptationMuStatistics.raState[laDbDistributionParameter->stationOrGroupIndex] = newState; 
			}
		}
	}
	else
	{
		laDbDistributionParameter->laStaGroupCommon->linkAdaptationState = LA_STA_NOT_CONNECT_RA_BYPASS;
		*totalIterationsCounter = 0;
	}

}
/********************************************************************************
laStateMachineInit


Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/

void laStateMachineInit(StaId stationIndex, uint8 uspIndex, bool isHeGroup)
{
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	LinkAdaptationDatabaseDistributionPack_t laDbDistributionParameter; 

	updateLaDbDistributionParam(&laDbDistributionParameter,stationIndex,uspIndex, isHeGroup);
	/*Change to state LA_WAIT_FOR_NEXT_PROBE_CYCLE*/
	laStateMachineChangeState(&laDbDistributionParameter, LA_STA_NOT_CONNECT_RA_BYPASS);
	currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);

	laDbDistributionParameter.laStaUspCommon->rateAdaptationDataBase.waitForProbeResponse = FALSE;
	laDbDistributionParameter.laStaGroupCommon->raEnterStateTime = currentTsf;
}

/********************************************************************************
laStateMachineEnterEvent

Description:
------------

Input:
-----


Output:
-------
Returns:
--------

********************************************************************************/
void laStateMachineEnterEvent(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter,LinkAdaptationEvent_e event)
{
	linkAdaptationStateMachine(laDbDistributionParameter, event);
}


/****************************************************************************
 **
 ** NAME:		 laStateMachineProcessBaaReport
 
 **
 ** PARAMETERS:  
 **
 **
 ** RETURN VALUES: 
 **
 **
 ** DESCRIPTION: 
 ** 
 ** 			 
 **
 **   
 ***************************************************************************/
void laStateMachineProcessBaaReport(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, BaaRateAdaptiveFullFifo_t* pCurrentBaaReport,BaaRateAdaptiveFullFifo_t* pLastBaaReport)
{
	bool           isProbingBa;
#ifdef ENET_INC_ARCH_WAVE600	
	uint8          uspIndex;
	bool           isHeGroup = (laDbDistributionParameter->laHeGroupUnique != NULL);
	uint16         userTcraStaAid;
	DlDataPhase_t* phasePtr;
#endif

	if (enableExtraLogs)
	{
		SLOG0(0, 0, BaaRateAdaptiveFullFifo_t, pCurrentBaaReport);
	}

// if HE-MU and dummy-AID we will indicate bitmap of probing as done since we will not receive BA report for this user while mask indicates that we should
#ifdef ENET_INC_ARCH_WAVE600
	if (isHeGroup)
	{
		/*Dummy users handling - for each MU transmission - for users that was not sent in the last MU transmission - increment probing counter and handle validation criteria*/
		/*Process is done on last reprt, hence the previous BA report should be HE type - first time pLastBaaReport is reset so  reportType will be zero*/
		if ((pLastBaaReport->commonReport.reportType == SEQ_REPORT_TYPE_SEQUENCER_HE_MU) && 
			(pLastBaaReport->commonReport.seqTransactionNumber != pCurrentBaaReport->commonReport.seqTransactionNumber))
		{
			linkAdaptationIncProbingCounterForTempDummyUser(pLastBaaReport);
			memset(&BaReportReceivedForUserIndication, 0, sizeof(HeUspBitmap_t));  
		}

		phasePtr = getWpPhaseAddress(laDbDistributionParameter->stationOrGroupIndex);
		for(uspIndex = 0; uspIndex < MAX_USP_IN_HE_GROUP; uspIndex++)
		{
			userTcraStaAid = phasePtr->userPhase[uspIndex].userTcraStaAid;
			if (userTcraStaAid == DUMMY_AID)
			{
				LA_SET_BIT_IN_BITMAP(laDbDistributionParameter->laHeGroupUnique->probingValidationBitmap.heUspBitmap, uspIndex, BITMAP_ARRAY_MAX_SIZE_2); //KW_FIX_FW_G Added array bound check
			}
		}
		linkAdaptationSetBaReportReceivedForRealUser(pCurrentBaaReport->userReport.txUserId);
	}
#endif	

	isProbingBa = (pCurrentBaaReport->userReport.altRateReasonFastProbing | pCurrentBaaReport->userReport.altRateReasonSlowProbing);
	/*per estimators have been changed*/
	if (isProbingBa == 0)
	{
		laStateMachineEnterEvent(laDbDistributionParameter,BA_TX_REPORT_RECEIVED);
	}
	else
	{
		laStateMachineEnterEvent(laDbDistributionParameter,BA_PROBE_RESPONSE_RECEIVED);
	}
	
}

/****************************************************************************
 **
 ** NAME:        laStateMachineProcessBaaReport
 **
 ** PARAMETERS:  
 **stationIndex
 **
 ** RETURN VALUES: 
 **
 **
 ** DESCRIPTION: 
 ** Check estimators indication and trigger Rate Adaptation accordingly
 **              
 **
 **   
 ***************************************************************************/

void laStateMachineProcessSequencerReport(LinkAdaptationDatabaseDistributionPack_t* laDbDistributionParameter, TxSequencerReport_t* pSeqReport)
{
	uint8 probing = (pSeqReport->altRateReasonFastProbing || pSeqReport->altRateReasonSlowProbing);

	/*per estimators have been changed*/
	if (probing == 0)
	{
		laStateMachineEnterEvent(laDbDistributionParameter, SEQ_TX_REPORT_RECEIVED);
	}
	else
	{
		laStateMachineEnterEvent(laDbDistributionParameter,SEQ_PROBE_RES_RECEIVED);
	}
}
/****************************************************************************
 **
 ** NAME:        laStateMachineUpdateCurrentSlowProbingBitMap
 **
 ** PARAMETERS:  
 **stationIndex
 **
 ** RETURN VALUES: 
 **
 **
 ** DESCRIPTION: 
 ** Check estimators indication and trigger Rate Adaptation accordingly
 **              
 **
 **   
 ***************************************************************************/

void laStateMachineUpdateCurrentSlowProbingBitMap(RaProbingIndication_e probingIndication, LaSlowProbingHandlerParams_t* pslowProbingHandlerData)
{
	RaSlowProbingPriority_e nextSlowProbePriority;

	if ((probingIndication != NO_PROBING_INDICATION) && 
		((probingIndication != FAST_PROBING_INDICATION)))/*Slow probing Indication*/
		{
			/*Update slowProbingCurrentTask according to slow probing priority*/
			nextSlowProbePriority = (RaSlowProbingPriority_e)(probingIndication - SLOW_PROBING_PRIORITY0_INDICATION);
			ASSERT(nextSlowProbePriority < SLOW_PROBING_NUMBER_OF_PRIORITIES);
			/*If some of slow probing tasks didn't finished at the last cycle, continue with those tasks in addition to higher priority if needed*/
			/*If all tasks have finished set bitmap according to priority*/
			if (nextSlowProbePriority == SLOW_PROBING_PRIORITY1)
			{
				pslowProbingHandlerData->slowProbingCurrentTaskBitmap |= (pslowProbingHandlerData->slowProbingTaskBitmap[nextSlowProbePriority] & rateAdaptationGetSlowProbingEnBitmap());
			}
			else if(pslowProbingHandlerData->slowProbingCurrentTaskBitmap == 0)
			{
				pslowProbingHandlerData->slowProbingCurrentTaskBitmap = (pslowProbingHandlerData->slowProbingTaskBitmap[nextSlowProbePriority] & rateAdaptationGetSlowProbingEnBitmap());
			}
			/* Clear probing tasks that may remain in slowProbingCurrentTaskBitmap but already cleared from slowProbingTaskBitmap */
			pslowProbingHandlerData->slowProbingCurrentTaskBitmap &= 
			(pslowProbingHandlerData->slowProbingTaskBitmap[SLOW_PROBING_PRIORITY0] | pslowProbingHandlerData->slowProbingTaskBitmap[SLOW_PROBING_PRIORITY1]);
		}
}
