/***********************************************************************************
 File:			Locker.c
 Module:			Locker
 Purpose: 		Manage all request to Hw Locker from FW modules
 Description:		Serialize and schedule all request  from FW modules to Hw Locker
 This module is the API to the Tx Selector responsible for the 
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "TxSelector_Api.h"
#include "QueueUtility_Api.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "Utils_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_LOCKER
#define LOG_LOCAL_FID 0


/* Total number of requests */
#define LOCKER_NUM_OF_REQUESTS			(16)
#define LOCKER_NUM_OF_GROUP_REQUESTS	(2)
#define LOCKER_NUM_OF_PLAN_REQUESTS		(128)

#ifdef ENET_INC_ARCH_WAVE600
#define LOCKER_SINGLE_TID_UPDATE_NUM_OF_BITS 	(0x4)
#else
#define LOCKER_SINGLE_TID_UPDATE_NUM_OF_BITS 	(0x3)
#endif //ENET_INC_ARCH_WAVE600

#define LOCKER_SINGLE_TID_UPDATE_MASK			((0x1 << LOCKER_SINGLE_TID_UPDATE_NUM_OF_BITS) - 0x1)
#define LOCKER_ALL_TID_UPDATE_START_BIT			(LOCKER_SINGLE_TID_UPDATE_NUM_OF_BITS)
#define LOCKER_ALL_TIDS_UPDATE					(0x1 << LOCKER_ALL_TID_UPDATE_START_BIT)


#define LOCKER_PLAN_INDEX_INVALID       (0xFF)
#define LOCKER_TID_INVALID              (0xFF)

typedef enum LockerRequestType
{
	LOCKER_REQUEST_TYPE_LOCK = 0,
	LOCKER_REQUEST_TYPE_DISABLE,
#ifdef ENET_INC_ARCH_WAVE600
    LOCKER_REQUEST_TYPE_PLAN_LOCK,
#endif //ENET_INC_ARCH_WAVE600
	LOCKER_TOTAL_NUM_OF_REQUESTS_TYPES
} LockerRequestType_e;

typedef enum LockerRequestMode
{
	LOCKER_REQUEST_MODE_FORCE = 0,
	LOCKER_REQUEST_MODE_TRY,
	LOCKET_TOTAL_NUM_OF_REQUESTS_MODES
} LockerRequestMode_e;

typedef enum LockerState
{
	LOCKER_STATE_IDLE = 0,
	LOCKER_STATE_BUSY,
	LOCKET_TOTAL_NUM_OF_STATES
} LockerState_e;

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Data Type Definition						
/----------------------------------------------------------------------------------*/
typedef struct LockRequest LockRequest_t;

struct LockRequest
{
	LockRequest_t *nextLockRequest;
	TxSelectorLockReqParams_t selectorLockReqParams;
	RequesterLockParams_t requesterLockParams;
	LockerRequestType_e	lockerRequestType;
};

typedef struct LockGroupRequest
{
	LockRequest_t*			nextLockRequest;
	uint32					groupId; 
	RequesterLockParams_t	requesterLockParams;

} LockGroupRequest_t;

typedef struct LockPlanRequest
{
	LockRequest_t*			nextLockRequest;
	uint8					planIndex;
	uint16					vapId;
	RequesterLockParams_t	requesterLockParams;

} LockPlanRequest_t;


typedef struct LockerDb
{
	GeneralQueueBlock_t lockRequestQueue;
	GeneralQueueBlock_t freeLockRequestPool;
	LockRequest_t *activeRequest;
	LockRequest_t  lockRequestsPool[LOCKER_NUM_OF_REQUESTS];
	LockerState_e lockerState;
} LockerDb_t;

typedef struct LockerGroupDb
{
	GeneralQueueBlock_t lockRequestQueue;
	GeneralQueueBlock_t freeLockRequestPool;
	LockGroupRequest_t *activeRequest;
	LockGroupRequest_t  lockRequestsPool[LOCKER_NUM_OF_GROUP_REQUESTS];
	LockerState_e lockerState;
} LockerGroupDb_t;

typedef struct LockerPlanDb
{
	GeneralQueueBlock_t lockRequestQueue;
	GeneralQueueBlock_t freeLockRequestPool;
	LockPlanRequest_t   *activeRequest;
	LockPlanRequest_t   lockRequestsPool[LOCKER_NUM_OF_PLAN_REQUESTS];
	LockerState_e 		lockerState;
} LockerPlanDb_t;

typedef void (*LockRequestHandler_t)(TxSelectorLockReqParams_t *txSelectorLockRequestParams);  

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

#ifdef ENET_INC_ARCH_WAVE600
static void lockerScheduleRequest(LockerRequestType_e lockRequestType,TxQueueType_e queueType,uint32 stationOrVapNum,uint8 planIndex,uint32 tid,RequesterLockParams_t *lockRequesterParams);
#else
static void lockerScheduleRequest(LockerRequestType_e lockRequestType,TxQueueType_e queueType,uint32 stationOrVapNum,uint32 tid,RequesterLockParams_t *lockRequesterParams);
#endif //ENET_INC_ARCH_WAVE600


/*---------------------------------------------------------------------------------
/							Static Variables									
/----------------------------------------------------------------------------------*/
LockerDb_t 		lockerDb;
LockerGroupDb_t	lockerGroupDb;
#ifdef ENET_INC_ARCH_WAVE600
LockerPlanDb_t  lockerPlanDb;
#endif


static const LockRequestHandler_t lockRequestHandlers[LOCKER_TOTAL_NUM_OF_REQUESTS_TYPES] = 
{
	TxSelector_LockRequest,
	TxSelector_DisableRequest,
};

/*---------------------------------------------------------------------------------
/							Debug									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

lockerScheduleRequest 

Description:
------------
	Common handler for sceduling & queueing  Lock/Disable requstes
	
Input:
-----
	lockRequestType		- The type of request (Lock\Disabled) for the Tx Selector
	queueType			- The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum 		- The Station or vap num that the Q belongs to 
	tid					- The required TID that should be locked
	lockRequetserParams - Caller unique params that should be returned when requst was serviced

**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
static void lockerScheduleRequest(LockerRequestType_e lockRequestType,TxQueueType_e queueType,uint32 stationOrVapNum,uint8 planIndex,uint32 tid,RequesterLockParams_t *lockRequesterParams)
#else
static void lockerScheduleRequest(LockerRequestType_e lockRequestType,TxQueueType_e queueType,uint32 stationOrVapNum,uint32 tid,RequesterLockParams_t *lockRequesterParams)
#endif //ENET_INC_ARCH_WAVE600
{
	LockRequest_t* lockRequest;
	TX_INTERRUPT_SAVE_AREA;

	//("lockerScheduleRequest lockRequestType= %d, queueType= %d, stationOrVapNum= %d, tid= %d ",lockRequestType,queueType,stationOrVapNum,tid);
	ILOG2_D("lockerScheduleRequest  lockRequesterParams= %x ",lockRequesterParams);
	
	lockRequest = (LockRequest_t*)GeneralQ_PopItemFromHead(&lockerDb.freeLockRequestPool);
	
	ASSERT(lockRequest);
		
	/* Fill Lock Request Params*/ 
	lockRequest->selectorLockReqParams.queueType = queueType;
	lockRequest->selectorLockReqParams.stationOrVapNum = stationOrVapNum;
#ifdef ENET_INC_ARCH_WAVE600
    lockRequest->selectorLockReqParams.planIndex = planIndex;
#endif //ENET_INC_ARCH_WAVE600
	lockRequest->selectorLockReqParams.tidOrAc = tid & LOCKER_SINGLE_TID_UPDATE_MASK;
	lockRequest->selectorLockReqParams.allTids = tid >> LOCKER_ALL_TID_UPDATE_START_BIT;
	lockRequest->requesterLockParams.callerContext = lockRequesterParams->callerContext;
	lockRequest->requesterLockParams.returnMsg = lockRequesterParams->returnMsg;
	lockRequest->requesterLockParams.returnTask = lockRequesterParams->returnTask;
	lockRequest->lockerRequestType = lockRequestType;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	if(lockerDb.lockerState == LOCKER_STATE_IDLE)
	{
		lockerDb.lockerState = LOCKER_STATE_BUSY;

		lockerDb.activeRequest = lockRequest;

		lockRequestHandlers[lockRequestType](&lockRequest->selectorLockReqParams);
	}
	else
	{
		GeneralQ_PushItemToTail(&lockerDb.lockRequestQueue,(GeneralQueueItem_t *)lockRequest);	

	}

	OSAL_ENABLE_INTERRUPTS(interrupt_save);

}


/**********************************************************************************

lockerScheduleGroupRequest 

Description:
------------
	Common handler for sceduling & queueing  MU group Lock requstes
	
Input:
-----
	groupId				- 	The group identification number 
	lockRequetserParams 	- 	Caller unique params that should be returned when requst was serviced

**********************************************************************************/
static void lockerScheduleGroupRequest(uint32 groupId,RequesterLockParams_t *lockRequesterParams)
{
	LockGroupRequest_t* lockGroupRequest;
	TX_INTERRUPT_SAVE_AREA;

	ILOG2_D("lockerScheduleGroupRequest  lockRequesterParams=0x %x ",lockRequesterParams);
	
	lockGroupRequest = (LockGroupRequest_t*)GeneralQ_PopItemFromHead(&lockerGroupDb.freeLockRequestPool);
	
	ASSERT(lockGroupRequest);
		
	/* Fill Group Lock Request Params*/ 
	lockGroupRequest->requesterLockParams.callerContext = lockRequesterParams->callerContext;
	lockGroupRequest->requesterLockParams.returnMsg = lockRequesterParams->returnMsg;
	lockGroupRequest->requesterLockParams.returnTask = lockRequesterParams->returnTask;
	lockGroupRequest->groupId= groupId;
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	if(lockerGroupDb.lockerState == LOCKER_STATE_IDLE)
	{
		lockerGroupDb.lockerState = LOCKER_STATE_BUSY;

		lockerGroupDb.activeRequest = lockGroupRequest;

		TxSelector_LockGroupRequest(groupId);
	}
	else
	{
		GeneralQ_PushItemToTail(&lockerGroupDb.lockRequestQueue,(GeneralQueueItem_t *)lockGroupRequest);	

	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

lockerSchedulePlanRequest 

Description:
------------
	Handler for sceduling & queueing  Lock\Disable Plan requests
	
Input:
-----
	lockRequestType		- The type of request (Lock\Disabled) for the Tx Selector
	stationOrVapNum 	- The Station or vap num that the Q belongs to 
	planIndex			- Index of the Plan to be locked
	lockRequetserParams - Caller unique params that should be returned when requst was serviced

**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
static void lockerSchedulePlanRequest(uint16 vapId, uint8 planIndex,RequesterLockParams_t *lockRequesterParams)
{
	LockPlanRequest_t* lockPlanRequest;
	TX_INTERRUPT_SAVE_AREA;

#ifdef DYNAMIC_GROUPING_DEBUG
	ILOG0_DDD("[lockerSchedulePlanRequest], lockRequesterParams = %x, planIndex: %d, locker state: %d",lockRequesterParams, planIndex, lockerPlanDb.lockerState);
#endif

	lockPlanRequest = (LockPlanRequest_t*)GeneralQ_PopItemFromHead(&lockerPlanDb.freeLockRequestPool);
	
	ASSERT(lockPlanRequest);
		
	/* Fill Lock Request Params*/ 
    lockPlanRequest->planIndex       					   = planIndex;
	lockPlanRequest->vapId								   = vapId;
	lockPlanRequest->requesterLockParams.callerContext     = lockRequesterParams->callerContext;
	lockPlanRequest->requesterLockParams.returnMsg		   = lockRequesterParams->returnMsg;
	lockPlanRequest->requesterLockParams.returnTask 	   = lockRequesterParams->returnTask;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	if(lockerPlanDb.lockerState == LOCKER_STATE_IDLE)
	{
		lockerPlanDb.lockerState = LOCKER_STATE_BUSY;

		lockerPlanDb.activeRequest = lockPlanRequest;

		TxSelector_LockPlanRequest(planIndex, vapId);
	}
	else
	{
		GeneralQ_PushItemToTail(&lockerPlanDb.lockRequestQueue,(GeneralQueueItem_t *)lockPlanRequest);	
	}

	OSAL_ENABLE_INTERRUPTS(interrupt_save);

}

#endif //ENET_INC_ARCH_WAVE600

/*---------------------------------------------------------------------------------
/						Public Function Definitions						
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Locker_LockRequestServiced 

Description:
------------
	Handle the confirmation from the HW locker that lock/disable request has been done
	
Input:
-----
	lockRequestStatus - Status of the request (Executed,Rejected)

Returns:
--------
	void 	

**********************************************************************************/
void Locker_LockRequestServiced(TxSelectorLockStatus_e lockRequestStatus)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG* lockerCfmMsg;
	LockReqCb_t* lockRegCbMsg;
	LockRequest_t * activeRequest = lockerDb.activeRequest;
	LockRequest_t * lockRequest; 
    K_MSG_TYPE  returnMsg_temp;
   	K_TASKID    returnTask_temp;
    
	lockerCfmMsg = OSAL_GET_MESSAGE(sizeof(LockReqCb_t));
	lockRegCbMsg = ((LockReqCb_t *)lockerCfmMsg->abData);

	lockRegCbMsg->callerContext =activeRequest->requesterLockParams.callerContext;
	lockRegCbMsg->stationOrVapNum = activeRequest->selectorLockReqParams.stationOrVapNum;
	lockRegCbMsg->tid = activeRequest->selectorLockReqParams.tidOrAc;
	lockRegCbMsg->lockStatus = lockRequestStatus;

	/* Send Message to requester*/
	ILOG2_DDD("Locker_LockRequestServiced  stationOrVapNum= %x tid = %x callerContext = %x",lockRegCbMsg->stationOrVapNum,lockRegCbMsg->tid,lockRegCbMsg->callerContext);

    // activeRequest might change until OSAL_SEND_MESSAGE
    returnTask_temp = activeRequest->requesterLockParams.returnTask;
    returnMsg_temp = activeRequest->requesterLockParams.returnMsg;

	GeneralQ_PushItemToTail(&lockerDb.freeLockRequestPool,(GeneralQueueItem_t *)lockerDb.activeRequest);

   	OSAL_SEND_MESSAGE(returnMsg_temp, returnTask_temp, lockerCfmMsg, VAP_ID_DO_NOT_CARE);
 

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/* Check For Pending Lock Requets*/
	lockRequest = (LockRequest_t *)GeneralQ_PopItemFromHead(&lockerDb.lockRequestQueue);

	if(lockRequest != NULL)
	{
		lockerDb.activeRequest = lockRequest;
		lockRequestHandlers[lockRequest->lockerRequestType](&lockRequest->selectorLockReqParams);
	}
	else
	{
		lockerDb.lockerState = LOCKER_STATE_IDLE;
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}


/**********************************************************************************

Locker_LockGroupRequestServiced 

Description:
------------
	Handle the confirmation from the HW locker that lock/disable request has been done
	
Input:
-----
	lockRequestStatus - Status of the request (Executed,Rejected)

Returns:
--------
	void	

**********************************************************************************/
void Locker_LockGroupRequestServiced(TxSelectorLockStatus_e lockRequestStatus)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG* lockerCfmMsg;
	LockGroupReqCb_t* lockRegCbMsg;
	LockGroupRequest_t * activeRequest = lockerGroupDb.activeRequest;
	LockGroupRequest_t * lockRequest; 

#ifdef DYNAMIC_GROUPING_DEBUG
	ILOG0_D("[Locker_LockGroupRequestServiced], status = %d", lockRequestStatus);
#endif

	lockerCfmMsg = OSAL_GET_MESSAGE(sizeof(LockGroupReqCb_t));
	lockRegCbMsg = ((LockGroupReqCb_t *)lockerCfmMsg->abData);

	lockRegCbMsg->callerContext =activeRequest->requesterLockParams.callerContext;
	lockRegCbMsg->groupId = activeRequest->groupId;
	lockRegCbMsg->lockStatus = lockRequestStatus;
	
	OSAL_SEND_MESSAGE(activeRequest->requesterLockParams.returnMsg,activeRequest->requesterLockParams.returnTask,lockerCfmMsg, VAP_ID_DO_NOT_CARE);
	GeneralQ_PushItemToTail(&lockerGroupDb.freeLockRequestPool,(GeneralQueueItem_t *)lockerGroupDb.activeRequest);
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/* Check For Pendong Lock Requets*/
	lockRequest = (LockGroupRequest_t *)GeneralQ_PopItemFromHead(&lockerGroupDb.lockRequestQueue);

	if(lockRequest != NULL)
	{
		lockerGroupDb.activeRequest = lockRequest;
		TxSelector_LockGroupRequest(lockRequest->groupId);
	}
	else
	{
		lockerGroupDb.lockerState = LOCKER_STATE_IDLE;
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save); 
}


/**********************************************************************************

Locker_LockPlanRequestServiced 

Description:
------------
	Handle the confirmation from the HW locker that lock\disable Plan request has been done
	
Input:
-----
	lockRequestStatus - Status of the request (Executed,Rejected)

Returns:
--------
	void	

**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
void Locker_LockPlanRequestServiced(TxSelectorLockStatus_e lockRequestStatus)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG* lockerCfmMsg;
	LockPlanReqCb_t* lockRegCbMsg;
	LockPlanRequest_t * activeRequest = lockerPlanDb.activeRequest;
	LockPlanRequest_t * lockRequest; 

#ifdef DYNAMIC_GROUPING_DEBUG
	ILOG0_D("[Locker_LockPlanRequestServiced], status = %d", lockRequestStatus);
#endif

	lockerCfmMsg = OSAL_GET_MESSAGE(sizeof(LockPlanReqCb_t));
	lockRegCbMsg = ((LockPlanReqCb_t *)lockerCfmMsg->abData);

	lockRegCbMsg->callerContext = activeRequest->requesterLockParams.callerContext;
	lockRegCbMsg->lockStatus 	= lockRequestStatus;
	
	OSAL_SEND_MESSAGE(activeRequest->requesterLockParams.returnMsg,activeRequest->requesterLockParams.returnTask,lockerCfmMsg, VAP_ID_DO_NOT_CARE);
	GeneralQ_PushItemToTail(&lockerPlanDb.freeLockRequestPool,(GeneralQueueItem_t *)lockerPlanDb.activeRequest);
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/* Check For Pending Plan Requets*/
	lockRequest = (LockPlanRequest_t *)GeneralQ_PopItemFromHead(&lockerPlanDb.lockRequestQueue);

	if(lockRequest != NULL)
	{
		lockerPlanDb.activeRequest = lockRequest;
		TxSelector_LockPlanRequest(lockRequest->planIndex, lockRequest->vapId);
	}
	else
	{
		lockerPlanDb.lockerState = LOCKER_STATE_IDLE;
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save); 
}

#endif

/**********************************************************************************

Locker_EnableStaQueues 

Description:
------------
	Request to enable all Sta/Vap queues for this Q type
	
Input:
-----
	queueType - The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum - The Station or vap num that the Q belongs to
Output:
-------
	None

**********************************************************************************/
void Locker_EnableStaQueues(TxQueueType_e queueType,uint32 stationOrVapNum)
{
	TxSelectorActionParams_t txSelectorActionParams;

//	ILOG0_DD("Locker_EnableStaQueues queueType= %d, stationOrVapId= %d",queueType,stationOrVapNum);


	txSelectorActionParams.stationOrVapNum = stationOrVapNum;
	txSelectorActionParams.queueType = queueType;

#ifndef ENET_INC_ARCH_WAVE600
	txSelectorActionParams.allTids = TX_SELECTOR_ALL_TIDS_UPDATE;
	txSelectorActionParams.tidOrAc = 0;
#endif //ENET_INC_ARCH_WAVE600

	TxSelector_EnableQueue(&txSelectorActionParams);
}


#ifdef ENET_INC_ARCH_WAVE600
/**********************************************************************************

Locker_UnLockPerTidQueues 

Description:
------------
	This procedure locks/unlocks the queue specified by the parameters queue, stationVapNum, tidNum 
	
Input:
-----
	queueType 	- The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	staOrVapId	- stationOrVapNum - The Station or vap num that the Q belongs to
	tidBitmap	- The required TIDs that should be unlocked
	
Returns:
--------
	void 
	
**********************************************************************************/
void Locker_UnLockPerTidQueues(TxQueueType_e queueType,uint16 stationOrVapNum, uint16 tidBitmap)
{
	TxSelectorActionParams_t txSelectorActionParams;

	//("Locker_UnLockSingleQueue queueType= %d, stationOrVapId= %d",queueType,stationOrVapNum);
	
	txSelectorActionParams.stationOrVapNum = stationOrVapNum;
	txSelectorActionParams.queueType = queueType;
	txSelectorActionParams.tidValueBitmap = TX_SELECTOR_UNLOCK_Q*tidBitmap;
	txSelectorActionParams.tidEnableBitmap = tidBitmap;
		
	TxSelector_UnlockQueue(&txSelectorActionParams);
}


/**********************************************************************************

Locker_UnLockAllStaQueues 

Description:
------------
	Request to unlock all Sta/Vap of this Q type
	
Input:
-----
	queueType - The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum - The Station or vap num that the Q belongs to
	
Returns:
--------
	void 	
**********************************************************************************/
void Locker_UnLockAllStaQueues(TxQueueType_e queueType,uint16 stationOrVapNum)
{
	TxSelectorActionParams_t txSelectorActionParams;

	memset (&txSelectorActionParams,0x0,sizeof(TxSelectorActionParams_t));

	//("Locker_UnLockAllStaQueues queueType= %d, stationOrVapId= %d",queueType,stationOrVapNum);

	txSelectorActionParams.stationOrVapNum = stationOrVapNum;
	txSelectorActionParams.queueType = queueType;
	txSelectorActionParams.tidValueBitmap = TX_SELECTOR_UNLOCK_Q*TID_BITMAP_ALL_TIDS;
	txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
		
	TxSelector_UnlockQueue(&txSelectorActionParams);
}


#else

/**********************************************************************************

Locker_UnLockAllStaQueues 

Description:
------------
	Request to unlock all Sta\Vap of this Q type
	
Input:
-----
	queueType - The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum - The Station or vap num that the Q belongs to
	
Returns:
--------
	void 	
**********************************************************************************/
void Locker_UnLockAllStaQueues(TxQueueType_e queueType,uint8 stationOrVapNum)
{
	TxSelectorActionParams_t txSelectorActionParams;

	//("Locker_UnLockAllStaQueues queueType= %d, stationOrVapId= %d",queueType,stationOrVapNum);

	txSelectorActionParams.stationOrVapNum = stationOrVapNum;
	txSelectorActionParams.queueType = queueType;
	txSelectorActionParams.allTids = TX_SELECTOR_ALL_TIDS_UPDATE;
	txSelectorActionParams.tidOrAc = 0;
		
	TxSelector_UnlockQueue(&txSelectorActionParams);
}


/**********************************************************************************

Locker_UnLockSingleQueue 

Description:
------------
	This procedure locks/unlocks the queue specified by the parameters queue, stationVapNum, tidNum 
	
Input:
-----
	queueType 	- The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	staOrVapId	- stationOrVapNum - The Station or vap num that the Q belongs to
	tid			- The required TID that should be unlocked
	
Returns:
--------
	void 
	
**********************************************************************************/
void Locker_UnLockPerTidQueues(TxQueueType_e queueType,uint8 stationOrVapNum, uint8 tidBitmap)
{
	TxSelectorActionParams_t txSelectorActionParams;

	//("Locker_UnLockSingleQueue queueType= %d, stationOrVapId= %d",queueType,stationOrVapNum);
	
	txSelectorActionParams.stationOrVapNum = stationOrVapNum;
	txSelectorActionParams.queueType = queueType;
	txSelectorActionParams.allTids = TX_SELECTOR_SINGLE_TID_UPDATE;
	txSelectorActionParams.tidOrAc = Utils_FindFirstSet((uint32)tidBitmap);
		
	TxSelector_UnlockQueue(&txSelectorActionParams);
}

#endif //ENET_INC_ARCH_WAVE600


/**********************************************************************************

Locker_DisableStaQueues 

Description:
------------
	
Input:
-----
	queueType - The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum - The Station or vap num that the Q belongs to 
	lockReqCbParams - Caller unique params that should be returned when requst was serviced
	
Returns:
--------
	void 	
	
**********************************************************************************/
void Locker_DisableStaQueues(TxQueueType_e queueType,uint32 stationOrVapNum,RequesterLockParams_t *lockRequesterParams)
{
#ifdef ENET_INC_ARCH_WAVE600
	lockerScheduleRequest(LOCKER_REQUEST_TYPE_DISABLE,queueType,stationOrVapNum,LOCKER_PLAN_INDEX_INVALID,LOCKER_ALL_TIDS_UPDATE,lockRequesterParams);
#else
    lockerScheduleRequest(LOCKER_REQUEST_TYPE_DISABLE,queueType,stationOrVapNum,LOCKER_ALL_TIDS_UPDATE,lockRequesterParams);
#endif //ENET_INC_ARCH_WAVE600
}


/**********************************************************************************

Locker_LockStaQueuesRequest 

Description:
------------
	
Input:
-----
	queueType - The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum - The Station or vap num that the Q belongs to
	lockReqCbParams - Caller unique params that should be returned when requst was serviced

Returns:
--------
	void 	
**********************************************************************************/
void Locker_LockAllStaQueues(TxQueueType_e queueType,StaId stationOrVapNum,RequesterLockParams_t *lockRequesterParams)
{
#ifdef ENET_INC_ARCH_WAVE600
	lockerScheduleRequest(LOCKER_REQUEST_TYPE_LOCK,queueType,stationOrVapNum,LOCKER_PLAN_INDEX_INVALID,LOCKER_ALL_TIDS_UPDATE,lockRequesterParams);	
#else
	lockerScheduleRequest(LOCKER_REQUEST_TYPE_LOCK,queueType,stationOrVapNum,LOCKER_ALL_TIDS_UPDATE,lockRequesterParams);	
#endif //ENET_INC_ARCH_WAVE600
}

/**********************************************************************************

Locker_LockSingleQueue 

Description:
------------
	Request to lock single queue
	
Input:
-----
	queueType - The queue type to be locked (Sta,Vap,Globa,GPLP,GPHP,Beacon)
	stationOrVapNum - The Station or vap num that the Q belongs to
	tid	- The required TID that should be locked
	lockRequesterParams - Caller unique params that should be returned when requst was serviced

Returns:
--------
	void 	
**********************************************************************************/
void Locker_LockSingleQueue(TxQueueType_e queueType,StaId stationOrVapNum, uint8 tid, RequesterLockParams_t *lockRequesterParams)
{
#ifdef ENET_INC_ARCH_WAVE600
    lockerScheduleRequest(LOCKER_REQUEST_TYPE_LOCK,queueType,stationOrVapNum,LOCKER_PLAN_INDEX_INVALID,tid,lockRequesterParams);    
#else
    lockerScheduleRequest(LOCKER_REQUEST_TYPE_LOCK,queueType,stationOrVapNum,tid,lockRequesterParams);    
#endif //ENET_INC_ARCH_WAVE600
}

#ifdef ENET_INC_ARCH_WAVE600
/**********************************************************************************

Locker_LockPlan 

Description:
------------
	Request to lock single plan
	
Input:
-----
	planIndex - The plan index to lock
	

vapNum - must have vap num for extracting the band (UMAC not aware of the band)
	lockRequesterParams - Caller unique params that should be returned when requst was serviced

Returns:
--------
	void 	
**********************************************************************************/
void Locker_LockPlan(uint8 planIndex, uint32 vapNum, RequesterLockParams_t *lockRequesterParams)
{
	lockerSchedulePlanRequest(vapNum, planIndex, lockRequesterParams);	
}

/**********************************************************************************

Locker_UnLockPlan 

Description:
------------
	Request to unlock single plan
	
Input:
-----
	

vapNum - must have vap num for extracting the band (UMAC not aware of the band)
	planIndex - The plan index to lock

Returns:
--------
	void 	
**********************************************************************************/
void Locker_UnLockPlan(uint8 planIndex, StaId staId)
{
    TxSelector_UnlockPlan(planIndex, staId);
}
#endif //ENET_INC_ARCH_WAVE600

/**********************************************************************************

Locker_LockGroup 

Description:
------------
	This procedure locks MU group specified by the group ID  
	
Input:
-----
	groupId 	- The group identification number 
	lockRequesterParams - Lock request callback and queue params 
	
Returns:
--------
	void 
	
**********************************************************************************/
void Locker_LockGroup(uint32 groupId,RequesterLockParams_t *lockRequesterParams)
{
	lockerScheduleGroupRequest(groupId,lockRequesterParams);	
}


/**********************************************************************************

Locker_UnLockGroup 

Description:
------------
	Unlocks a MU group specified by the group ID  
	
Input:
-----
	groupId 	- The group identification number 
	
Returns:
--------
	void 
	
**********************************************************************************/
void Locker_UnLockGroup(uint32 groupId)
{
	TxSelector_UnlockGroup(groupId);	
}


/**********************************************************************************

Locker_Init 

Description:
------------
	Initialization routine of the Sw Locker modlue
	1) Set t0 0 Locker DB 
	2) Init free lock request Q

Returns:
--------
	void 	

**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void Locker_Init(void)
{
	uint8 i;

	/* Init Internal Locker DB*/
	memset(&lockerDb,0,sizeof(LockerDb_t));	
	memset(&lockerGroupDb,0,sizeof(LockerGroupDb_t));	

	/*Init Free Lock Requests Queue*/
	for(i=0; i < LOCKER_NUM_OF_REQUESTS; i++)
	{
		GeneralQ_PushItemToTail(&lockerDb.freeLockRequestPool,(GeneralQueueItem_t *)&lockerDb.lockRequestsPool[i]);
	}

	/*Init Free Group Lock Requests Queue*/
	for(i=0; i < LOCKER_NUM_OF_GROUP_REQUESTS; i++)
	{
		GeneralQ_PushItemToTail(&lockerGroupDb.freeLockRequestPool,(GeneralQueueItem_t *)&lockerGroupDb.lockRequestsPool[i]);
	}
	
#ifdef ENET_INC_ARCH_WAVE600
	
	/*Init Free PlanDb Lock Requests Queue*/
	for(i=0; i < LOCKER_NUM_OF_PLAN_REQUESTS; i++)
	{
		GeneralQ_PushItemToTail(&lockerPlanDb.freeLockRequestPool,(GeneralQueueItem_t *)&lockerPlanDb.lockRequestsPool[i]);
	}
	
#endif

}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


