/***************************************************************
 File:		Mac2Index.c
 Module:	  	MAC 2 index
 Purpose: 	holds all the implementation of the MAC 2 index API functions
 Description:  This module contains the implementation of the MAC 2 index module 
 			which is responsible of converting MAC address to station id 
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Mac2index_Api.h"
#include "Mac2index.h"
#include "MacAddr2IndexRegs.h"
#include "StaDb.h" //should be removed when the SHRAM files will be defined
#include "VapDb.h" //should be removed when the SHRAM files will be defined
#include "ieee80211.h" //should be removed when the SHRAM files will be defined
#include "RegAccess_Api.h"
#include "ShramStationDatabase.h"
#include "ShramVapDatabase.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "MacAddr2IndexRegs.h"
#include "um_interface.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 12



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
uint8 mac2IndexInvokeCommand (IEEE_ADDR * macAddress,Mac2IndexCommands_e command);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

mac2IndexInvokeCommand  


Description:
------------
invokes a command which needs a MAC address 

Input:
-----
command -type of command to invoke 
macAddress - the MAC address of the station to add/search/remove	
		
Output:
-------
	

Returns:
--------
station id -the station id result written by the MAC 2 index module(not relevant in case of remove) 
	
**********************************************************************************/

uint8 mac2IndexInvokeCommand (IEEE_ADDR *macAddress,Mac2IndexCommands_e command)
{	
	uint32 timeStamp = 0;
	RegMacAddr2IndexMacAddr2IndexAddr0To31_u macAddress0to31Register;
	RegMacAddr2IndexMacAddr2IndexAddr32To47_u macAddress32to47Register;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;
	RegMacAddr2IndexMacAddr2IndexStatus_u  resultRegister;

    macAddress0to31Register.val = 0;
	macAddress32to47Register.val = 0;
	commandRegister.val = 0;
	resultRegister.val = 0;
	
	/* Write the MAC address of the station to the MAC 2 index registers */
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 = (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FORTH_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_FORTH_BYTE_OFFSET;
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_THIRD_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_THIRD_BYTE_OFFSET;
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_SECOND_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_SECOND_BYTE_OFFSET;
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FIRST_BYTE];
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_0TO31, macAddress0to31Register.val);
    		
	macAddress32to47Register.bitFields.macAddr2IndexAddr32To47 = (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_SIXTH_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_SIXTH_BYTE_OFFSET;
	macAddress32to47Register.bitFields.macAddr2IndexAddr32To47 |= macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FIFTH_BYTE];
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_32TO47, macAddress32to47Register.val);

	/* write the command to the MAC 2 index module */
	commandRegister.bitFields.macAddr2IndexCommand = command;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val);

    timeStamp = GET_TSF_TIMER_LOW();
    while(!resultRegister.bitFields.macAddr2IndexDone)
    {
    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= MAC_2_INDEX_MAX_MICROS_WAITING_FOR_DONE);
		RegAccess_Read(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_STATUS, &resultRegister.val);
    }
	ASSERT(!(MAC_2_INDEX_ERROR_INDICATION & resultRegister.val));

	return resultRegister.bitFields.macAddr2IndexDbIndex;
	
}




/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

Mac2Index_SearchStation  


Description:
------------
search for a station id with a MAC address

Input:
-----
macAddress - the address of the station to look for	
		
Output:
-------
	

Returns:
--------
station id - station id that was found (in case of failure there will be an assert) 
	
**********************************************************************************/

uint16 Mac2Index_SearchStation(IEEE_ADDR * macAddress)
{	
	StaId stationId = 0;

    stationId = mac2IndexInvokeCommand(macAddress, MAC_2_INDEX_COMMAND_SEARCH);
	return stationId; 	
}

/**********************************************************************************

Mac2Index_AddStation  


Description:
------------
add a new station to the MAC 2 index module

Input:
-----
macAddress - the address of the station to add
		
Output:
-------
	

Returns:
--------
station id -the station id that this station recieved (in case of failure there will be an assert) 
	
**********************************************************************************/

uint16 Mac2Index_AddStation(IEEE_ADDR *macAddress)
{	
	StaId stationId = 0;

    stationId = mac2IndexInvokeCommand(macAddress, MAC_2_INDEX_COMMAND_ADD);
	ASSERT(stationId != LoggerApi_GetLoggerStationIndex());
	return stationId; 	
}

/**********************************************************************************

Mac2Index_RemoveStationByAddress  


Description:
------------
remove a station from the MAC 2 index module

Input:
-----
macAddress - the address of the station to remove
		
Output:
-------
	

Returns:
--------
 void (in case of failure there will be an assert) 
	
**********************************************************************************/

void Mac2Index_RemoveStationByAddress(IEEE_ADDR * macAddress)
{	
 	mac2IndexInvokeCommand(macAddress, MAC_2_INDEX_COMMAND_REMOVE_BY_ADDRESS);		
}

/**********************************************************************************

Mac2Index_RemoveStationById  


Description:
------------
remove a station from the MAC 2 index module

Input:
-----
stationId - the station id of the station to remove
		
Output:
-------
	

Returns:
--------
void  (in case of failure there will be an assert) 
	
**********************************************************************************/

void Mac2Index_RemoveStationById(StaId stationId)
{	
 	uint32 timeStamp = 0;
	RegMacAddr2IndexMacAddr2IndexAddr32To47_u stationIdRegister;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;
	RegMacAddr2IndexMacAddr2IndexStatus_u  resultRegister;

	stationIdRegister.val = 0;
	commandRegister.val = 0;
	resultRegister.val = 0;
	
	/* Write the station id to remove to the MAC 2 index module */		
	stationIdRegister.bitFields.removeDbIndex = stationId;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_32TO47, stationIdRegister.val);

	/* write the command to the MAC 2 index module */
	commandRegister.bitFields.macAddr2IndexCommand = MAC_2_INDEX_COMMAND_REMOVE_BY_INDEX;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val);

    timeStamp = GET_TSF_TIMER_LOW();
    while(!resultRegister.bitFields.macAddr2IndexDone)
    {
    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= MAC_2_INDEX_MAX_MICROS_WAITING_FOR_DONE);    
    	RegAccess_Read(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_STATUS, &resultRegister.val);
    }
	ASSERT((MAC_2_INDEX_ERROR_INDICATION & resultRegister.val)== FALSE);		
}


/**********************************************************************************

Mac2Index_Initialize  


Description:
------------
initialize the MAC 2 index module

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#ifdef ENET_INC_UMAC
#pragma ghs section text=".initialization" 
#endif //ENET_INC_UMAC
void Mac2Index_Initialize()
{		
    RegMacAddr2IndexQueueStaDbBaseAddr_u stationDatabaseAddressRegister;
	RegMacAddr2IndexQueueStaDbSize_u stationDatabaseParametersRegister;
	RegMacAddr2IndexQueueVapDbBaseAddr_u vapDatabaseAddressRegister;
	RegMacAddr2IndexQueueVapDbSize_u vapDatabaseParametersRegister;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;       

	stationDatabaseAddressRegister.val = 0;
	stationDatabaseParametersRegister.val = 0;
	vapDatabaseAddressRegister.val = 0;
	vapDatabaseParametersRegister.val = 0;
	commandRegister.val = 0;
	
    /* configure the station data base parameters */
	stationDatabaseAddressRegister.bitFields.queueStaDbBaseAddr = (uint32)StaDbHwEntries;
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_STA_DB_BASE_ADDR, stationDatabaseAddressRegister.val);
	
    stationDatabaseParametersRegister.bitFields.queueStaDbEntry4BSize = (sizeof(StaDb_t))>>2;
	stationDatabaseParametersRegister.bitFields.queueStaSecondaryDbEntry4BOffset = (FM_STRUCT_OFFSET(StaDb_t,tid))>>2;
	stationDatabaseParametersRegister.bitFields.queueStaSecondaryDbEntry4BSize = (sizeof(StaDbTid_t))>>2;
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_STA_DB_SIZE, stationDatabaseParametersRegister.val);		

	/* configure the VAP data base parameters */
    vapDatabaseAddressRegister.bitFields.queueVapDbBaseAddr =  (uint32)VapDbHwEntries;
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_VAP_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	
    vapDatabaseParametersRegister.bitFields.queueVapDbEntry4BSize = (sizeof(VapDb_t))>>2;
	vapDatabaseParametersRegister.bitFields.queueVapSecondaryDbEntry4BOffset = (FM_STRUCT_OFFSET(VapDb_t,tid))>>2;
	vapDatabaseParametersRegister.bitFields.queueVapSecondaryDbEntry4BSize = (sizeof(VapDbTid_t))>>2;
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_VAP_DB_SIZE, vapDatabaseParametersRegister.val);
    
	
	// configure the GLOBAL data base parameters - currently we set it to VAP DB
    RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GLOBAL_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GLOBAL_DB_SIZE, vapDatabaseParametersRegister.val);

	
	//configure the GPLP data base parameters - currently we set it to VAP DB
    RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPLP_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPLP_DB_SIZE, vapDatabaseParametersRegister.val);

	
	// configure the GPHP data base parameters - currently we set it to VAP DB
    RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPHP_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPHP_DB_SIZE, vapDatabaseParametersRegister.val);


	// configure the beacon data base parameters - currently we set it to VAP DB
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_BEACON_DB_BASE_ADDR, vapDatabaseAddressRegister.val);	
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_BEACON_DB_SIZE, vapDatabaseParametersRegister.val);


	/* Initialize the MAC 2 index module */
	commandRegister.bitFields.macAddr2IndexCommand = MAC_2_INDEX_COMMAND_INITIALIZE;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


	
		













