/***************************************************************
 File:		Mac2Index.c
 Module:	  	MAC 2 index
 Purpose: 	holds all the implementation of the MAC 2 index API functions
 Description:  This module contains the implementation of the MAC 2 index module 
 			which is responsible of converting MAC address to station id 
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Mac2index_Api.h"
#include "Mac2index.h"
#include "MacAddr2IndexRegs.h"
#include "ieee80211.h" //should be removed when the SHRAM files will be defined
#include "RegAccess_Api.h"
#include "ShramStationDatabase.h"
#include "ShramVapDatabase.h"
#include "ConfigurationManager_api.h"
#include "database.h"
#include "Pac_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 13



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
uint8 mac2IndexInvokeCommand (IEEE_ADDR * macAddress,Mac2IndexCommands_e command);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

mac2IndexInvokeCommand  


Description:
------------
invokes a command which needs a MAC address 

Input:
-----
command -type of command to invoke 
macAddress - the MAC address of the station to add/search/remove	
		
Output:
-------
	

Returns:
--------
station id -the station id result written by the MAC 2 index module(not relevant in case of remove) 
	
**********************************************************************************/

uint8 mac2IndexInvokeCommand (IEEE_ADDR *macAddress,Mac2IndexCommands_e command)
{	
	uint32 timeStamp = 0;
	RegMacAddr2IndexMacAddr2IndexAddr0To31_u macAddress0to31Register;
	RegMacAddr2IndexMacAddr2IndexAddr32To47_u macAddress32to47Register;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;
	RegMacAddr2IndexMacAddr2IndexStatus_u  resultRegister;

    macAddress0to31Register.val = 0;
	macAddress32to47Register.val = 0;
	commandRegister.val = 0;
	resultRegister.val = 0;
	
	/* Write the MAC address of the station to the MAC 2 index registers */
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 = (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FORTH_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_FORTH_BYTE_OFFSET;
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_THIRD_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_THIRD_BYTE_OFFSET;
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_SECOND_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_SECOND_BYTE_OFFSET;
	macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 = (uint32)(macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 | macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FIRST_BYTE]);
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_0TO31, macAddress0to31Register.val);
    		
	macAddress32to47Register.bitFields.macAddr2IndexAddr32To47 = (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_SIXTH_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_SIXTH_BYTE_OFFSET;
	macAddress32to47Register.bitFields.macAddr2IndexAddr32To47 = (uint32)(macAddress32to47Register.bitFields.macAddr2IndexAddr32To47  | macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FIFTH_BYTE]);
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_32TO47, macAddress32to47Register.val);

	/* write the command to the MAC 2 index module */
	commandRegister.bitFields.macAddr2IndexCommand = command;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val);

    timeStamp = GET_TSF_TIMER_LOW();
    while(!resultRegister.bitFields.macAddr2IndexDone)
    {
    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= MAC_2_INDEX_MAX_MICROS_WAITING_FOR_DONE);
		RegAccess_Read(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_STATUS, &resultRegister.val);
    }
	ASSERT(!(MAC_2_INDEX_ERROR_INDICATION & resultRegister.val));

	return resultRegister.bitFields.macAddr2IndexDbIndex;
	
}




/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

#ifdef GENERATE_TRIGGER_FRAME_FROM_FW
// this funcion is not needed except for trigger frame patch (transmission from FW).
// when trigger frame is removed, this function can be removed as well, and mac2index.c should removed from UM compilation.
uint16 Mac2Index_SearchStationForTriggerFrame(IEEE_ADDR * macAddress)
{	
	// This function search for the STA in both bands (since we don't know which band it is in)
	uint32 band;
	uint32 timeStamp = 0;
	RegMacAddr2IndexMacAddr2IndexAddr0To31_u macAddress0to31Register;
	RegMacAddr2IndexMacAddr2IndexAddr32To47_u macAddress32to47Register;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;
	RegMacAddr2IndexMacAddr2IndexStatus_u  resultRegister;

    macAddress0to31Register.val = 0;
	macAddress32to47Register.val = 0;
	commandRegister.val = 0;
	resultRegister.val = 0;


	for (band = CONFIGURATION_MANAGER_BAND_0 ; band <= CONFIGURATION_MANAGER_BAND_1 ; band++)
	{
	
		/* Write the MAC address of the station to the MAC 2 index registers */
		macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 = (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FORTH_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_FORTH_BYTE_OFFSET;
		macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_THIRD_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_THIRD_BYTE_OFFSET;
		macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_SECOND_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_SECOND_BYTE_OFFSET;
		macAddress0to31Register.bitFields.macAddr2IndexAddr0To31 |= macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FIRST_BYTE];
		RegAccess_WritePerBand(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_0TO31, macAddress0to31Register.val, band);
	    		
		macAddress32to47Register.bitFields.macAddr2IndexAddr32To47 = (macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_SIXTH_BYTE]) << MAC_2_INDEX_MAC_ADDRESS_SIXTH_BYTE_OFFSET;
		macAddress32to47Register.bitFields.macAddr2IndexAddr32To47 |= macAddress->au8Addr[MAC_2_INDEX_MAC_ADDRESS_FIFTH_BYTE];
		RegAccess_WritePerBand(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_32TO47, macAddress32to47Register.val, band);

		/* write the command to the MAC 2 index module */
		commandRegister.bitFields.macAddr2IndexCommand = MAC_2_INDEX_COMMAND_SEARCH;
		RegAccess_WritePerBand(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val, band);

	    timeStamp = GET_TSF_TIMER_LOW();
	    while(!resultRegister.bitFields.macAddr2IndexDone)
	    {
	    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= MAC_2_INDEX_MAX_MICROS_WAITING_FOR_DONE);
			RegAccess_ReadPerBand(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_STATUS, &resultRegister.val, band);
	    }

		// If the SID was found - return it. If not, we will search again in the other band.
		if (resultRegister.bitFields.macAddr2IndexFail != 0)
		{
			return resultRegister.bitFields.macAddr2IndexDbIndex;
		}
	}

	ASSERT(0); // If we get here it means we didn't find the STA in any of the bands. It means trigger frame search for a bad mac addess. Trigger frame manager should fix it.
	return DB_ASYNC_SID;
}

#endif





/**********************************************************************************

Mac2Index_SearchStation  


Description:
------------
search for a station id with a MAC address

Input:
-----
macAddress - the address of the station to look for	
		
Output:
-------
	

Returns:
--------
station id - station id that was found (in case of failure there will be an assert) 
	
**********************************************************************************/

uint16 Mac2Index_SearchStation(IEEE_ADDR * macAddress)
{	
	uint16 stationId = 0;

    stationId = mac2IndexInvokeCommand(macAddress, MAC_2_INDEX_COMMAND_SEARCH);
	return stationId; 	
}

/**********************************************************************************

Mac2Index_AddStation  


Description:
------------
add a new station to the MAC 2 index module

Input:
-----
macAddress - the address of the station to add
		
Output:
-------
	

Returns:
--------
station id -the station id that this station recieved (in case of failure there will be an assert) 
	
**********************************************************************************/
uint16 Mac2Index_AddStation(IEEE_ADDR *macAddress)
{	
	uint16 stationId = 0;


// This is a temporary patch to allocate different SID every time (random).
// It is done in order to test all the TX and RX Queues of the HW.
// It should NOT be part of a real release. Only tests on FPGA.

    stationId = mac2IndexInvokeCommand(macAddress, MAC_2_INDEX_COMMAND_ADD);

	return stationId; 	
}

/**********************************************************************************

Mac2Index_RemoveStationByAddress  


Description:
------------
remove a station from the MAC 2 index module

Input:
-----
macAddress - the address of the station to remove
		
Output:
-------
	

Returns:
--------
 void (in case of failure there will be an assert) 
	
**********************************************************************************/

void Mac2Index_RemoveStationByAddress(IEEE_ADDR * macAddress)
{	
 	mac2IndexInvokeCommand(macAddress, MAC_2_INDEX_COMMAND_REMOVE_BY_ADDRESS);		
}

/**********************************************************************************

Mac2Index_RemoveStationById  


Description:
------------
remove a station from the MAC 2 index module

Input:
-----
stationId - the station id of the station to remove
		
Output:
-------
	

Returns:
--------
void  (in case of failure there will be an assert) 
	
**********************************************************************************/

void Mac2Index_RemoveStationById(uint16 stationId)
{	
 	uint32 timeStamp = 0;
	RegMacAddr2IndexMacAddr2IndexAddr32To47_u stationIdRegister;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;
	RegMacAddr2IndexMacAddr2IndexStatus_u  resultRegister;

	stationIdRegister.val = 0;
	commandRegister.val = 0;
	resultRegister.val = 0;
	
	/* Write the station id to remove to the MAC 2 index module */		
	stationIdRegister.bitFields.removeDbIndex = stationId;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_ADDR_32TO47, stationIdRegister.val);

	/* write the command to the MAC 2 index module */
	commandRegister.bitFields.macAddr2IndexCommand = MAC_2_INDEX_COMMAND_REMOVE_BY_INDEX;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val);

    timeStamp = GET_TSF_TIMER_LOW();
	
		//KW_FIX_FW_M - pragma ignore as register read & timeout check with an assert
    while(!resultRegister.bitFields.macAddr2IndexDone)
    {
    	ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= MAC_2_INDEX_MAX_MICROS_WAITING_FOR_DONE);    
    	RegAccess_Read(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_STATUS, &resultRegister.val);
    }
	ASSERT((MAC_2_INDEX_ERROR_INDICATION & resultRegister.val)== FALSE);		
}


/**********************************************************************************

Mac2Index_Initialize  


Description:
------------
initialize the MAC 2 index module

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void Mac2Index_Initialize()
{		
    RegMacAddr2IndexQueueStaDbBaseAddr_u stationDatabaseAddressRegister;
	RegMacAddr2IndexQueueStaDbSize_u stationDatabaseParametersRegister;
	RegMacAddr2IndexQueueVapDbBaseAddr_u vapDatabaseAddressRegister;
	RegMacAddr2IndexQueueVapDbSize_u vapDatabaseParametersRegister;
	RegMacAddr2IndexMacAddr2IndexCommand_u commandRegister;  
	RegMacAddr2IndexMacAddr2IndexInitParams_u initParamsRegister;

	stationDatabaseAddressRegister.val = 0;
	stationDatabaseParametersRegister.val = 0;
	vapDatabaseAddressRegister.val = 0;
	vapDatabaseParametersRegister.val = 0;
	commandRegister.val = 0;
	initParamsRegister.val = 0;

	/*for all DB configurations - mask only the first 20 bits (without the SHRAM offset, each HW block will add it by himself)*/
	
    /* configure the station data base parameters */
	stationDatabaseAddressRegister.bitFields.queueStaDbBaseAddr = ((uint32)StaDbHwEntries & 0xFFFFF); 
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_STA_DB_BASE_ADDR, stationDatabaseAddressRegister.val);
	
    stationDatabaseParametersRegister.bitFields.queueStaDbEntry4BSize = (sizeof(StaDb_t))>>2;
	stationDatabaseParametersRegister.bitFields.queueStaSecondaryDbEntry4BOffset = (FM_STRUCT_OFFSET(StaDb_t,tid))>>2;
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_STA_DB_SIZE, stationDatabaseParametersRegister.val);		

	/* configure the VAP data base parameters */
    vapDatabaseAddressRegister.bitFields.queueVapDbBaseAddr =  ((uint32)VapDbHwEntries & 0xFFFFF);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_VAP_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	
    vapDatabaseParametersRegister.bitFields.queueVapDbEntry4BSize = (sizeof(VapDb_t))>>2;
	vapDatabaseParametersRegister.bitFields.queueVapSecondaryDbEntry4BOffset = (FM_STRUCT_OFFSET(VapDb_t,tid))>>2;
	// goldman - Elik L. wrote: Currently, an identical register also resides within the MAC_TX_HC register file
	//			    The MAC_TX_HC register is still in use - both registers should be configured with the same value.
	vapDatabaseParametersRegister.bitFields.queueVapDbBssid4BOffset = (FM_STRUCT_OFFSET(VapDb_t,bssid) >> 0x2);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_VAP_DB_SIZE, vapDatabaseParametersRegister.val);
    
	
	// configure the GLOBAL data base parameters - currently we set it to VAP DB
    RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GLOBAL_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GLOBAL_DB_SIZE, vapDatabaseParametersRegister.val);

	
	//configure the GPLP data base parameters - currently we set it to VAP DB
    RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPLP_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPLP_DB_SIZE, vapDatabaseParametersRegister.val);

	
	// configure the GPHP data base parameters - currently we set it to VAP DB
    RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPHP_DB_BASE_ADDR, vapDatabaseAddressRegister.val);
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_GPHP_DB_SIZE, vapDatabaseParametersRegister.val);


	// configure the beacon data base parameters - currently we set it to VAP DB
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_BEACON_DB_BASE_ADDR, vapDatabaseAddressRegister.val);	
	RegAccess_Write(REG_MAC_ADDR2INDEX_QUEUE_BEACON_DB_SIZE, vapDatabaseParametersRegister.val);

	// configure init parameters - first index and num of indexes. 
	initParamsRegister.bitFields.firstIndex = ConfigurationManager_GetFirstSidInMyBand();
	initParamsRegister.bitFields.maxNumOfIndexes = ConfigurationManager_GetNumOfSupportedStationsInMyBand();
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_INIT_PARAMS, initParamsRegister.val);

	/* Initialize the MAC 2 index module */
	commandRegister.bitFields.macAddr2IndexCommand = MAC_2_INDEX_COMMAND_INITIALIZE;
	RegAccess_Write(REG_MAC_ADDR2INDEX_MAC_ADDR2INDEX_COMMAND, commandRegister.val);
}


