/***********************************************************************************
 File:			NdpaManager.c
 Module:		Ndpa Manager
 Purpose: 		Generate NDPA Message for transmission and monitors success of BF sequence for training
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"

#ifndef TRAINING_WAVE600_Z0

#include "NdpaManager.h"
#include "ResourceManager_API.h"
#include "TxPacketsClassifier_API.h"
#include "Locker_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ShramStationDatabase.h"
#include "ShramPacketDescriptors.h"
#include "HwQManager_API.h"
#include "ShramGenriscNdpa.h"
#include "linkAdaptation_api.h"
#if !defined(ENET_INC_ARCH_WAVE600)
#include "AggregationBuilder_Api.h"
#endif 


/*---------------------------------------------------------------------------------
/						Defines 					
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_NDPA_MANAGER 
#define LOG_LOCAL_FID 0

//#define NDPA_MANAGER_DEBUG

static void ndpaManagerStartTxReq(void *parameter);
static void ndpaManagerStopTxInWaitForPd(void* parameter);
static void ndpaManagerIgnoreEvent(void* parameter);
static void ndpaManagerStopTxReq(void* parameter);
static void ndpaManagerTxCfmInWaitTrainingEnd(void* parameter);
static void ndpaManagerTxCfmWaitForLock(void * parameter);
static void ndpaManagerTxCfmWaitForPdRelease(void *parameter);
static void ndpaManagerIllegalEvent(void *parameter);
static void ndpaManagerLaBfReportIndPdReleased(void *parameter);
static void ndpaManagerLaBfReportIndPdNotReleased(void *parameter);
static void ndpaManagerLaBfReportIndWaitForLock(void* parameter);
static void ndpaManagerLockCfmInWaitForLock(void *parameter);
static void ndpaManagerPdAllocatedInWaitForPd(void* parameter);
static void ndpaManagerPdAllocatedInPendingStop(void *parameter);
static void ndpaManagerLockCfmInPdReleased(void *parameter);
static void ndpaManagerLockCfmInTrainingEnded(void *parameter);
static void ndpaManagerRunStateMachine(NdpaManagerEvents_e event, void* parameters);	
static void ndpaManagerChangeState(NdpaManagerState_e state);
static void ndpaManagerBuildAndSendNdpaMessage(TxPd_t *trainingPd, uint8 numOfStations);
static bool ndpaManagerRemovePdFromQueue(TxPd_t *pd);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static NdpaManagerSm_t NdpaManagerSm[NDPA_MANAGER_STATE_NUM_STATES] = 
{
	// NDPA_MANAGER_STATE_IDLE
	{
		ndpaManagerStartTxReq,					// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},
	// NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_NOT_RELEASED
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerStopTxReq,					// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerTxCfmInWaitTrainingEnd,		// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LOCK_CONFIRMED,
		ndpaManagerLaBfReportIndPdNotReleased,	// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},
	// NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_RELEASED
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIgnoreEvent,				    // NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerLaBfReportIndPdReleased,		// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},
	// NDPA_MANAGER_STATE_WAIT_FOR_LOCK
	{
		ndpaManagerIllegalEvent, 				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerTxCfmWaitForLock,			// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerLockCfmInWaitForLock,		// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerLaBfReportIndWaitForLock,	// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},
	// NDPA_MANAGER_STATE_WAIT_FOR_PD
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerStopTxInWaitForPd,			// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerPdAllocatedInWaitForPd,		// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},	
	// NDPA_MANAGER_STATE_WAIT_FOR_PD_RELEASE
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIgnoreEvent,					// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerTxCfmWaitForPdRelease,		// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},	
	// 	NDPA_MANAGER_STATE_WAIT_FOR_PD_PENDING_STOP
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerPdAllocatedInPendingStop, 	// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
		
	},
	// NDPA_MANAGER_STATE_WAIT_FOR_LOCK_PD_RELEASED
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerLockCfmInPdReleased,			// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerLaBfReportIndWaitForLock,    // NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},
	// NDPA_MANAGER_STATE_WAIT_FOR_LOCK_TRAINING_ENDED
	{
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_START_TX_REQ
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_STOP_TX_REQ
		ndpaManagerTxCfmWaitForLock,			// NDPA_MANAGER_EVENT_TX_CFM
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_PD_ALLOCATED
		ndpaManagerLockCfmInTrainingEnded,		// NDPA_MANAGER_EVENT_LOCK_CONFIRMED
		ndpaManagerIllegalEvent,				// NDPA_MANAGER_EVENT_LA_BF_REPORT_IND
	},
				
};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

// NDPA manager Database
static NdpaManagerDb_t	NdpaManagerDb;

/*---------------------------------------------------------------------------------
/						Public Functions Definitions							
/----------------------------------------------------------------------------------*/
	
// the function is invoked by Training manager
void NdpaManager_StartTxReq(NdpaManagerStartTxReq_t *ndpaStartTxParams)
{
	ndpaManagerRunStateMachine(NDPA_MANAGER_EVENT_START_TX_REQ, ndpaStartTxParams);
}

// the function is invoked by Training manager
void NdpaManager_StopTxReq(void)
{
	ndpaManagerRunStateMachine(NDPA_MANAGER_EVENT_STOP_TX_REQ, PNULL);
}

void NdpaManager_TxCfm(K_MSG *ndpaManagerMessage)
{
	NdpaManagerPacketConfirmed_t*	packetCfmMsg = (NdpaManagerPacketConfirmed_t *)pK_MSG_DATA(ndpaManagerMessage);
	// Run the state machine
	ndpaManagerRunStateMachine(NDPA_MANAGER_EVENT_TX_CFM,  packetCfmMsg->pd); 
}

void NdpaManager_PdAllocated(K_MSG *ndpaManagerMessage)
{
	RmPdFreeDescResponse_t *pdAllockMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(ndpaManagerMessage);
	// Run the state machine
	ndpaManagerRunStateMachine(NDPA_MANAGER_EVENT_PD_ALLOCATED,  pdAllockMsg->packetDescriptor); 
}

void NdpaManager_QueueLockCfm(K_MSG *ndpaManagerMessage)
{
	LockReqCb_t*	lockCfmMsg = (LockReqCb_t *)pK_MSG_DATA(ndpaManagerMessage);
	// Run the state machine
	ndpaManagerRunStateMachine(NDPA_MANAGER_EVENT_LOCK_CONFIRMED,  lockCfmMsg); 
}

void NdpaManager_LaBfReportInd(K_MSG *ndpaManagerMessage)
{
	LaBfReportInd_t* laMsg = (LaBfReportInd_t*)pK_MSG_DATA(ndpaManagerMessage);
	ndpaManagerRunStateMachine(NDPA_MANAGER_EVENT_LA_BF_REPORT_IND,  laMsg); 	
}

/*---------------------------------------------------------------------------------
/						Static Functions Definitions							
/----------------------------------------------------------------------------------*/

static void ndpaManagerIllegalEvent(void *parameter)
{
	FATAL("ndpaManagerIllegalEvent");
}

static void ndpaManagerChangeState(NdpaManagerState_e state)
{
	ILOG0_D("ndpaManagerChangeState: newState = %d", state);

	NdpaManagerDb.state = state;
}

static void ndpaManagerRunStateMachine(NdpaManagerEvents_e event,void * parameter)
{
	DEBUG_ASSERT(event < NDPA_MANAGER_EVENT_NUM_EVENTS);

	ILOG0_DD("ndpaManagerRunStateMachine: CurrentState = %d, event = %d", NdpaManagerDb.state, event);

	// Send the event to the state machine		
	NdpaManagerSm[NdpaManagerDb.state].handler[event](parameter);	
}

static void ndpaManagerStartTxReq(void *parameter)
{
	NdpaManagerStartTxReq_t*				ndpaStartTxParams = (NdpaManagerStartTxReq_t*)parameter;
	uint8 									i;
	RmPdRequestFillParameters_t 			pdRequestFillParameters;
	uint8									allocationStatus = FALSE;	
#ifdef ENET_INC_ARCH_WAVE600
	K_MSG 									*pMsg = OSAL_GET_MESSAGE(sizeof(uint8));
	uint8 									*newBwLimit = (uint8 *)pK_MSG_DATA(pMsg);
#endif //ENET_INC_ARCH_WAVE600

	DEBUG_ASSERT(ndpaStartTxParams->numOfStations <= GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING);
	
#ifdef NDPA_MANAGER_DEBUG
	ILOG0_V("ndpaManagerStartTxReq: received NdpaStartTx !!!");
#endif
	
	// upon beginning of new training sequence reset all DBs
	memset(&NdpaManagerDb, 0, sizeof(NdpaManagerDb_t));
	memset(&SendersTrainingDb, 0, sizeof(TxSenderTrainingStationsDb_t)); // TBD CDB - this doesn't support CDB yet
	
	NdpaManagerDb.vapId = ndpaStartTxParams->vapId;
	NdpaManagerDb.bw = ndpaStartTxParams->currentBw;
	
	for(i=0; i < ndpaStartTxParams->numOfStations; i++)
	{
		SendersTrainingDb.staInfo[i].staId = ndpaStartTxParams->trainingVector[i];
		SendersTrainingDb.staInfo[i].ncIndex = StaDbHwEntries[ndpaStartTxParams->trainingVector[i]].common.maxNssTx; // extract number of special streams of specific station
	}
	
	// save number of stations in this training
	NdpaManagerDb.numOfStations = ndpaStartTxParams->numOfStations;
	
	// request PD for NDPA message
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	pdRequestFillParameters.returnMsgType = NDPA_MANAGER_PD_ALLOCATED;
	pdRequestFillParameters.returnTaskId = TASK_GROUP_MANAGER;
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_MANAGEMENT_FROM_FW, &pdRequestFillParameters);
	
	if (allocationStatus == TRUE) // PD was successfully allocated
	{
		// Build the message and send it
#ifdef ENET_INC_ARCH_WAVE600
		*newBwLimit = NdpaManagerDb.bw;
		OSAL_SEND_MESSAGE(PAC_MANAGER_SET_MU_TRAINING_BW_LIMIT, TASK_PAC_MANAGER, pMsg, NdpaManagerDb.vapId);
#else
		AggregationBuilder_SetMuBfTrainingBwLimit(NdpaManagerDb.bw);
#endif //ENET_INC_ARCH_WAVE600

		ndpaManagerBuildAndSendNdpaMessage(pdRequestFillParameters.packetDescriptor, ndpaStartTxParams->numOfStations);

		// Change state
		ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_NOT_RELEASED);
	}
	else // there was no available PD
	{
		// Save the request ID - in case StopTxReq will arrive in future and we will have to cancel PD request
		NdpaManagerDb.pdRequestId = pdRequestFillParameters.requestId;

		// Change state
		ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_FOR_PD);
	}	

}

static void ndpaManagerStopTxReq(void* parameter)
{
	RequesterLockParams_t 	lockRequesterParams;

#ifdef NDPA_MANAGER_DEBUG
	ILOG0_V("ndpaManagerStopTxReq: received NdpaStoptTx !!!");
#endif
	//Stop Training sequence arrived - try to lock queue and to extract NDPa message if was not transmitted yet
	
	// Lock GPLP queue and wait for LockCfm
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	lockRequesterParams.callerContext = NULL;
	lockRequesterParams.returnMsg = 	NDPA_MANAGER_LOCK_CONFIRMED;
	lockRequesterParams.returnTask = 	TASK_GROUP_MANAGER;	
	Locker_LockSingleQueue(HW_TX_Q_TYPE_GPLP, NdpaManagerDb.vapId, 0x0, &lockRequesterParams);

	// Change state
	ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_FOR_LOCK);
}


static void ndpaManagerTxCfmInWaitTrainingEnd(void* parameter)
{
	
	TxPd_t* pPd = (TxPd_t*)CONVERT_OFFSET_TO_PD(NdpaManagerDb.ndpaPdOffset);
	bool sendNdpaCfm = FALSE;
	
	// verify that returned pd is the one stored in NDPA manager db
	DEBUG_ASSERT((uint32)pPd == (uint32)parameter);
	
	// if packet was not transmitted  move to IDLE state and send NdpaCfm immediately
	if(pPd->retransmissionIndication == 0)
	{
		ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
		sendNdpaCfm = TRUE;
	}
	// if packet was transmitted keep waiting for training results in NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_RELEASED state
	else
	{
		ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_RELEASED);
		// save in db Pd was released
		NdpaManagerDb.pdReleased = TRUE;
	}
	// Release the PD
	ResourceManager_ReleaseDescriptor(pPd, DESC_POOL_MANAGEMENT_FROM_FW);
	
	// TxCfm signal indicates that packet was not transmitted, so training sequence has failed
	if(sendNdpaCfm == TRUE)
	{
		TrainingManagerNdpaCfm_t status;
		status.status = TRAINING_MANAGER_NDPA_STATUS_FAIL;
		TrainingManagerNdpaCfm(&status);
	}
	
}

static void ndpaManagerTxCfmWaitForLock(void * parameter)
{
	TxPd_t* pPd = (TxPd_t*)CONVERT_OFFSET_TO_PD(NdpaManagerDb.ndpaPdOffset);
	// verify that returned pd is the one stored in NDPA manager db
	DEBUG_ASSERT((uint32)pPd == (uint32)parameter);

	// Release the PD
	ResourceManager_ReleaseDescriptor(pPd, DESC_POOL_MANAGEMENT_FROM_FW);
	// save indication in DB that Pd was released
	NdpaManagerDb.pdReleased = TRUE;
	ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_FOR_LOCK_PD_RELEASED);
}

static void ndpaManagerTxCfmWaitForPdRelease(void *parameter)
{
	TxPd_t* pPd = (TxPd_t*)CONVERT_OFFSET_TO_PD(NdpaManagerDb.ndpaPdOffset);
	TrainingManagerNdpaCfm_t status;
		
	// verify that returned pd is the one stored in NDPA manager db
	DEBUG_ASSERT((uint32)pPd == (uint32)parameter);
	
	// Release the PD
	ResourceManager_ReleaseDescriptor(pPd, DESC_POOL_MANAGEMENT_FROM_FW);
	// send NDPA_CFM and move to IDLE state
	ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
	status.status = NdpaManagerDb.bfReportStatus;
	TrainingManagerNdpaCfm(&status);
}

static void ndpaManagerStopTxInWaitForPd(void* parameter)
{
	bool pdAllocationRequestCancelled = FALSE;
	// try to cancel the PD allocation pending request
	pdAllocationRequestCancelled = ResourceManager_RemoveRequest(NdpaManagerDb.pdRequestId, DESC_POOL_MANAGEMENT_FROM_FW);
	
	if (pdAllocationRequestCancelled == TRUE)
	{
		TrainingManagerNdpaCfm_t status;
		// Pd request was successfully canceled.
		// Change state to idle
		ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
		status.status = TRAINING_MANAGER_NDPA_STATUS_FAIL;
		// Send confirmation with status FAIL to Training manager
		TrainingManagerNdpaCfm(&status);
	}
	else
	{
		// PD request couldn't be canceled - will be provided later
		// Change state and wait for it.
		ndpaManagerChangeState( NDPA_MANAGER_STATE_WAIT_FOR_PD_PENDING_STOP);
	}

}

static void ndpaManagerIgnoreEvent(void* parameter)
{
#ifdef NDPA_MANAGER_DEBUG
	ILOG0_D("ndpaManagerIgnoreEvent: CurrentState = %d", NdpaManagerDb.state);
#endif
}


static void ndpaManagerLaBfReportIndPdReleased(void *parameter)
{
	TrainingManagerNdpaCfm_t status;

	status.status = (TrainingManagerNdpaCfmStatus_e)( (LaBfReportInd_t*)parameter)->status;

	ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
#ifdef NDPA_MANAGER_DEBUG
	ILOG0_D("ndpaManagerLaBfReportIndPdReleased: status = %d", status.status);
#endif
	TrainingManagerNdpaCfm(&status);
}

static void ndpaManagerLaBfReportIndPdNotReleased(void *parameter)
{
	NdpaManagerDb.bfIndReceived = TRUE;
	NdpaManagerDb.bfReportStatus = (TrainingManagerNdpaCfmStatus_e)((LaBfReportInd_t*)parameter)->status;
#ifdef NDPA_MANAGER_DEBUG
	ILOG0_D("ndpaManagerLaBfReportIndPdNotReleased: status = %d", NdpaManagerDb.bfReportStatus);
#endif
	ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_FOR_PD_RELEASE);
}

static void ndpaManagerLaBfReportIndWaitForLock(void* parameter)
{
	LaBfReportInd_t* bfReportInd = (LaBfReportInd_t*)parameter;	

	NdpaManagerDb.bfIndReceived = TRUE;
	NdpaManagerDb.bfReportStatus = (TrainingManagerNdpaCfmStatus_e)bfReportInd->status;
#ifdef NDPA_MANAGER_DEBUG
	ILOG0_D("ndpaManagerLaBfReportIndWaitForLock: status = %d", NdpaManagerDb.bfReportStatus);
#endif
	ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_FOR_LOCK_TRAINING_ENDED);
}

static void ndpaManagerPdAllocatedInWaitForPd(void* parameter)
{
	TxPd_t*	pd = (TxPd_t*)parameter;
#ifdef ENET_INC_ARCH_WAVE600
	K_MSG *pMsg = OSAL_GET_MESSAGE(sizeof(uint8));
	uint8 *newBwLimit = (uint8 *)pK_MSG_DATA(pMsg);;
#endif //ENET_INC_ARCH_WAVE600

	ResourceManager_ReleaseRequest(NdpaManagerDb.pdRequestId, DESC_POOL_MANAGEMENT_FROM_FW);
	// No need to fill in TrainingSendersDb it was done on StartTx
	// Build the message and send it

#ifdef ENET_INC_ARCH_WAVE600
	*newBwLimit = NdpaManagerDb.bw;
	OSAL_SEND_MESSAGE(PAC_MANAGER_SET_MU_TRAINING_BW_LIMIT, TASK_PAC_MANAGER, pMsg, NdpaManagerDb.vapId);
#else
	AggregationBuilder_SetMuBfTrainingBwLimit(NdpaManagerDb.bw);
#endif //ENET_INC_ARCH_WAVE600
	ndpaManagerBuildAndSendNdpaMessage(pd, NdpaManagerDb.numOfStations);
	
	// Change state
	ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_NOT_RELEASED);
}

// TODO: write common code for this function and for ndpaManagerTxCfmWaitForPdRelease (the difference is status in NDPA_CFM)
static void ndpaManagerPdAllocatedInPendingStop(void *parameter)
{
	TrainingManagerNdpaCfm_t status;

	ResourceManager_ReleaseDescriptor((TxPd_t*)parameter, DESC_POOL_MANAGEMENT_FROM_FW);
	ResourceManager_ReleaseRequest(NdpaManagerDb.pdRequestId, DESC_POOL_MANAGEMENT_FROM_FW);
		
	// Change state
	ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
	status.status = TRAINING_MANAGER_NDPA_STATUS_FAIL;
	TrainingManagerNdpaCfm(&status);
}

static void ndpaManagerLockCfmInWaitForLock(void *parameter)
{
	LockReqCb_t*			lockReqCb = (LockReqCb_t*)parameter;
	TxSelectorLockStatus_e	lockStatus = lockReqCb->lockStatus;
		
	TxPd_t*		pd = (TxPd_t*)CONVERT_OFFSET_TO_PD(NdpaManagerDb.ndpaPdOffset);
	bool					pdFound = FALSE;

	// check that correct vapId returned on lock
	DEBUG_ASSERT(lockReqCb->stationOrVapNum == NdpaManagerDb.vapId);

	ILOG2_D("ndpaManagerLockCfmInWaitForLock: Lock status %d", lockStatus);
	// In case of failure we will treat the pd as not found
	if (lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		// Queue is locked now. Try to find the PD.
		pdFound = ndpaManagerRemovePdFromQueue(pd);
		// Unlock the queue
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPLP, lockReqCb->stationOrVapNum, (0x1 << 0x0));
	}

	//Check if PD was found
	if (pdFound == TRUE)
	{
		TrainingManagerNdpaCfm_t status;
	
		// Release the PD we found
		ResourceManager_ReleaseDescriptor(pd, DESC_POOL_MANAGEMENT_FROM_FW);
		// Change state to idle and send NDPA_CFM
		ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
		status.status = TRAINING_MANAGER_NDPA_STATUS_FAIL;
		TrainingManagerNdpaCfm(&status);
	}
	else
	{
		// PD was not found. It means it is on the way back.
		// Change state and wait for it
		ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_NOT_RELEASED);		
	}
}

static void ndpaManagerLockCfmInPdReleased(void *parameter)
{
	LockReqCb_t*			lockReqCb = (LockReqCb_t*)parameter;
	TxSelectorLockStatus_e	lockStatus = lockReqCb->lockStatus;	
	
	// check that correct vapId returned on lock
	DEBUG_ASSERT(lockReqCb->stationOrVapNum == NdpaManagerDb.vapId);
		
	// If lock succeeded unlock the queue
	if (lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		// Unlock the queue because at this point Pd already released so not found in queue
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPLP, lockReqCb->stationOrVapNum, (0x1 << 0x0));
	}
		
	//Check BF report ind was received earlier - training ended, move to IDLE
	if (NdpaManagerDb.bfIndReceived == TRUE)
	{
		TrainingManagerNdpaCfm_t status;
	
		// Change state to idle and send NDPA_CFM
		ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
		status.status = NdpaManagerDb.bfReportStatus;
		TrainingManagerNdpaCfm(&status);
	}
	else
	{
		// wait for training sequence to end
		ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_TRAINING_END_PD_RELEASED);		
	}
}

static void ndpaManagerLockCfmInTrainingEnded(void *parameter)
{
	LockReqCb_t*			lockReqCb = (LockReqCb_t*)parameter;
	TxSelectorLockStatus_e	lockStatus = lockReqCb->lockStatus;	
	
	// check that correct vapId returned on lock
	DEBUG_ASSERT(lockReqCb->stationOrVapNum == NdpaManagerDb.vapId);
		
	// If lock succeeded unlock the queue
	if (lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		// Unlock the queue because at this point training already ended so not found in queue
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPLP, lockReqCb->stationOrVapNum, (0x1 << 0x0));
	}
		
	//Check if pd was released earlier - training ended, move to IDLE
	if (NdpaManagerDb.pdReleased == TRUE)
	{	
		TrainingManagerNdpaCfm_t status;

		status.status = NdpaManagerDb.bfReportStatus;
		// Change state to idle and send NDPA_CFM
		ndpaManagerChangeState(NDPA_MANAGER_STATE_IDLE);
		TrainingManagerNdpaCfm(&status);
	}
	else
	{
		// wait for pd
		ndpaManagerChangeState(NDPA_MANAGER_STATE_WAIT_FOR_PD_RELEASE);		
	}
}

static void ndpaManagerBuildAndSendNdpaMessage(TxPd_t *trainingPd, uint8 numOfStations)
{
	NdpaFrame_t*	ndpaFrame = PNULL;
	NdpaStaInfo*	stationsInfo = PNULL;		
	uint8 i = 0;
	
	// Save the PD offset
	NdpaManagerDb.ndpaPdOffset= CONVERT_PD_TO_OFFSET(trainingPd);
	
	// Fill PD fields
#ifdef ENET_INC_ARCH_WAVE600 
	trainingPd->ctrlDuration= MAX_NDPA_DURATION;
#else
	trainingPd->trainingDuration= MAX_NDPA_DURATION;
#endif
	trainingPd->mcUnicast = MULTICAST;
	trainingPd->pdType = PD_TYPE_MU_TRAINING;
	trainingPd->retransmissionIndication = 0;
	trainingPd->aggregationIndication = 0;
	trainingPd->txQVapId = NdpaManagerDb.vapId; 
	trainingPd->txQGroupId = HW_TX_Q_TYPE_GPLP; 
	trainingPd->numOfTrainingStas = (numOfStations - 1);
	// data length includes NDPA message 802.11 header size + payload size. 
	// payload contains 1 byte of sounding token + (number of stations in training * sizeof each station info)
	trainingPd->dataLength = sizeof(NDPA_FRAME_HEADER)+1+(numOfStations*sizeof(NdpaStaInfo));
	trainingPd->trainingPerStaStructPointer = (uint32)(&SendersTrainingDb);
	// in case numOfStations ==1, unicastMcast field must be unicast and dest mac addr must be the addr of teh station
	if(numOfStations == 1)
	{
		trainingPd->mcUnicast = UNICAST;
	}
	
	// Build Ndpa frame - fill in ndpa payload data
	ndpaFrame = (NdpaFrame_t *)trainingPd->packetPointer;
	// set to 0 ndpa frame content - some fields will be overwritten by Sender
	memset(ndpaFrame, 0, sizeof(NdpaFrame_t));
	stationsInfo = ndpaFrame->ndpaPayload.stationsInfo;
	for(i = 0; i < numOfStations; i++)
	{
		stationsInfo[i].aid = (SendersTrainingDb.staInfo[i].staId + 1); // convert SIDs provided by Training Mng to AIDs
		stationsInfo[i].feedbackType = NDPA_MULTI_USER;
		stationsInfo[i].ncIndex = SendersTrainingDb.staInfo[i].ncIndex;
	}
	
	// Put frame on GPLP queue
	TxPacketClassifier_SendNdpaMessage(trainingPd);

}

static bool ndpaManagerRemovePdFromQueue(TxPd_t *pd)
{
	HwQueueManagerRequestParams_t 	hwQueueManagerRequestParams;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = pd->txQVapId;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPLP;
	
	return HwQManager_RemovePdFromQueue(&hwQueueManagerRequestParams, pd);
}

void ndpaManager_Init()
{
	
}


#endif // TRAINING_WAVE600_Z0



