/***************************************************************
 File:		OtfAggregator.c
 Module:	  	OtfAggregator
 Purpose: 	holds all the implementation of the OtfAggregator  API functions
 Description:  This module contains the implementation of the functions of the OtfAggregator module 
 			which is responsible for creating the next transmission 
***************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "stringLibApi.h"
#include "OtfAggregator_Api.h"
#include "OtfAggregator.h"
#include "OtfAggregator_Ram.h"
#include "queue_utility.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "HwQManager_API.h"
#include "TxHandler_Api.h"
#include "DescriptorsDefinitions.h"
#include "OSAL_UpperMacMessages.h"
#include "ShramOtfAggregator.h"
#include "OtfaRegs.h"
#include "OtfaErrFifo_Descriptors.h"
#include "RegAccess_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "HeGroupManager_API.h"
#include "Utils_Api.h"
#include "ShramPacketDescriptors.h"
#include "PacketDescriptor.h"
#include "Locker_Api.h"
#include "linkAdaptation_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/

#define LOG_LOCAL_GID   GLOBAL_GID_TX_HW_ACCELERATORS
#define LOG_LOCAL_FID 	8


#define OTFA_MAX_MPDU_SIZE_PHY_MODE_11_A_G		(MAX_HT_MPDU_IN_A_MPDU_LENGTH_TX)	// Max MPDU size in case of PHY mode 11ag
#define OTFA_MAX_MPDU_SIZE_PHY_MODE_11_B		(MAX_HT_MPDU_IN_A_MPDU_LENGTH_TX)	// Max MPDU size in case of PHY mode 11b
#define OTFA_MAX_MPDU_SIZE_PHY_MODE_11_N		(MAX_MPDU_LENGTH_OPTION2_HT)  	// Max MPDU size in case of PHY mode 11n
#define OTFA_MAX_MPDU_SIZE_PHY_MODE_11_AC		(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_TX) // Max MPDU size in case of PHY mode 11ac
#define OTFA_MAX_MPDU_SIZE_PHY_MODE_11_AX		(MAX_HE_MPDU_IN_A_MPDU_LENGTH_TX) // Max MPDU size in case of PHY mode 11ax

#define OTFA_SLOWEST_HW_MIN_PD_SPACE			(50)							// units of 1/128 us (390ns)

#define OTFA_MANAGEMENT_TID_IN_ERROR_FIFO		(8)								// Usuall we use 0xF as management TID. Here they use 8.


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Debug Section									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void otfAggr_GetFifoEntry(OtfaErrFifo_t* pCurrentOtfAggrReport);
static void otfAggr_IncrementFifoEntryPtr(void);
static bool otfAggrr_IsReportsFifoNotEmpty(void);
static void otfAggr_DecFifoNumEntriesCount(uint8 value);
static void otfAggregator_RecoverFromWrongPsduOrMpduSize(OtfaErrFifo_t* currentOtfAggrReport);
static void otfAggregator_RecoverFromPdTypeNotAllowed(OtfaErrFifo_t* currentOtfAggrReport);



/*---------------------------------------------------------------------------------
						Static Variables									
/----------------------------------------------------------------------------------*/

#if defined (ENET_INC_LMAC0)
OtfaErrFifo_t*	readOtfAggrReportPtr = &(otfAggrFifoReport[0]);
#elif defined (ENET_INC_LMAC1)
OtfaErrFifo_t*	readOtfAggrReportPtr = &(otfAggrFifoReport_B1[0]);
#endif 

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/




/**********************************************************************************

otfAggr_GetFifoEntry  


Description: Copies the first report from the OTF_Aggregator status report, to a given local variable  
------------

Input:  reference to a local variable - current OTF report  
-----
		
Output:	Fills the variable with the first report content  
-------
	
**********************************************************************************/

static void otfAggr_GetFifoEntry(OtfaErrFifo_t* pCurrentOtfAggrReport)
{	
	MEMCPY (pCurrentOtfAggrReport, readOtfAggrReportPtr, sizeof (OtfaErrFifo_t));
}

/**********************************************************************************

otfAggr_IncrementFifoEntryPtr  


Description: 
------------
Increments the global otf_Fifo_pointer to point the next report in the OTF_Aggregator status fifo 


Input:	None 
-----
		
Output:	None
-------

**********************************************************************************/

static void otfAggr_IncrementFifoEntryPtr(void)
{
#if defined (ENET_INC_LMAC0)
	if ((readOtfAggrReportPtr ) == (&(otfAggrFifoReport[0]) + NUM_OF_OTF_AGGR_REPORTS - 1))
	{
		readOtfAggrReportPtr = &otfAggrFifoReport[0];
	}
#elif defined (ENET_INC_LMAC1)
	if ((readOtfAggrReportPtr ) == (&(otfAggrFifoReport_B1[0]) + NUM_OF_OTF_AGGR_REPORTS - 1))
	{
		readOtfAggrReportPtr = &otfAggrFifoReport_B1[0];
	}
#endif 
	else
	{
		readOtfAggrReportPtr++;
	}
}

/**********************************************************************************

otfAggr_DecFifoNumEntriesCount  


Description: 
------------
Decrements the number of pending reports in the OTF_Aggregator status fifo  


Input:	None 
-----
		
Output:	None
-------

**********************************************************************************/

static void otfAggr_DecFifoNumEntriesCount(uint8 value)
{
	RegOtfaTidErrorFifoRdEntriesNum_u 	otfaTidErrorFifoRdEntriesNumReg;

	/*Decrement number of reports that poped out from fifo*/
	otfaTidErrorFifoRdEntriesNumReg.val = 0;
	otfaTidErrorFifoRdEntriesNumReg.bitFields.tidErrorFifoRdEntriesNum = value;
	RegAccess_Write(REG_OTFA_TID_ERROR_FIFO_RD_ENTRIES_NUM,otfaTidErrorFifoRdEntriesNumReg.val);	
}

/**********************************************************************************

otfAggrr_IsReportsFifoNotEmpty  


Description:	Checks if there are more reports on the OTF_Aggregator status fifo 
------------

Input:	None 
-----
		
Output:	True/False - cooresponding to the fifo's current status
-------
	
**********************************************************************************/

static bool otfAggrr_IsReportsFifoNotEmpty(void)
{
	RegOtfaTidErrorFifoDebug_u OtfaTidErrorFifoDebugReg;
	/*Read TID error FIFO not empty indication and return it as the "not empty" boolian value*/
	RegAccess_Read(REG_OTFA_TID_ERROR_FIFO_DEBUG,&OtfaTidErrorFifoDebugReg.val);
	ASSERT(OtfaTidErrorFifoDebugReg.bitFields.tidErrorFifoFull == 0);
	return((bool)OtfaTidErrorFifoDebugReg.bitFields.tidErrorFifoNotEmpty);
}

/**********************************************************************************

otfAggregator_RecoverFromWrongPsduOrMpduSize

Description:
------------

Input:
-----
		
Output:
-------
	
**********************************************************************************/
static void otfAggregator_RecoverFromWrongPsduOrMpduSize(OtfaErrFifo_t* currentOtfAggrReport)
{
	HwQueueManagerRequestParams_t 		hwQueueManagerRequestParams;
	K_MSG*								pIndMsg =  NULL;
	OtfAggregatorRecoverFromError_t 	ErrorRecoverInd;	
	OtfAggregatorRecoverFromError_t* 	pErrorRecoverIndMsg = NULL;		
 
	memset(&ErrorRecoverInd, 0x0, sizeof(OtfAggregatorRecoverFromError_t));
	ErrorRecoverInd.stationIndex = currentOtfAggrReport->staId;
	ErrorRecoverInd.tid = currentOtfAggrReport->tid;
	
	// We should not have wrong PSDU size for management. Management should be transmitted in all rates. It's not AMSDUed.
	ASSERT(currentOtfAggrReport->tid != OTFA_MANAGEMENT_TID_IN_ERROR_FIFO);

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = currentOtfAggrReport->staId;
	hwQueueManagerRequestParams.secondaryAddr = currentOtfAggrReport->tid;
				
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	ErrorRecoverInd.headPd = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;

	// We should not get NULL PD when we peek the queue. It means we can't recover and something was wrong.
	ASSERT(ErrorRecoverInd.headPd != NULL_PD);

	/* Tell TX MANAGER if the PD we failed to transmit is ETSI complient or not (WLANRTSYS-14443) */
#ifdef ENET_INC_ARCH_WAVE600B	
	ErrorRecoverInd.etsi5GDuringPsduLimitError = LinkAdaptivePdIsEtsi5GDuringPsduLimitError((bool)currentOtfAggrReport->psduLimit, (TxPd_t *)ErrorRecoverInd.headPd);
#endif

	// Prepare a MSG to Tx Manager (UM)
	pIndMsg = OSAL_GET_MESSAGE(sizeof(OtfAggregatorRecoverFromError_t));
	pErrorRecoverIndMsg = (OtfAggregatorRecoverFromError_t *)pK_MSG_DATA(pIndMsg);
	MEMCPY(pErrorRecoverIndMsg, &ErrorRecoverInd, sizeof(OtfAggregatorRecoverFromError_t));	
	
	// send MSG to Tx Manager to continue the recovery
	OSAL_SEND_MESSAGE(TX_MANAGER_AGGREGATOR_ERROR_RECOVER, TASK_TX_MANAGER, pIndMsg, VAP_ID_DO_NOT_CARE);

}


/**********************************************************************************

otfAggregator_RecoverFromPdTypeNotAllowed


Description:
------------

Input:
-----
		
Output:
-------
	
**********************************************************************************/
static void otfAggregator_RecoverFromPdTypeNotAllowed(OtfaErrFifo_t* currentOtfAggrReport)
{
	K_MSG *pMsg = NULL;
	HeGm_RecoverFromPdTypeNotAllowedRequest_t *pRecoverFromPdTypeNotAllowedRequest = NULL;

	//verify we are handling a not allowed pd type from unicast queues
	ASSERT(currentOtfAggrReport->qType == TXQ_GROUP_ID_UNICAST);

#if 0
	//unlock queue to let this station be serviced again
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, currentOtfAggrReport.staId, (0x1 << currentOtfAggrReport.tid));
#else
	pMsg = OSAL_GET_MESSAGE(sizeof(HeGm_RecoverFromPdTypeNotAllowedRequest_t));
	pRecoverFromPdTypeNotAllowedRequest = (HeGm_RecoverFromPdTypeNotAllowedRequest_t *)pK_MSG_DATA(pMsg);

 	pRecoverFromPdTypeNotAllowedRequest->staId = currentOtfAggrReport->staId;
	pRecoverFromPdTypeNotAllowedRequest->tid = currentOtfAggrReport->tid;
				
	//send MSG to HE Group Manager to continue the recovery
	OSAL_SEND_MESSAGE(HE_GROUP_MANAGER_RECOVER_FROM_PD_TYPE_NOT_ALLOWED_REQ, TASK_HE_GROUP_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
#endif
}

#ifndef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************

OtfAggregator_ConfigureVapModeBitmap  


Description:
------------
Configure the OTFA VAP mode bitmap - each bit represent the VAP configuration - AP/STA

Input:
-----
	
		
Output:
-------
	
**********************************************************************************/
void OtfAggregator_ConfigureVapModeBitmap(uint8 vapId, bool vapMode)
{
	RegOtfaVapIsSta_u otfaVapIsSta; 

 	RegAccess_Read(REG_OTFA_VAP_IS_STA, &(otfaVapIsSta.val));
	if(vapMode == FALSE) //AP_MODE
	{
 		Utils_ZeroBitInBitmap(&(otfaVapIsSta.val), vapId); 
	}
	else //STA_MODE
	{
 		Utils_SetBitInBitmap(&(otfaVapIsSta.val), vapId); 
	}
	RegAccess_Write(REG_OTFA_VAP_IS_STA, otfaVapIsSta.val);
}
#endif

/**********************************************************************************

otfAggrProcessReportQueue  


Description:
------------

Input:
-----
		
Output:
-------
	
**********************************************************************************/
void OtfAggregator_ProcessReportQueue(void)
{
	OtfaErrFifo_t 	currentOtfAggrReport;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	TxPd_t *currentPacketDescriptor = NULL;
	do
	{
		otfAggr_GetFifoEntry(&currentOtfAggrReport);
		/*Increment ptr must be after the break, otherwize FW missed reports*/
		otfAggr_IncrementFifoEntryPtr();
		/*Decrement one element from sequencer fifo*/
		otfAggr_DecFifoNumEntriesCount(1);

		SLOG0(0, 0, OtfaErrFifo_t, &currentOtfAggrReport);
		
		if(currentOtfAggrReport.pdTypeNotAllowed == 1)
		{
			SERIAL_TRACE("OtfAggregator_ProcessReportQueue, pdTypeNotAllowed = 1",0,0,0);
		}
		else
		{
			SERIAL_TRACE("OtfAggregator_ProcessReportQueue",0,0,0);
		}
  
		/* Call the corresponding handler for each report */ 
		if ((currentOtfAggrReport.mpduLengthPerPhyModeLimit)	|| 
			(currentOtfAggrReport.mpduLengthPerStaLimit)		||
			(currentOtfAggrReport.msdusInMpduLimit)				||
			(currentOtfAggrReport.msduLengthLimit)				||
			(currentOtfAggrReport.dmaLengthLimit)				||
			(currentOtfAggrReport.dmaLimitFirstUserPd)			||
			(currentOtfAggrReport.psduLimit))
		{
			otfAggregator_RecoverFromWrongPsduOrMpduSize(&currentOtfAggrReport);
		}
		else if (currentOtfAggrReport.pdTypeNotAllowed)
		{
			//check the pd type			
			memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
			hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
			hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
			hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
			hwQueueManagerRequestParams.primaryAddr = currentOtfAggrReport.staId;
			hwQueueManagerRequestParams.secondaryAddr = currentOtfAggrReport.tid;
			HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
			currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
			SLOG0(0, 0, TxPd_t, currentPacketDescriptor);

#if defined	(WORKAROUND_FOR_HW_BUG_IN_TIM_IE)
			if (((TxPd_t *)NULL_PD != currentPacketDescriptor) && 
				((PD_TYPE_NDP == currentPacketDescriptor->pdType) &&  
				(HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW == currentPacketDescriptor->pdSource)))
			{
				HwQManager_PopPacket(&hwQueueManagerRequestParams);
				hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
				hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;				
				HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);					
				Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, currentOtfAggrReport.staId, (0x1 << currentOtfAggrReport.tid));
			}
			else
#endif	
			{
				//pd type is not allowed in mu - need to dissolve all mu groups this sta/tid is part of
				otfAggregator_RecoverFromPdTypeNotAllowed(&currentOtfAggrReport);
			}
		}
		else
		{
			FATAL("OtfAggregator_ProcessReportQueue");
		}
	} while(otfAggrr_IsReportsFifoNotEmpty());

	//unmask event
#if defined (ENET_INC_LMAC0)
	HwErrors_UnMaskEvent0(HW_ERROR_OTF_TID_ERROR_FIFO_NOT_EMPTY);
#elif defined (ENET_INC_LMAC1)
	HwErrors_UnMaskEvent1(HW_ERROR_OTF_TID_ERROR_FIFO_NOT_EMPTY);
#endif
}



/**********************************************************************************

OtfAggregator_Initialize  


Description:
------------
Initialize the memory and internal registers of the OTF_Aggregator module

Input:
-----
	
		
Output:
-------
	
**********************************************************************************/
void OtfAggregator_Initialize(void)
{	
	RegOtfaTidErrorFifoBaseAddr_u 		otfaTidErrorFifoBaseAddrReg;
	RegOtfaTidErrorFifoDepthMinusOne_u	otfaTidErrorFifoDepthReg; 
	RegOtfaTidErrorFifoClearStrb_u		otfaTidErrorFifoClearStrb;
	RegOtfaSenderDurationSettingEn_u	otfaSenderDurationSettingEn;
	RegOtfaMaxMpduLength11Ag_u			otfaMaxMpduLength11Ag;
	RegOtfaMaxMpduLength11B_u			otfaMaxMpduLength11B;
	RegOtfaMaxMpduLengthHt_u			otfaMaxMpduLength11Ht;
	RegOtfaMaxMpduLengthVht_u			otfaMaxMpduLength11Vht;
	RegOtfaMaxMpduLengthHe_u			otfaMaxMpduLength11He;
	RegOtfaGeneralControl_u 			otfaGeneralControl;
	RegOtfaSlowestHwMinPdSpace_u		otfaSlowestHwMinPdSpace;
    RegOtfaPdErrEnable_u                otfaPdErrEnable;
#ifdef ENET_INC_ARCH_WAVE600B
    RegOtfaErrorIrqEnable_u             otfaErrorIrqEnable;
#endif
#ifdef ENET_INC_ARCH_WAVE600D2
	RegOtfaSwUpdateFragDb_u				otfaSwUpdateFragDbReg;
#endif

	otfaTidErrorFifoBaseAddrReg.val = 0x0; 
	otfaTidErrorFifoDepthReg.val 	= 0x0; 
	otfaTidErrorFifoClearStrb.val 	= 0x0; 
	otfaSenderDurationSettingEn.val = 0x0;
	otfaMaxMpduLength11Ag.val		= 0x0;
	otfaMaxMpduLength11B.val		= 0x0;
	otfaMaxMpduLength11Ht.val		= 0x0;
	otfaMaxMpduLength11Vht.val		= 0x0;
	otfaMaxMpduLength11He.val		= 0x0;
	otfaGeneralControl.val			= 0x0;
	otfaSlowestHwMinPdSpace.val		= 0x0;
    otfaPdErrEnable.val             = 0x0;
#ifdef ENET_INC_ARCH_WAVE600B
    otfaErrorIrqEnable.val          = 0x0;
#endif
#ifdef ENET_INC_ARCH_WAVE600D2
	otfaSwUpdateFragDbReg.val		= 0x0;
#endif


	/* Set OTF HW with report queue address and depth*/
#if defined (ENET_INC_LMAC0)
	memset(&otfAggrFifoReport[0], 0, (sizeof(OtfaErrFifo_t)*NUM_OF_OTF_AGGR_REPORTS));
	otfaTidErrorFifoBaseAddrReg.bitFields.tidErrorFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&(otfAggrFifoReport[0]));
#endif
#if defined (ENET_INC_LMAC1)
	memset(&otfAggrFifoReport_B1[0], 0, (sizeof(OtfaErrFifo_t)*NUM_OF_OTF_AGGR_REPORTS));
	otfaTidErrorFifoBaseAddrReg.bitFields.tidErrorFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&(otfAggrFifoReport_B1[0]));
#endif
	RegAccess_Write(REG_OTFA_TID_ERROR_FIFO_BASE_ADDR,otfaTidErrorFifoBaseAddrReg.val);

	otfaTidErrorFifoDepthReg.bitFields.tidErrorFifoDepthMinusOne = NUM_OF_OTF_AGGR_REPORTS-1;
	RegAccess_Write(REG_OTFA_TID_ERROR_FIFO_DEPTH_MINUS_ONE,otfaTidErrorFifoDepthReg.val);

	/*Clear fifo strobs */
	otfaTidErrorFifoClearStrb.bitFields.tidErrorFifoClearStrb = 0x1; 
	otfaTidErrorFifoClearStrb.bitFields.tidErrorFifoClearFullDropCtrStrb = 0x1; 
	otfaTidErrorFifoClearStrb.bitFields.tidErrorFifoClearDecLessThanZeroStrb = 0x1; 
	RegAccess_Write(REG_OTFA_TID_ERROR_FIFO_CLEAR_STRB,otfaTidErrorFifoClearStrb.val);
	
	otfaSenderDurationSettingEn.bitFields.senderDurationSettingEn = TRUE;
	RegAccess_Write(REG_OTFA_SENDER_DURATION_SETTING_EN,otfaSenderDurationSettingEn.val);


	otfaMaxMpduLength11Ag.bitFields.maxMpduLength11Ag 	= OTFA_MAX_MPDU_SIZE_PHY_MODE_11_A_G;
	RegAccess_Write(REG_OTFA_MAX_MPDU_LENGTH_11AG, otfaMaxMpduLength11Ag.val);

	otfaMaxMpduLength11B.bitFields.maxMpduLength11B		= OTFA_MAX_MPDU_SIZE_PHY_MODE_11_B;
	RegAccess_Write(REG_OTFA_MAX_MPDU_LENGTH_11B, otfaMaxMpduLength11B.val);

	otfaMaxMpduLength11Ht.bitFields.maxMpduLengthHt 	= OTFA_MAX_MPDU_SIZE_PHY_MODE_11_N;	
	otfaMaxMpduLength11Ht.bitFields.maxMpduLengthHtAgg 	= OTFA_MAX_MPDU_SIZE_PHY_MODE_11_A_G;
	RegAccess_Write(REG_OTFA_MAX_MPDU_LENGTH_HT, otfaMaxMpduLength11Ht.val);

	otfaMaxMpduLength11Vht.bitFields.maxMpduLengthVht = OTFA_MAX_MPDU_SIZE_PHY_MODE_11_AC;
	RegAccess_Write(REG_OTFA_MAX_MPDU_LENGTH_VHT, otfaMaxMpduLength11Vht.val);

	otfaMaxMpduLength11He.bitFields.maxMpduLengthHe = OTFA_MAX_MPDU_SIZE_PHY_MODE_11_AX;
	RegAccess_Write(REG_OTFA_MAX_MPDU_LENGTH_HE, otfaMaxMpduLength11He.val);


	/*set OTFA that DMA works with 256 byte chunks (should be correlate to configuration in REG_DMAC_WRAPPER_DMA0_CH0_CONFIG_REG
	leave all other fields as their default value;
	*/
	otfaGeneralControl.bitFields.retryMaxMpduLengthLimitCheckEn = TRUE;
	otfaGeneralControl.bitFields.retryMaxMsdusAtAmsduLimitCheckEn = TRUE;
	otfaGeneralControl.bitFields.forceDelimiterInSingleMpduHt = FALSE;
	otfaGeneralControl.bitFields.dmaTran256Bytes = OTFA_DMA_TRANSACTION_LENGTH_256_BYTES;
	otfaGeneralControl.bitFields.setEofInTf = FALSE;
	otfaGeneralControl.bitFields.eofConfA = TRUE;
	otfaGeneralControl.bitFields.eofConfB = FALSE;
	otfaGeneralControl.bitFields.eofConfC = TRUE;
	otfaGeneralControl.bitFields.eofLastOnly = FALSE;	
	RegAccess_Write(REG_OTFA_GENERAL_CONTROL, otfaGeneralControl.val);	

	// according to Nissim G. change value to 390ns (50/128 usec) in order to avoid auto-fill in Delia for small PDs
	otfaSlowestHwMinPdSpace.bitFields.slowestHwMinPdSpace = OTFA_SLOWEST_HW_MIN_PD_SPACE;
	RegAccess_Write(REG_OTFA_SLOWEST_HW_MIN_PD_SPACE, otfaSlowestHwMinPdSpace.val);	
#ifdef ENET_INC_ARCH_WAVE600D2
	// according to Nissim G. need to set to zero in init 
	RegAccess_Write(REG_OTFA_SW_UPDATE_FRAG_DB, otfaSwUpdateFragDbReg.val);	
#endif	
	
	otfaPdErrEnable.bitFields.pdErrEnable = 0xFFF; // Enable all. bit 0: MPDU_length_per_PHY_mode_limit. 1: MPDU_length_per_STA_limit. 2: MSDUs_in_MPDU_limit. 3: A_MSDU_disabled. 4: PD_type_not_allowed. 5: MSDU_length_limit. 6: DMA_length_limit. 7: Frag_not_supported. 8: DMA_limit_first_user_PD. 9: PSDU_limit. 10: Max_frags_limit. 11: TF_plus_frag_not_supported.
    RegAccess_Write(REG_OTFA_PD_ERR_ENABLE, otfaPdErrEnable.val);
	
#ifdef ENET_INC_ARCH_WAVE600B
    // Enable OTFA TID error FIFO interrupts
    otfaErrorIrqEnable.bitFields.tidErrorFifoFullDropIrqEnabled = TRUE;
    otfaErrorIrqEnable.bitFields.tidErrorFifoDecrementLessThanZeroIrqEnabled = TRUE;
    RegAccess_Write(REG_OTFA_ERROR_IRQ_ENABLE, otfaErrorIrqEnable.val);	
#endif
}

