/***********************************************************************************
 File:			PacketTrafficArbitrator
 Module:			PacketTrafficArbitrator
 Purpose:				2.4GHz coexistance
 Description:	This src file is the implementation of the Packet Traffic Arbitrator module
			which is responsible for arbitrating air-time between WIFI and other 2.4GHz radios.
 Note: Document : https://sharepoint.amr.ith.intel.com/sites/WLAN_NG/SitePages/Home.aspx?RootFolder=%2Fsites%2FWLAN%5FNG%2FShared%20Documents%2FSoftware%2Fwave500FwDesign%2F2%2E4G%20coexistence&FolderCTID=0x01200094BEF99F6593D44E89102DDB34B18198&View={37C4B5A7-6CB5-4A23-951D-71851EB0E62D}
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/ 
 
#include "ErrorHandler_Api.h"
#include "InterruptManager_Api.h"
#include "loggerAPI.h"
#include "MacHtExtensionsRegs.h"
#include "OSAL_Api.h"
#include "PacketTrafficArbitrator.h"
#include "Pac_Api.h"
#include "PacketTrafficArbitrator_api.h"
#include "queue_utility.h"
#include "RegAccess_Api.h"
#include "System_Timers.h"
#include "PacketTrafficArbitratorCommon.h"
#include "StatisticsManager_api.h"
#include "Statistics_Descriptors.h"
#include "stringLibApi.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_PACKET_TRAFFIC_ARBITRATOR
#define LOG_LOCAL_FID 0



// This global is always needed, since it is part of statistics which are exported to driver. 
PtaStatistics_t 				PtaStatistics;





#ifdef PTA_BUILD_IN_PLAT

/* Timer value defines for testsing*/
#ifdef PTA_TEST
#define PTA_STARVATION_TIMER_100_MILLISECONDS				 (100)
#define PTA_HW_TIMER_VAL_5000_MICROSECONDS					(5000)
#endif

/* HW Timer values */
#define PTA_HW_TIMER_BT_ALLOW_200_MICROSECONDS				 (200)	/* Make sure BT is using the grant 				*/
#define PTA_HW_TIMER_BT_DENY_200_MICROSECONDS				 (200)	/* Make sure BT is NOT using the grant 			*/
#define PTA_HW_TIMER_PROPAGATION_TIME_80_MICROSECONDS		  (80)	/* Deleate time since board is BLE and not BT 	*/

#define PTA_HW_TIMER_BT_DENY								(PTA_HW_TIMER_BT_DENY_200_MICROSECONDS - PTA_HW_TIMER_PROPAGATION_TIME_80_MICROSECONDS) 

#define PTA_HW_TIMER_BT_LAST_CHANCE_200_MICROSECONDS		 (200)	/* Last change for BT before PTA indicates that BT has an error */

#define PTA_HW_TIMER_ZIGBEE_ALLOW_200_MICROSECONDS			 (200)	/* Make sure ZigBee is using the grant */
#define PTA_HW_TIMER_ZIGBEE_DENY_200_MICROSECONDS			 (200)	/* Make sure ZigBee is NOT using the grant (handles ZigBee TX) */
#define PTA_HW_TIMER_ZIGBEE_SHORT_200_MICROSECONDS			 (200)	/* Make sure ZigBee is (NOT) using the grant (handles ZigBee RX of short frames)*/
#define PTA_HW_TIMER_ZIGBEE_MID_1300_MICROSECONDS			(1300)	/* Make sure ZigBee is (NOT) using the grant (handles ZigBee RX of average frames)*/
#define PTA_HW_TIMER_ZIGBEE_LONG_2600_MICROSECONDS			(2600)	/* Make sure ZigBee is (NOT) using the grant (handles ZigBee RX of long frames)*/
#define PTA_HW_TIMER_ZIGBEE_LAST_CHANCE_200_MICROSECONDS	 (100)	/* Last change for ZigBee before PTA indicates that BT has an error */


#define PTA_HW_TIMER_POLLING_400_MICROSECONDS		 		 (400)	/* Polling on radio transmision */

#define GPIO_SHOULD_BE_HIGH			(0)
#define GPIO_SHOULD_BE_LOW			(1)

#define	PTA_PHY_TX_REGISTER_DISABLE_TX				(0)		/* When "FEATURE_ENABLE" (BIT[15] = 1): WIFI TX is disabled exept for ACK\BLOCK ACK\BEACON which have specific TCR bit set. */
#define	PTA_PHY_TX_REGISTER_DISABLE_AUTOREPLY		(1)		/* When "FEATURE_ENABLE" and "DISABLE_TX". Autoreply packets are not sent 													*/
#define	PTA_PHY_TX_REGISTER_ENABLE_GRACEFUL_TX_STOP	(2)		/* When "FEATURE_ENABLE" and "DISABLE_TX". [1] Stop TX after current frame transmission is completed. [0] Immediate TX stop */
#define PTA_PHY_TX_REGISTER_FEATURE_ENABLE			(15)	/* Feature is enabled. WIFI TX is as usual 																					*/

#ifdef PTA_TEST
#define  test_bUseRepetitionMoe				(0)
#define  test_repetitionSuspend				(1)
#define  test_repetitionResume				(2)
#define  test_zigbee_radioHoldOffValue		(3)
#define  test_zigbee_starvationTimeValue	(4)
#define  test_bt_radioHoldOffValue			(5)
#define  test_bt_starvationTimeValue		(6)
#endif

/*---------------------------------------------------------------------------------
/						Tables						
/----------------------------------------------------------------------------------*/

/* The following table describes the values used by PTA according to QoS valuse that Driver sent in the command: UMI_SET_2_4_G_COEX -> CoExActive


													Multi-Radio coordinator Other radio QoS request level	
				-----------------------------------------------------------------------------------------------------------------------------------------
															Other radios requests are ignored	BLE\BT	Only	ZigBee Only 	Both
															---------------------------------	----------- 	----------- 	-------------------------

															0									1				2				3		4		5		6
					Wifi radio parameters
					---------------------
Mutual params		gratuitousCtsToSelfInterval [mSec]		N/A 								N/A 			N/A 			N/A 	N/A		100 	100
					gratuitousCtsToSelfPeriod [mSec]		N/A 								N/A 			N/A 			N/A 	N/A		5		5
					bUseRepetitionMode [wave500B only]		N/A 								N/A 			N/A 			N/A 	yes 	yes 	yes
					repetitionSuspend [microSec]			N/A 								N/A 			N/A 			N/A 	20 		500 	600
					repetitionResume [microSec] 			N/A 								N/A 			N/A 			N/A 	100 	1000 	3000
					bAllowBeacon							N/A 								N/A 			N/A 			yes 	yes 	yes 	yes
					bAllowAutoRply							N/A 								N/A 			N/A 			yes 	yes 	yes 	yes
BT\BLE				bRadioDriverIsUp						0									1				0				1		1		1		1
					radioHoldOffValue						N/A 								21				N/A 			21		5		3		1
					starvationTimeValue [mSec]				N/A 								10				N/A 			10		20		30		40
					penaltyTime [Sec]						N/A 								10				N/A 			10		9		3		1
ZigBee				bRadioDriverIsUp						0									0				1				1		1		1		1
					radioHoldOffValue						N/A 								N/A 			21				21		5		3		1
					starvationTimeValue [mSec]				N/A 								N/A 			10				10		20		30		40
					penaltyTime  [mSec] 					N/A 								N/A 			10				10		9		3		1
*/
	
	
const PtaParams_t ptaQos[CO_EX_QOS_NUM_OF_TYPES]=
{
/*				Common																	ZigBee							BT
				--------------------------------------------------------------------	------------------------------- ------------------------------	*/	
/*				GCTS_I		GCTS_P	RepetitionMode	Suspend Resume	Beacon	AutoReply	Up	HoldOff starvation	penalty Up	HoldOff starvation penalty	*/
	{	/*0*/	0,			0,		0,				0,		0,		0,		0,			0,	0,		0,			0,		0,	0,		0,			0	},
	{	/*1*/	0,			0,		0,				0,		0,		0,		0,			0,	0, 		0, 			0, 		1,	1,		10,			10	},
	{	/*2*/	0,			0,		0,				0,		0,		0,		0,			1,	1,		10,			10,		0,	0, 		0, 			0	},
	{	/*3*/	0,			0,		0,				0,		0,		1,		1,			1,	1,		10,			10, 	1,	1,		10,			10	},
	{	/*4*/	1000,		5,		1,				20,		100,	1,		1,			1,	5,		20, 		9,		1,	5,		20, 		9	},
	{	/*5*/	100,		5,		1,				500,	1000,	1,		1,			1,	3,		30, 		3,		1,	3,		30, 		3	},
	{	/*6*/	100,		5,		1,				600,	3000,	1,		1,			1,	1,		40, 		1,		1,	1,		40, 		1	},
};
 
/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef void (*ptaSmHandler)(void *);

typedef struct PtaSm
{
	ptaSmHandler	handler[PTA_EVENT_NUM_OF_EVENTS];
}PtaSm_t;

typedef struct PtaGpioErrorParams
{
	uint32	gpioName;
	uint32	gpioError;
} PtaGpioErrorParams_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void PtaDbInit(void);
static void PtaSmInit(void);
static void ptaInit(void);
static void PtaPendingDriverReq(void);
static void PtaStateChange(PtaStates_e currentState, PtaStates_e nextState);
static void PtaRunPtaSm(PtaEvents_e event, void *eventInfo);
static void PtaCofigure(K_MSG *psMsg);
static void PtaSendCfmToDriver(K_MSG *psMsg);
#ifdef PTA_TEST
static void PtaSendMessage(K_MSG_TYPE ptaMsg, PtaGpioErrorParams_t *params);
#endif

/*************************************************************************************************************
PTA handlers
**************************************************************************************************************/
#ifdef PTA_TEST
void PtaTestCommand(void *eventInfo);
#endif
void PtaHandlePriorityGpio(void);
void PtaStartBtAllow(void *eventInfo);
void PtaStartZigBeeAllow(void *eventInfo);
void PtaStartBtDeny(void *eventInfo);
void PtaStartZigBeeDeny(void *eventInfo);
void PtaStarvationTimerExpired(void *eventInfo);
#ifdef PTA_TEST
void PtaStateTestGpio0(void *eventInfo);
void PtaStateTestStarvationTimerExpired(void *eventInfo);
#endif
void PtaInitPostTransitionCleanUp(void *eventInfo);
void PtaMoveToIdle(void *eventInfo);
void PtaZigBeeRxShort(void *eventInfo);
void PtaZigBeeRxMid(void *eventInfo);
void PtaZigBeeRxLong(void *eventInfo);
void PtaZigBeePoll(void *eventInfo);
void PtaZigBeeDenyError(void *eventInfo);
void PtaGrantNotUsed(void *eventInfo);
void PtaStartBtTx(void *eventInfo);
void PtaStartZigBeeTx(void *eventInfo);
void PtaTxOngoing(void *eventInfo);
#ifdef PTA_TEST
void PtaStateTestHwTimerOk(void *eventInfo);
#endif

/*************************************************************************************************************
ISRs
**************************************************************************************************************/
/* ISRs PROTOTYPES are defined at API file */

/*************************************************************************************************************
MAC and PHY HW handlers
**************************************************************************************************************/
static void PtaCcaSet(uint8 cca);
static void PtaPhyTxRegisterSet(uint32 phyTxReg);
/*************************************************************************************************************
HW timer
**************************************************************************************************************/
static PtaEvents_e PtaHwTimerDetermineEvent(void);
// already cleared at isr_Timers_TimerA void PtaHwTimerClearIrq(void);
static void PtaHwTimerStart(uint32 timeout);
/* static void PtaHwTimerStop(void); */
static void PtaHwTimerEnableInterrupt(void);
/* static void PtaHwTimerDisableInterrupt(void); */

/*************************************************************************************************************
Messages
**************************************************************************************************************/
static void PtaHandle24GCoExReq(K_MSG *psMsg);
static void PtaStarvationTimerExpiry(K_MSG *psMsg);
#ifdef PTA_TEST
static void PtaTestGpioValueIsWrong(K_MSG *psMsg);
static void PtaTestHwTimerIsOK(K_MSG *psMsg);
#endif
/*************************************************************************************************************
GPIOs
**************************************************************************************************************/
static PtaEvents_e PtaGpioDetermineEvent(void);
/*************************************************************************************************************
Starvation timer
**************************************************************************************************************/
static void PtaStarvationTimerStart(uint8 timeToGo);
static void PtaStarvationTimerStop(void);
#ifdef PTA_TEST
static void PtaStarvationTimerRestart(uint8 timeToGo);
#endif
/*************************************************************************************************************
PTA Errors
**************************************************************************************************************/
void PtaIgnoreEvent(void *eventInfo);
void PtaGpio0NotInIdle(void *eventInfo);
void PtaIllegalEvent(void *eventInfo);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
PtaSm_t							ptaSm[PTA_STATE_NUM_OF_STATES];	/* HW TIMER SM */ 
PtaDb_t							ptaDb;							/* PTA global DB */
PtaRepetitionParams_t			ptaRepDb;


static const FunctionEntry_t afpTaskTable[TASK_PTA_END - TASK_PTA_START] =
{
	{PtaHandle24GCoExReq,		DOUBLE_CHECK_MSG_TYPE(PTA_SET_2_4_G_COEX)},
	{PtaStarvationTimerExpiry,	DOUBLE_CHECK_MSG_TYPE(PTA_STARVATION_TIMER)},
	{PtaRepConfig,		 		DOUBLE_CHECK_MSG_TYPE(PTA_REP_CONFIG)},	
	{PtaRepTimerExpiry, 		DOUBLE_CHECK_MSG_TYPE(PTA_REP_TIMER_EXPIRED)},
	{PtaRepProcessStart, 		DOUBLE_CHECK_MSG_TYPE(PTA_REP_PROCESS_START)},
	{PtaRepProcessEnd,			DOUBLE_CHECK_MSG_TYPE(PTA_REP_PROCESS_END)},
#ifdef PTA_TEST
	{PtaTestGpioValueIsWrong,	DOUBLE_CHECK_MSG_TYPE(PTA_TEST_GPIO_VALUE_IS_WRONG)},
	{PtaTestHwTimerIsOK,		DOUBLE_CHECK_MSG_TYPE(PTA_TEST_HW_TIMER_IS_OK)},		
#endif		
};

/*---------------------------------------------------------------------------------
/						Debug code									
/----------------------------------------------------------------------------------*/

#if defined (PTA_MONITOR) || defined (PTA_TRACE) || defined (PTA_STATS)

#define AAA_PTA_MONITOR_ARRAY_SIZE (16)

typedef struct aaa_ptaHwTimer
{
	uint32 tsf;
	PtaStates_e state;
	PtaEvents_e event;
}	aaa_ptaHwTimer_t;

aaa_ptaHwTimer_t 			AAA_PTA_HW_TIMER[AAA_PTA_MONITOR_ARRAY_SIZE] = {0};

uint32 						AAA_PTA_HW_TIMER_INDEX = 0;

#endif
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
#ifdef PTA_TEST
/**********************************************************************************
ptaTest



Description:
------------
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PtaTest(void)
{
	PtaStates_e	currentState = ptaDb.state;
	void			*eventInfo = NULL;

	/* Hw Timer State should be INIT */
	ASSERT(ptaDb.state == PTA_STATE_INIT);

	/* 	Send event to PTA: TEST COMMAND */
	ptaSm[currentState].handler[PTA_EVENT_TEST_COMMAND](eventInfo);

#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaTest] Test command received");
#endif	
}
/**********************************************************************************
PtaTestDriverReq




Description: send Driver command
------------
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PtaTestDriverReq(uint8 data3)
{
	/* Create message */
	K_MSG *pMsg = NULL;
	UMI_SET_2_4_G_COEX	*ptaDriverCommand = PNULL;

	pMsg = OSAL_GET_MESSAGE(sizeof(UMI_SET_2_4_G_COEX));

	ptaDriverCommand = (UMI_SET_2_4_G_COEX *)pK_MSG_DATA(pMsg);

	memset(ptaDriverCommand, 0, sizeof(UMI_SET_2_4_G_COEX));

	/* Fill struct */
	ptaDriverCommand->bCoExActive = TRUE;
	ptaDriverCommand->coExQos = data3;

#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaTestDriverReq]");
#endif
	/* Send message to PTA task */
	// Dont send a message to Driver since there was no request: OSAL_SEND_MESSAGE(PTA_SET_2_4_G_COEX, TASK_PTA, pMsg, VAP_ID_DO_NOT_CARE);
	PtaCofigure(pMsg);

}


void PtaTestEmulateGpio0Int(/* BT_ACTIVE*/uint8 btActiveVal, /* TX_DEMAND */uint8 txDemandVal)
{
	
#ifdef PTA_TRACE	
	ILOG0_DD("[PTA][PtaTestEmulateGpio0Int] btActiveVal = %d txDemandVal = %d",btActiveVal,txDemandVal);
#endif

	/* Set GPIOs directly (not via BT wrapper for testing) */
	PtaGpioWrite(PTA_GPIO_NAME_BT_ACTIVE, btActiveVal);
	PtaGpioWrite(PTA_GPIO_NAME_TX_DEMAND, txDemandVal);

	/* Emultate GPIO0 - not required in wave600 since writing to TX_DEMAND triggers an interrup*/
	//Isr_PtaGpio0();
}


void PtaTestReset(void)
{

#ifdef PTA_TRACE	
	ILOG0_V("[PTA][PtaTestReset]");
#endif
	
	/* Reset the task as if we just initialized from power up */
	/* Init DB */
	PtaDbInit();	

	/* Init state of HW Timer state machine */
	ptaDb.state = PTA_STATE_INIT;

	/*Enable HW timer interrupt */
	PtaHwTimerEnableInterrupt();

	/* Reset the GPIOs to an init value of '0': */
	/* Write 0 to GPIO2 = BT_ACTIVE (WLAN_GPIO #1 wired to GRX_GPIO #11) */
	PtaGpioWrite(PTA_GPIO_NAME_GRANT, PTA_GPIO_VAL_LOW);
	/* Write 0 to GPIO1 = GRANT (WLAN_GPIO #2 wired to GRX GPIO #9) */
	PtaGpioWrite(PTA_GPIO_NAME_BT_ACTIVE, PTA_GPIO_VAL_LOW);
	/* Write 0 to GPIO0 = TX_DEMAND (WLAN_GPIO #0 wired to GRX GPIO W_GPIO_0) */
	PtaGpioWrite(PTA_GPIO_NAME_TX_DEMAND, PTA_GPIO_VAL_LOW);
}

void PtaTestRep(uint8 data2, uint8 data3)
{
	PtaParams_t *ptaParams = &ptaDb.ptaParams;
	
	/* Create message */
	K_MSG *pMsg = NULL;
	PtaRepConfig_t *ptaRepConfig = PNULL;

	pMsg = OSAL_GET_MESSAGE(sizeof(PtaRepConfig_t));
	ptaRepConfig = (PtaRepConfig_t *)pK_MSG_DATA(pMsg);

	memset(ptaRepConfig, 0, sizeof(PtaRepConfig_t));

	//ASSERT(0);

	/* Fill struct */
	ptaRepConfig->enable = (data2 == 0)? FALSE:TRUE;
	ptaRepConfig->activePeriod = data2;
	ptaRepConfig->inactivePeriod = data3;

	/* Store the parameters in PTA database so that it can be modified by PtaTestModifyQosParam */
	ptaParams->bUseRepetitionMode = ptaRepConfig->enable;
	ptaParams->repetitionResume = ptaRepConfig->activePeriod;
	ptaParams->repetitionSuspend = ptaRepConfig->inactivePeriod;

#ifdef PTA_TRACE
	ILOG0_V("[PTA][REP][PtaTestRep]");
#endif
	/* Send message to PTA task */
	OSAL_SEND_MESSAGE(PTA_REP_CONFIG, TASK_PTA, pMsg, VAP_ID_DO_NOT_CARE);
}

void PtaTestModifyQosParam(uint8 parameter, uint8 value)
{
	PtaParams_t *ptaParams = &ptaDb.ptaParams;
	/**/
	switch(parameter)
	{
		case test_bUseRepetitionMoe:
		{
			#ifdef PTA_TRACE
			ILOG0_D("[PTA][PtaTestModifyQosParam] bUseRepetitionMoe value = %d", value);
			#endif

			/* Send message to repetiotion (keep suspend\resume as is)  */
			PtaRepAssignNewParams((bool)value, ptaParams->repetitionResume, ptaParams->repetitionSuspend);
			break;
		}
		case test_repetitionSuspend:
		{
			/* Modify repetition suspend value (multiply by 10 since we have only uint8...) */
			ptaParams->repetitionSuspend = (10*value);

			#ifdef PTA_TRACE
			ILOG0_DDDD("[PTA][PtaTestModifyQosParam] Suspend new value = %d will use = %d old value = %d resume = %d", value, ptaParams->repetitionSuspend, ptaParams->repetitionSuspend, ptaParams->repetitionResume);
			#endif

			/* Send message to repetiotion (keep bUseRepetitionMode\repetitionResume as is)  */
			PtaRepAssignNewParams(ptaParams->bUseRepetitionMode, ptaParams->repetitionResume, ptaParams->repetitionSuspend);
			
			break;
		}			
		case test_repetitionResume:	
		{
			/* Modify repetition resume value (multiply by 100 since we have only uint8...) */
			ptaParams->repetitionResume = (100*value);
			
			#ifdef PTA_TRACE
			ILOG0_DDDD("[PTA][PtaTestModifyQosParam] Resume new value = %d will use = %d old value = %d suspend = %d", value, ptaParams->repetitionResume, ptaParams->repetitionResume, ptaParams->repetitionSuspend);
			#endif		
			
			/* Send message to repetiotion (keep bUseRepetitionMode\repetitionResume as is)  */
			PtaRepAssignNewParams(ptaParams->bUseRepetitionMode, ptaParams->repetitionResume, ptaParams->repetitionSuspend);			
			break;
		}
		case test_zigbee_radioHoldOffValue:	
		{
			#ifdef PTA_TRACE
			ILOG0_DD("[PTA][PtaTestModifyQosParam] ZigBee radioHoldOffValue new value = %d old value = %d", value, ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].radioHoldOffValue);
			#endif

			/* Modify ZigBee's radio hold off value */
			ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].radioHoldOffValue = value;
			break;
		}
		case test_zigbee_starvationTimeValue:
		{
			#ifdef PTA_TRACE
			ILOG0_DD("[PTA][PtaTestModifyQosParam] ZigBee starvationTimeValue new value = %d old value = %d", value, ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].starvationTimeValue);
			#endif

			/* Modify ZigBee's starvation time value */
			ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].starvationTimeValue = value;
			ptaDb.ZigBeeCounter = 0;
			break;
		}
		case test_bt_radioHoldOffValue:	
		{
			#ifdef PTA_TRACE
			ILOG0_DD("[PTA][PtaTestModifyQosParam] BT radioHoldOffValue new value = %d old value = %d", value, ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_BT].radioHoldOffValue);
			#endif

			/* Modify BT's radio hold off value */
			ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_BT].radioHoldOffValue = value;
			ptaDb.btCounter = 0;
			break;
		}
		case test_bt_starvationTimeValue:
		{
			#ifdef PTA_TRACE
			ILOG0_DD("[PTA][PtaTestModifyQosParam] BT starvationTimeValue new value = %d old value = %d", value, ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_BT].starvationTimeValue);
			#endif

			/* Modify BT's starvation time value */
			ptaParams->uniqueRadioParams[PTA_RADIO_TYPE_BT].starvationTimeValue = value;
			break;
		}

		default:
		{
			ILOG0_V("[PTA][PtaTestModifyQosParam] you are trying to change a PTA parameter but with wrong "parameter". allwoed values are 0 - 6");
		}		
	}
}

void PtaTestTracePtaParams(void)
{
	SLOG0(0, 0, PtaDb_t, &ptaDb);
	SLOG0(0, 0, PtaRepetitionParams_t, &ptaRepDb);
}
#endif // PTA_TEST

/**********************************************************************************
PtaDbInit


Description:
------------
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/

static void PtaDbInit(void)
{
	memset(&ptaDb, 0, sizeof(PtaDb_t));
}

/**********************************************************************************
PtaSmInit

Description:
------------
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/

void PtaSmInit(void)
{
	/* Initialize PTA state machine functions */

	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 			= PtaIgnoreEvent;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 				= PtaInitPostTransitionCleanUp;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 					= PtaInitPostTransitionCleanUp;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaInitPostTransitionCleanUp;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 				= PtaInitPostTransitionCleanUp;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 					= PtaInitPostTransitionCleanUp;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 					= PtaInitPostTransitionCleanUp;
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 	 									= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_INIT].handler[PTA_EVENT_TEST_COMMAND] 													= PtaTestCommand;
#endif

	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW]					= PtaIgnoreEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 				= PtaIgnoreEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 			= PtaIgnoreEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 				= PtaStartBtAllow;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 					= PtaStartZigBeeAllow;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaStartBtDeny;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 				= PtaStartZigBeeDeny;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 					= PtaIllegalEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 					= PtaIllegalEvent;
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 	 									= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_IDLE].handler[PTA_EVENT_TEST_COMMAND] 													= PtaIllegalEvent;
#endif

	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 			= PtaMoveToIdle;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 			= PtaMoveToIdle;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 				= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 				= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 					= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 	 								= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_BT_DENY].handler[PTA_EVENT_TEST_COMMAND] 												= PtaIllegalEvent;	
#endif
	
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 			= PtaGrantNotUsed;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 			= PtaGrantNotUsed;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 			= PtaStartBtTx;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 				= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED]			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 									= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_BT_ALLOW].handler[PTA_EVENT_TEST_COMMAND] 												= PtaIllegalEvent;	
#endif
	
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 				= PtaMoveToIdle;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 			= PtaMoveToIdle;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 			= PtaTxOngoing;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 				= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 				= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 					= PtaIllegalEvent;
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED]  									= PtaStarvationTimerExpired;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_BT_POLL].handler[PTA_EVENT_TEST_COMMAND] 												= PtaIllegalEvent;
#endif

	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 			= PtaMoveToIdle;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 		= PtaZigBeeRxShort;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 		= PtaZigBeeRxShort;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 	 							= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_DENY].handler[PTA_EVENT_TEST_COMMAND] 											= PtaIllegalEvent;	
#endif

	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 		= PtaGrantNotUsed;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 	= PtaStartZigBeeTx;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED]		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 								= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_ALLOW].handler[PTA_EVENT_TEST_COMMAND] 											= PtaIllegalEvent;	
#endif
	
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 		= PtaMoveToIdle;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 		= PtaZigBeeRxShort;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED]  								= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_TX].handler[PTA_EVENT_TEST_COMMAND] 												= PtaIllegalEvent;
#endif
	
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 		= PtaMoveToIdle;		/* ZigBee did not understand "DENY" unitl this point. Can move to IDLE now. */
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 	= PtaZigBeeRxMid;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 	= PtaZigBeeRxMid;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 	= PtaMoveToIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 	= PtaZigBeeRxMid;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED]  							= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_RX_SHORT].handler[PTA_EVENT_TEST_COMMAND] 										= PtaIllegalEvent;
#endif
	
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 		= PtaMoveToIdle;		/* ZigBee did not understand "DENY" unitl this point. Can move to IDLE now. */
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 		= PtaZigBeeRxLong;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 		= PtaZigBeeRxLong;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 	= PtaMoveToIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 	= PtaZigBeeRxLong;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED]  							= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_RX_MID].handler[PTA_EVENT_TEST_COMMAND] 											= PtaIllegalEvent;
#endif
	
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 		= PtaMoveToIdle;		/* ZigBee did not understand "DENY" unitl this point. Can move to IDLE now. */
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 		= PtaZigBeeDenyError;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 	= PtaZigBeeDenyError;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 	= PtaMoveToIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 	= PtaZigBeePoll;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 	= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED]  							= PtaIgnoreEvent;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_RX_LONG].handler[PTA_EVENT_TEST_COMMAND] 										= PtaIllegalEvent;
#endif
	
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 			= PtaMoveToIdle;		/* ZigBee did not understand "DENY" unitl this point. Can move to IDLE now. */
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 		= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 		= PtaMoveToIdle;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 		= PtaTxOngoing;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 			= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 		= PtaGpio0NotInIdle;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 			= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 				= PtaIllegalEvent;
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED]  								= PtaStarvationTimerExpired;
#ifdef PTA_TEST
	ptaSm[PTA_STATE_ZIGBEE_POLL].handler[PTA_EVENT_TEST_COMMAND] 											= PtaIllegalEvent;
#endif
	
#ifdef PTA_TEST
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW] 				= PtaStateTestHwTimerOk;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH] 				= PtaStateTestHwTimerOk;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH] 				= PtaStateTestHwTimerOk;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW] 				= PtaStateTestHwTimerOk;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH] 			= PtaStateTestHwTimerOk;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED] 				= PtaStateTestGpio0;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED] 					= PtaStateTestGpio0;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED] 			= PtaStateTestGpio0;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED] 				= PtaStateTestGpio0;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF] 					= PtaIllegalEvent;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_BT_OFF] 					= PtaIllegalEvent;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_STARVATION_TIMER_EXPIRED] 										= PtaStateTestStarvationTimerExpired;
	ptaSm[PTA_STATE_TEST].handler[PTA_EVENT_TEST_COMMAND] 													= PtaIllegalEvent;	
#endif	
}
	


static void ptaInit(void)
{

#ifdef PTA_TRACE
	ILOG0_V("[PTA][ptaInit]");
#endif	
	/* Stop starvation timer as we might get here as a result of an error */
	PtaStarvationTimerStop();

	/* Set CCA to NORMAL */
	PtaCcaSet(PTA_CCA_QUIET);
	/* Set the PHY to "TX ENABLE" */
	PtaPhyTxRegisterSet(1 << PTA_PHY_TX_REGISTER_FEATURE_ENABLE);

	/* If Driver configuration is pending start using it */
	PtaPendingDriverReq();
	/* If the feature is Disabled: exit */
	if (ptaDb.activeParams.bCoExActive == TRUE)
	{
		TX_INTERRUPT_SAVE_AREA;
		
		/* Set state = IDLE */
		PtaStateChange(ptaDb.state, PTA_STATE_IDLE);
		
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		/* Mask GPIO0 interrupt */
		PtaGpioMaskGpio0();
		/* Clear GPIO0 latch */
		ptaGpioLatchClearGpio0();
		/* Un-mask GPIO0 interrupt */
		ptaGpioUnmsakGpio0();
		/* Init GPIOs state */
		PtaGpioInitGpioState();
		
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
	else
	{
#ifdef PTA_TRACE
		ILOG0_V("[PTA][ptaInit] Feature is disabled");
#endif
	}
}

static void PtaPendingDriverReq(void)
{
	PtaDb_t *ptaDbLocal = &ptaDb;
	TX_INTERRUPT_SAVE_AREA;    
	
	/* Start using new params (disable interrupts to make sure PTA is working with new params) */
	if (ptaDbLocal->pendingDriverReq == TRUE)
	{
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		
		/* Set pending as false */
		ptaDbLocal->pendingDriverReq = FALSE;

		/* Copy information */
		//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
		MEMCPY(&ptaDbLocal->activeParams , &ptaDbLocal->driverReq , sizeof(UMI_SET_2_4_G_COEX));

		/* Extract parameters according to QoS */
		//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
		MEMCPY(&ptaDbLocal->ptaParams, &ptaQos[ptaDbLocal->activeParams.coExQos], sizeof(PtaParams_t));

	 	OSAL_ENABLE_INTERRUPTS(interrupt_save);
		
		/* Send message to repetiotion entity to start using new parameters */
		PtaRepAssignNewParams(ptaDbLocal->ptaParams.bUseRepetitionMode, ptaDbLocal->ptaParams.repetitionResume, ptaDbLocal->ptaParams.repetitionSuspend);		
	}

#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaPendingDriverReq]");
#endif	
}

static void PtaRunPtaSm(PtaEvents_e event, void *eventInfo)
{
	PtaStates_e currentState = ptaDb.state;

	ASSERT(event < PTA_EVENT_NUM_OF_EVENTS);

#ifdef PTA_MONITOR
	AAA_PTA_HW_TIMER[AAA_PTA_HW_TIMER_INDEX].tsf = GET_TSF_TIMER_LOW();
	AAA_PTA_HW_TIMER[AAA_PTA_HW_TIMER_INDEX].state = currentState;
	AAA_PTA_HW_TIMER[AAA_PTA_HW_TIMER_INDEX].event = event;
	AAA_PTA_HW_TIMER_INDEX = (AAA_PTA_HW_TIMER_INDEX + 1) % AAA_PTA_MONITOR_ARRAY_SIZE;
#endif

	ptaSm[currentState].handler[event](eventInfo);
}

/**********************************************************************************
PtaStateChange

Description:
------------
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
static void PtaStateChange(PtaStates_e currentState, PtaStates_e nextState)
{
	ASSERT(nextState < PTA_STATE_NUM_OF_STATES);
	UNUSED_PARAM(currentState);
#ifdef PTA_TRACE	
	ILOG0_DD("[PTA][PtaStateChange] currentState = %d nextState = %d",currentState, nextState);
#endif
	ptaDb.state = nextState;
}



/**********************************************************************************
PacketTrafficArbitratorTask_Init

Description:
------------
Initialization of the PTA at power up
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/

void PacketTrafficArbitratorTask_Init(void)
{
	/* Init DB */
	PtaDbInit();	

	/* Init PTA HT state machine */
	PtaSmInit();
	
	/* Init state of HW Timer state machine */
	ptaDb.state = PTA_STATE_INIT;

	/*Enable HW timer interrupt */
	PtaHwTimerEnableInterrupt();

	/* Set the polarity of the TX_DEMAND Interrupt (applicable only in WAVE600) */
	PtaInterruptManagerConfigureMacGeneraExternalIrqInvert();
	
#ifdef PTA_STATS
	memset(&PtaStatistics, 0, sizeof(PtaStatistics_t));
#endif

	/* Init repetition mode */
	PtaRepInit();
}

/**********************************************************************************
PTA_Task


Description:
------------
PTA task entry
	 
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PTA_Task(K_MSG *psMsg)
{
    /* Use common task switching and Table */
	vTaskDispatcher(psMsg, afpTaskTable, TASK_PTA_START, TASK_PTA_END);
}


#ifdef PTA_TEST
static void PtaSendMessage(K_MSG_TYPE ptaMsg, PtaGpioErrorParams_t *params)
{	
	K_MSG *pMsg = NULL;

	PtaGpioErrorParams_t *ptaGpioErrorParams = NULL;
	pMsg = OSAL_GET_MESSAGE(sizeof(PtaGpioErrorParams_t));
	ptaGpioErrorParams = (PtaGpioErrorParams_t*)pK_MSG_DATA(pMsg);

	ptaGpioErrorParams->gpioName = params->gpioName;
	ptaGpioErrorParams->gpioError = params->gpioError;

	OSAL_SEND_MESSAGE(ptaMsg, TASK_PTA, pMsg, VAP_ID_DO_NOT_CARE);
}

/*************************************************************************************************************
PTA handlers
**************************************************************************************************************/
void PtaTestCommand(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
	/* Change state to TEST */
	PtaStateChange(ptaDb.state, PTA_STATE_TEST);

	/* start starvation timer 100 Msec) */
	PtaStarvationTimerStart(PTA_STARVATION_TIMER_100_MILLISECONDS);

	/* Set GPIO0 as OUT */
	PtaGpioDirection(PTA_GPIO_NAME_TX_DEMAND, PTA_GPIO_DIRECTION_OUT); 
	
	/* Set GPIO1 as OUT */
	PtaGpioDirection(PTA_GPIO_NAME_BT_ACTIVE, PTA_GPIO_DIRECTION_OUT); 
	
	/* Set GPIO2 as OUT */
	PtaGpioDirection(PTA_GPIO_NAME_GRANT, PTA_GPIO_DIRECTION_OUT); 
	
	/* Read GPIO2 - should be 0 at power up, else assert */
	ASSERT(PtaGpioRead(PTA_GPIO_NAME_GRANT) == PTA_GPIO_VAL_LOW);
	/* Read GPIO1 - should be 0 at power up, else assert */
	ASSERT(PtaGpioRead(PTA_GPIO_NAME_BT_ACTIVE) == PTA_GPIO_VAL_LOW);
	/* Read GPIO0 - should be 0 at power up, else assert */
	ASSERT(PtaGpioRead(PTA_GPIO_NAME_TX_DEMAND) == PTA_GPIO_VAL_LOW);

	/* Write 1 to GPIO2 = BT_ACTIVE (WLAN_GPIO #1 wired to GRX_GPIO #11) */
	PtaGpioWrite(PTA_GPIO_NAME_GRANT, PTA_GPIO_VAL_HIGH);
	/* Write 1 to GPIO1 = GRANT (WLAN_GPIO #2 wired to GRX GPIO #9) */
	PtaGpioWrite(PTA_GPIO_NAME_BT_ACTIVE, PTA_GPIO_VAL_HIGH);
	/* Write 1 to GPIO0 = TX_DEMAND (WLAN_GPIO #0 wired to GRX GPIO W_GPIO_0) */
	PtaGpioWrite(PTA_GPIO_NAME_TX_DEMAND, PTA_GPIO_VAL_HIGH);

	/* Start HW timer 5000 micro seconds - this was added since the GPIO0 writing did not trigger interrupt */
	PtaHwTimerStart(PTA_HW_TIMER_VAL_5000_MICROSECONDS);

#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaStateInitTestCommand]");
#endif

}
#endif

    
/***********************************************************************
* PtaHandlePriorityGpio
* 
* Description:
* ------------
* This function tells phy to:
*								1. Stop TX.
* 								2. TX or not TX of autoreply and beacon frames 
* 								3. Allow graceful TX stop or cut the current frame in the middle of transmission
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PtaHandlePriorityGpio(void)
{
	/* Set PHY TX register: Start with enabling the feature and with no tx allowed */
	uint32 ptaPhyReq = ((1 << PTA_PHY_TX_REGISTER_FEATURE_ENABLE) | (1 << PTA_PHY_TX_REGISTER_DISABLE_TX));

	/* Read the PRIORITY GPIO and add graceful or autoreply bits to phy register */
	if (PtaGpioRead(PTA_GPIO_NAME_PRIORITY) == PTA_GPIO_NAME_PRIORITY)
	{
		/* Priority GPIO level indicates that traffic shall be suspended after current TX frame */
		ptaPhyReq |= (1 << PTA_PHY_TX_REGISTER_ENABLE_GRACEFUL_TX_STOP);
	}
	else
	{
		/* Priority GPIO level indicates that auto reply packets shall not be transmitted */
		ptaPhyReq |= (1 << PTA_PHY_TX_REGISTER_DISABLE_AUTOREPLY);
	}

#ifdef PTA_TRACE
	ILOG0_D("[PTA] PtaHandlePriorityGpio ptaPhyReq = 0x%x", ptaPhyReq);
#endif
	
	PtaPhyTxRegisterSet(ptaPhyReq);
}


void PtaStartBtAllow(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	/* BT is requesting */
	ptaDb.radioType = PTA_RADIO_TYPE_BT;

	/* Start starvation timer with a value defined by Driver */
	PtaStarvationTimerStart(ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_BT].starvationTimeValue);

	/* Set CCA to "Noisy" */
	PtaCcaSet(PTA_CCA_NOISY);

	/* Read Priority GPIO and act accordingly */
	PtaHandlePriorityGpio();
	
	/* Set GRANT = BT_ALLOW */
	PtaGpioBtAllow();

#ifdef PTA_STATS
	/* Reset the "on going" TX counter */
	PtaStatistics.btTxOngoing = 0;
#endif

	/* Clear the btCounter */
	ptaDb.btCounter = 0;
	
	/* Change state */	
	PtaStateChange(ptaDb.state, PTA_STATE_BT_ALLOW);

	/* Start HW timer for the ALLOW period (on expiry make sure BT is using the grant) */
	PtaHwTimerStart(PTA_HW_TIMER_BT_ALLOW_200_MICROSECONDS);	
}

void PtaStartZigBeeAllow(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
	/* ZigBee is requesting */
	ptaDb.radioType = PTA_RADIO_TYPE_ZIGBEE;

	/* Start starvation timer with a value defined by Driver */
	PtaStarvationTimerStart(ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].starvationTimeValue);

	/* Set CCA to "Noisy" */
	PtaCcaSet(PTA_CCA_NOISY);
	
	/* Read Priority GPIO and act accordingly */
	PtaHandlePriorityGpio();
	
	/* Set GRANT = ZIGBEE_ALLOW */
	PtaGpioZigBeeAllow();

#ifdef PTA_STATS
	/* Reset the "on going" TX counter */
	PtaStatistics.zigbeeTxOngoing = 0;
#endif

	/* Clear the ZigBeeCounter */
	ptaDb.ZigBeeCounter = 0;

	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_ALLOW);

	/* Start HW timer for the ALLOW period (on expiry make sure ZigBee is using the grant) */
	PtaHwTimerStart(PTA_HW_TIMER_ZIGBEE_ALLOW_200_MICROSECONDS);
}

void PtaStartBtDeny(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
	/* BT is requesting */
	ptaDb.radioType = PTA_RADIO_TYPE_BT;	
	/* Set GRANT = BT_DENY */
	PtaGpioBtDeny();
	/* Start HW timer for the DENY period (on expiry make sure BT is not using the grant) */
	PtaHwTimerStart(PTA_HW_TIMER_BT_DENY);
	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_BT_DENY);
}

void PtaStartZigBeeDeny(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
	/* ZigBee is requesting */
	ptaDb.radioType = PTA_RADIO_TYPE_ZIGBEE;	
	/* Set GRANT = ZIGBEE_DENY */
	PtaGpioZigBeeDeny();
	/* Start HW timer for the ALLOW period (on expiry make sure ZIGBEE is not using the grant) */
	PtaHwTimerStart(PTA_HW_TIMER_ZIGBEE_DENY_200_MICROSECONDS);
	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_DENY);	
}


void PtaStarvationTimerExpired(void *eventInfo)
{	
	//ASSERT(0); <= some races can occur
	UNUSED_PARAM(eventInfo);

	/* Read GRANT, BT_ACTIVE and TX_DEMAND values */
	bool isBtActiveHigh = PtaGpioReadBtActive();
	bool isTxDemandHigh = (PtaGpioRead(PTA_GPIO_NAME_TX_DEMAND) == PTA_GPIO_NAME_TX_DEMAND);
	bool isGrantHigh	= PtaGpioReadGrant();

	/* IF one of the conditions is met we have starvation:
	1. BT in TX and BT_ACTIVE is high (BT means isGrantHigh is LOW) 
	2. ZigBee in TX and TX_DEMAND is high (ZigBee means isGrantHigh is HIGH)*/
	
	if (((isGrantHigh == TRUE) && (isBtActiveHigh == TRUE)) || 
		((isGrantHigh == FALSE) && (isTxDemandHigh == TRUE)))
	{
#ifdef PTA_TRACE	
	ILOG0_DDDD("[PTA][PtaStateTxStarvationTimerExpired] State = %d isGrantHigh = %d isBtActiveHigh = %d isTxDemandHigh = %d", ptaDb.state, isGrantHigh, isBtActiveHigh, isTxDemandHigh);
#endif		
		/* Starvation has occured */
#ifdef PTA_DEBUG
	ASSERT(0);
#else
	ptaInit();
#endif
	}
		
}

#ifdef PTA_TEST
void PtaStateTestGpio0(void *eventInfo)
{
	PtaGpioErrorParams_t	params = {0};
	UNUSED_PARAM(eventInfo);
	
	/* restart starvation timer 100 mSec */
	PtaStarvationTimerRestart(PTA_STARVATION_TIMER_100_MILLISECONDS);
	
	/* read GPIO #1: if LOW send message to PTA */
	if (PtaGpioRead(PTA_GPIO_NAME_BT_ACTIVE) == PTA_GPIO_VAL_LOW)
	{
		params.gpioName = PTA_GPIO_NAME_BT_ACTIVE;
		params.gpioError = GPIO_SHOULD_BE_HIGH;
			
		PtaSendMessage(PTA_TEST_GPIO_VALUE_IS_WRONG, &params);	
	}
	/* read GPIO #2: if LOW send message to PTA */
	else if (PtaGpioRead(PTA_GPIO_NAME_GRANT) == PTA_GPIO_VAL_LOW)
	{
		params.gpioName = PTA_GPIO_NAME_GRANT;
		params.gpioError = GPIO_SHOULD_BE_HIGH;
			
		PtaSendMessage(PTA_TEST_GPIO_VALUE_IS_WRONG, &params);
	}
	/* GPIOs are OK */
	else
	{			
		/* Write 0 to GPIO2 */
		PtaGpioWrite(PTA_GPIO_NAME_GRANT, PTA_GPIO_VAL_LOW);
		/* Write 0 to GPIO1 */
		PtaGpioWrite(PTA_GPIO_NAME_BT_ACTIVE, PTA_GPIO_VAL_LOW);

		/* start HW timer 5000 micro seconds */
		PtaHwTimerStart(PTA_HW_TIMER_VAL_5000_MICROSECONDS);
	}
}

void PtaStateTestStarvationTimerExpired(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_TRACE	
	ILOG0_D("[PTA][PtaStateTestStarvationTimerExpired] State = %d ", ptaDb.state);
#endif
	ASSERT(0);
}
#endif

void PtaInitPostTransitionCleanUp(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
	/* Set CCA to "clear" */
	PtaCcaSet(PTA_CCA_QUIET);
	
	/* Set PHY TX disable register to "ENABLE TX" */
	PtaPhyTxRegisterSet(1 << PTA_PHY_TX_REGISTER_FEATURE_ENABLE);

	/* Clear GPIO0 latch */
	ptaGpioLatchClearGpio0();	
}

void PtaMoveToIdle(void *eventInfo)
{	
	TX_INTERRUPT_SAVE_AREA;
	
	UNUSED_PARAM(eventInfo);	
	
	/* Stop starvation timer */
	PtaStarvationTimerStop();
#ifdef PTA_TRACE	
	ILOG0_V("[PTA][PtaMoveToIdle]");
#endif

	/* Set CCA to "clear" */
	PtaCcaSet(PTA_CCA_QUIET);
	
	/* Set PHY TX disable register to "ENABLE TX" */
	PtaPhyTxRegisterSet(1 << PTA_PHY_TX_REGISTER_FEATURE_ENABLE);
	
	/* Handle pending configuration if any */
	PtaPendingDriverReq();

	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_IDLE);

    OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* Clear GPIO0 latch */
	ptaGpioLatchClearGpio0();

	/* Un-mask GPIO0 interrupt */
	ptaGpioUnmsakGpio0();

	/* Set GPIOs to their initial state */
	PtaGpioInitGpioState();

    OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

void PtaZigBeeRxShort(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS
	PtaStatistics.zigbeeTxOngoing++;
#endif
#ifdef PTA_TRACE	
	ILOG0_V("[PTA][PtaZigBeeRxShort]");
#endif

	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_RX_SHORT);
	/* Start HW timer 200 Micro */
	PtaHwTimerStart(PTA_HW_TIMER_ZIGBEE_SHORT_200_MICROSECONDS);
}

void PtaZigBeeRxMid(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS
	PtaStatistics.zigbeeTxOngoing++;
#endif
#ifdef PTA_TRACE	
	ILOG0_V("[PTA][PtaZigBeeRxMid]");
#endif
	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_RX_MID);
	/* Start HW timer 1300 Micro */
	PtaHwTimerStart(PTA_HW_TIMER_ZIGBEE_MID_1300_MICROSECONDS);
}

void PtaZigBeeRxLong(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS
	PtaStatistics.zigbeeTxOngoing++;
#endif
#ifdef PTA_TRACE	
	ILOG0_V("[PTA][PtaZigBeeRxLong]");
#endif

	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_RX_LONG);
	/* Start HW timer 2600 Micro */
	PtaHwTimerStart(PTA_HW_TIMER_ZIGBEE_LONG_2600_MICROSECONDS);
}

void PtaZigBeePoll(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS
	PtaStatistics.zigbeeTxOngoing++;
#endif
#ifdef PTA_TRACE	
	ILOG0_V("[PTA][PtaZigBeePoll]");
#endif

	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_POLL);	
	/* Start HW timer for polling on ZigBee TX 400 Micro */	
	PtaHwTimerStart(PTA_HW_TIMER_POLLING_400_MICROSECONDS);
}

void PtaZigBeeDenyError(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS
	PtaStatistics.zigbeeDenyOverTime++;
#endif

#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaZigBeeDenyError] ZigBee does not understand DENY");
#endif

#ifdef PTA_DEBUG
	ASSERT(0);
#else	 
	ptaInit();
#endif
}

void PtaGrantNotUsed(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS

	if (ptaDb.radioType == PTA_RADIO_TYPE_BT)
	{
		PtaStatistics.btGrnatNotUsed++;
	}
	else
	{
		PtaStatistics.zigbeeGrnatNotUsed++;
	}
#endif

#ifdef PTA_TRACE
	ILOG0_D("[PTA][PtaGrantNotUsed] radio = %d is not using the grant", ptaDb.radioType);
#endif


	/* Move to IDLE (assert in debug mode) */
#ifdef PTA_DEBUG
	ASSERT(0); 
#else
	ptaInit();
#endif
}

void PtaStartBtTx(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaStartBtTx]");
#endif	

#ifdef PTA_STATS  
	PtaStatistics.btGrnatUsed++;
#endif	
	/* Start HW timer with value BT_POLLING */
	PtaHwTimerStart(PTA_HW_TIMER_POLLING_400_MICROSECONDS);
	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_BT_POLL);
}

void PtaStartZigBeeTx(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaStartZigBeeTx]");
#endif	

#ifdef PTA_STATS
	PtaStatistics.zigbeeGrnatUsed++;
#endif	
	/* Change state */
	PtaStateChange(ptaDb.state, PTA_STATE_ZIGBEE_TX);

	/* Start HW timer with value ZIGBEE_POLLING */
	PtaHwTimerStart(PTA_HW_TIMER_POLLING_400_MICROSECONDS);
}

void PtaTxOngoing(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
	
#ifdef PTA_STATS
	if (ptaDb.radioType == PTA_RADIO_TYPE_BT)
	{
		PtaStatistics.btTxOngoing++;
	}
	else
	{
		PtaStatistics.zigbeeTxOngoing++;
	}
#endif

#ifdef PTA_TRACE
	ILOG0_D("[PTA][PtaTxOngoing] BT = %d", ptaDb.radioType);
#endif

	/* Continue polling: Set HW timer to 400 Micro */
	PtaHwTimerStart(PTA_HW_TIMER_POLLING_400_MICROSECONDS);
}

#ifdef PTA_TEST
void PtaStateTestHwTimerOk(void *eventInfo)
{
	PtaGpioErrorParams_t params = {0};
	
	UNUSED_PARAM(eventInfo);	
	
	/* Stop starvation timer */
	PtaStarvationTimerStop();

	/* read GPIO BT_ACTIVE: if "1" send error message to PTA */ // NOTE: gpio0 does not trigger an interrupt hence we jumped the writing of '0' to GPIOs #1 and #2 => we also need to jump the checking that GPIO #1 and #2 are '0'
//	if (PtaGpioRead(PTA_GPIO_NAME_BT_ACTIVE) == PTA_GPIO_VAL_HIGH)
//	{
//		params.gpioName = PTA_GPIO_NAME_BT_ACTIVE;
//		params.gpioError = GPIO_SHOULD_BE_LOW;
//			
//		PtaSendMessage(PTA_TEST_GPIO_VALUE_IS_WRONG, &params);		
//	}
//
//	/* read GPIO GRANT: if "1" send error message to PTA */
//	else if (PtaGpioRead(PTA_GPIO_NAME_GRANT) == PTA_GPIO_VAL_HIGH)
//	{
//		params.gpioName = PTA_GPIO_NAME_GRANT;
//		params.gpioError = GPIO_SHOULD_BE_LOW;
//		
//		PtaSendMessage(PTA_TEST_GPIO_VALUE_IS_WRONG, &params);
//	}
//	/* GPIOs are OK */
//	else
	{
		/* Send message to PTA: the HW timer is OK */
		PtaSendMessage(PTA_TEST_HW_TIMER_IS_OK, &params);
	}
	
}
#endif

/*************************************************************************************************************
messages
**************************************************************************************************************/

static void PtaStarvationTimerExpiry(K_MSG *psMsg)
{
	void *event = NULL;

	UNUSED_PARAM(psMsg);

#ifdef PTA_STATS
	PtaStatistics.illegalEvent[PTA_EVENT_STARVATION_TIMER_EXPIRED][ptaDb.state][ptaDb.radioType]++;
#endif

	PtaRunPtaSm(PTA_EVENT_STARVATION_TIMER_EXPIRED, event);
}
	
    
/***********************************************************************
* PtaHandle24GCoExReq
* 
* Description:
* ------------
* Driver message handler
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void PtaHandle24GCoExReq(K_MSG *psMsg)
{
	PtaCofigure(psMsg);
	PtaSendCfmToDriver(psMsg);
}

    
/***********************************************************************
* PtaCofigure
* 
* Description:
* ------------
* Configure PTA according to message from Driver
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void PtaCofigure(K_MSG *psMsg)
{
	/* Store command parameters */
	UMI_SET_2_4_G_COEX	*driverReq	= (UMI_SET_2_4_G_COEX*)pK_MSG_DATA(psMsg);
	UMI_SET_2_4_G_COEX	*storedReq	= &ptaDb.driverReq;

	/* Store Driver request (shall be handled at INIT or IDLE states) */
	//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
	MEMCPY(storedReq, driverReq, sizeof(UMI_SET_2_4_G_COEX));

	if (driverReq->getSetOperation == API_GET_OPERATION)
	{
		driverReq->bCoExActive = ptaDb.activeParams.bCoExActive;
		driverReq->coExQos = ptaDb.activeParams.coExQos;
	}
	else
	{
		/* Set pending Driver request */
		ptaDb.pendingDriverReq = TRUE;
		
		if (storedReq->bCoExActive == TRUE)
		{
			if (storedReq->coExQos != CO_EX_QOS_NONE)
			{
				#ifdef PTA_TRACE
					ILOG0_V("[PTA][PtaHandle24GCoExReq]  Driver: ON");
				#endif			
				if (ptaDb.state == PTA_STATE_INIT)
				{
					/* Initialize (configure GPIO direction etc.) */
					ptaInit();			
				}
				else if(ptaDb.state == PTA_STATE_IDLE)
				{
					/* Driver configuration is pending start using it */
					PtaPendingDriverReq();			
				}
				else
				{
					#ifdef PTA_TRACE
						ILOG0_D("[PTA][PtaHandle24GCoExReq] Delaying the handling of Driver command since PTA is not in IDLE. PTA State = %d", ptaDb.state);
					#endif
				}
			}
			else
			{
				PtaStateChange(ptaDb.state, PTA_STATE_INIT);
				#ifdef PTA_TRACE
					ILOG0_V("[PTA][PtaHandle24GCoExReq] Driver: all other radios off");
				#endif
				
				PtaRepAssignNewParams(FALSE, PTA_REP_ACTIVE_PERIOD_OFF, PTA_REP_INACTIVE_PERIOD_OFF);
				/* Mask GPIO0 interrupt */
				PtaGpioMaskGpio0();
			}
		}
		else
		{		
			PtaStateChange(ptaDb.state, PTA_STATE_INIT);
			#ifdef PTA_TRACE
				ILOG0_V("[PTA][PtaHandle24GCoExReq] Driver: 2.4G co-ex feature is off");
			#endif
		
			PtaRepAssignNewParams(FALSE, PTA_REP_ACTIVE_PERIOD_OFF, PTA_REP_INACTIVE_PERIOD_OFF);
			/* Mask GPIO0 interrupt */
			PtaGpioMaskGpio0(); 	
		}
	}
}

    
/***********************************************************************
* PtaSendCfmToDriver
* 
* Description:
* ------------
* Confirm the message from Driver
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void PtaSendCfmToDriver(K_MSG *psMsg)
{
	/* Send confirmation to Driver */
	OSAL_SEND_MESSAGE(UMI_MC_MAN_2_4GHZ_COEX_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);		
}

#ifdef PTA_TEST
static void PtaTestGpioValueIsWrong(K_MSG *psMsg)
{
#ifdef PTA_TRACE	
	PtaGpioErrorParams_t *ptaGpioErrorParams = NULL;

	ptaGpioErrorParams = (PtaGpioErrorParams_t *)psMsg->abData;

	UNUSED_PARAM(ptaGpioErrorParams);

	ILOG0_DD("[PTA][GPIO ERROR][PtaTestGpioValueIsWrong] GPIO = %d ERROR = %d",ptaGpioErrorParams->gpioName, ptaGpioErrorParams->gpioError);
#endif
}

static void PtaTestHwTimerIsOK(K_MSG *psMsg)
{
	UNUSED_PARAM(psMsg);
	
	/* Change state back to INIT */
	PtaStateChange(ptaDb.state, PTA_STATE_INIT);
	/* Clear GPIOs: set them to there INIT value (LOW): */

	/* Write 0 to GPIO2 = BT_ACTIVE (WLAN_GPIO #1 wired to GRX_GPIO #11) */
	PtaGpioWrite(PTA_GPIO_NAME_GRANT, PTA_GPIO_VAL_LOW);
	/* Write 0 to GPIO1 = GRANT (WLAN_GPIO #2 wired to GRX GPIO #9) */
	PtaGpioWrite(PTA_GPIO_NAME_BT_ACTIVE, PTA_GPIO_VAL_LOW);
	/* Write 0 to GPIO01 = TX_DEMAND (WLAN_GPIO #0 wired to W_GPIO #0) */
	PtaGpioWrite(PTA_GPIO_NAME_TX_DEMAND, PTA_GPIO_VAL_LOW);
	
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaTestHwTimerIsOK] Test completed OK");	
#endif
}
#endif

/*************************************************************************************************************
GPIOs
**************************************************************************************************************/
/**********************************************************************************
PtaGpioDetermineEvent


Description:
------------
This function determines which event has taken place:
1. Is BT requesting or ZigBee
2. Has their hold off counter reached its threshold
	 
Input:
-----
	None
Output:
-------
	The event
Returns:
--------
	None
**********************************************************************************/
static PtaEvents_e PtaGpioDetermineEvent(void)
{	
	PtaEvents_e ptaHwTimerSmEvents = PTA_EVENT_NUM_OF_EVENTS;
	PtaDb_t *localPtaDb = &ptaDb;
	PtaUniqueRadioParams_t *uniqueParams = &localPtaDb->ptaParams.uniqueRadioParams[0];
#ifdef PTA_TRACE
	ILOG0_DDDD("[PTA][PtaGpioDetermineEvent] btCounter = %d BT->radioHoldOffValue = %d ZigBeeCounter = %d ZigBee->radioHoldOffValue = %d", 
    localPtaDb->btCounter, uniqueParams[PTA_RADIO_TYPE_BT].radioHoldOffValue, localPtaDb->ZigBeeCounter, uniqueParams[PTA_RADIO_TYPE_ZIGBEE].radioHoldOffValue);
#endif

	/* Which radio is requesting (is BT connected to the board and BT_ACTIVE is high?) */
	if (PtaGpioReadBtActive() == TRUE)
	{		
		/* Increment specific radio threshold counter */
		localPtaDb->btCounter++;
		/* Start with deny */
		ptaHwTimerSmEvents = PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_NOT_REACHED;
		/* In case the TH is not zero (which means always deny) compare to counter */
		if (uniqueParams[PTA_RADIO_TYPE_BT].radioHoldOffValue != 0)
		{
			if (localPtaDb->btCounter == uniqueParams[PTA_RADIO_TYPE_BT].radioHoldOffValue)
			{			
				ptaHwTimerSmEvents = PTA_EVENT_GPIO0_BT_ACTIVE_HIGH_TX_DEMAND_HIGH_TH_REACHED;
			}
		}
	}
	else
	{
		/* In case we have received GPIO_0 and BT is not connected, PtaGpioReadBtActive will return FALSE. Now we must check that ZigBee is connected before incrementing the hold off counter. */
		if (uniqueParams[PTA_RADIO_TYPE_ZIGBEE].bRadioDriverIsUp)
		{
			/* Increment specific radio threshold counter */
			localPtaDb->ZigBeeCounter++;
			/* Start with deny */
			ptaHwTimerSmEvents = PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_NOT_REACHED;
			/* In case the TH is not zero (which means always deny) compare to counter */
			if (uniqueParams[PTA_RADIO_TYPE_ZIGBEE].radioHoldOffValue != 0)
			{
				if (localPtaDb->ZigBeeCounter == uniqueParams[PTA_RADIO_TYPE_ZIGBEE].radioHoldOffValue)
				{			
					ptaHwTimerSmEvents = PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_TH_REACHED;
				}
			}
		}
		else
		{
			ptaHwTimerSmEvents = PTA_EVENT_GPIO0_BT_ACTIVE_LOW_TX_DEMAND_HIGH_ZIGBEE_OFF;
		}
	}
	return(ptaHwTimerSmEvents);	
}
/*************************************************************************************************************
Starvation timer
**************************************************************************************************************/
static void	PtaStarvationTimerStart(uint8 timeToGo)
{
	OSAL_SET_TIMER_EXPLICIT(PTA_STARVATION_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(timeToGo), TASK_PTA);
#ifdef PTA_TRACE
	ILOG0_D("[PTA][PtaStarvationTimerStart] timeToGo = %d", timeToGo);
#endif
}

static void PtaStarvationTimerStop(void)
{
	/* Stop starvation timer */
	OSAL_RESET_TIMER_EXPLICIT(PTA_STARVATION_TIMER, TASK_PTA);
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaStarvationTimerStop]");
#endif
}

#ifdef PTA_TEST
static void PtaStarvationTimerRestart(uint8 timeToGo)
{
	/* Stop starvation timer */
	PtaStarvationTimerStop();
	/* Start starvation timer */
	PtaStarvationTimerStart(timeToGo);
}

/*************************************************************************************************************
TEST
**************************************************************************************************************/
/**********************************************************************************
PtaTestHandler


Description:
------------
Parser of the test command 
	 
Input:
-----
	User parameters
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PtaTestHandler(uint8 data1, uint8 data2, uint8 data3)
{
	switch (data1)
	{
		case 1:
			PtaTest();
			break;
	
		case 2:
			PtaTestDriverReq(data3);
			break;
	
		case 3:
			/* Set the GPIOs state according to user request and emulate a GPIO0 interrupt by calling Isr_PtaGpio0 */
			switch (data2)
			{
				case 0: 
					PtaTestEmulateGpio0Int(/* BT_ACTIVE */ PTA_GPIO_VAL_HIGH, /* TX_DEMAND */PTA_GPIO_VAL_HIGH);
					break;
				case 1:
					PtaTestEmulateGpio0Int(/* BT_ACTIVE */ PTA_GPIO_VAL_LOW, /* TX_DEMAND */PTA_GPIO_VAL_HIGH);
					break;
				case 2: 
					PtaTestEmulateGpio0Int(/* BT_ACTIVE */ PTA_GPIO_VAL_LOW, /* TX_DEMAND */PTA_GPIO_VAL_LOW);
					break;
				default:
					break;			
			}
			break;

		case 4:
			/*	data2 - active period
			   	data3 - inactive period
			   	In case data2 = 0 it means OFF */
			PtaTestRep(data2, data3);
			
			break;

		case 5:
			/*	data2 - parameter

				0 bUseRepetitionMode;
				1 repetitionSuspend;
				2 repetitionResume;
				3 radioHoldOffValue;
				4 starvationTimeValue;

				data3 - value
				bUseRepetitionMode		- 0: OFF 1:ON
				repetitionSuspend 		- value will be multiplied by 10
				repetitionResume  		- value will be multiplied by 100
				radioHoldOffValue		- value used as is
				starvationTimeValue		- value is used as is				*/
				
			PtaTestModifyQosParam(data2, data3);			
			break;
		case 6:
			PtaTestTracePtaParams();
			break;
		case 99:
			/* Reset PTA to be ready for next test command */
			PtaTestReset();
			break;
			
		default:
			break;	

	}
}
#endif // PTA_TEST	

/*************************************************************************************************************
ISRs
**************************************************************************************************************/


/**********************************************************************************
Isr_PtaGpio0


Description:
------------
 
	 
Input:
-----
	User parameters
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void Isr_PtaGpio0(void)
{
	//void *eventInfo = NULL;

	/* Clear GPIO interrupt (clear the latch) ===>>> clear the latch when mooving to IDLE */
	//InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_EXTERNAL);

	/* Mask GPIO interrupt ===>>> this is not possible in wave500, hence a single solution for both WAVE400 and WAVE500 shall use LATCH and IRQ enabling */
	//InterruptManager_MaskInterrupt(MAPPED_INT_EXTERNAL);

	/* MASK GPIO0 */
	PtaGpioMaskGpio0();

#ifdef PTA_TRACE
	ILOG0_DDD("[PTA][Isr_PtaGpio0] BT_ACTIVE = %d TX_DEMAND = %d state = %d", (PtaGpioReadBtActive() == TRUE), (PtaGpioRead(PTA_GPIO_NAME_TX_DEMAND) == PTA_GPIO_NAME_TX_DEMAND), ptaDb.state);
#endif

	/* Check that TX_DEMAND is high (if it is not it means we lost an interrupt (assert and recover)) */
	if (PtaGpioRead(PTA_GPIO_NAME_TX_DEMAND) == PTA_GPIO_NAME_TX_DEMAND)
	{
		/* Determine which event has occured */
		PtaEvents_e ptaHwTimerSmEvents = PtaGpioDetermineEvent();

#ifdef PTA_STATS
		PtaStatistics.gpio++;
#endif
		
#ifdef PTA_TRACE
		ILOG0_D("[PTA][Isr_PtaGpio0] ptaSmEvents = %d", ptaHwTimerSmEvents);
#endif

		/* Handle */
		PtaRunPtaSm(ptaHwTimerSmEvents, (void *)ptaHwTimerSmEvents);		
	}
	else
	{
		/* GPIO0 triggered interrupt. ISR is reading TX_DEMAND = LOW. We lost an interrupt? save statistics and re-init the PTA. Assert in debug mode. */
#ifdef PTA_STATS
		PtaStatistics.gpio0TxDemandLow++;
#endif

#ifdef PTA_DEBUG
		ASSERT(0);
#else
		ptaInit();
#endif
	}

}

void PtaHandleTimer1Expiry(void)
{
	PtaEvents_e event = PTA_EVENT_NUM_OF_EVENTS;

#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaHandleTimer1Expiry]");
#endif

	event = PtaHwTimerDetermineEvent();

	ASSERT(event < PTA_EVENT_NUM_OF_EVENTS);
	
// already cleared at isr_Timers_TimerA 	/* Clear the pending interrupt bit */
// already cleared at isr_Timers_TimerA 	PtaHwTimerClearIrq();

	/* Handle */
	PtaRunPtaSm(event, (void *)event);
}
/*************************************************************************************************************
HW timer
**************************************************************************************************************/

static PtaEvents_e PtaHwTimerDetermineEvent(void)
{
	PtaEvents_e event = PTA_EVENT_NUM_OF_EVENTS;
		
	bool isTxDemandHigh = FALSE;
	bool isBtActiveHigh = FALSE;
	bool isGrantHigh = FALSE;
	
	/* Read GRANT, BT_ACTIVE and TX_DEMAND values */
	isBtActiveHigh = PtaGpioReadBtActive();
	isTxDemandHigh = (PtaGpioRead(PTA_GPIO_NAME_TX_DEMAND) == PTA_GPIO_NAME_TX_DEMAND);
	isGrantHigh	= PtaGpioReadGrant();
	
	/* Determine the event */
	if (isGrantHigh == TRUE)
	{
		if (isBtActiveHigh == TRUE)
		{
			event = PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_HIGH_TX_DEMAND_HIGH;
		}
		else if(isTxDemandHigh == TRUE)
		{
			event = PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_HIGH;
		}
		else
		{
			event = PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_LOW_TX_DEMAND_LOW;
		}
	}
	else
	{
		if (isTxDemandHigh == TRUE)
		{
			event = PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_HIGH;
		}
		else
		{
			event = PTA_EVENT_POLL_TIMER_EXPIRED_BT_ACTIVE_MASKED_TX_DEMAND_LOW;
		}
	}

#ifdef PTA_TRACE	
	ILOG0_D("[PTA][PtaHwTimerDetermineEvent] event = %d", event);
#endif

	return (event);
}

// already cleared at isr_Timers_TimerA static void PtaHwTimerClearIrq(void)
// already cleared at isr_Timers_TimerA {
// already cleared at isr_Timers_TimerA 	/* Clear IRQ bit */	
// already cleared at isr_Timers_TimerA 	MT_ClearIsrTimer(MT_TIMER_1);
// already cleared at isr_Timers_TimerA }

static void PtaHwTimerStart(uint32 timeout)
{
#ifdef PTA_TRACE	
	ILOG0_D("[PTA][PtaHwTimerStart] timeout = %d", timeout);
#endif	
	MT_SetTimeTimer(MT_TIMER_1, timeout);
	MT_SetModeTimer(MT_TIMER_1, MT_TIMER_ENABLE, MT_TIMER_ONE_SHOT);
}

//static void PtaHwTimerStop(void)
//{
//	/* Disable interrupt */
//	MT_enableDisableGpTimerIrq(MT_MIPS_TIMER_1, MT_MIPS_TIMER_DISABLE);
//
//	/* Poll for finish */
//	MT_PollMipsTimer(MT_MIPS_TIMER_1);
//
//	/* clear IRQ  */
//	MT_ClearIsrMipsTimer(MT_MIPS_TIMER_1);
//
//	/* Disable Timer */
//	MT_SetModeMipsTimer(MT_MIPS_TIMER_1, MT_MIPS_TIMER_DISABLE, MT_MIPS_TIMER_ONE_SHOT);
//
//	/* Enable interrupt */
//	MT_enableDisableGpTimerIrq(MT_MIPS_TIMER_1, MT_MIPS_TIMER_ENABLE);
//}

static void PtaHwTimerEnableInterrupt(void)
{
	/* Enable interrupt */
	MT_enableDisableGpTimerIrq(MT_TIMER_1, MT_TIMER_ENABLE);
}

//static void PtaHwTimerDisableInterrupt(void)
//{
//	/* Disable interrupt */
//	MT_enableDisableGpTimerIrq(MT_MIPS_TIMER_1, MT_MIPS_TIMER_DISABLE);
//}

/*************************************************************************************************************
MAC and PHY HW handlers
**************************************************************************************************************/

static void PtaCcaSet(uint8 cca)
{
	/* Set the CCA*/
	if (cca == PTA_CCA_NOISY)
	{
		Pac_RxcSetForceCca();
	}
	else if (cca == PTA_CCA_QUIET)
	{
		Pac_RxcSetForceCcaToNormalMode();
	}
}


/**********************************************************************************

PtaPhyTxRegisterSet 


Description:
------------
Modify the state of the TX status to either ENABLE or DISABLE transmission.
Note: the address (0xAF320E74) is used by TX_SENDER

Input: 
-----
		
Output:
-------
	
Returns:
--------
None	
**********************************************************************************/
static void PtaPhyTxRegisterSet(uint32 phyTxReq)
{
	/* Set PHY TX register */
	RegAccess_Write(COEX_Wifi_Tx_Control, phyTxReq);

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaPhyTxRegisterSet] register = 0x%x write regVal = 0x%x", COEX_Wifi_Tx_Control, phyTxReq);
#endif // PTA_TRACE	

}



/**********************************************************************************

PtaRepAssignNewParams 


Description:
------------
Repetition mode has been disabled. Send message to repetiton entity.

Input: 
-----
		
Output:
-------
	
Returns:
--------
None
**********************************************************************************/
void PtaRepAssignNewParams(bool enable, uint32 activePeriod, uint32 inactivePeriod)
{
	K_MSG *pMsg = NULL;
	PtaRepConfig_t *ptaRepConfig = NULL;

	/* Send OFF command to repetition */
	pMsg = OSAL_GET_MESSAGE(sizeof(PtaRepConfig_t));
	ptaRepConfig = (PtaRepConfig_t*)pK_MSG_DATA(pMsg);
	ptaRepConfig->enable = enable;
	ptaRepConfig->activePeriod = activePeriod;
	ptaRepConfig->inactivePeriod = inactivePeriod;
	OSAL_SEND_MESSAGE(PTA_REP_CONFIG, TASK_PTA, pMsg, VAP_ID_DO_NOT_CARE);
}


/*************************************************************************************************************
PTA Errors
**************************************************************************************************************/
void PtaIgnoreEvent(void *eventInfo)
{
#if defined (PTA_TRACE) || defined (PTA_STATS)	
	PtaEvents_e event = PTA_EVENT_NUM_OF_EVENTS;
	AAA_PTA_HW_TIMER_INDEX = (AAA_PTA_HW_TIMER_INDEX + AAA_PTA_MONITOR_ARRAY_SIZE - 1) % AAA_PTA_MONITOR_ARRAY_SIZE;
	event = AAA_PTA_HW_TIMER[AAA_PTA_HW_TIMER_INDEX].event;
#endif

	UNUSED_PARAM(eventInfo);


#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaIgnoreEvent] State = %d event = %d	*** que pasa? ***", ptaDb.state, event);	
#endif

#ifdef PTA_DEBUG
	ASSERT(0);
#endif

#ifdef PTA_STATS
	PtaStatistics.ignoreEvent[event][ptaDb.state][ptaDb.radioType]++;
#endif

}

void PtaGpio0NotInIdle(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);	
	// For the debug stage lets assert. For the operational version do "PtaIllegalEvent"
	ASSERT(0);
	//PtaIllegalEvent(eventInfo);
}

void PtaIllegalEvent(void *eventInfo)
{
	UNUSED_PARAM(eventInfo);
#if defined (PTA_TRACE) || defined (PTA_STATS)	
	PtaEvents_e event = PTA_EVENT_NUM_OF_EVENTS;

	AAA_PTA_HW_TIMER_INDEX = (AAA_PTA_HW_TIMER_INDEX + AAA_PTA_MONITOR_ARRAY_SIZE - 1) % AAA_PTA_MONITOR_ARRAY_SIZE;
	event = AAA_PTA_HW_TIMER[AAA_PTA_HW_TIMER_INDEX].event;
#endif

#ifdef PTA_STATS
	PtaStatistics.illegalEvent[event][ptaDb.state][ptaDb.radioType]++;
#endif

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaIllegalEvent] State = %d event = %d									*** que pasa? ***", ptaDb.state, event);
#endif

#ifdef PTA_DEBUG
	ASSERT(0);
#else
	ptaInit();
#endif
}

#endif

/**********************************************************************************

PtaGetGlobalStatisticsAddress 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic	
**********************************************************************************/
uint32* PtaGetGlobalStatisticsAddress(void)
{
	return ((uint32*)&PtaStatistics);
}
