/***********************************************************************************
 File:			PacketTrafficArbitratorHW.c
 Module:			PacketTrafficArbitrator
 Purpose:				2.4GHz coexistance
 Description:	This src file is the implementation of the HW related actions for the Packet Traffic Arbitrator module		
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/

#include "ErrorHandler_Api.h"
#include "InterruptManager_Api.h"
#include "loggerAPI.h"
#include "MacHtExtensionsRegs.h"
#include "OSAL_Api.h"
#include "PacketTrafficArbitrator.h"
#include "Pac_Api.h"
#include "PacketTrafficArbitrator_api.h"
#include "queue_utility.h"
#include "RegAccess_Api.h"
#include "System_Timers.h"
#include "PacketTrafficArbitratorCommon.h"
#include "Pauser_Api.h"
#include "ServicesHandler_Api.h"
#include "ProcessManager_Api.h"
#include "BSSmanager_API.h"

#ifdef PTA_BUILD_IN_PLAT

extern PtaDb_t	ptaDb;
extern PtaRepetitionParams_t	ptaRepDb;


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_PACKET_TRAFFIC_ARBITRATOR
#define LOG_LOCAL_FID 2
/*************************************************************************************************************
Repetition mode (emulated in Gen4)
**************************************************************************************************************/
/* Repetition emulation events */
typedef enum
{
	PTA_REP_EVENT_CONFIG,
	PTA_REP_EVENT_TIMER,
	PTA_REP_EVENT_START,
	PTA_REP_EVENT_END,
	PTA_REP_EVENT_NUM_OF_EVENTS,
}PtaRepEvents;

/* Repetition emulation events to handlers */
typedef enum
{
	PTA_REP_EVENT_TO_STATE_FUNCTION_START,
	PTA_REP_EVENT_TO_STATE_FUNCTION_ACTIVE_TIMER,
	PTA_REP_EVENT_TO_STATE_FUNCTION_PROCESS_START,
	PTA_REP_EVENT_TO_STATE_FUNCTION_INACTIVE_TIMER,
	PTA_REP_EVENT_TO_STATE_FUNCTION_PROCESS_END,
	PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,
	PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,
	PTA_REP_EVENT_TO_STATE_FUNCTION_NUM_OF_FUNCTIONS,
}PtaRepEventsToStateFunction;

typedef void (*PtaRepHandler)(void);

/* GPIO_OEN_GPIO_OUTPUT_ENABLE */
#define REG_GPIO_OEN_GPIO_OUTPUT_ENABLE_MASK     	(0x0001ffff)
/* GPIO_DAT_GPIO_DATA */
#define REG_GPIO_DAT_GPIO_DATA_MASK     			(0x0000ffff)


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void PtaRepEventStart(void);
void PtaRepEventActiveTimer(void);
void PtaRepEventProcessStart(void); 
void PtaRepEventInactiveTimer(void);
void PtaRepEventProcessStop(void);
void PtaRepEventIgnore(void);
void PtaRepEventAssert(void);
static void PtaRepStateChange(PtaRepStates nextState);
static void PtaRepTimerStart(uint32 inactivePeriod);
void PtaGpioWriteWlanDeny(uint8 ptaGpioVal);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const PtaRepEventsToStateFunction PtaRepStateMachine[PTA_REP_STATE_NUM_OF_STATES][PTA_REP_EVENT_NUM_OF_EVENTS] =
{
	/* PTA_REP_STATE_INACTIVE */
	{
		PTA_REP_EVENT_TO_STATE_FUNCTION_START,				/* PTA_REP_EVENT_CONFIG	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_TIMER	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_START	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_END	*/	
	},
	/* PTA_REP_STATE_WAIT_FOR_ACTIVE_TIMER */
	{
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_CONFIG	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ACTIVE_TIMER,		/* PTA_REP_EVENT_TIMER	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_START	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_END	*/	
	},
	
	/* PTA_REP_STATE_WAIT_FOR_PROCESS_START  */
	{
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_CONFIG	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_TIMER	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_PROCESS_START,		/* PTA_REP_EVENT_START	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_END	*/	
	},
	
	/* PTA_REP_STATE_WAIT_FOR_INACTIVE_TIMER */
	{
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_CONFIG	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_INACTIVE_TIMER,		/* PTA_REP_EVENT_TIMER	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_START	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_ASSERT,				/* PTA_REP_EVENT_END	*/	
	},
	
	/* PTA_REP_STATE_WAIT_FOR_PROCESS_END	 */
	{
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_CONFIG */
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_TIMER	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_IGNORE,				/* PTA_REP_EVENT_START	*/
		PTA_REP_EVENT_TO_STATE_FUNCTION_PROCESS_END,		/* PTA_REP_EVENT_END	*/
	},	
};

/* */
static const PtaRepHandler PtaRepEventToStateFunctionTable[PTA_REP_EVENT_TO_STATE_FUNCTION_NUM_OF_FUNCTIONS] =
{
	PtaRepEventStart,
	PtaRepEventActiveTimer,
	PtaRepEventProcessStart,
	PtaRepEventInactiveTimer,
	PtaRepEventProcessStop,
	PtaRepEventIgnore,
	PtaRepEventAssert,
};

/*************************************************************************************************************
Repetition mode
**************************************************************************************************************/

void PtaRepEventStart(void)
{
#ifdef PTA_REP_TRACE
	ILOG0_D("[PTA][REP][PtaRepEventStart] repetition enable = %d", ptaRepDb.enable);
#endif
	
	if (ptaRepDb.enable == TRUE)
	{
		/* Start timer */
		PtaRepTimerStart(ptaRepDb.activePeriod);
		
		/* Waiting for the active period to end (TX is not halted) */
		PtaRepStateChange(PTA_REP_STATE_WAIT_FOR_ACTIVE_TIMER);	
	}
}

void PtaRepEventActiveTimer(void)
{
	K_MSG *processManagerMsg;
	ProcessRequestParams_t *processRequestParams;

#ifdef PTA_REP_TRACE
	ILOG0_D("[PTA][REP][PtaRepEventActiveTimer] repetition enable = %d", ptaRepDb.enable);
#endif

	/* Start a process that pauses WIFI TX */
	if (ptaRepDb.enable == TRUE)
	{
		/* Schedule Process*/	
		processManagerMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));

		processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(processManagerMsg);
		
		processRequestParams->processId 						= PROCESS_ID_PTA;
		processRequestParams->startProcessMsg					= PTA_REP_PROCESS_START;
		processRequestParams->endProcessMsg 					= PTA_REP_PROCESS_END;
		processRequestParams->requesterParams					= NULL;
		processRequestParams->preProcessServiceBitmap			= (1 << SERVICE_ID_PAUSE_AC_TX);
		processRequestParams->postProcessServiceBitmap			= (1 << SERVICE_ID_RESUME_AC_TX);
		processRequestParams->returnTask						= TASK_PTA;
		processRequestParams->updateParamsBeforeFinalizing		= FALSE;
		processRequestParams->serviceData 						= TX_PAUSE_ALL_ACS_EXCEPT_BEACON_AND_GPHP;
		processRequestParams->vapId								= ConfigurationManager_GetFirstVapForBand0();
		processRequestParams->dualBandProcess					= FALSE;
		processRequestParams->processMsgHandledByCdbProcessMan	= FALSE;

		OSAL_SEND_MESSAGE(PROCESS_MANAGER_SCHEDULE_PROCESS_REQUEST,TASK_PROCESS_MANAGER,processManagerMsg, processRequestParams->vapId);

		PtaRepStateChange(PTA_REP_STATE_WAIT_FOR_PROCESS_START);
	}
	else
	{
		PtaRepStateChange(PTA_REP_STATE_INACTIVE);
	}	
}
void PtaRepEventProcessStart(void)
{
	/* For TLOG purpose - used to be no data message, send dummy data instead */
	K_MSG *dummyMsg = NULL;
	
	/* Process manager indicates that the WIFI TX is paused. Start a timer and on expiry resume the WIFI TX */
#ifdef PTA_REP_TRACE
	ILOG0_D("[PTA][REP][PtaRepEventProcessStart] repetition enable = %d", ptaRepDb.enable);
#endif

	if (ptaRepDb.enable == TRUE)
	{
		/* Start inactive period */
		PtaRepTimerStart(ptaRepDb.inactivePeriod);
	
		/* Change state: wait for the timer to expire */
		PtaRepStateChange(PTA_REP_STATE_WAIT_FOR_INACTIVE_TIMER);	
	}
	else
	{
		/* Repetition disabled: tell process manager to stop the WIFI TX pause and change state */
		dummyMsg = BssManagerVapManager_GetDummyMessage();		
		OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, dummyMsg, ConfigurationManager_GetFirstVapForBand0()); 
		PtaRepStateChange(PTA_REP_STATE_WAIT_FOR_PROCESS_END);
	}
}
void PtaRepEventInactiveTimer(void)
{
	/* For TLOG purpose - used to be no data message, send dummy data instead */
	K_MSG *dummyMsg = NULL;

#ifdef PTA_REP_TRACE
	ILOG0_V("[PTA][REP][PtaRepEventInactiveTimer]");
#endif
	/* The inactive period has ended: resume WIFI TX by notifing process manager */
	dummyMsg = BssManagerVapManager_GetDummyMessage();
	OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, dummyMsg, ConfigurationManager_GetFirstVapForBand0()); 
	
	/* Change state: wait for the process manager to confirm that it resumed WIFI TX*/
	PtaRepStateChange(PTA_REP_STATE_WAIT_FOR_PROCESS_END);	
}
void PtaRepEventProcessStop(void)
{
#ifdef PTA_REP_TRACE
	ILOG0_D("[PTA][REP][PtaRepEventProcessStop] repetition enable = %d", ptaRepDb.enable);
#endif
	if (ptaRepDb.enable == TRUE)
	{
		/* Process manager indicates that it resumed WIFI TX. Set the active timer and on expiry suspend the WIFI TX */
		PtaRepTimerStart(ptaRepDb.activePeriod);
		
		/* Change state: wait for the timer to expire */
		PtaRepStateChange(PTA_REP_STATE_WAIT_FOR_ACTIVE_TIMER);
	}
	else
	{
		PtaRepStateChange(PTA_REP_STATE_INACTIVE);
	}
}
void PtaRepEventIgnore(void)
{
#ifdef PTA_REP_TRACE
	ILOG0_V("[PTA][REP][PtaRepEventIgnore]");
#endif
}
void PtaRepEventAssert(void)
{
#ifdef PTA_REP_TRACE
	ILOG0_V("[PTA][REP][PtaRepEventAssert] ASSERT === ASSERT === ASSERT");
#endif	
	//ASSERT(0);
}

static void PtaRepStateChange(PtaRepStates nextState)
{
#ifdef PTA_REP_TRACE
	ILOG0_DD("[PTA][REP][PtaRepStateChange] currrnet state = %d next state = %d", 	ptaRepDb.state, nextState);
#endif

	ptaRepDb.state = nextState;	
}

static void PtaRepTimerStart(uint32 timeToGo)
{
	OSAL_SET_TIMER_EXPLICIT(PTA_REP_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(timeToGo), TASK_PTA);
#ifdef PTA_REP_TRACE
	ILOG0_D("[PTA][REP][PtaRepTimerStart] timeToGo = %d", timeToGo);
#endif
}

/*---------------------------------------------------------------------------------
/						Global Functions Definitions									
/----------------------------------------------------------------------------------*/

void PtaRepConfig(K_MSG *ptaRepMessage)
{
	PtaRepEventsToStateFunction	functionIndex;
	PtaRepConfig_t 	*repConfig = (PtaRepConfig_t*)pK_MSG_DATA(ptaRepMessage);

	/* Get parameters */
	ptaRepDb.enable = repConfig->enable;
	ptaRepDb.activePeriod = repConfig->activePeriod;
	ptaRepDb.inactivePeriod = repConfig->inactivePeriod;

#ifdef PTA_REP_TRACE
	ILOG0_DDD("[PTA][REP][PtaRepConfig] enable = %d activePeriod = %d inactivePeriod = %d", ptaRepDb.enable, ptaRepDb.activePeriod, ptaRepDb.inactivePeriod);
#endif
	
	/* Call handler */
	functionIndex = PtaRepStateMachine[ptaRepDb.state][PTA_REP_EVENT_CONFIG];	
	PtaRepEventToStateFunctionTable[functionIndex]();	
}

void PtaRepTimerExpiry(K_MSG *ptaRepMessage)
{
	PtaRepEventsToStateFunction	functionIndex;
	UNUSED_PARAM(ptaRepMessage);

#ifdef PTA_REP_TRACE
	ILOG0_V("[PTA][REP][PtaRepTimerExpiry]");
#endif

	/* Call handler */
	functionIndex = PtaRepStateMachine[ptaRepDb.state][PTA_REP_EVENT_TIMER];
	PtaRepEventToStateFunctionTable[functionIndex]();
}

void PtaRepProcessStart(K_MSG *ptaRepMessage)
{
	PtaRepEventsToStateFunction	functionIndex;
	UNUSED_PARAM(ptaRepMessage);

#ifdef PTA_REP_TRACE
	ILOG0_V("[PTA][REP][PtaRepProcessStart]");
#endif

	/* Call handler */
	functionIndex = PtaRepStateMachine[ptaRepDb.state][PTA_REP_EVENT_START];
	PtaRepEventToStateFunctionTable[functionIndex]();
}
void PtaRepProcessEnd(K_MSG *ptaRepMessage)
{
	PtaRepEventsToStateFunction	functionIndex;
	UNUSED_PARAM(ptaRepMessage);
	
#ifdef PTA_REP_TRACE
	ILOG0_V("[PTA][REP][PtaRepProcessEnd]");
#endif

	/* Call handler */
	functionIndex = PtaRepStateMachine[ptaRepDb.state][PTA_REP_EVENT_END];
	PtaRepEventToStateFunctionTable[functionIndex]();
}

void PtaRepInit(void)
{
	memset(&ptaRepDb, 0, sizeof(ptaRepDb));
}

/*************************************************************************************************************
GPIO wrappers
**************************************************************************************************************/
bool PtaGpioReadBtActive(void)
{
	bool btActiveIsHigh = FALSE;
	if (ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_BT].bRadioDriverIsUp)
	{
		btActiveIsHigh = (PtaGpioRead(PTA_GPIO_NAME_BT_ACTIVE) == PTA_GPIO_NAME_BT_ACTIVE);
	}
#ifdef PTA_TRACE	
	else
	{
		ILOG0_V("[PTA][PtaGpioReadBtActive] BT not connected to board. BT_ACTIVE = LOW");
	}
#endif	
	return(btActiveIsHigh);
}

bool PtaGpioReadGrant(void)
{
	bool grantIsHigh = TRUE;
	if (ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].bRadioDriverIsUp) 
	{
		grantIsHigh = (PtaGpioRead(PTA_GPIO_NAME_GRANT) == PTA_GPIO_NAME_GRANT);
	}	
#ifdef PTA_TRACE	
	else
	{
		/* Grant is considered HIGH so that BT_ACTIVE is not treated as "masked" */
		ILOG0_V("[PTA][PtaGpioReadGrant] ZigBee is not connected to board. GRANT = HIGH");
	}
#endif
	return (grantIsHigh);
}

void PtaGpioWriteGrant(uint8 ptaGpioVal)
{
	if (ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].bRadioDriverIsUp)
	{
		PtaGpioWrite(PTA_GPIO_NAME_GRANT, ptaGpioVal);
	}
#ifdef PTA_TRACE
	else
	{

		ILOG0_V("[PTA][PtaGpioWriteGrant] ZigBee is not connected to board. GRANT is not set");
	}
#endif	
}

void PtaGpioDirectionGrant(bool isDirectionOut)
{
	if (ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_ZIGBEE].bRadioDriverIsUp)
	{
		PtaGpioDirection(PTA_GPIO_NAME_GRANT, isDirectionOut);
	}
#ifdef PTA_TRACE
	else
	{
		ILOG0_V("[PTA][PtaGpioDirectionGrant] ZigBee is not connected to board. Not setting GPIO direction for GRANT");
	}
#endif	
}
void PtaGpioDirectionWlanDeny(bool isDirectionOut)
{
	if (ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_BT].bRadioDriverIsUp)
	{
		PtaGpioDirection(PTA_GPIO_NAME_WLAN_DENY, isDirectionOut);
	}
#ifdef PTA_TRACE
	else
	{
		ILOG0_V("[PTA][PtaGpioDirectionWlanDeny] BT is not connected to board. Not setting GPIO direction for WLAN_DENY");
	}
#endif	
}

void PtaGpioWriteWlanDeny(uint8 ptaGpioVal)
{
	if (ptaDb.ptaParams.uniqueRadioParams[PTA_RADIO_TYPE_BT].bRadioDriverIsUp)
	{
		PtaGpioWrite(PTA_GPIO_NAME_WLAN_DENY, ptaGpioVal);
	}
#ifdef PTA_TRACE	
	else
	{
		ILOG0_V("[PTA][PtaGpioWriteWlanDeny] BT is not connected to board. WLAN_DENY is not set");
	}
#endif	
}



/*************************************************************************************************************
GPIO handlers
**************************************************************************************************************/
    
/***********************************************************************
* PtaGpioDirection
* 
* Description:
* ------------
* Set the direction for the given GPIO NAME
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PtaGpioDirection(uint32 gpioName, bool isDirectionOut)
{
	RegEmeraldEnvGpioOen_u readValue;

	RegAccess_Read(REG_EMERALD_ENV_GPIO_OEN, &readValue.val); 
	
	readValue.bitFields.gpioOutputEnable = (isDirectionOut == TRUE)? (readValue.bitFields.gpioOutputEnable | gpioName):(readValue.bitFields.gpioOutputEnable & (~gpioName));
	
	RegAccess_Write(REG_EMERALD_ENV_GPIO_OEN, readValue.val); 

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaGpioDirection] register = 0x%x value = 0x%x", REG_EMERALD_ENV_GPIO_OEN, readValue.bitFields.gpioOutputEnable);
#endif
}
    
/***********************************************************************
* PtaGpioRead
* 
* Description:
* ------------
* Read the value of the provided GPIO
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
uint32 PtaGpioRead(uint32 gpioName)
{
	RegEmeraldEnvGpioDat_u readValue;
	RegAccess_Read(REG_EMERALD_ENV_GPIO_DAT, &readValue.val);

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaGpioRead] before isolation register = 0x%x value = 0x%x", REG_EMERALD_ENV_GPIO_DAT, readValue.bitFields.gpioData);
#endif

	/* Isolate the GPIO */
	readValue.bitFields.gpioData &= gpioName;

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaGpioRead] after  isolation register = 0x%x value = 0x%x", REG_EMERALD_ENV_GPIO_DAT, readValue.bitFields.gpioData);
#endif

	return(readValue.bitFields.gpioData);
}
    
/***********************************************************************
* PtaGpioWrite
* 
* Description:
* ------------
* Write the given value to the provided GPIO
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PtaGpioWrite(uint32 gpioName, uint8 ptaGpioVal)
{
	RegEmeraldEnvGpioDat_u readValue;
	/* Read 32 bit GPIO register */
	RegAccess_Read(REG_EMERALD_ENV_GPIO_DAT, &readValue.val);

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaGpioWrite] Value read from register  = 0x%x value = 0x%x", REG_EMERALD_ENV_GPIO_DAT, readValue.bitFields.gpioData);
#endif

	/* Change the value */
	readValue.bitFields.gpioData = (ptaGpioVal == PTA_GPIO_VAL_HIGH)? (readValue.bitFields.gpioData | gpioName):(readValue.bitFields.gpioData & (~gpioName));

	/* Write new value to GPIO register */
	RegAccess_Write(REG_EMERALD_ENV_GPIO_DAT, readValue.val);

#ifdef PTA_TRACE
	ILOG0_DD("[PTA][PtaGpioWrite] Value written to register = 0x%x value = 0x%x", REG_EMERALD_ENV_GPIO_DAT, readValue.bitFields.gpioData);
#endif	
}

void PtaGpioMaskGpio0(void)
{
	/* GPIO0 is part of the "less important" interrupts */
	HwEvents_MaskEvent_Common_A(HW_EVENT_EXTERNAL_IRQ_GPIO_10);
#ifdef PTA_TRACE
	ILOG0_DDD("[PTA][PtaGpioMaskGpio0] Value written to register = 0x%x bit = %d value = 0x%x", REG_EMERALD_ENV_HW_EVENTS2A_INT_EN, HW_EVENT_EXTERNAL_IRQ_GPIO_10,  0);
#endif	
}

void PtaGpioInitGpioState(void)
{
	/* Set GPIO2 as OUT */
	PtaGpioDirectionGrant(PTA_GPIO_DIRECTION_OUT);
	/* Set GPIO7 as OUT */
	PtaGpioDirectionWlanDeny(PTA_GPIO_DIRECTION_OUT);
	
	/* Set GRANT as HIGH */
	PtaGpioWrite(PTA_GPIO_NAME_GRANT, PTA_GPIO_VAL_HIGH);
	/* Set WLAN_DENY as HIGH */
	PtaGpioWriteWlanDeny(PTA_GPIO_VAL_HIGH);	
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaGpioInitGpioState] setting GRANT as HIGH and WLAN_DENY as HIGH");
#endif
}

void ptaGpioLatchClearGpio0(void)
{
	/* Clear GPIO0 latch (in wave600 GPIO was replaced with GPIO10) */
	InterruptManager_ClearMacGeneralEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_GPIO_10);
#ifdef PTA_TRACE
	ILOG0_V("[PTA][ptaGpioLatchClearGpio0] Latch cleared");
#endif	
}

void ptaGpioUnmsakGpio0(void)
{
	/* GPIO0 is part of the "less important" interrupts */
	HwEvents_UnMaskEvent_Common_A(HW_EVENT_EXTERNAL_IRQ_GPIO_10);
	
#ifdef PTA_TRACE
	ILOG0_DDD("[PTA][ptaGpioUnmsakGpio0] Value written to register = 0x%x bit = %d value = 0x%x", REG_EMERALD_ENV_HW_EVENTS2A_INT_EN, HW_EVENT_EXTERNAL_IRQ_GPIO_10,  1);
#endif		
}

void PtaGpioBtAllow(void)
{
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaGpioBtAllow]");
#endif

	/* Wave 600: set WLAN_DENY as ALLOW and GRANT as DENY */
	PtaGpioWriteWlanDeny(PTA_GPIO_BT_ALLOW);
	PtaGpioWriteGrant(PTA_GPIO_ZIGBEE_DENY);	
}

void PtaGpioBtDeny(void)
{
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaGpioBtDeny]");
#endif

	/* Wave 600: set WLAN_DENY as DENY and GRANT as LOW */
	PtaGpioWriteGrant(PTA_GPIO_VAL_LOW);
	PtaGpioWriteWlanDeny(PTA_GPIO_BT_DENY);
}

void PtaGpioZigBeeAllow(void)
{
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaGpioZigBeeAllow]");
#endif

	/* Wave 500B: set GRANT as ALLOW and WLAN_DENY as DENY */
	PtaGpioWriteGrant(PTA_GPIO_ZIGBEE_ALLOW);
	PtaGpioWriteWlanDeny(PTA_GPIO_BT_DENY);
}

void PtaGpioZigBeeDeny(void)
{
#ifdef PTA_TRACE
	ILOG0_V("[PTA][PtaGpioZigBeeDeny]");
#endif

	/* Wave 500: set GRANT as DENY and WLAN_DENY as DENY */
	PtaGpioWriteGrant(PTA_GPIO_ZIGBEE_DENY);
	PtaGpioWriteWlanDeny(PTA_GPIO_BT_DENY);
}

    
/***********************************************************************
* InterruptManagerConfigureMacGeneraExternalIrqInvert
* 
* Description:
* ------------
* External interrupt may be triggered due LOW->HIGH or HIGH->LOW transission. We need to set it to LOW->HIGH.
* 
* Input:
* ------
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PtaInterruptManagerConfigureMacGeneraExternalIrqInvert(void)
{
	InterruptManagerConfigureMacGeneraExternalIrqInvert(EDGED_TRIGGERED_INT_GPIO_10, EXTERNAL_IRQ_INVERT_HIGH);
}

#endif
