/***********************************************************************************
 File:			TxPauser.c
 Module:		Pauser
 Purpose: 		To create a driver for the Tx pauser which pauses VAPs/ACs Tx
 Description:	This file is the implementation of the Tx Pauser module 
				which is responsible of pausing VAPs/ACs initiated transmissions
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Pauser_Api.h"
#include "TxHandlerRegs.h"
#include "ErrorHandler_Api.h"
#include "RegAccess_Api.h"
#include "ServicesHandler_Api.h"
#include "HwGlobalDefinitions.h"
#include "Pac_Api.h"
#include "loggerAPI.h"
#include "MacHtExtensionsRegs.h"
#include "AutoRespRegs.h"
#include "PacRxcRegs.h"
#include "mt_sysrst.h"

typedef struct pacRxcDbg
{
	uint32 							requestType;
	RegPacRxcPhyMacDbg_u 			pacRxcDbgReg;
	RegAutoRespAutoRespStatDebug_u	autoRespAutoRespStatDebug;
	RegAutoRespAutoRespSmStatDebug_u	autoRespAutoRespSmStatDebug;
	RegPacRxcRxcState_u				pacRxcRxcState;
	uint32 							TSF;

}pacRxcDbg_t;


pacRxcDbg_t pacRxcDbg[16];

uint32 pacRxcDbgInd =0;

//Yalva

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_PAUSER 
#define LOG_LOCAL_FID 1

#define RX_PAUSE_TIMER 10
#define PAUSER_MAX_WAITING_ITERATIONS_FOR_RX_PAUSE_REQ	((200/RX_PAUSE_TIMER) + 1)
#define PAUSER_MAX_RX_PAUSE_REQUEST_DURATION_ALLOWED 	(100000) /* 100 ms*/
#define PAUSER_RX_RESUME_BIT_VALUE						(0x0)
#define PAUSER_RX_PAUSE_BIT_VALUE						(0x1)
#define PAUSER_RX_PAUSED								(0x0)
#define PAUSER_RX_RESUMED								(0x1)



typedef enum 
{
	RX_PAUSER_STATE_IDLE 				= 0,
	RX_PAUSER_STATE_WAITING_PAUSE_CFM,
	RX_PAUSER_STATE_WAITING_RESUME_CFM,
} RxPauserState_e;

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static const uint8 acBitmapForPauseAc[TX_PAUSE_NUM_OF_AC_TYPES] = 
{
	((1 << HW_AC_BE) | (1 << HW_AC_BK) | (1 << HW_AC_VI) | (1 << HW_AC_VO) |  (1 << HW_AC_GPLP)),						/* TX_PAUSE_ALL_ACS_EXCEPT_BEACON_AND_GPHP */
	((1 << HW_AC_BE) | (1 << HW_AC_BK) | (1 << HW_AC_VI) | (1 << HW_AC_VO) |  (1 << HW_AC_GPHP)),						/* TX_PAUSE_ALL_ACS_EXCEPT_BEACON_AND_GPLP */
	((1 << HW_AC_BE) | (1 << HW_AC_BK) | (1 << HW_AC_VI) | (1 << HW_AC_VO) | (1 << HW_AC_GPHP) | (1 << HW_AC_GPLP)),	/* TX_PAUSE_ALL_ACS_EXCEPT_BEACON */
	((1 << HW_AC_BE) | (1 << HW_AC_BK) | (1 << HW_AC_VI) | (1 << HW_AC_VO) | (1 << HW_AC_GPHP) | (1 << HW_AC_BEACON)),	/* TX_PAUSE_ALL_ACS_EXCEPT_GPLP */
	((1 << HW_AC_BE) | (1 << HW_AC_BK) | (1 << HW_AC_VI) | (1 << HW_AC_VO) | (1 << HW_AC_GPLP) | (1 << HW_AC_BEACON)),	/* TX_PAUSE_ALL_ACS_EXCEPT_GPHP */
	((1 << HW_AC_GPLP)),																								/* TX_PAUSE_AC_GPLP */
};

typedef struct 
{
	uint8 rxPauserState;
	uint8 rxPasueCfmBitValue;
	uint8 rxPauserCurrentstate;
} txPauserDb_t;

txPauserDb_t txPauserDb= {0,0,PAUSER_RX_RESUMED};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

isr_TxPauser_PauseExecuted   


Description:
------------
handle the interupt from the Tx pauser - pause was complete

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

ISR_VOID isr_TxPauser_PauseExecuted(void)
{	
	RegTxHandlerTxhPauserClearIrq_u clearInterruptRegister;

	clearInterruptRegister.val = 0;

	clearInterruptRegister.bitFields.txhPauserClearIrq = TRUE;
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_CLEAR_IRQ, clearInterruptRegister.val);

	// to complete - call back to the Service handler module
	ServicesHandler_PauserCB(SERVICES_STAUTS_PASSED);
}


/**********************************************************************************

isr_TxPauser_RxPauseExecuted


Description:
------------
handle the interrupt from the Rx pauser - pause was complete

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void isr_TxPauser_RxPauseExecuted(void)
{	
	// Clear the RX pause/resume Interrupt
	Pac_RxcClearRxHaltIrq();

	if (txPauserDb.rxPauserState == RX_PAUSER_STATE_WAITING_RESUME_CFM)
	{
		TxPauser_ReadDebug(10);
	}
	else
	{
		TxPauser_ReadDebug(11);
	}

	
	//Rx halt should not be prompted when Rx Pauser is in idle state
	if (txPauserDb.rxPauserState != RX_PAUSER_STATE_IDLE)
	{
		txPauserDb.rxPauserState = RX_PAUSER_STATE_IDLE;

		// to complete - call back to the Service handler module
		ServicesHandler_PauserCB(SERVICES_STAUTS_PASSED);
	}
}


/**********************************************************************************

TxPauser_PauseOrResume 


Description:
------------
pause or resume specific Vaps or ACs

Input:
-----
	pauseResumeParams - a pointer to the pausing parameters structure
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxPauser_PauseVap(TxPauserPauseResumeParams_t *pauseResumeParams)
{

	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;

	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	DEBUG_ASSERT(pauseResumeParams->commandType <= TX_PAUSER_LAST_REGULAR_PAUSE_RESUME_COMMAND);

	commandRegister.bitFields.txhPauserCommand = TX_PAUSER_PAUSE_VAPS;
	regTxHandlerTxhPauserCommandVapVector.bitFields.txhPauserVapVector = 1 << (pauseResumeParams->vapIndex);

	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);

}

void TxPauser_ResumeVap(TxPauserPauseResumeParams_t *pauseResumeParams)
{
	
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;

	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	DEBUG_ASSERT(pauseResumeParams->commandType <= TX_PAUSER_LAST_REGULAR_PAUSE_RESUME_COMMAND);

	commandRegister.bitFields.txhPauserCommand = TX_PAUSER_RESUME_VAPS;
	regTxHandlerTxhPauserCommandVapVector.bitFields.txhPauserVapVector = 1 << (pauseResumeParams->vapIndex);

	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);
}

void TxPauser_PauseAc(TxPauserPauseResumeParams_t *pauseResumeParams)
{
	
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;

	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	DEBUG_ASSERT(pauseResumeParams->commandType <= TX_PAUSER_LAST_REGULAR_PAUSE_RESUME_COMMAND);

	commandRegister.bitFields.txhPauserCommand = TX_PAUSER_PAUSE_ACS;
	commandRegister.bitFields.txhPauserAcVector = acBitmapForPauseAc[pauseResumeParams->pauseAcType];

	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);
	
}

void TxPauser_ResumeAc(TxPauserPauseResumeParams_t *pauseResumeParams)
{
	
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;

	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	DEBUG_ASSERT(pauseResumeParams->commandType <= TX_PAUSER_LAST_REGULAR_PAUSE_RESUME_COMMAND);

	commandRegister.bitFields.txhPauserCommand = TX_PAUSER_RESUME_ACS;
	commandRegister.bitFields.txhPauserAcVector = acBitmapForPauseAc[pauseResumeParams->pauseAcType];

	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);

}


void TxPauser_PauseAll(TxPauserPauseResumeParams_t *pauseResumeParams)
{
	
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;

	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	DEBUG_ASSERT(pauseResumeParams->commandType <= TX_PAUSER_LAST_REGULAR_PAUSE_RESUME_COMMAND);

	commandRegister.bitFields.txhPauserCommand = TX_PAUSER_PAUSE_ALL;

	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);

}

void TxPauser_ResumeAll(TxPauserPauseResumeParams_t *pauseResumeParams)
{
	
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;

	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	DEBUG_ASSERT(pauseResumeParams->commandType <= TX_PAUSER_LAST_REGULAR_PAUSE_RESUME_COMMAND);

	commandRegister.bitFields.txhPauserCommand = TX_PAUSER_RESUME_ALL;

	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);

}

/**********************************************************************************

TxPauser_StartRepetitionMode 


Description:
------------
start repetition mode

Input:
-----
	repetitionParams - a pointer to the repetition parameters structure
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxPauser_StartRepetitionMode(TxPauserRepetitionParams_t *repetitionParams)
{
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;
	RegTxHandlerTxhPauserRepCycleCount_u repetitionCycleCounterRegister;
	RegTxHandlerTxhPauserRepDuration_u repetitionDurationRegister;
	
	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;
	repetitionCycleCounterRegister.val = 0;
	repetitionDurationRegister.val = 0;

	DEBUG_ASSERT(repetitionParams->repetitionType  < TX_PAUSER_REPETITION_NUM_OF_TYPES);
		
	repetitionCycleCounterRegister.bitFields.txhPauserRepCycleCount = repetitionParams->cycleCount;
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_REP_CYCLE_COUNT, repetitionCycleCounterRegister.val);
	
	repetitionDurationRegister.bitFields.txhPauserRepPauseDur = repetitionParams->pauseDuration;
	repetitionDurationRegister.bitFields.txhPauserRepResumeDur = repetitionParams->resumeDuration;
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_REP_DURATION, repetitionDurationRegister.val);

	commandRegister.bitFields.txhPauserCommand  = TX_PAUSER_START_REPETITION;
	commandRegister.bitFields.txhPauserRepetitionType = repetitionParams->repetitionType;
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);
}

/**********************************************************************************

TxPauser_EndRepetitionMode 


Description:
------------
end repetition mode


Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxPauser_EndRepetitionMode(void)
{
	RegTxHandlerTxhPauserCommand_u commandRegister;
	RegTxHandlerTxhPauserCommandVapVector_u regTxHandlerTxhPauserCommandVapVector;
	
	commandRegister.val = 0;
	regTxHandlerTxhPauserCommandVapVector.val = 0;

	commandRegister.bitFields.txhPauserCommand  = TX_PAUSER_END_REPETITION;
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND_VAP_VECTOR, regTxHandlerTxhPauserCommandVapVector.val);
	RegAccess_Write(REG_TX_HANDLER_TXH_PAUSER_COMMAND, commandRegister.val);
}


/**********************************************************************************
TxPauser_PauseRx 


Description:
------------
do Rx Pause

Input:
-----
		
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/



void TxPauser_PauseRx(void)
{

	TxPauser_ReadDebug(0);
	
	if	(txPauserDb.rxPauserCurrentstate == PAUSER_RX_PAUSED)
	{
		ASSERT(0);
	}
	else
	{
		// Change state to wait for pause confirm
		txPauserDb.rxPauserState = RX_PAUSER_STATE_WAITING_PAUSE_CFM;
		txPauserDb.rxPasueCfmBitValue = PAUSER_RX_PAUSE_BIT_VALUE;
		// Call RXC driver to pause the RX. An Interrupt will be triggered when pause is done.
		Pac_RxcPauseRx();
		txPauserDb.rxPauserCurrentstate = PAUSER_RX_PAUSED;
	}
}


/**********************************************************************************
TxPauser_ResumeRx 


Description:
------------
Do Rx resume

Input:
-----
		
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
void TxPauser_ResumeRx(void)
{


	TxPauser_ReadDebug(1);
	if	(txPauserDb.rxPauserCurrentstate == PAUSER_RX_RESUMED)
	{
		ASSERT(0);
	}
	else
	{
		// Change state to wait for resume confirm
		txPauserDb.rxPauserState = RX_PAUSER_STATE_WAITING_RESUME_CFM;
		txPauserDb.rxPasueCfmBitValue = PAUSER_RX_RESUME_BIT_VALUE;
		// Call RXC driver to resume the RX. An Interrupt will be triggered when resume is done.
		Pac_RxcResumeRx();
		txPauserDb.rxPauserCurrentstate = PAUSER_RX_RESUMED;
	}
}


/**********************************************************************************

TxPauser_Initialize 


Description:
------------
initialize internal structures and registers of the Tx Pauser module 
Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxPauser_Initialize(void)
{




}

void TxPauser_ReadDebug(uint32 type)
{
	RegAccess_Read(REG_PAC_RXC_PHY_MAC_DBG,&pacRxcDbg[pacRxcDbgInd].pacRxcDbgReg.val);
	RegAccess_Read(REG_AUTO_RESP_AUTO_RESP_STAT_DEBUG,&pacRxcDbg[pacRxcDbgInd].autoRespAutoRespStatDebug.val);
	RegAccess_Read(REG_PAC_RXC_RXC_STATE,&pacRxcDbg[pacRxcDbgInd].pacRxcRxcState.val);
	RegAccess_Read(REG_AUTO_RESP_AUTO_RESP_SM_STAT_DEBUG,&pacRxcDbg[pacRxcDbgInd].autoRespAutoRespSmStatDebug.val);

	pacRxcDbg[pacRxcDbgInd].TSF = Pac_TimGetTsfLow();

	pacRxcDbg[pacRxcDbgInd].requestType = type;

	pacRxcDbgInd++;

	if (pacRxcDbgInd == 16)
	{
		pacRxcDbgInd =0;
	}


}

