/***********************************************************************************
 File:			PlanManager.c
 Module:		Plan Manager
 Purpose: 		To allocate HW mu resources and set their parameters 
 Description:   This file is the implementation of the plan manager
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "linkAdaptation_api.h"
#include "LinkAdaptation.h"
#include "Locker_Api.h"
#include "Pac_Api.h"
#include "queue_utility.h"
#include "StaDatabase_Api.h"
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "SharedDbTypes.h"
#include "CommonPlan_Descriptors.h"
#include "ShramPlannerResources.h"
#include "VapDb_Descriptors.h"
#include "ShramVapDatabase.h"
#include "Rcr_Descriptors.h"
#include "HeGroupManager_API.h"
#include "PlanManager.h"
#include "Utils_Api.h"
#include "TsManager_API.h"
#include "ShramStationDatabase.h"
#include "PlanManager_API.h"
#include "PreAggregator_Api.h"
#include "RegAccess_Api.h"
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
#include "PreAggRegs.h"
#endif 
#ifdef TF_IN_HE_SU_FOR_PF
#include "TxSender_ScratchPadApi.h"
#endif
#include "loggerAPI.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "NfrpPhase_Descriptors.h"
#include "DescriptorsDefinitions.h"
#endif

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HE_PLAN_MANAGER
#define LOG_LOCAL_FID 0


#define	PLAN_MANAGER_INVALID_USP_OFFSET			0xFF
#define PROTECTION_PHASE_FIRST_OCCURENCE		(1 << PROTECTION_PHASE_BIT_INDEX)
#define SOUNDING_PHASE_FIRST_OCCURENCE			(1 << SOUNDING_PHASE_BIT_INDEX)
#define BSRP_PHASE_FIRST_OCCURENCE				(1 << BSRP_PHASE_BIT_INDEX)
#define DATA_PHASE_FIRST_OCCURENCE				(1 << DATA_PHASE_BIT_INDEX)
#ifdef ENET_INC_ARCH_WAVE600D2
#define NFRP_PHASE_FIRST_OCCURENCE				(1 << NFRP_PHASE_BIT_INDEX)
#define	NUMBER_OF_DATA_PHASES_IN_PLAN			(MAX_NUM_OF_PHASES_IN_A_PLAN - NFRP_PHASE_BIT_INDEX) //max number of phases - (protection+sounding+BSRP+ data + nfrp) 
#define LSIG_LENGHT_FOR_NFRP 					0x22
#else
#define	NUMBER_OF_DATA_PHASES_IN_PLAN			(MAX_NUM_OF_PHASES_IN_A_PLAN - DATA_PHASE_BIT_INDEX) //max number of phases - (protection+sounding+BSRP+ data + nfrp) 
#endif
#define	LSIG_LENGHT_FOR_8_TIDS_64_MU_BA_SIZE	265
#define	LSIG_LENGHT_FOR_UL						4045 // equivalent to 5430usec
#define TWENTY_MGHZ 							20 
#define NA_VALUE_FOR_FIELD_IN_PHASE 			0 
#define THIRD_BIT_MASK							0x4 
#define FIRST_4_BITS_MASK						0xF  
#define RU_ALLOCATION_MASK                      FIRST_4_BITS_MASK
#define RU_ALLOCATION_SHIFT                     4
#define RU_ALLOCATION_B_MASK                    0x7
#define RU_ALLOCATION_B_SHIFT                   3
#define SIZE_FROM_WORD0_TO_WORD4				CONVERT_WORDS_TO_BYTES(0x4)  //not including word4
#define DEBUG_RU_ALLOCATOR

/*
M-BA Size								
Max Bitmap Used (max BA Agreement)	L-SIG Length 1 TID	L-SIG Length 2 TID	L-SIG Length 3 TID	L-SIG Length 4 TID	L-SIG Length 5 TID	L-SIG Length 6 TID	L-SIG Length 7 TID	L-SIG Length 8 TID
		64									58				70						82				91					91					103						112					124
		128									70				82						91				112					124					145						157					166
		256									82				103						136				157					190					211						244					265
*/


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define	TURN_X_LSBITS_ON(x)					(~(1<<x))

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
// RU Allocation defines
#define RU_ALLOC_MASK_SIZE					(2)
#define MAX_NUM_OF_80MHz_BANDS				(2)

// bitmap allocation per BW:
#define BITMAP_ALLOC_BW80_1_INIT			(0x00040000)
#define BITMAP_ALLOC_BW80_2_INIT			(0xFFFFFFE0)
#define BITMAP_ALLOC_BW80_1_40MHZ			(0xFFFC0000)
#define BITMAP_ALLOC_BW80_2_40MHZ			(0xFFFFFFFF)
#define BITMAP_ALLOC_BW80_1_20MHZ			(0xFFFFFE00)
#define BITMAP_ALLOC_BW80_2_20MHZ			(0xFFFFFFFF)

// bitmap allocation per RU:
#define BITMAP_ALLOC_RU_26_1				(BITMAP_ALLOC_BW80_1_INIT)
#define BITMAP_ALLOC_RU_26_2			    (BITMAP_ALLOC_BW80_2_INIT)
#define BITMAP_ALLOC_RU_52_1				(0xAAD6B55A)
#define BITMAP_ALLOC_RU_52_2				(0xFFFFFFF5)
#define BITMAP_ALLOC_RU_106_1				(0xEEF7BDDE)
#define BITMAP_ALLOC_RU_106_2				(0xFFFFFFFD)
#define BITMAP_ALLOC_RU_242_1				(0xEFF7FDFE)
#define BITMAP_ALLOC_RU_242_2				(0xFFFFFFFF)
#define BITMAP_ALLOC_RU_484_1				(0xFFF7FFFE)
#define BITMAP_ALLOC_RU_484_2				(0xFFFFFFFF)
#define BITMAP_ALLOC_RU_996_1				(0xFFFFFFFE)
#define BITMAP_ALLOC_RU_996_2				(0xFFFFFFFF)

// last bit indexs for each SB
#define FIRST_BIT_INDEX_SB1					(9)
#define FIRST_BIT_INDEX_SB2					(19)
#define FIRST_BIT_INDEX_SB3					(28)
#define FIRST_BIT_INDEX_SB4					(37)
#define FIRST_BIT_INDEX_SB5					(46)
#define FIRST_BIT_INDEX_SB6					(56)
#define FIRST_BIT_INDEX_SB7					(65)
#define LAST_BIT_INDEX_SB7					(73)
#define RU_ALLOC_INVALID_BIT_INDEX			(99)
#define SIZE_OF_RU_ALLOC_BITMAP				(64) //size of (ruIndexMask64bit)

#define SOUNDING_PHASE_VALIDITY_MASK		(SOUNDING_PHASE_FIRST_OCCURENCE)
#define PROTECTION_PHASE_VALIDITY_MASK		(PROTECTION_PHASE_FIRST_OCCURENCE)
#define BSRP_PHASE_VALIDITY_MASK			(BSRP_PHASE_FIRST_OCCURENCE)
#ifdef ENET_INC_ARCH_WAVE600D2
#define NFRP_PHASE_VALIDITY_MASK			(NFRP_PHASE_FIRST_OCCURENCE)
#endif
#define SIZE_OF_SOUNDING_PHASE_STARTING_FROM_WORD5	(sizeof(SoundingPhase_t)-CONVERT_WORDS_TO_BYTES(0x5)) 
#define MIN_RU_SIZE_MH 2
/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum
{	
	ALLOCATION_SUCCESS = 0,
	PLAN_ALLOCATION_FAILED,
	PROTECTION_PHASE_ALLOCATION_FAILED,
	SOUNDING_PHASE_ALLOCATION_FAILED,
	BSRP_PHASE_ALLOCATION_FAILED,
	DATA_PHASE_ALLOCATION_FAILED,
	DATA_MIROR_PHASE_ALLOCATION_FAILED,
	RU_ALLOCATION_FAILED,
#ifdef ENET_INC_ARCH_WAVE600D2
	NFRP_PHASE_ALLOCATION_FAILED,
#endif
} PlanManagerAllocationStatus_e;

typedef struct _RuAllocationPerUser_t
{ 
	/* RU ALLOCATION PER USER */
	uint32	ul_SubBand  		: 3;  //bits [0..2]
	uint32	ul_StartRU    		: 4;  //bits [3..6]
	uint32	userRcrRuSize 	  	: 3;  //bits [7..9]
	
	uint32	dl_subBandPerUsp	: 3;  //bits [0..2]
	uint32	dl_startRuPerUsp	: 4;  //bits [3..6]
	uint32	dl_ruSizePerUsp		: 3;  //bits [7..9]

	uint32	reserved			: 12; 
} RuAllocationPerUser_t;

typedef struct RuAllocMask
{
	uint32 ruIndexMask64bit[RU_ALLOC_MASK_SIZE];
}RuAllocMask_t;
typedef struct RuSizeMap_t
{
	uint8 numberOfRusPerSubband;
	uint8 numberOfSubbandsForMinAllocatedBw;
	uint8 ruIndexPerSubband[NUM_OF_26_TONES_IN_A_SUB_BAND];
}RuSizeMap_t;
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
#ifdef OFDMA_DYNAMIC_PLAN_ON
static 	void							planManagerCreatePlan(K_MSG* groupManagerMessage); 
#endif
static 	void 							planManagerCreateStaticPlan(K_MSG* psMsg); 
static 	void 							planManagerRemoveStaticPlan(K_MSG* psMsg); 
static 	void 							planManagerRemovePlan(K_MSG* groupManagerMessage); 
static 	CommonPlan_t* 					planManagerFetchPlanFromPool(void); 
static 	void*							planManagerFetchPhaseFromPool(PhaseFormat_e phaseFormat);
static 	void 							planManagerReturnPhaseToPool(void* phasePtr, PhaseFormat_e phaseFormat);
static 	void 							planManagerReturnPlanToPool(CommonPlan_t* planPtr);
static 	void							planManagerSetStaticDlPhaseCommonFields(DlDataPhaseCommon_t* phaseCommonPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr); 
static 	void 							planManagerSetStaticDlPhaseCommonFieldValidUser(DlDataPhaseCommon_t* phaseCommonPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticDlPhaseCommonTcr_A_Fields(DlDataPhaseCommonTcra_t*  phaseCommonTcraPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr,uint8 vapId);
static 	void 							planManagerSetStaticDlPhaseCommonTcr_B_Fields(DlDataPhaseCommonTcrb_t*  phaseCommonTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr,uint8 vapId);
static 	void 							planManagerSetStaticDlPhaseCommonRcrFields(DlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId);
static 	void 							planManagerSetStaticDlPhaseCommonUserTCR_B_Fields(DlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticDlPhaseCommonTfParameters(DlDataPhase_t* dlDataPhasPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticDlPhaseCommonTfMuBar(DlDataPhaseTfMuBar_t* dlDataPhasPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticDlPhasePerUser(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, Bw_e bw);
static	void							planManagerSetStaticDlPhasePerUserWithUnEqualRus(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, Bw_e bw);
static 	bool 							planManagerAreUsersRuAllocationsEqual(UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static  void 							planManagerSetStaticDlMimoPhasePerUser(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticDlPhaseTfInfoPerUser(DlDataPhaseTfUserInfo_t* dlDataPhasTfInfoPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr); 

#ifdef OFDMA_DYNAMIC_PLAN_ON
static 	void							planManagerSetDlPhaseCommonFields(DlDataPhaseCommon_t* phaseCommonPtr,CreatePlanMsg_t* pMuCreatePlanReq, HeMuSequence_e muSequenceType); 
static 	void 							planManagerSetDlPhaseCommonTcr_A_Fields(DlDataPhaseCommonTcra_t*  phaseCommonTcraPtr,CreatePlanMsg_t* pMuCreatePlanReq,CommonPlan_t* planPtr, uint8 vapId, HeMuSequence_e muSequenceType);
static 	void 							planManagerSetDlPhaseCommonTcr_B_Fields(DlDataPhaseCommonTcrb_t*  phaseCommonTcrbPtr,CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t*  planPtr,uint8 vapId);
static 	void 							planManagerSetDlPhaseCommonRcrFields(DlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,CreatePlanMsg_t* pMuCreatePlanReq, uint8 vapId);
static 	void 							planManagerSetDlPhaseCommonUserTCR_B_Fields(DlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr, RuAllocationPerUser_t* ruAllocation);
static 	void 							planManagerSetDlPhaseCommonTfParameters(DlDataPhase_t* dlDataPhasPtr,CreatePlanMsg_t* pMuCreatePlanReq);
static 	void 							planManagerSetDlPhaseCommonTfMuBar(DlDataPhaseTfMuBar_t* dlDataPhasTfMuBarPtr);
static 	void 							planManagerSetDlPhasePerUser(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr,CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t*	ruAllocation, bool plannedBar, uint8 allocatedUsers);
static 	void 							planManagerSetDlPhaseTfInfoPerUser(DlDataPhaseTfUserInfo_t* dlDataPhasTfInfoPerUserPtr,CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation); 
static  void                            planManagerSetDlPhaseCommonFieldValidUser(DlDataPhaseCommon_t* phaseCommonPtr, CreatePlanMsg_t* pMuCreatePlanReq, uint8 allocatedUsers);
static  void                            planManagerSetUlPhase(UlDataPhase_t* ulDataPhasePointer, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId, RuAllocationPerUser_t* ruAllocation, HeMuSequence_e muSequenceType);
static 	void 							planManagerSetUlPhaseCommonFields(UlDataPhaseCommon_t* phaseCommonPtr, CreatePlanMsg_t* pMuCreatePlanReq, HeMuSequence_e muSequenceType);
static 	void 							planManagerSetUlPhaseCommonTcr_A_Fields(UlDataPhaseCommonTcra_t*  phaseCommonTcraPtr,CreatePlanMsg_t* pMuCreatePlanReq,CommonPlan_t* planPtr,uint8 vapId);
static 	void 							planManagerSetUlPhaseCommonTcr_B_Fields(UlDataPhaseCommonTcrb_t*  phaseCommonTcrbPtr,CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t*  planPtr, uint8 vapId);
static 	void 							planManagerSetUlPhaseCommonRcrFields(UlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,CreatePlanMsg_t* pMuCreatePlanReq, uint8 vapId);
static 	void 							planManagerSetUlPhaseCommonUserTCR_A_Fields(UlDataPhaseCommonUserTcra_t* commonUserTcrbPtr);
static 	void 							planManagerSetUlPhaseCommonUserTCR_B_Fields(UlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr);
static 	void 							planManagerSetUlPhaseCommonTfParameters(UlDataPhase_t* ulDataPhasPtr,CreatePlanMsg_t* pMuCreatePlanReq, uint8 muSequenceType);
static 	void 							planManagerSetUlPhasePerUser(UlDataPhaseUserPhase_t* ulDataPhasPerUserPtr, CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation, uint8 muSequenceType);
static 	void 							planManagerSetUlPhaseTfInfoPerUser(UlDataPhaseTfUserInfo_t* ulDataPhasTfInfoPerUserPtr,CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation);
static  void							planManagerSetNssPerUser(LaAddHeGroupParams_t* 	laAddHeGroupParameters, uint8 groupId); 
static  void 							planManagerSetProtectionPhase(ProtectionPhase_t* protectionPhasePtr,uint8 groupId, uint8 bw, RuAllocationPerUser_t* ruAllocationPerUser , uint8 vapId);
static  void 							planManagerSetSoundingPhase(SoundingPhase_t* soundingPhasePtr,CreatePlanMsg_t* pMuCreatePlanReq);
static PlanManagerAllocationStatus_e planManagerRuAllocator_GetAllocationForHeGroup(RuAllocationPerUser_t* ruAllocationArrayPerUser, uint8 numOfValidUsers, Bw_e bw, uint8* allocatedUsers);
#endif // OFDMA_DYNAMIC_PLAN_ON

static void 							planManagerSetStaticUlPhase(UlDataPhase_t* ulOrBsrpPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t*  planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId);
static 	void 							planManagerSetStaticUlPhaseCommonFields(UlDataPhaseCommon_t* phaseCommonPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticUlPhaseCommonTcr_A_Fields(UlDataPhaseCommonTcra_t*  phaseCommonTcraPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr,CommonPlan_t*  planPtr, uint8 vapId);
static 	void 							planManagerSetStaticUlPhaseCommonTcr_B_Fields(UlDataPhaseCommonTcrb_t*  phaseCommonTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr, uint8 vapId);
static 	void 							planManagerSetStaticUlPhaseCommonRcrFields(UlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId);
static 	void 							planManagerSetStaticUlPhaseCommonUserTCR_A_Fields(UlDataPhaseCommonUserTcra_t* commonUserTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticUlPhaseCommonUserTCR_B_Fields(UlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static void 							planManagerSetStaticUlPhaseCommonTfParameters(UlDataPhaseTfParameters_t* commonTfParamsPtr, UlDataPhaseTfCommonInfo_t* commonTfInfoPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticUlPhasePerUser(UlDataPhaseUserPhase_t* ulDataPhasPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticUlPhaseTfInfoPerUser(UlDataPhaseTfUserInfo_t* ulDataPhasTfInfoPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticSoundingPhase(SoundingPhase_t* soundingPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t*  planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId);
static 	void 							planManagerSetStaticSoundingPhaseCommonFields(SoundingPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, HeGroupManagerStaticGroupId_e groupId, uint8 vapId);
static void 							planManagerSetStaticSoundingPhaseNdpaPointer(UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 groupIndex);
static 	void 							planManagerSetStaticSoundingPhaseCommonTcr_A_Fields(SoundingPhaseCommonTcra_t* phaseCommonTcraPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr, uint8 vapId);
static 	void 							planManagerSetStaticSoundingPhaseCommonTcr_B_Fields(SoundingPhaseCommonTcrb_t* phaseCommonTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr, uint8 vapId);
static 	void 							planManagerSetStaticSoundingPhaseCommonRcrFields(SoundingPhaseCommonRcr_t* phaseCommonRcrPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId);
static  void 							planManagerSetStaticSoundingPhaseCommonUserTCR_A_Fields(SoundingPhaseCommonUserTcra_t* commonUserTcraPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static  void 							planManagerSetStaticSoundingPhaseCommonUserTCR_B_Fields(SoundingPhaseCommonUserTcrb_t* commonUserTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticSoundingPhasePerUser(SoundingPhaseUserPhase_t* soundingPhasPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetStaticSoundingCommonTfParameters(SoundingPhase_t* soundingPhasePtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static  void 							planManagerSetStaticSoundingPhaseTfInfoPerUser(SoundingPhaseTfUserInfo_t* soundingPhasTfInfoPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static  void 							planManagerSetStaticProtectionPhase(ProtectionPhase_t* protectionPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t*  planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId);
static  void 							planManagerSetStaticProtectionPhaseCommonFields(ProtectionPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, HeGroupManagerStaticGroupId_e groupId);
static  void							planManagerSetStaticProtectionPhaseCommonTcr_B_Fields(ProtectionPhaseCommonTcrb_t*  phaseCommonTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr, uint8 vapId);
static  void 							planManagerSetStaticProtectionPhaseCommonRcrFields(ProtectionPhaseCommonRcr_t* phaseCommonRcrPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId);
static  void 							planManagerSetStaticProtectionPhaseCommonUserTCR_B_Fields(ProtectionPhaseCommonUserTcrb_t* commonUserTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static  void 							planManagerSetStaticProtectionPhasePerUser(ProtectionPhaseUserPhase_t* protectionPhasePerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static  void 							planManagerSetStaticProtectionCommonTfParameters(ProtectionPhase_t* protectionPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig);
static  void 							planManagerSetStaticProtectionPhaseTfInfoPerUser(ProtectionPhaseTfUserInfo_t* protectionPhaseTfInfoPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static 	void 							planManagerSetPlanManagerDefaultSettings(void);
static 	void 							planManagerCreatePlanLaCfm(K_MSG* psMsg);
static 	void 							planManagerRemovePlanLaCfm(K_MSG* psMsg);
static  void 							planManagerSetStaticNssPerUser(UMI_STATIC_PLAN_CONFIG* 	pMuPlanConfig, LaAddHeGroupParams_t* laRequestParams, uint8 vapId);
static	void						 	planManagerHandleCreationFailure(K_MSG* psMsg,uint8 groupId, bool isStatic);
#ifdef ENET_INC_ARCH_WAVE600D2
static 	void 							planManagerFreeResourcesDueToFailure(CommonPlan_t* planPtr, ProtectionPhase_t* protectionPhasePtr, SoundingPhase_t* soundingPhasePtr, UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr ,NfrpPhase_t* nfrpPhasePtr );
static PlanManagerAllocationStatus_e    planManagerVerifyResourceAllocation(CommonPlan_t* planPtr ,ProtectionPhase_t* protectionPhasePtr,SoundingPhase_t* soundingPhasePtr , UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr, DlDataPhase_t* mirrorDataPhasePtr,NfrpPhase_t* nfrpPhasePtr );
#else
static 	void 							planManagerFreeResourcesDueToFailure(CommonPlan_t* planPtr, ProtectionPhase_t* protectionPhasePtr, SoundingPhase_t* soundingPhasePtr, UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr);
static 	PlanManagerAllocationStatus_e	planManagerVerifyResourceAllocation(CommonPlan_t* planPtr, ProtectionPhase_t* protectionPhasePtr, SoundingPhase_t* soundingPhasePtr, UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr, DlDataPhase_t* mirrorDataPhasePtr);
#endif
static void 							planManagerRuAllocator_updateRUAllocAccordingToBwNRuSize(PreAggRuSizeMapping_e ruSize, Bw_e bw);
static void 							planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80(PreAggRuSizeMapping_e ruSize, Bw_e bw, uint8 index80MHzBand);
static uint8 							planManagerRuAllocator_findLocationOfNextLeadingZero(uint8 lastLeadingZero);
static uint8 							planManagerRuAllocator_findLocationOfNextLeadingZeroPer80(uint8 lastLeadingZero, uint8 index80MHzBand);
static void 							planManagerRuAllocator_findSBIndexRUIndex(uint8 firstBitmapIndex, PreAggSubBandIndex_e* currentSBindex, PreAggStartRuIndex_e* currentRUindex);
static void 							planManagerRuAllocator_configureDummyUser(UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr, PreAggSubBandIndex_e currentSBindex, PreAggStartRuIndex_e currentRUindex, PreAggRuSizeMapping_e ruSize, bool ldpcDummyOn, uint8 userInputIndex);
static void 							planManagerRuAllocator_updateRuBitmap(PreAggRuSizeMapping_e ruSize, uint8 firstBitmapIndex);
static void 							planManagerRuAllocator_updateRuBitmapPer80(PreAggRuSizeMapping_e ruSize, uint8 firstBitmapIndex, uint8 index80MHzBand);

#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
static	bool 							planManagerCalcRuAllocationSecondaryBit(uint8 bw,uint8 ulSubBand,uint8 ulRuSize); 
static 	bool 							planManagerIsPrimarySubBandInLower80(void);
#endif //TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND

#ifdef ENET_INC_ARCH_WAVE600B
static 	void 							planManagerAddPhaseToStaticPlan(K_MSG* psMsg);
static void 							planManagerAddPhaseToStaticPlanPostLaRemove(K_MSG* psMsg, uint8 groupId);
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2
static void                             planManagerSetNfrpPhase(NfrpPhase_t* nfrpPhasePtr, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId, uint8 groupId);
static void 							planManagerSetStaticNfrpPhase(NfrpPhase_t* nfrpPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t* planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId);
static void 							planManagerSetStaticNfrpPhaseCommonFields(NfrpPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, HeGroupManagerStaticGroupId_e groupId);
static void								planManagerSetStaticNfrpPhasePerUser(NfrpPhaseUserPhase_t* phasPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);
static void 							planManagerSetStaticNfrpPhaseCommonRcrFields(RcrCommon_t*	 phaseCommonRcrPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId);
static void 							planManagerSetStaticNfrpPhaseTfInfoPerUser(NfrpPhaseTfUserInfo_t* tfInfoPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr);

#endif
static PreAggRuSizeMapping_e searchClosestRuSize(uint8 bwVal);

void PlanManager_TaskEntry(K_MSG *planManagerMessage);

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

HeMuResourcePools_t		heMuResourcesPool;
PlanManagerStatistics_t PlanManagerStatistics;


RuAllocMask_t RuAllocBitmap[MAX_NUM_OF_80MHz_BANDS]; 	//[0] - 160MHz low 80MHz; [1] - 160MHz High 80MHz;

//'1' indicates RU size 26 was already allocated
const RuAllocMask_t RuAllocIndexMask80[MAX_POSSIBLE_NUM_OF_BW-1] = 
{
	/*    RU index36-32      */  /*    RU index31-0      */
	{BITMAP_ALLOC_BW80_1_20MHZ, BITMAP_ALLOC_BW80_2_20MHZ}, /*20MHz*/
	{BITMAP_ALLOC_BW80_1_40MHZ, BITMAP_ALLOC_BW80_2_40MHZ}, /*40MHz*/
	{BITMAP_ALLOC_BW80_1_INIT , BITMAP_ALLOC_BW80_2_INIT },	/*80MHz*/
};

RuAllocMask_t RuSizeAllocIndexsMask[PRE_AGGREGATOR_NUM_OF_RU_SIZES-1] = 
{
	{BITMAP_ALLOC_RU_26_1 , BITMAP_ALLOC_RU_26_2},
	{BITMAP_ALLOC_RU_52_1 , BITMAP_ALLOC_RU_52_2},
	{BITMAP_ALLOC_RU_106_1, BITMAP_ALLOC_RU_106_2},
	{BITMAP_ALLOC_RU_242_1, BITMAP_ALLOC_RU_242_2},
	{BITMAP_ALLOC_RU_484_1, BITMAP_ALLOC_RU_484_2},
	{BITMAP_ALLOC_RU_996_1, BITMAP_ALLOC_RU_996_2},
};

const uint32 RuSizeUserMaskLength[PRE_AGGREGATOR_NUM_OF_RU_SIZES-1] = 
{
	1,  										/*26 TONES*/
	2,											/*52 TONES*/
	4,											/*106 TONES*/
	9,											/*242 TONES*/
	18,											/*484 TONES*/
	37,											/*996 TONES*/ 
};

//according to Std table 9-76c 
const uint8 SoundingRuEndIndex[MAX_POSSIBLE_NUM_OF_BW] = 
{
	8,											/*20MHz*/
	17,											/*40MHz*/
	36,											/*80MHz*/
	73, 										/*160MHz*/
};
const uint8 bwPerRuSize[PRE_AGGREGATOR_NUM_OF_RU_SIZES] = 
{
	2,  // 2MHz - PRE_AGGREGATOR_26_TONE_RU_SIZE
	5, 	//5MHz - PRE_AGGREGATOR_52_TONE_RU_SIZE
	10,  //10 MHz- PRE_AGGREGATOR_106_TONE_RU_SIZE
	20, // 20 MHz - PRE_AGGREGATOR_242_TONE_RU_SIZE
	40,	// 40 MHz - PRE_AGGREGATOR_484_TONE_RU_SIZE
	80, //PRE_AGGREGATOR_996_TONE_RU_SIZE
	160, //PRE_AGGREGATOR_996X2_TONE_RU_SIZE,	// 160 MHz
}	;	
const uint8 bwMapToNumOfSubbands[BW_LAST] = 
{
	1,
	2,
	4,
	8
};	
const uint8 bwMapToVal[BW_LAST] = 
{
	20,
	40,
	80,
	160
};	

RuSizeMap_t RuSizeMapping[PRE_AGGREGATOR_NUM_OF_RU_SIZES] = 
{
		{8,1,{0,1,2,3,5,6,7,8,8}} ,		//   8 RUs,  1 Subband,  users RUs indexes: 0,1,2,3,5,6,7,8
		{4,1,{0,2,5,7,7,7,7,7,7}} ,		//   4 RUs,  1 Subband,  users RUs indexes: 0,2,5,7
		{2,1,{0,5,5,5,5,5,5,5,5}} ,		//   2 RUs,  1 Subband,  users RUs indexes: 0,5
		{1,1,{0,0,0,0,0,0,0,0,0}} ,		//   1 RUs,  1 Subband,  users RUs indexes: 0
		{1,2,{0,0,0,0,0,0,0,0,0}} ,		//   1 RUs,  2 Subbands, users RUs indexes: 0
		{1,4,{0,0,0,0,0,0,0,0,0}} ,		//   1 RUs,  4 Subbands, users RUs indexes: 0
		{1,8,{0,0,0,0,0,0,0,0,0}} ,		//   1 RUs,  8 Subbands, users RUs indexes: 0
};

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const FunctionEntry_t afpTaskTable[TASK_PLAN_MANAGER_END - TASK_PLAN_MANAGER_START] =
{
	{planManagerCreateStaticPlan,	       	DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_CREATE_STATIC_PLAN)}, 
	{planManagerRemoveStaticPlan,			DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_REMOVE_STATIC_PLAN)}, 
#ifdef OFDMA_DYNAMIC_PLAN_ON	
	{planManagerCreatePlan,   				DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_CREATE_PLAN)}, 
#endif	
	{planManagerRemovePlan,					DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_REMOVE_PLAN)}, 
	{planManagerCreatePlanLaCfm, 			DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_CREATE_PLAN_LA_CFM)}, 
	{planManagerRemovePlanLaCfm, 			DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_REMOVE_PLAN_LA_CFM)}, 
#ifdef ENET_INC_ARCH_WAVE600B
	{planManagerAddPhaseToStaticPlan,		DOUBLE_CHECK_MSG_TYPE(PLAN_MANAGER_ADD_PHASE_TO_STATIC_PLAN)},
#endif
};

PlanManagerDefaultSettingValuesDb		planManagerDefaultSettings = {0};

/**********************************************************************************

PlanManager_TaskEntry 


Description:
------------
the entry point of the plan manager task

Input: 
-----
planManagerMessage - pointer to the message to handle	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void PlanManager_TaskEntry(K_MSG *planManagerMessage)
{
	/* Use common task switching and Table */ 
	vTaskDispatcher(planManagerMessage, afpTaskTable, TASK_PLAN_MANAGER_START, TASK_PLAN_MANAGER_END);
}

/*---------------------------------------------------------------------------------

/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
#ifdef OFDMA_DYNAMIC_PLAN_ON
/**********************************************************************************

planManagerCreatePlan 


Description:
------------
1. if available - pull plan& phase from pool according to the phase type
2. link between plan and phases  (set the phase address in the plan's phases ptr)
3. Call  RU allocator with plan pointer  
4. Call  LA allocator with plan pointer 
5. Fill Fixed plan parameters  
3. send a confirmation message to group manager : with status - success/fail

	
**********************************************************************************/
static void planManagerCreatePlan(K_MSG* psMsg)
{
	CreatePlanMsg_t* 				pMuCreatePlanReq = (CreatePlanMsg_t*)pK_MSG_DATA(psMsg);
	CommonPlan_t*					planPtr = NULL;
	DlDataPhase_t*					dataPhasePtr = NULL;	
	DlDataPhase_t*					mirrorDataPhasePtr = NULL;
    UlDataPhase_t*					bsrpPhasePtr = NULL;
	SoundingPhase_t*				soundingPhasePtr = NULL;	
	ProtectionPhase_t*				protectionPhasePtr = NULL;		
#ifdef ENET_INC_ARCH_WAVE600D2
	NfrpPhase_t*					nfrpPhasePtr = NULL;
#endif
	uint8							loopIndex; 
	uint32* 						phasePtrInPlanAddress = NULL; 
	DlDataPhase_t*					dlDataPhasePointer = NULL; 
	UlDataPhase_t*					ulDataPhasePointer = NULL;
	K_MSG*							laMessagePtr = NULL;
	LaAddHeGroupParams_t* 			laAddHeGroupParameters = NULL; 
	PlanManagerAllocationStatus_e	allocationStatus = ALLOCATION_SUCCESS; 
	RuAllocationPerUser_t			ruAllocation[NUM_OF_USERS_MU_HE_PHASE]; 
	uint8							allocatedUsers;
	ILOG0_V("[planManagerCreatePlan]");
	
	/*set plan manager state to memory allocation procedure */  
	heGroupManagerDb[pMuCreatePlanReq->groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_MEMORY_RESOURCE_ALLOCATION_STATE;  
	/*Allocate a plan from pool*/ 
	planPtr = planManagerFetchPlanFromPool();
	/*Allocate sounding, protection and BSRP phase*/ 
	protectionPhasePtr 	= 	planManagerFetchPhaseFromPool(PHASE_FORMAT_PROTECTION); 
	ILOG0_D("planManagerCreatePlan, phase Ptr = 0X%X", protectionPhasePtr);
	soundingPhasePtr 	=	planManagerFetchPhaseFromPool(PHASE_FORMAT_SOUNDING);
	ILOG0_D("planManagerCreatePlan, phase Ptr = 0X%X", soundingPhasePtr);
     bsrpPhasePtr		=	planManagerFetchPhaseFromPool(PHASE_FORMAT_BSRP);
	 ILOG0_D("planManagerCreatePlan, phase Ptr = 0X%X", bsrpPhasePtr);
	/*Allocate data and it's corespond mirror phase*/
	dataPhasePtr		= 	planManagerFetchPhaseFromPool(PHASE_FORMAT_DL_DATA); //We currently use the same type for UL/DL as they are at the same size, hence in the same pool  
	ILOG0_D("planManagerCreatePlan, phase Ptr = 0X%X", dataPhasePtr);

	mirrorDataPhasePtr	=	planManagerFetchPhaseFromPool(PHASE_FORMAT_DL_DATA);
	ILOG0_D("planManagerCreatePlan, phase Ptr = 0X%X", mirrorDataPhasePtr);

#ifdef ENET_INC_ARCH_WAVE600D2
	nfrpPhasePtr		=	planManagerFetchPhaseFromPool(PHASE_FORMAT_NFRP); 
	/* Verify that all allocation where successful 
	If not - send a confirmation message to GM with status - failed */	
			
	allocationStatus = planManagerVerifyResourceAllocation(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, mirrorDataPhasePtr, nfrpPhasePtr);
#else
	allocationStatus = planManagerVerifyResourceAllocation(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, mirrorDataPhasePtr);
#endif
	if(allocationStatus != ALLOCATION_SUCCESS)
	{
#ifdef ENET_INC_ARCH_WAVE600D2
		planManagerFreeResourcesDueToFailure(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, nfrpPhasePtr);
#else
		planManagerFreeResourcesDueToFailure(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr);
#endif
		SERIAL_TRACE("No available resource for HE group!", 0, 0, 0); 
		ILOG0_D("planManagerCreatePlan - No available resource for HE group!  reason - %d", allocationStatus);
		planManagerHandleCreationFailure(psMsg, pMuCreatePlanReq->groupId, FALSE);
	}
	else // All resource allocation succeeded 
	{
		//set plan ptr in HE group manager DB 
		heGroupManagerDb[pMuCreatePlanReq->groupId].planPtr = planPtr; 	
		dataPhasePtr->common.nextOrMirrorPhasePointer = (uint32)mirrorDataPhasePtr; // Make the data phase and its mirror phase pointing each other
		mirrorDataPhasePtr->common.nextOrMirrorPhasePointer = (uint32)dataPhasePtr; // Make the data phase and its mirror phase pointing each other

		planPtr->phase0Full32BitPointer = (uint32)protectionPhasePtr;   // set 1st phase in plan with protection phase 
		planPtr->phase1Full32BitPointer = (uint32)soundingPhasePtr;     // set 2nd phase in plan with sounding phase
		planPtr->phase2Full32BitPointer = (uint32)bsrpPhasePtr; 		// set 3rd phase in plan with bsrp phase 
#ifdef ENET_INC_ARCH_WAVE600D2
		planPtr->phase3Full32BitPointer = (uint32)nfrpPhasePtr; 		// set 4th phase in plan with nfrp phase 
#endif
		planPtr->phaseValid = 0x0;	// initialize the phase valid bitmap with zeroes
		planPtr->phaseFirstOccurrence  =	0x0; //init
		planPtr->phaseFirstOccurrence |=	PROTECTION_PHASE_FIRST_OCCURENCE;
		planPtr->phaseFirstOccurrence |=	SOUNDING_PHASE_FIRST_OCCURENCE;  
		planPtr->phaseFirstOccurrence |=	BSRP_PHASE_FIRST_OCCURENCE; 	 
		planPtr->phaseFirstOccurrence |=	DATA_PHASE_FIRST_OCCURENCE;  // set only the first data phase occurence bit 
#ifdef ENET_INC_ARCH_WAVE600D2
		planPtr->phaseFirstOccurrence |=	NFRP_PHASE_FIRST_OCCURENCE; 	 
#endif

		ASSERT(pMuCreatePlanReq->numberOfRepetitions < NUMBER_OF_DATA_PHASES_IN_PLAN); /*  In case of no repetitions- data phase would be transmitted only once. 
																							It is possible to configure only up to (NUMBER_OF_DATA_PHASES_IN_PLAN - 1) repetitions - 
																							which means NUMBER_OF_DATA_PHASES_IN_PLAN data phase tranmissions (the rest are reserved for sounding, protection, bsrp and nfrp)*/ 
		phasePtrInPlanAddress = (void*)((uint32*)&(planPtr->word5));	// &(planPtr->word5) is the address of the first data phase 
		for(loopIndex = 0; loopIndex < (pMuCreatePlanReq->numberOfRepetitions + 1); loopIndex++) 
		{
			// start from &(planPtr->word5) , data phase address starts from phase3Pointer(word5) while phases 0, 1 and 2 are for protection, sounding and bsrp
			*phasePtrInPlanAddress = (uint32)dataPhasePtr; // Set the phase pointer in plan with the current phase address
			phasePtrInPlanAddress++; // Make the phasePtr point to the next phase in plan 
			planPtr->phaseValid |=	0x1 << (loopIndex + DATA_PHASE_BIT_INDEX); 
		}
#ifdef ENET_INC_ARCH_WAVE600B
        planPtr->skipPhaseValidBitmap = 0x0; 
#endif
		/* set the plan manager status to wait for RU allocation state */ 
		heGroupManagerDb[pMuCreatePlanReq->groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_RU_ALLOCATION_STATE; 

		allocationStatus = planManagerRuAllocator_GetAllocationForHeGroup(ruAllocation, heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup, pMuCreatePlanReq->bw, &allocatedUsers);
		if (allocationStatus != ALLOCATION_SUCCESS)
		{
#ifdef ENET_INC_ARCH_WAVE600D2
			planManagerFreeResourcesDueToFailure(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, nfrpPhasePtr);
#else
			planManagerFreeResourcesDueToFailure(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr);
#endif
			planManagerHandleCreationFailure(psMsg, pMuCreatePlanReq->groupId, FALSE);
		}
		else // RU allocation succeeded
		{
            ILOG0_DD("[planManagerCreatePlan], group: %d, formationType: %d", pMuCreatePlanReq->groupId, heGroupManagerDb[pMuCreatePlanReq->groupId].formationType);
            SLOG0(0, 0, HeGroupManagerPhaseBitmap_t, &(heGroupManagerDb[pMuCreatePlanReq->groupId].phasesBitmap));
                
			// Set Common plan paramerters CommonPlan 
			planPtr->maxTxopDur = planManagerDefaultSettings.planMaxTxopDur; 
			planPtr->durationUpdateMode = planManagerDefaultSettings.planDurationUpdateMode;  
			planPtr->startBw = pMuCreatePlanReq->bw;	

            // Set protection phase parameters
            if (heGroupManagerDb[pMuCreatePlanReq->groupId].phasesBitmap.protectionPhase == 1)
            {
			    planManagerSetProtectionPhase(protectionPhasePtr, pMuCreatePlanReq->groupId, pMuCreatePlanReq->bw, ruAllocation, psMsg->header.vapId);
                planPtr->phaseValid |= PROTECTION_PHASE_VALIDITY_MASK;  //turn first phase (bit 0) on
            }
			
			// Set sounding phase parameters 
			if (heGroupManagerDb[pMuCreatePlanReq->groupId].phasesBitmap.soundingPhase == 1)
			{
                planManagerSetSoundingPhase(soundingPhasePtr, pMuCreatePlanReq);
			    planPtr->phaseValid |= SOUNDING_PHASE_VALIDITY_MASK;  //turn second phase (bit 1) on 
			}			
			
			// Set DATA phase parameters according to  USER configuration / fixed values / supported modes / station capabilities 
			//convert pointer to the phase type accordingly 
#ifdef ENET_INC_ARCH_WAVE600D2
			if ((heGroupManagerDb[pMuCreatePlanReq->groupId].formationType == UL_OFDMA) || (heGroupManagerDb[pMuCreatePlanReq->groupId].formationType == UL_MIMO))
#else
			if (heGroupManagerDb[pMuCreatePlanReq->groupId].formationType == UL_OFDMA)
#endif
			{
#ifdef DYNAMIC_GROUPING_DEBUG
                ILOG0_V("[planManagerCreatePlan], UL");
#endif
				ulDataPhasePointer = (UlDataPhase_t*)dataPhasePtr;
                planManagerSetUlPhase(ulDataPhasePointer, pMuCreatePlanReq, planPtr, psMsg->header.vapId, ruAllocation, HE_MU_UL);    

				if (heGroupManagerDb[pMuCreatePlanReq->groupId].phasesBitmap.bsrpPhase == 1)
				{
					planManagerSetUlPhase(bsrpPhasePtr, pMuCreatePlanReq, planPtr, psMsg->header.vapId, ruAllocation, HE_MU_BSRP_UL);
					planPtr->phaseValid |= BSRP_PHASE_VALIDITY_MASK;  //turn third phase (bit 2) on  
				}
#ifdef ENET_INC_ARCH_WAVE600D2
				if (heGroupManagerDb[pMuCreatePlanReq->groupId].phasesBitmap.nfrpPhase == 1)
				{
					planManagerSetNfrpPhase(nfrpPhasePtr, pMuCreatePlanReq, planPtr, psMsg->header.vapId, pMuCreatePlanReq->groupId);
					planPtr->phaseValid |= NFRP_PHASE_VALIDITY_MASK;  //turn fourth phase (bit 3) on  
				}	
#endif
			}
			else //DL OFDMA or MIMO 
			{
#ifdef DYNAMIC_GROUPING_DEBUG
                ILOG0_V("[planManagerCreatePlan], DL");
#endif
				dlDataPhasePointer = (DlDataPhase_t*)dataPhasePtr; 
				/*Set DL phase common fields */
				planManagerSetDlPhaseCommonFields(&(dlDataPhasePointer->common), pMuCreatePlanReq, heGroupManagerDb[pMuCreatePlanReq->groupId].sequence);
				/*Set DL phase TCR A (data) common fields */
				planManagerSetDlPhaseCommonTcr_A_Fields(&(dlDataPhasePointer->commonTcra),pMuCreatePlanReq,planPtr, psMsg->header.vapId, heGroupManagerDb[pMuCreatePlanReq->groupId].sequence);
				/*Set DL phase TCR B (MU BAR) common fields */
				planManagerSetDlPhaseCommonTcr_B_Fields(&(dlDataPhasePointer->commonTcrb),pMuCreatePlanReq,planPtr, psMsg->header.vapId);
				/*Set DL phase RCR common fields */
				planManagerSetDlPhaseCommonRcrFields(&(dlDataPhasePointer->commonRcr),pMuCreatePlanReq, psMsg->header.vapId);
				/*	Set DL phase TCR B (MU BAR) common user fields */
				planManagerSetDlPhaseCommonUserTCR_B_Fields((&(dlDataPhasePointer->commonUserTcrb)), ruAllocation);
				//Set TF parameters
				planManagerSetDlPhaseCommonTfParameters(dlDataPhasePointer,pMuCreatePlanReq);
				//Set TF MU BAR parameters
				planManagerSetDlPhaseCommonTfMuBar(&(dlDataPhasePointer->tfMuBar));
				//Set Data TCR per User parameters
				planManagerSetDlPhasePerUser(&(dlDataPhasePointer->userPhase[0]), pMuCreatePlanReq, ruAllocation, dlDataPhasePointer->common.plannedBar, allocatedUsers);
				//Set TF per User parameters
				planManagerSetDlPhaseTfInfoPerUser(&(dlDataPhasePointer->tfUserInfo[0]), pMuCreatePlanReq, ruAllocation);
                /*Fill DL phase common field Valid Users bitmap */
				planManagerSetDlPhaseCommonFieldValidUser(&(dlDataPhasePointer->common), pMuCreatePlanReq, allocatedUsers);
			}
            ILOG0_D("[planManagerCreatePlan], planPtr->phaseValid = 0x%x", planPtr->phaseValid);
			/* set the plan manager status to wait for LA setting state */ 
			heGroupManagerDb[pMuCreatePlanReq->groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_STATE; 
			/* Send message for Link adaptation module  */ 
			laMessagePtr                            = OSAL_GET_MESSAGE(sizeof(LaAddHeGroupParams_t));
			laAddHeGroupParameters                  = (LaAddHeGroupParams_t *)pK_MSG_DATA(laMessagePtr);
			laAddHeGroupParameters->isStaticGroup   = FALSE; 
			planManagerSetNssPerUser(laAddHeGroupParameters, pMuCreatePlanReq->groupId); 
			laAddHeGroupParameters->planIndex       = planPtr->planIndex; 
			laAddHeGroupParameters->groupId         = pMuCreatePlanReq->groupId; 
			OSAL_SEND_MESSAGE(LINK_ADAPTATION_ADD_HE_GROUP, TASK_LINK_ADAPTATION, laMessagePtr, psMsg->header.vapId);
		}
	}
}	
#endif

/**********************************************************************************

planManagerVerifyResourceAllocation 


Description:
------------
verify that all plan manager resource allocated successfully. 
In case of unseccessful allocation - it return a stat

	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600D2
static PlanManagerAllocationStatus_e planManagerVerifyResourceAllocation(CommonPlan_t* planPtr ,ProtectionPhase_t* protectionPhasePtr,SoundingPhase_t* soundingPhasePtr , UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr, DlDataPhase_t* mirrorDataPhasePtr,NfrpPhase_t* nfrpPhasePtr )
#else
static PlanManagerAllocationStatus_e planManagerVerifyResourceAllocation(CommonPlan_t* planPtr ,ProtectionPhase_t* protectionPhasePtr,SoundingPhase_t* soundingPhasePtr , UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr, DlDataPhase_t* mirrorDataPhasePtr)
#endif
{
	PlanManagerAllocationStatus_e	allocationStatus = ALLOCATION_SUCCESS; 

	if(planPtr == NULL)
	{
		allocationStatus  = PLAN_ALLOCATION_FAILED; 
	}
	if(protectionPhasePtr == NULL)
	{
		if(allocationStatus == ALLOCATION_SUCCESS)
		{
			allocationStatus  = PROTECTION_PHASE_ALLOCATION_FAILED; 
		}
	}
	if(soundingPhasePtr == NULL)
	{
		if(allocationStatus == ALLOCATION_SUCCESS)
		{
			allocationStatus  =	SOUNDING_PHASE_ALLOCATION_FAILED; 
		}
	}
	if(bsrpPhasePtr == NULL)
	{
		if(allocationStatus == ALLOCATION_SUCCESS)
		{
			allocationStatus  =	BSRP_PHASE_ALLOCATION_FAILED; 
		}
	}
	if(dataPhasePtr == NULL)
	{
		if(allocationStatus == ALLOCATION_SUCCESS)
		{
			allocationStatus  =	DATA_PHASE_ALLOCATION_FAILED; 
		}
	}
	if(mirrorDataPhasePtr == NULL)
	{
		if(allocationStatus == ALLOCATION_SUCCESS)
		{
			allocationStatus  =	DATA_MIROR_PHASE_ALLOCATION_FAILED; 
		}	
	}
#ifdef ENET_INC_ARCH_WAVE600D2

	if(nfrpPhasePtr == NULL)
	{
		if(allocationStatus == ALLOCATION_SUCCESS)
		{
			allocationStatus  =	NFRP_PHASE_ALLOCATION_FAILED; 
		}	
	}
#endif
	return allocationStatus; 
}
/**********************************************************************************

planManagerFreeResourcesDueToFailure 


Description:
------------
verify that all plan manager resource allocated successfully. 
In case of unseccessful allocation - it return a stat

	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600D2
static void planManagerFreeResourcesDueToFailure(CommonPlan_t* planPtr ,ProtectionPhase_t* protectionPhasePtr,SoundingPhase_t* soundingPhasePtr, UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr, NfrpPhase_t* nfrpPhasePtr)
#else
static void planManagerFreeResourcesDueToFailure(CommonPlan_t* planPtr ,ProtectionPhase_t* protectionPhasePtr,SoundingPhase_t* soundingPhasePtr, UlDataPhase_t* bsrpPhasePtr, DlDataPhase_t* dataPhasePtr)
#endif
{
	if(planPtr != NULL)
	{
		planManagerReturnPlanToPool(planPtr); 
	}
	if(protectionPhasePtr != NULL)
	{
		planManagerReturnPhaseToPool(protectionPhasePtr,PHASE_FORMAT_PROTECTION); 
	}
	if(soundingPhasePtr != NULL)
	{
		planManagerReturnPhaseToPool(soundingPhasePtr,PHASE_FORMAT_SOUNDING); 
	}
	if(bsrpPhasePtr != NULL)
	{
		planManagerReturnPhaseToPool(bsrpPhasePtr,PHASE_FORMAT_BSRP); 
	}
	if(dataPhasePtr != NULL)
	{
		//Release both data phase and its correspond mirror phase
		planManagerReturnPhaseToPool(dataPhasePtr,PHASE_FORMAT_DL_DATA); 	
	}
#ifdef ENET_INC_ARCH_WAVE600D2
	if(nfrpPhasePtr != NULL)
	{
		planManagerReturnPhaseToPool(nfrpPhasePtr,PHASE_FORMAT_NFRP); 
	}
#endif
}

/**********************************************************************************

planManagerHandleCreationFailure 


Description:
------------
Called in case of failure while creating an HE plan 
Send confirmation with failure status to GM  

	
**********************************************************************************/
static void planManagerHandleCreationFailure(K_MSG* psMsg,uint8 groupId, bool isStatic)
{
	CreatePlanCfmMsg_t* 			pMuCreatePlanCfm = NULL;
	
	pMuCreatePlanCfm = (CreatePlanCfmMsg_t*)pK_MSG_DATA(psMsg);
	pMuCreatePlanCfm->groupId = groupId; 	
	pMuCreatePlanCfm->isStatic = isStatic; 
	pMuCreatePlanCfm->status = CREATE_PLAN_FAILED; 
	OSAL_SEND_MESSAGE(HE_GROUP_MANAGER_CREATE_PLAN_CFM, TASK_HE_GROUP_MANAGER, psMsg, psMsg->header.vapId);

}
/**********************************************************************************

planManagerRemovePlan 


Description:
------------
	send message to LA	
	Free resources (put plan and phases back to pool) 
	send a confirmation message for the group manager 
	
**********************************************************************************/
static void planManagerRemovePlan(K_MSG* psMsg)
{
	RemovePlanMsg_t* 			removePlanMsgPtr = NULL; 
	K_MSG*						laMessagePtr = NULL;
	LaRemoveHeGroupParams_t*	laRemoveParamsPtr = NULL; 
	CommonPlan_t*				planPtr = NULL;

    ILOG0_D("planManagerRemovePlan, vapId = %d",psMsg->header.vapId);
	removePlanMsgPtr = (RemovePlanMsg_t *)pK_MSG_DATA(psMsg);
	heGroupManagerDb[removePlanMsgPtr->groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_LA_FREE_RESOURCES_STATE;

	laMessagePtr = OSAL_GET_MESSAGE(sizeof(LaRemoveHeGroupParams_t));
	laRemoveParamsPtr = (LaRemoveHeGroupParams_t *)pK_MSG_DATA(laMessagePtr);
	laRemoveParamsPtr->groupId = removePlanMsgPtr->groupId; 
	planPtr = heGroupManagerDb[removePlanMsgPtr->groupId].planPtr; 
	laRemoveParamsPtr->planIndex = planPtr->planIndex; 

	// send HE remove message to link adaptation  
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_REMOVE_HE_GROUP, TASK_LINK_ADAPTATION, laMessagePtr, psMsg->header.vapId);
}

/**********************************************************************************

planManagerFetchPlanFromPool 


Description:
------------
	If available - allocate a plan from free resources pool and pass it to the caller. 
	If no resource available - return NULL 

**********************************************************************************/

static CommonPlan_t* planManagerFetchPlanFromPool(void)
{
	CommonPlan_t* allocatedPlanPtr = NULL; 
	uint8 planIndex = 0;
	
	if(heMuResourcesPool.planPoolptr != NULL) 
	{
		allocatedPlanPtr = heMuResourcesPool.planPoolptr; 
		heMuResourcesPool.planPoolptr =(CommonPlan_t*) heMuResourcesPool.planPoolptr->nextPlanPointer; //set the list head with the next element 
		planIndex = allocatedPlanPtr->planIndex; //save plan index
		memset(allocatedPlanPtr, 0x0, sizeof(CommonPlan_t));
		allocatedPlanPtr->planIndex = planIndex;//restore plan index after reset the entry
	}
	return allocatedPlanPtr; 
}


/**********************************************************************************

planManagerFetchPhaseFromPool 


Description:
------------
	If available - allocate a phase from free resources pool and pass it to the caller. 
	If no resource available - return NULL 

**********************************************************************************/

static void* planManagerFetchPhaseFromPool(PhaseFormat_e phaseFormat)
{
	void* allocatedPhasePtr = NULL; 

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[planManagerFetchPhaseFromPool], phase format: %d", phaseFormat);
#endif

	switch (phaseFormat)
	{

	case PHASE_FORMAT_DL_DATA:
		if(heMuResourcesPool.dataPhasePoolPtr != NULL) 
		{
			allocatedPhasePtr = heMuResourcesPool.dataPhasePoolPtr; 
			heMuResourcesPool.dataPhasePoolPtr = (DataPhase_t*)(heMuResourcesPool.dataPhasePoolPtr->dlDataPhaseStructure.common.nextOrMirrorPhasePointer); //set the list head with the next element 
			((DlDataPhase_t*)allocatedPhasePtr)->common.nextOrMirrorPhasePointer = (uint32)NULL; 
			memset(allocatedPhasePtr,0x0, sizeof(DlDataPhase_t));
		}
		break;
	case PHASE_FORMAT_UL_DATA:
	case PHASE_FORMAT_BSRP:
		if(heMuResourcesPool.dataPhasePoolPtr != NULL) 
		{
			allocatedPhasePtr = heMuResourcesPool.dataPhasePoolPtr; 
			heMuResourcesPool.dataPhasePoolPtr = (DataPhase_t*)(heMuResourcesPool.dataPhasePoolPtr->ulDataPhaseStructure.common.nextOrMirrorPhasePointer); //set the list head with the next element 
			((UlDataPhase_t*)allocatedPhasePtr)->common.nextOrMirrorPhasePointer = (uint32)NULL; 
			memset(allocatedPhasePtr,0x0, sizeof(UlDataPhase_t));
		}
		break;
	case PHASE_FORMAT_SOUNDING:
		if(heMuResourcesPool.soundingPhasePoolPtr != NULL) 
		{
			allocatedPhasePtr = heMuResourcesPool.soundingPhasePoolPtr; 
			heMuResourcesPool.soundingPhasePoolPtr = (SoundingPhase_t*)(heMuResourcesPool.soundingPhasePoolPtr->common.nextPhasePointer); //set the list head with the next element 
			//((SoundingPhase_t*)allocatedPhasePtr)->common.nextPhasePointer = (uint32)NULL; 
			//Word4 holds the ndpaPointer , never reset it 
			memset(allocatedPhasePtr,0x0,SIZE_FROM_WORD0_TO_WORD4); //reset words 0-3
			memset(&(((SoundingPhase_t*)allocatedPhasePtr)->common.word5),0x0, SIZE_OF_SOUNDING_PHASE_STARTING_FROM_WORD5);// reset all words from word 5 
		}
		break;
	case PHASE_FORMAT_PROTECTION:
		if(heMuResourcesPool.protectionPhasePoolPtr != NULL) 
		{
			allocatedPhasePtr = heMuResourcesPool.protectionPhasePoolPtr; 
			heMuResourcesPool.protectionPhasePoolPtr = (ProtectionPhase_t*)(heMuResourcesPool.protectionPhasePoolPtr->common.nextPhasePointer); //set the list head with the next element 
			((ProtectionPhase_t*)allocatedPhasePtr)->common.nextPhasePointer = (uint32)NULL; 
			memset(allocatedPhasePtr,0x0, sizeof(ProtectionPhase_t));	
		}
		break;
#ifdef ENET_INC_ARCH_WAVE600D2

	case PHASE_FORMAT_NFRP:
		if(heMuResourcesPool.nfrpPhasePoolPtr != NULL) 
		{
			allocatedPhasePtr = heMuResourcesPool.nfrpPhasePoolPtr; 
			heMuResourcesPool.nfrpPhasePoolPtr = (NfrpPhase_t*)(heMuResourcesPool.nfrpPhasePoolPtr->common.nextPhasePointer); //set the list head with the next element 
			((NfrpPhase_t*)allocatedPhasePtr)->common.nextPhasePointer = (uint32)NULL; 
			memset(allocatedPhasePtr,0x0, sizeof(NfrpPhase_t));	
		}
		break;
#endif
	default:
		DEBUG_ASSERT(0);
	}
	return allocatedPhasePtr; 
}




/**********************************************************************************

planManagerReturnPhaseToPool 


Description:
------------
	Return Phaseptr to Phases Pool

	
**********************************************************************************/

static void planManagerReturnPhaseToPool(void* phasePtr, PhaseFormat_e phaseFormat)
{
	DataPhase_t*		dataPhasePtr = NULL; 	
	DataPhase_t*		bsrpPhasePtr = NULL; 	
	DataPhase_t*		mirrorDataPhasePtr = NULL; 
	SoundingPhase_t*	soundingPhasePtr = NULL; 
	ProtectionPhase_t*	protectionPhasePtr = NULL; 
#ifdef ENET_INC_ARCH_WAVE600D2
	NfrpPhase_t*		nfrpPhasePtr = NULL;
#endif

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[planManagerReturnPhaseToPool], phase format: %d", phaseFormat);
#endif

	switch (phaseFormat)
	{

	case PHASE_FORMAT_DL_DATA:
	case PHASE_FORMAT_UL_DATA:
		/* 			
			B4 -      heMuResourcesPool.dataPhasePoolPtr  :   originalHead -> ... 
			After-    heMuResourcesPool.dataPhasePoolPtr  :   returnedDataPhase  ->  returedDatamirrorPhase  ->  originalHead -> ...  
		*/
		dataPhasePtr = (DataPhase_t*)(phasePtr); 
		mirrorDataPhasePtr = (DataPhase_t*)(dataPhasePtr->dlDataPhaseStructure.common.nextOrMirrorPhasePointer); 
		mirrorDataPhasePtr->dlDataPhaseStructure.common.nextOrMirrorPhasePointer =  (uint32)(heMuResourcesPool.dataPhasePoolPtr);
		heMuResourcesPool.dataPhasePoolPtr = phasePtr;
		break;

	case PHASE_FORMAT_BSRP:
		bsrpPhasePtr = (DataPhase_t*)(phasePtr); 
 		bsrpPhasePtr->ulDataPhaseStructure.common.nextOrMirrorPhasePointer = (uint32)(heMuResourcesPool.dataPhasePoolPtr);
		heMuResourcesPool.dataPhasePoolPtr = phasePtr;
		break;

	case PHASE_FORMAT_SOUNDING:
		soundingPhasePtr = (SoundingPhase_t*)(phasePtr); 
		soundingPhasePtr->common.nextPhasePointer  =  (uint32)(heMuResourcesPool.soundingPhasePoolPtr);
		heMuResourcesPool.soundingPhasePoolPtr = phasePtr;
		break;

	case PHASE_FORMAT_PROTECTION:
		protectionPhasePtr = (ProtectionPhase_t*)(phasePtr); 
		protectionPhasePtr->common.nextPhasePointer  =  (uint32)(heMuResourcesPool.protectionPhasePoolPtr);
		heMuResourcesPool.protectionPhasePoolPtr = phasePtr;
		break;
#ifdef ENET_INC_ARCH_WAVE600D2

	case PHASE_FORMAT_NFRP:
		nfrpPhasePtr = (NfrpPhase_t*)(phasePtr); 
		nfrpPhasePtr->common.nextPhasePointer  =  (uint32)(heMuResourcesPool.nfrpPhasePoolPtr);
		heMuResourcesPool.nfrpPhasePoolPtr = phasePtr;
		break;
#endif		
	default:
		DEBUG_ASSERT(0);
	}

		
}

/**********************************************************************************

planManagerReturnPlanToPool 


Description:
------------
	Return PlanPtr to Plan Pool

	
**********************************************************************************/

static void planManagerReturnPlanToPool(CommonPlanCommon_t* planPtr)
{
	
	planPtr->nextPlanPointer= (uint32) heMuResourcesPool.planPoolptr;
	heMuResourcesPool.planPoolptr = planPtr;
}

/**********************************************************************************

planManagerCreateStaticPlan 


Description:
------------


1. if available - pull plan& phase from pool according to the phase type
2. link between plan and phases  (set the phase address in the plan's phases ptr)
3. Fill Fixed and configured plan parameters (from user configuration message)
4. send message to RU allocator (in static planner it may done internally)
5. Send message to LA 
6. send confirmation message to the group manager : with status - success/fail


	
**********************************************************************************/
static void planManagerCreateStaticPlan(K_MSG* psMsg)
{
	UMI_STATIC_PLAN_CONFIG* 		pMuPlanConfig = (UMI_STATIC_PLAN_CONFIG*)pK_MSG_DATA(psMsg); 
	CommonPlan_t*					planPtr = NULL;
	DlDataPhase_t*					dataPhasePtr= NULL;	
	DlDataPhase_t*					mirrorDataPhasePtr = NULL;
	UlDataPhase_t*					bsrpPhasePtr = NULL;	
	SoundingPhase_t*				soundingPhasePtr = NULL;	
	ProtectionPhase_t*				protectionPhasePtr = NULL; 	
#ifdef ENET_INC_ARCH_WAVE600D2
	NfrpPhase_t*					nfrpPhasePtr = NULL;
#endif
	uint8							loopIndex; 
	uint32* 						phasePtrInPlanAddress = NULL; 
	DlDataPhase_t*					dlDataPhasePointer = NULL; 
	UlDataPhase_t*					ulDataPhasePointer = NULL;
	UlDataPhase_t*					bsrpDataPhasePointer = NULL;
	K_MSG*							laMessagePtr = NULL;							
	LaAddHeGroupParams_t*			laRequestParams = NULL; 
//	RuAllocation_t					ruAllocation; 
	uint8							groupId = HE_GROUP_MANAGER_DL_GID; 
	PlanManagerAllocationStatus_e	allocationStatus = ALLOCATION_SUCCESS; 
	Bw_e							bw;

#if defined(HE_MU_RCR_EXPEDITE_VALIDITY_TIMER)
	// set a value for the RCR expedite validity timer (phy - RX TD)
	RegAccess_Write(0xA0b40208,	0x3C00);
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2
	if((pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_NFRP) || (pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_UL_DATA)||(pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_UL_MIXED))
#else
	if( (pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_UL_DATA)||(pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_UL_MIXED))
#endif
	{
		groupId = HE_GROUP_MANAGER_UL_GID;
	}
	else
	{
		ASSERT((pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_DL_DATA) || (pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_DL_MIXED) || (pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_SOUNDING));
	}
	
	/*set plan manager state to memory allocation procedure */	
	heGroupManagerDb[groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_MEMORY_RESOURCE_ALLOCATION_STATE;  
	/*Allocate a plan from pool*/ 
	planPtr = planManagerFetchPlanFromPool();
	/*Allocate sounding, protection phase and BSRP phase*/ 
	protectionPhasePtr	=	planManagerFetchPhaseFromPool(PHASE_FORMAT_PROTECTION); 
	soundingPhasePtr	=	planManagerFetchPhaseFromPool(PHASE_FORMAT_SOUNDING); 
	bsrpPhasePtr		=	planManagerFetchPhaseFromPool(PHASE_FORMAT_BSRP);
	/*Allocate data and it's corespond mirror phase*/
	dataPhasePtr		=	planManagerFetchPhaseFromPool(PHASE_FORMAT_DL_DATA); 
	mirrorDataPhasePtr	=	planManagerFetchPhaseFromPool(PHASE_FORMAT_DL_DATA);	
#ifdef ENET_INC_ARCH_WAVE600D2
	nfrpPhasePtr		=	planManagerFetchPhaseFromPool(PHASE_FORMAT_NFRP); 
	/* Verify that all allocation where successful 
		If not - send a confirmation message to GM with status - failed */	
		
	allocationStatus = planManagerVerifyResourceAllocation(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, mirrorDataPhasePtr, nfrpPhasePtr);
#else
	allocationStatus = planManagerVerifyResourceAllocation(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, mirrorDataPhasePtr);
#endif
	if(allocationStatus != ALLOCATION_SUCCESS)
	{
#ifdef ENET_INC_ARCH_WAVE600D2
		planManagerFreeResourcesDueToFailure(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr, nfrpPhasePtr);
#else
		planManagerFreeResourcesDueToFailure(planPtr, protectionPhasePtr, soundingPhasePtr, bsrpPhasePtr, dataPhasePtr);
#endif
		SERIAL_TRACE("No available resource for HE group!",0,0,0);
		ILOG0_D("planManagerCreateStaticPlan - No available resource for HE group!	reason - %d",allocationStatus);
		planManagerHandleCreationFailure(psMsg, groupId, TRUE);
	}
	else // All resource allocation succeeded 
	{
		//set plan ptr in HE group manager DB 
		heGroupManagerDb[groupId].planPtr = planPtr;	
		
		dataPhasePtr->common.nextOrMirrorPhasePointer = (uint32)mirrorDataPhasePtr; // Make the data phase and its mirror phase pointing each other
		mirrorDataPhasePtr->common.nextOrMirrorPhasePointer = (uint32)dataPhasePtr; // Make the data phase and its mirror phase pointing each other

		planPtr->phase0Full32BitPointer = (uint32)protectionPhasePtr; 	// set 1st phase in plan with protection phase 
		planPtr->phase1Full32BitPointer = (uint32)soundingPhasePtr; 	// set 2nd phase in plan with sounding phase 
		planPtr->phase2Full32BitPointer = (uint32)bsrpPhasePtr; 		// set 3rd phase in plan with bsrp phase 
#ifdef ENET_INC_ARCH_WAVE600D2
		planPtr->phase3Full32BitPointer = (uint32)nfrpPhasePtr; 		// set 4th phase in plan with nfrp phase 
#endif
		planPtr->phaseValid = 0x0;	// initialize the phase valid bitmap with zeroes
									
		planPtr->phaseFirstOccurrence  = 	0x0; //init
		planPtr->phaseFirstOccurrence |=	PROTECTION_PHASE_FIRST_OCCURENCE;
		planPtr->phaseFirstOccurrence |=	SOUNDING_PHASE_FIRST_OCCURENCE;  
		planPtr->phaseFirstOccurrence |=	BSRP_PHASE_FIRST_OCCURENCE;		 
		planPtr->phaseFirstOccurrence |=	DATA_PHASE_FIRST_OCCURENCE;  // set only the first data phase occurence bit 
#ifdef ENET_INC_ARCH_WAVE600D2
		planPtr->phaseFirstOccurrence |=	NFRP_PHASE_FIRST_OCCURENCE;		 
#endif
		
		/* set the plan manager status to wait for RU allocation state */ 
		heGroupManagerDb[groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_RU_ALLOCATION_STATE; 
		
		//	Set Common plan parameters CommonPlan 
		planPtr->maxTxopDur = pMuPlanConfig->commonSection.maxTxopDuration;  
		planPtr->durationUpdateMode = planManagerDefaultSettings.planDurationUpdateMode; //	full TXOP  
		bw = pMuPlanConfig->commonSection.startBwLimit;
		planPtr->startBw = bw;
		ASSERT(bw < BW_LAST);
		ILOG0_DD("[planManagerCreateStaticPlan], heGroupManagerDb[%d].formationType = %d", groupId, heGroupManagerDb[groupId].formationType);

		if(pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_SOUNDING) //Sounding only
		{ 	
			/* Set data phase ptr in plan in order to successfully release it in planManagerRemovePlanLaCfm */
			planPtr->phase3Full32BitPointer = (uint32)dataPhasePtr; 	

			planManagerSetStaticSoundingPhase(soundingPhasePtr, pMuPlanConfig, planPtr, psMsg->header.vapId, groupId);	
			// Turn on the phase valid bit for the sounding phase 
			planPtr->phaseValid |= SOUNDING_PHASE_VALIDITY_MASK;  //turn second phase (bit 1) on  

		}
		else // Not sounding only
		{
			ASSERT(pMuPlanConfig->commonSection.numberOfPhaseRepetitions < NUMBER_OF_DATA_PHASES_IN_PLAN);
			phasePtrInPlanAddress = (void*)((uint32*)&(planPtr->word5));					// address of the first data phase  
			for(loopIndex = 0; loopIndex < (pMuPlanConfig->commonSection.numberOfPhaseRepetitions + 1); loopIndex++) 
			{
				*phasePtrInPlanAddress = (uint32)dataPhasePtr; //Set the phase pointer in plan with the current phase address
				phasePtrInPlanAddress++; // Make the phasePtr point to the next phase in plan 

				planPtr->phaseValid |=	0x1 << (loopIndex + DATA_PHASE_BIT_INDEX); 
			}
			
#ifdef ENET_INC_ARCH_WAVE600B
	        planPtr->skipPhaseValidBitmap = 0x0; 
#endif



			// Set protection phase parameters 
			// planManagerSetProtectionPhase(protectionPhasePtr,groupId, pMuPlanConfig->commonSection.startBwLimit); 
			
			// Set DATA phase parameters according to  USER configuration / fixed values / supported modes / station capabilities 
			// convert pointer to the phase type accordingly 
#ifdef ENET_INC_ARCH_WAVE600D2
			if((heGroupManagerDb[groupId].formationType == UL_OFDMA) || (heGroupManagerDb[groupId].formationType == UL_MIMO))
#else
			if(heGroupManagerDb[groupId].formationType == UL_OFDMA)
#endif
			{
				ulDataPhasePointer = (UlDataPhase_t*)dataPhasePtr; 
				planManagerSetStaticUlPhase(ulDataPhasePointer, pMuPlanConfig, planPtr, psMsg->header.vapId, groupId);
				
				if((pMuPlanConfig->commonSection.muSequenceType == HE_MU_BSRP_UL) || (pMuPlanConfig->commonSection.muSequenceType == HE_MU_BSRP))
				{
					bsrpDataPhasePointer = (UlDataPhase_t*)bsrpPhasePtr; 
					planManagerSetStaticUlPhase(bsrpDataPhasePointer, pMuPlanConfig, planPtr, psMsg->header.vapId, groupId);
					planPtr->phaseValid |= BSRP_PHASE_VALIDITY_MASK;  //turn second phase (bit 1) on  
				}
#ifdef ENET_INC_ARCH_WAVE600D2
				else if(pMuPlanConfig->commonSection.muSequenceType == HE_MU_NFRP)
				{
					planManagerSetStaticNfrpPhase(nfrpPhasePtr, pMuPlanConfig, planPtr, psMsg->header.vapId, groupId);
					planPtr->phaseValid |= NFRP_PHASE_VALIDITY_MASK;  //turn  phase (bit 1) on  
				}	
#endif
			}
			else
			{
				dlDataPhasePointer = (DlDataPhase_t*)dataPhasePtr; 
				/*Fill DL phase common fields */
				planManagerSetStaticDlPhaseCommonFields(&(dlDataPhasePointer->common), pMuPlanConfig);
				/*Fill DL phase TCR A (data) common fields */
				planManagerSetStaticDlPhaseCommonTcr_A_Fields(&(dlDataPhasePointer->commonTcra), pMuPlanConfig, planPtr, psMsg->header.vapId);
				/*Fill DL phase TCR B (MU BAR) common fields */
				planManagerSetStaticDlPhaseCommonTcr_B_Fields(&(dlDataPhasePointer->commonTcrb), pMuPlanConfig, planPtr, psMsg->header.vapId);
				/*Fill DL phase RCR common fields */
				planManagerSetStaticDlPhaseCommonRcrFields(&(dlDataPhasePointer->commonRcr), pMuPlanConfig, psMsg->header.vapId);
				/*	Fill DL phase TCR B (MU BAR) common user fields */
				planManagerSetStaticDlPhaseCommonUserTCR_B_Fields((&(dlDataPhasePointer->commonUserTcrb)), pMuPlanConfig);
				//Fill TF parameters
				planManagerSetStaticDlPhaseCommonTfParameters(dlDataPhasePointer, pMuPlanConfig);
				//Fill TF MU BAR parameters
				planManagerSetStaticDlPhaseCommonTfMuBar(&(dlDataPhasePointer->tfMuBar), pMuPlanConfig);
				//Fill Data TCR per User parameters
				if(pMuPlanConfig->commonSection.muType == MIMO)
				{
					planManagerSetStaticDlMimoPhasePerUser(&(dlDataPhasePointer->userPhase[0]),pMuPlanConfig);
				}
				else //Not Mimo
				{
					if(planManagerAreUsersRuAllocationsEqual(pMuPlanConfig))
					{
						planManagerSetStaticDlPhasePerUser(&(dlDataPhasePointer->userPhase[0]),pMuPlanConfig, bw);
					}
					else
					{
						// Set DL phase per user - Unequal RUs 
						planManagerSetStaticDlPhasePerUserWithUnEqualRus(&(dlDataPhasePointer->userPhase[0]),pMuPlanConfig, bw);
					}
				}

				//Fill TF per User	parameters
				planManagerSetStaticDlPhaseTfInfoPerUser(&(dlDataPhasePointer->tfUserInfo[0]), pMuPlanConfig);
				/*Fill DL phase common field Valid Users bitmap */
				planManagerSetStaticDlPhaseCommonFieldValidUser(&(dlDataPhasePointer->common), pMuPlanConfig);

				// In case of DL MU-MIMO configure also sounding phase that will be transmitted prior to DL MU-MIMO phase
				if(pMuPlanConfig->commonSection.muType == MIMO) 
				{
					planManagerSetStaticSoundingPhase(soundingPhasePtr, pMuPlanConfig, planPtr, psMsg->header.vapId, groupId);  
					// Turn on the phase valid bit for the sounding phase 
					planPtr->phaseValid |= SOUNDING_PHASE_VALIDITY_MASK;  //turn second phase (bit 1) on  
				}
				if (pMuPlanConfig->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_PROTECTION)
				{
					planManagerSetStaticProtectionPhase(protectionPhasePtr, pMuPlanConfig, planPtr, psMsg->header.vapId, groupId);  
					// Turn on the phase valid bit for the protection phase 
					planPtr->phaseValid |= PROTECTION_PHASE_VALIDITY_MASK;  //turn bit 0 on  
				}
			}
		}
		ILOG0_D("[planManagerCreateStaticPlan], planPtr->phaseValid = 0x%x", planPtr->phaseValid);	
		heGroupManagerDb[groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_STATE; 
		laMessagePtr = OSAL_GET_MESSAGE(sizeof(LaAddHeGroupParams_t));
		laRequestParams = (LaAddHeGroupParams_t*)pK_MSG_DATA(laMessagePtr); 
		laRequestParams->planIndex= planPtr->planIndex; 
		laRequestParams->isStaticGroup = TRUE; 
		laRequestParams->groupId = groupId; 
		if(pMuPlanConfig->commonSection.muType == MIMO)
		{
			planManagerSetStaticNssPerUser(pMuPlanConfig, laRequestParams, psMsg->header.vapId);
		}
		/* Waterfall the TIDs participating in the PLAN */
		OSAL_SEND_MESSAGE(LINK_ADAPTATION_ADD_HE_GROUP, TASK_LINK_ADAPTATION, laMessagePtr, psMsg->header.vapId);	
	}	
}


/**********************************************************************************

planManagerSetStaticDlPhaseCommonFields 

Description: 
Sets the DATA common section in DL pahse 
	
**********************************************************************************/

static void planManagerSetStaticDlPhaseCommonFields(DlDataPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8			muSequenceType;
		
	phaseCommonPtr->phaseType = planManagerDefaultSettings.DlCommonPhase_phaseType; //0 - Protection Phase;  1 - Sounding Phase;  2 - DL Data Phase;  3 - UL Phase
	phaseCommonPtr->oneShotPhase = planManagerDefaultSettings.DlCommonPhase_oneShotPhase; 
	
	phaseCommonPtr->partialDuration = planManagerDefaultSettings.DlCommonPhase_partialDuration; // default NA FULL TXOP 
	phaseCommonPtr->keepPlanLock = planManagerDefaultSettings.DlCommonPhase_keepPlanLock; 
	phaseCommonPtr->moveStaToSu = planManagerDefaultSettings.DlCommonPhase_moveStaToSu; 

	phaseCommonPtr->plannedData = planManagerDefaultSettings.DlCommonPhase_plannedData;

	phaseCommonPtr->primaryUser = 0x0;
	phaseCommonPtr->primaryImmediateBa = planManagerDefaultSettings.DlCommonPhase_primaryImmediateBa; //  vht alike mode 

	muSequenceType = staticPlanConfigurationPtr->commonSection.muSequenceType;
	ILOG0_D("[planManagerSetStaticDlPhaseCommonFields] muSequenceType = %d", muSequenceType);
	
	switch(muSequenceType)
	{
		case HE_MU_SEQ_MU_BAR:
			phaseCommonPtr->tfParamTriggerType =  TF_TYPE_MU_BAR; 
			phaseCommonPtr->plannedMuBarTf = 0x1;
			phaseCommonPtr->plannedBar = 0x0; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x0;// 0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType = 0x2; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			break;
		case HE_MU_SEQ_VHT_LIKE:
		case HE_MU_SEQ_VHT_LIKE_PROTECTION:
			phaseCommonPtr->tfParamTriggerType =  TF_TYPE_BASIC; 
			phaseCommonPtr->plannedMuBarTf = 0x0; 
			phaseCommonPtr->plannedBar = 0x1; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x1; //0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType = 0x1; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			break;	
		case HE_MU_SEQ_DL_BASIC_TF:
			phaseCommonPtr->tfParamTriggerType =  TF_TYPE_BASIC; 
			phaseCommonPtr->plannedMuBarTf = 0x0; 
			phaseCommonPtr->plannedBar = 0x0; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x1; //0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType = 0x0; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			break;	
		case HE_MU_SEQ_VHT_LIKE_IMM_ACK:
			phaseCommonPtr->tfParamTriggerType =  TF_TYPE_BASIC; 
			phaseCommonPtr->plannedMuBarTf = 0x0; 
			phaseCommonPtr->plannedBar = 0x1; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x1; //0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType = 0x1; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			phaseCommonPtr->primaryImmediateBa = 0x1; //primary user will be configured with immediate ACK policy
			break;
		default:
			FATAL("UNKNOWN muSequenceType");
			break;
	}
	

	phaseCommonPtr->minimumNumberOfUsersForTransmission = planManagerDefaultSettings.DlCommonPhase_minimumNumberOfUsersForTransmission; 
	

	phaseCommonPtr->tfInfoBaseAddrDwOffset = CONVERT_BYTES_TO_WORDS(OFFSETOF(DlDataPhase_t, tfParameters));    // offset from phase start to tfParameters
	phaseCommonPtr->currentPsduMaxLengthLimit = planManagerDefaultSettings.DlCommonPhase_currentPsduMaxLengthLimit ; //Open issue  - 6500631 is taken from standart max, it should be calculated according to our capabilities 
#ifdef ENET_INC_ARCH_WAVE600D2
	phaseCommonPtr->dataMuMimo = staticPlanConfigurationPtr->commonSection.muType; // if OFDMA - 0    if MIMO - 1  //MIMO OR OFDMA 
	
	if(phaseCommonPtr->tfBroadcastUnicastMode == TF_BROADCAST_UNICAST_MODE_BROADCAST)
	{
		phaseCommonPtr->multiBssid	= heGroupManagerDb[HE_GROUP_MANAGER_DL_GID].mbssRxCtrlSupport; 
	}
	else
	{
		phaseCommonPtr->multiBssid	= MULTI_BSSID_NO_MULTI_BSSID;
	}
#else
	phaseCommonPtr->muMimo = staticPlanConfigurationPtr->commonSection.muType; // if OFDMA - 0    if MIMO - 1  //MIMO OR OFDMA 
#endif	
	//phaseCommonPtr->totalPreambleTime = // Filled by LA :	ltfDuration +sigbDuration +36 + 16 ;   /* 36micro+SIGB+HE LTF +packet extension (16micro) 
												//		HE SIGB =  commmon part and per user  (in MU mimo there is no common section)
												// 		for 2 users - 52 bit - so    (number of station / 2) * 52 - if need to add one user (if number is odd) the size of it is 31 
												// 		for Z0 -   20Mhz and 40Mhz - 31 , 80hz - 52 
												//CONVERT_SIGBRATE_2_DBPS - 0-26, 1- 52, 2- 78,3 104,4- 156,  5-208 
														
	phaseCommonPtr->maxPpduTransmissionTimeLimit = staticPlanConfigurationPtr->commonSection.maximumPpduTransmissionTimeLimit >> 0x1; //field is in resolution of 2uSec
	phaseCommonPtr->phaseTimeNotIncludingData = BAR_BACK_AIR_TIME_DURATION;/* phase_time_not_including_data  :   
																											Time shall include SIFS +MU BAR + SIFS + BA or (SIFS + BAR + SIFS + BA)x num of users (resolution of 2usec) */ 
#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	phaseCommonPtr->broadcastTfAidCfg = TRUE; // use original AID from phase
	phaseCommonPtr->unicastTfTypeCfg  = TRUE; // use original TF type from phase
#endif	

																												
}

/**********************************************************************************

planManagerSetStaticDlPhaseCommonTcr_A_Fields 

Description: 
Sets the DATA TCR common section in DL pahse 
	
**********************************************************************************/

static void planManagerSetStaticDlPhaseCommonTcr_A_Fields(DlDataPhaseCommonTcra_t*  phaseCommonTcraPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr,uint8 vapId)
{
	TcrCommon_t* phaseCommonDataTcrPtr = (TcrCommon_t*)(phaseCommonTcraPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	
	//phaseCommonDataTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonDataTcrPtr->ant0Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant1Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant2Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant3Boost//Filled by LA
	phaseCommonDataTcrPtr->rfPower = staticPlanConfigurationPtr->commonSection.rfPower; 
	phaseCommonDataTcrPtr->cbw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseCommonDataTcrPtr->phyMode = planManagerDefaultSettings.DlCommonPhaseTcrA_phyMode; 
	phaseCommonDataTcrPtr->heFullbandMu = (staticPlanConfigurationPtr->commonSection.muType == 0) ? 0 : 1 ;// if OFDMA - 0    if MIMO - 1 
	phaseCommonDataTcrPtr->txLoopMode = (staticPlanConfigurationPtr->commonSection.muType == 0) ? TX_LOOP_MODE_CLOSED_LOOP : TX_LOOP_MODE_OPEN_LOOP ;// if OFDMA - closed loop,  if MIMO - open loop 
	phaseCommonDataTcrPtr->hePuncturingMap = planManagerDefaultSettings.DlCommonPhaseTcrA_hePuncturingMap; 
	phaseCommonDataTcrPtr->bssColor =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonDataTcrPtr->customBf = planManagerDefaultSettings.DlCommonPhaseTcrA_customBf; // NA - for prouction 
	//phaseCommonDataTcrPtr->nHeltf   ////Filled by LA -  for MIMO - sum all NSS of users , for OFDMA - MAX NSS of all users ,value is always in power of 2 
	phaseCommonDataTcrPtr->ulDl = planManagerDefaultSettings.DlCommonPhaseTcrA_ulDl;   // set 0 for DL , 1 for UL ,See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonDataTcrPtr->txopPs = planManagerDefaultSettings.DlCommonPhaseTcrA_txopPs; 
	if(staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_DL_BASIC_TF)
	{
		phaseCommonDataTcrPtr->expectedRcr = 0X1; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	}
	else
	{
		phaseCommonDataTcrPtr->expectedRcr = planManagerDefaultSettings.DlCommonPhaseTcrA_expectedRcr; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	}
	phaseCommonDataTcrPtr->bwChange = planManagerDefaultSettings.DlCommonPhaseTcrA_bwChange; 
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 1; 
	}
	else //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 0;
	} 
	phaseCommonDataTcrPtr->aggregate = planManagerDefaultSettings.DlCommonPhaseTcrA_aggregate; 
	phaseCommonDataTcrPtr->dynamicBw = planManagerDefaultSettings.DlCommonPhaseTcrA_dynamicBw; 
	phaseCommonDataTcrPtr->muPhyNdp = 0;
	phaseCommonDataTcrPtr->muTraining = 0;
	phaseCommonDataTcrPtr->heCp = staticPlanConfigurationPtr->commonSection.dl_HeCp; 
	phaseCommonDataTcrPtr->heLtf = staticPlanConfigurationPtr->commonSection.dl_HeLtf; 
	phaseCommonDataTcrPtr->scp = planManagerDefaultSettings.DlCommonPhaseTcrA_scp;//don't care as phy mode is HE  (this field is relevant for HT/VHT)
	//phaseCommonDataTcrPtr->smoothing //Filled by LA 
	phaseCommonDataTcrPtr->notSounding  = planManagerDefaultSettings.DlCommonPhaseTcrA_notSounding; 
	//phaseCommonDataTcrPtr->stbc =  //Filled by LA  (HT/VHT)
	phaseCommonDataTcrPtr->spatialReuse =  planManagerDefaultSettings.DlCommonPhaseTcrA_spatialReuse; //NA
	//phaseCommonDataTcrPtr->triggerBasedDataLLenght  //owned by phy 
	if(staticPlanConfigurationPtr->commonSection.startBwLimit > BANDWIDTH_TWENTY)
	{
		phaseCommonDataTcrPtr->triggerBasedDataLdpc =  0x1; //We must turn LDPC if BW is above 20Mhz 
	}
	else //  BANDWIDTH_TWENTY
	{
		phaseCommonDataTcrPtr->triggerBasedDataLdpc =  0x0; // No LDPC for 20Mhz
	}
	phaseCommonDataTcrPtr->triggerBasedDataPktExt =  planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataPktExt; //Filled by LA , pre-FEC padding fator , 0 means 4 
	phaseCommonDataTcrPtr->triggerBasedDataPeDisam =  planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataPeDisam; //  
	phaseCommonDataTcrPtr->groupId =  HE_GROUP_MANAGER_DL_GID; // in the future it will be set from group manager   
	//phaseCommonDataTcrPtr->heSigRate  = 0x0;  //to be filled by LA  (MCS 0, NSS - according to users and DCM = 0 )
	phaseCommonDataTcrPtr->macDuration = planManagerDefaultSettings.DlCommonPhaseTcrA_macDuration; 
}


/**********************************************************************************

planManagerSetStaticDlPhaseCommonTcr_B_Fields 

Description: 
Sets the MU BAR TCR common section in DL pahse 
	
**********************************************************************************/

static void planManagerSetStaticDlPhaseCommonTcr_B_Fields(DlDataPhaseCommonTcrb_t*  phaseCommonTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr,uint8 vapId)
{
	TcrCommon_t* phaseCommonMuBarTcrPtr = (TcrCommon_t*)(phaseCommonTcrbPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	
	//phaseCommonMuBarTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonMuBarTcrPtr->ant0Boost//Filled by LA
	//phaseCommonMuBarTcrPtr->ant1Boost//Filled by LA
	//phaseCommonMuBarTcrPtr->ant2Boost//Filled by LA
	//phaseCommonMuBarTcrPtr->ant3Boost//Filled by LA
	phaseCommonMuBarTcrPtr->rfPower  = staticPlanConfigurationPtr->commonSection.rfPower;
	phaseCommonMuBarTcrPtr->txLoopMode = (staticPlanConfigurationPtr->commonSection.muType == 0) ? TX_LOOP_MODE_CLOSED_LOOP : TX_LOOP_MODE_OPEN_LOOP ;// if OFDMA - closed loop,  if MIMO - open loop ; 
	phaseCommonMuBarTcrPtr->cbw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseCommonMuBarTcrPtr->phyMode = planManagerDefaultSettings.DlCommonPhaseTcrB_phyMode;
	phaseCommonMuBarTcrPtr->heFullbandMu = (staticPlanConfigurationPtr->commonSection.muType == 0) ? 0 : 1;// if OFDMA - 0    if MIMO - 1 
	phaseCommonMuBarTcrPtr->hePuncturingMap = planManagerDefaultSettings.DlCommonPhaseTcrB_hePuncturingMap; 
	phaseCommonMuBarTcrPtr->bssColor =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonMuBarTcrPtr->customBf = planManagerDefaultSettings.DlCommonPhaseTcrB_customBf; // NA - for prouction 
	//phaseCommonMuBarTcrPtr->nHeltf = 4;  //Filled by LA 
	phaseCommonMuBarTcrPtr->ulDl = planManagerDefaultSettings.DlCommonPhaseTcrB_ulDl;    // set 0 for DL , 1 for UL ,See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonMuBarTcrPtr->txopPs = planManagerDefaultSettings.DlCommonPhaseTcrB_txopPs; 
	phaseCommonMuBarTcrPtr->expectedRcr = planManagerDefaultSettings.DlCommonPhaseTcrB_expectedRcr; //RCR expected , This packet is trigger frame - next receive is trigger based
	phaseCommonMuBarTcrPtr->bwChange = planManagerDefaultSettings.DlCommonPhaseTcrB_bwChange; 
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonMuBarTcrPtr->txopDurationFromMac = 1; 
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonMuBarTcrPtr->txopDurationFromMac = 0;
	}
	phaseCommonMuBarTcrPtr->aggregate = planManagerDefaultSettings.DlCommonPhaseTcrB_aggregate; //BAR is not aggregated 
	phaseCommonMuBarTcrPtr->dynamicBw = planManagerDefaultSettings.DlCommonPhaseTcrB_dynamicBw; 
	phaseCommonMuBarTcrPtr->muPhyNdp = 0;
	phaseCommonMuBarTcrPtr->muTraining = 0;
	//phaseCommonMuBarTcrPtr->heCp = 	// don't care - only for HE 
	//phaseCommonMuBarTcrPtr->heLtf =  // don't care - only for HE  
	//phaseCommonMuBarTcrPtr->scp = planManagerDefaultSettings.DlCommonPhaseTcrB_scp; // filed by LA 
	//phaseCommonMuBarTcrPtr->smoothing //Filled by LA 
	phaseCommonMuBarTcrPtr->notSounding  = planManagerDefaultSettings.DlCommonPhaseTcrB_notSounding; 
	//phaseCommonMuBarTcrPtr->stbc =  //Filled by LA  (HT/VHT)
	phaseCommonMuBarTcrPtr->spatialReuse =  planManagerDefaultSettings.DlCommonPhaseTcrB_spatialReuse; //NA
	//phaseCommonMuBarTcrPtr->triggerBasedDataLLenght  = // owned by the phy 
	phaseCommonMuBarTcrPtr->triggerBasedDataLdpc =   planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataLdpc;  //no LDPC
	phaseCommonMuBarTcrPtr->triggerBasedDataPktExt =  planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataPktExt;
	phaseCommonMuBarTcrPtr->triggerBasedDataPeDisam =  planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataPeDisam; //NA  
	phaseCommonMuBarTcrPtr->groupId =  HE_GROUP_MANAGER_DL_GID;  
	//phaseCommonMuBarTcrPtr->heSigRate  =  0x0; //Filled by LA - MCS 0, DCM0 , NSS  -1 
	phaseCommonMuBarTcrPtr->macDuration = planManagerDefaultSettings.DlCommonPhaseTcrB_macDuration; 
	

}

/**********************************************************************************

planManagerSetStaticDlPhaseCommonRcrFields 

Description: 
Sets the RCR common section in DL pahse 
	
**********************************************************************************/

static void planManagerSetStaticDlPhaseCommonRcrFields(DlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId)
{
	RcrCommon_t* phaseRcrCommonPtr = (RcrCommon_t*)(phaseCommonRcrPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];

	phaseRcrCommonPtr->heSigABssColor  =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseRcrCommonPtr->cbw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	//phaseRcrCommonPtr->nHeltf = //Filled by LA -
	phaseRcrCommonPtr->heCp = staticPlanConfigurationPtr->commonSection.ul_HeCp; 
	phaseRcrCommonPtr->heLtf = staticPlanConfigurationPtr->commonSection.ul_HeLtf;  
	phaseRcrCommonPtr->stbc = staticPlanConfigurationPtr->commonSection.rcr_Stbc;
	phaseRcrCommonPtr->heSigASpatialReuse = staticPlanConfigurationPtr->commonSection.rcr_heSigASpatialReuse; 
	phaseRcrCommonPtr->lSigLength = staticPlanConfigurationPtr->commonSection.tf_Length; //TBD 
	//phaseRcrCommonPtr->ldpcExtraSymbol = //Filled by LA 
	phaseRcrCommonPtr->peDisambiguty =  planManagerDefaultSettings.DlCommonPhaseRcr_peDisambiguty;// We force the station to set PE disambiguty   
}
/**********************************************************************************

planManagerSetStaticDlPhaseCommonUserTCR_B_Fields 

Description: 
Sets the common user TCRB common section in DL pahse  - how to transmit the MU BAR 
	
**********************************************************************************/

static void planManagerSetStaticDlPhaseCommonUserTCR_B_Fields(DlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	commonUserTcrbPtr->brdcstUserTcrbSubBand = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_SubBand; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbStartRu = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_StartRu; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbRuSize = staticPlanConfigurationPtr->perUserParameters[0].ul_ruSize; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbLdpc = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_Ldpc; 
	commonUserTcrbPtr->brdcstUserTcrbTxbf = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_Txbf; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPsduRate = 0;  //filed by LA 
	commonUserTcrbPtr->brdcstUserTcrbRelativeUserPower =planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_RelativeUserPower; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPacketExtension = // filed by LA 0; /
	commonUserTcrbPtr->brdcstUserTcrbStaAid = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_TcrbStaAid; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbBfIndex = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_BfIndex; // NA   trigger frame transmitted in legacy phy mode

}

/**********************************************************************************

planManagerSetStaticDlPhaseCommonTfParameters 

Description: 
Sets the common  Trigger frame  section in DL pahse 
	
**********************************************************************************/
static void planManagerSetStaticDlPhaseCommonTfParameters(DlDataPhase_t* dlDataPhasPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 muSequenceType;
	DlDataPhaseTfParameters_t* commonTfParamsPtr = &(dlDataPhasPtr->tfParameters); 
	DlDataPhaseTfCommonInfo_t* commonTfInfoPtr = &(dlDataPhasPtr->tfCommonInfo); 

	commonTfParamsPtr->tfOverrideEnableParams		= 0xFF; // make the default balue 1 for all 
	commonTfParamsPtr->tfBwOverride					=	planManagerDefaultSettings.DlCommonTf_tfBwOverride;
#ifdef ENET_INC_ARCH_WAVE600B
#ifdef ENET_INC_ARCH_WAVE600D2
	commonTfParamsPtr->tfMoreTfOverride 			=   0; 
#else
	commonTfParamsPtr->tfCascadeIndicationOverride 	=   0; 
#endif
	commonTfParamsPtr->tfRuIndexOverride			=	planManagerDefaultSettings.DlCommonTf_tfRuAllocationOverride;
#else
	commonTfParamsPtr->tfRuAllocationOverride		=	planManagerDefaultSettings.DlCommonTf_tfRuAllocationOverride;
#endif 
	commonTfParamsPtr->tfMcsOverride				=	planManagerDefaultSettings.DlCommonTf_tfMcsOverride;
	commonTfParamsPtr->tfDcmOverride				=	planManagerDefaultSettings.DlCommonTf_tfDcmOverride;
	commonTfParamsPtr->tfNssOverride				=	planManagerDefaultSettings.DlCommonTf_tfNssOverride;
	commonTfParamsPtr->tfTargetRssiOverride			=	planManagerDefaultSettings.DlCommonTf_tfTargetRssiOverride;
	commonTfParamsPtr->tfPreferredAcOverride		=	planManagerDefaultSettings.DlCommonTf_tfPreferredAcOverride;	
	commonTfParamsPtr->tfCommonInfoSwOverrideEnable = planManagerDefaultSettings.DlCommonTf_CommonInfoSwOverrideEnable;; // only in case we want to override the SW info with a configured value in TF generator register 
	commonTfParamsPtr->tfUserInfoSwOverrideEnable = planManagerDefaultSettings.DlCommonTf_UserInfoSwOverrideEnable; // only in case we want to override the SW info with a configured value in TF generator register 
	commonTfParamsPtr->tfLsigLengthOverride = 0; //LSIG length musn't be override  
	commonTfParamsPtr->tfTidAggrLimitOverride = 0; //in case of DL the limit should be set in TF to zero 
	commonTfParamsPtr->tfMultiTidOverride = 1; 
	commonTfParamsPtr->tfTidInfoOverride = 1; 
	commonTfParamsPtr->tfCompressedFnOverride = 1; 
	commonTfParamsPtr->tfCompressedSsnOverride = 1; 
	commonTfParamsPtr->tfMultiTidTidOverride = 1;
	commonTfParamsPtr->tfMultiTidFnOverride = 1;
	commonTfParamsPtr->tfMultiTidSsnOverride = 1;
    
    muSequenceType = staticPlanConfigurationPtr->commonSection.muSequenceType;
	switch(muSequenceType)
	{
		case HE_MU_SEQ_MU_BAR:
			commonTfInfoPtr->tfTriggerType =  TF_TYPE_MU_BAR; 
			break;
		case HE_MU_SEQ_VHT_LIKE:
		case HE_MU_SEQ_VHT_LIKE_IMM_ACK:
		case HE_MU_SEQ_VHT_LIKE_PROTECTION:
			//don't care in this case since there is no TF
			break;
		case HE_MU_SEQ_DL_BASIC_TF:
			commonTfInfoPtr->tfTriggerType =  TF_TYPE_BASIC; 
			break;
		default:
			FATAL("[planManagerSetStaticDlPhaseCommonTfParameters], UNKNOWN muSequenceType");
			break;
	}	
		
	commonTfInfoPtr->tfLength =  staticPlanConfigurationPtr->commonSection.tf_Length;//This is the L-SIG length we want the STAs to transmitt - amount of bytes for station to transmit
	
#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfInfoPtr->tfMoreTf = planManagerDefaultSettings.DlCommonTf_tfMoreTf;  //NA															
#else
	commonTfInfoPtr->tfCascadeIndication = planManagerDefaultSettings.DlCommonTf_CascadeIndication;  //NA															
#endif
	commonTfInfoPtr->tfCsRequired = planManagerDefaultSettings.DlCommonTf_CsRequired; // should a station consider CCA before UL  - only For MU RTS should be set with 1  
	commonTfInfoPtr->tfBw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	commonTfInfoPtr->tfGiAndLtfType = staticPlanConfigurationPtr->commonSection.tf_heGI_and_Ltf; 
	commonTfInfoPtr->tfMuMimoLtfMode =  planManagerDefaultSettings.DlCommonTf_MuMimoLtfMode; //0: Single Stream Pilots   1 : Mask on each spatial stream of the LTF sequence by a distinct orthogonal code
	//if mimo -N_HE_LTF - sum of all NSS , in case of  OFDMA - the max NSS of all users 
	//commonTfInfoPtr->tfNumberOfHeLtfSymbols = //to be set by LA 
	commonTfInfoPtr->tfStbc = 0; //to be set by LA 
	//commonTfInfoPtr->tfLdpcExtraSymbolSegment =1; // to be set by LA  
	
	//commonTfInfoPtr->tfApTxPower30 = //set by LA (staticPlanConfigurationPtr->commonSection.rfPower & 0x7); //Set 3 LSBs  
	//commonTfInfoPtr->tfApTxPower54 = //set by LA (staticPlanConfigurationPtr->commonSection.rfPower & 0x18); // set 2 MSBs 
	//commonTfInfoPtr->tfPacketExtension = // Station capability - to be filled by LA - Max of all users 
	commonTfInfoPtr->tfSpatialReuse = planManagerDefaultSettings.DlCommonTf_SpatialReuse; //NA 
	commonTfInfoPtr->tfDoppler = planManagerDefaultSettings.DlCommonTf_Doppler; //NA 
#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfInfoPtr->tfHeSigA2Reserved= planManagerDefaultSettings.tfHeSigA2Reserved; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#else
	commonTfInfoPtr->tfHeSigAReserved= planManagerDefaultSettings.tfHeSigAReserved; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#endif	
	commonTfInfoPtr->tfReservedCommonInfo= 0x0; 

}


/**********************************************************************************

planManagerSetStaticDlPhaseCommonTfMuBar 

Description: 
Sets the common  TF MU BAR section in DL pahse 
	
**********************************************************************************/

static void planManagerSetStaticDlPhaseCommonTfMuBar(DlDataPhaseTfMuBar_t* dlDataPhasTfMuBarPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	UNUSED_PARAM(staticPlanConfigurationPtr);	
	///dlDataPhasTfMuBarPtr->tfMuBarControl = // union of the below fields: 
	dlDataPhasTfMuBarPtr->tfBarAckPolicy = planManagerDefaultSettings.DlCommonTf_tfBarAckPolicy;
	dlDataPhasTfMuBarPtr->tfMultiTid = planManagerDefaultSettings.DlCommonTf_tfMultiTid;
	dlDataPhasTfMuBarPtr->tfCompressedBitmap = planManagerDefaultSettings.DlCommonTf_CompressedBitmap;
	dlDataPhasTfMuBarPtr->tfGcr = planManagerDefaultSettings.DlCommonTf_tfGcr;
	dlDataPhasTfMuBarPtr->tfReservedBarControl =planManagerDefaultSettings.DlCommonTf_ReservedBarControl;
	dlDataPhasTfMuBarPtr->tfTidInfo =  1;  //  filled by HW - pre-aggregator  
	//dlDataPhasTfMuBarPtr->tfCompressedMuBarInformation  // union of the below fields:
	dlDataPhasTfMuBarPtr->tfCompressedFn0  = planManagerDefaultSettings.DlCommonTf_CompressedFn0 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedFn1 = planManagerDefaultSettings.DlCommonTf_CompressedFn1 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedFn2 = planManagerDefaultSettings.DlCommonTf_CompressedFn2 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedFn3 = planManagerDefaultSettings.DlCommonTf_CompressedFn3 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedStartingSequenceNumber = planManagerDefaultSettings.DlCommonTf_CompressedStartingSequenceNumber;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidMuBarInformation = planManagerDefaultSettings.DlCommonTf_MultiTidMuBarInformation ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidReserved = planManagerDefaultSettings.DlCommonTf_MultiTidReserved ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidTid = planManagerDefaultSettings.DlCommonTf_MultiTidTid ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn0 = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn0 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn1 = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn1;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn2 = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn2 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn3 = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn3 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidStartingSequenceNumber = planManagerDefaultSettings.DlCommonTf_MultiTidStartingSequenceNumber ;//override by the RCR configuration 

	
}

/**********************************************************************************

planManagerSetStaticDlPhasePerUser 

Description: 
Sets the per user data section in DL phase 
1. Find RU size that should be filled per user (Static Planner assumes euqal allocation between users)
2. Iterate over RU bitmap to find RU configuration needed until all bitmap (specturm) is allocated
	2.1 find SB index and RU index of conifguration needed and see if user has entered this information 
		if user entered:
			2.1.1 fill all relevant information of user
			2.1.2 mark in bitmap RU allocated according to user's ru size
		else:
			2.1.1 fill all relevant information for dummy user
			2.1.2 mark bitmap accordingly
		go back to step #2
Input: 
-----
	Bw_e
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerSetStaticDlPhasePerUser(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, Bw_e bw)
{
    uint8   vapId;
	uint8 	userLoopIndex; 
	uint8   userInputIndex;
	
	PreAggSubBandIndex_e	currentSBindex;
	PreAggStartRuIndex_e   	currentRUindex;
	PreAggRuSizeMapping_e   ruSize;

	uint8  userStationIndex;
	uint8  firstBitmapIndex = 0; 						// index of first bit that is set to 0 in RUbitmap
	uint8  lastBitmapIndex = LAST_BIT_INDEX_SB7;  		// index of last bit that is set to 0 in RUbitmap
	bool   ldpcDummyOn = FALSE;

	// init 
	userLoopIndex = 0;  	// index over all users both active and dummy
	userInputIndex = 0; 	// index over configured users
	currentSBindex = PRE_AGG_SUB_BAND_0;
	currentRUindex = PRE_AGG_START_RU_0;

	// find RU size - get first user's RU size assuming equal allocation between users
	ruSize = staticPlanConfigurationPtr->perUserParameters[0].dl_ruSizePerUsp;
	// update RU bitmap according to RU size
	planManagerRuAllocator_updateRUAllocAccordingToBwNRuSize(ruSize, bw);
	// find first RU index in bitmap indicating hole from 0 to 63
	firstBitmapIndex = planManagerRuAllocator_findLocationOfNextLeadingZero(firstBitmapIndex);
#ifdef DEBUG_RU_ALLOCATOR	
    ILOG0_DDD("[planManagerSetStaticDlPhasePerUser] start ruSize = %d, firstBitmapIndex = %d, lastBitmapIndex = %d", ruSize, firstBitmapIndex, lastBitmapIndex);
#endif

	// get pointer to phase
	dlDataPhasPerUserPtr = &(dlDataPhasPerUserPtr[0]); 

	while(firstBitmapIndex <= lastBitmapIndex) // user was not configured need to fill this hole
	{	
		// intialize pointer to phase
		memset(dlDataPhasPerUserPtr, 0x0, sizeof(DlDataPhaseUserPhase_t)); // some fields (for example userRcrValid) must be set with zeros for invalid users

		//calc SB index and RU index that need to be filled
		planManagerRuAllocator_findSBIndexRUIndex(firstBitmapIndex, &currentSBindex, &currentRUindex);
#ifdef DEBUG_RU_ALLOCATOR			
        ILOG0_DD("[planManagerSetStaticDlPhasePerUser] after planManagerRuAllocator_findSBIndexRUIndex currentSBindex = %d, currentRUindex = %d", currentSBindex, currentRUindex);
		ILOG0_DD("[planManagerSetStaticDlPhasePerUser], userInputIndex = %d, staticPlanConfigurationPtr->commonSection.numOfParticipatingStations = %d", userInputIndex, staticPlanConfigurationPtr->commonSection.numOfParticipatingStations);
#endif		

		// if current user has allocation for the above SB index and RU index
		if ( (userInputIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations) && 
			(currentSBindex == staticPlanConfigurationPtr->perUserParameters[userInputIndex].dl_subBandPerUsp) && 
			(currentRUindex == staticPlanConfigurationPtr->perUserParameters[userInputIndex].dl_startRuPerUsp) ) //allocation was given by user
		{
			userStationIndex = staticPlanConfigurationPtr->perUserParameters[userInputIndex].uspStationIndexes;
							
			dlDataPhasPerUserPtr->staId =  userStationIndex;
			dlDataPhasPerUserPtr->userTcraStaAid = userStationIndex+1; 
            StaDb_GetVapId(userStationIndex, &vapId);
            dlDataPhasPerUserPtr->vapId = vapId;
			
			dlDataPhasPerUserPtr->tfPadding = staticPlanConfigurationPtr->perUserParameters[userInputIndex].tfPadding; 
			dlDataPhasPerUserPtr->numTfRepetition = planManagerDefaultSettings.DlPerUser_numTfRepetition; 
			if ((staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE) || (staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_IMM_ACK) || (staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_PROTECTION))
			{
				dlDataPhasPerUserPtr->mapTxUserToRxUser =  0x0; // ba is always in user 0 (same as in su)
			}
			else
			{
				dlDataPhasPerUserPtr->mapTxUserToRxUser =  userInputIndex; //Mapping is 1 X 1 for Z0
			}
			dlDataPhasPerUserPtr->tidAllocBitmap = staticPlanConfigurationPtr->perUserParameters[userInputIndex].tid_alloc_bitmap;
			heGroupManagerSetPlanTidsBitmap(dlDataPhasPerUserPtr->tidAllocBitmap, staticPlanConfigurationPtr->commonSection.muSequenceType);
			dlDataPhasPerUserPtr->maxTids = MIN(NUM_OF_TIDS , NUM_OF_TID_INDEXES/ staticPlanConfigurationPtr->commonSection.numOfParticipatingStations);// Minimum between  station cap  and  (36 / #users) 
			dlDataPhasPerUserPtr->tidSelectionPolicy = 0x2;   /*TID Selection Mode:
																00 Select TID with AC >= AC Primary
																01- Select TD with AC <= AC Primary
																10  Select Any TID (starting with AC>= AC Primary */
			dlDataPhasPerUserPtr->userPhaseReserve10 = 0x0; 
			dlDataPhasPerUserPtr->ampduLimit = StaDbHwEntries[userStationIndex].common.aMpduLimit; 
			dlDataPhasPerUserPtr->currUserToPrevPpduMap = 0x7F;// In Z0 Users must be in the same order in all phases
			dlDataPhasPerUserPtr->tid0RrWeight 	= planManagerDefaultSettings.DlPerUser_tid0RrWeight; //NA 
			dlDataPhasPerUserPtr->tid1RrWeight	= planManagerDefaultSettings.DlPerUser_tid1RrWeight; //NA 
			dlDataPhasPerUserPtr->tid2RrWeight	= planManagerDefaultSettings.DlPerUser_tid2RrWeight; //NA 
			dlDataPhasPerUserPtr->tid3RrWeight	= planManagerDefaultSettings.DlPerUser_tid3RrWeight; //NA 	
			dlDataPhasPerUserPtr->tid4RrWeight	= planManagerDefaultSettings.DlPerUser_tid4RrWeight; //NA 
			dlDataPhasPerUserPtr->tid5RrWeight	= planManagerDefaultSettings.DlPerUser_tid5RrWeight; //NA 
			dlDataPhasPerUserPtr->tid6RrWeight	= planManagerDefaultSettings.DlPerUser_tid6RrWeight; //NA 
			dlDataPhasPerUserPtr->tid7RrWeight	= planManagerDefaultSettings.DlPerUser_tid7RrWeight; //NA 
			//dlDataPhasPerUserPtr->currMaxMsduAtAmsdu => value is set by LA
			dlDataPhasPerUserPtr->heMngAgg = planManagerDefaultSettings.DlPerUser_heMngAgg; //No management in agg 
			dlDataPhasPerUserPtr->heBarAgg = planManagerDefaultSettings.DlPerUser_heBarAgg; 
			dlDataPhasPerUserPtr->aControlIncluded = planManagerDefaultSettings.DlPerUser_aControlIncluded;  /*A-Control (HT) Included  0  not included 1  included */
			/* RU ALLOCATOR SIMULATION */
			dlDataPhasPerUserPtr->userRcrSubBand = staticPlanConfigurationPtr->perUserParameters[userInputIndex].ul_SubBand; 
			dlDataPhasPerUserPtr->userRcrStartRu = staticPlanConfigurationPtr->perUserParameters[userInputIndex].ul_StartRU; 
			dlDataPhasPerUserPtr->userRcrRuSize = staticPlanConfigurationPtr->perUserParameters[userInputIndex].ul_ruSize; 
			dlDataPhasPerUserPtr->userRcrLdpc = staticPlanConfigurationPtr->perUserParameters[userInputIndex].ul_ldpc; 
			dlDataPhasPerUserPtr->userRcrTxbf = planManagerDefaultSettings.DlPerUser_userRcrTxbf;
			dlDataPhasPerUserPtr->userRcrPsduRate = staticPlanConfigurationPtr->perUserParameters[userInputIndex].ul_PsduRatePerUsp;
			dlDataPhasPerUserPtr->userRcrTargetRssi = staticPlanConfigurationPtr->perUserParameters[userInputIndex].targetRssi;
			dlDataPhasPerUserPtr->userRcrReserve00 = 0x0; 
			dlDataPhasPerUserPtr->userRcrValid = planManagerDefaultSettings.DlPerUser_RcrValid; 
			dlDataPhasPerUserPtr->userTcraSubBand = staticPlanConfigurationPtr->perUserParameters[userInputIndex].dl_subBandPerUsp;
			dlDataPhasPerUserPtr->userTcraStartRu = staticPlanConfigurationPtr->perUserParameters[userInputIndex].dl_startRuPerUsp;
			dlDataPhasPerUserPtr->userTcraRuSize = staticPlanConfigurationPtr->perUserParameters[userInputIndex].dl_ruSizePerUsp;
			dlDataPhasPerUserPtr->userTcraLdpc = staticPlanConfigurationPtr->perUserParameters[userInputIndex].codingType_BCC_OR_LDPC;
			ldpcDummyOn = dlDataPhasPerUserPtr->userTcraLdpc;
			dlDataPhasPerUserPtr->userTcraTxbf = 0; //this is dl phase for ofdma only
			dlDataPhasPerUserPtr->userTcraPsduRate = staticPlanConfigurationPtr->perUserParameters[userInputIndex].dl_PsduRatePerUsp; 
			dlDataPhasPerUserPtr->userTcraRelativeUserPower = planManagerDefaultSettings.DlPerUser_TcraRelativeUserPower; 
			dlDataPhasPerUserPtr->userTcraReserve00  = 0x0;  
			//dlDataPhasPerUserPtr->userTcraPacketExtension = 2;   //set by LA   : 0  0 usec ; 1  8 usec ; 2  16 usec
			dlDataPhasPerUserPtr->userTcraReserve01  = 0x0;  
			dlDataPhasPerUserPtr->userTcraReserve10 = 0x0;  
			dlDataPhasPerUserPtr->userTcraBfIndex  = userStationIndex; //SID in OFDMA
			dlDataPhasPerUserPtr->userTcraReserve11 = 0x0;  
			//Update statistics
			if(staticPlanConfigurationPtr->commonSection.muType == OFDMA)
			{
				PlanManagerStatistics.downlinkMuStats[userStationIndex].downlinkMuType = DL_MU_TYPE_HE;
			}
			else
			{
				PlanManagerStatistics.downlinkMuStats[userStationIndex].downlinkMuType = DL_MU_TYPE_MIMO;
			}
			PlanManagerStatistics.downlinkMuStats[userStationIndex].dlRuSubChannels = dlDataPhasPerUserPtr->userTcraSubBand;
			PlanManagerStatistics.downlinkMuStats[userStationIndex].dlRuType = dlDataPhasPerUserPtr->userTcraRuSize;
			
			userInputIndex++; //progress userIndex to next one
		}
		else // ran out of configured users or users were not allocated to specific SB and RU indexes, therefore have to configure dummy user
		{
			planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, currentSBindex, currentRUindex, ruSize, ldpcDummyOn, userInputIndex);
		}

		// update RU allocation bitmap according to RU that were filled
		planManagerRuAllocator_updateRuBitmap(ruSize, firstBitmapIndex);

		// find first index in bitmap indicating hole to indicate first RU allocation to fill from left(index #0) to right(index #64)
		firstBitmapIndex = planManagerRuAllocator_findLocationOfNextLeadingZero(firstBitmapIndex);
#ifdef DEBUG_RU_ALLOCATOR			
		ILOG0_D("[planManagerSetStaticDlPhasePerUser], firstBitmapIndex = %d", firstBitmapIndex);
#endif
		// move to next index
		dlDataPhasPerUserPtr++;
		userLoopIndex++;
	}		
	
	//update number of participating stations to include both active and dummy users
	staticPlanConfigurationPtr->commonSection.numOfParticipatingStations = userLoopIndex;
#ifdef DEBUG_RU_ALLOCATOR		
	ILOG0_D("[planManagerSetStaticDlPhasePerUser] end staticPlanConfigurationPtr->commonSection.numOfParticipatingStations = %d", staticPlanConfigurationPtr->commonSection.numOfParticipatingStations);
#endif
}


/**********************************************************************************

planManagerSetStaticDlMimoPhasePerUser 

Description: 

Input: 
-----
	Bw_e
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerSetStaticDlMimoPhasePerUser(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;

	// get pointer to phase
	dlDataPhasPerUserPtr = &(dlDataPhasPerUserPtr[0]); 

	for(userLoopIndex = 0; userLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; userLoopIndex++ )
	{	
		// intialize pointer to phase
		memset(dlDataPhasPerUserPtr, 0x0, sizeof(DlDataPhaseUserPhase_t)); // some fields (for example userRcrValid) must be set with zeros for invalid users
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes;
						
		dlDataPhasPerUserPtr->staId =  userStationIndex;
		dlDataPhasPerUserPtr->userTcraStaAid = userStationIndex+1; 
		
		//dlDataPhasPerUserPtr->vapId = StationDatabase.StaDbSwEntries[userStationIndex].vapIndex;
		dlDataPhasPerUserPtr->tfPadding = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfPadding; 
		dlDataPhasPerUserPtr->numTfRepetition = planManagerDefaultSettings.DlPerUser_numTfRepetition; 
		if ((staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE) || (staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_PROTECTION))
		{
			dlDataPhasPerUserPtr->mapTxUserToRxUser =  0x0; // ba is always in user 0 (same as in su)
		}
		else
		{
			dlDataPhasPerUserPtr->mapTxUserToRxUser =  userLoopIndex; //Mapping is 1 X 1 for Z0
		}
		dlDataPhasPerUserPtr->tidAllocBitmap = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tid_alloc_bitmap;		
		heGroupManagerSetPlanTidsBitmap(dlDataPhasPerUserPtr->tidAllocBitmap, staticPlanConfigurationPtr->commonSection.muSequenceType);
		dlDataPhasPerUserPtr->maxTids = MIN(NUM_OF_TIDS , NUM_OF_TID_INDEXES/ staticPlanConfigurationPtr->commonSection.numOfParticipatingStations);// Minimum between  station cap  and  (36 / #users) 
		dlDataPhasPerUserPtr->tidSelectionPolicy = 0x2;   /*TID Selection Mode:
															00 Select TID with AC >= AC Primary
															01- Select TD with AC <= AC Primary
															10  Select Any TID (starting with AC>= AC Primary */
		dlDataPhasPerUserPtr->userPhaseReserve10 = 0x0; 
		dlDataPhasPerUserPtr->ampduLimit = StaDbHwEntries[userStationIndex].common.aMpduLimit; 
		dlDataPhasPerUserPtr->currUserToPrevPpduMap = 0x7F;// In Z0 Users must be in the same order in all phases
		dlDataPhasPerUserPtr->tid0RrWeight 	= planManagerDefaultSettings.DlPerUser_tid0RrWeight; //NA 
		dlDataPhasPerUserPtr->tid1RrWeight	= planManagerDefaultSettings.DlPerUser_tid1RrWeight; //NA 
		dlDataPhasPerUserPtr->tid2RrWeight	= planManagerDefaultSettings.DlPerUser_tid2RrWeight; //NA 
		dlDataPhasPerUserPtr->tid3RrWeight	= planManagerDefaultSettings.DlPerUser_tid3RrWeight; //NA 	
		dlDataPhasPerUserPtr->tid4RrWeight	= planManagerDefaultSettings.DlPerUser_tid4RrWeight; //NA 
		dlDataPhasPerUserPtr->tid5RrWeight	= planManagerDefaultSettings.DlPerUser_tid5RrWeight; //NA 
		dlDataPhasPerUserPtr->tid6RrWeight	= planManagerDefaultSettings.DlPerUser_tid6RrWeight; //NA 
		dlDataPhasPerUserPtr->tid7RrWeight	= planManagerDefaultSettings.DlPerUser_tid7RrWeight; //NA 
		//dlDataPhasPerUserPtr->currMaxMsduAtAmsdu => value is set by LA
		dlDataPhasPerUserPtr->heMngAgg = planManagerDefaultSettings.DlPerUser_heMngAgg; //No management in agg 
		dlDataPhasPerUserPtr->heBarAgg = planManagerDefaultSettings.DlPerUser_heBarAgg; 
		dlDataPhasPerUserPtr->aControlIncluded = planManagerDefaultSettings.DlPerUser_aControlIncluded;  /*A-Control (HT) Included  0  not included 1  included */
		/* RU ALLOCATOR SIMULATION */
		dlDataPhasPerUserPtr->userRcrSubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand; 
		dlDataPhasPerUserPtr->userRcrStartRu = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_StartRU; 
		dlDataPhasPerUserPtr->userRcrRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize; 
		dlDataPhasPerUserPtr->userRcrLdpc = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc; 
		dlDataPhasPerUserPtr->userRcrTxbf  = planManagerDefaultSettings.DlPerUser_userRcrTxbf;
		dlDataPhasPerUserPtr->userRcrPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp;
		dlDataPhasPerUserPtr->userRcrTargetRssi = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		dlDataPhasPerUserPtr->userRcrReserve00 = 0x0; 
		dlDataPhasPerUserPtr->userRcrValid = planManagerDefaultSettings.DlPerUser_RcrValid; 
		dlDataPhasPerUserPtr->userTcraSubBand = 0;
		dlDataPhasPerUserPtr->userTcraStartRu = 0;
		// should be configured according to BW size
		dlDataPhasPerUserPtr->userTcraRuSize = CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit);
		dlDataPhasPerUserPtr->userTcraLdpc = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
		dlDataPhasPerUserPtr->userTcraTxbf = TRUE; // It is mimo
		dlDataPhasPerUserPtr->userTcraPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_PsduRatePerUsp; 
		dlDataPhasPerUserPtr->userTcraRelativeUserPower = planManagerDefaultSettings.DlPerUser_TcraRelativeUserPower; 
		dlDataPhasPerUserPtr->userTcraReserve00  = 0x0;  
		//dlDataPhasPerUserPtr->userTcraPacketExtension = 2;   //set by LA   : 0  0 usec ; 1  8 usec ; 2  16 usec
		dlDataPhasPerUserPtr->userTcraReserve01  = 0x0;  
		dlDataPhasPerUserPtr->userTcraReserve10 = 0x0;  
		dlDataPhasPerUserPtr->userTcraBfIndex  = CONVERT_USP_INDEX_TO_HE_MU_BF_INDEX(userLoopIndex); 
		dlDataPhasPerUserPtr->userTcraReserve11 = 0x0;  
		// move to next index
		dlDataPhasPerUserPtr++;
	}		
}





/****************************************************************************************************************

planManagerAreUsersRuAllocationsEqual 

Description: 
This function goes over the user's configuration for RU allocation betweeen the users and check if they are equal
Return TRUE if allocation is equal between users and FALSE otherwise 
	
****************************************************************************************************************/	
static bool planManagerAreUsersRuAllocationsEqual(UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8	userLoopIndex = 0;
	uint8	secondUserLoopIndex;
	bool	isEqualRuAllocation = TRUE; 

	// In case MU-MIMO isEqualRuAllocation == TRUE else check allocations given
	if(staticPlanConfigurationPtr->commonSection.muType == OFDMA)
	{
		while((userLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations) && (isEqualRuAllocation == TRUE))
		{
			for(secondUserLoopIndex = userLoopIndex+1; secondUserLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; secondUserLoopIndex++)
			{
				if(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp != staticPlanConfigurationPtr->perUserParameters[secondUserLoopIndex].dl_ruSizePerUsp)
				{	
					isEqualRuAllocation = FALSE; 
					break; 
				}
			}
			userLoopIndex++;
		}
	}

	return isEqualRuAllocation; 
}
/****************************************************************************************************************

planManagerSetStaticDlPhasePerUserWithUnEqualRus 

Description: 
Sets the per user data section in DL phase in case the RU allocation between users isn't equal 
Dummy users are being configured for all unallocated RUs, in size of the minimal allocated RU for a real user  

	
****************************************************************************************************************/	
static void planManagerSetStaticDlPhasePerUserWithUnEqualRus(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, Bw_e bw)
{
	uint32					minimalRuBitmap = 0x0;
	uint8 					userLoopIndex = 0; 
	bool   					ldpcDummyOn = FALSE; 
	uint8					maximalRusize_SubBand = 0; 
	uint8					tempSubBand = 0;
	uint8   				dummyUserLoopIndex;
	uint8  					userStationIndex;
	uint8					ruIndex;
	PreAggRuSizeMapping_e   minimalRuSize = PRE_AGGREGATOR_996X2_TONE_RU_SIZE;
	PreAggRuSizeMapping_e   maximalRuSize = PRE_AGGREGATOR_26_TONE_RU_SIZE;

	UNUSED_PARAM(bw);
	while(userLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations) // user was not configured need to fill this hole
	{
		// intialize pointer to phase
		memset(dlDataPhasPerUserPtr, 0x0, sizeof(DlDataPhaseUserPhase_t)); // some fields (for example userRcrValid) must be set with zeros for invalid users	
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes;						
		dlDataPhasPerUserPtr->staId =  userStationIndex;
		dlDataPhasPerUserPtr->userTcraStaAid = userStationIndex+1; 
		//dlDataPhasPerUserPtr->vapId = StationDatabase.StaDbSwEntries[userStationIndex].vapIndex;
		dlDataPhasPerUserPtr->tfPadding = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfPadding; 
		dlDataPhasPerUserPtr->numTfRepetition = planManagerDefaultSettings.DlPerUser_numTfRepetition; 
		if ((staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE) || (staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_PROTECTION))
		{
			dlDataPhasPerUserPtr->mapTxUserToRxUser =  0x0; // ba is always in user 0 (same as in su)
		}
		else
		{
			dlDataPhasPerUserPtr->mapTxUserToRxUser =  userLoopIndex; //Mapping is 1 X 1 for Z0
		}
		dlDataPhasPerUserPtr->tidAllocBitmap = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tid_alloc_bitmap;
		heGroupManagerSetPlanTidsBitmap(dlDataPhasPerUserPtr->tidAllocBitmap, staticPlanConfigurationPtr->commonSection.muSequenceType);
		dlDataPhasPerUserPtr->maxTids = MIN(NUM_OF_TIDS , NUM_OF_TID_INDEXES/ staticPlanConfigurationPtr->commonSection.numOfParticipatingStations);// Minimum between  station cap  and  (36 / #users) 
		dlDataPhasPerUserPtr->tidSelectionPolicy = 0x2;   /*TID Selection Mode:
															00 Select TID with AC >= AC Primary
															01- Select TD with AC <= AC Primary
															10  Select Any TID (starting with AC>= AC Primary */
		dlDataPhasPerUserPtr->userPhaseReserve10 = 0x0; 
		dlDataPhasPerUserPtr->ampduLimit = StaDbHwEntries[userStationIndex].common.aMpduLimit; 
		dlDataPhasPerUserPtr->currUserToPrevPpduMap = 0x7F;// In Z0 Users must be in the same order in all phases
		dlDataPhasPerUserPtr->tid0RrWeight 	= planManagerDefaultSettings.DlPerUser_tid0RrWeight; //NA 
		dlDataPhasPerUserPtr->tid1RrWeight	= planManagerDefaultSettings.DlPerUser_tid1RrWeight; //NA 
		dlDataPhasPerUserPtr->tid2RrWeight	= planManagerDefaultSettings.DlPerUser_tid2RrWeight; //NA 
		dlDataPhasPerUserPtr->tid3RrWeight	= planManagerDefaultSettings.DlPerUser_tid3RrWeight; //NA 	
		dlDataPhasPerUserPtr->tid4RrWeight	= planManagerDefaultSettings.DlPerUser_tid4RrWeight; //NA 
		dlDataPhasPerUserPtr->tid5RrWeight	= planManagerDefaultSettings.DlPerUser_tid5RrWeight; //NA 
		dlDataPhasPerUserPtr->tid6RrWeight	= planManagerDefaultSettings.DlPerUser_tid6RrWeight; //NA 
		dlDataPhasPerUserPtr->tid7RrWeight	= planManagerDefaultSettings.DlPerUser_tid7RrWeight; //NA 
		//dlDataPhasPerUserPtr->currMaxMsduAtAmsdu => value is set by LA
		dlDataPhasPerUserPtr->heMngAgg = planManagerDefaultSettings.DlPerUser_heMngAgg; //No management in agg 
		dlDataPhasPerUserPtr->heBarAgg = planManagerDefaultSettings.DlPerUser_heBarAgg; 
		dlDataPhasPerUserPtr->aControlIncluded = planManagerDefaultSettings.DlPerUser_aControlIncluded;  /*A-Control (HT) Included  0  not included 1  included */
		/* RU ALLOCATOR SIMULATION */
		dlDataPhasPerUserPtr->userRcrSubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand; 
		dlDataPhasPerUserPtr->userRcrStartRu = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_StartRU; 
		dlDataPhasPerUserPtr->userRcrRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize; 
		dlDataPhasPerUserPtr->userRcrLdpc = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc; 
		dlDataPhasPerUserPtr->userRcrTxbf  = planManagerDefaultSettings.DlPerUser_userRcrTxbf;
		dlDataPhasPerUserPtr->userRcrPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp;
		dlDataPhasPerUserPtr->userRcrTargetRssi = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		dlDataPhasPerUserPtr->userRcrReserve00 = 0x0; 
		dlDataPhasPerUserPtr->userRcrValid = planManagerDefaultSettings.DlPerUser_RcrValid; 
		dlDataPhasPerUserPtr->userTcraSubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_subBandPerUsp;
		dlDataPhasPerUserPtr->userTcraStartRu = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_startRuPerUsp;
		dlDataPhasPerUserPtr->userTcraRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp;
		dlDataPhasPerUserPtr->userTcraLdpc = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
		ldpcDummyOn = dlDataPhasPerUserPtr->userTcraLdpc;
		dlDataPhasPerUserPtr->userTcraTxbf = FALSE; //OFDMA
		dlDataPhasPerUserPtr->userTcraPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_PsduRatePerUsp; 
		dlDataPhasPerUserPtr->userTcraRelativeUserPower = planManagerDefaultSettings.DlPerUser_TcraRelativeUserPower; 
		dlDataPhasPerUserPtr->userTcraReserve00  = 0x0;  
		//dlDataPhasPerUserPtr->userTcraPacketExtension = 2;   //set by LA   : 0  0 usec ; 1  8 usec ; 2  16 usec
		dlDataPhasPerUserPtr->userTcraReserve01  = 0x0;  
		dlDataPhasPerUserPtr->userTcraReserve10 = 0x0;  
		dlDataPhasPerUserPtr->userTcraBfIndex  = userStationIndex;
		dlDataPhasPerUserPtr->userTcraReserve11 = 0x0;  
		if(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp <= minimalRuSize)
		{
			if(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp < minimalRuSize)
			{
				minimalRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp; 
				minimalRuBitmap = 0x0; 
			}
			minimalRuBitmap |= (1 << staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_startRuPerUsp);
			
		}
		if(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp > maximalRuSize)
		{
			maximalRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp; 
			maximalRusize_SubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_subBandPerUsp;
		}
		
		userLoopIndex++; //progress userIndex to next one		
		// move to next index
		dlDataPhasPerUserPtr++;
	}		 
	// fill Dummy users for non-Allocated RUs 
	dummyUserLoopIndex = userLoopIndex;
	minimalRuBitmap = ~minimalRuBitmap; //Now - the set bits are the non-allocated RU indexes  
	switch(maximalRuSize)
	{
		case PRE_AGGREGATOR_106_TONE_RU_SIZE:
			minimalRuBitmap &= POSIBLE_26_TONES_FOR_SECOND_RUS_FOR_106_TONES_MASK;  // only 5 posible RUs for this RU size
			while(minimalRuBitmap != 0)
			{
				ruIndex = Utils_FindFirstSet(minimalRuBitmap);  // find RU index which 
				planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, maximalRusize_SubBand, ruIndex, minimalRuSize, ldpcDummyOn, dummyUserLoopIndex); 
				minimalRuBitmap &=  ~(0x1 << (ruIndex));  // Turn bit off 
				dlDataPhasPerUserPtr++;
			}
			break; 
		case PRE_AGGREGATOR_484_TONE_RU_SIZE:
			switch(minimalRuSize)
			{
				case PRE_AGGREGATOR_26_TONE_RU_SIZE:
					minimalRuBitmap &= POSIBLE_26_TONES_FOR_SECOND_RUS_FOR_484_TONES_MASK;  // only 18 posible RUs for this RU size
					while(minimalRuBitmap != 0)
					{
						ruIndex = Utils_FindFirstSet(minimalRuBitmap);  // find RU index which 
						if(ruIndex < NUM_OF_26_TONES_IN_A_SUB_BAND)
						{
							tempSubBand = SUB_BAND2;
						}
						else
						{
							tempSubBand = SUB_BAND3; 
						}
						planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, tempSubBand, ruIndex, minimalRuSize, ldpcDummyOn, dummyUserLoopIndex); 
						minimalRuBitmap &=  ~(0x1 << (ruIndex));  // Turn bit off 
						dlDataPhasPerUserPtr++;
					}
					break; 
				case PRE_AGGREGATOR_52_TONE_RU_SIZE:
					minimalRuBitmap &= POSIBLE_52_TONES_FOR_SECOND_RUS_FOR_484_TONES_MASK;  // only 8 posible RUs for this RU size
					while(minimalRuBitmap != 0)
					{
						ruIndex = Utils_FindFirstSet(minimalRuBitmap);  // find RU index which 
						if(ruIndex < NUM_OF_52_TONES_IN_A_SUB_BAND)
						{
							tempSubBand = SUB_BAND2;
						}
						else
						{
							tempSubBand = SUB_BAND3; 
						}
						planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, tempSubBand, ruIndex, minimalRuSize, ldpcDummyOn, dummyUserLoopIndex); 
						minimalRuBitmap &=  ~(0x1 << (ruIndex));  // Turn bit off 
						dlDataPhasPerUserPtr++;
					}
					break; 
				case PRE_AGGREGATOR_106_TONE_RU_SIZE:
					minimalRuBitmap &= POSIBLE_106_TONES_FOR_SECOND_RUS_FOR_484_TONES_MASK;  // only 4 posible RUs for this RU size
					while(minimalRuBitmap != 0)
					{
						ruIndex = Utils_FindFirstSet(minimalRuBitmap);  // find RU index which 
						if(ruIndex < NUM_OF_106_TONES_IN_A_SUB_BAND)
						{
							tempSubBand = SUB_BAND2;
						}
						else
						{
							tempSubBand = SUB_BAND3; 
						}
						planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, tempSubBand, ruIndex, minimalRuSize, ldpcDummyOn, dummyUserLoopIndex); 
						minimalRuBitmap &=  ~(0x1 << (ruIndex));  // Turn bit off 
						dlDataPhasPerUserPtr++;
					}
					break;
				case PRE_AGGREGATOR_242_TONE_RU_SIZE:
					planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, 3, 0, PRE_AGGREGATOR_242_TONE_RU_SIZE, ldpcDummyOn, dummyUserLoopIndex); 
					break;
				default: 
					ASSERT(0);
					break;
			}	
			break; 
		case PRE_AGGREGATOR_996_TONE_RU_SIZE:
			switch(minimalRuSize)
			{
				case PRE_AGGREGATOR_26_TONE_RU_SIZE:
					ASSERT(0); // currently not supported 
				case PRE_AGGREGATOR_52_TONE_RU_SIZE:
					minimalRuBitmap &= POSIBLE_52_TONES_FOR_SECOND_RUS_FOR_996_TONES_MASK;  // only 16 posible RUs for this RU size
					while(minimalRuBitmap != 0)
					{
						ruIndex = Utils_FindFirstSet(minimalRuBitmap);  // find RU index which 
						if(ruIndex < NUM_OF_52_TONES_IN_A_SUB_BAND)
						{
							tempSubBand = SUB_BAND4;
						}
						else if (ruIndex < NUM_OF_52_TONES_IN_TWO_SUB_BANDS)
						{
							tempSubBand = SUB_BAND5; 
						}
						else if (ruIndex < NUM_OF_52_TONES_IN_THREE_SUB_BANDS)
						{
							tempSubBand = SUB_BAND6; 
						}
						else if (ruIndex < NUM_OF_52_TONES_IN_FOUR_SUB_BANDS)
						{
							tempSubBand = SUB_BAND7; 
						}
						planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, tempSubBand, ruIndex, minimalRuSize, ldpcDummyOn, dummyUserLoopIndex); 
						minimalRuBitmap &=  ~(0x1 << (ruIndex));  // Turn bit off 
						dlDataPhasPerUserPtr++;
					}
					break; 
				case PRE_AGGREGATOR_106_TONE_RU_SIZE:
					minimalRuBitmap &= POSIBLE_106_TONES_FOR_SECOND_RUS_FOR_996_TONES_MASK;  // only 8 posible RUs for this RU size
					while(minimalRuBitmap != 0)
					{
						ruIndex = Utils_FindFirstSet(minimalRuBitmap);  // find RU index which 
						if(ruIndex < NUM_OF_106_TONES_IN_TWO_SUB_BANDS)
						{
							tempSubBand = SUB_BAND2;
						}
						else
						{
							tempSubBand = SUB_BAND3; 
						}
						planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, tempSubBand, ruIndex, minimalRuSize, ldpcDummyOn, dummyUserLoopIndex); 
						minimalRuBitmap &=  ~(0x1 << (ruIndex));  // Turn bit off 
						dlDataPhasPerUserPtr++;
					}
					break;
				case PRE_AGGREGATOR_242_TONE_RU_SIZE:
					planManagerRuAllocator_configureDummyUser(staticPlanConfigurationPtr, dlDataPhasPerUserPtr, 3, 0, PRE_AGGREGATOR_242_TONE_RU_SIZE, ldpcDummyOn, dummyUserLoopIndex); 
					break;
				default: 
					ASSERT(0);
					break;
			}
		default: 
			ASSERT(0); 

	}
	
	//update number of participating stations to include both active and dummy users
	staticPlanConfigurationPtr->commonSection.numOfParticipatingStations = userLoopIndex;

}

/**********************************************************************************

planManagerSetStaticDlPhaseTfInfoPerUser 

Description: 
Sets per User  TF MU BAR section in DL pahse 
	
**********************************************************************************/
static void planManagerSetStaticDlPhaseTfInfoPerUser(DlDataPhaseTfUserInfo_t* dlDataPhasTfInfoPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;
	uint8	numOfLoopIterations;
	uint8	remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET;

	numOfLoopIterations = staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; 
	if(numOfLoopIterations % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset = numOfLoopIterations;
		numOfLoopIterations = numOfLoopIterations - 1; //loop will run over even number of users since in every loop two users are filled
	}

	for(userLoopIndex = 0; userLoopIndex < numOfLoopIterations; userLoopIndex = userLoopIndex+2)
	{
		/* First user in pair*/
		ASSERT(userLoopIndex < STATIC_PLAN_MANAGER_MAX_NUM_OF_MU_USERS);
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		dlDataPhasTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		dlDataPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else
		dlDataPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerDefaultSettings.DlPerUserTf_RuAllocation; 
#endif 
#if defined (ENET_INC_ARCH_WAVE600B)
#if defined (TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND)
        dlDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif 
        dlDataPhasTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.DlPerUserTf_RuAllocation);// override by the RCR value   
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		dlDataPhasTfInfoPerUserPtr->tfFecCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#else
		dlDataPhasTfInfoPerUserPtr->tfCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#endif
		dlDataPhasTfInfoPerUserPtr->tfMcs0 =  EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_PsduRatePerUsp); //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfDcm0 = planManagerDefaultSettings.DlPerUserTf_Dcm; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfStartingSS;
		dlDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_PsduRatePerUsp);
		dlDataPhasTfInfoPerUserPtr->tfTargetRssi0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		dlDataPhasTfInfoPerUserPtr->tfReservedUserInfo0  = 0x0; 
		//dlDataPhasTfInfoPerUserPtr->tfTriggerDependent0   union of the below fields 
		dlDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfMpduMuSpacingFactor;
		dlDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0 = planManagerDefaultSettings.DlPerUserTf_TidAggLimit; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfBasicReserved0  = 0x0; 
		dlDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0  = planManagerDefaultSettings.DlPerUserTf_PreferredAc; //override by the RCR value 
		/* Second user in pair*/
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].uspStationIndexes; 
		dlDataPhasTfInfoPerUserPtr->tfAid121 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		dlDataPhasTfInfoPerUserPtr->tfRuAllocation130 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_SubBand, staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ruSize); 
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		dlDataPhasTfInfoPerUserPtr->tfFecCodingType1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].codingType_BCC_OR_LDPC;
#else
		dlDataPhasTfInfoPerUserPtr->tfCodingType1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].codingType_BCC_OR_LDPC;
#endif
		dlDataPhasTfInfoPerUserPtr->tfMcs1 = EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].dl_PsduRatePerUsp); //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfDcm1 =  planManagerDefaultSettings.DlPerUserTf_Dcm; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfStartingSpatialStream1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].tfStartingSS;
		dlDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams1 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].dl_PsduRatePerUsp);
		dlDataPhasTfInfoPerUserPtr->tfTargetRssi1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].targetRssi;
		dlDataPhasTfInfoPerUserPtr->tfReservedUserInfo1 = 0x0; 
		//dlDataPhasTfInfoPerUserPtr->tfTriggerDependent1   union of the below fields 
		dlDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].tfMpduMuSpacingFactor; //set with 1 (no factor)
		dlDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit1 = planManagerDefaultSettings.DlPerUserTf_TidAggLimit; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfBasicReserved1 = 0x0; 
		dlDataPhasTfInfoPerUserPtr->tfBasicPreferredAc1  = planManagerDefaultSettings.DlPerUserTf_PreferredAc; //override by the RCR value 
		
		dlDataPhasTfInfoPerUserPtr++; 
	}
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userLoopIndex = remainderUspOffset -1; 
		ASSERT(remainderUspOffset < PLAN_MANAGER_INVALID_USP_OFFSET);
		ASSERT(userLoopIndex < STATIC_PLAN_MANAGER_MAX_NUM_OF_MU_USERS); //KW_FIX_FW_G Range Check
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		dlDataPhasTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		dlDataPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		dlDataPhasTfInfoPerUserPtr->tfRuAllocation0 =  planManagerDefaultSettings.DlPerUserTf_RuAllocation; 
#endif 
#ifdef	ENET_INC_ARCH_WAVE600B
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        dlDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif 
        dlDataPhasTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.DlPerUserTf_RuAllocation); //override by the RCR value        
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		dlDataPhasTfInfoPerUserPtr->tfFecCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#else
		dlDataPhasTfInfoPerUserPtr->tfCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#endif
		dlDataPhasTfInfoPerUserPtr->tfMcs0 =  EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_psduRate); //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfDcm0 = planManagerDefaultSettings.DlPerUserTf_Dcm; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfStartingSS;
		dlDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_psduRate);
		dlDataPhasTfInfoPerUserPtr->tfTargetRssi0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		dlDataPhasTfInfoPerUserPtr->tfReservedUserInfo0  = 0x0; 
		//dlDataPhasTfInfoPerUserPtr->tfTriggerDependent0  = 0x0; its a union of the 4 below fields 
		dlDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0  =  staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfMpduMuSpacingFactor; //set with 1 (no factor ) 
		dlDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0 = planManagerDefaultSettings.DlPerUserTf_TidAggLimit; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfBasicReserved0  = planManagerDefaultSettings.DlPerUserTf_BasicTrigger; 
		dlDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0  = 0x0; //override by the RCR value 
	}

}

/**********************************************************************************

planManagerSetStaticUlPhaseCommonFields 

Description: 
Sets the DATA common section in UL pahse 
	
**********************************************************************************/

static void planManagerSetStaticUlPhaseCommonFields(UlDataPhaseCommon_t* phaseCommonPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 			muSequenceType;
	uint8			loopIndex; 
	uint32			validUsersBitmap = 0x0;		

	for(loopIndex = 0; loopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; loopIndex++)
	{
		validUsersBitmap |= 1<<loopIndex; 
	}
	phaseCommonPtr->validUsers310 = validUsersBitmap; 		
	phaseCommonPtr->validUsers3532 = 0;
	//TBD fill USERs indexes which are greater than 31 (in field validUsers3532)

	muSequenceType = staticPlanConfigurationPtr->commonSection.muSequenceType;
	ILOG0_D("[planManagerSetStaticUlPhaseCommonFields] muSequenceType = %d", muSequenceType);
	
	switch(muSequenceType)
	{
		case HE_MU_UL:
			phaseCommonPtr->tfParamTriggerType =  TF_TYPE_BASIC; 
			break;
		case HE_MU_BSRP:
		case HE_MU_BSRP_UL:
			phaseCommonPtr->tfParamTriggerType =  TF_TYPE_BUFFER_STATUS_RPT;
			break;
		default:
			FATAL("[planManagerSetStaticUlPhaseCommonFields], UNKNOWN muSequenceType");
			break;
	}	

	phaseCommonPtr->phaseType = planManagerDefaultSettings.UlCommon_phaseType; //0 - Protection Phase;  1 - Sounding Phase;  2 - DL Data Phase;  3 - UL Phase
	phaseCommonPtr->oneShotPhase = planManagerDefaultSettings.UlCommon_oneShotPhase; 
	phaseCommonPtr->partialDuration = planManagerDefaultSettings.UlCommon_partialDuration; // FULL TXOP 
	phaseCommonPtr->keepPlanLock = planManagerDefaultSettings.UlCommon_keepPlanLock; 
	phaseCommonPtr->moveStaToSu = planManagerDefaultSettings.UlCommon_moveStaToSu; // For training only 
	phaseCommonPtr->plannedUlTf= planManagerDefaultSettings.UlCommon_plannedUlTf;  // bit 4 in action bitmap field 
#ifdef ENET_INC_ARCH_WAVE600D2	
	phaseCommonPtr->dataMuMimo = staticPlanConfigurationPtr->commonSection.muType;
	if (staticPlanConfigurationPtr->commonSection.muType == MIMO)
	{
		// In UL MU-MIMO, Minimum number of users for transmission should be set to 2 due to PHY limitation
		phaseCommonPtr->minimumNumberOfUsersForTransmission = HE_GROUP_MANAGER_MIN_USERS_FOR_MU_MIMO;
	}
	else
#endif
	{
		phaseCommonPtr->minimumNumberOfUsersForTransmission = planManagerDefaultSettings.UlCommon_minNumOfUsersForTransmission;
	}
	phaseCommonPtr->tfInfoBaseAddrDwOffset = CONVERT_BYTES_TO_WORDS(OFFSETOF(UlDataPhase_t, tfParameters));   // offset from phase start to tfParameters
	phaseCommonPtr->ulDataSeqType = planManagerDefaultSettings.UlCommon_ulDataSeqType; 
#ifdef ENET_INC_ARCH_WAVE600D2
	phaseCommonPtr->tfMode = planManagerDefaultSettings.UlCommon_tfMode; // BC 
	phaseCommonPtr->baMode = planManagerDefaultSettings.UlCommon_baMode; // BC 
	if(phaseCommonPtr->tfMode == TF_MODE_BROADCAST_TF)
	{
		phaseCommonPtr->multiBssid	= heGroupManagerDb[HE_GROUP_MANAGER_UL_GID].mbssRxCtrlSupport;  
	}
	else
	{
		phaseCommonPtr->multiBssid	= MULTI_BSSID_NO_MULTI_BSSID;
	}
#else
	phaseCommonPtr->tfBroadcastUnicastMode = planManagerDefaultSettings.UlCommon_tfBroadcastUnicastMode;  //BC 
	phaseCommonPtr->baBroadcastUnicastMode = planManagerDefaultSettings.UlCommon_baBroadcastUnicastMode; // BC 
#endif	
	
#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	phaseCommonPtr->broadcastTfAidCfg = TRUE; // use original AID from phase
	phaseCommonPtr->unicastTfTypeCfg  = TRUE; // use original TF type from phase
	phaseCommonPtr->lSigLengthBe	= staticPlanConfigurationPtr->commonSection.tf_Length;
	phaseCommonPtr->lSigLengthBk	= staticPlanConfigurationPtr->commonSection.tf_Length;
	phaseCommonPtr->lSigLengthVi	= staticPlanConfigurationPtr->commonSection.tf_Length;
	phaseCommonPtr->lSigLengthVo	= staticPlanConfigurationPtr->commonSection.tf_Length;
#endif	
}


/**********************************************************************************

planManagerSetStaticUlPhaseCommonTcr_A_Fields 

Description: 
Sets the DATA TCR common section in UL pahse  (The basic TF TCR) 
	
**********************************************************************************/

static void planManagerSetStaticUlPhaseCommonTcr_A_Fields(UlDataPhaseCommonTcra_t*  phaseCommonTcraPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr,CommonPlan_t*  planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonDataTcrPtr = (TcrCommon_t*)(phaseCommonTcraPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
#ifdef TF_IN_HE_SU_FOR_PF
	TxSender_ScratchPadApiParams_t *pScratchPadApiParams = NULL;
	BandId_e band;
#endif
	
	//phaseCommonDataTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonDataTcrPtr->ant0Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant1Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant2Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant3Boost//Filled by LA
	phaseCommonDataTcrPtr->rfPower = staticPlanConfigurationPtr->commonSection.rfPower; 
	phaseCommonDataTcrPtr->txLoopMode = planManagerDefaultSettings.UlCommonPhaseTcrA_txLoopMode; 
	phaseCommonDataTcrPtr->cbw = staticPlanConfigurationPtr->commonSection.startBwLimit;
#ifdef TF_IN_HE_SU_FOR_PF
	band = ConfigurationManager_GetBandForVap(vapId);

	if (CONFIGURATION_MANAGER_BAND_0 == band)
	{
		pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(B0_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
	}
	else		
	{
		pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(B1_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
	}
	
	// For PF6 TC 5.61 - maximumPpduTransmissionTimeLimit is used (the actual value is only relevant for DL phase, so we can hijack this field for other purpose in UL phase) to know whether to send TF in HE phy mode
	if (staticPlanConfigurationPtr->commonSection.maximumPpduTransmissionTimeLimit == 0)
	{
		phaseCommonDataTcrPtr->phyMode = PHY_MODE_11AX_SU;
		phaseCommonDataTcrPtr->heCp    = HE_CP_HE_3_2_CP; 
		phaseCommonDataTcrPtr->heLtf   = HE_LTF_HE_4X_LTF;
		pScratchPadApiParams->unSpecifiedDurationInHeSuUlBasicTriggerFrame = TRUE;
	}
	else
	{
		phaseCommonDataTcrPtr->phyMode = planManagerDefaultSettings.UlCommonPhaseTcrA_phyMode;
		pScratchPadApiParams->unSpecifiedDurationInHeSuUlBasicTriggerFrame = FALSE;
	}
#else
	phaseCommonDataTcrPtr->phyMode = planManagerDefaultSettings.UlCommonPhaseTcrA_phyMode;
#endif
	phaseCommonDataTcrPtr->scp = planManagerDefaultSettings.UlCommonPhaseTcrA_scp;//don't care as phy mode is legacy/HE  (this field is relevant for HT/VHT)
	phaseCommonDataTcrPtr->heFullbandMu = 0; // 0 for BC TF. If TF is sent in MU-MIMO, should be 1
	phaseCommonDataTcrPtr->hePuncturingMap = planManagerDefaultSettings.UlCommonPhaseTcrA_hePuncturingMap; 
	phaseCommonDataTcrPtr->bssColor =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonDataTcrPtr->customBf = 0; // NA - for prouction 
	//phaseCommonDataTcrPtr->nHeltf   ////Filled by LA -  for MIMO - sum all NSS of users , for OFDMA - MAX NSS of all users ,value is always in power of 2 
	phaseCommonDataTcrPtr->ulDl = planManagerDefaultSettings.UlCommonPhaseTcrA_ulDl;    // set 0 for DL , 1 for UL ,  See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonDataTcrPtr->txopPs = planManagerDefaultSettings.UlCommonPhaseTcrA_txopPs; 
	phaseCommonDataTcrPtr->expectedRcr =planManagerDefaultSettings.UlCommonPhaseTcrA_expectedRcr; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	phaseCommonDataTcrPtr->bwChange = planManagerDefaultSettings.UlCommonPhaseTcrA_bwChange; 
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 1;
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 0;
	}
	phaseCommonDataTcrPtr->aggregate = planManagerDefaultSettings.UlCommonPhaseTcrA_aggregate; 
	phaseCommonDataTcrPtr->dynamicBw = planManagerDefaultSettings.UlCommonPhaseTcrA_dynamicBw; 
	phaseCommonDataTcrPtr->muPhyNdp = 0;
	phaseCommonDataTcrPtr->muTraining = 0;
	//phaseCommonDataTcrPtr->smoothing //Filled by LA 
	phaseCommonDataTcrPtr->notSounding  = planManagerDefaultSettings.UlCommonPhaseTcrA_notSounding; 
	//phaseCommonDataTcrPtr->stbc =  //Filled by LA  (HT/VHT)
	phaseCommonDataTcrPtr->spatialReuse =  planManagerDefaultSettings.UlCommonPhaseTcrA_spatialReuse; //NA
	//phaseCommonDataTcrPtr->triggerBasedDataLLenght  //owned by phy 
						
	phaseCommonDataTcrPtr->triggerBasedDataLdpc =  planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataLdpc; // Used for station mode only
	//phaseCommonDataTcrPtr->triggerBasedDataPktExt = set by LA  //pre-FEC padding fator , 0 means 4 
	phaseCommonDataTcrPtr->triggerBasedDataPeDisam =  planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataPeDisam; //  Used for station mode only 
	phaseCommonDataTcrPtr->groupId =  HE_GROUP_MANAGER_UL_GID; // in the future it will be set from group manager   
	//phaseCommonDataTcrPtr->heSigRate  = 0x0;  //to be filled by LA  (MCS 0, NSS - according to users and DCM = 0 )
	phaseCommonDataTcrPtr->macDuration = planManagerDefaultSettings.UlCommonPhaseTcrA_macDuration; 
}




/**********************************************************************************

planManagerSetStaticUlPhaseCommonTcr_B_Fields 

Description: 
Sets the MU BA TCR common section in UL pahse 
	
**********************************************************************************/

static void planManagerSetStaticUlPhaseCommonTcr_B_Fields(UlDataPhaseCommonTcrb_t*  phaseCommonTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr,CommonPlan_t*  planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonMuBaTcrPtr = (TcrCommon_t*)(phaseCommonTcrbPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	
	//phaseCommonMuBaTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonMuBaTcrPtr->ant0Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->ant1Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->ant2Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->ant3Boost//Filled by LA
	phaseCommonMuBaTcrPtr->rfPower  = staticPlanConfigurationPtr->commonSection.rfPower;
	phaseCommonMuBaTcrPtr->txLoopMode = planManagerDefaultSettings.UlCommonPhaseTcrB_txLoopMode; 
	phaseCommonMuBaTcrPtr->cbw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseCommonMuBaTcrPtr->phyMode = planManagerDefaultSettings.UlCommonPhaseTcrB_phyMode; 
	phaseCommonMuBaTcrPtr->heFullbandMu = 0; // 0 for BC M-BA. If TF is sent in MU-MIMO, should be 1
	phaseCommonMuBaTcrPtr->hePuncturingMap = planManagerDefaultSettings.UlCommonPhaseTcrB_hePuncturingMap; 
	phaseCommonMuBaTcrPtr->bssColor =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonMuBaTcrPtr->customBf = planManagerDefaultSettings.UlCommonPhaseTcrB_customBf; // NA - for prouction 
	//phaseCommonMuBaTcrPtr->nHeltf = 4;//Filled by LA   
	phaseCommonMuBaTcrPtr->ulDl = planManagerDefaultSettings.UlCommonPhaseTcrB_ulDl;    // set 0 for DL , 1 for UL ,See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonMuBaTcrPtr->txopPs = planManagerDefaultSettings.UlCommonPhaseTcrB_txopPs; 
	phaseCommonMuBaTcrPtr->expectedRcr = planManagerDefaultSettings.UlCommonPhaseTcrB_expectedRcr; //RCR expected , This packet is trigger frame - next receive is trigger based
	phaseCommonMuBaTcrPtr->bwChange = planManagerDefaultSettings.UlCommonPhaseTcrB_bwChange; 
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonMuBaTcrPtr->txopDurationFromMac = 1;
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonMuBaTcrPtr->txopDurationFromMac = 0;
	}

	phaseCommonMuBaTcrPtr->aggregate = planManagerDefaultSettings.UlCommonPhaseTcrB_aggregate; //BAR is not aggregated 
	phaseCommonMuBaTcrPtr->dynamicBw = planManagerDefaultSettings.UlCommonPhaseTcrB_dynamicBw; 
	phaseCommonMuBaTcrPtr->muPhyNdp = 0;
	phaseCommonMuBaTcrPtr->muTraining = 0;
	//phaseCommonMuBaTcrPtr->heCp = //Filled by LA  
	//phaseCommonMuBaTcrPtr->heLtf = //Filled by LA 
	//phaseCommonMuBaTcrPtr->scp = //Filled by LA 
	//phaseCommonMuBaTcrPtr->smoothing //Filled by LA 
	phaseCommonMuBaTcrPtr->notSounding  = planManagerDefaultSettings.UlCommonPhaseTcrB_notSounding; 
	//phaseCommonMuBaTcrPtr->stbc =  //Filled by LA  
	phaseCommonMuBaTcrPtr->spatialReuse =  planManagerDefaultSettings.UlCommonPhaseTcrB_spatialReuse; //NA
	//phaseCommonMuBaTcrPtr->triggerBasedDataLLenght  = // owned by the phy 
	phaseCommonMuBaTcrPtr->triggerBasedDataLdpc =   planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataLdpc;  //for station mode ony
	// phaseCommonMuBaTcrPtr->triggerBasedDataPktExt =  // Filled by LA  - search for the max station packet extension in group  
	phaseCommonMuBaTcrPtr->triggerBasedDataPeDisam =  planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataPeDisam; //for station mode ony
	phaseCommonMuBaTcrPtr->groupId =  HE_GROUP_MANAGER_UL_GID; 
//	phaseCommonMuBaTcrPtr->heSigRate  =  0x0; //Filled by LA - MCS 0, DCM0 , NSS  -1  //Filled by LA  
	phaseCommonMuBaTcrPtr->macDuration = planManagerDefaultSettings.UlCommonPhaseTcrB_macDuration; 
}


/**********************************************************************************

planManagerSetStaticUlPhaseCommonRcrFields 

Description: 
Sets the RCR common section in UL pahse 
	
**********************************************************************************/

static void planManagerSetStaticUlPhaseCommonRcrFields(UlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId)
{
	RcrCommon_t* phaseRcrCommonPtr = (RcrCommon_t*)(phaseCommonRcrPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId]; 

	phaseRcrCommonPtr->heSigABssColor  =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseRcrCommonPtr->cbw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	//phaseRcrCommonPtr->nHeltf = //Filled by LA -
#ifdef ENET_INC_ARCH_WAVE600D2
	if (staticPlanConfigurationPtr->commonSection.muType == MIMO)
	{
		phaseRcrCommonPtr->heCp = HE_CP_HE_1_6_CP; 
		phaseRcrCommonPtr->heLtf = HE_LTF_HE_1X_LTF;
		phaseRcrCommonPtr->tbMode = TB_MODE_MU_MIMO;
	}
	else
	{
		phaseRcrCommonPtr->heCp = staticPlanConfigurationPtr->commonSection.ul_HeCp; 
		phaseRcrCommonPtr->heLtf = staticPlanConfigurationPtr->commonSection.ul_HeLtf;		
		phaseRcrCommonPtr->tbMode = TB_MODE_OFDMA;
	}
#else
	phaseRcrCommonPtr->heCp = staticPlanConfigurationPtr->commonSection.ul_HeCp; 
	phaseRcrCommonPtr->heLtf = staticPlanConfigurationPtr->commonSection.ul_HeLtf;	
#endif
	phaseRcrCommonPtr->stbc = staticPlanConfigurationPtr->commonSection.rcr_Stbc;
	phaseRcrCommonPtr->heSigASpatialReuse = staticPlanConfigurationPtr->commonSection.rcr_heSigASpatialReuse; 
	phaseRcrCommonPtr->lSigLength = staticPlanConfigurationPtr->commonSection.tf_Length;
	//phaseRcrCommonPtr->ldpcExtraSymbol = //Filled by LA
	phaseRcrCommonPtr->peDisambiguty =  planManagerDefaultSettings.DlCommonPhaseRcr_peDisambiguty;// We force the station to set PE disambiguty ;// We force the station to set PE disambiguty   
}

/**********************************************************************************

planManagerSetStaticUlPhaseCommonUserTCR_A_Fields 

Description: 
Sets the common user TCRB common section in DL pahse  - how to transmit the MU BAR 
	
**********************************************************************************/

static void planManagerSetStaticUlPhaseCommonUserTCR_A_Fields(UlDataPhaseCommonUserTcra_t* commonUserTcraPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	UNUSED_PARAM(staticPlanConfigurationPtr);	
	commonUserTcraPtr->brdcstUserTcraSubBand = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_SubBand; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcraPtr->brdcstUserTcraStartRu = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_StartRu; // NA   trigger frame transmitted in legacy phy mode
#ifdef TF_IN_HE_SU_FOR_PF
	// For PF6 TC 5.61 - maximumPpduTransmissionTimeLimit is used (relevant only in DL phase) to know whether to send TF in HE phy mode
	if (staticPlanConfigurationPtr->commonSection.maximumPpduTransmissionTimeLimit == 0)
	{
		// Use LDPC in HE TF only if BW > 20M
		if (staticPlanConfigurationPtr->commonSection.startBwLimit == BANDWIDTH_TWENTY)
		{
			commonUserTcraPtr->brdcstUserTcraLdpc = FALSE; 
		}
		else
		{
			commonUserTcraPtr->brdcstUserTcraLdpc = TRUE;
		}
		commonUserTcraPtr->brdcstUserTcraRuSize = CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit);
	}
	else
	{			
		commonUserTcraPtr->brdcstUserTcraLdpc = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Ldpc;		
		commonUserTcraPtr->brdcstUserTcraRuSize = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RuSize; // NA	 trigger frame transmitted in legacy phy mode
	}
#else
	commonUserTcraPtr->brdcstUserTcraLdpc = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Ldpc;
	commonUserTcraPtr->brdcstUserTcraRuSize = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RuSize; // NA	 trigger frame transmitted in legacy phy mode
#endif
	commonUserTcraPtr->brdcstUserTcraTxbf = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Txbf; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcraPtr->brdcstUserTcraPsduRate = 0;  //filed by LA 
	commonUserTcraPtr->brdcstUserTcraRelativeUserPower =planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RelativeUserPower;// NA   trigger frame transmitted in legacy phy mode
	//commonUserTcraPtr->brdcstUserTcraPacketExtension = // filed by LA 0; /
	commonUserTcraPtr->brdcstUserTcraStaAid = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_StaAid; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcraPtr->brdcstUserTcraBfIndex = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_BfIndex; // NA   trigger frame transmitted in legacy phy mode

}


/**********************************************************************************

planManagerSetStaticUlPhaseCommonUserTCR_B_Fields 

Description: 
Sets the common user TCRB common section in DL pahse  - how to transmit the MU BAR 
	
**********************************************************************************/

static void planManagerSetStaticUlPhaseCommonUserTCR_B_Fields(UlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	UNUSED_PARAM(staticPlanConfigurationPtr);	
	commonUserTcrbPtr->brdcstUserTcrbSubBand = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_SubBand; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbStartRu = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_StartRu; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbRuSize = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RuSize; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbLdpc = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_Ldpc; 
	commonUserTcrbPtr->brdcstUserTcrbTxbf = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_Txbf; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPsduRate = 0;  //filed by LA 
	commonUserTcrbPtr->brdcstUserTcrbRelativeUserPower =planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RelativeUserPower; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPacketExtension = // filed by LA 0; /
	commonUserTcrbPtr->brdcstUserTcrbStaAid = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_TcrbStaAid; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbBfIndex = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_BfIndex; // NA   trigger frame transmitted in legacy phy mode

}

/**********************************************************************************

planManagerSetStaticUlPhaseCommonTfParameters 

Description: 
Sets the common  Trigger frame  section in UL pahse 
	
**********************************************************************************/
static void planManagerSetStaticUlPhaseCommonTfParameters(UlDataPhaseTfParameters_t* commonTfParamsPtr, UlDataPhaseTfCommonInfo_t* commonTfInfoPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 muSequenceType;
	
	commonTfParamsPtr->tfLsigLengthOverride		= planManagerDefaultSettings.UlTf_LsigLengthOverride; 
	commonTfParamsPtr->tfBwOverride				= planManagerDefaultSettings.UlTf_BW_Override;
	muSequenceType = staticPlanConfigurationPtr->commonSection.muSequenceType;
	switch(muSequenceType)
	{
		case HE_MU_UL:
			commonTfInfoPtr->tfTriggerType =  TF_TYPE_BASIC; 
			commonTfInfoPtr->tfLength =  staticPlanConfigurationPtr->commonSection.tf_Length;//This is the L-SIG length we want the STAs to transmitt - amount of bytes for station to transmit
			break;
		case HE_MU_BSRP:
		case HE_MU_BSRP_UL:
			commonTfInfoPtr->tfTriggerType =  TF_TYPE_BUFFER_STATUS_RPT; 
			commonTfInfoPtr->tfLength =  staticPlanConfigurationPtr->commonSection.tf_Length;//This is the L-SIG length we want the STAs to transmitt - amount of bytes for station to transmit
			break;
#ifdef ENET_INC_ARCH_WAVE600D2
		case HE_MU_NFRP:
			commonTfInfoPtr->tfTriggerType =  TF_TYPE_NFRP; 
			commonTfInfoPtr->tfLength =  planManagerDefaultSettings.NfrpCommonPhaseRcr_lSigLength;
						
			commonTfParamsPtr->tfNfrpStartingAidOverride	  = SUPPORT;
			commonTfParamsPtr->tfNfrpMultiplexingFlagOverride = SUPPORT;						
			break;
#endif 

		default:
			FATAL("UNKNOWN muSequenceType");
			break;
	}	

#ifdef ENET_INC_ARCH_WAVE600B
	commonTfParamsPtr->tfRuIndexOverride		= planManagerDefaultSettings.UlTf_RU_Allocation_Override;
#else 
	commonTfParamsPtr->tfRuAllocationOverride	= planManagerDefaultSettings.UlTf_RU_Allocation_Override;
#endif 
	commonTfParamsPtr->tfMcsOverride			= planManagerDefaultSettings.UlTf_MCS_Override;
	commonTfParamsPtr->tfNssOverride			= planManagerDefaultSettings.UlTf_NSS_Override;
	commonTfParamsPtr->tfDcmOverride		 	= planManagerDefaultSettings.UlTf_DcmOverride;
	commonTfParamsPtr->tfTargetRssiOverride 	= planManagerDefaultSettings.UlTf_Target_RSSI_Override;
	commonTfParamsPtr->tfTidAggrLimitOverride 	= planManagerDefaultSettings.UlTf_TID_Aggr_Limit_Override;
	commonTfParamsPtr->tfPreferredAcOverride 	= planManagerDefaultSettings.UlTf_Preferred_AC_Override;
	commonTfParamsPtr->tfMultiTidOverride 		= planManagerDefaultSettings.UlTf_Multi_TID_Override;
	commonTfParamsPtr->tfTidInfoOverride 		= planManagerDefaultSettings.UlTf_TID_Info_Override;
	commonTfParamsPtr->tfCompressedFnOverride 	= planManagerDefaultSettings.UlTf_Compressed_FN_Override;
	commonTfParamsPtr->tfCompressedSsnOverride 	= planManagerDefaultSettings.UlTf_Compressed_SSN_Override;
	commonTfParamsPtr->tfMultiTidTidOverride 	= planManagerDefaultSettings.UlTf_Multi_TID_TID_Override;
	commonTfParamsPtr->tfMultiTidFnOverride 	= planManagerDefaultSettings.UlTf_Multi_TID_FN_Override;
	commonTfParamsPtr->tfMultiTidSsnOverride 	= planManagerDefaultSettings.UlTf_Multi_TID_SSN_Override;
	
	commonTfParamsPtr->tfCommonInfoSwOverrideEnable = planManagerDefaultSettings.UlCommonTf_CommonInfoSwOverrideEnable; // only in case we want to override the SW info with a configured value in TF generator register 
	commonTfParamsPtr->tfUserInfoSwOverrideEnable = planManagerDefaultSettings.UlCommonTf_UserInfoSwOverrideEnable; // only in case we want to override the SW info with a configured value in TF generator register 

#ifdef ENET_INC_ARCH_WAVE600D2
	commonTfParamsPtr->tfNumOfHeltfSymbolsOverride = 1;
	commonTfParamsPtr->tfFecCodingTypeOverride = staticPlanConfigurationPtr->commonSection.muType;	 // In UL MU-MIMO - can be changed due to lower number of users which will enlarge the RU to be larger than 242 (20Mhz)
	commonTfParamsPtr->tfLdpcExtraSymbolOverride = staticPlanConfigurationPtr->commonSection.muType; // can be changed in UL MU-MIMO
	commonTfParamsPtr->tfStartingSpatialStreamOverride = staticPlanConfigurationPtr->commonSection.muType; // Must be set *only* in UL MU-MIMO
#endif

#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfInfoPtr->tfMoreTf = planManagerDefaultSettings.UlCommonTf_tfMoreTf;  //NA															
#else
	commonTfInfoPtr->tfCascadeIndication = planManagerDefaultSettings.UlCommonTf_CascadeIndication;  //NA															
#endif	
	commonTfInfoPtr->tfCsRequired = planManagerDefaultSettings.UlCommonTf_CsRequired; // should a station consider CCA before UL  - only for MU RTS should be set with 1  
	commonTfInfoPtr->tfBw = staticPlanConfigurationPtr->commonSection.startBwLimit; 
#ifdef ENET_INC_ARCH_WAVE600D2	
	if (staticPlanConfigurationPtr->commonSection.muType == MIMO)
	{
		commonTfInfoPtr->tfGiAndLtfType = GI_AND_LTF_TYPE_1X_LTF_1_6_GI; 
	}
	else
#endif
	{
		commonTfInfoPtr->tfGiAndLtfType = staticPlanConfigurationPtr->commonSection.tf_heGI_and_Ltf; 
	}
	commonTfInfoPtr->tfMuMimoLtfMode =  planManagerDefaultSettings.UlCommonTf_MuMimoLtfMode; //0: Single Stream Pilots   1 : Mask on each spatial stream of the LTF sequence by a distinct orthogonal code
	//if mimo -N_HE_LTF - sum of all NSS , in case of  OFDMA - the max NSS of all users 
	//commonTfInfoPtr->tfNumberOfHeLtfSymbols = //to be set by LA 
	//commonTfInfoPtr->tfStbc = 0; //to be set by LA 
	//commonTfInfoPtr->tfLdpcExtraSymbolSegment =1; // to be set by LA  
	
	//commonTfInfoPtr->tfApTxPower30 = //to be set by LA 
	//commonTfInfoPtr->tfApTxPower54 = //to be set by LA 
	//commonTfInfoPtr->tfPacketExtension = // Station capability - to be filled by LA - Max of all users 
	commonTfInfoPtr->tfSpatialReuse = planManagerDefaultSettings.UlCommonTf_SpatialReuse; //NA 
	commonTfInfoPtr->tfDoppler = planManagerDefaultSettings.UlCommonTf_Doppler; //NA 
#ifdef ENET_INC_ARCH_WAVE600D2		
	commonTfInfoPtr->tfHeSigA2Reserved = 0x1FF; 
#else
	commonTfInfoPtr->tfHeSigAReserved = 0x1FF; 
#endif
	commonTfInfoPtr->tfReservedCommonInfo= 0x0; 

}

/**********************************************************************************

planManagerSetStaticUlPhasePerUser 

Description: 
Sets per user data section in UL pahse 
	
**********************************************************************************/
static void planManagerSetStaticUlPhasePerUser(UlDataPhaseUserPhase_t* ulDataPhasPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;
    uint8   vapId;

	ulDataPhasPerUserPtr = &(ulDataPhasPerUserPtr[0]); 

	for(userLoopIndex = 0; userLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; userLoopIndex++)
	{
		ASSERT(userLoopIndex < STATIC_PLAN_MANAGER_MAX_NUM_OF_MU_USERS);
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes;
		ulDataPhasPerUserPtr->staId =  userStationIndex;
		StaDb_GetVapId(userStationIndex, &vapId);
        ulDataPhasPerUserPtr->vapId = vapId;
		ulDataPhasPerUserPtr->tfPadding = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfPadding; 
		ulDataPhasPerUserPtr->numTfRepetition = planManagerDefaultSettings.UlPerUserTf_numTfRepetition; //0x1; 
		ulDataPhasPerUserPtr->mapTxUserToRxUser =  userLoopIndex; //Mapping is 1 X 1 for Z0
		ulDataPhasPerUserPtr->tidAllocBitmap = planManagerDefaultSettings.UlPerUser_tidAllocBitmap;
		heGroupManagerSetPlanTidsBitmap(ulDataPhasPerUserPtr->tidAllocBitmap, staticPlanConfigurationPtr->commonSection.muSequenceType);
		ulDataPhasPerUserPtr->maxTids  =  0x0; //if TF_TID_Aggr_Limit_Override = 1 then value in TF is taken from STD which is taken from phase field maxTids; tid_aggregation_limit = maxTids + 1
		/*for maxTids field per user - we should call an API routine from TS manager that provides the tid mumber, considering bitmap HW limitation. */
		/*	We should provide an API function for the TS manager to update the field OTF */
		
		ulDataPhasPerUserPtr->mapTxUserToTxUser = userLoopIndex; 
			
		ulDataPhasPerUserPtr->userPhaseReserve10 = 0x0; 
		//ulDataPhasPerUserPtr->ampduLimit = StationDatabase.pStaDbHwEntries[userStationIndex].common.aMpduLimit; 

		/* RU ALLOCATOR SIMULATION */
#ifdef ENET_INC_ARCH_WAVE600D2
		if (staticPlanConfigurationPtr->commonSection.muType == MIMO)
		{
			// In case of UL MU-MIMO, allocate all plan BW size as RU
			ulDataPhasPerUserPtr->userRcrSubBand = 0; 
			ulDataPhasPerUserPtr->userRcrStartRu = 0; 
			ulDataPhasPerUserPtr->userRcrRuSize = CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit);

			PlanManagerStatistics.uplinkMuStats[userStationIndex].uplinkMuType = UL_MU_TYPE_MIMO;
		}
		else
#endif
		{
			ulDataPhasPerUserPtr->userRcrSubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand; 
			ulDataPhasPerUserPtr->userRcrStartRu = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_StartRU; 
			ulDataPhasPerUserPtr->userRcrRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize;

			PlanManagerStatistics.uplinkMuStats[userStationIndex].uplinkMuType = UL_MU_TYPE_HE;
		}
		ulDataPhasPerUserPtr->userRcrLdpc = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc; 
		ulDataPhasPerUserPtr->userRcrTxbf  = planManagerDefaultSettings.UlPerUserTf_RcrTxbf;
		ulDataPhasPerUserPtr->userRcrPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp;
		ulDataPhasPerUserPtr->userRcrTargetRssi = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		ulDataPhasPerUserPtr->userRcrReserve00 = 0x0; 
		ulDataPhasPerUserPtr->userRcrValid = planManagerDefaultSettings.UlPerUserTf_RcrValid; 
		//TCR A per user 
		ulDataPhasPerUserPtr->userTcraSubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_subBandPerUsp;
		ulDataPhasPerUserPtr->userTcraStartRu = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_startRuPerUsp;
		ulDataPhasPerUserPtr->userTcraRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp;
		//ulDataPhasPerUserPtr->userTcraLdpc = 0x0; //set by LA 
		ulDataPhasPerUserPtr->userTcraTxbf = planManagerDefaultSettings.UlPerUserTf_TcraTxbf;  
		ulDataPhasPerUserPtr->userTcraPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_PsduRatePerUsp; 
		ulDataPhasPerUserPtr->userTcraRelativeUserPower = planManagerDefaultSettings.UlPerUserTf_TcraRelativeUserPower; 
		ulDataPhasPerUserPtr->userTcraReserve00  = 0x0;  
		//ulDataPhasPerUserPtr->userTcraPacketExtension = 2;   //set by LA   : 0  0 usec ; 1  8 usec ; 2  16 usec
		ulDataPhasPerUserPtr->userTcraReserve01  = 0x0;  
		ulDataPhasPerUserPtr->userTcraReserve10 = 0x0;  
		ulDataPhasPerUserPtr->userTcraBfIndex  = FALSE;
		ulDataPhasPerUserPtr->userTcraReserve11 = 0x0;  
		//TCR B per user 
		ulDataPhasPerUserPtr->userTcrbSubBand = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_subBandPerUsp;
		ulDataPhasPerUserPtr->userTcrbStartRu = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_startRuPerUsp;
		ulDataPhasPerUserPtr->userTcrbRuSize = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_ruSizePerUsp;
		ulDataPhasPerUserPtr->userTcrbLdpc = planManagerDefaultSettings.UlPerUserTf_TcrbLdpc;
		ulDataPhasPerUserPtr->userTcrbTxbf = planManagerDefaultSettings.UlPerUserTf_TcrbTxbf;  
		ulDataPhasPerUserPtr->userTcrbPsduRate = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].dl_PsduRatePerUsp; 
		ulDataPhasPerUserPtr->userTcrbRelativeUserPower = planManagerDefaultSettings.UlPerUserTf_TcrbRelativeUserPower; 
		ulDataPhasPerUserPtr->userTcrbReserve00  = 0x0;  
		//Update Statistics 
		PlanManagerStatistics.uplinkMuStats[userStationIndex].ulRuSubChannels = ulDataPhasPerUserPtr->userRcrSubBand;
		PlanManagerStatistics.uplinkMuStats[userStationIndex].ulRuType = ulDataPhasPerUserPtr->userRcrRuSize;
		//ulDataPhasPerUserPtr->userTcrbPacketExtension = //set by LA 2;  //0  0 usec ; 1  8 usec ; 2  16 usec	
		ulDataPhasPerUserPtr++;
	}
}

/**********************************************************************************

planManagerSetStaticUlPhaseTfInfoPerUser 

Description: 
Sets per user data section in UL pahse 

**********************************************************************************/
static void planManagerSetStaticUlPhaseTfInfoPerUser(UlDataPhaseTfUserInfo_t* ulDataPhasTfInfoPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;
	uint8	numOfLoopIterations;
	uint8	remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET; 

	numOfLoopIterations = staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; 
	if(numOfLoopIterations % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset = numOfLoopIterations;
		numOfLoopIterations = numOfLoopIterations - 1; //loop will run over even number of users since in every loop two users are filled
	}

	for(userLoopIndex = 0; userLoopIndex < numOfLoopIterations; userLoopIndex = userLoopIndex+2)
	{		
		/* First user in pair*/
		ASSERT(userLoopIndex < STATIC_PLAN_MANAGER_MAX_NUM_OF_MU_USERS);
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		ulDataPhasTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation0 =  planManagerDefaultSettings.UlPerUserTf_RuAllocation; 
#endif
#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        ulDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif 
        ulDataPhasTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation); // override by the RCR value  
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2		
		ulDataPhasTfInfoPerUserPtr->tfFecCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc;
#else
		ulDataPhasTfInfoPerUserPtr->tfCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc;
#endif
		ulDataPhasTfInfoPerUserPtr->tfMcs0 =  EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp); //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfDcm0 = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfStartingSS;
		ulDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp);
		ulDataPhasTfInfoPerUserPtr->tfTargetRssi0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		ulDataPhasTfInfoPerUserPtr->tfReservedUserInfo0  = 0x0; 
		//ulDataPhasTfInfoPerUserPtr->tfTriggerDependent0   union of the below fields 
		ulDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfMpduMuSpacingFactor;
		ulDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0 = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfBasicReserved0  = 0x0; 
		ulDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0  = planManagerDefaultSettings.UlPerUserTf_preferredAc0;; //override by the RCR value 
		/* Second user in pair*/
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].uspStationIndexes; 
		ulDataPhasTfInfoPerUserPtr->tfAid121 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation130 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ruSize); 
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		ulDataPhasTfInfoPerUserPtr->tfFecCodingType1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ldpc;
#else
		ulDataPhasTfInfoPerUserPtr->tfCodingType1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ldpc;
#endif
		ulDataPhasTfInfoPerUserPtr->tfMcs1 = EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_PsduRatePerUsp); //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfDcm1 =  planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfStartingSpatialStream1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].tfStartingSS;
		ulDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams1 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_PsduRatePerUsp);
		ulDataPhasTfInfoPerUserPtr->tfTargetRssi1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].targetRssi;
		ulDataPhasTfInfoPerUserPtr->tfReservedUserInfo1 = 0x0; 
		//ulDataPhasTfInfoPerUserPtr->tfTriggerDependent1   union of the below fields 
		ulDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].tfMpduMuSpacingFactor; //set with 1 (no factor)
		ulDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit1 = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfBasicReserved1 = 0x0; 
		ulDataPhasTfInfoPerUserPtr->tfBasicPreferredAc1  = planManagerDefaultSettings.UlPerUserTf_preferredAc0; //override by the RCR value 
		
		ulDataPhasTfInfoPerUserPtr++; 
	}
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userLoopIndex = remainderUspOffset -1; 
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		ulDataPhasTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation0 =  planManagerDefaultSettings.UlPerUserTf_RuAllocation; // override by the RCR value  
#endif 
#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        ulDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif
        ulDataPhasTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation);
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		ulDataPhasTfInfoPerUserPtr->tfFecCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc;
#else
		ulDataPhasTfInfoPerUserPtr->tfCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc;
#endif
		ulDataPhasTfInfoPerUserPtr->tfMcs0 =  EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp); //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfDcm0 = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfStartingSS;
		ulDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp);
		ulDataPhasTfInfoPerUserPtr->tfTargetRssi0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		ulDataPhasTfInfoPerUserPtr->tfReservedUserInfo0  = 0x0; 
		//ulDataPhasTfInfoPerUserPtr->tfTriggerDependent0  = 0x0; //union of the 4 below fields 
		ulDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0  =  staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfMpduMuSpacingFactor; //set with 1 (no factor)
		ulDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0 = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfBasicReserved0  = 0x0; 
		ulDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0  = planManagerDefaultSettings.UlPerUserTf_preferredAc0;; //override by the RCR value 
	}

}

#ifdef OFDMA_DYNAMIC_PLAN_ON
/**********************************************************************************

planManagerSetDlPhaseCommonFields 

Description: 

**********************************************************************************/

static void planManagerSetDlPhaseCommonFields(DlDataPhaseCommon_t* phaseCommonPtr, CreatePlanMsg_t* pMuCreatePlanReq, HeMuSequence_e muSequenceType)
{		
	phaseCommonPtr->phaseType           = planManagerDefaultSettings.DlCommonPhase_phaseType; //0 - Protection Phase;  1 - Sounding Phase;  2 - DL Data Phase;  3 - UL Phase
	phaseCommonPtr->oneShotPhase        = planManagerDefaultSettings.DlCommonPhase_oneShotPhase; 	
	phaseCommonPtr->partialDuration     = planManagerDefaultSettings.DlCommonPhase_partialDuration; // default NA FULL TXOP 
	phaseCommonPtr->keepPlanLock        = planManagerDefaultSettings.DlCommonPhase_keepPlanLock; 
	phaseCommonPtr->moveStaToSu         = planManagerDefaultSettings.DlCommonPhase_moveStaToSu; 
	phaseCommonPtr->plannedData         = planManagerDefaultSettings.DlCommonPhase_plannedData; 
    phaseCommonPtr->primaryUser         = 0x0; // in vht alike mode
	phaseCommonPtr->primaryImmediateBa  = planManagerDefaultSettings.DlCommonPhase_primaryImmediateBa; // relevant for vht like mode ACK policy for primary user 
    
    switch(muSequenceType)
	{
		case HE_MU_SEQ_MU_BAR:
			phaseCommonPtr->tfParamTriggerType     = TF_TYPE_MU_BAR; 
			phaseCommonPtr->plannedMuBarTf         = 0x1;
			phaseCommonPtr->plannedBar             = 0x0; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x0;// 0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType           = 0x2; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			break;
		case HE_MU_SEQ_VHT_LIKE:
		case HE_MU_SEQ_VHT_LIKE_PROTECTION:
			phaseCommonPtr->tfParamTriggerType     = TF_TYPE_BASIC; 
			phaseCommonPtr->plannedMuBarTf         = 0x0; 
			phaseCommonPtr->plannedBar             = 0x1; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x1; //0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType           = 0x1; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			break;	
		case HE_MU_SEQ_DL_BASIC_TF:
			phaseCommonPtr->tfParamTriggerType     = TF_TYPE_BASIC; 
			phaseCommonPtr->plannedMuBarTf         = 0x0; 
			phaseCommonPtr->plannedBar             = 0x0; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x1; //0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType           = 0x0; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			break;	
		case HE_MU_SEQ_VHT_LIKE_IMM_ACK:
			phaseCommonPtr->tfParamTriggerType     = TF_TYPE_BASIC; 
			phaseCommonPtr->plannedMuBarTf         = 0x0; 
			phaseCommonPtr->plannedBar             = 0x1; 
			phaseCommonPtr->tfBroadcastUnicastMode = 0x1; //0 - Broadcast;  1 - Unicast
			phaseCommonPtr->dlDataTxType           = 0x1; // '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
			phaseCommonPtr->primaryImmediateBa     = 0x1; //primary user will be configured with immediate ACK policy
			break;
		default:
			FATAL("UNKNOWN muSequenceType");
			break;
	}
	phaseCommonPtr->minimumNumberOfUsersForTransmission = planManagerDefaultSettings.DlCommonPhase_minimumNumberOfUsersForTransmission; 
	phaseCommonPtr->tfInfoBaseAddrDwOffset              = CONVERT_BYTES_TO_WORDS(OFFSETOF(DlDataPhase_t, tfParameters));    // offset from phase start to tfParameters
	phaseCommonPtr->currentPsduMaxLengthLimit           = planManagerDefaultSettings.DlCommonPhase_currentPsduMaxLengthLimit; //Open issue  - 6500631 is taken from standart max, it should be calculated according to our capabilities 

	
#ifdef ENET_INC_ARCH_WAVE600D2			
	phaseCommonPtr->dataMuMimo = pMuCreatePlanReq->muType; // if OFDMA - 0    if MIMO - 1   //MIMO OR OFDMA 
#else
	phaseCommonPtr->muMimo     = pMuCreatePlanReq->muType; // if OFDMA - 0    if MIMO - 1   //MIMO OR OFDMA 
#endif	
	//phaseCommonPtr->totalPreambleTime = // Filled by LA :	ltfDuration +sigbDuration +36 + 16 ;   /* 36micro+SIGB+HE LTF +packet extension (16micro) 
												//		HE SIGB =  commmon part and per user  (in MU mimo there is no common section)
												// 		for 2 users - 52 bit - so    (number of station / 2) * 52 - if need to add one user (if number is odd) the size of it is 31 
												// 		for Z0 -   20Mhz and 40Mhz - 31 , 80hz - 52 
												//CONVERT_SIGBRATE_2_DBPS - 0-26, 1- 52, 2- 78,3 104,4- 156,  5-208 
														
	phaseCommonPtr->maxPpduTransmissionTimeLimit = planManagerDefaultSettings.DlCommonPhase_maxPpduTransmissionTimeLimit; 
	phaseCommonPtr->phaseTimeNotIncludingData    = BAR_BACK_AIR_TIME_DURATION; /* phase_time_not_including_data  :   
																											Time shall include SIFS +MU BAR + SIFS + BA or (SIFS + BAR + SIFS + BA)x num of users (resolution of 2usec) */ 	
#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	phaseCommonPtr->broadcastTfAidCfg = TRUE; // use original AID from phase
	phaseCommonPtr->unicastTfTypeCfg  = TRUE; // use original TF type from phase
#endif	
}


/**********************************************************************************

planManagerSetDlPhaseCommonTcr_A_Fields 

Description: 
Sets the DATA TCR common section in DL phase 
	
**********************************************************************************/

static void planManagerSetDlPhaseCommonTcr_A_Fields(DlDataPhaseCommonTcra_t* phaseCommonTcraPtr, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId, HeMuSequence_e muSequenceType)
{
	TcrCommon_t* phaseCommonDataTcrPtr = (TcrCommon_t*)(phaseCommonTcraPtr);
    
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	
	//phaseCommonDataTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonDataTcrPtr->ant0Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant1Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant2Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant3Boost//Filled by LA
	//phaseCommonDataTcrPtr->rfPower = //Filled by LA
#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_V("[planManagerSetDlPhaseCommonTcr_A_Fields]");
#endif
	phaseCommonDataTcrPtr->cbw              = pMuCreatePlanReq->bw;
	phaseCommonDataTcrPtr->phyMode          = planManagerDefaultSettings.DlCommonPhaseTcrA_phyMode; 
	phaseCommonDataTcrPtr->heFullbandMu     = (pMuCreatePlanReq->muType == OFDMA) ? 0 : 1 ; // if OFDMA - 0    if MIMO - 1  
	phaseCommonDataTcrPtr->txLoopMode       = (pMuCreatePlanReq->muType == OFDMA) ? TX_LOOP_MODE_CLOSED_LOOP : TX_LOOP_MODE_OPEN_LOOP ;// if OFDMA - closed loop,  if MIMO - open loop 
	phaseCommonDataTcrPtr->hePuncturingMap  = planManagerDefaultSettings.DlCommonPhaseTcrA_hePuncturingMap; 
	phaseCommonDataTcrPtr->bssColor         = pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonDataTcrPtr->customBf         = planManagerDefaultSettings.DlCommonPhaseTcrA_customBf; // NA - for prouction 
	//phaseCommonDataTcrPtr->nHeltf         //Filled by LA -  for MIMO - sum all NSS of users , for OFDMA - MAX NSS of all users ,value is always in power of 2 
	phaseCommonDataTcrPtr->ulDl             = planManagerDefaultSettings.DlCommonPhaseTcrA_ulDl;   // set 0 for DL , 1 for UL ,See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonDataTcrPtr->txopPs           = planManagerDefaultSettings.DlCommonPhaseTcrA_txopPs; 
	if(muSequenceType == HE_MU_SEQ_DL_BASIC_TF)
	{
		phaseCommonDataTcrPtr->expectedRcr = 0X1; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	}
	else
	{
		phaseCommonDataTcrPtr->expectedRcr = planManagerDefaultSettings.DlCommonPhaseTcrA_expectedRcr; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	}
	phaseCommonDataTcrPtr->bwChange        = planManagerDefaultSettings.DlCommonPhaseTcrA_bwChange;
    
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 1; 
	}
	else //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 0;
	} 
    
	phaseCommonDataTcrPtr->aggregate    = planManagerDefaultSettings.DlCommonPhaseTcrA_aggregate; 
	phaseCommonDataTcrPtr->dynamicBw    = planManagerDefaultSettings.DlCommonPhaseTcrA_dynamicBw; 
	phaseCommonDataTcrPtr->muPhyNdp     = 0;
	phaseCommonDataTcrPtr->muTraining   = 0;
	//phaseCommonDataTcrPtr->heCp       = //Filled by LA  (HE)
	//phaseCommonDataTcrPtr->heLtf      = //Filled by LA 
	phaseCommonDataTcrPtr->scp          = planManagerDefaultSettings.DlCommonPhaseTcrA_scp;//don't care as phy mode is HE  (this field is relevant for HT/VHT)
	//phaseCommonDataTcrPtr->smoothing  //Filled by LA 
	phaseCommonDataTcrPtr->notSounding  = planManagerDefaultSettings.DlCommonPhaseTcrA_notSounding; 
	//phaseCommonDataTcrPtr->stbc       = //Filled by LA  (HT/VHT)
	phaseCommonDataTcrPtr->spatialReuse = planManagerDefaultSettings.DlCommonPhaseTcrA_spatialReuse; //NA
	//phaseCommonDataTcrPtr->triggerBasedDataLLenght  //owned by phy 					
	if(pMuCreatePlanReq->bw > BANDWIDTH_TWENTY)
	{
		phaseCommonDataTcrPtr->triggerBasedDataLdpc = 0x1; //We must turn LDPC if BW is above 20Mhz 
	}
	else //  BANDWIDTH_TWENTY
	{
		phaseCommonDataTcrPtr->triggerBasedDataLdpc = 0x0; // No LDPC for 20Mhz 
	}
	phaseCommonDataTcrPtr->triggerBasedDataPktExt  = planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataPktExt; //Filled by LA , pre-FEC padding fator , 0 means 4 
	phaseCommonDataTcrPtr->triggerBasedDataPeDisam = planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataPeDisam; //  
	phaseCommonDataTcrPtr->groupId                 = pMuCreatePlanReq->groupId;
	//phaseCommonDataTcrPtr->heSigRate             = 0x0;  //to be filled by LA  (MCS 0, NSS - according to users and DCM = 0 )
	phaseCommonDataTcrPtr->macDuration             = planManagerDefaultSettings.DlCommonPhaseTcrA_macDuration; 
}

/**********************************************************************************

planManagerSetDlPhaseCommonTcr_A_Fields 

Description: 
Sets the DATA TCR common section in DL phase 
	
**********************************************************************************/

static void planManagerSetDlPhaseCommonTcr_B_Fields(DlDataPhaseCommonTcrb_t* phaseCommonTcrbPtr,CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonDataTcrPtr = (TcrCommon_t*)(phaseCommonTcrbPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	
	//phaseCommonDataTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonDataTcrPtr->ant0Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant1Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant2Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant3Boost//Filled by LA
	//phaseCommonDataTcrPtr->rfPower = //Filled by LA
    phaseCommonDataTcrPtr->txLoopMode       = (pMuCreatePlanReq->muType == OFDMA) ? TX_LOOP_MODE_CLOSED_LOOP : TX_LOOP_MODE_OPEN_LOOP; // if OFDMA - closed loop,  if MIMO - open loop 
    phaseCommonDataTcrPtr->cbw              = pMuCreatePlanReq->bw; 
	phaseCommonDataTcrPtr->phyMode          = planManagerDefaultSettings.DlCommonPhaseTcrB_phyMode;
	phaseCommonDataTcrPtr->heFullbandMu     = (pMuCreatePlanReq->muType == OFDMA) ? 0 : 1 ; // if OFDMA - 0    if MIMO - 1 	
	phaseCommonDataTcrPtr->hePuncturingMap  = planManagerDefaultSettings.DlCommonPhaseTcrB_hePuncturingMap; 
	phaseCommonDataTcrPtr->bssColor         = pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonDataTcrPtr->customBf         = planManagerDefaultSettings.DlCommonPhaseTcrB_customBf; // NA - for prouction 
	//phaseCommonDataTcrPtr->nHeltf         //Filled by LA -  for MIMO - sum all NSS of users , for OFDMA - MAX NSS of all users ,value is always in power of 2 
	phaseCommonDataTcrPtr->ulDl             = planManagerDefaultSettings.DlCommonPhaseTcrB_ulDl;   // set 0 for DL , 1 for UL ,See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonDataTcrPtr->txopPs           = planManagerDefaultSettings.DlCommonPhaseTcrB_txopPs; 
	phaseCommonDataTcrPtr->expectedRcr      = planManagerDefaultSettings.DlCommonPhaseTcrB_expectedRcr; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	phaseCommonDataTcrPtr->bwChange         = planManagerDefaultSettings.DlCommonPhaseTcrB_bwChange;     
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 1; 
	}
	else //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 0;
	}     
	phaseCommonDataTcrPtr->aggregate    = planManagerDefaultSettings.DlCommonPhaseTcrB_aggregate; 
	phaseCommonDataTcrPtr->dynamicBw    = planManagerDefaultSettings.DlCommonPhaseTcrB_dynamicBw; 
	phaseCommonDataTcrPtr->muPhyNdp     = 0;
	phaseCommonDataTcrPtr->muTraining   = 0;
	//phaseCommonDataTcrPtr->heCp       = staticPlanConfigurationPtr->commonSection.heCp; //Filled by LA  (HE)
	//phaseCommonDataTcrPtr->heLtf      = //Filled by LA 
 	//phaseCommonDataTcrPtr->scp        = planManagerDefaultSettings.DlCommonPhaseTcrB_scp; // filled by LA
 	//phaseCommonDataTcrPtr->smoothing  //Filled by LA 
	phaseCommonDataTcrPtr->notSounding  = planManagerDefaultSettings.DlCommonPhaseTcrB_notSounding; 
	//phaseCommonDataTcrPtr->stbc       = //Filled by LA  (HT/VHT)
	phaseCommonDataTcrPtr->spatialReuse = planManagerDefaultSettings.DlCommonPhaseTcrB_spatialReuse; //NA
	//phaseCommonDataTcrPtr->triggerBasedDataLLenght  //owned by phy 						
	phaseCommonDataTcrPtr->triggerBasedDataLdpc    = planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataLdpc; // No LDPC 
	phaseCommonDataTcrPtr->triggerBasedDataPktExt  = planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataPktExt;
	phaseCommonDataTcrPtr->triggerBasedDataPeDisam = planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataPeDisam; // NA 
	phaseCommonDataTcrPtr->groupId                 = pMuCreatePlanReq->groupId;
	//phaseCommonDataTcrPtr->heSigRate             = 0x0;  //Filled by LA - MCS 0, DCM0 , NSS  -1
	phaseCommonDataTcrPtr->macDuration             = planManagerDefaultSettings.DlCommonPhaseTcrB_macDuration; 
}

/**********************************************************************************

planManagerSetDlPhaseCommonRcrFields 

Description: 
Sets the RCR common section in DL pahse 
	
**********************************************************************************/

static void planManagerSetDlPhaseCommonRcrFields(DlDataPhaseCommonRcr_t*  phaseCommonRcrPtr,CreatePlanMsg_t* pMuCreatePlanReq, uint8 vapId)
{
	RcrCommon_t* phaseRcrCommonPtr = (RcrCommon_t*)(phaseCommonRcrPtr);
	VapDb_t*     pVapDbHwEntry     = &VapDbHwEntries[vapId];

	phaseRcrCommonPtr->heSigABssColor     = pVapDbHwEntry->common.tcr1BssColor; 
	phaseRcrCommonPtr->cbw                = pMuCreatePlanReq->bw; 
	//phaseRcrCommonPtr->nHeltf           = //Filled by LA -
	//phaseRcrCommonPtr->heCp             = //Filled by LA -
	//phaseRcrCommonPtr->heLtf            = //Filled by LA -
	//phaseRcrCommonPtr->stbc             = //Filled by LA -
	phaseRcrCommonPtr->heSigASpatialReuse = 0x0; 
	phaseRcrCommonPtr->lSigLength         = planManagerDefaultSettings.DlCommonTf_tfLength; // dynamic grouping debug - was planManagerDefaultSettings.Dl_lSigLength;
	//phaseRcrCommonPtr->ldpcExtraSymbol  = //Filled by LA
	phaseRcrCommonPtr->peDisambiguty      = planManagerDefaultSettings.DlCommonPhaseRcr_peDisambiguty; // We force the station to set PE disambiguty   
}

/**********************************************************************************

planManagerSetDlPhaseCommonUserTCR_B_Fields 

Description: 
Sets the common user TCRB common section in DL phase  - how to transmit the MU BAR 
	
**********************************************************************************/

static void planManagerSetDlPhaseCommonUserTCR_B_Fields(DlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr, RuAllocationPerUser_t* ruAllocation)
{
	commonUserTcrbPtr->brdcstUserTcrbSubBand            = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_SubBand; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbStartRu            = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_StartRu; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbRuSize             = ruAllocation[0].userRcrRuSize; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbLdpc               = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_Ldpc; 
	commonUserTcrbPtr->brdcstUserTcrbTxbf               = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_Txbf; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPsduRate         = 0; //filed by LA 
	commonUserTcrbPtr->brdcstUserTcrbRelativeUserPower  = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_RelativeUserPower; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPacketExtension  = // filed by LA 0; /
	commonUserTcrbPtr->brdcstUserTcrbStaAid             = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_TcrbStaAid; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbBfIndex            = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_BfIndex; // NA   trigger frame transmitted in legacy phy mode

}
/**********************************************************************************

planManagerSetDlPhaseCommonTfParameters 

Description: 
Sets the common  Trigger frame  section in DL phase 
	
**********************************************************************************/
static void	planManagerSetDlPhaseCommonTfParameters(DlDataPhase_t* dlDataPhasPtr, CreatePlanMsg_t* pMuCreatePlanReq)
{
	DlDataPhaseTfParameters_t* commonTfParamsPtr = &(dlDataPhasPtr->tfParameters); 
	DlDataPhaseTfCommonInfo_t* commonTfInfoPtr   = &(dlDataPhasPtr->tfCommonInfo);
    HeMuSequence_e             muSequenceType;           

    commonTfParamsPtr->tfOverrideEnableParams		= 0xFF; // make the default value 1 for all
	commonTfParamsPtr->tfBwOverride					= planManagerDefaultSettings.DlCommonTf_tfBwOverride;
#ifdef ENET_INC_ARCH_WAVE600B
#ifdef ENET_INC_ARCH_WAVE600D2
	commonTfParamsPtr->tfMoreTfOverride 			= 0; 
#else
	commonTfParamsPtr->tfCascadeIndicationOverride 	= 0; 
#endif
	commonTfParamsPtr->tfRuIndexOverride			= planManagerDefaultSettings.DlCommonTf_tfRuAllocationOverride;
#else
	commonTfParamsPtr->tfRuAllocationOverride		= planManagerDefaultSettings.DlCommonTf_tfRuAllocationOverride;
#endif  
	commonTfParamsPtr->tfMcsOverride				= planManagerDefaultSettings.DlCommonTf_tfMcsOverride;
	commonTfParamsPtr->tfDcmOverride				= planManagerDefaultSettings.DlCommonTf_tfDcmOverride;
	commonTfParamsPtr->tfNssOverride				= planManagerDefaultSettings.DlCommonTf_tfNssOverride;
	commonTfParamsPtr->tfTargetRssiOverride			= planManagerDefaultSettings.DlCommonTf_tfTargetRssiOverride;
	commonTfParamsPtr->tfPreferredAcOverride		= planManagerDefaultSettings.DlCommonTf_tfPreferredAcOverride;
    commonTfParamsPtr->tfLsigLengthOverride         = planManagerDefaultSettings.DlCommonTf_tfLsigLengthOverride; 
	commonTfParamsPtr->tfTidAggrLimitOverride       = planManagerDefaultSettings.DlCommonTf_tfTidAggrLimitOverride;
	commonTfParamsPtr->tfMultiTidOverride           = planManagerDefaultSettings.DlCommonTf_tfMultiTidOverride; 
	commonTfParamsPtr->tfTidInfoOverride            = planManagerDefaultSettings.DlCommonTf_tfTidInfoOverride; 
	commonTfParamsPtr->tfCompressedFnOverride       = planManagerDefaultSettings.DlCommonTf_tfCompressedFnOverride;
	commonTfParamsPtr->tfCompressedSsnOverride      = planManagerDefaultSettings.DlCommonTf_tfCompressedSsnOverride;
	commonTfParamsPtr->tfMultiTidTidOverride        = planManagerDefaultSettings.DlCommonTf_tfMultiTidTidOverride;
	commonTfParamsPtr->tfMultiTidFnOverride         = planManagerDefaultSettings.DlCommonTf_tfMultiTidFnOverride;
	commonTfParamsPtr->tfMultiTidSsnOverride        = planManagerDefaultSettings.DlCommonTf_tfMultiTidSsnOverride;
    commonTfParamsPtr->tfCommonInfoSwOverrideEnable = planManagerDefaultSettings.DlCommonTf_CommonInfoSwOverrideEnable; // only in case we want to override the SW info with a configured value in TF generator register 
	commonTfParamsPtr->tfUserInfoSwOverrideEnable   = planManagerDefaultSettings.DlCommonTf_UserInfoSwOverrideEnable; // only in case we want to override the SW info with a configured value in TF generator register 

    commonTfParamsPtr->tfLsigLengthOverride         = 0; //LSIG length musn't be override  
	commonTfParamsPtr->tfTidAggrLimitOverride       = 0; //in case of DL the limit should be set in TF to zero

    commonTfParamsPtr->tfMultiTidOverride = 1; 
	commonTfParamsPtr->tfTidInfoOverride = 1; 
	commonTfParamsPtr->tfCompressedFnOverride = 1; 
	commonTfParamsPtr->tfCompressedSsnOverride = 1; 
	commonTfParamsPtr->tfMultiTidTidOverride = 1;
	commonTfParamsPtr->tfMultiTidFnOverride = 1;
	commonTfParamsPtr->tfMultiTidSsnOverride = 1;
    
    muSequenceType = heGroupManagerDb[pMuCreatePlanReq->groupId].sequence;
    switch(muSequenceType)
	{
		case HE_MU_SEQ_MU_BAR:
			commonTfInfoPtr->tfTriggerType = TF_TYPE_MU_BAR; 
			break;
		case HE_MU_SEQ_VHT_LIKE:
		case HE_MU_SEQ_VHT_LIKE_IMM_ACK:
		case HE_MU_SEQ_VHT_LIKE_PROTECTION:
			//don't care in this case since there is no TF
			break;
		case HE_MU_SEQ_DL_BASIC_TF:
			commonTfInfoPtr->tfTriggerType = TF_TYPE_BASIC; 
			break;
		default:
			FATAL("[planManagerSetDlPhaseCommonTfParameters], UNKNOWN muSequenceType");
			break;
	}

    commonTfInfoPtr->tfLength             = planManagerDefaultSettings.DlCommonTf_tfLength; //This is the L-SIG length we want the STAs to transmit - amount of bytes for station to transmit   
#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfInfoPtr->tfMoreTf             = planManagerDefaultSettings.DlCommonTf_tfMoreTf;  //NA															
#else
	commonTfInfoPtr->tfCascadeIndication  = planManagerDefaultSettings.DlCommonTf_CascadeIndication;  //NA															
#endif	
	commonTfInfoPtr->tfCsRequired         = planManagerDefaultSettings.DlCommonTf_CsRequired; // should a station consider CCA before UL  - only For MU RTS should be set with 1  
	commonTfInfoPtr->tfBw                 = pMuCreatePlanReq->bw;
    commonTfInfoPtr->tfGiAndLtfType       = planManagerDefaultSettings.tfGiAndLtfType;
	commonTfInfoPtr->tfMuMimoLtfMode      = planManagerDefaultSettings.DlCommonTf_MuMimoLtfMode; //0: Single Stream Pilots   1 : Mask on each spatial stream of the LTF sequence by a distinct orthogonal code
	//if mimo -N_HE_LTF - sum of all NSS , in case of  OFDMA - the max NSS of all users 
	//commonTfInfoPtr->tfNumberOfHeLtfSymbols   = //to be set by LA 
	commonTfInfoPtr->tfStbc                     = 0; //to be set by LA 
	//commonTfInfoPtr->tfLdpcExtraSymbolSegment = 1; //to be set by LA  	
    //commonTfInfoPtr->tfApTxPower30      =  // set by LA 
	//commonTfInfoPtr->tfApTxPower54      = // set by LA 
	//commonTfInfoPtr->tfPacketExtension  = // Station capability - to be filled by LA - Max of all users 
	commonTfInfoPtr->tfSpatialReuse       = planManagerDefaultSettings.DlCommonTf_SpatialReuse; //NA 
	commonTfInfoPtr->tfDoppler            = planManagerDefaultSettings.DlCommonTf_Doppler; //NA 
#ifdef ENET_INC_ARCH_WAVE600D2		
	commonTfInfoPtr->tfHeSigA2Reserved    = planManagerDefaultSettings.tfHeSigA2Reserved; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#else
	commonTfInfoPtr->tfHeSigAReserved     = planManagerDefaultSettings.tfHeSigAReserved; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#endif	
	commonTfInfoPtr->tfReservedCommonInfo = 0x0; 
}


/**********************************************************************************

planManagerSetDlPhaseCommonTfMuBar 

Description: 
Sets the common TF MU BAR section in DL phase 
	
**********************************************************************************/
static void planManagerSetDlPhaseCommonTfMuBar(DlDataPhaseTfMuBar_t* dlDataPhasTfMuBarPtr)
{
	///dlDataPhasTfMuBarPtr->tfMuBarControl = // union of the below fields: 
	dlDataPhasTfMuBarPtr->tfBarAckPolicy                        = planManagerDefaultSettings.DlCommonTf_tfBarAckPolicy;
	dlDataPhasTfMuBarPtr->tfMultiTid                            = planManagerDefaultSettings.DlCommonTf_tfMultiTid;
	dlDataPhasTfMuBarPtr->tfCompressedBitmap                    = planManagerDefaultSettings.DlCommonTf_CompressedBitmap;
	dlDataPhasTfMuBarPtr->tfGcr                                 = planManagerDefaultSettings.DlCommonTf_tfGcr;
	dlDataPhasTfMuBarPtr->tfReservedBarControl                  = planManagerDefaultSettings.DlCommonTf_ReservedBarControl;
	dlDataPhasTfMuBarPtr->tfTidInfo                             = 1;  //  filled by HW - pre-aggregator  
	//dlDataPhasTfMuBarPtr->tfCompressedMuBarInformation  // union of the below fields:
	dlDataPhasTfMuBarPtr->tfCompressedFn0                       = planManagerDefaultSettings.DlCommonTf_CompressedFn0 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedFn1                       = planManagerDefaultSettings.DlCommonTf_CompressedFn1 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedFn2                       = planManagerDefaultSettings.DlCommonTf_CompressedFn2 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedFn3                       = planManagerDefaultSettings.DlCommonTf_CompressedFn3 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfCompressedStartingSequenceNumber    = planManagerDefaultSettings.DlCommonTf_CompressedStartingSequenceNumber;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidMuBarInformation            = planManagerDefaultSettings.DlCommonTf_MultiTidMuBarInformation ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidReserved                    = planManagerDefaultSettings.DlCommonTf_MultiTidReserved ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidTid                         = planManagerDefaultSettings.DlCommonTf_MultiTidTid ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn0                         = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn0 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn1                         = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn1;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn2                         = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn2 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidFn3                         = planManagerDefaultSettings.DlCommonTf_tfMultiTidFn3 ;//override by the RCR configuration 
	dlDataPhasTfMuBarPtr->tfMultiTidStartingSequenceNumber      = planManagerDefaultSettings.DlCommonTf_MultiTidStartingSequenceNumber ;//override by the RCR configuration 	
}


/**********************************************************************************

planManagerSetDlPhasePerUser 

Description: 
Sets the per user data section in DL pahse 
	
**********************************************************************************/
static void planManagerSetDlPhasePerUser(DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr,CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation, bool plannedBar, uint8 allocatedUsers)
{
    uint8   vapId;
	uint8 	userLoopIndex; 
	StaId	userStationIndex;
	 	
	for(userLoopIndex = 0; userLoopIndex < heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup; userLoopIndex++)
	{
		userStationIndex                            = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex];
			
		memset(dlDataPhasPerUserPtr, 0x0, sizeof(DlDataPhaseUserPhase_t)); // some fields (for example userRcrValid) must be set with zeros for invalid users
		dlDataPhasPerUserPtr->staId                 = userStationIndex;
		dlDataPhasPerUserPtr->userTcraStaAid        = userStationIndex + 1; 
		StaDb_GetVapId(userStationIndex, &vapId);
        dlDataPhasPerUserPtr->vapId                 = vapId;

		//dlDataPhasPerUserPtr->tfPadding           = // filled by LA according to station capability
		dlDataPhasPerUserPtr->numTfRepetition       = planManagerDefaultSettings.DlPerUser_numTfRepetition; 

		if (plannedBar == TRUE) //this means we are using a vht-like sequence with sequential BARs
		{
			dlDataPhasPerUserPtr->mapTxUserToRxUser = 0x0; // ba is always in user 0 (same as in su)
		}
		else
		{
			dlDataPhasPerUserPtr->mapTxUserToRxUser = userLoopIndex; //Mapping is 1 X 1 for Z0
		}
        
		dlDataPhasPerUserPtr->tidAllocBitmap        = planManagerDefaultSettings.DlPerUser_tidAllocBitmap;
		heGroupManagerSetPlanTidsBitmap(dlDataPhasPerUserPtr->tidAllocBitmap, heGroupManagerDb[pMuCreatePlanReq->groupId].sequence);
        dlDataPhasPerUserPtr->maxTids               = MIN(NUM_OF_TIDS, NUM_OF_USERS_MU_HE/ (heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup));// Minimum between  station cap  and  (36 / #users)
		dlDataPhasPerUserPtr->tidSelectionPolicy    = 0x2;   /*TID Selection Mode:
															00 Select TID with AC >= AC Primary
															01- Select TD with AC <= AC Primary
															10  Select Any TID (starting with AC>= AC Primary */
		dlDataPhasPerUserPtr->userPhaseReserve10    = 0x0; 
		dlDataPhasPerUserPtr->ampduLimit            = StaDbHwEntries[userStationIndex].common.aMpduLimit; 
		dlDataPhasPerUserPtr->currUserToPrevPpduMap = 0x7F;// In Z0 Users must be in the same order in all phases
		dlDataPhasPerUserPtr->tid0RrWeight 	        = planManagerDefaultSettings.DlPerUser_tid0RrWeight; //NA 
		dlDataPhasPerUserPtr->tid1RrWeight	        = planManagerDefaultSettings.DlPerUser_tid1RrWeight; //NA 
		dlDataPhasPerUserPtr->tid2RrWeight	        = planManagerDefaultSettings.DlPerUser_tid2RrWeight; //NA 
		dlDataPhasPerUserPtr->tid3RrWeight	        = planManagerDefaultSettings.DlPerUser_tid3RrWeight; //NA 	
		dlDataPhasPerUserPtr->tid4RrWeight	        = planManagerDefaultSettings.DlPerUser_tid4RrWeight; //NA 
		dlDataPhasPerUserPtr->tid5RrWeight	        = planManagerDefaultSettings.DlPerUser_tid5RrWeight; //NA 
		dlDataPhasPerUserPtr->tid6RrWeight	        = planManagerDefaultSettings.DlPerUser_tid6RrWeight; //NA 
		dlDataPhasPerUserPtr->tid7RrWeight	        = planManagerDefaultSettings.DlPerUser_tid7RrWeight; //NA 
		//dlDataPhasPerUserPtr->currMaxMsduAtAmsdu  = LA
		dlDataPhasPerUserPtr->heMngAgg              = planManagerDefaultSettings.DlPerUser_heMngAgg; //No management in agg 
		dlDataPhasPerUserPtr->heBarAgg              = planManagerDefaultSettings.DlPerUser_heBarAgg; 
		dlDataPhasPerUserPtr->aControlIncluded      = planManagerDefaultSettings.DlPerUser_aControlIncluded;  /*A-Control (HT) Included  0  not included 1  included */
		/* RU allocator parameters  */
		dlDataPhasPerUserPtr->userRcrSubBand        = ruAllocation[userLoopIndex].ul_SubBand; 
		dlDataPhasPerUserPtr->userRcrStartRu        = ruAllocation[userLoopIndex].ul_StartRU; 
		dlDataPhasPerUserPtr->userRcrRuSize         = ruAllocation[userLoopIndex].userRcrRuSize; 

        if (pMuCreatePlanReq->muType == OFDMA)
        {
            dlDataPhasPerUserPtr->userTcraSubBand   = ruAllocation[userLoopIndex].dl_subBandPerUsp;
		    dlDataPhasPerUserPtr->userTcraStartRu   = ruAllocation[userLoopIndex].dl_startRuPerUsp;
		    dlDataPhasPerUserPtr->userTcraRuSize    = ruAllocation[userLoopIndex].dl_ruSizePerUsp;

            dlDataPhasPerUserPtr->userTcraBfIndex   = userStationIndex; //SID in OFDMA
			PlanManagerStatistics.downlinkMuStats[userStationIndex].downlinkMuType = DL_MU_TYPE_HE;
        }
        else // MIMO
        {
            dlDataPhasPerUserPtr->userTcraSubBand   = 0;
		    dlDataPhasPerUserPtr->userTcraStartRu   = 0;
            dlDataPhasPerUserPtr->userTcraRuSize    = CONVERT_BW_TO_RU_SIZE(pMuCreatePlanReq->bw);
        
            dlDataPhasPerUserPtr->userTcraBfIndex   = CONVERT_USP_INDEX_TO_HE_MU_BF_INDEX(userLoopIndex);
			PlanManagerStatistics.downlinkMuStats[userStationIndex].downlinkMuType = DL_MU_TYPE_MIMO;
        }
		//Update Statistics
		PlanManagerStatistics.downlinkMuStats[userStationIndex].dlRuSubChannels = dlDataPhasPerUserPtr->userTcraSubBand;
		PlanManagerStatistics.downlinkMuStats[userStationIndex].dlRuType = dlDataPhasPerUserPtr->userTcraRuSize;

		dlDataPhasPerUserPtr->userRcrLdpc               = LDPC; // reset by LA according to stations capabilities
	//	dlDataPhasPerUserPtr->userRcrPsduRate // set by LA  
	//	dlDataPhasPerUserPtr->userRcrTargetRssi // set by LA 
	//	dlDataPhasPerUserPtr->userTcraPsduRate // set by LA 

		dlDataPhasPerUserPtr->userRcrTxbf               = planManagerDefaultSettings.DlPerUser_userRcrTxbf;
		dlDataPhasPerUserPtr->userRcrReserve00          = 0x0; 
		dlDataPhasPerUserPtr->userRcrValid              = planManagerDefaultSettings.DlPerUser_RcrValid; 
		dlDataPhasPerUserPtr->userTcraLdpc              = LDPC; // reset by LA according to stations capabilities
		dlDataPhasPerUserPtr->userTcraTxbf              = (pMuCreatePlanReq->muType == OFDMA) ? 0 : 1; // if OFDMA - 0    if MIMO - 1   
		dlDataPhasPerUserPtr->userTcraRelativeUserPower = planManagerDefaultSettings.DlPerUser_TcraRelativeUserPower; 
		dlDataPhasPerUserPtr->userTcraReserve00         = 0x0;  
		//dlDataPhasPerUserPtr->userTcraPacketExtension = 2;   //set by LA   : 0  0 usec ; 1  8 usec ; 2  16 usec
		dlDataPhasPerUserPtr->userTcraReserve01         = 0x0;  
		dlDataPhasPerUserPtr->userTcraReserve10         = 0x0;          
		dlDataPhasPerUserPtr->userTcraReserve11         = 0x0;  
        dlDataPhasPerUserPtr++;
	}

    if (pMuCreatePlanReq->muType == OFDMA)
    {
        for (; userLoopIndex < allocatedUsers; userLoopIndex++) // for dummy users, assuming equal RUs
	    {
            dlDataPhasPerUserPtr->staId           = 0;
	        dlDataPhasPerUserPtr->userTcraStaAid  = DUMMY_AID; 

	        dlDataPhasPerUserPtr->userTcraSubBand = ruAllocation[userLoopIndex].dl_subBandPerUsp;
		dlDataPhasPerUserPtr->maxTids  = MIN(NUM_OF_TOTAL_TIDS ,NUM_OF_USERS_MU_HE/ (heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup));// Minimum between  station cap  and  (36 / #users)
	        dlDataPhasPerUserPtr->userTcraRuSize  = ruAllocation[userLoopIndex].dl_ruSizePerUsp;
            dlDataPhasPerUserPtr->userTcraLdpc    = LDPC;
            if (plannedBar == TRUE)
	        {
		        dlDataPhasPerUserPtr->mapTxUserToRxUser = 0x0; // ba is always in user 0 (same as in su)
	        }
	        else
	        {
		        dlDataPhasPerUserPtr->mapTxUserToRxUser = userLoopIndex; //Mapping is 1 X 1 for Z0
		    }
            dlDataPhasPerUserPtr++;
	    }
    }	
}

/**********************************************************************************

planManagerSetDlPhaseTfInfoPerUser 

Description: 
Sets per User  TF MU BAR section in DL pahse 
	
**********************************************************************************/
static void planManagerSetDlPhaseTfInfoPerUser(DlDataPhaseTfUserInfo_t* dlDataPhasTfInfoPerUserPtr,CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;
	uint8	numOfLoopIterations;
	uint8	remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET; 

    UNUSED_PARAM(ruAllocation);
	numOfLoopIterations = (heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup);
	
	if(numOfLoopIterations % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset  = numOfLoopIterations;
		numOfLoopIterations = numOfLoopIterations - 1; //loop will run over even number of users since in every loop two users are filled 
	}

	for(userLoopIndex = 0; userLoopIndex < numOfLoopIterations; userLoopIndex = userLoopIndex + 2)
	{
		/* First user in pair*/	
		userStationIndex                                        = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex]; 
		dlDataPhasTfInfoPerUserPtr->tfAid120                    = userStationIndex + 1; 
#ifdef TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        // only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
        dlDataPhasTfInfoPerUserPtr->tfRuAllocation0             = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize);
#else
        dlDataPhasTfInfoPerUserPtr->tfRuAllocation0             = planManagerDefaultSettings.DlPerUserTf_RuAllocation; //override by the RCR value
#endif 
#if defined ENET_INC_ARCH_WAVE600B
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		dlDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800     = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize);
#endif 
        dlDataPhasTfInfoPerUserPtr->tfRuIndex0                  = (planManagerDefaultSettings.DlPerUserTf_RuAllocation); //override by the RCR value 
#endif  
#ifdef ENET_INC_ARCH_WAVE600D2		
        dlDataPhasTfInfoPerUserPtr->tfFecCodingType0            = 1;
#else
        dlDataPhasTfInfoPerUserPtr->tfCodingType0               = 1;
#endif
		dlDataPhasTfInfoPerUserPtr->tfMcs0                      = EXTRACT_MCS_FROM_VHT_HE_RATE(0); //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfDcm0                      = planManagerDefaultSettings.DlPerUserTf_Dcm; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfSsAllocation0             = 0; 
		dlDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0    = 0; 
		//dlDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = //Filled by LA 
		//dlDataPhasTfInfoPerUserPtr->tfTargetRssi0             = //Filled by LA 
		dlDataPhasTfInfoPerUserPtr->tfReservedUserInfo0         = 0x0; 
		//dlDataPhasTfInfoPerUserPtr->tfTriggerDependent0   union of the below fields 
		dlDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0 = 0;
		dlDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0         = planManagerDefaultSettings.DlPerUserTf_TidAggLimit; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfBasicReserved0            = 0x0; 
		dlDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0         = planManagerDefaultSettings.DlPerUserTf_PreferredAc; //override by the RCR value 
		/* Second user in pair*/	
		userStationIndex                                        = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex+1]; 
		dlDataPhasTfInfoPerUserPtr->tfAid121                    = userStationIndex + 1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        // only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
        dlDataPhasTfInfoPerUserPtr->tfRuAllocation130           = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex+1].ul_SubBand, ruAllocation[userLoopIndex+1].userRcrRuSize);
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
        dlDataPhasTfInfoPerUserPtr->tfFecCodingType1            = 1;
#else
        dlDataPhasTfInfoPerUserPtr->tfCodingType1               = 1;
#endif
		//dlDataPhasTfInfoPerUserPtr->tfMcs1                    = //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfDcm1                      =  planManagerDefaultSettings.DlPerUserTf_Dcm; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfSsAllocation1             = 0; 
		dlDataPhasTfInfoPerUserPtr->tfStartingSpatialStream1    = 0; 
		//dlDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams1 = //Filled by LA 
		//dlDataPhasTfInfoPerUserPtr->tfTargetRssi1             = //Filled by LA 
		dlDataPhasTfInfoPerUserPtr->tfReservedUserInfo1         = 0x0; 
		//dlDataPhasTfInfoPerUserPtr->tfTriggerDependent1   union of the below fields 
		dlDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor1 = 0;
		dlDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit1         = planManagerDefaultSettings.DlPerUserTf_TidAggLimit; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfBasicReserved1            = 0;
		dlDataPhasTfInfoPerUserPtr->tfBasicPreferredAc1         = planManagerDefaultSettings.DlPerUserTf_PreferredAc; //override by the RCR value 
		
		dlDataPhasTfInfoPerUserPtr++; 
	}
    
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userLoopIndex                                           = remainderUspOffset - 1; 	
		userStationIndex                                        = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex]; 
		dlDataPhasTfInfoPerUserPtr->tfAid120                    = userStationIndex + 1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        dlDataPhasTfInfoPerUserPtr->tfRuAllocation0             = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw,ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize);
#else
        dlDataPhasTfInfoPerUserPtr->tfRuAllocation0             = planManagerDefaultSettings.DlPerUserTf_RuAllocation; //override by the RCR value 
#endif
#ifdef  ENET_INC_ARCH_WAVE600B
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND        
        dlDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800     = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw,ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize);  
#endif 
        dlDataPhasTfInfoPerUserPtr->tfRuIndex0                  = (planManagerDefaultSettings.DlPerUserTf_RuAllocation); //override by the RCR value 
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2		
        dlDataPhasTfInfoPerUserPtr->tfFecCodingType0            = 1;
#else
        dlDataPhasTfInfoPerUserPtr->tfCodingType0               = 1;
#endif
		dlDataPhasTfInfoPerUserPtr->tfMcs0                      = EXTRACT_MCS_FROM_VHT_HE_RATE(0); //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfDcm0                      = planManagerDefaultSettings.DlPerUserTf_Dcm; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfSsAllocation0 = 0; 
		dlDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0    = 0; 
		//dlDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = //Filled by LA 
		//dlDataPhasTfInfoPerUserPtr->tfTargetRssi0             = //Filled by LA 
		dlDataPhasTfInfoPerUserPtr->tfReservedUserInfo0         = 0x0; 
		//dlDataPhasTfInfoPerUserPtr->tfTriggerDependent0       = 0x0; its a union of the 4 below fields 
		dlDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0 = 0;
		dlDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0         = planManagerDefaultSettings.DlPerUserTf_TidAggLimit; //override by the RCR value 
		dlDataPhasTfInfoPerUserPtr->tfBasicReserved0            = planManagerDefaultSettings.DlPerUserTf_BasicTrigger; 
		dlDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0         = 0x0; //override by the RCR value 
	}
}


/**********************************************************************************
planManagerSetDlPhaseCommonFieldValidUser 

Description: signal validuser bitmap field in phase common section according to number of participating stations (including dummy users)
  
Input: 
-----
		
Output:
-------	

Returns:
--------
	void 	
**********************************************************************************/
static void planManagerSetDlPhaseCommonFieldValidUser(DlDataPhaseCommon_t* phaseCommonPtr, CreatePlanMsg_t* pMuCreatePlanReq, uint8 allocatedUsers)
{
    uint8	loopIndex;
    uint8   numOfUsers;
	uint32	validUsersBitmap = 0x0;

    if (pMuCreatePlanReq->muType == OFDMA)
    {
        numOfUsers = allocatedUsers;   // including dummy users 
    }
    else //MIMO
    {
        numOfUsers = heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup;
    }
	for(loopIndex = 0; loopIndex < numOfUsers; loopIndex++)
	{
		validUsersBitmap |= 1<<loopIndex; 
	}
	phaseCommonPtr->validUsers310  = validUsersBitmap;
    phaseCommonPtr->validUsers3532 = 0;
}

/**********************************************************************************

planManagerSetProtectionPhase 

Description: 
Sets protection phase parameters 
	
**********************************************************************************/
static void planManagerSetProtectionPhase(ProtectionPhase_t* protectionPhasePtr, uint8 groupId, uint8 bw, RuAllocationPerUser_t* ruAllocationPerUser, uint8 vapId)
{
	TcrCommon_t* 	phaseCommonMuBaTcrPtr;
	RcrCommon_t* 	phaseRcrCommonPtr;
	VapDb_t* 		pVapDbHwEntry = &VapDbHwEntries[vapId];
	uint16			userStationIndex; 
	uint8			userIndex; 
	uint8			numOfLoopIterations;
	uint8			remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET; 

    UNUSED_PARAM(ruAllocationPerUser);
	phaseCommonMuBaTcrPtr = (TcrCommon_t*)&(protectionPhasePtr->commonTcrb);
	phaseRcrCommonPtr = (RcrCommon_t*)&(protectionPhasePtr->commonRcr);
	
	/* Protection phase common section */ 
	/***********************************/
	protectionPhasePtr->common.plannedMuRts	   	= 	planManagerDefaultSettings.ProtectionPhasePlannedMuRts;
	protectionPhasePtr->common.plannedRtsCts	= 	planManagerDefaultSettings.ProtectionPhasePlannedRtsCts;
	protectionPhasePtr->common.plannedCts2Self	= 	planManagerDefaultSettings.ProtectionPhasePlannedCts2Self; // planned CTS_2_SELF;
	protectionPhasePtr->common.fillerType = 		planManagerDefaultSettings.ProtectionPhaseFillerType; 
	protectionPhasePtr->common.groupIndex = groupId; 
	protectionPhasePtr->common.keepPlanLock = planManagerDefaultSettings.ProtectionPhaseKeepPlanLock; 
	protectionPhasePtr->common.minimumNumberOfUsersForTransmission = planManagerDefaultSettings.ProtectionMinNumOfUsersForTx; 
	protectionPhasePtr->common.moveStaToSu = planManagerDefaultSettings.ProtectionPhaseMoveStaToSu; 
	protectionPhasePtr->common.oneShotPhase = 0x0;
	protectionPhasePtr->common.partialDuration = planManagerDefaultSettings.DlCommonPhase_partialDuration;
	protectionPhasePtr->common.phaseType = UMI_MU_PROTECTION_PHASE; 
	//protectionPhasePtr->common.rtsCtsFormat = RTS_CTS_FORMAT_INDICATION_LEGACY_RTS ?  ; 
	protectionPhasePtr->common.tfInfoBaseAddrDwOffset = CONVERT_BYTES_TO_WORDS(OFFSETOF(ProtectionPhase_t, tfParameters)); // offset from phase start to tfParameters
	//protectionPhasePtr->common.rtsCtsFormat //= N.A - Not relevant for MU RTS
#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	protectionPhasePtr->common.broadcastTfAidCfg = TRUE; // use original AID from phase
	protectionPhasePtr->common.unicastTfTypeCfg  = TRUE; // use original TF type from phase
#endif	

	/* Protection phase TCRB - Common TCR for MU-RTS in Broadcast   Transmitted in legacy phy_mode (11ag) */ 
	/*****************************************************************************************************/
	phaseCommonMuBaTcrPtr->cbw = bw; 
	//phaseCommonMuBaTcrPtr->rfPower = //To be set by LA   
	//phaseCommonMuBaTcrPtr->txLoopMode // To be set by LA   
	//phaseCommonMuBaTcrPtr->phyMode // To be set by LA   
	phaseCommonMuBaTcrPtr->heFullbandMu = (heGroupManagerDb[groupId].formationType == DL_OFDMA) ? 0 : 1 ;// if OFDMA - 0	if MIMO - 1 
	phaseCommonMuBaTcrPtr->hePuncturingMap= planManagerDefaultSettings.DlCommonPhaseTcrB_hePuncturingMap;
	//phaseCommonMuBaTcrPtr->forceTx =  ? 

	phaseCommonMuBaTcrPtr->bssColor = pVapDbHwEntry->common.tcr1BssColor; 
	//phaseCommonMuBaTcrPtr->nHeltf= //Filled by LA 

	phaseCommonMuBaTcrPtr->expectedRcr = NA_VALUE_FOR_FIELD_IN_PHASE; //phy does not using RCR for CTS recieve
	phaseCommonMuBaTcrPtr->bwChange = NA_VALUE_FOR_FIELD_IN_PHASE; 
	phaseCommonMuBaTcrPtr->txopDurationFromMac = NA_VALUE_FOR_FIELD_IN_PHASE; 
	phaseCommonMuBaTcrPtr->aggregate = NA_VALUE_FOR_FIELD_IN_PHASE; 
	phaseCommonMuBaTcrPtr->dynamicBw = NA_VALUE_FOR_FIELD_IN_PHASE; 		
	phaseCommonMuBaTcrPtr->muPhyNdp = NA_VALUE_FOR_FIELD_IN_PHASE; 
	phaseCommonMuBaTcrPtr->muTraining = NA_VALUE_FOR_FIELD_IN_PHASE; 
	//phaseCommonMuBaTcrPtr->heCp //N.A 
	//phaseCommonMuBaTcrPtr->heLtf //N.A 
	phaseCommonMuBaTcrPtr->scp = 0; 
	//phaseCommonMuBaTcrPtr->smoothing //Filled by LA 
	phaseCommonMuBaTcrPtr->notSounding  = 1; 
	phaseCommonMuBaTcrPtr->stbc = 0; 
	phaseCommonMuBaTcrPtr->spatialReuse = 0; 
	phaseCommonMuBaTcrPtr->triggerBasedDataLLenght = 0; 
	//phaseCommonMuBaTcrPtr->triggerBasedDataLdpc
	//phaseCommonMuBaTcrPtr->triggerBasedDataPktExt //N.A 
	//phaseCommonMuBaTcrPtr->triggerBasedDataPeDisam//N.A 

	phaseCommonMuBaTcrPtr->groupId = groupId; 
	//phaseCommonMuBaTcrPtr->heSigRate = //to be filled by LA
	//phaseCommonMuBaTcrPtr->macDuration //N.A 

	
	/* Protection phase common RCR */ 
	/*******************************/ 
	phaseRcrCommonPtr->heSigABssColor = pVapDbHwEntry->common.tcr1BssColor; 
	phaseRcrCommonPtr->cbw = bw; 
	//phaseRcrCommonPtr->nHeltf = //Filled by LA
	//phaseRcrCommonPtr->heCp = //Filled by LA
	//phaseRcrCommonPtr->heLtf = //Filled by LA
	//phaseRcrCommonPtr->stbc = //Filled by LA
	phaseRcrCommonPtr->heSigASpatialReuse = 0x0; 
	phaseRcrCommonPtr->lSigLength = 0x0; 
	//phaseRcrCommonPtr->ldpcExtraSymbol = //Filled by LA
	phaseRcrCommonPtr->peDisambiguty = planManagerDefaultSettings.DlCommonPhaseRcr_peDisambiguty;// We force the station to set PE disambiguty 	


	/* Protection phase commonUserTcrb  */  
	/**************************************************/

	protectionPhasePtr->commonUserTcrb.brdcstUserTcrbSubBand = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_SubBand;
	protectionPhasePtr->commonUserTcrb.brdcstUserTcrbStartRu = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_StartRu;
	protectionPhasePtr->commonUserTcrb.brdcstUserTcrbRuSize = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RuSize;
	//protectionPhasePtr->commonUserTcrb.brdcstUserTcrbLdpc = // Set by LA 
	//protectionPhasePtr->commonUserTcrb.brdcstUserTcrbTxbf = // Set by LA 
	//protectionPhasePtr->commonUserTcrb.brdcstUserTcrbPsduRate = // Set by LA 
	protectionPhasePtr->commonUserTcrb.brdcstUserTcrbRelativeUserPower = planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_RelativeUserPower; // NA   trigger frame transmitted in legacy phy mode
	//protectionPhasePtr->commonUserTcrb.brdcstUserTcrbPacketExtension = // Set by LA 
	//protectionPhasePtr->commonUserTcrb.brdcstUserTcrbStaAid =   N.A 
	protectionPhasePtr->commonUserTcrb.brdcstUserTcrbBfIndex = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_BfIndex; // NA   trigger frame transmitted in legacy phy mode

	/* Protection phase - TF paramters */ 
	/****************************************/
	protectionPhasePtr->tfCommonInfo.tfTriggerType = TF_TYPE_MU_RTS; 
	protectionPhasePtr->tfCommonInfo.tfLength = LSIG_LENGHT_FOR_8_TIDS_64_MU_BA_SIZE; //This is the L-SIG length we want the STAs to transmitt - amount of bytes for station to transmit 
#ifdef ENET_INC_ARCH_WAVE600D2	
	protectionPhasePtr->tfCommonInfo.tfMoreTf = planManagerDefaultSettings.UlCommonTf_tfMoreTf;  //NA	
#else
	protectionPhasePtr->tfCommonInfo.tfCascadeIndication = planManagerDefaultSettings.UlCommonTf_CascadeIndication;  //NA	
#endif	
	protectionPhasePtr->tfCommonInfo.tfCsRequired = 0x1; //only for MU RTS should be set with 1
	protectionPhasePtr->tfCommonInfo.tfBw = bw; 
	protectionPhasePtr->tfCommonInfo.tfGiAndLtfType = planManagerDefaultSettings.tfGiAndLtfType; 
	protectionPhasePtr->tfCommonInfo.tfMuMimoLtfMode =  planManagerDefaultSettings.UlCommonTf_MuMimoLtfMode; //0: Single Stream Pilots   1 : Mask on each spatial stream of the LTF sequence by a distinct orthogonal code
	//protectionPhasePtr->tfCommonInfo.tfNumberOfHeLtfSymbols //set by LA 
	//protectionPhasePtr->tfCommonInfo.tfStbc //to be set by LA 
	//protectionPhasePtr->tfCommonInfo.tfLdpcExtraSymbolSegment =1; //set by LA  
	//protectionPhasePtr->tfCommonInfo.tfApTxPower30 //set by LA  
	//protectionPhasePtr->tfCommonInfo.tfApTxPower54 //set by LA  
	//protectionPhasePtr->tfCommonInfo.tfPacketExtension // Station capability - to be filled by LA - Max of all users //set by LA  
	protectionPhasePtr->tfCommonInfo.tfSpatialReuse = planManagerDefaultSettings.UlCommonTf_SpatialReuse; //NA 
	protectionPhasePtr->tfCommonInfo.tfDoppler = planManagerDefaultSettings.UlCommonTf_Doppler; //NA ; 
	protectionPhasePtr->tfParameters.tfOverrideEnableParams = 0xFFFF; 
	protectionPhasePtr->tfParameters.tfLsigLengthOverride = NA_VALUE_FOR_FIELD_IN_PHASE; 
	protectionPhasePtr->tfParameters.tfTargetRssiOverride = NA_VALUE_FOR_FIELD_IN_PHASE; 
	protectionPhasePtr->tfParameters.tfCommonInfoSwOverrideEnable = NA_VALUE_FOR_FIELD_IN_PHASE; 
	protectionPhasePtr->tfParameters.tfUserInfoSwOverrideEnable = NA_VALUE_FOR_FIELD_IN_PHASE; 


	/* Set tfUserInfo per user section */
	/***********************************/
	numOfLoopIterations = (heGroupManagerDb[groupId].numOfStationsInGroup) >>1; //dived the number of users configured by 2
	if((heGroupManagerDb[groupId].numOfStationsInGroup) % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset = numOfLoopIterations+1; 
	}
	for(userIndex = 0; userIndex < numOfLoopIterations; userIndex++)
	{
		/* First user in pair*/
		userStationIndex = heGroupManagerDb[groupId].members[userIndex]; 
		protectionPhasePtr->tfUserInfo[userIndex].tfAid120 = userStationIndex+1; 		 
#ifndef ENET_INC_ARCH_WAVE600B
        protectionPhasePtr->tfUserInfo[userIndex].tfRuAllocation0 = planManagerDefaultSettings.UlPerUserTf_RuAllocation; //override by the RCR value
#else		
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        protectionPhasePtr->tfUserInfo[userIndex].tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(bw,ruAllocationPerUser[userIndex].ul_SubBand);
#endif 
        protectionPhasePtr->tfUserInfo[userIndex].tfRuIndex0 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation); //override by the RCR value 
#endif 
		//protectionPhasePtr->tfUserInfo[userIndex].tfCodingType0 = //Set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfMcs0 = //Set by LA 
		protectionPhasePtr->tfUserInfo[userIndex].tfDcm0 = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		//protectionPhasePtr->tfUserInfo[userIndex].tfSsAllocation0 = //Set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfStartingSpatialStream0 = //Set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfNumberOfSpatialStreams0 = //Set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfTargetRssi0 =  //Set by LA 
		protectionPhasePtr->tfUserInfo[userIndex].tfReservedUserInfo0  = NA_VALUE_FOR_FIELD_IN_PHASE; 
		//protectionPhasePtr->tfUserInfo[userIndex].tfTriggerDependent0	 union of the below fields 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicMpduMuSpacingFactor0 = planManagerDefaultSettings.tfMpduMuSpacingFactor;
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicTidAggLimit0 = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicReserved0  = NA_VALUE_FOR_FIELD_IN_PHASE; 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicPreferredAc0	= planManagerDefaultSettings.UlPerUserTf_preferredAc0;; //override by the RCR value 
		/* Second user in pair*/
		userStationIndex = heGroupManagerDb[groupId].members[userIndex+1]; 
		protectionPhasePtr->tfUserInfo[userIndex].tfAid121 = userStationIndex+1; 
#ifndef ENET_INC_ARCH_WAVE600B
		protectionPhasePtr->tfUserInfo[userIndex].tfRuAllocation130  = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) & RU_ALLOCATION_MASK; //override by the RCR value 
		protectionPhasePtr->tfUserInfo[userIndex].tfRuAllocation174 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) >> RU_ALLOCATION_SHIFT; //override by the RCR value 

#else
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        protectionPhasePtr->tfUserInfo[userIndex].tfRuPrimarySecondary801 = planManagerCalcRuAllocationSecondaryBit(bw,ruAllocationPerUser[userIndex+1].ul_SubBand);
#endif 
		protectionPhasePtr->tfUserInfo[userIndex].tfRuIndex2To01 = ((planManagerDefaultSettings.UlPerUserTf_RuAllocation) & RU_ALLOCATION_B_MASK); //override by the RCR value 
        protectionPhasePtr->tfUserInfo[userIndex].tfRuIndex6To31 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) >> RU_ALLOCATION_B_SHIFT; //override by the RCR value 
#endif
        //protectionPhasePtr->tfUserInfo[userIndex].tfCodingType1 = //to be set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfMcs1 = //to be set by LA 
		protectionPhasePtr->tfUserInfo[userIndex].tfDcm1 =  planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		//protectionPhasePtr->tfUserInfo[userIndex].tfSsAllocation1 = //to be set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfStartingSpatialStream1 = //to be set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfNumberOfSpatialStreams1 = //to be set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfTargetRssi1 = //to be set by LA 
		protectionPhasePtr->tfUserInfo[userIndex].tfReservedUserInfo1 = NA_VALUE_FOR_FIELD_IN_PHASE; 
		//protectionPhasePtr->tfUserInfo[userIndex].tfTriggerDependent1	 union of the below fields 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicMpduMuSpacingFactor1 = planManagerDefaultSettings.tfMpduMuSpacingFactor;
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicTidAggLimit1 = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicReserved1 = NA_VALUE_FOR_FIELD_IN_PHASE; 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicPreferredAc1	= planManagerDefaultSettings.UlPerUserTf_preferredAc0; //override by the RCR value 
	}
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userIndex = remainderUspOffset -1; 
		userStationIndex = heGroupManagerDb[groupId].members[userIndex]; 
		protectionPhasePtr->tfUserInfo[userIndex].tfAid120 = userStationIndex+1; 
#ifndef ENET_INC_ARCH_WAVE600B
        protectionPhasePtr->tfUserInfo[userIndex].tfRuAllocation0 = planManagerDefaultSettings.UlPerUserTf_RuAllocation; //override by the RCR value
#else	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        protectionPhasePtr->tfUserInfo[userIndex].tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(bw,ruAllocationPerUser[userIndex].ul_SubBand);
#endif 
		protectionPhasePtr->tfUserInfo[userIndex].tfRuIndex0 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation); //override by the RCR value 
#endif 
		//protectionPhasePtr->tfUserInfo[userIndex].tfCodingType0 // set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfMcs0 =	// set by LA 
		protectionPhasePtr->tfUserInfo[userIndex].tfDcm0 = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		//protectionPhasePtr->tfUserInfo[userIndex].tfSsAllocation0 =// set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfStartingSpatialStream0 = // set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfNumberOfSpatialStreams0 =  // set by LA 
		//protectionPhasePtr->tfUserInfo[userIndex].tfTargetRssi0 =  // set by LA 
		protectionPhasePtr->tfUserInfo[userIndex].tfReservedUserInfo0  = NA_VALUE_FOR_FIELD_IN_PHASE; 
		//protectionPhasePtr->tfUserInfo[userIndex].tfTriggerDependent0	= 0x0; //union of the 4 below fields 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicMpduMuSpacingFactor0	= planManagerDefaultSettings.tfMpduMuSpacingFactor;
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicTidAggLimit0 = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicReserved0  = NA_VALUE_FOR_FIELD_IN_PHASE; 
		protectionPhasePtr->tfUserInfo[userIndex].tfBasicPreferredAc0	= planManagerDefaultSettings.UlPerUserTf_preferredAc0;; //override by the RCR value 
	}

#if 0 

	/* Protection phase TCRB - per user TCR for MU-RTS */ 
	/**************************************************/
	for(userIndex = 0; userIndex < heGroupManagerDb[groupId].numOfStationsInGroup; userIndex++)
	{
		


		protectionPhasePtr->userPhase[userIndex].staId = heGroupManagerDb[groupId].members[userIndex];
		protectionPhasePtr->userPhase[userIndex].vapId = vapId; 
		protectionPhasePtr->userPhase[userIndex].tfPadding = planManagerDefaultSettings.ulTfPadding  ; //16usec
		protectionPhasePtr->userPhase[userIndex].userRcrSubBand = ruAllocationPerUser[userIndex].ul_SubBand; 
		protectionPhasePtr->userPhase[userIndex].userRcrStartRu = ruAllocationPerUser[userIndex].ul_StartRU; 
		protectionPhasePtr->userPhase[userIndex].userRcrRuSize = ruAllocationPerUser[userIndex].userRcrRuSize; 
		//protectionPhasePtr->userPhase[userIndex].userRcrLdpc = //To be set by LA 
		//protectionPhasePtr->userPhase[userIndex].userRcrPsduRate  = //To be set by LA 
		//protectionPhasePtr->userPhase[userIndex].userRcrTargetRssi = //To be set by LA 
		protectionPhasePtr->userPhase[userIndex].userRcrTxbf = 0; //NO UL BF 
		protectionPhasePtr->userPhase[userIndex].userRcrValid = 1; 
		


	}
#endif //If 0 
	
}

/**********************************************************************************

planManagerSetSoundingPhase 

Description: 
Sets sounding phase parameters 
	
**********************************************************************************/
static void planManagerSetSoundingPhase(SoundingPhase_t* soundingPhasePtr, CreatePlanMsg_t* pMuCreatePlanReq)
{
    UNUSED_PARAM(soundingPhasePtr);
    UNUSED_PARAM(pMuCreatePlanReq);
	/*TBD*/
}
#endif


/**********************************************************************************

planManagerSetUlPhase 

Description: 
Sets UL phase parameters which can be either UL or BSRP
	
**********************************************************************************/
static void planManagerSetStaticUlPhase(UlDataPhase_t* ulOrBsrpPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t* planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId)
{

	UNUSED_PARAM(groupId);
	/*Fill UL phase common fields */
	planManagerSetStaticUlPhaseCommonFields(&(ulOrBsrpPhasePtr->common), pMuPlanConfig);
	/*Fill UL phase TCR A (Basic TF) common fields */
	planManagerSetStaticUlPhaseCommonTcr_A_Fields(&(ulOrBsrpPhasePtr->commonTcra), pMuPlanConfig,planPtr, vapId);
	/*Fill UL phase TCR B (MU BA) common fields */
	planManagerSetStaticUlPhaseCommonTcr_B_Fields(&(ulOrBsrpPhasePtr->commonTcrb), pMuPlanConfig,planPtr, vapId);
	/*Fill UL phase RCR common fields */
	planManagerSetStaticUlPhaseCommonRcrFields(&(ulOrBsrpPhasePtr->commonRcr), pMuPlanConfig, vapId);
	/*	Fill UL phase TCR A (Basic TF) common user fields */
	planManagerSetStaticUlPhaseCommonUserTCR_A_Fields((&(ulOrBsrpPhasePtr->commonUserTcra)), pMuPlanConfig);
	/*	Fill UL phase TCR B (MU BA) common user fields */
	planManagerSetStaticUlPhaseCommonUserTCR_B_Fields((&(ulOrBsrpPhasePtr->commonUserTcrb)), pMuPlanConfig);
	//Fill TF parameters
	planManagerSetStaticUlPhaseCommonTfParameters(&ulOrBsrpPhasePtr->tfParameters, &ulOrBsrpPhasePtr->tfCommonInfo, pMuPlanConfig);
	//Fill Data TCR per User parameters
	planManagerSetStaticUlPhasePerUser(&(ulOrBsrpPhasePtr->userPhase[0]), pMuPlanConfig);
	//Fill TF per User	parameters
	planManagerSetStaticUlPhaseTfInfoPerUser(&(ulOrBsrpPhasePtr->tfUserInfo[0]), pMuPlanConfig);	
}

/**********************************************************************************

planManagerSetSoundingPhase 

Description: 
Sets sounding phase parameters 
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhase(SoundingPhase_t* soundingPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t*  planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId)
{
	
	heGroupManagerSetPlanTidsBitmap(TID_BITMAP_ALL_DATA_TIDS, pMuPlanConfig->commonSection.muSequenceType);
	// Fill Sounding phase Common fields 
	planManagerSetStaticSoundingPhaseCommonFields(&(soundingPhasePtr->common), pMuPlanConfig, groupId, vapId);
	// Fill Sounding phase TCR A (NDP & NDPA in broadcast, for NDPA TCR is updated by Sender) common fields 
	planManagerSetStaticSoundingPhaseCommonTcr_A_Fields(&(soundingPhasePtr->commonTcra), pMuPlanConfig, planPtr, vapId);
	// Fill Sounding phase TCR B (TF BFP broadcast/unicast) common fields 
	planManagerSetStaticSoundingPhaseCommonTcr_B_Fields(&(soundingPhasePtr->commonTcrb), pMuPlanConfig, planPtr, vapId);
	// Fill Sounding phase RCR (UL BF report) common fields 
	planManagerSetStaticSoundingPhaseCommonRcrFields(&(soundingPhasePtr->commonRcr), pMuPlanConfig, vapId);
	//	Fill Sounding phase TCR_A (NDP & NDPA in broadcast, for NDPA TCR is updated by Sender) common user
	planManagerSetStaticSoundingPhaseCommonUserTCR_A_Fields(&(soundingPhasePtr->commonUserTcra), pMuPlanConfig);
	//	Fill Sounding phase TCR_B (TF BFP broadcast/unicast) common user
	planManagerSetStaticSoundingPhaseCommonUserTCR_B_Fields(&(soundingPhasePtr->commonUserTcrb), pMuPlanConfig);
	// Fill Sounding phase per user
	planManagerSetStaticSoundingPhasePerUser(&(soundingPhasePtr->userPhase[0]), pMuPlanConfig);
	// Fill Soudning TF params
	planManagerSetStaticSoundingCommonTfParameters(soundingPhasePtr, pMuPlanConfig);
	// Fill Sounding TF per user params
	planManagerSetStaticSoundingPhaseTfInfoPerUser(&(soundingPhasePtr->tfUserInfo[0]), pMuPlanConfig);
}

/**********************************************************************************
planManagerSetStaticSoundingPhaseCommonFields 

Description: 
Sets the common section in Sounding phase 
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhaseCommonFields(SoundingPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, HeGroupManagerStaticGroupId_e groupId, uint8 vapId)
{
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	
	uint8 loopIndex; 
	//word0
	for(loopIndex = 0; loopIndex < heGroupManagerDb[groupId].numOfStationsInGroup; loopIndex++)
	{
		phaseCommonPtr->validUsers310 |= 1<<loopIndex; 
	}			
	//TBD fill USERs indexes which are greater than 31 (in field validUsers3532)
	//word1
	phaseCommonPtr->validUsers3532 						= 0;
	
	phaseCommonPtr->phaseType 							= PHASE_TYPE_SOUNDING_PHASE; 	// Sounding
	phaseCommonPtr->oneShotPhase 						= 0x0; 							// MU-MIMO vs OFDMA pre aggreagtor (need sounding every time before DL: in case of MU-MIMO sounding is needed every time; in case of DL-OFDMA sounding is needed only once)
	phaseCommonPtr->partialDuration 					= 0x0; 							// NA we don't use partial duration
#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	phaseCommonPtr->broadcastTfAidCfg 					= 0x1; 							 
	phaseCommonPtr->unicastTfTypeCfg  					= 0x1;
#endif
	phaseCommonPtr->keepPlanLock 						= 0x0; 							// NA 
	phaseCommonPtr->moveStaToSu 						= 0x0;							// NA
	phaseCommonPtr->tfParamTriggerType 					= TF_TYPE_BF_RPT_POLL;
//word2
	phaseCommonPtr->actionBitmap 						= 0x0;
	phaseCommonPtr->plannedBfrpTf 						= 0x1;
	phaseCommonPtr->plannedPhyNdp 						= 0x1;
	phaseCommonPtr->plannedBfReq 						= 0x1;
	phaseCommonPtr->minimumNumberOfUsersForTransmission = 0;
	phaseCommonPtr->tfInfoBaseAddrDwOffset 				= CONVERT_BYTES_TO_WORDS(OFFSETOF(SoundingPhase_t, tfParameters));    // offset from phase start to tfParameters						
//word3
	phaseCommonPtr->bfTxSeqType							= 0x3; 							// options: {00 - NO BF sequence; 01 - BF sequence w/o data; 11 - BF sequence before data PPDU}
	phaseCommonPtr->doubleSoundingEn 					= 0;							// MU-MIMO limited to 4 users and we don't support in OFDMA
	phaseCommonPtr->nssMaxNdp 							= pVapDbHwEntry->common.maxNssNdp;	// max number according to the maximum in vap DB (and should fit the NSS part of the rate that was filef for Soundig PHY NDP TCR (TCR_A)

// saveUserBitmapForData - options: {0 - Don't Save User Bitmap, 1 - Save User Bitmap} for MIMO require to set this bit so the Sounding and Data will be based on the same valid users
// soundingPartialSuccessCond - options: {0 - Full - success when received BF for all users (i.e in MIMO mode); 1 - Partial - success when received not for all BF users (i.e. OFDMA Mode)} define how to treat the BF results 
// muMimoDataTransmission - options: {0 - sounding is used for OFDMA (not for MIMO); 1 - sounding is used for MIMO operation the bit is used by the Sender for configuration of MBFM module} -	this configuration is used by the Sender to configure the MBFM and Sequencer HW. In MU-MIMO the Sequencer will activate the Data phase only if all the BF Reports successfully received from all (otherwise there is no point that PHY will transmit it)
if (staticPlanConfigurationPtr->commonSection.muType == OFDMA)
{
	phaseCommonPtr->saveUserBitmapForData 				= 0; 
	phaseCommonPtr->soundingPartialSuccessCond 			= 1;
	phaseCommonPtr->muMimoDataTransmission 				= 0; 			
}
else //MU-MIMO
{
	phaseCommonPtr->saveUserBitmapForData 				= 1; 
	phaseCommonPtr->soundingPartialSuccessCond 			= 0;
	phaseCommonPtr->muMimoDataTransmission 				= 1; 			
}
	phaseCommonPtr->tfBroadcastUnicastMode 				= 0;							// options: {0 - broadcast; 1 - unicast}	
#ifdef ENET_INC_ARCH_WAVE600D2	
		phaseCommonPtr->multiBssid							= heGroupManagerDb[groupId].mbssRxCtrlSupport; 
#endif

	phaseCommonPtr->groupIndex 							= groupId;						// only relevant for VHT group
//word4
	// configure NDPA pointer struct
	planManagerSetStaticSoundingPhaseNdpaPointer(staticPlanConfigurationPtr, groupId);
}

/**********************************************************************************
planManagerSetStaticSoundingPhaseNdpaPointer 

Description: 
Sets the NDPA pointer struct in Sounding phase 
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhaseNdpaPointer(UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 groupIndex)
{
	uint8 numOfParticipatingStations = heGroupManagerDb[groupIndex].numOfStationsInGroup; 
	uint8 userIndex;
	
	heSoundingNdpaArray[groupIndex].common.numOfStations = heGroupManagerDb[groupIndex].numOfStationsInGroup;
	for(userIndex = 0; userIndex < numOfParticipatingStations; userIndex++)
	{
		heSoundingNdpaArray[groupIndex].sta[userIndex].aid 				 = staticPlanConfigurationPtr->perUserParameters[userIndex].uspStationIndexes + 1;
		heSoundingNdpaArray[groupIndex].sta[userIndex].partialBwInfo 	 = 0;  //filled by Tx Sender
		heSoundingNdpaArray[groupIndex].sta[userIndex].feedbackTypeAndNg = 0x1;	//b26 = 0; b25 = 1; together with codeBook Size value should be 0x3
		heSoundingNdpaArray[groupIndex].sta[userIndex].disambiguation    = 1;
		heSoundingNdpaArray[groupIndex].sta[userIndex].codebookSize      = 1; //{7,9}//amount of bits per phase SU(OFDMA), MU (MIMO or OFDMA), CQI
	}
}

/**********************************************************************************
planManagerSetStaticSoundingPhaseCommonTcr_A_Fields 

Description: 
Sets the TCR_A common section in Sounding phase: used for NDPA and NDP in broadcast sent in HE-SU mode
NDPA TCR is updated by Sender	
**********************************************************************************/
static void	planManagerSetStaticSoundingPhaseCommonTcr_A_Fields(SoundingPhaseCommonTcra_t*  phaseCommonTcraPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t* planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonNdpTcrPtr = (TcrCommon_t*)(phaseCommonTcraPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];
	//word1	
	phaseCommonNdpTcrPtr->rfPower 					= staticPlanConfigurationPtr->commonSection.rfPower; 
    // In MU-MIMO open loop is used so that Tx power will not be differnt between TX and RX
	phaseCommonNdpTcrPtr->txLoopMode				= (staticPlanConfigurationPtr->commonSection.muType == OFDMA) ? TX_LOOP_MODE_CLOSED_LOOP : TX_LOOP_MODE_OPEN_LOOP; 
	phaseCommonNdpTcrPtr->cbw 						= staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseCommonNdpTcrPtr->phyMode 					= PHY_MODE_11AX_SU; 
	phaseCommonNdpTcrPtr->heFullbandMu 				= 0x0;  // NDP is HE-SU only one LTF per antenna
	phaseCommonNdpTcrPtr->hePuncturingMap 			= 0XFF; // NA puncturing is used to indicate which SB to use when BW=160MHz and interference is detected on one of the subsbands 
	phaseCommonNdpTcrPtr->forceTx 					= 0x0;	// Default in FW is 0x0 Sender might override e.g. in case of Beacon	
	// word4
	phaseCommonNdpTcrPtr->bssColor 					= pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonNdpTcrPtr->customBf 					= 0; 	// NA - for proudction 0x1, 0x2
	// we are always DL unless station mode
	phaseCommonNdpTcrPtr->ulDl 						= PHASE_FORMAT_DL_DATA; 
	phaseCommonNdpTcrPtr->txopPs 					= 0x0; 	// NA
	phaseCommonNdpTcrPtr->expectedRcr 				= 0x0;
	phaseCommonNdpTcrPtr->bwChange 					= 0;  	//NA
	// if partial txop mode - then take the duration from MAC 
	if(planPtr->durationUpdateMode == 0) 
	{
		phaseCommonNdpTcrPtr->txopDurationFromMac 	= 1; 
	}
	else //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonNdpTcrPtr->txopDurationFromMac 	= 0;
	} 
	phaseCommonNdpTcrPtr->aggregate 				= 0x0; // NA relevant in HT only
	phaseCommonNdpTcrPtr->dynamicBw 				= 0x0; // NA
	phaseCommonNdpTcrPtr->muPhyNdp 					= 0x1; // used for MU-MIMO
	phaseCommonNdpTcrPtr->muTraining 				= 0x0; // NA
	phaseCommonNdpTcrPtr->heCp 						= HE_CP_HE_1_6_CP; // 1.6us CP and 2X LTF is mandatory for HE NDP
	phaseCommonNdpTcrPtr->heLtf 					= HE_LTF_HE_2X_LTF; // 1.6us CP and 2X LTF is mandatory for HE NDP 
	phaseCommonNdpTcrPtr->scp 						= 0x0; // don't care as phy mode is HE  (this field is relevant for HT/VHT)
	phaseCommonNdpTcrPtr->notSounding 				= 0x0; // NA - this field is relevant for HT
	//word5
	phaseCommonNdpTcrPtr->spatialReuse 				= 0x0; // NA
	// all triggered based params are relevant for station mode only (he-trigerred based)
	phaseCommonNdpTcrPtr->triggerBasedDataLLenght 	= 0x0; 
	phaseCommonNdpTcrPtr->triggerBasedDataLdpc  	= 0x0;
	phaseCommonNdpTcrPtr->triggerBasedDataPktExt 	= 0x0; 
	phaseCommonNdpTcrPtr->triggerBasedDataPeDisam 	= 0x0;
	//word6
	phaseCommonNdpTcrPtr->groupId 					= 0x0; // NA - relevant for VHT only
	phaseCommonNdpTcrPtr->macDuration 				= MAC_DURATION; // NA
}

/**********************************************************************************
planManagerSetStaticSoundingPhaseCommonTcr_B_Fields 

Description: 
Sets the TCR_B common section in Sounding phase: used for BFP Broadcast/Unicast sent in Legacy mode
	
**********************************************************************************/
static void	planManagerSetStaticSoundingPhaseCommonTcr_B_Fields(SoundingPhaseCommonTcrb_t*  phaseCommonTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonBfpTcrPtr = (TcrCommon_t*)(phaseCommonTcrbPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];

	//word1
	phaseCommonBfpTcrPtr->rfPower  				  = staticPlanConfigurationPtr->commonSection.rfPower;
    // In MU-MIMO open loop is used so that Tx power will not be differnt between TX and RX
	phaseCommonBfpTcrPtr->txLoopMode 			  = (staticPlanConfigurationPtr->commonSection.muType == OFDMA) ? TX_LOOP_MODE_CLOSED_LOOP : TX_LOOP_MODE_OPEN_LOOP; 
	phaseCommonBfpTcrPtr->cbw 					  = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseCommonBfpTcrPtr->phyMode 				  = PHY_MODE_11AG;  // sent in legacy mode 
	phaseCommonBfpTcrPtr->heFullbandMu 			  = 0x0; 			// NA - BFP is legacy mode
	phaseCommonBfpTcrPtr->hePuncturingMap 		  = 0xFF; 			// puncturing is used to indicate which SB to use when BW=160MHz and interference is detected on one of the subsbands
	//word4
	phaseCommonBfpTcrPtr->bssColor 				  = pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonBfpTcrPtr->customBf 				  = 0x0;
	phaseCommonBfpTcrPtr->ulDl 					  = PHASE_FORMAT_DL_DATA;    
	phaseCommonBfpTcrPtr->txopPs 				  = 0x0; 
	phaseCommonBfpTcrPtr->expectedRcr 			  = 0x1;
	phaseCommonBfpTcrPtr->bwChange 			      = 0x0; 
	phaseCommonBfpTcrPtr->forceTx 	 			  = 0x0; // Default in FW is 0x0 Sender might override e.g. in case of Beacon
	// if partial txop mode - then take the duration from MAC 
	if(planPtr->durationUpdateMode == 0) 
	{
		phaseCommonBfpTcrPtr->txopDurationFromMac = 1;
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonBfpTcrPtr->txopDurationFromMac = 0;
	}

	phaseCommonBfpTcrPtr->aggregate 		      = 0x0; // NA; legacy
	phaseCommonBfpTcrPtr->dynamicBw 		      = 0x0; // NA
	phaseCommonBfpTcrPtr->muPhyNdp 			      = 0x0; // NA
	phaseCommonBfpTcrPtr->muTraining 			  = 0x0;
	phaseCommonBfpTcrPtr->notSounding  		      = 0x0; 
	//word5
	phaseCommonBfpTcrPtr->spatialReuse 			  = 0x0;
	// all triggered based params are relevant for station mode only (he-trigerred based)
	phaseCommonBfpTcrPtr->triggerBasedDataLLenght = 0x0;
	phaseCommonBfpTcrPtr->triggerBasedDataLdpc 	  = 0x0;
	phaseCommonBfpTcrPtr->triggerBasedDataPeDisam = 0x0;
	//word6
	phaseCommonBfpTcrPtr->groupId 				  = 0x0; // NA- relevant for VHT only
	phaseCommonBfpTcrPtr->macDuration 			  = MAC_DURATION; //??ask HW
}

/**********************************************************************************
planManagerSetStaticSoundingPhaseCommonRcrFields 

Description: 
Sets the RCR common section in Sounding phase: used for UL BF report
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhaseCommonRcrFields(SoundingPhaseCommonRcr_t* phaseCommonRcrPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId)
{
	RcrCommon_t* phaseRcrCommonPtr = (RcrCommon_t*)(phaseCommonRcrPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId]; 

	//word0
	phaseRcrCommonPtr->heSigABssColor 		= pVapDbHwEntry->common.tcr1BssColor; 
	phaseRcrCommonPtr->cbw 					= staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseRcrCommonPtr->heCp 				= staticPlanConfigurationPtr->commonSection.ul_HeCp; 
	phaseRcrCommonPtr->heLtf 				= staticPlanConfigurationPtr->commonSection.ul_HeLtf;  
	phaseRcrCommonPtr->stbc 				= staticPlanConfigurationPtr->commonSection.rcr_Stbc;
	//word1
	phaseRcrCommonPtr->heSigASpatialReuse 	= staticPlanConfigurationPtr->commonSection.rcr_heSigASpatialReuse; 
	phaseRcrCommonPtr->lSigLength 			= staticPlanConfigurationPtr->commonSection.tf_Length;
	phaseRcrCommonPtr->peDisambiguty 		= 0x1; // We force the station to set PE disambiguty
}

/**********************************************************************************
planManagerSetStaticUlPhaseCommonUserTCR_A_Fields 

Description: 
Sets the common user TCRA common section in Souding phase: used for NDPA and NDP in broadcast sent in HE-SU mode
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhaseCommonUserTCR_A_Fields(SoundingPhaseCommonUserTcra_t* commonUserTcraPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	//word0
	commonUserTcraPtr->brdcstUserTcraSubBand 			= 0x0;  
	commonUserTcraPtr->brdcstUserTcraStartRu 			= 0x0;   
	// should be configured according to BW size table % in MAC-PHY I/F
	commonUserTcraPtr->brdcstUserTcraRuSize 			= CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit);
	commonUserTcraPtr->brdcstUserTcraLdpc 				= 0x0;
	commonUserTcraPtr->brdcstUserTcraTxbf 				= 0x0;  // No BF in case of NDP
	commonUserTcraPtr->brdcstUserTcraRelativeUserPower  = 0x0;  // NA   
	//word1
	commonUserTcraPtr->brdcstUserTcraStaAid 		    = 0x0;  // NA   
	commonUserTcraPtr->brdcstUserTcraBfIndex 	        = 0x0;  // NA   
}

/**********************************************************************************

planManagerSetStaticSoundingPhaseCommonUserTCR_B_Fields 

Description: 
Sets the common user TCRB common section in Sounding phase: used for BFP Broadcast/Unicast sent in Legacy mode
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhaseCommonUserTCR_B_Fields(SoundingPhaseCommonUserTcrb_t* commonUserTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	//word0
	commonUserTcrbPtr->brdcstUserTcrbSubBand 			= 0x0; 
	commonUserTcrbPtr->brdcstUserTcrbStartRu 			= 0x0; 
	// should be configured according to BW size table % in MAC-PHY I/F
	commonUserTcrbPtr->brdcstUserTcrbRuSize 			= CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit);
	commonUserTcrbPtr->brdcstUserTcrbLdpc 				= 0x0; 
	commonUserTcrbPtr->brdcstUserTcrbTxbf 				= 0x0; //NA
	commonUserTcrbPtr->brdcstUserTcrbRelativeUserPower 	= 0x0; //NA
	//word1
	commonUserTcrbPtr->brdcstUserTcrbStaAid 			= 0x0; //NA
	commonUserTcrbPtr->brdcstUserTcrbBfIndex 			= 0x0; //NA
}

/**********************************************************************************
planManagerSetStaticSoundingPhasePerUser 

Description: 
Sets Sounding Phase Per User params
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhasePerUser(SoundingPhaseUserPhase_t* soundingPhasPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8	userLoopIndex; 
	uint8	userStationIndex;
    uint8   vapId;
 
	for(userLoopIndex = 0; userLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; userLoopIndex++)
	{
		//word0
		userStationIndex                          = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes;
		soundingPhasPerUserPtr->staId             = userStationIndex;
        StaDb_GetVapId(userStationIndex, &vapId);
        soundingPhasPerUserPtr->vapId			  = vapId;
		soundingPhasPerUserPtr->tfPadding         = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfPadding; 
		soundingPhasPerUserPtr->numTfRepetition   = 0x0; // NA only relevant in Unicast mode (should be set to 0x1 in Unicast mode)
		soundingPhasPerUserPtr->mapTxUserToRxUser = userLoopIndex; //Mapping is 1 X 1
		//word1
		soundingPhasPerUserPtr->muBfIndex 		  = CONVERT_USP_INDEX_TO_HE_MU_BF_INDEX(userLoopIndex); 
		soundingPhasPerUserPtr->mapTxUserToTxUser =  userLoopIndex; // Mapping TX Valid User to NDPA TX User 	Indicate the relevant NDPA TX User Index:	used by NDPA to map each TX to the transmitted STA in NDPA	Value 0  43
		//word2: BF report RCR
		soundingPhasPerUserPtr->userRcrSubBand 	  = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand; 
		soundingPhasPerUserPtr->userRcrStartRu 	  = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_StartRU; 
		soundingPhasPerUserPtr->userRcrRuSize     = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize; 

		soundingPhasPerUserPtr->userRcrLdpc       = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ldpc; 
		soundingPhasPerUserPtr->userRcrTxbf  	  = 0x0; 
		soundingPhasPerUserPtr->userRcrPsduRate   = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_PsduRatePerUsp;
		soundingPhasPerUserPtr->userRcrTargetRssi = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		soundingPhasPerUserPtr->userRcrValid      = 0x1; 
		soundingPhasPerUserPtr++;
	}
}

/**********************************************************************************
planManagerSetStaticSoundingCommonTfParameters 

Description: 
Sets Sounding TF Common params info 
	
**********************************************************************************/
static void planManagerSetStaticSoundingCommonTfParameters(SoundingPhase_t* soundingPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig)
{
	SoundingPhaseTfParameters_t* commonTfParamsPtr = &(soundingPhasePtr->tfParameters); 
	SoundingPhaseTfCommonInfo_t* commonTfInfoPtr = &(soundingPhasePtr->tfCommonInfo); 

	commonTfParamsPtr->tfLsigLengthOverride				= 0; 
	commonTfParamsPtr->tfBwOverride						= 1;
	commonTfParamsPtr->tfMcsOverride					= 1;
	commonTfParamsPtr->tfDcmOverride					= 1;
	commonTfParamsPtr->tfNssOverride					= 1;
	commonTfParamsPtr->tfTargetRssiOverride				= 0;
	commonTfParamsPtr->tfTidAggrLimitOverride			= 0;
	commonTfParamsPtr->tfPreferredAcOverride			= 0;
	commonTfParamsPtr->tfMultiTidOverride				= 0;
	commonTfParamsPtr->tfTidInfoOverride				= 0; 
	commonTfParamsPtr->tfCompressedFnOverride			= 0;
	commonTfParamsPtr->tfCompressedSsnOverride			= 0;
	commonTfParamsPtr->tfMultiTidTidOverride			= 0;
	commonTfParamsPtr->tfMultiTidFnOverride				= 0;
	commonTfParamsPtr->tfMultiTidSsnOverride			= 0;
	commonTfParamsPtr->tfCommonInfoSwOverrideEnable 	= 0; 
	commonTfParamsPtr->tfUserInfoSwOverrideEnable		= 0; 

#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfParamsPtr->tfMoreTfOverride 				= 0; 
#else
	commonTfParamsPtr->tfCascadeIndicationOverride 		= 0; 
#endif
	commonTfParamsPtr->tfRuPrimarySecondary80Override	= 1;  
	commonTfParamsPtr->tfRuIndexOverride				= 1;
#else
	commonTfParamsPtr->tfRuAllocationOverride			= 1;
#endif 

	//word0	
	commonTfInfoPtr->tfTriggerType 				= TF_TYPE_BF_RPT_POLL;
	commonTfInfoPtr->tfLength					= pMuPlanConfig->commonSection.tf_Length;
#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfInfoPtr->tfMoreTf 					= 0x0;	// no more tf in this txop													
#else
	commonTfInfoPtr->tfCascadeIndication 		= 0x0;	// no more tf in this txop													
#endif	
	commonTfInfoPtr->tfCsRequired 				= 0x0;  // should a station consider CCA before UL  - only for MU RTS should be set with 1  
	commonTfInfoPtr->tfBw 						= pMuPlanConfig->commonSection.startBwLimit; 
	commonTfInfoPtr->tfGiAndLtfType 			= pMuPlanConfig->commonSection.tf_heGI_and_Ltf;
	commonTfInfoPtr->tfMuMimoLtfMode			= 0x0; //0: Single Stream Pilots   1: Mask on each spatial stream of the LTF sequence by a distinct orthogonal code MimO on the TB NA
	commonTfInfoPtr->tfSpatialReuse				= 0x0; 
	//word1	
	commonTfInfoPtr->tfDoppler 					= 0x0; //NA 
#ifdef ENET_INC_ARCH_WAVE600D2		
	commonTfInfoPtr->tfHeSigA2Reserved			= 0x1FF; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#else
	commonTfInfoPtr->tfHeSigAReserved			= 0x1FF; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#endif	
}

/**********************************************************************************
planManagerSetStaticSoundingPhaseTfInfoPerUser 

Description: 
Sets Sounding TF info per user params
	
**********************************************************************************/
static void planManagerSetStaticSoundingPhaseTfInfoPerUser(SoundingPhaseTfUserInfo_t* soundingPhasTfInfoPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8	userLoopIndex; 
	uint8	userStationIndex;
	uint8	numOfLoopIterations;
	uint8	remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET; 

	numOfLoopIterations = staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; 
	if(numOfLoopIterations % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset = numOfLoopIterations;
		numOfLoopIterations = numOfLoopIterations - 1; //loop will run over even number of users since in every loop two users are filled
	}

	for(userLoopIndex = 0; userLoopIndex < numOfLoopIterations; userLoopIndex = userLoopIndex+2)
	{		
		/* First user in pair*/
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		soundingPhasTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		soundingPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		soundingPhasTfInfoPerUserPtr->tfRuAllocation0 =  planManagerDefaultSettings.UlPerUserTf_RuAllocation; 
#endif
#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		soundingPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif 
		soundingPhasTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation); // override by the RCR value  
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2		
		soundingPhasTfInfoPerUserPtr->tfFecCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#else
		soundingPhasTfInfoPerUserPtr->tfCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#endif
		soundingPhasTfInfoPerUserPtr->tfMcs0 =  EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_psduRate); //override by the RCR value 
		soundingPhasTfInfoPerUserPtr->tfDcm0 = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		soundingPhasTfInfoPerUserPtr->tfStartingSpatialStream0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfStartingSS;
		soundingPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 =  EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_psduRate);
		soundingPhasTfInfoPerUserPtr->tfTargetRssi0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		soundingPhasTfInfoPerUserPtr->tfTriggerDependent0 = 0xFF;
		/* Second user in pair*/
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].uspStationIndexes; 
		soundingPhasTfInfoPerUserPtr->tfAid121 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		soundingPhasTfInfoPerUserPtr->tfRuAllocation130 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ruSize); 
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		soundingPhasTfInfoPerUserPtr->tfFecCodingType1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ldpc;
#else
		soundingPhasTfInfoPerUserPtr->tfCodingType1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_ldpc;
#endif
		soundingPhasTfInfoPerUserPtr->tfMcs1 = EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_PsduRatePerUsp); //override by the RCR value 
		soundingPhasTfInfoPerUserPtr->tfDcm1 =  planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		soundingPhasTfInfoPerUserPtr->tfStartingSpatialStream1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].tfStartingSS;
		soundingPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams1 = EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_PsduRatePerUsp);
		soundingPhasTfInfoPerUserPtr->tfTargetRssi1 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].targetRssi;
		soundingPhasTfInfoPerUserPtr->tfTriggerDependent1 = 0xFF;

		soundingPhasTfInfoPerUserPtr++; 
	}
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userLoopIndex = remainderUspOffset -1; 
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		soundingPhasTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		soundingPhasTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		soundingPhasTfInfoPerUserPtr->tfRuAllocation0 =  planManagerDefaultSettings.UlPerUserTf_RuAllocation; // override by the RCR value  
#endif 
#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		soundingPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif
		soundingPhasTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation);
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		soundingPhasTfInfoPerUserPtr->tfFecCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#else
		soundingPhasTfInfoPerUserPtr->tfCodingType0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].codingType_BCC_OR_LDPC;
#endif
		soundingPhasTfInfoPerUserPtr->tfMcs0 =  0x4;// EXTRACT_MCS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_psduRate); //override by the RCR value 
		soundingPhasTfInfoPerUserPtr->tfDcm0 = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		soundingPhasTfInfoPerUserPtr->tfStartingSpatialStream0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfStartingSS;
		soundingPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = 0x0; //EXTRACT_NSS_FROM_VHT_HE_RATE(staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_psduRate);
		soundingPhasTfInfoPerUserPtr->tfTargetRssi0 = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].targetRssi;
		soundingPhasTfInfoPerUserPtr->tfTriggerDependent0	= 0xFF; //union of the 4 below fields 
	}
}

/**********************************************************************************

planManagerSetProtectionPhase 

Description: 
Sets sounding phase parameters 
	
**********************************************************************************/
static void planManagerSetStaticProtectionPhase(ProtectionPhase_t* protectionPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t*  planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId)
{
	heGroupManagerSetPlanTidsBitmap(TID_BITMAP_ALL_DATA_TIDS, pMuPlanConfig->commonSection.muSequenceType);
	// Fill Protection phase Common fields 
	planManagerSetStaticProtectionPhaseCommonFields(&(protectionPhasePtr->common), pMuPlanConfig, groupId);
	// Fill Protection phase TCR B (TF MU-RTS broadcast) common fields 
	planManagerSetStaticProtectionPhaseCommonTcr_B_Fields(&(protectionPhasePtr->commonTcrb), pMuPlanConfig, planPtr, vapId);
	// Fill Protection phase RCR (MU-RTS) common fields 
	planManagerSetStaticProtectionPhaseCommonRcrFields(&(protectionPhasePtr->commonRcr), pMuPlanConfig, vapId);
	// Fill Protection phase TCR_B (TF MU-RTS broadcast) common user
	planManagerSetStaticProtectionPhaseCommonUserTCR_B_Fields(&(protectionPhasePtr->commonUserTcrb), pMuPlanConfig);
	// Fill Protection phase per user
	planManagerSetStaticProtectionPhasePerUser(&(protectionPhasePtr->userPhase[0]), pMuPlanConfig);
	// Fill Protection TF params
	planManagerSetStaticProtectionCommonTfParameters(protectionPhasePtr, pMuPlanConfig);
	// Fill Protection TF per user params
	planManagerSetStaticProtectionPhaseTfInfoPerUser(&(protectionPhasePtr->tfUserInfo[0]), pMuPlanConfig);
}

/**********************************************************************************
planManagerSetStaticProtectionPhaseCommonFields 

Description: 
Sets the common section in Protection phase 
	
**********************************************************************************/
static void planManagerSetStaticProtectionPhaseCommonFields(ProtectionPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, HeGroupManagerStaticGroupId_e groupId)
{	
	uint8 loopIndex; 

	UNUSED_PARAM(staticPlanConfigurationPtr);
	//word0
	for(loopIndex = 0; loopIndex < heGroupManagerDb[groupId].numOfStationsInGroup; loopIndex++)
	{
		phaseCommonPtr->validUsers310 |= 1<<loopIndex; 
	}			
	//TBD fill USERs indexes which are greater than 31 (in field validUsers3532)
	//word1
	phaseCommonPtr->validUsers3532 						= 0;
	
	phaseCommonPtr->phaseType 							= PHASE_TYPE_PROTECTION_PHASE; 
	phaseCommonPtr->oneShotPhase 						= 0x0; 		
	phaseCommonPtr->partialDuration 					= 0x0; 							// NA we don't use partial duration
#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	phaseCommonPtr->broadcastTfAidCfg 					= 0x1; 							 
	phaseCommonPtr->unicastTfTypeCfg  					= 0x1;
#endif	
	phaseCommonPtr->keepPlanLock 						= 0x0; 							// NA 
	phaseCommonPtr->moveStaToSu 						= 0x0;							// NA
	phaseCommonPtr->tfParamTriggerType 					= TF_TYPE_MU_RTS;
	//word2
	phaseCommonPtr->actionBitmap 						= 0x0;
	phaseCommonPtr->plannedMuRts						= 0x1;
	phaseCommonPtr->minimumNumberOfUsersForTransmission = 0;
	phaseCommonPtr->tfInfoBaseAddrDwOffset 				= CONVERT_BYTES_TO_WORDS(OFFSETOF(ProtectionPhase_t, tfParameters));    // offset from phase start to tfParameters
	//word3
	phaseCommonPtr->rtsCtsFormat 						= RTS_CTS_FORMAT_INDICATION_INVALID_VALUE; // We use MU-RTS so RTS-CTS format is invalid
	phaseCommonPtr->fillerType 							= FILLER_TYPE_DISABLED;
	phaseCommonPtr->groupIndex 							= groupId;
	
#ifdef ENET_INC_ARCH_WAVE600D2	
	phaseCommonPtr->multiBssid							= heGroupManagerDb[groupId].mbssRxCtrlSupport; 
#endif
}

/**********************************************************************************
planManagerSetStaticProtectionPhaseCommonTcr_B_Fields 

Description: 
Sets the TCR_B common section in Protection phase: used for Broadcast MU-RTS, sent in Legacy mode
	
**********************************************************************************/
static void	planManagerSetStaticProtectionPhaseCommonTcr_B_Fields(ProtectionPhaseCommonTcrb_t*  phaseCommonTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, CommonPlan_t*  planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonMuRtsTcrPtr = (TcrCommon_t*)(phaseCommonTcrbPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId];

	//word1
	phaseCommonMuRtsTcrPtr->rfPower  			  = staticPlanConfigurationPtr->commonSection.rfPower;
	phaseCommonMuRtsTcrPtr->txLoopMode 			  = TX_LOOP_MODE_CLOSED_LOOP; 
	phaseCommonMuRtsTcrPtr->cbw 				  = staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseCommonMuRtsTcrPtr->phyMode 			  = PHY_MODE_11AG;  // sent in legacy mode 
	phaseCommonMuRtsTcrPtr->heFullbandMu 		  = 0x0; 			// NA - MU-RTS is legacy mode
	phaseCommonMuRtsTcrPtr->hePuncturingMap 	  = 0xFF; 			// puncturing is used to indicate which SB to use when BW=160MHz and interference is detected on one of the subsbands
	//word4
	phaseCommonMuRtsTcrPtr->bssColor 			  = pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonMuRtsTcrPtr->customBf 			  = 0x0;
	phaseCommonMuRtsTcrPtr->ulDl 				  = PHASE_FORMAT_DL_DATA;    
	phaseCommonMuRtsTcrPtr->txopPs 				  = 0x0; 
	phaseCommonMuRtsTcrPtr->expectedRcr 		  = 0x0;  // In MU-RTS - expected RCR = 0, as we don't expect for HE TB PPDU
	phaseCommonMuRtsTcrPtr->bwChange 			  = 0x0; 
	phaseCommonMuRtsTcrPtr->forceTx 	 		  = 0x0; // Default in FW is 0x0 Sender might override e.g. in case of Beacon
	// if partial txop mode - then take the duration from MAC 
	if(planPtr->durationUpdateMode == 0) 
	{
		phaseCommonMuRtsTcrPtr->txopDurationFromMac = 1;
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonMuRtsTcrPtr->txopDurationFromMac = 0;
	}

	phaseCommonMuRtsTcrPtr->aggregate 		      = 0x0; // NA; legacy
	phaseCommonMuRtsTcrPtr->dynamicBw 		      = 0x0; // NA
	phaseCommonMuRtsTcrPtr->muPhyNdp 			  = 0x0; // NA
	phaseCommonMuRtsTcrPtr->muTraining 			  = 0x0;
	phaseCommonMuRtsTcrPtr->notSounding  		  = 0x0; 
	//word5
	phaseCommonMuRtsTcrPtr->spatialReuse 		  = 0x0;
	// all triggered based params are relevant for station mode only (he-trigerred based)
	phaseCommonMuRtsTcrPtr->triggerBasedDataLLenght = 0x0;
	phaseCommonMuRtsTcrPtr->triggerBasedDataLdpc    = 0x0;
	phaseCommonMuRtsTcrPtr->triggerBasedDataPktExt  = 0x0;	
	phaseCommonMuRtsTcrPtr->triggerBasedDataPeDisam = 0x0;
	//word6
	phaseCommonMuRtsTcrPtr->groupId 			  = 0x0; // NA- relevant for VHT only
	phaseCommonMuRtsTcrPtr->macDuration 		  = MAC_DURATION; //??ask HW
}

/**********************************************************************************
planManagerSetStaticProtectionPhaseCommonRcrFields 

Description: 
Sets the RCR common section in Sounding phase: used for MU-RTS
	
**********************************************************************************/
static void planManagerSetStaticProtectionPhaseCommonRcrFields(ProtectionPhaseCommonRcr_t* phaseCommonRcrPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId)
{
	RcrCommon_t* phaseRcrCommonPtr = (RcrCommon_t*)(phaseCommonRcrPtr);
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId]; 

	//word0
	phaseRcrCommonPtr->heSigABssColor 		= pVapDbHwEntry->common.tcr1BssColor; 
	phaseRcrCommonPtr->cbw 					= staticPlanConfigurationPtr->commonSection.startBwLimit; 
	phaseRcrCommonPtr->heCp 				= 0x0; 
	phaseRcrCommonPtr->heLtf 				= 0x0;  
	phaseRcrCommonPtr->stbc 				= 0x0;
	//word1
	phaseRcrCommonPtr->heSigASpatialReuse 	= 0x0; 
	phaseRcrCommonPtr->lSigLength 			= 0x0;
	phaseRcrCommonPtr->peDisambiguty 		= 0x0; 
}
/**********************************************************************************

planManagerSetStaticProtectionPhaseCommonUserTCR_B_Fields 

Description: 
Sets the common user TCRB common section in Protection phase: used for Broadcast MU-RTS, sent in Legacy mode
	
**********************************************************************************/
static void planManagerSetStaticProtectionPhaseCommonUserTCR_B_Fields(ProtectionPhaseCommonUserTcrb_t* commonUserTcrbPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	//word0
	commonUserTcrbPtr->brdcstUserTcrbSubBand 			= 0x0; 
	commonUserTcrbPtr->brdcstUserTcrbStartRu 			= 0x0; 
	// should be configured according to BW size table % in MAC-PHY I/F
	commonUserTcrbPtr->brdcstUserTcrbRuSize 			= CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit);
	commonUserTcrbPtr->brdcstUserTcrbLdpc 				= 0x0; 
	commonUserTcrbPtr->brdcstUserTcrbTxbf 				= 0x0; //NA
	commonUserTcrbPtr->brdcstUserTcrbRelativeUserPower 	= 0x0; //NA
	//word1
	commonUserTcrbPtr->brdcstUserTcrbStaAid 			= 0x0; //NA
	commonUserTcrbPtr->brdcstUserTcrbBfIndex 			= 0x0; //NA
}

/**********************************************************************************
planManagerSetStaticProtectionPhasePerUser 

Description: 
Sets Protection Phase Per User params
	
**********************************************************************************/
static void planManagerSetStaticProtectionPhasePerUser(ProtectionPhaseUserPhase_t* protectionPhasePerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8	userLoopIndex; 
	uint8	userStationIndex;
    uint8   vapId;

	for(userLoopIndex = 0; userLoopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; userLoopIndex++)
	{
		//word0
		userStationIndex                              = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes;
		protectionPhasePerUserPtr->staId              = userStationIndex;
        StaDb_GetVapId(userStationIndex, &vapId);
		protectionPhasePerUserPtr->vapId		      = vapId;
		protectionPhasePerUserPtr->tfPadding          = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].tfPadding; 
		//word1
		protectionPhasePerUserPtr->userRcrSubBand 	  = 0x0; 
		protectionPhasePerUserPtr->userRcrStartRu 	  = 0x0; 
		protectionPhasePerUserPtr->userRcrRuSize      = CONVERT_BW_TO_RU_SIZE(staticPlanConfigurationPtr->commonSection.startBwLimit); 

		protectionPhasePerUserPtr->userRcrLdpc        = 0x0; 
		protectionPhasePerUserPtr->userRcrTxbf  	  = 0x0; 
		protectionPhasePerUserPtr->userRcrPsduRate    = 0x0;
		protectionPhasePerUserPtr->userRcrTargetRssi  = 0x0;
		protectionPhasePerUserPtr->userRcrValid       = 0x1; 
		protectionPhasePerUserPtr++;
	}
}

/**********************************************************************************
planManagerSetStaticProtectionCommonTfParameters 

Description: 
Sets Protection TF Common params info 
	
**********************************************************************************/
static void planManagerSetStaticProtectionCommonTfParameters(ProtectionPhase_t* protectionPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig)
{
	ProtectionPhaseTfParameters_t* commonTfParamsPtr = &(protectionPhasePtr->tfParameters); 
	ProtectionPhaseTfCommonInfo_t* commonTfInfoPtr = &(protectionPhasePtr->tfCommonInfo); 

	commonTfParamsPtr->tfOverrideEnableParams           = 0; 
	commonTfParamsPtr->tfBwOverride						= 1;
#ifdef ENET_INC_ARCH_WAVE600B	
	commonTfParamsPtr->tfRuPrimarySecondary80Override	= 1;  
	commonTfParamsPtr->tfRuIndexOverride				= 1;
#else
	commonTfParamsPtr->tfRuAllocationOverride			= 1;
#endif 

	//word0	
	commonTfInfoPtr->tfTriggerType 				= TF_TYPE_MU_RTS;
	commonTfInfoPtr->tfLength					= 0x0;
#ifdef ENET_INC_ARCH_WAVE600D2	
	commonTfInfoPtr->tfMoreTf					= 0x0;
#else
	commonTfInfoPtr->tfCascadeIndication 		= 0x0;
#endif
	commonTfInfoPtr->tfCsRequired 				= 0x1;
	commonTfInfoPtr->tfBw 						= pMuPlanConfig->commonSection.startBwLimit; 
	commonTfInfoPtr->tfGiAndLtfType 			= 0x0;
	commonTfInfoPtr->tfMuMimoLtfMode			= 0x0; //0: Single Stream Pilots   1: Mask on each spatial stream of the LTF sequence by a distinct orthogonal code MimO on the TB NA
	commonTfInfoPtr->tfNumberOfHeLtfSymbols 	= 0x0;	
	commonTfInfoPtr->tfStbc  					= 0x0;
	commonTfInfoPtr->tfLdpcExtraSymbolSegment   = 0x0;	
	commonTfInfoPtr->tfApTxPower30 				= 0x0;
	//word1		
	commonTfInfoPtr->tfApTxPower54 				= 0x0;	
	commonTfInfoPtr->tfPacketExtension			= 0x0;
	commonTfInfoPtr->tfSpatialReuse				= 0x0; 
	commonTfInfoPtr->tfDoppler 					= 0x0;  
#ifdef ENET_INC_ARCH_WAVE600D2			
	commonTfInfoPtr->tfHeSigA2Reserved			= 0x1FF; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#else
	commonTfInfoPtr->tfHeSigAReserved			= 0x1FF; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#endif	
}

/**********************************************************************************
planManagerSetStaticProtectionPhaseTfInfoPerUser 

Description: 
Sets Protection TF info per user params
	
**********************************************************************************/
static void planManagerSetStaticProtectionPhaseTfInfoPerUser(ProtectionPhaseTfUserInfo_t* protectionPhaseTfInfoPerUserPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8	userLoopIndex; 
	uint8	userStationIndex;
	uint8	numOfLoopIterations;
	uint8	remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET; 

	numOfLoopIterations = staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; 
	if(numOfLoopIterations % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset = numOfLoopIterations;
		numOfLoopIterations = numOfLoopIterations - 1; //loop will run over even number of users since in every loop two users are filled
	}

	for(userLoopIndex = 0; userLoopIndex < numOfLoopIterations; userLoopIndex = userLoopIndex+2)
	{		
		/* First user in pair*/
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		protectionPhaseTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		protectionPhaseTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		protectionPhaseTfInfoPerUserPtr->tfRuAllocation0 =  planManagerDefaultSettings.UlPerUserTf_RuAllocation; 
#endif
#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		protectionPhaseTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif 
		protectionPhaseTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.DlPerUserTf_RuAllocation); // override by the RCR value  
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2		
		protectionPhaseTfInfoPerUserPtr->tfFecCodingType0 = 0x0;
#else
		protectionPhaseTfInfoPerUserPtr->tfCodingType0 = 0x0;
#endif
		protectionPhaseTfInfoPerUserPtr->tfMcs0 = 0x0; 
		protectionPhaseTfInfoPerUserPtr->tfDcm0 = 0x0; 
		protectionPhaseTfInfoPerUserPtr->tfStartingSpatialStream0 = 0x0;
		protectionPhaseTfInfoPerUserPtr->tfNumberOfSpatialStreams0 =  0x0;
		protectionPhaseTfInfoPerUserPtr->tfTargetRssi0 = 0x0;
		protectionPhaseTfInfoPerUserPtr->tfTriggerDependent0 = 0x0;

		/* Second user in pair*/
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].uspStationIndexes; 
		protectionPhaseTfInfoPerUserPtr->tfAid121 = userStationIndex+1; 
#ifndef ENET_INC_ARCH_WAVE600B
		protectionPhaseTfInfoPerUserPtr->tfRuAllocation130  = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) & RU_ALLOCATION_MASK; //override by the RCR value 
		protectionPhaseTfInfoPerUserPtr->tfRuAllocation174 = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) >> RU_ALLOCATION_SHIFT; //override by the RCR value 
#else
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		protectionPhaseTfInfoPerUserPtr->tfRuPrimarySecondary801 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex+1].ul_SubBand);
#endif 
		protectionPhaseTfInfoPerUserPtr->tfRuIndex2To01 = ((planManagerDefaultSettings.DlPerUserTf_RuAllocation) & RU_ALLOCATION_B_MASK); //override by the RCR value
		protectionPhaseTfInfoPerUserPtr->tfRuIndex6To31 = (planManagerDefaultSettings.DlPerUserTf_RuAllocation) >> RU_ALLOCATION_B_SHIFT; //override by the RCR value
#endif
#ifdef ENET_INC_ARCH_WAVE600D2		
		protectionPhaseTfInfoPerUserPtr->tfFecCodingType1 = 0x0;
#else
		protectionPhaseTfInfoPerUserPtr->tfCodingType1 = 0x0;
#endif
		protectionPhaseTfInfoPerUserPtr->tfMcs1 = 0x0; 
		protectionPhaseTfInfoPerUserPtr->tfDcm1 = 0x0; 
		protectionPhaseTfInfoPerUserPtr->tfStartingSpatialStream1 = 0x0;
		protectionPhaseTfInfoPerUserPtr->tfNumberOfSpatialStreams1 =  0x0;
		protectionPhaseTfInfoPerUserPtr->tfTargetRssi1 = 0x0;
		protectionPhaseTfInfoPerUserPtr->tfTriggerDependent1 = 0x0;

		protectionPhaseTfInfoPerUserPtr++; 
	}
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userLoopIndex = remainderUspOffset - 1; 
		userStationIndex = staticPlanConfigurationPtr->perUserParameters[userLoopIndex].uspStationIndexes; 
		protectionPhaseTfInfoPerUserPtr->tfAid120 = userStationIndex+1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
		protectionPhaseTfInfoPerUserPtr->tfRuAllocation0 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_ruSize); 
#else 
		protectionPhaseTfInfoPerUserPtr->tfRuAllocation0 =	planManagerDefaultSettings.UlPerUserTf_RuAllocation; 
#endif
#ifdef ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
		protectionPhaseTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(staticPlanConfigurationPtr->commonSection.startBwLimit,staticPlanConfigurationPtr->perUserParameters[userLoopIndex].ul_SubBand);
#endif 
		protectionPhaseTfInfoPerUserPtr->tfRuIndex0 = (planManagerDefaultSettings.DlPerUserTf_RuAllocation); // override by the RCR value  
#endif 
#ifdef ENET_INC_ARCH_WAVE600D2		
		protectionPhaseTfInfoPerUserPtr->tfFecCodingType0 = 0x0;
#else
		protectionPhaseTfInfoPerUserPtr->tfCodingType0 = 0x0;
#endif
		protectionPhaseTfInfoPerUserPtr->tfMcs0 = 0x0; 
		protectionPhaseTfInfoPerUserPtr->tfDcm0 = 0x0; 
		protectionPhaseTfInfoPerUserPtr->tfStartingSpatialStream0 = 0x0;
		protectionPhaseTfInfoPerUserPtr->tfNumberOfSpatialStreams0 =  0x0;
		protectionPhaseTfInfoPerUserPtr->tfTargetRssi0 = 0x0;
		protectionPhaseTfInfoPerUserPtr->tfTriggerDependent0 = 0x0;
	}
}

/**********************************************************************************

planManagerRemoveStaticPlan 

Description: 
Remove static Plan 
	
**********************************************************************************/
void planManagerRemoveStaticPlan(K_MSG* psMsg)
{
	RemovePlanMsg_t*			removeStaticPlanMsgPtr;
	CommonPlan_t* 				planPtr = NULL;
	K_MSG*						laMessagePtr;
	LaRemoveHeGroupParams_t*	laRemoveParamsPtr; 
						
	removeStaticPlanMsgPtr = (RemovePlanMsg_t *) pK_MSG_DATA(psMsg);
	ASSERT((removeStaticPlanMsgPtr->groupId ==HE_GROUP_MANAGER_DL_GID )|| (removeStaticPlanMsgPtr->groupId ==HE_GROUP_MANAGER_UL_GID )); //Static plan could be only DL DATA or UL DATA 
	heGroupManagerDb[removeStaticPlanMsgPtr->groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_LA_FREE_RESOURCES_STATE;
	laMessagePtr = OSAL_GET_MESSAGE(sizeof(LaRemoveHeGroupParams_t));
	laRemoveParamsPtr = (LaRemoveHeGroupParams_t *)pK_MSG_DATA(laMessagePtr);
	laRemoveParamsPtr->groupId = removeStaticPlanMsgPtr->groupId; 
	planPtr = heGroupManagerDb[removeStaticPlanMsgPtr->groupId].planPtr; 
	laRemoveParamsPtr->planIndex = planPtr->planIndex; 
	// send HE remove message to link adaptation  
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_REMOVE_HE_GROUP, TASK_LINK_ADAPTATION, laMessagePtr, psMsg->header.vapId);
}

/**********************************************************************************

planManagerCreatePlanLaCfm 

	TBD : 
	1. send confirmation message to FM  

	
**********************************************************************************/
static void planManagerCreatePlanLaCfm(K_MSG* psMsg)
{
	LaAddHeGroupCfm_t* 		laAddHeGroupCfmParameters; 
	CreatePlanCfmMsg_t*		gmCreateCfmPtr; 
	K_MSG* 					psNewMsg;

	laAddHeGroupCfmParameters = (LaAddHeGroupCfm_t *)pK_MSG_DATA(psMsg); 

	ILOG0_D("planManagerCreatePlanLaCfm , status =0x%x",laAddHeGroupCfmParameters->status); 
	ASSERT((heGroupManagerDb[laAddHeGroupCfmParameters->groupId].planManagerState == PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_STATE)||
	(heGroupManagerDb[laAddHeGroupCfmParameters->groupId].planManagerState == PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_FOR_PHASE_ADDITION_STATE));

	heGroupManagerDb[laAddHeGroupCfmParameters->groupId].planManagerState = PLAN_MANAGER_READY_STATE;


	ASSERT(laAddHeGroupCfmParameters->status  == LA_ADD_HE_GROUP_SUCCEEDED);

	psNewMsg = OSAL_GET_MESSAGE(sizeof(CreatePlanCfmMsg_t)); 
	gmCreateCfmPtr = (CreatePlanCfmMsg_t *)pK_MSG_DATA(psNewMsg); 
	gmCreateCfmPtr->isStatic = laAddHeGroupCfmParameters->isStaticGroup;
	gmCreateCfmPtr->groupId = laAddHeGroupCfmParameters->groupId; 
	gmCreateCfmPtr->status = (laAddHeGroupCfmParameters->status == LA_ADD_HE_GROUP_SUCCEEDED) ? CREATE_PLAN_SUCCEEDED : CREATE_PLAN_FAILED; 	
	//send confirmation message to the group manager 
	OSAL_SEND_MESSAGE(HE_GROUP_MANAGER_CREATE_PLAN_CFM, TASK_HE_GROUP_MANAGER, psNewMsg, psNewMsg->header.vapId);
}

/**********************************************************************************

planManagerRemovePlanLaCfm 

	TBD : 
	1. send confirmation message to FM  

	
**********************************************************************************/
static void planManagerRemovePlanLaCfm(K_MSG* psMsg)
{
	LaRemoveHeGroupParams_t* 	pMuPlanRemove;
	CommonPlan_t* 				planPtr = NULL;
	DataPhase_t*				dataPhasePtr;
	RemovePlanMsg_t*			gmRemoveCfmPtr; 
	K_MSG* 						psNewMsg;
	uint8 						phaseIndex = 0;
	
	ILOG0_V("planManagerRemovePlanLaCfm"); 
	pMuPlanRemove = (LaRemoveHeGroupParams_t*)pK_MSG_DATA(psMsg);

#ifdef ENET_INC_ARCH_WAVE600B
	if(heGroupManagerDb[pMuPlanRemove->groupId].planManagerState == PLAN_MANAGER_WAIT_FOR_LA_REMOVE_FOR_PHASE_ADDITION_STATE)
	{
		/* In case we removed the plan from LA - due to second phase addition */
		planManagerAddPhaseToStaticPlanPostLaRemove(heGroupManagerDb[pMuPlanRemove->groupId].pSavedMsgforMixPlanLock, pMuPlanRemove->groupId); 
	}	
	else
#endif 
	{
		planPtr = heGroupManagerDb[pMuPlanRemove->groupId].planPtr; 

		planManagerReturnPhaseToPool((void*)(planPtr->phase0Full32BitPointer), PHASE_FORMAT_PROTECTION);
		ILOG0_D ("planManagerRemovePlanLaCfm, 0X%X", planPtr->phase0Full32BitPointer);
		phaseIndex++;
		planManagerReturnPhaseToPool((void*)(planPtr->phase1Full32BitPointer), PHASE_FORMAT_SOUNDING);
		ILOG0_D ("planManagerRemovePlanLaCfm, 0X%X", planPtr->phase1Full32BitPointer);
		phaseIndex++;
		planManagerReturnPhaseToPool((void*)(planPtr->phase2Full32BitPointer), PHASE_FORMAT_BSRP);
		ILOG0_D ("planManagerRemovePlanLaCfm, 0X%X", planPtr->phase2Full32BitPointer);
		phaseIndex++;
		dataPhasePtr = (DataPhase_t*)(planPtr->phase3Full32BitPointer);  //start from &(planPtr->word5) ,phase address starts from phase3Pointer - phase0,1&2 are for protection&sounding&bsrp 
 		do
		{
			if(planPtr->phaseFirstOccurrence & (1<<phaseIndex))
			{
				//release both the data phase and its correspond mirror phase
				ILOG0_D ("planManagerRemovePlanLaCfm, 0X%X", dataPhasePtr);
				planManagerReturnPhaseToPool((void*)dataPhasePtr,PHASE_FORMAT_DL_DATA);
			}
			dataPhasePtr++; 
			phaseIndex++; 
	    }while (planPtr->phaseValid & (1<<phaseIndex));
		
		// return plan to pool
		planManagerReturnPlanToPool((CommonPlan_t*)planPtr);

		heGroupManagerDb[pMuPlanRemove->groupId].planManagerState = PLAN_MANAGER_IDLE_STATE;

		psNewMsg = OSAL_GET_MESSAGE(sizeof(RemovePlanMsg_t)); 
		gmRemoveCfmPtr = (RemovePlanMsg_t *)pK_MSG_DATA(psNewMsg);
		gmRemoveCfmPtr->groupId = pMuPlanRemove->groupId;
		//send confirmation message to the group manager 
		OSAL_SEND_MESSAGE(HE_GROUP_MANAGER_REMOVE_PLAN_CFM, TASK_HE_GROUP_MANAGER, psNewMsg, psMsg->header.vapId);
	}
}



/**********************************************************************************

planManagerAddPhaseToStaticPlan 

	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600B
static void planManagerAddPhaseToStaticPlan(K_MSG* psMsg)
{
	RemovePlanMsg_t*			removePlanMsgPtr; 
	K_MSG*						laMessagePtr;
	LaRemoveHeGroupParams_t*	laRemoveParamsPtr; 
	CommonPlan_t*				planPtr;

	ILOG0_V("planManagerAddPhaseToStaticPlan");
	removePlanMsgPtr = (RemovePlanMsg_t *)pK_MSG_DATA(psMsg);
	heGroupManagerDb[removePlanMsgPtr->groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_LA_REMOVE_FOR_PHASE_ADDITION_STATE;

	laMessagePtr = OSAL_GET_MESSAGE(sizeof(LaRemoveHeGroupParams_t));
	laRemoveParamsPtr = (LaRemoveHeGroupParams_t *)pK_MSG_DATA(laMessagePtr);
	laRemoveParamsPtr->groupId = removePlanMsgPtr->groupId; 
	planPtr = heGroupManagerDb[removePlanMsgPtr->groupId].planPtr; 
	laRemoveParamsPtr->planIndex = planPtr->planIndex; 

	// send HE remove message to link adaptation  
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_REMOVE_HE_GROUP, TASK_LINK_ADAPTATION, laMessagePtr, psMsg->header.vapId);
}
#endif 

/**********************************************************************************

planManagerAddPhaseToStaticPlanPostLaRemove 

	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600B
static void planManagerAddPhaseToStaticPlanPostLaRemove(K_MSG* psMsg, uint8 groupId)
{
	UMI_STATIC_PLAN_CONFIG* pMuPlanConfig = (UMI_STATIC_PLAN_CONFIG*)pK_MSG_DATA(psMsg); 
	CommonPlan_t*			planPtr;
	DlDataPhase_t*			dataPhasePtr;	
	DlDataPhase_t*			mirrorDataPhasePtr;
	UlDataPhase_t*			ulDataPhasePointer;
	DlDataPhase_t*			dlDataPhasePointer;
	uint32* 				phasePtrInPlanAddress; 
	K_MSG*					laMessagePtr;
	LaAddHeGroupParams_t*	laRequestParams; 
	uint32					validPhaseBitmap; 
	uint8					availablePhaseEntryIndex; 
	

	//get plan ptr in HE group manager DB 
	planPtr = heGroupManagerDb[groupId].planPtr;
	ASSERT(planPtr != NULL); 
	/*Allocate data and it's corespond mirror phase*/
	dataPhasePtr		=	planManagerFetchPhaseFromPool(PHASE_FORMAT_DL_DATA); 
	mirrorDataPhasePtr	=	planManagerFetchPhaseFromPool(PHASE_FORMAT_DL_DATA);	
	ASSERT((dataPhasePtr != NULL) && (mirrorDataPhasePtr != NULL)); 

	dataPhasePtr->common.nextOrMirrorPhasePointer = (uint32)mirrorDataPhasePtr; // Make the data phase and its mirror phase pointing each other
	mirrorDataPhasePtr->common.nextOrMirrorPhasePointer = (uint32)dataPhasePtr; // Make the data phase and its mirror phase pointing each other
	validPhaseBitmap = (uint32)(planPtr->phaseValid);  // get the Valid phase bitmap from plan 
	availablePhaseEntryIndex = Utils_FindFirstSetAndClear(&validPhaseBitmap); //Find the first available entry for new data phse
	availablePhaseEntryIndex++;
	phasePtrInPlanAddress = (void*)((uint32*)&(planPtr->word2)); // point to the first phase address in plan 
	phasePtrInPlanAddress += availablePhaseEntryIndex;  // promote the pointer to the available phase entry we found before 
	*phasePtrInPlanAddress = (uint32)dataPhasePtr;   // Set the new Data phase pointer into the phase entry we got above
 	planPtr->phaseValid |=	0x1 << (availablePhaseEntryIndex); 
	
	if(pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_UL_MIXED) 
	{
		ulDataPhasePointer = (UlDataPhase_t*)dataPhasePtr; 
		/*Fill UL phase common fields */
		planManagerSetStaticUlPhaseCommonFields(&(ulDataPhasePointer->common),pMuPlanConfig);
		/*Fill UL phase TCR A (Basic TF) common fields */
		planManagerSetStaticUlPhaseCommonTcr_A_Fields(&(ulDataPhasePointer->commonTcra),pMuPlanConfig,planPtr, psMsg->header.vapId);
		/*Fill UL phase TCR B (MU BA) common fields */
		planManagerSetStaticUlPhaseCommonTcr_B_Fields(&(ulDataPhasePointer->commonTcrb),pMuPlanConfig,planPtr, psMsg->header.vapId);
		/*Fill UL phase RCR common fields */
		planManagerSetStaticUlPhaseCommonRcrFields(&(ulDataPhasePointer->commonRcr), pMuPlanConfig, psMsg->header.vapId);
		/*	Fill UL phase TCR A (Basic TF) common user fields */
		planManagerSetStaticUlPhaseCommonUserTCR_A_Fields((&(ulDataPhasePointer->commonUserTcra)),pMuPlanConfig);
		/*	Fill UL phase TCR B (MU BA) common user fields */
		planManagerSetStaticUlPhaseCommonUserTCR_B_Fields((&(ulDataPhasePointer->commonUserTcrb)),pMuPlanConfig);
		//Fill TF parameters
		planManagerSetStaticUlPhaseCommonTfParameters(&ulDataPhasePointer->tfParameters, &ulDataPhasePointer->tfCommonInfo, pMuPlanConfig);
		//Fill Data TCR per User parameters
		planManagerSetStaticUlPhasePerUser(&(ulDataPhasePointer->userPhase[0]),pMuPlanConfig);
		//Fill TF per User	parameters
		planManagerSetStaticUlPhaseTfInfoPerUser(&(ulDataPhasePointer->tfUserInfo[0]),pMuPlanConfig);
	}
	else if(pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_DL_MIXED) 
	{
		dlDataPhasePointer = (DlDataPhase_t*)dataPhasePtr; 
		/*Fill DL phase common fields */
		planManagerSetStaticDlPhaseCommonFields(&(dlDataPhasePointer->common),pMuPlanConfig);
		/*Fill DL phase TCR A (data) common fields */
		planManagerSetStaticDlPhaseCommonTcr_A_Fields(&(dlDataPhasePointer->commonTcra),pMuPlanConfig,planPtr, psMsg->header.vapId);
		/*Fill DL phase TCR B (MU BAR) common fields */
		planManagerSetStaticDlPhaseCommonTcr_B_Fields(&(dlDataPhasePointer->commonTcrb),pMuPlanConfig,planPtr, psMsg->header.vapId);
		/*Fill DL phase RCR common fields */
		planManagerSetStaticDlPhaseCommonRcrFields(&(dlDataPhasePointer->commonRcr), pMuPlanConfig,psMsg->header.vapId);
		/*	Fill DL phase TCR B (MU BAR) common user fields */
		planManagerSetStaticDlPhaseCommonUserTCR_B_Fields((&(dlDataPhasePointer->commonUserTcrb)),pMuPlanConfig);
		//Fill TF parameters
		planManagerSetStaticDlPhaseCommonTfParameters(dlDataPhasePointer,pMuPlanConfig);
		//Fill TF MU BAR parameters
		planManagerSetStaticDlPhaseCommonTfMuBar(&(dlDataPhasePointer->tfMuBar),pMuPlanConfig);
		//Fill Data TCR per User parameters
		planManagerSetStaticDlPhasePerUser(&(dlDataPhasePointer->userPhase[0]),pMuPlanConfig, pMuPlanConfig->commonSection.startBwLimit);
		//Fill TF per User	parameters
		planManagerSetStaticDlPhaseTfInfoPerUser(&(dlDataPhasePointer->tfUserInfo[0]),pMuPlanConfig);
		/*Fill DL phase common field Valid Users bitmap */
		planManagerSetStaticDlPhaseCommonFieldValidUser(&(dlDataPhasePointer->common),pMuPlanConfig);
		
	}
	else
	{
		ASSERT(0); 
	}
	// fill parameters from USER configuration / fixed values / supported modes / station capabilities 
	//convert pointer to the phase type accordingly 
	heGroupManagerDb[groupId].planManagerState = PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_FOR_PHASE_ADDITION_STATE; 
	laMessagePtr = OSAL_GET_MESSAGE(sizeof(LaAddHeGroupParams_t));
	laRequestParams = (LaAddHeGroupParams_t*)pK_MSG_DATA(laMessagePtr); 
	laRequestParams->planIndex= planPtr->planIndex; 
	laRequestParams->isStaticGroup = TRUE; 
	laRequestParams->groupId = groupId; 
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_ADD_HE_GROUP,TASK_LINK_ADAPTATION, laMessagePtr, psMsg->header.vapId);
}
#endif 
/*****************************************************************************************

planManagerGetPrimaryChannelIndex 

	Returns TRUE if primary sub-band located on the on the lowest 80Mzh, False otherwise 

	
*****************************************************************************************/
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
static bool planManagerIsPrimarySubBandInLower80(void)
{
	bool 							isPrimaryOnLowest80Channel = FALSE; 
	RegPreAggBw2SubBandMapping_u	bw2SubBandMappingReg;
	
	RegAccess_Read(REG_PRE_AGG_BW_2_SUB_BAND_MAPPING, &bw2SubBandMappingReg.val);
	
	if(bw2SubBandMappingReg.bitFields.bw2SubBandMapP80Mhz & FIRST_4_BITS_MASK) //At least one of the first 4 Bits are on - primary is on lowest 80Mhz
	{
		isPrimaryOnLowest80Channel = TRUE; 
	}
	return isPrimaryOnLowest80Channel; 
}
#endif //TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND



/**********************************************************************************

planManagerSetPlanManagerDefaultSettings  

	
**********************************************************************************/
static void planManagerSetPlanManagerDefaultSettings(void)
{
	/*****************************************/
	/*  DL phase common fields Default values*/
	/****************************************/	
#ifdef ENET_INC_ARCH_WAVE600B
    planManagerDefaultSettings.DlCommonPhase_currentPsduMaxLengthLimit = 6500631;
#else
    planManagerDefaultSettings.DlCommonPhase_currentPsduMaxLengthLimit = 1048575; // for HE MU DL in A0 the phy supports psdu length of 20 bits only, according to jira WLANVLSIIP-3814
#endif
	planManagerDefaultSettings.DlCommonPhase_minimumNumberOfUsersForTransmission = 0;
	planManagerDefaultSettings.DlCommonPhase_oneShotPhase = 0;
	planManagerDefaultSettings.DlCommonPhase_phaseType = PHASE_TYPE_DL_DATA_PHASE;
	planManagerDefaultSettings.DlCommonPhase_partialDuration = 0;
	planManagerDefaultSettings.DlCommonPhase_maxPpduTransmissionTimeLimit = (MAX_PSDU_TX_TIME_LIMIT>>1); 
	planManagerDefaultSettings.DlCommonPhase_keepPlanLock = 0;
	planManagerDefaultSettings.DlCommonPhase_primaryUser = 0 ;
#ifdef HE_MU_DYNAMIC_PLAN_USE_VHT_LIKE_DL_SEQ
	planManagerDefaultSettings.DlCommonPhase_tfParamTriggerType = TF_TYPE_BASIC;
	planManagerDefaultSettings.DlCommonPhase_dlDataTxType = 0x1;//0x2;// '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
	planManagerDefaultSettings.DlCommonPhase_plannedMuBarTf = 0;
	planManagerDefaultSettings.DlCommonPhase_tfMode = 1; // 0 - Broadcast;  1 - Unicast
#else
	planManagerDefaultSettings.DlCommonPhase_tfParamTriggerType = TF_TYPE_MU_BAR;
	planManagerDefaultSettings.DlCommonPhase_dlDataTxType = 0x2;// '00' - DL HE MU immediate;  '01' - DL HE MU non immediate;  '10' - DL HE MU non immediate MU-BAR 
	planManagerDefaultSettings.DlCommonPhase_plannedMuBarTf = 1;
	planManagerDefaultSettings.DlCommonPhase_tfMode = 0; // 0 - Broadcast;  1 - Unicast
#endif //HE_MU_DYNAMIC_PLAN_USE_VHT_LIKE_DL_SEQ
	planManagerDefaultSettings.DlCommonPhase_moveStaToSu = 0;
	planManagerDefaultSettings.DlCommonPhase_plannedData= 1; // bit 4 in action bitmap field 
	planManagerDefaultSettings.DlCommonPhase_primaryImmediateBa = 0; //  relevant for vht like mode ACK policy for primary user
	/****************************************/
	/*  DL phase TCR A fields Default values*/
	/***************************************/	
	planManagerDefaultSettings.DlCommonPhaseTcrA_txLoopMode = TX_LOOP_MODE_CLOSED_LOOP; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_phyMode = PHY_MODE_11AX_MU_DL; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_hePuncturingMap = 0XFF; //NA 
	planManagerDefaultSettings.DlCommonPhaseTcrA_customBf = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_ulDl = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_txopPs = 1; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_expectedRcr = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_bwChange = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_txopDurationFromMac = 1; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_aggregate = 1; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_dynamicBw = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_scp = 0;//don't care as phy mode is HE  (this field is relevant for HT/VHT)
	planManagerDefaultSettings.DlCommonPhaseTcrA_notSounding = 1; //notSounding should be enabled when DATA PHASE
	planManagerDefaultSettings.DlCommonPhaseTcrA_spatialReuse= 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrA_macDuration = MAC_DURATION;  //for HE only
#ifdef HE_MU_DYNAMIC_PLAN_USE_VHT_LIKE_DL_SEQ
	planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataLdpc = 0;
#else
	planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataLdpc = 1;
#endif //HE_MU_DYNAMIC_PLAN_USE_VHT_LIKE_DL_SEQ

	planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataPktExt = 0;
	planManagerDefaultSettings.DlCommonPhaseTcrA_triggerBasedDataPeDisam = 1; 
	/*****************************************/
	/*  DL phase TCR B fieldsDefault values */
	/****************************************/
	planManagerDefaultSettings.DlCommonPhaseTcrB_txLoopMode = TX_LOOP_MODE_CLOSED_LOOP; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_phyMode = PHY_MODE_11AG; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_hePuncturingMap = 0xFF; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_customBf = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_ulDl = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_txopPs = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_expectedRcr = 1; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_bwChange = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_txopDurationFromMac = 1; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_aggregate = 1; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_dynamicBw = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_scp = 0;//don't care as phy mode is HE  (this field is relevant for HT/VHT)
	planManagerDefaultSettings.DlCommonPhaseTcrB_notSounding = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_spatialReuse = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_macDuration = MAC_DURATION;  //for HE only
	planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataLdpc = 0; // No LDPC for BAR 
	planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataPktExt = 0; 
	planManagerDefaultSettings.DlCommonPhaseTcrB_triggerBasedDataPeDisam = 1; 

	/**********************************/
	/*  DL RCR 	 fields values Default*/
	/**********************************/
	planManagerDefaultSettings.DlCommonPhaseRcr_peDisambiguty = 1; 
	
	/*****************************************************/
	/*  DL phase CommonUserTCR_B_Fields Default	 values */
	/****************************************************/
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_TcrbStaAid = 0; // NA   trigger frame transmitted in legacy phy mode
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_Ldpc = 0; // NA   trigger frame transmitted in legacy phy mode
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_Txbf = 0; // NA   trigger frame transmitted in legacy phy mode								:1; 
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_BfIndex = 0; // NA
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_StartRu = 0; // NA
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_RelativeUserPower = 0; // NA
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_SubBand = 0; // NA
	planManagerDefaultSettings.DlCommonPhaseBrdcstUserTcrB_RuSize = 0; // NA

	/******************************************************/
	/*  DL phase Common TF fields Default values		 */
	/*****************************************************/

	planManagerDefaultSettings.DlCommonTf_SpatialReuse = 0x0; 
#ifdef HE_MU_DYNAMIC_PLAN_USE_VHT_LIKE_DL_SEQ
	planManagerDefaultSettings.DlCommonTf_tfTriggerType = TF_TYPE_BASIC; 
#else 
	planManagerDefaultSettings.DlCommonTf_tfTriggerType = TF_TYPE_MU_BAR;
#endif  
	planManagerDefaultSettings.DlCommonTf_CommonInfoSwOverrideEnable = 0; // only in case we want to override the SW info with a configured value in TF generator register 
	planManagerDefaultSettings.DlCommonTf_UserInfoSwOverrideEnable = 0; // only in case we want to override the SW info with a configured value in TF generator register 
#ifdef ENET_INC_ARCH_WAVE600D2					
	planManagerDefaultSettings.DlCommonTf_tfMoreTf= 0; 
#else
	planManagerDefaultSettings.DlCommonTf_CascadeIndication = 0; 
#endif
    planManagerDefaultSettings.DlCommonTf_tfLength = 310;
	planManagerDefaultSettings.DlCommonTf_CsRequired = 0 ; // should a station consider CCA before UL 
	planManagerDefaultSettings.DlCommonTf_MuMimoLtfMode = 0; //Single Stream Pilots   1 : Mask on each spatial stream of the LTF sequence by a distinct orthogonal code
	planManagerDefaultSettings.DlCommonTf_Doppler = 0; 
																
	planManagerDefaultSettings.DlCommonTf_tfLsigLengthOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfBwOverride = 1;
	planManagerDefaultSettings.DlCommonTf_tfRuAllocationOverride = 1; 
	planManagerDefaultSettings.DlCommonTf_tfMcsOverride = 1;
	planManagerDefaultSettings.DlCommonTf_tfDcmOverride = 1;
	planManagerDefaultSettings.DlCommonTf_tfNssOverride = 1;
	planManagerDefaultSettings.DlCommonTf_tfTargetRssiOverride = 1;
	planManagerDefaultSettings.DlCommonTf_tfTidAggrLimitOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfPreferredAcOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfMultiTidOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfTidInfoOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfCompressedFnOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfCompressedSsnOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfMultiTidTidOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfMultiTidFnOverride = 0;
	planManagerDefaultSettings.DlCommonTf_tfMultiTidSsnOverride = 0;

#ifdef ENET_INC_ARCH_WAVE600D2					
	planManagerDefaultSettings.tfHeSigA2Reserved = 0;   //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#else
	planManagerDefaultSettings.tfHeSigAReserved = 0;   //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#endif	
	planManagerDefaultSettings.DlCommonTf_tfBarAckPolicy = 0; 
	planManagerDefaultSettings.DlCommonTf_tfMultiTid= 0; 
	planManagerDefaultSettings.DlCommonTf_CompressedBitmap = 1; 
	planManagerDefaultSettings.DlCommonTf_tfGcr = 0; 
	planManagerDefaultSettings.DlCommonTf_CompressedFn0 = 0; 
	planManagerDefaultSettings.DlCommonTf_CompressedFn1 = 0; 
	planManagerDefaultSettings.DlCommonTf_CompressedFn2 = 0; 
	planManagerDefaultSettings.DlCommonTf_CompressedFn3 = 0; 
	planManagerDefaultSettings.DlCommonTf_MultiTidMuBarInformation = 0; 
	planManagerDefaultSettings.DlCommonTf_MultiTidReserved = 0; //was 0xFFF and changed to 0x0 according to D.B.
	planManagerDefaultSettings.DlCommonTf_MultiTidTid = 0; 
	planManagerDefaultSettings.DlCommonTf_tfMultiTidFn0 = 0; 
	planManagerDefaultSettings.DlCommonTf_tfMultiTidFn1 = 0; 
	planManagerDefaultSettings.DlCommonTf_tfMultiTidFn2 = 0; 
	planManagerDefaultSettings.DlCommonTf_tfMultiTidFn3 = 0; 
	planManagerDefaultSettings.DlCommonTf_MultiTidStartingSequenceNumber = 0; 
	planManagerDefaultSettings.DlCommonTf_CompressedStartingSequenceNumber = 0; 
	planManagerDefaultSettings.DlCommonTf_ReservedBarControl = 0x0; 


	planManagerDefaultSettings.DlPerUser_numTfRepetition = 0x0; //means no repetition - TF would be tranmitted once   
	planManagerDefaultSettings.DlPerUser_tidAllocBitmap = TID_BITMAP_ALL_DATA_TIDS; // dynamic grouping debug - was TID_BITMAP_ALL_TIDS; 
	planManagerDefaultSettings.DlPerUser_tidSelectionPolicy = 0x2;  //All combinations 
	planManagerDefaultSettings.DlPerUser_tid0RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid1RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid2RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid3RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid4RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid5RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid6RrWeight = 0x0; //NA 
	planManagerDefaultSettings.DlPerUser_tid7RrWeight = 0x0; //NA 
	
	planManagerDefaultSettings.DlPerUser_heMngAgg = 0; // management in agg 
	planManagerDefaultSettings.DlPerUser_heBarAgg = 0;
	planManagerDefaultSettings.DlPerUser_aControlIncluded = 0; // A-Control (HT) Included  0  not included 1  included 
	planManagerDefaultSettings.DlPerUser_userRcrTxbf = 0; //NO UL BF 
	planManagerDefaultSettings.DlPerUser_RcrValid = 1; 
	planManagerDefaultSettings.DlPerUser_TcraTxbf = 0;
	planManagerDefaultSettings.DlPerUser_TcraRelativeUserPower = 0; 
	planManagerDefaultSettings.DlPerUser_TcraBfIndex = 0; 

	planManagerDefaultSettings.DlPerUserTf_RuAllocation = 0x7a;//;//67; //override by the RCR value   // // Match to StartRU - 0 , RU size - 3 , Sub-Band -0 
	planManagerDefaultSettings.DlPerUserTf_Dcm = 0; 
	planManagerDefaultSettings.DlPerUserTf_TidAggLimit = 0x0; //override by the RCR value 
	planManagerDefaultSettings.DlPerUserTf_BasicTrigger = 0x0; //override by the RCR value 
	planManagerDefaultSettings.DlPerUserTf_PreferredAc = 0;  //override by the RCR value 
	planManagerDefaultSettings.DlPerUserTf_BfrpTriggerDependent = 0xFF; //NA make it transparent to the receiving STA

	/*****************************************/
	/*  UL phase common fields Default values*/
	/****************************************/	
	planManagerDefaultSettings.UlCommon_minNumOfUsersForTransmission = 1; 
	planManagerDefaultSettings.UlCommon_oneShotPhase = 0;
	planManagerDefaultSettings.UlCommon_tfParamTriggerType = TF_TYPE_BASIC;  
	planManagerDefaultSettings.UlCommon_moveStaToSu = 0; 
	planManagerDefaultSettings.UlCommon_plannedUlTf = 1; 
	planManagerDefaultSettings.UlCommon_ulDataSeqType = 0; 
#ifdef ENET_INC_ARCH_WAVE600D2
	planManagerDefaultSettings.UlCommon_tfMode = TF_BA_MODE_BROADCAST; 	// TfBaMode_e: 0 - BC
	planManagerDefaultSettings.UlCommon_baMode = TF_BA_MODE_BROADCAST; 	// TfBaMode_e: 0 - BC
#else	
	planManagerDefaultSettings.UlCommon_tfBroadcastUnicastMode = 0; 	//0 - BC mode ,1- Unicast mode
	planManagerDefaultSettings.UlCommon_baBroadcastUnicastMode = 0 ; 	//0 - BC mode ,1- Unicast mode
#endif	
	planManagerDefaultSettings.UlCommon_keepPlanLock = 0;
	planManagerDefaultSettings.UlCommon_phaseType = PHASE_TYPE_UL_DATA_PHASE;
	planManagerDefaultSettings.UlCommon_partialDuration = 0x0;  
	/****************************************/
	/*  UL phase TCR A fields Default values*/
	/***************************************/	
	planManagerDefaultSettings.UlCommonPhaseTcrA_macDuration = MAC_DURATION;  //for HE only
	planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataPktExt = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_txLoopMode = TX_LOOP_MODE_CLOSED_LOOP; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_ulDl = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_txopPs = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_expectedRcr = 1; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_bwChange = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_txopDurationFromMac = 1; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_aggregate = 1; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_dynamicBw = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_scp = 0; //don't care as phy mode is HE  (this field is relevant for HT/VHT)
	planManagerDefaultSettings.UlCommonPhaseTcrA_notSounding = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_spatialReuse = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataLdpc = 1; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataPeDisam = 1; //NA
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_StaAid =0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Ldpc = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_BfIndex = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_StartRu = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RelativeUserPower  =0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RuSize = 0 ;
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Txbf = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_SubBand = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_phyMode = PHY_MODE_11AG; 
	planManagerDefaultSettings.UlCommonPhaseTcrA_hePuncturingMap = 0xFF; 

	/****************************************/
	/*  UL phase TCR B fields Default values*/
	/***************************************/	
	planManagerDefaultSettings.UlCommonPhaseTcrB_txLoopMode = TX_LOOP_MODE_CLOSED_LOOP; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_hePuncturingMap = 0xFF; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_TcrbStaAid = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_BfIndex = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_StartRu = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RelativeUserPower =0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_SubBand = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_ulDl = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_txopPs = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RuSize = 0x7; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_phyMode = PHY_MODE_11AG; //If BA tranmitted in Broadcast then phy mode is PHY_MODE_11AG , otherwise - if unicast phy mode is PHY_MODE_11AX_MU_DL
	planManagerDefaultSettings.UlCommonPhaseTcrB_expectedRcr = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_bwChange = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_txopDurationFromMac = 1; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_Ldpc = 0; 
	planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_Txbf = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_aggregate = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_dynamicBw = 0 ;
	planManagerDefaultSettings.UlCommonPhaseTcrB_scp = 0; //relevant only if PHY mode is legacy 
	planManagerDefaultSettings.UlCommonPhaseTcrB_notSounding = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_spatialReuse = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataLdpc = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataPktExt = 0; 
	planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataPeDisam = 0;  //In BA we don't turn on this  bit 
	planManagerDefaultSettings.UlCommonPhaseTcrB_macDuration = MAC_DURATION; //for HE only 

	/********************************************/
	/*  UL phase common TF fields Default values*/
	/*******************************************/	

	planManagerDefaultSettings.UlTf_LsigLengthOverride = 0;
	planManagerDefaultSettings.UlTf_BW_Override = 1;
	planManagerDefaultSettings.UlTf_RU_Allocation_Override = 1; 
	planManagerDefaultSettings.UlTf_MCS_Override = 1;
	planManagerDefaultSettings.UlTf_NSS_Override = 1;
	planManagerDefaultSettings.UlTf_DcmOverride  = 1; 
	planManagerDefaultSettings.UlTf_Target_RSSI_Override = 0;
	planManagerDefaultSettings.UlTf_TID_Aggr_Limit_Override = 1;
	planManagerDefaultSettings.UlTf_Preferred_AC_Override = 0;
	planManagerDefaultSettings.UlTf_Multi_TID_Override = 0;
	planManagerDefaultSettings.UlTf_TID_Info_Override = 0;
	planManagerDefaultSettings.UlTf_Compressed_FN_Override = 0;
	planManagerDefaultSettings.UlTf_Compressed_SSN_Override = 0;
	planManagerDefaultSettings.UlTf_Multi_TID_TID_Override = 0;
	planManagerDefaultSettings.UlTf_Multi_TID_FN_Override = 0;
	planManagerDefaultSettings.UlTf_Multi_TID_SSN_Override = 0;
	
	planManagerDefaultSettings.UlCommonTf_tfTriggerType = TF_TYPE_BASIC; 
	planManagerDefaultSettings.UlCommonTf_UserInfoSwOverrideEnable = 0; 
	planManagerDefaultSettings.UlCommonTf_CommonInfoSwOverrideEnable = 0; 
#ifdef ENET_INC_ARCH_WAVE600D2						
	planManagerDefaultSettings.UlCommonTf_tfMoreTf = 0;  
#else
	planManagerDefaultSettings.UlCommonTf_CascadeIndication = 0;   
#endif
	planManagerDefaultSettings.UlCommonTf_CsRequired  = 1; // Changed from 0 to 1 due to plugfest requirement 
	planManagerDefaultSettings.UlCommonTf_MuMimoLtfMode = 0; 
	planManagerDefaultSettings.UlCommonTf_SpatialReuse = 0; 
	planManagerDefaultSettings.UlCommonTf_MultiTidStartingSequenceNumber = 0; 
	planManagerDefaultSettings.UlCommonTf_Doppler = 0; 
	planManagerDefaultSettings.UlCommonTf_MultiTidTid = 0; 
	planManagerDefaultSettings.UlCommonTf_tfMultiTidFn0 = 0; 
	planManagerDefaultSettings.UlCommonTf_tfMultiTidFn1 = 0; 
	planManagerDefaultSettings.UlCommonTf_tfMultiTidFn2 = 0; 
	planManagerDefaultSettings.UlCommonTf_tfMultiTidFn3 = 0;
    planManagerDefaultSettings.UlCommonTf_tfLength = 2914; // dynamic grouping debug TBD
	planManagerDefaultSettings.UlPerUserTf_numTfRepetition = 1; 
	planManagerDefaultSettings.UlPerUser_tidAllocBitmap = TID_BITMAP_ALL_TIDS; 
	planManagerDefaultSettings.UlPerUserTf_maxTids = 0; //should call TsManager_TsManagerRecipientGetMaxUlTids(userStationIndex);
	planManagerDefaultSettings.UlPerUserTf_RcrTxbf = 0;  // UL BF is not beamformed
	planManagerDefaultSettings.UlPerUserTf_RcrValid = 1; 
	planManagerDefaultSettings.UlPerUserTf_TcraTxbf = 0;
	planManagerDefaultSettings.UlPerUserTf_TcraBfIndex = 0;
	planManagerDefaultSettings.UlPerUserTf_TcrbRelativeUserPower = 0;
	planManagerDefaultSettings.UlPerUserTf_TcrbLdpc = 0; 
	planManagerDefaultSettings.UlPerUserTf_TcrbTxbf = 0;

	planManagerDefaultSettings.UlPerUserTf_RuAllocation = 0x7A; // Match to StartRU - 0 , RU size - 3 , Sub-Band -0 

	planManagerDefaultSettings.UlPerUserTf_Dcm = 0; 
	planManagerDefaultSettings.UlPerUserTf_TidAggLimit0 = 0;
	planManagerDefaultSettings.UlPerUserTf_preferredAc0 = 0; 
	planManagerDefaultSettings.UlPerUserTf_BfrpTriggerDependent0 = 0xFF; //NA make it transparent to the receiving STA
	
	planManagerDefaultSettings.planMaxTxopDur = PLAN_MAX_TXOP_DURATION;
	planManagerDefaultSettings.planDurationUpdateMode = DURATION_UPDATE_MODE_FULL_TXOP;
	planManagerDefaultSettings.ulTfPadding = 0x2; 
	planManagerDefaultSettings.tfMpduMuSpacingFactor = 1; // No spacing factor 
	planManagerDefaultSettings.Dl_lSigLength =  LSIG_LENGHT_FOR_8_TIDS_64_MU_BA_SIZE;
	planManagerDefaultSettings.Ul_lSigLength =  LSIG_LENGHT_FOR_UL;
	planManagerDefaultSettings.tfGiAndLtfType = GI_AND_LTF_TYPE_2X_LTF_1_6_GI;
	planManagerDefaultSettings.currMaxMsduAtAmsdu = 0x7;

	/*protection phase default values*/
	planManagerDefaultSettings.ProtectionPhasePlannedMuRts 		= 0x0; //only one of the planned protection bits shoudl be set
	planManagerDefaultSettings.ProtectionPhasePlannedRtsCts		= 0x0; 
	planManagerDefaultSettings.ProtectionPhasePlannedCts2Self	= 0x1; 
	planManagerDefaultSettings.ProtectionPhaseFillerType = 0x2; 
	planManagerDefaultSettings.ProtectionPhaseKeepPlanLock = 0x0; 
	planManagerDefaultSettings.ProtectionMinNumOfUsersForTx = 0x1; 
	planManagerDefaultSettings.ProtectionPhaseMoveStaToSu = 0x1; 

#ifdef ENET_INC_ARCH_WAVE600D2	
	planManagerDefaultSettings.NfrpCommonPhase_ValidUsers310 = 0x1;	
	planManagerDefaultSettings.NfrpCommonPhase_MinimumNumberOfUsersForTransmission = 0x0;	
	planManagerDefaultSettings.NfrpCommonPhase_TfinfoBaseAddrDWoffset = CONVERT_BYTES_TO_WORDS(OFFSETOF(NfrpPhase_t, tfParameters)); // offset from phase start to tfParameters

	
	planManagerDefaultSettings.NfrpCommonPhaseRcr_nHeltf = 0x1;
	planManagerDefaultSettings.NfrpCommonPhaseRcr_TBmode = TB_MODE_NDP_FEEDBACK;
	planManagerDefaultSettings.NfrpCommonPhaseRcr_heCp 	= HE_CP_HE_3_2_CP;											
	planManagerDefaultSettings.NfrpCommonPhaseRcr_heLtf = HE_LTF_HE_4X_LTF;													
	planManagerDefaultSettings.NfrpCommonPhaseRcr_stbc  = 0x0;												
	planManagerDefaultSettings.NfrpCommonPhaseRcr_lSigLength = LSIG_LENGHT_FOR_NFRP;											


#endif
}


static void planManagerSetStaticNssPerUser(UMI_STATIC_PLAN_CONFIG* 	pMuPlanConfig, LaAddHeGroupParams_t* laRequestParams, uint8 vapId)
{
	uint8 loopIndex = 0;
	uint8 maxNss[MAX_NUM_OF_USERS_IN_MIMO_GROUP] = {0};
	uint8 maxNssUserId = INVALID_MU_USP_INDEX;
	uint8 numOfAntennas = heGroupManager_GetNumOfActiveAntennas(vapId);
	uint8 ratePerUsp = 0;
	
	DEBUG_ASSERT(pMuPlanConfig->commonSection.numOfParticipatingStations <= numOfAntennas);

	ILOG0_DD("[planManagerSetStaticNssPerUser], numOfAntennas %d numOfSta %d", numOfAntennas, pMuPlanConfig->commonSection.numOfParticipatingStations);
	if (pMuPlanConfig->commonSection.numOfParticipatingStations == numOfAntennas)
	{
		/* 1 NSS for each user */
		memset(laRequestParams->maxNss, SPATIAL_STREAM_1, MAX_NUM_OF_USERS_IN_MIMO_GROUP); 
	}
	else
	{
		for(loopIndex = 0; loopIndex < pMuPlanConfig->commonSection.numOfParticipatingStations; loopIndex++)
		{
			if (pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_DL_DATA) // DL MIMO
			{
				ratePerUsp = pMuPlanConfig->perUserParameters[loopIndex].dl_PsduRatePerUsp;

			}
			else // UL MIMO
			{
				ratePerUsp = pMuPlanConfig->perUserParameters[loopIndex].ul_PsduRatePerUsp;
			}
			/* maxNss is the minimum between STA capabilty and static plan requested rate */
			maxNss[loopIndex] = MIN((StaDbHwEntries[pMuPlanConfig->perUserParameters[loopIndex].uspStationIndexes].common.maxNssTx), EXTRACT_NSS_FROM_VHT_HE_RATE(ratePerUsp));
			ILOG0_DD("maxNss[%d] = %d", loopIndex, maxNss[loopIndex]);
		}
		
		switch (numOfAntennas)
		{
			case 4: 		
				if (pMuPlanConfig->commonSection.numOfParticipatingStations == 3)
				{
					/* 4 antennas, 3 STAs */
					for(loopIndex = 0; loopIndex < pMuPlanConfig->commonSection.numOfParticipatingStations; loopIndex++)
					{
						if (maxNss[loopIndex] > SPATIAL_STREAM_1)
						{
							maxNssUserId = loopIndex;
							break;
						}
					}
					for(loopIndex = 0; loopIndex < pMuPlanConfig->commonSection.numOfParticipatingStations; loopIndex++)
					{
						if (loopIndex != maxNssUserId)
						{
							laRequestParams->maxNss[loopIndex] = SPATIAL_STREAM_1;
				
						}
					}
					if (maxNssUserId != INVALID_MU_USP_INDEX)
					{
						laRequestParams->maxNss[maxNssUserId] = SPATIAL_STREAM_2;
					}
				}
				else if (pMuPlanConfig->commonSection.numOfParticipatingStations == 2)
				{		
					/* 4 antennas, 2 STAs */
					if (maxNss[0] == SPATIAL_STREAM_1)
					{
						laRequestParams->maxNss[0] = SPATIAL_STREAM_1;
						laRequestParams->maxNss[1] = MIN(SPATIAL_STREAM_3, maxNss[1]);	
					}
					else if (maxNss[0] == SPATIAL_STREAM_2)
					{
						laRequestParams->maxNss[0] = SPATIAL_STREAM_2;
						laRequestParams->maxNss[1] = MIN(SPATIAL_STREAM_2, maxNss[1]);	
					}
					else if (maxNss[0] == SPATIAL_STREAM_3)
					{
						laRequestParams->maxNss[0] = SPATIAL_STREAM_3;
						laRequestParams->maxNss[1] = SPATIAL_STREAM_1;	
					}	
				}
				else
				{
					/* 4 antennas, 1 STA */
					laRequestParams->maxNss[0] = MIN(maxNss[0], SPATIAL_STREAM_4);
				}
				break;
			case 3: 
				/* 3 antennas, 2 STAs */
				if (pMuPlanConfig->commonSection.numOfParticipatingStations == 2)		
				{		
					if (maxNss[0] == SPATIAL_STREAM_1)
					{
						laRequestParams->maxNss[0] = SPATIAL_STREAM_1;
						laRequestParams->maxNss[1] = MIN(SPATIAL_STREAM_2, maxNss[1]);	
					}
					else
					{
						laRequestParams->maxNss[0] = SPATIAL_STREAM_2;
						laRequestParams->maxNss[1] = SPATIAL_STREAM_1;	
					}
				}
				else  
				{
					/* 3 antennas, 1 STA */
					laRequestParams->maxNss[0] = MIN(maxNss[0], SPATIAL_STREAM_3);
				}
				break;
			case 2:  
				/* 2 antennas, 1 STA */
				laRequestParams->maxNss[0] = MIN(maxNss[0], SPATIAL_STREAM_2);
				break;
		}
	}
	SLOG0(0, 0, LaAddHeGroupParams_t, laRequestParams);
}


#ifdef OFDMA_DYNAMIC_PLAN_ON
/**********************************************************************************
planManagerSetNssPerUser  


Set the NSS per user for LA add HW group request 
	
**********************************************************************************/
static void planManagerSetNssPerUser(LaAddHeGroupParams_t* laAddHeGroupParameters, uint8 groupId)
{
    uint8 loopIndex = 0;
	uint8 maxNss[MAX_NUM_OF_USERS_IN_MIMO_GROUP] = {0};
	uint8 maxNssUserId = INVALID_MU_USP_INDEX;
	uint8 numOfAntennas = heGroupManager_GetNumOfActiveAntennas(heGroupManagerDb[groupId].vapId);
    uint8 numOfStaInGroup = heGroupManagerDb[groupId].numOfStationsInGroup;
	
	if((heGroupManagerDb[groupId].formationType == DL_OFDMA) || (heGroupManagerDb[groupId].formationType == UL_OFDMA) ) 
	{
        return;
	}
	ASSERT(numOfStaInGroup <= numOfAntennas);
    ASSERT(heGroupManagerDb[groupId].numOfStationsInGroup <= MAX_NUM_OF_USERS_IN_MIMO_GROUP); 
	ILOG0_DD("[planManagerSetStaticNssPerUser], numOfAntennas %d numOfSta %d", numOfAntennas, numOfStaInGroup);
    
    /* will be changed in next phases according to dynamic grouping algorithm */
	if (numOfStaInGroup == numOfAntennas)
	{
		/* 1 NSS for each user */
		memset(laAddHeGroupParameters->maxNss, SPATIAL_STREAM_1, MAX_NUM_OF_USERS_IN_MIMO_GROUP); 
	}
	else
	{
		for(loopIndex = 0; loopIndex < numOfStaInGroup; loopIndex++)
		{
			
			maxNss[loopIndex] = StaDbHwEntries[heGroupManagerDb[groupId].members[loopIndex]].common.maxNssTx;
			ILOG0_DD("maxNss[%d] = %d", loopIndex, maxNss[loopIndex]);
		}
		
		switch (numOfAntennas)
		{
			case 4: 		
				if (numOfStaInGroup == 3)
				{
					/* 4 antennas, 3 STAs */
					for(loopIndex = 0; loopIndex < numOfStaInGroup; loopIndex++)
					{
						if (maxNss[loopIndex] > SPATIAL_STREAM_1)
						{
							maxNssUserId = loopIndex;
							break;
						}
					}
					for(loopIndex = 0; loopIndex < numOfStaInGroup; loopIndex++)
					{
						if (loopIndex != maxNssUserId)
						{
							laAddHeGroupParameters->maxNss[loopIndex] = SPATIAL_STREAM_1;
						}
					}
					if (maxNssUserId != INVALID_MU_USP_INDEX)
					{
						laAddHeGroupParameters->maxNss[maxNssUserId] = SPATIAL_STREAM_2;
					}
				}
				else if (numOfStaInGroup == 2)
				{		
					/* 4 antennas, 2 STAs */
					if (maxNss[0] == SPATIAL_STREAM_1)
					{
						laAddHeGroupParameters->maxNss[0] = SPATIAL_STREAM_1;
						laAddHeGroupParameters->maxNss[1] = MIN(SPATIAL_STREAM_3, maxNss[1]);	
					}
					else if (maxNss[0] == SPATIAL_STREAM_2)
					{
						laAddHeGroupParameters->maxNss[0] = SPATIAL_STREAM_2;
						laAddHeGroupParameters->maxNss[1] = MIN(SPATIAL_STREAM_2, maxNss[1]);	
					}
					else if (maxNss[0] == SPATIAL_STREAM_3)
					{
						laAddHeGroupParameters->maxNss[0] = SPATIAL_STREAM_3;
						laAddHeGroupParameters->maxNss[1] = SPATIAL_STREAM_1;	
					}	
				}
				else
				{
					/* 4 antennas, 1 STA */
					laAddHeGroupParameters->maxNss[0] = MIN(maxNss[0], SPATIAL_STREAM_4);
				}
				break;
			case 3: 
				/* 3 antennas, 2 STAs */
				if (numOfStaInGroup == 2)		
				{		
					if (maxNss[0] == SPATIAL_STREAM_1)
					{
						laAddHeGroupParameters->maxNss[0] = SPATIAL_STREAM_1;
						laAddHeGroupParameters->maxNss[1] = MIN(SPATIAL_STREAM_2, maxNss[1]);	
					}
					else
					{
						laAddHeGroupParameters->maxNss[0] = SPATIAL_STREAM_2;
						laAddHeGroupParameters->maxNss[1] = SPATIAL_STREAM_1;	
					}
				}
				else  
				{
					/* 3 antennas, 1 STA */
					laAddHeGroupParameters->maxNss[0] = MIN(maxNss[0], SPATIAL_STREAM_3);
				}
				break;
			case 2:  
				/* 2 antennas, 1 STA */
				laAddHeGroupParameters->maxNss[0] = MIN(maxNss[0], SPATIAL_STREAM_2);
				break;
		}
	}
	SLOG0(0, 0, LaAddHeGroupParams_t, laAddHeGroupParameters);
}


/**********************************************************************************

planManagerSetUlPhase

Description: 
Sets UL phase parameters which can be either UL or BSRP
	
**********************************************************************************/
static void planManagerSetUlPhase(UlDataPhase_t* ulDataPhasePointer, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId, RuAllocationPerUser_t* ruAllocation, HeMuSequence_e muSequenceType)
{
    /*Set UL phase common fields */
    planManagerSetUlPhaseCommonFields(&(ulDataPhasePointer->common), pMuCreatePlanReq, muSequenceType);
    /*Set UL phase TCR A (Basic TF) common fields */
    planManagerSetUlPhaseCommonTcr_A_Fields(&(ulDataPhasePointer->commonTcra), pMuCreatePlanReq, planPtr, vapId);
    /*Set UL phase TCR B (MU BA) common fields */
    planManagerSetUlPhaseCommonTcr_B_Fields(&(ulDataPhasePointer->commonTcrb), pMuCreatePlanReq, planPtr, vapId);
    /*Set UL phase RCR common fields */
    planManagerSetUlPhaseCommonRcrFields(&(ulDataPhasePointer->commonRcr), pMuCreatePlanReq, vapId);
    /*  Set UL phase TCR A (Basic TF) common user fields */
    planManagerSetUlPhaseCommonUserTCR_A_Fields((&(ulDataPhasePointer->commonUserTcra)));
    /*  Set UL phase TCR B (MU BA) common user fields */
    planManagerSetUlPhaseCommonUserTCR_B_Fields((&(ulDataPhasePointer->commonUserTcrb)));
    //Set TF parameters
    planManagerSetUlPhaseCommonTfParameters(ulDataPhasePointer, pMuCreatePlanReq, muSequenceType);
    //Set Data TCR per User parameters
    planManagerSetUlPhasePerUser(&(ulDataPhasePointer->userPhase[0]), pMuCreatePlanReq, ruAllocation, muSequenceType);
    //Set TF per User   parameters
    planManagerSetUlPhaseTfInfoPerUser(&(ulDataPhasePointer->tfUserInfo[0]), pMuCreatePlanReq, ruAllocation);
}


/**********************************************************************************

planManagerSetUlPhaseCommonFields 

Description: 
Sets the DATA common section in UL pahse 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonFields(UlDataPhaseCommon_t* phaseCommonPtr, CreatePlanMsg_t* pMuCreatePlanReq, HeMuSequence_e muSequenceType)
{
	uint8			loopIndex; 
	uint32			validUsersBitmap = 0x0;

	for(loopIndex = 0; loopIndex < heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup; loopIndex++)
	{
		validUsersBitmap |= 1<<loopIndex; 
	}
	phaseCommonPtr->validUsers310 = validUsersBitmap; 		
	phaseCommonPtr->validUsers3532 = 0;

    switch(muSequenceType)
	{
		case HE_MU_UL:
			phaseCommonPtr->tfParamTriggerType = TF_TYPE_BASIC; 
			break;
		case HE_MU_BSRP:
		case HE_MU_BSRP_UL:
			phaseCommonPtr->tfParamTriggerType = TF_TYPE_BUFFER_STATUS_RPT;
			break;
		default:
			FATAL("[planManagerSetUlPhaseCommonFields], UNKNOWN muSequenceType");
			break;
	}
	phaseCommonPtr->phaseType       = planManagerDefaultSettings.UlCommon_phaseType; //0 - Protection Phase;  1 - Sounding Phase;  2 - DL Data Phase;  3 - UL Phase
	phaseCommonPtr->oneShotPhase    = planManagerDefaultSettings.UlCommon_oneShotPhase; 
	phaseCommonPtr->partialDuration = planManagerDefaultSettings.UlCommon_partialDuration; // FULL TXOP 
	phaseCommonPtr->keepPlanLock    = planManagerDefaultSettings.UlCommon_keepPlanLock; 
	phaseCommonPtr->moveStaToSu     = planManagerDefaultSettings.UlCommon_moveStaToSu; // For training only 
	phaseCommonPtr->plannedUlTf     = planManagerDefaultSettings.UlCommon_plannedUlTf;  // bit 4 in action bitmap field 
#ifdef ENET_INC_ARCH_WAVE600D2
    phaseCommonPtr->dataMuMimo      = pMuCreatePlanReq->muType;
    if (heGroupManagerDb[pMuCreatePlanReq->groupId].formationType == UL_MIMO)
    {
        // In UL MU-MIMO, Minimum number of users for transmission should be set to 2 due to PHY limitation
        phaseCommonPtr->minimumNumberOfUsersForTransmission = HE_GROUP_MANAGER_MIN_USERS_FOR_MU_MIMO;
    }
    else
#endif
    {
        phaseCommonPtr->minimumNumberOfUsersForTransmission = planManagerDefaultSettings.UlCommon_minNumOfUsersForTransmission;
    }
	phaseCommonPtr->tfInfoBaseAddrDwOffset = CONVERT_BYTES_TO_WORDS(OFFSETOF(UlDataPhase_t, tfParameters));  // offset from phase start to tfParameters
	phaseCommonPtr->ulDataSeqType          = planManagerDefaultSettings.UlCommon_ulDataSeqType; 
#ifdef ENET_INC_ARCH_WAVE600D2	
	phaseCommonPtr->tfMode                 = planManagerDefaultSettings.UlCommon_tfMode;  //BC
	phaseCommonPtr->baMode                 = planManagerDefaultSettings.UlCommon_baMode;  //BC	
#else	
	phaseCommonPtr->tfBroadcastUnicastMode = planManagerDefaultSettings.UlCommon_tfBroadcastUnicastMode;  //BC 
	phaseCommonPtr->baBroadcastUnicastMode = planManagerDefaultSettings.UlCommon_baBroadcastUnicastMode; // BC 
#endif	

#ifdef ENET_INC_ARCH_WAVE600B	
	// 20MHz Occupancy (Keep User) configuration
	phaseCommonPtr->broadcastTfAidCfg = TRUE; // use original AID from phase
	phaseCommonPtr->unicastTfTypeCfg  = TRUE; // use original TF type from phase
	phaseCommonPtr->lSigLengthBe	  = planManagerDefaultSettings.UlCommonTf_tfLength;
	phaseCommonPtr->lSigLengthBk	  = planManagerDefaultSettings.UlCommonTf_tfLength;
	phaseCommonPtr->lSigLengthVi	  = planManagerDefaultSettings.UlCommonTf_tfLength;
	phaseCommonPtr->lSigLengthVo	  = planManagerDefaultSettings.UlCommonTf_tfLength;
#endif	
}


/**********************************************************************************
planManagerSetUlPhaseCommonTcr_A_Fields 

Description: 
Sets the DATA TCR common section in UL phase  (The basic TF TCR) 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonTcr_A_Fields(UlDataPhaseCommonTcra_t*  phaseCommonTcraPtr, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonDataTcrPtr = (TcrCommon_t*)&(phaseCommonTcraPtr);
	VapDb_t*     pVapDbHwEntry         = &VapDbHwEntries[vapId];

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_V("[planManagerSetUlPhaseCommonTcr_A_Fields]");
#endif
	//phaseCommonDataTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonDataTcrPtr->ant0Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant1Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant2Boost//Filled by LA
	//phaseCommonDataTcrPtr->ant3Boost//Filled by LA
	//phaseCommonDataTcrPtr->rfPower = filled by LA 
	phaseCommonDataTcrPtr->txLoopMode      = planManagerDefaultSettings.UlCommonPhaseTcrA_txLoopMode; 
	phaseCommonDataTcrPtr->cbw             = pMuCreatePlanReq->bw; 
	phaseCommonDataTcrPtr->phyMode         = planManagerDefaultSettings.UlCommonPhaseTcrA_phyMode;
    phaseCommonDataTcrPtr->scp             = planManagerDefaultSettings.UlCommonPhaseTcrA_scp; //don't care as phy mode is HE  (this field is relevant for HT/VHT)
	phaseCommonDataTcrPtr->heFullbandMu    = 0; // 0 for BC TF. If TF is sent in MU-MIMO, should be 1
	phaseCommonDataTcrPtr->hePuncturingMap = planManagerDefaultSettings.UlCommonPhaseTcrA_hePuncturingMap; 
	phaseCommonDataTcrPtr->bssColor        = pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonDataTcrPtr->customBf        = 0; // NA - for prouction 
	//phaseCommonDataTcrPtr->nHeltf        //Filled by LA -  for MIMO - sum all NSS of users , for OFDMA - MAX NSS of all users ,value is always in power of 2 
	phaseCommonDataTcrPtr->ulDl            = planManagerDefaultSettings.UlCommonPhaseTcrA_ulDl; // set 0 for DL , 1 for UL ,  See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonDataTcrPtr->txopPs          = planManagerDefaultSettings.UlCommonPhaseTcrA_txopPs; 
	phaseCommonDataTcrPtr->expectedRcr     = planManagerDefaultSettings.UlCommonPhaseTcrA_expectedRcr; //  RCR expected , This packet is not a trigger frame - next receive is trigger based
	phaseCommonDataTcrPtr->bwChange        = planManagerDefaultSettings.UlCommonPhaseTcrA_bwChange; 
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 1;
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonDataTcrPtr->txopDurationFromMac = 0;
	}
	phaseCommonDataTcrPtr->aggregate    = planManagerDefaultSettings.UlCommonPhaseTcrA_aggregate; 
	phaseCommonDataTcrPtr->dynamicBw    = planManagerDefaultSettings.UlCommonPhaseTcrA_dynamicBw; 
	phaseCommonDataTcrPtr->muPhyNdp     = 0;
	phaseCommonDataTcrPtr->muTraining   = 0;
	//phaseCommonDataTcrPtr->heCp       = staticPlanConfigurationPtr->commonSection.heCp; //Filled by LA  (HE)
	//phaseCommonDataTcrPtr->heLtf      = staticPlanConfigurationPtr->commonSection.heLtf; //Filled by LA 
	//phaseCommonDataTcrPtr->smoothing  //Filled by LA 
	phaseCommonDataTcrPtr->notSounding  = planManagerDefaultSettings.UlCommonPhaseTcrA_notSounding; 
	//phaseCommonDataTcrPtr->stbc       = //Filled by LA  (HT/VHT)
	phaseCommonDataTcrPtr->spatialReuse =  planManagerDefaultSettings.UlCommonPhaseTcrA_spatialReuse; //NA
	//phaseCommonDataTcrPtr->triggerBasedDataLLenght  //owned by phy 
						
	phaseCommonDataTcrPtr->triggerBasedDataLdpc     = planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataLdpc; // Used for station mode only 
	//phaseCommonDataTcrPtr->triggerBasedDataPktExt = set by LA  //pre-FEC padding fator , 0 means 4 
	phaseCommonDataTcrPtr->triggerBasedDataPeDisam  = planManagerDefaultSettings.UlCommonPhaseTcrA_triggerBasedDataPeDisam; // Used for station mode only
	phaseCommonDataTcrPtr->groupId                  = pMuCreatePlanReq->groupId;
	//phaseCommonDataTcrPtr->heSigRate              = 0x0;  //to be filled by LA  (MCS 0, NSS - according to users and DCM = 0 )
	phaseCommonDataTcrPtr->macDuration              = planManagerDefaultSettings.UlCommonPhaseTcrA_macDuration; 
}


/**********************************************************************************
planManagerSetUlPhaseCommonTcr_B_Fields 

Description: 
Sets the MU BA TCR common section in UL pahse 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonTcr_B_Fields(UlDataPhaseCommonTcrb_t* phaseCommonTcrbPtr, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t*  planPtr, uint8 vapId)
{
	TcrCommon_t* phaseCommonMuBaTcrPtr = (TcrCommon_t*)&(phaseCommonTcrbPtr);
	VapDb_t*     pVapDbHwEntry         = &VapDbHwEntries[vapId];
	
	//phaseCommonMuBaTcrPtr->antennaSelection =//Filled by LA -  might be depantent in CDB - TBD LA - API routine to get antenna selection per band ; 
	//phaseCommonMuBaTcrPtr->ant0Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->ant1Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->ant2Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->ant3Boost//Filled by LA
	//phaseCommonMuBaTcrPtr->rfPower  = //Filled by LA
	phaseCommonMuBaTcrPtr->txLoopMode       = planManagerDefaultSettings.UlCommonPhaseTcrB_txLoopMode; 
	phaseCommonMuBaTcrPtr->cbw              = pMuCreatePlanReq->bw; 
	phaseCommonMuBaTcrPtr->phyMode          = planManagerDefaultSettings.UlCommonPhaseTcrB_phyMode; 
	phaseCommonMuBaTcrPtr->heFullbandMu     = 0; // 0 for BC M-BA. If TF is sent in MU-MIMO, should be 1 
	phaseCommonMuBaTcrPtr->hePuncturingMap  = planManagerDefaultSettings.UlCommonPhaseTcrB_hePuncturingMap; 
	phaseCommonMuBaTcrPtr->forceTx = 1; // 2.4G COEX: set "force tx" bit as TRUE so that PHY shall transmit ACK\BA\MSBA when we are at "TX DISABLE".
	phaseCommonMuBaTcrPtr->bssColor         =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonMuBaTcrPtr->customBf         = planManagerDefaultSettings.UlCommonPhaseTcrB_customBf; // NA - for prouction 
	//phaseCommonMuBaTcrPtr->nHeltf         = 4;//Filled by LA   
	phaseCommonMuBaTcrPtr->ulDl             = planManagerDefaultSettings.UlCommonPhaseTcrB_ulDl;    // set 0 for DL , 1 for UL ,See the HE 1.3 draft sigA 28.3.10.7
	phaseCommonMuBaTcrPtr->txopPs           = planManagerDefaultSettings.UlCommonPhaseTcrB_txopPs; 
	phaseCommonMuBaTcrPtr->expectedRcr      = planManagerDefaultSettings.UlCommonPhaseTcrB_expectedRcr; //RCR expected , This packet is trigger frame - next receive is trigger based
	phaseCommonMuBaTcrPtr->bwChange         = planManagerDefaultSettings.UlCommonPhaseTcrB_bwChange; 
	if(planPtr->durationUpdateMode == 0) // if partial txop mode - then take the duration from MAC 
	{
		phaseCommonMuBaTcrPtr->txopDurationFromMac = 1;
	}
	else  //Full TXOP  - take txop duration from SIG A 
	{
		phaseCommonMuBaTcrPtr->txopDurationFromMac = 0;
	}

	phaseCommonMuBaTcrPtr->aggregate                 = planManagerDefaultSettings.UlCommonPhaseTcrB_aggregate; //BAR is not aggregated 
	phaseCommonMuBaTcrPtr->dynamicBw                 = planManagerDefaultSettings.UlCommonPhaseTcrB_dynamicBw; 
	phaseCommonMuBaTcrPtr->muPhyNdp                  = 0;
	phaseCommonMuBaTcrPtr->muTraining                = 0;
	//phaseCommonMuBaTcrPtr->heCp                    = //Filled by LA  
	//phaseCommonMuBaTcrPtr->heLtf                   = //Filled by LA 
	//phaseCommonMuBaTcrPtr->scp                     = //Filled by LA 
	//phaseCommonMuBaTcrPtr->smoothing               //Filled by LA 
	phaseCommonMuBaTcrPtr->notSounding               = planManagerDefaultSettings.UlCommonPhaseTcrB_notSounding; 
	//phaseCommonMuBaTcrPtr->stbc                    = //Filled by LA  
	phaseCommonMuBaTcrPtr->spatialReuse              = planManagerDefaultSettings.UlCommonPhaseTcrB_spatialReuse; //NA
	//phaseCommonMuBaTcrPtr->triggerBasedDataLLenght = // owned by the phy 
	phaseCommonMuBaTcrPtr->triggerBasedDataLdpc      = planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataLdpc;  // for station mode ony
	// phaseCommonMuBaTcrPtr->triggerBasedDataPktExt = // Filled by LA  - search for the max station packet extension in group  
	phaseCommonMuBaTcrPtr->triggerBasedDataPeDisam   = planManagerDefaultSettings.UlCommonPhaseTcrB_triggerBasedDataPeDisam; // for station mode ony 
	phaseCommonMuBaTcrPtr->groupId                   = pMuCreatePlanReq->groupId; 
    //phaseCommonMuBaTcrPtr->heSigRate               = 0x0; //Filled by LA - MCS 0, DCM0 , NSS  -1
	phaseCommonMuBaTcrPtr->macDuration               = planManagerDefaultSettings.UlCommonPhaseTcrB_macDuration; 
}



/**********************************************************************************
planManagerSetUlPhaseCommonRcrFields 

Description: 
Sets the RCR common section in UL pahse 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonRcrFields(UlDataPhaseCommonRcr_t* phaseCommonRcrPtr,CreatePlanMsg_t* pMuCreatePlanReq, uint8 vapId)
{
	RcrCommon_t* phaseRcrCommonPtr = (RcrCommon_t*)(phaseCommonRcrPtr);
    VapDb_t*     pVapDbHwEntry     = &VapDbHwEntries[vapId];

	phaseRcrCommonPtr->heSigABssColor = pVapDbHwEntry->common.tcr1BssColor;
	phaseRcrCommonPtr->cbw            = pMuCreatePlanReq->bw; 
	//phaseRcrCommonPtr->nHeltf       = //Filled by LA -
#ifdef ENET_INC_ARCH_WAVE600D2
	if (heGroupManagerDb[pMuCreatePlanReq->groupId].formationType == UL_MIMO)
	{
		phaseRcrCommonPtr->heCp   = HE_CP_HE_1_6_CP; 
		phaseRcrCommonPtr->heLtf  = HE_LTF_HE_1X_LTF;
		phaseRcrCommonPtr->tbMode = TB_MODE_MU_MIMO;
	}
	else
	{
		phaseRcrCommonPtr->heCp   = HE_CP_HE_1_6_CP; 
		phaseRcrCommonPtr->heLtf  = HE_LTF_HE_2X_LTF;		
		phaseRcrCommonPtr->tbMode = TB_MODE_OFDMA;
	}
#else
	phaseRcrCommonPtr->heCp  = HE_CP_HE_1_6_CP; 
	phaseRcrCommonPtr->heLtf = HE_LTF_HE_2X_LTF;	
#endif
    phaseRcrCommonPtr->stbc               = NOT_STBC_TRANSMISSION;
	phaseRcrCommonPtr->heSigASpatialReuse = 0x0; 
	phaseRcrCommonPtr->lSigLength         = planManagerDefaultSettings.Ul_lSigLength;
	//phaseRcrCommonPtr->ldpcExtraSymbol  = //Filled by LA
	phaseRcrCommonPtr->peDisambiguty      = planManagerDefaultSettings.DlCommonPhaseRcr_peDisambiguty; // We force the station to set PE disambiguty  
}


/**********************************************************************************

planManagerSetStaticUlPhaseCommonUserTCR_A_Fields 

Description: 
Sets the common user TCRB common section in DL pahse  - how to transmit the MU BAR 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonUserTCR_A_Fields(UlDataPhaseCommonUserTcra_t* commonUserTcraPtr)
{	
	commonUserTcraPtr->brdcstUserTcraSubBand            = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_SubBand; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcraPtr->brdcstUserTcraStartRu            = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_StartRu; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcraPtr->brdcstUserTcraRuSize             = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RuSize; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcraPtr->brdcstUserTcraLdpc               = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Ldpc;
	commonUserTcraPtr->brdcstUserTcraTxbf               = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_Txbf; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcraPtr->brdcstUserTcraPsduRate         = 0; //filed by LA 
	commonUserTcraPtr->brdcstUserTcraRelativeUserPower  = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_RelativeUserPower; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcraPtr->brdcstUserTcraPacketExtension  = // filed by LA 0; /
	commonUserTcraPtr->brdcstUserTcraStaAid             = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_StaAid; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcraPtr->brdcstUserTcraBfIndex            = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrA_BfIndex; // NA   trigger frame transmitted in legacy phy mode
}
 

/**********************************************************************************
planManagerSetStaticUlPhaseCommonUserTCR_B_Fields 

Description: 
Sets the common user TCRB common section in DL pahse  - how to transmit the MU BAR 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonUserTCR_B_Fields(UlDataPhaseCommonUserTcrb_t* commonUserTcrbPtr)
{
	commonUserTcrbPtr->brdcstUserTcrbSubBand            = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_SubBand; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbStartRu            = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_StartRu; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbRuSize             = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RuSize; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbLdpc               = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_Ldpc; 
	commonUserTcrbPtr->brdcstUserTcrbTxbf               = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_Txbf; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPsduRate         = 0;  //filed by LA 
	commonUserTcrbPtr->brdcstUserTcrbRelativeUserPower  = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_RelativeUserPower; // NA   trigger frame transmitted in legacy phy mode
	//commonUserTcrbPtr->brdcstUserTcrbPacketExtension  = // filed by LA 0; /
	commonUserTcrbPtr->brdcstUserTcrbStaAid             = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_TcrbStaAid; // NA   trigger frame transmitted in legacy phy mode
	commonUserTcrbPtr->brdcstUserTcrbBfIndex            = planManagerDefaultSettings.UlCommonPhaseBrdcstUserTcrB_BfIndex; // NA   trigger frame transmitted in legacy phy mode
}


/**********************************************************************************
planManagerSetUlPhaseCommonTfParameters 

Description: 
Sets the common Trigger frame section in UL phase 
	
**********************************************************************************/
static void planManagerSetUlPhaseCommonTfParameters(UlDataPhase_t* ulDataPhasPtr,CreatePlanMsg_t* pMuCreatePlanReq, uint8 muSequenceType)
{
	UlDataPhaseTfParameters_t* commonTfParamsPtr = &(ulDataPhasPtr->tfParameters); 
	UlDataPhaseTfCommonInfo_t* commonTfInfoPtr   = &(ulDataPhasPtr->tfCommonInfo); 
	
    commonTfParamsPtr->tfLsigLengthOverride		 = planManagerDefaultSettings.UlTf_LsigLengthOverride; 
	commonTfParamsPtr->tfBwOverride				 = planManagerDefaultSettings.UlTf_BW_Override;
    switch(muSequenceType)
	{
		case HE_MU_UL:
			commonTfInfoPtr->tfTriggerType = TF_TYPE_BASIC; 
			commonTfInfoPtr->tfLength      = planManagerDefaultSettings.UlCommonTf_tfLength; //This is the L-SIG length we want the STAs to transmitt - amount of bytes for station to transmit
			break;
		case HE_MU_BSRP:
		case HE_MU_BSRP_UL:
			commonTfInfoPtr->tfTriggerType = TF_TYPE_BUFFER_STATUS_RPT; 
			commonTfInfoPtr->tfLength      = planManagerDefaultSettings.UlCommonTf_tfLength; //This is the L-SIG length we want the STAs to transmitt - amount of bytes for station to transmit
			break;
#ifdef ENET_INC_ARCH_WAVE600D2
		case HE_MU_NFRP:
			commonTfInfoPtr->tfTriggerType = TF_TYPE_NFRP; 
			commonTfInfoPtr->tfLength      = planManagerDefaultSettings.NfrpCommonPhaseRcr_lSigLength;
						
			commonTfParamsPtr->tfNfrpStartingAidOverride	  = SUPPORT;
			commonTfParamsPtr->tfNfrpMultiplexingFlagOverride = SUPPORT;						
			break;
#endif 
		default:
			FATAL("UNKNOWN muSequenceType");
			break;
	}

#ifdef ENET_INC_ARCH_WAVE600B
    commonTfParamsPtr->tfRuIndexOverride        = planManagerDefaultSettings.UlTf_RU_Allocation_Override;
#else 
    commonTfParamsPtr->tfRuAllocationOverride   = planManagerDefaultSettings.UlTf_RU_Allocation_Override;
#endif 
    commonTfParamsPtr->tfMcsOverride            = planManagerDefaultSettings.UlTf_MCS_Override;
    commonTfParamsPtr->tfNssOverride            = planManagerDefaultSettings.UlTf_NSS_Override;
    commonTfParamsPtr->tfDcmOverride            = planManagerDefaultSettings.UlTf_DcmOverride;
    commonTfParamsPtr->tfTargetRssiOverride     = planManagerDefaultSettings.UlTf_Target_RSSI_Override;
    commonTfParamsPtr->tfTidAggrLimitOverride   = planManagerDefaultSettings.UlTf_TID_Aggr_Limit_Override;
    commonTfParamsPtr->tfPreferredAcOverride    = planManagerDefaultSettings.UlTf_Preferred_AC_Override;
    commonTfParamsPtr->tfMultiTidOverride       = planManagerDefaultSettings.UlTf_Multi_TID_Override;
    commonTfParamsPtr->tfTidInfoOverride        = planManagerDefaultSettings.UlTf_TID_Info_Override;
    commonTfParamsPtr->tfCompressedFnOverride   = planManagerDefaultSettings.UlTf_Compressed_FN_Override;
    commonTfParamsPtr->tfCompressedSsnOverride  = planManagerDefaultSettings.UlTf_Compressed_SSN_Override;
    commonTfParamsPtr->tfMultiTidTidOverride    = planManagerDefaultSettings.UlTf_Multi_TID_TID_Override;
    commonTfParamsPtr->tfMultiTidFnOverride     = planManagerDefaultSettings.UlTf_Multi_TID_FN_Override;
    commonTfParamsPtr->tfMultiTidSsnOverride    = planManagerDefaultSettings.UlTf_Multi_TID_SSN_Override;
        
    commonTfParamsPtr->tfCommonInfoSwOverrideEnable = planManagerDefaultSettings.UlCommonTf_CommonInfoSwOverrideEnable; // only in case we want to override the SW info with a configured value in TF generator register 
    commonTfParamsPtr->tfUserInfoSwOverrideEnable   = planManagerDefaultSettings.UlCommonTf_UserInfoSwOverrideEnable;   // only in case we want to override the SW info with a configured value in TF generator register 
    
#ifdef ENET_INC_ARCH_WAVE600D2
    commonTfParamsPtr->tfNumOfHeltfSymbolsOverride     = 1;
    commonTfParamsPtr->tfFecCodingTypeOverride         = pMuCreatePlanReq->muType; // In UL MU-MIMO - can be changed due to lower number of users which will enlarge the RU to be larger than 242 (20Mhz)
    commonTfParamsPtr->tfLdpcExtraSymbolOverride       = pMuCreatePlanReq->muType; // can be changed in UL MU-MIMO
    commonTfParamsPtr->tfStartingSpatialStreamOverride = pMuCreatePlanReq->muType; // Must be set *only* in UL MU-MIMO
#endif

#ifdef ENET_INC_ARCH_WAVE600D2		
	commonTfInfoPtr->tfMoreTf            = planManagerDefaultSettings.UlCommonTf_tfMoreTf;  //NA															
#else
	commonTfInfoPtr->tfCascadeIndication = planManagerDefaultSettings.UlCommonTf_CascadeIndication;  //NA															
#endif	
	commonTfInfoPtr->tfCsRequired        = planManagerDefaultSettings.UlCommonTf_CsRequired; // should a station consider CCA before UL  - only for MU RTS should be set with 1  
	commonTfInfoPtr->tfBw                = pMuCreatePlanReq->bw;
#ifdef ENET_INC_ARCH_WAVE600D2	
	if (pMuCreatePlanReq->muType == MIMO)
	{
		commonTfInfoPtr->tfGiAndLtfType = GI_AND_LTF_TYPE_1X_LTF_1_6_GI; 
	}
	else
#endif
	{
		commonTfInfoPtr->tfGiAndLtfType = planManagerDefaultSettings.tfGiAndLtfType; 
	}
	commonTfInfoPtr->tfMuMimoLtfMode    =  planManagerDefaultSettings.UlCommonTf_MuMimoLtfMode; //0: Single Stream Pilots   1 : Mask on each spatial stream of the LTF sequence by a distinct orthogonal code
	//commonTfInfoPtr->tfNumberOfHeLtfSymbols   = //to be set by LA 
	//commonTfInfoPtr->tfStbc                   = 0; //to be set by LA 
	//commonTfInfoPtr->tfLdpcExtraSymbolSegment = 1; // to be set by LA  
	//commonTfInfoPtr->tfApTxPower30            = // to be set by LA  
	//commonTfInfoPtr->tfApTxPower54            = // to be set by LA  
	//commonTfInfoPtr->tfPacketExtension        = // Station capability - to be filled by LA - Max of all users 
	commonTfInfoPtr->tfSpatialReuse     = planManagerDefaultSettings.UlCommonTf_SpatialReuse; //NA 
	commonTfInfoPtr->tfDoppler          = planManagerDefaultSettings.UlCommonTf_Doppler; //NA 
#ifdef ENET_INC_ARCH_WAVE600D2		
	commonTfInfoPtr->tfHeSigA2Reserved    = 0x1FF; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#else
	commonTfInfoPtr->tfHeSigAReserved     = 0x1FF; //Should be 0 for MU-BAR, 0x1FF for rest of TF types
#endif	
	commonTfInfoPtr->tfReservedCommonInfo = 0x0; 

}


/**********************************************************************************
planManagerSetUlPhasePerUser 

Description: 
Sets per user data section in UL pahse 
	
**********************************************************************************/
static void planManagerSetUlPhasePerUser(UlDataPhaseUserPhase_t* ulDataPhasPerUserPtr, CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation, uint8 muSequenceType)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;
    uint8   vapId;

    ulDataPhasPerUserPtr = &(ulDataPhasPerUserPtr[0]); 
    
    for(userLoopIndex = 0; userLoopIndex < heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup; userLoopIndex++)
	{	
		userStationIndex                        = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex];
		ulDataPhasPerUserPtr->staId             = userStationIndex;
		StaDb_GetVapId(userStationIndex, &vapId);
        ulDataPhasPerUserPtr->vapId             = vapId;
		ulDataPhasPerUserPtr->tfPadding         = planManagerDefaultSettings.ulTfPadding; 
		ulDataPhasPerUserPtr->numTfRepetition   = planManagerDefaultSettings.UlPerUserTf_numTfRepetition; //0x1; 
		ulDataPhasPerUserPtr->mapTxUserToRxUser = userLoopIndex; //Mapping is 1 X 1 for Z0
		ulDataPhasPerUserPtr->tidAllocBitmap    = planManagerDefaultSettings.UlPerUser_tidAllocBitmap;
		heGroupManagerSetPlanTidsBitmap(ulDataPhasPerUserPtr->tidAllocBitmap, muSequenceType);
		ulDataPhasPerUserPtr->maxTids           = TsManager_TsManagerRecipientGetMaxUlTids(userStationIndex);
        /*for maxTids field per user - we should call an API routine from TS manager that provides the tid mumber, considering bitmap HW limitation. */
		/*	We should provide an API function for the TS manager to update the field OTF */
		
		ulDataPhasPerUserPtr->mapTxUserToTxUser  = userLoopIndex; 			
		ulDataPhasPerUserPtr->userPhaseReserve10 = 0x0; 
		//ulDataPhasPerUserPtr->ampduLimit = StationDatabase.pStaDbHwEntries[userStationIndex].common.aMpduLimit; 

		/* RU ALLOCATOR SIMULATION */
#ifdef ENET_INC_ARCH_WAVE600D2
		if (pMuCreatePlanReq->muType == MIMO)
		{
			// In case of UL MU-MIMO, allocate all plan BW size as RU
			ulDataPhasPerUserPtr->userRcrSubBand = 0; 
			ulDataPhasPerUserPtr->userRcrStartRu = 0; 
			ulDataPhasPerUserPtr->userRcrRuSize  = CONVERT_BW_TO_RU_SIZE(pMuCreatePlanReq->bw);

			PlanManagerStatistics.uplinkMuStats[userStationIndex].uplinkMuType = UL_MU_TYPE_MIMO;
		}
		else
#endif
		{		
		    ulDataPhasPerUserPtr->userRcrSubBand = ruAllocation[userLoopIndex].ul_SubBand; 
		    ulDataPhasPerUserPtr->userRcrStartRu = ruAllocation[userLoopIndex].ul_StartRU; 
		    ulDataPhasPerUserPtr->userRcrRuSize  = ruAllocation[userLoopIndex].userRcrRuSize; 

			PlanManagerStatistics.uplinkMuStats[userStationIndex].uplinkMuType = UL_MU_TYPE_HE;
        }
		//Update Statistics
		PlanManagerStatistics.uplinkMuStats[userStationIndex].ulRuSubChannels = ulDataPhasPerUserPtr->userRcrSubBand;
		PlanManagerStatistics.uplinkMuStats[userStationIndex].ulRuType = ulDataPhasPerUserPtr->userRcrRuSize;

		//ulDataPhasPerUserPtr->userRcrLdpc =   //set by LA
		ulDataPhasPerUserPtr->userRcrTxbf       = planManagerDefaultSettings.UlPerUserTf_RcrTxbf;
		//ulDataPhasPerUserPtr->userRcrPsduRate = //set by LA 
		//DataPhasPerUserPtr->userRcrTargetRssi = //set by LA 
		ulDataPhasPerUserPtr->userRcrReserve00  = 0x0; 
		ulDataPhasPerUserPtr->userRcrValid      = planManagerDefaultSettings.UlPerUserTf_RcrValid; 
		//TCR A per user 
		ulDataPhasPerUserPtr->userTcraSubBand   = ruAllocation[userLoopIndex].dl_subBandPerUsp; 
		ulDataPhasPerUserPtr->userTcraStartRu   = ruAllocation[userLoopIndex].dl_startRuPerUsp;
		ulDataPhasPerUserPtr->userTcraRuSize    = ruAllocation[userLoopIndex].dl_ruSizePerUsp;
		//ulDataPhasPerUserPtr->userTcraLdpc    = 0x0; //set by LA 
		ulDataPhasPerUserPtr->userTcraTxbf      = planManagerDefaultSettings.UlPerUserTf_TcraTxbf;  
		//ulDataPhasPerUserPtr->userTcraPsduRate = //set by LA 
		ulDataPhasPerUserPtr->userTcraRelativeUserPower = planManagerDefaultSettings.UlPerUserTf_TcraRelativeUserPower; 
		ulDataPhasPerUserPtr->userTcraReserve00 = 0x0;  
		//ulDataPhasPerUserPtr->userTcraPacketExtension = 2;   //set by LA   : 0  0 usec ; 1  8 usec ; 2  16 usec
		ulDataPhasPerUserPtr->userTcraReserve01 = 0x0;  
		ulDataPhasPerUserPtr->userTcraReserve10 = 0x0;  
		ulDataPhasPerUserPtr->userTcraBfIndex   = planManagerDefaultSettings.UlPerUserTf_TcraBfIndex; //NA  
		ulDataPhasPerUserPtr->userTcraReserve11 = 0x0;  
		//TCR B per user 
		ulDataPhasPerUserPtr->userTcrbSubBand   = ruAllocation[userLoopIndex].dl_subBandPerUsp;
		ulDataPhasPerUserPtr->userTcrbStartRu   = ruAllocation[userLoopIndex].dl_startRuPerUsp;
		ulDataPhasPerUserPtr->userTcrbRuSize    = ruAllocation[userLoopIndex].dl_ruSizePerUsp;
		ulDataPhasPerUserPtr->userTcrbLdpc      = planManagerDefaultSettings.UlPerUserTf_TcrbLdpc;
		ulDataPhasPerUserPtr->userTcrbTxbf      = planManagerDefaultSettings.UlPerUserTf_TcrbTxbf;  
		//ulDataPhasPerUserPtr->userTcrbPsduRate        = //Set by LA 
		ulDataPhasPerUserPtr->userTcrbRelativeUserPower = planManagerDefaultSettings.UlPerUserTf_TcrbRelativeUserPower; 
		ulDataPhasPerUserPtr->userTcrbReserve00         = 0x0;  
		//ulDataPhasPerUserPtr->userTcrbPacketExtension = //set by LA 2;  //0  0 usec ; 1  8 usec ; 2  16 usec	
		ulDataPhasPerUserPtr++;
	}
}


/**********************************************************************************
planManagerSetUlPhaseTfInfoPerUser 

Description: 
Sets per user TF section in UL phase 

**********************************************************************************/
static void planManagerSetUlPhaseTfInfoPerUser(UlDataPhaseTfUserInfo_t* ulDataPhasTfInfoPerUserPtr,CreatePlanMsg_t* pMuCreatePlanReq, RuAllocationPerUser_t* ruAllocation)
{
	uint8 	userLoopIndex; 
	uint8	userStationIndex;
	uint8	numOfLoopIterations;
	uint8	remainderUspOffset = PLAN_MANAGER_INVALID_USP_OFFSET; 


	UNUSED_PARAM(ruAllocation);

	numOfLoopIterations = (heGroupManagerDb[pMuCreatePlanReq->groupId].numOfStationsInGroup);
	if((numOfLoopIterations) % 2 != 0) // if number of users is odd 
	{
		remainderUspOffset  = numOfLoopIterations; 
        numOfLoopIterations = numOfLoopIterations - 1; //loop will run over even number of users since in every loop two users are filled
	}

	for(userLoopIndex = 0; userLoopIndex < numOfLoopIterations; userLoopIndex = userLoopIndex + 2)
	{	
		/* First user in pair*/
		userStationIndex = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex]; 
		ulDataPhasTfInfoPerUserPtr->tfAid120                = userStationIndex + 1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        // only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
        ulDataPhasTfInfoPerUserPtr->tfRuAllocation0         = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize); 
#else 
        ulDataPhasTfInfoPerUserPtr->tfRuAllocation0         = planManagerDefaultSettings.UlPerUserTf_RuAllocation; 
#endif
#ifdef  ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        ulDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800 = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize);
#endif 
        ulDataPhasTfInfoPerUserPtr->tfRuIndex0              = planManagerDefaultSettings.UlPerUserTf_RuAllocation; // override by the RCR value  
#endif
		//ulDataPhasTfInfoPerUserPtr->tfCodingType0         = //Set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfMcs0                = //Set by LA 
		ulDataPhasTfInfoPerUserPtr->tfDcm0                  = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		//ulDataPhasTfInfoPerUserPtr->tfSsAllocation0       = //Set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0  = //Set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = //Set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfTargetRssi0         =  //Set by LA 
		ulDataPhasTfInfoPerUserPtr->tfReservedUserInfo0     = 0x0; 
		//ulDataPhasTfInfoPerUserPtr->tfTriggerDependent0   union of the below fields 
		ulDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0 = planManagerDefaultSettings.tfMpduMuSpacingFactor;
		ulDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0     = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfBasicReserved0        = 0x0; 
		ulDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0     = planManagerDefaultSettings.UlPerUserTf_preferredAc0; //override by the RCR value 
		/* Second user in pair*/
		userStationIndex = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex+1]; 
		ulDataPhasTfInfoPerUserPtr->tfAid121 = userStationIndex + 1; 
#ifndef ENET_INC_ARCH_WAVE600B
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation130       = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) & RU_ALLOCATION_MASK; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfRuAllocation174       = (planManagerDefaultSettings.UlPerUserTf_RuAllocation) >> RU_ALLOCATION_SHIFT; //override by the RCR value 
#else
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        ulDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary801 = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex+1].ul_SubBand, ruAllocation[userLoopIndex+1].userRcrRuSize);
        ulDataPhasTfInfoPerUserPtr->tfRuAllocation130       = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex+1].ul_SubBand, ruAllocation[userLoopIndex+1].userRcrRuSize); 
#endif 
	   	ulDataPhasTfInfoPerUserPtr->tfRuIndex2To01          = ((planManagerDefaultSettings.DlPerUserTf_RuAllocation) & RU_ALLOCATION_B_MASK); //override by the RCR value
        ulDataPhasTfInfoPerUserPtr->tfRuIndex6To31          = (planManagerDefaultSettings.DlPerUserTf_RuAllocation) >> RU_ALLOCATION_B_SHIFT; //override by the RCR value
#endif
		//ulDataPhasTfInfoPerUserPtr->tfCodingType1             = //to be set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfMcs1                    = //to be set by LA 
		ulDataPhasTfInfoPerUserPtr->tfDcm1                      = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		//ulDataPhasTfInfoPerUserPtr->tfSsAllocation1           = //to be set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfStartingSpatialStream1  = //to be set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams1 = //to be set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfTargetRssi1             = //to be set by LA 
		ulDataPhasTfInfoPerUserPtr->tfReservedUserInfo1         = 0x0; 
		//ulDataPhasTfInfoPerUserPtr->tfTriggerDependent1       union of the below fields 
		ulDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor1 = planManagerDefaultSettings.tfMpduMuSpacingFactor;
		ulDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit1         = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfBasicReserved1            = 0x0; 
		ulDataPhasTfInfoPerUserPtr->tfBasicPreferredAc1         = planManagerDefaultSettings.UlPerUserTf_preferredAc0; //override by the RCR value 
		
		ulDataPhasTfInfoPerUserPtr++; 
	}
	if(remainderUspOffset != PLAN_MANAGER_INVALID_USP_OFFSET)
	{
		userLoopIndex = remainderUspOffset - 1; 
		userStationIndex = heGroupManagerDb[pMuCreatePlanReq->groupId].members[userLoopIndex]; 
		ulDataPhasTfInfoPerUserPtr->tfAid120                    = userStationIndex + 1; 
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        // only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
        ulDataPhasTfInfoPerUserPtr->tfRuAllocation0             = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize); 
#else 
        ulDataPhasTfInfoPerUserPtr->tfRuAllocation0             = planManagerDefaultSettings.UlPerUserTf_RuAllocation; // override by the RCR value  
#endif 
#ifdef  ENET_INC_ARCH_WAVE600B	
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
        ulDataPhasTfInfoPerUserPtr->tfRuPrimarySecondary800     = planManagerCalcRuAllocationSecondaryBit(pMuCreatePlanReq->bw, ruAllocation[userLoopIndex].ul_SubBand, ruAllocation[userLoopIndex].userRcrRuSize);
#endif 
        ulDataPhasTfInfoPerUserPtr->tfRuIndex0                  = (planManagerDefaultSettings.UlPerUserTf_RuAllocation); //override by the RCR value
#endif
		//ulDataPhasTfInfoPerUserPtr->tfCodingType0 // set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfMcs0 =  // set by LA 
		ulDataPhasTfInfoPerUserPtr->tfDcm0                      = planManagerDefaultSettings.UlPerUserTf_Dcm; //override by the RCR value 
		//ulDataPhasTfInfoPerUserPtr->tfSsAllocation0           = // set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfStartingSpatialStream0  = // set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfNumberOfSpatialStreams0 = // set by LA 
		//ulDataPhasTfInfoPerUserPtr->tfTargetRssi0             = // set by LA 
		ulDataPhasTfInfoPerUserPtr->tfReservedUserInfo0         = 0x0; 
		//ulDataPhasTfInfoPerUserPtr->tfTriggerDependent0       = 0x0; //union of the 4 below fields 
		ulDataPhasTfInfoPerUserPtr->tfBasicMpduMuSpacingFactor0 = planManagerDefaultSettings.tfMpduMuSpacingFactor;
		ulDataPhasTfInfoPerUserPtr->tfBasicTidAggLimit0         = planManagerDefaultSettings.UlPerUserTf_TidAggLimit0; //override by the RCR value 
		ulDataPhasTfInfoPerUserPtr->tfBasicReserved0            = 0x0; 
		ulDataPhasTfInfoPerUserPtr->tfBasicPreferredAc0         = planManagerDefaultSettings.UlPerUserTf_preferredAc0;; //override by the RCR value 
	}

}


/**********************************************************************************

planManagerRuAllocator_GetAllocationForHeGroup 

Description: 


**********************************************************************************/
static PlanManagerAllocationStatus_e planManagerRuAllocator_GetAllocationForHeGroup(RuAllocationPerUser_t* ruAllocationArrayPerUser, uint8 numOfValidUsers, Bw_e bw, uint8* allocatedUsers)
{
	uint8 ruIndex; 
	PlanManagerAllocationStatus_e	allocationStatus = ALLOCATION_SUCCESS; 
	PreAggRuSizeMapping_e ruSizeIndex;
	uint8 subBand = 0;
	uint8 rusPerSubband = 0;
	uint8 numberOfSubbandsForMinAllocatedBw = 0;
	uint8* pArrRuIndexPerSubband;
	uint8 indexInArr = 0;
	uint8 userId = 0;
	uint8 minBwRoundDowm = bwMapToVal[bw]/numOfValidUsers;
	uint8 usedSubband; 
#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[planManagerRuAllocator_GetAllocationForHeGroup], bw: %d", bw);
#endif
	if(minBwRoundDowm < MIN_RU_SIZE_MH) 
	{
		allocationStatus = PLAN_ALLOCATION_FAILED; ; 
		ILOG0_V("planManagerRuAllocator_GetAllocationForHeGroup - Failed due to wrong number of user in BW configuration");
	}
	else
	{
		ruSizeIndex = searchClosestRuSize(minBwRoundDowm);
		rusPerSubband = RuSizeMapping[ruSizeIndex].numberOfRusPerSubband;
		numberOfSubbandsForMinAllocatedBw = RuSizeMapping[ruSizeIndex].numberOfSubbandsForMinAllocatedBw;
		usedSubband = bwMapToNumOfSubbands[bw];
		ASSERT(usedSubband >= numberOfSubbandsForMinAllocatedBw);
		pArrRuIndexPerSubband = &RuSizeMapping[ruSizeIndex].ruIndexPerSubband[0];
		*allocatedUsers = (usedSubband/numberOfSubbandsForMinAllocatedBw)*rusPerSubband;
		ILOG0_DDD("planManagerRuAllocator_GetAllocationForHeGroup - minBwRoundDowm = %d, Ru size = %d, rusPerSubband = %d, ",
					minBwRoundDowm, ruSizeIndex,rusPerSubband);
		ILOG0_DD("planManagerRuAllocator_GetAllocationForHeGroup - numberOfSubbandsForMinAllocatedBw = %d,allocatedUsers = %d",
					numberOfSubbandsForMinAllocatedBw,*allocatedUsers);
		
		for (subBand = 0, userId = 0; subBand < usedSubband ; subBand+=numberOfSubbandsForMinAllocatedBw)
		{
			for(indexInArr = 0; indexInArr < rusPerSubband; indexInArr++, userId++)
			{
				ruIndex = pArrRuIndexPerSubband[indexInArr];
				ruAllocationArrayPerUser[userId].ul_SubBand       = subBand; 
				ruAllocationArrayPerUser[userId].ul_StartRU       = ruIndex; 
				ruAllocationArrayPerUser[userId].userRcrRuSize    = ruSizeIndex; 

				ruAllocationArrayPerUser[userId].dl_subBandPerUsp = subBand; 
				ruAllocationArrayPerUser[userId].dl_startRuPerUsp = ruIndex; 
				ruAllocationArrayPerUser[userId].dl_ruSizePerUsp  = ruSizeIndex; 
				SLOG0(0, 0, RuAllocationPerUser_t, &ruAllocationArrayPerUser[userId]);
                //SERIAL_TRACE("sub band, start ru, ru size",subBand,ruIndex,ruSizeIndex);
			}
		}
	}
	return allocationStatus; 
}
#endif


/**********************************************************************************

planManagerRuAllocator_updateRUAllocAccordingToBwNRuSize 

Description: This function updates bitmap allocation according to RU size given to first user in static planner
assuming identical RU sizes for all users
  
Input: 
-----
	ruSize
	Bw_e
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerRuAllocator_updateRUAllocAccordingToBwNRuSize(PreAggRuSizeMapping_e ruSize, Bw_e bw)
{
	uint8 index80MHzBand;

	if (bw < BW_160MHZ)
	{
		index80MHzBand = 0;
		planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80(ruSize, bw, index80MHzBand);
		index80MHzBand = 1;
		RuAllocBitmap[index80MHzBand].ruIndexMask64bit[0] = BITMAP_ALLOC_BW80_2_20MHZ;
		RuAllocBitmap[index80MHzBand].ruIndexMask64bit[1] = BITMAP_ALLOC_BW80_2_20MHZ;
	}
	else //(bw >= BW_160MHZ)
	{		
		if (ruSize < PRE_AGGREGATOR_996X2_TONE_RU_SIZE)
		{
			index80MHzBand = 0;
			planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80(ruSize, BW_80MHZ, index80MHzBand);
			index80MHzBand = 1;
			planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80(ruSize, BW_80MHZ, index80MHzBand);
		}
		else //ruSize == PRE_AGGREGATOR_996X2_TONE_RU_SIZE
		{
			index80MHzBand = 0;
			ruSize = PRE_AGGREGATOR_996_TONE_RU_SIZE;
			planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80(ruSize, BW_80MHZ, index80MHzBand);
			index80MHzBand = 1;
			RuAllocBitmap[index80MHzBand].ruIndexMask64bit[0] = BITMAP_ALLOC_BW80_2_20MHZ;
			RuAllocBitmap[index80MHzBand].ruIndexMask64bit[1] = BITMAP_ALLOC_BW80_2_20MHZ;
		}			
	}
}


/**********************************************************************************

planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80 

Description: This function updates bitmap allocation according to RU size given to first user in static planner
assuming identical RU sizes for all users
  
Input: 
-----
	ruSize
	Bw_e
	index80MHzBand		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80(PreAggRuSizeMapping_e ruSize, Bw_e bw, uint8 index80MHzBand)
{
	uint32* pRuIndexMask64bit;
#ifdef DEBUG_RU_ALLOCATOR	
	ILOG0_DDD("[planManagerRuAllocator_updateRUAllocAccordingToBwNRuSizePer80], ruSize = %d, bw = %d, index80MHzBand = %d", ruSize, bw, index80MHzBand);
#endif
	RuAllocBitmap[index80MHzBand] = RuAllocIndexMask80[bw];
	pRuIndexMask64bit = &RuAllocBitmap[index80MHzBand].ruIndexMask64bit[0];
#ifdef DEBUG_RU_ALLOCATOR	
	SLOG0(0, 0, RuAllocMask_t, &RuAllocBitmap[index80MHzBand]);
	SLOG0(0, 0, RuAllocMask_t, &(RuSizeAllocIndexsMask[ruSize].ruIndexMask64bit));
#endif
	orOperator64Bit(pRuIndexMask64bit, RuSizeAllocIndexsMask[ruSize].ruIndexMask64bit, pRuIndexMask64bit);
#ifdef DEBUG_RU_ALLOCATOR
	SLOG0(0, 0, RuAllocMask_t, &RuAllocBitmap[index80MHzBand]);
#endif
}


/**********************************************************************************
planManagerRuAllocator_findLocationOfNextLeadingZero 

Description: This function finds first bit index from LSB that indicates RU that should be allocated
  
Input: 
-----
	lastLeadingZero
		
Output:
-------	
	firstBitmapIndex
Returns:
--------
	void 
	
**********************************************************************************/
static uint8 planManagerRuAllocator_findLocationOfNextLeadingZero(uint8 lastLeadingZero)
{
	uint8 index80MHzBand;
	uint8 firstBitmapIndex = RU_ALLOC_INVALID_BIT_INDEX;
	uint8 lastValidBitIndexInBitmap = RuSizeUserMaskLength[PRE_AGGREGATOR_996_TONE_RU_SIZE]; //37
	
#ifdef DEBUG_RU_ALLOCATOR		
	ILOG0_D("[planManagerRuAllocator_findLocationOfNextLeadingZero], lastLeadingZero =%d", lastLeadingZero);
#endif

	if (lastLeadingZero < lastValidBitIndexInBitmap)
	{
		index80MHzBand = 0;
		firstBitmapIndex = planManagerRuAllocator_findLocationOfNextLeadingZeroPer80(lastLeadingZero, index80MHzBand);
		if (firstBitmapIndex == RU_ALLOC_INVALID_BIT_INDEX)
		{
			index80MHzBand = 1;
			lastLeadingZero = 0;
			firstBitmapIndex = planManagerRuAllocator_findLocationOfNextLeadingZeroPer80(lastLeadingZero, index80MHzBand);
			firstBitmapIndex = firstBitmapIndex + lastValidBitIndexInBitmap;
		}
	}
	else 
	{
		index80MHzBand = 1;
		lastLeadingZero = lastLeadingZero - lastValidBitIndexInBitmap;
		firstBitmapIndex = planManagerRuAllocator_findLocationOfNextLeadingZeroPer80(lastLeadingZero, index80MHzBand);
		firstBitmapIndex = firstBitmapIndex + lastValidBitIndexInBitmap;
	}
	
#ifdef DEBUG_RU_ALLOCATOR		
	ILOG0_DDD("[planManagerRuAllocator_findLocationOfNextLeadingZero], lastLeadingZero =%d, firstBitmapIndex = %d, index80MHzBand = %d", lastLeadingZero, firstBitmapIndex, index80MHzBand);
#endif
	
	return(firstBitmapIndex);
}


/**********************************************************************************
planManagerRuAllocator_findLocationOfNextLeadingZeroPer80 

Description: This function finds first bit index from LSB that indicates RU that should be allocated
  
Input: 
-----
	lastLeadingZero
	index80MHzBand
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static uint8 planManagerRuAllocator_findLocationOfNextLeadingZeroPer80(uint8 lastLeadingZero, uint8 index80MHzBand)
{
	uint8 	indexBit = lastLeadingZero;
	uint8	firstBitmapIndex = RU_ALLOC_INVALID_BIT_INDEX;
	uint32	localRuMask[2] = {0};
	uint32 	mask = (1<<(indexBit));
	uint8   lastValidBitIndexInBitmap = RuSizeUserMaskLength[PRE_AGGREGATOR_996_TONE_RU_SIZE]; //37
	uint8 	numBitsInWord = CONVERT_WORDS_TO_BIT_INDEX(1); //32

	localRuMask[0] = RuAllocBitmap[index80MHzBand].ruIndexMask64bit[0];
	localRuMask[1] = RuAllocBitmap[index80MHzBand].ruIndexMask64bit[1];
#ifdef DEBUG_RU_ALLOCATOR	
	ILOG0_DD("[planManagerRuAllocator_findLocationOfNextLeadingZeroPer80] start, index80MHzBand = %d, lastLeadingZero = %d", index80MHzBand, lastLeadingZero);
	ILOG0_DD("[planManagerRuAllocator_findLocationOfNextLeadingZeroPer80] localRuMask[0] = %x, localRuMask[1] = %x", localRuMask[0], localRuMask[1]);
#endif	

	// loop over bitmap of RU to find the next unallocated RU to fill
	while( ((localRuMask[0] & mask) == mask) && (indexBit < numBitsInWord) )
	{
		indexBit++;
		mask = (1<<(indexBit));
	}
	if (indexBit >= numBitsInWord)
	{
		while( ((localRuMask[1] & mask) == mask) && (indexBit < lastValidBitIndexInBitmap) )
		{
				indexBit++;
				mask = (1<<(indexBit));
		}
		if (indexBit >= lastValidBitIndexInBitmap)
		{
			firstBitmapIndex = RU_ALLOC_INVALID_BIT_INDEX;		
		}
		else
		{
			firstBitmapIndex = indexBit;
		}
	}
	else
	{
		firstBitmapIndex = indexBit;
	}

	return(firstBitmapIndex);
}


/**********************************************************************************

planManagerRuAllocator_findSBIndex 

Description: This function converts bitmapIndex that should be filled to SBindex and RU index
  
Input: 
-----
	firstBitmapIndex
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerRuAllocator_findSBIndexRUIndex(uint8 firstBitmapIndex, PreAggSubBandIndex_e* currentSBindex, PreAggStartRuIndex_e* currentRUindex)
{
	if (firstBitmapIndex < FIRST_BIT_INDEX_SB1)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_0;
		*currentRUindex = firstBitmapIndex;
	}
	else if(firstBitmapIndex < FIRST_BIT_INDEX_SB2)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_1;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB1;
	}
	else if(firstBitmapIndex < FIRST_BIT_INDEX_SB3)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_2;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB2;
	}
	else if(firstBitmapIndex < FIRST_BIT_INDEX_SB4)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_3;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB3;
	}
	else if(firstBitmapIndex < FIRST_BIT_INDEX_SB5)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_4;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB4;
	}
	else if(firstBitmapIndex < FIRST_BIT_INDEX_SB6)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_5;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB5;
	}
	else if(firstBitmapIndex < FIRST_BIT_INDEX_SB7)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_6;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB6;
	}
	else if(firstBitmapIndex <= LAST_BIT_INDEX_SB7)
	{
		*currentSBindex = PRE_AGG_SUB_BAND_7;
		*currentRUindex = firstBitmapIndex-FIRST_BIT_INDEX_SB7;
	}
	else //not in range
	{
		ILOG0_D("[planManagerRuAllocator_findSBIndexRUIndex] firstBitmapIndex = %d out of range!!!", firstBitmapIndex);
		FATAL("[planManagerRuAllocator_findSBIndexRUIndex] firstBitmapIndex out of range!!!");
	}
#ifdef DEBUG_RU_ALLOCATOR		
	ILOG0_DDD("[planManagerRuAllocator_findSBIndexRUIndex] firstBitmapIndex = %d, currentSBindex = %d, currentRUindex = %d", firstBitmapIndex, *currentSBindex, *currentRUindex);
#endif
}


/**********************************************************************************

planManagerRuAllocator_configureDummyUser 

Description: This function updates dummy user given SB index, RU index and RU size
  
Input: 
-----
	dlDataPhasPerUserPtr
	currentSBindex - SB index of dummy user
	currentRUindex - RU index of dummy user
	ruSize	 	   - RU size of dummy user
	ldpcDummyOn	   - LDPC for dummy user should be configured same as coding type of non-dummy user; Currently waver given to HW that all users must have the same coding type
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerRuAllocator_configureDummyUser(UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, DlDataPhaseUserPhase_t* dlDataPhasPerUserPtr, PreAggSubBandIndex_e currentSBindex, PreAggStartRuIndex_e currentRUindex, PreAggRuSizeMapping_e ruSize, bool ldpcDummyOn, uint8 userInputIndex)
{
	dlDataPhasPerUserPtr->staId =  0;
	dlDataPhasPerUserPtr->userTcraStaAid = DUMMY_AID; 

	dlDataPhasPerUserPtr->userTcraSubBand = currentSBindex;
	dlDataPhasPerUserPtr->userTcraStartRu = currentRUindex;
	dlDataPhasPerUserPtr->userTcraRuSize = ruSize;	
	dlDataPhasPerUserPtr->userTcraLdpc = ldpcDummyOn;

	if ((staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE) || (staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_IMM_ACK) || (staticPlanConfigurationPtr->commonSection.muSequenceType == HE_MU_SEQ_VHT_LIKE_PROTECTION))
	{
		dlDataPhasPerUserPtr->mapTxUserToRxUser =  0x0; // ba is always in user 0 (same as in su)
	}
	else
	{
		dlDataPhasPerUserPtr->mapTxUserToRxUser =  userInputIndex; //Mapping is 1 X 1 for Z0
	}
	ILOG0_D("[planManagerRuAllocator_configureDummyUser], ldpcDummyOn = %d", ldpcDummyOn);
	ILOG0_DDDD("[planManagerRuAllocator_configureDummyUser], dlDataPhasPerUserPtr = %x, currentSBindex = %d, currentRUindex = %d, ruSize = %d", dlDataPhasPerUserPtr, currentSBindex,  currentRUindex, ruSize);
	ILOG0_DD("[planManagerRuAllocator_configureDummyUser], staticPlanConfigurationPtr->commonSection.muSequenceType = %d, dlDataPhasPerUserPtr->mapTxUserToRxUser = %d", staticPlanConfigurationPtr->commonSection.muSequenceType, dlDataPhasPerUserPtr->mapTxUserToRxUser);
	ILOG0_V("[planManagerRuAllocator_configureDummyUser] end");
}


/**********************************************************************************

planManagerRuAllocator_updateRuBitmap 

Description: This function updates bitmap according to allocation that was filled
  
Input: 
-----
	ruSize
	firstBitmapIndex	
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerRuAllocator_updateRuBitmap(PreAggRuSizeMapping_e ruSize, uint8 firstBitmapIndex)
{
	uint8 index80MHzBand;
	uint8 lastValidBitIndexInBitmap = RuSizeUserMaskLength[PRE_AGGREGATOR_996_TONE_RU_SIZE]; //37

	if (firstBitmapIndex < lastValidBitIndexInBitmap)
	{
		index80MHzBand = 0;
	}
	else // (firstBitmapIndex > lastValidBitIndexInBitmap) update high 80MHz band
	{
		index80MHzBand = 1;
		// normalize bitlocation in case 2nd 80MHz band is used
		firstBitmapIndex = firstBitmapIndex - lastValidBitIndexInBitmap;
	}
	if (ruSize < PRE_AGGREGATOR_996X2_TONE_RU_SIZE)
	{
		planManagerRuAllocator_updateRuBitmapPer80(ruSize, firstBitmapIndex, index80MHzBand);
	}
	else
	{
		//RU size equals whole 160MHz therefore need to update all bitmap to '1'
		RuAllocBitmap[index80MHzBand].ruIndexMask64bit[0] = BITMAP_ALLOC_BW80_2_20MHZ;
		RuAllocBitmap[index80MHzBand].ruIndexMask64bit[1] = BITMAP_ALLOC_BW80_2_20MHZ;
	}
}


/**********************************************************************************

planManagerRuAllocator_updateRuBitmapPer80 

Description: This function updates bitmap according to allocation that was filled
  
Input: 
-----
	ruSize
	firstBitmapIndex	
	index80MHzBand	
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void	planManagerRuAllocator_updateRuBitmapPer80(PreAggRuSizeMapping_e ruSize, uint8 firstBitmapIndex, uint8 index80MHzBand)
{
	uint8 iterNum;
	uint8 bitNumRuSize = RuSizeUserMaskLength[ruSize];
	uint32* pRuIndexMask64bit = &(RuAllocBitmap[index80MHzBand].ruIndexMask64bit[0]);
#ifdef DEBUG_RU_ALLOCATOR	
	ILOG0_D("[planManagerRuAllocator_updateRuBitmapPer80] start, index80MHzBand = %d", index80MHzBand);
#endif	

	for(iterNum = firstBitmapIndex; iterNum < firstBitmapIndex+bitNumRuSize; iterNum++)
	{
		Utils_SetBitInBitmap(pRuIndexMask64bit, iterNum);
	}
#ifdef DEBUG_RU_ALLOCATOR	
	SLOG0(0, 0, RuAllocMask_t, &(RuAllocBitmap[index80MHzBand].ruIndexMask64bit));
	ILOG0_V("[planManagerRuAllocator_updateRuBitmapPer80] end");
#endif	
}


/**********************************************************************************

planManagerSetStaticDlPhaseCommonFieldValidUser 

Description: signal validuser bitmap field in phase common section according to number of participating stations (including dummy users)
  
Input: 
-----
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void planManagerSetStaticDlPhaseCommonFieldValidUser(DlDataPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	uint8	loopIndex; 
	uint32	validUsersBitmap = 0x0;

	//static planner should support upto 8 clients
	if (staticPlanConfigurationPtr->commonSection.numOfParticipatingStations > HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP)
	{
		FATAL("[planManagerSetStaticDlPhaseCommonFieldValidUser], Number of stations configured in group exceeds allowed max number of stations!!");
	}

	for(loopIndex = 0; loopIndex < staticPlanConfigurationPtr->commonSection.numOfParticipatingStations; loopIndex++)
	{
		validUsersBitmap |= 1<<loopIndex; 
	}
	phaseCommonPtr->validUsers310 = validUsersBitmap; 		
	// TBD if HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP > 32 remove FATAL above and fill USERs indexes which are greater than 31 (in field validUsers3532)
}


/**********************************************************************************

planManagerCalcRuAllocationSecondaryBit 
  
Input: 
-----

		
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND
static	bool	planManagerCalcRuAllocationSecondaryBit(uint8 bw,uint8 ulSubBand,uint8 ulRuSize)
{
	bool isSecondaryFlag = FALSE; 
	bool isPrimarySubBandInLower80 = FALSE; 


	if(bw == BW_160MHZ_160MHZ)
	{
		if(ulRuSize == PRE_AGGREGATOR_996X2_TONE_RU_SIZE)
		{
			isSecondaryFlag = TRUE;
		}
		else
		{
			isPrimarySubBandInLower80 =  planManagerIsPrimarySubBandInLower80(); 
			// only zero bit is taken, all other bits are being override by the TF generator according to RCR value 
			if(isPrimarySubBandInLower80) // primary channel index is in range 0-3
			{
				if(ulSubBand & THIRD_BIT_MASK) // check if 3'd bit is on (sub band is in range of 4-7 )
				{
					/* it means sub-band is on the secondary channel - need to turn the bit on*/
					isSecondaryFlag = TRUE; 
				}
				//else	- if sub band is on primary - remain with FALSE 
			}
			else //  primary channel index is in range of 4-7
			{
				if(!(ulSubBand & THIRD_BIT_MASK)) // check if 3'd bit is off (sub band is in 0-3 range) 
				{
					/* it means sub-band is on the secondary channel - need to turn the bit on*/
					isSecondaryFlag = TRUE; 
				}
				//else	- if sub band is on primary - remain with FALSE 
			}
		}
	}
	return isSecondaryFlag; 	
}
#endif //#ifdef	TRIGGER_FRAME_GENERATOR_RU_ALLOCATION_WORKAROUND



#ifdef ENET_INC_ARCH_WAVE600D2

/**********************************************************************************

planManagerSetNfrpPhase

Description: 
Sets NFRP phase parameters 
	
**********************************************************************************/
static void planManagerSetNfrpPhase(NfrpPhase_t* nfrpPhasePtr, CreatePlanMsg_t* pMuCreatePlanReq, CommonPlan_t* planPtr, uint8 vapId, uint8 groupId)
{
    /* TBD */
    UNUSED_PARAM(nfrpPhasePtr);
    UNUSED_PARAM(pMuCreatePlanReq);
    UNUSED_PARAM(planPtr);
    UNUSED_PARAM(vapId);
    UNUSED_PARAM(groupId);
}


/**********************************************************************************

planManagerSetNfrpPhase 

Description: 
Sets NFRP phase parameters 
	
**********************************************************************************/
static void planManagerSetStaticNfrpPhase(NfrpPhase_t* nfrpPhasePtr, UMI_STATIC_PLAN_CONFIG* pMuPlanConfig, CommonPlan_t* planPtr, uint8 vapId, HeGroupManagerStaticGroupId_e groupId)
{
	/*Fill NFRP phase common fields */
	planManagerSetStaticNfrpPhaseCommonFields(&(nfrpPhasePtr->common), pMuPlanConfig,groupId);
	/*Fill NFRP phase TCR B common fields  (NFRP TCR B it's the same as UL)  */
	planManagerSetStaticUlPhaseCommonTcr_B_Fields((UlDataPhaseCommonTcrb_t*)&(nfrpPhasePtr->commonTcrb), pMuPlanConfig,planPtr, vapId);
	/*Fill UL phase RCR common fields */
	planManagerSetStaticNfrpPhaseCommonRcrFields((RcrCommon_t*)&(nfrpPhasePtr->commonRcr), pMuPlanConfig, vapId);
	/*	Fill NFRP phase TCR B common user fields  (NFRP TCR B it's the same as UL) */
	planManagerSetStaticUlPhaseCommonUserTCR_B_Fields((UlDataPhaseCommonUserTcrb_t*)&(nfrpPhasePtr->commonUserTcrb), pMuPlanConfig);	
	//Fill Data TCR per User parameters
	planManagerSetStaticNfrpPhasePerUser(&(nfrpPhasePtr->userPhase), pMuPlanConfig);
	//Fill TF common parameters	, the same as UL	
	planManagerSetStaticUlPhaseCommonTfParameters((UlDataPhaseTfParameters_t*)&(nfrpPhasePtr->tfParameters), (UlDataPhaseTfCommonInfo_t*)&(nfrpPhasePtr->tfCommonInfo) ,pMuPlanConfig);
	//Fill TF per User	parameters
	planManagerSetStaticNfrpPhaseTfInfoPerUser(&(nfrpPhasePtr->tfUserInfo), pMuPlanConfig);	
}


/**********************************************************************************

planManagerSetStaticNfrpPhaseCommonFields 

Description: 
Sets the NFRP common section in NFRP phase 
	
**********************************************************************************/	
static void planManagerSetStaticNfrpPhaseCommonFields(NfrpPhaseCommon_t* phaseCommonPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, HeGroupManagerStaticGroupId_e groupId)
{	
	uint16 startingAid     = 0;
	uint8 multiplexingFlag = 0;
	phaseCommonPtr->validUsers310 						= planManagerDefaultSettings.NfrpCommonPhase_ValidUsers310; // only one user at NFRP	
	phaseCommonPtr->validUsers3532 						= NOT_SUPPORT;
	phaseCommonPtr->phaseType 	  						= PHASE_TYPE_NDP_FEEDBACK; 		// NFRP
	
	phaseCommonPtr->oneShotPhase 						= SUPPORT; 	
	phaseCommonPtr->partialDuration 					= NOT_SUPPORT;						
	phaseCommonPtr->multiBssid							= heGroupManagerDb[groupId].mbssRxCtrlSupport; 
	phaseCommonPtr->broadcastTfAidCfg 					= SUPPORT; 				 
	phaseCommonPtr->unicastTfTypeCfg  					= SUPPORT;
	phaseCommonPtr->keepPlanLock 						= NOT_SUPPORT;
	phaseCommonPtr->moveStaToSu 						= NOT_SUPPORT;
	phaseCommonPtr->tfParamTriggerType 					= TF_TYPE_NFRP;
	
/*	phaseCommonPtr->actionBitmap 						= 0x0008; //bit3 :planned_ul_tf = nfrp bit ON. like ul. 	*/	
	phaseCommonPtr->plannedUlTf							= planManagerDefaultSettings.UlCommon_plannedUlTf;  //// 4th bit in action bitmap field 
	
	phaseCommonPtr->minimumNumberOfUsersForTransmission = planManagerDefaultSettings.NfrpCommonPhase_MinimumNumberOfUsersForTransmission;
	phaseCommonPtr->tfInfoBaseAddrDwOffset 				= CONVERT_BYTES_TO_WORDS(OFFSETOF(NfrpPhase_t, tfParameters));    // offset from phase start to tfParameters
	phaseCommonPtr->groupIndex 							= groupId;	
	startingAid 										= staticPlanConfigurationPtr->commonSection.startingAid;
	phaseCommonPtr->bw20StartingAid						= startingAid;
	phaseCommonPtr->bw40StartingAid 					= startingAid;
	phaseCommonPtr->bw80StartingAid 					= startingAid;	
	phaseCommonPtr->bw160StartingAid 					= startingAid;
	multiplexingFlag 									= staticPlanConfigurationPtr->commonSection.multiplexingFlag;
	phaseCommonPtr->bw20MultiplexFlag 					= multiplexingFlag;
	phaseCommonPtr->bw40MultiplexFlag					= multiplexingFlag;
	phaseCommonPtr->bw80MultiplexFlag  					= multiplexingFlag;
	phaseCommonPtr->bw160MultiplexFlag 					= multiplexingFlag;



}


/**********************************************************************************

planManagerSetStaticUlPhaseCommonRcrFields 

Description: 
Sets the RCR common section in UL pahse 
	
**********************************************************************************/
static void planManagerSetStaticNfrpPhaseCommonRcrFields(RcrCommon_t*  phaseCommonRcrPtr, UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr, uint8 vapId)
{
	VapDb_t* pVapDbHwEntry = &VapDbHwEntries[vapId]; 

	phaseCommonRcrPtr->heSigABssColor  =  pVapDbHwEntry->common.tcr1BssColor; 
	phaseCommonRcrPtr->cbw             =  staticPlanConfigurationPtr->commonSection.startBwLimit; 
	
	phaseCommonRcrPtr->tbMode             = planManagerDefaultSettings.NfrpCommonPhaseRcr_TBmode;
	phaseCommonRcrPtr->ndpFbMultiplexing  = staticPlanConfigurationPtr->commonSection.multiplexingFlag; 		
	phaseCommonRcrPtr->nHeltf 			  = planManagerDefaultSettings.NfrpCommonPhaseRcr_nHeltf;
	phaseCommonRcrPtr->heCp  			  = planManagerDefaultSettings.NfrpCommonPhaseRcr_heCp; 
	phaseCommonRcrPtr->heLtf  			  = planManagerDefaultSettings.NfrpCommonPhaseRcr_heLtf;  
	phaseCommonRcrPtr->stbc  			  = planManagerDefaultSettings.NfrpCommonPhaseRcr_stbc;
	
	phaseCommonRcrPtr->heSigASpatialReuse = staticPlanConfigurationPtr->commonSection.rcr_heSigASpatialReuse; 
	phaseCommonRcrPtr->lSigLength 		  = planManagerDefaultSettings.NfrpCommonPhaseRcr_lSigLength;

}


/**********************************************************************************

planManagerSetStaticNfrpPhasePerUser 

Description: 
Sets NFRP Phase per user
	
**********************************************************************************/
static void planManagerSetStaticNfrpPhasePerUser(NfrpPhaseUserPhase_t* phasPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{

	phasPerUserPtr->userPhaseReserve00 	= NOT_SUPPORT;  	
	phasPerUserPtr->tfPadding 			= staticPlanConfigurationPtr->perUserParameters[0].tfPadding; 
	phasPerUserPtr->userPhaseReserve01 	= NOT_SUPPORT;  

	
	phasPerUserPtr->userRcrSubBand 		= staticPlanConfigurationPtr->perUserParameters[0].ul_SubBand; 
	phasPerUserPtr->userRcrStartRu 		= staticPlanConfigurationPtr->perUserParameters[0].ul_StartRU; 
	phasPerUserPtr->userRcrRuSize 		= staticPlanConfigurationPtr->perUserParameters[0].ul_ruSize; 
	phasPerUserPtr->userRcrLdpc			= staticPlanConfigurationPtr->perUserParameters[0].ul_ldpc; 
	phasPerUserPtr->userRcrTxbf  		= planManagerDefaultSettings.UlPerUserTf_RcrTxbf; 
	// in psdu rate all 0x0
	phasPerUserPtr->userRcrPsduRate		= NOT_SUPPORT; 
	
	phasPerUserPtr->userRcrTargetRssi 	= staticPlanConfigurationPtr->perUserParameters[0].targetRssi;
	phasPerUserPtr->userRcrReserve00 	= NOT_SUPPORT;  
	phasPerUserPtr->userRcrValid 		= planManagerDefaultSettings.UlPerUserTf_RcrValid; 
	
}


/**********************************************************************************

planManagerSetStaticNfrpPhaseTfInfoPerUser


Description: 
Sets NFRP TF per user parameters
**********************************************************************************/
static void planManagerSetStaticNfrpPhaseTfInfoPerUser(NfrpPhaseTfUserInfo_t* tfInfoPerUserPtr,UMI_STATIC_PLAN_CONFIG* staticPlanConfigurationPtr)
{
	tfInfoPerUserPtr->tfStartingAid0 			  = staticPlanConfigurationPtr->commonSection.startingAid;
	tfInfoPerUserPtr->tfFeedbackType0 			  = staticPlanConfigurationPtr->commonSection.feedbackType;
	tfInfoPerUserPtr->tfTargetRssi0 			  = staticPlanConfigurationPtr->perUserParameters[0].targetRssi;
	tfInfoPerUserPtr->tfMultiplexingFlag0 		  = staticPlanConfigurationPtr->commonSection.multiplexingFlag;
	
	tfInfoPerUserPtr->tfReservedUserInfo00        = NOT_SUPPORT;	
	tfInfoPerUserPtr->tfReservedUserInfo10        = NOT_SUPPORT;	
	tfInfoPerUserPtr->tfReservedTriggerDependent0 = NOT_SUPPORT;	
	tfInfoPerUserPtr->tfReserved10                = NOT_SUPPORT;	

}



#endif

static PreAggRuSizeMapping_e searchClosestRuSize(uint8 bwVal)
{
	PreAggRuSizeMapping_e index = PRE_AGGREGATOR_NUM_OF_RU_SIZES-1;
	
	while ((bwVal < bwPerRuSize[index]) && (index != 0 ))
	{
		index--;	
	}
	return index;
}


/**********************************************************************************
PlanManager_Initialize 
  
Input: 
-----
		
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void PlanManager_Initialize(void)
{
	uint8	loopIterator; 

	memset(&RuAllocBitmap, 0, sizeof(RuAllocBitmap));
	memset(heMuPlansArray, 0, sizeof(CommonPlan_t) * MAX_NUMBER_OF_PLANS); 
	planManagerSetPlanManagerDefaultSettings(); //initialize default values for plan manager
	
	//allocate plan/phase resources 
	heMuResourcesPool.planPoolptr		= heMuPlansArray; 
	heMuResourcesPool.dataPhasePoolPtr	= heMuDataPhaseArray; 
	heMuResourcesPool.soundingPhasePoolPtr = heMuSoundingPhaseArray; 
	heMuResourcesPool.protectionPhasePoolPtr = heMuProtectionPhaseArray;   
#ifdef ENET_INC_ARCH_WAVE600D2
	heMuResourcesPool.nfrpPhasePoolPtr = heMuNfrpPhaseArray;   
#endif
	//prepare the plan elements pool along with the protection&sounding phases 
	for(loopIterator = 0; loopIterator < MAX_NUMBER_OF_PLANS; loopIterator++)
	{
		heMuPlansArray[loopIterator].nextPlanPointer = (uint32)(&(heMuPlansArray[loopIterator+1]));//first word in plan		
		heMuPlansArray[loopIterator].planIndex = loopIterator;
		// initialize ndpa pointer in sounding phase to ndpa array of the plan 
		heMuSoundingPhaseArray[loopIterator].common.ndpaPointer = CONVERT_TO_PHYSICAL_ADDR((uint32)(&(heSoundingNdpaArray[loopIterator])));
		// initialize the HE MU Plans Look Up Table
		heMuPlanLUT[loopIterator].planDbBaseAddr = (uint32) &heMuPlansArray[loopIterator];

		/*Allocte one sounding and one protection phase per each plan*/
		heMuSoundingPhaseArray[loopIterator].common.nextPhasePointer = (uint32)(&(heMuSoundingPhaseArray[loopIterator+1]));
		heMuProtectionPhaseArray[loopIterator].common.nextPhasePointer = (uint32)(&(heMuProtectionPhaseArray[loopIterator+1]));		
#ifdef ENET_INC_ARCH_WAVE600D2
		heMuNfrpPhaseArray[loopIterator].common.nextPhasePointer = (uint32)(&(heMuNfrpPhaseArray[loopIterator+1]));
#endif
	}
	//set the next of the last element with a NULL pointer 
	heMuPlansArray[MAX_NUMBER_OF_PLANS-1].nextPlanPointer= (uint32)NULL; 
	heMuSoundingPhaseArray[MAX_NUMBER_OF_PLANS-1].common.nextPhasePointer= (uint32)NULL; 
	heMuProtectionPhaseArray[MAX_NUMBER_OF_PLANS-1].common.nextPhasePointer= (uint32)NULL; 
#ifdef ENET_INC_ARCH_WAVE600D2
	heMuNfrpPhaseArray[MAX_NUMBER_OF_PLANS-1].common.nextPhasePointer= (uint32)NULL; 
#endif
	//prepare the data phase element pool 
	for(loopIterator = 0; loopIterator < MAX_NUMBER_OF_DATA_PHASES-1; loopIterator++)
	{
		heMuDataPhaseArray[loopIterator].dlDataPhaseStructure.common.nextOrMirrorPhasePointer = (uint32)(&(heMuDataPhaseArray[loopIterator+1]));//first word in plan   //first word in phase
	}
	heMuDataPhaseArray[MAX_NUMBER_OF_DATA_PHASES-1].dlDataPhaseStructure.common.nextOrMirrorPhasePointer = (uint32)NULL; 
}

