/***********************************************************************************
 File:			ProcessManagerCdb.c
 Module:		Process manager
 Purpose:		Coordinate and schedule all processes in the system
 Description:	This src file is the implementation of the process manager module 
				which is responsible for scheduling and prioritizing all processes 
				in the system.It is also the sole owner of all services related to 
				these processes.
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "stringLibApi.h"
#include "ProcessManager_Api.h"
#include "CalibrationManager.h"
#include "QueueUtility_Api.h"
#include "queue_utility.h"
#include "ProcessManagerCdb_Api.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_PROCESS_MANAGER 
#define LOG_LOCAL_FID 1

#define	PROCESS_MANAGER_NUMBER_OF_PENDING_PROCESSES 2


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum
{
	PROCESS_MANAGER_CDB_STATE_NOT_SUPPORTED,
	PROCESS_MANAGER_CDB_STATE_IDLE,
	PROCESS_MANAGER_CDB_STATE_DUAL_BAND_PROCESS_RUNNING,
	PROCESS_MANAGER_CDB_MAX_NUM_OF_STATES = 0xFF,
} processManagerCdBState_e;

typedef enum
{
	PROCESS_MANAGER_CDB_DUAL_BAND_PROCESS_EVENT,
	PROCESS_MANAGER_CDB_DUAL_BAND_PROCESS_ENDED_EVENT,
	PROCESS_MANAGER_CDB_NUM_OF_EVENTS = 0xFF,
} processManagerCdbEvents_e;

typedef struct ProcessManagerCdbQueueBuffer_ ProcessManagerCdbQueueBuffer_t;
	
struct ProcessManagerCdbQueueBuffer_
{
	ProcessManagerCdbQueueBuffer_t *nextProcessDb;
	ProcessManagerCdbQueueBuffer_t *prevProcessDb;
	ProcessRequestParams_t processRequestParams;
};

typedef struct 
{
	ProcessManagerCdbQueueBuffer_t pmCdbPendingMesgsPool[PROCESS_MANAGER_NUMBER_OF_PENDING_PROCESSES];
	processManagerCdBState_e pmCdbstate;
	GeneralQueueBlock_t pendingProcessesFreeQueue;
	GeneralQueueBlock_t pendingProcessesQueue;

}processManagerCdbDb_t;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void pmCdbScheduledProcessReq(K_MSG *pMsg);
static void pmCdbProecssEnded(K_MSG *pMsg);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static processManagerCdbDb_t processManagerCdbDb;
static const FunctionEntry_t afpTaskTable[TASK_PROCESS_MANAGER_CDB_END-TASK_PROCESS_MANAGER_CDB_START]=
{
	{pmCdbScheduledProcessReq,	DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_CDB_PROCESS_REQUEST)},
	{pmCdbProecssEnded,			DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_CDB_PROCESS_ENDED)}

};


/*---------------------------------------------------------------------------------
/						Debug Section									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

ProcessManagerCdb_TaskEntry 

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/

void ProcessManagerCdb_TaskEntry(K_MSG *processManagerMsg)
{
	/* Use common task switching and Table */
	vTaskDispatcher(processManagerMsg, afpTaskTable, TASK_PROCESS_MANAGER_CDB_START, TASK_PROCESS_MANAGER_CDB_END);
}

/**********************************************************************************

ProcessManagerCdbInit 

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
void ProcessManagerCdbInit(void)
{
	uint8 i;

	/*Init Free pending messages Queue*/
	for(i=0; i < PROCESS_MANAGER_NUMBER_OF_PENDING_PROCESSES; i++)
	{
		GeneralQ_PushItemToTail(&processManagerCdbDb.pendingProcessesFreeQueue,(GeneralQueueItem_t *)&processManagerCdbDb.pmCdbPendingMesgsPool[i]);
	}
	memset(&processManagerCdbDb, 0, sizeof(processManagerCdbDb));
	processManagerCdbDb.pmCdbstate = PROCESS_MANAGER_CDB_STATE_IDLE;
}
#endif
/**********************************************************************************

pmCdbScheduledProcessReq 

Description:
------------
Handle dual core processes - before handle process manager request in band it will be handled here,
If state is PROCESS_MANAGER_CDB_STATE_IDLE -> process will schedule in process manager band, 
if state is PROCESS_MANAGER_CDB_STATE_DUAL_BAND_PROCESS_RUNNING, ProcessRequestParams from schedule message will be pending on queue and will be sent to process manager when previous pdual band process will be finished

Input:
-----
	pMsg  - schedule process message sent by task to process manager in band	

Returns:
--------
	None
	
**********************************************************************************/

static void pmCdbScheduledProcessReq(K_MSG *pMsg)
{
	ProcessRequestParams_t* processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);
	ProcessManagerCdbQueueBuffer_t* pPendingProcessBuffer;

	if (processManagerCdbDb.pmCdbstate == PROCESS_MANAGER_CDB_STATE_IDLE)
	{
		/*Change state  only if message is from dual band process type*/
		processManagerCdbDb.pmCdbstate = PROCESS_MANAGER_CDB_STATE_DUAL_BAND_PROCESS_RUNNING;
		processRequestParams->processMsgHandledByCdbProcessMan = TRUE;
		/*State is idle -> forward message back to process manager in band*/
		OSAL_SEND_MESSAGE(PROCESS_MANAGER_SCHEDULE_PROCESS_REQUEST,TASK_PROCESS_MANAGER,pMsg, pMsg->header.vapId);
	}
	else
	{
		/*Get buffer from queue*/
		pPendingProcessBuffer = (ProcessManagerCdbQueueBuffer_t*)GeneralQ_PopItemFromHead(&processManagerCdbDb.pendingProcessesFreeQueue);
		MEMCPY(&pPendingProcessBuffer->processRequestParams, processRequestParams, sizeof(ProcessRequestParams_t));
		/*Push pending proecss to queue*/
		GeneralQ_PushItemToTail(&processManagerCdbDb.pendingProcessesQueue,(GeneralQueueItem_t *)pPendingProcessBuffer);
	}
}
/**********************************************************************************

pmCdbProecssEnded 

Description:
------------
Sent dual band process end messageto requester task.
Handling dual band process:
If pendingProcessesQueue is not empty, pop dual band schedule process message from queue and sent to process manager to scheduled

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/

static void pmCdbProecssEnded(K_MSG *pMsg)
{
	ProcessRequestParams_t* processEndedReqParamsFromPmBand = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);
	K_MSG *pManagerExecututionFinishedMsg;
	ProcessRequestParams_t* execFinishReqParamsToRequester; 
	ProcessRequestParams_t *pPendingProcessRequestParams;
	K_MSG *processManagerSchedMsg;
	ProcessManagerCdbQueueBuffer_t* pPendingProcessBuffer;

	pManagerExecututionFinishedMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));
	execFinishReqParamsToRequester = (ProcessRequestParams_t*)pK_MSG_DATA(pManagerExecututionFinishedMsg);
	/*copy process request params from pending message to new process msg to process manager*/
	MEMCPY(execFinishReqParamsToRequester, processEndedReqParamsFromPmBand, sizeof(ProcessRequestParams_t));

	/*Send process execution message to requester task*/
	OSAL_SEND_MESSAGE(processEndedReqParamsFromPmBand->endProcessMsg,processEndedReqParamsFromPmBand->returnTask,pManagerExecututionFinishedMsg, processEndedReqParamsFromPmBand->vapId);
	
	/*Get buffer from queue*/
	pPendingProcessBuffer = (ProcessManagerCdbQueueBuffer_t*)GeneralQ_PopItemFromHead(&processManagerCdbDb.pendingProcessesQueue);
	if (pPendingProcessBuffer != NULL)
	{
		/*Queue is not empty - take message from pending queue and send to peocess manager*/			

		/* Schedule new process*/	
		processManagerSchedMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));
		pPendingProcessRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(processManagerSchedMsg);
		/*copy process request params from pending message to new process msg to process manager*/
		MEMCPY(pPendingProcessRequestParams, &pPendingProcessBuffer->processRequestParams, sizeof(ProcessRequestParams_t));
		/*Send new schedule process message to process manager*/
		OSAL_SEND_MESSAGE(PROCESS_MANAGER_SCHEDULE_PROCESS_REQUEST,TASK_PROCESS_MANAGER,processManagerSchedMsg, pPendingProcessRequestParams->vapId);
	}
	else
	{
		/*Queue is empty - just change state*/
		processManagerCdbDb.pmCdbstate = PROCESS_MANAGER_CDB_STATE_IDLE;
		
	}
}
