/***********************************************************************************
 File:			ProcessManager.c
 Module:		Process manager
 Purpose:		Coordinate and schedule all processes in the system
 Description:	This src file is the implementation of the process manager module 
				which is responsible for scheduling and prioritizing all processes 
				in the system.It is also the sole owner of all services related to 
				these processes.
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "stringLibApi.h"
#include "ProcessManager_Api.h"
#include "ServicesHandler_Api.h"
#include "bss_manage_task.h"
#include "CalibrationManager.h"
#include "CoC_Api.h"
#include "ErrorHandler_Api.h"
#include "BSSmanager_API.h"
#include "Utils_Api.h"
#include "QueueUtility_Api.h"
#include "queue_utility.h"
#include "Dut_Api.h"
#include "loggerAPI.h"
#include "Pauser_Api.h"
#include "CtsManager_Api.h"



/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_PROCESS_MANAGER 
#define LOG_LOCAL_FID 0

typedef enum
{
	PROCESS_MANAGER_STATE_IDLE,
	PROCESS_MANAGER_STATE_ACTIVE,
	PROCESS_MANAGER_STATE_NUM_OF_STATES, 
	PROCESS_MANAGER_STATE_MAX_NUM_OF_STATES = 0xFF,
} processManagerState_e;

typedef enum
{
	PROCESS_MANAGER_SCHEDULE_PROCESS_EVENT,
	PROCESS_MANAGER_PRE_PROCESS_SERVICES_FINISHED_EVENT,
	PROCESS_MANAGER_PROCESS_EXCECUTION_FINISHED_EVENT,
	PROCESS_MANAGER_POST_PROCESS_SERVICES_FINISHED_EVENT,
	PROCESS_MANAGER_NUM_OF_EVENTS,
	PROCESS_MANAGER_MAX_NUM_OF_EVENTS = 0xFF,
} processManagerEvents_e;

typedef enum
{
	PROCESS_STATE_IDLE,
	PROCESS_STATE_PENDING,
	PROCESS_STATE_PRE_PROCESS_SERVICES_RUNNING,
	PROCESS_STATE_RUNNING,
	PROCESS_STATE_POST_PROCESS_SERVICES_RUNNING,
	PROCES_STATE_NUM_OF_STATES,
	PROCES_STATE_MAX_NUM_OF_STATES = 0xFF,
} processState_e;
typedef enum
{
	DUAL_BAND_PROCESS_STATE_IDLE,
	DUAL_BAND_PROCESS_PENDING,
	DUAL_BAND_PROCESS_READY,
		
} dualBandProcessState_e;


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct ProcessDb_ ProcessDb_t;
	
struct ProcessDb_
{
	ProcessDb_t *nextProcessDb;
	ProcessRequestParams_t processRequestParams;
	processState_e processState;
	bool dualBandProcessEnable;
};
typedef struct ProcessManagerParams_
{
	processManagerState_e state;
	GeneralQueueBlock_t pendingProcessesQueue;
	ProcessDb_t *currentRunningProcess;
} ProcessManagerParams_t;

typedef void (*processPreemptioAction_t)(ProcessDb_t *);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void processManagerScheduleProcessRequest(K_MSG* pMsg);
static void processManagerExecutionCB(K_MSG* pMsg);
static void processManagerRunServiceCB(K_MSG* pMsg);
static void processManagerStartProcess(ProcessDb_t *processDbEntry);
static void processManagerQueueProcess(ProcessDb_t *processDbEntry);
static void processManagerRejectProcess(ProcessDb_t *processDbEntry);
static void processManagerProcessExecutionFinished(ProcessDb_t*processDbEntry);
static void processManagerServicesExcecutionFinished(ProcessDb_t*processDbEntry);
static void processManagerClassifyProcessRequest(ProcessDb_t *processDbEntry);
static void processManagerScanPendingProcessRequests(void);
static void processManagerRunServices(uint32 *servicesBitMask);
static void processManagerCtsTxCfm(K_MSG *ctsCfmMessage);
static void processManagerCdbSyncServiceCfm(K_MSG *cdbSyncServiceMsg);
#if defined(HDK_CDB_SUPPORT)
static void sendDualBandProcessReqToDualBandProcessManager(ProcessRequestParams_t* processRequestParams);
#endif
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static const FunctionEntry_t afpTaskTable[TASK_PROCESS_MANAGER_END-TASK_PROCESS_MANAGER_START]=
{
	{processManagerScheduleProcessRequest,	DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_SCHEDULE_PROCESS_REQUEST)},
	{processManagerExecutionCB,				DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED)},
	{processManagerRunServiceCB,			DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_SERVICE_CB)},	
	{processManagerCtsTxCfm,				DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_CTS_TX_CFM)},
	{processManagerCdbSyncServiceCfm,		DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_CDB_SYNC_SERVICE_PROCESS_SCHEDULED_RES)},
	{processManagerCdbSyncServiceCfm,		DOUBLE_CHECK_MSG_TYPE(PROCESS_MANAGER_CDB_SYNC_SERVICE_CTS2SELF_SENT_RES)},

};

ProcessManagerParams_t		processManagerParams;
ProcessDb_t	processesDb[NUM_OF_PROCESSES];

/*---------------------------------------------------------------------------------
/						Debug Section									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
static void processManagerCtsTxCfm(K_MSG *ctsCfmMessage)
{
	CtsManagerTxCfmMessage_t *ctsCfmData = (CtsManagerTxCfmMessage_t *) ctsCfmMessage;
	ServicesHandler_ServiceCB ((ServicesStatus_e)!ctsCfmData->status); /*Status is TRUE or FALSE, Service Status is Passed = 0, Failed = 1*/
}
static void processManagerCdbSyncServiceCfm(K_MSG *cdbSyncServiceMsg)
{
	UNUSED_PARAM(cdbSyncServiceMsg);	
	ServicesHandler_ServiceCB (SERVICES_STAUTS_PASSED); /*Status is TRUE or FALSE, Service Status is Passed = 0, Failed = 1*/
}

/**********************************************************************************

processManagerScheduleProcessRequest 

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
static void processManagerScheduleProcessRequest(K_MSG* pMsg)
{
	ProcessRequestParams_t* processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);
	ProcessDb_t* processDbEntry;
	uint8 processId ;
	processId = processRequestParams->processId;
	processDbEntry = &processesDb[processId];

	ILOG0_DD("processManagerScheduleProcessRequest:Process Manager State = %x processId = %x",processManagerParams.state,processId);
	MEMCPY(&processDbEntry->processRequestParams,processRequestParams,sizeof(ProcessRequestParams_t));

								 
	switch(processManagerParams.state)
	{
		case PROCESS_MANAGER_STATE_IDLE:
#ifdef HDK_CDB_SUPPORT
			if ((processRequestParams->dualBandProcess == TRUE) 
				&& (processDbEntry->dualBandProcessEnable == TRUE)
				&& (processRequestParams->processMsgHandledByCdbProcessMan == FALSE))
			{
				/* Send dual band process request to dual band process manager task*/	
				sendDualBandProcessReqToDualBandProcessManager(processRequestParams);
			}
			else
#endif
			{
				processManagerParams.state = PROCESS_MANAGER_STATE_ACTIVE;
				processManagerParams.currentRunningProcess = processDbEntry;
				/* Send start Process to process SM */
				processManagerStartProcess(processDbEntry);
			}
			break;
			
		case PROCESS_MANAGER_STATE_ACTIVE:
#ifdef HDK_CDB_SUPPORT
			if ((processRequestParams->dualBandProcess == TRUE) 
				&& (processDbEntry->dualBandProcessEnable == TRUE)
				&& (processRequestParams->processMsgHandledByCdbProcessMan == FALSE))
			{
				/* Send dual band process request to dual band process manager task*/	
				sendDualBandProcessReqToDualBandProcessManager(processRequestParams);
			}
			else
#endif
			{
				/* Classify Process Request*/
				processManagerClassifyProcessRequest(processDbEntry);
			}
			break;
			
		default:
			DEBUG_FATAL("Process Manager Schedule Process Unknown State");
			break;
	}
}

/**********************************************************************************

processManagerProcessExcutionFinished 

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
static void processManagerExecutionCB(K_MSG* pMsg)
{
	ProcessRequestParams_t* processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);
	ProcessDb_t *processDbEntry = processManagerParams.currentRunningProcess;

	if (processDbEntry->processRequestParams.updateParamsBeforeFinalizing == TRUE)
	{
		MEMCPY(&processDbEntry->processRequestParams,processRequestParams,sizeof(ProcessRequestParams_t));
	}
	processManagerProcessExecutionFinished(processDbEntry);	
}


/**********************************************************************************

processManagerRunServiceCB 

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
static void processManagerRunServiceCB(K_MSG* pMsg)
{	
	ServiceReturnParams_t *serviceReturnParams = (ServiceReturnParams_t*)pK_MSG_DATA(pMsg);
	uint32 *remainingServicesBitMap;

	remainingServicesBitMap = (uint32 *)serviceReturnParams->serviceRequesterParams;
	processManagerRunServices(remainingServicesBitMap);	
}

/**********************************************************************************

processManagerStartProcess

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
static void processManagerStartProcess(ProcessDb_t *processDbEntry)
{
	ILOG0_D("processManagerStartProcess:Process Manager State = %x",processDbEntry->processState);
	switch(processDbEntry->processState)
	{
		case PROCESS_STATE_IDLE:
		case PROCESS_STATE_PENDING:
			processDbEntry->processState = PROCESS_STATE_PRE_PROCESS_SERVICES_RUNNING;
			processManagerRunServices(&processDbEntry->processRequestParams.preProcessServiceBitmap);
			break;
			
		default:
			DEBUG_FATAL("Process Manager Start Process Unknown State");
			break;		
	}
}

/**********************************************************************************

processManagerQueueProcess


Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
static void processManagerQueueProcess(ProcessDb_t *processDbEntry)
{
	switch(processDbEntry->processState)
	{
		case PROCESS_STATE_IDLE:
			processDbEntry->processState = PROCESS_STATE_PENDING;
			GeneralQ_PushItemToTail(&processManagerParams.pendingProcessesQueue,(GeneralQueueItem_t *)processDbEntry);
			break;
		default:
			DEBUG_FATAL("Process Manager Queue Process Unknown State");
			break;
	}
}

/**********************************************************************************

processManagerRejectProcess

Description:
------------
	

Input:
-----
	pMsg  - 		

Returns:
--------
	None
	
**********************************************************************************/
static void processManagerRejectProcess(ProcessDb_t *processDbEntry)
{
	K_MSG* processManagerReturnMsg;
	ProcessManagerReturnParams_t* processManagerReturnParams;
	ProcessRequestParams_t* processRequestParams = &processDbEntry->processRequestParams;
	

	switch(processDbEntry->processState)
	{
		case PROCESS_STATE_IDLE:
			processManagerReturnMsg = OSAL_GET_MESSAGE(sizeof(ProcessManagerReturnParams_t));
			processManagerReturnParams = (ProcessManagerReturnParams_t*)pK_MSG_DATA(processManagerReturnMsg);		
			processManagerReturnParams->processStatus = PROCESS_STATUS_REQUEST_REJECTED;
			processManagerReturnParams->requesterParams = processRequestParams->requesterParams;
			OSAL_SEND_MESSAGE(processRequestParams->startProcessMsg,processRequestParams->returnTask,processManagerReturnMsg, processRequestParams->vapId);
			break;
			
		default:
			DEBUG_FATAL("Process Manager Reject Process Unknown State");
			break;
	}			
}

/**********************************************************************************

	
processManagerProcessExcecutionFinished


Description:
------------
	

Input:
-----
	pMsg  - 		


	
**********************************************************************************/
static void processManagerProcessExecutionFinished(ProcessDb_t*processDbEntry)
{
	switch(processDbEntry->processState)
	{
		case PROCESS_STATE_RUNNING:
			processDbEntry->processState = PROCESS_STATE_POST_PROCESS_SERVICES_RUNNING;
			processManagerRunServices(&processDbEntry->processRequestParams.postProcessServiceBitmap);
			break;
		default:			
			DEBUG_FATAL("Process Manager Finish Process Unknown State");
			break;
	}
}

/**********************************************************************************

processManagerServicesExcecutionFinished


Description:
------------
	

Input:
-----
	pMsg  - 		


	
**********************************************************************************/
static void processManagerServicesExcecutionFinished(ProcessDb_t*processDbEntry)
{
	K_MSG* processManagerMsg;
	ProcessManagerReturnParams_t* processmanagerReturnParams;

	processManagerMsg = OSAL_GET_MESSAGE(sizeof(ProcessManagerReturnParams_t));
	processmanagerReturnParams = (ProcessManagerReturnParams_t *)pK_MSG_DATA(processManagerMsg);
	processmanagerReturnParams->requesterParams = processDbEntry->processRequestParams.requesterParams;
	switch(processDbEntry->processState)
	{
		case PROCESS_STATE_PRE_PROCESS_SERVICES_RUNNING:
			processDbEntry->processState = PROCESS_STATE_RUNNING;
			processmanagerReturnParams->processStatus = PROCESS_STATUS_REQUEST_ACCEPTED;
			OSAL_SEND_MESSAGE(processDbEntry->processRequestParams.startProcessMsg,processDbEntry->processRequestParams.returnTask,processManagerMsg, processDbEntry->processRequestParams.vapId);
			break;
			
		case PROCESS_STATE_POST_PROCESS_SERVICES_RUNNING:	
			processDbEntry->processState = PROCESS_STATE_IDLE;
			processmanagerReturnParams->processStatus = PROCESS_STATUS_EXECUTION_COMPLETED;
#ifdef HDK_CDB_SUPPORT
			if (processDbEntry->processRequestParams.processMsgHandledByCdbProcessMan == TRUE)
			{
				OSAL_SEND_MESSAGE(PROCESS_MANAGER_CDB_PROCESS_ENDED,TASK_PROCESS_MANAGER_CDB, processManagerMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
			}
			else
#endif
			{
			OSAL_SEND_MESSAGE(processDbEntry->processRequestParams.endProcessMsg,processDbEntry->processRequestParams.returnTask,processManagerMsg, processDbEntry->processRequestParams.vapId);
			}
			processManagerScanPendingProcessRequests();
			break;
		default:
			DEBUG_ASSERT(0);
			break;
	}
}

/**********************************************************************************

processManagerClassifyProcessRequest

Description:
------------
	

Input:
-----
	  - 		


	
**********************************************************************************/
static void processManagerClassifyProcessRequest(ProcessDb_t *processDbEntry)
{
    /* check invalid process request at current process manager queue */
    /* PROCESS_ID_ONLINE_CALIBRATION and PROCESS_ID_ERP can't run when a last remove vap is pending to start running,
       Pause requast of last vap remove is left paused, online calibaration will req a puase but will never get answered */ 
		if ((((processDbEntry->processRequestParams.processId == PROCESS_ID_ONLINE_CALIBRATION) ||
		  (processDbEntry->processRequestParams.processId == PROCESS_ID_ERP)||(processDbEntry->processRequestParams.processId == PROCESS_ID_DUTY_CYCLE)) &&
			((processesDb[PROCESS_ID_REMOVE_VAP].processState != PROCESS_STATE_IDLE) && // PROCESS_ID_REMOVE_VAP req for process manager started
			 (processesDb[PROCESS_ID_REMOVE_VAP].processRequestParams.serviceData == 0))) ||// serviceData == numOfActiveVaps (we need last remove vap)
				((processDbEntry->processRequestParams.processId == PROCESS_ID_ONLINE_CALIBRATION) && 
				(processesDb[PROCESS_ID_CHANNEL_SWITCH].processState != PROCESS_STATE_IDLE))) // during channel switch we want to disable online cal at the current band and cpntinue with the other band if enabled
    {
        processManagerRejectProcess(processDbEntry);
    }
    else
    {
        processManagerQueueProcess(processDbEntry);
    }
}

/**********************************************************************************

processManagerScanPendingProcessRequests 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	void
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void processManagerScanPendingProcessRequests(void)
{
	ProcessDb_t* processDbEntry;	

	processDbEntry = (ProcessDb_t *)GeneralQ_PopItemFromHead(&processManagerParams.pendingProcessesQueue); 
	if( processDbEntry != NULL)
	{	
		processManagerParams.state = PROCESS_MANAGER_STATE_ACTIVE;
		processManagerParams.currentRunningProcess = processDbEntry;
		/* start Process PreProcess Services Routines */
		processManagerStartProcess(processDbEntry);
	}
	else
	{
		processManagerParams.currentRunningProcess = NULL;
		processManagerParams.state = PROCESS_MANAGER_STATE_IDLE;
	}
				//OSAL_SEND_MESSAGE(discardedProcessRequestParams->returnTask,discardedProcessRequestParams->startProcessMsg,processManagerMsg);	
}

/**********************************************************************************

processManagerRunServices 

Description:
------------
<Description of the purpose of the function>

Input:
-----
	uint8 servicesBitMask
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void processManagerRunServices(uint32 *servicesBitMask)
{
	ServiceRequestParams_t serviceRequestParamas;
	ServicesId_e nextServiceId;
	ProcessDb_t*processDbEntry  = processManagerParams.currentRunningProcess;
	ILOG0_D("processManagerRunServices:servicesBitMask = %x",*servicesBitMask);
#ifdef LOG_PROCESS_MANAGER		
	ILOG0_D("processManagerRunServices:servicesBitMask = %x",*servicesBitMask);
#endif
	if(*servicesBitMask != 0)
	{
		nextServiceId = Utils_CountTrailingZeros(*servicesBitMask);
		*servicesBitMask &= (~(1 << nextServiceId));
		serviceRequestParamas.returnTask = TASK_PROCESS_MANAGER;
		serviceRequestParamas.returnMsg = PROCESS_MANAGER_SERVICE_CB;
		serviceRequestParamas.vapIndex = GET_DEFAULT_VAP_FOR_MY_BAND();		
		serviceRequestParamas.serviceRequesterParams = (void *)servicesBitMask;
		serviceRequestParamas.serviceId	= nextServiceId;
		serviceRequestParamas.serviceData = processDbEntry->processRequestParams.serviceData;
		ServicesHandler_RunService(&serviceRequestParamas);
	}
	else
	{
		processManagerServicesExcecutionFinished(processDbEntry);
	}
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

ProcessManager_TaskEntry 

Description:
------------
	
	
Input:
-----
	
**********************************************************************************/
void ProcessManager_TaskEntry(K_MSG *processManagerMsg)
{
	/* Use common task switching and Table */
	vTaskDispatcher(processManagerMsg, afpTaskTable, TASK_PROCESS_MANAGER_START, TASK_PROCESS_MANAGER_END);
}

K_MSG *ProcessManager_GetDummyMessage(void)
{
	K_MSG *dummyMsg = NULL;
	ProcessRequestParams_t *dummyStruct = NULL;
	dummyMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));
	dummyStruct = (ProcessRequestParams_t*)pK_MSG_DATA(dummyMsg);
	dummyStruct->preProcessServiceBitmap = 0xCAFECAFE;
	dummyStruct->postProcessServiceBitmap = 0xCAFECAFE;
	dummyStruct->serviceData = 0xCAFECAFE;
	dummyStruct->startProcessMsg = 0xCAFE;
	dummyStruct->endProcessMsg = 0xCAFE;
	dummyStruct->returnTask = 0xCA;
	dummyStruct->processId = 0xFE;
	return dummyMsg;
}


/**********************************************************************************

ProcessManager_Init 

Description:
------------
	Initializes all Process manger structures
Input:
-----
	void
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void ProcessManager_Init(void)
{
	memset(processesDb,0,NUM_OF_PROCESSES * sizeof(ProcessDb_t));
#ifdef HDK_DUAL_BAND_PROCESS_MECHANISM_ENABLED
	processesDb[PROCESS_ID_ONLINE_CALIBRATION].dualBandProcessEnable = TRUE;
	processesDb[PROCESS_ID_CHANNEL_SWITCH].dualBandProcessEnable = TRUE;
	processesDb[PROCESS_ID_CALIBRATE].dualBandProcessEnable = TRUE;
	processesDb[PROCESS_ID_HDK_USER_DEMAND].dualBandProcessEnable = TRUE;
	processesDb[PROCESS_ID_COC].dualBandProcessEnable = TRUE;
	processesDb[PROCESS_ID_CHANGE_RADIO_STATE].dualBandProcessEnable = TRUE;
	processesDb[PROCESS_ID_ERP].dualBandProcessEnable = TRUE;
#endif	
	memset(&processManagerParams,0,sizeof(ProcessManagerParams_t));			
	processManagerParams.state = PROCESS_MANAGER_STATE_IDLE;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

#ifdef HDK_CDB_SUPPORT
static void sendDualBandProcessReqToDualBandProcessManager(ProcessRequestParams_t* processRequestParams)
{
	ProcessRequestParams_t* dualBandProcessRequestParams;
	K_MSG* processSchedMsgDualBand;
	
	/* Send dual band process request to dual band process manager task*/	
	processSchedMsgDualBand = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));
	dualBandProcessRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(processSchedMsgDualBand);
	/*Copy request params from original process schedule process message to new message to be sent to dual band process manager*/
	MEMCPY(dualBandProcessRequestParams, processRequestParams, sizeof(ProcessRequestParams_t));
	/* Send dual band process schedule request */
	OSAL_SEND_MESSAGE(PROCESS_MANAGER_CDB_PROCESS_REQUEST,TASK_PROCESS_MANAGER_CDB,processSchedMsgDualBand, GET_DEFAULT_VAP_FOR_MY_BAND());
}
#endif
