/***************************************************************
 File:		PsManager.c
 Module:	  	Powe Save Manager
 Purpose: 	Driver for PS Manager
 Description:  This module contains the driver for PS Manager HW Module
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "PsManager_Api.h"
#include "PsSettingRegs.h"
#include "ShramPsManager.h"
#include "PsSettingFifoShram_Descriptors.h"
#include "HwQManager_API.h"
#include "Pac_Api.h"
#include "PacketDescriptor.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#include "ResourceManager_API.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PS_MANAGER
#define LOG_LOCAL_FID 1

// Global Enable
#define PS_SETTING_DISABLE					(0) 
#define PS_SETTING_ENABLE 					(1) 

// Ps Settings Mode
#define PS_SETTING_MODE_SELECTIVE_REQUEST	(0) 
#define PS_SETTING_MODE_FULL_REQUEST 		(1) 


#define PS_SETTING_PS_STATUS_NOTIFICATION_DISABLE_ALL_STATIONS			(0) 
#define PS_SETTING_PS_STATUS_NOTIFICATION_ENABLE_ALL_STATIONS			(0xFFFFFFFF) 

#define PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_DISABLE			(0) 
#define PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_ENABLE			(1) 

#define PS_SETTING_PS_POLL_FC_PM_ZERO_NOTIFICATION_DISABLE		(0) 
#define PS_SETTING_PS_POLL_FC_PM_ZERO_NOTIFICATION_ENABLE		(1) 

#define PS_SETTING_PS_POLL_NO_MATCH_AID_NOTIFICATION_DISABLE	(0) 
#define PS_SETTING_PS_POLL_NO_MATCH_AID_NOTIFICATION_ENABLE		(1) 

#define PS_SETTINGS_CLEAR_STRB 	(1)

#define PS_SETTING_START_NEW_RX_SESSION_WHILE_WAIT_AUTO_RESP_IRQ_DIS (0) 
#define PS_SETTING_START_NEW_RX_SESSION_WHILE_WAIT_AUTO_RESP_IRQ_EN (1) 

#define PS_SETTING_START_NEW_RX_SESSION_WHILE_PS_SETTING_ACTIVE_IRQ_DIS (0)
#define PS_SETTING_START_NEW_RX_SESSION_WHILE_PS_SETTING_ACTIVE_IRQ_EN (1)

#define PS_SETTING_FIFO_DECREMENT_LESS_THAN_ZERO_IRQ_DIS (0)
#define PS_SETTING_FIFO_DECREMENT_LESS_THAN_ZERO_IRQ_EN (1)

#define PS_SETTING_FIFO_FULL_DROP_IRQ_DIS (0)
#define PS_SETTING_FIFO_FULL_DROP_IRQ_EN (1)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct psManagerDb_
{
	uint16	fifoReadIndex;					// Next entry to provide to FW		
	uint16 fifoFullCounter;
	
} psManagerDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void psManagerCheckFifoError(void);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
psManagerDb_t PsManagerDb;
PsSettingFifoShram_t* pPsManagerStatusFifo;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
psManagerCheckFifoError 

Description:
------------
	Check Ps Setting Hw error scenarios
	 	1) Drooped events due to fifo full
	 	2) Fifo num of entries decreased to less than zero (Synch problem between Sw and Hw)
Input: 
-----
	None
Returns:
--------
	None 
**********************************************************************************/
void psManagerCheckFifoError(void)
{
	RegPsSettingPsFifoDebug_u	fifoDebugReg;
	RegPsSettingPsFifoClearStrb_u psFifoClearStrbReg;

	psFifoClearStrbReg.val = 0;
	
	//psManagerCheckFifoError	
	RegAccess_Read(REG_PS_SETTING_PS_FIFO_DEBUG,&fifoDebugReg.val);

	/* Dropped Events when fifo was full*/
	PsManagerDb.fifoFullCounter += fifoDebugReg.bitFields.psFifoFullDropCtr;

	// Clear fifo full drop counter
	psFifoClearStrbReg.bitFields.psFifoClearFullDropCtrStrb = PS_SETTINGS_CLEAR_STRB;
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_CLEAR_STRB,psFifoClearStrbReg.val);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
PsManager_GetFifoParameters 

Description:
------------
	
Output: 
-----
	fifoParams:
		1) Current Fifo Index
		2) Num of Entries in fifo waiting for FW processing

Returns:
--------
	None 
**********************************************************************************/
void PsManager_GetFifoParameters(PsManagerFifoParams_t *fifoParams)
{
	RegPsSettingPsFifoNumEntriesCount_u fifoNumOfEntriesCountReg;

	RegAccess_Read(REG_PS_SETTING_PS_FIFO_NUM_ENTRIES_COUNT,&fifoNumOfEntriesCountReg.val);
	
	fifoParams->currentIndexInFifo = PsManagerDb.fifoReadIndex;
	fifoParams->numOfValidEntries = fifoNumOfEntriesCountReg.bitFields.psFifoNumEntriesCount;

	ILOG2_D("PsManager_GetFifoParameters: fifoParams=%x",*(uint32 * )fifoParams);

	psManagerCheckFifoError();
}

/**********************************************************************************
PsManager_SetFifoParameters 

Description:
------------
	Sets the Fifo Params after FW finished processing all entries in fifo
Input: 
-----
	fifoParams
		1) Current Fifo Index
		2) Num of Entries in fifo waiting for FW processing

Returns:
--------
	None 
**********************************************************************************/	
void PsManager_SetFifoParameters(PsManagerFifoParams_t *fifoParams)

{
	RegPsSettingPsFifoRdEntriesNum_u fifoRdEntriesNumReg;

	ILOG2_D("PsManager_SetFifoParameters: fifoParams=%x",*(uint32 * )fifoParams);

	fifoRdEntriesNumReg.bitFields.psFifoRdEntriesNum = fifoParams->numOfValidEntries;

	// Decrement  Number Of Entries In Fifo 
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_RD_ENTRIES_NUM,fifoRdEntriesNumReg.val);

	PsManagerDb.fifoReadIndex = fifoParams->currentIndexInFifo;
}

/**********************************************************************************

PsManager_PsSettingsFifoNotEmpty 

Description:
------------
	Read and process Ps Setting Fifo 
Input: 
-----	
	psManagerMessage - Empty Message (No Data Message)
	

Returns:
--------
	void - 
	
**********************************************************************************/
void PsManager_PsSettingsFifoNotEmpty(void)
{
	PsManagerFifoParams_t psManagerFifoParams;
	PsSettingFifoShram_t psManagerStatusFifoEntry;
	uint8 fifoIndex;
	
	PsManager_GetFifoParameters(&psManagerFifoParams);

	for(fifoIndex = 0; fifoIndex < psManagerFifoParams.numOfValidEntries; fifoIndex++)
	{
		psManagerStatusFifoEntry = pPsManagerStatusFifo[psManagerFifoParams.currentIndexInFifo];

		SLOG0(0, 0, PsManagerStatusFifoEntry_t, &psManagerStatusFifoEntry);

		if(TRUE == psManagerStatusFifoEntry.noFreeNdp)
		{
			ASSERT(0); // Need more PDs... Or handle like they do in gen5, in rxManagerPsSettingsFifoNotEmpty
		}
		else if((TRUE == psManagerStatusFifoEntry.setPsStatus) || (TRUE == psManagerStatusFifoEntry.clearPsStatus))
		{
			ILOG2_DD("PsManager_PsSettingsFifoNotEmpty  Station Changed Ps Mode Set Ps Status %x Clear Ps Status %x", psManagerStatusFifoEntry.setPsStatus,psManagerStatusFifoEntry.clearPsStatus);
		}
				
		psManagerFifoParams.currentIndexInFifo++;
		psManagerFifoParams.currentIndexInFifo &= PS_MANAGER_STATUS_FIFO_SIZE_MASK;
	}  

	PsManager_SetFifoParameters(&psManagerFifoParams);
}




/**********************************************************************************
PsManager_Init 

Description:
------------
	Initial configuration of PS Manager
Input: 
-----
	None		
Output:
-------
	None
Returns:
--------
	None 
**********************************************************************************/
void PsManager_Init(void)
{
	RegPsSettingPsSettingControl_u psSettingControlReg;
	RegPsSettingPdNdpTxPdListIdx_u ndpPdListIdxReg;
	RegPsSettingPsFwNotificationEn_u psFwNotificationsEnReg;
	RegPsSettingPsFifoBaseAddr_u psFifoBaseAddrReg;
	RegPsSettingPsFifoDepthMinusOne_u psFifoDepthMinusOneReg;
	RegPsSettingPsFifoClearStrb_u psFifoClearStrbReg;
	RegPsSettingPsSettingErrIrqEn_u PsSettingErrIrqEn;
	uint32 statusNotificationRegAddr;

	psSettingControlReg.val = 0;
	ndpPdListIdxReg.val = 0;
	psFwNotificationsEnReg.val = 0;
	psFifoBaseAddrReg.val = 0;
	psFifoDepthMinusOneReg.val = 0;
	psFifoClearStrbReg.val = 0;
	PsSettingErrIrqEn.val = 0;
	
	
	/* Configuration:
		1) Enable Ps Setting Module ,
		2) Set Ps Request  Mode (selctive)
			Settings Mode
				0 - Selective - Set "PS Request" for all queues configured to PS type = legacy and data bit is set
				1- full - Set "PS Request" for all queues configured to (PS type = legacy) */
	psSettingControlReg.bitFields.psSettingEn = PS_SETTING_ENABLE;
	psSettingControlReg.bitFields.psRequestMode = PS_SETTING_MODE_SELECTIVE_REQUEST;
#ifdef ENET_INC_ARCH_WAVE600B
	psSettingControlReg.bitFields.psRequestTwtAvailEn = TRUE;
#endif
	RegAccess_Write(REG_PS_SETTING_PS_SETTING_CONTROL,psSettingControlReg.val);

	// Configure Null PD list index
	ndpPdListIdxReg.bitFields.pdNdpTxPdListIdx = HW_Q_MANAGER_TX_POOL_LIST_PS_SETTINGTS_NDP_PD;
	RegAccess_Write(REG_PS_SETTING_PD_NDP_TX_PD_LIST_IDX,ndpPdListIdxReg.val);
	
	// Enable NDP PD List Empty Notification
#ifdef PF6_TWT_PS_NDP
	psFwNotificationsEnReg.bitFields.ndpTxPdEmptyNotifyEn = PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_DISABLE;
#else
    psFwNotificationsEnReg.bitFields.ndpTxPdEmptyNotifyEn = PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_ENABLE;
#endif
	psFwNotificationsEnReg.bitFields.psPollFcPmZeroNotifyEn = PS_SETTING_PS_POLL_FC_PM_ZERO_NOTIFICATION_DISABLE;
	psFwNotificationsEnReg.bitFields.psPollNoMatchAidNotifyEn = PS_SETTING_PS_POLL_NO_MATCH_AID_NOTIFICATION_DISABLE;	
	RegAccess_Write(REG_PS_SETTING_PS_FW_NOTIFICATION_EN,psFwNotificationsEnReg.val);

	// Each band user a different place in SHRAM for status fifo
#if defined(ENET_INC_LMAC0) 
	pPsManagerStatusFifo = &PsManagerStatusFifo[0];
#endif
#if defined(ENET_INC_LMAC1) 
	pPsManagerStatusFifo = &PsManagerStatusFifo_B1[0];
#endif

	// Clear PS Status Fifo (the fifos of the 2 bands have the same size)
	memset32((void *)pPsManagerStatusFifo, 0, CONVERT_BYTES_TO_WORDS(sizeof(PsSettingFifoShram_t) * PS_MANAGER_STATUS_FIFO_SIZE));
	
	//Configure Fifo -  Base Address,size
	psFifoBaseAddrReg.bitFields.psFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET((uint32)pPsManagerStatusFifo);
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_BASE_ADDR,psFifoBaseAddrReg.val);
	psFifoDepthMinusOneReg.bitFields.psFifoDepthMinusOne = (PS_MANAGER_STATUS_FIFO_SIZE - 1);
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_DEPTH_MINUS_ONE,psFifoDepthMinusOneReg.val);
	
	psFifoClearStrbReg.bitFields.psFifoClearStrb = PS_SETTINGS_CLEAR_STRB;
	psFifoClearStrbReg.bitFields.psFifoClearFullDropCtrStrb = PS_SETTINGS_CLEAR_STRB;
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_CLEAR_STRB,psFifoClearStrbReg.val);

	// Enable PS Manager errors interrupt
	PsSettingErrIrqEn.bitFields.startNewRxSessionWhileWaitAutoRespIrqEn = PS_SETTING_START_NEW_RX_SESSION_WHILE_WAIT_AUTO_RESP_IRQ_EN;
	PsSettingErrIrqEn.bitFields.startNewRxSessionWhilePsSettingActiveIrqEn = PS_SETTING_START_NEW_RX_SESSION_WHILE_PS_SETTING_ACTIVE_IRQ_EN;
	PsSettingErrIrqEn.bitFields.psSettingFifoDecrementLessThanZeroIrqEn = PS_SETTING_FIFO_DECREMENT_LESS_THAN_ZERO_IRQ_EN;
	PsSettingErrIrqEn.bitFields.psSettingFifoFullDropIrqEn = PS_SETTING_FIFO_FULL_DROP_IRQ_EN;
	RegAccess_Write(REG_PS_SETTING_PS_SETTING_ERR_IRQ_EN,PsSettingErrIrqEn.val);	

	// Set Ps Status notification for all stations
	for(statusNotificationRegAddr = REG_PS_SETTING_PS_STATUS_CHANGE_STA_0TO31_NOTIFY_EN; statusNotificationRegAddr  <= REG_PS_SETTING_PS_STATUS_CHANGE_STA_224TO255_NOTIFY_EN; statusNotificationRegAddr+=HW_REG_SIZE)
	{
		RegAccess_Write(statusNotificationRegAddr,PS_SETTING_PS_STATUS_NOTIFICATION_ENABLE_ALL_STATIONS);	
	}
}












