/***************************************************************
 File:		PsManager.c
 Module:	  	Powe Save Manager
 Purpose: 	Driver for PS Manager
 Description:  This module contains the driver for PS Manager HW Module
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "PsManager_Api.h"
#include "PsSettingRegs.h"
#include "ShramPsManager.h"
#include "HwQManager_API.h"
#include "Pac_Api.h"
#include "PacketDescriptor.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#include "ResourceManager_API.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PS_MANAGER
#define LOG_LOCAL_FID 0

// Global Enable
#define PS_SETTING_DISABLE					(0) 
#define PS_SETTING_ENABLE 					(1) 

// Ps Settings Mode
#define PS_SETTING_MODE_SELECTIVE_REQUEST	(0) 
#define PS_SETTING_MODE_FULL_REQUEST 		(1) 


#define PS_SETTING_PS_STATUS_NOTIFICATION_DISABLE_ALL_STATIONS			(0) 
#define PS_SETTING_PS_STATUS_NOTIFICATION_ENABLE_ALL_STATIONS			(0xFFFFFFFF) 

#define PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_DISABLE			(0) 
#define PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_ENABLE			(1) 

#define PS_SETTING_PS_POLL_FC_PM_ZERO_NOTIFICATION_DISABLE		(0) 
#define PS_SETTING_PS_POLL_FC_PM_ZERO_NOTIFICATION_ENABLE		(1) 

#define PS_SETTING_PS_POLL_NO_MATCH_AID_NOTIFICATION_DISABLE	(0) 
#define PS_SETTING_PS_POLL_NO_MATCH_AID_NOTIFICATION_ENABLE		(1) 

#define PS_SETTINGS_CLEAR_STRB 	(1)


#define PS_SETTINGS_ACK_REPLY_RXD_LINES_A		((1 << RCD_ERR_CAT_ACK) | (1 << RCD_UQDN_ACK_AGG) | (1 << RCD_UQDN_ACK) | (1 << RCD_UDATA) | \
												 (1 << RCD_ACTION_LAST_IN_AGG) | (1 << RCD_UMGMT))
#define PS_SETTINGS_ACK_REPLY_RXD_LINES_B		(0x0)



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct psManagerDb_
{
	uint16	fifoReadIndex;					// Next entry to provide to FW		
	uint16 fifoFullCounter;
	
} psManagerDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void psManagerCheckFifoError(void);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
psManagerDb_t PsManagerDb;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
psManagerCheckFifoError 

Description:
------------
	Check Ps Setting Hw error scenarios
	 	1) Drooped events due to fifo full
	 	2) Fifo num of entries decreased to less than zero (Synch problem between Sw and Hw)
Input: 
-----
	None
Returns:
--------
	None 
**********************************************************************************/
void psManagerCheckFifoError(void)
{
	RegPsSettingPsFifoDebug_u	fifoDebugReg;
	RegPsSettingPsFifoClearStrb_u psFifoClearStrbReg;

	psFifoClearStrbReg.val = 0;
	
	//psManagerCheckFifoError	
	RegAccess_Read(REG_PS_SETTING_PS_FIFO_DEBUG,&fifoDebugReg.val);

	/* Dropped Events when fifo was full*/
	PsManagerDb.fifoFullCounter += fifoDebugReg.bitFields.psFifoFullDropCtr;

	// Clear fifo full drop counter
	psFifoClearStrbReg.bitFields.psFifoClearFullDropCtrStrb = PS_SETTINGS_CLEAR_STRB;
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_CLEAR_STRB,psFifoClearStrbReg.val);
	
	/* Decrement Less Than Zero */
	DEBUG_ASSERT(fifoDebugReg.bitFields.psFifoDecrementLessThanZero == FALSE);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
PsManager_EnableDisableLogs 

Description:
------------
	Enable or Disable the Hw logs of the Ps Setting Hw Module
		1) Disable Logs
		2) Wait for log activity to finish
		3) Enable\Disable logs as required
	
Input: 
-----
	isShortLogsLogEnabled - Enable or Disable fifo log
	isLongLogsLogEnabled - Not relevant for this module

Returns:
--------
	None 
**********************************************************************************/
void PsManager_EnableDisableLogs(bool isShortLogsLogEnabled, bool isLongLogsLogEnabled)
{
	RegPsSettingPsLogger_u psLoggerReg;
	RegPsSettingPsLoggerActive_u  psLoggerActiveReg;

	// disable	logs
	psLoggerReg.val = 0;
	RegAccess_Write(REG_PS_SETTING_PS_LOGGER,psLoggerReg.val);
	
	// need to check that we are not in the middle writing log to fifo
	RegAccess_Read(REG_PS_SETTING_PS_LOGGER_ACTIVE, &psLoggerActiveReg.val);

	while (psLoggerActiveReg.bitFields.psLoggerActive == 1)
	{
		RegAccess_Read(REG_PS_SETTING_PS_LOGGER_ACTIVE, &psLoggerActiveReg.val);
	}
	
	// enable logs
	psLoggerReg.bitFields.psLoggerEn = isShortLogsLogEnabled;
	psLoggerReg.bitFields.psLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PS_SETTING_PS_LOGGER, psLoggerReg.val);	
}

/**********************************************************************************
PsManager_TransmitNdpCfm 

Description:
------------
	Callback of the NDP frame sent by PS Setting Hw
Input: 
-----
	pPacketDesc - NDP PD
Returns:
--------
	None 
**********************************************************************************/
void PsManager_TransmitNdpCfm(TxPd_t* pPacketDesc)
{
    ResourceManager_ReleaseDescriptor(pPacketDesc,DESC_POOL_PS_SETTINGS_NDP);
}

/**********************************************************************************
PsManager_GetFifoParameters 

Description:
------------
	
Output: 
-----
	fifoParams:
		1) Current Fifo Index
		2) Num of Entries in fifo waiting for FW processing

Returns:
--------
	None 
**********************************************************************************/
void PsManager_GetFifoParameters(PsManagerFifoParams_t *fifoParams)
{
	RegPsSettingPsFifoNumEntriesCount_u fifoNumOfEntriesCountReg;

	RegAccess_Read(REG_PS_SETTING_PS_FIFO_NUM_ENTRIES_COUNT,&fifoNumOfEntriesCountReg.val);
	
	fifoParams->currentIndexInFifo = PsManagerDb.fifoReadIndex;
	fifoParams->numOfValidEntries = fifoNumOfEntriesCountReg.bitFields.psFifoNumEntriesCount;

//	ILOG0_D("PsManager_GetFifoParameters: fifoParams=%x",*(uint32 * )fifoParams);

	psManagerCheckFifoError();
}

/**********************************************************************************
PsManager_SetFifoParameters 

Description:
------------
	Sets the Fifo Params after FW finished processing all entries in fifo
Input: 
-----
	fifoParams
		1) Current Fifo Index
		2) Num of Entries in fifo waiting for FW processing

Returns:
--------
	None 
**********************************************************************************/	
void PsManager_SetFifoParameters(PsManagerFifoParams_t *fifoParams)

{
	RegPsSettingPsFifoRdEntriesNum_u fifoRdEntriesNumReg;

//	ILOG0_D("PsManager_SetFifoParameters: fifoParams=%x",*(uint32 * )fifoParams);

	fifoRdEntriesNumReg.bitFields.psFifoRdEntriesNum = fifoParams->numOfValidEntries;

	// Decrement  Number Of Entries In Fifo 
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_RD_ENTRIES_NUM,fifoRdEntriesNumReg.val);

	PsManagerDb.fifoReadIndex = fifoParams->currentIndexInFifo;
}



/**********************************************************************************
PsManager_Init 

Description:
------------
	Initial configuration of PS Manager
Input: 
-----
	None		
Output:
-------
	None
Returns:
--------
	None 
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void PsManager_Init(void)
{
	RegPsSettingPsSettingControl_u psSettingControlReg;
	RegPsSettingPsAckReplyRxdLinesA_u psAckReplyRxdLinesAReg;
	RegPsSettingPsAckReplyRxdLinesB_u psAckReplyRxdLinesBReg;
	RegPsSettingPdNdpTxPdListIdx_u ndpPdListIdxReg;
	RegPsSettingPsFwNotificationEn_u psFwNotificationsEnReg;
	RegPsSettingPsFifoBaseAddr_u psFifoBaseAddrReg;
	RegPsSettingPsFifoDepthMinusOne_u psFifoDepthMinusOneReg;
	RegPsSettingPsFifoClearStrb_u psFifoClearStrbReg;
	uint32 statusNotificationRegAddr;

	psSettingControlReg.val = 0;
	psAckReplyRxdLinesAReg.val = 0; 
	psAckReplyRxdLinesBReg.val = 0;
	ndpPdListIdxReg.val = 0;
	psFwNotificationsEnReg.val = 0;
	psFifoBaseAddrReg.val = 0;
	psFifoDepthMinusOneReg.val = 0;
	psFifoClearStrbReg.val = 0;
	
	
	/* Configuration:
		1) Enable Ps Setting Module ,
		2) Set Ps Request  Mode (selctive)
			Settings Mode
				0 - Selective - Set "PS Request" for all queues configured to PS type = legacy and data bit is 
				1- full - Set "PS Request" for all queues configured to (PS type = legacy) */
	psSettingControlReg.bitFields.psSettingEn = PS_SETTING_ENABLE;
	psSettingControlReg.bitFields.psRequestMode = PS_SETTING_MODE_SELECTIVE_REQUEST;
	RegAccess_Write(REG_PS_SETTING_PS_SETTING_CONTROL,psSettingControlReg.val);

	// Set Rxd Ack Reply  Lines
	psAckReplyRxdLinesAReg.bitFields.psAckReplyRxdLinesA = PS_SETTINGS_ACK_REPLY_RXD_LINES_A;
	RegAccess_Write(REG_PS_SETTING_PS_ACK_REPLY_RXD_LINES_A,psAckReplyRxdLinesAReg.val);
	psAckReplyRxdLinesBReg.bitFields.psAckReplyRxdLinesB = PS_SETTINGS_ACK_REPLY_RXD_LINES_B;
	RegAccess_Write(REG_PS_SETTING_PS_ACK_REPLY_RXD_LINES_B,psAckReplyRxdLinesBReg.val);
	
	// Configure Null PD list index
	ndpPdListIdxReg.bitFields.pdNdpTxPdListIdx = HW_Q_MANAGER_TX_POOL_LIST_PS_SETTINGTS_NDP_PD;
	RegAccess_Write(REG_PS_SETTING_PD_NDP_TX_PD_LIST_IDX,ndpPdListIdxReg.val);
	
	// Enable NDP PD List Empry Notification
	psFwNotificationsEnReg.bitFields.ndpTxPdEmptyNotifyEn = PS_SETTING_NDP_TX_PD_EMPTY_NOTIFICATION_ENABLE;
	psFwNotificationsEnReg.bitFields.psPollFcPmZeroNotifyEn = PS_SETTING_PS_POLL_FC_PM_ZERO_NOTIFICATION_DISABLE;
	psFwNotificationsEnReg.bitFields.psPollNoMatchAidNotifyEn = PS_SETTING_PS_POLL_NO_MATCH_AID_NOTIFICATION_DISABLE;	
	RegAccess_Write(REG_PS_SETTING_PS_FW_NOTIFICATION_EN,psFwNotificationsEnReg.val);

	// Clear PS Status Fifo
	memset32((void *)PsManagerStatusFifo, 0, CONVERT_BYTES_TO_WORDS(sizeof(PsManagerStatusFifo)));
	
	//Configure Fifo -  Base Address,size
	psFifoBaseAddrReg.bitFields.psFifoBaseAddr = (uint32)PsManagerStatusFifo;
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_BASE_ADDR,psFifoBaseAddrReg.val);
	psFifoDepthMinusOneReg.bitFields.psFifoDepthMinusOne = (PS_MANAGER_STATUS_FIFO_SIZE - 1);
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_DEPTH_MINUS_ONE,psFifoDepthMinusOneReg.val);

	
	//psFifoClearStrbReg.bitFields  - TBD
	
	psFifoClearStrbReg.bitFields.psFifoClearStrb = PS_SETTINGS_CLEAR_STRB;
	psFifoClearStrbReg.bitFields.psFifoClearFullDropCtrStrb = PS_SETTINGS_CLEAR_STRB;
	psFifoClearStrbReg.bitFields.psFifoClearDecLessThanZeroStrb = PS_SETTINGS_CLEAR_STRB;
	RegAccess_Write(REG_PS_SETTING_PS_FIFO_CLEAR_STRB,psFifoClearStrbReg.val);

	// Set Ps Status notification for all stations
	for(statusNotificationRegAddr = REG_PS_SETTING_PS_STATUS_CHANGE_STA_0TO31_NOTIFY_EN; statusNotificationRegAddr  <= REG_PS_SETTING_PS_STATUS_CHANGE_STA_96TO127_NOTIFY_EN; statusNotificationRegAddr+=HW_REG_SIZE)
	{
		RegAccess_Write(statusNotificationRegAddr,PS_SETTING_PS_STATUS_NOTIFICATION_ENABLE_ALL_STATIONS);	
	}	
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif












