/***********************************************************************************
 File:			Qos.c
 Module:		Qos
 Purpose: 		To handle discarded packets by the Ager and releasing PDs when number
                of free PDs is low
 Description:   This file is the implementation of the Qos module which is responsible 
 				to handle discarded packets by the Ager and releasing PDs when number
                of free PDs is low
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"	
#include "Qos_Api.h"
#include "Qos.h"
#include "Ager_Api.h"
#include "HostInterfaceAcc_Api.h"
#include "TsManager_API.h"
#include "HwEventsAndErrors_Api.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "StaDatabase_Api.h"
#include "Utils_Api.h"
#include "Locker_Api.h"
#include "HwGlobalDefinitions.h"
#include "BSSmanager_API.h"
#include "EventsManager_api.h"
#include "queue_utility.h"
#include "Atf_Api.h"
#include "StatisticsManager_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_QOS
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define QOS_MAXIMUM_AGING_INTERVAL_VALUE 	(HOST_INTERFACE_ACCELERATOR_CONVERT_TTL_UNITS_TO_TIME_UNITS((QOS_NUMBER_OF_QOS_AGING_UNITS - 1) << QOS_TTL_UNITS_TO_QOS_AGING_UNITS_SHIFT))
#define QOS_MAXIMUM_PD_DISCARD_LIMIT		0x3FFF

#define QOS_PD_THRESHOLD_STATION_AND_VAP_BITMAP_SIZE ((CONVERT_BIT_INDEX_TO_WORDS(HW_NUM_OF_STATIONS + HW_NUM_OF_VAPS - 1)) + 1)
//#define QOS_PD_THRESHOLD_MIN_AMOUNT_DEFAULT	0
//#define QOS_PD_THRESHOLD_MIN_DIFF_DEFAULT	300
#define QOS_PD_THRESHOLD_MIN_DIFF_DISABLED	0
#define QOS_PD_THRESHOLD_TIMEOUT			50

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef enum
{
	QOS_PD_THRESHOLD_STATE_OFF,
	QOS_PD_THRESHOLD_STATE_ON,
	QOS_PD_THRESHOLD_NUM_STATES
} QoSPdThresholdState_e;

typedef struct QoSPdThresholdDb_s
{
	uint32					stationAndVapBitmap[QOS_PD_THRESHOLD_STATION_AND_VAP_BITMAP_SIZE];
 	uint16				 	minPdDiff;		/*Minimum allowed Difference between Minimum and maximum PDs on queues with same AC*/
	QoSPdThresholdMode_e 	mode;
	QoSPdThresholdState_e	state;
	uint8				 	pollingTimer;
	uint8				 	minPdAmount;	/*Minimum Amount of PDs on a queues needed so a queue is taken into considerartion in Dynamic Mode*/
	uint16					hostInterfaceAcceleratorFreePdsThreshold;	/* initialized to HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD */
} QoSPdThresholdDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/* Messages functions */
static void qosAgerListNotEmpty(K_MSG* qosMessage);
static void qosFreePdsThresholdEvent(K_MSG* qosMessage);
static void qosAddStation(K_MSG* qosMessage);
static void qosRemoveStation(K_MSG* qosMessage);
static void qosAddVap(K_MSG* qosMessage);
static void qosRemoveVap(K_MSG* qosMessage);
static void qosSingleCheckDone(K_MSG* qosMessage);
static void qosSetBeaconInterval(K_MSG* qosMessage);
static void qosSetIsAtfStatic(K_MSG* qosMessage);
static void qosAgerSetRestrictedParams(K_MSG* psMsg);
static void qosEnablePdThreshold(void);
static void qosDisablePdThreshold(void);
static void qosPdThresholdTimer(K_MSG *qosMessage);
static void qosPdThresholdConfig(K_MSG *qosMessage);
/* General functions */
static void qosAddStationOrVap(uint16 stationOrVapIndex, uint16 agingInterval, uint8 bandId);
static void qosRemoveStationOrVap(uint16 stationOrVapIndex);
static void qosUpdateWindow(uint16 stationIndex, uint8 tid);
static void qosSendConfirmToStationManager(uint16 stationIndex);
static void qosProcessCurrentLockedQueue(uint16 stationIndex, uint8 tid, uint8 isDataPacketWasDiscarded);
static void QosAgerSetPeriodicParamsters(void);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static QosPeriodicCheckParameters_t QosPeriodicCheckParameters;
static QosSingleCheckParameters_t QosSingleCheckParameters;
static AgerSingleCheckParameters_t QosAgerSingleCheckConfigurationParameters;
static AgerPeriodicCheckParameters_t QosAgerPeriodicCheckConfigurationParameters;
static QosGeneralParameters_t QosGeneralParameters;
static HwQueueManagerRequestParams_t hwQueueManagerPopRequestParams;
static K_MSG *QosTsManagerPacketDiscardedBitmapMessage;
static uint8 QosTsManagerPacketDiscardedLastTid;

static const FunctionEntry_t afpTaskTable[TASK_QOS_END - TASK_QOS_START]=
{
	{qosAgerListNotEmpty,				      DOUBLE_CHECK_MSG_TYPE(QOS_AGER_LIST_NOT_EMPTY)}, 
	{qosFreePdsThresholdEvent,			      DOUBLE_CHECK_MSG_TYPE(QOS_NUMBER_OF_FREE_PDS_REACHED_THRESHOLD)}, 
	{qosAddStation,				              DOUBLE_CHECK_MSG_TYPE(QOS_ADD_STATION)}, 
	{qosRemoveStation,				          DOUBLE_CHECK_MSG_TYPE(QOS_REMOVE_STATION)},
	{qosAddVap,						          DOUBLE_CHECK_MSG_TYPE(QOS_ADD_VAP)},
	{qosRemoveVap,						      DOUBLE_CHECK_MSG_TYPE(QOS_REMOVE_VAP)},
	{qosSingleCheckDone,					  DOUBLE_CHECK_MSG_TYPE(QOS_SINGLE_CHECK_DONE)},	
	{qosPdThresholdTimer,					  DOUBLE_CHECK_MSG_TYPE(QOS_PD_THRESOLD_TIMER)},	
	{qosPdThresholdConfig,					  DOUBLE_CHECK_MSG_TYPE(QOS_PD_THRESHOLD_CONFIG)},	
	{qosSetBeaconInterval,					  DOUBLE_CHECK_MSG_TYPE(QOS_SET_BEACON_INTERVAL)},
	{qosSetIsAtfStatic,					      DOUBLE_CHECK_MSG_TYPE(QOS_SET_IS_ATF_STATIC)},        
	{qosAgerSetRestrictedParams,		      DOUBLE_CHECK_MSG_TYPE(QOS_AGER_SET_RESTRICTED_PARAMS)},
};

static const uint8 QosNextAcTable[ACCESS_CATEGORY_NUM] =
{
 	ACCESS_CATEGORY_VI             , /* ACCESS_CATEGORY_BE */
	ACCESS_CATEGORY_BE             , /* ACCESS_CATEGORY_BK */
	ACCESS_CATEGORY_VO             , /* ACCESS_CATEGORY_VI */
	ACCESS_CATEGORY_VO             , /* ACCESS_CATEGORY_VO */
};	

//TBD this strucrue is valid till the new FW configuration is implemented
static const uint16 QosDefaultPdsLimitsThresholds[QOS_DEFAULT_NUMBER_OF_PD_LIMITS] =
{
 	(NUM_OF_TX_DESC >> 5)            , /* 1/32 */
	(NUM_OF_TX_DESC >> 6)            , /* 1/64 */
	(NUM_OF_TX_DESC >> 7)			 , /* 1/128 */
	0			                     , /* threshold = 0 */
};	

static const uint8 QoSQueuesPerAcTable[ACCESS_CATEGORY_NUM][NUMBER_OF_TIDS_PER_AC] =
{
 	{IEEE802_1D_BE_1, IEEE802_1D_BE_2}, /* ACCESS_CATEGORY_BE */
	{IEEE802_1D_BK_1, IEEE802_1D_BK_2}, /* ACCESS_CATEGORY_BK */
	{IEEE802_1D_VI_1, IEEE802_1D_VI_2}, /* ACCESS_CATEGORY_VI */
	{IEEE802_1D_VO_1, IEEE802_1D_VO_2}, /* ACCESS_CATEGORY_VO */
};	

QoSPdThresholdDb_t QoSPdThresholdDb;

//

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

qosAgerListNotEmpty 


Description:
------------
Handles the event that the discarded PDs by the Ager list became not empty


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosAgerListNotEmpty(K_MSG* qosMessage)
{
	uint8 tid = 0;
	uint16 stationIndex = 0;
	uint8 isDataPacketWasDiscarded = FALSE;
	TxPd_t *packetDescriptor = NULL;
	TxPd_t *headDescriptor = NULL;
	TxPd_t *tailDescriptor = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	UNUSED_PARAM(qosMessage);
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_LOW_PR_READY_LIST_AGER;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;

	HwQManager_ClearTxListsInt(HW_Q_MANAGER_LOW_PR_READY_LIST_AGER);
	
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	ASSERT(hwQueueManagerRequestParams.pHeadDesc != NULL_PD);
	packetDescriptor = hwQueueManagerRequestParams.pHeadDesc;
    headDescriptor = packetDescriptor;
 
	stationIndex = packetDescriptor->txQStaId;
	tid = packetDescriptor->txQTid;
	
	while((TxPd_t *)NULL_PD != packetDescriptor)
	{
#if defined (AGER_STATUS_WORKAROUND)
		/*In Gen6 there is a bug that status is not set by ager when packet is discarded*/
		packetDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
#endif			
		if((packetDescriptor->txQStaId != stationIndex) || (packetDescriptor->txQTid != tid))
		{
			qosProcessCurrentLockedQueue(stationIndex, tid, isDataPacketWasDiscarded);

			isDataPacketWasDiscarded = FALSE;			
			stationIndex = packetDescriptor->txQStaId;
			tid = packetDescriptor->txQTid;
		}
		if(PD_TYPE_DATA == packetDescriptor->pdType)
		{
			isDataPacketWasDiscarded = TRUE;
			if(packetDescriptor->aggregationIndication)
			{
				/* Take the whole A-MSDU */
				packetDescriptor = (TxPd_t *)CONVERT_OFFSET_TO_PD(packetDescriptor->aMsduTailPointer);
			}	
		}
		tailDescriptor = packetDescriptor;
		packetDescriptor = (TxPd_t *)GET_NEXT_PD(packetDescriptor);
	}

	qosProcessCurrentLockedQueue(stationIndex, tid, isDataPacketWasDiscarded);

	/*Do we have any remaining STAs to send to TS Manager?*/
	if (QosTsManagerPacketDiscardedBitmapMessage != NULL)
	{
		OSAL_SEND_MESSAGE(TS_MANAGER_PACKET_DISCARDED_BITMAP, TASK_TS_MANAGER, QosTsManagerPacketDiscardedBitmapMessage, VAP_ID_DO_NOT_CARE);
		QosTsManagerPacketDiscardedBitmapMessage = NULL;
		QosTsManagerPacketDiscardedLastTid = QOS_INVALID_TID_INDEX;
	}
	
	/* Realse the all processed PDs - send to liberator Concatenated list in order to save DLM aproaches */
    hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	hwQueueManagerRequestParams.pHeadDesc = headDescriptor;
	hwQueueManagerRequestParams.pTailDesc = tailDescriptor;
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);

	EventManager_TurnOnEvent(EVENT_ID_TX_LIST_READY_PDS_LOW_PRI_NOT_EMPTY);

}


/**********************************************************************************

qosFreePdsThresholdEvent 


Description:
------------
Handles the event that the number of free PDs has reached to threshold


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosFreePdsThresholdEvent(K_MSG* qosMessage)
{
    uint32 currentNumberOfFreePds = 0;
	uint32 numberOfFreePdsNeeded = 0;
	uint8 bandId;
	UNUSED_PARAM(qosMessage);
	
	if(QosGeneralParameters.isAgerEnabled)
	{
		numberOfFreePdsNeeded = QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold + QosSingleCheckParameters.freePdsHysteresisNumber;
		currentNumberOfFreePds = HostIfAcc_GetCurrentFreePdsNumber();

		//ILOG0_DD("[qosFreePdsThresholdEvent], currentNumberOfFreePds = %d, numberOfFreePdsNeeded = %d", currentNumberOfFreePds, numberOfFreePdsNeeded);

		if(currentNumberOfFreePds < numberOfFreePdsNeeded)
		{
	//TBD currently the starting TTL will be taken from the TTL in the periodic check divided by 2. when an external module
	//will pass the configuration to the Qos the starting TTL will be permanenet */
			if (QosGeneralParameters.restrictedAc == TRUE)
			{
	        	QosSingleCheckParameters.startingTtl = 0;
			}
			else
			{
	        	QosSingleCheckParameters.startingTtl = QosAgerPeriodicCheckConfigurationParameters.generalParameters.acTtlCriteria[ACCESS_CATEGORY_BK] >> QOS_DEFAULT_STARTING_TTL_SHIFT;
			}
	///
			/* Return to initial parameters */
			QosSingleCheckParameters.currentAc = ACCESS_CATEGORY_BK;
			if (QosGeneralParameters.restrictedAc == TRUE)
			{
				QosSingleCheckParameters.currentPdLimitIndex = QosSingleCheckParameters.numberOfPdsLimits - 1;
			}
			else
			{
				QosSingleCheckParameters.currentPdLimitIndex = 0;
			}
			QosSingleCheckParameters.currentTtl = QosSingleCheckParameters.startingTtl;
		
			/* Return parameters to default - same as periodic check */
			MEMCPY(&QosAgerSingleCheckConfigurationParameters.generalParameters, &QosAgerPeriodicCheckConfigurationParameters.generalParameters, sizeof(AgerGeneralParameters_t));
			
			if (QosGeneralParameters.restrictedAc == TRUE)
			{
				QosAgerSingleCheckConfigurationParameters.generalParameters.acTtlCriteria[QosSingleCheckParameters.currentAc] = QosSingleCheckParameters.currentTtl;
				QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[QosSingleCheckParameters.currentAc] = QosSingleCheckParameters.pdLimitValues[QosSingleCheckParameters.currentPdLimitIndex];
				QosSingleCheckParameters.currentAc = ACCESS_CATEGORY_BK;
			}
   
		    /* Calculate the number of PDs to discard in order to reach the desired number of free PDs : threshold + hysteresis */
			QosAgerSingleCheckConfigurationParameters.maximumPdsToDiscard = numberOfFreePdsNeeded - currentNumberOfFreePds; 

	        /* Start from AC = BK , with initial values */
			QosAgerSingleCheckConfigurationParameters.generalParameters.acTtlCriteria[QosSingleCheckParameters.currentAc] = QosSingleCheckParameters.currentTtl;
			QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[QosSingleCheckParameters.currentAc] = QosSingleCheckParameters.pdLimitValues[QosSingleCheckParameters.currentPdLimitIndex];

			/*Overwrite minimum PDs */
			if (QosGeneralParameters.restrictedAc == FALSE)
			{
				QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BE] = QOS_MAXIMUM_PD_DISCARD_LIMIT;
			}
			QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI] = QOS_MAXIMUM_PD_DISCARD_LIMIT;
			QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VO] = QOS_MAXIMUM_PD_DISCARD_LIMIT;
			
			/* If number of stations and VAP is lower than threshold use force lock till the number of free PDs has reached destination */
	        QosAgerSingleCheckConfigurationParameters.forceLock = (QosGeneralParameters.restrictedAc) ? 1 : (QosGeneralParameters.numberOfValidVapsAndStations <= QOS_MINIMUM_NUMBER_OF_STATIONS_AND_VAPS_FOR_FORCE_LOCK);
			/* The purpose is to save registers access at the expense of 'if' */ 
			if(QosGeneralParameters.isPeriodicCheckActive)
			{
				//(0, 0, AgerSingleCheckParameters_t, &QosAgerSingleCheckConfigurationParameters);
				Ager_StopPeriodicCheckAndStartSingleCheck(&QosAgerSingleCheckConfigurationParameters, QosGeneralParameters.restrictedAc);		
			}
			else
			{
				//(0, 1, AgerSingleCheckParameters_t, &QosAgerSingleCheckConfigurationParameters);
				Ager_StartSingleCheck(&QosAgerSingleCheckConfigurationParameters, QosGeneralParameters.restrictedAc);
			}
			for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)		
			{
				if(QosGeneralParameters.isAtfStaticDistType[bandId])
	            {
	                OSAL_SEND_NO_DATA_MESSAGE(ATF_BUFFERING_DROP_ALL_BUFFERED ,TASK_ATF, ConfigurationManager_GetFirstVapForBand(bandId));
	            }
			}
			QosGeneralParameters.isDuringSingleCheck = TRUE;
		}
		else
		{
			/* Clear interrupt only here other wise the interrupt will we set again immidiately during operation */
			//("qosFreePdsThresholdEvent - no need to discard packets");
			HostIfAcc_ClearFreePdsThresholdEvent();
			EventManager_TurnOnEvent(EVENT_ID_PD_THRESHOLD_REACHED);
		}
	}
}

/**********************************************************************************

qosAddStation 


Description:
------------
Handles the event of adding station


Input: 
-----	
qosMessage - the pointer to the message that contains all the needed parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosAddStation(K_MSG* qosMessage)
{
	uint16 listenIntervalInTimeUnits = 0;
	uint8 vapIndex = 0;
	uint16 stationIndex = 0;
	uint16 beaconInterval = 0;
	BssManagerStaManagerReq_t *staManagerReq = NULL;
	K_MSG *originalAddStationMessage = NULL;
	UMI_STA_ADD *addStationMessageParameters = NULL;

	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(qosMessage);
	originalAddStationMessage = staManagerReq->psMsg;
	addStationMessageParameters = (UMI_STA_ADD *)pK_MSG_DATA(originalAddStationMessage);

	stationIndex = addStationMessageParameters->u16SID;
	StaDb_GetVapId(stationIndex, &vapIndex);

	beaconInterval = QosPeriodicCheckParameters.beaconIntervals[vapIndex];

	listenIntervalInTimeUnits = addStationMessageParameters->u8ListenInterval * beaconInterval;
	qosAddStationOrVap(stationIndex, listenIntervalInTimeUnits, ConfigurationManager_GetBandForVap(vapIndex));
	qosSendConfirmToStationManager(stationIndex);
}


/**********************************************************************************

qosRemoveStation 


Description:
------------
Handles the event of removing station


Input: 
-----	
qosMessage - the pointer to the message that contains all the needed parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosRemoveStation(K_MSG* qosMessage)
{
	uint16 stationIndex = 0;
	BssManagerStaManagerReq_t *staManagerReq = NULL;
	K_MSG *originalRemoveStationMessage = NULL;
	UMI_STOP_TRAFFIC *stopTrafficMessageParameters = NULL;

	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(qosMessage);
	originalRemoveStationMessage = staManagerReq->psMsg;
	stopTrafficMessageParameters = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(originalRemoveStationMessage);
  
	stationIndex = stopTrafficMessageParameters->u16SID;

	qosRemoveStationOrVap(stationIndex);
	qosSendConfirmToStationManager(stationIndex);
}

/**********************************************************************************

qosAddVap 


Description:
------------
Handles the event of adding VAP


Input: 
-----	
qosMessage - the pointer to the message that contains all the needed parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosAddVap(K_MSG* qosMessage)
{
	uint16 dtimIntervalInTimeUnits = 0;
	uint16 beaconInterval = 0;
	uint8 vapIndex = 0;
	UMI_SET_AP_BEACON_INFO* setApBeaconInfoStructurePtr = (UMI_SET_AP_BEACON_INFO*) pK_MSG_DATA(qosMessage);

	/* The VAP index is between 128 - 143. it is located in the same table so the function qosAddStationOrVap
	   can be called both for station and VAP transperently */
	beaconInterval = setApBeaconInfoStructurePtr->beaconInterval;
	vapIndex = setApBeaconInfoStructurePtr->vapId;
	QosPeriodicCheckParameters.beaconIntervals[vapIndex] = beaconInterval;
	dtimIntervalInTimeUnits = setApBeaconInfoStructurePtr->dtimInterval * beaconInterval;
	//("qosAddVap vap = %d , dtimIntervalInTimeUnits = %d ",vapIndex, dtimIntervalInTimeUnits);
	qosAddStationOrVap(vapIndex + HW_NUM_OF_STATIONS, dtimIntervalInTimeUnits, ConfigurationManager_GetBandForVap(vapIndex));

	OSAL_SEND_MESSAGE(UMI_MC_SET_AP_BEACON_INFO_CFM, TASK_UM_IF_TASK, qosMessage, vapIndex);
}


/**********************************************************************************

qosRemoveVap 


Description:
------------
Handles the event of removing VAP


Input: 
-----	
qosMessage - the pointer to the message that contains all the needed parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosRemoveVap(K_MSG* qosMessage)
{
	UMI_REMOVE_VAP *removeVapMessage = NULL;
	uint8 vapIndex = 0;
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent = NULL;
	K_MSG *vapMessage = NULL;
		
	vapMessage = (K_MSG *) (*((uint32 *)pK_MSG_DATA(qosMessage)));
	removeVapMessage = (UMI_REMOVE_VAP *) pK_MSG_DATA(vapMessage);
	vapIndex = removeVapMessage->vapId;

	qosRemoveStationOrVap(HW_NUM_OF_STATIONS + vapIndex);
	/* No need to reset beacon interval - when adding a new VAP it will be overwriten */

	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT *)(&(qosMessage->abData));
	confirmEvent->vapId = vapIndex;
	confirmEvent->eventIndex = VAP_MANAGER_REMOVE_VAP; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_QOS_CLIENT;
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, qosMessage, vapIndex);	
}


/**********************************************************************************

qosSingleCheckDone 


Description:
------------
Handles the event that the single check has finished


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosSingleCheckDone(K_MSG* qosMessage)
{
    uint32 currentNumberOfFreePds = 0;
	uint32 numberOfFreePdsNeeded = 0;
	AgerLastCheckParametes_t lastCheckParameters;
	AgerLastCheckCounters_t lastCheckCounters;
	UNUSED_PARAM(qosMessage);
	//ILOG0_V("[qosSingleCheckDone]");
	
#ifdef USE_AGER_EMULATOR
	OSAL_CPU_SERVICE_COOPERATION();  //This routine is being called after agerEmulator reduce its own priority, raised by the qos thread. We must let other threads in the same priority a chance to run at this point. 
#endif	
    memset(&lastCheckParameters, 0, sizeof(AgerLastCheckParametes_t));
	memset(&lastCheckCounters, 0, sizeof(AgerLastCheckCounters_t));
	
	Ager_GetLastCheckParameters(&lastCheckParameters);

	/* Calculate the next station/VAP index to start from */
	QosAgerSingleCheckConfigurationParameters.stationOrVapIndexToStart = lastCheckParameters.index;
	QosAgerSingleCheckConfigurationParameters.isStartingFromVap = lastCheckParameters.isVapQueue;
	QosAgerSingleCheckConfigurationParameters.stationOrVapIndexToStart ++;
	/* The check must be done on the HW number of VAPs since the Ager check them all */
	if(QosAgerSingleCheckConfigurationParameters.isStartingFromVap && (HW_NUM_OF_VAPS == QosAgerSingleCheckConfigurationParameters.stationOrVapIndexToStart))
	{
		QosAgerSingleCheckConfigurationParameters.stationOrVapIndexToStart = 0;
		QosAgerSingleCheckConfigurationParameters.isStartingFromVap = FALSE;
	}/* The check must be done on the HW number of stations since the Ager check them all */
	else if(HW_NUM_OF_STATIONS == QosAgerSingleCheckConfigurationParameters.stationOrVapIndexToStart)
	{
		QosAgerSingleCheckConfigurationParameters.stationOrVapIndexToStart = 0;
		QosAgerSingleCheckConfigurationParameters.isStartingFromVap = TRUE;
	}

	numberOfFreePdsNeeded = QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold;
	currentNumberOfFreePds = HostIfAcc_GetCurrentFreePdsNumber();
	//ILOG0_DD("[qosSingleCheckDone], numberOfFreePdsNeeded = %d, currentNumberOfFreePds = %d", numberOfFreePdsNeeded, currentNumberOfFreePds);
	if(currentNumberOfFreePds < numberOfFreePdsNeeded)
	{   
    	/* Calculate the number of PDs to discard in order to reach the desired number of free PDs : threshold + hysteresis */
		QosAgerSingleCheckConfigurationParameters.maximumPdsToDiscard = (numberOfFreePdsNeeded - currentNumberOfFreePds + QosSingleCheckParameters.freePdsHysteresisNumber); 
        
		QosSingleCheckParameters.currentPdLimitIndex++;
		//ILOG0_DD("[qosSingleCheckDone], QosAgerSingleCheckConfigurationParameters.maximumPdsToDiscard = %d, QosSingleCheckParameters.currentPdLimitIndex = %d", QosAgerSingleCheckConfigurationParameters.maximumPdsToDiscard, QosSingleCheckParameters.currentPdLimitIndex);

		/* If the configuration has changed during operation the index can be higher than the new size */
		if(QosSingleCheckParameters.numberOfPdsLimits <= QosSingleCheckParameters.currentPdLimitIndex)
        {
			if(0 == QosSingleCheckParameters.currentTtl)
			{
				//ILOG0_D("[qosSingleCheckDone], QosSingleCheckParameters.currentTtl = %d", QosSingleCheckParameters.currentTtl);
				if(ACCESS_CATEGORY_VO != QosSingleCheckParameters.currentAc)
				{
					QosSingleCheckParameters.currentPdLimitIndex = 0;
					QosSingleCheckParameters.currentTtl = QosSingleCheckParameters.startingTtl;
					QosSingleCheckParameters.currentAc = QosNextAcTable[QosSingleCheckParameters.currentAc];
				}
				else
				{
					QosSingleCheckParameters.currentPdLimitIndex = QosSingleCheckParameters.numberOfPdsLimits - 1;
				}
			}
			else
			{
				/* Dividing the TTL by 2 */
				QosSingleCheckParameters.currentTtl >>= QOS_DECREASE_TTL_CRITERIA_SHIFT;
				QosSingleCheckParameters.currentPdLimitIndex = 0;
			}			
        }
		Ager_GetLastCheckCounters(&lastCheckCounters);
		//(0, 0, AgerLastCheckParametes_t, &lastCheckCounters);
        QosSingleCheckParameters.numberOfLockFailures = lastCheckCounters.numberOfLockFailures;
		QosSingleCheckParameters.numberOfPdsHandledToLiberator += lastCheckCounters.numberOfPdsHandledToLiberator;
		QosSingleCheckParameters.numberOfPdsHandledToUmac += lastCheckCounters.numberOfPdsHandledToUmac;
		
        QosAgerSingleCheckConfigurationParameters.generalParameters.acTtlCriteria[QosSingleCheckParameters.currentAc] = QosSingleCheckParameters.currentTtl;
		QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[QosSingleCheckParameters.currentAc] = QosSingleCheckParameters.pdLimitValues[QosSingleCheckParameters.currentPdLimitIndex];
		if (QosGeneralParameters.restrictedAc == TRUE)
		{
			/* Lets keep the MIN at MAX :) */
			QosAgerSingleCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI] = QOS_MAXIMUM_PD_DISCARD_LIMIT;
		}

		/* If the number of lock failures is bigger than 1/2 of stations and VAPs use force lock till the number of free PDs has reached destination */
		if (QosGeneralParameters.restrictedAc == TRUE)
		{
			QosAgerSingleCheckConfigurationParameters.forceLock = 1;
		}
		else
		{
			QosAgerSingleCheckConfigurationParameters.forceLock |= (lastCheckCounters.numberOfLockFailures > (QosGeneralParameters.numberOfValidVapsAndStations >> QOS_FORCE_LOCK_CRITERIA_SHIFT));
		}
		//(1, 0, AgerSingleCheckParameters_t, &QosAgerSingleCheckConfigurationParameters);
		Ager_StartSingleCheck(&QosAgerSingleCheckConfigurationParameters, QosGeneralParameters.restrictedAc);
	}
	else
	{
		/* Clear interrupt only here other wise the interrupt will we set again immidiately during operation */
		//ILOG0_V("[qosSingleCheckDone] stop single check");

		HostIfAcc_ClearFreePdsThresholdEvent();
		EventManager_TurnOnEvent(EVENT_ID_PD_THRESHOLD_REACHED);
		//ILOG0_D("qosSingleCheckDone no need to discard more PDs is periodic check active = %d", QosGeneralParameters.isPeriodicCheckActive);

		QosGeneralParameters.isDuringSingleCheck = FALSE;
		if(QosGeneralParameters.isPeriodicCheckActive)
		{
			Ager_ConfigureAllAndActivatePeriodicCheck(&QosAgerPeriodicCheckConfigurationParameters);
		}
	}	
}

/**********************************************************************************

qosEnablePdThreshold 


Description:
------------
Enable the PD threshold


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosEnablePdThreshold()
{
	/*is the PD threshold already enabled*/
	if (QoSPdThresholdDb.state == QOS_PD_THRESHOLD_STATE_OFF)
	{
		/*It is not - enable it*/
		HostIfAcc_SetPdThreshold(QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold);
		/*and change state*/
		QoSPdThresholdDb.state = QOS_PD_THRESHOLD_STATE_ON;
	}
}

/**********************************************************************************

qosDisablePdThreshold 


Description:
------------
Diable the PD threshold


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosDisablePdThreshold()
{
	/*is the PD threshold already disabled*/
	if (QoSPdThresholdDb.state == QOS_PD_THRESHOLD_STATE_ON)
	{
		/*It is not - disable it*/
		HostIfAcc_SetPdThreshold(HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD_DISABLED);
		/*and change state*/
		QoSPdThresholdDb.state = QOS_PD_THRESHOLD_STATE_OFF;
	}
}

/**********************************************************************************

qosPdThresholdTimer 


Description:
------------
PD threshold timer. Scans the queues and sets the PD threshold accordingly


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosPdThresholdTimer(K_MSG *qosMessage)
{
	uint16 numQueuesWithMinPd[ACCESS_CATEGORY_NUM] = {0};
	uint8 numAcsWithMinPd = 0;
	uint16 minPdOnQueue[ACCESS_CATEGORY_NUM] = {0};
	uint16 maxPdOnQueue[ACCESS_CATEGORY_NUM] = {0};
	uint32 tempBitmap[QOS_PD_THRESHOLD_STATION_AND_VAP_BITMAP_SIZE];
	uint16 stationOrVapIndex;
	uint16 numberOfDataPdsOnAcQueues;
	uint8 acIndex;
	uint8 i;
	bool isVapQueues;
	uint8 firstTid, secondTid;
#ifdef ENET_INC_ARCH_WAVE600D2
	HostIfQosCounters_t *pHostIfQosCounters = (HostIfQosCounters_t *)statisticsGetHostIfQosCountersAddress();
#else
	HostIfCounters_t *pHostIfCounters = (HostIfCounters_t *)statisticsGetHostIfCountersAddress();
#endif
	UNUSED_PARAM(qosMessage);

	memset(numQueuesWithMinPd, 0, sizeof(uint16)*ACCESS_CATEGORY_NUM);
	memset(maxPdOnQueue, 0, sizeof(uint16)*ACCESS_CATEGORY_NUM);
	for (i = 0; i < ACCESS_CATEGORY_NUM; i++)
	{
		minPdOnQueue[i] = QOS_MAXIMUM_PD_DISCARD_LIMIT;
	}
	/*Check if we are in Dynamic mode and there are valid stations*/
	if ((QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_DYNAMIC) && (QosGeneralParameters.numberOfValidVapsAndStations > 0))
	{
		/*Scan queues of connected VAPs and STAs*/
		memcpy32(tempBitmap, QoSPdThresholdDb.stationAndVapBitmap, QOS_PD_THRESHOLD_STATION_AND_VAP_BITMAP_SIZE);
		/*Loop thru the number of val;id stations*/
		for (i = 0; i < QosGeneralParameters.numberOfValidVapsAndStations; i++)
		{
			isVapQueues = FALSE;
			stationOrVapIndex = Utils_FindLastSetAndClear(tempBitmap, CONVERT_WORDS_TO_BIT_INDEX(QOS_PD_THRESHOLD_STATION_AND_VAP_BITMAP_SIZE));
			if (HW_NUM_OF_STATIONS <= stationOrVapIndex)
			{
				stationOrVapIndex -= HW_NUM_OF_STATIONS;
				isVapQueues = TRUE;
			}
			/*Loop through all Access Categories*/
			for (acIndex = 0; acIndex < ACCESS_CATEGORY_NUM; acIndex++)
			{
				/*Get number of Data PDs on this STA or VAP AC*/
				if (isVapQueues == TRUE)
				{
#ifdef ENET_INC_ARCH_WAVE600D2
					numberOfDataPdsOnAcQueues = pHostIfQosCounters->qosTxVap[stationOrVapIndex][acIndex];
#else
					numberOfDataPdsOnAcQueues = pHostIfCounters->qosTxVap[stationOrVapIndex][acIndex];
#endif
				}
				else
				{
					firstTid = QoSQueuesPerAcTable[acIndex][0];
					secondTid = QoSQueuesPerAcTable[acIndex][1];
#ifdef ENET_INC_ARCH_WAVE600D2
					numberOfDataPdsOnAcQueues =  pHostIfQosCounters->qosTxSta[stationOrVapIndex][firstTid] + 
																	pHostIfQosCounters->qosTxSta[stationOrVapIndex][secondTid];
#else
					numberOfDataPdsOnAcQueues =  pHostIfCounters->qosTxSta[stationOrVapIndex][firstTid] + 
																	pHostIfCounters->qosTxSta[stationOrVapIndex][secondTid];
#endif
				}
				/*Check if number of data PDs on these queues exceeds the minimum amount of PDs*/
				if (numberOfDataPdsOnAcQueues > QoSPdThresholdDb.minPdAmount)
				{
					/*Increment the number of queues that exceed the minimum on this AC*/
					numQueuesWithMinPd[acIndex]++;
					/* Were these the first queues that exceeded the minimum?*/
					if (numQueuesWithMinPd[acIndex] == 1)
					{
						/*if it was increment the number of ACs with data queues which exceed the minimum*/
						numAcsWithMinPd++;
						/*Is the number of ACs greater than one?*/
						if (numAcsWithMinPd > 1)
						{
							/*We have data on more than one AC - enable PD threshold, restart the timer and exit*/
							qosEnablePdThreshold();
							/*Restart the timer*/
							OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
							return;
						}
					}
					/*Do we have a minimum difference configuired?*/
					if (QoSPdThresholdDb.minPdDiff == QOS_PD_THRESHOLD_MIN_DIFF_DISABLED)
					{
						/*If not once we have more than one queue that exceeds the minimum enable the PD threshold, restart the timer and exit*/
						if (numQueuesWithMinPd[acIndex] > 1)
						{
							qosEnablePdThreshold();
							/*Restart the timer*/
							OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
							return;
						}
					}
					else
					{
						/*Check if this is below the current minimum for this AC*/
						if (numberOfDataPdsOnAcQueues < minPdOnQueue[acIndex])
						{
							/*Set the minimum of this AC to the number of PDs on these queues*/
							minPdOnQueue[acIndex] = numberOfDataPdsOnAcQueues;
						}
						/*Check if this is above the current maximum for this AC */
						if (numberOfDataPdsOnAcQueues > maxPdOnQueue[acIndex])
						{
							/*Set the maximum of this AC to the number of PDs on these queues*/
							maxPdOnQueue[acIndex] = numberOfDataPdsOnAcQueues;
						}
						/*Is the difference between the minimum and maximum greater than the minimum allowed*/
						if ((maxPdOnQueue[acIndex] - minPdOnQueue[acIndex]) > QoSPdThresholdDb.minPdDiff)
						{
							/*If it is enable the PD threshold, restart the timer and exit*/
							qosEnablePdThreshold();
							/*Restart the timer*/
							OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
							return;
						}
					}
				}
			
			}

		}
		/*If we get here we can disable the PD threshold*/
		qosDisablePdThreshold();
		/*Restart the timer*/
		OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
	}
}

/**********************************************************************************

qosPdThresholdConfig 


Description:
------------
Handle PD Threshold configuration


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosPdThresholdConfig(K_MSG *qosMessage)
{
	QoSPdThresholdConfig_t *config = (QoSPdThresholdConfig_t *)pK_MSG_DATA(qosMessage);
	ILOG0_DDDD("[qosPdThresholdConfig], QoSPdThresholdDb.mode = %d, config->mode = %d, config->minPdDiff = %d , config->minPdAmount = %d ", QoSPdThresholdDb.mode, config->mode, config->minPdDiff, config->minPdAmount);

	if (config->getSetOperation == API_GET_OPERATION)
	{
		config->mode = QoSPdThresholdDb.mode;
		config->minPdAmount = QoSPdThresholdDb.minPdAmount;
		config->minPdDiff = QoSPdThresholdDb.minPdDiff;
	}
	else
	{
		//verify that UMI_SET_PD_THRESH.mode is valid
		ASSERT(QoSPdThresholdDb.mode < QOS_PD_THRESHOLD_NUM_MODES)
		
		if (config->mode == QOS_PD_THRESHOLD_FORCED)
		{
			/*Forced mode requested*/
			if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_FORCED)
			{
				/*Already in forced - nothing to do*/
			}
			else if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_DISABLED)
			{
				/*From disabled to forced - set Threshold*/
				HostIfAcc_SetPdThreshold(QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold);
			}
			else
			{
				/*From dynamic to Forced - enable PD threshold and stop timer*/
				qosEnablePdThreshold();
				OSAL_RESET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, TASK_QOS);
			}
		} 
		else if  (config->mode == QOS_PD_THRESHOLD_DISABLED)
		{
			ILOG0_V("[qosPdThresholdConfig], in QOS_PD_THRESHOLD_DISABLED");
			/*Disabled mode requested*/
			if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_DISABLED)
			{
				/*Already in disabled - nothing to do*/
			}
			else if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_FORCED)
			{
				/*From forced to disabled - set Threshold*/
				HostIfAcc_SetPdThreshold(HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD_DISABLED);
			}
			else
			{
				/*From dynamic to disabled - disable PD threshold and stop timer*/
				qosDisablePdThreshold();
				OSAL_RESET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, TASK_QOS);
			}
			// disable also aperiodic Ager
			QosGeneralParameters.isAgerEnabled = FALSE;
		}
		else if  (config->mode == QOS_PD_THRESHOLD_DYNAMIC)
		{
			/*Dynamic mode requested*/
			if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_DYNAMIC)
			{
				/*Already in dynamic - nothing to do*/
			}
			else if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_FORCED)
			{
				/*From forced to dynamic - start by disabling the threshold*/
				qosDisablePdThreshold();
				/*If STAs or VAPs connected start timer*/
				if (QosGeneralParameters.numberOfValidVapsAndStations !=  0)
				{
					OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
				}
			}
			else
			{
				/*From disabled to dynamic*/
				/*If STAs or VAPs connected start timer*/
				if (QosGeneralParameters.numberOfValidVapsAndStations !=  0)
				{
					OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
				}
			}
			/*Update parameters*/
			QoSPdThresholdDb.minPdDiff = config->minPdDiff;
			QoSPdThresholdDb.minPdAmount = config->minPdAmount;
		}
		/*Set mode*/
		QoSPdThresholdDb.mode = config->mode;
	}

	// Send confirmation to host
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_QOS_PD_THRESH_CFM, TASK_UM_IF_TASK, qosMessage,qosMessage->header.vapId);
}


/**********************************************************************************

qosSetBeaconInterval 


Description:
------------
Set Beacon Interval - used in STA mode, when VSTA 


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosSetBeaconInterval(K_MSG* qosMessage)
{
	uint8 vapIndex = 0;
	UMI_BEACON_INTERVAL_t *vapMessage = NULL;
	uint16 stationOrVapIndex;
	/*When VSTA VAP was added, beacon Interval was unknown*/
	vapMessage = (UMI_BEACON_INTERVAL_t *) (pK_MSG_DATA(qosMessage));
	vapIndex = vapMessage->vapID;
	/*Calculate the index*/
	stationOrVapIndex = vapIndex + HW_NUM_OF_STATIONS;
	/*Need to remove the VAP so current value is removed*/
	qosRemoveStationOrVap(stationOrVapIndex);
	/*Store Beacon Interval*/
	QosPeriodicCheckParameters.beaconIntervals[vapIndex] = vapMessage->beaconInterval;
	/*Now add it again with correct interval*/
	qosAddStationOrVap(stationOrVapIndex, vapMessage->beaconInterval, ConfigurationManager_GetBandForVap(vapIndex));
	//ILOG0_DD("Qos, pSetBeaconInterval->vapID %d, stationIndex %d", vapMessage->vapID, vapMessage->beaconInterval);
}


/**********************************************************************************

qosSetIsAtfStatic 


Description:
------------
Set ATF static - called by ATF when Static mode is used
In Static mode PDs may be buffered by ATF module. If we run outr of PDs we need
ATF to pushthe PDs to TX qeueues


Input: 
-----	
qosMessage 
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosSetIsAtfStatic(K_MSG* qosMessage)
{
	uint8 isStatic = 0;
	uint8 bandId = ConfigurationManager_GetBandForVap(qosMessage->header.vapId);

	
	isStatic = (uint8)(*(pK_MSG_DATA(qosMessage)));
	QosGeneralParameters.isAtfStaticDistType[bandId] = isStatic;
    //ILOG0_D("[qosSetIsAtfStatic] isStatic:%d",isStatic);
}
/***********************************************************************
* qosAgerSetParams
* 
* Description:
* ------------
* Restricted AC parameters have changed. Change AGER parameters accordingly.
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void qosAgerSetRestrictedParams(K_MSG* psMsg)
{
	
	UMI_SET_RESTRICTED_AC* pSetRestrictedAc = (UMI_SET_RESTRICTED_AC*)pK_MSG_DATA(psMsg);	

    QosGeneralParameters.restrictedAc = pSetRestrictedAc->restrictedAcModeEnable;
    QosGeneralParameters.restrictedAcBitmap = pSetRestrictedAc->acRestrictedBitmap;	

	/* Set default periodic parameters */
	QosAgerSetPeriodicParamsters();
	/* Modify HW register */
	if((QosGeneralParameters.isDuringSingleCheck == FALSE) && (QosGeneralParameters.isPeriodicCheckActive == TRUE))
	{
		Ager_ConfigureAllAndActivatePeriodicCheck(&QosAgerPeriodicCheckConfigurationParameters);
	}	
}

/**********************************************************************************

qosAddStationOrVap 


Description:
------------
Add station listen interval or VAP DTIM period to the Qos database


Input: 
-----	
stationOrVapIndex - the index of the station or the VAP to add (vap index is 128 - 143)
agingInterval - aging interval in timeuints
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosAddStationOrVap(uint16 stationOrVapIndex, uint16 agingInterval, uint8 bandId)
{
	uint8 i = 0;
	
	if(QosGeneralParameters.isAgerEnabled)
	{
		//ILOG0_DD("[qosAddStationOrVap] - stationOrVapIndex 0x%x agingInterval 0x%x",stationOrVapIndex,agingInterval);
		/* Set Ager discard interval to twice the size of Listen (or DTIM) interval  */
		agingInterval <<= 1;
		
		QosGeneralParameters.numberOfValidVapsAndStations ++;

	    //TBD currently the responsibility of acitivating the ager and disactivating is in the Qos. the logic is to
	    //activate it if there is a valid VAP or station. when an external module will take the resposibility to 
	    //configure the Qos and activating the periodic check this next line should be removed
		QosGeneralParameters.isPeriodicCheckActive = TRUE;

		/*Need to make sure that after the shifts agingInterval is in range*/
		agingInterval = MIN(agingInterval, QOS_MAXIMUM_AGING_INTERVAL_VALUE);
		agingInterval = HOST_INTERFACE_ACCELERATOR_CONVERT_TIME_UNITS_TO_TTL_UNITS(agingInterval);
		agingInterval >>= QOS_TTL_UNITS_TO_QOS_AGING_UNITS_SHIFT;

		ASSERT(agingInterval < QOS_NUMBER_OF_QOS_AGING_UNITS);
		QosPeriodicCheckParameters.qosAgingUnitsBitmap |= QOS_AGING_UNITS_BITMAP_SET << agingInterval;
		QosPeriodicCheckParameters.qosAgingUnitsCounters[agingInterval] ++;

		
		agingInterval <<= QOS_TTL_UNITS_TO_QOS_AGING_UNITS_SHIFT;
    	agingInterval += QOS_ADD_TO_GET_HIGHEST_VALUE_IN_CELL;
		HwQManagerAger_SetStaTtlVal(stationOrVapIndex,agingInterval);


		if(QosAgerPeriodicCheckConfigurationParameters.agerPeriod < agingInterval)
		{
			QosAgerPeriodicCheckConfigurationParameters.agerPeriod = agingInterval;

			/* Change all TTL criteria to the new aging interval */
			for(i = 0; i < ACCESS_CATEGORY_NUM; i++)
			{
				QosAgerPeriodicCheckConfigurationParameters.generalParameters.acTtlCriteria[i] = agingInterval;	
			}
		}

		if(!QosGeneralParameters.isDuringSingleCheck && QosGeneralParameters.isPeriodicCheckActive)
		{
			/* Reconfigure the Ager only if it is in the periodic check mode and the Qos is not during a single 
		   	check operation (at the end of the single check operation the Qos will reconfigure the Ager anyway) */
	    	Ager_ReconfigurePartialPeriodicCheck(&QosAgerPeriodicCheckConfigurationParameters);
		}

		// send the Criteria to ATF module need to changed once we will anable the ATF in wave 600
    	if (QosGeneralParameters.isAtfStaticDistType[bandId])
    	{
			K_MSG* qosMessage = OSAL_GET_MESSAGE(sizeof(uint8));
			uint8 *qosMessageParameters = ((uint8 *)qosMessage->abData);
        	*qosMessageParameters = agingInterval;
    		OSAL_SEND_MESSAGE(ATF_BUFFERING_QOS_TTL_CRITERIA, TASK_ATF, qosMessage, ConfigurationManager_GetFirstVapForBand(bandId));
    	}

		/*If this is the first VAP or STA and we are in Dynamic Mode start PD Threshold timer*/
		if ((QosGeneralParameters.numberOfValidVapsAndStations == 1) &&
			(QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_DYNAMIC))
		{
			OSAL_SET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(QoSPdThresholdDb.pollingTimer), TASK_QOS);
		}
		else if (QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_FORCED)
		{
			qosEnablePdThreshold();
		}
		Utils_SetBitInBitmap(QoSPdThresholdDb.stationAndVapBitmap, stationOrVapIndex);
	}
}


/**********************************************************************************

qosRemoveStationOrVap 


Description:
------------
Remove station listen interval or VAP DTIM period from the Qos database


Input: 
-----	
stationOrVapIndex - the index of the station or the VAP to remove (vap index is 128 - 143)
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosRemoveStationOrVap(uint16 stationOrVapIndex)
{	

	uint16 agingIntervalInQosAgingUnits;
	uint16 agingInterval;
	uint8  i = 0;
	
	if(QosGeneralParameters.isAgerEnabled)
	{

		agingIntervalInQosAgingUnits = HwQManagerAger_GetStaTtlVal(stationOrVapIndex);
		agingInterval = agingIntervalInQosAgingUnits;
		agingIntervalInQosAgingUnits >>= QOS_TTL_UNITS_TO_QOS_AGING_UNITS_SHIFT;

		ASSERT(agingIntervalInQosAgingUnits < QOS_NUMBER_OF_QOS_AGING_UNITS);

		QosPeriodicCheckParameters.qosAgingUnitsBitmap &= ~(QOS_AGING_UNITS_BITMAP_SET << agingIntervalInQosAgingUnits);
		QosPeriodicCheckParameters.qosAgingUnitsCounters[agingIntervalInQosAgingUnits] --;



		
		//ILOG0_D("[qosRemoveStationOrVap] - stationOrVapIndex 0x%x",stationOrVapIndex);
		QosGeneralParameters.numberOfValidVapsAndStations --;		

		//TBD This 'if' can be removed when an external module will take control of activation and disactivation
	    //of the periodic check - currently when no VAPs and station exist the Qos disable the Ager periodic 
		//check and enable if when the first VAP is added
		if(QosGeneralParameters.numberOfValidVapsAndStations == 0)
		{
			/* All stations and VAPs are removed - stop Ager periodic check if needed and reset values */
			if(!QosGeneralParameters.isDuringSingleCheck && QosGeneralParameters.isPeriodicCheckActive)
			{
				/* Reconfigure the Ager only if it is in the periodic check mode and the Qos is not during a single 
			   	   check operation (at the end of the single check operation the Qos will reconfigure the Ager anyway) */
   		    	Ager_StopPeriodicCheck();
			}
			/* No need to reconfigure the structure and the registers - the first VAP added will casue reconfiguration */
			//QosPeriodicCheckParameters.maximumQosAgingUnit = QOS_MAXIMUM_AGING_UNIT_DEFAULT_VALUE;
			QosGeneralParameters.isPeriodicCheckActive = FALSE;
	    }
		else if((QosPeriodicCheckParameters.qosAgingUnitsBitmap != 0) && (QosAgerPeriodicCheckConfigurationParameters.agerPeriod == agingInterval) && (QosPeriodicCheckParameters.qosAgingUnitsCounters[agingIntervalInQosAgingUnits] == 0))
	    {			
			agingInterval = Utils_FindFirstSet(QosPeriodicCheckParameters.qosAgingUnitsBitmap);
			agingInterval = agingInterval << QOS_TTL_UNITS_TO_QOS_AGING_UNITS_SHIFT;
			agingInterval += QOS_ADD_TO_GET_HIGHEST_VALUE_IN_CELL;

			QosAgerPeriodicCheckConfigurationParameters.agerPeriod = agingInterval;

			/* Change all TTL criteria to the new aging interval */
			for(i = 0; i < ACCESS_CATEGORY_NUM; i++)
			{
				QosAgerPeriodicCheckConfigurationParameters.generalParameters.acTtlCriteria[i] = agingInterval;	
			}

			if(!QosGeneralParameters.isDuringSingleCheck && QosGeneralParameters.isPeriodicCheckActive)
			{
				/* Reconfigure the Ager only if it is in the periodic check mode and the Qos is not during a single 
			   	check operation (at the end of the single check operation the Qos will reconfigure the Ager anyway) */
		    	Ager_ReconfigurePartialPeriodicCheck(&QosAgerPeriodicCheckConfigurationParameters);
			}
		}
		/*If this is the first VAP or STA stop PD Threshold timer*/
		if ((QosGeneralParameters.numberOfValidVapsAndStations == 0) &&
			(QoSPdThresholdDb.mode == QOS_PD_THRESHOLD_DYNAMIC))
		{
			OSAL_RESET_TIMER_EXPLICIT(QOS_PD_THRESOLD_TIMER, TASK_QOS);
		}
		Utils_ZeroBitInBitmap(QoSPdThresholdDb.stationAndVapBitmap, stationOrVapIndex);
 	}
}


/**********************************************************************************

qosUpdateWindow

Description:
------------
Updates the low sequence number in the station database

Input: 
-----	
stationIndex -  the station to update
tid - the tid to update
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosUpdateWindow(uint16 stationIndex, uint8 tid)
{
	hwQueueManagerPopRequestParams.primaryAddr = stationIndex;
	hwQueueManagerPopRequestParams.secondaryAddr = tid;
	//ILOG0_DD("[qosUpdateWindow] - stationIndex 0x%x tid 0x%x",stationIndex,tid);
	HwQManager_PeekHeadPacket(&hwQueueManagerPopRequestParams);
	if((NULL_PD == hwQueueManagerPopRequestParams.pHeadDesc) || !((TxPd_t *)hwQueueManagerPopRequestParams.pHeadDesc)->retransmissionIndication)
	{
		/* In case the queue is empty of the first PD hasnt been transmitted yet - take the sequence number from the station database */
		StaDB_UpdateLowSequenceNumberFromCurrent(stationIndex, tid);
	}
	else
	{
		/* Take the sequence number from the PD */
		StaDB_UpdateLowSequenceNumber(stationIndex, tid, ((TxPd_t *)hwQueueManagerPopRequestParams.pHeadDesc)->sn);
	}
}


/**********************************************************************************

qosSendConfirmToStationManager 


Description:
------------
Sends Confirmation message to STA manager

Input: 
-----
clientId -  the id that the Qos got when registered
stationIndex - the index of the station
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void qosSendConfirmToStationManager(uint16 stationIndex)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_QOS_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = stationIndex;
	/*Send confirmation message*/
	ILOG0_DD("Qos, Send Confirmation, Client %d, stationIndex %d", confirmMessage->clientId, stationIndex);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}


/**********************************************************************************

qosProcessCurrentLockedQueue


Description:
------------
This function handles a locked queue - it unlocks the queue and if needed it updates 
the sequence number and sends a message to the TS manager

Input: 
-----	
stationIndex -  the station index
tid - the TID of the queue
isDataPacketWasDiscarded - indicates if in this queue there was a data PD

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void qosProcessCurrentLockedQueue(uint16 stationIndex, uint8 tid, uint8 isDataPacketWasDiscarded)
{
	K_MSG* tsManagerMsg = QosTsManagerPacketDiscardedBitmapMessage; 
	TsManagerDataPacketDiscardedBitmapMessage_t  *tsManagerMessageParameters = NULL;
	
	if(isDataPacketWasDiscarded)
	{
		/* Updtae window */
		qosUpdateWindow(stationIndex, tid);

		/* Inform the TS manager that packet was discarded - to prevent TS manager thread queue overflow
		STA are aggregated in one message*/
		/*Ager works per TID so we wait for TID change*/
		if (QosTsManagerPacketDiscardedLastTid != tid)
		{
			/*TID changed - if we have a message in progress send it*/
			if (tsManagerMsg != NULL)
			{
				OSAL_SEND_MESSAGE(TS_MANAGER_PACKET_DISCARDED_BITMAP, TASK_TS_MANAGER, tsManagerMsg, VAP_ID_DO_NOT_CARE);
			}
			/*get a new message*/
			QosTsManagerPacketDiscardedBitmapMessage = OSAL_GET_MESSAGE(sizeof(TsManagerDataPacketDiscardedBitmapMessage_t));
			tsManagerMsg = QosTsManagerPacketDiscardedBitmapMessage;
			tsManagerMessageParameters = ((TsManagerDataPacketDiscardedBitmapMessage_t *)tsManagerMsg->abData);
			/*store the TID in the message*/
			tsManagerMessageParameters->tid = tid;
			/*Clear the STA bitmap*/
			memset(tsManagerMessageParameters->stationBitmap, 0, sizeof(uint32)*TS_MANAGER_STA_DISCARD_BITMAP_SIZE);
			/*Update last TID*/
			QosTsManagerPacketDiscardedLastTid = tid;
		}
		else
		{
			DEBUG_ASSERT(tsManagerMsg);
			tsManagerMessageParameters = ((TsManagerDataPacketDiscardedBitmapMessage_t *)tsManagerMsg->abData);
		}
		/*Set bit in STA bitmap*/
		Utils_SetBitInBitmap(tsManagerMessageParameters->stationBitmap, stationIndex);
		//("qosProcessCurrentLockedQueue packet discarded station = %d, tid = %d ", stationIndex , tid);
	}

	/* Unlock must be done here since the update window relies on the fact that the queue is locked */	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, stationIndex, (0x1 << tid));
}
    
/***********************************************************************
* QosSetAgerParams
* 
* Description:
* ------------
*  Modify AGER params
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void QosSetAgerParams(uint32 data1,  uint32 data2, uint32 data3)
{
	/*	data1: modify HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD
		data2: modify QOS_DEFAULT_FREE_PDS_HYSTERESIS
		data3: modify QOS_PD_THRESHOLD_TIMEOUT
	*/
	QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold = (uint16)data1;
	QosSingleCheckParameters.freePdsHysteresisNumber = (uint16)data2;
	QoSPdThresholdDb.pollingTimer = (uint8) data3;
	ILOG0_DDD("[DROP_DEBUG][QosSetAgerParams] hostInterfaceAcceleratorFreePdsThreshold = %d freePdsHysteresisNumber = %d pollingTimer = %d",
		QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold, QosSingleCheckParameters.freePdsHysteresisNumber, QoSPdThresholdDb.pollingTimer);
}
/***********************************************************************
* QosAgerSetPeriodicParamsters
* 
* Description:
* ------------
*  Initialize the threshold for Ager periodic check
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void QosAgerSetPeriodicParamsters()
{
	uint16 voThreshold = 0;
	uint16 viThreshold = 0;
	uint16 beThreshold = 0;
	uint16 bkThreshold = 0;	

	if (QosGeneralParameters.restrictedAc == TRUE)
	{
		/* Restricted AC is active: set threshold and TTL acording to bitmap */
		voThreshold = (QosGeneralParameters.restrictedAcBitmap & (1 << ACCESS_CATEGORY_VO)) ? QOS_MAXIMUM_PD_DISCARD_LIMIT : 0; //QOS_MAXIMUM_PD_DISCARD_LIMIT;
		viThreshold = (QosGeneralParameters.restrictedAcBitmap & (1 << ACCESS_CATEGORY_VI)) ? QOS_MAXIMUM_PD_DISCARD_LIMIT : 0; //QOS_MAXIMUM_PD_DISCARD_LIMIT;
		beThreshold = (QosGeneralParameters.restrictedAcBitmap & (1 << ACCESS_CATEGORY_BE)) ? QOS_MAXIMUM_PD_DISCARD_LIMIT : 0;
		bkThreshold = (QosGeneralParameters.restrictedAcBitmap & (1 << ACCESS_CATEGORY_BK)) ? QOS_MAXIMUM_PD_DISCARD_LIMIT : 0;
	}


	/* Set values for periodic aging (they are later copied to single check at qosFreePdsThresholdEvent) */
	QosAgerPeriodicCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VO] = voThreshold;	
	QosAgerPeriodicCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_VI] = viThreshold;
	QosAgerPeriodicCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BE] = beThreshold;
	QosAgerPeriodicCheckConfigurationParameters.generalParameters.minimumPdsToStartCheckForAc[ACCESS_CATEGORY_BK] = bkThreshold;
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

Qos_TaskEntry 


Description:
------------
the entry point of the Qos task

Input: 
-----
QosMessage - pointer to the message to handle	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void Qos_TaskEntry(K_MSG *QosMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(QosMessage, afpTaskTable, TASK_QOS_START, TASK_QOS_END);
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif


/**********************************************************************************

Qos_Initialize 


Description:
------------
initialize internal structures and parameters of the Qos module

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void Qos_Initialize(void)
{
	memset32(&QosPeriodicCheckParameters, 0, CONVERT_BYTES_TO_WORDS(sizeof(QosPeriodicCheckParameters_t)));
	memset32(&QosSingleCheckParameters, 0, CONVERT_BYTES_TO_WORDS(sizeof(QosSingleCheckParameters_t)));
	memset(&QosAgerSingleCheckConfigurationParameters, 0, sizeof(AgerSingleCheckParameters_t));
	memset(&QosAgerPeriodicCheckConfigurationParameters, 0, sizeof(AgerPeriodicCheckParameters_t));
	memset(&QosGeneralParameters, 0, sizeof(QosGeneralParameters_t));
	memset(&hwQueueManagerPopRequestParams, 0, sizeof(HwQueueManagerRequestParams_t));

	hwQueueManagerPopRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerPopRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerPopRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;

    //TBD this parameter is used to decide if to take an action when an event happens. When it is decided to use the
    //Ager permanently this parameter should be removed
#ifdef DISABLE_AGER
	QosGeneralParameters.isAgerEnabled = FALSE;
#else
	QosGeneralParameters.isAgerEnabled = TRUE;
#endif	
#ifdef ATF_INIT_DIST_TYPE_STATIC //KW_FIX_FW_G
	QosGeneralParameters.isAtfStaticDistType[CONFIGURATION_MANAGER_BAND_0] = TRUE;
	QosGeneralParameters.isAtfStaticDistType[CONFIGURATION_MANAGER_BAND_1] = TRUE;
#else
	QosGeneralParameters.isAtfStaticDistType[CONFIGURATION_MANAGER_BAND_0] = FALSE;
	QosGeneralParameters.isAtfStaticDistType[CONFIGURATION_MANAGER_BAND_1] = FALSE;
#endif
	/* Currently the only mode is discarding a specific amount of PDs during a single check */
	QosAgerSingleCheckConfigurationParameters.isEnabledMaximumLimit = TRUE;

	//QosPeriodicCheckParameters.maximumQosAgingUnit = QOS_MAXIMUM_AGING_UNIT_DEFAULT_VALUE;

//TBD this initialization will be used till the configuration of FW modules will be changed
   QosSingleCheckParameters.numberOfPdsLimits = QOS_DEFAULT_NUMBER_OF_PD_LIMITS;
   /* The values are uint16 se need to divide by 2 */
   memcpy32(QosSingleCheckParameters.pdLimitValues, &QosDefaultPdsLimitsThresholds, (QOS_DEFAULT_NUMBER_OF_PD_LIMITS >> 1));
   QosSingleCheckParameters.freePdsHysteresisNumber = QOS_DEFAULT_FREE_PDS_HYSTERESIS; 
	QosGeneralParameters.restrictedAc = FALSE;
///
	/*Initialize TS manager Packet Discarded Bitmap message*/
	QosTsManagerPacketDiscardedBitmapMessage = NULL;
	QosTsManagerPacketDiscardedLastTid = QOS_INVALID_TID_INDEX;

#ifdef DISABLE_AGER
	QoSPdThresholdDb.mode = QOS_PD_THRESHOLD_DISABLED;				
#else
	QoSPdThresholdDb.mode = QOS_PD_THRESHOLD_FORCED;			
#endif	
	QoSPdThresholdDb.state = QOS_PD_THRESHOLD_STATE_OFF;			/*PD Threshold check is off until needed*/
	QoSPdThresholdDb.minPdAmount = QOS_PD_THRESHOLD_MIN_AMOUNT_DEFAULT;
	QoSPdThresholdDb.minPdDiff = QOS_PD_THRESHOLD_MIN_DIFF_DEFAULT;
	QoSPdThresholdDb.pollingTimer = QOS_PD_THRESHOLD_TIMEOUT;
	QoSPdThresholdDb.hostInterfaceAcceleratorFreePdsThreshold = HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD;
	memset32(QoSPdThresholdDb.stationAndVapBitmap, 0, QOS_PD_THRESHOLD_STATION_AND_VAP_BITMAP_SIZE);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
