/***********************************************************************************
 File:			RegAccess.c
 Module:		RegAccess
 Purpose:		Access to Hw Registers
 Description:	This module is the API for accessing registers in the HW 
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "EmeraldEnvRegs.h"
#include "HwMemoryMap.h"
#include "mt_sysrst.h"
#include "loggerAPI.h"
#include "PhyRegsIncluder.h"



/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_REG_ACCESS
#define LOG_LOCAL_FID 0


#define SECURED_WRITE_STEP1_VALUE	0xaaaa
#define SECURED_WRITE_STEP2_VALUE	0x5555

#if defined (ENET_INC_ARCH_WAVE600)
#define BAND1_OFFSET_FROM_BAND0		BAND1_INTERNAL_SIZE
#endif


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

RegAccess_Read 

Description:
------------
	Read 32-bit value from Register

Input:
-----
	uint32 regAddress - Full register address
	uint32 * regValue - Pointr for the return value 
Output:
-------
	uint32 *regValue - Register Value
Returns:
--------
	void 
**********************************************************************************/
void RegAccess_Read(uint32 regAddress, volatile uint32 *regValue)
{
	*regValue = *((volatile uint32*)(regAddress));
}

/**********************************************************************************

RegAccess_ReadImmediate 

Description:
------------
	Read 32-bit value from Register

Input:
-----
	uint32 regAddress - Full register address
	uint32 * regValue - Pointr for the return value 
Output:
-------
	uint32 *regValue - Register Value
Returns:
--------
	void 
**********************************************************************************/
uint32 RegAccess_ReadImmediate(uint32 regAddress)
{
	uint32 regValue;
	
	regValue = *((volatile uint32*)(regAddress));	

	return regValue;
}
	

/**********************************************************************************

RegAccess_Write 

Description:
------------
	Write 32-Bit Value to Register

Input:
-----
	uint32 regAddress	- Full register address
	uint32 regValue		- The required value to write into the register
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_Write(uint32 regAddress, uint32 regValue)
{
	*((volatile uint32*)(regAddress)) = regValue;
	//reg_access_trace(regAddress,regValue);
}


/**********************************************************************************

RegAccess_WriteMasked 

Description:
------------
	Write 32-Bit Value masked with the relevant required mask value to Register 

Input:
-----
	uint32 regAddress	- Full register address
	uint32 mask			- The required Mask.
	uint32 data			- The required value to write into the register.
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WriteMasked(uint32 regAddress, uint32 mask, uint32 data)
{
	uint32 regCurrentValue;
	uint32 regNewValue;
	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_Read(regAddress, &regCurrentValue);
	regNewValue = (regCurrentValue & (~mask)) | (data & mask);
	RegAccess_Write(regAddress, regNewValue);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

RegAccess_WriteSecured 

Description:
------------
	Write 32-Bit Value to secured registers - 
		

Input:
-----
	uint32 regAddress	- Full register address
	uint32 mask		- The required Mask.
	uint32 data		- The required value to write into the register.
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WriteSecured(uint32 regAddress,uint32 data)
{
#if defined(ENET_INC_LMAC)
	DEBUG_FATAL("REG Access Write Secured");
#endif

	RegAccess_Write(REG_EMERALD_ENV_SECURE_WRITE, SECURED_WRITE_STEP1_VALUE);
	RegAccess_Write(REG_EMERALD_ENV_SECURE_WRITE, SECURED_WRITE_STEP2_VALUE);
	RegAccess_Write(regAddress,data );
}



#if defined (ENET_INC_ARCH_WAVE600)

/**********************************************************************************

RegAccess_ReadPerBand 

Description:
------------
	Read 32-bit value from Register in a given band

Input:
-----
	uint32 regAddress 	- Full register address
	uint32 * regValue 	- Pointr for the return value 
	uint32 band 		- band index (0/1)
	
Output:
-------
	uint32 *regValue - Register Value
Returns:
--------
	void 
**********************************************************************************/
void RegAccess_ReadPerBand(uint32 regAddress, volatile uint32 *regValue, uint32 band)
{
	uint32 regAddressInBand;


	regAddressInBand = regAddress + (BAND1_OFFSET_FROM_BAND0 * band);
	*regValue = *((volatile uint32*)(regAddressInBand));
}

/**********************************************************************************

RegAccess_ReadImmediatePerBand 

Description:
------------
	Read 32-bit value from Register in a given band

Input:
-----
	uint32 regAddress 	- Full register address
	uint32 * regValue 	- Pointr for the return value 
	uint32 band 		- band index (0/1)

Output:
-------
	uint32 *regValue - Register Value
Returns:
--------
	void 
**********************************************************************************/
uint32 RegAccess_ReadImmediatePerBand(uint32 regAddress, uint32 band)
{
	uint32 regValue;
	uint32 regAddressInBand;


	regAddressInBand = regAddress + (BAND1_OFFSET_FROM_BAND0 * band);
	regValue = *((volatile uint32*)(regAddressInBand));	

	return regValue;
}


/**********************************************************************************

RegAccess_WritePerBand 

Description:
------------
	Write 32-Bit Value to Register in a given band

Input:
-----
	uint32 regAddress	- Full register address
	uint32 regValue	- The required value to write into the register
	uint32 band		- band index (0/1)
	
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WritePerBand(uint32 regAddress, uint32 regValue, uint32 band)
{
	uint32 regAddressInBand;

	regAddressInBand = regAddress + (BAND1_OFFSET_FROM_BAND0 * band);
	
	*((volatile uint32*)(regAddressInBand)) = regValue;
}


/**********************************************************************************

RegAccess_WriteMaskedPerBand 

Description:
------------
	Write 32-Bit Value masked with the relevant required mask value to Register 

Input:
-----
	uint32 regAddress		- Full register address
	uint32 mask			- The required Mask.
	uint32 data			- The required value to write into the register.
	uint32 band 			- band index (0/1)
	
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WriteMaskedPerBand(uint32 regAddress, uint32 mask, uint32 data, uint32 band)
{
	uint32 regCurrentValue;
	uint32 regNewValue;	
	uint32 regAddressInBand;

	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	regAddressInBand = regAddress + (BAND1_OFFSET_FROM_BAND0 * band);
	RegAccess_Read(regAddressInBand, &regCurrentValue);
	regNewValue = (regCurrentValue & (~mask)) | (data & mask);
	RegAccess_Write(regAddressInBand, regNewValue);	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}


/**********************************************************************************

RegAccess_WritePerAntennaMasked 

Description:
------------
	Write 32-Bit Value masked with the relevant required mask value to Register 

Input:
-----
	uint32 regAddress	- Full register address
	uint32 mask			- The required Mask.
	uint32 data			- The required value to write into the register.
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WritePerAntMaskMasked(uint32 regAddress, uint32 mask, uint32 antmask, uint32 data)
{
	uint32 regCurrentValue;
	uint32 regNewValue;
	uint32 regNewAddress;
	uint32 ant;
	TX_INTERRUPT_SAVE_AREA;

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++) 
	{
		if (ANTENNA_MASK_IS_BIT_ON(antmask,ant))
			{
				regNewAddress = regAddress + (ant*ANT_REGS_OFFSET) ; 
				
				OSAL_DISABLE_INTERRUPTS(&interrupt_save);
				RegAccess_Read(regNewAddress, &regCurrentValue);
				regNewValue = (regCurrentValue & (~mask)) | (data & mask);
				RegAccess_Write(regNewAddress, regNewValue);
				
#ifdef PSD_DEBUG 
				RegAccess_Read(regNewAddress, &regNewValue);
				PhyRegAntDebug[ant].regAddress[CurrentRowIndex] = regNewAddress;
				PhyRegAntDebug[ant].prevdata[CurrentRowIndex]= regCurrentValue;
				PhyRegAntDebug[ant].newdata[CurrentRowIndex]= regNewValue;
#endif	
				
				OSAL_ENABLE_INTERRUPTS(interrupt_save);
			}
	}
	
}


/**********************************************************************************

RegAccess_WaitForFieldPerBand

Description:
------------
	Wait for field to be equal to expected value
		

Input:
-----
	uint32 regAddress	- Full register address
	uint8  shift		- field shift
	uint32 mask			- field mask
	uint32 expectedValue- expected value
	uint32 timeout		- field polling timeout [us]
	bool*  pSuccess		- TRUE if field value was expected before timeout reached
	uint8  bandId       - band Index (0/1)
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WaitForFieldPerBand(uint32 regAddress, uint8 shift, uint32 mask, uint32 expectedValue, uint32 timeout, bool* pSuccess, uint8 bandId)
{	
	uint32 start_time;

    start_time = TIME_STAMP(START_TIME, 0);
	*pSuccess = FALSE;
	while ((*pSuccess == FALSE) && (TIME_STAMP(END_TIME, start_time) < timeout))
	{
		RegAccess_ReadAndVerifyFieldPerBand(regAddress, shift, mask, expectedValue, pSuccess, bandId);
	}
}


/**********************************************************************************

RegAccess_ReadAndVerifyFieldPerBand 

Description:
------------
	Read and verify read field value from registerr
		

Input:
-----
	uint32 regAddress	- Full register address
	uint8  shift		- field shift
	uint32 mask			- field mask
	uint32 value		- expected value
	bool*  pSuccess		- TRUE if register value matches expected value
	uint8  bandId       - band Index (0/1)
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_ReadAndVerifyFieldPerBand(uint32 regAddress, uint8 shift, uint32 mask, uint32 value, bool* pSuccess, uint8 bandId)
{	
	uint32 fieldVal;

    RegAccess_ReadFieldPerBand(regAddress, shift, mask, &fieldVal, bandId);
	if (fieldVal == value)
	{
		*pSuccess = TRUE;
	}
	else
	{
		*pSuccess = FALSE;
	}
}


/**********************************************************************************

RegAccess_ReadFieldPerBand 

Description:
------------
	Read specific field from register		
Input:
-----
	uint32  regAddress	- Full register address
	uint8   shift		- field shift
	uint32  mask	    - field mask
	uint32* pFieldValue	- field value
	uint8   bandId      - band Index (0/1)
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_ReadFieldPerBand(uint32 regAddress, uint8 shift, uint32 mask, uint32* pFieldValue, uint8 bandId)
{
	RegAccess_ReadPerBand(regAddress, pFieldValue, bandId);
	(*pFieldValue) &= mask;
	(*pFieldValue) >>= shift;
}


#endif

/**********************************************************************************

RegAccess_Write64bitReg 

Description:
------------
	Write 64-Bit Value
		

Input:
-----
	uint32 regAddress	- Full register address
	uint32* RegValue	- array of 2 uint32 value (index 0 is for low address)
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_Write64bitReg(uint32 regAddress, uint32* regValue)
{
	// Warning - This function called from UM will access band0. If it's called from LM it will access the corresponding band. 
	// If needed - create new API for UM to read per band (like "RegAccess_ReadPerBand")
	RegAccess_Write(regAddress, regValue[0]);
	RegAccess_Write(regAddress + 4, regValue[1]);
}

/**********************************************************************************

RegAccess_ReadField 

Description:
------------
	Read specific field from register
		

Input:
-----
	uint32 regAddress	- Full register address
	uint8  shift		- field shift
	uint32 mask			- field mask
	uint32* fieldValue	- field value
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_ReadField(uint32 regAddress, uint8 shift, uint32 mask, uint32* fieldValue)
{
	// Warning - This function called from UM will access band0. If it's called from LM it will access the corresponding band. 
	// If needed - create new API for UM to read per band (like "RegAccess_ReadPerBand")
	RegAccess_Read(regAddress, fieldValue);
	(*fieldValue) &= mask;
	(*fieldValue) >>= shift;
}

/**********************************************************************************

RegAccess_ReadAndVerify 

Description:
------------
	Read and verify read value from registerr
		

Input:
-----
	uint32 regAddress	- Full register address
	uint32 value		- expected value
	bool   success		- TRUE if register value matches expected value
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_ReadAndVerify(uint32 regAddress, uint32 value, bool* success)
{
	// Warning - This function called from UM will access band0. If it's called from LM it will access the corresponding band. 
	// If needed - create new API for UM to read per band (like "RegAccess_ReadPerBand")
	
	uint32 regVal;

	RegAccess_Read(regAddress, &regVal);
	if (regVal == value)
	{
		*success = TRUE;
	}
	else
	{
		*success = FALSE;
	}
}

/**********************************************************************************

RegAccess_ReadAndVerifyField 

Description:
------------
	Read and verify read field value from registerr
		

Input:
-----
	uint32 regAddress	- Full register address
	uint8  shift		- field shift
	uint32 mask			- field mask
	uint32 value		- expected value
	bool   success		- TRUE if register value matches expected value
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_ReadAndVerifyField(uint32 regAddress, uint8 shift, uint32 mask, uint32 value, bool* success)
{
	// Warning - This function called from UM will access band0. If it's called from LM it will access the corresponding band. 
	// If needed - create new API for UM to read per band (like "RegAccess_ReadPerBand")
	
	uint32 fieldVal;
	RegAccess_ReadField(regAddress, shift, mask, &fieldVal);
	if (fieldVal == value)
	{
		*success = TRUE;
	}
	else
	{
		*success = FALSE;
	}
}

/**********************************************************************************

RegAccess_WriteInc4 

Description:
------------
	Write register and increament register address by 4 bytes
		

Input:
-----
	uint32 regAddress	- Full register address
	uint32 value		- Value to write
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WriteInc4(uint32* regAddress, uint32 value)
{
	// Warning - This function called from UM will access band0. If it's called from LM it will access the corresponding band. 
	// If needed - create new API for UM to read per band (like "RegAccess_ReadPerBand")
	
	RegAccess_Write(*regAddress, value);
	(*regAddress) += 4;
}

/**********************************************************************************

RegAccess_WaitForField 

Description:
------------
	Wait for field to be equal to expected value
		

Input:
-----
	uint32 regAddress	- Full register address
	uint8  shift		- field shift
	uint32 mask			- field mask
	uint32 expectedValue- expected value
	uint32 timeout		- field polling timeout [us]
	bool success		- TRUE if field value was expected before timeout reached
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void RegAccess_WaitForField(uint32 regAddress, uint8 shift, uint32 mask, uint32 expectedValue, uint32 timeout, bool* success)
{
	// Warning - This function called from UM will access band0. If it's called from LM it will access the corresponding band. 
	// If needed - create new API for UM to read per band (like "RegAccess_ReadPerBand")
	
	uint32 start_time = TIME_STAMP(START_TIME, 0);

	*success = FALSE;
	//KW_FW_FIX_J
	while ((*success == FALSE) && (TIME_STAMP(END_TIME, start_time) < timeout))
	{
		RegAccess_ReadAndVerifyField(regAddress, shift, mask, expectedValue, success);
	}
}



