/************************************************************************************
*    
*    File:		 ResourceManager.c
*    Class/Module: <Class/Module Name>
*    Description: manager the UMAC internal resources
*                        
*		 	  
*
*    COPYRIGHT: 
*        (C) Lantiq Israel Ltd.
*        All rights are strictly reserved. Reproduction or divulgence in any   
* 	  form whatsoever is not permitted without written authority from the 
*        copyright owner. Issued by Lantiq Israel Ltd
*
**************************************************************************************/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "ResourceManager_API.h"
#include "OSAL_Interrupts.h" 
#include "HwGlobalDefinitions.h"
#include "loggerAPI.h"
#include "frame.h"
#include "lmi.h"
#include "mhi_descriptors.h"
#include "stringLibApi.h"
#include "HwQManager_API.h"
#include "shramTxDesc.h"
#include "ShramPacketDescriptors.h"
#include "ShramDataTxDescriptors.h"
#include "TsManager_API.h"
#include "OSAL_UpperMacMessages.h"

#define LOG_LOCAL_GID   GLOBAL_GID_WAVE500
#define LOG_LOCAL_FID 0

#define RM_MSG_TYPE_WIDTH	(10)

#ifdef ENET_INC_ARCH_WAVE600
#define RM_ID_WIDTH			(14)
#define RM_CONTEXT_WIDTH	(13)
#else
#define RM_ID_WIDTH			(11)
#define RM_CONTEXT_WIDTH	(11)
#endif //ENET_INC_ARCH_WAVE600

#define	RM_RETURNED_TASK_WIDTH	(8)

#define RM_MSG_SENT_WIDTH		(1)

#if (!defined(TS_MANAGER_INITIATOR_PD_REQUEST_ID_WIDTH) || !defined(TS_MANAGER_RECIPIENT_PD_REQUEST_ID_WIDTH) || ((RM_ID_WIDTH < TS_MANAGER_INITIATOR_PD_REQUEST_ID_WIDTH) || (RM_ID_WIDTH < TS_MANAGER_RECIPIENT_PD_REQUEST_ID_WIDTH)))
#error "RM_ID_WIDTH Must greater or equal to TS_MANAGER_INITIATOR_PD_REQUEST_ID_WIDTH and TS_MANAGER_RECIPIENT_PD_REQUEST_ID_WIDTH"
#endif

#if (((HW_NUM_OF_STATIONS * NUM_TID * 2) - NUM_OF_TS_MANAGER_DESCRIPTORS) > (1 << RM_ID_WIDTH))
#error "RM_ID_WIDTH must accomdate max request ID"
#endif

#if (TS_MANAGER_RM_REQUEST_SIZE > RM_CONTEXT_WIDTH)
#error "RM_CONTEXT_WIDTH must accomdate TS_MANAGER_RM_REQUEST_SIZE"
#endif

#define RM_MAX_ID ((1 << RM_ID_WIDTH) - 1)


#ifdef ENET_INC_ARCH_WAVE600

#define PADDING_LENGTH_W0 (32 - (RM_ID_WIDTH + RM_ID_WIDTH + RM_MSG_SENT_WIDTH))
#define PADDING_LENGTH_W1 (32 - (RM_MSG_TYPE_WIDTH + RM_CONTEXT_WIDTH + RM_RETURNED_TASK_WIDTH))
#define PADDING_LENGTH_W2 (32 - (RM_ID_WIDTH))

#if (RM_ID_WIDTH+ RM_ID_WIDTH +RM_MSG_SENT_WIDTH > 32)
#error "RM_ID_WIDTH+RM_ID_WIDTH+RM_MSG_SENT_WIDTH must accomdate WORD size"
#endif
#if (RM_ID_WIDTH+ RM_ID_WIDTH +RM_MSG_SENT_WIDTH > 32)
#error "RM_MSG_TYPE_WIDTH+RM_CONTEXT_WIDTH+RM_RETURNED_TASK_WIDTH must accomdate WORD size"
#endif


typedef struct RmFreePendingDesc_tag
{
/* Word0 */
	uint32 nextReqId		: RM_ID_WIDTH;
	uint32 previousReqId	: RM_ID_WIDTH;
	uint32 msgSent			: RM_MSG_SENT_WIDTH;
	uint32 reserved_W0		: PADDING_LENGTH_W0;
/* Word1 */
	uint32 returnMsgType	: RM_MSG_TYPE_WIDTH;	/*Must accomodate max message size*/
	uint32 context			: RM_CONTEXT_WIDTH;
	uint32 returnTaskId		: RM_RETURNED_TASK_WIDTH;
	uint32 reserved_W1		: PADDING_LENGTH_W1;
/* Word2 */
	uint32 id				: RM_ID_WIDTH;
	uint32 reserved_W2		: PADDING_LENGTH_W2; 

}RmFreePendingDesc_t;

#else 

#define PADDING_LENGTH_W1 (32 - (RM_ID_WIDTH + RM_RETURNED_TASK_WIDTH + RM_MSG_SENT_WIDTH + RM_CONTEXT_WIDTH))

#if ((RM_ID_WIDTH + RM_RETURNED_TASK_WIDTH + RM_MSG_SENT_WIDTH + RM_CONTEXT_WIDTH) > 32)
#error "RM_ID_WIDTH+ RM_RETURNED_TASK_WIDTH +RM_MSG_SENT_WIDTH+ RM_CONTEXT_WIDTH must accomdate WORD size"
#endif


typedef struct RmFreePendingDesc_tag
{
/* Word0*/
	uint32 id				: RM_ID_WIDTH;
	uint32 nextReqId		: RM_ID_WIDTH;
	uint32 returnMsgType	: RM_MSG_TYPE_WIDTH;	/*Must accomodate max message size*/
/* Word1*/
	uint32 previousReqId	: RM_ID_WIDTH;
	uint32 returnTaskId		: RM_RETURNED_TASK_WIDTH;
	uint32 msgSent			: RM_MSG_SENT_WIDTH;
	uint32 context 			: RM_CONTEXT_WIDTH;
	uint32 reserved_W1		: PADDING_LENGTH_W1;
}RmFreePendingDesc_t;

#endif //ENET_INC_ARCH_WAVE600


typedef enum
{
	RM_FREE_PENDING_LIST_DESC_TYPE_FREE,
	RM_FREE_PENDING_LIST_DESC_TYPE_PENDING,
	RM_FREE_PENDING_LIST_DESC_TYPE_LAST
}RmFreePendingListDescType_e;

typedef struct RmFreePendingDescQueue_tag
{
	uint16 reqIdHead;
	uint16 reqIdTail;
}RmFreePendingDescQueue_t;

typedef enum 
{
	FREE_PENDING_DESC_POOL_TS_MANAGER = 0,
	FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW,
	FREE_PENDING_DESC_POOL_CSA_MANAGER,
	FREE_PENDING_DESC_POOL_MAX,	
}FreePendingDescPool_e;

typedef struct RmFreePendingDescQueues_tag
{
	RmFreePendingDescQueue_t rmFreePendingFreeListDescQueue[FREE_PENDING_DESC_POOL_MAX];
	RmFreePendingDescQueue_t rmFreePendingPendingListDescQueue[FREE_PENDING_DESC_POOL_MAX];
}RmFreePendingDescQueues_t;

#if defined (ENET_INC_ARCH_WAVE600)
#define MAX_RESOURCE_REQUEST_TS_MANAGER					(HW_NUM_OF_STATIONS * NUM_TID * 2)	/*Maximum is HW_NUM_OF_STATIONS * NUM_TID * 2*/
#else
//#define MAX_RESOURCE_REQUEST_TS_MANAGER					(384 + NUM_OF_TS_MANAGER_DESCRIPTORS)	/*Maximum is HW_NUM_OF_STATIONS * NUM_TID * 2*/
#define MAX_RESOURCE_REQUEST_TS_MANAGER					(256 + NUM_OF_TS_MANAGER_DESCRIPTORS)	/*Maximum is HW_NUM_OF_STATIONS * NUM_TID * 2*/
#endif

#define MAX_RESOURCE_REQUEST_MANAGEMENT_FOR_FW			(HW_NUM_OF_STATIONS)		/*SMPS per-STA*/
#define MAX_RESOURCE_REQUEST_CSA						(HW_NUM_OF_STATIONS + 1) /*CSA per-STA + one CTS*/

#define FREE_PENDING_DESC_POOL_TS_MANAGER_SIZE			(MAX_RESOURCE_REQUEST_TS_MANAGER - NUM_OF_TS_MANAGER_DESCRIPTORS)
#define FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW_SIZE	(MAX_RESOURCE_REQUEST_MANAGEMENT_FOR_FW - NUM_OF_MANAGEMENT_DESCRIPTORS)
#define FREE_PENDING_DESC_POOL_CSA_MANAGER_SIZE			(MAX_RESOURCE_REQUEST_CSA - NUM_OF_CSA_MANAGER_DESCRIPTORS)

typedef struct RmFreePendingDescPools_tag
{
	RmFreePendingDesc_t rmFreePendingDescPoolTsManager[FREE_PENDING_DESC_POOL_TS_MANAGER_SIZE];
	RmFreePendingDesc_t rmFreePendingDescPoolManagementFromFw[FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW_SIZE];
	RmFreePendingDesc_t rmFreePendingDescPoolCsaManager[FREE_PENDING_DESC_POOL_CSA_MANAGER_SIZE];

}RmFreePendingDescPools_t;

RmFreePendingDescQueues_t rmFreePendingDescQueues;
RmFreePendingDescPools_t rmFreePendingDescPools;
static PacketDescriptorPool_t PacketDescPool[NUM_OF_DESC_POOLS];

// local functions
void RmInitPayload(DescPoolTypes_e PoolType , TxPd_t* pDescriptors , uint32 EntrySize , uint8* pPayloadPool);
void RmInitPool(DescPoolTypes_e PoolType , uint8* pDescriptors, uint16 sizeOfDescriptorInWords,uint16 numOfDescInPool, uint8 PdSource);
RmFreePendingDesc_t *RmPopFromFreePendingList(FreePendingDescPool_e freePendingDescPool, RmFreePendingDescQueue_t *rmFreePendingDescQueue_p);
RmFreePendingDescQueue_t *RmGetFreePendingDescQueueByType(FreePendingDescPool_e freePendingDescPool, RmFreePendingListDescType_e rmFreePendingListDescType);
FreePendingDescPool_e RmGetPoolTypeIfSupportPendingList(DescPoolTypes_e poolType);
uint16 RmGetMaxFreePendingDescPoolSize(FreePendingDescPool_e freePendingDescPool);
RmFreePendingDescQueue_t *RmGetFreePendingDescQueueByType(FreePendingDescPool_e freePendingDescPool, RmFreePendingListDescType_e rmFreePendingListDescType);
RmFreePendingDesc_t *RmGetFreePendingDescElementPoolByType(uint16 reqId, FreePendingDescPool_e freePendingDescPool);
void RmRemoveFromFreePendingList(RmFreePendingDesc_t *descToRemove_p, RmFreePendingDescQueue_t *rmFreePendingDescQueue_p, FreePendingDescPool_e freePendingDescPool);
void RmPushToFreeList(RmFreePendingDesc_t *rmFreePendingDesc_p, FreePendingDescPool_e freePendingDescPool);
void RmPushToPendingList(RmFreePendingDesc_t *rmFreePendingDesc_p, FreePendingDescPool_e freePendingDescPool);
void RmInitFreePendingListPool(void);
void RmInitFreePendingListQueue(void);

/**********************************************************************************
RmGetPoolTypeIfSupportPendingList 


Description:
------------
returns pool of type FreePendingDescPool_e if pool supports pending list

Input:
-----
DescPoolTypes_e poolType

Output:
-------

Returns:
--------
pool of type FreePendingDescPool_e
**********************************************************************************/
FreePendingDescPool_e RmGetPoolTypeIfSupportPendingList(DescPoolTypes_e poolType)
{
	FreePendingDescPool_e retVal = FREE_PENDING_DESC_POOL_MAX;
	
	switch (poolType)
	{
		case DESC_POOL_TS_MANAGER:
			retVal = FREE_PENDING_DESC_POOL_TS_MANAGER;
			break;
		case DESC_POOL_MC_DEAUTH:
		case DESC_POOL_MANAGEMENT_FROM_FW:
			retVal = FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW;
			break;
		case DESC_POOL_CSA_MANAGER:
			retVal = FREE_PENDING_DESC_POOL_CSA_MANAGER;
			break;
		default:
			break;
	}
	return retVal;
}

/**********************************************************************************

RmGetMaxFreePendingDescPoolSize 


Description:
------------
returns pool max size

Input:
-----
FreePendingDescPool_e freePendingDescPool


Output:
-------

Returns:
--------
pool max size
**********************************************************************************/
uint16 RmGetMaxFreePendingDescPoolSize(FreePendingDescPool_e freePendingDescPool)
{
	uint16 retValMaxSize = 0;
	
	switch (freePendingDescPool)
	{
		case FREE_PENDING_DESC_POOL_TS_MANAGER:
			retValMaxSize = FREE_PENDING_DESC_POOL_TS_MANAGER_SIZE;
			break;
		case FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW:
			retValMaxSize = FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW_SIZE;
			break;
		case FREE_PENDING_DESC_POOL_CSA_MANAGER:
			retValMaxSize = FREE_PENDING_DESC_POOL_CSA_MANAGER_SIZE;
			break;
		default:
			break;
	}
	return retValMaxSize;
}

/**********************************************************************************

RmGetFreePendingDescQueueByType 


Description:
------------
returns pointer to free pending desc queue by type

Input:
-----
FreePendingDescPool_e freePendingDescPool


Output:
-------

Returns:
--------
RmFreePendingDescriptorQueue_t *
**********************************************************************************/
RmFreePendingDescQueue_t *RmGetFreePendingDescQueueByType(FreePendingDescPool_e freePendingDescPool, RmFreePendingListDescType_e rmFreePendingListDescType)
{
	RmFreePendingDescQueue_t *rmFreePendingDescQueue_p = PNULL;
	
	switch(rmFreePendingListDescType)
	{
		case RM_FREE_PENDING_LIST_DESC_TYPE_FREE:
			rmFreePendingDescQueue_p = &rmFreePendingDescQueues.rmFreePendingFreeListDescQueue[freePendingDescPool];
			break;
		case RM_FREE_PENDING_LIST_DESC_TYPE_PENDING:
			rmFreePendingDescQueue_p = &rmFreePendingDescQueues.rmFreePendingPendingListDescQueue[freePendingDescPool];
			break;
		default:
			DEBUG_ASSERT(0);
			break;
	}	
	return rmFreePendingDescQueue_p;
}

/**********************************************************************************

RmGetFreePendingDescElementPoolByType 


Description:
------------
returns pointer to free pending desc pool by type and req id

Input:
-----
DescPoolTypes_e PoolType

Output:
-------

Returns:
--------
RmFreePendingDesc_t *

**********************************************************************************/
RmFreePendingDesc_t *RmGetFreePendingDescElementPoolByType(uint16 reqId, FreePendingDescPool_e freePendingDescPool)
{
	RmFreePendingDesc_t *retVal = PNULL;
	
	switch (freePendingDescPool)
	{
		case FREE_PENDING_DESC_POOL_TS_MANAGER:
			ASSERT(reqId < FREE_PENDING_DESC_POOL_TS_MANAGER_SIZE);			
			retVal = &rmFreePendingDescPools.rmFreePendingDescPoolTsManager[reqId];
			break;
		case FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW:
			ASSERT(reqId < FREE_PENDING_DESC_POOL_MANAGEMENT_FROM_FW_SIZE);			
			retVal = &rmFreePendingDescPools.rmFreePendingDescPoolManagementFromFw[reqId];
			break;
		case FREE_PENDING_DESC_POOL_CSA_MANAGER:
			ASSERT(reqId < FREE_PENDING_DESC_POOL_CSA_MANAGER_SIZE);			
			retVal = &rmFreePendingDescPools.rmFreePendingDescPoolCsaManager[reqId];
			break;
		default:
			break;
	}
	DEBUG_ASSERT(retVal != PNULL);
	return retVal;
}

/**********************************************************************************

RmPopFromFreePendingList 


Description:
------------
gets pointer of Free Pending Id Block from resource managers free list

Input:
-----
DescPoolTypes_e PoolType

Output:
-------

Returns:
--------
RmFreePendingDesc_t *
**********************************************************************************/
RmFreePendingDesc_t *RmPopFromFreePendingList(FreePendingDescPool_e freePendingDescPool, RmFreePendingDescQueue_t *rmFreePendingDescQueue_p)
{
	RmFreePendingDesc_t *headDesc_p = PNULL;
	RmFreePendingDesc_t *nextDesc_p = PNULL;

	if (rmFreePendingDescQueue_p->reqIdHead != RM_MAX_ID)
	{
		headDesc_p = RmGetFreePendingDescElementPoolByType(rmFreePendingDescQueue_p->reqIdHead, freePendingDescPool);
		// update the next from the chain (it should always be the first - it's a pop function.)
		if (headDesc_p->nextReqId != RM_MAX_ID)
		{
			nextDesc_p = RmGetFreePendingDescElementPoolByType(headDesc_p->nextReqId, freePendingDescPool);
			/*Popping from head - previous must be NULL*/
			nextDesc_p->previousReqId = RM_MAX_ID;
		}
		else
		{
			/*Entry was head - set tail to NULL*/
			rmFreePendingDescQueue_p->reqIdTail = RM_MAX_ID;
		}
		/*Set head to next*/
		rmFreePendingDescQueue_p->reqIdHead = headDesc_p->nextReqId;
		// free pointers of the just popped free pending descriptor
		headDesc_p->nextReqId = RM_MAX_ID;
		headDesc_p->previousReqId = RM_MAX_ID;
	}
	// in case there are no free requestId, return PNULL
	return (headDesc_p);
}

/**********************************************************************************
RmRemoveFromFreePendingList 

Description:
------------
remove from free-pending list , update queue and pool

Input:
-----
RmFreePendingDesc_t *descToRemove_p, RmFreePendingDescQueue_t *rmFreePendingDescQueue_p


Output:
-------

Returns:
--------
none

**********************************************************************************/
void RmRemoveFromFreePendingList(RmFreePendingDesc_t *descToRemove_p, RmFreePendingDescQueue_t *rmFreePendingDescQueue_p, FreePendingDescPool_e freePendingDescPool)
{
	RmFreePendingDesc_t *previousDesc_p 	= PNULL;
	RmFreePendingDesc_t *nextDesc_p 		= PNULL;

	// remove from pending
	if (descToRemove_p->previousReqId != RM_MAX_ID)
	{
		previousDesc_p 	= RmGetFreePendingDescElementPoolByType(descToRemove_p->previousReqId, freePendingDescPool);
		previousDesc_p->nextReqId = descToRemove_p->nextReqId;
	}
	else
	{
		/*entry was head - set head to next*/
		rmFreePendingDescQueue_p->reqIdHead = descToRemove_p->nextReqId;
	}
	if (descToRemove_p->nextReqId != RM_MAX_ID)
	{
		nextDesc_p 		= RmGetFreePendingDescElementPoolByType(descToRemove_p->nextReqId, freePendingDescPool);
		nextDesc_p->previousReqId = descToRemove_p->previousReqId;
	}
	else
	{
		/*entry was tail - set tail to previous*/
		rmFreePendingDescQueue_p->reqIdTail = descToRemove_p->previousReqId;
	}
}

/**********************************************************************************

RmPushToFreeList 


Description:
------------
push element to Free list - to head

Input:
-----
RmFreePendingDesc_t *rmFreePendingDesc_p, FreePendingDescPool_e freePendingDescPool


Output:
-------

Returns:
--------
none
**********************************************************************************/
void RmPushToFreeList(RmFreePendingDesc_t *rmFreePendingDesc_p, FreePendingDescPool_e freePendingDescPool)
{
	RmFreePendingDescQueue_t *rmFreePendingDescQueue_p = RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_FREE);
	// save current head for later use
	RmFreePendingDesc_t *rmFreePendingDescQueueHead_p = PNULL;

	// update new descriptor next, to point to the current head
	rmFreePendingDesc_p->nextReqId = rmFreePendingDescQueue_p->reqIdHead;
	// update old head previous pointer to point to new current head
	if (rmFreePendingDescQueue_p->reqIdHead != RM_MAX_ID)
	{
		rmFreePendingDescQueueHead_p = RmGetFreePendingDescElementPoolByType(rmFreePendingDescQueue_p->reqIdHead, freePendingDescPool);
		rmFreePendingDescQueueHead_p->previousReqId = rmFreePendingDesc_p->id;
	}
	else
	{
		/*Queue was empty - so entry will also be tail*/
		rmFreePendingDescQueue_p->reqIdTail = rmFreePendingDesc_p->id;
	}
	// update queue to the new head 
	rmFreePendingDescQueue_p->reqIdHead = rmFreePendingDesc_p->id;
	// update new descriptor previous to point to null
	rmFreePendingDesc_p->previousReqId = RM_MAX_ID;
	// initialize new descriptor data
	rmFreePendingDesc_p->returnMsgType = OSAL_TASK_UNUSED_MSG;
	rmFreePendingDesc_p->returnTaskId = TASK_INVALID;
}

/**********************************************************************************
RmPushToPendingList 


Description:
------------
push element to Pending list - to tail

Input:
-----
RmFreePendingDesc_t *rmFreePendingDesc_p, FreePendingDescPool_e freePendingDescPool


Output:
-------

Returns:
--------
none
**********************************************************************************/
void RmPushToPendingList(RmFreePendingDesc_t *rmFreePendingDesc_p, FreePendingDescPool_e freePendingDescPool)
{
	RmFreePendingDescQueue_t *pendingQueue_p = RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_PENDING);
	// save current pending tail for later use
	RmFreePendingDesc_t *pendingQueueTail_p = PNULL;

	if (pendingQueue_p->reqIdTail != RM_MAX_ID)
	{
		pendingQueueTail_p = RmGetFreePendingDescElementPoolByType(pendingQueue_p->reqIdTail, freePendingDescPool);
		pendingQueueTail_p->nextReqId = rmFreePendingDesc_p->id;
	}
	else
	{
		/*Queue is empty - so entry will also be head*/
		pendingQueue_p->reqIdHead = rmFreePendingDesc_p->id;
	}
	// update new descriptor prev, to point to the current tail
	rmFreePendingDesc_p->previousReqId = pendingQueue_p->reqIdTail;
	// update queue to the new tail
	pendingQueue_p->reqIdTail = rmFreePendingDesc_p->id;
	// update new descriptor next to point to null
	rmFreePendingDesc_p->nextReqId = RM_MAX_ID;
}

/**********************************************************************************

ResourceManager_GetDescriptorRequest 


Description:
------------
allocate PD in case of success or request in case of failure

Input:
-----
PoolType - from which pool to allcate
pdRequestFillParameters - pointer to the structure to fill

		
Output:
-------
	status - allocation succeeded or not

Returns:
--------
	pdRequestFillParameters will be filled with PD address in case of success
	and request ID in case of failure
	
**********************************************************************************/
bool ResourceManager_GetDescriptorRequest(DescPoolTypes_e poolType, RmPdRequestFillParameters_t *pdRequestFillParameters)
{
	bool status = TRUE;
	FreePendingDescPool_e freePendingDescPool = FREE_PENDING_DESC_POOL_MAX;

	pdRequestFillParameters->packetDescriptor =  ResourceManager_GetDescriptor(poolType);
	freePendingDescPool = RmGetPoolTypeIfSupportPendingList(poolType);
	if(((TxPd_t *)NULL_PD == pdRequestFillParameters->packetDescriptor) && (freePendingDescPool < FREE_PENDING_DESC_POOL_MAX))
	{
		RmFreePendingDesc_t *rmFreePendingDesc_p = RmPopFromFreePendingList(freePendingDescPool, RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_FREE));

		if (rmFreePendingDesc_p != PNULL)
		{
			rmFreePendingDesc_p->returnMsgType = pdRequestFillParameters->returnMsgType;
			rmFreePendingDesc_p->returnTaskId = pdRequestFillParameters->returnTaskId;
			rmFreePendingDesc_p->context = pdRequestFillParameters->context;
			rmFreePendingDesc_p->msgSent = FALSE;
			pdRequestFillParameters->requestId = rmFreePendingDesc_p->id;
			RmPushToPendingList(rmFreePendingDesc_p, freePendingDescPool);

		}
		else
		{
			ILOG0_DDDD("Resource Manager - no ID available for pool type %d, return msg %d, return task %d, context %d", 
					poolType, pdRequestFillParameters->returnMsgType, pdRequestFillParameters->returnTaskId, pdRequestFillParameters->context);
			pdRequestFillParameters->requestId = INVALID_FREE_PENDING_LIST_ID;
			/*Should happen only for TS manager pool*/
			ASSERT(poolType == DESC_POOL_TS_MANAGER);
		}
		status = FALSE;
	}
	return status;
}

/**********************************************************************************

ResourceManager_RemoveRequest 


Description:
------------
remove request from the list of waiting requests

Input:
-----
requestId - the Id of the request to remove


		
Output:
-------
	status - remove succeeded or not

Returns:
--------
bool, True if succeed to remove from the list
	
**********************************************************************************/
bool ResourceManager_RemoveRequest(uint16 requestId, DescPoolTypes_e poolType)
{
	bool status = TRUE;
	FreePendingDescPool_e freePendingDescPool = RmGetPoolTypeIfSupportPendingList(poolType);
	RmFreePendingDesc_t *rmFreePendingDescPoolCurrent_p = RmGetFreePendingDescElementPoolByType(requestId, freePendingDescPool);

	DEBUG_ASSERT(freePendingDescPool < FREE_PENDING_DESC_POOL_MAX);
	DEBUG_ASSERT(requestId < RmGetMaxFreePendingDescPoolSize(freePendingDescPool));
	if (rmFreePendingDescPoolCurrent_p->msgSent == TRUE)
	{
		// message has already sent to requester task
		status = FALSE;
	}
	else
	{
		RmRemoveFromFreePendingList(rmFreePendingDescPoolCurrent_p, RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_PENDING), freePendingDescPool);
		RmPushToFreeList(rmFreePendingDescPoolCurrent_p, freePendingDescPool);
	}
	return status;
}

/**********************************************************************************

ResourceManager_ReleaseRequest 


Description:
------------
release request back to the list of free request

Input:
-----
uint32 requestId, FreePendingDescPool_e freePendingDescPool
		
Output:
-------

Returns:
--------
None
**********************************************************************************/
void ResourceManager_ReleaseRequest(uint16 requestId, DescPoolTypes_e poolType)
{
	FreePendingDescPool_e freePendingDescPool = RmGetPoolTypeIfSupportPendingList(poolType);
	RmFreePendingDesc_t *descToRelease_p = RmGetFreePendingDescElementPoolByType(requestId, freePendingDescPool);

	DEBUG_ASSERT(freePendingDescPool < FREE_PENDING_DESC_POOL_MAX);
	RmPushToFreeList(descToRelease_p, freePendingDescPool);
}

/********************************************************************************
ResourceManager_GetDescriptor

Description:
------------
Get descriptor from pool
Input:
-----
pool type

Output:
-------
Returns:
--------
<Descriptions of the function return value>
********************************************************************************/
TxPd_t* ResourceManager_GetDescriptor(DescPoolTypes_e poolType)
{
	TxPd_t* freeDesc = NULL;
	uint32 pdSource = 0;
	uint32 dramPointer =0;
	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	freeDesc = PacketDescPool[poolType].pHead;
	if ((TxPd_t *)NULL_PD != freeDesc)
	{
		ASSERT(freeDesc->pdSource != HW_Q_MANAGER_DONE_LIST_HOST_DATA);
		/*Save PD source and DRAM pointer*/
		pdSource = freeDesc->pdSource;
		dramPointer = freeDesc->packetPointer;

		ASSERT(PacketDescPool[poolType].numOfDescInPool > 0);
		PacketDescPool[poolType].numOfDescInPool--;
		/* The following field is located in the same offset for all PD types */
		PacketDescPool[poolType].pHead = (TxPd_t *)GET_NEXT_PD(freeDesc);
		/*Clear descriptor*/
		memset(freeDesc, 0, sizeof(TxPd_t));
		/*Restore PD source and DRAM pointer*/
		freeDesc->pdSource = pdSource;
		freeDesc->packetPointer = dramPointer;
		freeDesc->ethType = FRAME_TYPE_ETHERNET;
		freeDesc->nextPdPointer = NEXT_PD_NULL;
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
	return freeDesc;
}

/********************************************************************************
ResourceManager_ReleaseDescriptor

Description:
------------
return descriptor to pool
Input:
-----
pool type , descriptor

Output:
-------
Returns:
--------
<Descriptions of the function return value>
********************************************************************************/
void ResourceManager_ReleaseDescriptor(TxPd_t *pDesc,DescPoolTypes_e poolType)
{
	RmFreePendingDesc_t *rmFreePendingDesc_p = PNULL;
	FreePendingDescPool_e freePendingDescPool = FREE_PENDING_DESC_POOL_MAX;
	uint32 pdSource = 0;
	uint32 dramPointer =0;
	TX_INTERRUPT_SAVE_AREA;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save); 
	ASSERT(pDesc->pdSource != HW_Q_MANAGER_DONE_LIST_HOST_DATA);
	freePendingDescPool = RmGetPoolTypeIfSupportPendingList(poolType);
	if (freePendingDescPool < FREE_PENDING_DESC_POOL_MAX)
	{
		rmFreePendingDesc_p = RmPopFromFreePendingList(freePendingDescPool, RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_PENDING));
	}
	if (rmFreePendingDesc_p != PNULL)
	{
		K_MSG *kMsg_p = OSAL_GET_MESSAGE(sizeof(RmPdFreeDescResponse_t));
		RmPdFreeDescResponse_t *rmPdFreeDescResponse_p = (RmPdFreeDescResponse_t *)pK_MSG_DATA(kMsg_p);

		/*Before sending the descriptor to a pending task initialize it*/
		pdSource = pDesc->pdSource;
		dramPointer = pDesc->packetPointer;
		memset(pDesc, 0, sizeof(TxPd_t));
		pDesc->pdSource = pdSource;
		pDesc->packetPointer = dramPointer;
		pDesc->ethType = FRAME_TYPE_ETHERNET;
		pDesc->nextPdPointer = NEXT_PD_NULL;
		rmFreePendingDesc_p->msgSent = TRUE;
		rmPdFreeDescResponse_p->packetDescriptor = pDesc;
		rmPdFreeDescResponse_p->context = rmFreePendingDesc_p->context;
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
		OSAL_SEND_MESSAGE(rmFreePendingDesc_p->returnMsgType, rmFreePendingDesc_p->returnTaskId, kMsg_p, VAP_ID_DO_NOT_CARE);
	}
	else
	{
		/* All types of PDs have the following field in the same offset */	
		pDesc->nextPdPointer = SET_NEXT_PD(PacketDescPool[poolType].pHead);
		PacketDescPool[poolType].pHead = pDesc;
        ASSERT(PacketDescPool[poolType].numOfDescInPool < PacketDescPool[poolType].maxDescInPool);		
		PacketDescPool[poolType].numOfDescInPool++;		
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
/********************************************************************************
RmInitFreePendingListPool


Description:
------------
initializes the free pending list pool
Input:
-----
none

Output:
-------
Returns:
--------
none
********************************************************************************/
void RmInitFreePendingListPool()
{
	RmFreePendingDesc_t *rmFreePendingDescPoolCurrent_p = PNULL;
	uint16 prevReqId = RM_MAX_ID;
	FreePendingDescPool_e freePendingDescPool = FREE_PENDING_DESC_POOL_TS_MANAGER;
	uint16 poolSize = 0;
	uint16 reqId = 0;
	
	for (freePendingDescPool = FREE_PENDING_DESC_POOL_TS_MANAGER; freePendingDescPool < FREE_PENDING_DESC_POOL_MAX; freePendingDescPool++)
	{
		poolSize = RmGetMaxFreePendingDescPoolSize(freePendingDescPool);
		prevReqId = RM_MAX_ID;
		for (reqId = 0; reqId < poolSize; reqId++)
		{
			rmFreePendingDescPoolCurrent_p = RmGetFreePendingDescElementPoolByType(reqId, freePendingDescPool);
			rmFreePendingDescPoolCurrent_p->id = reqId;
			rmFreePendingDescPoolCurrent_p->msgSent = FALSE;
			rmFreePendingDescPoolCurrent_p->returnMsgType = OSAL_TASK_UNUSED_MSG;
			rmFreePendingDescPoolCurrent_p->returnTaskId = TASK_INVALID;
			// update next pointer (last pointer in array will be updated again after the loop)
			rmFreePendingDescPoolCurrent_p->nextReqId = reqId + 1;
			// update prev pointer (first pointer in array will be updated with PNULL )
			rmFreePendingDescPoolCurrent_p->previousReqId = prevReqId;
			prevReqId = reqId;
		}
		ASSERT(rmFreePendingDescPoolCurrent_p != NULL)
		// last pointer is updated with PNULL
		rmFreePendingDescPoolCurrent_p->nextReqId = RM_MAX_ID;
	}
}

/********************************************************************************
RmInitFreePendingListQueue


Description:
------------
initialize FreePendingListPool
Input:
-----

Output:
-------
Returns:
--------

********************************************************************************/
void RmInitFreePendingListQueue()
{
	RmFreePendingDescQueue_t *rmFreePendingFreeListDescQueue_p = PNULL;
	RmFreePendingDescQueue_t *rmFreePendingPendingListDescQueue_p = PNULL;
	FreePendingDescPool_e freePendingDescPool = FREE_PENDING_DESC_POOL_TS_MANAGER;

	for (freePendingDescPool = FREE_PENDING_DESC_POOL_TS_MANAGER; freePendingDescPool < FREE_PENDING_DESC_POOL_MAX; freePendingDescPool++)
	{
		rmFreePendingFreeListDescQueue_p = RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_FREE);
		rmFreePendingPendingListDescQueue_p = RmGetFreePendingDescQueueByType(freePendingDescPool, RM_FREE_PENDING_LIST_DESC_TYPE_PENDING);
		rmFreePendingFreeListDescQueue_p->reqIdHead = 0;
		rmFreePendingFreeListDescQueue_p->reqIdTail = RmGetMaxFreePendingDescPoolSize(freePendingDescPool) - 1;
		rmFreePendingPendingListDescQueue_p->reqIdHead = RM_MAX_ID;
		rmFreePendingPendingListDescQueue_p->reqIdTail = RM_MAX_ID;
	}
}

/********************************************************************************
RmInitPool


Description:
------------
initialize specific pool
Input:
-----
pool type, pool start pointer, , size of descriptor, number of element in pool, Tx done release Q

Output:
-------
Returns:
--------
<Descriptions of the function return value>
********************************************************************************/
void RmInitPool(DescPoolTypes_e PoolType ,uint8* pDescriptors, uint16 descriptorSize,uint16 numOfDescInPool, uint8 PdSource)
{
	uint32 j = 0;

	PacketDescPool[PoolType].pHead = (TxPd_t *)pDescriptors;
	PacketDescPool[PoolType].numOfDescInPool = numOfDescInPool;
	PacketDescPool[PoolType].maxDescInPool = numOfDescInPool;
	/*Clear pool*/
	memset32(pDescriptors, 0, CONVERT_BYTES_TO_WORDS(descriptorSize * numOfDescInPool));
	for (j = 0; j < numOfDescInPool; j++)
	{
		/* All types of pd have the following field in the same offset */
		((TxPd_t *)pDescriptors)->nextPdPointer = SET_NEXT_PD(pDescriptors + descriptorSize);
		((TxPd_t *)pDescriptors)->pdSource = PdSource;
		((TxPd_t *)pDescriptors)->ethType = FRAME_TYPE_ETHERNET;
		pDescriptors +=  descriptorSize;
	}
	/* Overwrite the next of the last PD */
	pDescriptors -=  descriptorSize;
	((TxPd_t *)pDescriptors)->nextPdPointer = NEXT_PD_NULL;
}

/********************************************************************************
RmInitPayload


Description:
------------
initialize the payload pointers in the pools
Input:
-----
pool type, pool start pointer, single entry size , payload pool start pointer

Output:
-------
Returns:
--------
<Descriptions of the function return value>
********************************************************************************/
void RmInitPayload(DescPoolTypes_e PoolType , TxPd_t* pDescriptors , uint32 EntrySize , uint8* pPayloadPool)
{
	uint32 j = 0;
	uint16 numOfDescInPool = 0;

	numOfDescInPool = PacketDescPool[PoolType].numOfDescInPool;
	/*Clear payload*/
	memset(pPayloadPool, 0, EntrySize * numOfDescInPool);
	for (j = 0 ; j < numOfDescInPool; j++)
	{
		pDescriptors->packetPointer = (uint32)pPayloadPool;
		pDescriptors ++;
		pPayloadPool = pPayloadPool + EntrySize;		
	}
}

/********************************************************************************
ResourceManager_Init

Description:
------------
initialize the pools 
Input:
-----


Output:
-------
Returns:
--------
<Descriptions of the function return value>
********************************************************************************/
void ResourceManager_Init(void)
{
	// init pools
#ifdef ENET_INC_ARCH_WAVE600
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
#endif //ENET_INC_ARCH_WAVE600
	memset(&PdDescPools, 0, sizeof(PdPools_t));

	RmInitPool(DESC_POOL_TS_MANAGER, (uint8 *)&PdDescPools.TsManDescPool[0], sizeof(TxPd_t), NUM_OF_TS_MANAGER_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW);
	RmInitPool(DESC_POOL_RELIABLE_MCAST, (uint8 *)&PdDescPools.ReliableMcastDescPool[0], sizeof(reliableMcastPacketDescriptor_t), NUM_OF_RELIABLE_MCAST_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST);
	RmInitPool(DESC_POOL_MANAGEMENT_FROM_FW, (uint8 *)&PdDescPools.ManDescPool[0], sizeof(TxPd_t), NUM_OF_MANAGEMENT_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW);
	RmInitPool(DESC_POOL_MANAGEMENT_FROM_HOST,(uint8 *)&PdDescPools.BssDescPool[0], sizeof(TxPd_t), NUM_OF_BSS_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST);
	RmInitPool(DESC_POOL_DATA, (uint8 *)&PdDescPools.DataDescPool[0], sizeof(TxPd_t), NUM_OF_TX_DESC, HW_Q_MANAGER_DONE_LIST_HOST_DATA);
	RmInitPool(DESC_POOL_CSA_MANAGER, (uint8 *)&PdDescPools.CsaManDescPool[0], sizeof(TxPd_t), NUM_OF_CSA_MANAGER_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW);
	RmInitPool(DESC_POOL_PS_SETTINGS_NDP, (uint8 *)&PdDescPools.PsSettingsNdpDescPool[0], sizeof(TxPd_t), NUM_OF_PS_SETTINGS_NDP_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW);	
	RmInitPool(DESC_POOL_MC_DEAUTH, (uint8 *)&PdDescPools.McDeauthDescPool[0], sizeof(TxPd_t), NUM_OF_MC_DEAUTH_DESCRIPTORS, HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW);

	/* Should be after the init pool */
	RmInitPayload(DESC_POOL_TS_MANAGER, &PdDescPools.TsManDescPool[0], TS_PACKET_SIZE_ALLINGED_TO_WORDS, &TsManPackets[0][0]);
	RmInitPayload(DESC_POOL_MANAGEMENT_FROM_FW, &PdDescPools.ManDescPool[0], sizeof(ManPacket_t), (uint8 *)&ManPkts[0]);
	RmInitPayload(DESC_POOL_CSA_MANAGER, &PdDescPools.CsaManDescPool[0], sizeof(CsaFrame_t), (uint8 *)&CsaPkts[0]);

#if defined (DMA_BUG_SKIP_BIT_WORKAROUND)
	// We give fake payload to the NDP PDs (we use the CSA Manager since they have the same number of PDs).
	// DMA will copy the payload (26 bytes - header only), and it will be overwritten by TX PD ACC.
	// The reason is that we can't use "SKIP" in DMA due to HW bug in CDB. This fix is also for single band for simplicity.
	ASSERT(PacketDescPool[DESC_POOL_PS_SETTINGS_NDP].numOfDescInPool == PacketDescPool[DESC_POOL_CSA_MANAGER].numOfDescInPool);
	RmInitPayload(DESC_POOL_PS_SETTINGS_NDP, &PdDescPools.PsSettingsNdpDescPool[0], sizeof(CsaFrame_t), (uint8 *)&CsaPkts[0]);
#endif
	
#ifdef ENET_INC_ARCH_WAVE600 //this code was removed in gen5 due to HW bug - JIRA3026 which is fixed in gen6
	// Push PDs to Ps Setting free  NDP Pool
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_TX_POOL_LIST_PS_SETTINGTS_NDP_PD;
	hwQueueManagerRequestParams.pHeadDesc = &PdDescPools.PsSettingsNdpDescPool[0];
	hwQueueManagerRequestParams.pTailDesc= &PdDescPools.PsSettingsNdpDescPool[NUM_OF_PS_SETTINGS_NDP_DESCRIPTORS-1];
#ifndef PF6_TWT_PS_NDP
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
#endif
#endif //ENET_INC_ARCH_WAVE600

	/*Init Request ID lists*/
	RmInitFreePendingListQueue();
	RmInitFreePendingListPool();
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


