/***********************************************************************************
 File:		RxCAlssifier.c
 Module:		Rx Classifier
 Purpose:		Owner for all access to the Rx Classifier
 Description:	This module is the API to the Rx Classifier which is responbile for frame filtering and classification 
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "RegAccess_Api.h"
#include "Utils_Api.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "RxClassifier_Api.h"
#include "RxClassifier.h"
#include "HwMemoryMap.h"
#include "RxClassifierRegs.h"
#include "RxClassifierRam.h"
#include "HwGlobalDefinitions.h"
#include "RxDescriptor.h"
#include "PacketDescriptor.h"
#include "protocol.h"
#include "frame.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 15


 /* RXQID Filter Definitions */
#define RX_CLASSIFIER_QXQ_ID_FILTER_GROUP_LOGIC_NUMBER			(4) /* Number of filters ande together - 1*/
#define WPA_SEC_PASS_THROUGH_VAP_EN 							(1) 

/*RTD ctrl register definitions*/
#define RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_LENGTH_CALC_NEGATIVE_DROP_EN_MASK		0x00000020
#define RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_MIS_DATA_PROT_FRAME_DB_DROP_EN_MASK    0x00000040
#define RESERVED_VAL (0)


/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
RxClassifierRam_t *RxClassifierRam;/* pointer to the classifier Ram 0xA6C3B800*/

const RamRxClassifierRdTypeConfiguration_t ramClassifierRdTypeConfiguration = 
{
	/* Unicast Data Configuration*/
	{
		RD_TYPE_UNICAST_NON_QOS_DATA,	/* Data */
		RD_TYPE_DROP,					/* Data + CF Ack */ 
		RD_TYPE_DROP,					/* Data + CF Poll */
		RD_TYPE_DROP,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_NDP, 					/* Null */ 
		RD_TYPE_DROP,					/* CF-Ack */
		RESERVED_VAL,
		RD_TYPE_DROP,					/* CF-Poll */
		RD_TYPE_DROP,					/* CF-Ack + CF Poll */
		RD_TYPE_UNICAST_QOS_DATA,		/* Qos Data */
		RD_TYPE_DROP,					/* Qos Data + CF Ack */
		RD_TYPE_DROP,					/* Qos Data + CF Poll */
		RD_TYPE_DROP,					/* Qos Data + CF-Ack+CF Poll */
		RESERVED_VAL,
		RD_TYPE_NDP,						/* Qos Null */
		RD_TYPE_DROP,					/* Reserved*/
		RD_TYPE_DROP,					/* Qos CF Poll*/
		RD_TYPE_DROP,					/* Qos CF-Ack + CF Poll */
		RESERVED_VAL,
	},
	/* Multicast Data Configuration */
	{
		RD_TYPE_MULTICAST_DATA,			/* Data */
		RD_TYPE_NOT_SUPPORTED,			/* Data + CF Ack */ 
		RD_TYPE_NOT_SUPPORTED, 			/* Data + CF Poll */
		RD_TYPE_NOT_SUPPORTED,			/* Data + CF-Ack+CF Poll */
		RD_TYPE_NDP,						/* Null */ 
		RD_TYPE_NOT_SUPPORTED, 			/* CF-Ack */
		RESERVED_VAL,
		RD_TYPE_NOT_SUPPORTED,			/* CF-Poll */
		RD_TYPE_NOT_SUPPORTED,			/* CF-Ack + CF Poll */
		RD_TYPE_MULTICAST_DATA,			/* Qos Data */
		RD_TYPE_NOT_SUPPORTED,			/* Qos Data + CF Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Qos Data + CF Poll */
		RD_TYPE_NOT_SUPPORTED,			/* Qos Data + CF-Ack+CF Poll */
		RESERVED_VAL,
		RD_TYPE_NDP,						/* Qos Null */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved*/
		RD_TYPE_NOT_SUPPORTED,			/* Qos CF Poll*/
		RD_TYPE_NOT_SUPPORTED, 			/* Qos CF-Ack + CF Poll */
		RESERVED_VAL,
	},
	/* Unicast Management Configuration*/
	{
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Association Req  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Association Rsp*/
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Reassociation Req */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Reassociation Rsp  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Probe Req  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,				/* Reserved  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,		 		/*ATIM */
		RD_TYPE_UNICAST_MGMT_TYPE_2, 	/* Disassociation*/
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Authentication */
		RESERVED_VAL,
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Deauthentication */
		RD_TYPE_UNICAST_MGMT_TYPE_1,	/* Action */
		RD_TYPE_DROP,					/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,				/* Reserved */
		RESERVED_VAL,
	},
	/* Multicast Management Configuration*/
	{
		RD_TYPE_MULTICAST_MGMT, 	/* Association Req	*/
		RD_TYPE_MULTICAST_MGMT, 	/* Association Rsp*/
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Req */
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Rsp  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Req  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_MULTICAST_MGMT, 	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_MULTICAST_MGMT, 	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_MULTICAST_MGMT, 	/* Disassociation*/
		RD_TYPE_MULTICAST_MGMT, 	/* Authentication */
		RESERVED_VAL,
		RD_TYPE_MULTICAST_MGMT, 	/* Deauthentication */
		RD_TYPE_MULTICAST_MGMT, 	/* Action */
		RD_TYPE_DROP, 				/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
		RESERVED_VAL,
	},	
	/* Control Frames  Configuration*/
	{
		RD_TYPE_DROP, 		/* Reserved	*/
		RD_TYPE_DROP, 		/* Reserved	*/
		RD_TYPE_DROP, 		/* Reserved*/
		RD_TYPE_DROP, 		/* Reserved */
		RD_TYPE_DROP, 		/* Beamforming Report Poll */
		RD_TYPE_DROP, 		/* VHT NDP Announcement  */
		RESERVED_VAL,
		RD_TYPE_DROP, 		/* Control Frame Extension  */
		RD_TYPE_DROP, 		/* Control Wrapper  */
		RD_TYPE_CONTROL,	/* BAR  */
		RD_TYPE_CONTROL,   	/* BA  */
		RD_TYPE_DROP,		/* Ps-Poll */
		RD_TYPE_DROP, 		/* RTS*/
		RESERVED_VAL,
		RD_TYPE_DROP, 		/* CTS */
		RD_TYPE_CONTROL, 	/* Ack */
		RD_TYPE_DROP, 		/* CF-End */
		RD_TYPE_DROP,		/*CF-End+ CF_Ack */
		RESERVED_VAL,
	},		
	/* Non Associated Unicast  Management Configuration*/
	{
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Association Req	*/
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Association Rsp*/
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Reassociation Req */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Reassociation Rsp  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Probe Req  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,					/* Reserved  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,					/*ATIM */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Disassociation*/
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Authentication */
		RESERVED_VAL,
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Deauthentication */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,		/* Action */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,		/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,					/* Reserved */
		RESERVED_VAL,
	},
	/* Non Associated Multicast  Management Configuration*/
	{	
		RD_TYPE_MULTICAST_MGMT, 	/* Association Req	*/
		RD_TYPE_MULTICAST_MGMT, 	/* Association Rsp*/
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Req */
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Rsp  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Req  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_MULTICAST_MGMT, 	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_MULTICAST_MGMT, 	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_MULTICAST_MGMT, 	/* Disassociation*/
		RD_TYPE_MULTICAST_MGMT, 	/* Authentication */
		RESERVED_VAL,
		RD_TYPE_MULTICAST_MGMT, 	/* Deauthentication */
		RD_TYPE_MULTICAST_MGMT, 	/* Action */
		RD_TYPE_MULTICAST_MGMT, 	/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
		RESERVED_VAL,
	},
	/* Unicast Management Configuration protected*/
	{
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Association Req	*/
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Association Rsp*/
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Reassociation Req */
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Reassociation Rsp  */
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Probe Req  */
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,						/* Reserved  */
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,						/*ATIM */
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Disassociation*/
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Authentication */
		RESERVED_VAL,
		RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED,	/* Deauthentication */
		RD_TYPE_UNICAST_MGMT_TYPE_1_PROTECTED,	/* Action */
		RD_TYPE_UNICAST_MGMT_TYPE_1_PROTECTED,	/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,						/* Reserved */
		RESERVED_VAL,
	},
	/* Multicast Management Configuration protected*/
	{
		RD_TYPE_BIP,						/* Association Req	*/
		RD_TYPE_BIP,						/* Association Rsp*/
		RD_TYPE_BIP,						/* Reassociation Req */
		RD_TYPE_BIP,						/* Reassociation Rsp  */
		RD_TYPE_BIP,						/* Probe Req  */
		RD_TYPE_BIP,						/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_BIP,						/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_BIP,						/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_BIP					,	/* Disassociation*/
		RD_TYPE_BIP,						/* Authentication */
		RESERVED_VAL,
		RD_TYPE_BIP,						/* Deauthentication */
		RD_TYPE_BIP,						/* Action */
		RD_TYPE_BIP,						/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
		RESERVED_VAL,
	},		
	/* Non Associated Unicast  Management Configuration protected*/
	{
		RD_TYPE_NOT_SUPPORTED,			/* Association Req	*/
		RD_TYPE_NOT_SUPPORTED,			/* Association Rsp*/
		RD_TYPE_NOT_SUPPORTED,			/* Reassociation Req */
		RD_TYPE_NOT_SUPPORTED,			/* Reassociation Rsp  */
		RD_TYPE_NOT_SUPPORTED,			/* Probe Req  */
		RD_TYPE_NOT_SUPPORTED,			/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_NOT_SUPPORTED,			/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_NOT_SUPPORTED,			/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_NOT_SUPPORTED,			/* Disassociation*/
		RD_TYPE_NOT_SUPPORTED,			/* Authentication */
		RESERVED_VAL,
		RD_TYPE_NOT_SUPPORTED,			/* Deauthentication */
		RD_TYPE_NOT_SUPPORTED,			/* Action */
		RD_TYPE_NOT_SUPPORTED,			/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
		RESERVED_VAL,
	},
	/* Non Associated Multicast  Management Configuration protected*/
	{	
		RD_TYPE_BIP,						/* Association Req	*/
		RD_TYPE_BIP,						/* Association Rsp*/
		RD_TYPE_BIP,						/* Reassociation Req */
		RD_TYPE_BIP,						/* Reassociation Rsp  */
		RD_TYPE_BIP,						/* Probe Req  */
		RD_TYPE_BIP,						/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_BIP,						/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_BIP,						/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_BIP,						/* Disassociation*/
		RD_TYPE_BIP,						/* Authentication */
		RESERVED_VAL,
		RD_TYPE_BIP,						/* Deauthentication */
		RD_TYPE_BIP,						/* Action */
		RD_TYPE_BIP,						/* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
		RESERVED_VAL,
	},
	/* Unicast Data Configuration protected*/
	{
		RD_TYPE_UNICAST_NON_QOS_DATA_PROTECTED,	/* Data */
		RD_TYPE_DROP,					/* Data + CF Ack */ 
		RD_TYPE_DROP,					/* Data + CF Poll */
		RD_TYPE_DROP,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_NDP, 					/* Null */ 
		RD_TYPE_DROP,					/* CF-Ack */
		RESERVED_VAL,
		RD_TYPE_DROP,					/* CF-Poll */
		RD_TYPE_DROP,					/* CF-Ack + CF Poll */
		RD_TYPE_UNICAST_QOS_DATA_PROTECTED,		/* Qos Data */
		RD_TYPE_DROP,					/* Qos Data + CF Ack */
		RD_TYPE_DROP,					/* Qos Data + CF Poll */
		RD_TYPE_DROP,					/* Qos Data + CF-Ack+CF Poll */
		RESERVED_VAL,
		RD_TYPE_NDP, 					/* Qos Null */
		RD_TYPE_DROP,					/* Reserved*/
		RD_TYPE_DROP,					/* Qos CF Poll*/
		RD_TYPE_DROP,					/* Qos CF-Ack + CF Poll */
		RESERVED_VAL,
	},
};	



const RamRxClassifierRdTypeConfiguration_t ramClassifierRdTypeForSnifferConfiguration = 
{
	/* Unicast Data Configuration*/
	{
		RD_TYPE_SNIFFER,					/* Data */
		RD_TYPE_SNIFFER,					/* Data + CF Ack */ 
		RD_TYPE_SNIFFER,					/* Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER, 				/* Null */ 
		RD_TYPE_SNIFFER,					/* CF-Ack */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* CF-Poll */
		RD_TYPE_SNIFFER,					/* CF-Ack + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Ack */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data + CF-Ack+CF Poll */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,						/* Qos Null */
		RD_TYPE_SNIFFER,					/* Reserved*/
		RD_TYPE_SNIFFER,					/* Qos CF Poll*/
		RD_TYPE_SNIFFER,					/* Qos CF-Ack + CF Poll */
		RESERVED_VAL,
	},
	/* Multicast Data Configuration */
	{
		RD_TYPE_SNIFFER,					/* Data */
		RD_TYPE_SNIFFER,					/* Data + CF Ack */ 
		RD_TYPE_SNIFFER, 				/* Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER,					/* Null */ 
		RD_TYPE_SNIFFER, 				/* CF-Ack */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* CF-Poll */
		RD_TYPE_SNIFFER,					/* CF-Ack + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Ack */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data + CF-Ack+CF Poll */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Qos Null */
		RD_TYPE_SNIFFER,					/* Reserved*/
		RD_TYPE_SNIFFER,					/* Qos CF Poll*/
		RD_TYPE_SNIFFER, 				/* Qos CF-Ack + CF Poll */
		RESERVED_VAL,
	},
	/* Unicast Management Configuration*/
	{
		RD_TYPE_SNIFFER,					/* Association Req  */
		RD_TYPE_SNIFFER,					/* Association Rsp*/
		RD_TYPE_SNIFFER,					/* Reassociation Req */
		RD_TYPE_SNIFFER,					/* Reassociation Rsp  */
		RD_TYPE_SNIFFER,					/* Probe Req  */
		RD_TYPE_SNIFFER,					/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER,					/* Beacon  */
		RD_TYPE_SNIFFER,		 			/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER,					/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Deauthentication */
		RD_TYPE_SNIFFER,					/* Action */
		RD_TYPE_SNIFFER,					/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},
	/* Multicast Management Configuration*/
	{
		RD_TYPE_SNIFFER, 				/* Association Req	*/
		RD_TYPE_SNIFFER, 				/* Association Rsp*/
		RD_TYPE_SNIFFER, 				/* Reassociation Req */
		RD_TYPE_SNIFFER, 				/* Reassociation Rsp  */
		RD_TYPE_SNIFFER, 				/* Probe Req  */
		RD_TYPE_SNIFFER, 				/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER, 				/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER, 				/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Deauthentication */
		RD_TYPE_SNIFFER, 				/* Action */
		RD_TYPE_SNIFFER, 				/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},	
	/* Control Frames  Configuration*/
	{
		RD_TYPE_SNIFFER, 				/* Reserved	*/
		RD_TYPE_SNIFFER, 				/* Reserved	*/
		RD_TYPE_SNIFFER, 				/* Reserved*/
		RD_TYPE_SNIFFER, 				/* Reserved */
		RD_TYPE_SNIFFER, 				/* Beamforming Report Poll */
		RD_TYPE_SNIFFER, 				/* VHT NDP Announcement  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Control Frame Extension  */
		RD_TYPE_SNIFFER, 				/* Control Wrapper  */
		RD_TYPE_SNIFFER,					/* BAR  */
		RD_TYPE_SNIFFER, 				/* BA  */
		RD_TYPE_SNIFFER,					/* Ps-Poll */
		RD_TYPE_SNIFFER, 				/* RTS*/
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* CTS */
		RD_TYPE_SNIFFER, 				/* Ack */
		RD_TYPE_SNIFFER, 				/* CF-End */
		RD_TYPE_SNIFFER,					/*CF-End+ CF_Ack */
		RESERVED_VAL,
	},		
	/* Non Associated Unicast  Management Configuration*/
	{
		RD_TYPE_SNIFFER,					/* Association Req	*/
		RD_TYPE_SNIFFER,					/* Association Rsp*/
		RD_TYPE_SNIFFER,					/* Reassociation Req */
		RD_TYPE_SNIFFER,					/* Reassociation Rsp  */
		RD_TYPE_SNIFFER,					/* Probe Req  */
		RD_TYPE_SNIFFER,					/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER,					/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER,					/* Disassociation*/
		RD_TYPE_SNIFFER,					/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Deauthentication */
		RD_TYPE_SNIFFER,					/* Action */
		RD_TYPE_SNIFFER,					/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},
	/* Non Associated Multicast  Management Configuration*/
	{	
		RD_TYPE_SNIFFER, 				/* Association Req	*/
		RD_TYPE_SNIFFER, 				/* Association Rsp*/
		RD_TYPE_SNIFFER, 				/* Reassociation Req */
		RD_TYPE_SNIFFER, 				/* Reassociation Rsp  */
		RD_TYPE_SNIFFER, 				/* Probe Req  */
		RD_TYPE_SNIFFER, 				/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER, 				/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER, 				/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Deauthentication */
		RD_TYPE_SNIFFER, 				/* Action */
		RD_TYPE_SNIFFER, 				/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},
	/* Unicast Management Configuration protected*/ 
	{
		RD_TYPE_SNIFFER,					/* Association Req  */
		RD_TYPE_SNIFFER,					/* Association Rsp*/
		RD_TYPE_SNIFFER,					/* Reassociation Req */
		RD_TYPE_SNIFFER,					/* Reassociation Rsp  */
		RD_TYPE_SNIFFER,					/* Probe Req  */
		RD_TYPE_SNIFFER,					/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER,					/* Beacon  */
		RD_TYPE_SNIFFER,		 			/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER,					/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Deauthentication */
		RD_TYPE_SNIFFER,					/* Action */
		RD_TYPE_SNIFFER,					/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},
	/* Multicast Management Configuration protected*/
	{
		RD_TYPE_SNIFFER, 				/* Association Req	*/
		RD_TYPE_SNIFFER, 				/* Association Rsp*/
		RD_TYPE_SNIFFER, 				/* Reassociation Req */
		RD_TYPE_SNIFFER, 				/* Reassociation Rsp  */
		RD_TYPE_SNIFFER, 				/* Probe Req  */
		RD_TYPE_SNIFFER, 				/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER, 				/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER, 				/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Deauthentication */
		RD_TYPE_SNIFFER, 				/* Action */
		RD_TYPE_SNIFFER, 				/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},	
	/* Non Associated Unicast  Management Configuration protected*/
	{
		RD_TYPE_SNIFFER,					/* Association Req	*/
		RD_TYPE_SNIFFER,					/* Association Rsp*/
		RD_TYPE_SNIFFER,					/* Reassociation Req */
		RD_TYPE_SNIFFER,					/* Reassociation Rsp  */
		RD_TYPE_SNIFFER,					/* Probe Req  */
		RD_TYPE_SNIFFER,					/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER,					/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER,					/* Disassociation*/
		RD_TYPE_SNIFFER,					/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Deauthentication */
		RD_TYPE_SNIFFER,					/* Action */
		RD_TYPE_SNIFFER,					/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},
	/* Non Associated Multicast  Management Configuration protected*/
	{	
		RD_TYPE_SNIFFER, 				/* Association Req	*/
		RD_TYPE_SNIFFER, 				/* Association Rsp*/
		RD_TYPE_SNIFFER, 				/* Reassociation Req */
		RD_TYPE_SNIFFER, 				/* Reassociation Rsp  */
		RD_TYPE_SNIFFER, 				/* Probe Req  */
		RD_TYPE_SNIFFER, 				/* Probe Rsp  */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER, 				/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER, 				/* Authentication */
		RESERVED_VAL,
		RD_TYPE_SNIFFER, 				/* Deauthentication */
		RD_TYPE_SNIFFER, 				/* Action */
		RD_TYPE_SNIFFER, 				/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
		RESERVED_VAL,
	},
	/* Unicast Data Configuration protected*/
	{
		RD_TYPE_SNIFFER,					/* Data */
		RD_TYPE_SNIFFER,					/* Data + CF Ack */ 
		RD_TYPE_SNIFFER, 				/* Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER,					/* Null */ 
		RD_TYPE_SNIFFER, 				/* CF-Ack */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* CF-Poll */
		RD_TYPE_SNIFFER,					/* CF-Ack + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Ack */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data + CF-Ack+CF Poll */
		RESERVED_VAL,
		RD_TYPE_SNIFFER,					/* Qos Null */
		RD_TYPE_SNIFFER,					/* Reserved*/
		RD_TYPE_SNIFFER,					/* Qos CF Poll*/
		RD_TYPE_SNIFFER, 				/* Qos CF-Ack + CF Poll */
		RESERVED_VAL,
	},
};	

/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/




/**********************************************************************************
RxClassifier_ConfigureRegularMode 

Description:
------------
	Configuration of Rx Classifier (Gen5) for regular mode (operational mode = NORMAL, DUT)
		1) Configure RD Type Table
		2) Filters Configuration
			1.1) Frame Class Violation Filter -Use default values for frame class configuration (according to type/subtype)
			1.2) RXQID Filter - And between filter 0-2 which sign as drop multicast frames except Beacon,Probe,Multicat Action (Public Action)	
Input:
-----
	None 
Output:
-------
	None
Returns:
--------
	None
*********************************************************************************/

void RxClassifier_ConfigureRegularMode(void)

{
	RegRxClassifierRxqIdFilter0_u rxqIdFilterConfiguration;
	RegRxClassifierRxqIdFilterControl_u rxqIdFilterControl;
	RegRxClassifierRxClassifierEnable_u rxClassifierEnable;

	rxqIdFilterConfiguration.val = 0;
	rxqIdFilterControl.val = 0;
	rxClassifierEnable.val = 0;

	/* RD Type Configuration*/
	memcpy32(&RxClassifierRam->rdTypeConfiguration,&ramClassifierRdTypeConfiguration, sizeof(RamRxClassifierRdTypeConfiguration_t) >> 2);
	

	/*RXQID Filter Configuration */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0DropEn = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameType = FM_CONTROL_FTYPE_MANAGEMENT;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0SubtypeEqual = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr2IdValNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1Unicast = FALSE;
	/*Accept Beacon Frames */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_BEACON;	
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER0,rxqIdFilterConfiguration.val);
	/*Accept Probe Req Frames */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_PROBE_REQ;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER1,rxqIdFilterConfiguration.val);
	/*Accept Multicast actions */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER2,rxqIdFilterConfiguration.val);
	/*Accept Multicast Deauthenticate */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_DEAUTHENTICATION;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER3,rxqIdFilterConfiguration.val);
	/*Accept Multicast Disassociate */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_DISASSOCIATION;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER4,rxqIdFilterConfiguration.val);
	/* Gather subfilters 0-4 as AND group */		
	rxqIdFilterControl.bitFields.rxqIdFilterGroupAndLogic = RX_CLASSIFIER_QXQ_ID_FILTER_GROUP_LOGIC_NUMBER;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER_CONTROL,rxqIdFilterControl.val);

	/* Reject Multicast Data frames*/	
	rxqIdFilterConfiguration.val = 0;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0DropEn = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameType = FM_CONTROL_FTYPE_DATA;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0SubtypeNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr2IdVal = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr2IdValNa = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1Unicast = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1UnicastNa = FALSE;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER5,rxqIdFilterConfiguration.val);

	/* Reject All frames with "frame type" = 3 (they might not be filtered by RXC in some cases) */	
	rxqIdFilterConfiguration.val = 0;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0DropEn = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameType = FM_CONTROL_FTYPE_EXTENSION;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0TypeNa = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0SubtypeNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr2IdValNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1UnicastNa = TRUE;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER6,rxqIdFilterConfiguration.val);

	//enable the rx classifier
	rxClassifierEnable.bitFields.rxClasTopEn = 1;			// Rx Classifier enable
	rxClassifierEnable.bitFields.rxClasSecurityEn = 1;		// Enable Security engine initiation
	rxClassifierEnable.bitFields.rxClasRtdGenerationEn = 1;	// Enable RD template generation
	RegAccess_Write(REG_RX_CLASSIFIER_RX_CLASSIFIER_ENABLE,rxClassifierEnable.val);
}


/**********************************************************************************
RxClassifier_HandleRxManagerAddVap 

Description:
------------
	Configuration of Rx Classifier according to operational mode
Input:
-----
	UMI_ADD_VAP* addVapStructurePtr
Output:
-------
	None
Returns:
--------
	None
*********************************************************************************/

void RxClassifier_HandleRxManagerAddVap(UMI_ADD_VAP* addVapStructurePtr)
{
	if (addVapStructurePtr->operationMode != OPERATION_MODE_SNIFFER)
	{
		RxClassifier_ConfigureRegularMode();
	}
	else
	{
		//Disable rx classifier filters
		RxClassifier_ConfigureSnifferMode();
	}
}
/**********************************************************************************
RxClassifier_HandleAddWpaPtVap 

Description:
------------
	Configuration of Rx Classifier according to operational mode
Input:
-----
	vap ID 
Output:
-------
	None
Returns:
--------
	None
*********************************************************************************/

void RxClassifier_HandleAddWpaPtVap(uint8 vapId)
{
	uint32 mask = WPA_SEC_PASS_THROUGH_VAP_EN << vapId;
	uint32 data = WPA_SEC_PASS_THROUGH_VAP_EN << vapId;
	RegAccess_WriteMasked(REG_RX_CLASSIFIER_WPA_SEC_PASS_THROUGH_VAP_EN, mask, data);
}
/**********************************************************************************

RxClassifier_ConfigureSnifferMode 

Description:
------------
	Configuration of Rx classifier for operation mode SNIFFER:
	
	- Configure RD Type = Sniffer for all frame types (this is done by writing the configuration table to specific place in Classifier's RAM).
	- Filters off:
		o	Frame class filter
		o	Length Calculation negative drop.
		o	RXQ_ID filter drop - configure for all 16 filters.
		o	RXQ_ID filter group and logic.

Input:
-----
	None 
	
Output:
-------
	None
Returns:
--------
	void
*********************************************************************************/
void RxClassifier_ConfigureSnifferMode(void)
{

	RegRxClassifierFrameClassFilterControl_u frameClassFilterControl;
	RegRxClassifierRxClasRtdCtrl_u	rxClasRtdCtrl;
	RegRxClassifierRxqIdFilter0_u rxqIdFilterConfiguration;
	RegRxClassifierWpaSecPassThroughVapEn_u wpaSecPassThroughVapEn;
	RegRxClassifierRdTypeDropValue_u	rdTypeDropValue;
	RegRxClassifierRxClassifierEnable_u rxClassifierEnable;
	uint32 regAddr;

	/* RD Type Configuration*/
	memcpy32(&RxClassifierRam->rdTypeConfiguration,&ramClassifierRdTypeForSnifferConfiguration, sizeof(RamRxClassifierRdTypeConfiguration_t) >> 2);

	//disable frame class filter - by default all filters were on
	frameClassFilterControl.val = 0x0;
	RegAccess_Write(REG_RX_CLASSIFIER_FRAME_CLASS_FILTER_CONTROL, frameClassFilterControl.val);	
	
	//disable negative length drop and mis data protection drop (were enabled by default)
	rxClasRtdCtrl.bitFields.lengthCalcNegativeDropEn = 0x0;
	rxClasRtdCtrl.bitFields.misDataProtFrameDbDropEn = 0x0;
	RegAccess_WriteMasked(REG_RX_CLASSIFIER_RX_CLAS_RTD_CTRL, (RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_LENGTH_CALC_NEGATIVE_DROP_EN_MASK | RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_MIS_DATA_PROT_FRAME_DB_DROP_EN_MASK), rxClasRtdCtrl.val);
	
	//remove rxqid drop for all 16 filters - when 'rxq_id_filter0_drop_en' field is FALSE, all other fields not relevant, thus we will zero the entire register
	rxqIdFilterConfiguration.val = 0x0;
	//write rxqIdFilterConfiguration for all 16 rxqid filters
	for(regAddr = REG_RX_CLASSIFIER_RXQ_ID_FILTER0; regAddr <= REG_RX_CLASSIFIER_RXQ_ID_FILTER15; regAddr += 4)
	{
		RegAccess_Write(regAddr, rxqIdFilterConfiguration.val);	
	}

	//reset the WPA reg
	wpaSecPassThroughVapEn.val = 0;
	RegAccess_Write(REG_RX_CLASSIFIER_WPA_SEC_PASS_THROUGH_VAP_EN, wpaSecPassThroughVapEn.val);	

	//set the drop index to be the sniffer index
	rdTypeDropValue.val = 0;
	rdTypeDropValue.bitFields.rdTypeDropValue = RD_TYPE_SNIFFER;	
	RegAccess_Write(REG_RX_CLASSIFIER_RD_TYPE_DROP_VALUE, rdTypeDropValue.val);

	//enable the rx classifier
	rxClassifierEnable.val = 0;
	rxClassifierEnable.bitFields.rxClasTopEn = 1;			// Rx Classifier enable
	rxClassifierEnable.bitFields.rxClasSecurityEn = 1;		// Enable Security engine initiation
	rxClassifierEnable.bitFields.rxClasRtdGenerationEn = 1;	// Enable RD template generation
	RegAccess_Write(REG_RX_CLASSIFIER_RX_CLASSIFIER_ENABLE,rxClassifierEnable.val);

	ILOG0_V("-------- Rx Classifier configured for Sniffer --------");
}


/**********************************************************************************

RxClassifier_Init 

Description:
------------
	Initialization of Rx Classifier parametes
		1) Set Internal Memory to 0
	
Input:
-----
	None
Output:
-------
	None
**********************************************************************************/
void RxClassifier_Init(void)
{
	RegRxClassifierFrameMgmtSubtypesClass_u frameMgmtSubtypesClassReg;

	
	RxClassifierRam = (RxClassifierRam_t *)(B0_RX_CLASSIFIER_BASE_ADDR + RX_CLASSIFIER_RAM_ADDR_OFFSET);
	/* Initial Internal Ram */
	memset32(RxClassifierRam,0,sizeof(RxClassifierRam_t) >> 2);

	//	Change Action frame to be class 1 so that it will not be thrown for unconnected STA.
	RegAccess_Read(REG_RX_CLASSIFIER_FRAME_MGMT_SUBTYPES_CLASS,&frameMgmtSubtypesClassReg.val);
	frameMgmtSubtypesClassReg.bitFields.frameMgmtSubtypesClass &= ~(FRAME_CLASS_3 << (MANAGEMENT_SUBTYPE_ACTION*2)); // clear configuration for action frame (each type is 2 bits)
	frameMgmtSubtypesClassReg.bitFields.frameMgmtSubtypesClass |=  (FRAME_CLASS_1 << (MANAGEMENT_SUBTYPE_ACTION*2)); // Set action frame to be class 1 (each type is 2 bits)
	RegAccess_Write(REG_RX_CLASSIFIER_FRAME_MGMT_SUBTYPES_CLASS,frameMgmtSubtypesClassReg.val);
	
}

