/***********************************************************************************
 File:		RxCAlssifier.c
 Module:		Rx Classifier
 Purpose:		Owner for all access to the Rx Classifier
 Description:	This module is the API to the Rx Classifier which is responbile for frame filtering and classification 
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "Utils_Api.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "RxClassifier_Api.h"
#include "RxClassifier.h"
#include "StaDatabase_Api.h"
#include "HwMemoryMap.h"
#include "RxClassifierRegs.h"
#include "RxClassifierRam.h"
#include "HwGlobalDefinitions.h"
#include "RxDescriptor.h"
#include "PacketDescriptor.h"
#include "frame.h"
#include "loggerAPI.h"
#include "Dut_Api.h"
#include "MacHtExtensionsRegs.h"


/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 14


#define RX_CLASSIFIER_INITIAL_MEMORY_VALUE		 (0)
#define RX_CLASSIFIER_STA_FRAME_CLASS_INIT_VALUE (0xAAAAAAAA)/* Frame Class Filter  Configuration - Set all Sta Frame Class to 1*/

 /* RXQID Filter Definitions */
#define RX_CLASSIFIER_QXQ_ID_FILTER_GROUP_LOGIC_NUMBER			(4) /* Number of filters ande together - 1*/
/*Frame Class Filter  Registers Definitions*/
#define RX_CLASSIFIER_FRAME_CLASS_ENTRY_WIDTH 					(2)	   /* Every 2 bit represent the Sta frame class value*/
#define RX_CLASSIFIER_FRAME_CLASS_ENTRY_MASK  					(0x3)  /* Mask of the sta frame class value */
#define RX_CLASSIFIER_FRAME_CLASS_NUM_OF_ENTRIES_PER_REG  	 	(16)   /* In each frame class registers there 16 entries */
#define RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_REG_ADDR_SHIFT  	(4)    /* Shift value to calculate reg addr from sta index  reg Address offset = Sta Index/16*/
#define RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_SHIFT	(1)	   /* Shift value to calculate bit offset in the register  bitoffset  = (Sta Index%16)* 2 */
#define RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_MASK	(0xF)	

/*RTD ctrl register definitions*/
#define RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_LENGTH_CALC_NEGATIVE_DROP_EN_MASK		0x00000020
#define RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_MIS_DATA_PROT_FRAME_DB_DROP_EN_MASK    0x00000040


/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
RxClassifierRam_t *RxClassifierRam;/* pointer to the classifier Ram 0xA6C3B800*/

const RamRxClassifierRdTypeConfiguration_t ramClassifierRdTypeConfiguration = 
{
	/* Unicast Data Configuration*/
	{
		RD_TYPE_UNICAST_NON_QOS_DATA,	/* Data */
		RD_TYPE_DROP,					/* Data + CF Ack */ 
		RD_TYPE_DROP,					/* Data + CF Poll */
		RD_TYPE_DROP,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_NDP, 					/* Null */ 
		RD_TYPE_DROP,					/* CF-Ack */
		RD_TYPE_DROP,					/* CF-Poll */
		RD_TYPE_DROP,					/* CF-Ack + CF Poll */
		RD_TYPE_UNICAST_QOS_DATA,		/* Qos Data */
		RD_TYPE_DROP,					/* Qos Data + CF Ack */
		RD_TYPE_DROP,					/* Qos Data + CF Poll */
		RD_TYPE_DROP,					/* Qos Data + CF-Ack+CF Poll */
		RD_TYPE_NDP,						/* Qos Null */
		RD_TYPE_DROP,					/* Reserved*/
		RD_TYPE_DROP,					/* Qos CF Poll*/
		RD_TYPE_DROP,					/* Qos CF-Ack + CF Poll */
	},

	/* Multicast Data Configuration */
	{
		RD_TYPE_MULTICAST_DATA,			/* Data */
		RD_TYPE_NOT_SUPPORTED,			/* Data + CF Ack */ 
		RD_TYPE_NOT_SUPPORTED, 			/* Data + CF Poll */
		RD_TYPE_NOT_SUPPORTED,			/* Data + CF-Ack+CF Poll */
		RD_TYPE_NDP,						/* Null */ 
		RD_TYPE_NOT_SUPPORTED, 			/* CF-Ack */
		RD_TYPE_NOT_SUPPORTED,			/* CF-Poll */
		RD_TYPE_NOT_SUPPORTED,			/* CF-Ack + CF Poll */
		RD_TYPE_MULTICAST_DATA,			/* Qos Data */
		RD_TYPE_NOT_SUPPORTED,			/* Qos Data + CF Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Qos Data + CF Poll */
		RD_TYPE_NOT_SUPPORTED,			/* Qos Data + CF-Ack+CF Poll */
		RD_TYPE_NDP,						/* Qos Null */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved*/
		RD_TYPE_NOT_SUPPORTED,			/* Qos CF Poll*/
		RD_TYPE_NOT_SUPPORTED, 			/* Qos CF-Ack + CF Poll */
	},
	/* Unicast Management Configuration*/
	{
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Association Req  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Association Rsp*/
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Reassociation Req */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Reassociation Rsp  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Probe Req  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Probe Rsp  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,				/* Reserved  */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,		 		/*ATIM */
		RD_TYPE_UNICAST_MGMT_TYPE_2, 	/* Disassociation*/
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Authentication */
		RD_TYPE_UNICAST_MGMT_TYPE_2,	/* Deauthentication */
		RD_TYPE_UNICAST_MGMT_TYPE_1,	/* Action */
		RD_TYPE_DROP,	                    /* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,				/* Reserved */
	},
	/* Multicast Management Configuration*/
	{
		RD_TYPE_MULTICAST_MGMT, 	/* Association Req	*/
		RD_TYPE_MULTICAST_MGMT, 	/* Association Rsp*/
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Req */
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Rsp  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Req  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Rsp  */
		RD_TYPE_MULTICAST_MGMT, 	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_MULTICAST_MGMT, 	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_MULTICAST_MGMT, 	/* Disassociation*/
		RD_TYPE_MULTICAST_MGMT, 	/* Authentication */
		RD_TYPE_MULTICAST_MGMT, 	/* Deauthentication */
		RD_TYPE_MULTICAST_MGMT, 	/* Action */
		RD_TYPE_DROP, 	                /* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
	},	
	/* Control Frames  Configuration*/
	{
		RD_TYPE_DROP, 		/* Reserved	*/
		RD_TYPE_DROP, 		/* Reserved	*/
		RD_TYPE_DROP, 		/* Reserved*/
		RD_TYPE_DROP, 		/* Reserved */
		RD_TYPE_DROP, 		/* Beamforming Report Poll */
		RD_TYPE_DROP, 		/* VHT NDP Announcement  */
		RD_TYPE_DROP, 		/* Control Frame Extension  */
		RD_TYPE_DROP, 		/* Control Wrapper  */
		RD_TYPE_CONTROL,		/* BAR  */
		RD_TYPE_DROP, 		/* BA  */
		RD_TYPE_DROP,		/* Ps-Poll */
		RD_TYPE_DROP, 		/* RTS*/
		RD_TYPE_DROP, 		/* CTS */
		RD_TYPE_DROP, 		/* Ack */
		RD_TYPE_DROP, 		/* CF-End */
		RD_TYPE_DROP,		/*CF-End+ CF_Ack */
	},
		
	/* Non Associated Unicast  Management Configuration*/
	{
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Association Req	*/
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Association Rsp*/
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Reassociation Req */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Reassociation Rsp  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Probe Req  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Probe Rsp  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,					/* Reserved  */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,					/*ATIM */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Disassociation*/
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Authentication */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2,	/* Deauthentication */
		RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1,		/* Action */
		RD_TYPE_DROP,		                    /* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,					/* Reserved */
	},
	/* Non Associated Multicast  Management Configuration*/
	{
	
		RD_TYPE_MULTICAST_MGMT, 	/* Association Req	*/
		RD_TYPE_MULTICAST_MGMT, 	/* Association Rsp*/
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Req */
		RD_TYPE_MULTICAST_MGMT, 	/* Reassociation Rsp  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Req  */
		RD_TYPE_MULTICAST_MGMT, 	/* Probe Rsp  */
		RD_TYPE_MULTICAST_MGMT, 	/* Timing Advertisement  */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved  */
		RD_TYPE_MULTICAST_MGMT, 	/* Beacon  */
		RD_TYPE_NOT_SUPPORTED,			/*ATIM */
		RD_TYPE_MULTICAST_MGMT, 	/* Disassociation*/
		RD_TYPE_MULTICAST_MGMT, 	/* Authentication */
		RD_TYPE_MULTICAST_MGMT, 	/* Deauthentication */
		RD_TYPE_MULTICAST_MGMT, 	/* Action */
		RD_TYPE_DROP, 	                /* Action No Ack */
		RD_TYPE_NOT_SUPPORTED,			/* Reserved */
	},
};	


//RSSI configuration of RAM-Classifier
const RamRxClassifierRdTypeConfiguration_t ramClassifierRdTypeForRssiConfiguration = 
{
	/* Unicast Data Configuration*/
	{
		RD_TYPE_RSSI,					/* Data */ 
		RD_TYPE_DROP,					/* Data + CF Ack */ 
		RD_TYPE_DROP,					/* Data + CF Poll */
		RD_TYPE_DROP,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_DROP, 					/* Null */ 
		RD_TYPE_DROP,					/* CF-Ack */
		RD_TYPE_DROP,					/* CF-Poll */
		RD_TYPE_DROP,					/* CF-Ack + CF Poll */
		RD_TYPE_RSSI,					/* Qos Data */
		RD_TYPE_DROP,					/* Qos Data + CF Ack */
		RD_TYPE_DROP,					/* Qos Data + CF Poll */
		RD_TYPE_DROP,					/* Qos Data + CF-Ack+CF Poll */
		RD_TYPE_DROP,					/* Qos Null */
		RD_TYPE_DROP,					/* Reserved*/
		RD_TYPE_DROP,					/* Qos CF Poll*/
		RD_TYPE_DROP,					/* Qos CF-Ack + CF Poll */
	},

	/* Multicast Data Configuration */
	{
		RD_TYPE_DROP,					/* Data */
		RD_TYPE_DROP,					/* Data + CF Ack */ 
		RD_TYPE_DROP, 					/* Data + CF Poll */
		RD_TYPE_DROP,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_DROP,					/* Null */ 
		RD_TYPE_DROP, 					/* CF-Ack */
		RD_TYPE_DROP,					/* CF-Poll */
		RD_TYPE_DROP,					/* CF-Ack + CF Poll */
		RD_TYPE_DROP,					/* Qos Data */
		RD_TYPE_DROP,					/* Qos Data + CF Ack */
		RD_TYPE_DROP,					/* Qos Data + CF Poll */
		RD_TYPE_DROP,					/* Qos Data + CF-Ack+CF Poll */
		RD_TYPE_DROP,					/* Qos Null */
		RD_TYPE_DROP,					/* Reserved*/
		RD_TYPE_DROP,					/* Qos CF Poll*/
		RD_TYPE_DROP, 					/* Qos CF-Ack + CF Poll */
	},
	/* Unicast Management Configuration*/
	{
		RD_TYPE_DROP,					/* Association Req  */
		RD_TYPE_DROP,					/* Association Rsp*/
		RD_TYPE_DROP,					/* Reassociation Req */
		RD_TYPE_DROP,					/* Reassociation Rsp  */
		RD_TYPE_DROP,					/* Probe Req  */
		RD_TYPE_DROP,					/* Probe Rsp  */
		RD_TYPE_DROP,					/* Timing Advertisement  */
		RD_TYPE_DROP,					/* Reserved  */
		RD_TYPE_DROP,					/* Beacon  */
		RD_TYPE_DROP,		 			/* ATIM */
		RD_TYPE_DROP, 			  		/* Disassociation*/
		RD_TYPE_DROP,					/* Authentication */
		RD_TYPE_DROP,					/* Deauthentication */
		RD_TYPE_DROP,					/* Action */
		RD_TYPE_DROP,					/* Action No Ack */
		RD_TYPE_DROP,					/* Reserved */
	},
	/* Multicast Management Configuration*/
	{
		RD_TYPE_DROP, 					/* Association Req	*/
		RD_TYPE_DROP, 					/* Association Rsp*/
		RD_TYPE_DROP, 					/* Reassociation Req */
		RD_TYPE_DROP, 					/* Reassociation Rsp  */
		RD_TYPE_DROP, 					/* Probe Req  */
		RD_TYPE_DROP, 					/* Probe Rsp  */
		RD_TYPE_DROP, 					/* Timing Advertisement  */
		RD_TYPE_DROP,					/* Reserved  */
		RD_TYPE_DROP, 					/* Beacon  */
		RD_TYPE_DROP,					/*ATIM */
		RD_TYPE_DROP, 					/* Disassociation*/
		RD_TYPE_DROP, 					/* Authentication */
		RD_TYPE_DROP, 					/* Deauthentication */
		RD_TYPE_DROP, 					/* Action */
		RD_TYPE_DROP, 					/* Action No Ack */
		RD_TYPE_DROP,					/* Reserved */
	},	
	/* Control Frames  Configuration*/
	{
		RD_TYPE_DROP, 					/* Reserved	*/
		RD_TYPE_DROP, 					/* Reserved	*/
		RD_TYPE_DROP, 					/* Reserved*/
		RD_TYPE_DROP, 					/* Reserved */
		RD_TYPE_DROP, 					/* Beamforming Report Poll */
		RD_TYPE_DROP, 					/* VHT NDP Announcement  */
		RD_TYPE_DROP, 					/* Control Frame Extension  */
		RD_TYPE_DROP, 					/* Control Wrapper  */
		RD_TYPE_DROP,					/* BAR  */
		RD_TYPE_DROP, 					/* BA  */
		RD_TYPE_DROP,					/* Ps-Poll */
		RD_TYPE_DROP, 					/* RTS*/
		RD_TYPE_DROP, 					/* CTS */
		RD_TYPE_DROP, 					/* Ack */
		RD_TYPE_DROP, 					/* CF-End */
		RD_TYPE_DROP,					/*CF-End+ CF_Ack */
	},
		
	/* Non Associated Unicast  Management Configuration*/
	{
		RD_TYPE_DROP,					/* Association Req	*/
		RD_TYPE_DROP,					/* Association Rsp*/
		RD_TYPE_DROP,					/* Reassociation Req */
		RD_TYPE_DROP,					/* Reassociation Rsp  */
		RD_TYPE_DROP,					/* Probe Req  */
		RD_TYPE_DROP,					/* Probe Rsp  */
		RD_TYPE_DROP,					/* Timing Advertisement  */
		RD_TYPE_DROP,					/* Reserved  */
		RD_TYPE_DROP,					/* Beacon  */
		RD_TYPE_DROP,					/*ATIM */
		RD_TYPE_DROP,					/* Disassociation*/
		RD_TYPE_DROP,					/* Authentication */
		RD_TYPE_DROP,					/* Deauthentication */
		RD_TYPE_DROP,					/* Action */
		RD_TYPE_DROP,					/* Action No Ack */
		RD_TYPE_DROP,					/* Reserved */
	},
	/* Non Associated Multicast  Management Configuration*/
	{
	
		RD_TYPE_DROP, 					/* Association Req	*/
		RD_TYPE_DROP, 					/* Association Rsp*/
		RD_TYPE_DROP, 					/* Reassociation Req */
		RD_TYPE_DROP, 					/* Reassociation Rsp  */
		RD_TYPE_DROP, 					/* Probe Req  */
		RD_TYPE_DROP, 					/* Probe Rsp  */
		RD_TYPE_DROP, 					/* Timing Advertisement  */
		RD_TYPE_DROP,					/* Reserved  */
		RD_TYPE_DROP, 					/* Beacon  */
		RD_TYPE_DROP,					/*ATIM */
		RD_TYPE_DROP, 					/* Disassociation*/
		RD_TYPE_DROP, 					/* Authentication */
		RD_TYPE_DROP, 					/* Deauthentication */
		RD_TYPE_DROP, 					/* Action */
		RD_TYPE_DROP, 					/* Action No Ack */
		RD_TYPE_DROP,					/* Reserved */
	},
};	




const RamRxClassifierRdTypeConfiguration_t ramClassifierRdTypeForSnifferConfiguration = 
{
	/* Unicast Data Configuration*/
	{
		RD_TYPE_SNIFFER,					/* Data */
		RD_TYPE_SNIFFER,					/* Data + CF Ack */ 
		RD_TYPE_SNIFFER,					/* Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER, 				/* Null */ 
		RD_TYPE_SNIFFER,					/* CF-Ack */
		RD_TYPE_SNIFFER,					/* CF-Poll */
		RD_TYPE_SNIFFER,					/* CF-Ack + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Ack */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER,						/* Qos Null */
		RD_TYPE_SNIFFER,					/* Reserved*/
		RD_TYPE_SNIFFER,					/* Qos CF Poll*/
		RD_TYPE_SNIFFER,					/* Qos CF-Ack + CF Poll */
	},

	/* Multicast Data Configuration */
	{
		RD_TYPE_SNIFFER,					/* Data */
		RD_TYPE_SNIFFER,					/* Data + CF Ack */ 
		RD_TYPE_SNIFFER, 				/* Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER,					/* Null */ 
		RD_TYPE_SNIFFER, 				/* CF-Ack */
		RD_TYPE_SNIFFER,					/* CF-Poll */
		RD_TYPE_SNIFFER,					/* CF-Ack + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Ack */
		RD_TYPE_SNIFFER,					/* Qos Data + CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Data + CF-Ack+CF Poll */
		RD_TYPE_SNIFFER,					/* Qos Null */
		RD_TYPE_SNIFFER,					/* Reserved*/
		RD_TYPE_SNIFFER,					/* Qos CF Poll*/
		RD_TYPE_SNIFFER, 				/* Qos CF-Ack + CF Poll */
	},
	/* Unicast Management Configuration*/
	{
		RD_TYPE_SNIFFER,					/* Association Req  */
		RD_TYPE_SNIFFER,					/* Association Rsp*/
		RD_TYPE_SNIFFER,					/* Reassociation Req */
		RD_TYPE_SNIFFER,					/* Reassociation Rsp  */
		RD_TYPE_SNIFFER,					/* Probe Req  */
		RD_TYPE_SNIFFER,					/* Probe Rsp  */
		RD_TYPE_SNIFFER,					/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER,					/* Beacon  */
		RD_TYPE_SNIFFER,		 			/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER,					/* Authentication */
		RD_TYPE_SNIFFER,					/* Deauthentication */
		RD_TYPE_SNIFFER,					/* Action */
		RD_TYPE_SNIFFER,					/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
	},
	/* Multicast Management Configuration*/
	{
		RD_TYPE_SNIFFER, 				/* Association Req	*/
		RD_TYPE_SNIFFER, 				/* Association Rsp*/
		RD_TYPE_SNIFFER, 				/* Reassociation Req */
		RD_TYPE_SNIFFER, 				/* Reassociation Rsp  */
		RD_TYPE_SNIFFER, 				/* Probe Req  */
		RD_TYPE_SNIFFER, 				/* Probe Rsp  */
		RD_TYPE_SNIFFER, 				/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER, 				/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER, 				/* Authentication */
		RD_TYPE_SNIFFER, 				/* Deauthentication */
		RD_TYPE_SNIFFER, 				/* Action */
		RD_TYPE_SNIFFER, 				/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
	},	
	/* Control Frames  Configuration*/
	{
		RD_TYPE_SNIFFER, 				/* Reserved	*/
		RD_TYPE_SNIFFER, 				/* Reserved	*/
		RD_TYPE_SNIFFER, 				/* Reserved*/
		RD_TYPE_SNIFFER, 				/* Reserved */
		RD_TYPE_SNIFFER, 				/* Beamforming Report Poll */
		RD_TYPE_SNIFFER, 				/* VHT NDP Announcement  */
		RD_TYPE_SNIFFER, 				/* Control Frame Extension  */
		RD_TYPE_SNIFFER, 				/* Control Wrapper  */
		RD_TYPE_SNIFFER,					/* BAR  */
		RD_TYPE_SNIFFER, 				/* BA  */
		RD_TYPE_SNIFFER,					/* Ps-Poll */
		RD_TYPE_SNIFFER, 				/* RTS*/
		RD_TYPE_SNIFFER, 				/* CTS */
		RD_TYPE_SNIFFER, 				/* Ack */
		RD_TYPE_SNIFFER, 				/* CF-End */
		RD_TYPE_SNIFFER,					/*CF-End+ CF_Ack */
	},
		
	/* Non Associated Unicast  Management Configuration*/
	{
		RD_TYPE_SNIFFER,					/* Association Req	*/
		RD_TYPE_SNIFFER,					/* Association Rsp*/
		RD_TYPE_SNIFFER,					/* Reassociation Req */
		RD_TYPE_SNIFFER,					/* Reassociation Rsp  */
		RD_TYPE_SNIFFER,					/* Probe Req  */
		RD_TYPE_SNIFFER,					/* Probe Rsp  */
		RD_TYPE_SNIFFER,					/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER,					/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER,					/* Disassociation*/
		RD_TYPE_SNIFFER,					/* Authentication */
		RD_TYPE_SNIFFER,					/* Deauthentication */
		RD_TYPE_SNIFFER,					/* Action */
		RD_TYPE_SNIFFER,					/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
	},
	/* Non Associated Multicast  Management Configuration*/
	{
	
		RD_TYPE_SNIFFER, 				/* Association Req	*/
		RD_TYPE_SNIFFER, 				/* Association Rsp*/
		RD_TYPE_SNIFFER, 				/* Reassociation Req */
		RD_TYPE_SNIFFER, 				/* Reassociation Rsp  */
		RD_TYPE_SNIFFER, 				/* Probe Req  */
		RD_TYPE_SNIFFER, 				/* Probe Rsp  */
		RD_TYPE_SNIFFER, 				/* Timing Advertisement  */
		RD_TYPE_SNIFFER,					/* Reserved  */
		RD_TYPE_SNIFFER, 				/* Beacon  */
		RD_TYPE_SNIFFER,					/*ATIM */
		RD_TYPE_SNIFFER, 				/* Disassociation*/
		RD_TYPE_SNIFFER, 				/* Authentication */
		RD_TYPE_SNIFFER, 				/* Deauthentication */
		RD_TYPE_SNIFFER, 				/* Action */
		RD_TYPE_SNIFFER, 				/* Action No Ack */
		RD_TYPE_SNIFFER,					/* Reserved */
	},
};	

/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/
uint32 rxCLassifierFullBufferCounter; 


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

RxClassifier_SetStaFrameClass 

Description:
------------
	Set allowed recieved frame class for the required station
	The register address and bit offset in the register are derived from
	the register structure: each register there 16 entries each one of them is 2 bit width
Input:
-----
	staIndex - The sta index to configure the allowed frame class
	allowedFrameClass - the allowed frmae class for the sta
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void RxClassifier_SetStaFrameClass(uint32 staIndex,uint32 allowedFrameClass)
{
	uint32 regAddress,regMask,regValue;
	uint8  regBitOffset;

	regAddress = REG_RX_CLASSIFIER_FRAME_CLASS_STA_0TO15 + ((staIndex >> RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_REG_ADDR_SHIFT)*4);
	regBitOffset = (staIndex & RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_MASK) << RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_SHIFT;
	regMask = RX_CLASSIFIER_FRAME_CLASS_ENTRY_MASK << regBitOffset;
	regValue = (allowedFrameClass & RX_CLASSIFIER_FRAME_CLASS_ENTRY_MASK) << regBitOffset;

	RegAccess_WriteMasked(regAddress,regMask,regValue);
}

/**********************************************************************************

RxClassifier_GetStaFrameClass 

Description:
------------
	Read the allowed recieved frame class for the required station
	The register address and bit offset in the register are derived from
	the register structure: each register there 16 entries each one of them is 2 bit width
Input:
-----
	staIndex - The sta index to configure the allowed frame class
Output:
-------
	allowedFrameClass - the allowed frmae class for the sta
Returns:
--------
	void
**********************************************************************************/
uint8 RxClassifier_GetStaFrameClass(uint32 staIndex)
{
	uint32 regAddress,regValue;
	uint32 allowedFrameClass;
	uint8  regBitOffset;
	
	

	regAddress = REG_RX_CLASSIFIER_FRAME_CLASS_STA_0TO15 + ((staIndex >> RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_REG_ADDR_SHIFT)*4);
	regBitOffset = (staIndex & RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_MASK) << RX_CLASSIFIER_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_SHIFT;
	
	RegAccess_Read(regAddress,&regValue);

	allowedFrameClass = (regValue >> regBitOffset)& RX_CLASSIFIER_FRAME_CLASS_ENTRY_MASK;

	return (uint8)allowedFrameClass;
}

/**********************************************************************************
RxClassifier_GetRssiTypeFlag 

Description:
------------
	Get RSSI type flag to indicate if channel switch case is RSSI or not
	NOTE: this is not needed in WAVE500B therefore should return FALSE always
Input: 
-----	
Output:
-------
	None
Returns:
--------
	ENABLED(1)/DISABLED(0) - RSSI type enabled/disabled
**********************************************************************************/
bool RxClassifier_GetRssiTypeFlag(void)
{
	// Currently not implemented in wave500.
	ASSERT(0);
	return 0;	
}

/**********************************************************************************

RxClassifier_ReadClassViolationCache 

Description:
------------
	Read an entry the frame class violation cache
Input:
-----
	None
	
Output:
-------
	frameClassViolationDetails
Returns:
--------
	void
**********************************************************************************/
void RxClassifier_ReadClassViolationCache(classViolationDetails_t *frameClassViolationDetails)
{
	RegRxClassifierClassViolIrq_u classViolationIrqReg;
	RamRxClassifierFrameClassViolationCacheEntry_t cacheEntry;

	uint32 cacheIndex;

	/* Read Int Status (Pending Indication)*/
	RegAccess_Read(REG_RX_CLASSIFIER_CLASS_VIOL_IRQ,&classViolationIrqReg.val);

	if(classViolationIrqReg.val)
	{
		cacheIndex = Utils_CountTrailingZeros(classViolationIrqReg.val);

		memcpy32(&cacheEntry, &RxClassifierRam->ClassViolationCache[cacheIndex], CONVERT_BYTES_TO_WORDS(sizeof(RamRxClassifierFrameClassViolationCacheEntry_t)));
		
		/* Copy the mac address which casued the frame violation*/
		memcpy(&frameClassViolationDetails->macAddress,&cacheEntry.macAddr0to4,sizeof(IEEE_ADDR));

		frameClassViolationDetails->vapId = cacheEntry.vapIndex;
		frameClassViolationDetails->indexInCache = cacheIndex;
		frameClassViolationDetails->valid = TRUE;
		
		/* Clear Int Status - by writing '1' */
		RegAccess_Write(REG_RX_CLASSIFIER_CLASS_VIOL_CLR_IRQ,(1 << cacheIndex));
	}
	else
	{
		frameClassViolationDetails->valid = FALSE;
	}
}

/**********************************************************************************

RxClassifier_ResetClass3ViolationValidEntry 

Description:
------------
	Reset frame class  violation  entry in cache
Input:
-----
	cacheIndex - The entry Index to clear 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void RxClassifier_ResetClass3ViolationValidEntry(uint32 cacheIndex)
{
	RegRxClassifierClassViolClrValidAdd_u classViolationClearValidAddrReg;
	RegRxClassifierClassViolBufFullCnt_u classViolationBufferFullCounterReg;

	classViolationClearValidAddrReg.val = 0;
	
	classViolationClearValidAddrReg.bitFields.frameClassViolClrValidMacAddr = 1 << cacheIndex;
	RegAccess_Write(REG_RX_CLASSIFIER_CLASS_VIOL_CLR_VALID_ADD,classViolationClearValidAddrReg.val);

	/* Read Full Buffer Counter and increment statistics*/
	RegAccess_Read(REG_RX_CLASSIFIER_CLASS_VIOL_BUF_FULL_CNT,&classViolationBufferFullCounterReg.val);
	rxCLassifierFullBufferCounter += classViolationBufferFullCounterReg.bitFields.frameClassViolBufFullCnt;
}


/**********************************************************************************
RxClassifier_ConfigureRegularMode 

Description:
------------
	Configuration of Rx Classifier (Gen5) for regular mode (operational mode = NORMAL, DUT)
Input:
-----
	None 
Output:
-------
	None
Returns:
--------
	None
*********************************************************************************/

void RxClassifier_ConfigureRegularMode(void)

{
	RegRxClassifierRxqIdFilter0_u rxqIdFilterConfiguration;
	RegRxClassifierRxqIdFilterControl_u rxqIdFilterControl;
	uint32 regAddr;

	rxqIdFilterConfiguration.val = 0;
	rxqIdFilterControl.val = 0;

	/* RD Type Configuration*/
	memcpy32(&RxClassifierRam->rdTypeConfiguration,&ramClassifierRdTypeConfiguration, sizeof(RamRxClassifierRdTypeConfiguration_t) >> 2);
	
	/* Frame Class Filter  Configuration - Set all Sta Frame Class to 1*/
 
	for(regAddr = REG_RX_CLASSIFIER_FRAME_CLASS_STA_0TO15; regAddr <= REG_RX_CLASSIFIER_FRAME_CLASS_STA_112TO127; regAddr += 4)
	{
		RegAccess_Write(regAddr,RX_CLASSIFIER_STA_FRAME_CLASS_INIT_VALUE);	
	}

	/*RXQID Filter Configuration */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0DropEn = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameType = FM_CONTROL_FTYPE_MANAGEMENT;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0SubtypeEqual = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1IdValNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr2IdValNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1Unicast = FALSE;
	/*Accept Beacon Frames */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_BEACON;	
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER0,rxqIdFilterConfiguration.val);
	/*Accept Probe Req Frames */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_PROBE_REQ;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER1,rxqIdFilterConfiguration.val);
	/*Accept Multicast actions */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER2,rxqIdFilterConfiguration.val);
	/*Accept Multicast Deauthenticate */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_DEAUTHENTICATION;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER3,rxqIdFilterConfiguration.val);
	/*Accept Multicast Disassociate */
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameSubtype = MGMT_FRAME_SUBTYPE_DISASSOCIATION;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER4,rxqIdFilterConfiguration.val);
	/* Gather subfilters 0-4 as AND group */		
	rxqIdFilterControl.bitFields.rxqIdFilterGroupAndLogic = RX_CLASSIFIER_QXQ_ID_FILTER_GROUP_LOGIC_NUMBER;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER_CONTROL,rxqIdFilterControl.val);
	/* Reject Multicast Data frames*/	
	rxqIdFilterConfiguration.val = 0;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0DropEn = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0FrameType = FM_CONTROL_FTYPE_DATA;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0SubtypeNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1IdValNa = TRUE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr2IdVal = FALSE;
	rxqIdFilterConfiguration.bitFields.rxqIdFilter0Addr1Unicast = FALSE;
	RegAccess_Write(REG_RX_CLASSIFIER_RXQ_ID_FILTER3,rxqIdFilterConfiguration.val);
}


/**********************************************************************************
RxClassifier_HandleRxManagerAddVap 

Description:
------------
	Configuration of Rx Classifier according to operational mode
Input:
-----
	UMI_ADD_VAP* addVapStructurePtr
Output:
-------
	None
Returns:
--------
	None
*********************************************************************************/
void RxClassifier_HandleRxManagerAddVap(UMI_ADD_VAP* addVapStructurePtr)
{
	RegMacHtExtensionsFc2Mode_u htExtension;
	if (addVapStructurePtr->operationMode != OPERATION_MODE_SNIFFER)
	{
		RxClassifier_ConfigureRegularMode();
	}
	else
	{
		//Disable rx classifier filters
		RxClassifier_ConfigureSnifferMode();
	}
	if (addVapStructurePtr->operationMode == OPERATION_MODE_DUT)
	{
		htExtension.val = REGISTER(REG_MAC_HT_EXTENSIONS_FC2_MODE);
		htExtension.bitFields.deliniatorVhtDebugMode =1;
		REGISTER(REG_MAC_HT_EXTENSIONS_FC2_MODE) = htExtension.val;
	}
}

/**********************************************************************************

RxClassifier_ConfigureNormalMode 

Description:
------------
	Configuration of Rx classifier for operation mode NORMAL_AFTER_RSSI:
		
Input:
-----
	None 
	
Output:
-------
	None
Returns:
--------
	void
*********************************************************************************/
void RxClassifier_ConfigureNormalMode(void)
{
	RegRxClassifierRxClasRtdCtrl_u	rxClasRtdCtrl;
	
	/* RD Type Configuration*/
	memcpy32(&RxClassifierRam->rdTypeConfiguration, &ramClassifierRdTypeConfiguration, sizeof(RamRxClassifierRdTypeConfiguration_t) >> 2);
#ifdef BEEROCK_DEBUG	
	ILOG0_V("[RxClassifier_ConfigureNormalMode]-------- Rx Classifier RAM configured for NORMAL_AFTER_RSSI --------");
#endif

	//enable decryption as in normal mode (staDB stays as before)
	rxClasRtdCtrl.bitFields.lengthCalcNegativeDropEn = ENABLED;
	rxClasRtdCtrl.bitFields.misDataProtFrameDbDropEn = ENABLED;

	RegAccess_WriteMasked(REG_RX_CLASSIFIER_RX_CLAS_RTD_CTRL, (RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_LENGTH_CALC_NEGATIVE_DROP_EN_MASK | RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_MIS_DATA_PROT_FRAME_DB_DROP_EN_MASK), rxClasRtdCtrl.val);
#ifdef BEEROCK_DEBUG	
	ILOG0_V("[RxClassifier_ConfigureNormalMode]--------------------------end");
#endif
}

/**********************************************************************************

RxClassifier_ConfigureRssiMode 

Description:
------------
	Configuration of Rx classifier for operation mode RSSI:
		
Input:
-----
	None 
	
Output:
-------
	None
Returns:
--------
	void
*********************************************************************************/
void RxClassifier_ConfigureRssiMode(void)
{
	RegRxClassifierRxClasRtdCtrl_u	rxClasRtdCtrl;
	//for DEBUG///////////////
#ifdef BEEROCK_DEBUG	
	StaId staIndex = 1;
#endif
	////end DEBUG/////////////
	
	/* RD Type Configuration*/
	memcpy32(&RxClassifierRam->rdTypeConfiguration, &ramClassifierRdTypeForRssiConfiguration, sizeof(RamRxClassifierRdTypeConfiguration_t) >> 2);
#ifdef BEEROCK_DEBUG
	ILOG0_V("[RxClassifier_ConfigureRssiMode]-------- Rx Classifier RAM configured for RSSI --------");
#endif

	//disable decryption
	//assuming STA_DB fields: dataEncryptionEnable:1;  managementEncryptionEnable:1; in StaDB Word 2 stay zero since no one is setting them
	//staDbFillDbEntry function is setting these params to DISABLE
#ifdef BEEROCK_DEBUG
	ILOG0_DDDD("[RxClassifier_ConfigureRssiMode], StaDB_IsDataEncrypted(%d) = %d, StaDB_IsManagementEncrypted(%d) = %d", staIndex, StaDB_IsDataEncrypted(staIndex), staIndex, StaDB_IsManagementEncrypted(staIndex));
#endif
	//disable negative length drop and mis data protection drop (were enabled by default)
	rxClasRtdCtrl.bitFields.lengthCalcNegativeDropEn = DISABLED;
	rxClasRtdCtrl.bitFields.misDataProtFrameDbDropEn = DISABLED;

	RegAccess_WriteMasked(REG_RX_CLASSIFIER_RX_CLAS_RTD_CTRL, (RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_LENGTH_CALC_NEGATIVE_DROP_EN_MASK | RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_MIS_DATA_PROT_FRAME_DB_DROP_EN_MASK), rxClasRtdCtrl.val);
#ifdef BEEROCK_DEBUG
	ILOG0_V("[RxClassifier_ConfigureRssiMode]------------------------------------end]");
#endif
}



/**********************************************************************************

RxClassifier_ConfigureSnifferMode 

Description:
------------
	Configuration of Rx classifier for operation mode SNIFFER:
	
	- Configure RD Type = Sniffer for all frame types (this is done by writing the configuration table to specific place in Classifier's RAM).
	- Filters off:
		o	Frame class filter
		o	Length Calculation negative drop.
		o	RXQ_ID filter drop - configure for all 16 filters.
		o	RXQ_ID filter group and logic.

Input:
-----
	None 
	
Output:
-------
	None
Returns:
--------
	void
*********************************************************************************/
void RxClassifier_ConfigureSnifferMode(void)
{

	RegRxClassifierFrameClassFilterControl_u frameClassFilterControl;
	RegRxClassifierRxClasRtdCtrl_u	rxClasRtdCtrl;
	RegRxClassifierRxqIdFilter0_u rxqIdFilterConfiguration;
	uint32 regAddr;

	/* RD Type Configuration*/
	memcpy32(&RxClassifierRam->rdTypeConfiguration,&ramClassifierRdTypeForSnifferConfiguration, sizeof(RamRxClassifierRdTypeConfiguration_t) >> 2);

	//disable frame class filter - by default all filters were on
	frameClassFilterControl.val = 0x0;
	RegAccess_Write(REG_RX_CLASSIFIER_FRAME_CLASS_FILTER_CONTROL, frameClassFilterControl.val);	
	
	//disable negative length drop and mis data protection drop (were enabled by default)
	rxClasRtdCtrl.bitFields.lengthCalcNegativeDropEn = 0x0;
	rxClasRtdCtrl.bitFields.misDataProtFrameDbDropEn = 0x0;
	RegAccess_WriteMasked(REG_RX_CLASSIFIER_RX_CLAS_RTD_CTRL, (RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_LENGTH_CALC_NEGATIVE_DROP_EN_MASK | RX_CLASSIFIER_RX_CLAS_RTD_CTRL_FIELD_MIS_DATA_PROT_FRAME_DB_DROP_EN_MASK), rxClasRtdCtrl.val);
	
	//remove rxqid drop for all 16 filters - when 'rxq_id_filter0_drop_en' field is FALSE, all other fields not relevant, thus we will zero the entire register
	rxqIdFilterConfiguration.val = 0x0;
	//write rxqIdFilterConfiguration for all 16 rxqid filters
	for(regAddr = REG_RX_CLASSIFIER_RXQ_ID_FILTER0; regAddr <= REG_RX_CLASSIFIER_RXQ_ID_FILTER15; regAddr += 4)
	{
		RegAccess_Write(regAddr, rxqIdFilterConfiguration.val);	
	}

#ifdef	SNIFFER_DEBUG
	ILOG0_V("-------- Rx Classifier configured for Sniffer --------");
#endif
}


/**********************************************************************************

RxClassifier_Init 

Description:
------------
	Initialization of Rx Classifier parametes
		1) Set Internal Memory to 0
		2) Configure RD Type Table
		2) Number Of Security Decriptors -> Use Default (= 63)
		3) Number of RTD ring entries      -> Use Default (= 63)
		4) Wep Confiuration - Key Location Vap or Sta DB -> use default value(= 0(Sta DB)) 
		5)Filters Configuration
			5.1) Frame Class Violation Filter -Use default values for frame class configuration (according to type\subtype)
			 							Set all Station Class to 0
			 5.2) RXQID Filter - And between filter 0-2 which sign as drop multicast frames except Beacon,Probe,Multicat Action (Public Action)		
Input:
-----
	None
Output:
-------
	None
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void RxClassifier_Init(void)
{
	RegRxClassifierMaxSecDescriptors_u maxSecDescriptorsReg;

	maxSecDescriptorsReg.val = 0;

	RxClassifierRam = (RxClassifierRam_t *)(RX_CLASSIFIER_BASE_ADDR + RX_CLASSIFIER_RAM_ADDR_OFFSET);
	/* Initial Internal Ram */
	memset32(RxClassifierRam,0,sizeof(RxClassifierRam_t) >> 2);
	
	/* Initialize number Of Security Descriptors */
	maxSecDescriptorsReg.bitFields.maxSecDescriptors = (RX_NUM_OF_SECURITY_DESCRIPTORS - 1);
	RegAccess_Write(REG_RX_CLASSIFIER_MAX_SEC_DESCRIPTORS,maxSecDescriptorsReg.val);
	
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

