/***********************************************************************************
 File:		RxCoordinator.c
 Module:		Rx Coordinator
 Purpose:		Owner for all access to the Rx Coordinator
 Description:	This module is the API to the Rx Coordinator which is responbile for frame filtering and classification 
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "RegAccess_Api.h"
#include "Utils_Api.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "RxCoordinator_Api.h"
#include "RxCoordinator.h"
#include "HwMemoryMap.h"
#include "HwQManager_API.h"
#include "RxCoordinatorRam.h"
#include "HwGlobalDefinitions.h"
#include "RxDescriptor.h"
#include "PacketDescriptor.h"
#include "frame.h"
#include "loggerAPI.h"
#include "ConfigurationManager_api.h"



/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 16

#define	RX_COORDINATOR_DISABLE_IDLE 		(0x0)

#define SNIFFER_MODE_DLM_LIST_BEHAVIOR_MASK	(0xFE) //mask for all list except RD_SOURCE_TO_HOST_MEMORY

/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
RxCoordinatorRam_t *pRxCoordinatorRam_Band0; /* pointer to the Coordinator Ram of band 0 */
RxCoordinatorRam_t *pRxCoordinatorRam_Band1; /* pointer to the Coordinator Ram of band 1 */


/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

RxCoordinator_ReadClassViolationCache 

Description:
------------
	Read an entry the frame class violation cache
Input:
-----

	
Output:
-------
	frameClassViolationDetails
Returns:
--------
	void
**********************************************************************************/
void RxCoordinator_ReadClassViolationCache(classViolationDetails_t *frameClassViolationDetails, uint8 vapId)
{
	RegRxCoordinatorFrameClassVioIrq_u 	classViolationIrqReg;
	uint32 								cacheIndex;
	uint8 								bandId = ConfigurationManager_GetBandForVap(vapId);
	RxCoordinatorRam_t*					pCoordinatorRam; // Pointer to the RAM of the correct band

	if (bandId == CONFIGURATION_MANAGER_BAND_0)
	{
		pCoordinatorRam = pRxCoordinatorRam_Band0;
	}
	else
	{
		pCoordinatorRam = pRxCoordinatorRam_Band1;
	}

	/* Read Int Status (Pending Indication)*/
	RegAccess_ReadPerBand(REG_RX_COORDINATOR_FRAME_CLASS_VIO_IRQ, &classViolationIrqReg.val, bandId);

	if(classViolationIrqReg.val)
	{
		cacheIndex = Utils_CountTrailingZeros(classViolationIrqReg.val);
		
		/* Copy the mac address which caused the frame violation*/
		MEMCPY(&frameClassViolationDetails->macAddress, &pCoordinatorRam->ClassViolationCache[cacheIndex], sizeof(IEEE_ADDR));

		frameClassViolationDetails->vapId = pCoordinatorRam->ClassViolationCache[cacheIndex].vapIndex;
		frameClassViolationDetails->indexInCache = cacheIndex;
		frameClassViolationDetails->valid = TRUE;
		
		/* Clear Int Status - by writing '1' */
		RegAccess_WritePerBand(REG_RX_COORDINATOR_FRAME_CLASS_VIO_CLR_IRQ, (1 << cacheIndex), bandId);
	}
	else
	{
		frameClassViolationDetails->valid = FALSE;
	}
}


/**********************************************************************************

RxCoordinator_ResetClass3ViolationValidEntry 

Description:
------------
	Reset frame class  violation  entry in cache (for both bands)
Input:
-----
	cacheIndex - The entry Index to clear 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void RxCoordinator_ResetClass3ViolationValidEntry(uint32 cacheIndex, uint8 vapId)
{
	RegRxCoordinatorFrameClassVioClrValidMacAddr_u classViolationClearValidAddrReg;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	classViolationClearValidAddrReg.val = 0;
	classViolationClearValidAddrReg.bitFields.frameClassViolValidMacAddrClr = 1 << cacheIndex;
	// Clear the cache of band 0 (always active).
	RegAccess_WritePerBand(REG_RX_COORDINATOR_FRAME_CLASS_VIO_CLR_VALID_MAC_ADDR,classViolationClearValidAddrReg.val, bandId);
}

/**********************************************************************************

RxCoordinator_SetSnifferModeConfiguration


Description:
------------

Input:
-----
	None
Output:
-------
	None
**********************************************************************************/
void RxCoordinator_SetSnifferModeConfiguration(uint8 vapId)
{
	RegRxCoordinatorDlmListRdFreeCfg1_u coordinatorDlmListRdFreeCfg1Reg;
	RegRxCoordinatorDlmListBehavior_u	coordinatorDlmListBehavioReg;
	uint8 bandId = ConfigurationManager_GetBandForVap(vapId);

	coordinatorDlmListRdFreeCfg1Reg.val = 0;
	coordinatorDlmListBehavioReg.val 	= 0;
	coordinatorDlmListRdFreeCfg1Reg.bitFields.rdFreeList0Idx = HW_Q_MANAGER_RX_POST_PROCESSING_INPUT_LIST;
	RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_RD_FREE_CFG1, coordinatorDlmListRdFreeCfg1Reg.val, bandId);

	coordinatorDlmListBehavioReg.bitFields.rdFreeListIsIndeedFree &= SNIFFER_MODE_DLM_LIST_BEHAVIOR_MASK; 
	RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_BEHAVIOR, coordinatorDlmListBehavioReg.val, bandId);
}
/**********************************************************************************

RxCoordinator_AddFirstVap

Description:
------------
	Activate the Rx coordinator module

Input:
-----
	None
Output:
-------
	None
**********************************************************************************/
void RxCoordinator_PostInit(void)
{
	RegRxCoordinatorEnableStateMachine_u coordinatorEnableStateMachine;
	uint8 bandId;

	coordinatorEnableStateMachine.val = 0;
	
	coordinatorEnableStateMachine.bitFields.swStatusTraceStayIdle = RX_COORDINATOR_DISABLE_IDLE;
	coordinatorEnableStateMachine.bitFields.swWrapUpStayIdle = RX_COORDINATOR_DISABLE_IDLE;

	for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
	{
		RegAccess_WritePerBand(REG_RX_COORDINATOR_ENABLE_STATE_MACHINE, coordinatorEnableStateMachine.val, bandId);
	}
}

/**********************************************************************************

RxCoordinator_Init 

Description:
------------
	Initialization of Rx coordinator parametes

Input:
-----
	None
Output:
-------
	None
**********************************************************************************/
void RxCoordinator_Init(void)
{
	RegRxCoordinatorDlmListCfg1_u 			coordinatorDlmListCfg1;
	RegRxCoordinatorDlmListCfg2_u 			coordinatorDlmListCfg2;
	RegRxCoordinatorDlmListRdFreeCfg1_u 	coordinatorDlmListRdFreeCfg1;
	RegRxCoordinatorDlmListRdFreeCfg2_u 	coordinatorDlmListRdFreeCfg2;
	RegRxCoordinatorDlmListBehavior_u	 	coordinatorDlmListBehavior;
	RegRxCoordinatorWrapUpRxhErrorPass_u 	coordinatorWrapUpRxhErrorPass;
	RegRxCoordinatorBaAgreementEn_u			coordinatorBaAgreementEn;
	uint8 bandId;


	coordinatorDlmListCfg1.val 			= 0;
	coordinatorDlmListCfg2.val 			= 0;
	coordinatorDlmListRdFreeCfg1.val 	= 0;
	coordinatorDlmListRdFreeCfg2.val 	= 0;
	coordinatorDlmListBehavior.val 		= 0;
	coordinatorWrapUpRxhErrorPass.val 	= 0;
	coordinatorBaAgreementEn.val 		= 0;

	// Init RAM of band 0 (always needed)	
	pRxCoordinatorRam_Band0 = (RxCoordinatorRam_t *)(B0_FRAME_CLASS_VIO_BUFFER_BASE_ADDR);
	memset32(pRxCoordinatorRam_Band0, 0, sizeof(RxCoordinatorRam_t) >> 2); // Band0

	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
	{
		// Init RAM of band 1 (if exists)
		pRxCoordinatorRam_Band1 = (RxCoordinatorRam_t *)(B1_FRAME_CLASS_VIO_BUFFER_BASE_ADDR);
		memset32(pRxCoordinatorRam_Band1, 0, sizeof(RxCoordinatorRam_t) >> 2); // Band1
	}

	coordinatorDlmListCfg1.bitFields.rxWrapUpInputListIdx 	= HW_Q_MANAGER_RX_MPDU_WRAP_UP_IN_LIST;
	coordinatorDlmListCfg1.bitFields.postPpduPhyStatusIdx 	= HW_Q_MANAGER_RX_MPDU_POST_PPDU_PHY_LIST;
	coordinatorDlmListCfg1.bitFields.rxhPassIdx				= HW_Q_MANAGER_RX_MPDU_RXH_IN_LIST;
	coordinatorDlmListCfg1.bitFields.rxhErrorListIdx 		= HW_Q_MANAGER_RX_MPDU_RXH_ERROR_LIST;

	coordinatorDlmListCfg2.bitFields.mpduFreeListIdx 		= HW_Q_MANAGER_RX_MPDU_RXC_IN_LIST;
	coordinatorDlmListCfg2.bitFields.rxPpInputListIdx 		= HW_Q_MANAGER_RX_POST_PROCESSING_INPUT_LIST;

#ifdef WORKAROUND_FOR_HW_BUG_IN_RX_COORDINATOR_FREE_LISTS	
	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList0Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList1Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList2Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList3Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;

	coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList4Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList5Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList6Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList7Idx 	= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;

	coordinatorDlmListBehavior.bitFields.rdFreeListIsIndeedFree = 0x0;

#else

	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList0Idx 	= HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList1Idx 	= HW_Q_MANAGER_RX_FREE_FW_RDS_LIST;
	coordinatorDlmListRdFreeCfg1.bitFields.rdFreeList3Idx 	= HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS;

	coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList4Idx 	= HW_Q_MANAGER_RX_FREE_MANAGEMENT_RDS_LIST;
	//coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList5Idx = ;
	//coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList6Idx = ;
	//coordinatorDlmListRdFreeCfg2.bitFields.rdFreeList7Idx = ;

	coordinatorDlmListBehavior.bitFields.rdFreeListIsIndeedFree = (0x1 << 0x0); //configure only list HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST which is FreeList0Idx since this is the only list that RD counter is decremented for in Rx Handler

#endif //WORKAROUND_FOR_HW_BUG_IN_RX_COORDINATOR_FREE_LISTS
	
	coordinatorWrapUpRxhErrorPass.bitFields.swRxhErrorBypass 			= TRUE;	 // Move the rxh_error category to rxh_error list.
	coordinatorWrapUpRxhErrorPass.bitFields.swRxhErrorPassFcsFail 		= FALSE; // Don't move MPDU to rxh_error category if FCS failed
	coordinatorWrapUpRxhErrorPass.bitFields.swRxhErrorPassRxhError 		= TRUE;	 // Move MPDU to rxh_error category if RXH_Error bit is set in MPDU descriptor 
	coordinatorWrapUpRxhErrorPass.bitFields.swRxhErrorPassRdType 		= FALSE; // Don't move MPDU to rxh_error category if RD_type in MPDU descriptor is "Drop"
	coordinatorWrapUpRxhErrorPass.bitFields.swRxhErrorPassDeAgg 		= TRUE;	 // Move MPDU to rxh_error category if there is a de-agg error in MPDU descriptor 
	coordinatorWrapUpRxhErrorPass.bitFields.swRxhErrorPassSecurityError = 0;	 // Don't move MPDU to rxh_error category if one of the security indication bits is set to error. There is a configuration bit per security status bit.

	coordinatorBaAgreementEn.bitFields.swEnBaAgreement					= TRUE;
	coordinatorBaAgreementEn.bitFields.swEnMulticastProbeReqUpdateBa	= FALSE;
	coordinatorBaAgreementEn.bitFields.swEnBeaconUpdateBa				= FALSE;
	
	for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
	{
		RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_CFG1,			coordinatorDlmListCfg1.val, 		bandId);
		RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_CFG2,			coordinatorDlmListCfg2.val, 		bandId);
		RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_RD_FREE_CFG1,	coordinatorDlmListRdFreeCfg1.val, 	bandId);
		RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_RD_FREE_CFG2,	coordinatorDlmListRdFreeCfg2.val, 	bandId);
		RegAccess_WritePerBand(REG_RX_COORDINATOR_DLM_LIST_BEHAVIOR,		coordinatorDlmListBehavior.val, 	bandId);
		RegAccess_WritePerBand(REG_RX_COORDINATOR_WRAP_UP_RXH_ERROR_PASS,	coordinatorWrapUpRxhErrorPass.val, 	bandId);
		RegAccess_WritePerBand(REG_RX_COORDINATOR_BA_AGREEMENT_EN,			coordinatorBaAgreementEn.val, 		bandId);		
	}
}


