/***************************************************************
 File:		RxHandler.c
 Module:	  	rx handler genRisc interface
 Purpose: 	holds all the implementation of the  FW - rx handler genRisc interface module
 Description:  This module contains the implementation of the  FW - rx handler genRisc interface
 			module
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "stringLibApi.h"
#include "ieee80211.h" //should be removed when the SHRAM files will be defined
#include "Pac_Api.h"
#include "RxHandler.h"
#include "RxHandler_Api.h"
#include "RxHandler_InitApi.h"
#include "RxHandler_ScratchPadApi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "PacketDescriptor.h"
#include "enet_pas.h"
#include "ShramRxH.h"
#include "HwMemoryMap.h"
#include "MacGenriscRxRegs.h"
#include "ShramSecurity.h"
#include "ShramRxDescriptors.h"
#include "ShramGenRiscMailboxes.h"
#include "HwQManager_API.h"
#include "MacHtExtensionsRegs.h"
#include "ShramPacketDescriptors.h"
#include "ShramRxDescriptorsPayload.h"
#include "mhi_umi.h"
#include "Utils_Api.h"
#include "EventsManager_api.h"
#include "ShramGenriscInitVectorStartAddr.h"
#include "PacManager_api.h"
#include "LinkAdaptation.h"
#ifdef SNIFFER_MU_ENABLED
#include "Rcr_Descriptors.h"
#include "PhyRxTdRegs.h"
#endif //SNIFFER_MU_ENABLED
#include "ShramPhyStatDb.h"
#include "PacExtrapolator_api.h"
#include "ShramStatistics.h"
#include "ChannelSwitchManager_Api.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_GENRISCS_INTERFACE
#define LOG_LOCAL_FID 2


#define SNIFFER_TEMPLATE_ALIGNMENT_SECTION_WORD_SIZE 	1

/*---------------------------------------------------------------------------------
/						Externals						
/----------------------------------------------------------------------------------*/
extern PerClientStatistics_t PerClientStatistics;
extern ChannelSwitchManagerChannelLoadDb_t channelSwitchManagerChannelLoadDb;
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void rxHandlerInterface_InitFwRxHandlerMailboxes(void);
#ifdef SNIFFER_MU_ENABLED
static void rxHandler_BuildCommonRcr(uint32 rcrAddr);
static void rxHandler_BuildUserRcr(uint32 rcrAddr, uint8 userId);
static void rxHandler_BuildRCR(void);
#endif //SNIFFER_MU_ENABLED

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
FwRxHandlerMailboxFifoInParams_t*	pFwRxHandlerMailboxFifoInParams; // pointer per band
FwRxHandlerMailboxFifoOutParams_t*	pFwRxHandlerMailboxFifoOutParams; // pointer per band


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
#define SNIFFER_MODE_GEN6_ENABLE

/*---------------------------------------------------------------------------------
/						Const tables									
/----------------------------------------------------------------------------------*/
#ifdef SNIFFER_MU_ENABLED

const TriggerFrameCpAndLtfTable_t CpAndLtfTable[GI_AND_LTF_TYPE_RESERVED] =
/*
			CP AND LTF						CP			LTF
*/
{
	{	/*0 - 1xLTF + 1.6us GI*/			0x1,		0x0,	},
	{	/*0 - 2xLTF + 1.6us GI*/			0x1,		0x1,	},
	{	/*0 - 4xLTF + 3.2us GI*/			0x2,		0x2,	},
};

const TriggerFrameManagerRuAllocTable_t  RuAllocationTable[TRIGGER_FRAME_NUM_OF_SUPPORTED_RU_ALLOCATIONS] = 
/*
		RU Allocation		Sub Band				Start RU			RU Size
*/
{
	{		/*0 */				0,						0,					0 ,	 },
	{		/*1 */				0,						1,					0 ,	 },
	{		/*2 */				0,						2,					0 ,	 },
	{		/*3 */				0,						3,					0 ,	 },
	{		/*4 */				0,						4,					0 ,	 },
	{		/*5 */				0,						5,					0 ,	 },
	{		/*6 */				0,						6,					0 ,	 },
	{		/*7 */				0,						7,					0 ,	 },
	{		/*8 */				0,						8,					0 ,	 },
	{		/*9 */				1,						0,					0 ,	 },
	{		/*10*/				1,						1,					0 ,	 },
	{		/*11*/				1,						2,					0 ,	 },
	{		/*12*/				1,						3,					0 ,	 },
	{		/*13*/				1,						4,					0 ,	 },
	{		/*14*/				1,						5,					0 ,	 },
	{		/*15*/				1,						6,					0 ,	 },
	{		/*16*/				1,						7,					0 ,	 },
	{		/*17*/				1,						8,					0 ,	 },
	{		/*18*/				1,						9,					0 ,	 },
	{		/*19*/				2,						0,					0 ,	 },
	{		/*20*/				2,						1,					0 ,	 },
	{		/*21*/				2,						2,					0 ,	 },
	{		/*22*/				2,						3,					0 ,	 },
	{		/*23*/				2,						4,					0 ,	 },
	{		/*24*/				2,						5,					0 ,	 },
	{		/*25*/				2,						6,					0 ,	 },
	{		/*26*/				2,						7,					0 ,	 },
	{		/*27*/				2,						8,					0 ,	 },
	{		/*28*/				3,						0,					0 ,	 },
	{		/*29*/				3,						1,					0 ,	 },
	{		/*30*/				3,						2,					0 ,	 },
	{		/*31*/				3,						3,					0 ,	 },
	{		/*32*/				3,						4,					0 ,	 },
	{		/*33*/				3,						5,					0 ,	 },
	{		/*34*/				3,						6,					0 ,	 },
	{		/*35*/				3,						7,					0 ,	 },
	{		/*36*/				3,						8,					0 ,	 },
	{		/*37*/				0,						0,					1 ,	 },
	{		/*38*/				0,						2,					1 ,	 },
	{		/*39*/				0,						4,					1 ,	 },
	{		/*40*/				0,						6,					1 ,	 },
	{		/*41*/				1,						0,					1 ,	 },
	{		/*42*/				1,						2,					1 ,	 },
	{		/*43*/				1,						4,					1 ,	 },
	{		/*44*/				1,						6,					1 ,	 },
	{		/*45*/				2,						0,					1 ,	 },
	{		/*46*/				2,						2,					1 ,	 },
	{		/*47*/				2,						4,					1 ,	 },
	{		/*48*/				2,						6,					1 ,	 },
	{		/*49*/				3,						0,					1 ,	 },
	{		/*50*/				3,						2,					1 ,	 },
	{		/*51*/				3,						4,					1 ,	 },
	{		/*52*/				3,						6,					1 ,	 },
	{		/*53*/				0,						0,					2 ,	 },
	{		/*54*/				0,						4,					2 ,	 },
	{		/*55*/				1,						0,					2 ,	 },
	{		/*56*/				1,						4,					2 ,	 },
	{		/*57*/				2,						0,					2 ,	 },
	{		/*58*/				2,						4,					2 ,	 },
	{		/*59*/				3,						0,					2 ,	 },
	{		/*60*/				3,						4,					2 ,	 },
	{		/*61*/				0,						0,					3 ,	 },
	{		/*62*/				1,						0,					3 ,	 },
	{		/*63*/				2,						0,					3 ,	 },
	{		/*64*/				3,						0,					3 ,	 },
	{		/*65*/				0,						0,					4 ,	 },
	{		/*66*/				2,						0,					4 ,	 },
	{		/*67*/				0,						0,					5 ,	 },
};

#endif //SNIFFER_MU_ENABLED
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
isr_RxHandler  


Description:
------------
interrupt routine that handles errors scenarions from the RXH genRisc 



Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID isr_RxHandler(void)
{
	RegMacGenriscRxGenriscUpperIrqClr_u irqClearRegister;
	RegMacGenriscRxGenriscUpperIrqStatus_u irqStatusRegister;
	uint32 eventIndex;
	uint32 idx;
	K_MSG *pMsg;
	FwRxHandlerMailboxFifoOutMessage_t *pBody1;
	RxHandler_ScratchPadApiParams_t *pScratchPadApiParams = NULL;

	irqClearRegister.val = 0;
	irqStatusRegister.val = 0;
	pScratchPadApiParams = (RxHandler_ScratchPadApiParams_t *)(B0_MAC_GENRISC_RX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_RX_HANDLER_SCRATCHPAD_API_STRUCTURE_START << 0x2));


	RegAccess_Read(REG_MAC_GENRISC_RX_GENRISC_UPPER_IRQ_STATUS, &irqStatusRegister.val);

	eventIndex = Utils_CountTrailingZeros(irqStatusRegister.val);
	irqClearRegister.bitFields.genriscUpperIrqClr = (0x1 << eventIndex);

	RegAccess_Write(REG_MAC_GENRISC_RX_GENRISC_UPPER_IRQ_CLR, irqClearRegister.val);

	switch (eventIndex)
	{	
	case RX_HANDLER_2_LOWER_MAC_INTERRUPTS_ERROR_START_BIT:
		FATAL("RX Handler Error Start");
		break;
	case RX_HANDLER_2_LOWER_MAC_INTERRUPTS_MAILBOX_FIFO_OUT_NOT_EMPTY_START_BIT:
		idx = 0;
		while (idx < FW_GENRISC_RX_HANDLER_MAILBOX_OUT_FIFO_NUMBER_OF_ENTRIES)
		{
			pMsg = RxHandlerInterface_Mailbox_Pop();
			if (pMsg)
			{
				pBody1 = (FwRxHandlerMailboxFifoOutMessage_t *)pK_MSG_DATA(pMsg);

				//send message to designated task
				switch(pBody1->messageId)
				{
				default:
					DEBUG_FATAL("Unsupported Message ID");
				}
			}
			else
			{
				break;
			}
			idx++;
		}
		break;
#ifdef SNIFFER_MU_ENABLED		
	case RX_HANDLER_2_LOWER_MAC_INTERRUPTS_SNIFFER_TRIGGER_FRAME_START_BIT:
		rxHandler_BuildRCR();
		break;
#endif //SNIFFER_MU_ENABLED	
	default:
		DEBUG_FATAL("RX Handler Unknown Event");
	}
}


/**********************************************************************************
RxHandler_Halt  


Description:
------------
stop the RXH genRisc

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint16 RxHandler_Halt(void)
{	
	RegMacGenriscRxStopOp_u haltRegister;
	RegMacGenriscRxLastPcExecuted_u lastPcExecuted;
	
	haltRegister.val = 0;


	/* Stop the genrisc */
	haltRegister.bitFields.stopOp = RXH_GENRISC_HALT_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_RX_STOP_OP, haltRegister.val);
	/* Read last pc executed */
	RegAccess_Read(REG_MAC_GENRISC_RX_LAST_PC_EXECUTED, &lastPcExecuted.val);

	return lastPcExecuted.bitFields.lastExecuted;
}



/**********************************************************************************
RxHandler_SetUdpHeader

Description:
------------
fill the Udp header structure in shared ram

Input:
-----

dutSnifferUdpHeaderParams *snifferUdpHeaderParams_p

			
Output:
-------
	none

Returns:
--------
	void - 
	
**********************************************************************************/
void RxHandler_SetUdpHeader(UMI_ADD_STREAM_REQ_t* addStreamStructurePtr)
{
	RxHandler_ScratchPadSnifferApiParams_t*	pScratchPadSnifferApiParams;
	RxHandler_ScratchPadSnifferApiParams_t localScratchPadSniffer;
	udpPacketHdr_t *pUdpPacketHdr = (udpPacketHdr_t *)&(localScratchPadSniffer.snifferTemplate[0]);

	pScratchPadSnifferApiParams = (RxHandler_ScratchPadSnifferApiParams_t *)(B0_MAC_GENRISC_RX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_RX_HANDLER_SNIFFER_TEMPLATE << 0x2));

	memset(&localScratchPadSniffer, 0x0, sizeof(RxHandler_ScratchPadSnifferApiParams_t));

	//Copy udp header template to local template
	MEMCPY(pUdpPacketHdr, addStreamStructurePtr->udpHeader, sizeof(udpPacketHdr_t));
		
	//Set the sniffer watermark (0xDEAF)
	pUdpPacketHdr->waterMark = RX_HANDLER_SNIFFER_UDP_HEADER_PADDING_BIG_ENDIAN;
	//Set RCB offset and 1 bytes alignment with zeros
	pUdpPacketHdr->rcbByteOffsetAlignment = 0x0;
	pUdpPacketHdr->reserved = 0x0; //reserved is used for bytes alignment

	//set sniffer Queue ID
	localScratchPadSniffer.snifferQueueId = (PacMngr_LoggerStationIndex << 4);
	//set pre calculated checksum
	localScratchPadSniffer.snifferPreCalculatedChecksum = SWITCH_ENDIANESS_TYPE_LONG(addStreamStructurePtr->swPreCalcChecksum);
	// set IP identification so IP fragments would start with zero
	localScratchPadSniffer.snifferIpIdentification = 0x0; 

#ifdef SNIFFER_MU_ENABLED
	// set Trigger Frame shared ptr
	localScratchPadSniffer.snifferTriggerFrameSharedPtr = (uint32)(&snifferTriggerFrameParams);
#endif //SNIFFER_MU_ENABLED
	//Copy the template to scpad
	memcpy32(pScratchPadSnifferApiParams, &localScratchPadSniffer , sizeof(RxHandler_ScratchPadSnifferApiParams_t) >> 2);
	
}	

#ifdef SNIFFER_MU_ENABLED		

/**********************************************************************************
rxHandler_BuildCommonRcr





Description:
------------
Configure Common RCR based on received trigger frame

Input:
-----

			
Output:
-------
	none

Returns:
--------
	void - 
	
**********************************************************************************/

static void rxHandler_BuildCommonRcr(uint32 rcrAddr)
{

	RcrCommon_t	*pRcr = (RcrCommon_t*) rcrAddr;
	RcrCommon_t  rcrCommon;

	TriggerFramePayloadCommon_t *pTriggerFrameCommonSection = (TriggerFramePayloadCommon_t *)&snifferTriggerFrameParams.triggerFrame;

	//zero struct before filling
	memset32(&rcrCommon, 0x0, CONVERT_BYTES_TO_WORDS(sizeof(RcrCommon_t)));

	rcrCommon.heSigASpatialReuse = 0;
	rcrCommon.peDisambiguty		 = pTriggerFrameCommonSection->peDisambiguty;
	rcrCommon.ldpcExtraSymbol	 = pTriggerFrameCommonSection->ldpcExtraSymbol;
	rcrCommon.nHeltf			 = pTriggerFrameCommonSection->nHeLtf;
	rcrCommon.stbc				 = pTriggerFrameCommonSection->stbc;
	rcrCommon.cbw				 = pTriggerFrameCommonSection->bw;
	rcrCommon.heCp				 = CpAndLtfTable[pTriggerFrameCommonSection->cpAndLtfType].cp;
	rcrCommon.heLtf				 = CpAndLtfTable[pTriggerFrameCommonSection->cpAndLtfType].ltf;
	rcrCommon.heSigABssColor	 = TRIGGER_FRAME_RCR_DEFAULT_BSS_COLOR;
	rcrCommon.lSigLength		 = pTriggerFrameCommonSection->length;

	//write to RCR area in TX circular buffer
	pRcr->word0 = rcrCommon.word0;
	pRcr->word1 = rcrCommon.word1;

	ILOG0_D("**********Rx Handler Build Common RCR rcrCommon.word0: 0x%x", rcrCommon.word0);

}


/**********************************************************************************
rxHandler_BuildUserRcr






Description:
------------
Configure per-user RCR based on received trigger frame

Input:
-----

			
Output:
-------
	none

Returns:
--------
	void - 
	
**********************************************************************************/

static void rxHandler_BuildUserRcr(uint32 rcrAddr, uint8 userId)
{
	RcrUser_t *pRcrUser = (RcrUser_t*)(rcrAddr + sizeof(RcrCommon_t) + userId*sizeof(RcrUser_t));
	RcrUser_t rcrUser;


	TriggerFramePayloadPerUser_t *pPerUserSection = (TriggerFramePayloadPerUser_t *)(((uint32)&snifferTriggerFrameParams.triggerFrame) + sizeof(TriggerFramePayloadCommon_t) + userId*sizeof(TriggerFramePayloadPerUser_t));
	
	//zero struct before filling
	memset32(&rcrUser, 0x0, CONVERT_BYTES_TO_WORDS(sizeof(RcrUser_t)));
	
	rcrUser.valid		= TRUE;
	rcrUser.subBand 	= RuAllocationTable[pPerUserSection->ruAllocation].subBand;
	rcrUser.startRu 	= RuAllocationTable[pPerUserSection->ruAllocation].startRu;
	rcrUser.ruSize 		= RuAllocationTable[pPerUserSection->ruAllocation].ruSize;
	rcrUser.ldpc		= pPerUserSection->codingType;
	rcrUser.mcs 		= pPerUserSection->mcs;
	rcrUser.nss			= pPerUserSection->ssAllocationNss;
	rcrUser.dcm			= pPerUserSection->dcm;
	rcrUser.targetRssi 	= pPerUserSection->targetRssi;
	rcrUser.txbf		= FALSE; //that was set in trigger frame

	ILOG0_D("**********Rx Handler Build RCR rcrUser.word0: 0x%x", rcrUser.word0);
	
	//update RCR
	pRcrUser->word0 = rcrUser.word0;		

}


/**********************************************************************************
rxHandler_BuildRCR




Description:
------------
Configure RCR based on received trigger frame

Input:
-----

			
Output:
-------
	none

Returns:
--------
	void - 
	
**********************************************************************************/
void rxHandler_BuildRCR(void)
{
	uint32   					   triggerFrameLength;
	uint32	 					   rcrAddr;
	BandId_e 					   bandId;
	uint8						   userId = 0;
	RegPhyRxTdPhyRxtdReg081_u	   phyRxTdPhyRxtdReg081;
	TriggerFramePayloadCommon_t    *pTriggerFrameCommonSection = (TriggerFramePayloadCommon_t *)&snifferTriggerFrameParams.triggerFrame;

	triggerFrameLength = snifferTriggerFrameParams.triggerFrameLength;

	if(pTriggerFrameCommonSection->triggerType == TRIGGER_FRAME_BASIC_TYPE)
	{ // Currently supported only Trigger Frame Basic
		
		bandId = ConfigurationManager_GetMyBand();
		// Sniffer is working only from Band0
		ASSERT(bandId == CONFIGURATION_MANAGER_BAND_0);
		rcrAddr = B0_RX_RCR_RCR_RAM_CDB0;

		rxHandler_BuildCommonRcr(rcrAddr);

		triggerFrameLength -= sizeof(TriggerFramePayloadCommon_t);

		// Assume Basic Trigger frame has no Trigger Dependent Common Info 
		while(triggerFrameLength > 0)
		{
			rxHandler_BuildUserRcr(rcrAddr, userId);
			userId ++;
			triggerFrameLength -= sizeof(TriggerFramePayloadPerUser_t);	
		}

		// write pulse to PHY
		phyRxTdPhyRxtdReg081.val = 0;
		phyRxTdPhyRxtdReg081.bitFields.txDoneDft = 1;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG081, phyRxTdPhyRxtdReg081.val);
	}
}

#endif // SNIFFER_MU_ENABLED	

/**********************************************************************************
RxHandler_Init  


Description:
------------
fill the initizlization structre of the RXH genRisc and start the RXH genRisc

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void RxHandler_Init(void)
{	
	RegMacGenriscRxStartOp_u startRegister;
	RegMacHtExtensionsGenriscRxFreeListIdx_u hostRxDescriptorListRegister;
	RxhGenriscInitVector_t *pRxhInitializationStructure;
	RegMacGenriscRxAbortCntLimit_u abortCntLimitReg;

    uint32 timeStamp = 0;
	startRegister.val = 0;
	hostRxDescriptorListRegister.val = 0;

	abortCntLimitReg.bitFields.abortCntLimit = 0x01FF;
	
	RegAccess_Write(REG_MAC_GENRISC_RX_ABORT_CNT_LIMIT, abortCntLimitReg.val);

	//init mailbox
	rxHandlerInterface_InitFwRxHandlerMailboxes();

#if defined(ENET_INC_LMAC0) 
	// Each band use a different place in SHRAM for init vector
	pRxhInitializationStructure = (RxhGenriscInitVector_t *)(&RxhInitializationStructure); 
	GenriscsInitVectorStartAddr.rxHandlerB0InitVectorPtr = (uint32)CONVERT_TO_PHYSICAL_ADDR(pRxhInitializationStructure);
#endif
	
#if defined(ENET_INC_LMAC1) 
	// Each band use a different place in SHRAM for init vector
	pRxhInitializationStructure = (RxhGenriscInitVector_t *)(&RxhInitializationStructure_B1); 
	GenriscsInitVectorStartAddr.rxHandlerB1InitVectorPtr = (uint32)CONVERT_TO_PHYSICAL_ADDR(pRxhInitializationStructure);
#endif


	memset((void*)pRxhInitializationStructure, 0, sizeof(RxhGenriscInitVector_t));
	pRxhInitializationStructure->driverFreeRdListIdx 						= HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	pRxhInitializationStructure->driverMngmtFreeRdListIdx					= HW_Q_MANAGER_RX_FREE_MANAGEMENT_RDS_LIST;
	pRxhInitializationStructure->fwFreeRdListIdx							= HW_Q_MANAGER_RX_FREE_FW_RDS_LIST;
	pRxhInitializationStructure->postPpduListIndex							= HW_Q_MANAGER_RX_MPDU_POST_PPDU_PHY_LIST;
	pRxhInitializationStructure->errorListIndex								= HW_Q_MANAGER_RX_MPDU_RXH_ERROR_LIST;
	pRxhInitializationStructure->passToRxhListIndex							= HW_Q_MANAGER_RX_MPDU_RXH_IN_LIST;
	pRxhInitializationStructure->rxLiberatorInputListIdx					= HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;	
	pRxhInitializationStructure->maxLocalCopyDataSize	 					= FW_RD_PAYLOAD_SIZE;
    pRxhInitializationStructure->categoryBitMapWord0 						= ACTION_FRAME_CATEGORY_INIT_WORD0;
    pRxhInitializationStructure->categoryBitMapWord1 						= ACTION_FRAME_CATEGORY_INIT_WORD1;
    pRxhInitializationStructure->categoryBitMapWord2 						= ACTION_FRAME_CATEGORY_INIT_WORD2;
    pRxhInitializationStructure->categoryBitMapWord3 						= ACTION_FRAME_CATEGORY_INIT_WORD3;
    pRxhInitializationStructure->categoryBitMapWord4 						= ACTION_FRAME_CATEGORY_INIT_WORD4;
    pRxhInitializationStructure->categoryBitMapWord5 						= ACTION_FRAME_CATEGORY_INIT_WORD5;
    pRxhInitializationStructure->categoryBitMapWord6 						= ACTION_FRAME_CATEGORY_INIT_WORD6;
    pRxhInitializationStructure->categoryBitMapWord7 						= ACTION_FRAME_CATEGORY_INIT_WORD7;
	pRxhInitializationStructure->wlanInterface 								= 0; 		/* TBD add API to get wlan IF*/
	pRxhInitializationStructure->loggerDestination 							= 0;
	pRxhInitializationStructure->coordinatorWrapUpListIdx					= HW_Q_MANAGER_RX_MPDU_WRAP_UP_IN_LIST;
	pRxhInitializationStructure->fwErrorListIdx								= HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS;
	pRxhInitializationStructure->badPointerRdListIdx						= HW_Q_MANAGER_RX_READY_LIST_BAD_PTR_RDS;
	pRxhInitializationStructure->mpduRetryCountStatisticsAddress			= CONVERT_TO_PHYSICAL_ADDR(PerClientStatistics.mpduRetryCount);
#ifdef	SNIFFER_MODE_GEN6_ENABLE
 	pRxhInitializationStructure->snifferRingStartAddress					= (uint32)(snifferHdrTemplatesRing);
#endif
 	pRxhInitializationStructure->mailboxFifoInStructStartAddress   			= CONVERT_TO_PHYSICAL_ADDR(pFwRxHandlerMailboxFifoInParams);
	pRxhInitializationStructure->mailboxFifoInNumOfEntries 		  			= FW_GENRISC_RX_HANDLER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES;
	pRxhInitializationStructure->mailboxFifoOutStructStartAddress  			= CONVERT_TO_PHYSICAL_ADDR(pFwRxHandlerMailboxFifoOutParams);
	pRxhInitializationStructure->mailboxFifoOutNumOfEntries 	      		= FW_GENRISC_RX_HANDLER_MAILBOX_OUT_FIFO_NUMBER_OF_ENTRIES;
	pRxhInitializationStructure->firstVapInBand 	      					= ConfigurationManager_GetFirstVapForMyBand();
	pRxhInitializationStructure->phyRxStatStaStartAddress					= (uint32)(phyRxStatusDb.staPhyRxStatus);
	pRxhInitializationStructure->phyRxStatStaEntrySize						= sizeof(stationPhyRxStatusDb_t);
	pRxhInitializationStructure->phyRxStatStaRxtimeUsageOffset				= CONVERT_WORDS_TO_BYTES(PAC_EXTRA_STA_RX_TIME_USAGE_OFFSET);
	pRxhInitializationStructure->phyRxStatStaNoiseOffset					= CONVERT_WORDS_TO_BYTES(PAC_EXTRA_STA_NOISE_OFFSET);
	pRxhInitializationStructure->mpduInAmpduStatisticsAddress				= CONVERT_TO_PHYSICAL_ADDR(PerClientStatistics.mpduInAmpdu);	 
	pRxhInitializationStructure->ampduStatisticsAddress						= CONVERT_TO_PHYSICAL_ADDR(PerClientStatistics.ampdu);
	hostRxDescriptorListRegister.bitFields.genriscRxFreeHostRdsListIdx		= HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	hostRxDescriptorListRegister.bitFields.genriscRxFreeDriverRdsListIdx	= HW_Q_MANAGER_RX_FREE_MANAGEMENT_RDS_LIST;
	hostRxDescriptorListRegister.bitFields.genriscRxFreeFwRdsListIdx		= HW_Q_MANAGER_RX_FREE_FW_RDS_LIST;
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_GENRISC_RX_FREE_LIST_IDX, hostRxDescriptorListRegister.val);
		
	/* Start the genrisc */
	startRegister.bitFields.startOp = RXH_GENRISC_START_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_RX_START_OP, startRegister.val);

#ifndef ARC_SIM
	timeStamp = GET_TSF_TIMER_LOW();
	while(RXH_GENRISC_COMPLETED_CHI_MAGIC != pRxhInitializationStructure->chiMagic)
	{
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= RXH_GENRISC_MAX_MICROSECONDS_FOR_INITIALIZATION); 
	}
#endif
}

static void rxHandlerInterface_InitFwRxHandlerMailboxes(void)
{
// Init mailbox pointers for band 0
#if defined(ENET_INC_LMAC0) 
	memset(&FwRxHandlerMailboxFifoInBuffer, 0x0, sizeof(FwRxHandlerMailboxFifoInBuffer));
	memset(&FwRxHandlerMailboxFifoOutBuffer, 0x0, sizeof(FwRxHandlerMailboxFifoOutBuffer));
	pFwRxHandlerMailboxFifoInParams = (FwRxHandlerMailboxFifoInParams_t *)(&FwRxHandlerMailboxFifoInBuffer);
	pFwRxHandlerMailboxFifoOutParams = (FwRxHandlerMailboxFifoOutParams_t *)(&FwRxHandlerMailboxFifoOutBuffer);
#endif

// Init mailbox pointers for band 1
#if defined(ENET_INC_LMAC1) 
	memset(&FwRxHandlerMailboxFifoInBuffer_B1, 0x0, sizeof(FwRxHandlerMailboxFifoInBuffer_B1));
	memset(&FwRxHandlerMailboxFifoOutBuffer_B1, 0x0, sizeof(FwRxHandlerMailboxFifoOutBuffer_B1));
	pFwRxHandlerMailboxFifoInParams = (FwRxHandlerMailboxFifoInParams_t *)(&FwRxHandlerMailboxFifoInBuffer_B1);
	pFwRxHandlerMailboxFifoOutParams = (FwRxHandlerMailboxFifoOutParams_t *)(&FwRxHandlerMailboxFifoOutBuffer_B1);
#endif
}

void RxHandlerInterface_Mailbox_Push(FwRxHandlerMailboxFifoInMessage_t *message)
{
	RegMacGenriscRxMips2GenriscIrqSet_u irqSetRegister;
	TX_INTERRUPT_SAVE_AREA;

	irqSetRegister.val = 0;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	DEBUG_ASSERT((pFwRxHandlerMailboxFifoInParams->writeCounter - pFwRxHandlerMailboxFifoInParams->readCounter) < FW_GENRISC_RX_HANDLER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES);

	memcpy32(&(pFwRxHandlerMailboxFifoInParams->messageArray[(pFwRxHandlerMailboxFifoInParams->writeCounter % FW_GENRISC_RX_HANDLER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES)]), 
				message, 
				(sizeof(FwRxHandlerMailboxFifoInMessage_t) >> 0x2));

	pFwRxHandlerMailboxFifoInParams->writeCounter++;

	/* set interrupt to genrisc */
	irqSetRegister.bitFields.mips2GenriscIrqSet = (TRUE << 0x0); //bit 0 for mailbox 0
	RegAccess_Write(REG_MAC_GENRISC_RX_MIPS2GENRISC_IRQ_SET, irqSetRegister.val);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

K_MSG *RxHandlerInterface_Mailbox_Pop(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *pMsg = NULL;
	FwRxHandlerMailboxFifoOutMessage_t *pBody;

	if (pFwRxHandlerMailboxFifoOutParams->readCounter != pFwRxHandlerMailboxFifoOutParams->writeCounter)
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(FwRxHandlerMailboxFifoOutMessage_t));
		pBody = (FwRxHandlerMailboxFifoOutMessage_t *)pK_MSG_DATA(pMsg);

		memcpy32(pBody,
			&(pFwRxHandlerMailboxFifoOutParams->messageArray[(pFwRxHandlerMailboxFifoOutParams->readCounter % FW_GENRISC_RX_HANDLER_MAILBOX_OUT_FIFO_NUMBER_OF_ENTRIES)]),
			(sizeof(FwRxHandlerMailboxFifoOutMessage_t) >> 0x2));

		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		pFwRxHandlerMailboxFifoOutParams->readCounter++;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
	
	return pMsg;
}














